"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
/**
 * Use an SQS queue as a subscription target
 */
class SqsSubscription {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
    }
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_1.Construct.isConstruct(this.queue)) {
            throw new Error(`The supplied Queue object must be an instance of Construct`);
        }
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new iam.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn }
            }
        }));
        return {
            subscriberScope: this.queue,
            subscriberId: topic.node.uniqueId,
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
        };
    }
}
exports.SqsSubscription = SqsSubscription;
//# sourceMappingURL=data:application/json;base64,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