"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
/**
 * Use a URL as a subscription target
 *
 * The message will be POSTed to the given URL.
 *
 * @see https://docs.aws.amazon.com/sns/latest/dg/sns-http-https-endpoint-as-subscriber.html
 */
class UrlSubscription {
    constructor(url, props = {}) {
        this.url = url;
        this.props = props;
        this.unresolvedUrl = core_1.Token.isUnresolved(url);
        if (!this.unresolvedUrl && !url.startsWith('http://') && !url.startsWith('https://')) {
            throw new Error('URL must start with either http:// or https://');
        }
        if (this.unresolvedUrl && props.protocol === undefined) {
            throw new Error('Must provide protocol if url is unresolved');
        }
        if (this.unresolvedUrl) {
            this.protocol = props.protocol;
        }
        else {
            this.protocol = this.url.startsWith('https:') ? sns.SubscriptionProtocol.HTTPS : sns.SubscriptionProtocol.HTTP;
        }
    }
    bind(_topic) {
        return {
            subscriberId: this.unresolvedUrl ? 'UnresolvedUrl' : this.url,
            endpoint: this.url,
            protocol: this.protocol,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
        };
    }
}
exports.UrlSubscription = UrlSubscription;
//# sourceMappingURL=data:application/json;base64,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