"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const subs = require("../lib");
// tslint:disable:object-literal-key-quotes
let stack;
let topic;
beforeEach(() => {
    stack = new core_1.Stack();
    topic = new sns.Topic(stack, 'MyTopic', {
        topicName: 'topicName',
        displayName: 'displayName'
    });
});
test('url subscription', () => {
    topic.addSubscription(new subs.UrlSubscription('https://foobar.com/'));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopichttpsfoobarcomDEA92AB5": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": "https://foobar.com/",
                    "Protocol": "https",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            }
        }
    });
});
test('url subscription (with raw delivery)', () => {
    topic.addSubscription(new subs.UrlSubscription('https://foobar.com/', {
        rawMessageDelivery: true
    }));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopichttpsfoobarcomDEA92AB5": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": "https://foobar.com/",
                    "Protocol": "https",
                    "TopicArn": { "Ref": "MyTopic86869434" },
                    "RawMessageDelivery": true
                }
            }
        }
    });
});
test('url subscription (unresolved url with protocol)', () => {
    const secret = core_1.SecretValue.secretsManager('my-secret');
    const url = secret.toString();
    topic.addSubscription(new subs.UrlSubscription(url, { protocol: sns.SubscriptionProtocol.HTTPS }));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopicUnresolvedUrlBA127FB3": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": "{{resolve:secretsmanager:my-secret:SecretString:::}}",
                    "Protocol": "https",
                    "TopicArn": { "Ref": "MyTopic86869434" },
                }
            }
        }
    });
});
test('url subscription (unknown protocol)', () => {
    expect(() => topic.addSubscription(new subs.UrlSubscription('some-protocol://foobar.com/')))
        .toThrowError(/URL must start with either http:\/\/ or https:\/\//);
});
test('url subscription (unresolved url without protocol)', () => {
    const secret = core_1.SecretValue.secretsManager('my-secret');
    const url = secret.toString();
    expect(() => topic.addSubscription(new subs.UrlSubscription(url)))
        .toThrowError(/Must provide protocol if url is unresolved/);
});
test('queue subscription', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    topic.addSubscription(new subs.SqsSubscription(queue));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyQueueE6CA6235": {
                "Type": "AWS::SQS::Queue"
            },
            "MyQueuePolicy6BBEDDAC": {
                "Type": "AWS::SQS::QueuePolicy",
                "Properties": {
                    "PolicyDocument": {
                        "Statement": [
                            {
                                "Action": "sqs:SendMessage",
                                "Condition": {
                                    "ArnEquals": {
                                        "aws:SourceArn": {
                                            "Ref": "MyTopic86869434"
                                        }
                                    }
                                },
                                "Effect": "Allow",
                                "Principal": {
                                    "Service": "sns.amazonaws.com"
                                },
                                "Resource": {
                                    "Fn::GetAtt": [
                                        "MyQueueE6CA6235",
                                        "Arn"
                                    ]
                                }
                            }
                        ],
                        "Version": "2012-10-17"
                    },
                    "Queues": [
                        {
                            "Ref": "MyQueueE6CA6235"
                        }
                    ]
                }
            },
            "MyQueueMyTopic9B00631B": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Protocol": "sqs",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    },
                    "Endpoint": {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                }
            }
        }
    });
});
test('queue subscription (with raw delivery)', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    topic.addSubscription(new subs.SqsSubscription(queue, { rawMessageDelivery: true }));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        "Endpoint": {
            "Fn::GetAtt": [
                "MyQueueE6CA6235",
                "Arn"
            ]
        },
        "Protocol": "sqs",
        "TopicArn": {
            "Ref": "MyTopic86869434"
        },
        "RawMessageDelivery": true
    });
});
test('lambda subscription', () => {
    const fction = new lambda.Function(stack, 'MyFunc', {
        runtime: lambda.Runtime.NODEJS_8_10,
        handler: 'index.handler',
        code: lambda.Code.fromInline('exports.handler = function(e, c, cb) { return cb() }')
    });
    topic.addSubscription(new subs.LambdaSubscription(fction));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyFuncServiceRole54065130": {
                "Type": "AWS::IAM::Role",
                "Properties": {
                    "AssumeRolePolicyDocument": {
                        "Statement": [
                            {
                                "Action": "sts:AssumeRole",
                                "Effect": "Allow",
                                "Principal": {
                                    "Service": "lambda.amazonaws.com"
                                }
                            }
                        ],
                        "Version": "2012-10-17"
                    },
                    "ManagedPolicyArns": [
                        {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        "Ref": "AWS::Partition"
                                    },
                                    ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                ]
                            ]
                        }
                    ]
                }
            },
            "MyFunc8A243A2C": {
                "Type": "AWS::Lambda::Function",
                "Properties": {
                    "Code": {
                        "ZipFile": "exports.handler = function(e, c, cb) { return cb() }"
                    },
                    "Handler": "index.handler",
                    "Role": {
                        "Fn::GetAtt": [
                            "MyFuncServiceRole54065130",
                            "Arn"
                        ]
                    },
                    "Runtime": "nodejs8.10"
                },
                "DependsOn": [
                    "MyFuncServiceRole54065130"
                ]
            },
            "MyFuncAllowInvokeMyTopicDD0A15B8": {
                "Type": "AWS::Lambda::Permission",
                "Properties": {
                    "Action": "lambda:InvokeFunction",
                    "FunctionName": {
                        "Fn::GetAtt": [
                            "MyFunc8A243A2C",
                            "Arn"
                        ]
                    },
                    "Principal": "sns.amazonaws.com",
                    "SourceArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            },
            "MyFuncMyTopic93B6FB00": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Protocol": "lambda",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    },
                    "Endpoint": {
                        "Fn::GetAtt": [
                            "MyFunc8A243A2C",
                            "Arn"
                        ]
                    }
                }
            }
        }
    });
});
test('email subscription', () => {
    topic.addSubscription(new subs.EmailSubscription('foo@bar.com'));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyTopicfoobarcomA344CADA": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Endpoint": "foo@bar.com",
                    "Protocol": "email",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            }
        }
    });
});
test('multiple subscriptions', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    const func = new lambda.Function(stack, 'MyFunc', {
        runtime: lambda.Runtime.NODEJS_8_10,
        handler: 'index.handler',
        code: lambda.Code.fromInline('exports.handler = function(e, c, cb) { return cb() }')
    });
    topic.addSubscription(new subs.SqsSubscription(queue));
    topic.addSubscription(new subs.LambdaSubscription(func));
    expect(stack).toMatchTemplate({
        "Resources": {
            "MyTopic86869434": {
                "Type": "AWS::SNS::Topic",
                "Properties": {
                    "DisplayName": "displayName",
                    "TopicName": "topicName"
                }
            },
            "MyQueueE6CA6235": {
                "Type": "AWS::SQS::Queue"
            },
            "MyQueuePolicy6BBEDDAC": {
                "Type": "AWS::SQS::QueuePolicy",
                "Properties": {
                    "PolicyDocument": {
                        "Statement": [
                            {
                                "Action": "sqs:SendMessage",
                                "Condition": {
                                    "ArnEquals": {
                                        "aws:SourceArn": {
                                            "Ref": "MyTopic86869434"
                                        }
                                    }
                                },
                                "Effect": "Allow",
                                "Principal": {
                                    "Service": "sns.amazonaws.com"
                                },
                                "Resource": {
                                    "Fn::GetAtt": [
                                        "MyQueueE6CA6235",
                                        "Arn"
                                    ]
                                }
                            }
                        ],
                        "Version": "2012-10-17"
                    },
                    "Queues": [
                        {
                            "Ref": "MyQueueE6CA6235"
                        }
                    ]
                }
            },
            "MyQueueMyTopic9B00631B": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Protocol": "sqs",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    },
                    "Endpoint": {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                }
            },
            "MyFuncServiceRole54065130": {
                "Type": "AWS::IAM::Role",
                "Properties": {
                    "AssumeRolePolicyDocument": {
                        "Statement": [
                            {
                                "Action": "sts:AssumeRole",
                                "Effect": "Allow",
                                "Principal": {
                                    "Service": "lambda.amazonaws.com"
                                }
                            }
                        ],
                        "Version": "2012-10-17"
                    },
                    "ManagedPolicyArns": [
                        {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        "Ref": "AWS::Partition"
                                    },
                                    ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                ]
                            ]
                        }
                    ]
                }
            },
            "MyFunc8A243A2C": {
                "Type": "AWS::Lambda::Function",
                "Properties": {
                    "Code": {
                        "ZipFile": "exports.handler = function(e, c, cb) { return cb() }"
                    },
                    "Handler": "index.handler",
                    "Role": {
                        "Fn::GetAtt": [
                            "MyFuncServiceRole54065130",
                            "Arn"
                        ]
                    },
                    "Runtime": "nodejs8.10"
                },
                "DependsOn": [
                    "MyFuncServiceRole54065130"
                ]
            },
            "MyFuncAllowInvokeMyTopicDD0A15B8": {
                "Type": "AWS::Lambda::Permission",
                "Properties": {
                    "Action": "lambda:InvokeFunction",
                    "FunctionName": {
                        "Fn::GetAtt": [
                            "MyFunc8A243A2C",
                            "Arn"
                        ]
                    },
                    "Principal": "sns.amazonaws.com",
                    "SourceArn": {
                        "Ref": "MyTopic86869434"
                    }
                }
            },
            "MyFuncMyTopic93B6FB00": {
                "Type": "AWS::SNS::Subscription",
                "Properties": {
                    "Protocol": "lambda",
                    "TopicArn": {
                        "Ref": "MyTopic86869434"
                    },
                    "Endpoint": {
                        "Fn::GetAtt": [
                            "MyFunc8A243A2C",
                            "Arn"
                        ]
                    }
                }
            }
        }
    });
});
test('throws with mutliple subscriptions of the same subscriber', () => {
    const queue = new sqs.Queue(stack, 'MyQueue');
    topic.addSubscription(new subs.SqsSubscription(queue));
    expect(() => topic.addSubscription(new subs.SqsSubscription(queue)))
        .toThrowError(/A subscription with id \"MyTopic\" already exists under the scope MyQueue/);
});
test('with filter policy', () => {
    const fction = new lambda.Function(stack, 'MyFunc', {
        runtime: lambda.Runtime.NODEJS_8_10,
        handler: 'index.handler',
        code: lambda.Code.fromInline('exports.handler = function(e, c, cb) { return cb() }')
    });
    topic.addSubscription(new subs.LambdaSubscription(fction, {
        filterPolicy: {
            color: sns.SubscriptionFilter.stringFilter({
                whitelist: ['red'],
                matchPrefixes: ['bl', 'ye'],
            }),
            size: sns.SubscriptionFilter.stringFilter({
                blacklist: ['small', 'medium'],
            }),
            price: sns.SubscriptionFilter.numericFilter({
                between: { start: 100, stop: 200 }
            })
        }
    }));
    expect(stack).toHaveResource('AWS::SNS::Subscription', {
        "FilterPolicy": {
            "color": [
                "red",
                {
                    "prefix": "bl"
                },
                {
                    "prefix": "ye"
                }
            ],
            "size": [
                {
                    "anything-but": [
                        "small",
                        "medium"
                    ]
                }
            ],
            "price": [
                {
                    "numeric": [
                        ">=",
                        100,
                        "<=",
                        200
                    ]
                }
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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