"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Runtime = exports.Test = void 0;
const crypto = require("crypto");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const synthetics_generated_1 = require("./synthetics.generated");
/**
 * Specify a test that the canary should run
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * Specify a custom test with your own code
     *
     * @returns `Test` associated with the specified Code object
     * @param options The configuration options
     */
    static custom(options) {
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
/**
 * Runtime options for a canary
 */
class Runtime {
    /**
     * @param name The name of the runtime version
     */
    constructor(name) {
        this.name = name;
    }
}
exports.Runtime = Runtime;
/**
 * `syn-1.0` includes the following:
 *
 * - Synthetics library 1.0
 * - Synthetics handler code 1.0
 * - Lambda runtime Node.js 10.x
 * - Puppeteer-core version 1.14.0
 * - The Chromium version that matches Puppeteer-core 1.14.0
 */
Runtime.SYNTHETICS_1_0 = new Runtime('syn-1.0');
/**
 * Define a new Canary
 */
class Canary extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.stringValue({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = (_b = (_a = props.artifactsBucketLocation) === null || _a === void 0 ? void 0 : _a.bucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
        });
        this.role = (_c = props.role) !== null && _c !== void 0 ? _c : this.createDefaultRole((_d = props.artifactsBucketLocation) === null || _d === void 0 ? void 0 : _d.prefix);
        const resource = new synthetics_generated_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject((_e = props.artifactsBucketLocation) === null || _e === void 0 ? void 0 : _e.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: (_f = props.startAfterCreation) !== null && _f !== void 0 ? _f : true,
            runtimeVersion: (_h = (_g = props.runtime) === null || _g === void 0 ? void 0 : _g.name) !== null && _h !== void 0 ? _h : Runtime.SYNTHETICS_1_0.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: (_j = props.failureRetentionPeriod) === null || _j === void 0 ? void 0 : _j.toDays(),
            successRetentionPeriod: (_k = props.successRetentionPeriod) === null || _k === void 0 ? void 0 : _k.toDays(),
            code: this.createCode(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricDuration(options) {
        return this.metric('Duration', options);
    }
    /**
     * Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricSuccessPercent(options) {
        return this.metric('SuccessPercent', options);
    }
    /**
     * Measure the number of failed canary runs over a given time period.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricFailed(options) {
        return this.metric('Failed', options);
    }
    /**
     * @param metricName - the name of the metric
     * @param options - configuration options for the metric
     *
     * @returns a CloudWatch metric associated with the canary.
     * @default avg over 5 minutes
     */
    metric(metricName, options) {
        return new aws_cloudwatch_1.Metric({
            metricName,
            namespace: 'CloudWatchSynthetics',
            dimensions: { CanaryName: this.canaryName },
            statistic: 'avg',
            ...options,
        }).attachTo(this);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(prefix) {
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject', 's3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: ['arn:aws:logs:::*'],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        var _a, _b, _c;
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: (_a = codeConfig.s3Location) === null || _a === void 0 ? void 0 : _a.bucketName,
            s3Key: (_b = codeConfig.s3Location) === null || _b === void 0 ? void 0 : _b.objectKey,
            s3ObjectVersion: (_c = codeConfig.s3Location) === null || _c === void 0 ? void 0 : _c.objectVersion,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        var _a, _b, _c, _d;
        return {
            durationInSeconds: String(`${(_b = (_a = props.timeToLive) === null || _a === void 0 ? void 0 : _a.toSeconds()) !== null && _b !== void 0 ? _b : 0}`),
            expression: (_d = (_c = props.schedule) === null || _c === void 0 ? void 0 : _c.expressionString) !== null && _d !== void 0 ? _d : 'rate(5 minutes)',
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = this.node.uniqueId.toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
}
exports.Canary = Canary;
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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