"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Code = exports.InlineCode = exports.AssetCode = exports.Code = void 0;
const fs = require("fs");
const path = require("path");
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * The code the canary should execute
 */
class Code {
    /**
     * Specify code inline.
     *
     * @param code The actual handler code (limited to 4KiB)
     *
     * @returns `InlineCode` with inline code.
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * Specify code from a local path. Path must include the folder structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param assetPath Either a directory or a .zip file
     *
     * @returns `AssetCode` associated with the specified path.
     */
    static fromAsset(assetPath, options) {
        return new AssetCode(assetPath, options);
    }
    /**
     * Specify code from an s3 bucket. The object in the s3 bucket must be a .zip file that contains
     * the structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     *
     * @returns `S3Code` associated with the specified S3 object.
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
}
exports.Code = Code;
/**
 * Canary code from an Asset
 */
class AssetCode extends Code {
    /**
     * @param assetPath The path to the asset file or directory.
     */
    constructor(assetPath, options) {
        super();
        this.assetPath = assetPath;
        this.options = options;
        if (!fs.existsSync(this.assetPath)) {
            throw new Error(`${this.assetPath} is not a valid path`);
        }
    }
    bind(scope, handler) {
        this.validateCanaryAsset(handler);
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.assetPath,
                ...this.options,
            });
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Validates requirements specified by the canary resource. For example, the canary code with handler `index.handler`
     * must be found in the file structure `nodejs/node_modules/index.js`.
     *
     * Requires path to be either zip file or directory.
     * Requires asset directory to have the structure 'nodejs/node_modules'.
     * Requires canary file to be directly inside node_modules folder.
     * Requires canary file name matches the handler name.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html
     *
     * @param handler the canary handler
     */
    validateCanaryAsset(handler) {
        if (path.extname(this.assetPath) !== '.zip') {
            if (!fs.lstatSync(this.assetPath).isDirectory()) {
                throw new Error(`Asset must be a .zip file or a directory (${this.assetPath})`);
            }
            const filename = `${handler.split('.')[0]}.js`;
            if (!fs.existsSync(path.join(this.assetPath, 'nodejs', 'node_modules', filename))) {
                throw new Error(`The canary resource requires that the handler is present at "nodejs/node_modules/${filename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch)`);
            }
        }
    }
}
exports.AssetCode = AssetCode;
/**
 * Canary code from an inline string.
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        if (code.length === 0) {
            throw new Error('Canary inline code cannot be empty');
        }
    }
    bind(_scope, handler) {
        if (handler !== 'index.handler') {
            throw new Error(`The handler for inline code must be "index.handler" (got "${handler}")`);
        }
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * S3 bucket path to the code zip file
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.bucket = bucket;
        this.key = key;
        this.objectVersion = objectVersion;
    }
    bind(_scope, _handler) {
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
//# sourceMappingURL=data:application/json;base64,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