"""
    pangocairocffi
    ~~~~~~~~~

    CFFI-based pangocairo bindings for Python
    See README for details.

"""
import os
import warnings
from typing import List
import ctypes.util
from .ffi_build import ffi
import cairocffi


def _dlopen(dl_name: str, generated_ffi, names: List[str]):
    """
    :param dl_name:
        The name of the dynamic library. This is also used to determine the
        environment variable name to lookup. For example, if dl_name is
        "pangocairo", this function will attempt to load "PANGOCAIRO_LOCATION".
    :param generated_ffi:
        The FFI for pangocairo, generated by pangocairocffi.
    :param names:
        An array of library names commonly used across different platforms.
    :return:
        A FFILibrary instance for the library.
    """

    # Try environment locations if set
    env_location = os.getenv(f'{dl_name.upper()}_LOCATION')
    if env_location:
        names.append(env_location)
        try:
            return generated_ffi.dlopen(env_location)
        except OSError:
            warnings.warn(f"dlopen() failed to load {dl_name} library:"
                          f" '{env_location}'. Falling back.")

    # Try various names for the same library, for different platforms.
    for name in names:
        for lib_name in (name, 'lib' + name):
            try:
                path = ctypes.util.find_library(lib_name)
                lib = generated_ffi.dlopen(path or lib_name)
                if lib:
                    return lib
            except OSError:
                pass
    raise OSError(
        f"dlopen() failed to load {dl_name} library: {' / '.join(names)}"
    )


pangocairo = _dlopen('pangocairo', ffi, ['pangocairo-1.0', 'pangocairo-1.0-0'])


# Imports are normally always put at the top of the file.
# But the wrapper API requires that the pango library be loaded first.
# Therefore, we have to disable linting rules for these lines.
# from .constants import *  # noqa

# from .pango_cairo_font_map import PangoCairoFontMap

def _get_cairo_t_from_cairo_ctx(cairo_context: cairocffi.Context):
    # Cairo does not give access to the pointer publicly. So this is the best
    # we can do.
    # noinspection PyProtectedMember
    return cairo_context._pointer  # noqa


from .render_functions import *  # noqa
from .create_update_functions import *  # noqa
from .font_functions import *  # noqa
from .font_map import PangoCairoFontMap  # noqa
