"""Tests for the Jira client module."""

from unittest.mock import MagicMock, patch

from atlassian_client.jira.client import JiraClient
from atlassian_client.jira.config import JiraConfig


def test_init_with_basic_auth():
    """Test initializing the client with basic auth configuration."""
    with patch("atlassian_client.jira.client.Jira") as mock_jira:
        config = JiraConfig(
            url="https://test.atlassian.net",
            auth_type="basic",
            username="test_username",
            api_token="test_token",
        )

        client = JiraClient(config=config)

        # Verify Jira was initialized correctly
        mock_jira.assert_called_once_with(
            url="https://test.atlassian.net",
            username="test_username",
            password="test_token",
            cloud=True,
            verify_ssl=True,
        )

        assert client.config == config
        assert client._field_ids is None
        assert client._current_user_account_id is None


def test_init_with_token_auth():
    """Test initializing the client with token auth configuration."""
    with patch("atlassian_client.jira.client.Jira") as mock_jira:
        config = JiraConfig(
            url="https://jira.example.com",
            auth_type="token",
            personal_token="test_personal_token",
            ssl_verify=False,
        )

        client = JiraClient(config=config)

        # Verify Jira was initialized correctly
        mock_jira.assert_called_once_with(
            url="https://jira.example.com",
            token="test_personal_token",
            cloud=False,
            verify_ssl=False,
        )

        assert client.config == config


def test_init_from_env():
    """Test initializing the client from environment variables."""
    with (
        patch("atlassian_client.jira.config.JiraConfig.from_env") as mock_from_env,
        patch("atlassian_client.jira.client.Jira") as mock_jira,
    ):
        mock_config = MagicMock()
        mock_config.auth_type = "basic"  # needed for the if condition
        mock_from_env.return_value = mock_config

        client = JiraClient()

        mock_from_env.assert_called_once()
        assert client.config == mock_config


def test_clean_text():
    """Test the _clean_text method."""
    with patch("atlassian_client.jira.client.Jira"):
        client = JiraClient(
            config=JiraConfig(
                url="https://test.atlassian.net",
                auth_type="basic",
                username="test_username",
                api_token="test_token",
            )
        )

        # Test with HTML
        assert client._clean_text("<p>Test content</p>") == "Test content"

        # Test with empty string
        assert client._clean_text("") == ""

        # Test with None
        assert client._clean_text(None) == ""

        # Test with spaces and newlines
        assert client._clean_text("  \n  Test with spaces  \n  ") == "Test with spaces"
