"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateHostedZone = exports.PublicHostedZone = exports.HostedZone = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * Container for records, and records contain information about how to route traffic for a specific domain, such as example.com and its subdomains (acme.example.com, zenith.example.com).
 */
class HostedZone extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added.
         */
        this.vpcs = new Array();
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.any({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs }),
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    /**
     * ARN of this hosted zone, such as arn:${Partition}:route53:::hostedzone/${Id}.
     */
    get hostedZoneArn() {
        return util_1.makeHostedZoneArn(this, this.hostedZoneId);
    }
    /**
     * Import a Route 53 hosted zone defined either outside the CDK, or in a different CDK stack.
     *
     * Use when hosted zone ID is known. Hosted zone name becomes unavailable through this query.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param hostedZoneId the ID of the hosted zone to import.
     */
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error('HostedZone.fromHostedZoneId doesn\'t support "zoneName"');
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     *
     * Use when both hosted zone ID and hosted zone name are known.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param attrs the HostedZoneAttributes (hosted zone ID and hosted zone name).
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     *
     * Requires environment, you must specify env for the stack.
     *
     * Use to easily query hosted zones.
     *
     * @see https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    static fromLookup(scope, id, query) {
        const DEFAULT_HOSTED_ZONE = {
            Id: 'DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query,
        }).value;
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        response.Id = response.Id.replace('/hostedzone/', '');
        return HostedZone.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    /**
     * Import a Route 53 public hosted zone defined either outside the CDK, or in a different CDK stack.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param publicHostedZoneId the ID of the public hosted zone to import.
     */
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this,
            });
        }
    }
    /**
     * Add another VPC to this private hosted zone.
     */
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    /**
     * Import a Route 53 private hosted zone defined either outside the CDK, or in a different CDK stack.
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param privateHostedZoneId the ID of the private hosted zone to import.
     */
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.addVpc(props.vpc);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
//# sourceMappingURL=data:application/json;base64,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