# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pydelica']

package_data = \
{'': ['*']}

install_requires = \
['defusedxml>=0.7.1,<0.8.0',
 'lxml>=4.6.2,<5.0.0',
 'pandas>=1.2.2,<2.0.0',
 'scipy>=1.6.0,<2.0.0']

setup_kwargs = {
    'name': 'pydelica',
    'version': '0.1.17',
    'description': 'Light-weight serverless API for OpenModelica',
    'long_description': '# PyDelica: Serverless OpenModelica with Python\n\n## About\n\nPyDelica is an API providing a quick and easy to use interface to compile, customise and run OpenModelica models with Python. Unlike OMPython it does not require the launch of a server session to use OMShell but rather dynamically adjusts files produced after model compilation in order to update options and parameters. The lack of server requirement means models can be run in tandem without calling multiple OMC sessions which can be very taxing on CPU. Furthermore PyDelica is able to detect errors during model runs by reading the `stderr` and `stdout` from OpenModelica and throw appropriate exceptions to terminate the program execution.\n\n## Installation\n\nTo use PyDelica you will require an installation of OpenModelica on your system, the API has been confirmed to work on both Windows and Linux, but if OM is installable on macOS it should also be possible to still use it. \n\nTo install run:\n\n```bash\npip install pydelica\n```\n\n## Getting Started\nFor the purposes of demonstration the included test model `tests/models/SineCurrent.mo` will be used.\n### PyDelica Session\nAll uses of PyDelica require an instance of the `Session` class:\n\n```python\nfrom pydelica import Session\n\nsession = Session()\n```\n\n#### Logging\nThe `Session` class has one optional argument which is used to set the log level output within OpenModelica itself. The options are based on the `-lv` flag within OM. By default the level is set to `Normal` which means no log level output.\n\nAs an example if you wanted to run with statistics logging `-lv=LOG_STATS` you would setup with the following:\n\n```python\nfrom pydelica import Session\nfrom pydelica.logger import OMLogLevel\n\nsession = Session(log_level=OMLogLevel.STATS)\n```\n\nSee the source for more options [here](https://gitlab.com/krizar/pydelica/-/blob/master/pydelica/logging.py).\n\n#### Building/Compiling Models\nBefore you can run a model you must first compile it. This is done using the `build_model` member function which takes the path to the Modelica source file.\n\n```python\nmodel_path = os.path.join(\'tests\', \'models\', \'SineCurrent.mo\')\nsession.build_model(model_path)\n```\n\nIf the required model is not top level, that is to say it exists within a module or , we can optionally specify the address within Modelica. This is also required if the required model is not the default. For example say model `A` existed within module `M`:\n\n```python\nmodel_path = \'FictionalModelFile.mo\'\nsession.build_model(model_path, \'M.A\')\n```\n\n#### Examining Parameters and Options\nWe can examine all parameters for a given model using the `get_parameters` method which will return a Python dictionary:\n\n```python\nsession.get_parameters(\'SineCurrentModel\')\n```\n\nif the parameter is unique to a single model then the model name argument can be dropped. Returning the value for a single parameter is as simple as:\n\n```python\nsession.get_parameter(<parameter-name>)\n```\n\nFor simulation options the analogous methods are `get_simulation_options` and `get_simulation_option` respectively for general case, for more specific see below.\n\n#### Setting Parameters and Options\nSet a parameter to a different value using the `set_parameter` function:\n\n```python\nsession.set_parameter(<parameter-name>, <new-value>)\n```\n\n#### Further Configuration\nThe output file type can be specified:\n\n```python\nfrom pydelica.options import OutputFormat\nsession.set_output_format(OutputFormat.CSV) # Other options are MAT and PLT\n```\n\nSet the solver:\n\n```python\nfrom pydelica.options import Solver\nsession.set_solver(Solver.DASSL)    # Other options are EULER and RUNGE_KUTTA\n```\n\nSet the time range:\n\n```python\n# Each argument is optional\nsession.set_time_range(start_time=0, stop_time=10, model_name=\'SineCurrentModel\')\n```\n\nSet tolerance:\n\n```python\n# Model name is optional\nsession.set_tolerance(tolerance=1E-9, model_name=\'SineCurrentModel\')\n```\n\nSet variable filter for outputs:\n\n```python\n# Model name is optional\nsession.set_variable_filter(filter_str=\'output*\', model_name=\'SineCurrentModel\')\n```\n\n#### Failing Simulation on Lower Assertion Level\nBy default PyDelica will look for the expression `assert | error` as an indication of a Modelica assertion\nfailure and then terminate when this is violated. You can override this behaviour using the `fail_on_assert_level`\nmethod of the `Session` class:\n\n```python\nfrom pydelica import Session\n\npd_session = Session()\npd_session.fail_on_assert_level(\'warning\')\n```\n\nPossible values ranked by order (highest at the top):\n\n|**Value**|**Description**|\n|---|---|\n|`\'never\'`|Do not throw an exception on Modelica assertion violation|\n|`\'error\'`|Default. Throw an exception on an assertion of level `AssertionLevel.error`|\n|`\'warning\'`|Throw an exception on assertion of level `AssertionLevel.warning`|\n|`\'info\'`|Throw an exception on any `assert | info` statement|\n|`\'debug\'`|Throw an exception on any `assert | debug` statement|\n\n#### Running the Simulation\nTo run the simulation use the `simulate` method. If a model name is specified then that model is run,\nelse this is the first model in the model list. At the simulation step parameter values are written to the\nXML file read by the binary before the model binary is executed.\n\n```python\n# Model name is optional, verbosity is also optional and overwrites that of the session\nsession.simulate(model_name=\'SineCurrentModel\', verbosity=OMLogLevel.DEBUG)\n```\n\n#### Retrieving Results\nTo view the results use the `get_solutions` method which will return a python dictionary containing\nthe solutions for all models after a model run:\n\n```python\nsolutions = session.get_solutions()\n```\nThe variables for each model are stored as a Pandas dataframe.\n\n#### Using Alternative Libraries\n\n**NOTE:** Currently only works in WSL on Windows machines.\n\nYou can use an alternative library version via the `use_library` method:\n```python3\nsession.use_library("Modelica", "3.2.3")\n```\nyou can also optionally specify the location of this library:\n```python3\nsession.use_library("Modelica", "3.2.3", library_directory="/home/user/my_om_libraries")\n```\n\n## Troubleshooting\n\n### PyDelica cannot find OMC\nPyDelica relies on either locating OMC on UNIX using the `which` command, or in the case of Windows using the `OPENMODELICAHOME` environment variable. Ensure at least one of these is available after installating OpenModelica.\n\n## Docker\nPyDelica is available on Docker, images can be found [here]()\n',
    'author': 'Kristian Zarebski',
    'author_email': 'krizar312@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/krizar/pydelica',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.1,<4.0.0',
}


setup(**setup_kwargs)
