#!/usr/bin/env python

"""Tests for the `heart_rate` functions in fitnick."""

import datetime
import os

from decimal import Decimal

import pytest

from fitnick.database.database import Database
from fitnick.heart_rate.models import heart_daily_table, heart_intraday_table
from fitnick.heart_rate.time_series import HeartRateTimeSeries
from fitnick.time_series import plot_rolling_average

EXPECTED_ROWS = [
    ('Out of Range', Decimal('1267.00000'), datetime.date(2020, 9, 5), Decimal('2086.83184'), 68),
    ('Fat Burn', Decimal('115.00000'), datetime.date(2020, 9, 5), Decimal('721.58848'), 68),
    ('Cardio', Decimal('3.00000'), datetime.date(2020, 9, 5), Decimal('30.91792'), 68),
    ('Peak', Decimal('0.00000'), datetime.date(2020, 9, 5), Decimal('0.00000'), 68)
]

EXPECTED_PERIOD_ROWS = [
    ('Out of Range', Decimal('1267.00000'), datetime.date(2020, 9, 5), Decimal('2086.83184'), 68),
    ('Cardio', Decimal('3.00000'), datetime.date(2020, 9, 5), Decimal('30.91792'), 68),
    ('Out of Range', Decimal('1210.00000'), datetime.date(2020, 9, 6), Decimal('1967.73432'), 69),
    ('Cardio', Decimal('2.00000'), datetime.date(2020, 9, 6), Decimal('24.22690'), 69),
    ('Fat Burn', Decimal('115.00000'), datetime.date(2020, 9, 5), Decimal('721.58848'), 68),
    ('Peak', Decimal('0.00000'), datetime.date(2020, 9, 5), Decimal('0.00000'), 68),
    ('Fat Burn', Decimal('178.00000'), datetime.date(2020, 9, 6), Decimal('950.84207'), 69),
    ('Peak', Decimal('0.00000'), datetime.date(2020, 9, 6), Decimal('0.00000'), 69)
]

EXPECTED_INTRADAY_ROWS = [
    ('2020-10-05', '00:00:00', 77),
    ('2020-10-05', '00:01:00', 77),
    ('2020-10-05', '00:02:00', 77),
    ('2020-10-05', '00:03:00', 75),
    ('2020-10-05', '00:04:00', 73),
    ('2020-10-05', '00:05:00', 71),
    ('2020-10-05', '00:06:00', 76),
    ('2020-10-05', '00:07:00', 75),
    ('2020-10-05', '00:08:00', 73),
    ('2020-10-05', '00:09:00', 69)
]

EXPECTED_DATA = {'activities-heart': [
    {'dateTime': '2020-09-05', 'value': {'customHeartRateZones': [], 'heartRateZones': [
        {'caloriesOut': 2086.83184, 'max': 96, 'min': 30, 'minutes': 1267, 'name': 'Out of Range'},
        {'caloriesOut': 721.58848, 'max': 134, 'min': 96, 'minutes': 115, 'name': 'Fat Burn'},
        {'caloriesOut': 30.91792, 'max': 163, 'min': 134, 'minutes': 3, 'name': 'Cardio'},
        {'caloriesOut': 0, 'max': 220, 'min': 163, 'minutes': 0, 'name': 'Peak'}
    ], 'restingHeartRate': 68}}],
}

EXPECTED_INTRADAY_DATA = {'activities-heart-intraday': {'dataset': [
    {'time': '00:00:00', 'value': 77}, {'time': '00:01:00', 'value': 77}, {'time': '00:02:00', 'value': 77},{'time': '00:03:00', 'value': 75}, {'time': '00:04:00', 'value': 73}, {'time': '00:05:00', 'value': 71}, {'time': '00:06:00', 'value': 76}, {'time': '00:07:00', 'value': 75}, {'time': '00:08:00', 'value': 73}, {'time': '00:09:00', 'value': 69}, {'time': '00:10:00', 'value': 67}, {'time': '00:11:00', 'value': 72}, {'time': '00:12:00', 'value': 73}, {'time': '00:13:00', 'value': 72}, {'time': '00:14:00', 'value': 74}, {'time': '00:15:00', 'value': 75}, {'time': '00:16:00', 'value': 72}, {'time': '00:17:00', 'value': 75}, {'time': '00:18:00', 'value': 74}, {'time': '00:19:00', 'value': 73}, {'time': '00:20:00', 'value': 71}, {'time': '00:21:00', 'value': 74}, {'time': '00:22:00', 'value': 73}, {'time': '00:23:00', 'value': 74}, {'time': '00:24:00', 'value': 76}, {'time': '00:25:00', 'value': 93}, {'time': '00:26:00', 'value': 100}, {'time': '00:27:00', 'value': 99}, {'time': '00:28:00', 'value': 97}, {'time': '00:29:00', 'value': 93}, {'time': '00:30:00', 'value': 98}, {'time': '00:31:00', 'value': 94}, {'time': '00:32:00', 'value': 93}, {'time': '00:33:00', 'value': 103}, {'time': '00:34:00', 'value': 101}, {'time': '00:35:00', 'value': 100}, {'time': '00:36:00', 'value': 96}, {'time': '00:37:00', 'value': 100}, {'time': '00:38:00', 'value': 93}, {'time': '00:39:00', 'value': 105}, {'time': '00:40:00', 'value': 100}, {'time': '00:41:00', 'value': 107}, {'time': '00:42:00', 'value': 104}, {'time': '00:43:00', 'value': 115}, {'time': '00:44:00', 'value': 92}, {'time': '00:45:00', 'value': 91}, {'time': '00:46:00', 'value': 86}, {'time': '00:47:00', 'value': 79}, {'time': '00:48:00', 'value': 78}, {'time': '00:49:00', 'value': 84}, {'time': '01:17:00', 'value': 97}, {'time': '01:18:00', 'value': 83}, {'time': '01:19:00', 'value': 80}, {'time': '01:20:00', 'value': 86}, {'time': '01:21:00', 'value': 86}, {'time': '01:22:00', 'value': 85}, {'time': '01:23:00', 'value': 88}, {'time': '01:24:00', 'value': 83}, {'time': '01:25:00', 'value': 80}, {'time': '01:26:00', 'value': 84}, {'time': '01:27:00', 'value': 84}, {'time': '01:28:00', 'value': 84}, {'time': '01:29:00', 'value': 78}, {'time': '01:30:00', 'value': 75}, {'time': '01:31:00', 'value': 69}, {'time': '01:32:00', 'value': 69}, {'time': '01:33:00', 'value': 73}, {'time': '01:34:00', 'value': 72}, {'time': '01:35:00', 'value': 74}, {'time': '01:36:00', 'value': 73}, {'time': '01:37:00', 'value': 74}, {'time': '01:38:00', 'value': 73}, {'time': '01:39:00', 'value': 68}, {'time': '01:40:00', 'value': 72}, {'time': '01:41:00', 'value': 70}, {'time': '01:42:00', 'value': 71}, {'time': '01:43:00', 'value': 72}, {'time': '01:44:00', 'value': 76}, {'time': '01:45:00', 'value': 78}, {'time': '01:46:00', 'value': 75}, {'time': '01:47:00', 'value': 74}, {'time': '01:48:00', 'value': 84}, {'time': '01:49:00', 'value': 87}, {'time': '01:50:00', 'value': 84}, {'time': '01:51:00', 'value': 74}, {'time': '01:52:00', 'value': 71}, {'time': '01:53:00', 'value': 71}, {'time': '01:54:00', 'value': 63}, {'time': '01:55:00', 'value': 66}, {'time': '01:56:00', 'value': 68}, {'time': '01:57:00', 'value': 68}, {'time': '01:58:00', 'value': 67}, {'time': '01:59:00', 'value': 69}, {'time': '02:00:00', 'value': 73}, {'time': '02:01:00', 'value': 84}, {'time': '02:02:00', 'value': 77}, {'time': '02:03:00', 'value': 83}, {'time': '02:04:00', 'value': 87}, {'time': '02:05:00', 'value': 82}, {'time': '02:06:00', 'value': 78}, {'time': '02:07:00', 'value': 73}, {'time': '02:08:00', 'value': 73}, {'time': '02:09:00', 'value': 75}, {'time': '02:10:00', 'value': 73}, {'time': '02:11:00', 'value': 74}, {'time': '02:12:00', 'value': 70}, {'time': '02:13:00', 'value': 72}, {'time': '02:14:00', 'value': 71}, {'time': '02:15:00', 'value': 72}, {'time': '02:16:00', 'value': 71}, {'time': '02:17:00', 'value': 71}, {'time': '02:18:00', 'value': 73}, {'time': '02:19:00', 'value': 72}, {'time': '02:20:00', 'value': 70}, {'time': '02:21:00', 'value': 71}, {'time': '02:22:00', 'value': 73}, {'time': '02:23:00', 'value': 71}, {'time': '02:24:00', 'value': 71}, {'time': '02:25:00', 'value': 72}, {'time': '02:26:00', 'value': 72}, {'time': '02:27:00', 'value': 68}, {'time': '02:28:00', 'value': 68}, {'time': '02:29:00', 'value': 68}, {'time': '02:30:00', 'value': 70}, {'time': '02:31:00', 'value': 69}, {'time': '02:32:00', 'value': 70}, {'time': '02:33:00', 'value': 69}, {'time': '02:34:00', 'value': 72}, {'time': '02:35:00', 'value': 69}, {'time': '02:36:00', 'value': 67}, {'time': '02:37:00', 'value': 69}, {'time': '02:38:00', 'value': 68}, {'time': '02:39:00', 'value': 69}, {'time': '02:40:00', 'value': 70}, {'time': '02:41:00', 'value': 67}, {'time': '02:42:00', 'value': 68}, {'time': '02:43:00', 'value': 67}, {'time': '02:44:00', 'value': 68}, {'time': '02:45:00', 'value': 70}, {'time': '02:46:00', 'value': 69}, {'time': '02:47:00', 'value': 70}, {'time': '02:48:00', 'value': 73}, {'time': '02:49:00', 'value': 71}, {'time': '02:50:00', 'value': 70}, {'time': '02:51:00', 'value': 71}, {'time': '02:52:00', 'value': 72}, {'time': '02:53:00', 'value': 68}, {'time': '02:54:00', 'value': 66}, {'time': '02:55:00', 'value': 67}, {'time': '02:56:00', 'value': 69}, {'time': '02:57:00', 'value': 70}, {'time': '02:58:00', 'value': 69}, {'time': '02:59:00', 'value': 70}, {'time': '03:00:00', 'value': 69}, {'time': '03:01:00', 'value': 70}, {'time': '03:02:00', 'value': 69}, {'time': '03:03:00', 'value': 69}, {'time': '03:04:00', 'value': 68}, {'time': '03:05:00', 'value': 69}, {'time': '03:06:00', 'value': 68}, {'time': '03:07:00', 'value': 69}, {'time': '03:08:00', 'value': 71}, {'time': '03:09:00', 'value': 73}, {'time': '03:10:00', 'value': 71}, {'time': '03:11:00', 'value': 69}, {'time': '03:12:00', 'value': 68}, {'time': '03:13:00', 'value': 69}, {'time': '03:14:00', 'value': 69}, {'time': '03:15:00', 'value': 69}, {'time': '03:16:00', 'value': 65}, {'time': '03:17:00', 'value': 70}, {'time': '03:18:00', 'value': 68}, {'time': '03:19:00', 'value': 68}, {'time': '03:20:00', 'value': 68}, {'time': '03:21:00', 'value': 69}, {'time': '03:22:00', 'value': 70}, {'time': '03:23:00', 'value': 67}, {'time': '03:24:00', 'value': 69}, {'time': '03:25:00', 'value': 70}, {'time': '03:26:00', 'value': 71}, {'time': '03:27:00', 'value': 73}, {'time': '03:28:00', 'value': 73}, {'time': '03:29:00', 'value': 70}, {'time': '03:30:00', 'value': 67}, {'time': '03:31:00', 'value': 67}, {'time': '03:32:00', 'value': 67}, {'time': '03:33:00', 'value': 69}, {'time': '03:34:00', 'value': 70}, {'time': '03:35:00', 'value': 72}, {'time': '03:36:00', 'value': 71}, {'time': '03:37:00', 'value': 72}, {'time': '03:38:00', 'value': 73}, {'time': '03:39:00', 'value': 72}, {'time': '03:40:00', 'value': 72}, {'time': '03:41:00', 'value': 73}, {'time': '03:42:00', 'value': 73}, {'time': '03:43:00', 'value': 74}, {'time': '03:44:00', 'value': 74}, {'time': '03:45:00', 'value': 72}, {'time': '03:46:00', 'value': 69}, {'time': '03:47:00', 'value': 71}, {'time': '03:48:00', 'value': 72}, {'time': '03:49:00', 'value': 73}, {'time': '03:50:00', 'value': 74}, {'time': '03:51:00', 'value': 75}, {'time': '03:52:00', 'value': 68}, {'time': '03:53:00', 'value': 68}, {'time': '03:54:00', 'value': 68}, {'time': '03:55:00', 'value': 69}, {'time': '03:56:00', 'value': 68}, {'time': '03:57:00', 'value': 67}, {'time': '03:58:00', 'value': 66}, {'time': '03:59:00', 'value': 65},
    {'time': '04:00:00', 'value': 65}, {'time': '04:01:00', 'value': 65}, {'time': '04:02:00', 'value': 66}, {'time': '04:03:00', 'value': 65}, {'time': '04:04:00', 'value': 65}, {'time': '04:05:00', 'value': 65}, {'time': '04:06:00', 'value': 65}, {'time': '04:07:00', 'value': 64}, {'time': '04:08:00', 'value': 62}, {'time': '04:09:00', 'value': 64}, {'time': '04:10:00', 'value': 64}, {'time': '04:11:00', 'value': 64}, {'time': '04:12:00', 'value': 63}, {'time': '04:13:00', 'value': 64}, {'time': '04:14:00', 'value': 64}, {'time': '04:15:00', 'value': 63}, {'time': '04:16:00', 'value': 63}, {'time': '04:17:00', 'value': 63}, {'time': '04:18:00', 'value': 63}, {'time': '04:19:00', 'value': 63}, {'time': '04:20:00', 'value': 65}, {'time': '04:21:00', 'value': 63}, {'time': '04:22:00', 'value': 65}, {'time': '04:23:00', 'value': 65}, {'time': '04:24:00', 'value': 66}, {'time': '04:25:00', 'value': 69}, {'time': '04:26:00', 'value': 71}, {'time': '04:27:00', 'value': 62}, {'time': '04:28:00', 'value': 61}, {'time': '04:29:00', 'value': 58}, {'time': '04:30:00', 'value': 61}, {'time': '04:31:00', 'value': 59}, {'time': '04:32:00', 'value': 58}, {'time': '04:33:00', 'value': 63}, {'time': '04:34:00', 'value': 62}, {'time': '04:35:00', 'value': 62}, {'time': '04:36:00', 'value': 59}, {'time': '04:37:00', 'value': 63}, {'time': '04:38:00', 'value': 57}, {'time': '04:39:00', 'value': 61}, {'time': '04:40:00', 'value': 59}, {'time': '04:41:00', 'value': 61}, {'time': '04:42:00', 'value': 58}, {'time': '04:43:00', 'value': 61}, {'time': '04:44:00', 'value': 58}, {'time': '04:45:00', 'value': 58}, {'time': '04:46:00', 'value': 59}, {'time': '04:47:00', 'value': 60}, {'time': '04:48:00', 'value': 58}, {'time': '04:49:00', 'value': 60}, {'time': '04:50:00', 'value': 68}, {'time': '04:51:00', 'value': 56}, {'time': '04:52:00', 'value': 54}, {'time': '04:53:00', 'value': 57}, {'time': '04:54:00', 'value': 55}, {'time': '04:55:00', 'value': 57}, {'time': '04:56:00', 'value': 58}, {'time': '04:57:00', 'value': 55}, {'time': '04:58:00', 'value': 55}, {'time': '04:59:00', 'value': 56}, {'time': '05:00:00', 'value': 55}, {'time': '05:01:00', 'value': 55}, {'time': '05:02:00', 'value': 59}, {'time': '05:03:00', 'value': 58}, {'time': '05:04:00', 'value': 55}, {'time': '05:05:00', 'value': 61}, {'time': '05:06:00', 'value': 58}, {'time': '05:07:00', 'value': 54}, {'time': '05:08:00', 'value': 56}, {'time': '05:09:00', 'value': 59}, {'time': '05:10:00', 'value': 61}, {'time': '05:11:00', 'value': 59}, {'time': '05:12:00', 'value': 58}, {'time': '05:13:00', 'value': 56}, {'time': '05:14:00', 'value': 57}, {'time': '05:15:00', 'value': 57}, {'time': '05:16:00', 'value': 57}, {'time': '05:17:00', 'value': 58}, {'time': '05:18:00', 'value': 57}, {'time': '05:19:00', 'value': 56}, {'time': '05:20:00', 'value': 57}, {'time': '05:21:00', 'value': 57}, {'time': '05:22:00', 'value': 59}, {'time': '05:23:00', 'value': 60}, {'time': '05:26:00', 'value': 60}, {'time': '05:27:00', 'value': 60}, {'time': '05:28:00', 'value': 60}, {'time': '05:42:00', 'value': 60}, {'time': '05:43:00', 'value': 60}, {'time': '05:44:00', 'value': 77}, {'time': '05:45:00', 'value': 60}, {'time': '05:46:00', 'value': 60}, {'time': '05:47:00', 'value': 61}, {'time': '05:48:00', 'value': 61}, {'time': '05:49:00', 'value': 62}, {'time': '05:50:00', 'value': 62}, {'time': '05:51:00', 'value': 62}, {'time': '05:52:00', 'value': 61}, {'time': '05:53:00', 'value': 62}, {'time': '05:54:00', 'value': 61}, {'time': '05:55:00', 'value': 62}, {'time': '05:56:00', 'value': 62}, {'time': '05:57:00', 'value': 63}, {'time': '05:58:00', 'value': 64}, {'time': '05:59:00', 'value': 63}, {'time': '06:00:00', 'value': 63}, {'time': '06:01:00', 'value': 64}, {'time': '06:02:00', 'value': 65}, {'time': '06:03:00', 'value': 65}, {'time': '06:04:00', 'value': 64}, {'time': '06:05:00', 'value': 66}, {'time': '06:06:00', 'value': 60}, {'time': '06:07:00', 'value': 57}, {'time': '06:08:00', 'value': 58}, {'time': '06:09:00', 'value': 57}, {'time': '06:10:00', 'value': 57}, {'time': '06:11:00', 'value': 58}, {'time': '06:12:00', 'value': 61}, {'time': '06:13:00', 'value': 60}, {'time': '06:14:00', 'value': 61}, {'time': '06:15:00', 'value': 61}, {'time': '06:16:00', 'value': 62}, {'time': '06:17:00', 'value': 60}, {'time': '06:18:00', 'value': 63}, {'time': '06:19:00', 'value': 64}, {'time': '06:20:00', 'value': 57}, {'time': '06:21:00', 'value': 56}, {'time': '06:22:00', 'value': 59}, {'time': '06:23:00', 'value': 58}, {'time': '06:24:00', 'value': 65}, {'time': '06:25:00', 'value': 61}, {'time': '06:26:00', 'value': 59}, {'time': '06:27:00', 'value': 62}, {'time': '06:28:00', 'value': 56}, {'time': '06:29:00', 'value': 58}, {'time': '06:30:00', 'value': 60}, {'time': '06:31:00', 'value': 59}, {'time': '06:32:00', 'value': 59}, {'time': '06:33:00', 'value': 59}, {'time': '06:34:00', 'value': 59}, {'time': '06:35:00', 'value': 59}, {'time': '06:36:00', 'value': 54}, {'time': '06:37:00', 'value': 60}, {'time': '06:38:00', 'value': 61}, {'time': '06:39:00', 'value': 59}, {'time': '06:40:00', 'value': 56}, {'time': '06:41:00', 'value': 59}, {'time': '06:42:00', 'value': 58}, {'time': '06:43:00', 'value': 58}, {'time': '06:44:00', 'value': 57}, {'time': '06:45:00', 'value': 59}, {'time': '06:46:00', 'value': 57}, {'time': '06:47:00', 'value': 59}, {'time': '06:48:00', 'value': 59}, {'time': '06:49:00', 'value': 59}, {'time': '06:50:00', 'value': 61}, {'time': '06:51:00', 'value': 61}, {'time': '06:52:00', 'value': 61}, {'time': '06:53:00', 'value': 58}, {'time': '06:54:00', 'value': 61}, {'time': '06:55:00', 'value': 56}, {'time': '06:56:00', 'value': 57}, {'time': '06:57:00', 'value': 57}, {'time': '06:58:00', 'value': 58}, {'time': '06:59:00', 'value': 58}, {'time': '07:00:00', 'value': 57}, {'time': '07:01:00', 'value': 57}, {'time': '07:02:00', 'value': 58}, {'time': '07:03:00', 'value': 58}, {'time': '07:04:00', 'value': 58}, {'time': '07:05:00', 'value': 57}, {'time': '07:06:00', 'value': 59}, {'time': '07:07:00', 'value': 59}, {'time': '07:08:00', 'value': 57}, {'time': '07:09:00', 'value': 57}, {'time': '07:10:00', 'value': 58}, {'time': '07:11:00', 'value': 57}, {'time': '07:12:00', 'value': 59}, {'time': '07:13:00', 'value': 58}, {'time': '07:14:00', 'value': 52}, {'time': '07:15:00', 'value': 59}, {'time': '07:16:00', 'value': 55}, {'time': '07:17:00', 'value': 57}, {'time': '07:18:00', 'value': 57}, {'time': '07:19:00', 'value': 60}, {'time': '07:20:00', 'value': 59}, {'time': '07:21:00', 'value': 54}, {'time': '07:22:00', 'value': 56}, {'time': '07:23:00', 'value': 58}, {'time': '07:24:00', 'value': 57}, {'time': '07:25:00', 'value': 56}, {'time': '07:26:00', 'value': 59}, {'time': '07:27:00', 'value': 53}, {'time': '07:28:00', 'value': 57}, {'time': '07:29:00', 'value': 59}, {'time': '07:30:00', 'value': 88}, {'time': '07:31:00', 'value': 58}, {'time': '07:32:00', 'value': 57}, {'time': '07:33:00', 'value': 58}, {'time': '07:34:00', 'value': 59}, {'time': '07:35:00', 'value': 57}, {'time': '07:36:00', 'value': 53}, {'time': '07:37:00', 'value': 57}, {'time': '07:38:00', 'value': 58}, {'time': '07:39:00', 'value': 60}, {'time': '07:40:00', 'value': 57}, {'time': '07:41:00', 'value': 55}, {'time': '07:42:00', 'value': 55}, {'time': '07:43:00', 'value': 57}, {'time': '07:44:00', 'value': 51}, {'time': '07:45:00', 'value': 56}, {'time': '07:46:00', 'value': 56}, {'time': '07:47:00', 'value': 55}, {'time': '07:48:00', 'value': 57}, {'time': '07:49:00', 'value': 61}, {'time': '07:50:00', 'value': 59}, {'time': '07:51:00', 'value': 55}, {'time': '07:52:00', 'value': 57}, {'time': '07:53:00', 'value': 56}, {'time': '07:54:00', 'value': 59}, {'time': '07:55:00', 'value': 59}, {'time': '07:56:00', 'value': 57}, {'time': '07:57:00', 'value': 58}, {'time': '07:58:00', 'value': 58}, {'time': '07:59:00', 'value': 57},
    {'time': '08:00:00', 'value': 58}, {'time': '08:01:00', 'value': 62}, {'time': '08:02:00', 'value': 60}, {'time': '08:03:00', 'value': 61}, {'time': '08:04:00', 'value': 62}, {'time': '08:05:00', 'value': 69}, {'time': '08:06:00', 'value': 72}, {'time': '08:07:00', 'value': 75}, {'time': '08:08:00', 'value': 80}, {'time': '08:09:00', 'value': 90}, {'time': '08:10:00', 'value': 95}, {'time': '08:11:00', 'value': 98}, {'time': '08:12:00', 'value': 102}, {'time': '08:13:00', 'value': 79}, {'time': '08:14:00', 'value': 63}, {'time': '08:15:00', 'value': 64}, {'time': '08:16:00', 'value': 65}, {'time': '08:17:00', 'value': 66}, {'time': '08:18:00', 'value': 66}, {'time': '08:19:00', 'value': 66}, {'time': '08:20:00', 'value': 68}, {'time': '08:21:00', 'value': 68}, {'time': '08:22:00', 'value': 69}, {'time': '08:23:00', 'value': 70}, {'time': '08:24:00', 'value': 70}, {'time': '08:25:00', 'value': 69}, {'time': '08:26:00', 'value': 69}, {'time': '08:27:00', 'value': 70}, {'time': '08:28:00', 'value': 71}, {'time': '08:29:00', 'value': 70}, {'time': '08:30:00', 'value': 70}, {'time': '08:31:00', 'value': 71}, {'time': '08:32:00', 'value': 71}, {'time': '08:33:00', 'value': 69}, {'time': '08:34:00', 'value': 71}, {'time': '08:35:00', 'value': 70}, {'time': '08:36:00', 'value': 71}, {'time': '08:37:00', 'value': 73}, {'time': '08:38:00', 'value': 73}, {'time': '08:39:00', 'value': 72}, {'time': '08:40:00', 'value': 73}, {'time': '08:41:00', 'value': 72}, {'time': '08:42:00', 'value': 74}, {'time': '08:43:00', 'value': 73}, {'time': '08:44:00', 'value': 74}, {'time': '08:45:00', 'value': 72}, {'time': '08:46:00', 'value': 76}, {'time': '08:47:00', 'value': 61}, {'time': '08:48:00', 'value': 66}, {'time': '08:49:00', 'value': 62}, {'time': '08:50:00', 'value': 63}, {'time': '08:51:00', 'value': 63}, {'time': '08:52:00', 'value': 65}, {'time': '08:53:00', 'value': 65}, {'time': '08:54:00', 'value': 66}, {'time': '08:55:00', 'value': 68}, {'time': '08:56:00', 'value': 70}, {'time': '08:57:00', 'value': 61}, {'time': '08:58:00', 'value': 62}, {'time': '08:59:00', 'value': 61}, {'time': '09:00:00', 'value': 61}, {'time': '09:01:00', 'value': 63}, {'time': '09:02:00', 'value': 65}, {'time': '09:03:00', 'value': 63}, {'time': '09:04:00', 'value': 64}, {'time': '09:05:00', 'value': 66}, {'time': '09:06:00', 'value': 66}, {'time': '09:07:00', 'value': 66}, {'time': '09:08:00', 'value': 67}, {'time': '09:09:00', 'value': 64}, {'time': '09:10:00', 'value': 58}, {'time': '09:11:00', 'value': 64}, {'time': '09:12:00', 'value': 62}, {'time': '09:13:00', 'value': 64}, {'time': '09:14:00', 'value': 65}, {'time': '09:15:00', 'value': 64}, {'time': '09:16:00', 'value': 62}, {'time': '09:17:00', 'value': 65}, {'time': '09:18:00', 'value': 64}, {'time': '09:19:00', 'value': 60}, {'time': '09:20:00', 'value': 64}, {'time': '09:21:00', 'value': 63}, {'time': '09:22:00', 'value': 61}, {'time': '09:23:00', 'value': 64}, {'time': '09:24:00', 'value': 63}, {'time': '09:25:00', 'value': 66}, {'time': '09:26:00', 'value': 61}, {'time': '09:27:00', 'value': 60}, {'time': '09:28:00', 'value': 61}, {'time': '09:29:00', 'value': 64}, {'time': '09:30:00', 'value': 62}, {'time': '09:31:00', 'value': 61}, {'time': '09:32:00', 'value': 62}, {'time': '09:33:00', 'value': 62}, {'time': '09:34:00', 'value': 62}, {'time': '09:35:00', 'value': 67}, {'time': '09:36:00', 'value': 64}, {'time': '09:37:00', 'value': 61}, {'time': '09:38:00', 'value': 64}, {'time': '09:39:00', 'value': 64}, {'time': '09:40:00', 'value': 64}, {'time': '09:41:00', 'value': 62}, {'time': '09:42:00', 'value': 63}, {'time': '09:43:00', 'value': 64}, {'time': '09:44:00', 'value': 59}, {'time': '09:45:00', 'value': 61}, {'time': '09:46:00', 'value': 60}, {'time': '09:47:00', 'value': 65}, {'time': '09:48:00', 'value': 64}, {'time': '09:49:00', 'value': 59}, {'time': '09:50:00', 'value': 58}, {'time': '09:51:00', 'value': 65}, {'time': '09:52:00', 'value': 61}, {'time': '09:53:00', 'value': 66}, {'time': '09:54:00', 'value': 62}, {'time': '09:55:00', 'value': 62}, {'time': '09:56:00', 'value': 58}, {'time': '09:57:00', 'value': 60}, {'time': '09:58:00', 'value': 62}, {'time': '09:59:00', 'value': 58}, {'time': '10:00:00', 'value': 62}, {'time': '10:01:00', 'value': 65}, {'time': '10:02:00', 'value': 63}, {'time': '10:03:00', 'value': 61}, {'time': '10:04:00', 'value': 63}, {'time': '10:05:00', 'value': 61}, {'time': '10:06:00', 'value': 59}, {'time': '10:07:00', 'value': 61}, {'time': '10:08:00', 'value': 61}, {'time': '10:09:00', 'value': 63}, {'time': '10:10:00', 'value': 58}, {'time': '10:11:00', 'value': 61}, {'time': '10:12:00', 'value': 61}, {'time': '10:13:00', 'value': 60}, {'time': '10:14:00', 'value': 69}, {'time': '10:15:00', 'value': 67}, {'time': '10:16:00', 'value': 66}, {'time': '10:17:00', 'value': 62}, {'time': '10:18:00', 'value': 61}, {'time': '10:19:00', 'value': 62}, {'time': '10:20:00', 'value': 64}, {'time': '10:21:00', 'value': 62}, {'time': '10:22:00', 'value': 63}, {'time': '10:23:00', 'value': 61}, {'time': '10:24:00', 'value': 64}, {'time': '10:25:00', 'value': 62}, {'time': '10:26:00', 'value': 64}, {'time': '10:27:00', 'value': 65}, {'time': '10:28:00', 'value': 65}, {'time': '10:29:00', 'value': 63}, {'time': '10:30:00', 'value': 63}, {'time': '10:31:00', 'value': 66}, {'time': '10:32:00', 'value': 64}, {'time': '10:33:00', 'value': 65}, {'time': '10:34:00', 'value': 64}, {'time': '10:35:00', 'value': 65}, {'time': '10:36:00', 'value': 64}, {'time': '10:37:00', 'value': 64}, {'time': '10:38:00', 'value': 66}, {'time': '10:39:00', 'value': 66}, {'time': '10:40:00', 'value': 66}, {'time': '10:41:00', 'value': 67}, {'time': '10:42:00', 'value': 66}, {'time': '10:43:00', 'value': 66}, {'time': '10:44:00', 'value': 67}, {'time': '10:45:00', 'value': 67}, {'time': '10:46:00', 'value': 67}, {'time': '10:47:00', 'value': 64}, {'time': '10:48:00', 'value': 65}, {'time': '10:49:00', 'value': 64}, {'time': '10:50:00', 'value': 64}, {'time': '10:51:00', 'value': 64}, {'time': '10:52:00', 'value': 65}, {'time': '10:53:00', 'value': 65}, {'time': '10:54:00', 'value': 65}, {'time': '10:55:00', 'value': 65}, {'time': '10:56:00', 'value': 65}, {'time': '10:57:00', 'value': 66}, {'time': '10:58:00', 'value': 66}, {'time': '10:59:00', 'value': 67}, {'time': '11:00:00', 'value': 66}, {'time': '11:01:00', 'value': 65}, {'time': '11:02:00', 'value': 65}, {'time': '11:03:00', 'value': 66}, {'time': '11:04:00', 'value': 67}, {'time': '11:05:00', 'value': 66}, {'time': '11:06:00', 'value': 66}, {'time': '11:07:00', 'value': 67}, {'time': '11:08:00', 'value': 68}, {'time': '11:09:00', 'value': 68}, {'time': '11:10:00', 'value': 69}, {'time': '11:11:00', 'value': 72}, {'time': '11:12:00', 'value': 67}, {'time': '11:13:00', 'value': 69}, {'time': '11:14:00', 'value': 68}, {'time': '11:15:00', 'value': 67}, {'time': '11:16:00', 'value': 68}, {'time': '11:17:00', 'value': 67}, {'time': '11:18:00', 'value': 67}, {'time': '11:19:00', 'value': 71}, {'time': '11:20:00', 'value': 65}, {'time': '11:21:00', 'value': 69}, {'time': '11:22:00', 'value': 68}, {'time': '11:23:00', 'value': 68}, {'time': '11:24:00', 'value': 67}, {'time': '11:25:00', 'value': 66}, {'time': '11:26:00', 'value': 65}, {'time': '11:27:00', 'value': 64}, {'time': '11:28:00', 'value': 63}, {'time': '11:29:00', 'value': 64}, {'time': '11:30:00', 'value': 63}, {'time': '11:31:00', 'value': 68}, {'time': '11:32:00', 'value': 68}, {'time': '11:33:00', 'value': 67}, {'time': '11:34:00', 'value': 68}, {'time': '11:35:00', 'value': 71}, {'time': '11:36:00', 'value': 69}, {'time': '11:37:00', 'value': 70}, {'time': '11:38:00', 'value': 73}, {'time': '11:39:00', 'value': 71}, {'time': '11:40:00', 'value': 70}, {'time': '11:41:00', 'value': 69}, {'time': '11:42:00', 'value': 67}, {'time': '11:43:00', 'value': 72}, {'time': '11:44:00', 'value': 72}, {'time': '11:45:00', 'value': 68}, {'time': '11:46:00', 'value': 63}, {'time': '11:47:00', 'value': 69}, {'time': '11:48:00', 'value': 72}, {'time': '11:49:00', 'value': 74}, {'time': '11:50:00', 'value': 75}, {'time': '11:51:00', 'value': 65}, {'time': '11:52:00', 'value': 63}, {'time': '11:53:00', 'value': 62}, {'time': '11:54:00', 'value': 63}, {'time': '11:55:00', 'value': 62}, {'time': '11:56:00', 'value': 62}, {'time': '11:57:00', 'value': 64}, {'time': '11:58:00', 'value': 64}, {'time': '11:59:00', 'value': 68},
    {'time': '12:00:00', 'value': 66}, {'time': '12:01:00', 'value': 66}, {'time': '12:02:00', 'value': 66}, {'time': '12:03:00', 'value': 67}, {'time': '12:04:00', 'value': 67}, {'time': '12:05:00', 'value': 66}, {'time': '12:06:00', 'value': 65}, {'time': '12:07:00', 'value': 67}, {'time': '12:08:00', 'value': 66}, {'time': '12:09:00', 'value': 66}, {'time': '12:10:00', 'value': 66}, {'time': '12:11:00', 'value': 69}, {'time': '12:12:00', 'value': 64}, {'time': '12:13:00', 'value': 64}, {'time': '12:14:00', 'value': 63}, {'time': '12:15:00', 'value': 63}, {'time': '12:16:00', 'value': 65}, {'time': '12:17:00', 'value': 66}, {'time': '12:18:00', 'value': 70}, {'time': '12:19:00', 'value': 76}, {'time': '12:20:00', 'value': 71}, {'time': '12:21:00', 'value': 68}, {'time': '12:22:00', 'value': 67}, {'time': '12:23:00', 'value': 68}, {'time': '12:24:00', 'value': 67}, {'time': '12:25:00', 'value': 66}, {'time': '12:26:00', 'value': 65}, {'time': '12:27:00', 'value': 66}, {'time': '12:28:00', 'value': 69}, {'time': '12:29:00', 'value': 69}, {'time': '12:30:00', 'value': 68}, {'time': '12:31:00', 'value': 69}, {'time': '12:32:00', 'value': 72}, {'time': '12:33:00', 'value': 73}, {'time': '12:34:00', 'value': 74}, {'time': '12:35:00', 'value': 72}, {'time': '12:36:00', 'value': 73}, {'time': '12:37:00', 'value': 73}, {'time': '12:39:00', 'value': 73}, {'time': '12:40:00', 'value': 73}, {'time': '12:52:00', 'value': 73}, {'time': '12:53:00', 'value': 77}, {'time': '12:54:00', 'value': 83}, {'time': '12:55:00', 'value': 89}, {'time': '12:56:00', 'value': 98}, {'time': '12:57:00', 'value': 101}, {'time': '12:58:00', 'value': 105}, {'time': '12:59:00', 'value': 104}, {'time': '13:00:00', 'value': 109}, {'time': '13:01:00', 'value': 117}, {'time': '13:02:00', 'value': 113}, {'time': '13:03:00', 'value': 109}, {'time': '13:04:00', 'value': 104}, {'time': '13:05:00', 'value': 102}, {'time': '13:06:00', 'value': 104}, {'time': '13:07:00', 'value': 109}, {'time': '13:08:00', 'value': 110}, {'time': '13:09:00', 'value': 106}, {'time': '13:10:00', 'value': 99}, {'time': '13:11:00', 'value': 101}, {'time': '13:12:00', 'value': 100}, {'time': '13:13:00', 'value': 100}, {'time': '13:14:00', 'value': 100}, {'time': '13:15:00', 'value': 100}, {'time': '13:16:00', 'value': 65}, {'time': '13:17:00', 'value': 83}, {'time': '13:18:00', 'value': 106}, {'time': '13:19:00', 'value': 117}, {'time': '13:20:00', 'value': 109}, {'time': '13:21:00', 'value': 89}, {'time': '13:22:00', 'value': 77}, {'time': '13:23:00', 'value': 75}, {'time': '13:24:00', 'value': 77}, {'time': '13:25:00', 'value': 91}, {'time': '13:26:00', 'value': 81}, {'time': '13:27:00', 'value': 76}, {'time': '13:28:00', 'value': 80}, {'time': '13:29:00', 'value': 80}, {'time': '13:30:00', 'value': 80}, {'time': '13:31:00', 'value': 77}, {'time': '13:32:00', 'value': 77}, {'time': '13:33:00', 'value': 78}, {'time': '13:34:00', 'value': 77}, {'time': '13:35:00', 'value': 74}, {'time': '13:36:00', 'value': 73}, {'time': '13:37:00', 'value': 68}, {'time': '13:38:00', 'value': 75}, {'time': '13:39:00', 'value': 75}, {'time': '13:40:00', 'value': 74}, {'time': '13:41:00', 'value': 74}, {'time': '13:42:00', 'value': 75}, {'time': '13:43:00', 'value': 72}, {'time': '13:44:00', 'value': 73}, {'time': '13:46:00', 'value': 80}, {'time': '13:47:00', 'value': 79}, {'time': '13:48:00', 'value': 77}, {'time': '13:49:00', 'value': 79}, {'time': '13:50:00', 'value': 81}, {'time': '13:51:00', 'value': 75}, {'time': '13:52:00', 'value': 74}, {'time': '13:53:00', 'value': 76}, {'time': '13:54:00', 'value': 72}, {'time': '13:55:00', 'value': 76}, {'time': '13:56:00', 'value': 77}, {'time': '13:57:00', 'value': 79}, {'time': '13:58:00', 'value': 74}, {'time': '13:59:00', 'value': 80}, {'time': '14:00:00', 'value': 79}, {'time': '14:01:00', 'value': 79}, {'time': '14:02:00', 'value': 77}, {'time': '14:03:00', 'value': 81}, {'time': '14:04:00', 'value': 78}, {'time': '14:05:00', 'value': 82}, {'time': '14:06:00', 'value': 82}, {'time': '14:07:00', 'value': 79}, {'time': '14:08:00', 'value': 80}, {'time': '14:09:00', 'value': 81}, {'time': '14:10:00', 'value': 82}, {'time': '14:11:00', 'value': 80}, {'time': '14:12:00', 'value': 77}, {'time': '14:13:00', 'value': 77}, {'time': '14:14:00', 'value': 79}, {'time': '14:15:00', 'value': 77}, {'time': '14:16:00', 'value': 74}, {'time': '14:17:00', 'value': 74}, {'time': '14:18:00', 'value': 76}, {'time': '14:19:00', 'value': 75}, {'time': '14:20:00', 'value': 79}, {'time': '14:21:00', 'value': 78}, {'time': '14:22:00', 'value': 75}, {'time': '14:23:00', 'value': 72}, {'time': '14:24:00', 'value': 79}, {'time': '14:25:00', 'value': 78}, {'time': '14:26:00', 'value': 79}, {'time': '14:27:00', 'value': 80}, {'time': '14:28:00', 'value': 76}, {'time': '14:29:00', 'value': 75}, {'time': '14:30:00', 'value': 79}, {'time': '14:31:00', 'value': 82}, {'time': '14:32:00', 'value': 80}, {'time': '14:33:00', 'value': 83}, {'time': '14:34:00', 'value': 100}, {'time': '14:35:00', 'value': 109}, {'time': '14:36:00', 'value': 120}, {'time': '14:37:00', 'value': 141}, {'time': '14:38:00', 'value': 138}, {'time': '14:39:00', 'value': 109}, {'time': '14:40:00', 'value': 109}, {'time': '14:41:00', 'value': 101}, {'time': '14:42:00', 'value': 99}, {'time': '14:43:00', 'value': 100}, {'time': '14:44:00', 'value': 100}, {'time': '14:45:00', 'value': 98}, {'time': '14:46:00', 'value': 94}, {'time': '14:47:00', 'value': 98}, {'time': '14:48:00', 'value': 102}, {'time': '14:49:00', 'value': 114}, {'time': '14:50:00', 'value': 111}, {'time': '14:51:00', 'value': 109}, {'time': '14:52:00', 'value': 112}, {'time': '14:53:00', 'value': 111}, {'time': '14:54:00', 'value': 115}, {'time': '14:55:00', 'value': 114}, {'time': '14:56:00', 'value': 106}, {'time': '14:57:00', 'value': 100}, {'time': '14:58:00', 'value': 104}, {'time': '14:59:00', 'value': 97}, {'time': '15:00:00', 'value': 92}, {'time': '15:01:00', 'value': 95}, {'time': '15:02:00', 'value': 95}, {'time': '15:03:00', 'value': 93}, {'time': '15:04:00', 'value': 91}, {'time': '15:05:00', 'value': 91}, {'time': '15:06:00', 'value': 91}, {'time': '15:07:00', 'value': 95}, {'time': '15:08:00', 'value': 97}, {'time': '15:09:00', 'value': 96}, {'time': '15:10:00', 'value': 96}, {'time': '15:11:00', 'value': 95}, {'time': '15:12:00', 'value': 93}, {'time': '15:13:00', 'value': 88}, {'time': '15:14:00', 'value': 88}, {'time': '15:15:00', 'value': 89}, {'time': '15:16:00', 'value': 85}, {'time': '15:17:00', 'value': 83}, {'time': '15:18:00', 'value': 84}, {'time': '15:19:00', 'value': 84}, {'time': '15:20:00', 'value': 83}, {'time': '15:21:00', 'value': 81}, {'time': '15:22:00', 'value': 85}, {'time': '15:23:00', 'value': 86}, {'time': '15:24:00', 'value': 87}, {'time': '15:25:00', 'value': 84}, {'time': '15:26:00', 'value': 85}, {'time': '15:27:00', 'value': 86}, {'time': '15:28:00', 'value': 82}, {'time': '15:29:00', 'value': 76}, {'time': '15:30:00', 'value': 81}, {'time': '15:31:00', 'value': 83}, {'time': '15:32:00', 'value': 76}, {'time': '15:33:00', 'value': 69}, {'time': '15:34:00', 'value': 76}, {'time': '15:35:00', 'value': 77}, {'time': '15:36:00', 'value': 74}, {'time': '15:37:00', 'value': 76}, {'time': '15:38:00', 'value': 75}, {'time': '15:39:00', 'value': 76}, {'time': '15:40:00', 'value': 83}, {'time': '15:41:00', 'value': 102}, {'time': '15:42:00', 'value': 96}, {'time': '15:43:00', 'value': 99}, {'time': '15:44:00', 'value': 98}, {'time': '15:45:00', 'value': 101}, {'time': '15:46:00', 'value': 79}, {'time': '15:47:00', 'value': 78}, {'time': '15:48:00', 'value': 92}, {'time': '15:49:00', 'value': 83}, {'time': '15:50:00', 'value': 80}, {'time': '15:51:00', 'value': 80}, {'time': '15:52:00', 'value': 88}, {'time': '15:53:00', 'value': 94}, {'time': '15:54:00', 'value': 94}, {'time': '15:55:00', 'value': 84}, {'time': '15:56:00', 'value': 85}, {'time': '15:57:00', 'value': 87}, {'time': '15:58:00', 'value': 85}, {'time': '15:59:00', 'value': 84},
    {'time': '16:00:00', 'value': 81}, {'time': '16:01:00', 'value': 82}, {'time': '16:02:00', 'value': 76}, {'time': '16:03:00', 'value': 82}, {'time': '16:04:00', 'value': 81}, {'time': '16:05:00', 'value': 78}, {'time': '16:06:00', 'value': 79}, {'time': '16:07:00', 'value': 78}, {'time': '16:08:00', 'value': 79}, {'time': '16:09:00', 'value': 80}, {'time': '16:10:00', 'value': 81}, {'time': '16:11:00', 'value': 80}, {'time': '16:12:00', 'value': 80}, {'time': '16:13:00', 'value': 81}, {'time': '16:14:00', 'value': 78}, {'time': '16:15:00', 'value': 78}, {'time': '16:16:00', 'value': 76}, {'time': '16:17:00', 'value': 81}, {'time': '16:18:00', 'value': 78}, {'time': '16:19:00', 'value': 80}, {'time': '16:20:00', 'value': 78}, {'time': '16:21:00', 'value': 76}, {'time': '16:22:00', 'value': 78}, {'time': '16:23:00', 'value': 84}, {'time': '16:24:00', 'value': 93}, {'time': '16:25:00', 'value': 101}, {'time': '16:26:00', 'value': 102}, {'time': '16:27:00', 'value': 89}, {'time': '16:28:00', 'value': 78}, {'time': '16:29:00', 'value': 75}, {'time': '16:30:00', 'value': 71}, {'time': '16:31:00', 'value': 78}, {'time': '16:32:00', 'value': 75}, {'time': '16:33:00', 'value': 73}, {'time': '16:34:00', 'value': 72}, {'time': '16:35:00', 'value': 80}, {'time': '16:36:00', 'value': 72}, {'time': '16:37:00', 'value': 71}, {'time': '16:38:00', 'value': 74}, {'time': '16:39:00', 'value': 73}, {'time': '16:40:00', 'value': 67}, {'time': '16:41:00', 'value': 73}, {'time': '16:42:00', 'value': 69}, {'time': '16:43:00', 'value': 77}, {'time': '16:44:00', 'value': 79}, {'time': '16:45:00', 'value': 75}, {'time': '16:46:00', 'value': 72}, {'time': '16:47:00', 'value': 74}, {'time': '16:48:00', 'value': 72}, {'time': '16:49:00', 'value': 71}, {'time': '16:50:00', 'value': 70}, {'time': '16:51:00', 'value': 70}, {'time': '16:52:00', 'value': 69}, {'time': '16:53:00', 'value': 69}, {'time': '16:54:00', 'value': 71}, {'time': '16:55:00', 'value': 68}, {'time': '16:56:00', 'value': 69}, {'time': '16:57:00', 'value': 75}, {'time': '16:58:00', 'value': 73}, {'time': '16:59:00', 'value': 73}, {'time': '17:00:00', 'value': 71}, {'time': '17:01:00', 'value': 73}, {'time': '17:02:00', 'value': 81}, {'time': '17:03:00', 'value': 87}, {'time': '17:04:00', 'value': 86}, {'time': '17:05:00', 'value': 79}, {'time': '17:06:00', 'value': 69}, {'time': '17:07:00', 'value': 69}, {'time': '17:08:00', 'value': 71}, {'time': '17:09:00', 'value': 78}, {'time': '17:10:00', 'value': 80}, {'time': '17:11:00', 'value': 82}, {'time': '17:12:00', 'value': 86}, {'time': '17:13:00', 'value': 88}, {'time': '17:14:00', 'value': 88}, {'time': '17:15:00', 'value': 87}, {'time': '17:16:00', 'value': 96}, {'time': '17:17:00', 'value': 100}, {'time': '17:18:00', 'value': 111}, {'time': '17:19:00', 'value': 127}, {'time': '17:20:00', 'value': 112}, {'time': '17:21:00', 'value': 103}, {'time': '17:22:00', 'value': 103}, {'time': '17:23:00', 'value': 84}, {'time': '17:24:00', 'value': 82}, {'time': '17:25:00', 'value': 80}, {'time': '17:26:00', 'value': 85}, {'time': '17:27:00', 'value': 84}, {'time': '17:28:00', 'value': 75}, {'time': '17:29:00', 'value': 78}, {'time': '17:30:00', 'value': 84}, {'time': '17:31:00', 'value': 83}, {'time': '17:32:00', 'value': 85}, {'time': '17:33:00', 'value': 86}, {'time': '17:34:00', 'value': 90}, {'time': '17:35:00', 'value': 92}, {'time': '17:36:00', 'value': 92}, {'time': '17:37:00', 'value': 90}, {'time': '17:38:00', 'value': 92}, {'time': '17:39:00', 'value': 85}, {'time': '17:40:00', 'value': 91}, {'time': '17:41:00', 'value': 91}, {'time': '17:42:00', 'value': 91}, {'time': '17:43:00', 'value': 90}, {'time': '17:44:00', 'value': 88}, {'time': '17:45:00', 'value': 86}, {'time': '17:46:00', 'value': 92}, {'time': '17:47:00', 'value': 91}, {'time': '17:48:00', 'value': 88}, {'time': '17:49:00', 'value': 87}, {'time': '17:50:00', 'value': 89}, {'time': '17:51:00', 'value': 92}, {'time': '17:52:00', 'value': 90}, {'time': '17:53:00', 'value': 91}, {'time': '17:54:00', 'value': 87}, {'time': '17:55:00', 'value': 89}, {'time': '17:56:00', 'value': 89}, {'time': '17:57:00', 'value': 91}, {'time': '17:58:00', 'value': 88}, {'time': '17:59:00', 'value': 87}, {'time': '18:00:00', 'value': 86}, {'time': '18:01:00', 'value': 87}, {'time': '18:02:00', 'value': 86}, {'time': '18:03:00', 'value': 86}, {'time': '18:04:00', 'value': 86}, {'time': '18:05:00', 'value': 83}, {'time': '18:06:00', 'value': 86}, {'time': '18:07:00', 'value': 85}, {'time': '18:08:00', 'value': 86}, {'time': '18:09:00', 'value': 82}, {'time': '18:10:00', 'value': 82}, {'time': '18:11:00', 'value': 81}, {'time': '18:12:00', 'value': 84}, {'time': '18:13:00', 'value': 84}, {'time': '18:14:00', 'value': 84}, {'time': '18:15:00', 'value': 80}, {'time': '18:16:00', 'value': 84}, {'time': '18:17:00', 'value': 84}, {'time': '18:18:00', 'value': 81}, {'time': '18:19:00', 'value': 83}, {'time': '18:20:00', 'value': 81}, {'time': '18:21:00', 'value': 80}, {'time': '18:22:00', 'value': 82}, {'time': '18:23:00', 'value': 85}, {'time': '18:24:00', 'value': 73}, {'time': '18:25:00', 'value': 75}, {'time': '18:26:00', 'value': 77}, {'time': '18:27:00', 'value': 84}, {'time': '18:28:00', 'value': 92}, {'time': '18:29:00', 'value': 92}, {'time': '18:30:00', 'value': 83}, {'time': '18:31:00', 'value': 84}, {'time': '18:32:00', 'value': 84}, {'time': '18:33:00', 'value': 86}, {'time': '18:34:00', 'value': 85}, {'time': '18:35:00', 'value': 84}, {'time': '18:36:00', 'value': 86}, {'time': '18:37:00', 'value': 86}, {'time': '18:38:00', 'value': 86}, {'time': '18:39:00', 'value': 87}, {'time': '18:40:00', 'value': 84}, {'time': '18:41:00', 'value': 86}, {'time': '18:42:00', 'value': 86}, {'time': '18:43:00', 'value': 87}, {'time': '18:44:00', 'value': 88}, {'time': '18:45:00', 'value': 85}, {'time': '18:46:00', 'value': 88}, {'time': '18:47:00', 'value': 87}, {'time': '18:48:00', 'value': 86}, {'time': '18:49:00', 'value': 85}, {'time': '18:50:00', 'value': 86}, {'time': '18:51:00', 'value': 87}, {'time': '18:52:00', 'value': 85}, {'time': '18:53:00', 'value': 88}, {'time': '18:54:00', 'value': 86}, {'time': '18:55:00', 'value': 83}, {'time': '18:56:00', 'value': 82}, {'time': '18:57:00', 'value': 82}, {'time': '18:58:00', 'value': 81}, {'time': '18:59:00', 'value': 82}, {'time': '19:00:00', 'value': 80}, {'time': '19:01:00', 'value': 84}, {'time': '19:02:00', 'value': 85}, {'time': '19:03:00', 'value': 92}, {'time': '19:04:00', 'value': 97}, {'time': '19:05:00', 'value': 101}, {'time': '19:06:00', 'value': 110}, {'time': '19:07:00', 'value': 106}, {'time': '19:08:00', 'value': 102}, {'time': '19:09:00', 'value': 105}, {'time': '19:10:00', 'value': 102}, {'time': '19:11:00', 'value': 103}, {'time': '19:12:00', 'value': 100}, {'time': '19:13:00', 'value': 121}, {'time': '19:14:00', 'value': 135}, {'time': '19:15:00', 'value': 111}, {'time': '19:16:00', 'value': 121}, {'time': '19:17:00', 'value': 116}, {'time': '19:18:00', 'value': 104}, {'time': '19:19:00', 'value': 98}, {'time': '19:20:00', 'value': 91}, {'time': '19:21:00', 'value': 91}, {'time': '19:22:00', 'value': 91}, {'time': '19:23:00', 'value': 103}, {'time': '19:24:00', 'value': 102}, {'time': '19:25:00', 'value': 93}, {'time': '19:26:00', 'value': 97}, {'time': '19:27:00', 'value': 93}, {'time': '19:28:00', 'value': 88}, {'time': '19:29:00', 'value': 85}, {'time': '19:30:00', 'value': 88}, {'time': '19:31:00', 'value': 87}, {'time': '19:32:00', 'value': 88}, {'time': '19:33:00', 'value': 86}, {'time': '19:34:00', 'value': 84}, {'time': '19:35:00', 'value': 85}, {'time': '19:36:00', 'value': 84}, {'time': '19:37:00', 'value': 85}, {'time': '19:38:00', 'value': 84}, {'time': '19:39:00', 'value': 80}, {'time': '19:40:00', 'value': 85}, {'time': '19:41:00', 'value': 84}, {'time': '19:42:00', 'value': 86}, {'time': '19:43:00', 'value': 84}, {'time': '19:44:00', 'value': 86}, {'time': '19:45:00', 'value': 88}, {'time': '19:46:00', 'value': 84}, {'time': '19:47:00', 'value': 83}, {'time': '19:48:00', 'value': 87}, {'time': '19:49:00', 'value': 91}, {'time': '19:50:00', 'value': 88}, {'time': '19:51:00', 'value': 88}, {'time': '19:52:00', 'value': 88}, {'time': '19:53:00', 'value': 85}, {'time': '19:54:00', 'value': 89}, {'time': '19:55:00', 'value': 85}, {'time': '19:56:00', 'value': 87}, {'time': '19:57:00', 'value': 87}, {'time': '19:58:00', 'value': 80}, {'time': '19:59:00', 'value': 83},
    {'time': '20:00:00', 'value': 84}, {'time': '20:01:00', 'value': 80}, {'time': '20:02:00', 'value': 82}, {'time': '20:03:00', 'value': 79}, {'time': '20:04:00', 'value': 81}, {'time': '20:05:00', 'value': 87}, {'time': '20:06:00', 'value': 80}, {'time': '20:07:00', 'value': 83}, {'time': '20:08:00', 'value': 81}, {'time': '20:09:00', 'value': 81}, {'time': '20:10:00', 'value': 79}, {'time': '20:11:00', 'value': 79}, {'time': '20:12:00', 'value': 81}, {'time': '20:13:00', 'value': 79}, {'time': '20:14:00', 'value': 79}, {'time': '20:15:00', 'value': 79}, {'time': '20:16:00', 'value': 80}, {'time': '20:17:00', 'value': 77}, {'time': '20:18:00', 'value': 77}, {'time': '20:19:00', 'value': 76}, {'time': '20:20:00', 'value': 79}, {'time': '20:21:00', 'value': 79}, {'time': '20:22:00', 'value': 79}, {'time': '20:23:00', 'value': 79}, {'time': '20:24:00', 'value': 79}, {'time': '20:25:00', 'value': 77}, {'time': '20:26:00', 'value': 75}, {'time': '20:27:00', 'value': 78}, {'time': '20:28:00', 'value': 78}, {'time': '20:29:00', 'value': 79}, {'time': '20:30:00', 'value': 75}, {'time': '20:31:00', 'value': 78}, {'time': '20:32:00', 'value': 80}, {'time': '20:33:00', 'value': 81}, {'time': '20:34:00', 'value': 83}, {'time': '20:35:00', 'value': 79}, {'time': '20:36:00', 'value': 76}, {'time': '20:37:00', 'value': 79}, {'time': '20:38:00', 'value': 76}, {'time': '20:39:00', 'value': 76}, {'time': '20:40:00', 'value': 76}, {'time': '20:41:00', 'value': 78}, {'time': '20:42:00', 'value': 77}, {'time': '20:43:00', 'value': 75}, {'time': '20:44:00', 'value': 79}, {'time': '20:45:00', 'value': 78}, {'time': '20:46:00', 'value': 76}, {'time': '20:47:00', 'value': 74}, {'time': '20:48:00', 'value': 77}, {'time': '20:49:00', 'value': 77}, {'time': '20:50:00', 'value': 78}, {'time': '20:51:00', 'value': 81}, {'time': '20:52:00', 'value': 82}, {'time': '20:53:00', 'value': 76}, {'time': '20:54:00', 'value': 74}, {'time': '20:55:00', 'value': 78}, {'time': '20:56:00', 'value': 73}, {'time': '20:57:00', 'value': 71}, {'time': '20:58:00', 'value': 74}, {'time': '20:59:00', 'value': 78}, {'time': '21:00:00', 'value': 74}, {'time': '21:01:00', 'value': 68}, {'time': '21:02:00', 'value': 75}, {'time': '21:03:00', 'value': 77}, {'time': '21:04:00', 'value': 80}, {'time': '21:05:00', 'value': 86}, {'time': '21:06:00', 'value': 84}, {'time': '21:07:00', 'value': 77}, {'time': '21:08:00', 'value': 73}, {'time': '21:09:00', 'value': 78}, {'time': '21:10:00', 'value': 86}, {'time': '21:11:00', 'value': 74}, {'time': '21:12:00', 'value': 81}, {'time': '21:13:00', 'value': 87}, {'time': '21:14:00', 'value': 83}, {'time': '21:15:00', 'value': 77}, {'time': '21:16:00', 'value': 80}, {'time': '21:17:00', 'value': 79}, {'time': '21:18:00', 'value': 86}, {'time': '21:19:00', 'value': 95}, {'time': '21:20:00', 'value': 83}, {'time': '21:21:00', 'value': 77}, {'time': '21:22:00', 'value': 73}, {'time': '21:23:00', 'value': 72}, {'time': '21:24:00', 'value': 77}, {'time': '21:25:00', 'value': 80}, {'time': '21:26:00', 'value': 76}, {'time': '21:27:00', 'value': 78}, {'time': '21:28:00', 'value': 72}, {'time': '21:29:00', 'value': 73}, {'time': '21:30:00', 'value': 72}, {'time': '21:31:00', 'value': 80}, {'time': '21:32:00', 'value': 77}, {'time': '21:33:00', 'value': 79}, {'time': '21:34:00', 'value': 76}, {'time': '21:35:00', 'value': 76}, {'time': '21:36:00', 'value': 74}, {'time': '21:37:00', 'value': 75}, {'time': '21:38:00', 'value': 80}, {'time': '21:39:00', 'value': 80}, {'time': '21:40:00', 'value': 74}, {'time': '21:41:00', 'value': 78}, {'time': '21:42:00', 'value': 75}, {'time': '21:43:00', 'value': 75}, {'time': '21:44:00', 'value': 76}, {'time': '21:45:00', 'value': 77}, {'time': '21:46:00', 'value': 75}, {'time': '21:47:00', 'value': 78}, {'time': '21:48:00', 'value': 82}, {'time': '21:49:00', 'value': 81}, {'time': '21:50:00', 'value': 81}, {'time': '21:51:00', 'value': 78}, {'time': '21:52:00', 'value': 77}, {'time': '21:53:00', 'value': 73}, {'time': '21:54:00', 'value': 72}, {'time': '21:55:00', 'value': 76}, {'time': '21:56:00', 'value': 78}, {'time': '21:57:00', 'value': 79}, {'time': '21:58:00', 'value': 77}, {'time': '21:59:00', 'value': 75}, {'time': '22:00:00', 'value': 79}, {'time': '22:01:00', 'value': 88}, {'time': '22:02:00', 'value': 96}, {'time': '22:03:00', 'value': 89}, {'time': '22:04:00', 'value': 77}, {'time': '22:05:00', 'value': 72}, {'time': '22:06:00', 'value': 74}, {'time': '22:07:00', 'value': 77}, {'time': '22:08:00', 'value': 83}, {'time': '22:09:00', 'value': 97}, {'time': '22:10:00', 'value': 110}, {'time': '22:11:00', 'value': 114}, {'time': '22:12:00', 'value': 108}, {'time': '22:13:00', 'value': 106}, {'time': '22:14:00', 'value': 107}, {'time': '22:15:00', 'value': 106}, {'time': '22:16:00', 'value': 100}, {'time': '22:17:00', 'value': 93}, {'time': '22:18:00', 'value': 95}, {'time': '22:19:00', 'value': 97}, {'time': '22:20:00', 'value': 101}, {'time': '22:21:00', 'value': 105}, {'time': '22:22:00', 'value': 103}, {'time': '22:23:00', 'value': 100}, {'time': '22:24:00', 'value': 96}, {'time': '22:25:00', 'value': 100}, {'time': '22:26:00', 'value': 100}, {'time': '22:27:00', 'value': 91}, {'time': '22:28:00', 'value': 79}, {'time': '22:29:00', 'value': 78}, {'time': '22:30:00', 'value': 77}, {'time': '22:31:00', 'value': 77}, {'time': '22:32:00', 'value': 76}, {'time': '22:33:00', 'value': 76}, {'time': '22:34:00', 'value': 85}, {'time': '22:35:00', 'value': 90}, {'time': '22:36:00', 'value': 87}, {'time': '22:37:00', 'value': 87}, {'time': '22:38:00', 'value': 88}, {'time': '22:39:00', 'value': 92}, {'time': '22:40:00', 'value': 90}, {'time': '22:41:00', 'value': 83}, {'time': '22:42:00', 'value': 85}, {'time': '22:43:00', 'value': 86}, {'time': '22:44:00', 'value': 79}, {'time': '22:45:00', 'value': 74}, {'time': '22:46:00', 'value': 75}, {'time': '22:47:00', 'value': 77}, {'time': '22:48:00', 'value': 75}, {'time': '22:49:00', 'value': 75}, {'time': '22:50:00', 'value': 73}, {'time': '22:51:00', 'value': 74}, {'time': '22:52:00', 'value': 75}, {'time': '22:53:00', 'value': 74}, {'time': '22:54:00', 'value': 73}, {'time': '22:55:00', 'value': 75}, {'time': '22:56:00', 'value': 74}, {'time': '22:57:00', 'value': 73}, {'time': '22:58:00', 'value': 75}, {'time': '22:59:00', 'value': 74}, {'time': '23:00:00', 'value': 69}, {'time': '23:01:00', 'value': 73}, {'time': '23:02:00', 'value': 74}, {'time': '23:03:00', 'value': 73}, {'time': '23:04:00', 'value': 73}, {'time': '23:05:00', 'value': 73}, {'time': '23:06:00', 'value': 76}, {'time': '23:07:00', 'value': 73}, {'time': '23:08:00', 'value': 74}, {'time': '23:09:00', 'value': 74}, {'time': '23:10:00', 'value': 69}, {'time': '23:11:00', 'value': 72}, {'time': '23:12:00', 'value': 72}, {'time': '23:13:00', 'value': 76}, {'time': '23:14:00', 'value': 71}, {'time': '23:15:00', 'value': 69}, {'time': '23:16:00', 'value': 72}, {'time': '23:17:00', 'value': 72}, {'time': '23:18:00', 'value': 70}, {'time': '23:19:00', 'value': 73}, {'time': '23:20:00', 'value': 72}, {'time': '23:21:00', 'value': 73}, {'time': '23:22:00', 'value': 71}, {'time': '23:23:00', 'value': 72}, {'time': '23:24:00', 'value': 70}, {'time': '23:25:00', 'value': 70}, {'time': '23:26:00', 'value': 71}, {'time': '23:27:00', 'value': 71}, {'time': '23:28:00', 'value': 73}, {'time': '23:29:00', 'value': 69}, {'time': '23:30:00', 'value': 70}, {'time': '23:31:00', 'value': 74}, {'time': '23:32:00', 'value': 72}, {'time': '23:33:00', 'value': 72}, {'time': '23:34:00', 'value': 73}, {'time': '23:35:00', 'value': 72}, {'time': '23:36:00', 'value': 72}, {'time': '23:37:00', 'value': 75}, {'time': '23:38:00', 'value': 75}, {'time': '23:39:00', 'value': 77}, {'time': '23:40:00', 'value': 80}, {'time': '23:41:00', 'value': 80}, {'time': '23:42:00', 'value': 79}, {'time': '23:43:00', 'value': 76}, {'time': '23:44:00', 'value': 84}, {'time': '23:45:00', 'value': 73}, {'time': '23:46:00', 'value': 78}, {'time': '23:47:00', 'value': 80}, {'time': '23:48:00', 'value': 78}, {'time': '23:49:00', 'value': 78}, {'time': '23:50:00', 'value': 76}, {'time': '23:51:00', 'value': 80}, {'time': '23:52:00', 'value': 79}, {'time': '23:53:00', 'value': 80}, {'time': '23:54:00', 'value': 76}, {'time': '23:55:00', 'value': 74}, {'time': '23:56:00', 'value': 74}, {'time': '23:57:00', 'value': 77}, {'time': '23:58:00', 'value': 76}, {'time': '23:59:00', 'value': 71}],
    'datasetInterval': 1,
    'datasetType': 'minute'}
}


@pytest.mark.skipif(os.getenv("TEST_LEVEL") != "local", reason='Travis-CI issues')
def test_get_heart_rate_time_series_period():
    database = Database('fitbit_test', 'heart')
    connection = database.engine.connect()

    connection.execute(heart_daily_table.delete())

    HeartRateTimeSeries(config={
        'database': 'fitbit_test',
        'base_date': '2020-09-05',
        'period': '1d'}
    ).insert_data(database, heart_daily_table)

    rows = [row for row in connection.execute(heart_daily_table.select())]
    connection.close()

    assert sorted(rows) == sorted(EXPECTED_PERIOD_ROWS)


def test_query_heart_rate_zone_time_series():
    data = HeartRateTimeSeries(config={
        'database': 'fitbit_test',
        'base_date': '2020-09-05',
        'end_date': '2020-09-05'}
    ).query()
    data.pop('activities-heart-intraday')  # quickfix for not yet parsing intraday heart data

    assert data == EXPECTED_DATA


def test_parse_response():
    rows = HeartRateTimeSeries(config={
        'database': 'fitbit_test',
        'base_date': '2020-09-05',
        'period': '1d'}
    ).parse_response(EXPECTED_DATA)

    for index, row in enumerate(rows):
        assert row.type == EXPECTED_ROWS[index][0]
        assert row.minutes == EXPECTED_ROWS[index][1]
        assert row.date == EXPECTED_ROWS[index][2].strftime('%Y-%m-%d')
        assert row.calories == float(EXPECTED_ROWS[index][3])
        assert row.resting_heart_rate == EXPECTED_ROWS[index][4]


@pytest.mark.skipif(os.getenv("TEST_LEVEL") != "local", reason='Travis-CI issues')
def test_get_total_calories():
    heart_rate_zone = HeartRateTimeSeries(config={'database': 'fitbit'})
    df = heart_rate_zone.get_total_calories_df(show=False)

    testing_date = (df.where(df.date == '2020-08-08').withColumnRenamed('sum(calories)', 'sum_calories')).take(1)[0]
    assert testing_date.sum_calories == Decimal('3063.96122')


def test_parse_intraday_response():
    heart_rate_zone = HeartRateTimeSeries(config={'database': 'fitbit'})
    rows = heart_rate_zone.parse_intraday_response('2020-10-05', EXPECTED_INTRADAY_DATA)
    for index, row in enumerate(rows[:10]):
        assert row.date == EXPECTED_INTRADAY_ROWS[index][0]
        assert row.time == EXPECTED_INTRADAY_ROWS[index][1]
        assert row.value == EXPECTED_INTRADAY_ROWS[index][2]


@pytest.mark.skip(reason='takes inordinately long - test locally.')
def test_insert_intraday_data():
    database = Database('fitbit_test', 'heart')
    connection = database.engine.connect()

    connection.execute(heart_intraday_table.delete())

    heart_rate_zone = HeartRateTimeSeries(config={
        'database': 'fitbit_test',
        'base_date': '2020-10-04'
        })

    rows = heart_rate_zone.insert_intraday_data()
    assert len(rows) == 1381


@pytest.mark.skipif(os.getenv("TEST_LEVEL") != "local", reason='Travis-CI issues')
def test_plot_rolling_average():
    hrz = HeartRateTimeSeries(config={'database': 'fitbit_test',
                                      'base_date': '2020-10-01',
                                      'end_date': '2020-10-15',
                                      'table': 'daily',
                                      'sum_column': 'calories'})
    plot_rolling_average(hrz.config)
