"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplicaProvider = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
class ReplicaProvider extends core_1.NestedStack {
    constructor(scope, id, props = {}) {
        super(scope, id);
        const code = lambda.Code.fromAsset(path.join(__dirname, 'replica-handler'));
        // Issues UpdateTable API calls
        this.onEventHandler = new lambda.Function(this, 'OnEventHandler', {
            code,
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.onEventHandler',
            timeout: core_1.Duration.minutes(5),
        });
        // Checks if table is back to `ACTIVE` state
        this.isCompleteHandler = new lambda.Function(this, 'IsCompleteHandler', {
            code,
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.isCompleteHandler',
            timeout: core_1.Duration.seconds(30),
        });
        // Allows the creation of the `AWSServiceRoleForDynamoDBReplication` service linked role
        this.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: [core_1.Stack.of(this).formatArn({
                    service: 'iam',
                    region: '',
                    resource: 'role',
                    resourceName: 'aws-service-role/replication.dynamodb.amazonaws.com/AWSServiceRoleForDynamoDBReplication',
                })],
        }));
        // Required for replica table creation
        this.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['dynamodb:DescribeLimits'],
            resources: ['*'],
        }));
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: this.onEventHandler,
            isCompleteHandler: this.isCompleteHandler,
            queryInterval: core_1.Duration.seconds(10),
            totalTimeout: props.timeout,
        });
    }
    /**
     * Creates a stack-singleton resource provider nested stack.
     */
    static getOrCreate(scope, props = {}) {
        var _a;
        const stack = core_1.Stack.of(scope);
        const uid = '@aws-cdk/aws-dynamodb.ReplicaProvider';
        return (_a = stack.node.tryFindChild(uid)) !== null && _a !== void 0 ? _a : new ReplicaProvider(stack, uid, props);
    }
}
exports.ReplicaProvider = ReplicaProvider;
//# sourceMappingURL=data:application/json;base64,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