"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Topic = void 0;
const core_1 = require("@aws-cdk/core");
const sns_generated_1 = require("./sns.generated");
const topic_base_1 = require("./topic-base");
/**
 * A new SNS topic.
 *
 * @stability stable
 */
class Topic extends topic_base_1.TopicBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.topicName,
        });
        /**
         * Controls automatic creation of policy objects.
         *
         * Set by subclasses.
         *
         * @stability stable
         */
        this.autoCreatePolicy = true;
        if (props.fifo && !props.topicName) {
            // NOTE: Workaround for CloudFormation problem reported in CDK issue 12386
            // see https://github.com/aws/aws-cdk/issues/12386
            throw new Error('FIFO SNS topics must be given a topic name.');
        }
        if (props.contentBasedDeduplication && !props.fifo) {
            throw new Error('Content based deduplication can only be enabled for FIFO SNS topics.');
        }
        let cfnTopicName;
        if (props.fifo && props.topicName && !props.topicName.endsWith('.fifo')) {
            cfnTopicName = this.physicalName + '.fifo';
        }
        else {
            cfnTopicName = this.physicalName;
        }
        const resource = new sns_generated_1.CfnTopic(this, 'Resource', {
            displayName: props.displayName,
            topicName: cfnTopicName,
            kmsMasterKeyId: props.masterKey && props.masterKey.keyArn,
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoTopic: props.fifo,
        });
        this.topicArn = this.getResourceArnAttribute(resource.ref, {
            service: 'sns',
            resource: this.physicalName,
        });
        this.topicName = this.getResourceNameAttribute(resource.attrTopicName);
    }
    /**
     * Import an existing SNS topic provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param topicArn topic ARN (i.e. arn:aws:sns:us-east-2:444455556666:MyTopic).
     * @stability stable
     */
    static fromTopicArn(scope, id, topicArn) {
        class Import extends topic_base_1.TopicBase {
            constructor() {
                super(...arguments);
                this.topicArn = topicArn;
                this.topicName = core_1.Stack.of(scope).parseArn(topicArn).resource;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.Topic = Topic;
//# sourceMappingURL=data:application/json;base64,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