"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationTargetGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancingv2_canned_metrics_generated_1 = require("../elasticloadbalancingv2-canned-metrics.generated");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define an Application Target Group.
 *
 * @stability stable
 */
class ApplicationTargetGroup extends base_target_group_1.TargetGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        const { protocolVersion } = props;
        super(scope, id, { ...props }, {
            protocol,
            protocolVersion,
            port,
        });
        this.protocol = protocol;
        this.port = port;
        this.connectableMembers = [];
        this.listeners = [];
        if (props) {
            if (props.slowStart !== undefined) {
                if (props.slowStart.toSeconds() < 30 || props.slowStart.toSeconds() > 900) {
                    throw new Error('Slow start duration value must be between 30 and 900 seconds.');
                }
                this.setAttribute('slow_start.duration_seconds', props.slowStart.toSeconds().toString());
            }
            if (props.stickinessCookieDuration) {
                this.enableCookieStickiness(props.stickinessCookieDuration, props.stickinessCookieName);
            }
            if (props.loadBalancingAlgorithmType) {
                this.setAttribute('load_balancing.algorithm.type', props.loadBalancingAlgorithmType);
            }
            this.addTarget(...(props.targets || []));
        }
    }
    /**
     * Import an existing target group.
     *
     * @stability stable
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedApplicationTargetGroup(scope, id, attrs);
    }
    /**
     * (deprecated) Import an existing target group.
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return ApplicationTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group.
     *
     * @stability stable
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Enable sticky routing via a cookie to members of this target group.
     *
     * Note: If the `cookieName` parameter is set, application-based stickiness will be applied,
     * otherwise it defaults to duration-based stickiness attributes (`lb_cookie`).
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/sticky-sessions.html
     * @stability stable
     */
    enableCookieStickiness(duration, cookieName) {
        if (duration.toSeconds() < 1 || duration.toSeconds() > 604800) {
            throw new Error('Stickiness cookie duration value must be between 1 second and 7 days (604800 seconds).');
        }
        if (cookieName !== undefined) {
            if (!core_1.Token.isUnresolved(cookieName) && (cookieName.startsWith('AWSALB') || cookieName.startsWith('AWSALBAPP') || cookieName.startsWith('AWSALBTG'))) {
                throw new Error('App cookie names that start with the following prefixes are not allowed: AWSALB, AWSALBAPP, and AWSALBTG; they\'re reserved for use by the load balancer.');
            }
            if (cookieName === '') {
                throw new Error('App cookie name cannot be an empty string.');
            }
        }
        this.setAttribute('stickiness.enabled', 'true');
        if (cookieName) {
            this.setAttribute('stickiness.type', 'app_cookie');
            this.setAttribute('stickiness.app_cookie.cookie_name', cookieName);
            this.setAttribute('stickiness.app_cookie.duration_seconds', duration.toSeconds().toString());
        }
        else {
            this.setAttribute('stickiness.type', 'lb_cookie');
            this.setAttribute('stickiness.lb_cookie.duration_seconds', duration.toSeconds().toString());
        }
    }
    /**
     * Register a connectable as a member of this target group.
     *
     * Don't call this directly. It will be called by load balancing targets.
     *
     * @stability stable
     */
    registerConnectable(connectable, portRange) {
        portRange = portRange || ec2.Port.tcp(this.defaultPort);
        // Notify all listeners that we already know about of this new connectable.
        // Then remember for new listeners that might get added later.
        this.connectableMembers.push({ connectable, portRange });
        for (const listener of this.listeners) {
            listener.registerConnectable(connectable, portRange);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     *
     * @stability stable
     */
    registerListener(listener, associatingConstruct) {
        // Notify this listener of all connectables that we know about.
        // Then remember for new connectables that might get added later.
        for (const member of this.connectableMembers) {
            listener.registerConnectable(member.connectable, member.portRange);
        }
        this.listeners.push(listener);
        this.loadBalancerAttachedDependencies.add((associatingConstruct || listener));
    }
    /**
     * Full name of first load balancer.
     *
     * @stability stable
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * Return the given named metric for this Application Load Balancer Target Group.
     *
     * Returns the metric for this target group from the point of view of the first
     * load balancer load balancing to it. If you have multiple load balancers load
     * sending traffic to the same target group, you will have to override the dimensions
     * on this metric.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: {
                TargetGroup: this.targetGroupFullName,
                LoadBalancer: this.firstLoadBalancerFullName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of IPv6 requests received by the target group.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricIpv6RequestCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.ApplicationELBMetrics.iPv6RequestCountSum, props);
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricRequestCount(props) {
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.ApplicationELBMetrics.requestCountSum, props);
    }
    /**
     * The number of healthy hosts in the target group.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of unhealthy hosts in the target group.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricUnhealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets in this target group.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The average number of requests received by each target in a target group.
     *
     * The only valid statistic is Sum. Note that this represents the average not the sum.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricRequestCountPerTarget(props) {
        return this.metric('RequestCountPerTarget', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const ret = super.validate();
        if (this.targetType !== undefined && this.targetType !== enums_1.TargetType.LAMBDA
            && (this.protocol === undefined || this.port === undefined)) {
            ret.push('At least one of \'port\' or \'protocol\' is required for a non-Lambda TargetGroup');
        }
        if (this.healthCheck && this.healthCheck.protocol) {
            if (ALB_HEALTH_CHECK_PROTOCOLS.includes(this.healthCheck.protocol)) {
                if (this.healthCheck.interval && this.healthCheck.timeout &&
                    this.healthCheck.interval.toMilliseconds() <= this.healthCheck.timeout.toMilliseconds()) {
                    ret.push(`Healthcheck interval ${this.healthCheck.interval.toHumanString()} must be greater than the timeout ${this.healthCheck.timeout.toHumanString()}`);
                }
            }
            if (!ALB_HEALTH_CHECK_PROTOCOLS.includes(this.healthCheck.protocol)) {
                ret.push([
                    `Health check protocol '${this.healthCheck.protocol}' is not supported. `,
                    `Must be one of [${ALB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
                ].join(''));
            }
        }
        return ret;
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({
                LoadBalancer: this.firstLoadBalancerFullName,
                TargetGroup: this.targetGroupFullName,
            }),
            ...props,
        }).attachTo(this);
    }
}
exports.ApplicationTargetGroup = ApplicationTargetGroup;
_a = JSII_RTTI_SYMBOL_1;
ApplicationTargetGroup[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.ApplicationTargetGroup", version: "1.129.0" };
/**
 * An imported application target group
 */
class ImportedApplicationTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener, _associatingConstruct) {
        // Nothing to do, we know nothing of our members
        core_1.Annotations.of(this).addWarning('Cannot register listener on imported target group -- security groups might need to be updated manually');
    }
    registerConnectable(_connectable, _portRange) {
        core_1.Annotations.of(this).addWarning('Cannot register connectable on imported target group -- security groups might need to be updated manually');
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const ALB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
//# sourceMappingURL=data:application/json;base64,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