"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobExecutable = exports.JobType = exports.PythonVersion = exports.JobLanguage = exports.GlueVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * AWS Glue version determines the versions of Apache Spark and Python that are available to the job.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/add-job.html.
 *
 * If you need to use a GlueVersion that doesn't exist as a static member, you
 * can instantiate a `GlueVersion` object, e.g: `GlueVersion.of('1.5')`.
 */
class GlueVersion {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom Glue version
     * @param version custom version
     */
    static of(version) {
        return new GlueVersion(version);
    }
}
exports.GlueVersion = GlueVersion;
_a = JSII_RTTI_SYMBOL_1;
GlueVersion[_a] = { fqn: "@aws-cdk/aws-glue-alpha.GlueVersion", version: "2.71.0-alpha.0" };
/**
 * Glue version using Spark 2.2.1 and Python 2.7
 */
GlueVersion.V0_9 = new GlueVersion('0.9');
/**
 * Glue version using Spark 2.4.3, Python 2.7 and Python 3.6
 */
GlueVersion.V1_0 = new GlueVersion('1.0');
/**
 * Glue version using Spark 2.4.3 and Python 3.7
 */
GlueVersion.V2_0 = new GlueVersion('2.0');
/**
 * Glue version using Spark 3.1.1 and Python 3.7
 */
GlueVersion.V3_0 = new GlueVersion('3.0');
/**
 * Glue version using Spark 3.3.0 and Python 3.10
 */
GlueVersion.V4_0 = new GlueVersion('4.0');
/**
 * Runtime language of the Glue job
 */
var JobLanguage;
(function (JobLanguage) {
    /**
     * Scala
     */
    JobLanguage["SCALA"] = "scala";
    /**
     * Python
     */
    JobLanguage["PYTHON"] = "python";
})(JobLanguage = exports.JobLanguage || (exports.JobLanguage = {}));
/**
 * Python version
 */
var PythonVersion;
(function (PythonVersion) {
    /**
     * Python 2 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["TWO"] = "2";
    /**
     * Python 3 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["THREE"] = "3";
    /**
     * Python 3.9 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["THREE_NINE"] = "3.9";
})(PythonVersion = exports.PythonVersion || (exports.PythonVersion = {}));
/**
 * The job type.
 *
 * If you need to use a JobType that doesn't exist as a static member, you
 * can instantiate a `JobType` object, e.g: `JobType.of('other name')`.
 */
class JobType {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom type name
     * @param name type name
     */
    static of(name) {
        return new JobType(name);
    }
}
exports.JobType = JobType;
_b = JSII_RTTI_SYMBOL_1;
JobType[_b] = { fqn: "@aws-cdk/aws-glue-alpha.JobType", version: "2.71.0-alpha.0" };
/**
 * Command for running a Glue Spark job.
 */
JobType.ETL = new JobType('glueetl');
/**
 * Command for running a Glue Spark streaming job.
 */
JobType.STREAMING = new JobType('gluestreaming');
/**
 * Command for running a Glue python shell job.
 */
JobType.PYTHON_SHELL = new JobType('pythonshell');
/**
 * Command for running a Glue Ray job.
 */
JobType.RAY = new JobType('glueray');
/**
 * The executable properties related to the Glue job's GlueVersion, JobType and code
 */
class JobExecutable {
    constructor(config) {
        if (JobType.PYTHON_SHELL === config.type) {
            if (config.language !== JobLanguage.PYTHON) {
                throw new Error('Python shell requires the language to be set to Python');
            }
            if ([GlueVersion.V0_9, GlueVersion.V3_0, GlueVersion.V4_0].includes(config.glueVersion)) {
                throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support Python Shell`);
            }
        }
        if (JobType.RAY === config.type) {
            if (config.language !== JobLanguage.PYTHON) {
                throw new Error('Ray requires the language to be set to Python');
            }
            if ([GlueVersion.V0_9, GlueVersion.V1_0, GlueVersion.V2_0, GlueVersion.V3_0].includes(config.glueVersion)) {
                throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support Ray`);
            }
        }
        if (config.extraJarsFirst && [GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support extraJarsFirst`);
        }
        if (config.pythonVersion === PythonVersion.TWO && ![GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support PythonVersion ${config.pythonVersion}`);
        }
        if (JobLanguage.PYTHON !== config.language && config.extraPythonFiles) {
            throw new Error('extraPythonFiles is not supported for languages other than JobLanguage.PYTHON');
        }
        if (config.pythonVersion === PythonVersion.THREE_NINE && config.type !== JobType.PYTHON_SHELL && config.type !== JobType.RAY) {
            throw new Error('Specified PythonVersion PythonVersion.THREE_NINE is only supported for JobType Python Shell and Ray');
        }
        if (config.pythonVersion === PythonVersion.THREE && config.type === JobType.RAY) {
            throw new Error('Specified PythonVersion PythonVersion.THREE is not supported for Ray');
        }
        this.config = config;
    }
    /**
     * Create Scala executable props for Apache Spark ETL job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Scala executable props for Apache Spark Streaming job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Python executable props for Apache Spark ETL job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for Apache Spark Streaming job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for python shell jobs.
     *
     * @param props Python Shell Job props.
     */
    static pythonShell(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonShellExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonShell);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.PYTHON_SHELL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for Ray jobs.
     *
     * @param props Ray Job props.
     */
    static pythonRay(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonRayExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonRay);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.RAY,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create a custom JobExecutable.
     *
     * @param config custom job executable configuration.
     */
    static of(config) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobExecutableConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.of);
            }
            throw error;
        }
        return new JobExecutable(config);
    }
    /**
     * Called during Job initialization to get JobExecutableConfig.
     */
    bind() {
        return this.config;
    }
}
exports.JobExecutable = JobExecutable;
_c = JSII_RTTI_SYMBOL_1;
JobExecutable[_c] = { fqn: "@aws-cdk/aws-glue-alpha.JobExecutable", version: "2.71.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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