import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import { DockerImageAssetProps } from '@aws-cdk/aws-ecr-assets';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as s3 from '@aws-cdk/aws-s3';
import * as sfn from '@aws-cdk/aws-stepfunctions';
import { Duration, Size } from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * Task to train a machine learning model using Amazon SageMaker.
 *
 * @stability stable
 */
export interface ISageMakerTask extends iam.IGrantable {
}
/**
 * Specify the training algorithm and algorithm-specific metadata.
 *
 * @stability stable
 */
export interface AlgorithmSpecification {
    /**
     * Name of the algorithm resource to use for the training job.
     *
     * This must be an algorithm resource that you created or subscribe to on AWS Marketplace.
     * If you specify a value for this parameter, you can't specify a value for TrainingImage.
     *
     * @default - No algorithm is specified
     * @stability stable
     */
    readonly algorithmName?: string;
    /**
     * List of metric definition objects.
     *
     * Each object specifies the metric name and regular expressions used to parse algorithm logs.
     *
     * @default - No metrics
     * @stability stable
     */
    readonly metricDefinitions?: MetricDefinition[];
    /**
     * Registry path of the Docker image that contains the training algorithm.
     *
     * @default - No Docker image is specified
     * @stability stable
     */
    readonly trainingImage?: DockerImage;
    /**
     * Input mode that the algorithm supports.
     *
     * @default 'File' mode
     * @stability stable
     */
    readonly trainingInputMode?: InputMode;
}
/**
 * Describes the training, validation or test dataset and the Amazon S3 location where it is stored.
 *
 * @stability stable
 */
export interface Channel {
    /**
     * Name of the channel.
     *
     * @stability stable
     */
    readonly channelName: string;
    /**
     * Compression type if training data is compressed.
     *
     * @default - None
     * @stability stable
     */
    readonly compressionType?: CompressionType;
    /**
     * The MIME type of the data.
     *
     * @default - None
     * @stability stable
     */
    readonly contentType?: string;
    /**
     * Location of the channel data.
     *
     * @stability stable
     */
    readonly dataSource: DataSource;
    /**
     * Input mode to use for the data channel in a training job.
     *
     * @default - None
     * @stability stable
     */
    readonly inputMode?: InputMode;
    /**
     * Specify RecordIO as the value when input data is in raw format but the training algorithm requires the RecordIO format.
     *
     * In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record.
     * If the input data is already in RecordIO format, you don't need to set this attribute.
     *
     * @default - None
     * @stability stable
     */
    readonly recordWrapperType?: RecordWrapperType;
    /**
     * Shuffle config option for input data in a channel.
     *
     * @default - None
     * @stability stable
     */
    readonly shuffleConfig?: ShuffleConfig;
}
/**
 * Configuration for a shuffle option for input data in a channel.
 *
 * @stability stable
 */
export interface ShuffleConfig {
    /**
     * Determines the shuffling order.
     *
     * @stability stable
     */
    readonly seed: number;
}
/**
 * Location of the channel data.
 *
 * @stability stable
 */
export interface DataSource {
    /**
     * S3 location of the data source that is associated with a channel.
     *
     * @stability stable
     */
    readonly s3DataSource: S3DataSource;
}
/**
 * S3 location of the channel data.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_S3DataSource.html
 * @stability stable
 */
export interface S3DataSource {
    /**
     * List of one or more attribute names to use that are found in a specified augmented manifest file.
     *
     * @default - No attribute names
     * @stability stable
     */
    readonly attributeNames?: string[];
    /**
     * S3 Data Distribution Type.
     *
     * @default - None
     * @stability stable
     */
    readonly s3DataDistributionType?: S3DataDistributionType;
    /**
     * S3 Data Type.
     *
     * @default S3_PREFIX
     * @stability stable
     */
    readonly s3DataType?: S3DataType;
    /**
     * S3 Uri.
     *
     * @stability stable
     */
    readonly s3Location: S3Location;
}
/**
 * Configures the S3 bucket where SageMaker will save the result of model training.
 *
 * @stability stable
 */
export interface OutputDataConfig {
    /**
     * Optional KMS encryption key that Amazon SageMaker uses to encrypt the model artifacts at rest using Amazon S3 server-side encryption.
     *
     * @default - Amazon SageMaker uses the default KMS key for Amazon S3 for your role's account
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Identifies the S3 path where you want Amazon SageMaker to store the model artifacts.
     *
     * @stability stable
     */
    readonly s3OutputLocation: S3Location;
}
/**
 * Specifies a limit to how long a model training job can run.
 *
 * When the job reaches the time limit, Amazon SageMaker ends the training job.
 *
 * @stability stable
 */
export interface StoppingCondition {
    /**
     * The maximum length of time, in seconds, that the training or compilation job can run.
     *
     * @default - 1 hour
     * @stability stable
     */
    readonly maxRuntime?: Duration;
}
/**
 * Specifies the resources, ML compute instances, and ML storage volumes to deploy for model training.
 *
 * @stability stable
 */
export interface ResourceConfig {
    /**
     * The number of ML compute instances to use.
     *
     * @default 1 instance.
     * @stability stable
     */
    readonly instanceCount: number;
    /**
     * ML compute instance type.
     *
     * @default is the 'm4.xlarge' instance type.
     * @stability stable
     */
    readonly instanceType: ec2.InstanceType;
    /**
     * KMS key that Amazon SageMaker uses to encrypt data on the storage volume attached to the ML compute instance(s) that run the training job.
     *
     * @default - Amazon SageMaker uses the default KMS key for Amazon S3 for your role's account
     * @stability stable
     */
    readonly volumeEncryptionKey?: kms.IKey;
    /**
     * Size of the ML storage volume that you want to provision.
     *
     * @default 10 GB EBS volume.
     * @stability stable
     */
    readonly volumeSize: Size;
}
/**
 * Specifies the VPC that you want your Amazon SageMaker training job to connect to.
 *
 * @stability stable
 */
export interface VpcConfig {
    /**
     * VPC.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
    /**
     * VPC subnets.
     *
     * @default - Private Subnets are selected
     * @stability stable
     */
    readonly subnets?: ec2.SubnetSelection;
}
/**
 * Specifies the metric name and regular expressions used to parse algorithm logs.
 *
 * @stability stable
 */
export interface MetricDefinition {
    /**
     * Name of the metric.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * Regular expression that searches the output of a training job and gets the value of the metric.
     *
     * @stability stable
     */
    readonly regex: string;
}
/**
 * Stores information about the location of an object in Amazon S3.
 *
 * @stability stable
 */
export interface S3LocationConfig {
    /**
     * Uniquely identifies the resource in Amazon S3.
     *
     * @stability stable
     */
    readonly uri: string;
}
/**
 * Constructs `IS3Location` objects.
 *
 * @stability stable
 */
export declare abstract class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     * @stability stable
     */
    static fromBucket(bucket: s3.IBucket, keyPrefix: string): S3Location;
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     * @stability stable
     */
    static fromJsonExpression(expression: string): S3Location;
    /**
     * Called when the S3Location is bound to a StepFunctions task.
     *
     * @stability stable
     */
    abstract bind(task: ISageMakerTask, opts: S3LocationBindOptions): S3LocationConfig;
}
/**
 * Options for binding an S3 Location.
 *
 * @stability stable
 */
export interface S3LocationBindOptions {
    /**
     * Allow reading from the S3 Location.
     *
     * @default false
     * @stability stable
     */
    readonly forReading?: boolean;
    /**
     * Allow writing to the S3 Location.
     *
     * @default false
     * @stability stable
     */
    readonly forWriting?: boolean;
}
/**
 * Configuration for a using Docker image.
 *
 * @stability stable
 */
export interface DockerImageConfig {
    /**
     * The fully qualified URI of the Docker image.
     *
     * @stability stable
     */
    readonly imageUri: string;
}
/**
 * Creates `IDockerImage` instances.
 *
 * @stability stable
 */
export declare abstract class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`.
     * @stability stable
     */
    static fromEcrRepository(repository: ecr.IRepository, tag?: string): DockerImage;
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     * @stability stable
     */
    static fromJsonExpression(expression: string, allowAnyEcrImagePull?: boolean): DockerImage;
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     * @stability stable
     */
    static fromRegistry(imageUri: string): DockerImage;
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     * @stability stable
     */
    static fromAsset(scope: Construct, id: string, props: DockerImageAssetProps): DockerImage;
    /**
     * Called when the image is used by a SageMaker task.
     *
     * @stability stable
     */
    abstract bind(task: ISageMakerTask): DockerImageConfig;
}
/**
 * S3 Data Type.
 *
 * @stability stable
 */
export declare enum S3DataType {
    /**
     * Manifest File Data Type.
     *
     * @stability stable
     */
    MANIFEST_FILE = "ManifestFile",
    /**
     * S3 Prefix Data Type.
     *
     * @stability stable
     */
    S3_PREFIX = "S3Prefix",
    /**
     * Augmented Manifest File Data Type.
     *
     * @stability stable
     */
    AUGMENTED_MANIFEST_FILE = "AugmentedManifestFile"
}
/**
 * S3 Data Distribution Type.
 *
 * @stability stable
 */
export declare enum S3DataDistributionType {
    /**
     * Fully replicated S3 Data Distribution Type.
     *
     * @stability stable
     */
    FULLY_REPLICATED = "FullyReplicated",
    /**
     * Sharded By S3 Key Data Distribution Type.
     *
     * @stability stable
     */
    SHARDED_BY_S3_KEY = "ShardedByS3Key"
}
/**
 * Define the format of the input data.
 *
 * @stability stable
 */
export declare enum RecordWrapperType {
    /**
     * None record wrapper type.
     *
     * @stability stable
     */
    NONE = "None",
    /**
     * RecordIO record wrapper type.
     *
     * @stability stable
     */
    RECORD_IO = "RecordIO"
}
/**
 * Input mode that the algorithm supports.
 *
 * @stability stable
 */
export declare enum InputMode {
    /**
     * Pipe mode.
     *
     * @stability stable
     */
    PIPE = "Pipe",
    /**
     * File mode.
     *
     * @stability stable
     */
    FILE = "File"
}
/**
 * Compression type of the data.
 *
 * @stability stable
 */
export declare enum CompressionType {
    /**
     * None compression type.
     *
     * @stability stable
     */
    NONE = "None",
    /**
     * Gzip compression type.
     *
     * @stability stable
     */
    GZIP = "Gzip"
}
/**
 * Configures the timeout and maximum number of retries for processing a transform job invocation.
 *
 * @stability stable
 */
export interface ModelClientOptions {
    /**
     * The maximum number of retries when invocation requests are failing.
     *
     * @default 0
     * @stability stable
     */
    readonly invocationsMaxRetries?: number;
    /**
     * The timeout duration for an invocation request.
     *
     * @default Duration.minutes(1)
     * @stability stable
     */
    readonly invocationsTimeout?: Duration;
}
/**
 * Dataset to be transformed and the Amazon S3 location where it is stored.
 *
 * @stability stable
 */
export interface TransformInput {
    /**
     * The compression type of the transform data.
     *
     * @default NONE
     * @stability stable
     */
    readonly compressionType?: CompressionType;
    /**
     * Multipurpose internet mail extension (MIME) type of the data.
     *
     * @default - None
     * @stability stable
     */
    readonly contentType?: string;
    /**
     * S3 location of the channel data.
     *
     * @stability stable
     */
    readonly transformDataSource: TransformDataSource;
    /**
     * Method to use to split the transform job's data files into smaller batches.
     *
     * @default NONE
     * @stability stable
     */
    readonly splitType?: SplitType;
}
/**
 * S3 location of the input data that the model can consume.
 *
 * @stability stable
 */
export interface TransformDataSource {
    /**
     * S3 location of the input data.
     *
     * @stability stable
     */
    readonly s3DataSource: TransformS3DataSource;
}
/**
 * Location of the channel data.
 *
 * @stability stable
 */
export interface TransformS3DataSource {
    /**
     * S3 Data Type.
     *
     * @default 'S3Prefix'
     * @stability stable
     */
    readonly s3DataType?: S3DataType;
    /**
     * Identifies either a key name prefix or a manifest.
     *
     * @stability stable
     */
    readonly s3Uri: string;
}
/**
 * S3 location where you want Amazon SageMaker to save the results from the transform job.
 *
 * @stability stable
 */
export interface TransformOutput {
    /**
     * MIME type used to specify the output data.
     *
     * @default - None
     * @stability stable
     */
    readonly accept?: string;
    /**
     * Defines how to assemble the results of the transform job as a single S3 object.
     *
     * @default - None
     * @stability stable
     */
    readonly assembleWith?: AssembleWith;
    /**
     * AWS KMS key that Amazon SageMaker uses to encrypt the model artifacts at rest using Amazon S3 server-side encryption.
     *
     * @default - default KMS key for Amazon S3 for your role's account.
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * S3 path where you want Amazon SageMaker to store the results of the transform job.
     *
     * @stability stable
     */
    readonly s3OutputPath: string;
}
/**
 * ML compute instances for the transform job.
 *
 * @stability stable
 */
export interface TransformResources {
    /**
     * Number of ML compute instances to use in the transform job.
     *
     * @stability stable
     */
    readonly instanceCount: number;
    /**
     * ML compute instance type for the transform job.
     *
     * @stability stable
     */
    readonly instanceType: ec2.InstanceType;
    /**
     * AWS KMS key that Amazon SageMaker uses to encrypt data on the storage volume attached to the ML compute instance(s).
     *
     * @default - None
     * @stability stable
     */
    readonly volumeEncryptionKey?: kms.IKey;
}
/**
 * Properties to define a ContainerDefinition.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 * @stability stable
 */
export interface ContainerDefinitionOptions {
    /**
     * The Amazon EC2 Container Registry (Amazon ECR) path where inference code is stored.
     *
     * @default - None
     * @stability stable
     */
    readonly image?: DockerImage;
    /**
     * The environment variables to set in the Docker container.
     *
     * @default - No variables
     * @stability stable
     */
    readonly environmentVariables?: sfn.TaskInput;
    /**
     * The name or Amazon Resource Name (ARN) of the model package to use to create the model.
     *
     * @default - None
     * @stability stable
     */
    readonly modelPackageName?: string;
    /**
     * Defines how many models the container hosts.
     *
     * @default - Mode.SINGLE_MODEL
     * @stability stable
     */
    readonly mode?: Mode;
    /**
     * This parameter is ignored for models that contain only a PrimaryContainer.
     *
     * When a ContainerDefinition is part of an inference pipeline,
     * the value of the parameter uniquely identifies the container for the purposes of logging and metrics.
     *
     * @default - None
     * @stability stable
     */
    readonly containerHostName?: string;
    /**
     * The S3 path where the model artifacts, which result from model training, are stored.
     *
     * This path must point to a single gzip compressed tar archive (.tar.gz suffix).
     * The S3 path is required for Amazon SageMaker built-in algorithms, but not if you use your own algorithms.
     *
     * @default - None
     * @stability stable
     */
    readonly modelS3Location?: S3Location;
}
/**
 * Describes the container, as part of model definition.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 * @stability stable
 */
export declare class ContainerDefinition implements IContainerDefinition {
    private readonly options;
    /**
     * @stability stable
     */
    constructor(options: ContainerDefinitionOptions);
    /**
     * Called when the ContainerDefinition type configured on Sagemaker Task.
     *
     * @stability stable
     */
    bind(task: ISageMakerTask): ContainerDefinitionConfig;
}
/**
 * Configuration of the container used to host the model.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 * @stability stable
 */
export interface IContainerDefinition {
    /**
     * Called when the ContainerDefinition is used by a SageMaker task.
     *
     * @stability stable
     */
    bind(task: ISageMakerTask): ContainerDefinitionConfig;
}
/**
 * Configuration options for the ContainerDefinition.
 *
 * @stability stable
 */
export interface ContainerDefinitionConfig {
    /**
     * Additional parameters to pass to the base task.
     *
     * @default - No additional parameters passed
     * @stability stable
     */
    readonly parameters?: {
        [key: string]: any;
    };
}
/**
 * Specifies how many models the container hosts.
 *
 * @stability stable
 */
export declare enum Mode {
    /**
     * Container hosts a single model.
     *
     * @stability stable
     */
    SINGLE_MODEL = "SingleModel",
    /**
     * Container hosts multiple models.
     *
     * @see https://docs.aws.amazon.com/sagemaker/latest/dg/multi-model-endpoints.html
     * @stability stable
     */
    MULTI_MODEL = "MultiModel"
}
/**
 * Identifies a model that you want to host and the resources to deploy for hosting it.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ProductionVariant.html
 * @stability stable
 */
export interface ProductionVariant {
    /**
     * The size of the Elastic Inference (EI) instance to use for the production variant.
     *
     * @default - None
     * @stability stable
     */
    readonly acceleratorType?: AcceleratorType;
    /**
     * Number of instances to launch initially.
     *
     * @default - 1
     * @stability stable
     */
    readonly initialInstanceCount?: number;
    /**
     * Determines initial traffic distribution among all of the models that you specify in the endpoint configuration.
     *
     * @default - 1.0
     * @stability stable
     */
    readonly initialVariantWeight?: number;
    /**
     * The ML compute instance type.
     *
     * @stability stable
     */
    readonly instanceType: ec2.InstanceType;
    /**
     * The name of the production variant.
     *
     * @stability stable
     */
    readonly variantName: string;
    /**
     * The name of the model that you want to host.
     *
     * This is the name that you specified when creating the model.
     *
     * @stability stable
     */
    readonly modelName: string;
}
/**
 * The generation of Elastic Inference (EI) instance.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 * @stability stable
 */
export declare class AcceleratorClass {
    readonly version: string;
    /**
     * Elastic Inference accelerator 1st generation.
     *
     * @stability stable
     */
    static readonly EIA1: AcceleratorClass;
    /**
     * Elastic Inference accelerator 2nd generation.
     *
     * @stability stable
     */
    static readonly EIA2: AcceleratorClass;
    /**
     * Custom AcceleratorType.
     *
     * @param version - Elastic Inference accelerator generation.
     * @stability stable
     */
    static of(version: string): AcceleratorClass;
    /**
     * @param version - Elastic Inference accelerator generation
     */
    private constructor();
}
/**
 * The size of the Elastic Inference (EI) instance to use for the production variant.
 *
 * EI instances provide on-demand GPU computing for inference
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 * @stability stable
 */
export declare class AcceleratorType {
    private readonly instanceTypeIdentifier;
    /**
     * AcceleratorType.
     *
     * This class takes a combination of a class and size.
     *
     * @stability stable
     */
    static of(acceleratorClass: AcceleratorClass, instanceSize: ec2.InstanceSize): AcceleratorType;
    /**
     * @stability stable
     */
    constructor(instanceTypeIdentifier: string);
    /**
     * Return the accelerator type as a dotted string.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @stability stable
 */
export declare enum BatchStrategy {
    /**
     * Fits multiple records in a mini-batch.
     *
     * @stability stable
     */
    MULTI_RECORD = "MultiRecord",
    /**
     * Use a single record when making an invocation request.
     *
     * @stability stable
     */
    SINGLE_RECORD = "SingleRecord"
}
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @stability stable
 */
export declare enum SplitType {
    /**
     * Input data files are not split,.
     *
     * @stability stable
     */
    NONE = "None",
    /**
     * Split records on a newline character boundary.
     *
     * @stability stable
     */
    LINE = "Line",
    /**
     * Split using MXNet RecordIO format.
     *
     * @stability stable
     */
    RECORD_IO = "RecordIO",
    /**
     * Split using TensorFlow TFRecord format.
     *
     * @stability stable
     */
    TF_RECORD = "TFRecord"
}
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @stability stable
 */
export declare enum AssembleWith {
    /**
     * Concatenate the results in binary format.
     *
     * @stability stable
     */
    NONE = "None",
    /**
     * Add a newline character at the end of every transformed record.
     *
     * @stability stable
     */
    LINE = "Line"
}
