"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const ecr = require("../lib");
module.exports = {
    'construct repository'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo');
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                Repo02AC86CF: {
                    Type: "AWS::ECR::Repository",
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                }
            }
        });
        test.done();
    },
    'tag-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ tagPrefixList: ['abc'], maxImageCount: 1 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"abc\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":1},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'add day-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = new ecr.Repository(stack, 'Repo');
        repo.addLifecycleRule({
            maxImageAge: cdk.Duration.days(5),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"sinceImagePushed\",\"countNumber\":5,\"countUnit\":\"days\"},\"action\":{\"type\":\"expire\"}}]}",
            }
        }));
        test.done();
    },
    'add count-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({
            maxImageCount: 5,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}}]}",
            }
        }));
        test.done();
    },
    'mixing numbered and unnumbered rules'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['a'], maxImageCount: 5 });
        repo.addLifecycleRule({ rulePriority: 10, tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['b'], maxImageCount: 5 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":10,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"b\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}},{\"rulePriority\":11,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"a\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'tagstatus Any is automatically sorted to the back'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ maxImageCount: 5 });
        repo.addLifecycleRule({ tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['important'], maxImageCount: 999 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"important\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":999},\"action\":{\"type\":\"expire\"}},{\"rulePriority\":2,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'lifecycle rules can be added upon initialization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo', {
            lifecycleRules: [
                { maxImageCount: 3 }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            "LifecyclePolicy": {
                // tslint:disable-next-line:max-line-length
                "LifecyclePolicyText": "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"imageCountMoreThan\",\"countNumber\":3},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'calculate repository URI'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        const uri = repo.repositoryUri;
        // THEN
        const arnSplit = { 'Fn::Split': [':', { 'Fn::GetAtt': ['Repo02AC86CF', 'Arn'] }] };
        test.deepEqual(stack.resolve(uri), {
            'Fn::Join': ['', [
                    { 'Fn::Select': [4, arnSplit] },
                    '.dkr.ecr.',
                    { 'Fn::Select': [3, arnSplit] },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'Repo02AC86CF' }
                ]]
        });
        test.done();
    },
    'import with concrete arn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo2 = ecr.Repository.fromRepositoryArn(stack, 'repo', 'arn:aws:ecr:us-east-1:585695036304:repository/foo/bar/foo/fooo');
        // THEN
        test.deepEqual(stack.resolve(repo2.repositoryArn), 'arn:aws:ecr:us-east-1:585695036304:repository/foo/bar/foo/fooo');
        test.deepEqual(stack.resolve(repo2.repositoryName), 'foo/bar/foo/fooo');
        test.done();
    },
    'fails if importing with token arn and no name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN/THEN
        test.throws(() => ecr.Repository.fromRepositoryArn(stack, 'arn', cdk.Fn.getAtt('Boom', 'Boom').toString()), /\"repositoryArn\" is a late-bound value, and therefore \"repositoryName\" is required\. Use \`fromRepositoryAttributes\` instead/);
        test.done();
    },
    'import with token arn and repository name (see awslabs/aws-cdk#1232)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = ecr.Repository.fromRepositoryAttributes(stack, 'Repo', {
            repositoryArn: cdk.Fn.getAtt('Boom', 'Arn').toString(),
            repositoryName: cdk.Fn.getAtt('Boom', 'Name').toString()
        });
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryArn), { 'Fn::GetAtt': ['Boom', 'Arn'] });
        test.deepEqual(stack.resolve(repo.repositoryName), { 'Fn::GetAtt': ['Boom', 'Name'] });
        test.done();
    },
    'import only with a repository name (arn is deduced)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = ecr.Repository.fromRepositoryName(stack, 'just-name', 'my-repo');
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ecr:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':repository/my-repo'
                ]
            ]
        });
        test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
        test.done();
    },
    'arnForLocalRepository can be used to render an ARN for a local repository'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repoName = cdk.Fn.getAtt('Boom', 'Name').toString();
        // WHEN
        const repo = ecr.Repository.fromRepositoryAttributes(stack, 'Repo', {
            repositoryArn: ecr.Repository.arnForLocalRepository(repoName, stack),
            repositoryName: repoName
        });
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryName), { 'Fn::GetAtt': ['Boom', 'Name'] });
        test.deepEqual(stack.resolve(repo.repositoryArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ecr:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':repository/',
                    { 'Fn::GetAtt': ['Boom', 'Name'] }
                ]]
        });
        test.done();
    },
    'resource policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addToResourcePolicy(new iam.PolicyStatement({ actions: ['*'] }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            RepositoryPolicyText: {
                Statement: [
                    {
                        Action: "*",
                        Effect: "Allow"
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
    'events': {
        'onImagePushed without imageTag creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onCloudTrailImagePushed('EventRule', {
                target: {
                    bind: () => ({ arn: 'ARN', id: '' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.ecr",
                    ],
                    "detail": {
                        "eventName": [
                            "PutImage",
                        ],
                        "requestParameters": {
                            "repositoryName": [
                                {
                                    "Ref": "Repo02AC86CF"
                                }
                            ],
                        },
                    },
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        'onImageScanCompleted without imageTags creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onImageScanCompleted('EventRule', {
                target: {
                    bind: () => ({ arn: 'ARN', id: '' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.ecr",
                    ],
                    "detail": {
                        "repository-name": [
                            {
                                "Ref": "Repo02AC86CF"
                            }
                        ],
                        "scan-status": [
                            "COMPLETE"
                        ]
                    }
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        'onImageScanCompleted with one imageTag creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onImageScanCompleted('EventRule', {
                imageTags: ['some-tag'],
                target: {
                    bind: () => ({ arn: 'ARN', id: '' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.ecr",
                    ],
                    "detail": {
                        "repository-name": [
                            {
                                "Ref": "Repo02AC86CF"
                            }
                        ],
                        "image-tags": [
                            "some-tag"
                        ],
                        "scan-status": [
                            "COMPLETE"
                        ]
                    }
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        'onImageScanCompleted with multiple imageTags creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onImageScanCompleted('EventRule', {
                imageTags: ['tag1', 'tag2', 'tag3'],
                target: {
                    bind: () => ({ arn: 'ARN', id: '' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.ecr",
                    ],
                    "detail": {
                        "repository-name": [
                            {
                                "Ref": "Repo02AC86CF"
                            }
                        ],
                        "image-tags": [
                            "tag1",
                            "tag2",
                            "tag3"
                        ],
                        "scan-status": [
                            "COMPLETE"
                        ]
                    }
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        'removal policy is "Retain" by default'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new ecr.Repository(stack, 'Repo');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                "Type": "AWS::ECR::Repository",
                "DeletionPolicy": "Retain"
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        '"Delete" removal policy can be set explicitly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new ecr.Repository(stack, 'Repo', {
                removalPolicy: core_1.RemovalPolicy.DESTROY
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                "Type": "AWS::ECR::Repository",
                "DeletionPolicy": "Delete"
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'grant adds appropriate resource-*'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const repo = new ecr.Repository(stack, 'TestHarnessRepo');
            // WHEN
            repo.grantPull(new iam.AnyPrincipal());
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                "RepositoryPolicyText": {
                    "Statement": [
                        {
                            "Action": [
                                "ecr:BatchCheckLayerAvailability",
                                "ecr:GetDownloadUrlForLayer",
                                "ecr:BatchGetImage"
                            ],
                            "Effect": "Allow",
                            "Principal": "*",
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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