"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobExecutable = exports.JobType = exports.PythonVersion = exports.JobLanguage = exports.GlueVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * AWS Glue version determines the versions of Apache Spark and Python that are available to the job.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/add-job.html.
 *
 * If you need to use a GlueVersion that doesn't exist as a static member, you
 * can instantiate a `GlueVersion` object, e.g: `GlueVersion.of('1.5')`.
 */
class GlueVersion {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom Glue version
     * @param version custom version
     */
    static of(version) {
        return new GlueVersion(version);
    }
}
exports.GlueVersion = GlueVersion;
_a = JSII_RTTI_SYMBOL_1;
GlueVersion[_a] = { fqn: "@aws-cdk/aws-glue-alpha.GlueVersion", version: "2.63.0-alpha.0" };
/**
 * Glue version using Spark 2.2.1 and Python 2.7
 */
GlueVersion.V0_9 = new GlueVersion('0.9');
/**
 * Glue version using Spark 2.4.3, Python 2.7 and Python 3.6
 */
GlueVersion.V1_0 = new GlueVersion('1.0');
/**
 * Glue version using Spark 2.4.3 and Python 3.7
 */
GlueVersion.V2_0 = new GlueVersion('2.0');
/**
 * Glue version using Spark 3.1.1 and Python 3.7
 */
GlueVersion.V3_0 = new GlueVersion('3.0');
/**
 * Glue version using Spark 3.3.0 and Python 3.10
 */
GlueVersion.V4_0 = new GlueVersion('4.0');
/**
 * Runtime language of the Glue job
 */
var JobLanguage;
(function (JobLanguage) {
    /**
     * Scala
     */
    JobLanguage["SCALA"] = "scala";
    /**
     * Python
     */
    JobLanguage["PYTHON"] = "python";
})(JobLanguage = exports.JobLanguage || (exports.JobLanguage = {}));
/**
 * Python version
 */
var PythonVersion;
(function (PythonVersion) {
    /**
     * Python 2 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["TWO"] = "2";
    /**
     * Python 3 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["THREE"] = "3";
    /**
     * Python 3.9 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["THREE_NINE"] = "3.9";
})(PythonVersion = exports.PythonVersion || (exports.PythonVersion = {}));
/**
 * The job type.
 *
 * If you need to use a JobType that doesn't exist as a static member, you
 * can instantiate a `JobType` object, e.g: `JobType.of('other name')`.
 */
class JobType {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom type name
     * @param name type name
     */
    static of(name) {
        return new JobType(name);
    }
}
exports.JobType = JobType;
_b = JSII_RTTI_SYMBOL_1;
JobType[_b] = { fqn: "@aws-cdk/aws-glue-alpha.JobType", version: "2.63.0-alpha.0" };
/**
 * Command for running a Glue ETL job.
 */
JobType.ETL = new JobType('glueetl');
/**
 * Command for running a Glue streaming job.
 */
JobType.STREAMING = new JobType('gluestreaming');
/**
 * Command for running a Glue python shell job.
 */
JobType.PYTHON_SHELL = new JobType('pythonshell');
/**
 * The executable properties related to the Glue job's GlueVersion, JobType and code
 */
class JobExecutable {
    constructor(config) {
        if (JobType.PYTHON_SHELL === config.type) {
            if (config.language !== JobLanguage.PYTHON) {
                throw new Error('Python shell requires the language to be set to Python');
            }
            if ([GlueVersion.V0_9, GlueVersion.V2_0, GlueVersion.V3_0, GlueVersion.V4_0].includes(config.glueVersion)) {
                throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support Python Shell`);
            }
        }
        if (config.extraJarsFirst && [GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support extraJarsFirst`);
        }
        if (config.pythonVersion === PythonVersion.TWO && ![GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support PythonVersion ${config.pythonVersion}`);
        }
        if (JobLanguage.PYTHON !== config.language && config.extraPythonFiles) {
            throw new Error('extraPythonFiles is not supported for languages other than JobLanguage.PYTHON');
        }
        if (config.pythonVersion === PythonVersion.THREE_NINE && config.type !== JobType.PYTHON_SHELL) {
            throw new Error('Specified PythonVersion PythonVersion.THREE_NINE is only supported for JobType Python Shell');
        }
        this.config = config;
    }
    /**
     * Create Scala executable props for Apache Spark ETL job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Scala executable props for Apache Spark Streaming job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Python executable props for Apache Spark ETL job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for Apache Spark Streaming job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for python shell jobs.
     *
     * @param props Python Shell Job props.
     */
    static pythonShell(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonShellExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonShell);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.PYTHON_SHELL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create a custom JobExecutable.
     *
     * @param config custom job executable configuration.
     */
    static of(config) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobExecutableConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.of);
            }
            throw error;
        }
        return new JobExecutable(config);
    }
    /**
     * Called during Job initialization to get JobExecutableConfig.
     */
    bind() {
        return this.config;
    }
}
exports.JobExecutable = JobExecutable;
_c = JSII_RTTI_SYMBOL_1;
JobExecutable[_c] = { fqn: "@aws-cdk/aws-glue-alpha.JobExecutable", version: "2.63.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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