"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Table = exports.TableEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const s3 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cr = require("aws-cdk-lib/custom-resources");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    TableEncryption["KMS"] = "SSE-KMS";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * A Glue table.
 */
class Table extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName ??
                aws_cdk_lib_1.Lazy.string({
                    produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, {}).toLowerCase(),
                }),
        });
        /**
         * Partition indexes must be created one at a time. To avoid
         * race conditions, we store the resource and add dependencies
         * each time a new partition index is created.
         */
        this.partitionIndexCustomResources = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Table);
            }
            throw error;
        }
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = props.s3Prefix ?? '';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = props.compressed ?? false;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new aws_glue_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${this.physicalName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    'classification': props.dataFormat.classificationString?.value,
                    'has_encrypted_data': this.encryption !== TableEncryption.UNENCRYPTED,
                    'partition_filtering.enabled': props.enablePartitionFiltering,
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: props.storedAsSubDirectories ?? false,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className,
                    },
                },
                tableType: 'EXTERNAL_TABLE',
            },
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`,
        });
        this.node.defaultChild = tableResource;
        // Partition index creation relies on created table.
        if (props.partitionIndexes) {
            this.partitionIndexes = props.partitionIndexes;
            this.partitionIndexes.forEach((index) => this.addPartitionIndex(index));
        }
    }
    static fromTableArn(scope, id, tableArn) {
        const tableName = aws_cdk_lib_1.Fn.select(1, aws_cdk_lib_1.Fn.split('/', aws_cdk_lib_1.Stack.of(scope).splitArn(tableArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName,
        });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a partition index to the table. You can have a maximum of 3 partition
     * indexes to a table. Partition index keys must be a subset of the table's
     * partition keys.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/partition-indexes.html
     */
    addPartitionIndex(index) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PartitionIndex(index);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPartitionIndex);
            }
            throw error;
        }
        const numPartitions = this.partitionIndexCustomResources.length;
        if (numPartitions >= 3) {
            throw new Error('Maximum number of partition indexes allowed is 3');
        }
        this.validatePartitionIndex(index);
        const indexName = index.indexName ?? this.generateIndexName(index.keyNames);
        const partitionIndexCustomResource = new cr.AwsCustomResource(this, `partition-index-${indexName}`, {
            onCreate: {
                service: 'Glue',
                action: 'createPartitionIndex',
                parameters: {
                    DatabaseName: this.database.databaseName,
                    TableName: this.tableName,
                    PartitionIndex: {
                        IndexName: indexName,
                        Keys: index.keyNames,
                    },
                },
                physicalResourceId: cr.PhysicalResourceId.of(indexName),
            },
            policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                resources: cr.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
            // APIs are available in 2.1055.0
            installLatestAwsSdk: false,
        });
        this.grantToUnderlyingResources(partitionIndexCustomResource, ['glue:UpdateTable']);
        // Depend on previous partition index if possible, to avoid race condition
        if (numPartitions > 0) {
            this.partitionIndexCustomResources[numPartitions - 1].node.addDependency(partitionIndexCustomResource);
        }
        this.partitionIndexCustomResources.push(partitionIndexCustomResource);
    }
    generateIndexName(keys) {
        const prefix = keys.join('-') + '-';
        const uniqueId = aws_cdk_lib_1.Names.uniqueId(this);
        const maxIndexLength = 80; // arbitrarily specified
        const startIndex = Math.max(0, uniqueId.length - (maxIndexLength - prefix.length));
        return prefix + uniqueId.substring(startIndex);
    }
    validatePartitionIndex(index) {
        if (index.indexName !== undefined && (index.indexName.length < 1 || index.indexName.length > 255)) {
            throw new Error(`Index name must be between 1 and 255 characters, but got ${index.indexName.length}`);
        }
        if (!this.partitionKeys || this.partitionKeys.length === 0) {
            throw new Error('The table must have partition keys to create a partition index');
        }
        const keyNames = this.partitionKeys.map(pk => pk.name);
        if (!index.keyNames.every(k => keyNames.includes(k))) {
            throw new Error(`All index keys must also be partition keys. Got ${index.keyNames} but partition key names are ${keyNames}`);
        }
    }
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant the given identity custom permissions.
     */
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
    /**
     * Grant the given identity custom permissions to ALL underlying resources of the table.
     * Permissions will be granted to the catalog, the database, and the table.
     */
    grantToUnderlyingResources(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [
                this.tableArn,
                this.database.catalogArn,
                this.database.databaseArn,
            ],
            actions,
        });
    }
    getS3PrefixForGrant() {
        return this.s3Prefix + '*';
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-glue-alpha.Table", version: "2.63.0-alpha.0" };
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but \'${column.name}\' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey,
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey,
    };
}
const readPermissions = [
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersion',
    'glue:GetTableVersions',
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition',
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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