#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const version = require('../package.json').version;
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const [moduleFamily, moduleBaseName] = (namespace === 'AWS::Serverless' ? 'AWS::SAM' : namespace).split('::');
        const moduleName = `${moduleFamily}-${moduleBaseName.replace(/V\d+$/, '')}`.toLocaleLowerCase();
        const packagePath = path.join(root, moduleName);
        const lowcaseModuleName = moduleBaseName.toLocaleLowerCase();
        const packageName = `@aws-cdk/${moduleName}`;
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${moduleFamily}-${moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        // dotnet names
        const dotnetPackage = `Amazon.CDK.${moduleFamily}.${moduleBaseName}`;
        // java names
        const javaGroupId = 'software.amazon.awscdk';
        const javaPackage = moduleFamily === 'AWS'
            ? `services.${lowcaseModuleName}`
            : `${moduleFamily.toLocaleLowerCase()}.${lowcaseModuleName}`;
        const javaArtifactId = moduleFamily === 'AWS'
            ? lowcaseModuleName
            : `${moduleFamily.toLocaleLowerCase()}-${lowcaseModuleName}`;
        // python names
        const pythonDistName = `aws-cdk.${moduleName}`;
        const pythonModuleName = pythonDistName.replace(/-/g, '_');
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${packageName}...`);
        await write('package.json', {
            name: packageName,
            version,
            description: `The CDK Construct Library for ${namespace}`,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: dotnetPackage,
                        packageId: dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/master/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${javaGroupId}.${javaPackage}`,
                        maven: {
                            groupId: javaGroupId,
                            artifactId: javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 1',
                        ],
                        distName: pythonDistName,
                        module: pythonModuleName,
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
                'build+extract': 'yarn build && yarn rosetta:extract',
                'build+test+extract': 'yarn build+test && yarn rosetta:extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@types/jest': '^26.0.22',
                '@aws-cdk/assertions': version,
                'cdk-build-tools': version,
                'cfn2ts': version,
                'pkglint': version,
            },
            dependencies: {
                '@aws-cdk/core': version,
            },
            peerDependencies: {
                '@aws-cdk/core': version,
            },
            engines: {
                node: '>= 10.13.0 <13 || >=13.7.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
            '!*.lit.ts',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assertions';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await write('README.md', [
            `# ${namespace} Construct Library`,
            '<!--BEGIN STABILITY BANNER-->',
            '',
            '---',
            '',
            '![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)',
            '',
            '> All classes with the `Cfn` prefix in this module ([CFN Resources]) are always stable and safe to use.',
            '>',
            '> [CFN Resources]: https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib',
            '',
            '---',
            '',
            '<!--END STABILITY BANNER-->',
            '',
            'This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.',
            '',
            '```ts',
            `import ${lowcaseModuleName} = require('${packageName}');`,
            '```',
        ]);
        await write('.eslintrc.js', [
            "const baseConfig = require('cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('monocdk', packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('decdk', packageName, version, ['dependencies']);
    }
}
/**
 * A few of our packages (e.g., decdk, aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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