"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateSpecification = void 0;
const schema = require("../lib/schema");
function validateSpecification(test, specification) {
    validateResourceTypes(test, specification);
    validatePropertyTypes(test, specification);
}
exports.validateSpecification = validateSpecification;
function validateResourceTypes(test, specification) {
    for (const typeName of Object.keys(specification.ResourceTypes)) {
        test.ok(typeName, 'Resource type name is not empty');
        const type = specification.ResourceTypes[typeName];
        test.notEqual(type.Documentation, null, `${typeName} is documented`);
        if (type.ScrutinyType) {
            test.ok(schema.isResourceScrutinyType(type.ScrutinyType), `${typeName}.ScrutinyType is not a valid ResourceScrutinyType`);
        }
        if (type.Properties) {
            validateProperties(typeName, test, type.Properties, specification);
        }
        if (type.Attributes) {
            validateAttributes(typeName, test, type.Attributes, specification);
        }
    }
}
function validatePropertyTypes(test, specification) {
    for (const typeName of Object.keys(specification.PropertyTypes)) {
        test.ok(typeName, 'Property type name is not empty');
        const type = specification.PropertyTypes[typeName];
        if (schema.isRecordType(type)) {
            validateProperties(typeName, test, type.Properties, specification);
        }
        else {
            validateProperties(typeName, test, { '<this>': type }, specification);
        }
    }
}
function validateProperties(typeName, test, properties, specification) {
    const expectedKeys = ['Documentation', 'Required', 'UpdateType', 'ScrutinyType'];
    for (const name of Object.keys(properties)) {
        const property = properties[name];
        test.notEqual(property.Documentation, '', `${typeName}.Properties.${name} is documented`);
        test.ok(!property.UpdateType || schema.isUpdateType(property.UpdateType), `${typeName}.Properties.${name} has valid UpdateType`);
        if (property.ScrutinyType !== undefined) {
            test.ok(schema.isPropertyScrutinyType(property.ScrutinyType), `${typeName}.Properties.${name} has valid ScrutinyType`);
        }
        if (schema.isPrimitiveProperty(property)) {
            test.ok(schema.isPrimitiveType(property.PrimitiveType), `${typeName}.Properties.${name} has a valid PrimitiveType`);
            expectedKeys.push('PrimitiveType');
        }
        else if (schema.isPrimitiveListProperty(property)) {
            expectedKeys.push('Type', 'DuplicatesAllowed', 'PrimitiveItemType');
            test.ok(schema.isPrimitiveType(property.PrimitiveItemType), `${typeName}.Properties.${name} has a valid PrimitiveItemType`);
        }
        else if (schema.isPrimitiveMapProperty(property)) {
            expectedKeys.push('Type', 'DuplicatesAllowed', 'PrimitiveItemType', 'Type');
            test.ok(schema.isPrimitiveType(property.PrimitiveItemType), `${typeName}.Properties.${name} has a valid PrimitiveItemType`);
            test.ok(!property.DuplicatesAllowed, `${typeName}.Properties.${name} does not allow duplicates`);
        }
        else if (schema.isComplexListProperty(property)) {
            expectedKeys.push('Type', 'DuplicatesAllowed', 'ItemType', 'Type');
            test.ok(property.ItemType, `${typeName}.Properties.${name} has a valid ItemType`);
            if (property.ItemType !== 'Tag') {
                const fqn = `${typeName.split('.')[0]}.${property.ItemType}`;
                const resolvedType = specification.PropertyTypes && specification.PropertyTypes[fqn];
                test.ok(resolvedType, `${typeName}.Properties.${name} ItemType (${fqn}) resolves`);
            }
        }
        else if (schema.isMapOfStructsProperty(property)) {
            expectedKeys.push('Type', 'DuplicatesAllowed', 'ItemType', 'Type');
            test.ok(property.ItemType, `${typeName}.Properties.${name} has a valid ItemType`);
            const fqn = `${typeName.split('.')[0]}.${property.ItemType}`;
            const resolvedType = specification.PropertyTypes && specification.PropertyTypes[fqn];
            test.ok(resolvedType, `${typeName}.Properties.${name} ItemType (${fqn}) resolves`);
            test.ok(!property.DuplicatesAllowed, `${typeName}.Properties.${name} does not allow duplicates`);
        }
        else if (schema.isMapOfListsOfPrimitivesProperty(property)) {
            expectedKeys.push('Type', 'DuplicatesAllowed', 'ItemType', 'PrimitiveItemItemType', 'Type');
            test.ok(schema.isPrimitiveType(property.PrimitiveItemItemType), `${typeName}.Properties.${name} has a valid PrimitiveItemItemType`);
            test.ok(!property.DuplicatesAllowed, `${typeName}.Properties.${name} does not allow duplicates`);
        }
        else if (schema.isComplexProperty(property)) {
            expectedKeys.push('Type');
            test.ok(property.Type, `${typeName}.Properties.${name} has a valid type`);
            const fqn = `${typeName.split('.')[0]}.${property.Type}`;
            const resolvedType = specification.PropertyTypes && specification.PropertyTypes[fqn];
            test.ok(resolvedType, `${typeName}.Properties.${name} type (${fqn}) resolves`);
        }
        else if (schema.isUnionProperty(property)) {
            expectedKeys.push('PrimitiveTypes', 'PrimitiveItemTypes', 'ItemTypes', 'Types');
            if (property.PrimitiveTypes) {
                for (const type of property.PrimitiveTypes) {
                    test.ok(schema.isPrimitiveType(type), `${typeName}.Properties.${name} has only valid PrimitiveTypes`);
                }
            }
            if (property.ItemTypes) {
                for (const type of property.ItemTypes) {
                    const fqn = `${typeName.split('.')[0]}.${type}`;
                    const resolvedType = specification.PropertyTypes && specification.PropertyTypes[fqn];
                    test.ok(resolvedType, `${typeName}.Properties.${name} type (${fqn}) resolves`);
                }
            }
            if (property.Types) {
                for (const type of property.Types) {
                    const fqn = `${typeName.split('.')[0]}.${type}`;
                    const resolvedType = specification.PropertyTypes && specification.PropertyTypes[fqn];
                    test.ok(resolvedType, `${typeName}.Properties.${name} type (${fqn}) resolves`);
                }
            }
        }
        else {
            test.ok(false, `${typeName}.Properties.${name} has known type`);
        }
        test.deepEqual(without(Object.keys(property), expectedKeys), [], `${typeName}.Properties.${name} has no extra properties`);
    }
}
function validateAttributes(typeName, test, attributes, specification) {
    for (const name of Object.keys(attributes)) {
        const attribute = attributes[name];
        test.ok(('Type' in attribute) !== ('PrimitiveType' in attribute), 'One of, and only one of, Type or PrimitiveType must be present');
        if (schema.isPrimitiveAttribute(attribute)) {
            test.ok(!schema.isListAttribute(attribute), `${typeName}.Attributes.${name} is only a Primitive type`);
            test.ok(schema.isPrimitiveType(attribute.PrimitiveType), `${typeName}.Attributes.${name} has a valid PrimitiveType`);
            test.ok(!('PrimitiveItemType' in attribute), `${typeName}.Attributes.${name} has no PrimitiveItemType`);
            test.ok(!('ItemType' in attribute), `${typeName}.Attributes.${name} has no ItemType`);
        }
        else if (schema.isPrimitiveListAttribute(attribute)) {
            test.ok(!schema.isComplexListAttribute(attribute), `${typeName}.Attributes.${name} is only a List<Primitive> type`);
            test.ok(schema.isPrimitiveType(attribute.PrimitiveItemType), `${typeName}.Attributes.${name} has a valid PrimitiveItemType`);
            test.ok(!('ItemType' in attribute), `${typeName}.Attributes.${name} has no ItemType`);
        }
        else if (schema.isComplexListAttribute(attribute)) {
            test.ok(attribute.ItemType, `${typeName}.Attributes.${name} is a valid List<Complex> type`);
            const fqn = `${typeName.split('.')[0]}.${attribute.ItemType}`;
            const resolvedType = specification.PropertyTypes && specification.PropertyTypes[fqn];
            test.ok(resolvedType, `${typeName}.Attributes.${name} ItemType (${fqn}) resolves`);
            test.ok(!('PrimitiveItemType' in attribute), `${typeName}.Attributes.${name} has no PrimitiveItemType`);
        }
        else if (schema.isPrimitiveMapAttribute(attribute)) {
            test.ok(schema.isPrimitiveType(attribute.PrimitiveItemType), `${typeName}.Attributes.${name} has a valid PrimitiveItemType`);
            test.ok(!('ItemType' in attribute), `${typeName}.Attributes.${name} has no ItemType`);
        }
        else {
            test.ok(false, `${typeName}.Attributes.${name} has a valid type`);
        }
    }
}
/**
 * Remove elements from a set
 */
function without(xs, ...sets) {
    const ret = new Set(xs);
    for (const set of sets) {
        for (const element of set) {
            if (ret.has(element)) {
                ret.delete(element);
            }
        }
    }
    return Array.from(ret);
}
//# sourceMappingURL=data:application/json;base64,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