import { Construct } from 'constructs';
import { Duration } from '../duration';
import { Size } from '../size';
import { Construct as CoreConstruct } from '../construct-compat';
/**
 * Initialization properties for `CustomResourceProvider`.
 *
 * @stability stable
 */
export interface CustomResourceProviderProps {
    /**
     * A local file system directory with the provider's code.
     *
     * The code will be
     * bundled into a zip asset and wired to the provider's AWS Lambda function.
     *
     * @stability stable
     */
    readonly codeDirectory: string;
    /**
     * The AWS Lambda runtime and version to use for the provider.
     *
     * @stability stable
     */
    readonly runtime: CustomResourceProviderRuntime;
    /**
     * A set of IAM policy statements to include in the inline policy of the provider's lambda function.
     *
     * @default - no additional inline policy
     * @stability stable
     * @example
     *
     *   [{ Effect: 'Allow', Action: 's3:PutObject*', Resource: '*' }]
     */
    readonly policyStatements?: any[];
    /**
     * AWS Lambda timeout for the provider.
     *
     * @default Duration.minutes(15)
     * @stability stable
     */
    readonly timeout?: Duration;
    /**
     * The amount of memory that your function has access to.
     *
     * Increasing the
     * function's memory also increases its CPU allocation.
     *
     * @default Size.mebibytes(128)
     * @stability stable
     */
    readonly memorySize?: Size;
    /**
     * Key-value pairs that are passed to Lambda as Environment.
     *
     * @default - No environment variables.
     * @stability stable
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * A description of the function.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
}
/**
 * The lambda runtime to use for the resource provider.
 *
 * This also indicates
 * which language is used for the handler.
 *
 * @stability stable
 */
export declare enum CustomResourceProviderRuntime {
    /**
     * (deprecated) Node.js 12.x.
     *
     * @deprecated Use {@link NODEJS_12_X}
     */
    NODEJS_12 = "nodejs12.x",
    /**
     * Node.js 12.x
     */
    NODEJS_12_X = "nodejs12.x",
    /**
     * Node.js 14.x.
     *
     * @stability stable
     */
    NODEJS_14_X = "nodejs14.x"
}
/**
 * An AWS-Lambda backed custom resource provider.
 *
 * @stability stable
 */
export declare class CustomResourceProvider extends CoreConstruct {
    /**
     * Returns a stack-level singleton ARN (service token) for the custom resource provider.
     *
     * @param scope Construct scope.
     * @param uniqueid A globally unique id that will be used for the stack-level construct.
     * @param props Provider properties which will only be applied when the provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     * @stability stable
     */
    static getOrCreate(scope: Construct, uniqueid: string, props: CustomResourceProviderProps): string;
    /**
     * Returns a stack-level singleton for the custom resource provider.
     *
     * @param scope Construct scope.
     * @param uniqueid A globally unique id that will be used for the stack-level construct.
     * @param props Provider properties which will only be applied when the provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     * @stability stable
     */
    static getOrCreateProvider(scope: Construct, uniqueid: string, props: CustomResourceProviderProps): CustomResourceProvider;
    /**
     * The ARN of the provider's AWS Lambda function which should be used as the `serviceToken` when defining a custom resource.
     *
     * @stability stable
     * @example
     *
     *   new CustomResource(this, 'MyCustomResource', {
     *     // ...
     *     serviceToken: myProvider.serviceToken, // <--- here
     *   })
     */
    readonly serviceToken: string;
    /**
     * The ARN of the provider's AWS Lambda function role.
     *
     * @stability stable
     */
    readonly roleArn: string;
    /**
     * @stability stable
     */
    protected constructor(scope: Construct, id: string, props: CustomResourceProviderProps);
    private renderEnvironmentVariables;
}
