"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const default_1 = require("../lib/default");
const aws_entities_1 = require("./aws-entities");
const fact_tables_1 = require("./fact-tables");
async function main() {
    checkRegions(fact_tables_1.APPMESH_ECR_ACCOUNTS);
    checkRegions(fact_tables_1.DLC_REPOSITORY_ACCOUNTS);
    checkRegions(fact_tables_1.ELBV2_ACCOUNTS);
    checkRegions(fact_tables_1.FIREHOSE_CIDR_BLOCKS);
    checkRegions(fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS);
    checkRegionsSubMap(fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS);
    const lines = [
        "import { Fact, FactName } from './fact';",
        '',
        '/* eslint-disable quote-props */',
        '/* eslint-disable max-len */',
        '',
        '/**',
        ' * Built-in regional information, re-generated by `npm run build`.',
        ' *',
        ` * @generated ${new Date().toISOString()}`,
        ' */',
        'export class BuiltIns {',
        '  /**',
        '   * Registers all the built in regional data in the RegionInfo database.',
        '   */',
        '  public static register(): void {',
    ];
    const defaultMap = 'default';
    for (const region of aws_entities_1.AWS_REGIONS) {
        let partition = fact_tables_1.PARTITION_MAP[defaultMap].partition;
        let domainSuffix = fact_tables_1.PARTITION_MAP[defaultMap].domainSuffix;
        for (const key in fact_tables_1.PARTITION_MAP) {
            if (region.startsWith(key)) {
                partition = fact_tables_1.PARTITION_MAP[key].partition;
                domainSuffix = fact_tables_1.PARTITION_MAP[key].domainSuffix;
            }
        }
        registerFact(region, 'PARTITION', partition);
        registerFact(region, 'DOMAIN_SUFFIX', domainSuffix);
        registerFact(region, 'CDK_METADATA_RESOURCE_AVAILABLE', fact_tables_1.AWS_CDK_METADATA.has(region) ? 'YES' : 'NO');
        registerFact(region, 'S3_STATIC_WEBSITE_ENDPOINT', fact_tables_1.AWS_OLDER_REGIONS.has(region)
            ? `s3-website-${region}.${domainSuffix}`
            : `s3-website.${region}.${domainSuffix}`);
        registerFact(region, 'S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID', fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS[region] || '');
        registerFact(region, 'ELBV2_ACCOUNT', fact_tables_1.ELBV2_ACCOUNTS[region]);
        registerFact(region, 'DLC_REPOSITORY_ACCOUNT', fact_tables_1.DLC_REPOSITORY_ACCOUNTS[region]);
        registerFact(region, 'APPMESH_ECR_ACCOUNT', fact_tables_1.APPMESH_ECR_ACCOUNTS[region]);
        const firehoseCidrBlock = fact_tables_1.FIREHOSE_CIDR_BLOCKS[region];
        if (firehoseCidrBlock) {
            registerFact(region, 'FIREHOSE_CIDR_BLOCK', `${fact_tables_1.FIREHOSE_CIDR_BLOCKS[region]}/27`);
        }
        const vpcEndpointServiceNamePrefix = `${domainSuffix.split('.').reverse().join('.')}.vpce`;
        registerFact(region, 'VPC_ENDPOINT_SERVICE_NAME_PREFIX', vpcEndpointServiceNamePrefix);
        for (const service of aws_entities_1.AWS_SERVICES) {
            registerFact(region, ['servicePrincipal', service], default_1.Default.servicePrincipal(service, region, domainSuffix));
        }
        for (const version in fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS) {
            registerFact(region, ['cloudwatchLambdaInsightsVersion', version], fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS[version][region]);
        }
    }
    lines.push('  }');
    lines.push('');
    lines.push('  private constructor() {}');
    lines.push('}');
    await fs.writeFile(path.resolve(__dirname, '..', 'lib', 'built-ins.generated.ts'), lines.join('\n'));
    function registerFact(region, name, value) {
        const factName = typeof name === 'string' ? name : `${name[0]}(${name.slice(1).map(s => JSON.stringify(s)).join(', ')})`;
        lines.push(`    Fact.register({ region: ${JSON.stringify(region)}, name: FactName.${factName}, value: ${JSON.stringify(value)} });`);
    }
}
/**
 * Verifies that the provided map of region to fact does not contain an entry
 * for a region that was not registered in `AWS_REGIONS`.
 */
function checkRegions(map) {
    const allRegions = new Set(aws_entities_1.AWS_REGIONS);
    for (const region of Object.keys(map)) {
        if (!allRegions.has(region)) {
            throw new Error(`Un-registered region fact found: ${region}. Add to AWS_REGIONS list!`);
        }
    }
}
/**
 * Verifies that the provided map of <KEY> to region to fact does not contain an entry
 * for a region that was not registered in `AWS_REGIONS`.
 */
function checkRegionsSubMap(map) {
    const allRegions = new Set(aws_entities_1.AWS_REGIONS);
    for (const key of Object.keys(map)) {
        for (const region of Object.keys(map[key])) {
            if (!allRegions.has(region)) {
                throw new Error(`Un-registered region fact found: ${region}. Add to AWS_REGIONS list!`);
            }
        }
    }
}
main().catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exit(-1);
});
//# sourceMappingURL=data:application/json;base64,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