"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bootstrapVersionFromTemplate = exports.BootstrapStack = void 0;
const os = require("os");
const path = require("path");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const aws_auth_1 = require("../aws-auth");
const deploy_stack_1 = require("../deploy-stack");
const toolkit_info_1 = require("../toolkit-info");
const bootstrap_props_1 = require("./bootstrap-props");
/**
 * A class to hold state around stack bootstrapping
 *
 * This class exists so we can break bootstrapping into 2 phases:
 *
 * ```ts
 * const current = BootstrapStack.lookup(...);
 * // ...
 * current.update(newTemplate, ...);
 * ```
 *
 * And do something in between the two phases (such as look at the
 * current bootstrap stack and doing something intelligent).
 */
class BootstrapStack {
    constructor(sdkProvider, sdk, resolvedEnvironment, toolkitStackName, currentToolkitInfo) {
        this.sdkProvider = sdkProvider;
        this.sdk = sdk;
        this.resolvedEnvironment = resolvedEnvironment;
        this.toolkitStackName = toolkitStackName;
        this.currentToolkitInfo = currentToolkitInfo;
    }
    static async lookup(sdkProvider, environment, toolkitStackName) {
        toolkitStackName = toolkitStackName !== null && toolkitStackName !== void 0 ? toolkitStackName : toolkit_info_1.DEFAULT_TOOLKIT_STACK_NAME;
        const resolvedEnvironment = await sdkProvider.resolveEnvironment(environment);
        const sdk = await sdkProvider.forEnvironment(resolvedEnvironment, aws_auth_1.Mode.ForWriting);
        const currentToolkitInfo = await toolkit_info_1.ToolkitInfo.lookup(resolvedEnvironment, sdk, toolkitStackName);
        return new BootstrapStack(sdkProvider, sdk, resolvedEnvironment, toolkitStackName, currentToolkitInfo);
    }
    get parameters() {
        return this.currentToolkitInfo.found ? this.currentToolkitInfo.bootstrapStack.parameters : {};
    }
    get terminationProtection() {
        return this.currentToolkitInfo.found ? this.currentToolkitInfo.bootstrapStack.terminationProtection : undefined;
    }
    async partition() {
        return (await this.sdk.currentAccount()).partition;
    }
    /**
     * Perform the actual deployment of a bootstrap stack, given a template and some parameters
     */
    async update(template, parameters, options) {
        var _a;
        const newVersion = bootstrapVersionFromTemplate(template);
        if (this.currentToolkitInfo.found && newVersion < this.currentToolkitInfo.version && !options.force) {
            throw new Error(`Not downgrading existing bootstrap stack from version '${this.currentToolkitInfo.version}' to version '${newVersion}'. Use --force to force.`);
        }
        const outdir = await fs.mkdtemp(path.join(os.tmpdir(), 'cdk-bootstrap'));
        const builder = new cxapi.CloudAssemblyBuilder(outdir);
        const templateFile = `${this.toolkitStackName}.template.json`;
        await fs.writeJson(path.join(builder.outdir, templateFile), template, { spaces: 2 });
        builder.addArtifact(this.toolkitStackName, {
            type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
            environment: cxapi.EnvironmentUtils.format(this.resolvedEnvironment.account, this.resolvedEnvironment.region),
            properties: {
                templateFile,
                terminationProtection: (_a = options.terminationProtection) !== null && _a !== void 0 ? _a : false,
            },
        });
        const assembly = builder.buildAssembly();
        return deploy_stack_1.deployStack({
            stack: assembly.getStackByName(this.toolkitStackName),
            resolvedEnvironment: this.resolvedEnvironment,
            sdk: this.sdk,
            sdkProvider: this.sdkProvider,
            force: options.force,
            roleArn: options.roleArn,
            tags: options.tags,
            execute: options.execute,
            parameters,
            usePreviousParameters: true,
            // Obviously we can't need a bootstrap stack to deploy a bootstrap stack
            toolkitInfo: toolkit_info_1.ToolkitInfo.bootstraplessDeploymentsOnly(this.sdk),
        });
    }
}
exports.BootstrapStack = BootstrapStack;
function bootstrapVersionFromTemplate(template) {
    var _a, _b, _c, _d, _e;
    const versionSources = [
        (_b = (_a = template.Outputs) === null || _a === void 0 ? void 0 : _a[bootstrap_props_1.BOOTSTRAP_VERSION_OUTPUT]) === null || _b === void 0 ? void 0 : _b.Value,
        (_e = (_d = (_c = template.Resources) === null || _c === void 0 ? void 0 : _c[bootstrap_props_1.BOOTSTRAP_VERSION_RESOURCE]) === null || _d === void 0 ? void 0 : _d.Properties) === null || _e === void 0 ? void 0 : _e.Value,
    ];
    for (const vs of versionSources) {
        if (typeof vs === 'number') {
            return vs;
        }
        if (typeof vs === 'string' && !isNaN(parseInt(vs, 10))) {
            return parseInt(vs, 10);
        }
    }
    return 0;
}
exports.bootstrapVersionFromTemplate = bootstrapVersionFromTemplate;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwbG95LWJvb3RzdHJhcC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlcGxveS1ib290c3RyYXAudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3QiwyREFBMkQ7QUFDM0QseUNBQXlDO0FBQ3pDLCtCQUErQjtBQUMvQiwwQ0FBc0Q7QUFDdEQsa0RBQWlFO0FBQ2pFLGtEQUEwRTtBQUMxRSx1REFBc0g7QUFFdEg7Ozs7Ozs7Ozs7Ozs7R0FhRztBQUNILE1BQWEsY0FBYztJQVl6QixZQUNtQixXQUF3QixFQUN4QixHQUFTLEVBQ1QsbUJBQXNDLEVBQ3RDLGdCQUF3QixFQUN4QixrQkFBK0I7UUFKL0IsZ0JBQVcsR0FBWCxXQUFXLENBQWE7UUFDeEIsUUFBRyxHQUFILEdBQUcsQ0FBTTtRQUNULHdCQUFtQixHQUFuQixtQkFBbUIsQ0FBbUI7UUFDdEMscUJBQWdCLEdBQWhCLGdCQUFnQixDQUFRO1FBQ3hCLHVCQUFrQixHQUFsQixrQkFBa0IsQ0FBYTtJQUNsRCxDQUFDO0lBakJNLE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLFdBQXdCLEVBQUUsV0FBOEIsRUFBRSxnQkFBeUI7UUFDNUcsZ0JBQWdCLEdBQUcsZ0JBQWdCLGFBQWhCLGdCQUFnQixjQUFoQixnQkFBZ0IsR0FBSSx5Q0FBMEIsQ0FBQztRQUVsRSxNQUFNLG1CQUFtQixHQUFHLE1BQU0sV0FBVyxDQUFDLGtCQUFrQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQzlFLE1BQU0sR0FBRyxHQUFHLE1BQU0sV0FBVyxDQUFDLGNBQWMsQ0FBQyxtQkFBbUIsRUFBRSxlQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFbkYsTUFBTSxrQkFBa0IsR0FBRyxNQUFNLDBCQUFXLENBQUMsTUFBTSxDQUFDLG1CQUFtQixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1FBRWhHLE9BQU8sSUFBSSxjQUFjLENBQUMsV0FBVyxFQUFFLEdBQUcsRUFBRSxtQkFBbUIsRUFBRSxnQkFBZ0IsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3pHLENBQUM7SUFVRCxJQUFXLFVBQVU7UUFDbkIsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsY0FBYyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO0lBQ2hHLENBQUM7SUFFRCxJQUFXLHFCQUFxQjtRQUM5QixPQUFPLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxjQUFjLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztJQUNsSCxDQUFDO0lBRU0sS0FBSyxDQUFDLFNBQVM7UUFDcEIsT0FBTyxDQUFDLE1BQU0sSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQztJQUNyRCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxLQUFLLENBQUMsTUFBTSxDQUNqQixRQUFhLEVBQ2IsVUFBOEMsRUFDOUMsT0FBd0Q7O1FBR3hELE1BQU0sVUFBVSxHQUFHLDRCQUE0QixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzFELElBQUksSUFBSSxDQUFDLGtCQUFrQixDQUFDLEtBQUssSUFBSSxVQUFVLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUU7WUFDbkcsTUFBTSxJQUFJLEtBQUssQ0FBQywwREFBMEQsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE9BQU8saUJBQWlCLFVBQVUsMEJBQTBCLENBQUMsQ0FBQztTQUNqSztRQUVELE1BQU0sTUFBTSxHQUFHLE1BQU0sRUFBRSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxlQUFlLENBQUMsQ0FBQyxDQUFDO1FBQ3pFLE1BQU0sT0FBTyxHQUFHLElBQUksS0FBSyxDQUFDLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3ZELE1BQU0sWUFBWSxHQUFHLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixnQkFBZ0IsQ0FBQztRQUM5RCxNQUFNLEVBQUUsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLFlBQVksQ0FBQyxFQUFFLFFBQVEsRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRXJGLE9BQU8sQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQ3pDLElBQUksRUFBRSxRQUFRLENBQUMsWUFBWSxDQUFDLHdCQUF3QjtZQUNwRCxXQUFXLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUM7WUFDN0csVUFBVSxFQUFFO2dCQUNWLFlBQVk7Z0JBQ1oscUJBQXFCLFFBQUUsT0FBTyxDQUFDLHFCQUFxQixtQ0FBSSxLQUFLO2FBQzlEO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUFDO1FBRXpDLE9BQU8sMEJBQVcsQ0FBQztZQUNqQixLQUFLLEVBQUUsUUFBUSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUM7WUFDckQsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLG1CQUFtQjtZQUM3QyxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVc7WUFDN0IsS0FBSyxFQUFFLE9BQU8sQ0FBQyxLQUFLO1lBQ3BCLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTztZQUN4QixJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUk7WUFDbEIsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPO1lBQ3hCLFVBQVU7WUFDVixxQkFBcUIsRUFBRSxJQUFJO1lBQzNCLHdFQUF3RTtZQUN4RSxXQUFXLEVBQUUsMEJBQVcsQ0FBQyw0QkFBNEIsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO1NBQ2hFLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQTdFRCx3Q0E2RUM7QUFFRCxTQUFnQiw0QkFBNEIsQ0FBQyxRQUFhOztJQUN4RCxNQUFNLGNBQWMsR0FBRztvQkFDckIsUUFBUSxDQUFDLE9BQU8sMENBQUcsMENBQXdCLDJDQUFHLEtBQUs7MEJBQ25ELFFBQVEsQ0FBQyxTQUFTLDBDQUFHLDRDQUEwQiwyQ0FBRyxVQUFVLDBDQUFFLEtBQUs7S0FDcEUsQ0FBQztJQUVGLEtBQUssTUFBTSxFQUFFLElBQUksY0FBYyxFQUFFO1FBQy9CLElBQUksT0FBTyxFQUFFLEtBQUssUUFBUSxFQUFFO1lBQUUsT0FBTyxFQUFFLENBQUM7U0FBRTtRQUMxQyxJQUFJLE9BQU8sRUFBRSxLQUFLLFFBQVEsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUU7WUFDdEQsT0FBTyxRQUFRLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1NBQ3pCO0tBQ0Y7SUFDRCxPQUFPLENBQUMsQ0FBQztBQUNYLENBQUM7QUFiRCxvRUFhQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIG9zIGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBjeHNjaGVtYSBmcm9tICdAYXdzLWNkay9jbG91ZC1hc3NlbWJseS1zY2hlbWEnO1xuaW1wb3J0ICogYXMgY3hhcGkgZnJvbSAnQGF3cy1jZGsvY3gtYXBpJztcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzLWV4dHJhJztcbmltcG9ydCB7IE1vZGUsIFNka1Byb3ZpZGVyLCBJU0RLIH0gZnJvbSAnLi4vYXdzLWF1dGgnO1xuaW1wb3J0IHsgZGVwbG95U3RhY2ssIERlcGxveVN0YWNrUmVzdWx0IH0gZnJvbSAnLi4vZGVwbG95LXN0YWNrJztcbmltcG9ydCB7IERFRkFVTFRfVE9PTEtJVF9TVEFDS19OQU1FLCBUb29sa2l0SW5mbyB9IGZyb20gJy4uL3Rvb2xraXQtaW5mbyc7XG5pbXBvcnQgeyBCT09UU1RSQVBfVkVSU0lPTl9PVVRQVVQsIEJvb3RzdHJhcEVudmlyb25tZW50T3B0aW9ucywgQk9PVFNUUkFQX1ZFUlNJT05fUkVTT1VSQ0UgfSBmcm9tICcuL2Jvb3RzdHJhcC1wcm9wcyc7XG5cbi8qKlxuICogQSBjbGFzcyB0byBob2xkIHN0YXRlIGFyb3VuZCBzdGFjayBib290c3RyYXBwaW5nXG4gKlxuICogVGhpcyBjbGFzcyBleGlzdHMgc28gd2UgY2FuIGJyZWFrIGJvb3RzdHJhcHBpbmcgaW50byAyIHBoYXNlczpcbiAqXG4gKiBgYGB0c1xuICogY29uc3QgY3VycmVudCA9IEJvb3RzdHJhcFN0YWNrLmxvb2t1cCguLi4pO1xuICogLy8gLi4uXG4gKiBjdXJyZW50LnVwZGF0ZShuZXdUZW1wbGF0ZSwgLi4uKTtcbiAqIGBgYFxuICpcbiAqIEFuZCBkbyBzb21ldGhpbmcgaW4gYmV0d2VlbiB0aGUgdHdvIHBoYXNlcyAoc3VjaCBhcyBsb29rIGF0IHRoZVxuICogY3VycmVudCBib290c3RyYXAgc3RhY2sgYW5kIGRvaW5nIHNvbWV0aGluZyBpbnRlbGxpZ2VudCkuXG4gKi9cbmV4cG9ydCBjbGFzcyBCb290c3RyYXBTdGFjayB7XG4gIHB1YmxpYyBzdGF0aWMgYXN5bmMgbG9va3VwKHNka1Byb3ZpZGVyOiBTZGtQcm92aWRlciwgZW52aXJvbm1lbnQ6IGN4YXBpLkVudmlyb25tZW50LCB0b29sa2l0U3RhY2tOYW1lPzogc3RyaW5nKSB7XG4gICAgdG9vbGtpdFN0YWNrTmFtZSA9IHRvb2xraXRTdGFja05hbWUgPz8gREVGQVVMVF9UT09MS0lUX1NUQUNLX05BTUU7XG5cbiAgICBjb25zdCByZXNvbHZlZEVudmlyb25tZW50ID0gYXdhaXQgc2RrUHJvdmlkZXIucmVzb2x2ZUVudmlyb25tZW50KGVudmlyb25tZW50KTtcbiAgICBjb25zdCBzZGsgPSBhd2FpdCBzZGtQcm92aWRlci5mb3JFbnZpcm9ubWVudChyZXNvbHZlZEVudmlyb25tZW50LCBNb2RlLkZvcldyaXRpbmcpO1xuXG4gICAgY29uc3QgY3VycmVudFRvb2xraXRJbmZvID0gYXdhaXQgVG9vbGtpdEluZm8ubG9va3VwKHJlc29sdmVkRW52aXJvbm1lbnQsIHNkaywgdG9vbGtpdFN0YWNrTmFtZSk7XG5cbiAgICByZXR1cm4gbmV3IEJvb3RzdHJhcFN0YWNrKHNka1Byb3ZpZGVyLCBzZGssIHJlc29sdmVkRW52aXJvbm1lbnQsIHRvb2xraXRTdGFja05hbWUsIGN1cnJlbnRUb29sa2l0SW5mbyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgY29uc3RydWN0b3IoXG4gICAgcHJpdmF0ZSByZWFkb25seSBzZGtQcm92aWRlcjogU2RrUHJvdmlkZXIsXG4gICAgcHJpdmF0ZSByZWFkb25seSBzZGs6IElTREssXG4gICAgcHJpdmF0ZSByZWFkb25seSByZXNvbHZlZEVudmlyb25tZW50OiBjeGFwaS5FbnZpcm9ubWVudCxcbiAgICBwcml2YXRlIHJlYWRvbmx5IHRvb2xraXRTdGFja05hbWU6IHN0cmluZyxcbiAgICBwcml2YXRlIHJlYWRvbmx5IGN1cnJlbnRUb29sa2l0SW5mbzogVG9vbGtpdEluZm8pIHtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgcGFyYW1ldGVycygpOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+IHtcbiAgICByZXR1cm4gdGhpcy5jdXJyZW50VG9vbGtpdEluZm8uZm91bmQgPyB0aGlzLmN1cnJlbnRUb29sa2l0SW5mby5ib290c3RyYXBTdGFjay5wYXJhbWV0ZXJzIDoge307XG4gIH1cblxuICBwdWJsaWMgZ2V0IHRlcm1pbmF0aW9uUHJvdGVjdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy5jdXJyZW50VG9vbGtpdEluZm8uZm91bmQgPyB0aGlzLmN1cnJlbnRUb29sa2l0SW5mby5ib290c3RyYXBTdGFjay50ZXJtaW5hdGlvblByb3RlY3Rpb24gOiB1bmRlZmluZWQ7XG4gIH1cblxuICBwdWJsaWMgYXN5bmMgcGFydGl0aW9uKCk6IFByb21pc2U8c3RyaW5nPiB7XG4gICAgcmV0dXJuIChhd2FpdCB0aGlzLnNkay5jdXJyZW50QWNjb3VudCgpKS5wYXJ0aXRpb247XG4gIH1cblxuICAvKipcbiAgICogUGVyZm9ybSB0aGUgYWN0dWFsIGRlcGxveW1lbnQgb2YgYSBib290c3RyYXAgc3RhY2ssIGdpdmVuIGEgdGVtcGxhdGUgYW5kIHNvbWUgcGFyYW1ldGVyc1xuICAgKi9cbiAgcHVibGljIGFzeW5jIHVwZGF0ZShcbiAgICB0ZW1wbGF0ZTogYW55LFxuICAgIHBhcmFtZXRlcnM6IFJlY29yZDxzdHJpbmcsIHN0cmluZyB8IHVuZGVmaW5lZD4sXG4gICAgb3B0aW9uczogT21pdDxCb290c3RyYXBFbnZpcm9ubWVudE9wdGlvbnMsICdwYXJhbWV0ZXJzJz4sXG4gICk6IFByb21pc2U8RGVwbG95U3RhY2tSZXN1bHQ+IHtcblxuICAgIGNvbnN0IG5ld1ZlcnNpb24gPSBib290c3RyYXBWZXJzaW9uRnJvbVRlbXBsYXRlKHRlbXBsYXRlKTtcbiAgICBpZiAodGhpcy5jdXJyZW50VG9vbGtpdEluZm8uZm91bmQgJiYgbmV3VmVyc2lvbiA8IHRoaXMuY3VycmVudFRvb2xraXRJbmZvLnZlcnNpb24gJiYgIW9wdGlvbnMuZm9yY2UpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTm90IGRvd25ncmFkaW5nIGV4aXN0aW5nIGJvb3RzdHJhcCBzdGFjayBmcm9tIHZlcnNpb24gJyR7dGhpcy5jdXJyZW50VG9vbGtpdEluZm8udmVyc2lvbn0nIHRvIHZlcnNpb24gJyR7bmV3VmVyc2lvbn0nLiBVc2UgLS1mb3JjZSB0byBmb3JjZS5gKTtcbiAgICB9XG5cbiAgICBjb25zdCBvdXRkaXIgPSBhd2FpdCBmcy5ta2R0ZW1wKHBhdGguam9pbihvcy50bXBkaXIoKSwgJ2Nkay1ib290c3RyYXAnKSk7XG4gICAgY29uc3QgYnVpbGRlciA9IG5ldyBjeGFwaS5DbG91ZEFzc2VtYmx5QnVpbGRlcihvdXRkaXIpO1xuICAgIGNvbnN0IHRlbXBsYXRlRmlsZSA9IGAke3RoaXMudG9vbGtpdFN0YWNrTmFtZX0udGVtcGxhdGUuanNvbmA7XG4gICAgYXdhaXQgZnMud3JpdGVKc29uKHBhdGguam9pbihidWlsZGVyLm91dGRpciwgdGVtcGxhdGVGaWxlKSwgdGVtcGxhdGUsIHsgc3BhY2VzOiAyIH0pO1xuXG4gICAgYnVpbGRlci5hZGRBcnRpZmFjdCh0aGlzLnRvb2xraXRTdGFja05hbWUsIHtcbiAgICAgIHR5cGU6IGN4c2NoZW1hLkFydGlmYWN0VHlwZS5BV1NfQ0xPVURGT1JNQVRJT05fU1RBQ0ssXG4gICAgICBlbnZpcm9ubWVudDogY3hhcGkuRW52aXJvbm1lbnRVdGlscy5mb3JtYXQodGhpcy5yZXNvbHZlZEVudmlyb25tZW50LmFjY291bnQsIHRoaXMucmVzb2x2ZWRFbnZpcm9ubWVudC5yZWdpb24pLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICB0ZW1wbGF0ZUZpbGUsXG4gICAgICAgIHRlcm1pbmF0aW9uUHJvdGVjdGlvbjogb3B0aW9ucy50ZXJtaW5hdGlvblByb3RlY3Rpb24gPz8gZmFsc2UsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgY29uc3QgYXNzZW1ibHkgPSBidWlsZGVyLmJ1aWxkQXNzZW1ibHkoKTtcblxuICAgIHJldHVybiBkZXBsb3lTdGFjayh7XG4gICAgICBzdGFjazogYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUodGhpcy50b29sa2l0U3RhY2tOYW1lKSxcbiAgICAgIHJlc29sdmVkRW52aXJvbm1lbnQ6IHRoaXMucmVzb2x2ZWRFbnZpcm9ubWVudCxcbiAgICAgIHNkazogdGhpcy5zZGssXG4gICAgICBzZGtQcm92aWRlcjogdGhpcy5zZGtQcm92aWRlcixcbiAgICAgIGZvcmNlOiBvcHRpb25zLmZvcmNlLFxuICAgICAgcm9sZUFybjogb3B0aW9ucy5yb2xlQXJuLFxuICAgICAgdGFnczogb3B0aW9ucy50YWdzLFxuICAgICAgZXhlY3V0ZTogb3B0aW9ucy5leGVjdXRlLFxuICAgICAgcGFyYW1ldGVycyxcbiAgICAgIHVzZVByZXZpb3VzUGFyYW1ldGVyczogdHJ1ZSxcbiAgICAgIC8vIE9idmlvdXNseSB3ZSBjYW4ndCBuZWVkIGEgYm9vdHN0cmFwIHN0YWNrIHRvIGRlcGxveSBhIGJvb3RzdHJhcCBzdGFja1xuICAgICAgdG9vbGtpdEluZm86IFRvb2xraXRJbmZvLmJvb3RzdHJhcGxlc3NEZXBsb3ltZW50c09ubHkodGhpcy5zZGspLFxuICAgIH0pO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBib290c3RyYXBWZXJzaW9uRnJvbVRlbXBsYXRlKHRlbXBsYXRlOiBhbnkpOiBudW1iZXIge1xuICBjb25zdCB2ZXJzaW9uU291cmNlcyA9IFtcbiAgICB0ZW1wbGF0ZS5PdXRwdXRzPy5bQk9PVFNUUkFQX1ZFUlNJT05fT1VUUFVUXT8uVmFsdWUsXG4gICAgdGVtcGxhdGUuUmVzb3VyY2VzPy5bQk9PVFNUUkFQX1ZFUlNJT05fUkVTT1VSQ0VdPy5Qcm9wZXJ0aWVzPy5WYWx1ZSxcbiAgXTtcblxuICBmb3IgKGNvbnN0IHZzIG9mIHZlcnNpb25Tb3VyY2VzKSB7XG4gICAgaWYgKHR5cGVvZiB2cyA9PT0gJ251bWJlcicpIHsgcmV0dXJuIHZzOyB9XG4gICAgaWYgKHR5cGVvZiB2cyA9PT0gJ3N0cmluZycgJiYgIWlzTmFOKHBhcnNlSW50KHZzLCAxMCkpKSB7XG4gICAgICByZXR1cm4gcGFyc2VJbnQodnMsIDEwKTtcbiAgICB9XG4gIH1cbiAgcmV0dXJuIDA7XG59Il19