"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcNetworkContextProviderPlugin = void 0;
const cxapi = require("@aws-cdk/cx-api");
const api_1 = require("../api");
const logging_1 = require("../logging");
class VpcNetworkContextProviderPlugin {
    constructor(aws) {
        this.aws = aws;
    }
    async getValue(args) {
        const account = args.account;
        const region = args.region;
        const options = { assumeRoleArn: args.lookupRoleArn };
        const ec2 = (await this.aws.forEnvironment(cxapi.EnvironmentUtils.make(account, region), api_1.Mode.ForReading, options)).ec2();
        const vpcId = await this.findVpc(ec2, args);
        return this.readVpcProps(ec2, vpcId, args);
    }
    async findVpc(ec2, args) {
        // Build request filter (map { Name -> Value } to list of [{ Name, Values }])
        const filters = Object.entries(args.filter).map(([tag, value]) => ({ Name: tag, Values: [value] }));
        logging_1.debug(`Listing VPCs in ${args.account}:${args.region}`);
        const response = await ec2.describeVpcs({ Filters: filters }).promise();
        const vpcs = response.Vpcs || [];
        if (vpcs.length === 0) {
            throw new Error(`Could not find any VPCs matching ${JSON.stringify(args)}`);
        }
        if (vpcs.length > 1) {
            throw new Error(`Found ${vpcs.length} VPCs matching ${JSON.stringify(args)}; please narrow the search criteria`);
        }
        return vpcs[0];
    }
    async readVpcProps(ec2, vpc, args) {
        const vpcId = vpc.VpcId;
        logging_1.debug(`Describing VPC ${vpcId}`);
        const filters = { Filters: [{ Name: 'vpc-id', Values: [vpcId] }] };
        const subnetsResponse = await ec2.describeSubnets(filters).promise();
        const listedSubnets = subnetsResponse.Subnets || [];
        const routeTablesResponse = await ec2.describeRouteTables(filters).promise();
        const routeTables = new RouteTables(routeTablesResponse.RouteTables || []);
        // Now comes our job to separate these subnets out into AZs and subnet groups (Public, Private, Isolated)
        // We have the following attributes to go on:
        // - Type tag, we tag subnets with their type. In absence of this tag, we
        //   fall back to MapPublicIpOnLaunch => must be a Public subnet, anything
        //   else is considered Priate.
        // - Name tag, we tag subnets with their subnet group name. In absence of this tag,
        //   we use the type as the name.
        const azs = Array.from(new Set(listedSubnets.map(s => s.AvailabilityZone)));
        azs.sort();
        const subnets = listedSubnets.map(subnet => {
            let type = getTag('aws-cdk:subnet-type', subnet.Tags);
            if (type === undefined && subnet.MapPublicIpOnLaunch) {
                type = SubnetType.Public;
            }
            if (type === undefined && routeTables.hasRouteToIgw(subnet.SubnetId)) {
                type = SubnetType.Public;
            }
            if (type === undefined) {
                type = SubnetType.Private;
            }
            if (!isValidSubnetType(type)) {
                // eslint-disable-next-line max-len
                throw new Error(`Subnet ${subnet.SubnetArn} has invalid subnet type ${type} (must be ${SubnetType.Public}, ${SubnetType.Private} or ${SubnetType.Isolated})`);
            }
            const name = getTag(args.subnetGroupNameTag || 'aws-cdk:subnet-name', subnet.Tags) || type;
            const routeTableId = routeTables.routeTableIdForSubnetId(subnet.SubnetId);
            if (!routeTableId) {
                throw new Error(`Subnet ${subnet.SubnetArn} does not have an associated route table (and there is no "main" table)`);
            }
            return {
                az: subnet.AvailabilityZone,
                cidr: subnet.CidrBlock,
                type,
                name,
                subnetId: subnet.SubnetId,
                routeTableId,
            };
        });
        let grouped;
        let assymetricSubnetGroups;
        if (args.returnAsymmetricSubnets) {
            grouped = { azs: [], groups: [] };
            assymetricSubnetGroups = groupAsymmetricSubnets(subnets);
        }
        else {
            grouped = groupSubnets(subnets);
            assymetricSubnetGroups = undefined;
        }
        // Find attached+available VPN gateway for this VPC
        const vpnGatewayResponse = await ec2.describeVpnGateways({
            Filters: [
                {
                    Name: 'attachment.vpc-id',
                    Values: [vpcId],
                },
                {
                    Name: 'attachment.state',
                    Values: ['attached'],
                },
                {
                    Name: 'state',
                    Values: ['available'],
                },
            ],
        }).promise();
        const vpnGatewayId = vpnGatewayResponse.VpnGateways && vpnGatewayResponse.VpnGateways.length === 1
            ? vpnGatewayResponse.VpnGateways[0].VpnGatewayId
            : undefined;
        return {
            vpcId,
            vpcCidrBlock: vpc.CidrBlock,
            availabilityZones: grouped.azs,
            isolatedSubnetIds: collapse(flatMap(findGroups(SubnetType.Isolated, grouped), group => group.subnets.map(s => s.subnetId))),
            isolatedSubnetNames: collapse(flatMap(findGroups(SubnetType.Isolated, grouped), group => group.name ? [group.name] : [])),
            isolatedSubnetRouteTableIds: collapse(flatMap(findGroups(SubnetType.Isolated, grouped), group => group.subnets.map(s => s.routeTableId))),
            privateSubnetIds: collapse(flatMap(findGroups(SubnetType.Private, grouped), group => group.subnets.map(s => s.subnetId))),
            privateSubnetNames: collapse(flatMap(findGroups(SubnetType.Private, grouped), group => group.name ? [group.name] : [])),
            privateSubnetRouteTableIds: collapse(flatMap(findGroups(SubnetType.Private, grouped), group => group.subnets.map(s => s.routeTableId))),
            publicSubnetIds: collapse(flatMap(findGroups(SubnetType.Public, grouped), group => group.subnets.map(s => s.subnetId))),
            publicSubnetNames: collapse(flatMap(findGroups(SubnetType.Public, grouped), group => group.name ? [group.name] : [])),
            publicSubnetRouteTableIds: collapse(flatMap(findGroups(SubnetType.Public, grouped), group => group.subnets.map(s => s.routeTableId))),
            vpnGatewayId,
            subnetGroups: assymetricSubnetGroups,
        };
    }
}
exports.VpcNetworkContextProviderPlugin = VpcNetworkContextProviderPlugin;
class RouteTables {
    constructor(tables) {
        this.tables = tables;
        this.mainRouteTable = this.tables.find(table => !!table.Associations && table.Associations.some(assoc => !!assoc.Main));
    }
    routeTableIdForSubnetId(subnetId) {
        const table = this.tableForSubnet(subnetId);
        return (table && table.RouteTableId) || (this.mainRouteTable && this.mainRouteTable.RouteTableId);
    }
    /**
     * Whether the given subnet has a route to an IGW
     */
    hasRouteToIgw(subnetId) {
        const table = this.tableForSubnet(subnetId);
        return !!table && !!table.Routes && table.Routes.some(route => !!route.GatewayId && route.GatewayId.startsWith('igw-'));
    }
    tableForSubnet(subnetId) {
        return this.tables.find(table => !!table.Associations && table.Associations.some(assoc => assoc.SubnetId === subnetId));
    }
}
/**
 * Return the value of a tag from a set of tags
 */
function getTag(name, tags) {
    for (const tag of tags || []) {
        if (tag.Key === name) {
            return tag.Value;
        }
    }
    return undefined;
}
/**
 * Group subnets of the same type together, and order by AZ
 */
function groupSubnets(subnets) {
    const grouping = {};
    for (const subnet of subnets) {
        const key = [subnet.type, subnet.name].toString();
        if (!(key in grouping)) {
            grouping[key] = [];
        }
        grouping[key].push(subnet);
    }
    const groups = Object.values(grouping).map(sns => {
        sns.sort((a, b) => a.az.localeCompare(b.az));
        return {
            type: sns[0].type,
            name: sns[0].name,
            subnets: sns,
        };
    });
    const azs = groups[0].subnets.map(s => s.az);
    for (const group of groups) {
        const groupAZs = group.subnets.map(s => s.az);
        if (!arraysEqual(groupAZs, azs)) {
            throw new Error(`Not all subnets in VPC have the same AZs: ${groupAZs} vs ${azs}`);
        }
    }
    return { azs, groups };
}
function groupAsymmetricSubnets(subnets) {
    const grouping = {};
    for (const subnet of subnets) {
        const key = [subnet.type, subnet.name].toString();
        if (!(key in grouping)) {
            grouping[key] = [];
        }
        grouping[key].push(subnet);
    }
    return Object.values(grouping).map(subnetArray => {
        subnetArray.sort((subnet1, subnet2) => subnet1.az.localeCompare(subnet2.az));
        return {
            name: subnetArray[0].name,
            type: subnetTypeToVpcSubnetType(subnetArray[0].type),
            subnets: subnetArray.map(subnet => ({
                subnetId: subnet.subnetId,
                cidr: subnet.cidr,
                availabilityZone: subnet.az,
                routeTableId: subnet.routeTableId,
            })),
        };
    });
}
function subnetTypeToVpcSubnetType(type) {
    switch (type) {
        case SubnetType.Isolated: return cxapi.VpcSubnetGroupType.ISOLATED;
        case SubnetType.Private: return cxapi.VpcSubnetGroupType.PRIVATE;
        case SubnetType.Public: return cxapi.VpcSubnetGroupType.PUBLIC;
    }
}
var SubnetType;
(function (SubnetType) {
    SubnetType["Public"] = "Public";
    SubnetType["Private"] = "Private";
    SubnetType["Isolated"] = "Isolated";
})(SubnetType || (SubnetType = {}));
function isValidSubnetType(val) {
    return val === SubnetType.Public
        || val === SubnetType.Private
        || val === SubnetType.Isolated;
}
function arraysEqual(as, bs) {
    if (as.length !== bs.length) {
        return false;
    }
    for (let i = 0; i < as.length; i++) {
        if (as[i] !== bs[i]) {
            return false;
        }
    }
    return true;
}
function findGroups(type, groups) {
    return groups.groups.filter(g => g.type === type);
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
function collapse(xs) {
    if (xs.length > 0) {
        return xs;
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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