"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.printAvailableTemplates = exports.availableInitLanguages = exports.availableInitTemplates = exports.InitTemplate = exports.cliInit = void 0;
const childProcess = require("child_process");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const colors = require("colors/safe");
const fs = require("fs-extra");
const semver = require("semver");
const logging_1 = require("./logging");
const directories_1 = require("./util/directories");
const version_1 = require("./version");
/* eslint-disable @typescript-eslint/no-var-requires */ // Packages don't have @types module
// eslint-disable-next-line @typescript-eslint/no-require-imports
const camelCase = require('camelcase');
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
/**
 * Initialize a CDK package in the current directory
 */
async function cliInit(type, language, canUseNetwork = true, generateOnly = false, workDir = process.cwd()) {
    if (!type && !language) {
        await printAvailableTemplates();
        return;
    }
    type = type || 'default'; // "default" is the default type (and maps to "app")
    const template = (await availableInitTemplates()).find(t => t.hasName(type));
    if (!template) {
        await printAvailableTemplates(language);
        throw new Error(`Unknown init template: ${type}`);
    }
    if (!language && template.languages.length === 1) {
        language = template.languages[0];
        logging_1.warning(`No --language was provided, but '${type}' supports only '${language}', so defaulting to --language=${language}`);
    }
    if (!language) {
        logging_1.print(`Available languages for ${colors.green(type)}: ${template.languages.map(l => colors.blue(l)).join(', ')}`);
        throw new Error('No language was selected');
    }
    await initializeProject(template, language, canUseNetwork, generateOnly, workDir);
}
exports.cliInit = cliInit;
/**
 * Returns the name of the Python executable for this OS
 */
function pythonExecutable() {
    let python = 'python3';
    if (process.platform === 'win32') {
        python = 'python';
    }
    return python;
}
const INFO_DOT_JSON = 'info.json';
class InitTemplate {
    constructor(basePath, name, languages, info) {
        this.basePath = basePath;
        this.name = name;
        this.languages = languages;
        this.aliases = new Set();
        this.description = info.description;
        for (const alias of info.aliases || []) {
            this.aliases.add(alias);
        }
    }
    static async fromName(templatesDir, name) {
        const basePath = path.join(templatesDir, name);
        const languages = (await listDirectory(basePath)).filter(f => f !== INFO_DOT_JSON);
        const info = await fs.readJson(path.join(basePath, INFO_DOT_JSON));
        return new InitTemplate(basePath, name, languages, info);
    }
    /**
     * @param name the name that is being checked
     * @returns ``true`` if ``name`` is the name of this template or an alias of it.
     */
    hasName(name) {
        return name === this.name || this.aliases.has(name);
    }
    /**
     * Creates a new instance of this ``InitTemplate`` for a given language to a specified folder.
     *
     * @param language    the language to instantiate this template with
     * @param targetDirectory the directory where the template is to be instantiated into
     */
    async install(language, targetDirectory) {
        if (this.languages.indexOf(language) === -1) {
            logging_1.error(`The ${colors.blue(language)} language is not supported for ${colors.green(this.name)} `
                + `(it supports: ${this.languages.map(l => colors.blue(l)).join(', ')})`);
            throw new Error(`Unsupported language: ${language}`);
        }
        const sourceDirectory = path.join(this.basePath, language);
        const hookTempDirectory = path.join(targetDirectory, 'tmp');
        await fs.mkdir(hookTempDirectory);
        await this.installFiles(sourceDirectory, targetDirectory, {
            name: decamelize(path.basename(path.resolve(targetDirectory))),
        });
        await this.applyFutureFlags(targetDirectory);
        await this.invokeHooks(hookTempDirectory, targetDirectory);
        await fs.remove(hookTempDirectory);
    }
    async installFiles(sourceDirectory, targetDirectory, project) {
        for (const file of await fs.readdir(sourceDirectory)) {
            const fromFile = path.join(sourceDirectory, file);
            const toFile = path.join(targetDirectory, this.expand(file, project));
            if ((await fs.stat(fromFile)).isDirectory()) {
                await fs.mkdir(toFile);
                await this.installFiles(fromFile, toFile, project);
                continue;
            }
            else if (file.match(/^.*\.template\.[^.]+$/)) {
                await this.installProcessed(fromFile, toFile.replace(/\.template(\.[^.]+)$/, '$1'), project);
                continue;
            }
            else if (file.match(/^.*\.hook\.(d.)?[^.]+$/)) {
                await this.installProcessed(fromFile, path.join(targetDirectory, 'tmp', file), project);
                continue;
            }
            else {
                await fs.copy(fromFile, toFile);
            }
        }
    }
    /**
     * @summary   Invoke any javascript hooks that exist in the template.
     * @description Sometimes templates need more complex logic than just replacing tokens. A 'hook' is
     *        any file that ends in .hook.js. It should export a single function called "invoke"
     *        that accepts a single string parameter. When the template is installed, each hook
     *        will be invoked, passing the target directory as the only argument. Hooks are invoked
     *        in lexical order.
     */
    async invokeHooks(sourceDirectory, targetDirectory) {
        const files = await fs.readdir(sourceDirectory);
        files.sort(); // Sorting allows template authors to control the order in which hooks are invoked.
        for (const file of files) {
            if (file.match(/^.*\.hook\.js$/)) {
                // eslint-disable-next-line @typescript-eslint/no-require-imports
                const invoke = require(path.join(sourceDirectory, file)).invoke;
                await invoke(targetDirectory);
            }
        }
    }
    async installProcessed(templatePath, toFile, project) {
        const template = await fs.readFile(templatePath, { encoding: 'utf-8' });
        await fs.writeFile(toFile, this.expand(template, project));
    }
    expand(template, project) {
        const MATCH_VER_BUILD = /\+[a-f0-9]+$/; // Matches "+BUILD" in "x.y.z-beta+BUILD"
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const cdkVersion = require('../package.json').version.replace(MATCH_VER_BUILD, '');
        return template.replace(/%name%/g, project.name)
            .replace(/%name\.camelCased%/g, camelCase(project.name))
            .replace(/%name\.PascalCased%/g, camelCase(project.name, { pascalCase: true }))
            .replace(/%cdk-version%/g, cdkVersion)
            .replace(/%cdk-home%/g, directories_1.cdkHomeDir())
            .replace(/%name\.PythonModule%/g, project.name.replace(/-/g, '_'))
            .replace(/%python-executable%/g, pythonExecutable())
            .replace(/%name\.StackName%/g, project.name.replace(/[^A-Za-z0-9-]/g, '-'));
    }
    /**
     * Adds context variables to `cdk.json` in the generated project directory to
     * enable future behavior for new projects.
     */
    async applyFutureFlags(projectDir) {
        const cdkJson = path.join(projectDir, 'cdk.json');
        if (!await fs.pathExists(cdkJson)) {
            return;
        }
        const futureFlags = {};
        Object.entries(cxapi.FUTURE_FLAGS)
            .filter(([k, _]) => !cxapi.FUTURE_FLAGS_EXPIRED.includes(k))
            .forEach(([k, v]) => futureFlags[k] = v);
        const config = await fs.readJson(cdkJson);
        config.context = {
            ...config.context,
            ...futureFlags,
        };
        await fs.writeJson(cdkJson, config, { spaces: 2 });
    }
}
exports.InitTemplate = InitTemplate;
function versionedTemplatesDir() {
    return new Promise(async (resolve) => {
        let currentVersion = version_1.versionNumber();
        // If the CLI is invoked from source (i.e., developement), rather than from a packaged distribution,
        // the version number will be '0.0.0'. We will (currently) default to the v1 templates in this case.
        if (currentVersion === '0.0.0') {
            currentVersion = '1.0.0';
        }
        const majorVersion = semver.major(currentVersion);
        resolve(path.join(__dirname, 'init-templates', `v${majorVersion}`));
    });
}
async function availableInitTemplates() {
    return new Promise(async (resolve) => {
        const templatesDir = await versionedTemplatesDir();
        const templateNames = await listDirectory(templatesDir);
        const templates = new Array();
        for (const templateName of templateNames) {
            templates.push(await InitTemplate.fromName(templatesDir, templateName));
        }
        resolve(templates);
    });
}
exports.availableInitTemplates = availableInitTemplates;
async function availableInitLanguages() {
    return new Promise(async (resolve) => {
        const templates = await availableInitTemplates();
        const result = new Set();
        for (const template of templates) {
            for (const language of template.languages) {
                result.add(language);
            }
        }
        resolve([...result]);
    });
}
exports.availableInitLanguages = availableInitLanguages;
/**
 * @param dirPath is the directory to be listed.
 * @returns the list of file or directory names contained in ``dirPath``, excluding any dot-file, and sorted.
 */
async function listDirectory(dirPath) {
    return (await fs.readdir(dirPath))
        .filter(p => !p.startsWith('.'))
        .sort();
}
async function printAvailableTemplates(language) {
    logging_1.print('Available templates:');
    for (const template of await availableInitTemplates()) {
        if (language && template.languages.indexOf(language) === -1) {
            continue;
        }
        logging_1.print(`* ${colors.green(template.name)}: ${template.description}`);
        const languageArg = language ? colors.bold(language)
            : template.languages.length > 1 ? `[${template.languages.map(t => colors.bold(t)).join('|')}]`
                : colors.bold(template.languages[0]);
        logging_1.print(`   └─ ${colors.blue(`cdk init ${colors.bold(template.name)} --language=${languageArg}`)}`);
    }
}
exports.printAvailableTemplates = printAvailableTemplates;
async function initializeProject(template, language, canUseNetwork, generateOnly, workDir) {
    await assertIsEmptyDirectory(workDir);
    logging_1.print(`Applying project template ${colors.green(template.name)} for ${colors.blue(language)}`);
    await template.install(language, workDir);
    if (await fs.pathExists('README.md')) {
        logging_1.print(colors.green(await fs.readFile('README.md', { encoding: 'utf-8' })));
    }
    if (!generateOnly) {
        await initializeGitRepository(workDir);
        await postInstall(language, canUseNetwork, workDir);
    }
    logging_1.print('✅ All done!');
}
async function assertIsEmptyDirectory(workDir) {
    const files = await fs.readdir(workDir);
    if (files.filter(f => !f.startsWith('.')).length !== 0) {
        throw new Error('`cdk init` cannot be run in a non-empty directory!');
    }
}
async function initializeGitRepository(workDir) {
    if (await isInGitRepository(workDir)) {
        return;
    }
    logging_1.print('Initializing a new git repository...');
    try {
        await execute('git', ['init'], { cwd: workDir });
        await execute('git', ['add', '.'], { cwd: workDir });
        await execute('git', ['commit', '--message="Initial commit"', '--no-gpg-sign'], { cwd: workDir });
    }
    catch (e) {
        logging_1.warning('Unable to initialize git repository for your project.');
    }
}
async function postInstall(language, canUseNetwork, workDir) {
    switch (language) {
        case 'javascript':
            return postInstallJavascript(canUseNetwork, workDir);
        case 'typescript':
            return postInstallTypescript(canUseNetwork, workDir);
        case 'java':
            return postInstallJava(canUseNetwork, workDir);
        case 'python':
            return postInstallPython(workDir);
    }
}
async function postInstallJavascript(canUseNetwork, cwd) {
    return postInstallTypescript(canUseNetwork, cwd);
}
async function postInstallTypescript(canUseNetwork, cwd) {
    const command = 'npm';
    if (!canUseNetwork) {
        logging_1.warning(`Please run '${command} install'!`);
        return;
    }
    logging_1.print(`Executing ${colors.green(`${command} install`)}...`);
    try {
        await execute(command, ['install'], { cwd });
    }
    catch (e) {
        logging_1.warning(`${command} install failed: ` + e.message);
    }
}
async function postInstallJava(canUseNetwork, cwd) {
    const mvnPackageWarning = 'Please run \'mvn package\'!';
    if (!canUseNetwork) {
        logging_1.warning(mvnPackageWarning);
        return;
    }
    logging_1.print('Executing \'mvn package\'');
    try {
        await execute('mvn', ['package'], { cwd });
    }
    catch (e) {
        logging_1.warning('Unable to package compiled code as JAR');
        logging_1.warning(mvnPackageWarning);
    }
}
async function postInstallPython(cwd) {
    const python = pythonExecutable();
    logging_1.warning(`Please run '${python} -m venv .venv'!`);
    logging_1.print(`Executing ${colors.green('Creating virtualenv...')}`);
    try {
        await execute(python, ['-m venv', '.venv'], { cwd });
    }
    catch (e) {
        logging_1.warning('Unable to create virtualenv automatically');
        logging_1.warning(`Please run '${python} -m venv .venv'!`);
    }
}
/**
 * @param dir a directory to be checked
 * @returns true if ``dir`` is within a git repository.
 */
async function isInGitRepository(dir) {
    while (true) {
        if (await fs.pathExists(path.join(dir, '.git'))) {
            return true;
        }
        if (isRoot(dir)) {
            return false;
        }
        dir = path.dirname(dir);
    }
}
/**
 * @param dir a directory to be checked.
 * @returns true if ``dir`` is the root of a filesystem.
 */
function isRoot(dir) {
    return path.dirname(dir) === dir;
}
/**
 * Executes `command`. STDERR is emitted in real-time.
 *
 * If command exits with non-zero exit code, an exceprion is thrown and includes
 * the contents of STDOUT.
 *
 * @returns STDOUT (if successful).
 */
async function execute(cmd, args, { cwd }) {
    const child = childProcess.spawn(cmd, args, { cwd, shell: true, stdio: ['ignore', 'pipe', 'inherit'] });
    let stdout = '';
    child.stdout.on('data', chunk => stdout += chunk.toString());
    return new Promise((ok, fail) => {
        child.once('error', err => fail(err));
        child.once('exit', status => {
            if (status === 0) {
                return ok(stdout);
            }
            else {
                process.stderr.write(stdout);
                return fail(new Error(`${cmd} exited with status ${status}`));
            }
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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