"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const settings_1 = require("../lib/settings");
test('can delete values from Context object', () => {
    // GIVEN
    const settings1 = new settings_1.Settings({ foo: 'bar' });
    const settings2 = new settings_1.Settings({ boo: 'baz' });
    const context = new settings_1.Context(settings1, settings2);
    // WHEN
    context.unset('foo');
    // THEN
    expect(context.all).toEqual({ boo: 'baz' });
    expect(settings1.all).toEqual({});
    expect(settings2.all).toEqual({ boo: 'baz' });
});
test('can set values in Context object', () => {
    // GIVEN
    const settings1 = new settings_1.Settings();
    const settings2 = new settings_1.Settings();
    const context = new settings_1.Context(settings1, settings2);
    // WHEN
    context.set('foo', 'bar');
    // THEN
    expect(context.all).toEqual({ foo: 'bar' });
    expect(settings1.all).toEqual({ foo: 'bar' });
    expect(settings2.all).toEqual({});
});
test('can set values in Context object if first is immutable', () => {
    // GIVEN
    const settings1 = new settings_1.Settings();
    const settings2 = new settings_1.Settings();
    const context = new settings_1.Context(settings1.makeReadOnly(), settings2);
    // WHEN
    context.set('foo', 'bar');
    // THEN
    expect(context.all).toEqual({ foo: 'bar' });
    expect(settings1.all).toEqual({});
    expect(settings2.all).toEqual({ foo: 'bar' });
});
test('can clear all values in all objects', () => {
    // GIVEN
    const settings1 = new settings_1.Settings({ foo: 'bar' });
    const settings2 = new settings_1.Settings({ foo: 'snar', boo: 'gar' });
    const context = new settings_1.Context(settings1, settings2);
    // WHEN
    context.clear();
    // THEN
    expect(context.all).toEqual({});
    expect(settings1.all).toEqual({});
    expect(settings2.all).toEqual({});
});
test('can parse string context from command line arguments', () => {
    // GIVEN
    const settings1 = settings_1.Settings.fromCommandLineArguments({ context: ['foo=bar'], _: [settings_1.Command.DEPLOY] });
    const settings2 = settings_1.Settings.fromCommandLineArguments({ context: ['foo='], _: [settings_1.Command.DEPLOY] });
    // THEN
    expect(settings1.get(['context']).foo).toEqual('bar');
    expect(settings2.get(['context']).foo).toEqual('');
});
test('can parse string context from command line arguments with equals sign in value', () => {
    // GIVEN
    const settings1 = settings_1.Settings.fromCommandLineArguments({ context: ['foo==bar='], _: [settings_1.Command.DEPLOY] });
    const settings2 = settings_1.Settings.fromCommandLineArguments({ context: ['foo=bar='], _: [settings_1.Command.DEPLOY] });
    // THEN
    expect(settings1.get(['context']).foo).toEqual('=bar=');
    expect(settings2.get(['context']).foo).toEqual('bar=');
});
test('bundling stacks defaults to an empty list', () => {
    // GIVEN
    const settings = settings_1.Settings.fromCommandLineArguments({
        _: [settings_1.Command.LIST],
    });
    // THEN
    expect(settings.get(['bundlingStacks'])).toEqual([]);
});
test('bundling stacks defaults to * for deploy', () => {
    // GIVEN
    const settings = settings_1.Settings.fromCommandLineArguments({
        _: [settings_1.Command.DEPLOY],
    });
    // THEN
    expect(settings.get(['bundlingStacks'])).toEqual(['*']);
});
test('bundling stacks with deploy exclusively', () => {
    // GIVEN
    const settings = settings_1.Settings.fromCommandLineArguments({
        _: [settings_1.Command.DEPLOY],
        exclusively: true,
        STACKS: ['cool-stack'],
    });
    // THEN
    expect(settings.get(['bundlingStacks'])).toEqual(['cool-stack']);
});
test('should include outputs-file in settings', () => {
    // GIVEN
    const settings = settings_1.Settings.fromCommandLineArguments({
        _: [settings_1.Command.DEPLOY],
        outputsFile: 'my-outputs-file.json',
    });
    // THEN
    expect(settings.get(['outputsFile'])).toEqual('my-outputs-file.json');
});
//# sourceMappingURL=data:application/json;base64,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