"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudformation_1 = require("../../lib/api/util/cloudformation");
const mock_sdk_1 = require("./mock-sdk");
const PARAM = 'TheParameter';
const DEFAULT = 'TheDefault';
const OVERRIDE = 'TheOverride';
const USE_OVERRIDE = { ParameterKey: PARAM, ParameterValue: OVERRIDE };
const USE_PREVIOUS = { ParameterKey: PARAM, UsePreviousValue: true };
let sdkProvider;
let cfnMocks;
let cfn;
beforeEach(async () => {
    sdkProvider = new mock_sdk_1.MockSdkProvider();
    cfnMocks = {
        describeStacks: jest.fn()
            // No stacks exist
            .mockImplementation(() => ({ Stacks: [] })),
    };
    sdkProvider.stubCloudFormation(cfnMocks);
    cfn = (await sdkProvider.forEnvironment()).cloudFormation();
});
test('A non-existent stack pretends to have an empty template', async () => {
    // WHEN
    const stack = await cloudformation_1.CloudFormationStack.lookup(cfn, 'Dummy');
    // THEN
    expect(await stack.template()).toEqual({});
});
test.each([
    [false, false],
    [false, true],
    [true, false],
    [true, true],
])('given override, always use the override (parameter has a default: %p, parameter previously supplied: %p)', (haveDefault, havePrevious) => {
    expect(makeParams(haveDefault, havePrevious, true)).toEqual({
        apiParameters: [USE_OVERRIDE],
        changed: true,
    });
});
test('no default, no prev, no override => error', () => {
    expect(() => makeParams(false, false, false)).toThrow(/missing a value: TheParameter/);
});
test('no default, yes prev, no override => use previous', () => {
    expect(makeParams(false, true, false)).toEqual({
        apiParameters: [USE_PREVIOUS],
        changed: false,
    });
});
test('default, no prev, no override => empty param set (and obviously changes to be applied)', () => {
    expect(makeParams(true, false, false)).toEqual({
        apiParameters: [],
        changed: true,
    });
});
test('default, prev, no override => use previous', () => {
    expect(makeParams(true, true, false)).toEqual({
        apiParameters: [USE_PREVIOUS],
        changed: false,
    });
});
test('if a parameter is retrieved from SSM, the parameters always count as changed', () => {
    const params = cloudformation_1.TemplateParameters.fromTemplate({
        Parameters: {
            Foo: {
                Type: 'AWS::SSM::Parameter::Name',
                Default: '/Some/Key',
            },
        },
    });
    const oldValues = { Foo: '/Some/Key' };
    // If we don't pass a new value
    expect(params.updateExisting({}, oldValues).hasChanges(oldValues)).toEqual(true);
    // If we do pass a new value but it's the same as the old one
    expect(params.updateExisting({ Foo: '/Some/Key' }, oldValues).hasChanges(oldValues)).toEqual(true);
});
test('empty string is a valid update value', () => {
    const params = cloudformation_1.TemplateParameters.fromTemplate({
        Parameters: {
            Foo: { Type: 'String', Default: 'Foo' },
        },
    });
    expect(params.updateExisting({ Foo: '' }, { Foo: 'ThisIsOld' }).apiParameters).toEqual([
        { ParameterKey: 'Foo', ParameterValue: '' },
    ]);
});
test('unknown parameter in overrides, pass it anyway', () => {
    // Not sure if we really want this. It seems like it would be nice
    // to not pass parameters that aren't expected, given that CFN will
    // just error out. But maybe we want to be warned of typos...
    const params = cloudformation_1.TemplateParameters.fromTemplate({
        Parameters: {
            Foo: { Type: 'String', Default: 'Foo' },
        },
    });
    expect(params.updateExisting({ Bar: 'Bar' }, {}).apiParameters).toEqual([
        { ParameterKey: 'Bar', ParameterValue: 'Bar' },
    ]);
});
test('if an unsupplied parameter reverts to its default, it can still be dirty', () => {
    // GIVEN
    const templateParams = cloudformation_1.TemplateParameters.fromTemplate({
        Parameters: {
            Foo: { Type: 'String', Default: 'Foo' },
        },
    });
    // WHEN
    const stackParams = templateParams.supplyAll({});
    // THEN
    expect(stackParams.hasChanges({ Foo: 'NonStandard' })).toEqual(true);
    expect(stackParams.hasChanges({ Foo: 'Foo' })).toEqual(false);
});
function makeParams(defaultValue, hasPrevValue, override) {
    const params = cloudformation_1.TemplateParameters.fromTemplate({
        Parameters: {
            [PARAM]: {
                Type: 'String',
                Default: defaultValue ? DEFAULT : undefined,
            },
        },
    });
    const prevParams = hasPrevValue ? { [PARAM]: 'Foo' } : {};
    const stackParams = params.updateExisting({ [PARAM]: override ? OVERRIDE : undefined }, prevParams);
    return { apiParameters: stackParams.apiParameters, changed: stackParams.hasChanges(prevParams) };
}
//# sourceMappingURL=data:application/json;base64,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