"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.errorWithCode = exports.mockResolvedEnvironment = exports.mockToolkitInfo = exports.mockBootstrapStack = exports.MockSdk = exports.MockSdkProvider = void 0;
const AWS = require("aws-sdk");
const aws_auth_1 = require("../../lib/api/aws-auth");
const toolkit_info_1 = require("../../lib/api/toolkit-info");
const cloudformation_1 = require("../../lib/api/util/cloudformation");
const FAKE_CREDENTIALS = new AWS.Credentials({ accessKeyId: 'ACCESS', secretAccessKey: 'SECRET', sessionToken: 'TOKEN ' });
const FAKE_CREDENTIAL_CHAIN = new AWS.CredentialProviderChain([
    () => FAKE_CREDENTIALS,
]);
/**
 * An SDK that allows replacing (some of) the clients
 *
 * Its the responsibility of the consumer to replace all calls that
 * actually will be called.
 */
class MockSdkProvider extends aws_auth_1.SdkProvider {
    constructor(options = {}) {
        var _a;
        super(FAKE_CREDENTIAL_CHAIN, 'bermuda-triangle-1337', { customUserAgent: 'aws-cdk/jest' });
        // SDK contains a real SDK, since some test use 'AWS-mock' to replace the underlying
        // AWS calls which a real SDK would do, and some tests use the 'stub' functionality below.
        if ((_a = options.realSdk) !== null && _a !== void 0 ? _a : true) {
            this.sdk = new aws_auth_1.SDK(FAKE_CREDENTIALS, this.defaultRegion, { customUserAgent: 'aws-cdk/jest' });
        }
        else {
            this.sdk = new MockSdk();
        }
    }
    async baseCredentialsPartition(_environment, _mode) {
        return undefined;
    }
    defaultAccount() {
        return Promise.resolve({ accountId: '123456789012', partition: 'aws' });
    }
    forEnvironment() {
        return Promise.resolve(this.sdk);
    }
    /**
     * Replace the CloudFormation client with the given object
     */
    stubCloudFormation(stubs) {
        this.sdk.cloudFormation = jest.fn().mockReturnValue(partialAwsService(stubs));
    }
    /**
     * Replace the ECR client with the given object
     */
    stubEcr(stubs) {
        this.sdk.ecr = jest.fn().mockReturnValue(partialAwsService(stubs));
    }
    /**
     * Replace the S3 client with the given object
     */
    stubS3(stubs) {
        this.sdk.s3 = jest.fn().mockReturnValue(partialAwsService(stubs));
    }
    /**
     * Replace the STS client with the given object
     */
    stubSTS(stubs) {
        this.sdk.sts = jest.fn().mockReturnValue(partialAwsService(stubs));
    }
    /**
     * Replace the ELBv2 client with the given object
     */
    stubELBv2(stubs) {
        this.sdk.elbv2 = jest.fn().mockReturnValue(partialAwsService(stubs));
    }
    /**
     * Replace the SSM client with the given object
     */
    stubSSM(stubs) {
        this.sdk.ssm = jest.fn().mockReturnValue(partialAwsService(stubs));
    }
}
exports.MockSdkProvider = MockSdkProvider;
class MockSdk {
    constructor() {
        this.currentRegion = 'bermuda-triangle-1337';
        this.cloudFormation = jest.fn();
        this.ec2 = jest.fn();
        this.ssm = jest.fn();
        this.s3 = jest.fn();
        this.route53 = jest.fn();
        this.ecr = jest.fn();
        this.elbv2 = jest.fn();
        this.secretsManager = jest.fn();
    }
    currentAccount() {
        return Promise.resolve({ accountId: '123456789012', partition: 'aws' });
    }
    /**
     * Replace the CloudFormation client with the given object
     */
    stubCloudFormation(stubs) {
        this.cloudFormation.mockReturnValue(partialAwsService(stubs));
    }
    /**
     * Replace the ECR client with the given object
     */
    stubEcr(stubs) {
        this.ecr.mockReturnValue(partialAwsService(stubs));
    }
    /**
     * Replace the SSM client with the given object
     */
    stubSsm(stubs) {
        this.ssm.mockReturnValue(partialAwsService(stubs));
    }
}
exports.MockSdk = MockSdk;
/**
 * Wrap synchronous fake handlers so that they sort-of function like a real AWS client
 *
 * For example, turns an object like this:
 *
 * ```ts
 * {
 *   someCall(opts: AWS.Service.SomeCallInput): AWS.Service.SomeCallOutput {
 *     return {...whatever...};
 *   }
 * }
 * ```
 *
 * Into an object that in the type system pretends to be an 'AWS.Service'
 * class (even though it really isn't) and can be called like this:
 *
 * ```ts
 * const service = await sdk.someService(...);
 * const response = await service.someCall(...).promise();
 * ```
 *
 * We only implement the narrow subset of the AWS SDK API that the CDK actually
 * uses, and we cheat on the types to make TypeScript happy on the rest of the API.
 *
 * Most important feature of this class is that it will derive the input and output
 * types of the handlers on the input object from the ACTUAL AWS Service class,
 * so that you don't have to declare them.
 */
function partialAwsService(fns) {
    // Super unsafe in here because I don't know how to make TypeScript happy,
    // but at least the outer types make sure everything that happens in here works out.
    const ret = {};
    for (const [key, handler] of Object.entries(fns)) {
        ret[key] = (args) => new FakeAWSResponse(handler(args));
    }
    return ret;
}
/**
 * Fake AWS response.
 *
 * We only ever 'await response.promise()' so that's the only thing we implement here.
 */
class FakeAWSResponse {
    constructor(x) {
        this.x = x;
    }
    promise() {
        return Promise.resolve(this.x);
    }
}
function mockBootstrapStack(sdk, stack) {
    var _a;
    return cloudformation_1.CloudFormationStack.fromStaticInformation((sdk !== null && sdk !== void 0 ? sdk : new MockSdk()).cloudFormation(), 'CDKToolkit', {
        CreationTime: new Date(),
        StackName: 'CDKToolkit',
        StackStatus: 'CREATE_COMPLETE',
        ...stack,
        Outputs: [
            { OutputKey: 'BucketName', OutputValue: 'BUCKET_NAME' },
            { OutputKey: 'BucketDomainName', OutputValue: 'BUCKET_ENDPOINT' },
            { OutputKey: 'BootstrapVersion', OutputValue: '1' },
            ...(_a = stack === null || stack === void 0 ? void 0 : stack.Outputs) !== null && _a !== void 0 ? _a : [],
        ],
    });
}
exports.mockBootstrapStack = mockBootstrapStack;
function mockToolkitInfo(stack) {
    const sdk = new MockSdk();
    return toolkit_info_1.ToolkitInfo.fromStack(mockBootstrapStack(sdk, stack), sdk);
}
exports.mockToolkitInfo = mockToolkitInfo;
function mockResolvedEnvironment() {
    return {
        account: '123456789',
        region: 'bermuda-triangle-1337',
        name: 'aws://123456789/bermuda-triangle-1337',
    };
}
exports.mockResolvedEnvironment = mockResolvedEnvironment;
function errorWithCode(code, message) {
    const ret = new Error(message);
    ret.code = code;
    return ret;
}
exports.errorWithCode = errorWithCode;
//# sourceMappingURL=data:application/json;base64,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