"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const timers_1 = require("timers");
const util_1 = require("util");
const fs = require("fs-extra");
const sinon = require("sinon");
const logging = require("../lib/logging");
const version_1 = require("../lib/version");
jest.setTimeout(10000);
const setTimeout = util_1.promisify(timers_1.setTimeout);
function tmpfile() {
    return `/tmp/version-${Math.floor(Math.random() * 10000)}`;
}
afterEach(done => {
    sinon.restore();
    done();
});
test('initialization fails on unwritable directory', () => {
    const cacheFile = tmpfile();
    sinon.stub(fs, 'mkdirsSync').withArgs(path.dirname(cacheFile)).throws('Cannot make directory');
    expect(() => new version_1.VersionCheckTTL(cacheFile)).toThrow(/not writable/);
});
test('cache file responds correctly when file is not present', async () => {
    const cache = new version_1.VersionCheckTTL(tmpfile(), 1);
    expect(await cache.hasExpired()).toBeTruthy();
});
test('cache file honours the specified TTL', async () => {
    const cache = new version_1.VersionCheckTTL(tmpfile(), 1);
    await cache.update();
    expect(await cache.hasExpired()).toBeFalsy();
    await setTimeout(1001); // Just above 1 sec in ms
    expect(await cache.hasExpired()).toBeTruthy();
});
test('Skip version check if cache has not expired', async () => {
    const cache = new version_1.VersionCheckTTL(tmpfile(), 100);
    await cache.update();
    expect(await version_1.latestVersionIfHigher('0.0.0', cache)).toBeNull();
});
test('Return later version when exists & skip recent re-check', async () => {
    const cache = new version_1.VersionCheckTTL(tmpfile(), 100);
    const result = await version_1.latestVersionIfHigher('0.0.0', cache);
    expect(result).not.toBeNull();
    expect(result.length).toBeGreaterThan(0);
    const result2 = await version_1.latestVersionIfHigher('0.0.0', cache);
    expect(result2).toBeNull();
});
test('Return null if version is higher than npm', async () => {
    const cache = new version_1.VersionCheckTTL(tmpfile(), 100);
    const result = await version_1.latestVersionIfHigher('100.100.100', cache);
    expect(result).toBeNull();
});
test('Version specified is stored in the TTL file', async () => {
    const cacheFile = tmpfile();
    const cache = new version_1.VersionCheckTTL(cacheFile, 1);
    await cache.update('1.1.1');
    const storedVersion = fs.readFileSync(cacheFile, 'utf8');
    expect(storedVersion).toBe('1.1.1');
});
test('No Version specified for storage in the TTL file', async () => {
    const cacheFile = tmpfile();
    const cache = new version_1.VersionCheckTTL(cacheFile, 1);
    await cache.update();
    const storedVersion = fs.readFileSync(cacheFile, 'utf8');
    expect(storedVersion).toBe('');
});
test('Skip version check if environment variable is set', async () => {
    process.stdout.isTTY = true;
    process.env.CDK_DISABLE_VERSION_CHECK = '1';
    const printStub = sinon.stub(logging, 'print');
    await version_1.displayVersionMessage();
    expect(printStub.called).toEqual(false);
});
//# sourceMappingURL=data:application/json;base64,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