"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultAwsClient = void 0;
const os = require("os");
/**
 * AWS client using the AWS SDK for JS with no special configuration
 */
class DefaultAwsClient {
    constructor(profile) {
        // Force AWS SDK to look in ~/.aws/credentials and potentially use the configured profile.
        process.env.AWS_SDK_LOAD_CONFIG = '1';
        process.env.AWS_STS_REGIONAL_ENDPOINTS = 'regional';
        process.env.AWS_NODEJS_CONNECTION_REUSE_ENABLED = '1';
        if (profile) {
            process.env.AWS_PROFILE = profile;
        }
        // We need to set the environment before we load this library for the first time.
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        this.AWS = require('aws-sdk');
    }
    async s3Client(options) {
        return new this.AWS.S3(await this.awsOptions(options));
    }
    async ecrClient(options) {
        return new this.AWS.ECR(await this.awsOptions(options));
    }
    async secretsManagerClient(options) {
        return new this.AWS.SecretsManager(await this.awsOptions(options));
    }
    async discoverPartition() {
        return (await this.discoverCurrentAccount()).partition;
    }
    async discoverDefaultRegion() {
        return this.AWS.config.region || 'us-east-1';
    }
    async discoverCurrentAccount() {
        if (this.account === undefined) {
            const sts = new this.AWS.STS();
            const response = await sts.getCallerIdentity().promise();
            if (!response.Account || !response.Arn) {
                throw new Error(`Unrecognized reponse from STS: '${JSON.stringify(response)}'`);
            }
            this.account = {
                accountId: response.Account,
                partition: response.Arn.split(':')[1],
            };
        }
        return this.account;
    }
    async awsOptions(options) {
        let credentials;
        if (options.assumeRoleArn) {
            credentials = await this.assumeRole(options.region, options.assumeRoleArn, options.assumeRoleExternalId);
        }
        return {
            region: options.region,
            customUserAgent: 'cdk-assets',
            credentials,
        };
    }
    /**
     * Explicit manual AssumeRole call
     *
     * Necessary since I can't seem to get the built-in support for ChainableTemporaryCredentials to work.
     *
     * It needs an explicit configuration of `masterCredentials`, we need to put
     * a `DefaultCredentialProverChain()` in there but that is not possible.
     */
    async assumeRole(region, roleArn, externalId) {
        return new this.AWS.ChainableTemporaryCredentials({
            params: {
                RoleArn: roleArn,
                ExternalId: externalId,
                RoleSessionName: `cdk-assets-${safeUsername()}`,
            },
            stsConfig: {
                region,
                customUserAgent: 'cdk-assets',
            },
        });
    }
}
exports.DefaultAwsClient = DefaultAwsClient;
/**
 * Return the username with characters invalid for a RoleSessionName removed
 *
 * @see https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html#API_AssumeRole_RequestParameters
 */
function safeUsername() {
    return os.userInfo().username.replace(/[^\w+=,.@-]/g, '@');
}
//# sourceMappingURL=data:application/json;base64,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