# functions to load the data generated by the plate reader
# and to parse the annotation file
import re

import numpy as np
import pandas as pd

import omniplate.admin as admin
import omniplate.clogger as clogger
import omniplate.omerrors as errors
import omniplate.omgenutils as gu
from omniplate.parseplate import parseplate


@clogger.log
def load(
    self,
    dnames,
    anames=False,
    platereadertype="Tecan",
    dsheetnumbers=False,
    asheetnumbers=False,
    info=True,
    combine=False,
    min_dt=0.05,
):
    """
    Load raw data files.

    Two files are required: one generated by the plate reader and the other
    the corresponding annotation files - both assumed to be xlsx.

    Parameters
    ----------
    dnames: string or list of strings, optional
        The name of the file containing the data from the plate reader
        or a list of file names.
    anames: string or list of strings, optional
        The name of file containing the corresponding annotation or a list
        of file names.
    platereadertype: string
        The type of plate reader, currently either 'Tecan' or 'Sunrise'
        or 'tidy'.
    dsheetnumbers: integer or list of integers, optional
        The relevant sheets of the Excel files storing the data.
    asheetnumbers: integer or list of integers, optional
        The relevant sheets of the corresponding Excel files for the
        annotation.
    info: boolean
        If True (default), display summary information on the data once
        loaded.
    combine: boolean
        If True (default is False), combine all the data loaded into one
        single experiment.
    min_dt: float, optional
        The minimum time interval taken by the plate reader between
        measurements. Used only when combine=True.

    Examples
    -------
    >>> p.load('Data.xlsx', 'DataContents.xlsx')
    >>> p.load('Data.xlsx', 'DataContents.xlsx', info= False)
    """
    dnames = gu.makelist(dnames)
    if not anames:
        anames = [dname.split(".")[0] + "Contents.xlsx" for dname in dnames]
    else:
        anames = gu.makelist(anames)
    if not dsheetnumbers:
        dsheetnumbers = [0 for dname in dnames]
    if not asheetnumbers:
        asheetnumbers = [0 for dname in dnames]

    alldata = {}
    for i, dname in enumerate(dnames):
        # get dataframe for raw data
        (
            rdf,
            allconditionssingle,
            allstrainssingle,
            alldatasingle,
            experiment,
            datatypes,
        ) = loaddatafiles(
            platereadertype,
            self.wdirpath,
            dname,
            dsheetnumbers[i],
            anames[i],
            asheetnumbers[i],
        )
        self.allexperiments.append(experiment)
        self.allconditions[experiment] = allconditionssingle
        self.allstrains[experiment] = allstrainssingle
        self.datatypes[experiment] = datatypes
        self.allstrainsconditions[experiment] = list(
            (rdf.strain + " in " + rdf.condition).dropna().unique()
        )
        alldata.update(alldatasingle)
        # make r dataframe of raw data
        self.r = (
            pd.merge(self.r, rdf, how="outer") if hasattr(self, "r") else rdf
        )

        # update progress dictionary
        admin.initialiseprogress(self, experiment)
    # make sc dataframe for summary stats and corrections
    alldfs = []
    for exp in alldata:
        strs, cons = [], []
        for cs in alldata[exp]:
            strs.append(cs.split(" in ")[0])
            cons.append(cs.split(" in ")[1])
        corrdict = {
            "experiment": exp,
            "strain": strs,
            "condition": cons,
            "OD_corrected": False,
            "OD_corrected_for_media": False,
        }
        corrdict.update(
            {dtype + "_measured": True for dtype in self.datatypes[exp]}
        )
        alldfs.append(pd.DataFrame(corrdict))
    self.sc = pd.concat(alldfs)
    # replace NaN with False for datatypes_measured
    for column in self.sc.columns[self.sc.columns.str.contains("_measured")]:
        self.sc[column] = (
            self.sc[column].infer_objects(copy=False).fillna(False)
        )
    # make backup dataframe of original data
    self.origr = self.r.copy()
    # make dataframe for well contents
    self.wellsdf = admin.makewellsdf(self.r)
    # make s dataframe for summary data
    self.s = admin.make_s(self)
    # display info on experiment, conditions and strains
    if info:
        self.info
    print(
        '\nWarning: wells with no strains have been changed to "Null"'
        "\nto avoid confusion with pandas.\n"
    )
    # combine into one experiment
    if combine:
        self.combine_experiments(min_dt=min_dt)


def loaddatafiles(
    platereadertype, wdirpath, dname, dsheetnumber, aname, asheetnumber
):
    """Call functions to parse data and metadate from the input files."""
    experiment = dname.split(".")[0]
    # import and parse plate contents file
    allconditions, allstrains, alldata, wellcontents = analyseContentsofWells(
        wdirpath, experiment, aname, asheetnumber
    )
    # import and parse data file created by plate reader
    try:
        print("Loading", dname)
        rdf = parseplate(dname, platereadertype, wdirpath, dsheetnumber)
    except FileNotFoundError as exc:
        raise errors.FileNotFound(str(wdirpath / dname)) from exc
    # get datatypes
    cols = rdf.columns.to_list()
    datatypes = [col for col in cols if col != "time" and col != "well"]
    # add condition and strain to dataframe
    rdf["experiment"] = experiment
    rdf["condition"] = rdf["well"].map(
        {well: wellcontents[well][0] for well in wellcontents}
    )
    rdf["strain"] = rdf["well"].map(
        {well: wellcontents[well][1] for well in wellcontents}
    )
    # drop wells with condition=strain=None
    rdf = rdf.drop(
        rdf[rdf.condition.isnull() & rdf.strain.isnull()].index
    ).reset_index(drop=True)
    return rdf, allconditions, allstrains, alldata, experiment, datatypes


def analyseContentsofWells(wdirpath, experiment, aname, asheetnumber):
    """
    Load and parse ContentsofWells file.

    Return wellcontents, a dictionary with the contents of each well indexed
    by well, and alldata, a dictionary describing the contents of each well
    indexed by experiment.
    """
    try:
        alldata = {}
        # import contents of the wells
        anno = pd.read_excel(
            str(wdirpath / aname), index_col=0, sheet_name=asheetnumber
        )
        alldata[experiment] = []
        wellcontents = {}
        # run through and parse content of each well
        for x in np.arange(1, 13):
            for y in "ABCDEFGH":
                well = y + str(x)
                if (
                    isinstance(anno[x][y], str)
                    and anno[x][y] != "contaminated"
                ):
                    s, c = anno[x][y].split(" in ")
                    # standardise naming of wells with no strains
                    s = re.sub("(null|NULL)", "Null", s)
                    wellcontents[well] = [c.strip(), s.strip()]
                    alldata[experiment].append(
                        wellcontents[well][1] + " in " + wellcontents[well][0]
                    )
                else:
                    wellcontents[well] = [None, None]
        # create summary descriptions of the well contents
        alldata[experiment] = list(np.unique(alldata[experiment]))
        allconditions = list(
            np.unique(
                [
                    wellcontents[well][0]
                    for well in wellcontents
                    if wellcontents[well][0] is not None
                ]
            )
        )
        allstrains = list(
            np.unique(
                [
                    wellcontents[well][1]
                    for well in wellcontents
                    if wellcontents[well][0] is not None
                ]
            )
        )
        return allconditions, allstrains, alldata, wellcontents
    except FileNotFoundError as exc:
        raise errors.FileNotFound(str(wdirpath / aname)) from exc


def combine_experiments(self, min_dt, combined="combined"):
    """
    Combine raw data from all loaded experiments into one experiment.

    Only to be run immediately after first loading the data.

    Wells are relabelled with a prefix denoting their original
    experiment.
    p.expt_dict gives the mapping back to the original experiment
    names.

    Parameters
    ----------
    combined: str (optional)
        The name of the combined experiment.
    """
    print("Combining experiments...")
    # to make new r dataframe
    rdict = {col: [] for col in list(self.r.columns)}
    # find data types present in all experiments
    datatypes = [
        col
        for col in list(self.r.columns)
        if col not in ["experiment", "condition", "strain", "well", "time"]
    ]
    # define a new universal time
    unique_t = np.sort(self.r.time.unique())
    ut_values, ut_counts = np.unique(
        np.round(np.diff(unique_t), 2), return_counts=True
    )
    dt = ut_values[ut_values > min_dt][
        np.argmax(ut_counts[ut_values > min_dt])
    ]
    print(f" Using dt = {dt:.2f}.")
    nt = np.arange(0, unique_t.max(), dt)
    # find index for all wells
    wells_index = list(
        self.r.groupby(["experiment", "condition", "strain", "well"])
        .mean()
        .index
    )
    # interpolate data for each well to new universal time
    for wi in wells_index:
        print(f" Processing {wi[0]}; {wi[3]}")
        df = self.r[
            (
                (self.r.experiment == wi[0])
                & (self.r.condition == wi[1])
                & (self.r.strain == wi[2])
                & (self.r.well == wi[3])
            )
        ]
        # local universal time depending on length of experiments
        l_nt = nt[
            np.argmin((df.time.min() - nt) ** 2) : np.argmin(
                (df.time.max() - nt) ** 2
            )
            + 1
        ]
        x = df.time.values
        xs = np.sort(x)
        # interpolate data for this well to universal time
        new_y = {}
        for datatype in datatypes:
            try:
                y = df[datatype].values
                ys = y[np.argsort(x)]
                new_y[datatype] = np.interp(l_nt, xs, ys)
            except KeyError:
                new_y[datatype] = np.nan * np.ones(l_nt.shape)
        # add to dict
        for i in range(l_nt.size):
            rdict["experiment"].append(wi[0])
            rdict["condition"].append(wi[1])
            rdict["strain"].append(wi[2])
            rdict["well"].append(wi[3])
            rdict["time"].append(l_nt[i])
            for datatype in datatypes:
                rdict[datatype].append(new_y[datatype][i])
    # re-define r dataframe, now with universal time
    self.r = pd.DataFrame(data=rdict)
    # make dict mapping experiment number to original name
    self.expt_dict = {
        expt: str(i) for i, expt in enumerate(self.r.experiment.unique())
    }
    # rename wells
    self.r.well = (
        self.r.experiment.astype("string").replace(self.expt_dict)
        + "_"
        + self.r.well
    )
    # save names of original experiments
    self.r["original_experiment"] = self.r.experiment.copy()
    self.r["experiment_number"] = (
        self.r.experiment.astype("string").replace(self.expt_dict).copy()
    )
    # write over r.experiment and sort
    self.r.experiment = combined
    self.r = self.r.sort_values(by=["time", "well"]).reset_index(drop=True)
    # remake summary attributes
    self.allexperiments = [combined]
    for mattr in [
        "allconditions",
        "allstrains",
        "allstrainsconditions",
        "datatypes",
    ]:
        uniques = set(
            [
                lattr
                for e in getattr(self, mattr)
                for lattr in getattr(self, mattr)[e]
            ]
        )
        setattr(self, mattr, {combined: list(uniques)})
    # reset self.progress
    for key in self.progress:
        self.progress[key] = {
            combined: self.progress[key][next(iter(self.progress[key]))]
        }
    # remake dataframe for well content
    self.wellsdf = admin.makewellsdf(self.r)
    # remake s dataframe
    self.s = admin.make_s(self)
    # add original experiments to sc dataframe
    self.sc["original_experiment"] = self.sc.experiment.copy()
    self.sc["experiment_number"] = (
        self.sc.experiment.astype("string").replace(self.expt_dict).copy()
    )
    self.sc.experiment = combined
