"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetApiDefinition = exports.InlineApiDefinition = exports.S3ApiDefinition = exports.ApiDefinition = void 0;
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * Represents an OpenAPI definition asset.
 * @experimental
 */
class ApiDefinition {
    /**
     * Creates an API definition from a specification file in an S3 bucket
     * @experimental
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3ApiDefinition(bucket, key, objectVersion);
    }
    /**
     * Create an API definition from an inline object. The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @example
     *   ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition) {
        return new InlineApiDefinition(definition);
    }
    /**
     * Loads the API specification from a local disk asset.
     * @experimental
     */
    static fromAsset(file, options) {
        return new AssetApiDefinition(file, options);
    }
}
exports.ApiDefinition = ApiDefinition;
/**
 * OpenAPI specification from an S3 archive.
 * @experimental
 */
class S3ApiDefinition extends ApiDefinition {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucket: this.bucketName,
                key: this.key,
                version: this.objectVersion,
            },
        };
    }
}
exports.S3ApiDefinition = S3ApiDefinition;
/**
 * OpenAPI specification from an inline JSON object.
 * @experimental
 */
class InlineApiDefinition extends ApiDefinition {
    constructor(definition) {
        super();
        this.definition = definition;
        if (typeof (definition) !== 'object') {
            throw new Error('definition should be of type object');
        }
        if (Object.keys(definition).length === 0) {
            throw new Error('JSON definition cannot be empty');
        }
    }
    bind(_scope) {
        return {
            inlineDefinition: this.definition,
        };
    }
}
exports.InlineApiDefinition = InlineApiDefinition;
/**
 * OpenAPI specification from a local file.
 * @experimental
 */
class AssetApiDefinition extends ApiDefinition {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope) {
        // If the same AssetAPIDefinition is used multiple times, retain only the first instantiation.
        if (this.asset === undefined) {
            this.asset = new s3_assets.Asset(scope, 'APIDefinition', {
                path: this.path,
                ...this.options,
            });
        }
        if (this.asset.isZipArchive) {
            throw new Error(`Asset cannot be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucket: this.asset.s3BucketName,
                key: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.AssetApiDefinition = AssetApiDefinition;
//# sourceMappingURL=data:application/json;base64,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