"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResponseType = exports.GatewayResponse = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
/**
 * Configure the response received by clients, produced from the API Gateway backend.
 *
 * @resource AWS::ApiGateway::GatewayResponse
 */
class GatewayResponse extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new apigateway_generated_1.CfnGatewayResponse(this, 'Resource', {
            restApiId: props.restApi.restApiId,
            responseType: props.type.responseType,
            responseParameters: this.buildResponseParameters(props.responseHeaders),
            responseTemplates: props.templates,
            statusCode: props.statusCode,
        });
        this.node.defaultChild = resource;
    }
    buildResponseParameters(responseHeaders) {
        if (!responseHeaders) {
            return undefined;
        }
        const responseParameters = {};
        for (const [header, value] of Object.entries(responseHeaders)) {
            responseParameters[`gatewayresponse.header.${header}`] = value;
        }
        return responseParameters;
    }
}
exports.GatewayResponse = GatewayResponse;
/**
 * Supported types of gateway responses.
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html
 */
class ResponseType {
    constructor(type) {
        this.responseType = type;
    }
    /** A custom response type to suppport future cases. */
    static of(type) {
        return new ResponseType(type.toUpperCase());
    }
}
exports.ResponseType = ResponseType;
/**
 * The gateway response for authorization failure.
 */
ResponseType.ACCESS_DENIED = new ResponseType('ACCESS_DENIED');
/**
 * The gateway response for an invalid API configuration.
 */
ResponseType.API_CONFIGURATION_ERROR = new ResponseType('API_CONFIGURATION_ERROR');
/**
 * The gateway response when a custom or Amazon Cognito authorizer failed to authenticate the caller.
 */
ResponseType.AUTHORIZER_FAILURE = new ResponseType('AUTHORIZER_FAILURE');
/**
 * The gateway response for failing to connect to a custom or Amazon Cognito authorizer.
 */
ResponseType.AUTHORIZER_CONFIGURATION_ERROR = new ResponseType('AUTHORIZER_CONFIGURATION_ERROR');
/**
 * The gateway response when the request parameter cannot be validated according to an enabled request validator.
 */
ResponseType.BAD_REQUEST_PARAMETERS = new ResponseType('BAD_REQUEST_PARAMETERS');
/**
 * The gateway response when the request body cannot be validated according to an enabled request validator.
 */
ResponseType.BAD_REQUEST_BODY = new ResponseType('BAD_REQUEST_BODY');
/**
 * The default gateway response for an unspecified response type with the status code of 4XX.
 */
ResponseType.DEFAULT_4XX = new ResponseType('DEFAULT_4XX');
/**
 * The default gateway response for an unspecified response type with a status code of 5XX.
 */
ResponseType.DEFAULT_5XX = new ResponseType('DEFAULT_5XX');
/**
 * The gateway response for an AWS authentication token expired error.
 */
ResponseType.EXPIRED_TOKEN = new ResponseType('EXPIRED_TOKEN');
/**
 * The gateway response for an invalid AWS signature error.
 */
ResponseType.INVALID_SIGNATURE = new ResponseType('INVALID_SIGNATURE');
/**
 * The gateway response for an integration failed error.
 */
ResponseType.INTEGRATION_FAILURE = new ResponseType('INTEGRATION_FAILURE');
/**
 * The gateway response for an integration timed out error.
 */
ResponseType.INTEGRATION_TIMEOUT = new ResponseType('INTEGRATION_TIMEOUT');
/**
 * The gateway response for an invalid API key submitted for a method requiring an API key.
 */
ResponseType.INVALID_API_KEY = new ResponseType('INVALID_API_KEY');
/**
 * The gateway response for a missing authentication token error,
 * including the cases when the client attempts to invoke an unsupported API method or resource.
 */
ResponseType.MISSING_AUTHENTICATION_TOKEN = new ResponseType('MISSING_AUTHENTICATION_TOKEN');
/**
 * The gateway response for the usage plan quota exceeded error.
 */
ResponseType.QUOTA_EXCEEDED = new ResponseType('QUOTA_EXCEEDED');
/**
 * The gateway response for the request too large error.
 */
ResponseType.REQUEST_TOO_LARGE = new ResponseType('REQUEST_TOO_LARGE');
/**
 * The gateway response when API Gateway cannot find the specified resource
 * after an API request passes authentication and authorization.
 */
ResponseType.RESOURCE_NOT_FOUND = new ResponseType('RESOURCE_NOT_FOUND');
/**
 * The gateway response when usage plan, method, stage, or account level throttling limits exceeded.
 */
ResponseType.THROTTLED = new ResponseType('THROTTLED');
/**
 * The gateway response when the custom or Amazon Cognito authorizer failed to authenticate the caller.
 */
ResponseType.UNAUTHORIZED = new ResponseType('UNAUTHORIZED');
/**
 * The gateway response when a payload is of an unsupported media type, if strict passthrough behavior is enabled.
 */
ResponseType.UNSUPPORTED_MEDIA_TYPE = new ResponseType('UNSUPPORTED_MEDIA_TYPE');
/**
 * The gateway response when a request is blocked by AWS WAF.
 */
ResponseType.WAF_FILTERED = new ResponseType('WAF_FILTERED');
//# sourceMappingURL=data:application/json;base64,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