"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = exports.MethodLoggingLevel = void 0;
const core_1 = require("@aws-cdk/core");
const access_log_1 = require("./access-log");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
class Stage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined &&
                !core_1.Token.isUnresolved(accessLogFormat.toString()) &&
                !/.*\$context.requestId.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include at least `AccessLogFormat.contextRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination === null || accessLogDestination === void 0 ? void 0 : accessLogDestination.bind(this).destinationArn,
                format: (accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString()) ? accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString() : access_log_1.AccessLogFormat.clf().toString(),
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * Returns the invoke URL for a certain path.
     * @param path The resource path
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted,
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod,
                resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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