"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
module.exports = {
    'default token authorizer'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.Authorization',
            AuthorizerUri: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':apigateway:',
                        {
                            Ref: 'AWS::Region',
                        },
                        ':lambda:path/2015-03-31/functions/',
                        {
                            'Fn::GetAtt': ['myfunction9B95E948', 'Arn'],
                        },
                        '/invocations',
                    ],
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'apigateway.amazonaws.com',
        }));
        test.ok(auth.authorizerArn.endsWith(`/authorizers/${auth.authorizerId}`), 'Malformed authorizer ARN');
        test.done();
    },
    'default request authorizer'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            resultsCacheTtl: core_1.Duration.seconds(0),
            identitySources: [],
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
            AuthorizerUri: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':apigateway:',
                        {
                            Ref: 'AWS::Region',
                        },
                        ':lambda:path/2015-03-31/functions/',
                        {
                            'Fn::GetAtt': ['myfunction9B95E948', 'Arn'],
                        },
                        '/invocations',
                    ],
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'apigateway.amazonaws.com',
        }));
        test.ok(auth.authorizerArn.endsWith(`/authorizers/${auth.authorizerId}`), 'Malformed authorizer ARN');
        test.done();
    },
    'invalid request authorizer config'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        test.throws(() => new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            resultsCacheTtl: core_1.Duration.seconds(1),
            identitySources: [],
        }), Error, 'At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        test.done();
    },
    'token authorizer with all parameters specified'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySource: 'method.request.header.whoami',
            validationRegex: 'a-hacker',
            authorizerName: 'myauthorizer',
            resultsCacheTtl: core_1.Duration.minutes(1),
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.whoami',
            IdentityValidationExpression: 'a-hacker',
            Name: 'myauthorizer',
            AuthorizerResultTtlInSeconds: 60,
            AuthorizerUri: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':apigateway:',
                        {
                            Ref: 'AWS::Region',
                        },
                        ':lambda:path/2015-03-31/functions/',
                        {
                            'Fn::GetAtt': ['myfunction9B95E948', 'Arn'],
                        },
                        '/invocations',
                    ],
                ],
            },
        }));
        test.done();
    },
    'request authorizer with all parameters specified'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySources: [lib_1.IdentitySource.header('whoami')],
            authorizerName: 'myauthorizer',
            resultsCacheTtl: core_1.Duration.minutes(1),
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.whoami',
            Name: 'myauthorizer',
            AuthorizerResultTtlInSeconds: 60,
            AuthorizerUri: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':apigateway:',
                        {
                            Ref: 'AWS::Region',
                        },
                        ':lambda:path/2015-03-31/functions/',
                        {
                            'Fn::GetAtt': ['myfunction9B95E948', 'Arn'],
                        },
                        '/invocations',
                    ],
                ],
            },
        }));
        test.done();
    },
    'token authorizer with assume role'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const role = new iam.Role(stack, 'authorizerassumerole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            roleName: 'authorizerassumerole',
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
            assumeRole: role,
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
            AuthorizerUri: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':apigateway:',
                        {
                            Ref: 'AWS::Region',
                        },
                        ':lambda:path/2015-03-31/functions/',
                        {
                            'Fn::GetAtt': ['myfunction9B95E948', 'Arn'],
                        },
                        '/invocations',
                    ],
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            Roles: [
                stack.resolve(role.roleName),
            ],
            PolicyDocument: {
                Statement: [
                    {
                        Resource: stack.resolve(func.functionArn),
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                    },
                ],
            },
        }, assert_1.ResourcePart.Properties, true));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Permission'));
        test.done();
    },
    'request authorizer with assume role'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const role = new iam.Role(stack, 'authorizerassumerole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            roleName: 'authorizerassumerole',
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            assumeRole: role,
            resultsCacheTtl: core_1.Duration.seconds(0),
            identitySources: [],
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
            AuthorizerUri: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':apigateway:',
                        {
                            Ref: 'AWS::Region',
                        },
                        ':lambda:path/2015-03-31/functions/',
                        {
                            'Fn::GetAtt': ['myfunction9B95E948', 'Arn'],
                        },
                        '/invocations',
                    ],
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            Roles: [
                stack.resolve(role.roleName),
            ],
            PolicyDocument: {
                Statement: [
                    {
                        Resource: stack.resolve(func.functionArn),
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                    },
                ],
            },
        }, assert_1.ResourcePart.Properties, true));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Permission'));
        test.done();
    },
    'token authorizer throws when not attached to a rest api'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
        });
        test.throws(() => stack.resolve(auth.authorizerArn), /must be attached to a RestApi/);
        test.done();
    },
    'request authorizer throws when not attached to a rest api'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySources: [lib_1.IdentitySource.header('myheader')],
        });
        test.throws(() => stack.resolve(auth.authorizerArn), /must be attached to a RestApi/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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