"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool Client', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            AllowedOAuthFlows: ['implicit', 'code'],
            AllowedOAuthScopes: ['profile', 'phone', 'email', 'openid', 'aws.cognito.signin.user.admin'],
            CallbackURLs: ['https://example.com'],
            SupportedIdentityProviders: ['COGNITO'],
        });
    });
    test('client name', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        const client1 = new lib_1.UserPoolClient(stack, 'Client1', {
            userPool: pool,
            userPoolClientName: 'myclient',
        });
        const client2 = new lib_1.UserPoolClient(stack, 'Client2', {
            userPool: pool,
        });
        // THEN
        expect(client1.userPoolClientName).toEqual('myclient');
        expect(() => client2.userPoolClientName).toThrow(/available only if specified on the UserPoolClient during initialization/);
    });
    test('import', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const client = lib_1.UserPoolClient.fromUserPoolClientId(stack, 'Client', 'client-id-1');
        // THEN
        expect(client.userPoolClientId).toEqual('client-id-1');
    });
    test('ExplicitAuthFlows is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ExplicitAuthFlows: assert_1.ABSENT,
        });
    });
    test('ExplicitAuthFlows are correctly named', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client', {
            authFlows: {
                adminUserPassword: true,
                custom: true,
                refreshToken: true,
                userPassword: true,
                userSrp: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ExplicitAuthFlows: [
                'ALLOW_USER_PASSWORD_AUTH',
                'ALLOW_ADMIN_USER_PASSWORD_AUTH',
                'ALLOW_CUSTOM_AUTH',
                'ALLOW_USER_SRP_AUTH',
                'ALLOW_REFRESH_TOKEN_AUTH',
            ],
        });
    });
    test('AllowedOAuthFlows are correctly named', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client1', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                    implicitCodeGrant: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        pool.addClient('Client2', {
            oAuth: {
                flows: {
                    clientCredentials: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['implicit', 'code'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['client_credentials'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
    });
    test('callbackUrl defaults are correctly chosen', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        pool.addClient('Client1', {
            oAuth: {
                flows: {
                    clientCredentials: true,
                },
            },
        });
        pool.addClient('Client2', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
            },
        });
        pool.addClient('Client3', {
            oAuth: {
                flows: {
                    implicitCodeGrant: true,
                },
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['client_credentials'],
            CallbackURLs: assert_1.ABSENT,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['implicit'],
            CallbackURLs: ['https://example.com'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['code'],
            CallbackURLs: ['https://example.com'],
        });
    });
    test('fails when callbackUrls is empty for codeGrant or implicitGrant', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client1', {
            oAuth: {
                flows: { implicitCodeGrant: true },
                callbackUrls: [],
            },
        })).toThrow(/callbackUrl must not be empty/);
        expect(() => pool.addClient('Client3', {
            oAuth: {
                flows: { authorizationCodeGrant: true },
                callbackUrls: [],
            },
        })).toThrow(/callbackUrl must not be empty/);
        expect(() => pool.addClient('Client4', {
            oAuth: {
                flows: { clientCredentials: true },
                callbackUrls: [],
            },
        })).not.toThrow();
    });
    test('logoutUrls can be set', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        pool.addClient('Client', {
            oAuth: {
                logoutUrls: ['https://example.com'],
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            LogoutURLs: ['https://example.com'],
        });
    });
    test('fails when clientCredentials OAuth flow is selected along with codeGrant or implicitGrant', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client1', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                    clientCredentials: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/clientCredentials OAuth flow cannot be selected/);
        expect(() => pool.addClient('Client2', {
            oAuth: {
                flows: {
                    implicitCodeGrant: true,
                    clientCredentials: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/clientCredentials OAuth flow cannot be selected/);
    });
    test('OAuth scopes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client', {
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [
                    lib_1.OAuthScope.PHONE,
                    lib_1.OAuthScope.EMAIL,
                    lib_1.OAuthScope.OPENID,
                    lib_1.OAuthScope.PROFILE,
                    lib_1.OAuthScope.COGNITO_ADMIN,
                    lib_1.OAuthScope.custom('my-resource-server/my-own-scope'),
                ],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthScopes: [
                'phone',
                'email',
                'openid',
                'profile',
                'aws.cognito.signin.user.admin',
                'my-resource-server/my-own-scope',
            ],
        });
    });
    test('OAuthScope - openid is included when email or phone is specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client1', {
            userPoolClientName: 'Client1',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        pool.addClient('Client2', {
            userPoolClientName: 'Client2',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.EMAIL],
            },
        });
        pool.addClient('Client3', {
            userPoolClientName: 'Client3',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.PROFILE],
            },
        });
        pool.addClient('Client4', {
            userPoolClientName: 'Client4',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.COGNITO_ADMIN],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client1',
            AllowedOAuthScopes: ['phone', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client2',
            AllowedOAuthScopes: ['email', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client3',
            AllowedOAuthScopes: ['profile', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client4',
            AllowedOAuthScopes: ['aws.cognito.signin.user.admin'],
        });
    });
    test('enable user existence errors prevention', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
            preventUserExistenceErrors: true,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: 'ENABLED',
        });
    });
    test('disable user existence errors prevention', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
            preventUserExistenceErrors: false,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: 'LEGACY',
        });
    });
    test('user existence errors prevention is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: assert_1.ABSENT,
        });
    });
    test('default supportedIdentityProviders', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const idp = lib_1.UserPoolIdentityProvider.fromProviderName(stack, 'imported', 'userpool-idp');
        pool.registerIdentityProvider(idp);
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            SupportedIdentityProviders: [
                'userpool-idp',
                'COGNITO',
            ],
        });
    });
    test('supportedIdentityProviders', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('AllEnabled', {
            userPoolClientName: 'AllEnabled',
            supportedIdentityProviders: [
                lib_1.UserPoolClientIdentityProvider.COGNITO,
                lib_1.UserPoolClientIdentityProvider.FACEBOOK,
                lib_1.UserPoolClientIdentityProvider.AMAZON,
            ],
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            ClientName: 'AllEnabled',
            SupportedIdentityProviders: ['COGNITO', 'Facebook', 'LoginWithAmazon'],
        });
    });
    test('disableOAuth', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('OAuthDisabled', {
            userPoolClientName: 'OAuthDisabled',
            disableOAuth: true,
        });
        pool.addClient('OAuthEnabled', {
            userPoolClientName: 'OAuthEnabled',
            disableOAuth: false,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            ClientName: 'OAuthDisabled',
            AllowedOAuthFlows: assert_1.ABSENT,
            AllowedOAuthScopes: assert_1.ABSENT,
            AllowedOAuthFlowsUserPoolClient: false,
        });
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            ClientName: 'OAuthEnabled',
            AllowedOAuthFlows: ['implicit', 'code'],
            AllowedOAuthScopes: ['profile', 'phone', 'email', 'openid', 'aws.cognito.signin.user.admin'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
    });
    test('fails when oAuth is specified but is disableOAuth is set', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client', {
            disableOAuth: true,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
            },
        })).toThrow(/disableOAuth is set/);
    });
});
//# sourceMappingURL=data:application/json;base64,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