"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 *
 * @stability stable
 */
class S3Origin {
    /**
     * @stability stable
     */
    constructor(bucket, props = {}) {
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    /**
     * The method called when a given Origin is added (for the first time) to a Distribution.
     *
     * @stability stable
     */
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
_a = JSII_RTTI_SYMBOL_1;
S3Origin[_a] = { fqn: "@aws-cdk/aws-cloudfront-origins.S3Origin", version: "1.117.0" };
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, { originAccessIdentity, ...props }) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
        if (originAccessIdentity) {
            this.originAccessIdentity = originAccessIdentity;
        }
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            // Using a bucket from another stack creates a cyclic reference with
            // the bucket taking a dependency on the generated S3CanonicalUserId for the grant principal,
            // and the distribution having a dependency on the bucket's domain name.
            // Fix this by parenting the OAI in the bucket's stack when cross-stack usage is detected.
            const bucketStack = cdk.Stack.of(this.bucket);
            const bucketInDifferentStack = bucketStack !== cdk.Stack.of(scope);
            const oaiScope = bucketInDifferentStack ? bucketStack : scope;
            const oaiId = bucketInDifferentStack ? `${cdk.Names.uniqueId(scope)}S3Origin` : 'S3Origin';
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(oaiScope, oaiId, {
                comment: `Identity for ${options.originId}`,
            });
            // Used rather than `grantRead` because `grantRead` will grant overly-permissive policies.
            // Only GetObject is needed to retrieve objects for the distribution.
            // This also excludes KMS permissions; currently, OAI only supports SSE-S3 for buckets.
            // Source: https://aws.amazon.com/blogs/networking-and-content-delivery/serving-sse-kms-encrypted-content-from-s3-using-cloudfront/
            this.bucket.addToResourcePolicy(new iam.PolicyStatement({
                resources: [this.bucket.arnForObjects('*')],
                actions: ['s3:GetObject'],
                principals: [this.originAccessIdentity.grantPrincipal],
            }));
        }
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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