# Broadie

A powerful multi-agent framework with persistence, API server, and agent-to-agent communication.

## Features

- **Agent & SubAgent System**: Create intelligent agents with specialized sub-agents
- **Multiple Model Support**: Google Gemini and Vertex AI integration
- **Persistence**: SQLite and PostgreSQL backends with vector storage
- **API Server**: FastAPI-based REST and WebSocket endpoints
- **Agent-to-Agent Communication**: Secure peer discovery and function invocation
- **Tool System**: Extensible tool registry for agent capabilities
- **Memory Management**: Persistent memory with semantic search
- **CLI Tools**: Command-line interface for agent management
- **Notifications**: Slack integration for alerts and updates
  
## Getting Started

Follow these steps to set up and run Broadie locally.

### Prerequisites

- Python 3.10 or newer
- pip
- git

### Setup

```bash
git clone https://github.com/your-org/broadie.git
cd broadie
python3 -m venv venv
source venv/bin/activate        # On Windows: venv\Scripts\activate
pip install -e .[dev]
```

Alternatively, install the latest release from PyPI:

```bash
pip install broadie
```

### Configuration

Copy the example environment file and update with your settings:

```bash
cp .env.example .env
```
  
## Quick Start

### Installation

```bash
pip install broadie
```

### Basic Usage

#### 1. Configuration-driven Agent

Create `main.json` (minimal, tool-less agent configuration):

```json
{  
  "name": "my_agent",
  "description": "A helpful assistant",
  "instruction": "You are a helpful AI assistant.",
  "model": {"provider": "google", "name": "gemini-2.0-flash"}
}
```

Run the agent:

```bash
broadie run main.json
```

#### 2. Programmatic Agent

```python
from broadie import Agent, SubAgent, tool

# Define a custom tool
@tool(name="analyze", description="Analyze data")
def analyze(data: str) -> str:
    return f"Analysis of: {data}"

# Sub-Agent defines its own configuration
class AnalysisAgent(SubAgent):
    def build_config(self):
        return {
            "name": "analysis_agent",
            "description": "Performs data analysis tasks",
            "instruction": "You are a data analysis specialist."
        }

# Main Agent coordinates sub-agents
class CoordinatorAgent(Agent):
    def build_config(self):
        return {
            "name": "coordinator",
            "description": "Coordinates tasks",
            "instruction": "You break tasks among sub-agents.",
            "sub_agents": [AnalysisAgent()]
        }

# Use the agent
agent = CoordinatorAgent()
response = await agent.process_message("Analyze this data: sample")
print(response)
```

## Configuration

### Environment Variables

Copy `.env.example` to `.env` and configure:

```bash
# Google AI
GOOGLE_API_KEY=your_api_key_here
DEFAULT_GEMINI_MODEL=gemini-2.0-flash

# Agent-to-Agent
A2A_ENABLED=true
A2A_AGENT_ID=agent-12345
A2A_AGENT_NAME=my_agent
A2A_TRUSTED_AGENTS=agent-456,agent-789

# Database
DB_DRIVER=sqlite
DB_NAME=broadie.db

# Slack (optional)
SLACK_BOT_TOKEN=xoxb-your-token
SLACK_CHANNEL=#alerts
```

### Agent Configuration

Agent configurations are JSON files defining agent identity and model settings. These configs are minimal and do not include custom tools; to extend with tools, use the programmatic API.

```json
{
  "name": "agent_name",
  "description": "Agent description",
  "instruction": "System prompt for the agent",
  "model": {
    "provider": "google",
    "name": "gemini-2.0-flash"
  },
  "model_settings": {
    "temperature": 0.1,
    "max_tokens": 8000
  },
  "sub_agents": ["subagent1.json", "subagent2.json"]
}
```

## CLI Commands

```bash
# Initialize new project (scaffolds .json configs, agent.py, .env.example, README.md)
broadie init my_project

# Run agent from config or code
broadie run main.json

# Start API server with A2A heartbeat (reads settings from .env)
broadie serve main.json
# alternative: serve a programmatic agent
broadie serve module.path:AgentClass

# Create a minimal JSON agent config in agents/ directory
broadie create agent <agent_name>

# Create a minimal JSON sub-agent config in agents/ directory
broadie create sub-agent <subagent_name>
```

## API Server

Start the API server by pointing to a JSON config or a Python agent reference:

```bash
# Serve JSON-configured agent
broadie serve main.json

# Serve programmatic agent from code
broadie serve module.path:AgentClass
```

### Endpoints

- `GET /health` - Health check
- `POST /agents/{id}/invoke` - Invoke agent function
- `GET /threads` - List all conversation threads
- `GET /threads/{thread_id}/messages` - Get messages for a specific thread
- `WebSocket /ws` - Real-time chat interface

### Example API Usage

```python
import httpx

# Invoke agent
response = httpx.post(
    "http://localhost:8000/agents/my_agent/invoke",
    json={"message": "Hello, agent!"}
)
print(response.json())
```

## Agent-to-Agent Communication

Agents can discover and communicate with each other:

```python
# Agent A can invoke functions on Agent B
result = await agent_a.invoke_peer_agent(
    "agent_b_id", 
    "analyze_threat",
    {"threat_data": "suspicious_activity"}
)
```

### Security Model

- Agents can see all peers via registry
- Only trusted agents (configured in `A2A_TRUSTED_AGENTS`) can invoke functions
- All communications include agent identity verification

## Memory System

Agents have persistent memory with semantic search:

```python
# Store memory
memory_id = await agent.memory_manager.remember(
    "Important information",
    importance=0.8,
    tags=["important", "project_x"]
)

# Search memories
memories = await agent.memory_manager.search(
    "project information",
    limit=5
)

# Recall specific memory
memory = await agent.memory_manager.recall(memory_id)
```

## Tools

Create custom tools for agents:

```python
from broadie.core.tools import tool

@tool(name="weather", description="Get weather information")
async def get_weather(location: str) -> str:
    # Weather API integration
    return f"Weather in {location}: Sunny, 72°F"

# Register tool
from broadie.core.tools import get_global_registry
registry = get_global_registry()
# Tool is auto-registered via decorator
```

## Development

### Setup Development Environment

```bash
git clone https://github.com/your-org/broadie
cd broadie
pip install -e ".[dev]"
```

### Running Tests

```bash
pytest
pytest --cov=src/broadie --cov-report=html
```

### Code Quality

```bash
black src/ tests/
isort src/ tests/
flake8 src/ tests/
mypy src/
```

## Architecture

Broadie follows SOLID principles with clean separation of concerns:

- **Core**: Agent, SubAgent, Tool, Memory abstractions
- **Config**: Environment and JSON configuration management  
- **Persistence**: SQLite, PostgreSQL, and vector storage backends
- **API**: FastAPI server with REST and WebSocket support
- **A2A**: Agent-to-agent communication and discovery
- **CLI**: Command-line tools for agent management
- **Utils**: Logging, exceptions, and schema validation

<!-- Removed examples section: no examples directory exists -->

## Contributing

1. Fork the repository
2. Create a feature branch (`git checkout -b feature/amazing-feature`)
3. Make your changes
4. Add tests for your changes
5. Run the test suite (`pytest`)
6. Commit your changes (`git commit -m 'Add amazing feature'`)
7. Push to your branch (`git push origin feature/amazing-feature`)
8. Open a Pull Request

## License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## Support

- Documentation: [https://docs.broadie.ai](https://docs.broadie.ai)
- Issues: [https://github.com/your-org/broadie/issues](https://github.com/your-org/broadie/issues)
- Discussions: [https://github.com/your-org/broadie/discussions](https://github.com/your-org/broadie/discussions)


## Developer Quick Reference

The following notes are adapted from the internal CLAUDE.md guide for convenience.

### Testing Commands
- Lint: `python -m black --check .` or `python -m ruff check .`
- Type checking: `python -m mypy broadie/`
- Tests: `python -m pytest`
- Build: `python -m build` or `pip install -e .`

### Available Tools

#### Built-in Tools (Always Available)
- `write_todos` - Create and manage task lists  
- `ls` - List files in filesystem
- `read_file` - Read file contents
- `write_file` - Write content to files
- `edit_file` - Make exact string replacements

#### Google Drive Tools (Auto-included when configured)
- `search_google_drive` - Search files and folders by name/content
- `read_google_drive_file` - Read document/sheet content
- `write_google_drive_file` - Create/update documents
- `create_google_sheet` - Create new spreadsheets
- `update_google_sheet` - Update sheet data
- `list_drive_folders` - List folders and contents

#### Slack Tools (Auto-included when configured)
- `search_slack_messages` - Search messages across workspace
- `send_slack_message` - Send messages to channels
- `send_slack_dm` - Send direct messages
- `list_slack_channels` - List available channels
- `list_slack_users` - List workspace users
- `get_slack_user_info` - Get user details
- `create_slack_thread` - Reply to messages
- `upload_slack_file` - Upload files to channels

### Architecture Notes
- All tools are automatically available — no need to specify them in agent configs
- Built-in tools are always included (write_todos, file operations, etc.)
- Integration tools are automatically included when properly configured
- Tools use a global registry with auto-registration
- Configuration managed through `broadie.config.integrations`
- Tool registry supports categories: builtin, google_drive, slack
- Integration tools gracefully fail when dependencies are unavailable
- Production-quality error handling with helpful setup instructions

### Agent Configuration
Agents automatically get all available tools. You can still specify explicit tools in the `tools` list to add additional custom tools, but built-ins and integrations are always included.



## Packaging & Release

Follow these steps to produce installable distributions and publish a release.

1. Build the UI (optional but recommended for releases)
   - cd broad-ui
   - npm install
   - npm run build
   - This outputs the SPA to broadie/static/ui/, which is included in the package.

2. Build Python distributions
   - Ensure build tools are available: pip install -U build twine
   - Or install dev extras: pip install -e .[dev]
   - Build: python -m build
     - Produces dist/broadie-<version>.tar.gz and dist/broadie-<version>-py3-none-any.whl

3. Verify contents
   - Inspect the wheel/sdist (tar -tf dist/*.tar.gz) for:
     - broadie/static/ui/**/* (UI assets)
     - broadie/templates/*.json
     - broadie/py.typed

4. Upload to PyPI (or TestPyPI)
   - TestPyPI: twine upload -r testpypi dist/*
   - PyPI: twine upload dist/*

Notes
- Optional extras:
  - dev: tooling for tests and builds (pytest, black, isort, mypy, ruff, build, twine)
  - google, slack, postgres, sqlite: install integration-specific dependencies
- Source distribution is governed by MANIFEST.in; wheels include package-data via pyproject.toml.

### Troubleshooting installed CLI

- Symptom: `ModuleNotFoundError: No module named 'broadie'` right after running `broadie`.
  - Cause: You may have an older wheel that didn’t include the package modules.
  - Fix: Upgrade to the latest version: `python -m pip install -U broadie`.
  - If testing from a local build, ensure the wheel includes the `broadie/` package:
    - Build: `python -m build`
    - Inspect wheel: `python -m pip install wheel-inspect && wheel2zip dist/*.whl && unzip -l dist/*.whl.zip | sed -n '1,120p'`
    - You should see entries under `broadie/...`. Rebuild if missing.
