"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LustreFileSystem = exports.LustreDeploymentType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * The different kinds of file system deployments used by Lustre.
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    /**
     * Original type for shorter term data processing. Data is not replicated and does not persist on server fail.
     */
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    /**
     * Newer type for shorter term data processing. Data is not replicated and does not persist on server fail.
     * Provides better support for spiky workloads.
     */
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    /**
     * Long term storage. Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
    /**
     * Newer type of long term storage with higher throughput tiers.
     * Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_2"] = "PERSISTENT_2";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 *
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_fsx_LustreFileSystemProps(props);
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: (_b = props.lustreConfiguration.weeklyMaintenanceStartTime) === null || _b === void 0 ? void 0 : _b.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.stack.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * Import an existing FSx for Lustre file system from the given properties.
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_fsx_FileSystemAttributes(attrs);
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1 && deploymentType !== LustreDeploymentType.PERSISTENT_2) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1/PERSISTENT_2 deployment types, received: ' + deploymentType);
        }
        if (deploymentType === LustreDeploymentType.PERSISTENT_1) {
            if (![50, 100, 200].includes(perUnitStorageThroughput)) {
                throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB for PERSISTENT_1 deployment type, got: ' + perUnitStorageThroughput);
            }
        }
        if (deploymentType === LustreDeploymentType.PERSISTENT_2) {
            if (![125, 250, 500, 1000].includes(perUnitStorageThroughput)) {
                throw new Error('perUnitStorageThroughput must be 125, 250, 500 or 1000 MB/s/TiB for PERSISTENT_2 deployment type, got: ' + perUnitStorageThroughput);
            }
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
exports.LustreFileSystem = LustreFileSystem;
_a = JSII_RTTI_SYMBOL_1;
LustreFileSystem[_a] = { fqn: "@aws-cdk/aws-fsx.LustreFileSystem", version: "1.147.0" };
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
//# sourceMappingURL=data:application/json;base64,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