/*******************************************************************************
 * Copyright 2013-2022 Aerospike, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/

#include <Python.h>

#include <aerospike/aerospike_info.h>
#include <aerospike/as_error.h>
#include <aerospike/as_node.h>
#include <aerospike/as_cluster.h>

#include "client.h"
#include "conversions.h"
#include "exceptions.h"
#include "policy.h"

/**
 *******************************************************************************************************
 * Return the base64 representation of the passed in expression.
 *
 * @param self                  AerospikeClient object
 * @param args                  The args is a tuple object containing an argument
 *                              list passed from Python to a C function
 * @param kwds                  Dictionary of keywords
 *
 * Returns a string containing the base64 representation of the passed in expression.
 * In case of error, appropriate exceptions will be raised.
 *******************************************************************************************************
 */
PyObject *AerospikeClient_GetExpressionBase64(AerospikeClient *self,
                                              PyObject *args, PyObject *kwds)
{
    // function args
    PyObject *py_expression_filter = NULL;

    // utility vars
    char *base64_filter = NULL;
    as_exp *exp_list_p = NULL;

    PyObject *py_response = NULL;

    as_error err;
    as_error_init(&err);

    static char *kwlist[] = {"expression", NULL};
    if (PyArg_ParseTupleAndKeywords(args, kwds, "O:get_expression_base64",
                                    kwlist, &py_expression_filter) == false) {
        return NULL;
    }

    //convert filter to base64
    if (py_expression_filter == NULL || !PyList_Check(py_expression_filter)) {
        as_error_update(&err, AEROSPIKE_ERR_PARAM,
                        "expression must be a non empty list of 4 element "
                        "tuples, generated by a compiled aerospike expression");
        goto CLEANUP;
    }

    if (convert_exp_list(self, py_expression_filter, &exp_list_p, &err) !=
        AEROSPIKE_OK) {
        goto CLEANUP;
    }

    base64_filter = as_exp_compile_b64(exp_list_p);

    py_response = PyUnicode_FromString((const char *)base64_filter);

CLEANUP:

    if (exp_list_p != NULL) {
        as_exp_destroy(exp_list_p);
    }

    if (base64_filter != NULL) {
        as_exp_destroy_b64(base64_filter);
    }

    if (err.code != AEROSPIKE_OK) {
        raise_exception(&err);
        return NULL;
    }

    return py_response;
}
