"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeECRRepositoryAspect = exports.StepFunctionsSageMakerTrainingJob = exports.ECSTaskDefinition = exports.ECRRepositoryAspect = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const FN_SUB = 'Fn::Sub';
/**
 * Abtract aspect for ECR repository.
 *
 * You must provide the account id in props or set BSS_IMAGE_ASSET_ACCOUNT_ID in env
 */
class ECRRepositoryAspect {
    constructor(props = {}) {
        var _a;
        this.account = (_a = props.imageAssetAccountId) !== null && _a !== void 0 ? _a : process.env.BSS_IMAGE_ASSET_ACCOUNT_ID;
    }
    getRepoName(imageUri) {
        const matches = /\d{12}\.dkr\.ecr\..*\/(.*):.*/g.exec(imageUri);
        if (matches) {
            return matches[1];
        }
        return undefined;
    }
    crossAccountECRPolicy(stack, repoName) {
        const policy = ECRRepositoryAspect._repoPolicies.get(repoName);
        if (policy) {
            return policy;
        }
        const newPolicy = new aws_iam_1.Policy(stack, `CrossAccountECR-${repoName}`, {
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'ecr:BatchCheckLayerAvailability',
                        'ecr:GetDownloadUrlForLayer',
                        'ecr:BatchGetImage',
                    ],
                    resources: [
                        aws_cdk_lib_1.Arn.format({
                            account: this.account,
                            service: 'ecr',
                            resource: 'repository',
                            resourceName: repoName,
                        }, stack),
                    ],
                }),
            ],
        });
        ECRRepositoryAspect._repoPolicies.set(repoName, newPolicy);
        return newPolicy;
    }
}
exports.ECRRepositoryAspect = ECRRepositoryAspect;
ECRRepositoryAspect._repoPolicies = new Map();
/**
 * Process the image assets in ECS task definition
 */
class ECSTaskDefinition extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
    }
    hasBeReplaced(prop) {
        if (typeof prop.image === 'object' && FN_SUB in prop.image &&
            prop.image[FN_SUB].indexOf(this.account) > -1) {
            return prop.image[FN_SUB];
        }
        else if (prop.image && prop.image && prop.image.indexOf(this.account) > -1) {
            return prop.image;
        }
        return undefined;
    }
    visit(construct) {
        if (construct instanceof aws_ecs_1.TaskDefinition) {
            const containers = construct.stack.resolve(construct.node.defaultChild.containerDefinitions);
            let imageUri = undefined;
            if (containers instanceof Array) {
                for (const container of containers) {
                    if (container) {
                        imageUri = this.hasBeReplaced(container);
                        if (imageUri) {
                            break;
                        }
                    }
                }
            }
            else if (containers) {
                imageUri = this.hasBeReplaced(containers);
            }
            if (imageUri) {
                const repoName = this.getRepoName(imageUri);
                if (repoName) {
                    construct.executionRole.attachInlinePolicy(this.crossAccountECRPolicy(construct.stack, repoName));
                }
            }
        }
    }
}
exports.ECSTaskDefinition = ECSTaskDefinition;
/**
 * Process the image assets in SageMaker training job in Step Functions
 */
class StepFunctionsSageMakerTrainingJob extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
    }
    visit(construct) {
        if (construct instanceof aws_stepfunctions_tasks_1.SageMakerCreateTrainingJob) {
            const stack = aws_cdk_lib_1.Stack.of(construct);
            const state = construct.toStateJson();
            const image = stack.resolve(state.Parameters.AlgorithmSpecification.TrainingImage);
            if (FN_SUB in image) {
                const repoName = this.getRepoName(image[FN_SUB]);
                if (repoName) {
                    construct.role.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                }
            }
        }
    }
}
exports.StepFunctionsSageMakerTrainingJob = StepFunctionsSageMakerTrainingJob;
/**
 * Default ECR asset aspect, support using ECR assets in below services,
 *
 * - ECS task definition
 * - SageMaker training job in Step Functions
 */
class CompositeECRRepositoryAspect extends ECRRepositoryAspect {
    constructor(props = {}) {
        super(props);
        this._aspects = [
            new ECSTaskDefinition(props),
            new StepFunctionsSageMakerTrainingJob(props),
        ];
    }
    visit(construct) {
        for (const _aspect of this._aspects) {
            _aspect.visit(construct);
        }
    }
}
exports.CompositeECRRepositoryAspect = CompositeECRRepositoryAspect;
//# sourceMappingURL=data:application/json;base64,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