"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cloudformation_generated_1 = require("./cloudformation.generated");
/**
 * Represents a provider for an AWS CloudFormation custom resources.
 */
class CustomResourceProvider {
    /**
     * @param serviceToken the ServiceToken which contains the ARN for this provider.
     */
    constructor(serviceToken) {
        this.serviceToken = serviceToken;
    }
    /**
     * The Lambda provider that implements this custom resource.
     *
     * We recommend using a lambda.SingletonFunction for this.
     */
    static fromLambda(handler) {
        return new CustomResourceProvider(handler.functionArn);
    }
    /**
     * The SNS Topic for the provider that implements this custom resource.
     */
    static fromTopic(topic) {
        return new CustomResourceProvider(topic.topicArn);
    }
    /**
     * Use AWS Lambda as a provider.
     * @deprecated use `fromLambda`
     */
    static lambda(handler) { return this.fromLambda(handler); }
    /**
     * Use an SNS topic as the provider.
     * @deprecated use `fromTopic`
     */
    static topic(topic) { return this.fromTopic(topic); }
    bind(_) {
        return { serviceToken: this.serviceToken };
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
/**
 * Custom resource that is implemented using a Lambda
 *
 * As a custom resource author, you should be publishing a subclass of this class
 * that hides the choice of provider, and accepts a strongly-typed properties
 * object with the properties your provider accepts.
 */
class CustomResource extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const type = renderResourceType(props.resourceType);
        const providerConfig = props.provider.bind(this);
        this.resource = new core_1.CfnResource(this, 'Default', {
            type,
            properties: {
                ServiceToken: providerConfig.serviceToken,
                ...uppercaseProperties(props.properties || {})
            }
        });
        this.resource.applyRemovalPolicy(props.removalPolicy, {
            default: core_1.RemovalPolicy.DESTROY
        });
    }
    /**
     * The physical name of this custom resource.
     */
    get ref() {
        return this.resource.ref;
    }
    /**
     * Returns the value of an attribute of the custom resource of an arbitrary
     * type. Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt`. Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getAttString` for string attributes.
     */
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    /**
     * Returns the value of an attribute of the custom resource of type string.
     * Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt` encoded as a string.
     */
    getAttString(attributeName) {
        return core_1.Token.asString(this.getAtt(attributeName));
    }
}
exports.CustomResource = CustomResource;
/**
 * Uppercase the first letter of every property name
 *
 * It's customary for CloudFormation properties to start with capitals, and our
 * properties to start with lowercase, so this function translates from one
 * to the other
 */
function uppercaseProperties(props) {
    const ret = {};
    Object.keys(props).forEach(key => {
        const upper = key.substr(0, 1).toUpperCase() + key.substr(1);
        ret[upper] = props[key];
    });
    return ret;
}
function renderResourceType(resourceType) {
    if (!resourceType) {
        return cloudformation_generated_1.CfnCustomResource.CFN_RESOURCE_TYPE_NAME;
    }
    if (!resourceType.startsWith('Custom::')) {
        throw new Error(`Custom resource type must begin with "Custom::" (${resourceType})`);
    }
    const typeName = resourceType.substr(resourceType.indexOf('::') + 2);
    if (typeName.length > 60) {
        throw new Error(`Custom resource type length > 60 (${resourceType})`);
    }
    if (!/^[a-z0-9_@-]+$/i.test(typeName)) {
        throw new Error(`Custom resource type name can only include alphanumeric characters and _@- (${typeName})`);
    }
    return resourceType;
}
//# sourceMappingURL=data:application/json;base64,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