"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
const nested_stack_1 = require("../lib/nested-stack");
module.exports = {
    'fails if defined as a root'(test) {
        // THEN
        test.throws(() => new nested_stack_1.NestedStack(undefined, 'boom'), /Nested stacks cannot be defined as a root construct/);
        test.done();
    },
    'fails if defined without a parent stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const group = new core_1.Construct(app, 'group');
        // THEN
        test.throws(() => new nested_stack_1.NestedStack(app, 'boom'), /must be defined within scope of another non-nested stack/);
        test.throws(() => new nested_stack_1.NestedStack(group, 'bam'), /must be defined within scope of another non-nested stack/);
        test.done();
    },
    'can be defined as a direct child or an indirect child of a Stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        // THEN
        new nested_stack_1.NestedStack(parent, 'direct');
        new nested_stack_1.NestedStack(new core_1.Construct(parent, 'group'), 'indirect');
        test.done();
    },
    'nested stack is not synthesized as a stack artifact into the assembly'(test) {
        // GIVEN
        const app = new core_1.App();
        const parentStack = new core_1.Stack(app, 'parent-stack');
        new nested_stack_1.NestedStack(parentStack, 'nested-stack');
        // WHEN
        const assembly = app.synth();
        // THEN
        test.deepEqual(assembly.artifacts.length, 2);
        test.done();
    },
    'the template of the nested stack is synthesized into the cloud assembly'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent-stack');
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        new core_1.CfnResource(nested, 'ResourceInNestedStack', { type: 'AWS::Resource::Nested' });
        // WHEN
        const assembly = app.synth();
        // THEN
        const template = JSON.parse(fs.readFileSync(path.join(assembly.directory, `${nested.node.uniqueId}.nested.template.json`), 'utf-8'));
        test.deepEqual(template, {
            Resources: {
                ResourceInNestedStack: {
                    Type: 'AWS::Resource::Nested'
                }
            }
        });
        test.done();
    },
    'file asset metadata is associated with the parent stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent-stack');
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        new core_1.CfnResource(nested, 'ResourceInNestedStack', { type: 'AWS::Resource::Nested' });
        // WHEN
        const assembly = app.synth();
        // THEN
        test.deepEqual(assembly.getStackByName(parent.stackName).assets, [{
                path: 'parentstacknestedstack844892C0.nested.template.json',
                id: 'c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096',
                packaging: 'file',
                sourceHash: 'c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096',
                s3BucketParameter: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3BucketDA8C3345',
                s3KeyParameter: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6',
                artifactHashParameter: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096ArtifactHash8DE450C7'
            }]);
        test.done();
    },
    'aws::cloudformation::stack is synthesized in the parent scope'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent-stack');
        // WHEN
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        new core_1.CfnResource(nested, 'ResourceInNestedStack', { type: 'AWS::Resource::Nested' });
        // THEN
        const assembly = app.synth();
        // assembly has one stack (the parent)
        test.deepEqual(assembly.stacks.length, 1);
        // but this stack has an asset that points to the synthesized template
        test.deepEqual(assembly.stacks[0].assets[0].path, 'parentstacknestedstack844892C0.nested.template.json');
        // the template includes our resource
        const filePath = path.join(assembly.directory, assembly.stacks[0].assets[0].path);
        test.deepEqual(JSON.parse(fs.readFileSync(filePath).toString('utf-8')), {
            Resources: { ResourceInNestedStack: { Type: 'AWS::Resource::Nested' } }
        });
        // the parent template includes the parameters and the nested stack resource which points to the s3 url
        assert_1.expect(parent).toMatch({
            Resources: {
                nestedstackNestedStacknestedstackNestedStackResource71CDD241: {
                    Type: "AWS::CloudFormation::Stack",
                    Properties: {
                        TemplateURL: {
                            "Fn::Join": [
                                "",
                                [
                                    "https://s3.",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ".",
                                    {
                                        Ref: "AWS::URLSuffix"
                                    },
                                    "/",
                                    {
                                        Ref: "AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3BucketDA8C3345"
                                    },
                                    "/",
                                    {
                                        "Fn::Select": [
                                            0,
                                            {
                                                "Fn::Split": [
                                                    "||",
                                                    {
                                                        Ref: "AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6"
                                                    }
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        "Fn::Select": [
                                            1,
                                            {
                                                "Fn::Split": [
                                                    "||",
                                                    {
                                                        Ref: "AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6"
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            ]
                        }
                    }
                }
            },
            Parameters: {
                AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3BucketDA8C3345: {
                    Type: "String",
                    Description: "S3 bucket for asset \"c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096\""
                },
                AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6: {
                    Type: "String",
                    Description: "S3 key for asset version \"c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096\""
                },
                AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096ArtifactHash8DE450C7: {
                    Type: "String",
                    Description: "Artifact hash for asset \"c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096\""
                }
            }
        });
        test.done();
    },
    'Stack.of()'(test) {
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id) {
                super(scope, id);
                const param = new core_1.CfnParameter(this, 'param', { type: 'String' });
                this.stackOfChild = core_1.Stack.of(param);
            }
        }
        const parent = new core_1.Stack();
        const nested = new MyNestedStack(parent, 'nested');
        test.ok(nested.stackOfChild === nested);
        test.ok(core_1.Stack.of(nested) === nested);
        test.done();
    },
    'references within the nested stack are not reported as cross stack references'(test) {
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id) {
                super(scope, id);
                const param = new core_1.CfnParameter(this, 'param', { type: 'String' });
                new core_1.CfnResource(this, 'resource', {
                    type: 'My::Resource',
                    properties: {
                        SomeProp: param.valueAsString
                    }
                });
            }
        }
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent');
        new MyNestedStack(parent, 'nested');
        // references are added during "prepare"
        const assembly = app.synth();
        test.deepEqual(assembly.stacks.length, 1);
        test.deepEqual(assembly.stacks[0].dependencies, []);
        test.done();
    },
    'references to a resource from the parent stack in a nested stack is translated into a cfn parameter'(test) {
        // WHEN
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id, resourceFromParent) {
                super(scope, id);
                new core_1.CfnResource(this, 'resource', {
                    type: 'AWS::Child::Resource',
                    properties: {
                        ReferenceToResourceInParentStack: resourceFromParent.ref
                    }
                });
                new core_1.CfnResource(this, 'resource2', {
                    type: 'My::Resource::2',
                    properties: {
                        Prop1: resourceFromParent.getAtt('Attr'),
                        Prop2: resourceFromParent.ref,
                    }
                });
            }
        }
        const app = new core_1.App();
        const parentStack = new core_1.Stack(app, 'parent');
        const resource = new core_1.CfnResource(parentStack, 'parent-resource', { type: 'AWS::Parent::Resource' });
        const nested = new MyNestedStack(parentStack, 'nested', resource);
        // THEN
        app.synth();
        // nested template should use a parameter to reference the resource from the parent stack
        assert_1.expect(nested).toMatch({
            Resources: {
                resource: {
                    Type: 'AWS::Child::Resource',
                    Properties: { ReferenceToResourceInParentStack: { Ref: 'referencetoparentparentresourceD56EA8F7Ref' } }
                },
                resource2: {
                    Type: 'My::Resource::2',
                    Properties: {
                        Prop1: { Ref: 'referencetoparentparentresourceD56EA8F7Attr' },
                        Prop2: { Ref: 'referencetoparentparentresourceD56EA8F7Ref' }
                    }
                }
            },
            Parameters: {
                referencetoparentparentresourceD56EA8F7Ref: { Type: 'String' },
                referencetoparentparentresourceD56EA8F7Attr: { Type: 'String' }
            }
        });
        // parent template should pass in the value through the parameter
        assert_1.expect(parentStack).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetoparentparentresourceD56EA8F7Ref: {
                    Ref: "parentresource"
                },
                referencetoparentparentresourceD56EA8F7Attr: {
                    "Fn::GetAtt": [
                        "parentresource",
                        "Attr"
                    ]
                }
            },
        }));
        test.done();
    },
    'references to a resource in the nested stack in the parent is translated into a cfn output'(test) {
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id) {
                super(scope, id);
                this.resourceFromChild = new core_1.CfnResource(this, 'resource', {
                    type: 'AWS::Child::Resource',
                });
            }
        }
        const app = new core_1.App();
        const parentStack = new core_1.Stack(app, 'parent');
        const nested = new MyNestedStack(parentStack, 'nested');
        new core_1.CfnResource(parentStack, 'another-parent-resource', {
            type: 'AWS::Parent::Resource',
            properties: {
                RefToResourceInNestedStack: nested.resourceFromChild.ref
            }
        });
        // references are added during "prepare"
        app.synth();
        // nested template should use a parameter to reference the resource from the parent stack
        assert_1.expect(nested).toMatch({
            Resources: {
                resource: { Type: 'AWS::Child::Resource' }
            },
            Outputs: {
                parentnestedresource4D680677Ref: { Value: { Ref: 'resource' } }
            }
        });
        // parent template should pass in the value through the parameter
        assert_1.expect(parentStack).to(assert_1.haveResource('AWS::Parent::Resource', {
            RefToResourceInNestedStack: {
                "Fn::GetAtt": [
                    "nestedNestedStacknestedNestedStackResource3DD143BF",
                    "Outputs.parentnestedresource4D680677Ref"
                ]
            }
        }));
        test.done();
    },
    'nested stack references a resource from another non-nested stack (not the parent)'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const stack2 = new core_1.Stack(app, 'Stack2');
        const nestedUnderStack1 = new nested_stack_1.NestedStack(stack1, 'NestedUnderStack1');
        const resourceInStack2 = new core_1.CfnResource(stack2, 'ResourceInStack2', { type: 'MyResource' });
        // WHEN
        new core_1.CfnResource(nestedUnderStack1, 'ResourceInNestedStack1', {
            type: 'Nested::Resource',
            properties: {
                RefToSibling: resourceInStack2.getAtt('MyAttribute')
            }
        });
        // THEN
        const assembly = app.synth();
        // producing stack should have an export
        assert_1.expect(stack2).toMatch({
            Resources: {
                ResourceInStack2: { Type: "MyResource" }
            },
            Outputs: {
                ExportsOutputFnGetAttResourceInStack2MyAttributeC15F1009: {
                    Value: { "Fn::GetAtt": ["ResourceInStack2", "MyAttribute"] },
                    Export: { Name: "Stack2:ExportsOutputFnGetAttResourceInStack2MyAttributeC15F1009" }
                }
            }
        });
        // nested stack uses Fn::ImportValue like normal
        assert_1.expect(nestedUnderStack1).toMatch({
            Resources: {
                ResourceInNestedStack1: {
                    Type: "Nested::Resource",
                    Properties: {
                        RefToSibling: {
                            "Fn::ImportValue": "Stack2:ExportsOutputFnGetAttResourceInStack2MyAttributeC15F1009"
                        }
                    }
                }
            }
        });
        // verify a depedency was established between the parents
        const stack1Artifact = assembly.getStackByName(stack1.stackName);
        const stack2Artifact = assembly.getStackByName(stack2.stackName);
        test.deepEqual(stack1Artifact.dependencies.length, 1);
        test.deepEqual(stack2Artifact.dependencies.length, 0);
        test.same(stack1Artifact.dependencies[0], stack2Artifact);
        test.done();
    },
    'another non-nested stack takes a reference on a resource within the nested stack (the parent exports)'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const stack2 = new core_1.Stack(app, 'Stack2');
        const nestedUnderStack1 = new nested_stack_1.NestedStack(stack1, 'NestedUnderStack1');
        const resourceInNestedStack = new core_1.CfnResource(nestedUnderStack1, 'ResourceInNestedStack', { type: 'MyResource' });
        // WHEN
        new core_1.CfnResource(stack2, 'ResourceInStack2', {
            type: 'JustResource',
            properties: {
                RefToSibling: resourceInNestedStack.getAtt('MyAttribute')
            }
        });
        // THEN
        const assembly = app.synth();
        // nested stack should output this value as if it was referenced by the parent (without the export)
        assert_1.expect(nestedUnderStack1).toMatch({
            Resources: {
                ResourceInNestedStack: {
                    Type: "MyResource"
                }
            },
            Outputs: {
                Stack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute: {
                    Value: {
                        "Fn::GetAtt": [
                            "ResourceInNestedStack",
                            "MyAttribute"
                        ]
                    }
                }
            }
        });
        // parent stack (stack1) should export this value
        test.deepEqual(assembly.getStackByName(stack1.stackName).template.Outputs, {
            ExportsOutputFnGetAttNestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305BOutputsStack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute564EECF3: {
                Value: { 'Fn::GetAtt': ['NestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305B', 'Outputs.Stack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute'] },
                Export: { Name: 'Stack1:ExportsOutputFnGetAttNestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305BOutputsStack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute564EECF3' }
            }
        });
        // consuming stack should use ImportValue to import the value from the parent stack
        assert_1.expect(stack2).toMatch({
            Resources: {
                ResourceInStack2: {
                    Type: "JustResource",
                    Properties: {
                        RefToSibling: {
                            "Fn::ImportValue": "Stack1:ExportsOutputFnGetAttNestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305BOutputsStack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute564EECF3"
                        }
                    }
                }
            }
        });
        test.deepEqual(assembly.stacks.length, 2);
        const stack1Artifact = assembly.getStackByName(stack1.stackName);
        const stack2Artifact = assembly.getStackByName(stack2.stackName);
        test.deepEqual(stack1Artifact.dependencies.length, 0);
        test.deepEqual(stack2Artifact.dependencies.length, 1);
        test.same(stack2Artifact.dependencies[0], stack1Artifact);
        test.done();
    },
    'references between sibling nested stacks should output from one and getAtt from the other'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'Parent');
        const nested1 = new nested_stack_1.NestedStack(parent, 'Nested1');
        const nested2 = new nested_stack_1.NestedStack(parent, 'Nested2');
        const resource1 = new core_1.CfnResource(nested1, 'Resource1', { type: 'Resource1' });
        // WHEN
        new core_1.CfnResource(nested2, 'Resource2', {
            type: 'Resource2',
            properties: {
                RefToResource1: resource1.ref
            }
        });
        // THEN
        app.synth();
        // producing nested stack
        assert_1.expect(nested1).toMatch({
            Resources: {
                Resource1: {
                    Type: "Resource1"
                }
            },
            Outputs: {
                ParentNested1Resource15F3F0657Ref: {
                    Value: {
                        Ref: "Resource1"
                    }
                }
            }
        });
        // consuming nested stack
        assert_1.expect(nested2).toMatch({
            Resources: {
                Resource2: {
                    Type: "Resource2",
                    Properties: {
                        RefToResource1: {
                            Ref: "referencetoParentNested1NestedStackNested1NestedStackResource9C05342COutputsParentNested1Resource15F3F0657Ref"
                        }
                    }
                }
            },
            Parameters: {
                referencetoParentNested1NestedStackNested1NestedStackResource9C05342COutputsParentNested1Resource15F3F0657Ref: {
                    Type: "String"
                }
            }
        });
        // parent
        assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetoParentNested1NestedStackNested1NestedStackResource9C05342COutputsParentNested1Resource15F3F0657Ref: {
                    "Fn::GetAtt": [
                        "Nested1NestedStackNested1NestedStackResourceCD0AD36B",
                        "Outputs.ParentNested1Resource15F3F0657Ref"
                    ]
                }
            }
        }));
        test.done();
    },
    'stackId returns AWS::StackId when referenced from the context of the nested stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(nested, 'NestedResource', {
            type: 'Nested::Resource',
            properties: { MyStackId: nested.stackId }
        });
        // THEN
        assert_1.expect(nested).to(assert_1.haveResource('Nested::Resource', {
            MyStackId: { Ref: "AWS::StackId" }
        }));
        test.done();
    },
    'stackId returns the REF of the CloudFormation::Stack resource when referenced from the parent stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(parent, 'ParentResource', {
            type: 'Parent::Resource',
            properties: { NestedStackId: nested.stackId }
        });
        // THEN
        assert_1.expect(parent).to(assert_1.haveResource('Parent::Resource', {
            NestedStackId: { Ref: "NestedStackNestedStackNestedStackNestedStackResourceB70834FD" }
        }));
        test.done();
    },
    'stackName returns AWS::StackName when referenced from the context of the nested stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(nested, 'NestedResource', {
            type: 'Nested::Resource',
            properties: { MyStackName: nested.stackName }
        });
        // THEN
        assert_1.expect(nested).to(assert_1.haveResource('Nested::Resource', {
            MyStackName: { Ref: "AWS::StackName" }
        }));
        test.done();
    },
    'stackName returns the REF of the CloudFormation::Stack resource when referenced from the parent stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(parent, 'ParentResource', {
            type: 'Parent::Resource',
            properties: { NestedStackName: nested.stackName }
        });
        // THEN
        assert_1.expect(parent).to(assert_1.haveResource('Parent::Resource', {
            NestedStackName: {
                "Fn::Select": [
                    1,
                    {
                        "Fn::Split": [
                            "/",
                            {
                                Ref: "NestedStackNestedStackNestedStackNestedStackResourceB70834FD"
                            }
                        ]
                    }
                ]
            }
        }));
        test.done();
    },
    '"account", "region" and "environment" are all derived from the parent'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'ParentStack', { env: { account: '1234account', region: 'us-east-44' } });
        // WHEN
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // THEN
        test.deepEqual(nested.environment, parent.environment);
        test.deepEqual(nested.account, parent.account);
        test.deepEqual(nested.region, parent.region);
        test.done();
    },
    'double-nested stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'stack');
        // WHEN
        const nested1 = new nested_stack_1.NestedStack(parent, 'Nested1');
        const nested2 = new nested_stack_1.NestedStack(nested1, 'Nested2');
        new core_1.CfnResource(nested1, 'Resource1', { type: 'Resource::1' });
        new core_1.CfnResource(nested2, 'Resource2', { type: 'Resource::2' });
        // THEN
        const assembly = app.synth();
        // nested2 is a "leaf", so it's just the resource
        assert_1.expect(nested2).toMatch({
            Resources: {
                Resource2: { Type: "Resource::2" }
            }
        });
        // nested1 wires the nested2 template through parameters, so we expect those
        assert_1.expect(nested1).to(assert_1.haveResource('Resource::1'));
        const nested2Template = assert_1.SynthUtils.toCloudFormation(nested1);
        test.deepEqual(nested2Template.Parameters, {
            referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketE8768F5CRef: { Type: 'String' },
            referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey49DD83A2Ref: { Type: 'String' },
        });
        // parent stack should have two sets of parameters. one for the first nested stack and the second
        // for the second nested stack, passed in as parameters to the first
        const template = assert_1.SynthUtils.toCloudFormation(parent);
        test.deepEqual(template.Parameters, {
            AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketDE3B88D6: { Type: 'String', Description: 'S3 bucket for asset "8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235c"' },
            AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey3A62EFEA: { Type: 'String', Description: 'S3 key for asset version "8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235c"' },
            AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cArtifactHash7DC546E0: { Type: 'String', Description: 'Artifact hash for asset "8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235c"' },
            AssetParameters8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bfS3Bucket76ACFB38: { Type: 'String', Description: 'S3 bucket for asset "8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bf"' },
            AssetParameters8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bfS3VersionKey04162EF1: { Type: 'String', Description: 'S3 key for asset version "8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bf"' },
            AssetParameters8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bfArtifactHashF227ADD3: { Type: 'String', Description: 'Artifact hash for asset "8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bf"' }
        });
        // proxy asset params to nested stack
        assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketE8768F5CRef: { Ref: "AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketDE3B88D6" },
                referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey49DD83A2Ref: { Ref: "AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey3A62EFEA" }
            }
        }));
        // parent stack should have 2 assets
        test.deepEqual(assembly.getStackByName(parent.stackName).assets.length, 2);
        test.done();
    },
    'assets within nested stacks are proxied from the parent'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'ParentStack');
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        const asset = new s3_assets.Asset(nested, 'asset', {
            path: path.join(__dirname, 'asset-fixture.txt')
        });
        new core_1.CfnResource(nested, 'NestedResource', {
            type: 'Nested::Resource',
            properties: {
                AssetBucket: asset.s3BucketName,
                AssetKey: asset.s3ObjectKey
            }
        });
        // THEN
        const assembly = app.synth();
        const template = assert_1.SynthUtils.toCloudFormation(parent);
        // two sets of asset parameters: one for the nested stack itself and one as a proxy for the asset within the stack
        test.deepEqual(template.Parameters, {
            AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3BucketC188F637: { Type: 'String', Description: 'S3 bucket for asset "db01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281"' },
            AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3VersionKeyC7F4DBF2: { Type: 'String', Description: 'S3 key for asset version "db01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281"' },
            AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281ArtifactHash373B14D2: { Type: 'String', Description: 'Artifact hash for asset "db01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281"' },
            AssetParameters46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71S3Bucket3C4265E9: { Type: 'String', Description: 'S3 bucket for asset "46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71"' },
            AssetParameters46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71S3VersionKey8E981535: { Type: 'String', Description: 'S3 key for asset version "46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71"' },
            AssetParameters46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71ArtifactHash45A28583: { Type: 'String', Description: 'Artifact hash for asset "46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71"' }
        });
        // asset proxy parameters are passed to the nested stack
        assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetoParentStackAssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3Bucket82C55B96Ref: { Ref: "AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3BucketC188F637" },
                referencetoParentStackAssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3VersionKeyA43C3CC6Ref: { Ref: "AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3VersionKeyC7F4DBF2" },
            }
        }));
        // parent stack should have 2 assets
        test.deepEqual(assembly.getStackByName(parent.stackName).assets.length, 2);
        test.done();
    },
    'docker image assets are wired through the top-level stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'my-stack');
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        // WHEN
        const location = nested.addDockerImageAsset({
            directoryName: 'my-image',
            dockerBuildArgs: { key: 'value', boom: 'bam' },
            dockerBuildTarget: 'buildTarget',
            sourceHash: 'hash-of-source',
        });
        // use the asset, so the parameters will be wired.
        new sns.Topic(nested, 'MyTopic', {
            displayName: `image location is ${location.imageUri}`
        });
        // THEN
        const asm = app.synth();
        test.deepEqual(asm.getStackArtifact(parent.artifactId).assets, [
            {
                repositoryName: 'aws-cdk/assets',
                imageTag: 'hash-of-source',
                id: 'hash-of-source',
                packaging: 'container-image',
                path: 'my-image',
                sourceHash: 'hash-of-source',
                buildArgs: { key: 'value', boom: 'bam' },
                target: 'buildTarget'
            },
            {
                path: 'mystacknestedstackFAE12FB5.nested.template.json',
                id: 'fcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5',
                packaging: 'file',
                sourceHash: 'fcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5',
                s3BucketParameter: 'AssetParametersfcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5S3Bucket67A749F8',
                s3KeyParameter: 'AssetParametersfcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5S3VersionKeyE1E6A8D4',
                artifactHashParameter: 'AssetParametersfcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5ArtifactHash0AEDBE8A'
            }
        ]);
        test.done();
    },
    'metadata defined in nested stacks is reported at the parent stack level in the cloud assembly'(test) {
        // GIVEN
        const app = new core_1.App({ stackTraces: false });
        const parent = new core_1.Stack(app, 'parent');
        const child = new core_1.Stack(parent, 'child');
        const nested = new nested_stack_1.NestedStack(child, 'nested');
        const resource = new core_1.CfnResource(nested, 'resource', { type: 'foo' });
        // WHEN
        resource.node.addMetadata('foo', 'bar');
        // THEN: the first non-nested stack records the assembly metadata
        const asm = app.synth();
        test.deepEqual(asm.stacks.length, 2); // only one stack is defined as an artifact
        test.deepEqual(asm.getStackByName(parent.stackName).findMetadataByType('foo'), []);
        test.deepEqual(asm.getStackByName(child.stackName).findMetadataByType('foo'), [
            {
                path: '/parent/child/nested/resource',
                type: 'foo',
                data: 'bar'
            }
        ]);
        test.done();
    },
    'referencing attributes with period across stacks'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'nested');
        const consumed = new core_1.CfnResource(nested, 'resource-in-nested', { type: 'CONSUMED' });
        // WHEN
        new core_1.CfnResource(parent, 'resource-in-parent', {
            type: 'CONSUMER',
            properties: {
                ConsumedAttribute: consumed.getAtt('Consumed.Attribute')
            }
        });
        // THEN
        assert_1.expect(nested).toMatch({
            Resources: {
                resourceinnested: {
                    Type: "CONSUMED"
                }
            },
            Outputs: {
                nestedresourceinnested59B1F01CConsumedAttribute: {
                    Value: {
                        "Fn::GetAtt": [
                            "resourceinnested",
                            "Consumed.Attribute"
                        ]
                    }
                }
            }
        });
        assert_1.expect(parent).to(assert_1.haveResource('CONSUMER', {
            ConsumedAttribute: {
                "Fn::GetAtt": [
                    "nestedNestedStacknestedNestedStackResource3DD143BF",
                    "Outputs.nestedresourceinnested59B1F01CConsumedAttribute"
                ]
            }
        }));
        test.done();
    },
    'missing context in nested stack is reported if the context is not available'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'ParentStack', { env: { account: '1234account', region: 'us-east-44' } });
        const nestedStack = new nested_stack_1.NestedStack(stack, 'nested');
        const provider = 'dummyProvider';
        const expectedKey = core_1.ContextProvider.getKey(nestedStack, {
            provider
        }).key;
        // WHEN
        core_1.ContextProvider.getValue(nestedStack, {
            provider,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        });
        // THEN: missing context is reported in the cloud assembly
        const asm = app.synth();
        const missing = asm.manifest.missing;
        test.ok(missing && missing.find(m => {
            return (m.key === expectedKey);
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5uZXN0ZWQtc3RhY2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0Lm5lc3RlZC1zdGFjay50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQW1FO0FBQ25FLG9EQUFvRDtBQUNwRCx3Q0FBd0M7QUFDeEMsd0NBQWtHO0FBQ2xHLHlCQUF5QjtBQUV6Qiw2QkFBNkI7QUFDN0Isc0RBQWtEO0FBSWxELGlCQUFTO0lBQ1AsNEJBQTRCLENBQUMsSUFBVTtRQUNyQyxPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLDBCQUFXLENBQUMsU0FBZ0IsRUFBRSxNQUFNLENBQUMsRUFBRSxxREFBcUQsQ0FBQyxDQUFDO1FBQ3BILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx5Q0FBeUMsQ0FBQyxJQUFVO1FBQ2xELFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksZ0JBQVMsQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFMUMsT0FBTztRQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSwwQkFBVyxDQUFDLEdBQUcsRUFBRSxNQUFNLENBQUMsRUFBRSwwREFBMEQsQ0FBQyxDQUFDO1FBQzVHLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSwwQkFBVyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsRUFBRSwwREFBMEQsQ0FBQyxDQUFDO1FBQzdHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrRUFBa0UsQ0FBQyxJQUFVO1FBQzNFLFFBQVE7UUFDUixNQUFNLE1BQU0sR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTNCLE9BQU87UUFDUCxJQUFJLDBCQUFXLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ2xDLElBQUksMEJBQVcsQ0FBQyxJQUFJLGdCQUFTLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQzVELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1RUFBdUUsQ0FBQyxJQUFVO1FBQ2hGLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sV0FBVyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUNuRCxJQUFJLDBCQUFXLENBQUMsV0FBVyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRTdDLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFN0IsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDN0MsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHlFQUF5RSxDQUFDLElBQVU7UUFDbEYsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQzlDLE1BQU0sTUFBTSxHQUFHLElBQUksMEJBQVcsQ0FBQyxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDdkQsSUFBSSxrQkFBVyxDQUFDLE1BQU0sRUFBRSx1QkFBdUIsRUFBRSxFQUFFLElBQUksRUFBRSx1QkFBdUIsRUFBRSxDQUFDLENBQUM7UUFFcEYsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU3QixPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSx1QkFBdUIsQ0FBQyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDckksSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLEVBQUU7WUFDdkIsU0FBUyxFQUFFO2dCQUNULHFCQUFxQixFQUFFO29CQUNyQixJQUFJLEVBQUUsdUJBQXVCO2lCQUM5QjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHlEQUF5RCxDQUFDLElBQVU7UUFDbEUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQzlDLE1BQU0sTUFBTSxHQUFHLElBQUksMEJBQVcsQ0FBQyxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDdkQsSUFBSSxrQkFBVyxDQUFDLE1BQU0sRUFBRSx1QkFBdUIsRUFBRSxFQUFFLElBQUksRUFBRSx1QkFBdUIsRUFBRSxDQUFDLENBQUM7UUFFcEYsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU3QixPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQztnQkFDaEUsSUFBSSxFQUFFLHFEQUFxRDtnQkFDM0QsRUFBRSxFQUFFLGtFQUFrRTtnQkFDdEUsU0FBUyxFQUFFLE1BQU07Z0JBQ2pCLFVBQVUsRUFBRSxrRUFBa0U7Z0JBQzlFLGlCQUFpQixFQUFFLGlHQUFpRztnQkFDcEgsY0FBYyxFQUFFLHFHQUFxRztnQkFDckgscUJBQXFCLEVBQUUscUdBQXFHO2FBQzdILENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtEQUErRCxDQUFDLElBQVU7UUFDeEUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRTlDLE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLDBCQUFXLENBQUMsTUFBTSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3ZELElBQUksa0JBQVcsQ0FBQyxNQUFNLEVBQUUsdUJBQXVCLEVBQUUsRUFBRSxJQUFJLEVBQUUsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDO1FBRXBGLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFN0Isc0NBQXNDO1FBQ3RDLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFMUMsc0VBQXNFO1FBQ3RFLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLHFEQUFxRCxDQUFDLENBQUM7UUFFekcscUNBQXFDO1FBQ3JDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNsRixJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsRUFBRTtZQUN0RSxTQUFTLEVBQUUsRUFBRSxxQkFBcUIsRUFBRSxFQUFFLElBQUksRUFBRSx1QkFBdUIsRUFBRSxFQUFFO1NBQ3hFLENBQUMsQ0FBQztRQUVILHVHQUF1RztRQUN2RyxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3JCLFNBQVMsRUFBRTtnQkFDVCw0REFBNEQsRUFBRTtvQkFDNUQsSUFBSSxFQUFFLDRCQUE0QjtvQkFDbEMsVUFBVSxFQUFFO3dCQUNWLFdBQVcsRUFBRTs0QkFDWCxVQUFVLEVBQUU7Z0NBQ1YsRUFBRTtnQ0FDRjtvQ0FDRSxhQUFhO29DQUNiO3dDQUNFLEdBQUcsRUFBRSxhQUFhO3FDQUNuQjtvQ0FDRCxHQUFHO29DQUNIO3dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUNBQ3RCO29DQUNELEdBQUc7b0NBQ0g7d0NBQ0UsR0FBRyxFQUFFLGlHQUFpRztxQ0FDdkc7b0NBQ0QsR0FBRztvQ0FDSDt3Q0FDRSxZQUFZLEVBQUU7NENBQ1osQ0FBQzs0Q0FDRDtnREFDRSxXQUFXLEVBQUU7b0RBQ1gsSUFBSTtvREFDSjt3REFDRSxHQUFHLEVBQUUscUdBQXFHO3FEQUMzRztpREFDRjs2Q0FDRjt5Q0FDRjtxQ0FDRjtvQ0FDRDt3Q0FDRSxZQUFZLEVBQUU7NENBQ1osQ0FBQzs0Q0FDRDtnREFDRSxXQUFXLEVBQUU7b0RBQ1gsSUFBSTtvREFDSjt3REFDRSxHQUFHLEVBQUUscUdBQXFHO3FEQUMzRztpREFDRjs2Q0FDRjt5Q0FDRjtxQ0FDRjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLCtGQUErRixFQUFFO29CQUMvRixJQUFJLEVBQUUsUUFBUTtvQkFDZCxXQUFXLEVBQUUsMEZBQTBGO2lCQUN4RztnQkFDRCxtR0FBbUcsRUFBRTtvQkFDbkcsSUFBSSxFQUFFLFFBQVE7b0JBQ2QsV0FBVyxFQUFFLCtGQUErRjtpQkFDN0c7Z0JBQ0QsbUdBQW1HLEVBQUU7b0JBQ25HLElBQUksRUFBRSxRQUFRO29CQUNkLFdBQVcsRUFBRSw4RkFBOEY7aUJBQzVHO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsWUFBWSxDQUFDLElBQVU7UUFDckIsTUFBTSxhQUFjLFNBQVEsMEJBQVc7WUFHckMsWUFBWSxLQUFnQixFQUFFLEVBQVU7Z0JBQ3RDLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7Z0JBRWpCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQVksQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7Z0JBQ2xFLElBQUksQ0FBQyxZQUFZLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUN0QyxDQUFDO1NBQ0Y7UUFFRCxNQUFNLE1BQU0sR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzNCLE1BQU0sTUFBTSxHQUFHLElBQUksYUFBYSxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUVuRCxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxZQUFZLEtBQUssTUFBTSxDQUFDLENBQUM7UUFDeEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFLLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLE1BQU0sQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrRUFBK0UsQ0FBQyxJQUFVO1FBQ3hGLE1BQU0sYUFBYyxTQUFRLDBCQUFXO1lBQ3JDLFlBQVksS0FBZ0IsRUFBRSxFQUFVO2dCQUN0QyxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO2dCQUVqQixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFZLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO2dCQUNsRSxJQUFJLGtCQUFXLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtvQkFDaEMsSUFBSSxFQUFFLGNBQWM7b0JBQ3BCLFVBQVUsRUFBRTt3QkFDVixRQUFRLEVBQUUsS0FBSyxDQUFDLGFBQWE7cUJBQzlCO2lCQUNGLENBQUMsQ0FBQztZQUNMLENBQUM7U0FDRjtRQUVELE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLElBQUksYUFBYSxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUVwQyx3Q0FBd0M7UUFDeEMsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTdCLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDMUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQVksRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNwRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQscUdBQXFHLENBQUMsSUFBVTtRQUM5RyxPQUFPO1FBQ1AsTUFBTSxhQUFjLFNBQVEsMEJBQVc7WUFFckMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxrQkFBK0I7Z0JBQ3ZFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7Z0JBRWpCLElBQUksa0JBQVcsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO29CQUNoQyxJQUFJLEVBQUUsc0JBQXNCO29CQUM1QixVQUFVLEVBQUU7d0JBQ1YsZ0NBQWdDLEVBQUUsa0JBQWtCLENBQUMsR0FBRztxQkFDekQ7aUJBQ0YsQ0FBQyxDQUFDO2dCQUVILElBQUksa0JBQVcsQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFO29CQUNqQyxJQUFJLEVBQUUsaUJBQWlCO29CQUN2QixVQUFVLEVBQUU7d0JBQ1YsS0FBSyxFQUFFLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUM7d0JBQ3hDLEtBQUssRUFBRSxrQkFBa0IsQ0FBQyxHQUFHO3FCQUM5QjtpQkFDRixDQUFDLENBQUM7WUFDTCxDQUFDO1NBQ0Y7UUFFRCxNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sV0FBVyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUU3QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGtCQUFXLENBQUMsV0FBVyxFQUFFLGlCQUFpQixFQUFFLEVBQUUsSUFBSSxFQUFFLHVCQUF1QixFQUFFLENBQUMsQ0FBQztRQUVwRyxNQUFNLE1BQU0sR0FBRyxJQUFJLGFBQWEsQ0FBQyxXQUFXLEVBQUUsUUFBUSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRWxFLE9BQU87UUFDUCxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFWix5RkFBeUY7UUFDekYsZUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNyQixTQUFTLEVBQ1Q7Z0JBQ0UsUUFBUSxFQUNSO29CQUNFLElBQUksRUFBRSxzQkFBc0I7b0JBQzVCLFVBQVUsRUFDUixFQUFFLGdDQUFnQyxFQUFFLEVBQUUsR0FBRyxFQUFFLDRDQUE0QyxFQUFFLEVBQUU7aUJBQzlGO2dCQUNELFNBQVMsRUFDVDtvQkFDRSxJQUFJLEVBQUUsaUJBQWlCO29CQUN2QixVQUFVLEVBQ1Y7d0JBQ0UsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLDZDQUE2QyxFQUFFO3dCQUM3RCxLQUFLLEVBQUUsRUFBRSxHQUFHLEVBQUUsNENBQTRDLEVBQUU7cUJBQzdEO2lCQUNGO2FBQ0Y7WUFDRCxVQUFVLEVBQ1Y7Z0JBQ0UsMENBQTBDLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFO2dCQUM5RCwyQ0FBMkMsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUU7YUFDaEU7U0FDRixDQUFDLENBQUM7UUFFSCxpRUFBaUU7UUFDakUsZUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDRCQUE0QixFQUFFO1lBQ2hFLFVBQVUsRUFBRTtnQkFDViwwQ0FBMEMsRUFBRTtvQkFDMUMsR0FBRyxFQUFFLGdCQUFnQjtpQkFDdEI7Z0JBQ0QsMkNBQTJDLEVBQUU7b0JBQzNDLFlBQVksRUFBRTt3QkFDWixnQkFBZ0I7d0JBQ2hCLE1BQU07cUJBQ1A7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDRGQUE0RixDQUFDLElBQVU7UUFDckcsTUFBTSxhQUFjLFNBQVEsMEJBQVc7WUFHckMsWUFBWSxLQUFnQixFQUFFLEVBQVU7Z0JBQ3RDLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7Z0JBRWpCLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLGtCQUFXLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtvQkFDekQsSUFBSSxFQUFFLHNCQUFzQjtpQkFDN0IsQ0FBQyxDQUFDO1lBQ0wsQ0FBQztTQUNGO1FBRUQsTUFBTSxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLFdBQVcsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFN0MsTUFBTSxNQUFNLEdBQUcsSUFBSSxhQUFhLENBQUMsV0FBVyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXhELElBQUksa0JBQVcsQ0FBQyxXQUFXLEVBQUUseUJBQXlCLEVBQUU7WUFDdEQsSUFBSSxFQUFFLHVCQUF1QjtZQUM3QixVQUFVLEVBQUU7Z0JBQ1YsMEJBQTBCLEVBQUUsTUFBTSxDQUFDLGlCQUFpQixDQUFDLEdBQUc7YUFDekQ7U0FDRixDQUFDLENBQUM7UUFFSCx3Q0FBd0M7UUFDeEMsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRVoseUZBQXlGO1FBQ3pGLGVBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDckIsU0FBUyxFQUFFO2dCQUNULFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRSxzQkFBc0IsRUFBRTthQUMzQztZQUNELE9BQU8sRUFBRTtnQkFDUCwrQkFBK0IsRUFBRSxFQUFFLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxVQUFVLEVBQUUsRUFBRTthQUNoRTtTQUNGLENBQUMsQ0FBQztRQUVILGlFQUFpRTtRQUNqRSxlQUFNLENBQUMsV0FBVyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7WUFDM0QsMEJBQTBCLEVBQUU7Z0JBQzFCLFlBQVksRUFBRTtvQkFDWixvREFBb0Q7b0JBQ3BELHlDQUF5QztpQkFDMUM7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1GQUFtRixDQUFDLElBQVU7UUFDNUYsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sTUFBTSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLGlCQUFpQixHQUFHLElBQUksMEJBQVcsQ0FBQyxNQUFNLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUN2RSxNQUFNLGdCQUFnQixHQUFHLElBQUksa0JBQVcsQ0FBQyxNQUFNLEVBQUUsa0JBQWtCLEVBQUUsRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUU3RixPQUFPO1FBQ1AsSUFBSSxrQkFBVyxDQUFDLGlCQUFpQixFQUFFLHdCQUF3QixFQUFFO1lBQzNELElBQUksRUFBRSxrQkFBa0I7WUFDeEIsVUFBVSxFQUFFO2dCQUNWLFlBQVksRUFBRSxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDO2FBQ3JEO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU3Qix3Q0FBd0M7UUFDeEMsZUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNyQixTQUFTLEVBQUU7Z0JBQ1QsZ0JBQWdCLEVBQUUsRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFO2FBQ3pDO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLHdEQUF3RCxFQUFFO29CQUN4RCxLQUFLLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxrQkFBa0IsRUFBRSxhQUFhLENBQUMsRUFBRTtvQkFDNUQsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLGlFQUFpRSxFQUFFO2lCQUNwRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsZ0RBQWdEO1FBQ2hELGVBQU0sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNoQyxTQUFTLEVBQUU7Z0JBQ1Qsc0JBQXNCLEVBQUU7b0JBQ3RCLElBQUksRUFBRSxrQkFBa0I7b0JBQ3hCLFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUU7NEJBQ1osaUJBQWlCLEVBQUUsaUVBQWlFO3lCQUNyRjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgseURBQXlEO1FBQ3pELE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ2pFLE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ2pFLElBQUksQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDMUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHVHQUF1RyxDQUFDLElBQVU7UUFDaEgsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sTUFBTSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLGlCQUFpQixHQUFHLElBQUksMEJBQVcsQ0FBQyxNQUFNLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUN2RSxNQUFNLHFCQUFxQixHQUFHLElBQUksa0JBQVcsQ0FBQyxpQkFBaUIsRUFBRSx1QkFBdUIsRUFBRSxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUsQ0FBQyxDQUFDO1FBRWxILE9BQU87UUFDUCxJQUFJLGtCQUFXLENBQUMsTUFBTSxFQUFFLGtCQUFrQixFQUFFO1lBQzFDLElBQUksRUFBRSxjQUFjO1lBQ3BCLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUscUJBQXFCLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQzthQUMxRDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFN0IsbUdBQW1HO1FBQ25HLGVBQU0sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNoQyxTQUFTLEVBQUU7Z0JBQ1QscUJBQXFCLEVBQUU7b0JBQ3JCLElBQUksRUFBRSxZQUFZO2lCQUNuQjthQUNGO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLCtEQUErRCxFQUFFO29CQUMvRCxLQUFLLEVBQUU7d0JBQ0wsWUFBWSxFQUFFOzRCQUNaLHVCQUF1Qjs0QkFDdkIsYUFBYTt5QkFDZDtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsaURBQWlEO1FBQ2pELElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRTtZQUN6RSwyS0FBMkssRUFBRTtnQkFDM0ssS0FBSyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsMEVBQTBFLEVBQUUseUVBQXlFLENBQUMsRUFBRTtnQkFDaEwsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLG9MQUFvTCxFQUFFO2FBQ3ZNO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsbUZBQW1GO1FBQ25GLGVBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDckIsU0FBUyxFQUFFO2dCQUNULGdCQUFnQixFQUFFO29CQUNoQixJQUFJLEVBQUUsY0FBYztvQkFDcEIsVUFBVSxFQUFFO3dCQUNWLFlBQVksRUFBRTs0QkFDWixpQkFBaUIsRUFBRSxvTEFBb0w7eUJBQ3hNO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQzFDLE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ2pFLE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ2pFLElBQUksQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDMUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJGQUEyRixDQUFDLElBQVU7UUFDcEcsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sT0FBTyxHQUFHLElBQUksMEJBQVcsQ0FBQyxNQUFNLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDbkQsTUFBTSxPQUFPLEdBQUcsSUFBSSwwQkFBVyxDQUFDLE1BQU0sRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNuRCxNQUFNLFNBQVMsR0FBRyxJQUFJLGtCQUFXLENBQUMsT0FBTyxFQUFFLFdBQVcsRUFBRSxFQUFFLElBQUksRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBRS9FLE9BQU87UUFDUCxJQUFJLGtCQUFXLENBQUMsT0FBTyxFQUFFLFdBQVcsRUFBRTtZQUNwQyxJQUFJLEVBQUUsV0FBVztZQUNqQixVQUFVLEVBQUU7Z0JBQ1YsY0FBYyxFQUFFLFNBQVMsQ0FBQyxHQUFHO2FBQzlCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUVaLHlCQUF5QjtRQUN6QixlQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3RCLFNBQVMsRUFBRTtnQkFDVCxTQUFTLEVBQUU7b0JBQ1QsSUFBSSxFQUFFLFdBQVc7aUJBQ2xCO2FBQ0Y7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsaUNBQWlDLEVBQUU7b0JBQ2pDLEtBQUssRUFBRTt3QkFDTCxHQUFHLEVBQUUsV0FBVztxQkFDakI7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILHlCQUF5QjtRQUN6QixlQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3RCLFNBQVMsRUFBRTtnQkFDVCxTQUFTLEVBQUU7b0JBQ1QsSUFBSSxFQUFFLFdBQVc7b0JBQ2pCLFVBQVUsRUFBRTt3QkFDVixjQUFjLEVBQUU7NEJBQ2QsR0FBRyxFQUFFLCtHQUErRzt5QkFDckg7cUJBQ0Y7aUJBQ0Y7YUFDRjtZQUNELFVBQVUsRUFBRTtnQkFDViw2R0FBNkcsRUFBRTtvQkFDN0csSUFBSSxFQUFFLFFBQVE7aUJBQ2Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILFNBQVM7UUFDVCxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNEJBQTRCLEVBQUU7WUFDM0QsVUFBVSxFQUFFO2dCQUNWLDZHQUE2RyxFQUFFO29CQUM3RyxZQUFZLEVBQUU7d0JBQ1osc0RBQXNEO3dCQUN0RCwyQ0FBMkM7cUJBQzVDO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtRkFBbUYsQ0FBQyxJQUFVO1FBQzVGLFFBQVE7UUFDUixNQUFNLE1BQU0sR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzNCLE1BQU0sTUFBTSxHQUFHLElBQUksMEJBQVcsQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFFdEQsT0FBTztRQUNQLElBQUksa0JBQVcsQ0FBQyxNQUFNLEVBQUUsZ0JBQWdCLEVBQUU7WUFDeEMsSUFBSSxFQUFFLGtCQUFrQjtZQUN4QixVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsTUFBTSxDQUFDLE9BQU8sRUFBRTtTQUMxQyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO1lBQ2pELFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxjQUFjLEVBQUU7U0FDbkMsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQscUdBQXFHLENBQUMsSUFBVTtRQUM5RyxRQUFRO1FBQ1IsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMzQixNQUFNLE1BQU0sR0FBRyxJQUFJLDBCQUFXLENBQUMsTUFBTSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRXRELE9BQU87UUFDUCxJQUFJLGtCQUFXLENBQUMsTUFBTSxFQUFFLGdCQUFnQixFQUFFO1lBQ3hDLElBQUksRUFBRSxrQkFBa0I7WUFDeEIsVUFBVSxFQUFFLEVBQUUsYUFBYSxFQUFFLE1BQU0sQ0FBQyxPQUFPLEVBQUU7U0FDOUMsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtZQUNqRCxhQUFhLEVBQUUsRUFBRSxHQUFHLEVBQUUsOERBQThELEVBQUU7U0FDdkYsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsdUZBQXVGLENBQUMsSUFBVTtRQUNoRyxRQUFRO1FBQ1IsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMzQixNQUFNLE1BQU0sR0FBRyxJQUFJLDBCQUFXLENBQUMsTUFBTSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRXRELE9BQU87UUFDUCxJQUFJLGtCQUFXLENBQUMsTUFBTSxFQUFFLGdCQUFnQixFQUFFO1lBQ3hDLElBQUksRUFBRSxrQkFBa0I7WUFDeEIsVUFBVSxFQUFFLEVBQUUsV0FBVyxFQUFFLE1BQU0sQ0FBQyxTQUFTLEVBQUU7U0FDOUMsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtZQUNqRCxXQUFXLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7U0FDdkMsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsdUdBQXVHLENBQUMsSUFBVTtRQUNoSCxRQUFRO1FBQ1IsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMzQixNQUFNLE1BQU0sR0FBRyxJQUFJLDBCQUFXLENBQUMsTUFBTSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRXRELE9BQU87UUFDUCxJQUFJLGtCQUFXLENBQUMsTUFBTSxFQUFFLGdCQUFnQixFQUFFO1lBQ3hDLElBQUksRUFBRSxrQkFBa0I7WUFDeEIsVUFBVSxFQUFFLEVBQUUsZUFBZSxFQUFFLE1BQU0sQ0FBQyxTQUFTLEVBQUU7U0FDbEQsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtZQUNqRCxlQUFlLEVBQUU7Z0JBQ2YsWUFBWSxFQUFFO29CQUNaLENBQUM7b0JBQ0Q7d0JBQ0UsV0FBVyxFQUFFOzRCQUNYLEdBQUc7NEJBQ0g7Z0NBQ0UsR0FBRyxFQUFFLDhEQUE4RDs2QkFDcEU7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHVFQUF1RSxDQUFDLElBQVU7UUFDaEYsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxhQUFhLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRSxFQUFFLENBQUMsQ0FBQztRQUV4RyxPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSwwQkFBVyxDQUFDLE1BQU0sRUFBRSxhQUFhLENBQUMsQ0FBQztRQUV0RCxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUN2RCxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDN0MsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFCQUFxQixDQUFDLElBQVU7UUFDOUIsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXZDLE9BQU87UUFDUCxNQUFNLE9BQU8sR0FBRyxJQUFJLDBCQUFXLENBQUMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ25ELE1BQU0sT0FBTyxHQUFHLElBQUksMEJBQVcsQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFcEQsSUFBSSxrQkFBVyxDQUFDLE9BQU8sRUFBRSxXQUFXLEVBQUUsRUFBRSxJQUFJLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQztRQUMvRCxJQUFJLGtCQUFXLENBQUMsT0FBTyxFQUFFLFdBQVcsRUFBRSxFQUFFLElBQUksRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDO1FBRS9ELE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFN0IsaURBQWlEO1FBQ2pELGVBQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDdEIsU0FBUyxFQUFFO2dCQUNULFNBQVMsRUFBRSxFQUFFLElBQUksRUFBRSxhQUFhLEVBQUU7YUFDbkM7U0FDRixDQUFDLENBQUM7UUFFSCw0RUFBNEU7UUFDNUUsZUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUM7UUFDaEQsTUFBTSxlQUFlLEdBQUcsbUJBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUM3RCxJQUFJLENBQUMsU0FBUyxDQUFDLGVBQWUsQ0FBQyxVQUFVLEVBQUU7WUFDekMsa0hBQWtILEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFO1lBQ3RJLHNIQUFzSCxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRTtTQUMzSSxDQUFDLENBQUM7UUFFSCxpR0FBaUc7UUFDakcsb0VBQW9FO1FBQ3BFLE1BQU0sUUFBUSxHQUFHLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDckQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsVUFBVSxFQUFFO1lBQ2xDLCtGQUErRixFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxXQUFXLEVBQUUsd0ZBQXdGLEVBQUU7WUFDMU4sbUdBQW1HLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSw2RkFBNkYsRUFBRTtZQUNuTyxtR0FBbUcsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLDRGQUE0RixFQUFFO1lBQ2xPLCtGQUErRixFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxXQUFXLEVBQUUsd0ZBQXdGLEVBQUU7WUFDMU4sbUdBQW1HLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSw2RkFBNkYsRUFBRTtZQUNuTyxtR0FBbUcsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLDRGQUE0RixFQUFFO1NBQ25PLENBQUMsQ0FBQztRQUVILHFDQUFxQztRQUNyQyxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNEJBQTRCLEVBQUU7WUFDM0QsVUFBVSxFQUFFO2dCQUNWLGtIQUFrSCxFQUFFLEVBQUUsR0FBRyxFQUFFLGlHQUFpRyxFQUFFO2dCQUM5TixzSEFBc0gsRUFBRSxFQUFFLEdBQUcsRUFBRSxxR0FBcUcsRUFBRTthQUN2TztTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosb0NBQW9DO1FBQ3BDLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztRQUMzRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseURBQXlELENBQUMsSUFBVTtRQUNsRSxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLE1BQU0sR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDN0MsTUFBTSxNQUFNLEdBQUcsSUFBSSwwQkFBVyxDQUFDLE1BQU0sRUFBRSxhQUFhLENBQUMsQ0FBQztRQUV0RCxPQUFPO1FBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSxTQUFTLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxPQUFPLEVBQUU7WUFDakQsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLG1CQUFtQixDQUFDO1NBQ2hELENBQUMsQ0FBQztRQUVILElBQUksa0JBQVcsQ0FBQyxNQUFNLEVBQUUsZ0JBQWdCLEVBQUU7WUFDeEMsSUFBSSxFQUFFLGtCQUFrQjtZQUN4QixVQUFVLEVBQUU7Z0JBQ1YsV0FBVyxFQUFFLEtBQUssQ0FBQyxZQUFZO2dCQUMvQixRQUFRLEVBQUUsS0FBSyxDQUFDLFdBQVc7YUFDNUI7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzdCLE1BQU0sUUFBUSxHQUFHLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFckQsa0hBQWtIO1FBQ2xILElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLFVBQVUsRUFBRTtZQUNsQywrRkFBK0YsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLHdGQUF3RixFQUFFO1lBQzFOLG1HQUFtRyxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxXQUFXLEVBQUUsNkZBQTZGLEVBQUU7WUFDbk8sbUdBQW1HLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSw0RkFBNEYsRUFBRTtZQUNsTywrRkFBK0YsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLHdGQUF3RixFQUFFO1lBQzFOLG1HQUFtRyxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxXQUFXLEVBQUUsNkZBQTZGLEVBQUU7WUFDbk8sbUdBQW1HLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSw0RkFBNEYsRUFBRTtTQUNuTyxDQUFDLENBQUM7UUFFSCx3REFBd0Q7UUFDeEQsZUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDRCQUE0QixFQUFFO1lBQzNELFVBQVUsRUFBRTtnQkFDVix3SEFBd0gsRUFBRSxFQUFFLEdBQUcsRUFBRSxpR0FBaUcsRUFBRTtnQkFDcE8sNEhBQTRILEVBQUUsRUFBRSxHQUFHLEVBQUUscUdBQXFHLEVBQUU7YUFDN087U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLG9DQUFvQztRQUNwQyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDM0UsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJEQUEyRCxDQUFDLElBQVU7UUFDcEUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQzFDLE1BQU0sTUFBTSxHQUFHLElBQUksMEJBQVcsQ0FBQyxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFFdkQsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQztZQUMxQyxhQUFhLEVBQUUsVUFBVTtZQUN6QixlQUFlLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUU7WUFDOUMsaUJBQWlCLEVBQUUsYUFBYTtZQUNoQyxVQUFVLEVBQUUsZ0JBQWdCO1NBQzdCLENBQUMsQ0FBQztRQUVILGtEQUFrRDtRQUNsRCxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLFNBQVMsRUFBRTtZQUMvQixXQUFXLEVBQUUscUJBQXFCLFFBQVEsQ0FBQyxRQUFRLEVBQUU7U0FDdEQsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUN4QixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsTUFBTSxFQUFFO1lBQzdEO2dCQUNFLGNBQWMsRUFBRSxnQkFBZ0I7Z0JBQ2hDLFFBQVEsRUFBRSxnQkFBZ0I7Z0JBQzFCLEVBQUUsRUFBRSxnQkFBZ0I7Z0JBQ3BCLFNBQVMsRUFBRSxpQkFBaUI7Z0JBQzVCLElBQUksRUFBRSxVQUFVO2dCQUNoQixVQUFVLEVBQUUsZ0JBQWdCO2dCQUM1QixTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUU7Z0JBQ3hDLE1BQU0sRUFBRSxhQUFhO2FBQ3RCO1lBQ0Q7Z0JBQ0UsSUFBSSxFQUFFLGlEQUFpRDtnQkFDdkQsRUFBRSxFQUFFLGtFQUFrRTtnQkFDdEUsU0FBUyxFQUFFLE1BQU07Z0JBQ2pCLFVBQVUsRUFBRSxrRUFBa0U7Z0JBQzlFLGlCQUFpQixFQUFFLGlHQUFpRztnQkFDcEgsY0FBYyxFQUFFLHFHQUFxRztnQkFDckgscUJBQXFCLEVBQUUscUdBQXFHO2FBQzdIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtGQUErRixDQUFDLElBQVU7UUFDeEcsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxDQUFDLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDNUMsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN6QyxNQUFNLE1BQU0sR0FBRyxJQUFJLDBCQUFXLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQVcsQ0FBQyxNQUFNLEVBQUUsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFdEUsT0FBTztRQUNQLFFBQVEsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUV4QyxpRUFBaUU7UUFDakUsTUFBTSxHQUFHLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ3hCLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQywyQ0FBMkM7UUFDakYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNuRixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQzVFO2dCQUNFLElBQUksRUFBRSwrQkFBK0I7Z0JBQ3JDLElBQUksRUFBRSxLQUFLO2dCQUNYLElBQUksRUFBRSxLQUFLO2FBQ1o7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0RBQWtELENBQUMsSUFBVTtRQUMzRCxRQUFRO1FBQ1IsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMzQixNQUFNLE1BQU0sR0FBRyxJQUFJLDBCQUFXLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ2pELE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQVcsQ0FBQyxNQUFNLEVBQUUsb0JBQW9CLEVBQUUsRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQztRQUVyRixPQUFPO1FBQ1AsSUFBSSxrQkFBVyxDQUFDLE1BQU0sRUFBRSxvQkFBb0IsRUFBRTtZQUM1QyxJQUFJLEVBQUUsVUFBVTtZQUNoQixVQUFVLEVBQUU7Z0JBQ1YsaUJBQWlCLEVBQUUsUUFBUSxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQzthQUN6RDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3JCLFNBQVMsRUFBRTtnQkFDVCxnQkFBZ0IsRUFBRTtvQkFDaEIsSUFBSSxFQUFFLFVBQVU7aUJBQ2pCO2FBQ0Y7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsK0NBQStDLEVBQUU7b0JBQy9DLEtBQUssRUFBRTt3QkFDTCxZQUFZLEVBQUU7NEJBQ1osa0JBQWtCOzRCQUNsQixvQkFBb0I7eUJBQ3JCO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsVUFBVSxFQUFFO1lBQ3pDLGlCQUFpQixFQUFFO2dCQUNqQixZQUFZLEVBQUU7b0JBQ1osb0RBQW9EO29CQUNwRCx5REFBeUQ7aUJBQzFEO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2RUFBNkUsQ0FBQyxJQUFVO1FBQ3RGLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxhQUFhLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsYUFBYSxFQUFFLE1BQU0sRUFBRSxZQUFZLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDdkcsTUFBTSxXQUFXLEdBQUcsSUFBSSwwQkFBVyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUNyRCxNQUFNLFFBQVEsR0FBRyxlQUFlLENBQUM7UUFDakMsTUFBTSxXQUFXLEdBQUcsc0JBQWUsQ0FBQyxNQUFNLENBQUMsV0FBVyxFQUFFO1lBQ3RELFFBQVE7U0FDVCxDQUFDLENBQUMsR0FBRyxDQUFDO1FBRVAsT0FBTztRQUNQLHNCQUFlLENBQUMsUUFBUSxDQUFDLFdBQVcsRUFBRTtZQUNwQyxRQUFRO1lBQ1IsVUFBVSxFQUFFLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxTQUFTLENBQUM7U0FDOUMsQ0FBQyxDQUFDO1FBRUgsMERBQTBEO1FBQzFELE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUN4QixNQUFNLE9BQU8sR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQztRQUVyQyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQU8sSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ2xDLE9BQU8sQ0FBQyxDQUFDLENBQUMsR0FBRyxLQUFLLFdBQVcsQ0FBQyxDQUFDO1FBQ2pDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlLCBTeW50aFV0aWxzIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAqIGFzIHMzX2Fzc2V0cyBmcm9tICdAYXdzLWNkay9hd3MtczMtYXNzZXRzJztcbmltcG9ydCAqIGFzIHNucyBmcm9tICdAYXdzLWNkay9hd3Mtc25zJztcbmltcG9ydCB7IEFwcCwgQ2ZuUGFyYW1ldGVyLCBDZm5SZXNvdXJjZSwgQ29uc3RydWN0LCBDb250ZXh0UHJvdmlkZXIsIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IE5lc3RlZFN0YWNrIH0gZnJvbSAnLi4vbGliL25lc3RlZC1zdGFjayc7XG5cbi8vIHRzbGludDpkaXNhYmxlOm1heC1saW5lLWxlbmd0aFxuXG5leHBvcnQgPSB7XG4gICdmYWlscyBpZiBkZWZpbmVkIGFzIGEgcm9vdCcodGVzdDogVGVzdCkge1xuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgTmVzdGVkU3RhY2sodW5kZWZpbmVkIGFzIGFueSwgJ2Jvb20nKSwgL05lc3RlZCBzdGFja3MgY2Fubm90IGJlIGRlZmluZWQgYXMgYSByb290IGNvbnN0cnVjdC8pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdmYWlscyBpZiBkZWZpbmVkIHdpdGhvdXQgYSBwYXJlbnQgc3RhY2snKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBncm91cCA9IG5ldyBDb25zdHJ1Y3QoYXBwLCAnZ3JvdXAnKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgTmVzdGVkU3RhY2soYXBwLCAnYm9vbScpLCAvbXVzdCBiZSBkZWZpbmVkIHdpdGhpbiBzY29wZSBvZiBhbm90aGVyIG5vbi1uZXN0ZWQgc3RhY2svKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgTmVzdGVkU3RhY2soZ3JvdXAsICdiYW0nKSwgL211c3QgYmUgZGVmaW5lZCB3aXRoaW4gc2NvcGUgb2YgYW5vdGhlciBub24tbmVzdGVkIHN0YWNrLyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NhbiBiZSBkZWZpbmVkIGFzIGEgZGlyZWN0IGNoaWxkIG9yIGFuIGluZGlyZWN0IGNoaWxkIG9mIGEgU3RhY2snKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHBhcmVudCA9IG5ldyBTdGFjaygpO1xuXG4gICAgLy8gVEhFTlxuICAgIG5ldyBOZXN0ZWRTdGFjayhwYXJlbnQsICdkaXJlY3QnKTtcbiAgICBuZXcgTmVzdGVkU3RhY2sobmV3IENvbnN0cnVjdChwYXJlbnQsICdncm91cCcpLCAnaW5kaXJlY3QnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnbmVzdGVkIHN0YWNrIGlzIG5vdCBzeW50aGVzaXplZCBhcyBhIHN0YWNrIGFydGlmYWN0IGludG8gdGhlIGFzc2VtYmx5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgcGFyZW50U3RhY2sgPSBuZXcgU3RhY2soYXBwLCAncGFyZW50LXN0YWNrJyk7XG4gICAgbmV3IE5lc3RlZFN0YWNrKHBhcmVudFN0YWNrLCAnbmVzdGVkLXN0YWNrJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXNzZW1ibHkgPSBhcHAuc3ludGgoKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5hcnRpZmFjdHMubGVuZ3RoLCAyKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndGhlIHRlbXBsYXRlIG9mIHRoZSBuZXN0ZWQgc3RhY2sgaXMgc3ludGhlc2l6ZWQgaW50byB0aGUgY2xvdWQgYXNzZW1ibHknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBwYXJlbnQgPSBuZXcgU3RhY2soYXBwLCAncGFyZW50LXN0YWNrJyk7XG4gICAgY29uc3QgbmVzdGVkID0gbmV3IE5lc3RlZFN0YWNrKHBhcmVudCwgJ25lc3RlZC1zdGFjaycpO1xuICAgIG5ldyBDZm5SZXNvdXJjZShuZXN0ZWQsICdSZXNvdXJjZUluTmVzdGVkU3RhY2snLCB7IHR5cGU6ICdBV1M6OlJlc291cmNlOjpOZXN0ZWQnIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgY29uc3QgdGVtcGxhdGUgPSBKU09OLnBhcnNlKGZzLnJlYWRGaWxlU3luYyhwYXRoLmpvaW4oYXNzZW1ibHkuZGlyZWN0b3J5LCBgJHtuZXN0ZWQubm9kZS51bmlxdWVJZH0ubmVzdGVkLnRlbXBsYXRlLmpzb25gKSwgJ3V0Zi04JykpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHRlbXBsYXRlLCB7XG4gICAgICBSZXNvdXJjZXM6IHtcbiAgICAgICAgUmVzb3VyY2VJbk5lc3RlZFN0YWNrOiB7XG4gICAgICAgICAgVHlwZTogJ0FXUzo6UmVzb3VyY2U6Ok5lc3RlZCdcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdmaWxlIGFzc2V0IG1ldGFkYXRhIGlzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcGFyZW50IHN0YWNrJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgcGFyZW50ID0gbmV3IFN0YWNrKGFwcCwgJ3BhcmVudC1zdGFjaycpO1xuICAgIGNvbnN0IG5lc3RlZCA9IG5ldyBOZXN0ZWRTdGFjayhwYXJlbnQsICduZXN0ZWQtc3RhY2snKTtcbiAgICBuZXcgQ2ZuUmVzb3VyY2UobmVzdGVkLCAnUmVzb3VyY2VJbk5lc3RlZFN0YWNrJywgeyB0eXBlOiAnQVdTOjpSZXNvdXJjZTo6TmVzdGVkJyB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKGFzc2VtYmx5LmdldFN0YWNrQnlOYW1lKHBhcmVudC5zdGFja05hbWUpLmFzc2V0cywgW3tcbiAgICAgIHBhdGg6ICdwYXJlbnRzdGFja25lc3RlZHN0YWNrODQ0ODkyQzAubmVzdGVkLnRlbXBsYXRlLmpzb24nLFxuICAgICAgaWQ6ICdjNjM5YzBhNWU3MzIwNzU4YWEyMjU4OTY2OWVjZWJjOThmMTg1YjcxMTMwMGIwNzRmNTM5OThjOGY5YTQ1MDk2JyxcbiAgICAgIHBhY2thZ2luZzogJ2ZpbGUnLFxuICAgICAgc291cmNlSGFzaDogJ2M2MzljMGE1ZTczMjA3NThhYTIyNTg5NjY5ZWNlYmM5OGYxODViNzExMzAwYjA3NGY1Mzk5OGM4ZjlhNDUwOTYnLFxuICAgICAgczNCdWNrZXRQYXJhbWV0ZXI6ICdBc3NldFBhcmFtZXRlcnNjNjM5YzBhNWU3MzIwNzU4YWEyMjU4OTY2OWVjZWJjOThmMTg1YjcxMTMwMGIwNzRmNTM5OThjOGY5YTQ1MDk2UzNCdWNrZXREQThDMzM0NScsXG4gICAgICBzM0tleVBhcmFtZXRlcjogJ0Fzc2V0UGFyYW1ldGVyc2M2MzljMGE1ZTczMjA3NThhYTIyNTg5NjY5ZWNlYmM5OGYxODViNzExMzAwYjA3NGY1Mzk5OGM4ZjlhNDUwOTZTM1ZlcnNpb25LZXkwOUQwM0VFNicsXG4gICAgICBhcnRpZmFjdEhhc2hQYXJhbWV0ZXI6ICdBc3NldFBhcmFtZXRlcnNjNjM5YzBhNWU3MzIwNzU4YWEyMjU4OTY2OWVjZWJjOThmMTg1YjcxMTMwMGIwNzRmNTM5OThjOGY5YTQ1MDk2QXJ0aWZhY3RIYXNoOERFNDUwQzcnXG4gICAgfV0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdhd3M6OmNsb3VkZm9ybWF0aW9uOjpzdGFjayBpcyBzeW50aGVzaXplZCBpbiB0aGUgcGFyZW50IHNjb3BlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgcGFyZW50ID0gbmV3IFN0YWNrKGFwcCwgJ3BhcmVudC1zdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IG5lc3RlZCA9IG5ldyBOZXN0ZWRTdGFjayhwYXJlbnQsICduZXN0ZWQtc3RhY2snKTtcbiAgICBuZXcgQ2ZuUmVzb3VyY2UobmVzdGVkLCAnUmVzb3VyY2VJbk5lc3RlZFN0YWNrJywgeyB0eXBlOiAnQVdTOjpSZXNvdXJjZTo6TmVzdGVkJyB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuXG4gICAgLy8gYXNzZW1ibHkgaGFzIG9uZSBzdGFjayAodGhlIHBhcmVudClcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5zdGFja3MubGVuZ3RoLCAxKTtcblxuICAgIC8vIGJ1dCB0aGlzIHN0YWNrIGhhcyBhbiBhc3NldCB0aGF0IHBvaW50cyB0byB0aGUgc3ludGhlc2l6ZWQgdGVtcGxhdGVcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5zdGFja3NbMF0uYXNzZXRzWzBdLnBhdGgsICdwYXJlbnRzdGFja25lc3RlZHN0YWNrODQ0ODkyQzAubmVzdGVkLnRlbXBsYXRlLmpzb24nKTtcblxuICAgIC8vIHRoZSB0ZW1wbGF0ZSBpbmNsdWRlcyBvdXIgcmVzb3VyY2VcbiAgICBjb25zdCBmaWxlUGF0aCA9IHBhdGguam9pbihhc3NlbWJseS5kaXJlY3RvcnksIGFzc2VtYmx5LnN0YWNrc1swXS5hc3NldHNbMF0ucGF0aCk7XG4gICAgdGVzdC5kZWVwRXF1YWwoSlNPTi5wYXJzZShmcy5yZWFkRmlsZVN5bmMoZmlsZVBhdGgpLnRvU3RyaW5nKCd1dGYtOCcpKSwge1xuICAgICAgUmVzb3VyY2VzOiB7IFJlc291cmNlSW5OZXN0ZWRTdGFjazogeyBUeXBlOiAnQVdTOjpSZXNvdXJjZTo6TmVzdGVkJyB9IH1cbiAgICB9KTtcblxuICAgIC8vIHRoZSBwYXJlbnQgdGVtcGxhdGUgaW5jbHVkZXMgdGhlIHBhcmFtZXRlcnMgYW5kIHRoZSBuZXN0ZWQgc3RhY2sgcmVzb3VyY2Ugd2hpY2ggcG9pbnRzIHRvIHRoZSBzMyB1cmxcbiAgICBleHBlY3QocGFyZW50KS50b01hdGNoKHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBuZXN0ZWRzdGFja05lc3RlZFN0YWNrbmVzdGVkc3RhY2tOZXN0ZWRTdGFja1Jlc291cmNlNzFDREQyNDE6IHtcbiAgICAgICAgICBUeXBlOiBcIkFXUzo6Q2xvdWRGb3JtYXRpb246OlN0YWNrXCIsXG4gICAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgICAgVGVtcGxhdGVVUkw6IHtcbiAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICBcImh0dHBzOi8vczMuXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlJlZ2lvblwiXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXCIuXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXCIvXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJBc3NldFBhcmFtZXRlcnNjNjM5YzBhNWU3MzIwNzU4YWEyMjU4OTY2OWVjZWJjOThmMTg1YjcxMTMwMGIwNzRmNTM5OThjOGY5YTQ1MDk2UzNCdWNrZXREQThDMzM0NVwiXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXCIvXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFwiRm46OlNlbGVjdFwiOiBbXG4gICAgICAgICAgICAgICAgICAgICAgMCxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBcIkZuOjpTcGxpdFwiOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgIFwifHxcIixcbiAgICAgICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJBc3NldFBhcmFtZXRlcnNjNjM5YzBhNWU3MzIwNzU4YWEyMjU4OTY2OWVjZWJjOThmMTg1YjcxMTMwMGIwNzRmNTM5OThjOGY5YTQ1MDk2UzNWZXJzaW9uS2V5MDlEMDNFRTZcIlxuICAgICAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBcIkZuOjpTZWxlY3RcIjogW1xuICAgICAgICAgICAgICAgICAgICAgIDEsXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgXCJGbjo6U3BsaXRcIjogW1xuICAgICAgICAgICAgICAgICAgICAgICAgICBcInx8XCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQXNzZXRQYXJhbWV0ZXJzYzYzOWMwYTVlNzMyMDc1OGFhMjI1ODk2NjllY2ViYzk4ZjE4NWI3MTEzMDBiMDc0ZjUzOTk4YzhmOWE0NTA5NlMzVmVyc2lvbktleTA5RDAzRUU2XCJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICAgQXNzZXRQYXJhbWV0ZXJzYzYzOWMwYTVlNzMyMDc1OGFhMjI1ODk2NjllY2ViYzk4ZjE4NWI3MTEzMDBiMDc0ZjUzOTk4YzhmOWE0NTA5NlMzQnVja2V0REE4QzMzNDU6IHtcbiAgICAgICAgICBUeXBlOiBcIlN0cmluZ1wiLFxuICAgICAgICAgIERlc2NyaXB0aW9uOiBcIlMzIGJ1Y2tldCBmb3IgYXNzZXQgXFxcImM2MzljMGE1ZTczMjA3NThhYTIyNTg5NjY5ZWNlYmM5OGYxODViNzExMzAwYjA3NGY1Mzk5OGM4ZjlhNDUwOTZcXFwiXCJcbiAgICAgICAgfSxcbiAgICAgICAgQXNzZXRQYXJhbWV0ZXJzYzYzOWMwYTVlNzMyMDc1OGFhMjI1ODk2NjllY2ViYzk4ZjE4NWI3MTEzMDBiMDc0ZjUzOTk4YzhmOWE0NTA5NlMzVmVyc2lvbktleTA5RDAzRUU2OiB7XG4gICAgICAgICAgVHlwZTogXCJTdHJpbmdcIixcbiAgICAgICAgICBEZXNjcmlwdGlvbjogXCJTMyBrZXkgZm9yIGFzc2V0IHZlcnNpb24gXFxcImM2MzljMGE1ZTczMjA3NThhYTIyNTg5NjY5ZWNlYmM5OGYxODViNzExMzAwYjA3NGY1Mzk5OGM4ZjlhNDUwOTZcXFwiXCJcbiAgICAgICAgfSxcbiAgICAgICAgQXNzZXRQYXJhbWV0ZXJzYzYzOWMwYTVlNzMyMDc1OGFhMjI1ODk2NjllY2ViYzk4ZjE4NWI3MTEzMDBiMDc0ZjUzOTk4YzhmOWE0NTA5NkFydGlmYWN0SGFzaDhERTQ1MEM3OiB7XG4gICAgICAgICAgVHlwZTogXCJTdHJpbmdcIixcbiAgICAgICAgICBEZXNjcmlwdGlvbjogXCJBcnRpZmFjdCBoYXNoIGZvciBhc3NldCBcXFwiYzYzOWMwYTVlNzMyMDc1OGFhMjI1ODk2NjllY2ViYzk4ZjE4NWI3MTEzMDBiMDc0ZjUzOTk4YzhmOWE0NTA5NlxcXCJcIlxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrLm9mKCknKHRlc3Q6IFRlc3QpIHtcbiAgICBjbGFzcyBNeU5lc3RlZFN0YWNrIGV4dGVuZHMgTmVzdGVkU3RhY2sge1xuICAgICAgcHVibGljIHJlYWRvbmx5IHN0YWNrT2ZDaGlsZDogU3RhY2s7XG5cbiAgICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgICAgICBjb25zdCBwYXJhbSA9IG5ldyBDZm5QYXJhbWV0ZXIodGhpcywgJ3BhcmFtJywgeyB0eXBlOiAnU3RyaW5nJyB9KTtcbiAgICAgICAgdGhpcy5zdGFja09mQ2hpbGQgPSBTdGFjay5vZihwYXJhbSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgcGFyZW50ID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgbmVzdGVkID0gbmV3IE15TmVzdGVkU3RhY2socGFyZW50LCAnbmVzdGVkJyk7XG5cbiAgICB0ZXN0Lm9rKG5lc3RlZC5zdGFja09mQ2hpbGQgPT09IG5lc3RlZCk7XG4gICAgdGVzdC5vayhTdGFjay5vZihuZXN0ZWQpID09PSBuZXN0ZWQpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdyZWZlcmVuY2VzIHdpdGhpbiB0aGUgbmVzdGVkIHN0YWNrIGFyZSBub3QgcmVwb3J0ZWQgYXMgY3Jvc3Mgc3RhY2sgcmVmZXJlbmNlcycodGVzdDogVGVzdCkge1xuICAgIGNsYXNzIE15TmVzdGVkU3RhY2sgZXh0ZW5kcyBOZXN0ZWRTdGFjayB7XG4gICAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAgICAgY29uc3QgcGFyYW0gPSBuZXcgQ2ZuUGFyYW1ldGVyKHRoaXMsICdwYXJhbScsIHsgdHlwZTogJ1N0cmluZycgfSk7XG4gICAgICAgIG5ldyBDZm5SZXNvdXJjZSh0aGlzLCAncmVzb3VyY2UnLCB7XG4gICAgICAgICAgdHlwZTogJ015OjpSZXNvdXJjZScsXG4gICAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgICAgU29tZVByb3A6IHBhcmFtLnZhbHVlQXNTdHJpbmdcbiAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cblxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBwYXJlbnQgPSBuZXcgU3RhY2soYXBwLCAncGFyZW50Jyk7XG5cbiAgICBuZXcgTXlOZXN0ZWRTdGFjayhwYXJlbnQsICduZXN0ZWQnKTtcblxuICAgIC8vIHJlZmVyZW5jZXMgYXJlIGFkZGVkIGR1cmluZyBcInByZXBhcmVcIlxuICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5zdGFja3MubGVuZ3RoLCAxKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5zdGFja3NbMF0uZGVwZW5kZW5jaWVzLCBbXSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3JlZmVyZW5jZXMgdG8gYSByZXNvdXJjZSBmcm9tIHRoZSBwYXJlbnQgc3RhY2sgaW4gYSBuZXN0ZWQgc3RhY2sgaXMgdHJhbnNsYXRlZCBpbnRvIGEgY2ZuIHBhcmFtZXRlcicodGVzdDogVGVzdCkge1xuICAgIC8vIFdIRU5cbiAgICBjbGFzcyBNeU5lc3RlZFN0YWNrIGV4dGVuZHMgTmVzdGVkU3RhY2sge1xuXG4gICAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCByZXNvdXJjZUZyb21QYXJlbnQ6IENmblJlc291cmNlKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAgICAgbmV3IENmblJlc291cmNlKHRoaXMsICdyZXNvdXJjZScsIHtcbiAgICAgICAgICB0eXBlOiAnQVdTOjpDaGlsZDo6UmVzb3VyY2UnLFxuICAgICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIFJlZmVyZW5jZVRvUmVzb3VyY2VJblBhcmVudFN0YWNrOiByZXNvdXJjZUZyb21QYXJlbnQucmVmXG4gICAgICAgICAgfVxuICAgICAgICB9KTtcblxuICAgICAgICBuZXcgQ2ZuUmVzb3VyY2UodGhpcywgJ3Jlc291cmNlMicsIHtcbiAgICAgICAgICB0eXBlOiAnTXk6OlJlc291cmNlOjoyJyxcbiAgICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICBQcm9wMTogcmVzb3VyY2VGcm9tUGFyZW50LmdldEF0dCgnQXR0cicpLFxuICAgICAgICAgICAgUHJvcDI6IHJlc291cmNlRnJvbVBhcmVudC5yZWYsXG4gICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgcGFyZW50U3RhY2sgPSBuZXcgU3RhY2soYXBwLCAncGFyZW50Jyk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5SZXNvdXJjZShwYXJlbnRTdGFjaywgJ3BhcmVudC1yZXNvdXJjZScsIHsgdHlwZTogJ0FXUzo6UGFyZW50OjpSZXNvdXJjZScgfSk7XG5cbiAgICBjb25zdCBuZXN0ZWQgPSBuZXcgTXlOZXN0ZWRTdGFjayhwYXJlbnRTdGFjaywgJ25lc3RlZCcsIHJlc291cmNlKTtcblxuICAgIC8vIFRIRU5cbiAgICBhcHAuc3ludGgoKTtcblxuICAgIC8vIG5lc3RlZCB0ZW1wbGF0ZSBzaG91bGQgdXNlIGEgcGFyYW1ldGVyIHRvIHJlZmVyZW5jZSB0aGUgcmVzb3VyY2UgZnJvbSB0aGUgcGFyZW50IHN0YWNrXG4gICAgZXhwZWN0KG5lc3RlZCkudG9NYXRjaCh7XG4gICAgICBSZXNvdXJjZXM6XG4gICAgICB7XG4gICAgICAgIHJlc291cmNlOlxuICAgICAgICB7XG4gICAgICAgICAgVHlwZTogJ0FXUzo6Q2hpbGQ6OlJlc291cmNlJyxcbiAgICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgICAgeyBSZWZlcmVuY2VUb1Jlc291cmNlSW5QYXJlbnRTdGFjazogeyBSZWY6ICdyZWZlcmVuY2V0b3BhcmVudHBhcmVudHJlc291cmNlRDU2RUE4RjdSZWYnIH0gfVxuICAgICAgICB9LFxuICAgICAgICByZXNvdXJjZTI6XG4gICAgICAgIHtcbiAgICAgICAgICBUeXBlOiAnTXk6OlJlc291cmNlOjoyJyxcbiAgICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFByb3AxOiB7IFJlZjogJ3JlZmVyZW5jZXRvcGFyZW50cGFyZW50cmVzb3VyY2VENTZFQThGN0F0dHInIH0sXG4gICAgICAgICAgICBQcm9wMjogeyBSZWY6ICdyZWZlcmVuY2V0b3BhcmVudHBhcmVudHJlc291cmNlRDU2RUE4RjdSZWYnIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBQYXJhbWV0ZXJzOlxuICAgICAge1xuICAgICAgICByZWZlcmVuY2V0b3BhcmVudHBhcmVudHJlc291cmNlRDU2RUE4RjdSZWY6IHsgVHlwZTogJ1N0cmluZycgfSxcbiAgICAgICAgcmVmZXJlbmNldG9wYXJlbnRwYXJlbnRyZXNvdXJjZUQ1NkVBOEY3QXR0cjogeyBUeXBlOiAnU3RyaW5nJyB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICAvLyBwYXJlbnQgdGVtcGxhdGUgc2hvdWxkIHBhc3MgaW4gdGhlIHZhbHVlIHRocm91Z2ggdGhlIHBhcmFtZXRlclxuICAgIGV4cGVjdChwYXJlbnRTdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNsb3VkRm9ybWF0aW9uOjpTdGFjaycsIHtcbiAgICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICAgcmVmZXJlbmNldG9wYXJlbnRwYXJlbnRyZXNvdXJjZUQ1NkVBOEY3UmVmOiB7XG4gICAgICAgICAgUmVmOiBcInBhcmVudHJlc291cmNlXCJcbiAgICAgICAgfSxcbiAgICAgICAgcmVmZXJlbmNldG9wYXJlbnRwYXJlbnRyZXNvdXJjZUQ1NkVBOEY3QXR0cjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcInBhcmVudHJlc291cmNlXCIsXG4gICAgICAgICAgICBcIkF0dHJcIlxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAncmVmZXJlbmNlcyB0byBhIHJlc291cmNlIGluIHRoZSBuZXN0ZWQgc3RhY2sgaW4gdGhlIHBhcmVudCBpcyB0cmFuc2xhdGVkIGludG8gYSBjZm4gb3V0cHV0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY2xhc3MgTXlOZXN0ZWRTdGFjayBleHRlbmRzIE5lc3RlZFN0YWNrIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSByZXNvdXJjZUZyb21DaGlsZDogQ2ZuUmVzb3VyY2U7XG5cbiAgICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgICAgICB0aGlzLnJlc291cmNlRnJvbUNoaWxkID0gbmV3IENmblJlc291cmNlKHRoaXMsICdyZXNvdXJjZScsIHtcbiAgICAgICAgICB0eXBlOiAnQVdTOjpDaGlsZDo6UmVzb3VyY2UnLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgcGFyZW50U3RhY2sgPSBuZXcgU3RhY2soYXBwLCAncGFyZW50Jyk7XG5cbiAgICBjb25zdCBuZXN0ZWQgPSBuZXcgTXlOZXN0ZWRTdGFjayhwYXJlbnRTdGFjaywgJ25lc3RlZCcpO1xuXG4gICAgbmV3IENmblJlc291cmNlKHBhcmVudFN0YWNrLCAnYW5vdGhlci1wYXJlbnQtcmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpQYXJlbnQ6OlJlc291cmNlJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgUmVmVG9SZXNvdXJjZUluTmVzdGVkU3RhY2s6IG5lc3RlZC5yZXNvdXJjZUZyb21DaGlsZC5yZWZcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIC8vIHJlZmVyZW5jZXMgYXJlIGFkZGVkIGR1cmluZyBcInByZXBhcmVcIlxuICAgIGFwcC5zeW50aCgpO1xuXG4gICAgLy8gbmVzdGVkIHRlbXBsYXRlIHNob3VsZCB1c2UgYSBwYXJhbWV0ZXIgdG8gcmVmZXJlbmNlIHRoZSByZXNvdXJjZSBmcm9tIHRoZSBwYXJlbnQgc3RhY2tcbiAgICBleHBlY3QobmVzdGVkKS50b01hdGNoKHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICByZXNvdXJjZTogeyBUeXBlOiAnQVdTOjpDaGlsZDo6UmVzb3VyY2UnIH1cbiAgICAgIH0sXG4gICAgICBPdXRwdXRzOiB7XG4gICAgICAgIHBhcmVudG5lc3RlZHJlc291cmNlNEQ2ODA2NzdSZWY6IHsgVmFsdWU6IHsgUmVmOiAncmVzb3VyY2UnIH0gfVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gcGFyZW50IHRlbXBsYXRlIHNob3VsZCBwYXNzIGluIHRoZSB2YWx1ZSB0aHJvdWdoIHRoZSBwYXJhbWV0ZXJcbiAgICBleHBlY3QocGFyZW50U3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpQYXJlbnQ6OlJlc291cmNlJywge1xuICAgICAgUmVmVG9SZXNvdXJjZUluTmVzdGVkU3RhY2s6IHtcbiAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICBcIm5lc3RlZE5lc3RlZFN0YWNrbmVzdGVkTmVzdGVkU3RhY2tSZXNvdXJjZTNERDE0M0JGXCIsXG4gICAgICAgICAgXCJPdXRwdXRzLnBhcmVudG5lc3RlZHJlc291cmNlNEQ2ODA2NzdSZWZcIlxuICAgICAgICBdXG4gICAgICB9XG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ25lc3RlZCBzdGFjayByZWZlcmVuY2VzIGEgcmVzb3VyY2UgZnJvbSBhbm90aGVyIG5vbi1uZXN0ZWQgc3RhY2sgKG5vdCB0aGUgcGFyZW50KScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdTdGFjazEnKTtcbiAgICBjb25zdCBzdGFjazIgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2syJyk7XG4gICAgY29uc3QgbmVzdGVkVW5kZXJTdGFjazEgPSBuZXcgTmVzdGVkU3RhY2soc3RhY2sxLCAnTmVzdGVkVW5kZXJTdGFjazEnKTtcbiAgICBjb25zdCByZXNvdXJjZUluU3RhY2syID0gbmV3IENmblJlc291cmNlKHN0YWNrMiwgJ1Jlc291cmNlSW5TdGFjazInLCB7IHR5cGU6ICdNeVJlc291cmNlJyB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgQ2ZuUmVzb3VyY2UobmVzdGVkVW5kZXJTdGFjazEsICdSZXNvdXJjZUluTmVzdGVkU3RhY2sxJywge1xuICAgICAgdHlwZTogJ05lc3RlZDo6UmVzb3VyY2UnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBSZWZUb1NpYmxpbmc6IHJlc291cmNlSW5TdGFjazIuZ2V0QXR0KCdNeUF0dHJpYnV0ZScpXG4gICAgICB9XG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgY29uc3QgYXNzZW1ibHkgPSBhcHAuc3ludGgoKTtcblxuICAgIC8vIHByb2R1Y2luZyBzdGFjayBzaG91bGQgaGF2ZSBhbiBleHBvcnRcbiAgICBleHBlY3Qoc3RhY2syKS50b01hdGNoKHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBSZXNvdXJjZUluU3RhY2syOiB7IFR5cGU6IFwiTXlSZXNvdXJjZVwiIH1cbiAgICAgIH0sXG4gICAgICBPdXRwdXRzOiB7XG4gICAgICAgIEV4cG9ydHNPdXRwdXRGbkdldEF0dFJlc291cmNlSW5TdGFjazJNeUF0dHJpYnV0ZUMxNUYxMDA5OiB7XG4gICAgICAgICAgVmFsdWU6IHsgXCJGbjo6R2V0QXR0XCI6IFtcIlJlc291cmNlSW5TdGFjazJcIiwgXCJNeUF0dHJpYnV0ZVwiXSB9LFxuICAgICAgICAgIEV4cG9ydDogeyBOYW1lOiBcIlN0YWNrMjpFeHBvcnRzT3V0cHV0Rm5HZXRBdHRSZXNvdXJjZUluU3RhY2syTXlBdHRyaWJ1dGVDMTVGMTAwOVwiIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gbmVzdGVkIHN0YWNrIHVzZXMgRm46OkltcG9ydFZhbHVlIGxpa2Ugbm9ybWFsXG4gICAgZXhwZWN0KG5lc3RlZFVuZGVyU3RhY2sxKS50b01hdGNoKHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBSZXNvdXJjZUluTmVzdGVkU3RhY2sxOiB7XG4gICAgICAgICAgVHlwZTogXCJOZXN0ZWQ6OlJlc291cmNlXCIsXG4gICAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgICAgUmVmVG9TaWJsaW5nOiB7XG4gICAgICAgICAgICAgIFwiRm46OkltcG9ydFZhbHVlXCI6IFwiU3RhY2syOkV4cG9ydHNPdXRwdXRGbkdldEF0dFJlc291cmNlSW5TdGFjazJNeUF0dHJpYnV0ZUMxNUYxMDA5XCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIC8vIHZlcmlmeSBhIGRlcGVkZW5jeSB3YXMgZXN0YWJsaXNoZWQgYmV0d2VlbiB0aGUgcGFyZW50c1xuICAgIGNvbnN0IHN0YWNrMUFydGlmYWN0ID0gYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUoc3RhY2sxLnN0YWNrTmFtZSk7XG4gICAgY29uc3Qgc3RhY2syQXJ0aWZhY3QgPSBhc3NlbWJseS5nZXRTdGFja0J5TmFtZShzdGFjazIuc3RhY2tOYW1lKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazFBcnRpZmFjdC5kZXBlbmRlbmNpZXMubGVuZ3RoLCAxKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazJBcnRpZmFjdC5kZXBlbmRlbmNpZXMubGVuZ3RoLCAwKTtcbiAgICB0ZXN0LnNhbWUoc3RhY2sxQXJ0aWZhY3QuZGVwZW5kZW5jaWVzWzBdLCBzdGFjazJBcnRpZmFjdCk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Fub3RoZXIgbm9uLW5lc3RlZCBzdGFjayB0YWtlcyBhIHJlZmVyZW5jZSBvbiBhIHJlc291cmNlIHdpdGhpbiB0aGUgbmVzdGVkIHN0YWNrICh0aGUgcGFyZW50IGV4cG9ydHMpJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMScpO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhhcHAsICdTdGFjazInKTtcbiAgICBjb25zdCBuZXN0ZWRVbmRlclN0YWNrMSA9IG5ldyBOZXN0ZWRTdGFjayhzdGFjazEsICdOZXN0ZWRVbmRlclN0YWNrMScpO1xuICAgIGNvbnN0IHJlc291cmNlSW5OZXN0ZWRTdGFjayA9IG5ldyBDZm5SZXNvdXJjZShuZXN0ZWRVbmRlclN0YWNrMSwgJ1Jlc291cmNlSW5OZXN0ZWRTdGFjaycsIHsgdHlwZTogJ015UmVzb3VyY2UnIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDZm5SZXNvdXJjZShzdGFjazIsICdSZXNvdXJjZUluU3RhY2syJywge1xuICAgICAgdHlwZTogJ0p1c3RSZXNvdXJjZScsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIFJlZlRvU2libGluZzogcmVzb3VyY2VJbk5lc3RlZFN0YWNrLmdldEF0dCgnTXlBdHRyaWJ1dGUnKVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG5cbiAgICAvLyBuZXN0ZWQgc3RhY2sgc2hvdWxkIG91dHB1dCB0aGlzIHZhbHVlIGFzIGlmIGl0IHdhcyByZWZlcmVuY2VkIGJ5IHRoZSBwYXJlbnQgKHdpdGhvdXQgdGhlIGV4cG9ydClcbiAgICBleHBlY3QobmVzdGVkVW5kZXJTdGFjazEpLnRvTWF0Y2goe1xuICAgICAgUmVzb3VyY2VzOiB7XG4gICAgICAgIFJlc291cmNlSW5OZXN0ZWRTdGFjazoge1xuICAgICAgICAgIFR5cGU6IFwiTXlSZXNvdXJjZVwiXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBPdXRwdXRzOiB7XG4gICAgICAgIFN0YWNrMU5lc3RlZFVuZGVyU3RhY2sxUmVzb3VyY2VJbk5lc3RlZFN0YWNrNkVFOURDRDJNeUF0dHJpYnV0ZToge1xuICAgICAgICAgIFZhbHVlOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcIlJlc291cmNlSW5OZXN0ZWRTdGFja1wiLFxuICAgICAgICAgICAgICBcIk15QXR0cmlidXRlXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIC8vIHBhcmVudCBzdGFjayAoc3RhY2sxKSBzaG91bGQgZXhwb3J0IHRoaXMgdmFsdWVcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5nZXRTdGFja0J5TmFtZShzdGFjazEuc3RhY2tOYW1lKS50ZW1wbGF0ZS5PdXRwdXRzLCB7XG4gICAgICBFeHBvcnRzT3V0cHV0Rm5HZXRBdHROZXN0ZWRVbmRlclN0YWNrMU5lc3RlZFN0YWNrTmVzdGVkVW5kZXJTdGFjazFOZXN0ZWRTdGFja1Jlc291cmNlRjYxNjMwNUJPdXRwdXRzU3RhY2sxTmVzdGVkVW5kZXJTdGFjazFSZXNvdXJjZUluTmVzdGVkU3RhY2s2RUU5RENEMk15QXR0cmlidXRlNTY0RUVDRjM6IHtcbiAgICAgICAgVmFsdWU6IHsgJ0ZuOjpHZXRBdHQnOiBbJ05lc3RlZFVuZGVyU3RhY2sxTmVzdGVkU3RhY2tOZXN0ZWRVbmRlclN0YWNrMU5lc3RlZFN0YWNrUmVzb3VyY2VGNjE2MzA1QicsICdPdXRwdXRzLlN0YWNrMU5lc3RlZFVuZGVyU3RhY2sxUmVzb3VyY2VJbk5lc3RlZFN0YWNrNkVFOURDRDJNeUF0dHJpYnV0ZSddIH0sXG4gICAgICAgIEV4cG9ydDogeyBOYW1lOiAnU3RhY2sxOkV4cG9ydHNPdXRwdXRGbkdldEF0dE5lc3RlZFVuZGVyU3RhY2sxTmVzdGVkU3RhY2tOZXN0ZWRVbmRlclN0YWNrMU5lc3RlZFN0YWNrUmVzb3VyY2VGNjE2MzA1Qk91dHB1dHNTdGFjazFOZXN0ZWRVbmRlclN0YWNrMVJlc291cmNlSW5OZXN0ZWRTdGFjazZFRTlEQ0QyTXlBdHRyaWJ1dGU1NjRFRUNGMycgfVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gY29uc3VtaW5nIHN0YWNrIHNob3VsZCB1c2UgSW1wb3J0VmFsdWUgdG8gaW1wb3J0IHRoZSB2YWx1ZSBmcm9tIHRoZSBwYXJlbnQgc3RhY2tcbiAgICBleHBlY3Qoc3RhY2syKS50b01hdGNoKHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBSZXNvdXJjZUluU3RhY2syOiB7XG4gICAgICAgICAgVHlwZTogXCJKdXN0UmVzb3VyY2VcIixcbiAgICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICBSZWZUb1NpYmxpbmc6IHtcbiAgICAgICAgICAgICAgXCJGbjo6SW1wb3J0VmFsdWVcIjogXCJTdGFjazE6RXhwb3J0c091dHB1dEZuR2V0QXR0TmVzdGVkVW5kZXJTdGFjazFOZXN0ZWRTdGFja05lc3RlZFVuZGVyU3RhY2sxTmVzdGVkU3RhY2tSZXNvdXJjZUY2MTYzMDVCT3V0cHV0c1N0YWNrMU5lc3RlZFVuZGVyU3RhY2sxUmVzb3VyY2VJbk5lc3RlZFN0YWNrNkVFOURDRDJNeUF0dHJpYnV0ZTU2NEVFQ0YzXCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKGFzc2VtYmx5LnN0YWNrcy5sZW5ndGgsIDIpO1xuICAgIGNvbnN0IHN0YWNrMUFydGlmYWN0ID0gYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUoc3RhY2sxLnN0YWNrTmFtZSk7XG4gICAgY29uc3Qgc3RhY2syQXJ0aWZhY3QgPSBhc3NlbWJseS5nZXRTdGFja0J5TmFtZShzdGFjazIuc3RhY2tOYW1lKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazFBcnRpZmFjdC5kZXBlbmRlbmNpZXMubGVuZ3RoLCAwKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazJBcnRpZmFjdC5kZXBlbmRlbmNpZXMubGVuZ3RoLCAxKTtcbiAgICB0ZXN0LnNhbWUoc3RhY2syQXJ0aWZhY3QuZGVwZW5kZW5jaWVzWzBdLCBzdGFjazFBcnRpZmFjdCk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3JlZmVyZW5jZXMgYmV0d2VlbiBzaWJsaW5nIG5lc3RlZCBzdGFja3Mgc2hvdWxkIG91dHB1dCBmcm9tIG9uZSBhbmQgZ2V0QXR0IGZyb20gdGhlIG90aGVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgcGFyZW50ID0gbmV3IFN0YWNrKGFwcCwgJ1BhcmVudCcpO1xuICAgIGNvbnN0IG5lc3RlZDEgPSBuZXcgTmVzdGVkU3RhY2socGFyZW50LCAnTmVzdGVkMScpO1xuICAgIGNvbnN0IG5lc3RlZDIgPSBuZXcgTmVzdGVkU3RhY2socGFyZW50LCAnTmVzdGVkMicpO1xuICAgIGNvbnN0IHJlc291cmNlMSA9IG5ldyBDZm5SZXNvdXJjZShuZXN0ZWQxLCAnUmVzb3VyY2UxJywgeyB0eXBlOiAnUmVzb3VyY2UxJyB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgQ2ZuUmVzb3VyY2UobmVzdGVkMiwgJ1Jlc291cmNlMicsIHtcbiAgICAgIHR5cGU6ICdSZXNvdXJjZTInLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBSZWZUb1Jlc291cmNlMTogcmVzb3VyY2UxLnJlZlxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGFwcC5zeW50aCgpO1xuXG4gICAgLy8gcHJvZHVjaW5nIG5lc3RlZCBzdGFja1xuICAgIGV4cGVjdChuZXN0ZWQxKS50b01hdGNoKHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBSZXNvdXJjZTE6IHtcbiAgICAgICAgICBUeXBlOiBcIlJlc291cmNlMVwiXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBPdXRwdXRzOiB7XG4gICAgICAgIFBhcmVudE5lc3RlZDFSZXNvdXJjZTE1RjNGMDY1N1JlZjoge1xuICAgICAgICAgIFZhbHVlOiB7XG4gICAgICAgICAgICBSZWY6IFwiUmVzb3VyY2UxXCJcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIC8vIGNvbnN1bWluZyBuZXN0ZWQgc3RhY2tcbiAgICBleHBlY3QobmVzdGVkMikudG9NYXRjaCh7XG4gICAgICBSZXNvdXJjZXM6IHtcbiAgICAgICAgUmVzb3VyY2UyOiB7XG4gICAgICAgICAgVHlwZTogXCJSZXNvdXJjZTJcIixcbiAgICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICBSZWZUb1Jlc291cmNlMToge1xuICAgICAgICAgICAgICBSZWY6IFwicmVmZXJlbmNldG9QYXJlbnROZXN0ZWQxTmVzdGVkU3RhY2tOZXN0ZWQxTmVzdGVkU3RhY2tSZXNvdXJjZTlDMDUzNDJDT3V0cHV0c1BhcmVudE5lc3RlZDFSZXNvdXJjZTE1RjNGMDY1N1JlZlwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICByZWZlcmVuY2V0b1BhcmVudE5lc3RlZDFOZXN0ZWRTdGFja05lc3RlZDFOZXN0ZWRTdGFja1Jlc291cmNlOUMwNTM0MkNPdXRwdXRzUGFyZW50TmVzdGVkMVJlc291cmNlMTVGM0YwNjU3UmVmOiB7XG4gICAgICAgICAgVHlwZTogXCJTdHJpbmdcIlxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICAvLyBwYXJlbnRcbiAgICBleHBlY3QocGFyZW50KS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6Q2xvdWRGb3JtYXRpb246OlN0YWNrJywge1xuICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICByZWZlcmVuY2V0b1BhcmVudE5lc3RlZDFOZXN0ZWRTdGFja05lc3RlZDFOZXN0ZWRTdGFja1Jlc291cmNlOUMwNTM0MkNPdXRwdXRzUGFyZW50TmVzdGVkMVJlc291cmNlMTVGM0YwNjU3UmVmOiB7XG4gICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwiTmVzdGVkMU5lc3RlZFN0YWNrTmVzdGVkMU5lc3RlZFN0YWNrUmVzb3VyY2VDRDBBRDM2QlwiLFxuICAgICAgICAgICAgXCJPdXRwdXRzLlBhcmVudE5lc3RlZDFSZXNvdXJjZTE1RjNGMDY1N1JlZlwiXG4gICAgICAgICAgXVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3N0YWNrSWQgcmV0dXJucyBBV1M6OlN0YWNrSWQgd2hlbiByZWZlcmVuY2VkIGZyb20gdGhlIGNvbnRleHQgb2YgdGhlIG5lc3RlZCBzdGFjaycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgcGFyZW50ID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgbmVzdGVkID0gbmV3IE5lc3RlZFN0YWNrKHBhcmVudCwgJ05lc3RlZFN0YWNrJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IENmblJlc291cmNlKG5lc3RlZCwgJ05lc3RlZFJlc291cmNlJywge1xuICAgICAgdHlwZTogJ05lc3RlZDo6UmVzb3VyY2UnLFxuICAgICAgcHJvcGVydGllczogeyBNeVN0YWNrSWQ6IG5lc3RlZC5zdGFja0lkIH1cbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QobmVzdGVkKS50byhoYXZlUmVzb3VyY2UoJ05lc3RlZDo6UmVzb3VyY2UnLCB7XG4gICAgICBNeVN0YWNrSWQ6IHsgUmVmOiBcIkFXUzo6U3RhY2tJZFwiIH1cbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhY2tJZCByZXR1cm5zIHRoZSBSRUYgb2YgdGhlIENsb3VkRm9ybWF0aW9uOjpTdGFjayByZXNvdXJjZSB3aGVuIHJlZmVyZW5jZWQgZnJvbSB0aGUgcGFyZW50IHN0YWNrJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBwYXJlbnQgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBuZXN0ZWQgPSBuZXcgTmVzdGVkU3RhY2socGFyZW50LCAnTmVzdGVkU3RhY2snKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgQ2ZuUmVzb3VyY2UocGFyZW50LCAnUGFyZW50UmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnUGFyZW50OjpSZXNvdXJjZScsXG4gICAgICBwcm9wZXJ0aWVzOiB7IE5lc3RlZFN0YWNrSWQ6IG5lc3RlZC5zdGFja0lkIH1cbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QocGFyZW50KS50byhoYXZlUmVzb3VyY2UoJ1BhcmVudDo6UmVzb3VyY2UnLCB7XG4gICAgICBOZXN0ZWRTdGFja0lkOiB7IFJlZjogXCJOZXN0ZWRTdGFja05lc3RlZFN0YWNrTmVzdGVkU3RhY2tOZXN0ZWRTdGFja1Jlc291cmNlQjcwODM0RkRcIiB9XG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3N0YWNrTmFtZSByZXR1cm5zIEFXUzo6U3RhY2tOYW1lIHdoZW4gcmVmZXJlbmNlZCBmcm9tIHRoZSBjb250ZXh0IG9mIHRoZSBuZXN0ZWQgc3RhY2snKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHBhcmVudCA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IG5lc3RlZCA9IG5ldyBOZXN0ZWRTdGFjayhwYXJlbnQsICdOZXN0ZWRTdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDZm5SZXNvdXJjZShuZXN0ZWQsICdOZXN0ZWRSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdOZXN0ZWQ6OlJlc291cmNlJyxcbiAgICAgIHByb3BlcnRpZXM6IHsgTXlTdGFja05hbWU6IG5lc3RlZC5zdGFja05hbWUgfVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChuZXN0ZWQpLnRvKGhhdmVSZXNvdXJjZSgnTmVzdGVkOjpSZXNvdXJjZScsIHtcbiAgICAgIE15U3RhY2tOYW1lOiB7IFJlZjogXCJBV1M6OlN0YWNrTmFtZVwiIH1cbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhY2tOYW1lIHJldHVybnMgdGhlIFJFRiBvZiB0aGUgQ2xvdWRGb3JtYXRpb246OlN0YWNrIHJlc291cmNlIHdoZW4gcmVmZXJlbmNlZCBmcm9tIHRoZSBwYXJlbnQgc3RhY2snKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHBhcmVudCA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IG5lc3RlZCA9IG5ldyBOZXN0ZWRTdGFjayhwYXJlbnQsICdOZXN0ZWRTdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDZm5SZXNvdXJjZShwYXJlbnQsICdQYXJlbnRSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdQYXJlbnQ6OlJlc291cmNlJyxcbiAgICAgIHByb3BlcnRpZXM6IHsgTmVzdGVkU3RhY2tOYW1lOiBuZXN0ZWQuc3RhY2tOYW1lIH1cbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QocGFyZW50KS50byhoYXZlUmVzb3VyY2UoJ1BhcmVudDo6UmVzb3VyY2UnLCB7XG4gICAgICBOZXN0ZWRTdGFja05hbWU6IHtcbiAgICAgICAgXCJGbjo6U2VsZWN0XCI6IFtcbiAgICAgICAgICAxLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiRm46OlNwbGl0XCI6IFtcbiAgICAgICAgICAgICAgXCIvXCIsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBSZWY6IFwiTmVzdGVkU3RhY2tOZXN0ZWRTdGFja05lc3RlZFN0YWNrTmVzdGVkU3RhY2tSZXNvdXJjZUI3MDgzNEZEXCJcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgXVxuICAgICAgfVxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdcImFjY291bnRcIiwgXCJyZWdpb25cIiBhbmQgXCJlbnZpcm9ubWVudFwiIGFyZSBhbGwgZGVyaXZlZCBmcm9tIHRoZSBwYXJlbnQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBwYXJlbnQgPSBuZXcgU3RhY2soYXBwLCAnUGFyZW50U3RhY2snLCB7IGVudjogeyBhY2NvdW50OiAnMTIzNGFjY291bnQnLCByZWdpb246ICd1cy1lYXN0LTQ0JyB9IH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IG5lc3RlZCA9IG5ldyBOZXN0ZWRTdGFjayhwYXJlbnQsICdOZXN0ZWRTdGFjaycpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKG5lc3RlZC5lbnZpcm9ubWVudCwgcGFyZW50LmVudmlyb25tZW50KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChuZXN0ZWQuYWNjb3VudCwgcGFyZW50LmFjY291bnQpO1xuICAgIHRlc3QuZGVlcEVxdWFsKG5lc3RlZC5yZWdpb24sIHBhcmVudC5yZWdpb24pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkb3VibGUtbmVzdGVkIHN0YWNrJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgcGFyZW50ID0gbmV3IFN0YWNrKGFwcCwgJ3N0YWNrJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgbmVzdGVkMSA9IG5ldyBOZXN0ZWRTdGFjayhwYXJlbnQsICdOZXN0ZWQxJyk7XG4gICAgY29uc3QgbmVzdGVkMiA9IG5ldyBOZXN0ZWRTdGFjayhuZXN0ZWQxLCAnTmVzdGVkMicpO1xuXG4gICAgbmV3IENmblJlc291cmNlKG5lc3RlZDEsICdSZXNvdXJjZTEnLCB7IHR5cGU6ICdSZXNvdXJjZTo6MScgfSk7XG4gICAgbmV3IENmblJlc291cmNlKG5lc3RlZDIsICdSZXNvdXJjZTInLCB7IHR5cGU6ICdSZXNvdXJjZTo6MicgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgY29uc3QgYXNzZW1ibHkgPSBhcHAuc3ludGgoKTtcblxuICAgIC8vIG5lc3RlZDIgaXMgYSBcImxlYWZcIiwgc28gaXQncyBqdXN0IHRoZSByZXNvdXJjZVxuICAgIGV4cGVjdChuZXN0ZWQyKS50b01hdGNoKHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBSZXNvdXJjZTI6IHsgVHlwZTogXCJSZXNvdXJjZTo6MlwiIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIC8vIG5lc3RlZDEgd2lyZXMgdGhlIG5lc3RlZDIgdGVtcGxhdGUgdGhyb3VnaCBwYXJhbWV0ZXJzLCBzbyB3ZSBleHBlY3QgdGhvc2VcbiAgICBleHBlY3QobmVzdGVkMSkudG8oaGF2ZVJlc291cmNlKCdSZXNvdXJjZTo6MScpKTtcbiAgICBjb25zdCBuZXN0ZWQyVGVtcGxhdGUgPSBTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24obmVzdGVkMSk7XG4gICAgdGVzdC5kZWVwRXF1YWwobmVzdGVkMlRlbXBsYXRlLlBhcmFtZXRlcnMsIHtcbiAgICAgIHJlZmVyZW5jZXRvc3RhY2tBc3NldFBhcmFtZXRlcnM4MTY5YzZmOGFhZWFmNWUyZTg2MjBmNWY4OTVmZmUyMDk5MjAyY2NiNGI2ODg5ZGY0OGZlMDk2N2E4OTQyMzVjUzNCdWNrZXRFODc2OEY1Q1JlZjogeyBUeXBlOiAnU3RyaW5nJyB9LFxuICAgICAgcmVmZXJlbmNldG9zdGFja0Fzc2V0UGFyYW1ldGVyczgxNjljNmY4YWFlYWY1ZTJlODYyMGY1Zjg5NWZmZTIwOTkyMDJjY2I0YjY4ODlkZjQ4ZmUwOTY3YTg5NDIzNWNTM1ZlcnNpb25LZXk0OUREODNBMlJlZjogeyBUeXBlOiAnU3RyaW5nJyB9LFxuICAgIH0pO1xuXG4gICAgLy8gcGFyZW50IHN0YWNrIHNob3VsZCBoYXZlIHR3byBzZXRzIG9mIHBhcmFtZXRlcnMuIG9uZSBmb3IgdGhlIGZpcnN0IG5lc3RlZCBzdGFjayBhbmQgdGhlIHNlY29uZFxuICAgIC8vIGZvciB0aGUgc2Vjb25kIG5lc3RlZCBzdGFjaywgcGFzc2VkIGluIGFzIHBhcmFtZXRlcnMgdG8gdGhlIGZpcnN0XG4gICAgY29uc3QgdGVtcGxhdGUgPSBTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24ocGFyZW50KTtcbiAgICB0ZXN0LmRlZXBFcXVhbCh0ZW1wbGF0ZS5QYXJhbWV0ZXJzLCB7XG4gICAgICBBc3NldFBhcmFtZXRlcnM4MTY5YzZmOGFhZWFmNWUyZTg2MjBmNWY4OTVmZmUyMDk5MjAyY2NiNGI2ODg5ZGY0OGZlMDk2N2E4OTQyMzVjUzNCdWNrZXRERTNCODhENjogeyBUeXBlOiAnU3RyaW5nJywgRGVzY3JpcHRpb246ICdTMyBidWNrZXQgZm9yIGFzc2V0IFwiODE2OWM2ZjhhYWVhZjVlMmU4NjIwZjVmODk1ZmZlMjA5OTIwMmNjYjRiNjg4OWRmNDhmZTA5NjdhODk0MjM1Y1wiJyB9LFxuICAgICAgQXNzZXRQYXJhbWV0ZXJzODE2OWM2ZjhhYWVhZjVlMmU4NjIwZjVmODk1ZmZlMjA5OTIwMmNjYjRiNjg4OWRmNDhmZTA5NjdhODk0MjM1Y1MzVmVyc2lvbktleTNBNjJFRkVBOiB7IFR5cGU6ICdTdHJpbmcnLCBEZXNjcmlwdGlvbjogJ1MzIGtleSBmb3IgYXNzZXQgdmVyc2lvbiBcIjgxNjljNmY4YWFlYWY1ZTJlODYyMGY1Zjg5NWZmZTIwOTkyMDJjY2I0YjY4ODlkZjQ4ZmUwOTY3YTg5NDIzNWNcIicgfSxcbiAgICAgIEFzc2V0UGFyYW1ldGVyczgxNjljNmY4YWFlYWY1ZTJlODYyMGY1Zjg5NWZmZTIwOTkyMDJjY2I0YjY4ODlkZjQ4ZmUwOTY3YTg5NDIzNWNBcnRpZmFjdEhhc2g3REM1NDZFMDogeyBUeXBlOiAnU3RyaW5nJywgRGVzY3JpcHRpb246ICdBcnRpZmFjdCBoYXNoIGZvciBhc3NldCBcIjgxNjljNmY4YWFlYWY1ZTJlODYyMGY1Zjg5NWZmZTIwOTkyMDJjY2I0YjY4ODlkZjQ4ZmUwOTY3YTg5NDIzNWNcIicgfSxcbiAgICAgIEFzc2V0UGFyYW1ldGVyczhiNTA3OTVhOTUwY2NhNmIwMTM1MmYxNjJjNDVkOWQyNzRkZWU2YmM0MDlmMmYyYjJlZDAyOWFkNjgyOGIzYmZTM0J1Y2tldDc2QUNGQjM4OiB7IFR5cGU6ICdTdHJpbmcnLCBEZXNjcmlwdGlvbjogJ1MzIGJ1Y2tldCBmb3IgYXNzZXQgXCI4YjUwNzk1YTk1MGNjYTZiMDEzNTJmMTYyYzQ1ZDlkMjc0ZGVlNmJjNDA5ZjJmMmIyZWQwMjlhZDY4MjhiM2JmXCInIH0sXG4gICAgICBBc3NldFBhcmFtZXRlcnM4YjUwNzk1YTk1MGNjYTZiMDEzNTJmMTYyYzQ1ZDlkMjc0ZGVlNmJjNDA5ZjJmMmIyZWQwMjlhZDY4MjhiM2JmUzNWZXJzaW9uS2V5MDQxNjJFRjE6IHsgVHlwZTogJ1N0cmluZycsIERlc2NyaXB0aW9uOiAnUzMga2V5IGZvciBhc3NldCB2ZXJzaW9uIFwiOGI1MDc5NWE5NTBjY2E2YjAxMzUyZjE2MmM0NWQ5ZDI3NGRlZTZiYzQwOWYyZjJiMmVkMDI5YWQ2ODI4YjNiZlwiJyB9LFxuICAgICAgQXNzZXRQYXJhbWV0ZXJzOGI1MDc5NWE5NTBjY2E2YjAxMzUyZjE2MmM0NWQ5ZDI3NGRlZTZiYzQwOWYyZjJiMmVkMDI5YWQ2ODI4YjNiZkFydGlmYWN0SGFzaEYyMjdBREQzOiB7IFR5cGU6ICdTdHJpbmcnLCBEZXNjcmlwdGlvbjogJ0FydGlmYWN0IGhhc2ggZm9yIGFzc2V0IFwiOGI1MDc5NWE5NTBjY2E2YjAxMzUyZjE2MmM0NWQ5ZDI3NGRlZTZiYzQwOWYyZjJiMmVkMDI5YWQ2ODI4YjNiZlwiJyB9XG4gICAgfSk7XG5cbiAgICAvLyBwcm94eSBhc3NldCBwYXJhbXMgdG8gbmVzdGVkIHN0YWNrXG4gICAgZXhwZWN0KHBhcmVudCkudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNsb3VkRm9ybWF0aW9uOjpTdGFjaycsIHtcbiAgICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICAgcmVmZXJlbmNldG9zdGFja0Fzc2V0UGFyYW1ldGVyczgxNjljNmY4YWFlYWY1ZTJlODYyMGY1Zjg5NWZmZTIwOTkyMDJjY2I0YjY4ODlkZjQ4ZmUwOTY3YTg5NDIzNWNTM0J1Y2tldEU4NzY4RjVDUmVmOiB7IFJlZjogXCJBc3NldFBhcmFtZXRlcnM4MTY5YzZmOGFhZWFmNWUyZTg2MjBmNWY4OTVmZmUyMDk5MjAyY2NiNGI2ODg5ZGY0OGZlMDk2N2E4OTQyMzVjUzNCdWNrZXRERTNCODhENlwiIH0sXG4gICAgICAgIHJlZmVyZW5jZXRvc3RhY2tBc3NldFBhcmFtZXRlcnM4MTY5YzZmOGFhZWFmNWUyZTg2MjBmNWY4OTVmZmUyMDk5MjAyY2NiNGI2ODg5ZGY0OGZlMDk2N2E4OTQyMzVjUzNWZXJzaW9uS2V5NDlERDgzQTJSZWY6IHsgUmVmOiBcIkFzc2V0UGFyYW1ldGVyczgxNjljNmY4YWFlYWY1ZTJlODYyMGY1Zjg5NWZmZTIwOTkyMDJjY2I0YjY4ODlkZjQ4ZmUwOTY3YTg5NDIzNWNTM1ZlcnNpb25LZXkzQTYyRUZFQVwiIH1cbiAgICAgIH1cbiAgICB9KSk7XG5cbiAgICAvLyBwYXJlbnQgc3RhY2sgc2hvdWxkIGhhdmUgMiBhc3NldHNcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5nZXRTdGFja0J5TmFtZShwYXJlbnQuc3RhY2tOYW1lKS5hc3NldHMubGVuZ3RoLCAyKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYXNzZXRzIHdpdGhpbiBuZXN0ZWQgc3RhY2tzIGFyZSBwcm94aWVkIGZyb20gdGhlIHBhcmVudCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHBhcmVudCA9IG5ldyBTdGFjayhhcHAsICdQYXJlbnRTdGFjaycpO1xuICAgIGNvbnN0IG5lc3RlZCA9IG5ldyBOZXN0ZWRTdGFjayhwYXJlbnQsICdOZXN0ZWRTdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFzc2V0ID0gbmV3IHMzX2Fzc2V0cy5Bc3NldChuZXN0ZWQsICdhc3NldCcsIHtcbiAgICAgIHBhdGg6IHBhdGguam9pbihfX2Rpcm5hbWUsICdhc3NldC1maXh0dXJlLnR4dCcpXG4gICAgfSk7XG5cbiAgICBuZXcgQ2ZuUmVzb3VyY2UobmVzdGVkLCAnTmVzdGVkUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnTmVzdGVkOjpSZXNvdXJjZScsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIEFzc2V0QnVja2V0OiBhc3NldC5zM0J1Y2tldE5hbWUsXG4gICAgICAgIEFzc2V0S2V5OiBhc3NldC5zM09iamVjdEtleVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG4gICAgY29uc3QgdGVtcGxhdGUgPSBTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24ocGFyZW50KTtcblxuICAgIC8vIHR3byBzZXRzIG9mIGFzc2V0IHBhcmFtZXRlcnM6IG9uZSBmb3IgdGhlIG5lc3RlZCBzdGFjayBpdHNlbGYgYW5kIG9uZSBhcyBhIHByb3h5IGZvciB0aGUgYXNzZXQgd2l0aGluIHRoZSBzdGFja1xuICAgIHRlc3QuZGVlcEVxdWFsKHRlbXBsYXRlLlBhcmFtZXRlcnMsIHtcbiAgICAgIEFzc2V0UGFyYW1ldGVyc2RiMDFlZTJlYjdhZGM3OTE1ZTM2NGRjNDEwZDg2MWU1Njk1NDNmOWJlMzc2MWQ1MzVhNjhkNWMyY2MxODEyODFTM0J1Y2tldEMxODhGNjM3OiB7IFR5cGU6ICdTdHJpbmcnLCBEZXNjcmlwdGlvbjogJ1MzIGJ1Y2tldCBmb3IgYXNzZXQgXCJkYjAxZWUyZWI3YWRjNzkxNWUzNjRkYzQxMGQ4NjFlNTY5NTQzZjliZTM3NjFkNTM1YTY4ZDVjMmNjMTgxMjgxXCInIH0sXG4gICAgICBBc3NldFBhcmFtZXRlcnNkYjAxZWUyZWI3YWRjNzkxNWUzNjRkYzQxMGQ4NjFlNTY5NTQzZjliZTM3NjFkNTM1YTY4ZDVjMmNjMTgxMjgxUzNWZXJzaW9uS2V5QzdGNERCRjI6IHsgVHlwZTogJ1N0cmluZycsIERlc2NyaXB0aW9uOiAnUzMga2V5IGZvciBhc3NldCB2ZXJzaW9uIFwiZGIwMWVlMmViN2FkYzc5MTVlMzY0ZGM0MTBkODYxZTU2OTU0M2Y5YmUzNzYxZDUzNWE2OGQ1YzJjYzE4MTI4MVwiJyB9LFxuICAgICAgQXNzZXRQYXJhbWV0ZXJzZGIwMWVlMmViN2FkYzc5MTVlMzY0ZGM0MTBkODYxZTU2OTU0M2Y5YmUzNzYxZDUzNWE2OGQ1YzJjYzE4MTI4MUFydGlmYWN0SGFzaDM3M0IxNEQyOiB7IFR5cGU6ICdTdHJpbmcnLCBEZXNjcmlwdGlvbjogJ0FydGlmYWN0IGhhc2ggZm9yIGFzc2V0IFwiZGIwMWVlMmViN2FkYzc5MTVlMzY0ZGM0MTBkODYxZTU2OTU0M2Y5YmUzNzYxZDUzNWE2OGQ1YzJjYzE4MTI4MVwiJyB9LFxuICAgICAgQXNzZXRQYXJhbWV0ZXJzNDZiMTA3ZDZkYjc5OGNhNDYwNDZiODY2OWQwNTdhNGRlYmNiZGJhYWRkYjYxNzA0MDA3NDhjMmY5ZTRmOWQ3MVMzQnVja2V0M0M0MjY1RTk6IHsgVHlwZTogJ1N0cmluZycsIERlc2NyaXB0aW9uOiAnUzMgYnVja2V0IGZvciBhc3NldCBcIjQ2YjEwN2Q2ZGI3OThjYTQ2MDQ2Yjg2NjlkMDU3YTRkZWJjYmRiYWFkZGI2MTcwNDAwNzQ4YzJmOWU0ZjlkNzFcIicgfSxcbiAgICAgIEFzc2V0UGFyYW1ldGVyczQ2YjEwN2Q2ZGI3OThjYTQ2MDQ2Yjg2NjlkMDU3YTRkZWJjYmRiYWFkZGI2MTcwNDAwNzQ4YzJmOWU0ZjlkNzFTM1ZlcnNpb25LZXk4RTk4MTUzNTogeyBUeXBlOiAnU3RyaW5nJywgRGVzY3JpcHRpb246ICdTMyBrZXkgZm9yIGFzc2V0IHZlcnNpb24gXCI0NmIxMDdkNmRiNzk4Y2E0NjA0NmI4NjY5ZDA1N2E0ZGViY2JkYmFhZGRiNjE3MDQwMDc0OGMyZjllNGY5ZDcxXCInIH0sXG4gICAgICBBc3NldFBhcmFtZXRlcnM0NmIxMDdkNmRiNzk4Y2E0NjA0NmI4NjY5ZDA1N2E0ZGViY2JkYmFhZGRiNjE3MDQwMDc0OGMyZjllNGY5ZDcxQXJ0aWZhY3RIYXNoNDVBMjg1ODM6IHsgVHlwZTogJ1N0cmluZycsIERlc2NyaXB0aW9uOiAnQXJ0aWZhY3QgaGFzaCBmb3IgYXNzZXQgXCI0NmIxMDdkNmRiNzk4Y2E0NjA0NmI4NjY5ZDA1N2E0ZGViY2JkYmFhZGRiNjE3MDQwMDc0OGMyZjllNGY5ZDcxXCInIH1cbiAgICB9KTtcblxuICAgIC8vIGFzc2V0IHByb3h5IHBhcmFtZXRlcnMgYXJlIHBhc3NlZCB0byB0aGUgbmVzdGVkIHN0YWNrXG4gICAgZXhwZWN0KHBhcmVudCkudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNsb3VkRm9ybWF0aW9uOjpTdGFjaycsIHtcbiAgICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICAgcmVmZXJlbmNldG9QYXJlbnRTdGFja0Fzc2V0UGFyYW1ldGVyc2RiMDFlZTJlYjdhZGM3OTE1ZTM2NGRjNDEwZDg2MWU1Njk1NDNmOWJlMzc2MWQ1MzVhNjhkNWMyY2MxODEyODFTM0J1Y2tldDgyQzU1Qjk2UmVmOiB7IFJlZjogXCJBc3NldFBhcmFtZXRlcnNkYjAxZWUyZWI3YWRjNzkxNWUzNjRkYzQxMGQ4NjFlNTY5NTQzZjliZTM3NjFkNTM1YTY4ZDVjMmNjMTgxMjgxUzNCdWNrZXRDMTg4RjYzN1wiIH0sXG4gICAgICAgIHJlZmVyZW5jZXRvUGFyZW50U3RhY2tBc3NldFBhcmFtZXRlcnNkYjAxZWUyZWI3YWRjNzkxNWUzNjRkYzQxMGQ4NjFlNTY5NTQzZjliZTM3NjFkNTM1YTY4ZDVjMmNjMTgxMjgxUzNWZXJzaW9uS2V5QTQzQzNDQzZSZWY6IHsgUmVmOiBcIkFzc2V0UGFyYW1ldGVyc2RiMDFlZTJlYjdhZGM3OTE1ZTM2NGRjNDEwZDg2MWU1Njk1NDNmOWJlMzc2MWQ1MzVhNjhkNWMyY2MxODEyODFTM1ZlcnNpb25LZXlDN0Y0REJGMlwiIH0sXG4gICAgICB9XG4gICAgfSkpO1xuXG4gICAgLy8gcGFyZW50IHN0YWNrIHNob3VsZCBoYXZlIDIgYXNzZXRzXG4gICAgdGVzdC5kZWVwRXF1YWwoYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUocGFyZW50LnN0YWNrTmFtZSkuYXNzZXRzLmxlbmd0aCwgMik7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2RvY2tlciBpbWFnZSBhc3NldHMgYXJlIHdpcmVkIHRocm91Z2ggdGhlIHRvcC1sZXZlbCBzdGFjaycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHBhcmVudCA9IG5ldyBTdGFjayhhcHAsICdteS1zdGFjaycpO1xuICAgIGNvbnN0IG5lc3RlZCA9IG5ldyBOZXN0ZWRTdGFjayhwYXJlbnQsICduZXN0ZWQtc3RhY2snKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBsb2NhdGlvbiA9IG5lc3RlZC5hZGREb2NrZXJJbWFnZUFzc2V0KHtcbiAgICAgIGRpcmVjdG9yeU5hbWU6ICdteS1pbWFnZScsXG4gICAgICBkb2NrZXJCdWlsZEFyZ3M6IHsga2V5OiAndmFsdWUnLCBib29tOiAnYmFtJyB9LFxuICAgICAgZG9ja2VyQnVpbGRUYXJnZXQ6ICdidWlsZFRhcmdldCcsXG4gICAgICBzb3VyY2VIYXNoOiAnaGFzaC1vZi1zb3VyY2UnLFxuICAgIH0pO1xuXG4gICAgLy8gdXNlIHRoZSBhc3NldCwgc28gdGhlIHBhcmFtZXRlcnMgd2lsbCBiZSB3aXJlZC5cbiAgICBuZXcgc25zLlRvcGljKG5lc3RlZCwgJ015VG9waWMnLCB7XG4gICAgICBkaXNwbGF5TmFtZTogYGltYWdlIGxvY2F0aW9uIGlzICR7bG9jYXRpb24uaW1hZ2VVcml9YFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGFzbSA9IGFwcC5zeW50aCgpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzbS5nZXRTdGFja0FydGlmYWN0KHBhcmVudC5hcnRpZmFjdElkKS5hc3NldHMsIFtcbiAgICAgIHtcbiAgICAgICAgcmVwb3NpdG9yeU5hbWU6ICdhd3MtY2RrL2Fzc2V0cycsXG4gICAgICAgIGltYWdlVGFnOiAnaGFzaC1vZi1zb3VyY2UnLFxuICAgICAgICBpZDogJ2hhc2gtb2Ytc291cmNlJyxcbiAgICAgICAgcGFja2FnaW5nOiAnY29udGFpbmVyLWltYWdlJyxcbiAgICAgICAgcGF0aDogJ215LWltYWdlJyxcbiAgICAgICAgc291cmNlSGFzaDogJ2hhc2gtb2Ytc291cmNlJyxcbiAgICAgICAgYnVpbGRBcmdzOiB7IGtleTogJ3ZhbHVlJywgYm9vbTogJ2JhbScgfSxcbiAgICAgICAgdGFyZ2V0OiAnYnVpbGRUYXJnZXQnXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBwYXRoOiAnbXlzdGFja25lc3RlZHN0YWNrRkFFMTJGQjUubmVzdGVkLnRlbXBsYXRlLmpzb24nLFxuICAgICAgICBpZDogJ2ZjZGFlZTc5ZWI3OWYzN2VjYTNhOWIxY2MwY2M5YmExNTBlNGVlYThjNWQ2ZDBjMzQzY2I2Y2Q5ZGM2OGUyZTUnLFxuICAgICAgICBwYWNrYWdpbmc6ICdmaWxlJyxcbiAgICAgICAgc291cmNlSGFzaDogJ2ZjZGFlZTc5ZWI3OWYzN2VjYTNhOWIxY2MwY2M5YmExNTBlNGVlYThjNWQ2ZDBjMzQzY2I2Y2Q5ZGM2OGUyZTUnLFxuICAgICAgICBzM0J1Y2tldFBhcmFtZXRlcjogJ0Fzc2V0UGFyYW1ldGVyc2ZjZGFlZTc5ZWI3OWYzN2VjYTNhOWIxY2MwY2M5YmExNTBlNGVlYThjNWQ2ZDBjMzQzY2I2Y2Q5ZGM2OGUyZTVTM0J1Y2tldDY3QTc0OUY4JyxcbiAgICAgICAgczNLZXlQYXJhbWV0ZXI6ICdBc3NldFBhcmFtZXRlcnNmY2RhZWU3OWViNzlmMzdlY2EzYTliMWNjMGNjOWJhMTUwZTRlZWE4YzVkNmQwYzM0M2NiNmNkOWRjNjhlMmU1UzNWZXJzaW9uS2V5RTFFNkE4RDQnLFxuICAgICAgICBhcnRpZmFjdEhhc2hQYXJhbWV0ZXI6ICdBc3NldFBhcmFtZXRlcnNmY2RhZWU3OWViNzlmMzdlY2EzYTliMWNjMGNjOWJhMTUwZTRlZWE4YzVkNmQwYzM0M2NiNmNkOWRjNjhlMmU1QXJ0aWZhY3RIYXNoMEFFREJFOEEnXG4gICAgICB9XG4gICAgXSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnbWV0YWRhdGEgZGVmaW5lZCBpbiBuZXN0ZWQgc3RhY2tzIGlzIHJlcG9ydGVkIGF0IHRoZSBwYXJlbnQgc3RhY2sgbGV2ZWwgaW4gdGhlIGNsb3VkIGFzc2VtYmx5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKHsgc3RhY2tUcmFjZXM6IGZhbHNlIH0pO1xuICAgIGNvbnN0IHBhcmVudCA9IG5ldyBTdGFjayhhcHAsICdwYXJlbnQnKTtcbiAgICBjb25zdCBjaGlsZCA9IG5ldyBTdGFjayhwYXJlbnQsICdjaGlsZCcpO1xuICAgIGNvbnN0IG5lc3RlZCA9IG5ldyBOZXN0ZWRTdGFjayhjaGlsZCwgJ25lc3RlZCcpO1xuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmblJlc291cmNlKG5lc3RlZCwgJ3Jlc291cmNlJywgeyB0eXBlOiAnZm9vJyB9KTtcblxuICAgIC8vIFdIRU5cbiAgICByZXNvdXJjZS5ub2RlLmFkZE1ldGFkYXRhKCdmb28nLCAnYmFyJyk7XG5cbiAgICAvLyBUSEVOOiB0aGUgZmlyc3Qgbm9uLW5lc3RlZCBzdGFjayByZWNvcmRzIHRoZSBhc3NlbWJseSBtZXRhZGF0YVxuICAgIGNvbnN0IGFzbSA9IGFwcC5zeW50aCgpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzbS5zdGFja3MubGVuZ3RoLCAyKTsgLy8gb25seSBvbmUgc3RhY2sgaXMgZGVmaW5lZCBhcyBhbiBhcnRpZmFjdFxuICAgIHRlc3QuZGVlcEVxdWFsKGFzbS5nZXRTdGFja0J5TmFtZShwYXJlbnQuc3RhY2tOYW1lKS5maW5kTWV0YWRhdGFCeVR5cGUoJ2ZvbycpLCBbXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNtLmdldFN0YWNrQnlOYW1lKGNoaWxkLnN0YWNrTmFtZSkuZmluZE1ldGFkYXRhQnlUeXBlKCdmb28nKSwgW1xuICAgICAge1xuICAgICAgICBwYXRoOiAnL3BhcmVudC9jaGlsZC9uZXN0ZWQvcmVzb3VyY2UnLFxuICAgICAgICB0eXBlOiAnZm9vJyxcbiAgICAgICAgZGF0YTogJ2JhcidcbiAgICAgIH1cbiAgICBdKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAncmVmZXJlbmNpbmcgYXR0cmlidXRlcyB3aXRoIHBlcmlvZCBhY3Jvc3Mgc3RhY2tzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBwYXJlbnQgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBuZXN0ZWQgPSBuZXcgTmVzdGVkU3RhY2socGFyZW50LCAnbmVzdGVkJyk7XG4gICAgY29uc3QgY29uc3VtZWQgPSBuZXcgQ2ZuUmVzb3VyY2UobmVzdGVkLCAncmVzb3VyY2UtaW4tbmVzdGVkJywgeyB0eXBlOiAnQ09OU1VNRUQnIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDZm5SZXNvdXJjZShwYXJlbnQsICdyZXNvdXJjZS1pbi1wYXJlbnQnLCB7XG4gICAgICB0eXBlOiAnQ09OU1VNRVInLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBDb25zdW1lZEF0dHJpYnV0ZTogY29uc3VtZWQuZ2V0QXR0KCdDb25zdW1lZC5BdHRyaWJ1dGUnKVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChuZXN0ZWQpLnRvTWF0Y2goe1xuICAgICAgUmVzb3VyY2VzOiB7XG4gICAgICAgIHJlc291cmNlaW5uZXN0ZWQ6IHtcbiAgICAgICAgICBUeXBlOiBcIkNPTlNVTUVEXCJcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIE91dHB1dHM6IHtcbiAgICAgICAgbmVzdGVkcmVzb3VyY2Vpbm5lc3RlZDU5QjFGMDFDQ29uc3VtZWRBdHRyaWJ1dGU6IHtcbiAgICAgICAgICBWYWx1ZToge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJyZXNvdXJjZWlubmVzdGVkXCIsXG4gICAgICAgICAgICAgIFwiQ29uc3VtZWQuQXR0cmlidXRlXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcbiAgICBleHBlY3QocGFyZW50KS50byhoYXZlUmVzb3VyY2UoJ0NPTlNVTUVSJywge1xuICAgICAgQ29uc3VtZWRBdHRyaWJ1dGU6IHtcbiAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICBcIm5lc3RlZE5lc3RlZFN0YWNrbmVzdGVkTmVzdGVkU3RhY2tSZXNvdXJjZTNERDE0M0JGXCIsXG4gICAgICAgICAgXCJPdXRwdXRzLm5lc3RlZHJlc291cmNlaW5uZXN0ZWQ1OUIxRjAxQ0NvbnN1bWVkQXR0cmlidXRlXCJcbiAgICAgICAgXVxuICAgICAgfVxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdtaXNzaW5nIGNvbnRleHQgaW4gbmVzdGVkIHN0YWNrIGlzIHJlcG9ydGVkIGlmIHRoZSBjb250ZXh0IGlzIG5vdCBhdmFpbGFibGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdQYXJlbnRTdGFjaycsIHsgZW52OiB7IGFjY291bnQ6ICcxMjM0YWNjb3VudCcsIHJlZ2lvbjogJ3VzLWVhc3QtNDQnIH0gfSk7XG4gICAgY29uc3QgbmVzdGVkU3RhY2sgPSBuZXcgTmVzdGVkU3RhY2soc3RhY2ssICduZXN0ZWQnKTtcbiAgICBjb25zdCBwcm92aWRlciA9ICdkdW1teVByb3ZpZGVyJztcbiAgICBjb25zdCBleHBlY3RlZEtleSA9IENvbnRleHRQcm92aWRlci5nZXRLZXkobmVzdGVkU3RhY2ssIHtcbiAgICAgIHByb3ZpZGVyXG4gICAgfSkua2V5O1xuXG4gICAgLy8gV0hFTlxuICAgIENvbnRleHRQcm92aWRlci5nZXRWYWx1ZShuZXN0ZWRTdGFjaywge1xuICAgICAgcHJvdmlkZXIsXG4gICAgICBkdW1teVZhbHVlOiBbJ2R1bW15MWEnLCAnZHVtbXkxYicsICdkdW1teTFjJ10sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOOiBtaXNzaW5nIGNvbnRleHQgaXMgcmVwb3J0ZWQgaW4gdGhlIGNsb3VkIGFzc2VtYmx5XG4gICAgY29uc3QgYXNtID0gYXBwLnN5bnRoKCk7XG4gICAgY29uc3QgbWlzc2luZyA9IGFzbS5tYW5pZmVzdC5taXNzaW5nO1xuXG4gICAgdGVzdC5vayhtaXNzaW5nICYmIG1pc3NpbmcuZmluZChtID0+IHtcbiAgICAgIHJldHVybiAobS5rZXkgPT09IGV4cGVjdGVkS2V5KTtcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbn07XG4iXX0=