"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'fixed retention'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: lib_1.RetentionDays.ONE_WEEK
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 7
        }));
        test.done();
    },
    'default retention'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 731
        }));
        test.done();
    },
    'infinite retention/dont delete log group by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: lib_1.RetentionDays.INFINITE,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: "AWS::Logs::LogGroup",
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'infinite retention via legacy method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            // Don't know why TypeScript doesn't complain about passing Infinity to
            // something where an enum is expected, but better keep this behavior for
            // existing clients.
            retention: Infinity
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: "AWS::Logs::LogGroup",
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'will delete log group if asked to'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: Infinity,
            removalPolicy: core_1.RemovalPolicy.DESTROY
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: "AWS::Logs::LogGroup",
                    DeletionPolicy: "Delete",
                    UpdateReplacePolicy: "Delete"
                }
            }
        }));
        test.done();
    },
    'import from arn'(test) {
        // GIVEN
        const stack2 = new core_1.Stack();
        // WHEN
        const imported = lib_1.LogGroup.fromLogGroupArn(stack2, 'lg', 'arn:aws:logs:us-east-1:123456789012:log-group:my-log-group');
        imported.addStream('MakeMeAStream');
        // THEN
        test.deepEqual(imported.logGroupName, 'my-log-group');
        test.deepEqual(imported.logGroupArn, 'arn:aws:logs:us-east-1:123456789012:log-group:my-log-group');
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::Logs::LogStream', {
            LogGroupName: "my-log-group"
        }));
        test.done();
    },
    'import from name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const imported = lib_1.LogGroup.fromLogGroupName(stack, 'lg', 'my-log-group');
        imported.addStream('MakeMeAStream');
        // THEN
        test.deepEqual(imported.logGroupName, 'my-log-group');
        test.ok(/^arn:.+:logs:.+:.+:log-group:my-log-group$/.test(imported.logGroupArn), `LogGroup ARN ${imported.logGroupArn} does not match the expected pattern`);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogStream', {
            LogGroupName: 'my-log-group'
        }));
        test.done();
    },
    'extractMetric'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        // WHEN
        const metric = lg.extractMetric('$.myField', 'MyService', 'Field');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::MetricFilter', {
            FilterPattern: "{ $.myField = \"*\" }",
            LogGroupName: { Ref: "LogGroupF5B46931" },
            MetricTransformations: [
                {
                    MetricName: "Field",
                    MetricNamespace: "MyService",
                    MetricValue: "$.myField"
                }
            ]
        }));
        test.equal(metric.namespace, 'MyService');
        test.equal(metric.metricName, 'Field');
        test.done();
    },
    'extractMetric allows passing in namespaces with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        // WHEN
        const metric = lg.extractMetric('$.myField', 'MyNamespace/MyService', 'Field');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::MetricFilter', {
            FilterPattern: "{ $.myField = \"*\" }",
            MetricTransformations: [
                {
                    MetricName: "Field",
                    MetricNamespace: "MyNamespace/MyService",
                    MetricValue: "$.myField"
                }
            ]
        }));
        test.equal(metric.namespace, 'MyNamespace/MyService');
        test.equal(metric.metricName, 'Field');
        test.done();
    },
    'grant'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        const user = new iam.User(stack, 'User');
        // WHEN
        lg.grantWrite(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: ["logs:CreateLogStream", "logs:PutLogEvents"],
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["LogGroupF5B46931", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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