import numpy as np
import numba
import scipy.optimize as sopt
import json

sin = np.sin
cos = np.cos
atan2 = np.arctan2
sqrt = np.sqrt 
sign = np.sign 


class grid_trafo_2bus_class: 

    def __init__(self): 

        self.t_end = 10.000000 
        self.Dt = 0.0010000 
        self.decimation = 10.000000 
        self.itol = 1e-6 
        self.Dt_max = 0.001000 
        self.Dt_min = 0.001000 
        self.solvern = 5 
        self.imax = 100 
        self.N_x = 1
        self.N_y = 192 
        self.N_z = 75 
        self.N_store = 10000 
        self.params_list = [] 
        self.params_values_list  = [] 
        self.inputs_ini_list = ['v_R0_a_r', 'v_R0_a_i', 'v_R0_b_r', 'v_R0_b_i', 'v_R0_c_r', 'v_R0_c_i', 'i_R1_n_r', 'i_R1_n_i', 'i_R11_n_r', 'i_R11_n_i', 'i_R15_n_r', 'i_R15_n_i', 'i_R16_n_r', 'i_R16_n_i', 'i_R17_n_r', 'i_R17_n_i', 'i_R18_n_r', 'i_R18_n_i', 'i_R2_a_r', 'i_R2_a_i', 'i_R2_b_r', 'i_R2_b_i', 'i_R2_c_r', 'i_R2_c_i', 'i_R2_n_r', 'i_R2_n_i', 'i_R3_a_r', 'i_R3_a_i', 'i_R3_b_r', 'i_R3_b_i', 'i_R3_c_r', 'i_R3_c_i', 'i_R3_n_r', 'i_R3_n_i', 'i_R4_a_r', 'i_R4_a_i', 'i_R4_b_r', 'i_R4_b_i', 'i_R4_c_r', 'i_R4_c_i', 'i_R4_n_r', 'i_R4_n_i', 'i_R5_a_r', 'i_R5_a_i', 'i_R5_b_r', 'i_R5_b_i', 'i_R5_c_r', 'i_R5_c_i', 'i_R5_n_r', 'i_R5_n_i', 'i_R6_a_r', 'i_R6_a_i', 'i_R6_b_r', 'i_R6_b_i', 'i_R6_c_r', 'i_R6_c_i', 'i_R6_n_r', 'i_R6_n_i', 'i_R7_a_r', 'i_R7_a_i', 'i_R7_b_r', 'i_R7_b_i', 'i_R7_c_r', 'i_R7_c_i', 'i_R7_n_r', 'i_R7_n_i', 'i_R8_a_r', 'i_R8_a_i', 'i_R8_b_r', 'i_R8_b_i', 'i_R8_c_r', 'i_R8_c_i', 'i_R8_n_r', 'i_R8_n_i', 'i_R9_a_r', 'i_R9_a_i', 'i_R9_b_r', 'i_R9_b_i', 'i_R9_c_r', 'i_R9_c_i', 'i_R9_n_r', 'i_R9_n_i', 'i_R10_a_r', 'i_R10_a_i', 'i_R10_b_r', 'i_R10_b_i', 'i_R10_c_r', 'i_R10_c_i', 'i_R10_n_r', 'i_R10_n_i', 'i_R12_a_r', 'i_R12_a_i', 'i_R12_b_r', 'i_R12_b_i', 'i_R12_c_r', 'i_R12_c_i', 'i_R12_n_r', 'i_R12_n_i', 'i_R13_a_r', 'i_R13_a_i', 'i_R13_b_r', 'i_R13_b_i', 'i_R13_c_r', 'i_R13_c_i', 'i_R13_n_r', 'i_R13_n_i', 'i_R14_a_r', 'i_R14_a_i', 'i_R14_b_r', 'i_R14_b_i', 'i_R14_c_r', 'i_R14_c_i', 'i_R14_n_r', 'i_R14_n_i', 'p_R1_a', 'q_R1_a', 'p_R1_b', 'q_R1_b', 'p_R1_c', 'q_R1_c', 'p_R11_a', 'q_R11_a', 'p_R11_b', 'q_R11_b', 'p_R11_c', 'q_R11_c', 'p_R15_a', 'q_R15_a', 'p_R15_b', 'q_R15_b', 'p_R15_c', 'q_R15_c', 'p_R16_a', 'q_R16_a', 'p_R16_b', 'q_R16_b', 'p_R16_c', 'q_R16_c', 'p_R17_a', 'q_R17_a', 'p_R17_b', 'q_R17_b', 'p_R17_c', 'q_R17_c', 'p_R18_a', 'q_R18_a', 'p_R18_b', 'q_R18_b', 'p_R18_c', 'q_R18_c', 'u_dummy'] 
        self.inputs_ini_values_list  = [11547.0, 0.0, -5773.499999999997, -9999.995337498915, -5773.5000000000055, 9999.99533749891, -0.1137718099519418, -0.062435917080051695, 0.038921773016749484, -0.10489637569849819, 0.3445405949857374, -0.7980789653137919, 0.37357657074305806, -0.8707524289912385, 0.3054824958108071, -0.699068183990839, 0.4382865505353877, -0.9973969673568632, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -63333.33333323794, -20816.65999458629, -63333.33333316128, -20816.659994702903, -63333.33333361338, -20816.659994702255, -4749.999999991674, -1561.2494996136027, -4749.999999961012, -1561.2494996070764, -4750.000000051293, -1561.2494995734658, -16466.666666612546, -5412.331598952091, -16466.666666233108, -5412.331598746236, -16466.666667180372, -5412.331598103374, -17416.666666636127, -5724.581498685594, -17416.66666639193, -5724.581498594195, -17416.666666998393, -5724.581498278923, -11083.333333311602, -3642.9154991873775, -11083.333333126331, -3642.915499108254, -11083.333333581077, -3642.91549887427, -14883.333333304252, -4891.915098923226, -14883.333333039369, -4891.915098810083, -14883.333333685177, -4891.915098464485, 1.0] 
        self.inputs_run_list = ['v_R0_a_r', 'v_R0_a_i', 'v_R0_b_r', 'v_R0_b_i', 'v_R0_c_r', 'v_R0_c_i', 'i_R1_n_r', 'i_R1_n_i', 'i_R11_n_r', 'i_R11_n_i', 'i_R15_n_r', 'i_R15_n_i', 'i_R16_n_r', 'i_R16_n_i', 'i_R17_n_r', 'i_R17_n_i', 'i_R18_n_r', 'i_R18_n_i', 'i_R2_a_r', 'i_R2_a_i', 'i_R2_b_r', 'i_R2_b_i', 'i_R2_c_r', 'i_R2_c_i', 'i_R2_n_r', 'i_R2_n_i', 'i_R3_a_r', 'i_R3_a_i', 'i_R3_b_r', 'i_R3_b_i', 'i_R3_c_r', 'i_R3_c_i', 'i_R3_n_r', 'i_R3_n_i', 'i_R4_a_r', 'i_R4_a_i', 'i_R4_b_r', 'i_R4_b_i', 'i_R4_c_r', 'i_R4_c_i', 'i_R4_n_r', 'i_R4_n_i', 'i_R5_a_r', 'i_R5_a_i', 'i_R5_b_r', 'i_R5_b_i', 'i_R5_c_r', 'i_R5_c_i', 'i_R5_n_r', 'i_R5_n_i', 'i_R6_a_r', 'i_R6_a_i', 'i_R6_b_r', 'i_R6_b_i', 'i_R6_c_r', 'i_R6_c_i', 'i_R6_n_r', 'i_R6_n_i', 'i_R7_a_r', 'i_R7_a_i', 'i_R7_b_r', 'i_R7_b_i', 'i_R7_c_r', 'i_R7_c_i', 'i_R7_n_r', 'i_R7_n_i', 'i_R8_a_r', 'i_R8_a_i', 'i_R8_b_r', 'i_R8_b_i', 'i_R8_c_r', 'i_R8_c_i', 'i_R8_n_r', 'i_R8_n_i', 'i_R9_a_r', 'i_R9_a_i', 'i_R9_b_r', 'i_R9_b_i', 'i_R9_c_r', 'i_R9_c_i', 'i_R9_n_r', 'i_R9_n_i', 'i_R10_a_r', 'i_R10_a_i', 'i_R10_b_r', 'i_R10_b_i', 'i_R10_c_r', 'i_R10_c_i', 'i_R10_n_r', 'i_R10_n_i', 'i_R12_a_r', 'i_R12_a_i', 'i_R12_b_r', 'i_R12_b_i', 'i_R12_c_r', 'i_R12_c_i', 'i_R12_n_r', 'i_R12_n_i', 'i_R13_a_r', 'i_R13_a_i', 'i_R13_b_r', 'i_R13_b_i', 'i_R13_c_r', 'i_R13_c_i', 'i_R13_n_r', 'i_R13_n_i', 'i_R14_a_r', 'i_R14_a_i', 'i_R14_b_r', 'i_R14_b_i', 'i_R14_c_r', 'i_R14_c_i', 'i_R14_n_r', 'i_R14_n_i', 'p_R1_a', 'q_R1_a', 'p_R1_b', 'q_R1_b', 'p_R1_c', 'q_R1_c', 'p_R11_a', 'q_R11_a', 'p_R11_b', 'q_R11_b', 'p_R11_c', 'q_R11_c', 'p_R15_a', 'q_R15_a', 'p_R15_b', 'q_R15_b', 'p_R15_c', 'q_R15_c', 'p_R16_a', 'q_R16_a', 'p_R16_b', 'q_R16_b', 'p_R16_c', 'q_R16_c', 'p_R17_a', 'q_R17_a', 'p_R17_b', 'q_R17_b', 'p_R17_c', 'q_R17_c', 'p_R18_a', 'q_R18_a', 'p_R18_b', 'q_R18_b', 'p_R18_c', 'q_R18_c', 'u_dummy'] 
        self.inputs_run_values_list = [11547.0, 0.0, -5773.499999999997, -9999.995337498915, -5773.5000000000055, 9999.99533749891, -0.1137718099519418, -0.062435917080051695, 0.038921773016749484, -0.10489637569849819, 0.3445405949857374, -0.7980789653137919, 0.37357657074305806, -0.8707524289912385, 0.3054824958108071, -0.699068183990839, 0.4382865505353877, -0.9973969673568632, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -63333.33333323794, -20816.65999458629, -63333.33333316128, -20816.659994702903, -63333.33333361338, -20816.659994702255, -4749.999999991674, -1561.2494996136027, -4749.999999961012, -1561.2494996070764, -4750.000000051293, -1561.2494995734658, -16466.666666612546, -5412.331598952091, -16466.666666233108, -5412.331598746236, -16466.666667180372, -5412.331598103374, -17416.666666636127, -5724.581498685594, -17416.66666639193, -5724.581498594195, -17416.666666998393, -5724.581498278923, -11083.333333311602, -3642.9154991873775, -11083.333333126331, -3642.915499108254, -11083.333333581077, -3642.91549887427, -14883.333333304252, -4891.915098923226, -14883.333333039369, -4891.915098810083, -14883.333333685177, -4891.915098464485, 1.0] 
        self.outputs_list = ['v_R0_a_m', 'v_R0_b_m', 'v_R0_c_m', 'v_R1_a_m', 'v_R1_b_m', 'v_R1_c_m', 'v_R1_n_m', 'v_R11_a_m', 'v_R11_b_m', 'v_R11_c_m', 'v_R11_n_m', 'v_R15_a_m', 'v_R15_b_m', 'v_R15_c_m', 'v_R15_n_m', 'v_R16_a_m', 'v_R16_b_m', 'v_R16_c_m', 'v_R16_n_m', 'v_R17_a_m', 'v_R17_b_m', 'v_R17_c_m', 'v_R17_n_m', 'v_R18_a_m', 'v_R18_b_m', 'v_R18_c_m', 'v_R18_n_m', 'v_R2_a_m', 'v_R2_b_m', 'v_R2_c_m', 'v_R2_n_m', 'v_R3_a_m', 'v_R3_b_m', 'v_R3_c_m', 'v_R3_n_m', 'v_R4_a_m', 'v_R4_b_m', 'v_R4_c_m', 'v_R4_n_m', 'v_R5_a_m', 'v_R5_b_m', 'v_R5_c_m', 'v_R5_n_m', 'v_R6_a_m', 'v_R6_b_m', 'v_R6_c_m', 'v_R6_n_m', 'v_R7_a_m', 'v_R7_b_m', 'v_R7_c_m', 'v_R7_n_m', 'v_R8_a_m', 'v_R8_b_m', 'v_R8_c_m', 'v_R8_n_m', 'v_R9_a_m', 'v_R9_b_m', 'v_R9_c_m', 'v_R9_n_m', 'v_R10_a_m', 'v_R10_b_m', 'v_R10_c_m', 'v_R10_n_m', 'v_R12_a_m', 'v_R12_b_m', 'v_R12_c_m', 'v_R12_n_m', 'v_R13_a_m', 'v_R13_b_m', 'v_R13_c_m', 'v_R13_n_m', 'v_R14_a_m', 'v_R14_b_m', 'v_R14_c_m', 'v_R14_n_m'] 
        self.x_list = ['x_dummy'] 
        self.y_run_list = ['v_R1_a_r', 'v_R1_a_i', 'v_R1_b_r', 'v_R1_b_i', 'v_R1_c_r', 'v_R1_c_i', 'v_R1_n_r', 'v_R1_n_i', 'v_R11_a_r', 'v_R11_a_i', 'v_R11_b_r', 'v_R11_b_i', 'v_R11_c_r', 'v_R11_c_i', 'v_R11_n_r', 'v_R11_n_i', 'v_R15_a_r', 'v_R15_a_i', 'v_R15_b_r', 'v_R15_b_i', 'v_R15_c_r', 'v_R15_c_i', 'v_R15_n_r', 'v_R15_n_i', 'v_R16_a_r', 'v_R16_a_i', 'v_R16_b_r', 'v_R16_b_i', 'v_R16_c_r', 'v_R16_c_i', 'v_R16_n_r', 'v_R16_n_i', 'v_R17_a_r', 'v_R17_a_i', 'v_R17_b_r', 'v_R17_b_i', 'v_R17_c_r', 'v_R17_c_i', 'v_R17_n_r', 'v_R17_n_i', 'v_R18_a_r', 'v_R18_a_i', 'v_R18_b_r', 'v_R18_b_i', 'v_R18_c_r', 'v_R18_c_i', 'v_R18_n_r', 'v_R18_n_i', 'v_R2_a_r', 'v_R2_a_i', 'v_R2_b_r', 'v_R2_b_i', 'v_R2_c_r', 'v_R2_c_i', 'v_R2_n_r', 'v_R2_n_i', 'v_R3_a_r', 'v_R3_a_i', 'v_R3_b_r', 'v_R3_b_i', 'v_R3_c_r', 'v_R3_c_i', 'v_R3_n_r', 'v_R3_n_i', 'v_R4_a_r', 'v_R4_a_i', 'v_R4_b_r', 'v_R4_b_i', 'v_R4_c_r', 'v_R4_c_i', 'v_R4_n_r', 'v_R4_n_i', 'v_R5_a_r', 'v_R5_a_i', 'v_R5_b_r', 'v_R5_b_i', 'v_R5_c_r', 'v_R5_c_i', 'v_R5_n_r', 'v_R5_n_i', 'v_R6_a_r', 'v_R6_a_i', 'v_R6_b_r', 'v_R6_b_i', 'v_R6_c_r', 'v_R6_c_i', 'v_R6_n_r', 'v_R6_n_i', 'v_R7_a_r', 'v_R7_a_i', 'v_R7_b_r', 'v_R7_b_i', 'v_R7_c_r', 'v_R7_c_i', 'v_R7_n_r', 'v_R7_n_i', 'v_R8_a_r', 'v_R8_a_i', 'v_R8_b_r', 'v_R8_b_i', 'v_R8_c_r', 'v_R8_c_i', 'v_R8_n_r', 'v_R8_n_i', 'v_R9_a_r', 'v_R9_a_i', 'v_R9_b_r', 'v_R9_b_i', 'v_R9_c_r', 'v_R9_c_i', 'v_R9_n_r', 'v_R9_n_i', 'v_R10_a_r', 'v_R10_a_i', 'v_R10_b_r', 'v_R10_b_i', 'v_R10_c_r', 'v_R10_c_i', 'v_R10_n_r', 'v_R10_n_i', 'v_R12_a_r', 'v_R12_a_i', 'v_R12_b_r', 'v_R12_b_i', 'v_R12_c_r', 'v_R12_c_i', 'v_R12_n_r', 'v_R12_n_i', 'v_R13_a_r', 'v_R13_a_i', 'v_R13_b_r', 'v_R13_b_i', 'v_R13_c_r', 'v_R13_c_i', 'v_R13_n_r', 'v_R13_n_i', 'v_R14_a_r', 'v_R14_a_i', 'v_R14_b_r', 'v_R14_b_i', 'v_R14_c_r', 'v_R14_c_i', 'v_R14_n_r', 'v_R14_n_i', 'i_R1_a_r', 'i_R1_a_i', 'i_R1_b_r', 'i_R1_b_i', 'i_R1_c_r', 'i_R1_c_i', 'i_R1_n_r', 'i_R1_n_i', 'i_R11_a_r', 'i_R11_a_i', 'i_R11_b_r', 'i_R11_b_i', 'i_R11_c_r', 'i_R11_c_i', 'i_R11_n_r', 'i_R11_n_i', 'i_R15_a_r', 'i_R15_a_i', 'i_R15_b_r', 'i_R15_b_i', 'i_R15_c_r', 'i_R15_c_i', 'i_R15_n_r', 'i_R15_n_i', 'i_R16_a_r', 'i_R16_a_i', 'i_R16_b_r', 'i_R16_b_i', 'i_R16_c_r', 'i_R16_c_i', 'i_R16_n_r', 'i_R16_n_i', 'i_R17_a_r', 'i_R17_a_i', 'i_R17_b_r', 'i_R17_b_i', 'i_R17_c_r', 'i_R17_c_i', 'i_R17_n_r', 'i_R17_n_i', 'i_R18_a_r', 'i_R18_a_i', 'i_R18_b_r', 'i_R18_b_i', 'i_R18_c_r', 'i_R18_c_i', 'i_R18_n_r', 'i_R18_n_i'] 
        self.xy_list = self.x_list + self.y_run_list 
        self.y_ini_list = ['v_R1_a_r', 'v_R1_a_i', 'v_R1_b_r', 'v_R1_b_i', 'v_R1_c_r', 'v_R1_c_i', 'v_R1_n_r', 'v_R1_n_i', 'v_R11_a_r', 'v_R11_a_i', 'v_R11_b_r', 'v_R11_b_i', 'v_R11_c_r', 'v_R11_c_i', 'v_R11_n_r', 'v_R11_n_i', 'v_R15_a_r', 'v_R15_a_i', 'v_R15_b_r', 'v_R15_b_i', 'v_R15_c_r', 'v_R15_c_i', 'v_R15_n_r', 'v_R15_n_i', 'v_R16_a_r', 'v_R16_a_i', 'v_R16_b_r', 'v_R16_b_i', 'v_R16_c_r', 'v_R16_c_i', 'v_R16_n_r', 'v_R16_n_i', 'v_R17_a_r', 'v_R17_a_i', 'v_R17_b_r', 'v_R17_b_i', 'v_R17_c_r', 'v_R17_c_i', 'v_R17_n_r', 'v_R17_n_i', 'v_R18_a_r', 'v_R18_a_i', 'v_R18_b_r', 'v_R18_b_i', 'v_R18_c_r', 'v_R18_c_i', 'v_R18_n_r', 'v_R18_n_i', 'v_R2_a_r', 'v_R2_a_i', 'v_R2_b_r', 'v_R2_b_i', 'v_R2_c_r', 'v_R2_c_i', 'v_R2_n_r', 'v_R2_n_i', 'v_R3_a_r', 'v_R3_a_i', 'v_R3_b_r', 'v_R3_b_i', 'v_R3_c_r', 'v_R3_c_i', 'v_R3_n_r', 'v_R3_n_i', 'v_R4_a_r', 'v_R4_a_i', 'v_R4_b_r', 'v_R4_b_i', 'v_R4_c_r', 'v_R4_c_i', 'v_R4_n_r', 'v_R4_n_i', 'v_R5_a_r', 'v_R5_a_i', 'v_R5_b_r', 'v_R5_b_i', 'v_R5_c_r', 'v_R5_c_i', 'v_R5_n_r', 'v_R5_n_i', 'v_R6_a_r', 'v_R6_a_i', 'v_R6_b_r', 'v_R6_b_i', 'v_R6_c_r', 'v_R6_c_i', 'v_R6_n_r', 'v_R6_n_i', 'v_R7_a_r', 'v_R7_a_i', 'v_R7_b_r', 'v_R7_b_i', 'v_R7_c_r', 'v_R7_c_i', 'v_R7_n_r', 'v_R7_n_i', 'v_R8_a_r', 'v_R8_a_i', 'v_R8_b_r', 'v_R8_b_i', 'v_R8_c_r', 'v_R8_c_i', 'v_R8_n_r', 'v_R8_n_i', 'v_R9_a_r', 'v_R9_a_i', 'v_R9_b_r', 'v_R9_b_i', 'v_R9_c_r', 'v_R9_c_i', 'v_R9_n_r', 'v_R9_n_i', 'v_R10_a_r', 'v_R10_a_i', 'v_R10_b_r', 'v_R10_b_i', 'v_R10_c_r', 'v_R10_c_i', 'v_R10_n_r', 'v_R10_n_i', 'v_R12_a_r', 'v_R12_a_i', 'v_R12_b_r', 'v_R12_b_i', 'v_R12_c_r', 'v_R12_c_i', 'v_R12_n_r', 'v_R12_n_i', 'v_R13_a_r', 'v_R13_a_i', 'v_R13_b_r', 'v_R13_b_i', 'v_R13_c_r', 'v_R13_c_i', 'v_R13_n_r', 'v_R13_n_i', 'v_R14_a_r', 'v_R14_a_i', 'v_R14_b_r', 'v_R14_b_i', 'v_R14_c_r', 'v_R14_c_i', 'v_R14_n_r', 'v_R14_n_i', 'i_R1_a_r', 'i_R1_a_i', 'i_R1_b_r', 'i_R1_b_i', 'i_R1_c_r', 'i_R1_c_i', 'i_R1_n_r', 'i_R1_n_i', 'i_R11_a_r', 'i_R11_a_i', 'i_R11_b_r', 'i_R11_b_i', 'i_R11_c_r', 'i_R11_c_i', 'i_R11_n_r', 'i_R11_n_i', 'i_R15_a_r', 'i_R15_a_i', 'i_R15_b_r', 'i_R15_b_i', 'i_R15_c_r', 'i_R15_c_i', 'i_R15_n_r', 'i_R15_n_i', 'i_R16_a_r', 'i_R16_a_i', 'i_R16_b_r', 'i_R16_b_i', 'i_R16_c_r', 'i_R16_c_i', 'i_R16_n_r', 'i_R16_n_i', 'i_R17_a_r', 'i_R17_a_i', 'i_R17_b_r', 'i_R17_b_i', 'i_R17_c_r', 'i_R17_c_i', 'i_R17_n_r', 'i_R17_n_i', 'i_R18_a_r', 'i_R18_a_i', 'i_R18_b_r', 'i_R18_b_i', 'i_R18_c_r', 'i_R18_c_i', 'i_R18_n_r', 'i_R18_n_i'] 
        self.xy_ini_list = self.x_list + self.y_ini_list 
        self.t = 0.0
        self.it = 0
        self.it_store = 0
        self.xy_prev = np.zeros((self.N_x+self.N_y,1))
        self.initialization_tol = 1e-6
        self.N_u = len(self.inputs_run_list) 
        self.sopt_root_method='hybr'
        self.sopt_root_jac=True
        self.u_ini_list = self.inputs_ini_list
        self.u_ini_values_list = self.inputs_ini_values_list
        self.u_run_list = self.inputs_run_list
        self.u_run_values_list = self.inputs_run_values_list
        self.N_u = len(self.u_run_list)
        self.update() 


    def update(self): 

        self.N_steps = int(np.ceil(self.t_end/self.Dt)) 
        dt = [  
              ('t_end', np.float64),
              ('Dt', np.float64),
              ('decimation', np.float64),
              ('itol', np.float64),
              ('Dt_max', np.float64),
              ('Dt_min', np.float64),
              ('solvern', np.int64),
              ('imax', np.int64),
              ('N_steps', np.int64),
              ('N_store', np.int64),
              ('N_x', np.int64),
              ('N_y', np.int64),
              ('N_z', np.int64),
              ('t', np.float64),
              ('it', np.int64),
              ('it_store', np.int64),
              ('idx', np.int64),
              ('idy', np.int64),
              ('f', np.float64, (self.N_x,1)),
              ('x', np.float64, (self.N_x,1)),
              ('x_0', np.float64, (self.N_x,1)),
              ('g', np.float64, (self.N_y,1)),
              ('y_run', np.float64, (self.N_y,1)),
              ('y_ini', np.float64, (self.N_y,1)),
              ('u_run', np.float64, (self.N_u,1)),
              ('y_0', np.float64, (self.N_y,1)),
              ('h', np.float64, (self.N_z,1)),
              ('Fx', np.float64, (self.N_x,self.N_x)),
              ('Fy', np.float64, (self.N_x,self.N_y)),
              ('Gx', np.float64, (self.N_y,self.N_x)),
              ('Gy', np.float64, (self.N_y,self.N_y)),
              ('Fu', np.float64, (self.N_x,self.N_u)),
              ('Gu', np.float64, (self.N_y,self.N_u)),
              ('Hx', np.float64, (self.N_z,self.N_x)),
              ('Hy', np.float64, (self.N_z,self.N_y)),
              ('Hu', np.float64, (self.N_z,self.N_u)),
              ('Fx_ini', np.float64, (self.N_x,self.N_x)),
              ('Fy_ini', np.float64, (self.N_x,self.N_y)),
              ('Gx_ini', np.float64, (self.N_y,self.N_x)),
              ('Gy_ini', np.float64, (self.N_y,self.N_y)),
              ('T', np.float64, (self.N_store+1,1)),
              ('X', np.float64, (self.N_store+1,self.N_x)),
              ('Y', np.float64, (self.N_store+1,self.N_y)),
              ('Z', np.float64, (self.N_store+1,self.N_z)),
              ('iters', np.float64, (self.N_store+1,1)),
              ('store', np.int64),
             ]

        values = [
                self.t_end,                          
                self.Dt,
                self.decimation,
                self.itol,
                self.Dt_max,
                self.Dt_min,
                self.solvern,
                self.imax,
                self.N_steps,
                self.N_store,
                self.N_x,
                self.N_y,
                self.N_z,
                self.t,
                self.it,
                self.it_store,
                0,                                     # idx
                0,                                     # idy
                np.zeros((self.N_x,1)),                # f
                np.zeros((self.N_x,1)),                # x
                np.zeros((self.N_x,1)),                # x_0
                np.zeros((self.N_y,1)),                # g
                np.zeros((self.N_y,1)),                # y_run
                np.zeros((self.N_y,1)),                # y_ini
                np.zeros((self.N_u,1)),                # u_run
                np.zeros((self.N_y,1)),                # y_0
                np.zeros((self.N_z,1)),                # h
                np.zeros((self.N_x,self.N_x)),         # Fx   
                np.zeros((self.N_x,self.N_y)),         # Fy 
                np.zeros((self.N_y,self.N_x)),         # Gx 
                np.zeros((self.N_y,self.N_y)),         # Fy
                np.zeros((self.N_x,self.N_u)),         # Fu 
                np.zeros((self.N_y,self.N_u)),         # Gu 
                np.zeros((self.N_z,self.N_x)),         # Hx 
                np.zeros((self.N_z,self.N_y)),         # Hy 
                np.zeros((self.N_z,self.N_u)),         # Hu 
                np.zeros((self.N_x,self.N_x)),         # Fx_ini  
                np.zeros((self.N_x,self.N_y)),         # Fy_ini 
                np.zeros((self.N_y,self.N_x)),         # Gx_ini 
                np.zeros((self.N_y,self.N_y)),         # Fy_ini 
                np.zeros((self.N_store+1,1)),          # T
                np.zeros((self.N_store+1,self.N_x)),   # X
                np.zeros((self.N_store+1,self.N_y)),   # Y
                np.zeros((self.N_store+1,self.N_z)),   # Z
                np.zeros((self.N_store+1,1)),          # iters
                1,
                ]  

        dt += [(item,np.float64) for item in self.params_list]
        values += [item for item in self.params_values_list]

        for item_id,item_val in zip(self.inputs_ini_list,self.inputs_ini_values_list):
            if item_id in self.inputs_run_list: continue
            dt += [(item_id,np.float64)]
            values += [item_val]

        dt += [(item,np.float64) for item in self.inputs_run_list]
        values += [item for item in self.inputs_run_values_list]

        self.struct = np.rec.array([tuple(values)], dtype=np.dtype(dt))
        
        xy0 = np.zeros((self.N_x+self.N_y,))
        self.ini_dae_jacobian_nn(xy0)
        self.run_dae_jacobian_nn(xy0)

    def load_params(self,data_input):

        if type(data_input) == str:
            json_file = data_input
            self.json_file = json_file
            self.json_data = open(json_file).read().replace("'",'"')
            data = json.loads(self.json_data)
        elif type(data_input) == dict:
            data = data_input

        self.data = data
        for item in self.data:
            self.struct[0][item] = self.data[item]
            self.params_values_list[self.params_list.index(item)] = self.data[item]



    def ini_problem(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_ini[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        if self.compile:
            ini(self.struct,2)
            ini(self.struct,3)       
        else:
            ini.py_func(self.struct,2)
            ini.py_func(self.struct,3)                   
        fg = np.vstack((self.struct[0].f,self.struct[0].g))[:,0]
        return fg

    def run_problem(self,x):
        t = self.struct[0].t
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_run[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        
        if self.compile:
            run(t,self.struct,2)
            run(t,self.struct,3)
            run(t,self.struct,10)
            run(t,self.struct,11)
            run(t,self.struct,12)
            run(t,self.struct,13)
        else:
            run.py_func(t,self.struct,2)
            run.py_func(t,self.struct,3)
            run.py_func(t,self.struct,10)
            run.py_func(t,self.struct,11)
            run.py_func(t,self.struct,12)
            run.py_func(t,self.struct,13)            
        
        fg = np.vstack((self.struct[0].f,self.struct[0].g))[:,0]
        return fg
    

    def run_dae_jacobian(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_run[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        run(0.0,self.struct,10)
        run(0.0,self.struct,11)     
        run(0.0,self.struct,12)
        run(0.0,self.struct,13)
        A_c = np.block([[self.struct[0].Fx,self.struct[0].Fy],
                        [self.struct[0].Gx,self.struct[0].Gy]])
        return A_c

    def run_dae_jacobian_nn(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_run[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        run_nn(0.0,self.struct,10)
        run_nn(0.0,self.struct,11)     
        run_nn(0.0,self.struct,12)
        run_nn(0.0,self.struct,13)
 

    
    def eval_jacobians(self):

        run(0.0,self.struct,10)
        run(0.0,self.struct,11)  
        run(0.0,self.struct,12) 

        return 1


    def ini_dae_jacobian(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_ini[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        if self.compile:
            ini(self.struct,10)
            ini(self.struct,11) 
        else:
            ini.py_func(self.struct,10)
            ini.py_func(self.struct,11)             
        A_c = np.block([[self.struct[0].Fx_ini,self.struct[0].Fy_ini],
                        [self.struct[0].Gx_ini,self.struct[0].Gy_ini]])
        return A_c

    def ini_dae_jacobian_nn(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_ini[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        ini_nn(self.struct,10)
        ini_nn(self.struct,11)       
 

    def f_ode(self,x):
        self.struct[0].x[:,0] = x
        run(self.struct,1)
        return self.struct[0].f[:,0]

    def f_odeint(self,x,t):
        self.struct[0].x[:,0] = x
        run(self.struct,1)
        return self.struct[0].f[:,0]

    def f_ivp(self,t,x):
        self.struct[0].x[:,0] = x
        run(self.struct,1)
        return self.struct[0].f[:,0]

    def Fx_ode(self,x):
        self.struct[0].x[:,0] = x
        run(self.struct,10)
        return self.struct[0].Fx

    def eval_A(self):
        
        Fx = self.struct[0].Fx
        Fy = self.struct[0].Fy
        Gx = self.struct[0].Gx
        Gy = self.struct[0].Gy
        
        A = Fx - Fy @ np.linalg.solve(Gy,Gx)
        
        self.A = A
        
        return A

    def eval_A_ini(self):
        
        Fx = self.struct[0].Fx_ini
        Fy = self.struct[0].Fy_ini
        Gx = self.struct[0].Gx_ini
        Gy = self.struct[0].Gy_ini
        
        A = Fx - Fy @ np.linalg.solve(Gy,Gx)
        
        
        return A
    
    def reset(self):
        for param,param_value in zip(self.params_list,self.params_values_list):
            self.struct[0][param] = param_value
        for input_name,input_value in zip(self.inputs_ini_list,self.inputs_ini_values_list):
            self.struct[0][input_name] = input_value   
        for input_name,input_value in zip(self.inputs_run_list,self.inputs_run_values_list):
            self.struct[0][input_name] = input_value  

    def simulate(self,events,xy0=0):
        
        # initialize both the ini and the run system
        self.initialize(events,xy0=xy0)
        
        # simulation run
        for event in events:  
            # make all the desired changes
            self.run([event]) 
            
        # post process
        T,X,Y,Z = self.post()
        
        return T,X,Y,Z
    

    
    def run(self,events):
        

        # simulation run
        for event in events:  
            # make all the desired changes
            for item in event:
                self.struct[0][item] = event[item]
            daesolver(self.struct)    # run until next event
            
        return 1
 
    def rtrun(self,events):
        

        # simulation run
        for event in events:  
            # make all the desired changes
            for item in event:
                self.struct[0][item] = event[item]
            self.struct[0].it_store = self.struct[0].N_store-1
            daesolver(self.struct)    # run until next event
            
            
        return 1
    
    def post(self):
        
        # post process result    
        T = self.struct[0]['T'][:self.struct[0].it_store]
        X = self.struct[0]['X'][:self.struct[0].it_store,:]
        Y = self.struct[0]['Y'][:self.struct[0].it_store,:]
        Z = self.struct[0]['Z'][:self.struct[0].it_store,:]
        iters = self.struct[0]['iters'][:self.struct[0].it_store,:]
    
        self.T = T
        self.X = X
        self.Y = Y
        self.Z = Z
        self.iters = iters
        
        return T,X,Y,Z
        
    def save_0(self,file_name = 'xy_0.json'):
        xy_0_dict = {}
        for item in self.x_list:
            xy_0_dict.update({item:self.get_value(item)})
        for item in self.y_ini_list:
            xy_0_dict.update({item:self.get_value(item)})
    
        xy_0_str = json.dumps(xy_0_dict, indent=4)
        with open(file_name,'w') as fobj:
            fobj.write(xy_0_str)

    def load_0(self,file_name = 'xy_0.json'):
        with open(file_name) as fobj:
            xy_0_str = fobj.read()
        xy_0_dict = json.loads(xy_0_str)
    
        for item in xy_0_dict:
            if item in self.x_list:
                self.xy_prev[self.x_list.index(item)] = xy_0_dict[item]
            if item in self.y_ini_list:
                self.xy_prev[self.y_ini_list.index(item)+self.N_x] = xy_0_dict[item]
                
            
    def initialize(self,events=[{}],xy0=0,compile=True):
        '''
        

        Parameters
        ----------
        events : dictionary 
            Dictionary with at least 't_end' and all inputs and parameters 
            that need to be changed.
        xy0 : float or string, optional
            0 means all states should be zero as initial guess. 
            If not zero all the states initial guess are the given input.
            If 'prev' it uses the last known initialization result as initial guess.

        Returns
        -------
        T : TYPE
            DESCRIPTION.
        X : TYPE
            DESCRIPTION.
        Y : TYPE
            DESCRIPTION.
        Z : TYPE
            DESCRIPTION.

        '''
        
        self.compile = compile
        
        # simulation parameters
        self.struct[0].it = 0       # set time step to zero
        self.struct[0].it_store = 0 # set storage to zero
        self.struct[0].t = 0.0      # set time to zero
                    
        # initialization
        it_event = 0
        event = events[it_event]
        for item in event:
            self.struct[0][item] = event[item]
            
        
        ## compute initial conditions using x and y_ini 
        if type(xy0) == str:
            if xy0 == 'prev':
                xy0 = self.xy_prev
            else:
                self.load_0(xy0)
                xy0 = self.xy_prev
        elif type(xy0) == dict:
            with open('xy_0.json','w') as fobj:
                fobj.write(json.dumps(xy0))
            self.load_0('xy_0.json')
            xy0 = self.xy_prev            
        else:
            if xy0 == 0:
                xy0 = np.zeros(self.N_x+self.N_y)
            elif xy0 == 1:
                xy0 = np.ones(self.N_x+self.N_y)
            else:
                xy0 = xy0*np.ones(self.N_x+self.N_y)

        #xy = sopt.fsolve(self.ini_problem,xy0, jac=self.ini_dae_jacobian )

        
        if self.sopt_root_jac:
            sol = sopt.root(self.ini_problem, xy0, 
                            jac=self.ini_dae_jacobian, 
                            method=self.sopt_root_method, tol=self.initialization_tol)
        else:
            sol = sopt.root(self.ini_problem, xy0, method=self.sopt_root_method)

        self.initialization_ok = True
        if sol.success == False:
            print('initialization not found!')
            self.initialization_ok = False

            T = self.struct[0]['T'][:self.struct[0].it_store]
            X = self.struct[0]['X'][:self.struct[0].it_store,:]
            Y = self.struct[0]['Y'][:self.struct[0].it_store,:]
            Z = self.struct[0]['Z'][:self.struct[0].it_store,:]
            iters = self.struct[0]['iters'][:self.struct[0].it_store,:]

        if self.initialization_ok:
            xy = sol.x
            self.xy_prev = xy
            self.struct[0].x[:,0] = xy[0:self.N_x]
            self.struct[0].y_run[:,0] = xy[self.N_x:]

            ## y_ini to u_run
            for item in self.inputs_run_list:
                if item in self.y_ini_list:
                    self.struct[0][item] = self.struct[0].y_ini[self.y_ini_list.index(item)]

            ## u_ini to y_run
            for item in self.inputs_ini_list:
                if item in self.y_run_list:
                    self.struct[0].y_run[self.y_run_list.index(item)] = self.struct[0][item]


            #xy = sopt.fsolve(self.ini_problem,xy0, jac=self.ini_dae_jacobian )
            if self.sopt_root_jac:
                sol = sopt.root(self.run_problem, xy0, 
                                jac=self.run_dae_jacobian, 
                                method=self.sopt_root_method, tol=self.initialization_tol)
            else:
                sol = sopt.root(self.run_problem, xy0, method=self.sopt_root_method)

            if self.compile:
                # evaluate f and g
                run(0.0,self.struct,2)
                run(0.0,self.struct,3)                
    
                # evaluate run jacobians 
                run(0.0,self.struct,10)
                run(0.0,self.struct,11)                
                run(0.0,self.struct,12) 
                run(0.0,self.struct,14) 
                
            else:
                # evaluate f and g
                run.py_func(0.0,self.struct,2)
                run.py_func(0.0,self.struct,3)                
    
                # evaluate run jacobians 
                run.py_func(0.0,self.struct,10)
                run.py_func(0.0,self.struct,11)                
                run.py_func(0.0,self.struct,12) 
                run.py_func(0.0,self.struct,14)                 
                
             
            # post process result    
            T = self.struct[0]['T'][:self.struct[0].it_store]
            X = self.struct[0]['X'][:self.struct[0].it_store,:]
            Y = self.struct[0]['Y'][:self.struct[0].it_store,:]
            Z = self.struct[0]['Z'][:self.struct[0].it_store,:]
            iters = self.struct[0]['iters'][:self.struct[0].it_store,:]
        
            self.T = T
            self.X = X
            self.Y = Y
            self.Z = Z
            self.iters = iters
            
        return self.initialization_ok
    
    
    def get_value(self,name):
        if name in self.inputs_run_list:
            value = self.struct[0][name]
        if name in self.x_list:
            idx = self.x_list.index(name)
            value = self.struct[0].x[idx,0]
        if name in self.y_run_list:
            idy = self.y_run_list.index(name)
            value = self.struct[0].y_run[idy,0]
        if name in self.params_list:
            value = self.struct[0][name]
        if name in self.outputs_list:
            value = self.struct[0].h[self.outputs_list.index(name),0] 

        return value
    
    def get_values(self,name):
        if name in self.x_list:
            values = self.X[:,self.x_list.index(name)]
        if name in self.y_run_list:
            values = self.Y[:,self.y_run_list.index(name)]
        if name in self.outputs_list:
            values = self.Z[:,self.outputs_list.index(name)]
                        
        return values

    def get_mvalue(self,names):
        '''

        Parameters
        ----------
        names : list
            list of variables names to return each value.

        Returns
        -------
        mvalue : TYPE
            list of value of each variable.

        '''
        mvalue = []
        for name in names:
            mvalue += [self.get_value(name)]
                        
        return mvalue
    
    def set_value(self,name,value):
        if name in self.inputs_run_list:
            self.struct[0][name] = value
        if name in self.params_list:
            self.struct[0][name] = value
            
    def report_x(self,value_format='5.2f'):
        for item in self.x_list:
            print(f'{item:5s} = {self.get_value(item):5.2f}')

    def report_y(self,value_format='5.2f'):
        for item in self.y_run_list:
            print(f'{item:5s} = {self.get_value(item):5.2f}')
            
    def report_u(self,value_format='5.2f'):
        for item in self.inputs_run_list:
            print(f'{item:5s} = {self.get_value(item):5.2f}')

    def report_z(self,value_format='5.2f'):
        for item in self.outputs_list:
            print(f'{item:5s} = {self.get_value(item):5.2f}')

    def report_params(self,value_format='5.2f'):
        for item in self.params_list:
            print(f'{item:5s} = {self.get_value(item):5.2f}')
            
    def get_x(self):
        return self.struct[0].x


@numba.njit(cache=True)
def ini(struct,mode):

    # Parameters:
    
    # Inputs:
    v_R0_a_r = struct[0].v_R0_a_r
    v_R0_a_i = struct[0].v_R0_a_i
    v_R0_b_r = struct[0].v_R0_b_r
    v_R0_b_i = struct[0].v_R0_b_i
    v_R0_c_r = struct[0].v_R0_c_r
    v_R0_c_i = struct[0].v_R0_c_i
    i_R1_n_r = struct[0].i_R1_n_r
    i_R1_n_i = struct[0].i_R1_n_i
    i_R11_n_r = struct[0].i_R11_n_r
    i_R11_n_i = struct[0].i_R11_n_i
    i_R15_n_r = struct[0].i_R15_n_r
    i_R15_n_i = struct[0].i_R15_n_i
    i_R16_n_r = struct[0].i_R16_n_r
    i_R16_n_i = struct[0].i_R16_n_i
    i_R17_n_r = struct[0].i_R17_n_r
    i_R17_n_i = struct[0].i_R17_n_i
    i_R18_n_r = struct[0].i_R18_n_r
    i_R18_n_i = struct[0].i_R18_n_i
    i_R2_a_r = struct[0].i_R2_a_r
    i_R2_a_i = struct[0].i_R2_a_i
    i_R2_b_r = struct[0].i_R2_b_r
    i_R2_b_i = struct[0].i_R2_b_i
    i_R2_c_r = struct[0].i_R2_c_r
    i_R2_c_i = struct[0].i_R2_c_i
    i_R2_n_r = struct[0].i_R2_n_r
    i_R2_n_i = struct[0].i_R2_n_i
    i_R3_a_r = struct[0].i_R3_a_r
    i_R3_a_i = struct[0].i_R3_a_i
    i_R3_b_r = struct[0].i_R3_b_r
    i_R3_b_i = struct[0].i_R3_b_i
    i_R3_c_r = struct[0].i_R3_c_r
    i_R3_c_i = struct[0].i_R3_c_i
    i_R3_n_r = struct[0].i_R3_n_r
    i_R3_n_i = struct[0].i_R3_n_i
    i_R4_a_r = struct[0].i_R4_a_r
    i_R4_a_i = struct[0].i_R4_a_i
    i_R4_b_r = struct[0].i_R4_b_r
    i_R4_b_i = struct[0].i_R4_b_i
    i_R4_c_r = struct[0].i_R4_c_r
    i_R4_c_i = struct[0].i_R4_c_i
    i_R4_n_r = struct[0].i_R4_n_r
    i_R4_n_i = struct[0].i_R4_n_i
    i_R5_a_r = struct[0].i_R5_a_r
    i_R5_a_i = struct[0].i_R5_a_i
    i_R5_b_r = struct[0].i_R5_b_r
    i_R5_b_i = struct[0].i_R5_b_i
    i_R5_c_r = struct[0].i_R5_c_r
    i_R5_c_i = struct[0].i_R5_c_i
    i_R5_n_r = struct[0].i_R5_n_r
    i_R5_n_i = struct[0].i_R5_n_i
    i_R6_a_r = struct[0].i_R6_a_r
    i_R6_a_i = struct[0].i_R6_a_i
    i_R6_b_r = struct[0].i_R6_b_r
    i_R6_b_i = struct[0].i_R6_b_i
    i_R6_c_r = struct[0].i_R6_c_r
    i_R6_c_i = struct[0].i_R6_c_i
    i_R6_n_r = struct[0].i_R6_n_r
    i_R6_n_i = struct[0].i_R6_n_i
    i_R7_a_r = struct[0].i_R7_a_r
    i_R7_a_i = struct[0].i_R7_a_i
    i_R7_b_r = struct[0].i_R7_b_r
    i_R7_b_i = struct[0].i_R7_b_i
    i_R7_c_r = struct[0].i_R7_c_r
    i_R7_c_i = struct[0].i_R7_c_i
    i_R7_n_r = struct[0].i_R7_n_r
    i_R7_n_i = struct[0].i_R7_n_i
    i_R8_a_r = struct[0].i_R8_a_r
    i_R8_a_i = struct[0].i_R8_a_i
    i_R8_b_r = struct[0].i_R8_b_r
    i_R8_b_i = struct[0].i_R8_b_i
    i_R8_c_r = struct[0].i_R8_c_r
    i_R8_c_i = struct[0].i_R8_c_i
    i_R8_n_r = struct[0].i_R8_n_r
    i_R8_n_i = struct[0].i_R8_n_i
    i_R9_a_r = struct[0].i_R9_a_r
    i_R9_a_i = struct[0].i_R9_a_i
    i_R9_b_r = struct[0].i_R9_b_r
    i_R9_b_i = struct[0].i_R9_b_i
    i_R9_c_r = struct[0].i_R9_c_r
    i_R9_c_i = struct[0].i_R9_c_i
    i_R9_n_r = struct[0].i_R9_n_r
    i_R9_n_i = struct[0].i_R9_n_i
    i_R10_a_r = struct[0].i_R10_a_r
    i_R10_a_i = struct[0].i_R10_a_i
    i_R10_b_r = struct[0].i_R10_b_r
    i_R10_b_i = struct[0].i_R10_b_i
    i_R10_c_r = struct[0].i_R10_c_r
    i_R10_c_i = struct[0].i_R10_c_i
    i_R10_n_r = struct[0].i_R10_n_r
    i_R10_n_i = struct[0].i_R10_n_i
    i_R12_a_r = struct[0].i_R12_a_r
    i_R12_a_i = struct[0].i_R12_a_i
    i_R12_b_r = struct[0].i_R12_b_r
    i_R12_b_i = struct[0].i_R12_b_i
    i_R12_c_r = struct[0].i_R12_c_r
    i_R12_c_i = struct[0].i_R12_c_i
    i_R12_n_r = struct[0].i_R12_n_r
    i_R12_n_i = struct[0].i_R12_n_i
    i_R13_a_r = struct[0].i_R13_a_r
    i_R13_a_i = struct[0].i_R13_a_i
    i_R13_b_r = struct[0].i_R13_b_r
    i_R13_b_i = struct[0].i_R13_b_i
    i_R13_c_r = struct[0].i_R13_c_r
    i_R13_c_i = struct[0].i_R13_c_i
    i_R13_n_r = struct[0].i_R13_n_r
    i_R13_n_i = struct[0].i_R13_n_i
    i_R14_a_r = struct[0].i_R14_a_r
    i_R14_a_i = struct[0].i_R14_a_i
    i_R14_b_r = struct[0].i_R14_b_r
    i_R14_b_i = struct[0].i_R14_b_i
    i_R14_c_r = struct[0].i_R14_c_r
    i_R14_c_i = struct[0].i_R14_c_i
    i_R14_n_r = struct[0].i_R14_n_r
    i_R14_n_i = struct[0].i_R14_n_i
    p_R1_a = struct[0].p_R1_a
    q_R1_a = struct[0].q_R1_a
    p_R1_b = struct[0].p_R1_b
    q_R1_b = struct[0].q_R1_b
    p_R1_c = struct[0].p_R1_c
    q_R1_c = struct[0].q_R1_c
    p_R11_a = struct[0].p_R11_a
    q_R11_a = struct[0].q_R11_a
    p_R11_b = struct[0].p_R11_b
    q_R11_b = struct[0].q_R11_b
    p_R11_c = struct[0].p_R11_c
    q_R11_c = struct[0].q_R11_c
    p_R15_a = struct[0].p_R15_a
    q_R15_a = struct[0].q_R15_a
    p_R15_b = struct[0].p_R15_b
    q_R15_b = struct[0].q_R15_b
    p_R15_c = struct[0].p_R15_c
    q_R15_c = struct[0].q_R15_c
    p_R16_a = struct[0].p_R16_a
    q_R16_a = struct[0].q_R16_a
    p_R16_b = struct[0].p_R16_b
    q_R16_b = struct[0].q_R16_b
    p_R16_c = struct[0].p_R16_c
    q_R16_c = struct[0].q_R16_c
    p_R17_a = struct[0].p_R17_a
    q_R17_a = struct[0].q_R17_a
    p_R17_b = struct[0].p_R17_b
    q_R17_b = struct[0].q_R17_b
    p_R17_c = struct[0].p_R17_c
    q_R17_c = struct[0].q_R17_c
    p_R18_a = struct[0].p_R18_a
    q_R18_a = struct[0].q_R18_a
    p_R18_b = struct[0].p_R18_b
    q_R18_b = struct[0].q_R18_b
    p_R18_c = struct[0].p_R18_c
    q_R18_c = struct[0].q_R18_c
    u_dummy = struct[0].u_dummy
    
    # Dynamical states:
    x_dummy = struct[0].x[0,0]
    
    # Algebraic states:
    v_R1_a_r = struct[0].y_ini[0,0]
    v_R1_a_i = struct[0].y_ini[1,0]
    v_R1_b_r = struct[0].y_ini[2,0]
    v_R1_b_i = struct[0].y_ini[3,0]
    v_R1_c_r = struct[0].y_ini[4,0]
    v_R1_c_i = struct[0].y_ini[5,0]
    v_R1_n_r = struct[0].y_ini[6,0]
    v_R1_n_i = struct[0].y_ini[7,0]
    v_R11_a_r = struct[0].y_ini[8,0]
    v_R11_a_i = struct[0].y_ini[9,0]
    v_R11_b_r = struct[0].y_ini[10,0]
    v_R11_b_i = struct[0].y_ini[11,0]
    v_R11_c_r = struct[0].y_ini[12,0]
    v_R11_c_i = struct[0].y_ini[13,0]
    v_R11_n_r = struct[0].y_ini[14,0]
    v_R11_n_i = struct[0].y_ini[15,0]
    v_R15_a_r = struct[0].y_ini[16,0]
    v_R15_a_i = struct[0].y_ini[17,0]
    v_R15_b_r = struct[0].y_ini[18,0]
    v_R15_b_i = struct[0].y_ini[19,0]
    v_R15_c_r = struct[0].y_ini[20,0]
    v_R15_c_i = struct[0].y_ini[21,0]
    v_R15_n_r = struct[0].y_ini[22,0]
    v_R15_n_i = struct[0].y_ini[23,0]
    v_R16_a_r = struct[0].y_ini[24,0]
    v_R16_a_i = struct[0].y_ini[25,0]
    v_R16_b_r = struct[0].y_ini[26,0]
    v_R16_b_i = struct[0].y_ini[27,0]
    v_R16_c_r = struct[0].y_ini[28,0]
    v_R16_c_i = struct[0].y_ini[29,0]
    v_R16_n_r = struct[0].y_ini[30,0]
    v_R16_n_i = struct[0].y_ini[31,0]
    v_R17_a_r = struct[0].y_ini[32,0]
    v_R17_a_i = struct[0].y_ini[33,0]
    v_R17_b_r = struct[0].y_ini[34,0]
    v_R17_b_i = struct[0].y_ini[35,0]
    v_R17_c_r = struct[0].y_ini[36,0]
    v_R17_c_i = struct[0].y_ini[37,0]
    v_R17_n_r = struct[0].y_ini[38,0]
    v_R17_n_i = struct[0].y_ini[39,0]
    v_R18_a_r = struct[0].y_ini[40,0]
    v_R18_a_i = struct[0].y_ini[41,0]
    v_R18_b_r = struct[0].y_ini[42,0]
    v_R18_b_i = struct[0].y_ini[43,0]
    v_R18_c_r = struct[0].y_ini[44,0]
    v_R18_c_i = struct[0].y_ini[45,0]
    v_R18_n_r = struct[0].y_ini[46,0]
    v_R18_n_i = struct[0].y_ini[47,0]
    v_R2_a_r = struct[0].y_ini[48,0]
    v_R2_a_i = struct[0].y_ini[49,0]
    v_R2_b_r = struct[0].y_ini[50,0]
    v_R2_b_i = struct[0].y_ini[51,0]
    v_R2_c_r = struct[0].y_ini[52,0]
    v_R2_c_i = struct[0].y_ini[53,0]
    v_R2_n_r = struct[0].y_ini[54,0]
    v_R2_n_i = struct[0].y_ini[55,0]
    v_R3_a_r = struct[0].y_ini[56,0]
    v_R3_a_i = struct[0].y_ini[57,0]
    v_R3_b_r = struct[0].y_ini[58,0]
    v_R3_b_i = struct[0].y_ini[59,0]
    v_R3_c_r = struct[0].y_ini[60,0]
    v_R3_c_i = struct[0].y_ini[61,0]
    v_R3_n_r = struct[0].y_ini[62,0]
    v_R3_n_i = struct[0].y_ini[63,0]
    v_R4_a_r = struct[0].y_ini[64,0]
    v_R4_a_i = struct[0].y_ini[65,0]
    v_R4_b_r = struct[0].y_ini[66,0]
    v_R4_b_i = struct[0].y_ini[67,0]
    v_R4_c_r = struct[0].y_ini[68,0]
    v_R4_c_i = struct[0].y_ini[69,0]
    v_R4_n_r = struct[0].y_ini[70,0]
    v_R4_n_i = struct[0].y_ini[71,0]
    v_R5_a_r = struct[0].y_ini[72,0]
    v_R5_a_i = struct[0].y_ini[73,0]
    v_R5_b_r = struct[0].y_ini[74,0]
    v_R5_b_i = struct[0].y_ini[75,0]
    v_R5_c_r = struct[0].y_ini[76,0]
    v_R5_c_i = struct[0].y_ini[77,0]
    v_R5_n_r = struct[0].y_ini[78,0]
    v_R5_n_i = struct[0].y_ini[79,0]
    v_R6_a_r = struct[0].y_ini[80,0]
    v_R6_a_i = struct[0].y_ini[81,0]
    v_R6_b_r = struct[0].y_ini[82,0]
    v_R6_b_i = struct[0].y_ini[83,0]
    v_R6_c_r = struct[0].y_ini[84,0]
    v_R6_c_i = struct[0].y_ini[85,0]
    v_R6_n_r = struct[0].y_ini[86,0]
    v_R6_n_i = struct[0].y_ini[87,0]
    v_R7_a_r = struct[0].y_ini[88,0]
    v_R7_a_i = struct[0].y_ini[89,0]
    v_R7_b_r = struct[0].y_ini[90,0]
    v_R7_b_i = struct[0].y_ini[91,0]
    v_R7_c_r = struct[0].y_ini[92,0]
    v_R7_c_i = struct[0].y_ini[93,0]
    v_R7_n_r = struct[0].y_ini[94,0]
    v_R7_n_i = struct[0].y_ini[95,0]
    v_R8_a_r = struct[0].y_ini[96,0]
    v_R8_a_i = struct[0].y_ini[97,0]
    v_R8_b_r = struct[0].y_ini[98,0]
    v_R8_b_i = struct[0].y_ini[99,0]
    v_R8_c_r = struct[0].y_ini[100,0]
    v_R8_c_i = struct[0].y_ini[101,0]
    v_R8_n_r = struct[0].y_ini[102,0]
    v_R8_n_i = struct[0].y_ini[103,0]
    v_R9_a_r = struct[0].y_ini[104,0]
    v_R9_a_i = struct[0].y_ini[105,0]
    v_R9_b_r = struct[0].y_ini[106,0]
    v_R9_b_i = struct[0].y_ini[107,0]
    v_R9_c_r = struct[0].y_ini[108,0]
    v_R9_c_i = struct[0].y_ini[109,0]
    v_R9_n_r = struct[0].y_ini[110,0]
    v_R9_n_i = struct[0].y_ini[111,0]
    v_R10_a_r = struct[0].y_ini[112,0]
    v_R10_a_i = struct[0].y_ini[113,0]
    v_R10_b_r = struct[0].y_ini[114,0]
    v_R10_b_i = struct[0].y_ini[115,0]
    v_R10_c_r = struct[0].y_ini[116,0]
    v_R10_c_i = struct[0].y_ini[117,0]
    v_R10_n_r = struct[0].y_ini[118,0]
    v_R10_n_i = struct[0].y_ini[119,0]
    v_R12_a_r = struct[0].y_ini[120,0]
    v_R12_a_i = struct[0].y_ini[121,0]
    v_R12_b_r = struct[0].y_ini[122,0]
    v_R12_b_i = struct[0].y_ini[123,0]
    v_R12_c_r = struct[0].y_ini[124,0]
    v_R12_c_i = struct[0].y_ini[125,0]
    v_R12_n_r = struct[0].y_ini[126,0]
    v_R12_n_i = struct[0].y_ini[127,0]
    v_R13_a_r = struct[0].y_ini[128,0]
    v_R13_a_i = struct[0].y_ini[129,0]
    v_R13_b_r = struct[0].y_ini[130,0]
    v_R13_b_i = struct[0].y_ini[131,0]
    v_R13_c_r = struct[0].y_ini[132,0]
    v_R13_c_i = struct[0].y_ini[133,0]
    v_R13_n_r = struct[0].y_ini[134,0]
    v_R13_n_i = struct[0].y_ini[135,0]
    v_R14_a_r = struct[0].y_ini[136,0]
    v_R14_a_i = struct[0].y_ini[137,0]
    v_R14_b_r = struct[0].y_ini[138,0]
    v_R14_b_i = struct[0].y_ini[139,0]
    v_R14_c_r = struct[0].y_ini[140,0]
    v_R14_c_i = struct[0].y_ini[141,0]
    v_R14_n_r = struct[0].y_ini[142,0]
    v_R14_n_i = struct[0].y_ini[143,0]
    i_R1_a_r = struct[0].y_ini[144,0]
    i_R1_a_i = struct[0].y_ini[145,0]
    i_R1_b_r = struct[0].y_ini[146,0]
    i_R1_b_i = struct[0].y_ini[147,0]
    i_R1_c_r = struct[0].y_ini[148,0]
    i_R1_c_i = struct[0].y_ini[149,0]
    i_R1_n_r = struct[0].y_ini[150,0]
    i_R1_n_i = struct[0].y_ini[151,0]
    i_R11_a_r = struct[0].y_ini[152,0]
    i_R11_a_i = struct[0].y_ini[153,0]
    i_R11_b_r = struct[0].y_ini[154,0]
    i_R11_b_i = struct[0].y_ini[155,0]
    i_R11_c_r = struct[0].y_ini[156,0]
    i_R11_c_i = struct[0].y_ini[157,0]
    i_R11_n_r = struct[0].y_ini[158,0]
    i_R11_n_i = struct[0].y_ini[159,0]
    i_R15_a_r = struct[0].y_ini[160,0]
    i_R15_a_i = struct[0].y_ini[161,0]
    i_R15_b_r = struct[0].y_ini[162,0]
    i_R15_b_i = struct[0].y_ini[163,0]
    i_R15_c_r = struct[0].y_ini[164,0]
    i_R15_c_i = struct[0].y_ini[165,0]
    i_R15_n_r = struct[0].y_ini[166,0]
    i_R15_n_i = struct[0].y_ini[167,0]
    i_R16_a_r = struct[0].y_ini[168,0]
    i_R16_a_i = struct[0].y_ini[169,0]
    i_R16_b_r = struct[0].y_ini[170,0]
    i_R16_b_i = struct[0].y_ini[171,0]
    i_R16_c_r = struct[0].y_ini[172,0]
    i_R16_c_i = struct[0].y_ini[173,0]
    i_R16_n_r = struct[0].y_ini[174,0]
    i_R16_n_i = struct[0].y_ini[175,0]
    i_R17_a_r = struct[0].y_ini[176,0]
    i_R17_a_i = struct[0].y_ini[177,0]
    i_R17_b_r = struct[0].y_ini[178,0]
    i_R17_b_i = struct[0].y_ini[179,0]
    i_R17_c_r = struct[0].y_ini[180,0]
    i_R17_c_i = struct[0].y_ini[181,0]
    i_R17_n_r = struct[0].y_ini[182,0]
    i_R17_n_i = struct[0].y_ini[183,0]
    i_R18_a_r = struct[0].y_ini[184,0]
    i_R18_a_i = struct[0].y_ini[185,0]
    i_R18_b_r = struct[0].y_ini[186,0]
    i_R18_b_i = struct[0].y_ini[187,0]
    i_R18_c_r = struct[0].y_ini[188,0]
    i_R18_c_i = struct[0].y_ini[189,0]
    i_R18_n_r = struct[0].y_ini[190,0]
    i_R18_n_i = struct[0].y_ini[191,0]
    
    # Differential equations:
    if mode == 2:


        struct[0].f[0,0] = u_dummy - x_dummy
    
    # Algebraic equations:
    if mode == 3:

        g_n = np.ascontiguousarray(struct[0].Gy_ini) @ np.ascontiguousarray(struct[0].y_ini)

        struct[0].g[0,0] = i_R1_a_r + 0.849044513514155*v_R0_a_i + 0.212261128378539*v_R0_a_r - 0.849044513514155*v_R0_c_i - 0.212261128378539*v_R0_c_r - 127.595184532966*v_R1_a_i - 123.954122254357*v_R1_a_r + 10.2713736253513*v_R1_b_i + 39.6392229058202*v_R1_b_r + 23.284964480954*v_R1_c_i + 24.9575997948692*v_R1_c_r + 83.8007853900572*v_R1_n_i + 58.0215758469966*v_R1_n_r + 54.0657727682604*v_R2_a_i + 105.57176931318*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r - 23.284964480954*v_R2_c_i - 24.9575997948692*v_R2_c_r - 10.2713736253513*v_R2_n_i - 39.6392229058202*v_R2_n_r
        struct[0].g[1,0] = i_R1_a_i + 0.212261128378539*v_R0_a_i - 0.849044513514155*v_R0_a_r - 0.212261128378539*v_R0_c_i + 0.849044513514155*v_R0_c_r - 123.954122254357*v_R1_a_i + 127.595184532966*v_R1_a_r + 39.6392229058202*v_R1_b_i - 10.2713736253513*v_R1_b_r + 24.9575997948692*v_R1_c_i - 23.284964480954*v_R1_c_r + 58.0215758469966*v_R1_n_i - 83.8007853900572*v_R1_n_r + 105.57176931318*v_R2_a_i - 54.0657727682604*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r - 24.9575997948692*v_R2_c_i + 23.284964480954*v_R2_c_r - 39.6392229058202*v_R2_n_i + 10.2713736253513*v_R2_n_r
        struct[0].g[2,0] = i_R1_b_r - 0.849044513514155*v_R0_a_i - 0.212261128378539*v_R0_a_r + 0.849044513514155*v_R0_b_i + 0.212261128378539*v_R0_b_r + 10.2713736253513*v_R1_a_i + 39.6392229058202*v_R1_a_r - 127.595184532966*v_R1_b_i - 123.954122254357*v_R1_b_r + 10.2713736253513*v_R1_c_i + 39.6392229058201*v_R1_c_r + 96.8143762456598*v_R1_n_i + 43.3399527360457*v_R1_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r + 54.0657727682604*v_R2_b_i + 105.57176931318*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058201*v_R2_c_r - 23.284964480954*v_R2_n_i - 24.9575997948692*v_R2_n_r
        struct[0].g[3,0] = i_R1_b_i - 0.212261128378539*v_R0_a_i + 0.849044513514155*v_R0_a_r + 0.212261128378539*v_R0_b_i - 0.849044513514155*v_R0_b_r + 39.6392229058202*v_R1_a_i - 10.2713736253513*v_R1_a_r - 123.954122254357*v_R1_b_i + 127.595184532966*v_R1_b_r + 39.6392229058201*v_R1_c_i - 10.2713736253513*v_R1_c_r + 43.3399527360457*v_R1_n_i - 96.8143762456598*v_R1_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r + 105.57176931318*v_R2_b_i - 54.0657727682604*v_R2_b_r - 39.6392229058201*v_R2_c_i + 10.2713736253513*v_R2_c_r - 24.9575997948692*v_R2_n_i + 23.284964480954*v_R2_n_r
        struct[0].g[4,0] = i_R1_c_r - 0.849044513514155*v_R0_b_i - 0.212261128378539*v_R0_b_r + 0.849044513514155*v_R0_c_i + 0.212261128378539*v_R0_c_r + 23.284964480954*v_R1_a_i + 24.9575997948692*v_R1_a_r + 10.2713736253513*v_R1_b_i + 39.6392229058202*v_R1_b_r - 127.595184532966*v_R1_c_i - 123.954122254357*v_R1_c_r + 83.8007853900572*v_R1_n_i + 58.0215758469966*v_R1_n_r - 23.284964480954*v_R2_a_i - 24.9575997948692*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r + 54.0657727682604*v_R2_c_i + 105.57176931318*v_R2_c_r - 10.2713736253514*v_R2_n_i - 39.6392229058201*v_R2_n_r
        struct[0].g[5,0] = i_R1_c_i - 0.212261128378539*v_R0_b_i + 0.849044513514155*v_R0_b_r + 0.212261128378539*v_R0_c_i - 0.849044513514155*v_R0_c_r + 24.9575997948692*v_R1_a_i - 23.284964480954*v_R1_a_r + 39.6392229058202*v_R1_b_i - 10.2713736253513*v_R1_b_r - 123.954122254357*v_R1_c_i + 127.595184532966*v_R1_c_r + 58.0215758469966*v_R1_n_i - 83.8007853900572*v_R1_n_r - 24.9575997948692*v_R2_a_i + 23.284964480954*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r + 105.57176931318*v_R2_c_i - 54.0657727682604*v_R2_c_r - 39.6392229058201*v_R2_n_i + 10.2713736253514*v_R2_n_r
        struct[0].g[6,0] = g_n[6,0]
        struct[0].g[7,0] = g_n[7,0]
        struct[0].g[8,0] = g_n[8,0]
        struct[0].g[9,0] = g_n[9,0]
        struct[0].g[10,0] = g_n[10,0]
        struct[0].g[11,0] = g_n[11,0]
        struct[0].g[12,0] = g_n[12,0]
        struct[0].g[13,0] = g_n[13,0]
        struct[0].g[14,0] = g_n[14,0]
        struct[0].g[15,0] = g_n[15,0]
        struct[0].g[16,0] = g_n[16,0]
        struct[0].g[17,0] = g_n[17,0]
        struct[0].g[18,0] = g_n[18,0]
        struct[0].g[19,0] = g_n[19,0]
        struct[0].g[20,0] = g_n[20,0]
        struct[0].g[21,0] = g_n[21,0]
        struct[0].g[22,0] = g_n[22,0]
        struct[0].g[23,0] = g_n[23,0]
        struct[0].g[24,0] = g_n[24,0]
        struct[0].g[25,0] = g_n[25,0]
        struct[0].g[26,0] = g_n[26,0]
        struct[0].g[27,0] = g_n[27,0]
        struct[0].g[28,0] = g_n[28,0]
        struct[0].g[29,0] = g_n[29,0]
        struct[0].g[30,0] = g_n[30,0]
        struct[0].g[31,0] = g_n[31,0]
        struct[0].g[32,0] = g_n[32,0]
        struct[0].g[33,0] = g_n[33,0]
        struct[0].g[34,0] = g_n[34,0]
        struct[0].g[35,0] = g_n[35,0]
        struct[0].g[36,0] = g_n[36,0]
        struct[0].g[37,0] = g_n[37,0]
        struct[0].g[38,0] = g_n[38,0]
        struct[0].g[39,0] = g_n[39,0]
        struct[0].g[40,0] = g_n[40,0]
        struct[0].g[41,0] = g_n[41,0]
        struct[0].g[42,0] = g_n[42,0]
        struct[0].g[43,0] = g_n[43,0]
        struct[0].g[44,0] = g_n[44,0]
        struct[0].g[45,0] = g_n[45,0]
        struct[0].g[46,0] = g_n[46,0]
        struct[0].g[47,0] = g_n[47,0]
        struct[0].g[48,0] = i_R2_a_r + 54.0657727682604*v_R1_a_i + 105.57176931318*v_R1_a_r - 10.2713736253513*v_R1_b_i - 39.6392229058202*v_R1_b_r - 23.284964480954*v_R1_c_i - 24.9575997948692*v_R1_c_r - 10.2713736253513*v_R1_n_i - 39.6392229058202*v_R1_n_r - 108.131545536521*v_R2_a_i - 211.14353862636*v_R2_a_r + 20.5427472507027*v_R2_b_i + 79.2784458116403*v_R2_b_r + 46.569928961908*v_R2_c_i + 49.9151995897384*v_R2_c_r + 20.5427472507027*v_R2_n_i + 79.2784458116403*v_R2_n_r + 54.0657727682604*v_R3_a_i + 105.57176931318*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r - 23.284964480954*v_R3_c_i - 24.9575997948692*v_R3_c_r - 10.2713736253513*v_R3_n_i - 39.6392229058202*v_R3_n_r
        struct[0].g[49,0] = i_R2_a_i + 105.57176931318*v_R1_a_i - 54.0657727682604*v_R1_a_r - 39.6392229058202*v_R1_b_i + 10.2713736253513*v_R1_b_r - 24.9575997948692*v_R1_c_i + 23.284964480954*v_R1_c_r - 39.6392229058202*v_R1_n_i + 10.2713736253513*v_R1_n_r - 211.14353862636*v_R2_a_i + 108.131545536521*v_R2_a_r + 79.2784458116403*v_R2_b_i - 20.5427472507027*v_R2_b_r + 49.9151995897384*v_R2_c_i - 46.569928961908*v_R2_c_r + 79.2784458116403*v_R2_n_i - 20.5427472507027*v_R2_n_r + 105.57176931318*v_R3_a_i - 54.0657727682604*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r - 24.9575997948692*v_R3_c_i + 23.284964480954*v_R3_c_r - 39.6392229058202*v_R3_n_i + 10.2713736253513*v_R3_n_r
        struct[0].g[50,0] = i_R2_b_r - 10.2713736253513*v_R1_a_i - 39.6392229058202*v_R1_a_r + 54.0657727682604*v_R1_b_i + 105.57176931318*v_R1_b_r - 10.2713736253513*v_R1_c_i - 39.6392229058201*v_R1_c_r - 23.284964480954*v_R1_n_i - 24.9575997948692*v_R1_n_r + 20.5427472507027*v_R2_a_i + 79.2784458116403*v_R2_a_r - 108.131545536521*v_R2_b_i - 211.14353862636*v_R2_b_r + 20.5427472507027*v_R2_c_i + 79.2784458116403*v_R2_c_r + 46.569928961908*v_R2_n_i + 49.9151995897384*v_R2_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r + 54.0657727682604*v_R3_b_i + 105.57176931318*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058201*v_R3_c_r - 23.284964480954*v_R3_n_i - 24.9575997948692*v_R3_n_r
        struct[0].g[51,0] = i_R2_b_i - 39.6392229058202*v_R1_a_i + 10.2713736253513*v_R1_a_r + 105.57176931318*v_R1_b_i - 54.0657727682604*v_R1_b_r - 39.6392229058201*v_R1_c_i + 10.2713736253513*v_R1_c_r - 24.9575997948692*v_R1_n_i + 23.284964480954*v_R1_n_r + 79.2784458116403*v_R2_a_i - 20.5427472507027*v_R2_a_r - 211.14353862636*v_R2_b_i + 108.131545536521*v_R2_b_r + 79.2784458116403*v_R2_c_i - 20.5427472507027*v_R2_c_r + 49.9151995897384*v_R2_n_i - 46.569928961908*v_R2_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r + 105.57176931318*v_R3_b_i - 54.0657727682604*v_R3_b_r - 39.6392229058201*v_R3_c_i + 10.2713736253513*v_R3_c_r - 24.9575997948692*v_R3_n_i + 23.284964480954*v_R3_n_r
        struct[0].g[52,0] = i_R2_c_r - 23.284964480954*v_R1_a_i - 24.9575997948692*v_R1_a_r - 10.2713736253513*v_R1_b_i - 39.6392229058202*v_R1_b_r + 54.0657727682604*v_R1_c_i + 105.57176931318*v_R1_c_r - 10.2713736253514*v_R1_n_i - 39.6392229058201*v_R1_n_r + 46.569928961908*v_R2_a_i + 49.9151995897384*v_R2_a_r + 20.5427472507027*v_R2_b_i + 79.2784458116404*v_R2_b_r - 108.131545536521*v_R2_c_i - 211.14353862636*v_R2_c_r + 20.5427472507027*v_R2_n_i + 79.2784458116403*v_R2_n_r - 23.284964480954*v_R3_a_i - 24.9575997948692*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r + 54.0657727682604*v_R3_c_i + 105.57176931318*v_R3_c_r - 10.2713736253514*v_R3_n_i - 39.6392229058201*v_R3_n_r
        struct[0].g[53,0] = i_R2_c_i - 24.9575997948692*v_R1_a_i + 23.284964480954*v_R1_a_r - 39.6392229058202*v_R1_b_i + 10.2713736253513*v_R1_b_r + 105.57176931318*v_R1_c_i - 54.0657727682604*v_R1_c_r - 39.6392229058201*v_R1_n_i + 10.2713736253514*v_R1_n_r + 49.9151995897384*v_R2_a_i - 46.569928961908*v_R2_a_r + 79.2784458116404*v_R2_b_i - 20.5427472507027*v_R2_b_r - 211.14353862636*v_R2_c_i + 108.131545536521*v_R2_c_r + 79.2784458116403*v_R2_n_i - 20.5427472507027*v_R2_n_r - 24.9575997948692*v_R3_a_i + 23.284964480954*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r + 105.57176931318*v_R3_c_i - 54.0657727682604*v_R3_c_r - 39.6392229058201*v_R3_n_i + 10.2713736253514*v_R3_n_r
        struct[0].g[54,0] = i_R2_n_r - 10.2713736253513*v_R1_a_i - 39.6392229058202*v_R1_a_r - 23.284964480954*v_R1_b_i - 24.9575997948692*v_R1_b_r - 10.2713736253513*v_R1_c_i - 39.6392229058202*v_R1_c_r + 54.0657727682604*v_R1_n_i + 105.57176931318*v_R1_n_r + 20.5427472507027*v_R2_a_i + 79.2784458116403*v_R2_a_r + 46.569928961908*v_R2_b_i + 49.9151995897384*v_R2_b_r + 20.5427472507027*v_R2_c_i + 79.2784458116403*v_R2_c_r - 108.131545536521*v_R2_n_i - 211.16853862636*v_R2_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r - 23.284964480954*v_R3_b_i - 24.9575997948692*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058202*v_R3_c_r + 54.0657727682604*v_R3_n_i + 105.57176931318*v_R3_n_r
        struct[0].g[55,0] = i_R2_n_i - 39.6392229058202*v_R1_a_i + 10.2713736253513*v_R1_a_r - 24.9575997948692*v_R1_b_i + 23.284964480954*v_R1_b_r - 39.6392229058202*v_R1_c_i + 10.2713736253513*v_R1_c_r + 105.57176931318*v_R1_n_i - 54.0657727682604*v_R1_n_r + 79.2784458116403*v_R2_a_i - 20.5427472507027*v_R2_a_r + 49.9151995897384*v_R2_b_i - 46.569928961908*v_R2_b_r + 79.2784458116403*v_R2_c_i - 20.5427472507027*v_R2_c_r - 211.16853862636*v_R2_n_i + 108.131545536521*v_R2_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r - 24.9575997948692*v_R3_b_i + 23.284964480954*v_R3_b_r - 39.6392229058202*v_R3_c_i + 10.2713736253513*v_R3_c_r + 105.57176931318*v_R3_n_i - 54.0657727682604*v_R3_n_r
        struct[0].g[56,0] = i_R3_a_r + 4.84676915585801*v_R11_a_i + 26.5300693004806*v_R11_a_r + 1.58482814504483*v_R11_b_i - 7.8946133751477*v_R11_b_r + 0.67991845328531*v_R11_c_i - 7.72144347466701*v_R11_c_r + 1.58482814504483*v_R11_n_i - 7.89461337514771*v_R11_n_r + 54.0657727682604*v_R2_a_i + 105.57176931318*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r - 23.284964480954*v_R2_c_i - 24.9575997948692*v_R2_c_r - 10.2713736253513*v_R2_n_i - 39.6392229058202*v_R2_n_r - 112.978314692379*v_R3_a_i - 237.673607926841*v_R3_a_r + 18.9579191056578*v_R3_b_i + 87.173059186788*v_R3_b_r + 45.8900105086227*v_R3_c_i + 57.6366430644055*v_R3_c_r + 18.9579191056579*v_R3_n_i + 87.173059186788*v_R3_n_r + 54.0657727682604*v_R4_a_i + 105.57176931318*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r - 23.284964480954*v_R4_c_i - 24.9575997948692*v_R4_c_r - 10.2713736253513*v_R4_n_i - 39.6392229058202*v_R4_n_r
        struct[0].g[57,0] = i_R3_a_i + 26.5300693004806*v_R11_a_i - 4.84676915585801*v_R11_a_r - 7.8946133751477*v_R11_b_i - 1.58482814504483*v_R11_b_r - 7.72144347466701*v_R11_c_i - 0.67991845328531*v_R11_c_r - 7.89461337514771*v_R11_n_i - 1.58482814504483*v_R11_n_r + 105.57176931318*v_R2_a_i - 54.0657727682604*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r - 24.9575997948692*v_R2_c_i + 23.284964480954*v_R2_c_r - 39.6392229058202*v_R2_n_i + 10.2713736253513*v_R2_n_r - 237.673607926841*v_R3_a_i + 112.978314692379*v_R3_a_r + 87.173059186788*v_R3_b_i - 18.9579191056578*v_R3_b_r + 57.6366430644055*v_R3_c_i - 45.8900105086227*v_R3_c_r + 87.173059186788*v_R3_n_i - 18.9579191056579*v_R3_n_r + 105.57176931318*v_R4_a_i - 54.0657727682604*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r - 24.9575997948692*v_R4_c_i + 23.284964480954*v_R4_c_r - 39.6392229058202*v_R4_n_i + 10.2713736253513*v_R4_n_r
        struct[0].g[58,0] = i_R3_b_r + 1.58482814504483*v_R11_a_i - 7.89461337514771*v_R11_a_r + 4.84676915585801*v_R11_b_i + 26.5300693004806*v_R11_b_r + 1.58482814504483*v_R11_c_i - 7.89461337514771*v_R11_c_r + 0.679918453285308*v_R11_n_i - 7.72144347466701*v_R11_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r + 54.0657727682604*v_R2_b_i + 105.57176931318*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058201*v_R2_c_r - 23.284964480954*v_R2_n_i - 24.9575997948692*v_R2_n_r + 18.9579191056579*v_R3_a_i + 87.173059186788*v_R3_a_r - 112.978314692379*v_R3_b_i - 237.673607926841*v_R3_b_r + 18.9579191056579*v_R3_c_i + 87.173059186788*v_R3_c_r + 45.8900105086227*v_R3_n_i + 57.6366430644054*v_R3_n_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r + 54.0657727682604*v_R4_b_i + 105.57176931318*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058201*v_R4_c_r - 23.284964480954*v_R4_n_i - 24.9575997948692*v_R4_n_r
        struct[0].g[59,0] = i_R3_b_i - 7.89461337514771*v_R11_a_i - 1.58482814504483*v_R11_a_r + 26.5300693004806*v_R11_b_i - 4.84676915585801*v_R11_b_r - 7.89461337514771*v_R11_c_i - 1.58482814504483*v_R11_c_r - 7.72144347466701*v_R11_n_i - 0.679918453285308*v_R11_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r + 105.57176931318*v_R2_b_i - 54.0657727682604*v_R2_b_r - 39.6392229058201*v_R2_c_i + 10.2713736253513*v_R2_c_r - 24.9575997948692*v_R2_n_i + 23.284964480954*v_R2_n_r + 87.173059186788*v_R3_a_i - 18.9579191056579*v_R3_a_r - 237.673607926841*v_R3_b_i + 112.978314692379*v_R3_b_r + 87.173059186788*v_R3_c_i - 18.9579191056579*v_R3_c_r + 57.6366430644054*v_R3_n_i - 45.8900105086227*v_R3_n_r - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r + 105.57176931318*v_R4_b_i - 54.0657727682604*v_R4_b_r - 39.6392229058201*v_R4_c_i + 10.2713736253513*v_R4_c_r - 24.9575997948692*v_R4_n_i + 23.284964480954*v_R4_n_r
        struct[0].g[60,0] = i_R3_c_r + 0.67991845328531*v_R11_a_i - 7.72144347466701*v_R11_a_r + 1.58482814504483*v_R11_b_i - 7.8946133751477*v_R11_b_r + 4.84676915585801*v_R11_c_i + 26.5300693004806*v_R11_c_r + 1.58482814504483*v_R11_n_i - 7.89461337514771*v_R11_n_r - 23.284964480954*v_R2_a_i - 24.9575997948692*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r + 54.0657727682604*v_R2_c_i + 105.57176931318*v_R2_c_r - 10.2713736253514*v_R2_n_i - 39.6392229058201*v_R2_n_r + 45.8900105086227*v_R3_a_i + 57.6366430644054*v_R3_a_r + 18.9579191056578*v_R3_b_i + 87.1730591867881*v_R3_b_r - 112.978314692379*v_R3_c_i - 237.673607926841*v_R3_c_r + 18.9579191056579*v_R3_n_i + 87.173059186788*v_R3_n_r - 23.284964480954*v_R4_a_i - 24.9575997948692*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r + 54.0657727682604*v_R4_c_i + 105.57176931318*v_R4_c_r - 10.2713736253514*v_R4_n_i - 39.6392229058201*v_R4_n_r
        struct[0].g[61,0] = i_R3_c_i - 7.72144347466701*v_R11_a_i - 0.67991845328531*v_R11_a_r - 7.8946133751477*v_R11_b_i - 1.58482814504483*v_R11_b_r + 26.5300693004806*v_R11_c_i - 4.84676915585801*v_R11_c_r - 7.89461337514771*v_R11_n_i - 1.58482814504483*v_R11_n_r - 24.9575997948692*v_R2_a_i + 23.284964480954*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r + 105.57176931318*v_R2_c_i - 54.0657727682604*v_R2_c_r - 39.6392229058201*v_R2_n_i + 10.2713736253514*v_R2_n_r + 57.6366430644054*v_R3_a_i - 45.8900105086227*v_R3_a_r + 87.1730591867881*v_R3_b_i - 18.9579191056578*v_R3_b_r - 237.673607926841*v_R3_c_i + 112.978314692379*v_R3_c_r + 87.173059186788*v_R3_n_i - 18.9579191056579*v_R3_n_r - 24.9575997948692*v_R4_a_i + 23.284964480954*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r + 105.57176931318*v_R4_c_i - 54.0657727682604*v_R4_c_r - 39.6392229058201*v_R4_n_i + 10.2713736253514*v_R4_n_r
        struct[0].g[62,0] = i_R3_n_r + 1.58482814504483*v_R11_a_i - 7.89461337514771*v_R11_a_r + 0.679918453285308*v_R11_b_i - 7.72144347466701*v_R11_b_r + 1.58482814504483*v_R11_c_i - 7.89461337514771*v_R11_c_r + 4.84676915585801*v_R11_n_i + 26.5300693004806*v_R11_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r - 23.284964480954*v_R2_b_i - 24.9575997948692*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058202*v_R2_c_r + 54.0657727682604*v_R2_n_i + 105.57176931318*v_R2_n_r + 18.9579191056578*v_R3_a_i + 87.173059186788*v_R3_a_r + 45.8900105086227*v_R3_b_i + 57.6366430644054*v_R3_b_r + 18.9579191056579*v_R3_c_i + 87.173059186788*v_R3_c_r - 112.978314692379*v_R3_n_i - 237.673607926841*v_R3_n_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r - 23.284964480954*v_R4_b_i - 24.9575997948692*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058202*v_R4_c_r + 54.0657727682604*v_R4_n_i + 105.57176931318*v_R4_n_r
        struct[0].g[63,0] = i_R3_n_i - 7.89461337514771*v_R11_a_i - 1.58482814504483*v_R11_a_r - 7.72144347466701*v_R11_b_i - 0.679918453285308*v_R11_b_r - 7.89461337514771*v_R11_c_i - 1.58482814504483*v_R11_c_r + 26.5300693004806*v_R11_n_i - 4.84676915585801*v_R11_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r - 24.9575997948692*v_R2_b_i + 23.284964480954*v_R2_b_r - 39.6392229058202*v_R2_c_i + 10.2713736253513*v_R2_c_r + 105.57176931318*v_R2_n_i - 54.0657727682604*v_R2_n_r + 87.173059186788*v_R3_a_i - 18.9579191056578*v_R3_a_r + 57.6366430644054*v_R3_b_i - 45.8900105086227*v_R3_b_r + 87.173059186788*v_R3_c_i - 18.9579191056579*v_R3_c_r - 237.673607926841*v_R3_n_i + 112.978314692379*v_R3_n_r - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r - 24.9575997948692*v_R4_b_i + 23.284964480954*v_R4_b_r - 39.6392229058202*v_R4_c_i + 10.2713736253513*v_R4_c_r + 105.57176931318*v_R4_n_i - 54.0657727682604*v_R4_n_r
        struct[0].g[64,0] = i_R4_a_r + 4.84676915585801*v_R12_a_i + 26.5300693004806*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 0.67991845328531*v_R12_c_i - 7.72144347466701*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r + 54.0657727682604*v_R3_a_i + 105.57176931318*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r - 23.284964480954*v_R3_c_i - 24.9575997948692*v_R3_c_r - 10.2713736253513*v_R3_n_i - 39.6392229058202*v_R3_n_r - 112.978314692379*v_R4_a_i - 237.673607926841*v_R4_a_r + 18.9579191056578*v_R4_b_i + 87.173059186788*v_R4_b_r + 45.8900105086227*v_R4_c_i + 57.6366430644055*v_R4_c_r + 18.9579191056579*v_R4_n_i + 87.173059186788*v_R4_n_r + 54.0657727682604*v_R5_a_i + 105.57176931318*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r - 23.284964480954*v_R5_c_i - 24.9575997948692*v_R5_c_r - 10.2713736253513*v_R5_n_i - 39.6392229058202*v_R5_n_r
        struct[0].g[65,0] = i_R4_a_i + 26.5300693004806*v_R12_a_i - 4.84676915585801*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r - 7.72144347466701*v_R12_c_i - 0.67991845328531*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r + 105.57176931318*v_R3_a_i - 54.0657727682604*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r - 24.9575997948692*v_R3_c_i + 23.284964480954*v_R3_c_r - 39.6392229058202*v_R3_n_i + 10.2713736253513*v_R3_n_r - 237.673607926841*v_R4_a_i + 112.978314692379*v_R4_a_r + 87.173059186788*v_R4_b_i - 18.9579191056578*v_R4_b_r + 57.6366430644055*v_R4_c_i - 45.8900105086227*v_R4_c_r + 87.173059186788*v_R4_n_i - 18.9579191056579*v_R4_n_r + 105.57176931318*v_R5_a_i - 54.0657727682604*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r - 24.9575997948692*v_R5_c_i + 23.284964480954*v_R5_c_r - 39.6392229058202*v_R5_n_i + 10.2713736253513*v_R5_n_r
        struct[0].g[66,0] = i_R4_b_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 4.84676915585801*v_R12_b_i + 26.5300693004806*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 0.679918453285308*v_R12_n_i - 7.72144347466701*v_R12_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r + 54.0657727682604*v_R3_b_i + 105.57176931318*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058201*v_R3_c_r - 23.284964480954*v_R3_n_i - 24.9575997948692*v_R3_n_r + 18.9579191056579*v_R4_a_i + 87.173059186788*v_R4_a_r - 112.978314692379*v_R4_b_i - 237.673607926841*v_R4_b_r + 18.9579191056579*v_R4_c_i + 87.173059186788*v_R4_c_r + 45.8900105086227*v_R4_n_i + 57.6366430644054*v_R4_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r + 54.0657727682604*v_R5_b_i + 105.57176931318*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058201*v_R5_c_r - 23.284964480954*v_R5_n_i - 24.9575997948692*v_R5_n_r
        struct[0].g[67,0] = i_R4_b_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r + 26.5300693004806*v_R12_b_i - 4.84676915585801*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r - 7.72144347466701*v_R12_n_i - 0.679918453285308*v_R12_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r + 105.57176931318*v_R3_b_i - 54.0657727682604*v_R3_b_r - 39.6392229058201*v_R3_c_i + 10.2713736253513*v_R3_c_r - 24.9575997948692*v_R3_n_i + 23.284964480954*v_R3_n_r + 87.173059186788*v_R4_a_i - 18.9579191056579*v_R4_a_r - 237.673607926841*v_R4_b_i + 112.978314692379*v_R4_b_r + 87.173059186788*v_R4_c_i - 18.9579191056579*v_R4_c_r + 57.6366430644054*v_R4_n_i - 45.8900105086227*v_R4_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r + 105.57176931318*v_R5_b_i - 54.0657727682604*v_R5_b_r - 39.6392229058201*v_R5_c_i + 10.2713736253513*v_R5_c_r - 24.9575997948692*v_R5_n_i + 23.284964480954*v_R5_n_r
        struct[0].g[68,0] = i_R4_c_r + 0.67991845328531*v_R12_a_i - 7.72144347466701*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 4.84676915585801*v_R12_c_i + 26.5300693004806*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r - 23.284964480954*v_R3_a_i - 24.9575997948692*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r + 54.0657727682604*v_R3_c_i + 105.57176931318*v_R3_c_r - 10.2713736253514*v_R3_n_i - 39.6392229058201*v_R3_n_r + 45.8900105086227*v_R4_a_i + 57.6366430644054*v_R4_a_r + 18.9579191056578*v_R4_b_i + 87.1730591867881*v_R4_b_r - 112.978314692379*v_R4_c_i - 237.673607926841*v_R4_c_r + 18.9579191056579*v_R4_n_i + 87.173059186788*v_R4_n_r - 23.284964480954*v_R5_a_i - 24.9575997948692*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r + 54.0657727682604*v_R5_c_i + 105.57176931318*v_R5_c_r - 10.2713736253514*v_R5_n_i - 39.6392229058201*v_R5_n_r
        struct[0].g[69,0] = i_R4_c_i - 7.72144347466701*v_R12_a_i - 0.67991845328531*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r + 26.5300693004806*v_R12_c_i - 4.84676915585801*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r - 24.9575997948692*v_R3_a_i + 23.284964480954*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r + 105.57176931318*v_R3_c_i - 54.0657727682604*v_R3_c_r - 39.6392229058201*v_R3_n_i + 10.2713736253514*v_R3_n_r + 57.6366430644054*v_R4_a_i - 45.8900105086227*v_R4_a_r + 87.1730591867881*v_R4_b_i - 18.9579191056578*v_R4_b_r - 237.673607926841*v_R4_c_i + 112.978314692379*v_R4_c_r + 87.173059186788*v_R4_n_i - 18.9579191056579*v_R4_n_r - 24.9575997948692*v_R5_a_i + 23.284964480954*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r + 105.57176931318*v_R5_c_i - 54.0657727682604*v_R5_c_r - 39.6392229058201*v_R5_n_i + 10.2713736253514*v_R5_n_r
        struct[0].g[70,0] = i_R4_n_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 0.679918453285308*v_R12_b_i - 7.72144347466701*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 4.84676915585801*v_R12_n_i + 26.5300693004806*v_R12_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r - 23.284964480954*v_R3_b_i - 24.9575997948692*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058202*v_R3_c_r + 54.0657727682604*v_R3_n_i + 105.57176931318*v_R3_n_r + 18.9579191056578*v_R4_a_i + 87.173059186788*v_R4_a_r + 45.8900105086227*v_R4_b_i + 57.6366430644054*v_R4_b_r + 18.9579191056579*v_R4_c_i + 87.173059186788*v_R4_c_r - 112.978314692379*v_R4_n_i - 237.698607926841*v_R4_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r - 23.284964480954*v_R5_b_i - 24.9575997948692*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058202*v_R5_c_r + 54.0657727682604*v_R5_n_i + 105.57176931318*v_R5_n_r
        struct[0].g[71,0] = i_R4_n_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r - 7.72144347466701*v_R12_b_i - 0.679918453285308*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r + 26.5300693004806*v_R12_n_i - 4.84676915585801*v_R12_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r - 24.9575997948692*v_R3_b_i + 23.284964480954*v_R3_b_r - 39.6392229058202*v_R3_c_i + 10.2713736253513*v_R3_c_r + 105.57176931318*v_R3_n_i - 54.0657727682604*v_R3_n_r + 87.173059186788*v_R4_a_i - 18.9579191056578*v_R4_a_r + 57.6366430644054*v_R4_b_i - 45.8900105086227*v_R4_b_r + 87.173059186788*v_R4_c_i - 18.9579191056579*v_R4_c_r - 237.698607926841*v_R4_n_i + 112.978314692379*v_R4_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r - 24.9575997948692*v_R5_b_i + 23.284964480954*v_R5_b_r - 39.6392229058202*v_R5_c_i + 10.2713736253513*v_R5_c_r + 105.57176931318*v_R5_n_i - 54.0657727682604*v_R5_n_r
        struct[0].g[72,0] = i_R5_a_r + 54.0657727682604*v_R4_a_i + 105.57176931318*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r - 23.284964480954*v_R4_c_i - 24.9575997948692*v_R4_c_r - 10.2713736253513*v_R4_n_i - 39.6392229058202*v_R4_n_r - 108.131545536521*v_R5_a_i - 211.14353862636*v_R5_a_r + 20.5427472507027*v_R5_b_i + 79.2784458116403*v_R5_b_r + 46.569928961908*v_R5_c_i + 49.9151995897384*v_R5_c_r + 20.5427472507027*v_R5_n_i + 79.2784458116403*v_R5_n_r + 54.0657727682604*v_R6_a_i + 105.57176931318*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r - 23.284964480954*v_R6_c_i - 24.9575997948692*v_R6_c_r - 10.2713736253513*v_R6_n_i - 39.6392229058202*v_R6_n_r
        struct[0].g[73,0] = i_R5_a_i + 105.57176931318*v_R4_a_i - 54.0657727682604*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r - 24.9575997948692*v_R4_c_i + 23.284964480954*v_R4_c_r - 39.6392229058202*v_R4_n_i + 10.2713736253513*v_R4_n_r - 211.14353862636*v_R5_a_i + 108.131545536521*v_R5_a_r + 79.2784458116403*v_R5_b_i - 20.5427472507027*v_R5_b_r + 49.9151995897384*v_R5_c_i - 46.569928961908*v_R5_c_r + 79.2784458116403*v_R5_n_i - 20.5427472507027*v_R5_n_r + 105.57176931318*v_R6_a_i - 54.0657727682604*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r - 24.9575997948692*v_R6_c_i + 23.284964480954*v_R6_c_r - 39.6392229058202*v_R6_n_i + 10.2713736253513*v_R6_n_r
        struct[0].g[74,0] = i_R5_b_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r + 54.0657727682604*v_R4_b_i + 105.57176931318*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058201*v_R4_c_r - 23.284964480954*v_R4_n_i - 24.9575997948692*v_R4_n_r + 20.5427472507027*v_R5_a_i + 79.2784458116403*v_R5_a_r - 108.131545536521*v_R5_b_i - 211.14353862636*v_R5_b_r + 20.5427472507027*v_R5_c_i + 79.2784458116403*v_R5_c_r + 46.569928961908*v_R5_n_i + 49.9151995897384*v_R5_n_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r + 54.0657727682604*v_R6_b_i + 105.57176931318*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058201*v_R6_c_r - 23.284964480954*v_R6_n_i - 24.9575997948692*v_R6_n_r
        struct[0].g[75,0] = i_R5_b_i - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r + 105.57176931318*v_R4_b_i - 54.0657727682604*v_R4_b_r - 39.6392229058201*v_R4_c_i + 10.2713736253513*v_R4_c_r - 24.9575997948692*v_R4_n_i + 23.284964480954*v_R4_n_r + 79.2784458116403*v_R5_a_i - 20.5427472507027*v_R5_a_r - 211.14353862636*v_R5_b_i + 108.131545536521*v_R5_b_r + 79.2784458116403*v_R5_c_i - 20.5427472507027*v_R5_c_r + 49.9151995897384*v_R5_n_i - 46.569928961908*v_R5_n_r - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r + 105.57176931318*v_R6_b_i - 54.0657727682604*v_R6_b_r - 39.6392229058201*v_R6_c_i + 10.2713736253513*v_R6_c_r - 24.9575997948692*v_R6_n_i + 23.284964480954*v_R6_n_r
        struct[0].g[76,0] = i_R5_c_r - 23.284964480954*v_R4_a_i - 24.9575997948692*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r + 54.0657727682604*v_R4_c_i + 105.57176931318*v_R4_c_r - 10.2713736253514*v_R4_n_i - 39.6392229058201*v_R4_n_r + 46.569928961908*v_R5_a_i + 49.9151995897384*v_R5_a_r + 20.5427472507027*v_R5_b_i + 79.2784458116404*v_R5_b_r - 108.131545536521*v_R5_c_i - 211.14353862636*v_R5_c_r + 20.5427472507027*v_R5_n_i + 79.2784458116403*v_R5_n_r - 23.284964480954*v_R6_a_i - 24.9575997948692*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r + 54.0657727682604*v_R6_c_i + 105.57176931318*v_R6_c_r - 10.2713736253514*v_R6_n_i - 39.6392229058201*v_R6_n_r
        struct[0].g[77,0] = i_R5_c_i - 24.9575997948692*v_R4_a_i + 23.284964480954*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r + 105.57176931318*v_R4_c_i - 54.0657727682604*v_R4_c_r - 39.6392229058201*v_R4_n_i + 10.2713736253514*v_R4_n_r + 49.9151995897384*v_R5_a_i - 46.569928961908*v_R5_a_r + 79.2784458116404*v_R5_b_i - 20.5427472507027*v_R5_b_r - 211.14353862636*v_R5_c_i + 108.131545536521*v_R5_c_r + 79.2784458116403*v_R5_n_i - 20.5427472507027*v_R5_n_r - 24.9575997948692*v_R6_a_i + 23.284964480954*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r + 105.57176931318*v_R6_c_i - 54.0657727682604*v_R6_c_r - 39.6392229058201*v_R6_n_i + 10.2713736253514*v_R6_n_r
        struct[0].g[78,0] = i_R5_n_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r - 23.284964480954*v_R4_b_i - 24.9575997948692*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058202*v_R4_c_r + 54.0657727682604*v_R4_n_i + 105.57176931318*v_R4_n_r + 20.5427472507027*v_R5_a_i + 79.2784458116403*v_R5_a_r + 46.569928961908*v_R5_b_i + 49.9151995897384*v_R5_b_r + 20.5427472507027*v_R5_c_i + 79.2784458116403*v_R5_c_r - 108.131545536521*v_R5_n_i - 211.14353862636*v_R5_n_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r - 23.284964480954*v_R6_b_i - 24.9575997948692*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058202*v_R6_c_r + 54.0657727682604*v_R6_n_i + 105.57176931318*v_R6_n_r
        struct[0].g[79,0] = i_R5_n_i - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r - 24.9575997948692*v_R4_b_i + 23.284964480954*v_R4_b_r - 39.6392229058202*v_R4_c_i + 10.2713736253513*v_R4_c_r + 105.57176931318*v_R4_n_i - 54.0657727682604*v_R4_n_r + 79.2784458116403*v_R5_a_i - 20.5427472507027*v_R5_a_r + 49.9151995897384*v_R5_b_i - 46.569928961908*v_R5_b_r + 79.2784458116403*v_R5_c_i - 20.5427472507027*v_R5_c_r - 211.14353862636*v_R5_n_i + 108.131545536521*v_R5_n_r - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r - 24.9575997948692*v_R6_b_i + 23.284964480954*v_R6_b_r - 39.6392229058202*v_R6_c_i + 10.2713736253513*v_R6_c_r + 105.57176931318*v_R6_n_i - 54.0657727682604*v_R6_n_r
        struct[0].g[80,0] = i_R6_a_r + 5.65456401516768*v_R16_a_i + 30.9517475172273*v_R16_a_r + 1.84896616921897*v_R16_b_i - 9.21038227100566*v_R16_b_r + 0.793238195499529*v_R16_c_i - 9.00835072044485*v_R16_c_r + 1.84896616921897*v_R16_n_i - 9.21038227100566*v_R16_n_r + 54.0657727682604*v_R5_a_i + 105.57176931318*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r - 23.284964480954*v_R5_c_i - 24.9575997948692*v_R5_c_r - 10.2713736253513*v_R5_n_i - 39.6392229058202*v_R5_n_r - 113.786109551688*v_R6_a_i - 242.095286143588*v_R6_a_r + 18.6937810814837*v_R6_b_i + 88.488828082646*v_R6_b_r + 45.7766907664085*v_R6_c_i + 58.9235503101833*v_R6_c_r + 18.6937810814837*v_R6_n_i + 88.488828082646*v_R6_n_r + 54.0657727682604*v_R7_a_i + 105.57176931318*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r - 23.284964480954*v_R7_c_i - 24.9575997948692*v_R7_c_r - 10.2713736253513*v_R7_n_i - 39.6392229058202*v_R7_n_r
        struct[0].g[81,0] = i_R6_a_i + 30.9517475172273*v_R16_a_i - 5.65456401516768*v_R16_a_r - 9.21038227100566*v_R16_b_i - 1.84896616921897*v_R16_b_r - 9.00835072044485*v_R16_c_i - 0.793238195499529*v_R16_c_r - 9.21038227100566*v_R16_n_i - 1.84896616921897*v_R16_n_r + 105.57176931318*v_R5_a_i - 54.0657727682604*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r - 24.9575997948692*v_R5_c_i + 23.284964480954*v_R5_c_r - 39.6392229058202*v_R5_n_i + 10.2713736253513*v_R5_n_r - 242.095286143588*v_R6_a_i + 113.786109551688*v_R6_a_r + 88.488828082646*v_R6_b_i - 18.6937810814837*v_R6_b_r + 58.9235503101833*v_R6_c_i - 45.7766907664085*v_R6_c_r + 88.488828082646*v_R6_n_i - 18.6937810814837*v_R6_n_r + 105.57176931318*v_R7_a_i - 54.0657727682604*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r - 24.9575997948692*v_R7_c_i + 23.284964480954*v_R7_c_r - 39.6392229058202*v_R7_n_i + 10.2713736253513*v_R7_n_r
        struct[0].g[82,0] = i_R6_b_r + 1.84896616921897*v_R16_a_i - 9.21038227100566*v_R16_a_r + 5.65456401516768*v_R16_b_i + 30.9517475172273*v_R16_b_r + 1.84896616921897*v_R16_c_i - 9.21038227100566*v_R16_c_r + 0.793238195499528*v_R16_n_i - 9.00835072044485*v_R16_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r + 54.0657727682604*v_R5_b_i + 105.57176931318*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058201*v_R5_c_r - 23.284964480954*v_R5_n_i - 24.9575997948692*v_R5_n_r + 18.6937810814837*v_R6_a_i + 88.488828082646*v_R6_a_r - 113.786109551688*v_R6_b_i - 242.095286143588*v_R6_b_r + 18.6937810814837*v_R6_c_i + 88.488828082646*v_R6_c_r + 45.7766907664085*v_R6_n_i + 58.9235503101833*v_R6_n_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r + 54.0657727682604*v_R7_b_i + 105.57176931318*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058201*v_R7_c_r - 23.284964480954*v_R7_n_i - 24.9575997948692*v_R7_n_r
        struct[0].g[83,0] = i_R6_b_i - 9.21038227100566*v_R16_a_i - 1.84896616921897*v_R16_a_r + 30.9517475172273*v_R16_b_i - 5.65456401516768*v_R16_b_r - 9.21038227100566*v_R16_c_i - 1.84896616921897*v_R16_c_r - 9.00835072044485*v_R16_n_i - 0.793238195499528*v_R16_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r + 105.57176931318*v_R5_b_i - 54.0657727682604*v_R5_b_r - 39.6392229058201*v_R5_c_i + 10.2713736253513*v_R5_c_r - 24.9575997948692*v_R5_n_i + 23.284964480954*v_R5_n_r + 88.488828082646*v_R6_a_i - 18.6937810814837*v_R6_a_r - 242.095286143588*v_R6_b_i + 113.786109551688*v_R6_b_r + 88.488828082646*v_R6_c_i - 18.6937810814837*v_R6_c_r + 58.9235503101833*v_R6_n_i - 45.7766907664085*v_R6_n_r - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r + 105.57176931318*v_R7_b_i - 54.0657727682604*v_R7_b_r - 39.6392229058201*v_R7_c_i + 10.2713736253513*v_R7_c_r - 24.9575997948692*v_R7_n_i + 23.284964480954*v_R7_n_r
        struct[0].g[84,0] = i_R6_c_r + 0.793238195499527*v_R16_a_i - 9.00835072044484*v_R16_a_r + 1.84896616921897*v_R16_b_i - 9.21038227100566*v_R16_b_r + 5.65456401516768*v_R16_c_i + 30.9517475172273*v_R16_c_r + 1.84896616921897*v_R16_n_i - 9.21038227100566*v_R16_n_r - 23.284964480954*v_R5_a_i - 24.9575997948692*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r + 54.0657727682604*v_R5_c_i + 105.57176931318*v_R5_c_r - 10.2713736253514*v_R5_n_i - 39.6392229058201*v_R5_n_r + 45.7766907664085*v_R6_a_i + 58.9235503101833*v_R6_a_r + 18.6937810814837*v_R6_b_i + 88.4888280826461*v_R6_b_r - 113.786109551688*v_R6_c_i - 242.095286143588*v_R6_c_r + 18.6937810814837*v_R6_n_i + 88.4888280826459*v_R6_n_r - 23.284964480954*v_R7_a_i - 24.9575997948692*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r + 54.0657727682604*v_R7_c_i + 105.57176931318*v_R7_c_r - 10.2713736253514*v_R7_n_i - 39.6392229058201*v_R7_n_r
        struct[0].g[85,0] = i_R6_c_i - 9.00835072044484*v_R16_a_i - 0.793238195499527*v_R16_a_r - 9.21038227100566*v_R16_b_i - 1.84896616921897*v_R16_b_r + 30.9517475172273*v_R16_c_i - 5.65456401516768*v_R16_c_r - 9.21038227100566*v_R16_n_i - 1.84896616921897*v_R16_n_r - 24.9575997948692*v_R5_a_i + 23.284964480954*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r + 105.57176931318*v_R5_c_i - 54.0657727682604*v_R5_c_r - 39.6392229058201*v_R5_n_i + 10.2713736253514*v_R5_n_r + 58.9235503101833*v_R6_a_i - 45.7766907664085*v_R6_a_r + 88.4888280826461*v_R6_b_i - 18.6937810814837*v_R6_b_r - 242.095286143588*v_R6_c_i + 113.786109551688*v_R6_c_r + 88.4888280826459*v_R6_n_i - 18.6937810814837*v_R6_n_r - 24.9575997948692*v_R7_a_i + 23.284964480954*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r + 105.57176931318*v_R7_c_i - 54.0657727682604*v_R7_c_r - 39.6392229058201*v_R7_n_i + 10.2713736253514*v_R7_n_r
        struct[0].g[86,0] = i_R6_n_r + 1.84896616921897*v_R16_a_i - 9.21038227100566*v_R16_a_r + 0.793238195499527*v_R16_b_i - 9.00835072044485*v_R16_b_r + 1.84896616921897*v_R16_c_i - 9.21038227100566*v_R16_c_r + 5.65456401516768*v_R16_n_i + 30.9517475172273*v_R16_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r - 23.284964480954*v_R5_b_i - 24.9575997948692*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058202*v_R5_c_r + 54.0657727682604*v_R5_n_i + 105.57176931318*v_R5_n_r + 18.6937810814837*v_R6_a_i + 88.488828082646*v_R6_a_r + 45.7766907664085*v_R6_b_i + 58.9235503101832*v_R6_b_r + 18.6937810814837*v_R6_c_i + 88.488828082646*v_R6_c_r - 113.786109551688*v_R6_n_i - 242.120286143588*v_R6_n_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r - 23.284964480954*v_R7_b_i - 24.9575997948692*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058202*v_R7_c_r + 54.0657727682604*v_R7_n_i + 105.57176931318*v_R7_n_r
        struct[0].g[87,0] = i_R6_n_i - 9.21038227100566*v_R16_a_i - 1.84896616921897*v_R16_a_r - 9.00835072044485*v_R16_b_i - 0.793238195499527*v_R16_b_r - 9.21038227100566*v_R16_c_i - 1.84896616921897*v_R16_c_r + 30.9517475172273*v_R16_n_i - 5.65456401516768*v_R16_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r - 24.9575997948692*v_R5_b_i + 23.284964480954*v_R5_b_r - 39.6392229058202*v_R5_c_i + 10.2713736253513*v_R5_c_r + 105.57176931318*v_R5_n_i - 54.0657727682604*v_R5_n_r + 88.488828082646*v_R6_a_i - 18.6937810814837*v_R6_a_r + 58.9235503101832*v_R6_b_i - 45.7766907664085*v_R6_b_r + 88.488828082646*v_R6_c_i - 18.6937810814837*v_R6_c_r - 242.120286143588*v_R6_n_i + 113.786109551688*v_R6_n_r - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r - 24.9575997948692*v_R7_b_i + 23.284964480954*v_R7_b_r - 39.6392229058202*v_R7_c_i + 10.2713736253513*v_R7_c_r + 105.57176931318*v_R7_n_i - 54.0657727682604*v_R7_n_r
        struct[0].g[88,0] = i_R7_a_r + 54.0657727682604*v_R6_a_i + 105.57176931318*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r - 23.284964480954*v_R6_c_i - 24.9575997948692*v_R6_c_r - 10.2713736253513*v_R6_n_i - 39.6392229058202*v_R6_n_r - 108.131545536521*v_R7_a_i - 211.14353862636*v_R7_a_r + 20.5427472507027*v_R7_b_i + 79.2784458116403*v_R7_b_r + 46.569928961908*v_R7_c_i + 49.9151995897384*v_R7_c_r + 20.5427472507027*v_R7_n_i + 79.2784458116403*v_R7_n_r + 54.0657727682604*v_R8_a_i + 105.57176931318*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r - 23.284964480954*v_R8_c_i - 24.9575997948692*v_R8_c_r - 10.2713736253513*v_R8_n_i - 39.6392229058202*v_R8_n_r
        struct[0].g[89,0] = i_R7_a_i + 105.57176931318*v_R6_a_i - 54.0657727682604*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r - 24.9575997948692*v_R6_c_i + 23.284964480954*v_R6_c_r - 39.6392229058202*v_R6_n_i + 10.2713736253513*v_R6_n_r - 211.14353862636*v_R7_a_i + 108.131545536521*v_R7_a_r + 79.2784458116403*v_R7_b_i - 20.5427472507027*v_R7_b_r + 49.9151995897384*v_R7_c_i - 46.569928961908*v_R7_c_r + 79.2784458116403*v_R7_n_i - 20.5427472507027*v_R7_n_r + 105.57176931318*v_R8_a_i - 54.0657727682604*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r - 24.9575997948692*v_R8_c_i + 23.284964480954*v_R8_c_r - 39.6392229058202*v_R8_n_i + 10.2713736253513*v_R8_n_r
        struct[0].g[90,0] = i_R7_b_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r + 54.0657727682604*v_R6_b_i + 105.57176931318*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058201*v_R6_c_r - 23.284964480954*v_R6_n_i - 24.9575997948692*v_R6_n_r + 20.5427472507027*v_R7_a_i + 79.2784458116403*v_R7_a_r - 108.131545536521*v_R7_b_i - 211.14353862636*v_R7_b_r + 20.5427472507027*v_R7_c_i + 79.2784458116403*v_R7_c_r + 46.569928961908*v_R7_n_i + 49.9151995897384*v_R7_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r + 54.0657727682604*v_R8_b_i + 105.57176931318*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058201*v_R8_c_r - 23.284964480954*v_R8_n_i - 24.9575997948692*v_R8_n_r
        struct[0].g[91,0] = i_R7_b_i - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r + 105.57176931318*v_R6_b_i - 54.0657727682604*v_R6_b_r - 39.6392229058201*v_R6_c_i + 10.2713736253513*v_R6_c_r - 24.9575997948692*v_R6_n_i + 23.284964480954*v_R6_n_r + 79.2784458116403*v_R7_a_i - 20.5427472507027*v_R7_a_r - 211.14353862636*v_R7_b_i + 108.131545536521*v_R7_b_r + 79.2784458116403*v_R7_c_i - 20.5427472507027*v_R7_c_r + 49.9151995897384*v_R7_n_i - 46.569928961908*v_R7_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r + 105.57176931318*v_R8_b_i - 54.0657727682604*v_R8_b_r - 39.6392229058201*v_R8_c_i + 10.2713736253513*v_R8_c_r - 24.9575997948692*v_R8_n_i + 23.284964480954*v_R8_n_r
        struct[0].g[92,0] = i_R7_c_r - 23.284964480954*v_R6_a_i - 24.9575997948692*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r + 54.0657727682604*v_R6_c_i + 105.57176931318*v_R6_c_r - 10.2713736253514*v_R6_n_i - 39.6392229058201*v_R6_n_r + 46.569928961908*v_R7_a_i + 49.9151995897384*v_R7_a_r + 20.5427472507027*v_R7_b_i + 79.2784458116404*v_R7_b_r - 108.131545536521*v_R7_c_i - 211.14353862636*v_R7_c_r + 20.5427472507027*v_R7_n_i + 79.2784458116403*v_R7_n_r - 23.284964480954*v_R8_a_i - 24.9575997948692*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r + 54.0657727682604*v_R8_c_i + 105.57176931318*v_R8_c_r - 10.2713736253514*v_R8_n_i - 39.6392229058201*v_R8_n_r
        struct[0].g[93,0] = i_R7_c_i - 24.9575997948692*v_R6_a_i + 23.284964480954*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r + 105.57176931318*v_R6_c_i - 54.0657727682604*v_R6_c_r - 39.6392229058201*v_R6_n_i + 10.2713736253514*v_R6_n_r + 49.9151995897384*v_R7_a_i - 46.569928961908*v_R7_a_r + 79.2784458116404*v_R7_b_i - 20.5427472507027*v_R7_b_r - 211.14353862636*v_R7_c_i + 108.131545536521*v_R7_c_r + 79.2784458116403*v_R7_n_i - 20.5427472507027*v_R7_n_r - 24.9575997948692*v_R8_a_i + 23.284964480954*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r + 105.57176931318*v_R8_c_i - 54.0657727682604*v_R8_c_r - 39.6392229058201*v_R8_n_i + 10.2713736253514*v_R8_n_r
        struct[0].g[94,0] = i_R7_n_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r - 23.284964480954*v_R6_b_i - 24.9575997948692*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058202*v_R6_c_r + 54.0657727682604*v_R6_n_i + 105.57176931318*v_R6_n_r + 20.5427472507027*v_R7_a_i + 79.2784458116403*v_R7_a_r + 46.569928961908*v_R7_b_i + 49.9151995897384*v_R7_b_r + 20.5427472507027*v_R7_c_i + 79.2784458116403*v_R7_c_r - 108.131545536521*v_R7_n_i - 211.14353862636*v_R7_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r - 23.284964480954*v_R8_b_i - 24.9575997948692*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058202*v_R8_c_r + 54.0657727682604*v_R8_n_i + 105.57176931318*v_R8_n_r
        struct[0].g[95,0] = i_R7_n_i - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r - 24.9575997948692*v_R6_b_i + 23.284964480954*v_R6_b_r - 39.6392229058202*v_R6_c_i + 10.2713736253513*v_R6_c_r + 105.57176931318*v_R6_n_i - 54.0657727682604*v_R6_n_r + 79.2784458116403*v_R7_a_i - 20.5427472507027*v_R7_a_r + 49.9151995897384*v_R7_b_i - 46.569928961908*v_R7_b_r + 79.2784458116403*v_R7_c_i - 20.5427472507027*v_R7_c_r - 211.14353862636*v_R7_n_i + 108.131545536521*v_R7_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r - 24.9575997948692*v_R8_b_i + 23.284964480954*v_R8_b_r - 39.6392229058202*v_R8_c_i + 10.2713736253513*v_R8_c_r + 105.57176931318*v_R8_n_i - 54.0657727682604*v_R8_n_r
        struct[0].g[96,0] = i_R8_a_r + 54.0657727682604*v_R7_a_i + 105.57176931318*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r - 23.284964480954*v_R7_c_i - 24.9575997948692*v_R7_c_r - 10.2713736253513*v_R7_n_i - 39.6392229058202*v_R7_n_r - 108.131545536521*v_R8_a_i - 211.14353862636*v_R8_a_r + 20.5427472507027*v_R8_b_i + 79.2784458116403*v_R8_b_r + 46.569928961908*v_R8_c_i + 49.9151995897384*v_R8_c_r + 20.5427472507027*v_R8_n_i + 79.2784458116403*v_R8_n_r + 54.0657727682604*v_R9_a_i + 105.57176931318*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r - 23.284964480954*v_R9_c_i - 24.9575997948692*v_R9_c_r - 10.2713736253513*v_R9_n_i - 39.6392229058202*v_R9_n_r
        struct[0].g[97,0] = i_R8_a_i + 105.57176931318*v_R7_a_i - 54.0657727682604*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r - 24.9575997948692*v_R7_c_i + 23.284964480954*v_R7_c_r - 39.6392229058202*v_R7_n_i + 10.2713736253513*v_R7_n_r - 211.14353862636*v_R8_a_i + 108.131545536521*v_R8_a_r + 79.2784458116403*v_R8_b_i - 20.5427472507027*v_R8_b_r + 49.9151995897384*v_R8_c_i - 46.569928961908*v_R8_c_r + 79.2784458116403*v_R8_n_i - 20.5427472507027*v_R8_n_r + 105.57176931318*v_R9_a_i - 54.0657727682604*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r - 24.9575997948692*v_R9_c_i + 23.284964480954*v_R9_c_r - 39.6392229058202*v_R9_n_i + 10.2713736253513*v_R9_n_r
        struct[0].g[98,0] = i_R8_b_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r + 54.0657727682604*v_R7_b_i + 105.57176931318*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058201*v_R7_c_r - 23.284964480954*v_R7_n_i - 24.9575997948692*v_R7_n_r + 20.5427472507027*v_R8_a_i + 79.2784458116403*v_R8_a_r - 108.131545536521*v_R8_b_i - 211.14353862636*v_R8_b_r + 20.5427472507027*v_R8_c_i + 79.2784458116403*v_R8_c_r + 46.569928961908*v_R8_n_i + 49.9151995897384*v_R8_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r + 54.0657727682604*v_R9_b_i + 105.57176931318*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058201*v_R9_c_r - 23.284964480954*v_R9_n_i - 24.9575997948692*v_R9_n_r
        struct[0].g[99,0] = i_R8_b_i - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r + 105.57176931318*v_R7_b_i - 54.0657727682604*v_R7_b_r - 39.6392229058201*v_R7_c_i + 10.2713736253513*v_R7_c_r - 24.9575997948692*v_R7_n_i + 23.284964480954*v_R7_n_r + 79.2784458116403*v_R8_a_i - 20.5427472507027*v_R8_a_r - 211.14353862636*v_R8_b_i + 108.131545536521*v_R8_b_r + 79.2784458116403*v_R8_c_i - 20.5427472507027*v_R8_c_r + 49.9151995897384*v_R8_n_i - 46.569928961908*v_R8_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r + 105.57176931318*v_R9_b_i - 54.0657727682604*v_R9_b_r - 39.6392229058201*v_R9_c_i + 10.2713736253513*v_R9_c_r - 24.9575997948692*v_R9_n_i + 23.284964480954*v_R9_n_r
        struct[0].g[100,0] = i_R8_c_r - 23.284964480954*v_R7_a_i - 24.9575997948692*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r + 54.0657727682604*v_R7_c_i + 105.57176931318*v_R7_c_r - 10.2713736253514*v_R7_n_i - 39.6392229058201*v_R7_n_r + 46.569928961908*v_R8_a_i + 49.9151995897384*v_R8_a_r + 20.5427472507027*v_R8_b_i + 79.2784458116404*v_R8_b_r - 108.131545536521*v_R8_c_i - 211.14353862636*v_R8_c_r + 20.5427472507027*v_R8_n_i + 79.2784458116403*v_R8_n_r - 23.284964480954*v_R9_a_i - 24.9575997948692*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r + 54.0657727682604*v_R9_c_i + 105.57176931318*v_R9_c_r - 10.2713736253514*v_R9_n_i - 39.6392229058201*v_R9_n_r
        struct[0].g[101,0] = i_R8_c_i - 24.9575997948692*v_R7_a_i + 23.284964480954*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r + 105.57176931318*v_R7_c_i - 54.0657727682604*v_R7_c_r - 39.6392229058201*v_R7_n_i + 10.2713736253514*v_R7_n_r + 49.9151995897384*v_R8_a_i - 46.569928961908*v_R8_a_r + 79.2784458116404*v_R8_b_i - 20.5427472507027*v_R8_b_r - 211.14353862636*v_R8_c_i + 108.131545536521*v_R8_c_r + 79.2784458116403*v_R8_n_i - 20.5427472507027*v_R8_n_r - 24.9575997948692*v_R9_a_i + 23.284964480954*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r + 105.57176931318*v_R9_c_i - 54.0657727682604*v_R9_c_r - 39.6392229058201*v_R9_n_i + 10.2713736253514*v_R9_n_r
        struct[0].g[102,0] = i_R8_n_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r - 23.284964480954*v_R7_b_i - 24.9575997948692*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058202*v_R7_c_r + 54.0657727682604*v_R7_n_i + 105.57176931318*v_R7_n_r + 20.5427472507027*v_R8_a_i + 79.2784458116403*v_R8_a_r + 46.569928961908*v_R8_b_i + 49.9151995897384*v_R8_b_r + 20.5427472507027*v_R8_c_i + 79.2784458116403*v_R8_c_r - 108.131545536521*v_R8_n_i - 211.16853862636*v_R8_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r - 23.284964480954*v_R9_b_i - 24.9575997948692*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058202*v_R9_c_r + 54.0657727682604*v_R9_n_i + 105.57176931318*v_R9_n_r
        struct[0].g[103,0] = i_R8_n_i - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r - 24.9575997948692*v_R7_b_i + 23.284964480954*v_R7_b_r - 39.6392229058202*v_R7_c_i + 10.2713736253513*v_R7_c_r + 105.57176931318*v_R7_n_i - 54.0657727682604*v_R7_n_r + 79.2784458116403*v_R8_a_i - 20.5427472507027*v_R8_a_r + 49.9151995897384*v_R8_b_i - 46.569928961908*v_R8_b_r + 79.2784458116403*v_R8_c_i - 20.5427472507027*v_R8_c_r - 211.16853862636*v_R8_n_i + 108.131545536521*v_R8_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r - 24.9575997948692*v_R9_b_i + 23.284964480954*v_R9_b_r - 39.6392229058202*v_R9_c_i + 10.2713736253513*v_R9_c_r + 105.57176931318*v_R9_n_i - 54.0657727682604*v_R9_n_r
        struct[0].g[104,0] = i_R9_a_r + 54.0657727682604*v_R10_a_i + 105.57176931318*v_R10_a_r - 10.2713736253513*v_R10_b_i - 39.6392229058202*v_R10_b_r - 23.284964480954*v_R10_c_i - 24.9575997948692*v_R10_c_r - 10.2713736253513*v_R10_n_i - 39.6392229058202*v_R10_n_r + 5.65456401516768*v_R17_a_i + 30.9517475172273*v_R17_a_r + 1.84896616921897*v_R17_b_i - 9.21038227100566*v_R17_b_r + 0.793238195499529*v_R17_c_i - 9.00835072044485*v_R17_c_r + 1.84896616921897*v_R17_n_i - 9.21038227100566*v_R17_n_r + 54.0657727682604*v_R8_a_i + 105.57176931318*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r - 23.284964480954*v_R8_c_i - 24.9575997948692*v_R8_c_r - 10.2713736253513*v_R8_n_i - 39.6392229058202*v_R8_n_r - 113.786109551688*v_R9_a_i - 242.095286143588*v_R9_a_r + 18.6937810814837*v_R9_b_i + 88.488828082646*v_R9_b_r + 45.7766907664085*v_R9_c_i + 58.9235503101833*v_R9_c_r + 18.6937810814837*v_R9_n_i + 88.488828082646*v_R9_n_r
        struct[0].g[105,0] = i_R9_a_i + 105.57176931318*v_R10_a_i - 54.0657727682604*v_R10_a_r - 39.6392229058202*v_R10_b_i + 10.2713736253513*v_R10_b_r - 24.9575997948692*v_R10_c_i + 23.284964480954*v_R10_c_r - 39.6392229058202*v_R10_n_i + 10.2713736253513*v_R10_n_r + 30.9517475172273*v_R17_a_i - 5.65456401516768*v_R17_a_r - 9.21038227100566*v_R17_b_i - 1.84896616921897*v_R17_b_r - 9.00835072044485*v_R17_c_i - 0.793238195499529*v_R17_c_r - 9.21038227100566*v_R17_n_i - 1.84896616921897*v_R17_n_r + 105.57176931318*v_R8_a_i - 54.0657727682604*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r - 24.9575997948692*v_R8_c_i + 23.284964480954*v_R8_c_r - 39.6392229058202*v_R8_n_i + 10.2713736253513*v_R8_n_r - 242.095286143588*v_R9_a_i + 113.786109551688*v_R9_a_r + 88.488828082646*v_R9_b_i - 18.6937810814837*v_R9_b_r + 58.9235503101833*v_R9_c_i - 45.7766907664085*v_R9_c_r + 88.488828082646*v_R9_n_i - 18.6937810814837*v_R9_n_r
        struct[0].g[106,0] = i_R9_b_r - 10.2713736253513*v_R10_a_i - 39.6392229058202*v_R10_a_r + 54.0657727682604*v_R10_b_i + 105.57176931318*v_R10_b_r - 10.2713736253513*v_R10_c_i - 39.6392229058201*v_R10_c_r - 23.284964480954*v_R10_n_i - 24.9575997948692*v_R10_n_r + 1.84896616921897*v_R17_a_i - 9.21038227100566*v_R17_a_r + 5.65456401516768*v_R17_b_i + 30.9517475172273*v_R17_b_r + 1.84896616921897*v_R17_c_i - 9.21038227100566*v_R17_c_r + 0.793238195499528*v_R17_n_i - 9.00835072044485*v_R17_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r + 54.0657727682604*v_R8_b_i + 105.57176931318*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058201*v_R8_c_r - 23.284964480954*v_R8_n_i - 24.9575997948692*v_R8_n_r + 18.6937810814837*v_R9_a_i + 88.488828082646*v_R9_a_r - 113.786109551688*v_R9_b_i - 242.095286143588*v_R9_b_r + 18.6937810814837*v_R9_c_i + 88.488828082646*v_R9_c_r + 45.7766907664085*v_R9_n_i + 58.9235503101833*v_R9_n_r
        struct[0].g[107,0] = i_R9_b_i - 39.6392229058202*v_R10_a_i + 10.2713736253513*v_R10_a_r + 105.57176931318*v_R10_b_i - 54.0657727682604*v_R10_b_r - 39.6392229058201*v_R10_c_i + 10.2713736253513*v_R10_c_r - 24.9575997948692*v_R10_n_i + 23.284964480954*v_R10_n_r - 9.21038227100566*v_R17_a_i - 1.84896616921897*v_R17_a_r + 30.9517475172273*v_R17_b_i - 5.65456401516768*v_R17_b_r - 9.21038227100566*v_R17_c_i - 1.84896616921897*v_R17_c_r - 9.00835072044485*v_R17_n_i - 0.793238195499528*v_R17_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r + 105.57176931318*v_R8_b_i - 54.0657727682604*v_R8_b_r - 39.6392229058201*v_R8_c_i + 10.2713736253513*v_R8_c_r - 24.9575997948692*v_R8_n_i + 23.284964480954*v_R8_n_r + 88.488828082646*v_R9_a_i - 18.6937810814837*v_R9_a_r - 242.095286143588*v_R9_b_i + 113.786109551688*v_R9_b_r + 88.488828082646*v_R9_c_i - 18.6937810814837*v_R9_c_r + 58.9235503101833*v_R9_n_i - 45.7766907664085*v_R9_n_r
        struct[0].g[108,0] = i_R9_c_r - 23.284964480954*v_R10_a_i - 24.9575997948692*v_R10_a_r - 10.2713736253513*v_R10_b_i - 39.6392229058202*v_R10_b_r + 54.0657727682604*v_R10_c_i + 105.57176931318*v_R10_c_r - 10.2713736253514*v_R10_n_i - 39.6392229058201*v_R10_n_r + 0.793238195499527*v_R17_a_i - 9.00835072044484*v_R17_a_r + 1.84896616921897*v_R17_b_i - 9.21038227100566*v_R17_b_r + 5.65456401516768*v_R17_c_i + 30.9517475172273*v_R17_c_r + 1.84896616921897*v_R17_n_i - 9.21038227100566*v_R17_n_r - 23.284964480954*v_R8_a_i - 24.9575997948692*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r + 54.0657727682604*v_R8_c_i + 105.57176931318*v_R8_c_r - 10.2713736253514*v_R8_n_i - 39.6392229058201*v_R8_n_r + 45.7766907664085*v_R9_a_i + 58.9235503101833*v_R9_a_r + 18.6937810814837*v_R9_b_i + 88.4888280826461*v_R9_b_r - 113.786109551688*v_R9_c_i - 242.095286143588*v_R9_c_r + 18.6937810814837*v_R9_n_i + 88.4888280826459*v_R9_n_r
        struct[0].g[109,0] = i_R9_c_i - 24.9575997948692*v_R10_a_i + 23.284964480954*v_R10_a_r - 39.6392229058202*v_R10_b_i + 10.2713736253513*v_R10_b_r + 105.57176931318*v_R10_c_i - 54.0657727682604*v_R10_c_r - 39.6392229058201*v_R10_n_i + 10.2713736253514*v_R10_n_r - 9.00835072044484*v_R17_a_i - 0.793238195499527*v_R17_a_r - 9.21038227100566*v_R17_b_i - 1.84896616921897*v_R17_b_r + 30.9517475172273*v_R17_c_i - 5.65456401516768*v_R17_c_r - 9.21038227100566*v_R17_n_i - 1.84896616921897*v_R17_n_r - 24.9575997948692*v_R8_a_i + 23.284964480954*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r + 105.57176931318*v_R8_c_i - 54.0657727682604*v_R8_c_r - 39.6392229058201*v_R8_n_i + 10.2713736253514*v_R8_n_r + 58.9235503101833*v_R9_a_i - 45.7766907664085*v_R9_a_r + 88.4888280826461*v_R9_b_i - 18.6937810814837*v_R9_b_r - 242.095286143588*v_R9_c_i + 113.786109551688*v_R9_c_r + 88.4888280826459*v_R9_n_i - 18.6937810814837*v_R9_n_r
        struct[0].g[110,0] = i_R9_n_r - 10.2713736253513*v_R10_a_i - 39.6392229058202*v_R10_a_r - 23.284964480954*v_R10_b_i - 24.9575997948692*v_R10_b_r - 10.2713736253513*v_R10_c_i - 39.6392229058202*v_R10_c_r + 54.0657727682604*v_R10_n_i + 105.57176931318*v_R10_n_r + 1.84896616921897*v_R17_a_i - 9.21038227100566*v_R17_a_r + 0.793238195499527*v_R17_b_i - 9.00835072044485*v_R17_b_r + 1.84896616921897*v_R17_c_i - 9.21038227100566*v_R17_c_r + 5.65456401516768*v_R17_n_i + 30.9517475172273*v_R17_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r - 23.284964480954*v_R8_b_i - 24.9575997948692*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058202*v_R8_c_r + 54.0657727682604*v_R8_n_i + 105.57176931318*v_R8_n_r + 18.6937810814837*v_R9_a_i + 88.488828082646*v_R9_a_r + 45.7766907664085*v_R9_b_i + 58.9235503101832*v_R9_b_r + 18.6937810814837*v_R9_c_i + 88.488828082646*v_R9_c_r - 113.786109551688*v_R9_n_i - 242.095286143588*v_R9_n_r
        struct[0].g[111,0] = i_R9_n_i - 39.6392229058202*v_R10_a_i + 10.2713736253513*v_R10_a_r - 24.9575997948692*v_R10_b_i + 23.284964480954*v_R10_b_r - 39.6392229058202*v_R10_c_i + 10.2713736253513*v_R10_c_r + 105.57176931318*v_R10_n_i - 54.0657727682604*v_R10_n_r - 9.21038227100566*v_R17_a_i - 1.84896616921897*v_R17_a_r - 9.00835072044485*v_R17_b_i - 0.793238195499527*v_R17_b_r - 9.21038227100566*v_R17_c_i - 1.84896616921897*v_R17_c_r + 30.9517475172273*v_R17_n_i - 5.65456401516768*v_R17_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r - 24.9575997948692*v_R8_b_i + 23.284964480954*v_R8_b_r - 39.6392229058202*v_R8_c_i + 10.2713736253513*v_R8_c_r + 105.57176931318*v_R8_n_i - 54.0657727682604*v_R8_n_r + 88.488828082646*v_R9_a_i - 18.6937810814837*v_R9_a_r + 58.9235503101832*v_R9_b_i - 45.7766907664085*v_R9_b_r + 88.488828082646*v_R9_c_i - 18.6937810814837*v_R9_c_r - 242.095286143588*v_R9_n_i + 113.786109551688*v_R9_n_r
        struct[0].g[112,0] = i_R10_a_r - 59.7203367834281*v_R10_a_i - 136.523516830407*v_R10_a_r + 8.42240745613237*v_R10_b_i + 48.8496051768258*v_R10_b_r + 22.4917262854545*v_R10_c_i + 33.9659505153141*v_R10_c_r + 8.42240745613237*v_R10_n_i + 48.8496051768258*v_R10_n_r + 5.65456401516768*v_R18_a_i + 30.9517475172273*v_R18_a_r + 1.84896616921897*v_R18_b_i - 9.21038227100566*v_R18_b_r + 0.793238195499529*v_R18_c_i - 9.00835072044485*v_R18_c_r + 1.84896616921897*v_R18_n_i - 9.21038227100566*v_R18_n_r + 54.0657727682604*v_R9_a_i + 105.57176931318*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r - 23.284964480954*v_R9_c_i - 24.9575997948692*v_R9_c_r - 10.2713736253513*v_R9_n_i - 39.6392229058202*v_R9_n_r
        struct[0].g[113,0] = i_R10_a_i - 136.523516830407*v_R10_a_i + 59.7203367834281*v_R10_a_r + 48.8496051768258*v_R10_b_i - 8.42240745613237*v_R10_b_r + 33.9659505153141*v_R10_c_i - 22.4917262854545*v_R10_c_r + 48.8496051768258*v_R10_n_i - 8.42240745613237*v_R10_n_r + 30.9517475172273*v_R18_a_i - 5.65456401516768*v_R18_a_r - 9.21038227100566*v_R18_b_i - 1.84896616921897*v_R18_b_r - 9.00835072044485*v_R18_c_i - 0.793238195499529*v_R18_c_r - 9.21038227100566*v_R18_n_i - 1.84896616921897*v_R18_n_r + 105.57176931318*v_R9_a_i - 54.0657727682604*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r - 24.9575997948692*v_R9_c_i + 23.284964480954*v_R9_c_r - 39.6392229058202*v_R9_n_i + 10.2713736253513*v_R9_n_r
        struct[0].g[114,0] = i_R10_b_r + 8.42240745613237*v_R10_a_i + 48.8496051768258*v_R10_a_r - 59.720336783428*v_R10_b_i - 136.523516830407*v_R10_b_r + 8.42240745613238*v_R10_c_i + 48.8496051768258*v_R10_c_r + 22.4917262854545*v_R10_n_i + 33.9659505153141*v_R10_n_r + 1.84896616921897*v_R18_a_i - 9.21038227100566*v_R18_a_r + 5.65456401516768*v_R18_b_i + 30.9517475172273*v_R18_b_r + 1.84896616921897*v_R18_c_i - 9.21038227100566*v_R18_c_r + 0.793238195499528*v_R18_n_i - 9.00835072044485*v_R18_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r + 54.0657727682604*v_R9_b_i + 105.57176931318*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058201*v_R9_c_r - 23.284964480954*v_R9_n_i - 24.9575997948692*v_R9_n_r
        struct[0].g[115,0] = i_R10_b_i + 48.8496051768258*v_R10_a_i - 8.42240745613237*v_R10_a_r - 136.523516830407*v_R10_b_i + 59.720336783428*v_R10_b_r + 48.8496051768258*v_R10_c_i - 8.42240745613238*v_R10_c_r + 33.9659505153141*v_R10_n_i - 22.4917262854545*v_R10_n_r - 9.21038227100566*v_R18_a_i - 1.84896616921897*v_R18_a_r + 30.9517475172273*v_R18_b_i - 5.65456401516768*v_R18_b_r - 9.21038227100566*v_R18_c_i - 1.84896616921897*v_R18_c_r - 9.00835072044485*v_R18_n_i - 0.793238195499528*v_R18_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r + 105.57176931318*v_R9_b_i - 54.0657727682604*v_R9_b_r - 39.6392229058201*v_R9_c_i + 10.2713736253513*v_R9_c_r - 24.9575997948692*v_R9_n_i + 23.284964480954*v_R9_n_r
        struct[0].g[116,0] = i_R10_c_r + 22.4917262854545*v_R10_a_i + 33.9659505153141*v_R10_a_r + 8.42240745613236*v_R10_b_i + 48.8496051768259*v_R10_b_r - 59.7203367834281*v_R10_c_i - 136.523516830407*v_R10_c_r + 8.42240745613239*v_R10_n_i + 48.8496051768258*v_R10_n_r + 0.793238195499527*v_R18_a_i - 9.00835072044484*v_R18_a_r + 1.84896616921897*v_R18_b_i - 9.21038227100566*v_R18_b_r + 5.65456401516768*v_R18_c_i + 30.9517475172273*v_R18_c_r + 1.84896616921897*v_R18_n_i - 9.21038227100566*v_R18_n_r - 23.284964480954*v_R9_a_i - 24.9575997948692*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r + 54.0657727682604*v_R9_c_i + 105.57176931318*v_R9_c_r - 10.2713736253514*v_R9_n_i - 39.6392229058201*v_R9_n_r
        struct[0].g[117,0] = i_R10_c_i + 33.9659505153141*v_R10_a_i - 22.4917262854545*v_R10_a_r + 48.8496051768259*v_R10_b_i - 8.42240745613236*v_R10_b_r - 136.523516830407*v_R10_c_i + 59.7203367834281*v_R10_c_r + 48.8496051768258*v_R10_n_i - 8.42240745613239*v_R10_n_r - 9.00835072044484*v_R18_a_i - 0.793238195499527*v_R18_a_r - 9.21038227100566*v_R18_b_i - 1.84896616921897*v_R18_b_r + 30.9517475172273*v_R18_c_i - 5.65456401516768*v_R18_c_r - 9.21038227100566*v_R18_n_i - 1.84896616921897*v_R18_n_r - 24.9575997948692*v_R9_a_i + 23.284964480954*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r + 105.57176931318*v_R9_c_i - 54.0657727682604*v_R9_c_r - 39.6392229058201*v_R9_n_i + 10.2713736253514*v_R9_n_r
        struct[0].g[118,0] = i_R10_n_r + 8.42240745613237*v_R10_a_i + 48.8496051768258*v_R10_a_r + 22.4917262854545*v_R10_b_i + 33.965950515314*v_R10_b_r + 8.42240745613237*v_R10_c_i + 48.8496051768258*v_R10_c_r - 59.7203367834281*v_R10_n_i - 136.548516830407*v_R10_n_r + 1.84896616921897*v_R18_a_i - 9.21038227100566*v_R18_a_r + 0.793238195499527*v_R18_b_i - 9.00835072044485*v_R18_b_r + 1.84896616921897*v_R18_c_i - 9.21038227100566*v_R18_c_r + 5.65456401516768*v_R18_n_i + 30.9517475172273*v_R18_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r - 23.284964480954*v_R9_b_i - 24.9575997948692*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058202*v_R9_c_r + 54.0657727682604*v_R9_n_i + 105.57176931318*v_R9_n_r
        struct[0].g[119,0] = i_R10_n_i + 48.8496051768258*v_R10_a_i - 8.42240745613237*v_R10_a_r + 33.965950515314*v_R10_b_i - 22.4917262854545*v_R10_b_r + 48.8496051768258*v_R10_c_i - 8.42240745613237*v_R10_c_r - 136.548516830407*v_R10_n_i + 59.7203367834281*v_R10_n_r - 9.21038227100566*v_R18_a_i - 1.84896616921897*v_R18_a_r - 9.00835072044485*v_R18_b_i - 0.793238195499527*v_R18_b_r - 9.21038227100566*v_R18_c_i - 1.84896616921897*v_R18_c_r + 30.9517475172273*v_R18_n_i - 5.65456401516768*v_R18_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r - 24.9575997948692*v_R9_b_i + 23.284964480954*v_R9_b_r - 39.6392229058202*v_R9_c_i + 10.2713736253513*v_R9_c_r + 105.57176931318*v_R9_n_i - 54.0657727682604*v_R9_n_r
        struct[0].g[120,0] = i_R12_a_r - 9.69353831171601*v_R12_a_i - 53.0601386009611*v_R12_a_r - 3.16965629008966*v_R12_b_i + 15.7892267502954*v_R12_b_r - 1.35983690657062*v_R12_c_i + 15.442886949334*v_R12_c_r - 3.16965629008966*v_R12_n_i + 15.7892267502954*v_R12_n_r + 4.84676915585801*v_R13_a_i + 26.5300693004806*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 0.67991845328531*v_R13_c_i - 7.72144347466701*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r + 4.84676915585801*v_R4_a_i + 26.5300693004806*v_R4_a_r + 1.58482814504483*v_R4_b_i - 7.8946133751477*v_R4_b_r + 0.67991845328531*v_R4_c_i - 7.72144347466701*v_R4_c_r + 1.58482814504483*v_R4_n_i - 7.89461337514771*v_R4_n_r
        struct[0].g[121,0] = i_R12_a_i - 53.0601386009611*v_R12_a_i + 9.69353831171601*v_R12_a_r + 15.7892267502954*v_R12_b_i + 3.16965629008966*v_R12_b_r + 15.442886949334*v_R12_c_i + 1.35983690657062*v_R12_c_r + 15.7892267502954*v_R12_n_i + 3.16965629008966*v_R12_n_r + 26.5300693004806*v_R13_a_i - 4.84676915585801*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r - 7.72144347466701*v_R13_c_i - 0.67991845328531*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r + 26.5300693004806*v_R4_a_i - 4.84676915585801*v_R4_a_r - 7.8946133751477*v_R4_b_i - 1.58482814504483*v_R4_b_r - 7.72144347466701*v_R4_c_i - 0.67991845328531*v_R4_c_r - 7.89461337514771*v_R4_n_i - 1.58482814504483*v_R4_n_r
        struct[0].g[122,0] = i_R12_b_r - 3.16965629008966*v_R12_a_i + 15.7892267502954*v_R12_a_r - 9.69353831171602*v_R12_b_i - 53.0601386009611*v_R12_b_r - 3.16965629008967*v_R12_c_i + 15.7892267502954*v_R12_c_r - 1.35983690657062*v_R12_n_i + 15.442886949334*v_R12_n_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 4.84676915585801*v_R13_b_i + 26.5300693004806*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 0.679918453285308*v_R13_n_i - 7.72144347466701*v_R13_n_r + 1.58482814504483*v_R4_a_i - 7.89461337514771*v_R4_a_r + 4.84676915585801*v_R4_b_i + 26.5300693004806*v_R4_b_r + 1.58482814504483*v_R4_c_i - 7.89461337514771*v_R4_c_r + 0.679918453285308*v_R4_n_i - 7.72144347466701*v_R4_n_r
        struct[0].g[123,0] = i_R12_b_i + 15.7892267502954*v_R12_a_i + 3.16965629008966*v_R12_a_r - 53.0601386009611*v_R12_b_i + 9.69353831171602*v_R12_b_r + 15.7892267502954*v_R12_c_i + 3.16965629008967*v_R12_c_r + 15.442886949334*v_R12_n_i + 1.35983690657062*v_R12_n_r - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r + 26.5300693004806*v_R13_b_i - 4.84676915585801*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r - 7.72144347466701*v_R13_n_i - 0.679918453285308*v_R13_n_r - 7.89461337514771*v_R4_a_i - 1.58482814504483*v_R4_a_r + 26.5300693004806*v_R4_b_i - 4.84676915585801*v_R4_b_r - 7.89461337514771*v_R4_c_i - 1.58482814504483*v_R4_c_r - 7.72144347466701*v_R4_n_i - 0.679918453285308*v_R4_n_r
        struct[0].g[124,0] = i_R12_c_r - 1.35983690657062*v_R12_a_i + 15.442886949334*v_R12_a_r - 3.16965629008966*v_R12_b_i + 15.7892267502954*v_R12_b_r - 9.69353831171602*v_R12_c_i - 53.0601386009611*v_R12_c_r - 3.16965629008967*v_R12_n_i + 15.7892267502954*v_R12_n_r + 0.67991845328531*v_R13_a_i - 7.72144347466701*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 4.84676915585801*v_R13_c_i + 26.5300693004806*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r + 0.67991845328531*v_R4_a_i - 7.72144347466701*v_R4_a_r + 1.58482814504483*v_R4_b_i - 7.8946133751477*v_R4_b_r + 4.84676915585801*v_R4_c_i + 26.5300693004806*v_R4_c_r + 1.58482814504483*v_R4_n_i - 7.89461337514771*v_R4_n_r
        struct[0].g[125,0] = i_R12_c_i + 15.442886949334*v_R12_a_i + 1.35983690657062*v_R12_a_r + 15.7892267502954*v_R12_b_i + 3.16965629008966*v_R12_b_r - 53.0601386009611*v_R12_c_i + 9.69353831171602*v_R12_c_r + 15.7892267502954*v_R12_n_i + 3.16965629008967*v_R12_n_r - 7.72144347466701*v_R13_a_i - 0.67991845328531*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r + 26.5300693004806*v_R13_c_i - 4.84676915585801*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r - 7.72144347466701*v_R4_a_i - 0.67991845328531*v_R4_a_r - 7.8946133751477*v_R4_b_i - 1.58482814504483*v_R4_b_r + 26.5300693004806*v_R4_c_i - 4.84676915585801*v_R4_c_r - 7.89461337514771*v_R4_n_i - 1.58482814504483*v_R4_n_r
        struct[0].g[126,0] = i_R12_n_r - 3.16965629008966*v_R12_a_i + 15.7892267502954*v_R12_a_r - 1.35983690657062*v_R12_b_i + 15.442886949334*v_R12_b_r - 3.16965629008966*v_R12_c_i + 15.7892267502954*v_R12_c_r - 9.69353831171602*v_R12_n_i - 53.0601386009611*v_R12_n_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 0.679918453285308*v_R13_b_i - 7.72144347466701*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 4.84676915585801*v_R13_n_i + 26.5300693004806*v_R13_n_r + 1.58482814504483*v_R4_a_i - 7.89461337514771*v_R4_a_r + 0.679918453285308*v_R4_b_i - 7.72144347466701*v_R4_b_r + 1.58482814504483*v_R4_c_i - 7.89461337514771*v_R4_c_r + 4.84676915585801*v_R4_n_i + 26.5300693004806*v_R4_n_r
        struct[0].g[127,0] = i_R12_n_i + 15.7892267502954*v_R12_a_i + 3.16965629008966*v_R12_a_r + 15.442886949334*v_R12_b_i + 1.35983690657062*v_R12_b_r + 15.7892267502954*v_R12_c_i + 3.16965629008966*v_R12_c_r - 53.0601386009611*v_R12_n_i + 9.69353831171602*v_R12_n_r - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r - 7.72144347466701*v_R13_b_i - 0.679918453285308*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r + 26.5300693004806*v_R13_n_i - 4.84676915585801*v_R13_n_r - 7.89461337514771*v_R4_a_i - 1.58482814504483*v_R4_a_r - 7.72144347466701*v_R4_b_i - 0.679918453285308*v_R4_b_r - 7.89461337514771*v_R4_c_i - 1.58482814504483*v_R4_c_r + 26.5300693004806*v_R4_n_i - 4.84676915585801*v_R4_n_r
        struct[0].g[128,0] = i_R13_a_r + 4.84676915585801*v_R12_a_i + 26.5300693004806*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 0.67991845328531*v_R12_c_i - 7.72144347466701*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r - 9.69353831171601*v_R13_a_i - 53.0601386009611*v_R13_a_r - 3.16965629008966*v_R13_b_i + 15.7892267502954*v_R13_b_r - 1.35983690657062*v_R13_c_i + 15.442886949334*v_R13_c_r - 3.16965629008966*v_R13_n_i + 15.7892267502954*v_R13_n_r + 4.84676915585801*v_R14_a_i + 26.5300693004806*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 0.67991845328531*v_R14_c_i - 7.72144347466701*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r
        struct[0].g[129,0] = i_R13_a_i + 26.5300693004806*v_R12_a_i - 4.84676915585801*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r - 7.72144347466701*v_R12_c_i - 0.67991845328531*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r - 53.0601386009611*v_R13_a_i + 9.69353831171601*v_R13_a_r + 15.7892267502954*v_R13_b_i + 3.16965629008966*v_R13_b_r + 15.442886949334*v_R13_c_i + 1.35983690657062*v_R13_c_r + 15.7892267502954*v_R13_n_i + 3.16965629008966*v_R13_n_r + 26.5300693004806*v_R14_a_i - 4.84676915585801*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r - 7.72144347466701*v_R14_c_i - 0.67991845328531*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r
        struct[0].g[130,0] = i_R13_b_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 4.84676915585801*v_R12_b_i + 26.5300693004806*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 0.679918453285308*v_R12_n_i - 7.72144347466701*v_R12_n_r - 3.16965629008966*v_R13_a_i + 15.7892267502954*v_R13_a_r - 9.69353831171602*v_R13_b_i - 53.0601386009611*v_R13_b_r - 3.16965629008967*v_R13_c_i + 15.7892267502954*v_R13_c_r - 1.35983690657062*v_R13_n_i + 15.442886949334*v_R13_n_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 4.84676915585801*v_R14_b_i + 26.5300693004806*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 0.679918453285308*v_R14_n_i - 7.72144347466701*v_R14_n_r
        struct[0].g[131,0] = i_R13_b_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r + 26.5300693004806*v_R12_b_i - 4.84676915585801*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r - 7.72144347466701*v_R12_n_i - 0.679918453285308*v_R12_n_r + 15.7892267502954*v_R13_a_i + 3.16965629008966*v_R13_a_r - 53.0601386009611*v_R13_b_i + 9.69353831171602*v_R13_b_r + 15.7892267502954*v_R13_c_i + 3.16965629008967*v_R13_c_r + 15.442886949334*v_R13_n_i + 1.35983690657062*v_R13_n_r - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r + 26.5300693004806*v_R14_b_i - 4.84676915585801*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r - 7.72144347466701*v_R14_n_i - 0.679918453285308*v_R14_n_r
        struct[0].g[132,0] = i_R13_c_r + 0.67991845328531*v_R12_a_i - 7.72144347466701*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 4.84676915585801*v_R12_c_i + 26.5300693004806*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r - 1.35983690657062*v_R13_a_i + 15.442886949334*v_R13_a_r - 3.16965629008966*v_R13_b_i + 15.7892267502954*v_R13_b_r - 9.69353831171602*v_R13_c_i - 53.0601386009611*v_R13_c_r - 3.16965629008967*v_R13_n_i + 15.7892267502954*v_R13_n_r + 0.67991845328531*v_R14_a_i - 7.72144347466701*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 4.84676915585801*v_R14_c_i + 26.5300693004806*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r
        struct[0].g[133,0] = i_R13_c_i - 7.72144347466701*v_R12_a_i - 0.67991845328531*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r + 26.5300693004806*v_R12_c_i - 4.84676915585801*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r + 15.442886949334*v_R13_a_i + 1.35983690657062*v_R13_a_r + 15.7892267502954*v_R13_b_i + 3.16965629008966*v_R13_b_r - 53.0601386009611*v_R13_c_i + 9.69353831171602*v_R13_c_r + 15.7892267502954*v_R13_n_i + 3.16965629008967*v_R13_n_r - 7.72144347466701*v_R14_a_i - 0.67991845328531*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r + 26.5300693004806*v_R14_c_i - 4.84676915585801*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r
        struct[0].g[134,0] = i_R13_n_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 0.679918453285308*v_R12_b_i - 7.72144347466701*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 4.84676915585801*v_R12_n_i + 26.5300693004806*v_R12_n_r - 3.16965629008966*v_R13_a_i + 15.7892267502954*v_R13_a_r - 1.35983690657062*v_R13_b_i + 15.442886949334*v_R13_b_r - 3.16965629008966*v_R13_c_i + 15.7892267502954*v_R13_c_r - 9.69353831171602*v_R13_n_i - 53.0851386009611*v_R13_n_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 0.679918453285308*v_R14_b_i - 7.72144347466701*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 4.84676915585801*v_R14_n_i + 26.5300693004806*v_R14_n_r
        struct[0].g[135,0] = i_R13_n_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r - 7.72144347466701*v_R12_b_i - 0.679918453285308*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r + 26.5300693004806*v_R12_n_i - 4.84676915585801*v_R12_n_r + 15.7892267502954*v_R13_a_i + 3.16965629008966*v_R13_a_r + 15.442886949334*v_R13_b_i + 1.35983690657062*v_R13_b_r + 15.7892267502954*v_R13_c_i + 3.16965629008966*v_R13_c_r - 53.0851386009611*v_R13_n_i + 9.69353831171602*v_R13_n_r - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r - 7.72144347466701*v_R14_b_i - 0.679918453285308*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r + 26.5300693004806*v_R14_n_i - 4.84676915585801*v_R14_n_r
        struct[0].g[136,0] = i_R14_a_r + 4.84676915585801*v_R13_a_i + 26.5300693004806*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 0.67991845328531*v_R13_c_i - 7.72144347466701*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r - 9.69353831171601*v_R14_a_i - 53.0601386009611*v_R14_a_r - 3.16965629008966*v_R14_b_i + 15.7892267502954*v_R14_b_r - 1.35983690657062*v_R14_c_i + 15.442886949334*v_R14_c_r - 3.16965629008966*v_R14_n_i + 15.7892267502954*v_R14_n_r + 4.84676915585801*v_R15_a_i + 26.5300693004806*v_R15_a_r + 1.58482814504483*v_R15_b_i - 7.8946133751477*v_R15_b_r + 0.67991845328531*v_R15_c_i - 7.72144347466701*v_R15_c_r + 1.58482814504483*v_R15_n_i - 7.89461337514771*v_R15_n_r
        struct[0].g[137,0] = i_R14_a_i + 26.5300693004806*v_R13_a_i - 4.84676915585801*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r - 7.72144347466701*v_R13_c_i - 0.67991845328531*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r - 53.0601386009611*v_R14_a_i + 9.69353831171601*v_R14_a_r + 15.7892267502954*v_R14_b_i + 3.16965629008966*v_R14_b_r + 15.442886949334*v_R14_c_i + 1.35983690657062*v_R14_c_r + 15.7892267502954*v_R14_n_i + 3.16965629008966*v_R14_n_r + 26.5300693004806*v_R15_a_i - 4.84676915585801*v_R15_a_r - 7.8946133751477*v_R15_b_i - 1.58482814504483*v_R15_b_r - 7.72144347466701*v_R15_c_i - 0.67991845328531*v_R15_c_r - 7.89461337514771*v_R15_n_i - 1.58482814504483*v_R15_n_r
        struct[0].g[138,0] = i_R14_b_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 4.84676915585801*v_R13_b_i + 26.5300693004806*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 0.679918453285308*v_R13_n_i - 7.72144347466701*v_R13_n_r - 3.16965629008966*v_R14_a_i + 15.7892267502954*v_R14_a_r - 9.69353831171602*v_R14_b_i - 53.0601386009611*v_R14_b_r - 3.16965629008967*v_R14_c_i + 15.7892267502954*v_R14_c_r - 1.35983690657062*v_R14_n_i + 15.442886949334*v_R14_n_r + 1.58482814504483*v_R15_a_i - 7.89461337514771*v_R15_a_r + 4.84676915585801*v_R15_b_i + 26.5300693004806*v_R15_b_r + 1.58482814504483*v_R15_c_i - 7.89461337514771*v_R15_c_r + 0.679918453285308*v_R15_n_i - 7.72144347466701*v_R15_n_r
        struct[0].g[139,0] = i_R14_b_i - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r + 26.5300693004806*v_R13_b_i - 4.84676915585801*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r - 7.72144347466701*v_R13_n_i - 0.679918453285308*v_R13_n_r + 15.7892267502954*v_R14_a_i + 3.16965629008966*v_R14_a_r - 53.0601386009611*v_R14_b_i + 9.69353831171602*v_R14_b_r + 15.7892267502954*v_R14_c_i + 3.16965629008967*v_R14_c_r + 15.442886949334*v_R14_n_i + 1.35983690657062*v_R14_n_r - 7.89461337514771*v_R15_a_i - 1.58482814504483*v_R15_a_r + 26.5300693004806*v_R15_b_i - 4.84676915585801*v_R15_b_r - 7.89461337514771*v_R15_c_i - 1.58482814504483*v_R15_c_r - 7.72144347466701*v_R15_n_i - 0.679918453285308*v_R15_n_r
        struct[0].g[140,0] = i_R14_c_r + 0.67991845328531*v_R13_a_i - 7.72144347466701*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 4.84676915585801*v_R13_c_i + 26.5300693004806*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r - 1.35983690657062*v_R14_a_i + 15.442886949334*v_R14_a_r - 3.16965629008966*v_R14_b_i + 15.7892267502954*v_R14_b_r - 9.69353831171602*v_R14_c_i - 53.0601386009611*v_R14_c_r - 3.16965629008967*v_R14_n_i + 15.7892267502954*v_R14_n_r + 0.67991845328531*v_R15_a_i - 7.72144347466701*v_R15_a_r + 1.58482814504483*v_R15_b_i - 7.8946133751477*v_R15_b_r + 4.84676915585801*v_R15_c_i + 26.5300693004806*v_R15_c_r + 1.58482814504483*v_R15_n_i - 7.89461337514771*v_R15_n_r
        struct[0].g[141,0] = i_R14_c_i - 7.72144347466701*v_R13_a_i - 0.67991845328531*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r + 26.5300693004806*v_R13_c_i - 4.84676915585801*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r + 15.442886949334*v_R14_a_i + 1.35983690657062*v_R14_a_r + 15.7892267502954*v_R14_b_i + 3.16965629008966*v_R14_b_r - 53.0601386009611*v_R14_c_i + 9.69353831171602*v_R14_c_r + 15.7892267502954*v_R14_n_i + 3.16965629008967*v_R14_n_r - 7.72144347466701*v_R15_a_i - 0.67991845328531*v_R15_a_r - 7.8946133751477*v_R15_b_i - 1.58482814504483*v_R15_b_r + 26.5300693004806*v_R15_c_i - 4.84676915585801*v_R15_c_r - 7.89461337514771*v_R15_n_i - 1.58482814504483*v_R15_n_r
        struct[0].g[142,0] = i_R14_n_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 0.679918453285308*v_R13_b_i - 7.72144347466701*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 4.84676915585801*v_R13_n_i + 26.5300693004806*v_R13_n_r - 3.16965629008966*v_R14_a_i + 15.7892267502954*v_R14_a_r - 1.35983690657062*v_R14_b_i + 15.442886949334*v_R14_b_r - 3.16965629008966*v_R14_c_i + 15.7892267502954*v_R14_c_r - 9.69353831171602*v_R14_n_i - 53.0601386009611*v_R14_n_r + 1.58482814504483*v_R15_a_i - 7.89461337514771*v_R15_a_r + 0.679918453285308*v_R15_b_i - 7.72144347466701*v_R15_b_r + 1.58482814504483*v_R15_c_i - 7.89461337514771*v_R15_c_r + 4.84676915585801*v_R15_n_i + 26.5300693004806*v_R15_n_r
        struct[0].g[143,0] = i_R14_n_i - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r - 7.72144347466701*v_R13_b_i - 0.679918453285308*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r + 26.5300693004806*v_R13_n_i - 4.84676915585801*v_R13_n_r + 15.7892267502954*v_R14_a_i + 3.16965629008966*v_R14_a_r + 15.442886949334*v_R14_b_i + 1.35983690657062*v_R14_b_r + 15.7892267502954*v_R14_c_i + 3.16965629008966*v_R14_c_r - 53.0601386009611*v_R14_n_i + 9.69353831171602*v_R14_n_r - 7.89461337514771*v_R15_a_i - 1.58482814504483*v_R15_a_r - 7.72144347466701*v_R15_b_i - 0.679918453285308*v_R15_b_r - 7.89461337514771*v_R15_c_i - 1.58482814504483*v_R15_c_r + 26.5300693004806*v_R15_n_i - 4.84676915585801*v_R15_n_r
        struct[0].g[144,0] = i_R1_a_i*v_R1_a_i - i_R1_a_i*v_R1_n_i + i_R1_a_r*v_R1_a_r - i_R1_a_r*v_R1_n_r - p_R1_a
        struct[0].g[145,0] = i_R1_b_i*v_R1_b_i - i_R1_b_i*v_R1_n_i + i_R1_b_r*v_R1_b_r - i_R1_b_r*v_R1_n_r - p_R1_b
        struct[0].g[146,0] = i_R1_c_i*v_R1_c_i - i_R1_c_i*v_R1_n_i + i_R1_c_r*v_R1_c_r - i_R1_c_r*v_R1_n_r - p_R1_c
        struct[0].g[147,0] = -i_R1_a_i*v_R1_a_r + i_R1_a_i*v_R1_n_r + i_R1_a_r*v_R1_a_i - i_R1_a_r*v_R1_n_i - q_R1_a
        struct[0].g[148,0] = -i_R1_b_i*v_R1_b_r + i_R1_b_i*v_R1_n_r + i_R1_b_r*v_R1_b_i - i_R1_b_r*v_R1_n_i - q_R1_b
        struct[0].g[149,0] = -i_R1_c_i*v_R1_c_r + i_R1_c_i*v_R1_n_r + i_R1_c_r*v_R1_c_i - i_R1_c_r*v_R1_n_i - q_R1_c
        struct[0].g[150,0] = g_n[150,0]
        struct[0].g[151,0] = g_n[151,0]
        struct[0].g[152,0] = i_R11_a_i*v_R11_a_i - i_R11_a_i*v_R11_n_i + i_R11_a_r*v_R11_a_r - i_R11_a_r*v_R11_n_r - p_R11_a
        struct[0].g[153,0] = i_R11_b_i*v_R11_b_i - i_R11_b_i*v_R11_n_i + i_R11_b_r*v_R11_b_r - i_R11_b_r*v_R11_n_r - p_R11_b
        struct[0].g[154,0] = i_R11_c_i*v_R11_c_i - i_R11_c_i*v_R11_n_i + i_R11_c_r*v_R11_c_r - i_R11_c_r*v_R11_n_r - p_R11_c
        struct[0].g[155,0] = -i_R11_a_i*v_R11_a_r + i_R11_a_i*v_R11_n_r + i_R11_a_r*v_R11_a_i - i_R11_a_r*v_R11_n_i - q_R11_a
        struct[0].g[156,0] = -i_R11_b_i*v_R11_b_r + i_R11_b_i*v_R11_n_r + i_R11_b_r*v_R11_b_i - i_R11_b_r*v_R11_n_i - q_R11_b
        struct[0].g[157,0] = -i_R11_c_i*v_R11_c_r + i_R11_c_i*v_R11_n_r + i_R11_c_r*v_R11_c_i - i_R11_c_r*v_R11_n_i - q_R11_c
        struct[0].g[158,0] = g_n[158,0]
        struct[0].g[159,0] = g_n[159,0]
        struct[0].g[160,0] = i_R15_a_i*v_R15_a_i - i_R15_a_i*v_R15_n_i + i_R15_a_r*v_R15_a_r - i_R15_a_r*v_R15_n_r - p_R15_a
        struct[0].g[161,0] = i_R15_b_i*v_R15_b_i - i_R15_b_i*v_R15_n_i + i_R15_b_r*v_R15_b_r - i_R15_b_r*v_R15_n_r - p_R15_b
        struct[0].g[162,0] = i_R15_c_i*v_R15_c_i - i_R15_c_i*v_R15_n_i + i_R15_c_r*v_R15_c_r - i_R15_c_r*v_R15_n_r - p_R15_c
        struct[0].g[163,0] = -i_R15_a_i*v_R15_a_r + i_R15_a_i*v_R15_n_r + i_R15_a_r*v_R15_a_i - i_R15_a_r*v_R15_n_i - q_R15_a
        struct[0].g[164,0] = -i_R15_b_i*v_R15_b_r + i_R15_b_i*v_R15_n_r + i_R15_b_r*v_R15_b_i - i_R15_b_r*v_R15_n_i - q_R15_b
        struct[0].g[165,0] = -i_R15_c_i*v_R15_c_r + i_R15_c_i*v_R15_n_r + i_R15_c_r*v_R15_c_i - i_R15_c_r*v_R15_n_i - q_R15_c
        struct[0].g[166,0] = g_n[166,0]
        struct[0].g[167,0] = g_n[167,0]
        struct[0].g[168,0] = i_R16_a_i*v_R16_a_i - i_R16_a_i*v_R16_n_i + i_R16_a_r*v_R16_a_r - i_R16_a_r*v_R16_n_r - p_R16_a
        struct[0].g[169,0] = i_R16_b_i*v_R16_b_i - i_R16_b_i*v_R16_n_i + i_R16_b_r*v_R16_b_r - i_R16_b_r*v_R16_n_r - p_R16_b
        struct[0].g[170,0] = i_R16_c_i*v_R16_c_i - i_R16_c_i*v_R16_n_i + i_R16_c_r*v_R16_c_r - i_R16_c_r*v_R16_n_r - p_R16_c
        struct[0].g[171,0] = -i_R16_a_i*v_R16_a_r + i_R16_a_i*v_R16_n_r + i_R16_a_r*v_R16_a_i - i_R16_a_r*v_R16_n_i - q_R16_a
        struct[0].g[172,0] = -i_R16_b_i*v_R16_b_r + i_R16_b_i*v_R16_n_r + i_R16_b_r*v_R16_b_i - i_R16_b_r*v_R16_n_i - q_R16_b
        struct[0].g[173,0] = -i_R16_c_i*v_R16_c_r + i_R16_c_i*v_R16_n_r + i_R16_c_r*v_R16_c_i - i_R16_c_r*v_R16_n_i - q_R16_c
        struct[0].g[174,0] = g_n[174,0]
        struct[0].g[175,0] = g_n[175,0]
        struct[0].g[176,0] = i_R17_a_i*v_R17_a_i - i_R17_a_i*v_R17_n_i + i_R17_a_r*v_R17_a_r - i_R17_a_r*v_R17_n_r - p_R17_a
        struct[0].g[177,0] = i_R17_b_i*v_R17_b_i - i_R17_b_i*v_R17_n_i + i_R17_b_r*v_R17_b_r - i_R17_b_r*v_R17_n_r - p_R17_b
        struct[0].g[178,0] = i_R17_c_i*v_R17_c_i - i_R17_c_i*v_R17_n_i + i_R17_c_r*v_R17_c_r - i_R17_c_r*v_R17_n_r - p_R17_c
        struct[0].g[179,0] = -i_R17_a_i*v_R17_a_r + i_R17_a_i*v_R17_n_r + i_R17_a_r*v_R17_a_i - i_R17_a_r*v_R17_n_i - q_R17_a
        struct[0].g[180,0] = -i_R17_b_i*v_R17_b_r + i_R17_b_i*v_R17_n_r + i_R17_b_r*v_R17_b_i - i_R17_b_r*v_R17_n_i - q_R17_b
        struct[0].g[181,0] = -i_R17_c_i*v_R17_c_r + i_R17_c_i*v_R17_n_r + i_R17_c_r*v_R17_c_i - i_R17_c_r*v_R17_n_i - q_R17_c
        struct[0].g[182,0] = g_n[182,0]
        struct[0].g[183,0] = g_n[183,0]
        struct[0].g[184,0] = i_R18_a_i*v_R18_a_i - i_R18_a_i*v_R18_n_i + i_R18_a_r*v_R18_a_r - i_R18_a_r*v_R18_n_r - p_R18_a
        struct[0].g[185,0] = i_R18_b_i*v_R18_b_i - i_R18_b_i*v_R18_n_i + i_R18_b_r*v_R18_b_r - i_R18_b_r*v_R18_n_r - p_R18_b
        struct[0].g[186,0] = i_R18_c_i*v_R18_c_i - i_R18_c_i*v_R18_n_i + i_R18_c_r*v_R18_c_r - i_R18_c_r*v_R18_n_r - p_R18_c
        struct[0].g[187,0] = -i_R18_a_i*v_R18_a_r + i_R18_a_i*v_R18_n_r + i_R18_a_r*v_R18_a_i - i_R18_a_r*v_R18_n_i - q_R18_a
        struct[0].g[188,0] = -i_R18_b_i*v_R18_b_r + i_R18_b_i*v_R18_n_r + i_R18_b_r*v_R18_b_i - i_R18_b_r*v_R18_n_i - q_R18_b
        struct[0].g[189,0] = -i_R18_c_i*v_R18_c_r + i_R18_c_i*v_R18_n_r + i_R18_c_r*v_R18_c_i - i_R18_c_r*v_R18_n_i - q_R18_c
        struct[0].g[190,0] = g_n[190,0]
        struct[0].g[191,0] = g_n[191,0]
    
    # Outputs:
    if mode == 3:

        struct[0].h[0,0] = (v_R0_a_i**2 + v_R0_a_r**2)**0.5
        struct[0].h[1,0] = (v_R0_b_i**2 + v_R0_b_r**2)**0.5
        struct[0].h[2,0] = (v_R0_c_i**2 + v_R0_c_r**2)**0.5
        struct[0].h[3,0] = (v_R1_a_i**2 + v_R1_a_r**2)**0.5
        struct[0].h[4,0] = (v_R1_b_i**2 + v_R1_b_r**2)**0.5
        struct[0].h[5,0] = (v_R1_c_i**2 + v_R1_c_r**2)**0.5
        struct[0].h[6,0] = (v_R1_n_i**2 + v_R1_n_r**2)**0.5
        struct[0].h[7,0] = (v_R11_a_i**2 + v_R11_a_r**2)**0.5
        struct[0].h[8,0] = (v_R11_b_i**2 + v_R11_b_r**2)**0.5
        struct[0].h[9,0] = (v_R11_c_i**2 + v_R11_c_r**2)**0.5
        struct[0].h[10,0] = (v_R11_n_i**2 + v_R11_n_r**2)**0.5
        struct[0].h[11,0] = (v_R15_a_i**2 + v_R15_a_r**2)**0.5
        struct[0].h[12,0] = (v_R15_b_i**2 + v_R15_b_r**2)**0.5
        struct[0].h[13,0] = (v_R15_c_i**2 + v_R15_c_r**2)**0.5
        struct[0].h[14,0] = (v_R15_n_i**2 + v_R15_n_r**2)**0.5
        struct[0].h[15,0] = (v_R16_a_i**2 + v_R16_a_r**2)**0.5
        struct[0].h[16,0] = (v_R16_b_i**2 + v_R16_b_r**2)**0.5
        struct[0].h[17,0] = (v_R16_c_i**2 + v_R16_c_r**2)**0.5
        struct[0].h[18,0] = (v_R16_n_i**2 + v_R16_n_r**2)**0.5
        struct[0].h[19,0] = (v_R17_a_i**2 + v_R17_a_r**2)**0.5
        struct[0].h[20,0] = (v_R17_b_i**2 + v_R17_b_r**2)**0.5
        struct[0].h[21,0] = (v_R17_c_i**2 + v_R17_c_r**2)**0.5
        struct[0].h[22,0] = (v_R17_n_i**2 + v_R17_n_r**2)**0.5
        struct[0].h[23,0] = (v_R18_a_i**2 + v_R18_a_r**2)**0.5
        struct[0].h[24,0] = (v_R18_b_i**2 + v_R18_b_r**2)**0.5
        struct[0].h[25,0] = (v_R18_c_i**2 + v_R18_c_r**2)**0.5
        struct[0].h[26,0] = (v_R18_n_i**2 + v_R18_n_r**2)**0.5
        struct[0].h[27,0] = (v_R2_a_i**2 + v_R2_a_r**2)**0.5
        struct[0].h[28,0] = (v_R2_b_i**2 + v_R2_b_r**2)**0.5
        struct[0].h[29,0] = (v_R2_c_i**2 + v_R2_c_r**2)**0.5
        struct[0].h[30,0] = (v_R2_n_i**2 + v_R2_n_r**2)**0.5
        struct[0].h[31,0] = (v_R3_a_i**2 + v_R3_a_r**2)**0.5
        struct[0].h[32,0] = (v_R3_b_i**2 + v_R3_b_r**2)**0.5
        struct[0].h[33,0] = (v_R3_c_i**2 + v_R3_c_r**2)**0.5
        struct[0].h[34,0] = (v_R3_n_i**2 + v_R3_n_r**2)**0.5
        struct[0].h[35,0] = (v_R4_a_i**2 + v_R4_a_r**2)**0.5
        struct[0].h[36,0] = (v_R4_b_i**2 + v_R4_b_r**2)**0.5
        struct[0].h[37,0] = (v_R4_c_i**2 + v_R4_c_r**2)**0.5
        struct[0].h[38,0] = (v_R4_n_i**2 + v_R4_n_r**2)**0.5
        struct[0].h[39,0] = (v_R5_a_i**2 + v_R5_a_r**2)**0.5
        struct[0].h[40,0] = (v_R5_b_i**2 + v_R5_b_r**2)**0.5
        struct[0].h[41,0] = (v_R5_c_i**2 + v_R5_c_r**2)**0.5
        struct[0].h[42,0] = (v_R5_n_i**2 + v_R5_n_r**2)**0.5
        struct[0].h[43,0] = (v_R6_a_i**2 + v_R6_a_r**2)**0.5
        struct[0].h[44,0] = (v_R6_b_i**2 + v_R6_b_r**2)**0.5
        struct[0].h[45,0] = (v_R6_c_i**2 + v_R6_c_r**2)**0.5
        struct[0].h[46,0] = (v_R6_n_i**2 + v_R6_n_r**2)**0.5
        struct[0].h[47,0] = (v_R7_a_i**2 + v_R7_a_r**2)**0.5
        struct[0].h[48,0] = (v_R7_b_i**2 + v_R7_b_r**2)**0.5
        struct[0].h[49,0] = (v_R7_c_i**2 + v_R7_c_r**2)**0.5
        struct[0].h[50,0] = (v_R7_n_i**2 + v_R7_n_r**2)**0.5
        struct[0].h[51,0] = (v_R8_a_i**2 + v_R8_a_r**2)**0.5
        struct[0].h[52,0] = (v_R8_b_i**2 + v_R8_b_r**2)**0.5
        struct[0].h[53,0] = (v_R8_c_i**2 + v_R8_c_r**2)**0.5
        struct[0].h[54,0] = (v_R8_n_i**2 + v_R8_n_r**2)**0.5
        struct[0].h[55,0] = (v_R9_a_i**2 + v_R9_a_r**2)**0.5
        struct[0].h[56,0] = (v_R9_b_i**2 + v_R9_b_r**2)**0.5
        struct[0].h[57,0] = (v_R9_c_i**2 + v_R9_c_r**2)**0.5
        struct[0].h[58,0] = (v_R9_n_i**2 + v_R9_n_r**2)**0.5
        struct[0].h[59,0] = (v_R10_a_i**2 + v_R10_a_r**2)**0.5
        struct[0].h[60,0] = (v_R10_b_i**2 + v_R10_b_r**2)**0.5
        struct[0].h[61,0] = (v_R10_c_i**2 + v_R10_c_r**2)**0.5
        struct[0].h[62,0] = (v_R10_n_i**2 + v_R10_n_r**2)**0.5
        struct[0].h[63,0] = (v_R12_a_i**2 + v_R12_a_r**2)**0.5
        struct[0].h[64,0] = (v_R12_b_i**2 + v_R12_b_r**2)**0.5
        struct[0].h[65,0] = (v_R12_c_i**2 + v_R12_c_r**2)**0.5
        struct[0].h[66,0] = (v_R12_n_i**2 + v_R12_n_r**2)**0.5
        struct[0].h[67,0] = (v_R13_a_i**2 + v_R13_a_r**2)**0.5
        struct[0].h[68,0] = (v_R13_b_i**2 + v_R13_b_r**2)**0.5
        struct[0].h[69,0] = (v_R13_c_i**2 + v_R13_c_r**2)**0.5
        struct[0].h[70,0] = (v_R13_n_i**2 + v_R13_n_r**2)**0.5
        struct[0].h[71,0] = (v_R14_a_i**2 + v_R14_a_r**2)**0.5
        struct[0].h[72,0] = (v_R14_b_i**2 + v_R14_b_r**2)**0.5
        struct[0].h[73,0] = (v_R14_c_i**2 + v_R14_c_r**2)**0.5
        struct[0].h[74,0] = (v_R14_n_i**2 + v_R14_n_r**2)**0.5
    

    if mode == 10:

        pass

    if mode == 11:



        struct[0].Gy_ini[144,0] = i_R1_a_r
        struct[0].Gy_ini[144,1] = i_R1_a_i
        struct[0].Gy_ini[144,6] = -i_R1_a_r
        struct[0].Gy_ini[144,7] = -i_R1_a_i
        struct[0].Gy_ini[144,144] = v_R1_a_r - v_R1_n_r
        struct[0].Gy_ini[144,145] = v_R1_a_i - v_R1_n_i
        struct[0].Gy_ini[145,2] = i_R1_b_r
        struct[0].Gy_ini[145,3] = i_R1_b_i
        struct[0].Gy_ini[145,6] = -i_R1_b_r
        struct[0].Gy_ini[145,7] = -i_R1_b_i
        struct[0].Gy_ini[145,146] = v_R1_b_r - v_R1_n_r
        struct[0].Gy_ini[145,147] = v_R1_b_i - v_R1_n_i
        struct[0].Gy_ini[146,4] = i_R1_c_r
        struct[0].Gy_ini[146,5] = i_R1_c_i
        struct[0].Gy_ini[146,6] = -i_R1_c_r
        struct[0].Gy_ini[146,7] = -i_R1_c_i
        struct[0].Gy_ini[146,148] = v_R1_c_r - v_R1_n_r
        struct[0].Gy_ini[146,149] = v_R1_c_i - v_R1_n_i
        struct[0].Gy_ini[147,0] = -i_R1_a_i
        struct[0].Gy_ini[147,1] = i_R1_a_r
        struct[0].Gy_ini[147,6] = i_R1_a_i
        struct[0].Gy_ini[147,7] = -i_R1_a_r
        struct[0].Gy_ini[147,144] = v_R1_a_i - v_R1_n_i
        struct[0].Gy_ini[147,145] = -v_R1_a_r + v_R1_n_r
        struct[0].Gy_ini[148,2] = -i_R1_b_i
        struct[0].Gy_ini[148,3] = i_R1_b_r
        struct[0].Gy_ini[148,6] = i_R1_b_i
        struct[0].Gy_ini[148,7] = -i_R1_b_r
        struct[0].Gy_ini[148,146] = v_R1_b_i - v_R1_n_i
        struct[0].Gy_ini[148,147] = -v_R1_b_r + v_R1_n_r
        struct[0].Gy_ini[149,4] = -i_R1_c_i
        struct[0].Gy_ini[149,5] = i_R1_c_r
        struct[0].Gy_ini[149,6] = i_R1_c_i
        struct[0].Gy_ini[149,7] = -i_R1_c_r
        struct[0].Gy_ini[149,148] = v_R1_c_i - v_R1_n_i
        struct[0].Gy_ini[149,149] = -v_R1_c_r + v_R1_n_r
        struct[0].Gy_ini[152,8] = i_R11_a_r
        struct[0].Gy_ini[152,9] = i_R11_a_i
        struct[0].Gy_ini[152,14] = -i_R11_a_r
        struct[0].Gy_ini[152,15] = -i_R11_a_i
        struct[0].Gy_ini[152,152] = v_R11_a_r - v_R11_n_r
        struct[0].Gy_ini[152,153] = v_R11_a_i - v_R11_n_i
        struct[0].Gy_ini[153,10] = i_R11_b_r
        struct[0].Gy_ini[153,11] = i_R11_b_i
        struct[0].Gy_ini[153,14] = -i_R11_b_r
        struct[0].Gy_ini[153,15] = -i_R11_b_i
        struct[0].Gy_ini[153,154] = v_R11_b_r - v_R11_n_r
        struct[0].Gy_ini[153,155] = v_R11_b_i - v_R11_n_i
        struct[0].Gy_ini[154,12] = i_R11_c_r
        struct[0].Gy_ini[154,13] = i_R11_c_i
        struct[0].Gy_ini[154,14] = -i_R11_c_r
        struct[0].Gy_ini[154,15] = -i_R11_c_i
        struct[0].Gy_ini[154,156] = v_R11_c_r - v_R11_n_r
        struct[0].Gy_ini[154,157] = v_R11_c_i - v_R11_n_i
        struct[0].Gy_ini[155,8] = -i_R11_a_i
        struct[0].Gy_ini[155,9] = i_R11_a_r
        struct[0].Gy_ini[155,14] = i_R11_a_i
        struct[0].Gy_ini[155,15] = -i_R11_a_r
        struct[0].Gy_ini[155,152] = v_R11_a_i - v_R11_n_i
        struct[0].Gy_ini[155,153] = -v_R11_a_r + v_R11_n_r
        struct[0].Gy_ini[156,10] = -i_R11_b_i
        struct[0].Gy_ini[156,11] = i_R11_b_r
        struct[0].Gy_ini[156,14] = i_R11_b_i
        struct[0].Gy_ini[156,15] = -i_R11_b_r
        struct[0].Gy_ini[156,154] = v_R11_b_i - v_R11_n_i
        struct[0].Gy_ini[156,155] = -v_R11_b_r + v_R11_n_r
        struct[0].Gy_ini[157,12] = -i_R11_c_i
        struct[0].Gy_ini[157,13] = i_R11_c_r
        struct[0].Gy_ini[157,14] = i_R11_c_i
        struct[0].Gy_ini[157,15] = -i_R11_c_r
        struct[0].Gy_ini[157,156] = v_R11_c_i - v_R11_n_i
        struct[0].Gy_ini[157,157] = -v_R11_c_r + v_R11_n_r
        struct[0].Gy_ini[160,16] = i_R15_a_r
        struct[0].Gy_ini[160,17] = i_R15_a_i
        struct[0].Gy_ini[160,22] = -i_R15_a_r
        struct[0].Gy_ini[160,23] = -i_R15_a_i
        struct[0].Gy_ini[160,160] = v_R15_a_r - v_R15_n_r
        struct[0].Gy_ini[160,161] = v_R15_a_i - v_R15_n_i
        struct[0].Gy_ini[161,18] = i_R15_b_r
        struct[0].Gy_ini[161,19] = i_R15_b_i
        struct[0].Gy_ini[161,22] = -i_R15_b_r
        struct[0].Gy_ini[161,23] = -i_R15_b_i
        struct[0].Gy_ini[161,162] = v_R15_b_r - v_R15_n_r
        struct[0].Gy_ini[161,163] = v_R15_b_i - v_R15_n_i
        struct[0].Gy_ini[162,20] = i_R15_c_r
        struct[0].Gy_ini[162,21] = i_R15_c_i
        struct[0].Gy_ini[162,22] = -i_R15_c_r
        struct[0].Gy_ini[162,23] = -i_R15_c_i
        struct[0].Gy_ini[162,164] = v_R15_c_r - v_R15_n_r
        struct[0].Gy_ini[162,165] = v_R15_c_i - v_R15_n_i
        struct[0].Gy_ini[163,16] = -i_R15_a_i
        struct[0].Gy_ini[163,17] = i_R15_a_r
        struct[0].Gy_ini[163,22] = i_R15_a_i
        struct[0].Gy_ini[163,23] = -i_R15_a_r
        struct[0].Gy_ini[163,160] = v_R15_a_i - v_R15_n_i
        struct[0].Gy_ini[163,161] = -v_R15_a_r + v_R15_n_r
        struct[0].Gy_ini[164,18] = -i_R15_b_i
        struct[0].Gy_ini[164,19] = i_R15_b_r
        struct[0].Gy_ini[164,22] = i_R15_b_i
        struct[0].Gy_ini[164,23] = -i_R15_b_r
        struct[0].Gy_ini[164,162] = v_R15_b_i - v_R15_n_i
        struct[0].Gy_ini[164,163] = -v_R15_b_r + v_R15_n_r
        struct[0].Gy_ini[165,20] = -i_R15_c_i
        struct[0].Gy_ini[165,21] = i_R15_c_r
        struct[0].Gy_ini[165,22] = i_R15_c_i
        struct[0].Gy_ini[165,23] = -i_R15_c_r
        struct[0].Gy_ini[165,164] = v_R15_c_i - v_R15_n_i
        struct[0].Gy_ini[165,165] = -v_R15_c_r + v_R15_n_r
        struct[0].Gy_ini[168,24] = i_R16_a_r
        struct[0].Gy_ini[168,25] = i_R16_a_i
        struct[0].Gy_ini[168,30] = -i_R16_a_r
        struct[0].Gy_ini[168,31] = -i_R16_a_i
        struct[0].Gy_ini[168,168] = v_R16_a_r - v_R16_n_r
        struct[0].Gy_ini[168,169] = v_R16_a_i - v_R16_n_i
        struct[0].Gy_ini[169,26] = i_R16_b_r
        struct[0].Gy_ini[169,27] = i_R16_b_i
        struct[0].Gy_ini[169,30] = -i_R16_b_r
        struct[0].Gy_ini[169,31] = -i_R16_b_i
        struct[0].Gy_ini[169,170] = v_R16_b_r - v_R16_n_r
        struct[0].Gy_ini[169,171] = v_R16_b_i - v_R16_n_i
        struct[0].Gy_ini[170,28] = i_R16_c_r
        struct[0].Gy_ini[170,29] = i_R16_c_i
        struct[0].Gy_ini[170,30] = -i_R16_c_r
        struct[0].Gy_ini[170,31] = -i_R16_c_i
        struct[0].Gy_ini[170,172] = v_R16_c_r - v_R16_n_r
        struct[0].Gy_ini[170,173] = v_R16_c_i - v_R16_n_i
        struct[0].Gy_ini[171,24] = -i_R16_a_i
        struct[0].Gy_ini[171,25] = i_R16_a_r
        struct[0].Gy_ini[171,30] = i_R16_a_i
        struct[0].Gy_ini[171,31] = -i_R16_a_r
        struct[0].Gy_ini[171,168] = v_R16_a_i - v_R16_n_i
        struct[0].Gy_ini[171,169] = -v_R16_a_r + v_R16_n_r
        struct[0].Gy_ini[172,26] = -i_R16_b_i
        struct[0].Gy_ini[172,27] = i_R16_b_r
        struct[0].Gy_ini[172,30] = i_R16_b_i
        struct[0].Gy_ini[172,31] = -i_R16_b_r
        struct[0].Gy_ini[172,170] = v_R16_b_i - v_R16_n_i
        struct[0].Gy_ini[172,171] = -v_R16_b_r + v_R16_n_r
        struct[0].Gy_ini[173,28] = -i_R16_c_i
        struct[0].Gy_ini[173,29] = i_R16_c_r
        struct[0].Gy_ini[173,30] = i_R16_c_i
        struct[0].Gy_ini[173,31] = -i_R16_c_r
        struct[0].Gy_ini[173,172] = v_R16_c_i - v_R16_n_i
        struct[0].Gy_ini[173,173] = -v_R16_c_r + v_R16_n_r
        struct[0].Gy_ini[176,32] = i_R17_a_r
        struct[0].Gy_ini[176,33] = i_R17_a_i
        struct[0].Gy_ini[176,38] = -i_R17_a_r
        struct[0].Gy_ini[176,39] = -i_R17_a_i
        struct[0].Gy_ini[176,176] = v_R17_a_r - v_R17_n_r
        struct[0].Gy_ini[176,177] = v_R17_a_i - v_R17_n_i
        struct[0].Gy_ini[177,34] = i_R17_b_r
        struct[0].Gy_ini[177,35] = i_R17_b_i
        struct[0].Gy_ini[177,38] = -i_R17_b_r
        struct[0].Gy_ini[177,39] = -i_R17_b_i
        struct[0].Gy_ini[177,178] = v_R17_b_r - v_R17_n_r
        struct[0].Gy_ini[177,179] = v_R17_b_i - v_R17_n_i
        struct[0].Gy_ini[178,36] = i_R17_c_r
        struct[0].Gy_ini[178,37] = i_R17_c_i
        struct[0].Gy_ini[178,38] = -i_R17_c_r
        struct[0].Gy_ini[178,39] = -i_R17_c_i
        struct[0].Gy_ini[178,180] = v_R17_c_r - v_R17_n_r
        struct[0].Gy_ini[178,181] = v_R17_c_i - v_R17_n_i
        struct[0].Gy_ini[179,32] = -i_R17_a_i
        struct[0].Gy_ini[179,33] = i_R17_a_r
        struct[0].Gy_ini[179,38] = i_R17_a_i
        struct[0].Gy_ini[179,39] = -i_R17_a_r
        struct[0].Gy_ini[179,176] = v_R17_a_i - v_R17_n_i
        struct[0].Gy_ini[179,177] = -v_R17_a_r + v_R17_n_r
        struct[0].Gy_ini[180,34] = -i_R17_b_i
        struct[0].Gy_ini[180,35] = i_R17_b_r
        struct[0].Gy_ini[180,38] = i_R17_b_i
        struct[0].Gy_ini[180,39] = -i_R17_b_r
        struct[0].Gy_ini[180,178] = v_R17_b_i - v_R17_n_i
        struct[0].Gy_ini[180,179] = -v_R17_b_r + v_R17_n_r
        struct[0].Gy_ini[181,36] = -i_R17_c_i
        struct[0].Gy_ini[181,37] = i_R17_c_r
        struct[0].Gy_ini[181,38] = i_R17_c_i
        struct[0].Gy_ini[181,39] = -i_R17_c_r
        struct[0].Gy_ini[181,180] = v_R17_c_i - v_R17_n_i
        struct[0].Gy_ini[181,181] = -v_R17_c_r + v_R17_n_r
        struct[0].Gy_ini[184,40] = i_R18_a_r
        struct[0].Gy_ini[184,41] = i_R18_a_i
        struct[0].Gy_ini[184,46] = -i_R18_a_r
        struct[0].Gy_ini[184,47] = -i_R18_a_i
        struct[0].Gy_ini[184,184] = v_R18_a_r - v_R18_n_r
        struct[0].Gy_ini[184,185] = v_R18_a_i - v_R18_n_i
        struct[0].Gy_ini[185,42] = i_R18_b_r
        struct[0].Gy_ini[185,43] = i_R18_b_i
        struct[0].Gy_ini[185,46] = -i_R18_b_r
        struct[0].Gy_ini[185,47] = -i_R18_b_i
        struct[0].Gy_ini[185,186] = v_R18_b_r - v_R18_n_r
        struct[0].Gy_ini[185,187] = v_R18_b_i - v_R18_n_i
        struct[0].Gy_ini[186,44] = i_R18_c_r
        struct[0].Gy_ini[186,45] = i_R18_c_i
        struct[0].Gy_ini[186,46] = -i_R18_c_r
        struct[0].Gy_ini[186,47] = -i_R18_c_i
        struct[0].Gy_ini[186,188] = v_R18_c_r - v_R18_n_r
        struct[0].Gy_ini[186,189] = v_R18_c_i - v_R18_n_i
        struct[0].Gy_ini[187,40] = -i_R18_a_i
        struct[0].Gy_ini[187,41] = i_R18_a_r
        struct[0].Gy_ini[187,46] = i_R18_a_i
        struct[0].Gy_ini[187,47] = -i_R18_a_r
        struct[0].Gy_ini[187,184] = v_R18_a_i - v_R18_n_i
        struct[0].Gy_ini[187,185] = -v_R18_a_r + v_R18_n_r
        struct[0].Gy_ini[188,42] = -i_R18_b_i
        struct[0].Gy_ini[188,43] = i_R18_b_r
        struct[0].Gy_ini[188,46] = i_R18_b_i
        struct[0].Gy_ini[188,47] = -i_R18_b_r
        struct[0].Gy_ini[188,186] = v_R18_b_i - v_R18_n_i
        struct[0].Gy_ini[188,187] = -v_R18_b_r + v_R18_n_r
        struct[0].Gy_ini[189,44] = -i_R18_c_i
        struct[0].Gy_ini[189,45] = i_R18_c_r
        struct[0].Gy_ini[189,46] = i_R18_c_i
        struct[0].Gy_ini[189,47] = -i_R18_c_r
        struct[0].Gy_ini[189,188] = v_R18_c_i - v_R18_n_i
        struct[0].Gy_ini[189,189] = -v_R18_c_r + v_R18_n_r



@numba.njit(cache=True)
def run(t,struct,mode):

    # Parameters:
    
    # Inputs:
    v_R0_a_r = struct[0].v_R0_a_r
    v_R0_a_i = struct[0].v_R0_a_i
    v_R0_b_r = struct[0].v_R0_b_r
    v_R0_b_i = struct[0].v_R0_b_i
    v_R0_c_r = struct[0].v_R0_c_r
    v_R0_c_i = struct[0].v_R0_c_i
    i_R1_n_r = struct[0].i_R1_n_r
    i_R1_n_i = struct[0].i_R1_n_i
    i_R11_n_r = struct[0].i_R11_n_r
    i_R11_n_i = struct[0].i_R11_n_i
    i_R15_n_r = struct[0].i_R15_n_r
    i_R15_n_i = struct[0].i_R15_n_i
    i_R16_n_r = struct[0].i_R16_n_r
    i_R16_n_i = struct[0].i_R16_n_i
    i_R17_n_r = struct[0].i_R17_n_r
    i_R17_n_i = struct[0].i_R17_n_i
    i_R18_n_r = struct[0].i_R18_n_r
    i_R18_n_i = struct[0].i_R18_n_i
    i_R2_a_r = struct[0].i_R2_a_r
    i_R2_a_i = struct[0].i_R2_a_i
    i_R2_b_r = struct[0].i_R2_b_r
    i_R2_b_i = struct[0].i_R2_b_i
    i_R2_c_r = struct[0].i_R2_c_r
    i_R2_c_i = struct[0].i_R2_c_i
    i_R2_n_r = struct[0].i_R2_n_r
    i_R2_n_i = struct[0].i_R2_n_i
    i_R3_a_r = struct[0].i_R3_a_r
    i_R3_a_i = struct[0].i_R3_a_i
    i_R3_b_r = struct[0].i_R3_b_r
    i_R3_b_i = struct[0].i_R3_b_i
    i_R3_c_r = struct[0].i_R3_c_r
    i_R3_c_i = struct[0].i_R3_c_i
    i_R3_n_r = struct[0].i_R3_n_r
    i_R3_n_i = struct[0].i_R3_n_i
    i_R4_a_r = struct[0].i_R4_a_r
    i_R4_a_i = struct[0].i_R4_a_i
    i_R4_b_r = struct[0].i_R4_b_r
    i_R4_b_i = struct[0].i_R4_b_i
    i_R4_c_r = struct[0].i_R4_c_r
    i_R4_c_i = struct[0].i_R4_c_i
    i_R4_n_r = struct[0].i_R4_n_r
    i_R4_n_i = struct[0].i_R4_n_i
    i_R5_a_r = struct[0].i_R5_a_r
    i_R5_a_i = struct[0].i_R5_a_i
    i_R5_b_r = struct[0].i_R5_b_r
    i_R5_b_i = struct[0].i_R5_b_i
    i_R5_c_r = struct[0].i_R5_c_r
    i_R5_c_i = struct[0].i_R5_c_i
    i_R5_n_r = struct[0].i_R5_n_r
    i_R5_n_i = struct[0].i_R5_n_i
    i_R6_a_r = struct[0].i_R6_a_r
    i_R6_a_i = struct[0].i_R6_a_i
    i_R6_b_r = struct[0].i_R6_b_r
    i_R6_b_i = struct[0].i_R6_b_i
    i_R6_c_r = struct[0].i_R6_c_r
    i_R6_c_i = struct[0].i_R6_c_i
    i_R6_n_r = struct[0].i_R6_n_r
    i_R6_n_i = struct[0].i_R6_n_i
    i_R7_a_r = struct[0].i_R7_a_r
    i_R7_a_i = struct[0].i_R7_a_i
    i_R7_b_r = struct[0].i_R7_b_r
    i_R7_b_i = struct[0].i_R7_b_i
    i_R7_c_r = struct[0].i_R7_c_r
    i_R7_c_i = struct[0].i_R7_c_i
    i_R7_n_r = struct[0].i_R7_n_r
    i_R7_n_i = struct[0].i_R7_n_i
    i_R8_a_r = struct[0].i_R8_a_r
    i_R8_a_i = struct[0].i_R8_a_i
    i_R8_b_r = struct[0].i_R8_b_r
    i_R8_b_i = struct[0].i_R8_b_i
    i_R8_c_r = struct[0].i_R8_c_r
    i_R8_c_i = struct[0].i_R8_c_i
    i_R8_n_r = struct[0].i_R8_n_r
    i_R8_n_i = struct[0].i_R8_n_i
    i_R9_a_r = struct[0].i_R9_a_r
    i_R9_a_i = struct[0].i_R9_a_i
    i_R9_b_r = struct[0].i_R9_b_r
    i_R9_b_i = struct[0].i_R9_b_i
    i_R9_c_r = struct[0].i_R9_c_r
    i_R9_c_i = struct[0].i_R9_c_i
    i_R9_n_r = struct[0].i_R9_n_r
    i_R9_n_i = struct[0].i_R9_n_i
    i_R10_a_r = struct[0].i_R10_a_r
    i_R10_a_i = struct[0].i_R10_a_i
    i_R10_b_r = struct[0].i_R10_b_r
    i_R10_b_i = struct[0].i_R10_b_i
    i_R10_c_r = struct[0].i_R10_c_r
    i_R10_c_i = struct[0].i_R10_c_i
    i_R10_n_r = struct[0].i_R10_n_r
    i_R10_n_i = struct[0].i_R10_n_i
    i_R12_a_r = struct[0].i_R12_a_r
    i_R12_a_i = struct[0].i_R12_a_i
    i_R12_b_r = struct[0].i_R12_b_r
    i_R12_b_i = struct[0].i_R12_b_i
    i_R12_c_r = struct[0].i_R12_c_r
    i_R12_c_i = struct[0].i_R12_c_i
    i_R12_n_r = struct[0].i_R12_n_r
    i_R12_n_i = struct[0].i_R12_n_i
    i_R13_a_r = struct[0].i_R13_a_r
    i_R13_a_i = struct[0].i_R13_a_i
    i_R13_b_r = struct[0].i_R13_b_r
    i_R13_b_i = struct[0].i_R13_b_i
    i_R13_c_r = struct[0].i_R13_c_r
    i_R13_c_i = struct[0].i_R13_c_i
    i_R13_n_r = struct[0].i_R13_n_r
    i_R13_n_i = struct[0].i_R13_n_i
    i_R14_a_r = struct[0].i_R14_a_r
    i_R14_a_i = struct[0].i_R14_a_i
    i_R14_b_r = struct[0].i_R14_b_r
    i_R14_b_i = struct[0].i_R14_b_i
    i_R14_c_r = struct[0].i_R14_c_r
    i_R14_c_i = struct[0].i_R14_c_i
    i_R14_n_r = struct[0].i_R14_n_r
    i_R14_n_i = struct[0].i_R14_n_i
    p_R1_a = struct[0].p_R1_a
    q_R1_a = struct[0].q_R1_a
    p_R1_b = struct[0].p_R1_b
    q_R1_b = struct[0].q_R1_b
    p_R1_c = struct[0].p_R1_c
    q_R1_c = struct[0].q_R1_c
    p_R11_a = struct[0].p_R11_a
    q_R11_a = struct[0].q_R11_a
    p_R11_b = struct[0].p_R11_b
    q_R11_b = struct[0].q_R11_b
    p_R11_c = struct[0].p_R11_c
    q_R11_c = struct[0].q_R11_c
    p_R15_a = struct[0].p_R15_a
    q_R15_a = struct[0].q_R15_a
    p_R15_b = struct[0].p_R15_b
    q_R15_b = struct[0].q_R15_b
    p_R15_c = struct[0].p_R15_c
    q_R15_c = struct[0].q_R15_c
    p_R16_a = struct[0].p_R16_a
    q_R16_a = struct[0].q_R16_a
    p_R16_b = struct[0].p_R16_b
    q_R16_b = struct[0].q_R16_b
    p_R16_c = struct[0].p_R16_c
    q_R16_c = struct[0].q_R16_c
    p_R17_a = struct[0].p_R17_a
    q_R17_a = struct[0].q_R17_a
    p_R17_b = struct[0].p_R17_b
    q_R17_b = struct[0].q_R17_b
    p_R17_c = struct[0].p_R17_c
    q_R17_c = struct[0].q_R17_c
    p_R18_a = struct[0].p_R18_a
    q_R18_a = struct[0].q_R18_a
    p_R18_b = struct[0].p_R18_b
    q_R18_b = struct[0].q_R18_b
    p_R18_c = struct[0].p_R18_c
    q_R18_c = struct[0].q_R18_c
    u_dummy = struct[0].u_dummy
    
    # Dynamical states:
    x_dummy = struct[0].x[0,0]
    
    # Algebraic states:
    v_R1_a_r = struct[0].y_run[0,0]
    v_R1_a_i = struct[0].y_run[1,0]
    v_R1_b_r = struct[0].y_run[2,0]
    v_R1_b_i = struct[0].y_run[3,0]
    v_R1_c_r = struct[0].y_run[4,0]
    v_R1_c_i = struct[0].y_run[5,0]
    v_R1_n_r = struct[0].y_run[6,0]
    v_R1_n_i = struct[0].y_run[7,0]
    v_R11_a_r = struct[0].y_run[8,0]
    v_R11_a_i = struct[0].y_run[9,0]
    v_R11_b_r = struct[0].y_run[10,0]
    v_R11_b_i = struct[0].y_run[11,0]
    v_R11_c_r = struct[0].y_run[12,0]
    v_R11_c_i = struct[0].y_run[13,0]
    v_R11_n_r = struct[0].y_run[14,0]
    v_R11_n_i = struct[0].y_run[15,0]
    v_R15_a_r = struct[0].y_run[16,0]
    v_R15_a_i = struct[0].y_run[17,0]
    v_R15_b_r = struct[0].y_run[18,0]
    v_R15_b_i = struct[0].y_run[19,0]
    v_R15_c_r = struct[0].y_run[20,0]
    v_R15_c_i = struct[0].y_run[21,0]
    v_R15_n_r = struct[0].y_run[22,0]
    v_R15_n_i = struct[0].y_run[23,0]
    v_R16_a_r = struct[0].y_run[24,0]
    v_R16_a_i = struct[0].y_run[25,0]
    v_R16_b_r = struct[0].y_run[26,0]
    v_R16_b_i = struct[0].y_run[27,0]
    v_R16_c_r = struct[0].y_run[28,0]
    v_R16_c_i = struct[0].y_run[29,0]
    v_R16_n_r = struct[0].y_run[30,0]
    v_R16_n_i = struct[0].y_run[31,0]
    v_R17_a_r = struct[0].y_run[32,0]
    v_R17_a_i = struct[0].y_run[33,0]
    v_R17_b_r = struct[0].y_run[34,0]
    v_R17_b_i = struct[0].y_run[35,0]
    v_R17_c_r = struct[0].y_run[36,0]
    v_R17_c_i = struct[0].y_run[37,0]
    v_R17_n_r = struct[0].y_run[38,0]
    v_R17_n_i = struct[0].y_run[39,0]
    v_R18_a_r = struct[0].y_run[40,0]
    v_R18_a_i = struct[0].y_run[41,0]
    v_R18_b_r = struct[0].y_run[42,0]
    v_R18_b_i = struct[0].y_run[43,0]
    v_R18_c_r = struct[0].y_run[44,0]
    v_R18_c_i = struct[0].y_run[45,0]
    v_R18_n_r = struct[0].y_run[46,0]
    v_R18_n_i = struct[0].y_run[47,0]
    v_R2_a_r = struct[0].y_run[48,0]
    v_R2_a_i = struct[0].y_run[49,0]
    v_R2_b_r = struct[0].y_run[50,0]
    v_R2_b_i = struct[0].y_run[51,0]
    v_R2_c_r = struct[0].y_run[52,0]
    v_R2_c_i = struct[0].y_run[53,0]
    v_R2_n_r = struct[0].y_run[54,0]
    v_R2_n_i = struct[0].y_run[55,0]
    v_R3_a_r = struct[0].y_run[56,0]
    v_R3_a_i = struct[0].y_run[57,0]
    v_R3_b_r = struct[0].y_run[58,0]
    v_R3_b_i = struct[0].y_run[59,0]
    v_R3_c_r = struct[0].y_run[60,0]
    v_R3_c_i = struct[0].y_run[61,0]
    v_R3_n_r = struct[0].y_run[62,0]
    v_R3_n_i = struct[0].y_run[63,0]
    v_R4_a_r = struct[0].y_run[64,0]
    v_R4_a_i = struct[0].y_run[65,0]
    v_R4_b_r = struct[0].y_run[66,0]
    v_R4_b_i = struct[0].y_run[67,0]
    v_R4_c_r = struct[0].y_run[68,0]
    v_R4_c_i = struct[0].y_run[69,0]
    v_R4_n_r = struct[0].y_run[70,0]
    v_R4_n_i = struct[0].y_run[71,0]
    v_R5_a_r = struct[0].y_run[72,0]
    v_R5_a_i = struct[0].y_run[73,0]
    v_R5_b_r = struct[0].y_run[74,0]
    v_R5_b_i = struct[0].y_run[75,0]
    v_R5_c_r = struct[0].y_run[76,0]
    v_R5_c_i = struct[0].y_run[77,0]
    v_R5_n_r = struct[0].y_run[78,0]
    v_R5_n_i = struct[0].y_run[79,0]
    v_R6_a_r = struct[0].y_run[80,0]
    v_R6_a_i = struct[0].y_run[81,0]
    v_R6_b_r = struct[0].y_run[82,0]
    v_R6_b_i = struct[0].y_run[83,0]
    v_R6_c_r = struct[0].y_run[84,0]
    v_R6_c_i = struct[0].y_run[85,0]
    v_R6_n_r = struct[0].y_run[86,0]
    v_R6_n_i = struct[0].y_run[87,0]
    v_R7_a_r = struct[0].y_run[88,0]
    v_R7_a_i = struct[0].y_run[89,0]
    v_R7_b_r = struct[0].y_run[90,0]
    v_R7_b_i = struct[0].y_run[91,0]
    v_R7_c_r = struct[0].y_run[92,0]
    v_R7_c_i = struct[0].y_run[93,0]
    v_R7_n_r = struct[0].y_run[94,0]
    v_R7_n_i = struct[0].y_run[95,0]
    v_R8_a_r = struct[0].y_run[96,0]
    v_R8_a_i = struct[0].y_run[97,0]
    v_R8_b_r = struct[0].y_run[98,0]
    v_R8_b_i = struct[0].y_run[99,0]
    v_R8_c_r = struct[0].y_run[100,0]
    v_R8_c_i = struct[0].y_run[101,0]
    v_R8_n_r = struct[0].y_run[102,0]
    v_R8_n_i = struct[0].y_run[103,0]
    v_R9_a_r = struct[0].y_run[104,0]
    v_R9_a_i = struct[0].y_run[105,0]
    v_R9_b_r = struct[0].y_run[106,0]
    v_R9_b_i = struct[0].y_run[107,0]
    v_R9_c_r = struct[0].y_run[108,0]
    v_R9_c_i = struct[0].y_run[109,0]
    v_R9_n_r = struct[0].y_run[110,0]
    v_R9_n_i = struct[0].y_run[111,0]
    v_R10_a_r = struct[0].y_run[112,0]
    v_R10_a_i = struct[0].y_run[113,0]
    v_R10_b_r = struct[0].y_run[114,0]
    v_R10_b_i = struct[0].y_run[115,0]
    v_R10_c_r = struct[0].y_run[116,0]
    v_R10_c_i = struct[0].y_run[117,0]
    v_R10_n_r = struct[0].y_run[118,0]
    v_R10_n_i = struct[0].y_run[119,0]
    v_R12_a_r = struct[0].y_run[120,0]
    v_R12_a_i = struct[0].y_run[121,0]
    v_R12_b_r = struct[0].y_run[122,0]
    v_R12_b_i = struct[0].y_run[123,0]
    v_R12_c_r = struct[0].y_run[124,0]
    v_R12_c_i = struct[0].y_run[125,0]
    v_R12_n_r = struct[0].y_run[126,0]
    v_R12_n_i = struct[0].y_run[127,0]
    v_R13_a_r = struct[0].y_run[128,0]
    v_R13_a_i = struct[0].y_run[129,0]
    v_R13_b_r = struct[0].y_run[130,0]
    v_R13_b_i = struct[0].y_run[131,0]
    v_R13_c_r = struct[0].y_run[132,0]
    v_R13_c_i = struct[0].y_run[133,0]
    v_R13_n_r = struct[0].y_run[134,0]
    v_R13_n_i = struct[0].y_run[135,0]
    v_R14_a_r = struct[0].y_run[136,0]
    v_R14_a_i = struct[0].y_run[137,0]
    v_R14_b_r = struct[0].y_run[138,0]
    v_R14_b_i = struct[0].y_run[139,0]
    v_R14_c_r = struct[0].y_run[140,0]
    v_R14_c_i = struct[0].y_run[141,0]
    v_R14_n_r = struct[0].y_run[142,0]
    v_R14_n_i = struct[0].y_run[143,0]
    i_R1_a_r = struct[0].y_run[144,0]
    i_R1_a_i = struct[0].y_run[145,0]
    i_R1_b_r = struct[0].y_run[146,0]
    i_R1_b_i = struct[0].y_run[147,0]
    i_R1_c_r = struct[0].y_run[148,0]
    i_R1_c_i = struct[0].y_run[149,0]
    i_R1_n_r = struct[0].y_run[150,0]
    i_R1_n_i = struct[0].y_run[151,0]
    i_R11_a_r = struct[0].y_run[152,0]
    i_R11_a_i = struct[0].y_run[153,0]
    i_R11_b_r = struct[0].y_run[154,0]
    i_R11_b_i = struct[0].y_run[155,0]
    i_R11_c_r = struct[0].y_run[156,0]
    i_R11_c_i = struct[0].y_run[157,0]
    i_R11_n_r = struct[0].y_run[158,0]
    i_R11_n_i = struct[0].y_run[159,0]
    i_R15_a_r = struct[0].y_run[160,0]
    i_R15_a_i = struct[0].y_run[161,0]
    i_R15_b_r = struct[0].y_run[162,0]
    i_R15_b_i = struct[0].y_run[163,0]
    i_R15_c_r = struct[0].y_run[164,0]
    i_R15_c_i = struct[0].y_run[165,0]
    i_R15_n_r = struct[0].y_run[166,0]
    i_R15_n_i = struct[0].y_run[167,0]
    i_R16_a_r = struct[0].y_run[168,0]
    i_R16_a_i = struct[0].y_run[169,0]
    i_R16_b_r = struct[0].y_run[170,0]
    i_R16_b_i = struct[0].y_run[171,0]
    i_R16_c_r = struct[0].y_run[172,0]
    i_R16_c_i = struct[0].y_run[173,0]
    i_R16_n_r = struct[0].y_run[174,0]
    i_R16_n_i = struct[0].y_run[175,0]
    i_R17_a_r = struct[0].y_run[176,0]
    i_R17_a_i = struct[0].y_run[177,0]
    i_R17_b_r = struct[0].y_run[178,0]
    i_R17_b_i = struct[0].y_run[179,0]
    i_R17_c_r = struct[0].y_run[180,0]
    i_R17_c_i = struct[0].y_run[181,0]
    i_R17_n_r = struct[0].y_run[182,0]
    i_R17_n_i = struct[0].y_run[183,0]
    i_R18_a_r = struct[0].y_run[184,0]
    i_R18_a_i = struct[0].y_run[185,0]
    i_R18_b_r = struct[0].y_run[186,0]
    i_R18_b_i = struct[0].y_run[187,0]
    i_R18_c_r = struct[0].y_run[188,0]
    i_R18_c_i = struct[0].y_run[189,0]
    i_R18_n_r = struct[0].y_run[190,0]
    i_R18_n_i = struct[0].y_run[191,0]
    
    struct[0].u_run[0,0] = v_R0_a_r
    struct[0].u_run[1,0] = v_R0_a_i
    struct[0].u_run[2,0] = v_R0_b_r
    struct[0].u_run[3,0] = v_R0_b_i
    struct[0].u_run[4,0] = v_R0_c_r
    struct[0].u_run[5,0] = v_R0_c_i
    struct[0].u_run[6,0] = i_R1_n_r
    struct[0].u_run[7,0] = i_R1_n_i
    struct[0].u_run[8,0] = i_R11_n_r
    struct[0].u_run[9,0] = i_R11_n_i
    struct[0].u_run[10,0] = i_R15_n_r
    struct[0].u_run[11,0] = i_R15_n_i
    struct[0].u_run[12,0] = i_R16_n_r
    struct[0].u_run[13,0] = i_R16_n_i
    struct[0].u_run[14,0] = i_R17_n_r
    struct[0].u_run[15,0] = i_R17_n_i
    struct[0].u_run[16,0] = i_R18_n_r
    struct[0].u_run[17,0] = i_R18_n_i
    struct[0].u_run[18,0] = i_R2_a_r
    struct[0].u_run[19,0] = i_R2_a_i
    struct[0].u_run[20,0] = i_R2_b_r
    struct[0].u_run[21,0] = i_R2_b_i
    struct[0].u_run[22,0] = i_R2_c_r
    struct[0].u_run[23,0] = i_R2_c_i
    struct[0].u_run[24,0] = i_R2_n_r
    struct[0].u_run[25,0] = i_R2_n_i
    struct[0].u_run[26,0] = i_R3_a_r
    struct[0].u_run[27,0] = i_R3_a_i
    struct[0].u_run[28,0] = i_R3_b_r
    struct[0].u_run[29,0] = i_R3_b_i
    struct[0].u_run[30,0] = i_R3_c_r
    struct[0].u_run[31,0] = i_R3_c_i
    struct[0].u_run[32,0] = i_R3_n_r
    struct[0].u_run[33,0] = i_R3_n_i
    struct[0].u_run[34,0] = i_R4_a_r
    struct[0].u_run[35,0] = i_R4_a_i
    struct[0].u_run[36,0] = i_R4_b_r
    struct[0].u_run[37,0] = i_R4_b_i
    struct[0].u_run[38,0] = i_R4_c_r
    struct[0].u_run[39,0] = i_R4_c_i
    struct[0].u_run[40,0] = i_R4_n_r
    struct[0].u_run[41,0] = i_R4_n_i
    struct[0].u_run[42,0] = i_R5_a_r
    struct[0].u_run[43,0] = i_R5_a_i
    struct[0].u_run[44,0] = i_R5_b_r
    struct[0].u_run[45,0] = i_R5_b_i
    struct[0].u_run[46,0] = i_R5_c_r
    struct[0].u_run[47,0] = i_R5_c_i
    struct[0].u_run[48,0] = i_R5_n_r
    struct[0].u_run[49,0] = i_R5_n_i
    struct[0].u_run[50,0] = i_R6_a_r
    struct[0].u_run[51,0] = i_R6_a_i
    struct[0].u_run[52,0] = i_R6_b_r
    struct[0].u_run[53,0] = i_R6_b_i
    struct[0].u_run[54,0] = i_R6_c_r
    struct[0].u_run[55,0] = i_R6_c_i
    struct[0].u_run[56,0] = i_R6_n_r
    struct[0].u_run[57,0] = i_R6_n_i
    struct[0].u_run[58,0] = i_R7_a_r
    struct[0].u_run[59,0] = i_R7_a_i
    struct[0].u_run[60,0] = i_R7_b_r
    struct[0].u_run[61,0] = i_R7_b_i
    struct[0].u_run[62,0] = i_R7_c_r
    struct[0].u_run[63,0] = i_R7_c_i
    struct[0].u_run[64,0] = i_R7_n_r
    struct[0].u_run[65,0] = i_R7_n_i
    struct[0].u_run[66,0] = i_R8_a_r
    struct[0].u_run[67,0] = i_R8_a_i
    struct[0].u_run[68,0] = i_R8_b_r
    struct[0].u_run[69,0] = i_R8_b_i
    struct[0].u_run[70,0] = i_R8_c_r
    struct[0].u_run[71,0] = i_R8_c_i
    struct[0].u_run[72,0] = i_R8_n_r
    struct[0].u_run[73,0] = i_R8_n_i
    struct[0].u_run[74,0] = i_R9_a_r
    struct[0].u_run[75,0] = i_R9_a_i
    struct[0].u_run[76,0] = i_R9_b_r
    struct[0].u_run[77,0] = i_R9_b_i
    struct[0].u_run[78,0] = i_R9_c_r
    struct[0].u_run[79,0] = i_R9_c_i
    struct[0].u_run[80,0] = i_R9_n_r
    struct[0].u_run[81,0] = i_R9_n_i
    struct[0].u_run[82,0] = i_R10_a_r
    struct[0].u_run[83,0] = i_R10_a_i
    struct[0].u_run[84,0] = i_R10_b_r
    struct[0].u_run[85,0] = i_R10_b_i
    struct[0].u_run[86,0] = i_R10_c_r
    struct[0].u_run[87,0] = i_R10_c_i
    struct[0].u_run[88,0] = i_R10_n_r
    struct[0].u_run[89,0] = i_R10_n_i
    struct[0].u_run[90,0] = i_R12_a_r
    struct[0].u_run[91,0] = i_R12_a_i
    struct[0].u_run[92,0] = i_R12_b_r
    struct[0].u_run[93,0] = i_R12_b_i
    struct[0].u_run[94,0] = i_R12_c_r
    struct[0].u_run[95,0] = i_R12_c_i
    struct[0].u_run[96,0] = i_R12_n_r
    struct[0].u_run[97,0] = i_R12_n_i
    struct[0].u_run[98,0] = i_R13_a_r
    struct[0].u_run[99,0] = i_R13_a_i
    struct[0].u_run[100,0] = i_R13_b_r
    struct[0].u_run[101,0] = i_R13_b_i
    struct[0].u_run[102,0] = i_R13_c_r
    struct[0].u_run[103,0] = i_R13_c_i
    struct[0].u_run[104,0] = i_R13_n_r
    struct[0].u_run[105,0] = i_R13_n_i
    struct[0].u_run[106,0] = i_R14_a_r
    struct[0].u_run[107,0] = i_R14_a_i
    struct[0].u_run[108,0] = i_R14_b_r
    struct[0].u_run[109,0] = i_R14_b_i
    struct[0].u_run[110,0] = i_R14_c_r
    struct[0].u_run[111,0] = i_R14_c_i
    struct[0].u_run[112,0] = i_R14_n_r
    struct[0].u_run[113,0] = i_R14_n_i
    struct[0].u_run[114,0] = p_R1_a
    struct[0].u_run[115,0] = q_R1_a
    struct[0].u_run[116,0] = p_R1_b
    struct[0].u_run[117,0] = q_R1_b
    struct[0].u_run[118,0] = p_R1_c
    struct[0].u_run[119,0] = q_R1_c
    struct[0].u_run[120,0] = p_R11_a
    struct[0].u_run[121,0] = q_R11_a
    struct[0].u_run[122,0] = p_R11_b
    struct[0].u_run[123,0] = q_R11_b
    struct[0].u_run[124,0] = p_R11_c
    struct[0].u_run[125,0] = q_R11_c
    struct[0].u_run[126,0] = p_R15_a
    struct[0].u_run[127,0] = q_R15_a
    struct[0].u_run[128,0] = p_R15_b
    struct[0].u_run[129,0] = q_R15_b
    struct[0].u_run[130,0] = p_R15_c
    struct[0].u_run[131,0] = q_R15_c
    struct[0].u_run[132,0] = p_R16_a
    struct[0].u_run[133,0] = q_R16_a
    struct[0].u_run[134,0] = p_R16_b
    struct[0].u_run[135,0] = q_R16_b
    struct[0].u_run[136,0] = p_R16_c
    struct[0].u_run[137,0] = q_R16_c
    struct[0].u_run[138,0] = p_R17_a
    struct[0].u_run[139,0] = q_R17_a
    struct[0].u_run[140,0] = p_R17_b
    struct[0].u_run[141,0] = q_R17_b
    struct[0].u_run[142,0] = p_R17_c
    struct[0].u_run[143,0] = q_R17_c
    struct[0].u_run[144,0] = p_R18_a
    struct[0].u_run[145,0] = q_R18_a
    struct[0].u_run[146,0] = p_R18_b
    struct[0].u_run[147,0] = q_R18_b
    struct[0].u_run[148,0] = p_R18_c
    struct[0].u_run[149,0] = q_R18_c
    struct[0].u_run[150,0] = u_dummy
    # Differential equations:
    if mode == 2:


        struct[0].f[0,0] = u_dummy - x_dummy
    
    # Algebraic equations:
    if mode == 3:

        g_n = np.ascontiguousarray(struct[0].Gy) @ np.ascontiguousarray(struct[0].y_run) + np.ascontiguousarray(struct[0].Gu) @ np.ascontiguousarray(struct[0].u_run)

        struct[0].g[0,0] = g_n[0,0]
        struct[0].g[1,0] = g_n[1,0]
        struct[0].g[2,0] = g_n[2,0]
        struct[0].g[3,0] = g_n[3,0]
        struct[0].g[4,0] = g_n[4,0]
        struct[0].g[5,0] = g_n[5,0]
        struct[0].g[6,0] = i_R1_n_r + 83.8007853900572*v_R1_a_i + 58.0215758469966*v_R1_a_r + 96.8143762456598*v_R1_b_i + 43.3399527360457*v_R1_b_r + 83.8007853900572*v_R1_c_i + 58.0215758469966*v_R1_c_r - 274.654008062378*v_R1_n_i - 161.052161470043*v_R1_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r - 23.284964480954*v_R2_b_i - 24.9575997948692*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058202*v_R2_c_r + 54.0657727682604*v_R2_n_i + 105.57176931318*v_R2_n_r
        struct[0].g[7,0] = i_R1_n_i + 58.0215758469966*v_R1_a_i - 83.8007853900572*v_R1_a_r + 43.3399527360457*v_R1_b_i - 96.8143762456598*v_R1_b_r + 58.0215758469966*v_R1_c_i - 83.8007853900572*v_R1_c_r - 161.052161470043*v_R1_n_i + 274.654008062378*v_R1_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r - 24.9575997948692*v_R2_b_i + 23.284964480954*v_R2_b_r - 39.6392229058202*v_R2_c_i + 10.2713736253513*v_R2_c_r + 105.57176931318*v_R2_n_i - 54.0657727682604*v_R2_n_r
        struct[0].g[8,0] = g_n[8,0]
        struct[0].g[9,0] = g_n[9,0]
        struct[0].g[10,0] = g_n[10,0]
        struct[0].g[11,0] = g_n[11,0]
        struct[0].g[12,0] = g_n[12,0]
        struct[0].g[13,0] = g_n[13,0]
        struct[0].g[14,0] = i_R11_n_r - 1.58482814504483*v_R11_a_i + 7.89461337514771*v_R11_a_r - 0.679918453285308*v_R11_b_i + 7.72144347466701*v_R11_b_r - 1.58482814504483*v_R11_c_i + 7.89461337514771*v_R11_c_r - 4.84676915585801*v_R11_n_i - 26.5550693004806*v_R11_n_r + 1.58482814504483*v_R3_a_i - 7.89461337514771*v_R3_a_r + 0.679918453285308*v_R3_b_i - 7.72144347466701*v_R3_b_r + 1.58482814504483*v_R3_c_i - 7.89461337514771*v_R3_c_r + 4.84676915585801*v_R3_n_i + 26.5300693004806*v_R3_n_r
        struct[0].g[15,0] = i_R11_n_i + 7.89461337514771*v_R11_a_i + 1.58482814504483*v_R11_a_r + 7.72144347466701*v_R11_b_i + 0.679918453285308*v_R11_b_r + 7.89461337514771*v_R11_c_i + 1.58482814504483*v_R11_c_r - 26.5550693004806*v_R11_n_i + 4.84676915585801*v_R11_n_r - 7.89461337514771*v_R3_a_i - 1.58482814504483*v_R3_a_r - 7.72144347466701*v_R3_b_i - 0.679918453285308*v_R3_b_r - 7.89461337514771*v_R3_c_i - 1.58482814504483*v_R3_c_r + 26.5300693004806*v_R3_n_i - 4.84676915585801*v_R3_n_r
        struct[0].g[16,0] = g_n[16,0]
        struct[0].g[17,0] = g_n[17,0]
        struct[0].g[18,0] = g_n[18,0]
        struct[0].g[19,0] = g_n[19,0]
        struct[0].g[20,0] = g_n[20,0]
        struct[0].g[21,0] = g_n[21,0]
        struct[0].g[22,0] = i_R15_n_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 0.679918453285308*v_R14_b_i - 7.72144347466701*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 4.84676915585801*v_R14_n_i + 26.5300693004806*v_R14_n_r - 1.58482814504483*v_R15_a_i + 7.89461337514771*v_R15_a_r - 0.679918453285308*v_R15_b_i + 7.72144347466701*v_R15_b_r - 1.58482814504483*v_R15_c_i + 7.89461337514771*v_R15_c_r - 4.84676915585801*v_R15_n_i - 26.5550693004806*v_R15_n_r
        struct[0].g[23,0] = i_R15_n_i - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r - 7.72144347466701*v_R14_b_i - 0.679918453285308*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r + 26.5300693004806*v_R14_n_i - 4.84676915585801*v_R14_n_r + 7.89461337514771*v_R15_a_i + 1.58482814504483*v_R15_a_r + 7.72144347466701*v_R15_b_i + 0.679918453285308*v_R15_b_r + 7.89461337514771*v_R15_c_i + 1.58482814504483*v_R15_c_r - 26.5550693004806*v_R15_n_i + 4.84676915585801*v_R15_n_r
        struct[0].g[24,0] = g_n[24,0]
        struct[0].g[25,0] = g_n[25,0]
        struct[0].g[26,0] = g_n[26,0]
        struct[0].g[27,0] = g_n[27,0]
        struct[0].g[28,0] = g_n[28,0]
        struct[0].g[29,0] = g_n[29,0]
        struct[0].g[30,0] = i_R16_n_r - 1.84896616921897*v_R16_a_i + 9.21038227100566*v_R16_a_r - 0.793238195499527*v_R16_b_i + 9.00835072044485*v_R16_b_r - 1.84896616921897*v_R16_c_i + 9.21038227100566*v_R16_c_r - 5.65456401516768*v_R16_n_i - 30.9767475172273*v_R16_n_r + 1.84896616921897*v_R6_a_i - 9.21038227100566*v_R6_a_r + 0.793238195499527*v_R6_b_i - 9.00835072044485*v_R6_b_r + 1.84896616921897*v_R6_c_i - 9.21038227100566*v_R6_c_r + 5.65456401516768*v_R6_n_i + 30.9517475172273*v_R6_n_r
        struct[0].g[31,0] = i_R16_n_i + 9.21038227100566*v_R16_a_i + 1.84896616921897*v_R16_a_r + 9.00835072044485*v_R16_b_i + 0.793238195499527*v_R16_b_r + 9.21038227100566*v_R16_c_i + 1.84896616921897*v_R16_c_r - 30.9767475172273*v_R16_n_i + 5.65456401516768*v_R16_n_r - 9.21038227100566*v_R6_a_i - 1.84896616921897*v_R6_a_r - 9.00835072044485*v_R6_b_i - 0.793238195499527*v_R6_b_r - 9.21038227100566*v_R6_c_i - 1.84896616921897*v_R6_c_r + 30.9517475172273*v_R6_n_i - 5.65456401516768*v_R6_n_r
        struct[0].g[32,0] = g_n[32,0]
        struct[0].g[33,0] = g_n[33,0]
        struct[0].g[34,0] = g_n[34,0]
        struct[0].g[35,0] = g_n[35,0]
        struct[0].g[36,0] = g_n[36,0]
        struct[0].g[37,0] = g_n[37,0]
        struct[0].g[38,0] = i_R17_n_r - 1.84896616921897*v_R17_a_i + 9.21038227100566*v_R17_a_r - 0.793238195499527*v_R17_b_i + 9.00835072044485*v_R17_b_r - 1.84896616921897*v_R17_c_i + 9.21038227100566*v_R17_c_r - 5.65456401516768*v_R17_n_i - 30.9767475172273*v_R17_n_r + 1.84896616921897*v_R9_a_i - 9.21038227100566*v_R9_a_r + 0.793238195499527*v_R9_b_i - 9.00835072044485*v_R9_b_r + 1.84896616921897*v_R9_c_i - 9.21038227100566*v_R9_c_r + 5.65456401516768*v_R9_n_i + 30.9517475172273*v_R9_n_r
        struct[0].g[39,0] = i_R17_n_i + 9.21038227100566*v_R17_a_i + 1.84896616921897*v_R17_a_r + 9.00835072044485*v_R17_b_i + 0.793238195499527*v_R17_b_r + 9.21038227100566*v_R17_c_i + 1.84896616921897*v_R17_c_r - 30.9767475172273*v_R17_n_i + 5.65456401516768*v_R17_n_r - 9.21038227100566*v_R9_a_i - 1.84896616921897*v_R9_a_r - 9.00835072044485*v_R9_b_i - 0.793238195499527*v_R9_b_r - 9.21038227100566*v_R9_c_i - 1.84896616921897*v_R9_c_r + 30.9517475172273*v_R9_n_i - 5.65456401516768*v_R9_n_r
        struct[0].g[40,0] = g_n[40,0]
        struct[0].g[41,0] = g_n[41,0]
        struct[0].g[42,0] = g_n[42,0]
        struct[0].g[43,0] = g_n[43,0]
        struct[0].g[44,0] = g_n[44,0]
        struct[0].g[45,0] = g_n[45,0]
        struct[0].g[46,0] = i_R18_n_r + 1.84896616921897*v_R10_a_i - 9.21038227100566*v_R10_a_r + 0.793238195499527*v_R10_b_i - 9.00835072044485*v_R10_b_r + 1.84896616921897*v_R10_c_i - 9.21038227100566*v_R10_c_r + 5.65456401516768*v_R10_n_i + 30.9517475172273*v_R10_n_r - 1.84896616921897*v_R18_a_i + 9.21038227100566*v_R18_a_r - 0.793238195499527*v_R18_b_i + 9.00835072044485*v_R18_b_r - 1.84896616921897*v_R18_c_i + 9.21038227100566*v_R18_c_r - 5.65456401516768*v_R18_n_i - 30.9767475172273*v_R18_n_r
        struct[0].g[47,0] = i_R18_n_i - 9.21038227100566*v_R10_a_i - 1.84896616921897*v_R10_a_r - 9.00835072044485*v_R10_b_i - 0.793238195499527*v_R10_b_r - 9.21038227100566*v_R10_c_i - 1.84896616921897*v_R10_c_r + 30.9517475172273*v_R10_n_i - 5.65456401516768*v_R10_n_r + 9.21038227100566*v_R18_a_i + 1.84896616921897*v_R18_a_r + 9.00835072044485*v_R18_b_i + 0.793238195499527*v_R18_b_r + 9.21038227100566*v_R18_c_i + 1.84896616921897*v_R18_c_r - 30.9767475172273*v_R18_n_i + 5.65456401516768*v_R18_n_r
        struct[0].g[48,0] = g_n[48,0]
        struct[0].g[49,0] = g_n[49,0]
        struct[0].g[50,0] = g_n[50,0]
        struct[0].g[51,0] = g_n[51,0]
        struct[0].g[52,0] = g_n[52,0]
        struct[0].g[53,0] = g_n[53,0]
        struct[0].g[54,0] = g_n[54,0]
        struct[0].g[55,0] = g_n[55,0]
        struct[0].g[56,0] = g_n[56,0]
        struct[0].g[57,0] = g_n[57,0]
        struct[0].g[58,0] = g_n[58,0]
        struct[0].g[59,0] = g_n[59,0]
        struct[0].g[60,0] = g_n[60,0]
        struct[0].g[61,0] = g_n[61,0]
        struct[0].g[62,0] = g_n[62,0]
        struct[0].g[63,0] = g_n[63,0]
        struct[0].g[64,0] = g_n[64,0]
        struct[0].g[65,0] = g_n[65,0]
        struct[0].g[66,0] = g_n[66,0]
        struct[0].g[67,0] = g_n[67,0]
        struct[0].g[68,0] = g_n[68,0]
        struct[0].g[69,0] = g_n[69,0]
        struct[0].g[70,0] = g_n[70,0]
        struct[0].g[71,0] = g_n[71,0]
        struct[0].g[72,0] = g_n[72,0]
        struct[0].g[73,0] = g_n[73,0]
        struct[0].g[74,0] = g_n[74,0]
        struct[0].g[75,0] = g_n[75,0]
        struct[0].g[76,0] = g_n[76,0]
        struct[0].g[77,0] = g_n[77,0]
        struct[0].g[78,0] = g_n[78,0]
        struct[0].g[79,0] = g_n[79,0]
        struct[0].g[80,0] = g_n[80,0]
        struct[0].g[81,0] = g_n[81,0]
        struct[0].g[82,0] = g_n[82,0]
        struct[0].g[83,0] = g_n[83,0]
        struct[0].g[84,0] = g_n[84,0]
        struct[0].g[85,0] = g_n[85,0]
        struct[0].g[86,0] = g_n[86,0]
        struct[0].g[87,0] = g_n[87,0]
        struct[0].g[88,0] = g_n[88,0]
        struct[0].g[89,0] = g_n[89,0]
        struct[0].g[90,0] = g_n[90,0]
        struct[0].g[91,0] = g_n[91,0]
        struct[0].g[92,0] = g_n[92,0]
        struct[0].g[93,0] = g_n[93,0]
        struct[0].g[94,0] = g_n[94,0]
        struct[0].g[95,0] = g_n[95,0]
        struct[0].g[96,0] = g_n[96,0]
        struct[0].g[97,0] = g_n[97,0]
        struct[0].g[98,0] = g_n[98,0]
        struct[0].g[99,0] = g_n[99,0]
        struct[0].g[100,0] = g_n[100,0]
        struct[0].g[101,0] = g_n[101,0]
        struct[0].g[102,0] = g_n[102,0]
        struct[0].g[103,0] = g_n[103,0]
        struct[0].g[104,0] = g_n[104,0]
        struct[0].g[105,0] = g_n[105,0]
        struct[0].g[106,0] = g_n[106,0]
        struct[0].g[107,0] = g_n[107,0]
        struct[0].g[108,0] = g_n[108,0]
        struct[0].g[109,0] = g_n[109,0]
        struct[0].g[110,0] = g_n[110,0]
        struct[0].g[111,0] = g_n[111,0]
        struct[0].g[112,0] = g_n[112,0]
        struct[0].g[113,0] = g_n[113,0]
        struct[0].g[114,0] = g_n[114,0]
        struct[0].g[115,0] = g_n[115,0]
        struct[0].g[116,0] = g_n[116,0]
        struct[0].g[117,0] = g_n[117,0]
        struct[0].g[118,0] = g_n[118,0]
        struct[0].g[119,0] = g_n[119,0]
        struct[0].g[120,0] = g_n[120,0]
        struct[0].g[121,0] = g_n[121,0]
        struct[0].g[122,0] = g_n[122,0]
        struct[0].g[123,0] = g_n[123,0]
        struct[0].g[124,0] = g_n[124,0]
        struct[0].g[125,0] = g_n[125,0]
        struct[0].g[126,0] = g_n[126,0]
        struct[0].g[127,0] = g_n[127,0]
        struct[0].g[128,0] = g_n[128,0]
        struct[0].g[129,0] = g_n[129,0]
        struct[0].g[130,0] = g_n[130,0]
        struct[0].g[131,0] = g_n[131,0]
        struct[0].g[132,0] = g_n[132,0]
        struct[0].g[133,0] = g_n[133,0]
        struct[0].g[134,0] = g_n[134,0]
        struct[0].g[135,0] = g_n[135,0]
        struct[0].g[136,0] = g_n[136,0]
        struct[0].g[137,0] = g_n[137,0]
        struct[0].g[138,0] = g_n[138,0]
        struct[0].g[139,0] = g_n[139,0]
        struct[0].g[140,0] = g_n[140,0]
        struct[0].g[141,0] = g_n[141,0]
        struct[0].g[142,0] = g_n[142,0]
        struct[0].g[143,0] = g_n[143,0]
        struct[0].g[144,0] = i_R1_a_i*v_R1_a_i - i_R1_a_i*v_R1_n_i + i_R1_a_r*v_R1_a_r - i_R1_a_r*v_R1_n_r - p_R1_a
        struct[0].g[145,0] = i_R1_b_i*v_R1_b_i - i_R1_b_i*v_R1_n_i + i_R1_b_r*v_R1_b_r - i_R1_b_r*v_R1_n_r - p_R1_b
        struct[0].g[146,0] = i_R1_c_i*v_R1_c_i - i_R1_c_i*v_R1_n_i + i_R1_c_r*v_R1_c_r - i_R1_c_r*v_R1_n_r - p_R1_c
        struct[0].g[147,0] = -i_R1_a_i*v_R1_a_r + i_R1_a_i*v_R1_n_r + i_R1_a_r*v_R1_a_i - i_R1_a_r*v_R1_n_i - q_R1_a
        struct[0].g[148,0] = -i_R1_b_i*v_R1_b_r + i_R1_b_i*v_R1_n_r + i_R1_b_r*v_R1_b_i - i_R1_b_r*v_R1_n_i - q_R1_b
        struct[0].g[149,0] = -i_R1_c_i*v_R1_c_r + i_R1_c_i*v_R1_n_r + i_R1_c_r*v_R1_c_i - i_R1_c_r*v_R1_n_i - q_R1_c
        struct[0].g[150,0] = i_R1_a_r + i_R1_b_r + i_R1_c_r + i_R1_n_r
        struct[0].g[151,0] = i_R1_a_i + i_R1_b_i + i_R1_c_i + i_R1_n_i
        struct[0].g[152,0] = i_R11_a_i*v_R11_a_i - i_R11_a_i*v_R11_n_i + i_R11_a_r*v_R11_a_r - i_R11_a_r*v_R11_n_r - p_R11_a
        struct[0].g[153,0] = i_R11_b_i*v_R11_b_i - i_R11_b_i*v_R11_n_i + i_R11_b_r*v_R11_b_r - i_R11_b_r*v_R11_n_r - p_R11_b
        struct[0].g[154,0] = i_R11_c_i*v_R11_c_i - i_R11_c_i*v_R11_n_i + i_R11_c_r*v_R11_c_r - i_R11_c_r*v_R11_n_r - p_R11_c
        struct[0].g[155,0] = -i_R11_a_i*v_R11_a_r + i_R11_a_i*v_R11_n_r + i_R11_a_r*v_R11_a_i - i_R11_a_r*v_R11_n_i - q_R11_a
        struct[0].g[156,0] = -i_R11_b_i*v_R11_b_r + i_R11_b_i*v_R11_n_r + i_R11_b_r*v_R11_b_i - i_R11_b_r*v_R11_n_i - q_R11_b
        struct[0].g[157,0] = -i_R11_c_i*v_R11_c_r + i_R11_c_i*v_R11_n_r + i_R11_c_r*v_R11_c_i - i_R11_c_r*v_R11_n_i - q_R11_c
        struct[0].g[158,0] = i_R11_a_r + i_R11_b_r + i_R11_c_r + i_R11_n_r
        struct[0].g[159,0] = i_R11_a_i + i_R11_b_i + i_R11_c_i + i_R11_n_i
        struct[0].g[160,0] = i_R15_a_i*v_R15_a_i - i_R15_a_i*v_R15_n_i + i_R15_a_r*v_R15_a_r - i_R15_a_r*v_R15_n_r - p_R15_a
        struct[0].g[161,0] = i_R15_b_i*v_R15_b_i - i_R15_b_i*v_R15_n_i + i_R15_b_r*v_R15_b_r - i_R15_b_r*v_R15_n_r - p_R15_b
        struct[0].g[162,0] = i_R15_c_i*v_R15_c_i - i_R15_c_i*v_R15_n_i + i_R15_c_r*v_R15_c_r - i_R15_c_r*v_R15_n_r - p_R15_c
        struct[0].g[163,0] = -i_R15_a_i*v_R15_a_r + i_R15_a_i*v_R15_n_r + i_R15_a_r*v_R15_a_i - i_R15_a_r*v_R15_n_i - q_R15_a
        struct[0].g[164,0] = -i_R15_b_i*v_R15_b_r + i_R15_b_i*v_R15_n_r + i_R15_b_r*v_R15_b_i - i_R15_b_r*v_R15_n_i - q_R15_b
        struct[0].g[165,0] = -i_R15_c_i*v_R15_c_r + i_R15_c_i*v_R15_n_r + i_R15_c_r*v_R15_c_i - i_R15_c_r*v_R15_n_i - q_R15_c
        struct[0].g[166,0] = i_R15_a_r + i_R15_b_r + i_R15_c_r + i_R15_n_r
        struct[0].g[167,0] = i_R15_a_i + i_R15_b_i + i_R15_c_i + i_R15_n_i
        struct[0].g[168,0] = i_R16_a_i*v_R16_a_i - i_R16_a_i*v_R16_n_i + i_R16_a_r*v_R16_a_r - i_R16_a_r*v_R16_n_r - p_R16_a
        struct[0].g[169,0] = i_R16_b_i*v_R16_b_i - i_R16_b_i*v_R16_n_i + i_R16_b_r*v_R16_b_r - i_R16_b_r*v_R16_n_r - p_R16_b
        struct[0].g[170,0] = i_R16_c_i*v_R16_c_i - i_R16_c_i*v_R16_n_i + i_R16_c_r*v_R16_c_r - i_R16_c_r*v_R16_n_r - p_R16_c
        struct[0].g[171,0] = -i_R16_a_i*v_R16_a_r + i_R16_a_i*v_R16_n_r + i_R16_a_r*v_R16_a_i - i_R16_a_r*v_R16_n_i - q_R16_a
        struct[0].g[172,0] = -i_R16_b_i*v_R16_b_r + i_R16_b_i*v_R16_n_r + i_R16_b_r*v_R16_b_i - i_R16_b_r*v_R16_n_i - q_R16_b
        struct[0].g[173,0] = -i_R16_c_i*v_R16_c_r + i_R16_c_i*v_R16_n_r + i_R16_c_r*v_R16_c_i - i_R16_c_r*v_R16_n_i - q_R16_c
        struct[0].g[174,0] = i_R16_a_r + i_R16_b_r + i_R16_c_r + i_R16_n_r
        struct[0].g[175,0] = i_R16_a_i + i_R16_b_i + i_R16_c_i + i_R16_n_i
        struct[0].g[176,0] = i_R17_a_i*v_R17_a_i - i_R17_a_i*v_R17_n_i + i_R17_a_r*v_R17_a_r - i_R17_a_r*v_R17_n_r - p_R17_a
        struct[0].g[177,0] = i_R17_b_i*v_R17_b_i - i_R17_b_i*v_R17_n_i + i_R17_b_r*v_R17_b_r - i_R17_b_r*v_R17_n_r - p_R17_b
        struct[0].g[178,0] = i_R17_c_i*v_R17_c_i - i_R17_c_i*v_R17_n_i + i_R17_c_r*v_R17_c_r - i_R17_c_r*v_R17_n_r - p_R17_c
        struct[0].g[179,0] = -i_R17_a_i*v_R17_a_r + i_R17_a_i*v_R17_n_r + i_R17_a_r*v_R17_a_i - i_R17_a_r*v_R17_n_i - q_R17_a
        struct[0].g[180,0] = -i_R17_b_i*v_R17_b_r + i_R17_b_i*v_R17_n_r + i_R17_b_r*v_R17_b_i - i_R17_b_r*v_R17_n_i - q_R17_b
        struct[0].g[181,0] = -i_R17_c_i*v_R17_c_r + i_R17_c_i*v_R17_n_r + i_R17_c_r*v_R17_c_i - i_R17_c_r*v_R17_n_i - q_R17_c
        struct[0].g[182,0] = i_R17_a_r + i_R17_b_r + i_R17_c_r + i_R17_n_r
        struct[0].g[183,0] = i_R17_a_i + i_R17_b_i + i_R17_c_i + i_R17_n_i
        struct[0].g[184,0] = i_R18_a_i*v_R18_a_i - i_R18_a_i*v_R18_n_i + i_R18_a_r*v_R18_a_r - i_R18_a_r*v_R18_n_r - p_R18_a
        struct[0].g[185,0] = i_R18_b_i*v_R18_b_i - i_R18_b_i*v_R18_n_i + i_R18_b_r*v_R18_b_r - i_R18_b_r*v_R18_n_r - p_R18_b
        struct[0].g[186,0] = i_R18_c_i*v_R18_c_i - i_R18_c_i*v_R18_n_i + i_R18_c_r*v_R18_c_r - i_R18_c_r*v_R18_n_r - p_R18_c
        struct[0].g[187,0] = -i_R18_a_i*v_R18_a_r + i_R18_a_i*v_R18_n_r + i_R18_a_r*v_R18_a_i - i_R18_a_r*v_R18_n_i - q_R18_a
        struct[0].g[188,0] = -i_R18_b_i*v_R18_b_r + i_R18_b_i*v_R18_n_r + i_R18_b_r*v_R18_b_i - i_R18_b_r*v_R18_n_i - q_R18_b
        struct[0].g[189,0] = -i_R18_c_i*v_R18_c_r + i_R18_c_i*v_R18_n_r + i_R18_c_r*v_R18_c_i - i_R18_c_r*v_R18_n_i - q_R18_c
        struct[0].g[190,0] = i_R18_a_r + i_R18_b_r + i_R18_c_r + i_R18_n_r
        struct[0].g[191,0] = i_R18_a_i + i_R18_b_i + i_R18_c_i + i_R18_n_i
    
    # Outputs:
    if mode == 3:

        struct[0].h[0,0] = (v_R0_a_i**2 + v_R0_a_r**2)**0.5
        struct[0].h[1,0] = (v_R0_b_i**2 + v_R0_b_r**2)**0.5
        struct[0].h[2,0] = (v_R0_c_i**2 + v_R0_c_r**2)**0.5
        struct[0].h[3,0] = (v_R1_a_i**2 + v_R1_a_r**2)**0.5
        struct[0].h[4,0] = (v_R1_b_i**2 + v_R1_b_r**2)**0.5
        struct[0].h[5,0] = (v_R1_c_i**2 + v_R1_c_r**2)**0.5
        struct[0].h[6,0] = (v_R1_n_i**2 + v_R1_n_r**2)**0.5
        struct[0].h[7,0] = (v_R11_a_i**2 + v_R11_a_r**2)**0.5
        struct[0].h[8,0] = (v_R11_b_i**2 + v_R11_b_r**2)**0.5
        struct[0].h[9,0] = (v_R11_c_i**2 + v_R11_c_r**2)**0.5
        struct[0].h[10,0] = (v_R11_n_i**2 + v_R11_n_r**2)**0.5
        struct[0].h[11,0] = (v_R15_a_i**2 + v_R15_a_r**2)**0.5
        struct[0].h[12,0] = (v_R15_b_i**2 + v_R15_b_r**2)**0.5
        struct[0].h[13,0] = (v_R15_c_i**2 + v_R15_c_r**2)**0.5
        struct[0].h[14,0] = (v_R15_n_i**2 + v_R15_n_r**2)**0.5
        struct[0].h[15,0] = (v_R16_a_i**2 + v_R16_a_r**2)**0.5
        struct[0].h[16,0] = (v_R16_b_i**2 + v_R16_b_r**2)**0.5
        struct[0].h[17,0] = (v_R16_c_i**2 + v_R16_c_r**2)**0.5
        struct[0].h[18,0] = (v_R16_n_i**2 + v_R16_n_r**2)**0.5
        struct[0].h[19,0] = (v_R17_a_i**2 + v_R17_a_r**2)**0.5
        struct[0].h[20,0] = (v_R17_b_i**2 + v_R17_b_r**2)**0.5
        struct[0].h[21,0] = (v_R17_c_i**2 + v_R17_c_r**2)**0.5
        struct[0].h[22,0] = (v_R17_n_i**2 + v_R17_n_r**2)**0.5
        struct[0].h[23,0] = (v_R18_a_i**2 + v_R18_a_r**2)**0.5
        struct[0].h[24,0] = (v_R18_b_i**2 + v_R18_b_r**2)**0.5
        struct[0].h[25,0] = (v_R18_c_i**2 + v_R18_c_r**2)**0.5
        struct[0].h[26,0] = (v_R18_n_i**2 + v_R18_n_r**2)**0.5
        struct[0].h[27,0] = (v_R2_a_i**2 + v_R2_a_r**2)**0.5
        struct[0].h[28,0] = (v_R2_b_i**2 + v_R2_b_r**2)**0.5
        struct[0].h[29,0] = (v_R2_c_i**2 + v_R2_c_r**2)**0.5
        struct[0].h[30,0] = (v_R2_n_i**2 + v_R2_n_r**2)**0.5
        struct[0].h[31,0] = (v_R3_a_i**2 + v_R3_a_r**2)**0.5
        struct[0].h[32,0] = (v_R3_b_i**2 + v_R3_b_r**2)**0.5
        struct[0].h[33,0] = (v_R3_c_i**2 + v_R3_c_r**2)**0.5
        struct[0].h[34,0] = (v_R3_n_i**2 + v_R3_n_r**2)**0.5
        struct[0].h[35,0] = (v_R4_a_i**2 + v_R4_a_r**2)**0.5
        struct[0].h[36,0] = (v_R4_b_i**2 + v_R4_b_r**2)**0.5
        struct[0].h[37,0] = (v_R4_c_i**2 + v_R4_c_r**2)**0.5
        struct[0].h[38,0] = (v_R4_n_i**2 + v_R4_n_r**2)**0.5
        struct[0].h[39,0] = (v_R5_a_i**2 + v_R5_a_r**2)**0.5
        struct[0].h[40,0] = (v_R5_b_i**2 + v_R5_b_r**2)**0.5
        struct[0].h[41,0] = (v_R5_c_i**2 + v_R5_c_r**2)**0.5
        struct[0].h[42,0] = (v_R5_n_i**2 + v_R5_n_r**2)**0.5
        struct[0].h[43,0] = (v_R6_a_i**2 + v_R6_a_r**2)**0.5
        struct[0].h[44,0] = (v_R6_b_i**2 + v_R6_b_r**2)**0.5
        struct[0].h[45,0] = (v_R6_c_i**2 + v_R6_c_r**2)**0.5
        struct[0].h[46,0] = (v_R6_n_i**2 + v_R6_n_r**2)**0.5
        struct[0].h[47,0] = (v_R7_a_i**2 + v_R7_a_r**2)**0.5
        struct[0].h[48,0] = (v_R7_b_i**2 + v_R7_b_r**2)**0.5
        struct[0].h[49,0] = (v_R7_c_i**2 + v_R7_c_r**2)**0.5
        struct[0].h[50,0] = (v_R7_n_i**2 + v_R7_n_r**2)**0.5
        struct[0].h[51,0] = (v_R8_a_i**2 + v_R8_a_r**2)**0.5
        struct[0].h[52,0] = (v_R8_b_i**2 + v_R8_b_r**2)**0.5
        struct[0].h[53,0] = (v_R8_c_i**2 + v_R8_c_r**2)**0.5
        struct[0].h[54,0] = (v_R8_n_i**2 + v_R8_n_r**2)**0.5
        struct[0].h[55,0] = (v_R9_a_i**2 + v_R9_a_r**2)**0.5
        struct[0].h[56,0] = (v_R9_b_i**2 + v_R9_b_r**2)**0.5
        struct[0].h[57,0] = (v_R9_c_i**2 + v_R9_c_r**2)**0.5
        struct[0].h[58,0] = (v_R9_n_i**2 + v_R9_n_r**2)**0.5
        struct[0].h[59,0] = (v_R10_a_i**2 + v_R10_a_r**2)**0.5
        struct[0].h[60,0] = (v_R10_b_i**2 + v_R10_b_r**2)**0.5
        struct[0].h[61,0] = (v_R10_c_i**2 + v_R10_c_r**2)**0.5
        struct[0].h[62,0] = (v_R10_n_i**2 + v_R10_n_r**2)**0.5
        struct[0].h[63,0] = (v_R12_a_i**2 + v_R12_a_r**2)**0.5
        struct[0].h[64,0] = (v_R12_b_i**2 + v_R12_b_r**2)**0.5
        struct[0].h[65,0] = (v_R12_c_i**2 + v_R12_c_r**2)**0.5
        struct[0].h[66,0] = (v_R12_n_i**2 + v_R12_n_r**2)**0.5
        struct[0].h[67,0] = (v_R13_a_i**2 + v_R13_a_r**2)**0.5
        struct[0].h[68,0] = (v_R13_b_i**2 + v_R13_b_r**2)**0.5
        struct[0].h[69,0] = (v_R13_c_i**2 + v_R13_c_r**2)**0.5
        struct[0].h[70,0] = (v_R13_n_i**2 + v_R13_n_r**2)**0.5
        struct[0].h[71,0] = (v_R14_a_i**2 + v_R14_a_r**2)**0.5
        struct[0].h[72,0] = (v_R14_b_i**2 + v_R14_b_r**2)**0.5
        struct[0].h[73,0] = (v_R14_c_i**2 + v_R14_c_r**2)**0.5
        struct[0].h[74,0] = (v_R14_n_i**2 + v_R14_n_r**2)**0.5
    

    if mode == 10:

        pass

    if mode == 11:



        struct[0].Gy[144,0] = i_R1_a_r
        struct[0].Gy[144,1] = i_R1_a_i
        struct[0].Gy[144,6] = -i_R1_a_r
        struct[0].Gy[144,7] = -i_R1_a_i
        struct[0].Gy[144,144] = v_R1_a_r - v_R1_n_r
        struct[0].Gy[144,145] = v_R1_a_i - v_R1_n_i
        struct[0].Gy[145,2] = i_R1_b_r
        struct[0].Gy[145,3] = i_R1_b_i
        struct[0].Gy[145,6] = -i_R1_b_r
        struct[0].Gy[145,7] = -i_R1_b_i
        struct[0].Gy[145,146] = v_R1_b_r - v_R1_n_r
        struct[0].Gy[145,147] = v_R1_b_i - v_R1_n_i
        struct[0].Gy[146,4] = i_R1_c_r
        struct[0].Gy[146,5] = i_R1_c_i
        struct[0].Gy[146,6] = -i_R1_c_r
        struct[0].Gy[146,7] = -i_R1_c_i
        struct[0].Gy[146,148] = v_R1_c_r - v_R1_n_r
        struct[0].Gy[146,149] = v_R1_c_i - v_R1_n_i
        struct[0].Gy[147,0] = -i_R1_a_i
        struct[0].Gy[147,1] = i_R1_a_r
        struct[0].Gy[147,6] = i_R1_a_i
        struct[0].Gy[147,7] = -i_R1_a_r
        struct[0].Gy[147,144] = v_R1_a_i - v_R1_n_i
        struct[0].Gy[147,145] = -v_R1_a_r + v_R1_n_r
        struct[0].Gy[148,2] = -i_R1_b_i
        struct[0].Gy[148,3] = i_R1_b_r
        struct[0].Gy[148,6] = i_R1_b_i
        struct[0].Gy[148,7] = -i_R1_b_r
        struct[0].Gy[148,146] = v_R1_b_i - v_R1_n_i
        struct[0].Gy[148,147] = -v_R1_b_r + v_R1_n_r
        struct[0].Gy[149,4] = -i_R1_c_i
        struct[0].Gy[149,5] = i_R1_c_r
        struct[0].Gy[149,6] = i_R1_c_i
        struct[0].Gy[149,7] = -i_R1_c_r
        struct[0].Gy[149,148] = v_R1_c_i - v_R1_n_i
        struct[0].Gy[149,149] = -v_R1_c_r + v_R1_n_r
        struct[0].Gy[152,8] = i_R11_a_r
        struct[0].Gy[152,9] = i_R11_a_i
        struct[0].Gy[152,14] = -i_R11_a_r
        struct[0].Gy[152,15] = -i_R11_a_i
        struct[0].Gy[152,152] = v_R11_a_r - v_R11_n_r
        struct[0].Gy[152,153] = v_R11_a_i - v_R11_n_i
        struct[0].Gy[153,10] = i_R11_b_r
        struct[0].Gy[153,11] = i_R11_b_i
        struct[0].Gy[153,14] = -i_R11_b_r
        struct[0].Gy[153,15] = -i_R11_b_i
        struct[0].Gy[153,154] = v_R11_b_r - v_R11_n_r
        struct[0].Gy[153,155] = v_R11_b_i - v_R11_n_i
        struct[0].Gy[154,12] = i_R11_c_r
        struct[0].Gy[154,13] = i_R11_c_i
        struct[0].Gy[154,14] = -i_R11_c_r
        struct[0].Gy[154,15] = -i_R11_c_i
        struct[0].Gy[154,156] = v_R11_c_r - v_R11_n_r
        struct[0].Gy[154,157] = v_R11_c_i - v_R11_n_i
        struct[0].Gy[155,8] = -i_R11_a_i
        struct[0].Gy[155,9] = i_R11_a_r
        struct[0].Gy[155,14] = i_R11_a_i
        struct[0].Gy[155,15] = -i_R11_a_r
        struct[0].Gy[155,152] = v_R11_a_i - v_R11_n_i
        struct[0].Gy[155,153] = -v_R11_a_r + v_R11_n_r
        struct[0].Gy[156,10] = -i_R11_b_i
        struct[0].Gy[156,11] = i_R11_b_r
        struct[0].Gy[156,14] = i_R11_b_i
        struct[0].Gy[156,15] = -i_R11_b_r
        struct[0].Gy[156,154] = v_R11_b_i - v_R11_n_i
        struct[0].Gy[156,155] = -v_R11_b_r + v_R11_n_r
        struct[0].Gy[157,12] = -i_R11_c_i
        struct[0].Gy[157,13] = i_R11_c_r
        struct[0].Gy[157,14] = i_R11_c_i
        struct[0].Gy[157,15] = -i_R11_c_r
        struct[0].Gy[157,156] = v_R11_c_i - v_R11_n_i
        struct[0].Gy[157,157] = -v_R11_c_r + v_R11_n_r
        struct[0].Gy[160,16] = i_R15_a_r
        struct[0].Gy[160,17] = i_R15_a_i
        struct[0].Gy[160,22] = -i_R15_a_r
        struct[0].Gy[160,23] = -i_R15_a_i
        struct[0].Gy[160,160] = v_R15_a_r - v_R15_n_r
        struct[0].Gy[160,161] = v_R15_a_i - v_R15_n_i
        struct[0].Gy[161,18] = i_R15_b_r
        struct[0].Gy[161,19] = i_R15_b_i
        struct[0].Gy[161,22] = -i_R15_b_r
        struct[0].Gy[161,23] = -i_R15_b_i
        struct[0].Gy[161,162] = v_R15_b_r - v_R15_n_r
        struct[0].Gy[161,163] = v_R15_b_i - v_R15_n_i
        struct[0].Gy[162,20] = i_R15_c_r
        struct[0].Gy[162,21] = i_R15_c_i
        struct[0].Gy[162,22] = -i_R15_c_r
        struct[0].Gy[162,23] = -i_R15_c_i
        struct[0].Gy[162,164] = v_R15_c_r - v_R15_n_r
        struct[0].Gy[162,165] = v_R15_c_i - v_R15_n_i
        struct[0].Gy[163,16] = -i_R15_a_i
        struct[0].Gy[163,17] = i_R15_a_r
        struct[0].Gy[163,22] = i_R15_a_i
        struct[0].Gy[163,23] = -i_R15_a_r
        struct[0].Gy[163,160] = v_R15_a_i - v_R15_n_i
        struct[0].Gy[163,161] = -v_R15_a_r + v_R15_n_r
        struct[0].Gy[164,18] = -i_R15_b_i
        struct[0].Gy[164,19] = i_R15_b_r
        struct[0].Gy[164,22] = i_R15_b_i
        struct[0].Gy[164,23] = -i_R15_b_r
        struct[0].Gy[164,162] = v_R15_b_i - v_R15_n_i
        struct[0].Gy[164,163] = -v_R15_b_r + v_R15_n_r
        struct[0].Gy[165,20] = -i_R15_c_i
        struct[0].Gy[165,21] = i_R15_c_r
        struct[0].Gy[165,22] = i_R15_c_i
        struct[0].Gy[165,23] = -i_R15_c_r
        struct[0].Gy[165,164] = v_R15_c_i - v_R15_n_i
        struct[0].Gy[165,165] = -v_R15_c_r + v_R15_n_r
        struct[0].Gy[168,24] = i_R16_a_r
        struct[0].Gy[168,25] = i_R16_a_i
        struct[0].Gy[168,30] = -i_R16_a_r
        struct[0].Gy[168,31] = -i_R16_a_i
        struct[0].Gy[168,168] = v_R16_a_r - v_R16_n_r
        struct[0].Gy[168,169] = v_R16_a_i - v_R16_n_i
        struct[0].Gy[169,26] = i_R16_b_r
        struct[0].Gy[169,27] = i_R16_b_i
        struct[0].Gy[169,30] = -i_R16_b_r
        struct[0].Gy[169,31] = -i_R16_b_i
        struct[0].Gy[169,170] = v_R16_b_r - v_R16_n_r
        struct[0].Gy[169,171] = v_R16_b_i - v_R16_n_i
        struct[0].Gy[170,28] = i_R16_c_r
        struct[0].Gy[170,29] = i_R16_c_i
        struct[0].Gy[170,30] = -i_R16_c_r
        struct[0].Gy[170,31] = -i_R16_c_i
        struct[0].Gy[170,172] = v_R16_c_r - v_R16_n_r
        struct[0].Gy[170,173] = v_R16_c_i - v_R16_n_i
        struct[0].Gy[171,24] = -i_R16_a_i
        struct[0].Gy[171,25] = i_R16_a_r
        struct[0].Gy[171,30] = i_R16_a_i
        struct[0].Gy[171,31] = -i_R16_a_r
        struct[0].Gy[171,168] = v_R16_a_i - v_R16_n_i
        struct[0].Gy[171,169] = -v_R16_a_r + v_R16_n_r
        struct[0].Gy[172,26] = -i_R16_b_i
        struct[0].Gy[172,27] = i_R16_b_r
        struct[0].Gy[172,30] = i_R16_b_i
        struct[0].Gy[172,31] = -i_R16_b_r
        struct[0].Gy[172,170] = v_R16_b_i - v_R16_n_i
        struct[0].Gy[172,171] = -v_R16_b_r + v_R16_n_r
        struct[0].Gy[173,28] = -i_R16_c_i
        struct[0].Gy[173,29] = i_R16_c_r
        struct[0].Gy[173,30] = i_R16_c_i
        struct[0].Gy[173,31] = -i_R16_c_r
        struct[0].Gy[173,172] = v_R16_c_i - v_R16_n_i
        struct[0].Gy[173,173] = -v_R16_c_r + v_R16_n_r
        struct[0].Gy[176,32] = i_R17_a_r
        struct[0].Gy[176,33] = i_R17_a_i
        struct[0].Gy[176,38] = -i_R17_a_r
        struct[0].Gy[176,39] = -i_R17_a_i
        struct[0].Gy[176,176] = v_R17_a_r - v_R17_n_r
        struct[0].Gy[176,177] = v_R17_a_i - v_R17_n_i
        struct[0].Gy[177,34] = i_R17_b_r
        struct[0].Gy[177,35] = i_R17_b_i
        struct[0].Gy[177,38] = -i_R17_b_r
        struct[0].Gy[177,39] = -i_R17_b_i
        struct[0].Gy[177,178] = v_R17_b_r - v_R17_n_r
        struct[0].Gy[177,179] = v_R17_b_i - v_R17_n_i
        struct[0].Gy[178,36] = i_R17_c_r
        struct[0].Gy[178,37] = i_R17_c_i
        struct[0].Gy[178,38] = -i_R17_c_r
        struct[0].Gy[178,39] = -i_R17_c_i
        struct[0].Gy[178,180] = v_R17_c_r - v_R17_n_r
        struct[0].Gy[178,181] = v_R17_c_i - v_R17_n_i
        struct[0].Gy[179,32] = -i_R17_a_i
        struct[0].Gy[179,33] = i_R17_a_r
        struct[0].Gy[179,38] = i_R17_a_i
        struct[0].Gy[179,39] = -i_R17_a_r
        struct[0].Gy[179,176] = v_R17_a_i - v_R17_n_i
        struct[0].Gy[179,177] = -v_R17_a_r + v_R17_n_r
        struct[0].Gy[180,34] = -i_R17_b_i
        struct[0].Gy[180,35] = i_R17_b_r
        struct[0].Gy[180,38] = i_R17_b_i
        struct[0].Gy[180,39] = -i_R17_b_r
        struct[0].Gy[180,178] = v_R17_b_i - v_R17_n_i
        struct[0].Gy[180,179] = -v_R17_b_r + v_R17_n_r
        struct[0].Gy[181,36] = -i_R17_c_i
        struct[0].Gy[181,37] = i_R17_c_r
        struct[0].Gy[181,38] = i_R17_c_i
        struct[0].Gy[181,39] = -i_R17_c_r
        struct[0].Gy[181,180] = v_R17_c_i - v_R17_n_i
        struct[0].Gy[181,181] = -v_R17_c_r + v_R17_n_r
        struct[0].Gy[184,40] = i_R18_a_r
        struct[0].Gy[184,41] = i_R18_a_i
        struct[0].Gy[184,46] = -i_R18_a_r
        struct[0].Gy[184,47] = -i_R18_a_i
        struct[0].Gy[184,184] = v_R18_a_r - v_R18_n_r
        struct[0].Gy[184,185] = v_R18_a_i - v_R18_n_i
        struct[0].Gy[185,42] = i_R18_b_r
        struct[0].Gy[185,43] = i_R18_b_i
        struct[0].Gy[185,46] = -i_R18_b_r
        struct[0].Gy[185,47] = -i_R18_b_i
        struct[0].Gy[185,186] = v_R18_b_r - v_R18_n_r
        struct[0].Gy[185,187] = v_R18_b_i - v_R18_n_i
        struct[0].Gy[186,44] = i_R18_c_r
        struct[0].Gy[186,45] = i_R18_c_i
        struct[0].Gy[186,46] = -i_R18_c_r
        struct[0].Gy[186,47] = -i_R18_c_i
        struct[0].Gy[186,188] = v_R18_c_r - v_R18_n_r
        struct[0].Gy[186,189] = v_R18_c_i - v_R18_n_i
        struct[0].Gy[187,40] = -i_R18_a_i
        struct[0].Gy[187,41] = i_R18_a_r
        struct[0].Gy[187,46] = i_R18_a_i
        struct[0].Gy[187,47] = -i_R18_a_r
        struct[0].Gy[187,184] = v_R18_a_i - v_R18_n_i
        struct[0].Gy[187,185] = -v_R18_a_r + v_R18_n_r
        struct[0].Gy[188,42] = -i_R18_b_i
        struct[0].Gy[188,43] = i_R18_b_r
        struct[0].Gy[188,46] = i_R18_b_i
        struct[0].Gy[188,47] = -i_R18_b_r
        struct[0].Gy[188,186] = v_R18_b_i - v_R18_n_i
        struct[0].Gy[188,187] = -v_R18_b_r + v_R18_n_r
        struct[0].Gy[189,44] = -i_R18_c_i
        struct[0].Gy[189,45] = i_R18_c_r
        struct[0].Gy[189,46] = i_R18_c_i
        struct[0].Gy[189,47] = -i_R18_c_r
        struct[0].Gy[189,188] = v_R18_c_i - v_R18_n_i
        struct[0].Gy[189,189] = -v_R18_c_r + v_R18_n_r

    if mode > 12:




        struct[0].Hy[3,0] = 1.0*v_R1_a_r*(v_R1_a_i**2 + v_R1_a_r**2)**(-0.5)
        struct[0].Hy[3,1] = 1.0*v_R1_a_i*(v_R1_a_i**2 + v_R1_a_r**2)**(-0.5)
        struct[0].Hy[4,2] = 1.0*v_R1_b_r*(v_R1_b_i**2 + v_R1_b_r**2)**(-0.5)
        struct[0].Hy[4,3] = 1.0*v_R1_b_i*(v_R1_b_i**2 + v_R1_b_r**2)**(-0.5)
        struct[0].Hy[5,4] = 1.0*v_R1_c_r*(v_R1_c_i**2 + v_R1_c_r**2)**(-0.5)
        struct[0].Hy[5,5] = 1.0*v_R1_c_i*(v_R1_c_i**2 + v_R1_c_r**2)**(-0.5)
        struct[0].Hy[6,6] = 1.0*v_R1_n_r*(v_R1_n_i**2 + v_R1_n_r**2)**(-0.5)
        struct[0].Hy[6,7] = 1.0*v_R1_n_i*(v_R1_n_i**2 + v_R1_n_r**2)**(-0.5)
        struct[0].Hy[7,8] = 1.0*v_R11_a_r*(v_R11_a_i**2 + v_R11_a_r**2)**(-0.5)
        struct[0].Hy[7,9] = 1.0*v_R11_a_i*(v_R11_a_i**2 + v_R11_a_r**2)**(-0.5)
        struct[0].Hy[8,10] = 1.0*v_R11_b_r*(v_R11_b_i**2 + v_R11_b_r**2)**(-0.5)
        struct[0].Hy[8,11] = 1.0*v_R11_b_i*(v_R11_b_i**2 + v_R11_b_r**2)**(-0.5)
        struct[0].Hy[9,12] = 1.0*v_R11_c_r*(v_R11_c_i**2 + v_R11_c_r**2)**(-0.5)
        struct[0].Hy[9,13] = 1.0*v_R11_c_i*(v_R11_c_i**2 + v_R11_c_r**2)**(-0.5)
        struct[0].Hy[10,14] = 1.0*v_R11_n_r*(v_R11_n_i**2 + v_R11_n_r**2)**(-0.5)
        struct[0].Hy[10,15] = 1.0*v_R11_n_i*(v_R11_n_i**2 + v_R11_n_r**2)**(-0.5)
        struct[0].Hy[11,16] = 1.0*v_R15_a_r*(v_R15_a_i**2 + v_R15_a_r**2)**(-0.5)
        struct[0].Hy[11,17] = 1.0*v_R15_a_i*(v_R15_a_i**2 + v_R15_a_r**2)**(-0.5)
        struct[0].Hy[12,18] = 1.0*v_R15_b_r*(v_R15_b_i**2 + v_R15_b_r**2)**(-0.5)
        struct[0].Hy[12,19] = 1.0*v_R15_b_i*(v_R15_b_i**2 + v_R15_b_r**2)**(-0.5)
        struct[0].Hy[13,20] = 1.0*v_R15_c_r*(v_R15_c_i**2 + v_R15_c_r**2)**(-0.5)
        struct[0].Hy[13,21] = 1.0*v_R15_c_i*(v_R15_c_i**2 + v_R15_c_r**2)**(-0.5)
        struct[0].Hy[14,22] = 1.0*v_R15_n_r*(v_R15_n_i**2 + v_R15_n_r**2)**(-0.5)
        struct[0].Hy[14,23] = 1.0*v_R15_n_i*(v_R15_n_i**2 + v_R15_n_r**2)**(-0.5)
        struct[0].Hy[15,24] = 1.0*v_R16_a_r*(v_R16_a_i**2 + v_R16_a_r**2)**(-0.5)
        struct[0].Hy[15,25] = 1.0*v_R16_a_i*(v_R16_a_i**2 + v_R16_a_r**2)**(-0.5)
        struct[0].Hy[16,26] = 1.0*v_R16_b_r*(v_R16_b_i**2 + v_R16_b_r**2)**(-0.5)
        struct[0].Hy[16,27] = 1.0*v_R16_b_i*(v_R16_b_i**2 + v_R16_b_r**2)**(-0.5)
        struct[0].Hy[17,28] = 1.0*v_R16_c_r*(v_R16_c_i**2 + v_R16_c_r**2)**(-0.5)
        struct[0].Hy[17,29] = 1.0*v_R16_c_i*(v_R16_c_i**2 + v_R16_c_r**2)**(-0.5)
        struct[0].Hy[18,30] = 1.0*v_R16_n_r*(v_R16_n_i**2 + v_R16_n_r**2)**(-0.5)
        struct[0].Hy[18,31] = 1.0*v_R16_n_i*(v_R16_n_i**2 + v_R16_n_r**2)**(-0.5)
        struct[0].Hy[19,32] = 1.0*v_R17_a_r*(v_R17_a_i**2 + v_R17_a_r**2)**(-0.5)
        struct[0].Hy[19,33] = 1.0*v_R17_a_i*(v_R17_a_i**2 + v_R17_a_r**2)**(-0.5)
        struct[0].Hy[20,34] = 1.0*v_R17_b_r*(v_R17_b_i**2 + v_R17_b_r**2)**(-0.5)
        struct[0].Hy[20,35] = 1.0*v_R17_b_i*(v_R17_b_i**2 + v_R17_b_r**2)**(-0.5)
        struct[0].Hy[21,36] = 1.0*v_R17_c_r*(v_R17_c_i**2 + v_R17_c_r**2)**(-0.5)
        struct[0].Hy[21,37] = 1.0*v_R17_c_i*(v_R17_c_i**2 + v_R17_c_r**2)**(-0.5)
        struct[0].Hy[22,38] = 1.0*v_R17_n_r*(v_R17_n_i**2 + v_R17_n_r**2)**(-0.5)
        struct[0].Hy[22,39] = 1.0*v_R17_n_i*(v_R17_n_i**2 + v_R17_n_r**2)**(-0.5)
        struct[0].Hy[23,40] = 1.0*v_R18_a_r*(v_R18_a_i**2 + v_R18_a_r**2)**(-0.5)
        struct[0].Hy[23,41] = 1.0*v_R18_a_i*(v_R18_a_i**2 + v_R18_a_r**2)**(-0.5)
        struct[0].Hy[24,42] = 1.0*v_R18_b_r*(v_R18_b_i**2 + v_R18_b_r**2)**(-0.5)
        struct[0].Hy[24,43] = 1.0*v_R18_b_i*(v_R18_b_i**2 + v_R18_b_r**2)**(-0.5)
        struct[0].Hy[25,44] = 1.0*v_R18_c_r*(v_R18_c_i**2 + v_R18_c_r**2)**(-0.5)
        struct[0].Hy[25,45] = 1.0*v_R18_c_i*(v_R18_c_i**2 + v_R18_c_r**2)**(-0.5)
        struct[0].Hy[26,46] = 1.0*v_R18_n_r*(v_R18_n_i**2 + v_R18_n_r**2)**(-0.5)
        struct[0].Hy[26,47] = 1.0*v_R18_n_i*(v_R18_n_i**2 + v_R18_n_r**2)**(-0.5)
        struct[0].Hy[27,48] = 1.0*v_R2_a_r*(v_R2_a_i**2 + v_R2_a_r**2)**(-0.5)
        struct[0].Hy[27,49] = 1.0*v_R2_a_i*(v_R2_a_i**2 + v_R2_a_r**2)**(-0.5)
        struct[0].Hy[28,50] = 1.0*v_R2_b_r*(v_R2_b_i**2 + v_R2_b_r**2)**(-0.5)
        struct[0].Hy[28,51] = 1.0*v_R2_b_i*(v_R2_b_i**2 + v_R2_b_r**2)**(-0.5)
        struct[0].Hy[29,52] = 1.0*v_R2_c_r*(v_R2_c_i**2 + v_R2_c_r**2)**(-0.5)
        struct[0].Hy[29,53] = 1.0*v_R2_c_i*(v_R2_c_i**2 + v_R2_c_r**2)**(-0.5)
        struct[0].Hy[30,54] = 1.0*v_R2_n_r*(v_R2_n_i**2 + v_R2_n_r**2)**(-0.5)
        struct[0].Hy[30,55] = 1.0*v_R2_n_i*(v_R2_n_i**2 + v_R2_n_r**2)**(-0.5)
        struct[0].Hy[31,56] = 1.0*v_R3_a_r*(v_R3_a_i**2 + v_R3_a_r**2)**(-0.5)
        struct[0].Hy[31,57] = 1.0*v_R3_a_i*(v_R3_a_i**2 + v_R3_a_r**2)**(-0.5)
        struct[0].Hy[32,58] = 1.0*v_R3_b_r*(v_R3_b_i**2 + v_R3_b_r**2)**(-0.5)
        struct[0].Hy[32,59] = 1.0*v_R3_b_i*(v_R3_b_i**2 + v_R3_b_r**2)**(-0.5)
        struct[0].Hy[33,60] = 1.0*v_R3_c_r*(v_R3_c_i**2 + v_R3_c_r**2)**(-0.5)
        struct[0].Hy[33,61] = 1.0*v_R3_c_i*(v_R3_c_i**2 + v_R3_c_r**2)**(-0.5)
        struct[0].Hy[34,62] = 1.0*v_R3_n_r*(v_R3_n_i**2 + v_R3_n_r**2)**(-0.5)
        struct[0].Hy[34,63] = 1.0*v_R3_n_i*(v_R3_n_i**2 + v_R3_n_r**2)**(-0.5)
        struct[0].Hy[35,64] = 1.0*v_R4_a_r*(v_R4_a_i**2 + v_R4_a_r**2)**(-0.5)
        struct[0].Hy[35,65] = 1.0*v_R4_a_i*(v_R4_a_i**2 + v_R4_a_r**2)**(-0.5)
        struct[0].Hy[36,66] = 1.0*v_R4_b_r*(v_R4_b_i**2 + v_R4_b_r**2)**(-0.5)
        struct[0].Hy[36,67] = 1.0*v_R4_b_i*(v_R4_b_i**2 + v_R4_b_r**2)**(-0.5)
        struct[0].Hy[37,68] = 1.0*v_R4_c_r*(v_R4_c_i**2 + v_R4_c_r**2)**(-0.5)
        struct[0].Hy[37,69] = 1.0*v_R4_c_i*(v_R4_c_i**2 + v_R4_c_r**2)**(-0.5)
        struct[0].Hy[38,70] = 1.0*v_R4_n_r*(v_R4_n_i**2 + v_R4_n_r**2)**(-0.5)
        struct[0].Hy[38,71] = 1.0*v_R4_n_i*(v_R4_n_i**2 + v_R4_n_r**2)**(-0.5)
        struct[0].Hy[39,72] = 1.0*v_R5_a_r*(v_R5_a_i**2 + v_R5_a_r**2)**(-0.5)
        struct[0].Hy[39,73] = 1.0*v_R5_a_i*(v_R5_a_i**2 + v_R5_a_r**2)**(-0.5)
        struct[0].Hy[40,74] = 1.0*v_R5_b_r*(v_R5_b_i**2 + v_R5_b_r**2)**(-0.5)
        struct[0].Hy[40,75] = 1.0*v_R5_b_i*(v_R5_b_i**2 + v_R5_b_r**2)**(-0.5)
        struct[0].Hy[41,76] = 1.0*v_R5_c_r*(v_R5_c_i**2 + v_R5_c_r**2)**(-0.5)
        struct[0].Hy[41,77] = 1.0*v_R5_c_i*(v_R5_c_i**2 + v_R5_c_r**2)**(-0.5)
        struct[0].Hy[42,78] = 1.0*v_R5_n_r*(v_R5_n_i**2 + v_R5_n_r**2)**(-0.5)
        struct[0].Hy[42,79] = 1.0*v_R5_n_i*(v_R5_n_i**2 + v_R5_n_r**2)**(-0.5)
        struct[0].Hy[43,80] = 1.0*v_R6_a_r*(v_R6_a_i**2 + v_R6_a_r**2)**(-0.5)
        struct[0].Hy[43,81] = 1.0*v_R6_a_i*(v_R6_a_i**2 + v_R6_a_r**2)**(-0.5)
        struct[0].Hy[44,82] = 1.0*v_R6_b_r*(v_R6_b_i**2 + v_R6_b_r**2)**(-0.5)
        struct[0].Hy[44,83] = 1.0*v_R6_b_i*(v_R6_b_i**2 + v_R6_b_r**2)**(-0.5)
        struct[0].Hy[45,84] = 1.0*v_R6_c_r*(v_R6_c_i**2 + v_R6_c_r**2)**(-0.5)
        struct[0].Hy[45,85] = 1.0*v_R6_c_i*(v_R6_c_i**2 + v_R6_c_r**2)**(-0.5)
        struct[0].Hy[46,86] = 1.0*v_R6_n_r*(v_R6_n_i**2 + v_R6_n_r**2)**(-0.5)
        struct[0].Hy[46,87] = 1.0*v_R6_n_i*(v_R6_n_i**2 + v_R6_n_r**2)**(-0.5)
        struct[0].Hy[47,88] = 1.0*v_R7_a_r*(v_R7_a_i**2 + v_R7_a_r**2)**(-0.5)
        struct[0].Hy[47,89] = 1.0*v_R7_a_i*(v_R7_a_i**2 + v_R7_a_r**2)**(-0.5)
        struct[0].Hy[48,90] = 1.0*v_R7_b_r*(v_R7_b_i**2 + v_R7_b_r**2)**(-0.5)
        struct[0].Hy[48,91] = 1.0*v_R7_b_i*(v_R7_b_i**2 + v_R7_b_r**2)**(-0.5)
        struct[0].Hy[49,92] = 1.0*v_R7_c_r*(v_R7_c_i**2 + v_R7_c_r**2)**(-0.5)
        struct[0].Hy[49,93] = 1.0*v_R7_c_i*(v_R7_c_i**2 + v_R7_c_r**2)**(-0.5)
        struct[0].Hy[50,94] = 1.0*v_R7_n_r*(v_R7_n_i**2 + v_R7_n_r**2)**(-0.5)
        struct[0].Hy[50,95] = 1.0*v_R7_n_i*(v_R7_n_i**2 + v_R7_n_r**2)**(-0.5)
        struct[0].Hy[51,96] = 1.0*v_R8_a_r*(v_R8_a_i**2 + v_R8_a_r**2)**(-0.5)
        struct[0].Hy[51,97] = 1.0*v_R8_a_i*(v_R8_a_i**2 + v_R8_a_r**2)**(-0.5)
        struct[0].Hy[52,98] = 1.0*v_R8_b_r*(v_R8_b_i**2 + v_R8_b_r**2)**(-0.5)
        struct[0].Hy[52,99] = 1.0*v_R8_b_i*(v_R8_b_i**2 + v_R8_b_r**2)**(-0.5)
        struct[0].Hy[53,100] = 1.0*v_R8_c_r*(v_R8_c_i**2 + v_R8_c_r**2)**(-0.5)
        struct[0].Hy[53,101] = 1.0*v_R8_c_i*(v_R8_c_i**2 + v_R8_c_r**2)**(-0.5)
        struct[0].Hy[54,102] = 1.0*v_R8_n_r*(v_R8_n_i**2 + v_R8_n_r**2)**(-0.5)
        struct[0].Hy[54,103] = 1.0*v_R8_n_i*(v_R8_n_i**2 + v_R8_n_r**2)**(-0.5)
        struct[0].Hy[55,104] = 1.0*v_R9_a_r*(v_R9_a_i**2 + v_R9_a_r**2)**(-0.5)
        struct[0].Hy[55,105] = 1.0*v_R9_a_i*(v_R9_a_i**2 + v_R9_a_r**2)**(-0.5)
        struct[0].Hy[56,106] = 1.0*v_R9_b_r*(v_R9_b_i**2 + v_R9_b_r**2)**(-0.5)
        struct[0].Hy[56,107] = 1.0*v_R9_b_i*(v_R9_b_i**2 + v_R9_b_r**2)**(-0.5)
        struct[0].Hy[57,108] = 1.0*v_R9_c_r*(v_R9_c_i**2 + v_R9_c_r**2)**(-0.5)
        struct[0].Hy[57,109] = 1.0*v_R9_c_i*(v_R9_c_i**2 + v_R9_c_r**2)**(-0.5)
        struct[0].Hy[58,110] = 1.0*v_R9_n_r*(v_R9_n_i**2 + v_R9_n_r**2)**(-0.5)
        struct[0].Hy[58,111] = 1.0*v_R9_n_i*(v_R9_n_i**2 + v_R9_n_r**2)**(-0.5)
        struct[0].Hy[59,112] = 1.0*v_R10_a_r*(v_R10_a_i**2 + v_R10_a_r**2)**(-0.5)
        struct[0].Hy[59,113] = 1.0*v_R10_a_i*(v_R10_a_i**2 + v_R10_a_r**2)**(-0.5)
        struct[0].Hy[60,114] = 1.0*v_R10_b_r*(v_R10_b_i**2 + v_R10_b_r**2)**(-0.5)
        struct[0].Hy[60,115] = 1.0*v_R10_b_i*(v_R10_b_i**2 + v_R10_b_r**2)**(-0.5)
        struct[0].Hy[61,116] = 1.0*v_R10_c_r*(v_R10_c_i**2 + v_R10_c_r**2)**(-0.5)
        struct[0].Hy[61,117] = 1.0*v_R10_c_i*(v_R10_c_i**2 + v_R10_c_r**2)**(-0.5)
        struct[0].Hy[62,118] = 1.0*v_R10_n_r*(v_R10_n_i**2 + v_R10_n_r**2)**(-0.5)
        struct[0].Hy[62,119] = 1.0*v_R10_n_i*(v_R10_n_i**2 + v_R10_n_r**2)**(-0.5)
        struct[0].Hy[63,120] = 1.0*v_R12_a_r*(v_R12_a_i**2 + v_R12_a_r**2)**(-0.5)
        struct[0].Hy[63,121] = 1.0*v_R12_a_i*(v_R12_a_i**2 + v_R12_a_r**2)**(-0.5)
        struct[0].Hy[64,122] = 1.0*v_R12_b_r*(v_R12_b_i**2 + v_R12_b_r**2)**(-0.5)
        struct[0].Hy[64,123] = 1.0*v_R12_b_i*(v_R12_b_i**2 + v_R12_b_r**2)**(-0.5)
        struct[0].Hy[65,124] = 1.0*v_R12_c_r*(v_R12_c_i**2 + v_R12_c_r**2)**(-0.5)
        struct[0].Hy[65,125] = 1.0*v_R12_c_i*(v_R12_c_i**2 + v_R12_c_r**2)**(-0.5)
        struct[0].Hy[66,126] = 1.0*v_R12_n_r*(v_R12_n_i**2 + v_R12_n_r**2)**(-0.5)
        struct[0].Hy[66,127] = 1.0*v_R12_n_i*(v_R12_n_i**2 + v_R12_n_r**2)**(-0.5)
        struct[0].Hy[67,128] = 1.0*v_R13_a_r*(v_R13_a_i**2 + v_R13_a_r**2)**(-0.5)
        struct[0].Hy[67,129] = 1.0*v_R13_a_i*(v_R13_a_i**2 + v_R13_a_r**2)**(-0.5)
        struct[0].Hy[68,130] = 1.0*v_R13_b_r*(v_R13_b_i**2 + v_R13_b_r**2)**(-0.5)
        struct[0].Hy[68,131] = 1.0*v_R13_b_i*(v_R13_b_i**2 + v_R13_b_r**2)**(-0.5)
        struct[0].Hy[69,132] = 1.0*v_R13_c_r*(v_R13_c_i**2 + v_R13_c_r**2)**(-0.5)
        struct[0].Hy[69,133] = 1.0*v_R13_c_i*(v_R13_c_i**2 + v_R13_c_r**2)**(-0.5)
        struct[0].Hy[70,134] = 1.0*v_R13_n_r*(v_R13_n_i**2 + v_R13_n_r**2)**(-0.5)
        struct[0].Hy[70,135] = 1.0*v_R13_n_i*(v_R13_n_i**2 + v_R13_n_r**2)**(-0.5)
        struct[0].Hy[71,136] = 1.0*v_R14_a_r*(v_R14_a_i**2 + v_R14_a_r**2)**(-0.5)
        struct[0].Hy[71,137] = 1.0*v_R14_a_i*(v_R14_a_i**2 + v_R14_a_r**2)**(-0.5)
        struct[0].Hy[72,138] = 1.0*v_R14_b_r*(v_R14_b_i**2 + v_R14_b_r**2)**(-0.5)
        struct[0].Hy[72,139] = 1.0*v_R14_b_i*(v_R14_b_i**2 + v_R14_b_r**2)**(-0.5)
        struct[0].Hy[73,140] = 1.0*v_R14_c_r*(v_R14_c_i**2 + v_R14_c_r**2)**(-0.5)
        struct[0].Hy[73,141] = 1.0*v_R14_c_i*(v_R14_c_i**2 + v_R14_c_r**2)**(-0.5)
        struct[0].Hy[74,142] = 1.0*v_R14_n_r*(v_R14_n_i**2 + v_R14_n_r**2)**(-0.5)
        struct[0].Hy[74,143] = 1.0*v_R14_n_i*(v_R14_n_i**2 + v_R14_n_r**2)**(-0.5)

        struct[0].Hu[0,0] = 1.0*v_R0_a_r*(v_R0_a_i**2 + v_R0_a_r**2)**(-0.5)
        struct[0].Hu[0,1] = 1.0*v_R0_a_i*(v_R0_a_i**2 + v_R0_a_r**2)**(-0.5)
        struct[0].Hu[1,2] = 1.0*v_R0_b_r*(v_R0_b_i**2 + v_R0_b_r**2)**(-0.5)
        struct[0].Hu[1,3] = 1.0*v_R0_b_i*(v_R0_b_i**2 + v_R0_b_r**2)**(-0.5)
        struct[0].Hu[2,4] = 1.0*v_R0_c_r*(v_R0_c_i**2 + v_R0_c_r**2)**(-0.5)
        struct[0].Hu[2,5] = 1.0*v_R0_c_i*(v_R0_c_i**2 + v_R0_c_r**2)**(-0.5)



def ini_nn(struct,mode):

    # Parameters:
    
    # Inputs:
    v_R0_a_r = struct[0].v_R0_a_r
    v_R0_a_i = struct[0].v_R0_a_i
    v_R0_b_r = struct[0].v_R0_b_r
    v_R0_b_i = struct[0].v_R0_b_i
    v_R0_c_r = struct[0].v_R0_c_r
    v_R0_c_i = struct[0].v_R0_c_i
    i_R1_n_r = struct[0].i_R1_n_r
    i_R1_n_i = struct[0].i_R1_n_i
    i_R11_n_r = struct[0].i_R11_n_r
    i_R11_n_i = struct[0].i_R11_n_i
    i_R15_n_r = struct[0].i_R15_n_r
    i_R15_n_i = struct[0].i_R15_n_i
    i_R16_n_r = struct[0].i_R16_n_r
    i_R16_n_i = struct[0].i_R16_n_i
    i_R17_n_r = struct[0].i_R17_n_r
    i_R17_n_i = struct[0].i_R17_n_i
    i_R18_n_r = struct[0].i_R18_n_r
    i_R18_n_i = struct[0].i_R18_n_i
    i_R2_a_r = struct[0].i_R2_a_r
    i_R2_a_i = struct[0].i_R2_a_i
    i_R2_b_r = struct[0].i_R2_b_r
    i_R2_b_i = struct[0].i_R2_b_i
    i_R2_c_r = struct[0].i_R2_c_r
    i_R2_c_i = struct[0].i_R2_c_i
    i_R2_n_r = struct[0].i_R2_n_r
    i_R2_n_i = struct[0].i_R2_n_i
    i_R3_a_r = struct[0].i_R3_a_r
    i_R3_a_i = struct[0].i_R3_a_i
    i_R3_b_r = struct[0].i_R3_b_r
    i_R3_b_i = struct[0].i_R3_b_i
    i_R3_c_r = struct[0].i_R3_c_r
    i_R3_c_i = struct[0].i_R3_c_i
    i_R3_n_r = struct[0].i_R3_n_r
    i_R3_n_i = struct[0].i_R3_n_i
    i_R4_a_r = struct[0].i_R4_a_r
    i_R4_a_i = struct[0].i_R4_a_i
    i_R4_b_r = struct[0].i_R4_b_r
    i_R4_b_i = struct[0].i_R4_b_i
    i_R4_c_r = struct[0].i_R4_c_r
    i_R4_c_i = struct[0].i_R4_c_i
    i_R4_n_r = struct[0].i_R4_n_r
    i_R4_n_i = struct[0].i_R4_n_i
    i_R5_a_r = struct[0].i_R5_a_r
    i_R5_a_i = struct[0].i_R5_a_i
    i_R5_b_r = struct[0].i_R5_b_r
    i_R5_b_i = struct[0].i_R5_b_i
    i_R5_c_r = struct[0].i_R5_c_r
    i_R5_c_i = struct[0].i_R5_c_i
    i_R5_n_r = struct[0].i_R5_n_r
    i_R5_n_i = struct[0].i_R5_n_i
    i_R6_a_r = struct[0].i_R6_a_r
    i_R6_a_i = struct[0].i_R6_a_i
    i_R6_b_r = struct[0].i_R6_b_r
    i_R6_b_i = struct[0].i_R6_b_i
    i_R6_c_r = struct[0].i_R6_c_r
    i_R6_c_i = struct[0].i_R6_c_i
    i_R6_n_r = struct[0].i_R6_n_r
    i_R6_n_i = struct[0].i_R6_n_i
    i_R7_a_r = struct[0].i_R7_a_r
    i_R7_a_i = struct[0].i_R7_a_i
    i_R7_b_r = struct[0].i_R7_b_r
    i_R7_b_i = struct[0].i_R7_b_i
    i_R7_c_r = struct[0].i_R7_c_r
    i_R7_c_i = struct[0].i_R7_c_i
    i_R7_n_r = struct[0].i_R7_n_r
    i_R7_n_i = struct[0].i_R7_n_i
    i_R8_a_r = struct[0].i_R8_a_r
    i_R8_a_i = struct[0].i_R8_a_i
    i_R8_b_r = struct[0].i_R8_b_r
    i_R8_b_i = struct[0].i_R8_b_i
    i_R8_c_r = struct[0].i_R8_c_r
    i_R8_c_i = struct[0].i_R8_c_i
    i_R8_n_r = struct[0].i_R8_n_r
    i_R8_n_i = struct[0].i_R8_n_i
    i_R9_a_r = struct[0].i_R9_a_r
    i_R9_a_i = struct[0].i_R9_a_i
    i_R9_b_r = struct[0].i_R9_b_r
    i_R9_b_i = struct[0].i_R9_b_i
    i_R9_c_r = struct[0].i_R9_c_r
    i_R9_c_i = struct[0].i_R9_c_i
    i_R9_n_r = struct[0].i_R9_n_r
    i_R9_n_i = struct[0].i_R9_n_i
    i_R10_a_r = struct[0].i_R10_a_r
    i_R10_a_i = struct[0].i_R10_a_i
    i_R10_b_r = struct[0].i_R10_b_r
    i_R10_b_i = struct[0].i_R10_b_i
    i_R10_c_r = struct[0].i_R10_c_r
    i_R10_c_i = struct[0].i_R10_c_i
    i_R10_n_r = struct[0].i_R10_n_r
    i_R10_n_i = struct[0].i_R10_n_i
    i_R12_a_r = struct[0].i_R12_a_r
    i_R12_a_i = struct[0].i_R12_a_i
    i_R12_b_r = struct[0].i_R12_b_r
    i_R12_b_i = struct[0].i_R12_b_i
    i_R12_c_r = struct[0].i_R12_c_r
    i_R12_c_i = struct[0].i_R12_c_i
    i_R12_n_r = struct[0].i_R12_n_r
    i_R12_n_i = struct[0].i_R12_n_i
    i_R13_a_r = struct[0].i_R13_a_r
    i_R13_a_i = struct[0].i_R13_a_i
    i_R13_b_r = struct[0].i_R13_b_r
    i_R13_b_i = struct[0].i_R13_b_i
    i_R13_c_r = struct[0].i_R13_c_r
    i_R13_c_i = struct[0].i_R13_c_i
    i_R13_n_r = struct[0].i_R13_n_r
    i_R13_n_i = struct[0].i_R13_n_i
    i_R14_a_r = struct[0].i_R14_a_r
    i_R14_a_i = struct[0].i_R14_a_i
    i_R14_b_r = struct[0].i_R14_b_r
    i_R14_b_i = struct[0].i_R14_b_i
    i_R14_c_r = struct[0].i_R14_c_r
    i_R14_c_i = struct[0].i_R14_c_i
    i_R14_n_r = struct[0].i_R14_n_r
    i_R14_n_i = struct[0].i_R14_n_i
    p_R1_a = struct[0].p_R1_a
    q_R1_a = struct[0].q_R1_a
    p_R1_b = struct[0].p_R1_b
    q_R1_b = struct[0].q_R1_b
    p_R1_c = struct[0].p_R1_c
    q_R1_c = struct[0].q_R1_c
    p_R11_a = struct[0].p_R11_a
    q_R11_a = struct[0].q_R11_a
    p_R11_b = struct[0].p_R11_b
    q_R11_b = struct[0].q_R11_b
    p_R11_c = struct[0].p_R11_c
    q_R11_c = struct[0].q_R11_c
    p_R15_a = struct[0].p_R15_a
    q_R15_a = struct[0].q_R15_a
    p_R15_b = struct[0].p_R15_b
    q_R15_b = struct[0].q_R15_b
    p_R15_c = struct[0].p_R15_c
    q_R15_c = struct[0].q_R15_c
    p_R16_a = struct[0].p_R16_a
    q_R16_a = struct[0].q_R16_a
    p_R16_b = struct[0].p_R16_b
    q_R16_b = struct[0].q_R16_b
    p_R16_c = struct[0].p_R16_c
    q_R16_c = struct[0].q_R16_c
    p_R17_a = struct[0].p_R17_a
    q_R17_a = struct[0].q_R17_a
    p_R17_b = struct[0].p_R17_b
    q_R17_b = struct[0].q_R17_b
    p_R17_c = struct[0].p_R17_c
    q_R17_c = struct[0].q_R17_c
    p_R18_a = struct[0].p_R18_a
    q_R18_a = struct[0].q_R18_a
    p_R18_b = struct[0].p_R18_b
    q_R18_b = struct[0].q_R18_b
    p_R18_c = struct[0].p_R18_c
    q_R18_c = struct[0].q_R18_c
    u_dummy = struct[0].u_dummy
    
    # Dynamical states:
    x_dummy = struct[0].x[0,0]
    
    # Algebraic states:
    v_R1_a_r = struct[0].y_ini[0,0]
    v_R1_a_i = struct[0].y_ini[1,0]
    v_R1_b_r = struct[0].y_ini[2,0]
    v_R1_b_i = struct[0].y_ini[3,0]
    v_R1_c_r = struct[0].y_ini[4,0]
    v_R1_c_i = struct[0].y_ini[5,0]
    v_R1_n_r = struct[0].y_ini[6,0]
    v_R1_n_i = struct[0].y_ini[7,0]
    v_R11_a_r = struct[0].y_ini[8,0]
    v_R11_a_i = struct[0].y_ini[9,0]
    v_R11_b_r = struct[0].y_ini[10,0]
    v_R11_b_i = struct[0].y_ini[11,0]
    v_R11_c_r = struct[0].y_ini[12,0]
    v_R11_c_i = struct[0].y_ini[13,0]
    v_R11_n_r = struct[0].y_ini[14,0]
    v_R11_n_i = struct[0].y_ini[15,0]
    v_R15_a_r = struct[0].y_ini[16,0]
    v_R15_a_i = struct[0].y_ini[17,0]
    v_R15_b_r = struct[0].y_ini[18,0]
    v_R15_b_i = struct[0].y_ini[19,0]
    v_R15_c_r = struct[0].y_ini[20,0]
    v_R15_c_i = struct[0].y_ini[21,0]
    v_R15_n_r = struct[0].y_ini[22,0]
    v_R15_n_i = struct[0].y_ini[23,0]
    v_R16_a_r = struct[0].y_ini[24,0]
    v_R16_a_i = struct[0].y_ini[25,0]
    v_R16_b_r = struct[0].y_ini[26,0]
    v_R16_b_i = struct[0].y_ini[27,0]
    v_R16_c_r = struct[0].y_ini[28,0]
    v_R16_c_i = struct[0].y_ini[29,0]
    v_R16_n_r = struct[0].y_ini[30,0]
    v_R16_n_i = struct[0].y_ini[31,0]
    v_R17_a_r = struct[0].y_ini[32,0]
    v_R17_a_i = struct[0].y_ini[33,0]
    v_R17_b_r = struct[0].y_ini[34,0]
    v_R17_b_i = struct[0].y_ini[35,0]
    v_R17_c_r = struct[0].y_ini[36,0]
    v_R17_c_i = struct[0].y_ini[37,0]
    v_R17_n_r = struct[0].y_ini[38,0]
    v_R17_n_i = struct[0].y_ini[39,0]
    v_R18_a_r = struct[0].y_ini[40,0]
    v_R18_a_i = struct[0].y_ini[41,0]
    v_R18_b_r = struct[0].y_ini[42,0]
    v_R18_b_i = struct[0].y_ini[43,0]
    v_R18_c_r = struct[0].y_ini[44,0]
    v_R18_c_i = struct[0].y_ini[45,0]
    v_R18_n_r = struct[0].y_ini[46,0]
    v_R18_n_i = struct[0].y_ini[47,0]
    v_R2_a_r = struct[0].y_ini[48,0]
    v_R2_a_i = struct[0].y_ini[49,0]
    v_R2_b_r = struct[0].y_ini[50,0]
    v_R2_b_i = struct[0].y_ini[51,0]
    v_R2_c_r = struct[0].y_ini[52,0]
    v_R2_c_i = struct[0].y_ini[53,0]
    v_R2_n_r = struct[0].y_ini[54,0]
    v_R2_n_i = struct[0].y_ini[55,0]
    v_R3_a_r = struct[0].y_ini[56,0]
    v_R3_a_i = struct[0].y_ini[57,0]
    v_R3_b_r = struct[0].y_ini[58,0]
    v_R3_b_i = struct[0].y_ini[59,0]
    v_R3_c_r = struct[0].y_ini[60,0]
    v_R3_c_i = struct[0].y_ini[61,0]
    v_R3_n_r = struct[0].y_ini[62,0]
    v_R3_n_i = struct[0].y_ini[63,0]
    v_R4_a_r = struct[0].y_ini[64,0]
    v_R4_a_i = struct[0].y_ini[65,0]
    v_R4_b_r = struct[0].y_ini[66,0]
    v_R4_b_i = struct[0].y_ini[67,0]
    v_R4_c_r = struct[0].y_ini[68,0]
    v_R4_c_i = struct[0].y_ini[69,0]
    v_R4_n_r = struct[0].y_ini[70,0]
    v_R4_n_i = struct[0].y_ini[71,0]
    v_R5_a_r = struct[0].y_ini[72,0]
    v_R5_a_i = struct[0].y_ini[73,0]
    v_R5_b_r = struct[0].y_ini[74,0]
    v_R5_b_i = struct[0].y_ini[75,0]
    v_R5_c_r = struct[0].y_ini[76,0]
    v_R5_c_i = struct[0].y_ini[77,0]
    v_R5_n_r = struct[0].y_ini[78,0]
    v_R5_n_i = struct[0].y_ini[79,0]
    v_R6_a_r = struct[0].y_ini[80,0]
    v_R6_a_i = struct[0].y_ini[81,0]
    v_R6_b_r = struct[0].y_ini[82,0]
    v_R6_b_i = struct[0].y_ini[83,0]
    v_R6_c_r = struct[0].y_ini[84,0]
    v_R6_c_i = struct[0].y_ini[85,0]
    v_R6_n_r = struct[0].y_ini[86,0]
    v_R6_n_i = struct[0].y_ini[87,0]
    v_R7_a_r = struct[0].y_ini[88,0]
    v_R7_a_i = struct[0].y_ini[89,0]
    v_R7_b_r = struct[0].y_ini[90,0]
    v_R7_b_i = struct[0].y_ini[91,0]
    v_R7_c_r = struct[0].y_ini[92,0]
    v_R7_c_i = struct[0].y_ini[93,0]
    v_R7_n_r = struct[0].y_ini[94,0]
    v_R7_n_i = struct[0].y_ini[95,0]
    v_R8_a_r = struct[0].y_ini[96,0]
    v_R8_a_i = struct[0].y_ini[97,0]
    v_R8_b_r = struct[0].y_ini[98,0]
    v_R8_b_i = struct[0].y_ini[99,0]
    v_R8_c_r = struct[0].y_ini[100,0]
    v_R8_c_i = struct[0].y_ini[101,0]
    v_R8_n_r = struct[0].y_ini[102,0]
    v_R8_n_i = struct[0].y_ini[103,0]
    v_R9_a_r = struct[0].y_ini[104,0]
    v_R9_a_i = struct[0].y_ini[105,0]
    v_R9_b_r = struct[0].y_ini[106,0]
    v_R9_b_i = struct[0].y_ini[107,0]
    v_R9_c_r = struct[0].y_ini[108,0]
    v_R9_c_i = struct[0].y_ini[109,0]
    v_R9_n_r = struct[0].y_ini[110,0]
    v_R9_n_i = struct[0].y_ini[111,0]
    v_R10_a_r = struct[0].y_ini[112,0]
    v_R10_a_i = struct[0].y_ini[113,0]
    v_R10_b_r = struct[0].y_ini[114,0]
    v_R10_b_i = struct[0].y_ini[115,0]
    v_R10_c_r = struct[0].y_ini[116,0]
    v_R10_c_i = struct[0].y_ini[117,0]
    v_R10_n_r = struct[0].y_ini[118,0]
    v_R10_n_i = struct[0].y_ini[119,0]
    v_R12_a_r = struct[0].y_ini[120,0]
    v_R12_a_i = struct[0].y_ini[121,0]
    v_R12_b_r = struct[0].y_ini[122,0]
    v_R12_b_i = struct[0].y_ini[123,0]
    v_R12_c_r = struct[0].y_ini[124,0]
    v_R12_c_i = struct[0].y_ini[125,0]
    v_R12_n_r = struct[0].y_ini[126,0]
    v_R12_n_i = struct[0].y_ini[127,0]
    v_R13_a_r = struct[0].y_ini[128,0]
    v_R13_a_i = struct[0].y_ini[129,0]
    v_R13_b_r = struct[0].y_ini[130,0]
    v_R13_b_i = struct[0].y_ini[131,0]
    v_R13_c_r = struct[0].y_ini[132,0]
    v_R13_c_i = struct[0].y_ini[133,0]
    v_R13_n_r = struct[0].y_ini[134,0]
    v_R13_n_i = struct[0].y_ini[135,0]
    v_R14_a_r = struct[0].y_ini[136,0]
    v_R14_a_i = struct[0].y_ini[137,0]
    v_R14_b_r = struct[0].y_ini[138,0]
    v_R14_b_i = struct[0].y_ini[139,0]
    v_R14_c_r = struct[0].y_ini[140,0]
    v_R14_c_i = struct[0].y_ini[141,0]
    v_R14_n_r = struct[0].y_ini[142,0]
    v_R14_n_i = struct[0].y_ini[143,0]
    i_R1_a_r = struct[0].y_ini[144,0]
    i_R1_a_i = struct[0].y_ini[145,0]
    i_R1_b_r = struct[0].y_ini[146,0]
    i_R1_b_i = struct[0].y_ini[147,0]
    i_R1_c_r = struct[0].y_ini[148,0]
    i_R1_c_i = struct[0].y_ini[149,0]
    i_R1_n_r = struct[0].y_ini[150,0]
    i_R1_n_i = struct[0].y_ini[151,0]
    i_R11_a_r = struct[0].y_ini[152,0]
    i_R11_a_i = struct[0].y_ini[153,0]
    i_R11_b_r = struct[0].y_ini[154,0]
    i_R11_b_i = struct[0].y_ini[155,0]
    i_R11_c_r = struct[0].y_ini[156,0]
    i_R11_c_i = struct[0].y_ini[157,0]
    i_R11_n_r = struct[0].y_ini[158,0]
    i_R11_n_i = struct[0].y_ini[159,0]
    i_R15_a_r = struct[0].y_ini[160,0]
    i_R15_a_i = struct[0].y_ini[161,0]
    i_R15_b_r = struct[0].y_ini[162,0]
    i_R15_b_i = struct[0].y_ini[163,0]
    i_R15_c_r = struct[0].y_ini[164,0]
    i_R15_c_i = struct[0].y_ini[165,0]
    i_R15_n_r = struct[0].y_ini[166,0]
    i_R15_n_i = struct[0].y_ini[167,0]
    i_R16_a_r = struct[0].y_ini[168,0]
    i_R16_a_i = struct[0].y_ini[169,0]
    i_R16_b_r = struct[0].y_ini[170,0]
    i_R16_b_i = struct[0].y_ini[171,0]
    i_R16_c_r = struct[0].y_ini[172,0]
    i_R16_c_i = struct[0].y_ini[173,0]
    i_R16_n_r = struct[0].y_ini[174,0]
    i_R16_n_i = struct[0].y_ini[175,0]
    i_R17_a_r = struct[0].y_ini[176,0]
    i_R17_a_i = struct[0].y_ini[177,0]
    i_R17_b_r = struct[0].y_ini[178,0]
    i_R17_b_i = struct[0].y_ini[179,0]
    i_R17_c_r = struct[0].y_ini[180,0]
    i_R17_c_i = struct[0].y_ini[181,0]
    i_R17_n_r = struct[0].y_ini[182,0]
    i_R17_n_i = struct[0].y_ini[183,0]
    i_R18_a_r = struct[0].y_ini[184,0]
    i_R18_a_i = struct[0].y_ini[185,0]
    i_R18_b_r = struct[0].y_ini[186,0]
    i_R18_b_i = struct[0].y_ini[187,0]
    i_R18_c_r = struct[0].y_ini[188,0]
    i_R18_c_i = struct[0].y_ini[189,0]
    i_R18_n_r = struct[0].y_ini[190,0]
    i_R18_n_i = struct[0].y_ini[191,0]
    
    # Differential equations:
    if mode == 2:


        struct[0].f[0,0] = u_dummy - x_dummy
    
    # Algebraic equations:
    if mode == 3:


        struct[0].g[0,0] = i_R1_a_r + 0.849044513514155*v_R0_a_i + 0.212261128378539*v_R0_a_r - 0.849044513514155*v_R0_c_i - 0.212261128378539*v_R0_c_r - 127.595184532966*v_R1_a_i - 123.954122254357*v_R1_a_r + 10.2713736253513*v_R1_b_i + 39.6392229058202*v_R1_b_r + 23.284964480954*v_R1_c_i + 24.9575997948692*v_R1_c_r + 83.8007853900572*v_R1_n_i + 58.0215758469966*v_R1_n_r + 54.0657727682604*v_R2_a_i + 105.57176931318*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r - 23.284964480954*v_R2_c_i - 24.9575997948692*v_R2_c_r - 10.2713736253513*v_R2_n_i - 39.6392229058202*v_R2_n_r
        struct[0].g[1,0] = i_R1_a_i + 0.212261128378539*v_R0_a_i - 0.849044513514155*v_R0_a_r - 0.212261128378539*v_R0_c_i + 0.849044513514155*v_R0_c_r - 123.954122254357*v_R1_a_i + 127.595184532966*v_R1_a_r + 39.6392229058202*v_R1_b_i - 10.2713736253513*v_R1_b_r + 24.9575997948692*v_R1_c_i - 23.284964480954*v_R1_c_r + 58.0215758469966*v_R1_n_i - 83.8007853900572*v_R1_n_r + 105.57176931318*v_R2_a_i - 54.0657727682604*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r - 24.9575997948692*v_R2_c_i + 23.284964480954*v_R2_c_r - 39.6392229058202*v_R2_n_i + 10.2713736253513*v_R2_n_r
        struct[0].g[2,0] = i_R1_b_r - 0.849044513514155*v_R0_a_i - 0.212261128378539*v_R0_a_r + 0.849044513514155*v_R0_b_i + 0.212261128378539*v_R0_b_r + 10.2713736253513*v_R1_a_i + 39.6392229058202*v_R1_a_r - 127.595184532966*v_R1_b_i - 123.954122254357*v_R1_b_r + 10.2713736253513*v_R1_c_i + 39.6392229058201*v_R1_c_r + 96.8143762456598*v_R1_n_i + 43.3399527360457*v_R1_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r + 54.0657727682604*v_R2_b_i + 105.57176931318*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058201*v_R2_c_r - 23.284964480954*v_R2_n_i - 24.9575997948692*v_R2_n_r
        struct[0].g[3,0] = i_R1_b_i - 0.212261128378539*v_R0_a_i + 0.849044513514155*v_R0_a_r + 0.212261128378539*v_R0_b_i - 0.849044513514155*v_R0_b_r + 39.6392229058202*v_R1_a_i - 10.2713736253513*v_R1_a_r - 123.954122254357*v_R1_b_i + 127.595184532966*v_R1_b_r + 39.6392229058201*v_R1_c_i - 10.2713736253513*v_R1_c_r + 43.3399527360457*v_R1_n_i - 96.8143762456598*v_R1_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r + 105.57176931318*v_R2_b_i - 54.0657727682604*v_R2_b_r - 39.6392229058201*v_R2_c_i + 10.2713736253513*v_R2_c_r - 24.9575997948692*v_R2_n_i + 23.284964480954*v_R2_n_r
        struct[0].g[4,0] = i_R1_c_r - 0.849044513514155*v_R0_b_i - 0.212261128378539*v_R0_b_r + 0.849044513514155*v_R0_c_i + 0.212261128378539*v_R0_c_r + 23.284964480954*v_R1_a_i + 24.9575997948692*v_R1_a_r + 10.2713736253513*v_R1_b_i + 39.6392229058202*v_R1_b_r - 127.595184532966*v_R1_c_i - 123.954122254357*v_R1_c_r + 83.8007853900572*v_R1_n_i + 58.0215758469966*v_R1_n_r - 23.284964480954*v_R2_a_i - 24.9575997948692*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r + 54.0657727682604*v_R2_c_i + 105.57176931318*v_R2_c_r - 10.2713736253514*v_R2_n_i - 39.6392229058201*v_R2_n_r
        struct[0].g[5,0] = i_R1_c_i - 0.212261128378539*v_R0_b_i + 0.849044513514155*v_R0_b_r + 0.212261128378539*v_R0_c_i - 0.849044513514155*v_R0_c_r + 24.9575997948692*v_R1_a_i - 23.284964480954*v_R1_a_r + 39.6392229058202*v_R1_b_i - 10.2713736253513*v_R1_b_r - 123.954122254357*v_R1_c_i + 127.595184532966*v_R1_c_r + 58.0215758469966*v_R1_n_i - 83.8007853900572*v_R1_n_r - 24.9575997948692*v_R2_a_i + 23.284964480954*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r + 105.57176931318*v_R2_c_i - 54.0657727682604*v_R2_c_r - 39.6392229058201*v_R2_n_i + 10.2713736253514*v_R2_n_r
        struct[0].g[6,0] = i_R1_n_r + 83.8007853900572*v_R1_a_i + 58.0215758469966*v_R1_a_r + 96.8143762456598*v_R1_b_i + 43.3399527360457*v_R1_b_r + 83.8007853900572*v_R1_c_i + 58.0215758469966*v_R1_c_r - 274.654008062378*v_R1_n_i - 161.052161470043*v_R1_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r - 23.284964480954*v_R2_b_i - 24.9575997948692*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058202*v_R2_c_r + 54.0657727682604*v_R2_n_i + 105.57176931318*v_R2_n_r
        struct[0].g[7,0] = i_R1_n_i + 58.0215758469966*v_R1_a_i - 83.8007853900572*v_R1_a_r + 43.3399527360457*v_R1_b_i - 96.8143762456598*v_R1_b_r + 58.0215758469966*v_R1_c_i - 83.8007853900572*v_R1_c_r - 161.052161470043*v_R1_n_i + 274.654008062378*v_R1_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r - 24.9575997948692*v_R2_b_i + 23.284964480954*v_R2_b_r - 39.6392229058202*v_R2_c_i + 10.2713736253513*v_R2_c_r + 105.57176931318*v_R2_n_i - 54.0657727682604*v_R2_n_r
        struct[0].g[8,0] = i_R11_a_r - 4.84676915585801*v_R11_a_i - 26.5300693004806*v_R11_a_r - 1.58482814504483*v_R11_b_i + 7.8946133751477*v_R11_b_r - 0.67991845328531*v_R11_c_i + 7.72144347466701*v_R11_c_r - 1.58482814504483*v_R11_n_i + 7.89461337514771*v_R11_n_r + 4.84676915585801*v_R3_a_i + 26.5300693004806*v_R3_a_r + 1.58482814504483*v_R3_b_i - 7.8946133751477*v_R3_b_r + 0.67991845328531*v_R3_c_i - 7.72144347466701*v_R3_c_r + 1.58482814504483*v_R3_n_i - 7.89461337514771*v_R3_n_r
        struct[0].g[9,0] = i_R11_a_i - 26.5300693004806*v_R11_a_i + 4.84676915585801*v_R11_a_r + 7.8946133751477*v_R11_b_i + 1.58482814504483*v_R11_b_r + 7.72144347466701*v_R11_c_i + 0.67991845328531*v_R11_c_r + 7.89461337514771*v_R11_n_i + 1.58482814504483*v_R11_n_r + 26.5300693004806*v_R3_a_i - 4.84676915585801*v_R3_a_r - 7.8946133751477*v_R3_b_i - 1.58482814504483*v_R3_b_r - 7.72144347466701*v_R3_c_i - 0.67991845328531*v_R3_c_r - 7.89461337514771*v_R3_n_i - 1.58482814504483*v_R3_n_r
        struct[0].g[10,0] = i_R11_b_r - 1.58482814504483*v_R11_a_i + 7.89461337514771*v_R11_a_r - 4.84676915585801*v_R11_b_i - 26.5300693004806*v_R11_b_r - 1.58482814504483*v_R11_c_i + 7.89461337514771*v_R11_c_r - 0.679918453285308*v_R11_n_i + 7.72144347466701*v_R11_n_r + 1.58482814504483*v_R3_a_i - 7.89461337514771*v_R3_a_r + 4.84676915585801*v_R3_b_i + 26.5300693004806*v_R3_b_r + 1.58482814504483*v_R3_c_i - 7.89461337514771*v_R3_c_r + 0.679918453285308*v_R3_n_i - 7.72144347466701*v_R3_n_r
        struct[0].g[11,0] = i_R11_b_i + 7.89461337514771*v_R11_a_i + 1.58482814504483*v_R11_a_r - 26.5300693004806*v_R11_b_i + 4.84676915585801*v_R11_b_r + 7.89461337514771*v_R11_c_i + 1.58482814504483*v_R11_c_r + 7.72144347466701*v_R11_n_i + 0.679918453285308*v_R11_n_r - 7.89461337514771*v_R3_a_i - 1.58482814504483*v_R3_a_r + 26.5300693004806*v_R3_b_i - 4.84676915585801*v_R3_b_r - 7.89461337514771*v_R3_c_i - 1.58482814504483*v_R3_c_r - 7.72144347466701*v_R3_n_i - 0.679918453285308*v_R3_n_r
        struct[0].g[12,0] = i_R11_c_r - 0.67991845328531*v_R11_a_i + 7.72144347466701*v_R11_a_r - 1.58482814504483*v_R11_b_i + 7.8946133751477*v_R11_b_r - 4.84676915585801*v_R11_c_i - 26.5300693004806*v_R11_c_r - 1.58482814504483*v_R11_n_i + 7.89461337514771*v_R11_n_r + 0.67991845328531*v_R3_a_i - 7.72144347466701*v_R3_a_r + 1.58482814504483*v_R3_b_i - 7.8946133751477*v_R3_b_r + 4.84676915585801*v_R3_c_i + 26.5300693004806*v_R3_c_r + 1.58482814504483*v_R3_n_i - 7.89461337514771*v_R3_n_r
        struct[0].g[13,0] = i_R11_c_i + 7.72144347466701*v_R11_a_i + 0.67991845328531*v_R11_a_r + 7.8946133751477*v_R11_b_i + 1.58482814504483*v_R11_b_r - 26.5300693004806*v_R11_c_i + 4.84676915585801*v_R11_c_r + 7.89461337514771*v_R11_n_i + 1.58482814504483*v_R11_n_r - 7.72144347466701*v_R3_a_i - 0.67991845328531*v_R3_a_r - 7.8946133751477*v_R3_b_i - 1.58482814504483*v_R3_b_r + 26.5300693004806*v_R3_c_i - 4.84676915585801*v_R3_c_r - 7.89461337514771*v_R3_n_i - 1.58482814504483*v_R3_n_r
        struct[0].g[14,0] = i_R11_n_r - 1.58482814504483*v_R11_a_i + 7.89461337514771*v_R11_a_r - 0.679918453285308*v_R11_b_i + 7.72144347466701*v_R11_b_r - 1.58482814504483*v_R11_c_i + 7.89461337514771*v_R11_c_r - 4.84676915585801*v_R11_n_i - 26.5550693004806*v_R11_n_r + 1.58482814504483*v_R3_a_i - 7.89461337514771*v_R3_a_r + 0.679918453285308*v_R3_b_i - 7.72144347466701*v_R3_b_r + 1.58482814504483*v_R3_c_i - 7.89461337514771*v_R3_c_r + 4.84676915585801*v_R3_n_i + 26.5300693004806*v_R3_n_r
        struct[0].g[15,0] = i_R11_n_i + 7.89461337514771*v_R11_a_i + 1.58482814504483*v_R11_a_r + 7.72144347466701*v_R11_b_i + 0.679918453285308*v_R11_b_r + 7.89461337514771*v_R11_c_i + 1.58482814504483*v_R11_c_r - 26.5550693004806*v_R11_n_i + 4.84676915585801*v_R11_n_r - 7.89461337514771*v_R3_a_i - 1.58482814504483*v_R3_a_r - 7.72144347466701*v_R3_b_i - 0.679918453285308*v_R3_b_r - 7.89461337514771*v_R3_c_i - 1.58482814504483*v_R3_c_r + 26.5300693004806*v_R3_n_i - 4.84676915585801*v_R3_n_r
        struct[0].g[16,0] = i_R15_a_r + 4.84676915585801*v_R14_a_i + 26.5300693004806*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 0.67991845328531*v_R14_c_i - 7.72144347466701*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r - 4.84676915585801*v_R15_a_i - 26.5300693004806*v_R15_a_r - 1.58482814504483*v_R15_b_i + 7.8946133751477*v_R15_b_r - 0.67991845328531*v_R15_c_i + 7.72144347466701*v_R15_c_r - 1.58482814504483*v_R15_n_i + 7.89461337514771*v_R15_n_r
        struct[0].g[17,0] = i_R15_a_i + 26.5300693004806*v_R14_a_i - 4.84676915585801*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r - 7.72144347466701*v_R14_c_i - 0.67991845328531*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r - 26.5300693004806*v_R15_a_i + 4.84676915585801*v_R15_a_r + 7.8946133751477*v_R15_b_i + 1.58482814504483*v_R15_b_r + 7.72144347466701*v_R15_c_i + 0.67991845328531*v_R15_c_r + 7.89461337514771*v_R15_n_i + 1.58482814504483*v_R15_n_r
        struct[0].g[18,0] = i_R15_b_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 4.84676915585801*v_R14_b_i + 26.5300693004806*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 0.679918453285308*v_R14_n_i - 7.72144347466701*v_R14_n_r - 1.58482814504483*v_R15_a_i + 7.89461337514771*v_R15_a_r - 4.84676915585801*v_R15_b_i - 26.5300693004806*v_R15_b_r - 1.58482814504483*v_R15_c_i + 7.89461337514771*v_R15_c_r - 0.679918453285308*v_R15_n_i + 7.72144347466701*v_R15_n_r
        struct[0].g[19,0] = i_R15_b_i - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r + 26.5300693004806*v_R14_b_i - 4.84676915585801*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r - 7.72144347466701*v_R14_n_i - 0.679918453285308*v_R14_n_r + 7.89461337514771*v_R15_a_i + 1.58482814504483*v_R15_a_r - 26.5300693004806*v_R15_b_i + 4.84676915585801*v_R15_b_r + 7.89461337514771*v_R15_c_i + 1.58482814504483*v_R15_c_r + 7.72144347466701*v_R15_n_i + 0.679918453285308*v_R15_n_r
        struct[0].g[20,0] = i_R15_c_r + 0.67991845328531*v_R14_a_i - 7.72144347466701*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 4.84676915585801*v_R14_c_i + 26.5300693004806*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r - 0.67991845328531*v_R15_a_i + 7.72144347466701*v_R15_a_r - 1.58482814504483*v_R15_b_i + 7.8946133751477*v_R15_b_r - 4.84676915585801*v_R15_c_i - 26.5300693004806*v_R15_c_r - 1.58482814504483*v_R15_n_i + 7.89461337514771*v_R15_n_r
        struct[0].g[21,0] = i_R15_c_i - 7.72144347466701*v_R14_a_i - 0.67991845328531*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r + 26.5300693004806*v_R14_c_i - 4.84676915585801*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r + 7.72144347466701*v_R15_a_i + 0.67991845328531*v_R15_a_r + 7.8946133751477*v_R15_b_i + 1.58482814504483*v_R15_b_r - 26.5300693004806*v_R15_c_i + 4.84676915585801*v_R15_c_r + 7.89461337514771*v_R15_n_i + 1.58482814504483*v_R15_n_r
        struct[0].g[22,0] = i_R15_n_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 0.679918453285308*v_R14_b_i - 7.72144347466701*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 4.84676915585801*v_R14_n_i + 26.5300693004806*v_R14_n_r - 1.58482814504483*v_R15_a_i + 7.89461337514771*v_R15_a_r - 0.679918453285308*v_R15_b_i + 7.72144347466701*v_R15_b_r - 1.58482814504483*v_R15_c_i + 7.89461337514771*v_R15_c_r - 4.84676915585801*v_R15_n_i - 26.5550693004806*v_R15_n_r
        struct[0].g[23,0] = i_R15_n_i - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r - 7.72144347466701*v_R14_b_i - 0.679918453285308*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r + 26.5300693004806*v_R14_n_i - 4.84676915585801*v_R14_n_r + 7.89461337514771*v_R15_a_i + 1.58482814504483*v_R15_a_r + 7.72144347466701*v_R15_b_i + 0.679918453285308*v_R15_b_r + 7.89461337514771*v_R15_c_i + 1.58482814504483*v_R15_c_r - 26.5550693004806*v_R15_n_i + 4.84676915585801*v_R15_n_r
        struct[0].g[24,0] = i_R16_a_r - 5.65456401516768*v_R16_a_i - 30.9517475172273*v_R16_a_r - 1.84896616921897*v_R16_b_i + 9.21038227100566*v_R16_b_r - 0.793238195499529*v_R16_c_i + 9.00835072044485*v_R16_c_r - 1.84896616921897*v_R16_n_i + 9.21038227100566*v_R16_n_r + 5.65456401516768*v_R6_a_i + 30.9517475172273*v_R6_a_r + 1.84896616921897*v_R6_b_i - 9.21038227100566*v_R6_b_r + 0.793238195499529*v_R6_c_i - 9.00835072044485*v_R6_c_r + 1.84896616921897*v_R6_n_i - 9.21038227100566*v_R6_n_r
        struct[0].g[25,0] = i_R16_a_i - 30.9517475172273*v_R16_a_i + 5.65456401516768*v_R16_a_r + 9.21038227100566*v_R16_b_i + 1.84896616921897*v_R16_b_r + 9.00835072044485*v_R16_c_i + 0.793238195499529*v_R16_c_r + 9.21038227100566*v_R16_n_i + 1.84896616921897*v_R16_n_r + 30.9517475172273*v_R6_a_i - 5.65456401516768*v_R6_a_r - 9.21038227100566*v_R6_b_i - 1.84896616921897*v_R6_b_r - 9.00835072044485*v_R6_c_i - 0.793238195499529*v_R6_c_r - 9.21038227100566*v_R6_n_i - 1.84896616921897*v_R6_n_r
        struct[0].g[26,0] = i_R16_b_r - 1.84896616921897*v_R16_a_i + 9.21038227100566*v_R16_a_r - 5.65456401516768*v_R16_b_i - 30.9517475172273*v_R16_b_r - 1.84896616921897*v_R16_c_i + 9.21038227100566*v_R16_c_r - 0.793238195499528*v_R16_n_i + 9.00835072044485*v_R16_n_r + 1.84896616921897*v_R6_a_i - 9.21038227100566*v_R6_a_r + 5.65456401516768*v_R6_b_i + 30.9517475172273*v_R6_b_r + 1.84896616921897*v_R6_c_i - 9.21038227100566*v_R6_c_r + 0.793238195499528*v_R6_n_i - 9.00835072044485*v_R6_n_r
        struct[0].g[27,0] = i_R16_b_i + 9.21038227100566*v_R16_a_i + 1.84896616921897*v_R16_a_r - 30.9517475172273*v_R16_b_i + 5.65456401516768*v_R16_b_r + 9.21038227100566*v_R16_c_i + 1.84896616921897*v_R16_c_r + 9.00835072044485*v_R16_n_i + 0.793238195499528*v_R16_n_r - 9.21038227100566*v_R6_a_i - 1.84896616921897*v_R6_a_r + 30.9517475172273*v_R6_b_i - 5.65456401516768*v_R6_b_r - 9.21038227100566*v_R6_c_i - 1.84896616921897*v_R6_c_r - 9.00835072044485*v_R6_n_i - 0.793238195499528*v_R6_n_r
        struct[0].g[28,0] = i_R16_c_r - 0.793238195499527*v_R16_a_i + 9.00835072044484*v_R16_a_r - 1.84896616921897*v_R16_b_i + 9.21038227100566*v_R16_b_r - 5.65456401516768*v_R16_c_i - 30.9517475172273*v_R16_c_r - 1.84896616921897*v_R16_n_i + 9.21038227100566*v_R16_n_r + 0.793238195499527*v_R6_a_i - 9.00835072044484*v_R6_a_r + 1.84896616921897*v_R6_b_i - 9.21038227100566*v_R6_b_r + 5.65456401516768*v_R6_c_i + 30.9517475172273*v_R6_c_r + 1.84896616921897*v_R6_n_i - 9.21038227100566*v_R6_n_r
        struct[0].g[29,0] = i_R16_c_i + 9.00835072044484*v_R16_a_i + 0.793238195499527*v_R16_a_r + 9.21038227100566*v_R16_b_i + 1.84896616921897*v_R16_b_r - 30.9517475172273*v_R16_c_i + 5.65456401516768*v_R16_c_r + 9.21038227100566*v_R16_n_i + 1.84896616921897*v_R16_n_r - 9.00835072044484*v_R6_a_i - 0.793238195499527*v_R6_a_r - 9.21038227100566*v_R6_b_i - 1.84896616921897*v_R6_b_r + 30.9517475172273*v_R6_c_i - 5.65456401516768*v_R6_c_r - 9.21038227100566*v_R6_n_i - 1.84896616921897*v_R6_n_r
        struct[0].g[30,0] = i_R16_n_r - 1.84896616921897*v_R16_a_i + 9.21038227100566*v_R16_a_r - 0.793238195499527*v_R16_b_i + 9.00835072044485*v_R16_b_r - 1.84896616921897*v_R16_c_i + 9.21038227100566*v_R16_c_r - 5.65456401516768*v_R16_n_i - 30.9767475172273*v_R16_n_r + 1.84896616921897*v_R6_a_i - 9.21038227100566*v_R6_a_r + 0.793238195499527*v_R6_b_i - 9.00835072044485*v_R6_b_r + 1.84896616921897*v_R6_c_i - 9.21038227100566*v_R6_c_r + 5.65456401516768*v_R6_n_i + 30.9517475172273*v_R6_n_r
        struct[0].g[31,0] = i_R16_n_i + 9.21038227100566*v_R16_a_i + 1.84896616921897*v_R16_a_r + 9.00835072044485*v_R16_b_i + 0.793238195499527*v_R16_b_r + 9.21038227100566*v_R16_c_i + 1.84896616921897*v_R16_c_r - 30.9767475172273*v_R16_n_i + 5.65456401516768*v_R16_n_r - 9.21038227100566*v_R6_a_i - 1.84896616921897*v_R6_a_r - 9.00835072044485*v_R6_b_i - 0.793238195499527*v_R6_b_r - 9.21038227100566*v_R6_c_i - 1.84896616921897*v_R6_c_r + 30.9517475172273*v_R6_n_i - 5.65456401516768*v_R6_n_r
        struct[0].g[32,0] = i_R17_a_r - 5.65456401516768*v_R17_a_i - 30.9517475172273*v_R17_a_r - 1.84896616921897*v_R17_b_i + 9.21038227100566*v_R17_b_r - 0.793238195499529*v_R17_c_i + 9.00835072044485*v_R17_c_r - 1.84896616921897*v_R17_n_i + 9.21038227100566*v_R17_n_r + 5.65456401516768*v_R9_a_i + 30.9517475172273*v_R9_a_r + 1.84896616921897*v_R9_b_i - 9.21038227100566*v_R9_b_r + 0.793238195499529*v_R9_c_i - 9.00835072044485*v_R9_c_r + 1.84896616921897*v_R9_n_i - 9.21038227100566*v_R9_n_r
        struct[0].g[33,0] = i_R17_a_i - 30.9517475172273*v_R17_a_i + 5.65456401516768*v_R17_a_r + 9.21038227100566*v_R17_b_i + 1.84896616921897*v_R17_b_r + 9.00835072044485*v_R17_c_i + 0.793238195499529*v_R17_c_r + 9.21038227100566*v_R17_n_i + 1.84896616921897*v_R17_n_r + 30.9517475172273*v_R9_a_i - 5.65456401516768*v_R9_a_r - 9.21038227100566*v_R9_b_i - 1.84896616921897*v_R9_b_r - 9.00835072044485*v_R9_c_i - 0.793238195499529*v_R9_c_r - 9.21038227100566*v_R9_n_i - 1.84896616921897*v_R9_n_r
        struct[0].g[34,0] = i_R17_b_r - 1.84896616921897*v_R17_a_i + 9.21038227100566*v_R17_a_r - 5.65456401516768*v_R17_b_i - 30.9517475172273*v_R17_b_r - 1.84896616921897*v_R17_c_i + 9.21038227100566*v_R17_c_r - 0.793238195499528*v_R17_n_i + 9.00835072044485*v_R17_n_r + 1.84896616921897*v_R9_a_i - 9.21038227100566*v_R9_a_r + 5.65456401516768*v_R9_b_i + 30.9517475172273*v_R9_b_r + 1.84896616921897*v_R9_c_i - 9.21038227100566*v_R9_c_r + 0.793238195499528*v_R9_n_i - 9.00835072044485*v_R9_n_r
        struct[0].g[35,0] = i_R17_b_i + 9.21038227100566*v_R17_a_i + 1.84896616921897*v_R17_a_r - 30.9517475172273*v_R17_b_i + 5.65456401516768*v_R17_b_r + 9.21038227100566*v_R17_c_i + 1.84896616921897*v_R17_c_r + 9.00835072044485*v_R17_n_i + 0.793238195499528*v_R17_n_r - 9.21038227100566*v_R9_a_i - 1.84896616921897*v_R9_a_r + 30.9517475172273*v_R9_b_i - 5.65456401516768*v_R9_b_r - 9.21038227100566*v_R9_c_i - 1.84896616921897*v_R9_c_r - 9.00835072044485*v_R9_n_i - 0.793238195499528*v_R9_n_r
        struct[0].g[36,0] = i_R17_c_r - 0.793238195499527*v_R17_a_i + 9.00835072044484*v_R17_a_r - 1.84896616921897*v_R17_b_i + 9.21038227100566*v_R17_b_r - 5.65456401516768*v_R17_c_i - 30.9517475172273*v_R17_c_r - 1.84896616921897*v_R17_n_i + 9.21038227100566*v_R17_n_r + 0.793238195499527*v_R9_a_i - 9.00835072044484*v_R9_a_r + 1.84896616921897*v_R9_b_i - 9.21038227100566*v_R9_b_r + 5.65456401516768*v_R9_c_i + 30.9517475172273*v_R9_c_r + 1.84896616921897*v_R9_n_i - 9.21038227100566*v_R9_n_r
        struct[0].g[37,0] = i_R17_c_i + 9.00835072044484*v_R17_a_i + 0.793238195499527*v_R17_a_r + 9.21038227100566*v_R17_b_i + 1.84896616921897*v_R17_b_r - 30.9517475172273*v_R17_c_i + 5.65456401516768*v_R17_c_r + 9.21038227100566*v_R17_n_i + 1.84896616921897*v_R17_n_r - 9.00835072044484*v_R9_a_i - 0.793238195499527*v_R9_a_r - 9.21038227100566*v_R9_b_i - 1.84896616921897*v_R9_b_r + 30.9517475172273*v_R9_c_i - 5.65456401516768*v_R9_c_r - 9.21038227100566*v_R9_n_i - 1.84896616921897*v_R9_n_r
        struct[0].g[38,0] = i_R17_n_r - 1.84896616921897*v_R17_a_i + 9.21038227100566*v_R17_a_r - 0.793238195499527*v_R17_b_i + 9.00835072044485*v_R17_b_r - 1.84896616921897*v_R17_c_i + 9.21038227100566*v_R17_c_r - 5.65456401516768*v_R17_n_i - 30.9767475172273*v_R17_n_r + 1.84896616921897*v_R9_a_i - 9.21038227100566*v_R9_a_r + 0.793238195499527*v_R9_b_i - 9.00835072044485*v_R9_b_r + 1.84896616921897*v_R9_c_i - 9.21038227100566*v_R9_c_r + 5.65456401516768*v_R9_n_i + 30.9517475172273*v_R9_n_r
        struct[0].g[39,0] = i_R17_n_i + 9.21038227100566*v_R17_a_i + 1.84896616921897*v_R17_a_r + 9.00835072044485*v_R17_b_i + 0.793238195499527*v_R17_b_r + 9.21038227100566*v_R17_c_i + 1.84896616921897*v_R17_c_r - 30.9767475172273*v_R17_n_i + 5.65456401516768*v_R17_n_r - 9.21038227100566*v_R9_a_i - 1.84896616921897*v_R9_a_r - 9.00835072044485*v_R9_b_i - 0.793238195499527*v_R9_b_r - 9.21038227100566*v_R9_c_i - 1.84896616921897*v_R9_c_r + 30.9517475172273*v_R9_n_i - 5.65456401516768*v_R9_n_r
        struct[0].g[40,0] = i_R18_a_r + 5.65456401516768*v_R10_a_i + 30.9517475172273*v_R10_a_r + 1.84896616921897*v_R10_b_i - 9.21038227100566*v_R10_b_r + 0.793238195499529*v_R10_c_i - 9.00835072044485*v_R10_c_r + 1.84896616921897*v_R10_n_i - 9.21038227100566*v_R10_n_r - 5.65456401516768*v_R18_a_i - 30.9517475172273*v_R18_a_r - 1.84896616921897*v_R18_b_i + 9.21038227100566*v_R18_b_r - 0.793238195499529*v_R18_c_i + 9.00835072044485*v_R18_c_r - 1.84896616921897*v_R18_n_i + 9.21038227100566*v_R18_n_r
        struct[0].g[41,0] = i_R18_a_i + 30.9517475172273*v_R10_a_i - 5.65456401516768*v_R10_a_r - 9.21038227100566*v_R10_b_i - 1.84896616921897*v_R10_b_r - 9.00835072044485*v_R10_c_i - 0.793238195499529*v_R10_c_r - 9.21038227100566*v_R10_n_i - 1.84896616921897*v_R10_n_r - 30.9517475172273*v_R18_a_i + 5.65456401516768*v_R18_a_r + 9.21038227100566*v_R18_b_i + 1.84896616921897*v_R18_b_r + 9.00835072044485*v_R18_c_i + 0.793238195499529*v_R18_c_r + 9.21038227100566*v_R18_n_i + 1.84896616921897*v_R18_n_r
        struct[0].g[42,0] = i_R18_b_r + 1.84896616921897*v_R10_a_i - 9.21038227100566*v_R10_a_r + 5.65456401516768*v_R10_b_i + 30.9517475172273*v_R10_b_r + 1.84896616921897*v_R10_c_i - 9.21038227100566*v_R10_c_r + 0.793238195499528*v_R10_n_i - 9.00835072044485*v_R10_n_r - 1.84896616921897*v_R18_a_i + 9.21038227100566*v_R18_a_r - 5.65456401516768*v_R18_b_i - 30.9517475172273*v_R18_b_r - 1.84896616921897*v_R18_c_i + 9.21038227100566*v_R18_c_r - 0.793238195499528*v_R18_n_i + 9.00835072044485*v_R18_n_r
        struct[0].g[43,0] = i_R18_b_i - 9.21038227100566*v_R10_a_i - 1.84896616921897*v_R10_a_r + 30.9517475172273*v_R10_b_i - 5.65456401516768*v_R10_b_r - 9.21038227100566*v_R10_c_i - 1.84896616921897*v_R10_c_r - 9.00835072044485*v_R10_n_i - 0.793238195499528*v_R10_n_r + 9.21038227100566*v_R18_a_i + 1.84896616921897*v_R18_a_r - 30.9517475172273*v_R18_b_i + 5.65456401516768*v_R18_b_r + 9.21038227100566*v_R18_c_i + 1.84896616921897*v_R18_c_r + 9.00835072044485*v_R18_n_i + 0.793238195499528*v_R18_n_r
        struct[0].g[44,0] = i_R18_c_r + 0.793238195499527*v_R10_a_i - 9.00835072044484*v_R10_a_r + 1.84896616921897*v_R10_b_i - 9.21038227100566*v_R10_b_r + 5.65456401516768*v_R10_c_i + 30.9517475172273*v_R10_c_r + 1.84896616921897*v_R10_n_i - 9.21038227100566*v_R10_n_r - 0.793238195499527*v_R18_a_i + 9.00835072044484*v_R18_a_r - 1.84896616921897*v_R18_b_i + 9.21038227100566*v_R18_b_r - 5.65456401516768*v_R18_c_i - 30.9517475172273*v_R18_c_r - 1.84896616921897*v_R18_n_i + 9.21038227100566*v_R18_n_r
        struct[0].g[45,0] = i_R18_c_i - 9.00835072044484*v_R10_a_i - 0.793238195499527*v_R10_a_r - 9.21038227100566*v_R10_b_i - 1.84896616921897*v_R10_b_r + 30.9517475172273*v_R10_c_i - 5.65456401516768*v_R10_c_r - 9.21038227100566*v_R10_n_i - 1.84896616921897*v_R10_n_r + 9.00835072044484*v_R18_a_i + 0.793238195499527*v_R18_a_r + 9.21038227100566*v_R18_b_i + 1.84896616921897*v_R18_b_r - 30.9517475172273*v_R18_c_i + 5.65456401516768*v_R18_c_r + 9.21038227100566*v_R18_n_i + 1.84896616921897*v_R18_n_r
        struct[0].g[46,0] = i_R18_n_r + 1.84896616921897*v_R10_a_i - 9.21038227100566*v_R10_a_r + 0.793238195499527*v_R10_b_i - 9.00835072044485*v_R10_b_r + 1.84896616921897*v_R10_c_i - 9.21038227100566*v_R10_c_r + 5.65456401516768*v_R10_n_i + 30.9517475172273*v_R10_n_r - 1.84896616921897*v_R18_a_i + 9.21038227100566*v_R18_a_r - 0.793238195499527*v_R18_b_i + 9.00835072044485*v_R18_b_r - 1.84896616921897*v_R18_c_i + 9.21038227100566*v_R18_c_r - 5.65456401516768*v_R18_n_i - 30.9767475172273*v_R18_n_r
        struct[0].g[47,0] = i_R18_n_i - 9.21038227100566*v_R10_a_i - 1.84896616921897*v_R10_a_r - 9.00835072044485*v_R10_b_i - 0.793238195499527*v_R10_b_r - 9.21038227100566*v_R10_c_i - 1.84896616921897*v_R10_c_r + 30.9517475172273*v_R10_n_i - 5.65456401516768*v_R10_n_r + 9.21038227100566*v_R18_a_i + 1.84896616921897*v_R18_a_r + 9.00835072044485*v_R18_b_i + 0.793238195499527*v_R18_b_r + 9.21038227100566*v_R18_c_i + 1.84896616921897*v_R18_c_r - 30.9767475172273*v_R18_n_i + 5.65456401516768*v_R18_n_r
        struct[0].g[48,0] = i_R2_a_r + 54.0657727682604*v_R1_a_i + 105.57176931318*v_R1_a_r - 10.2713736253513*v_R1_b_i - 39.6392229058202*v_R1_b_r - 23.284964480954*v_R1_c_i - 24.9575997948692*v_R1_c_r - 10.2713736253513*v_R1_n_i - 39.6392229058202*v_R1_n_r - 108.131545536521*v_R2_a_i - 211.14353862636*v_R2_a_r + 20.5427472507027*v_R2_b_i + 79.2784458116403*v_R2_b_r + 46.569928961908*v_R2_c_i + 49.9151995897384*v_R2_c_r + 20.5427472507027*v_R2_n_i + 79.2784458116403*v_R2_n_r + 54.0657727682604*v_R3_a_i + 105.57176931318*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r - 23.284964480954*v_R3_c_i - 24.9575997948692*v_R3_c_r - 10.2713736253513*v_R3_n_i - 39.6392229058202*v_R3_n_r
        struct[0].g[49,0] = i_R2_a_i + 105.57176931318*v_R1_a_i - 54.0657727682604*v_R1_a_r - 39.6392229058202*v_R1_b_i + 10.2713736253513*v_R1_b_r - 24.9575997948692*v_R1_c_i + 23.284964480954*v_R1_c_r - 39.6392229058202*v_R1_n_i + 10.2713736253513*v_R1_n_r - 211.14353862636*v_R2_a_i + 108.131545536521*v_R2_a_r + 79.2784458116403*v_R2_b_i - 20.5427472507027*v_R2_b_r + 49.9151995897384*v_R2_c_i - 46.569928961908*v_R2_c_r + 79.2784458116403*v_R2_n_i - 20.5427472507027*v_R2_n_r + 105.57176931318*v_R3_a_i - 54.0657727682604*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r - 24.9575997948692*v_R3_c_i + 23.284964480954*v_R3_c_r - 39.6392229058202*v_R3_n_i + 10.2713736253513*v_R3_n_r
        struct[0].g[50,0] = i_R2_b_r - 10.2713736253513*v_R1_a_i - 39.6392229058202*v_R1_a_r + 54.0657727682604*v_R1_b_i + 105.57176931318*v_R1_b_r - 10.2713736253513*v_R1_c_i - 39.6392229058201*v_R1_c_r - 23.284964480954*v_R1_n_i - 24.9575997948692*v_R1_n_r + 20.5427472507027*v_R2_a_i + 79.2784458116403*v_R2_a_r - 108.131545536521*v_R2_b_i - 211.14353862636*v_R2_b_r + 20.5427472507027*v_R2_c_i + 79.2784458116403*v_R2_c_r + 46.569928961908*v_R2_n_i + 49.9151995897384*v_R2_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r + 54.0657727682604*v_R3_b_i + 105.57176931318*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058201*v_R3_c_r - 23.284964480954*v_R3_n_i - 24.9575997948692*v_R3_n_r
        struct[0].g[51,0] = i_R2_b_i - 39.6392229058202*v_R1_a_i + 10.2713736253513*v_R1_a_r + 105.57176931318*v_R1_b_i - 54.0657727682604*v_R1_b_r - 39.6392229058201*v_R1_c_i + 10.2713736253513*v_R1_c_r - 24.9575997948692*v_R1_n_i + 23.284964480954*v_R1_n_r + 79.2784458116403*v_R2_a_i - 20.5427472507027*v_R2_a_r - 211.14353862636*v_R2_b_i + 108.131545536521*v_R2_b_r + 79.2784458116403*v_R2_c_i - 20.5427472507027*v_R2_c_r + 49.9151995897384*v_R2_n_i - 46.569928961908*v_R2_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r + 105.57176931318*v_R3_b_i - 54.0657727682604*v_R3_b_r - 39.6392229058201*v_R3_c_i + 10.2713736253513*v_R3_c_r - 24.9575997948692*v_R3_n_i + 23.284964480954*v_R3_n_r
        struct[0].g[52,0] = i_R2_c_r - 23.284964480954*v_R1_a_i - 24.9575997948692*v_R1_a_r - 10.2713736253513*v_R1_b_i - 39.6392229058202*v_R1_b_r + 54.0657727682604*v_R1_c_i + 105.57176931318*v_R1_c_r - 10.2713736253514*v_R1_n_i - 39.6392229058201*v_R1_n_r + 46.569928961908*v_R2_a_i + 49.9151995897384*v_R2_a_r + 20.5427472507027*v_R2_b_i + 79.2784458116404*v_R2_b_r - 108.131545536521*v_R2_c_i - 211.14353862636*v_R2_c_r + 20.5427472507027*v_R2_n_i + 79.2784458116403*v_R2_n_r - 23.284964480954*v_R3_a_i - 24.9575997948692*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r + 54.0657727682604*v_R3_c_i + 105.57176931318*v_R3_c_r - 10.2713736253514*v_R3_n_i - 39.6392229058201*v_R3_n_r
        struct[0].g[53,0] = i_R2_c_i - 24.9575997948692*v_R1_a_i + 23.284964480954*v_R1_a_r - 39.6392229058202*v_R1_b_i + 10.2713736253513*v_R1_b_r + 105.57176931318*v_R1_c_i - 54.0657727682604*v_R1_c_r - 39.6392229058201*v_R1_n_i + 10.2713736253514*v_R1_n_r + 49.9151995897384*v_R2_a_i - 46.569928961908*v_R2_a_r + 79.2784458116404*v_R2_b_i - 20.5427472507027*v_R2_b_r - 211.14353862636*v_R2_c_i + 108.131545536521*v_R2_c_r + 79.2784458116403*v_R2_n_i - 20.5427472507027*v_R2_n_r - 24.9575997948692*v_R3_a_i + 23.284964480954*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r + 105.57176931318*v_R3_c_i - 54.0657727682604*v_R3_c_r - 39.6392229058201*v_R3_n_i + 10.2713736253514*v_R3_n_r
        struct[0].g[54,0] = i_R2_n_r - 10.2713736253513*v_R1_a_i - 39.6392229058202*v_R1_a_r - 23.284964480954*v_R1_b_i - 24.9575997948692*v_R1_b_r - 10.2713736253513*v_R1_c_i - 39.6392229058202*v_R1_c_r + 54.0657727682604*v_R1_n_i + 105.57176931318*v_R1_n_r + 20.5427472507027*v_R2_a_i + 79.2784458116403*v_R2_a_r + 46.569928961908*v_R2_b_i + 49.9151995897384*v_R2_b_r + 20.5427472507027*v_R2_c_i + 79.2784458116403*v_R2_c_r - 108.131545536521*v_R2_n_i - 211.16853862636*v_R2_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r - 23.284964480954*v_R3_b_i - 24.9575997948692*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058202*v_R3_c_r + 54.0657727682604*v_R3_n_i + 105.57176931318*v_R3_n_r
        struct[0].g[55,0] = i_R2_n_i - 39.6392229058202*v_R1_a_i + 10.2713736253513*v_R1_a_r - 24.9575997948692*v_R1_b_i + 23.284964480954*v_R1_b_r - 39.6392229058202*v_R1_c_i + 10.2713736253513*v_R1_c_r + 105.57176931318*v_R1_n_i - 54.0657727682604*v_R1_n_r + 79.2784458116403*v_R2_a_i - 20.5427472507027*v_R2_a_r + 49.9151995897384*v_R2_b_i - 46.569928961908*v_R2_b_r + 79.2784458116403*v_R2_c_i - 20.5427472507027*v_R2_c_r - 211.16853862636*v_R2_n_i + 108.131545536521*v_R2_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r - 24.9575997948692*v_R3_b_i + 23.284964480954*v_R3_b_r - 39.6392229058202*v_R3_c_i + 10.2713736253513*v_R3_c_r + 105.57176931318*v_R3_n_i - 54.0657727682604*v_R3_n_r
        struct[0].g[56,0] = i_R3_a_r + 4.84676915585801*v_R11_a_i + 26.5300693004806*v_R11_a_r + 1.58482814504483*v_R11_b_i - 7.8946133751477*v_R11_b_r + 0.67991845328531*v_R11_c_i - 7.72144347466701*v_R11_c_r + 1.58482814504483*v_R11_n_i - 7.89461337514771*v_R11_n_r + 54.0657727682604*v_R2_a_i + 105.57176931318*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r - 23.284964480954*v_R2_c_i - 24.9575997948692*v_R2_c_r - 10.2713736253513*v_R2_n_i - 39.6392229058202*v_R2_n_r - 112.978314692379*v_R3_a_i - 237.673607926841*v_R3_a_r + 18.9579191056578*v_R3_b_i + 87.173059186788*v_R3_b_r + 45.8900105086227*v_R3_c_i + 57.6366430644055*v_R3_c_r + 18.9579191056579*v_R3_n_i + 87.173059186788*v_R3_n_r + 54.0657727682604*v_R4_a_i + 105.57176931318*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r - 23.284964480954*v_R4_c_i - 24.9575997948692*v_R4_c_r - 10.2713736253513*v_R4_n_i - 39.6392229058202*v_R4_n_r
        struct[0].g[57,0] = i_R3_a_i + 26.5300693004806*v_R11_a_i - 4.84676915585801*v_R11_a_r - 7.8946133751477*v_R11_b_i - 1.58482814504483*v_R11_b_r - 7.72144347466701*v_R11_c_i - 0.67991845328531*v_R11_c_r - 7.89461337514771*v_R11_n_i - 1.58482814504483*v_R11_n_r + 105.57176931318*v_R2_a_i - 54.0657727682604*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r - 24.9575997948692*v_R2_c_i + 23.284964480954*v_R2_c_r - 39.6392229058202*v_R2_n_i + 10.2713736253513*v_R2_n_r - 237.673607926841*v_R3_a_i + 112.978314692379*v_R3_a_r + 87.173059186788*v_R3_b_i - 18.9579191056578*v_R3_b_r + 57.6366430644055*v_R3_c_i - 45.8900105086227*v_R3_c_r + 87.173059186788*v_R3_n_i - 18.9579191056579*v_R3_n_r + 105.57176931318*v_R4_a_i - 54.0657727682604*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r - 24.9575997948692*v_R4_c_i + 23.284964480954*v_R4_c_r - 39.6392229058202*v_R4_n_i + 10.2713736253513*v_R4_n_r
        struct[0].g[58,0] = i_R3_b_r + 1.58482814504483*v_R11_a_i - 7.89461337514771*v_R11_a_r + 4.84676915585801*v_R11_b_i + 26.5300693004806*v_R11_b_r + 1.58482814504483*v_R11_c_i - 7.89461337514771*v_R11_c_r + 0.679918453285308*v_R11_n_i - 7.72144347466701*v_R11_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r + 54.0657727682604*v_R2_b_i + 105.57176931318*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058201*v_R2_c_r - 23.284964480954*v_R2_n_i - 24.9575997948692*v_R2_n_r + 18.9579191056579*v_R3_a_i + 87.173059186788*v_R3_a_r - 112.978314692379*v_R3_b_i - 237.673607926841*v_R3_b_r + 18.9579191056579*v_R3_c_i + 87.173059186788*v_R3_c_r + 45.8900105086227*v_R3_n_i + 57.6366430644054*v_R3_n_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r + 54.0657727682604*v_R4_b_i + 105.57176931318*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058201*v_R4_c_r - 23.284964480954*v_R4_n_i - 24.9575997948692*v_R4_n_r
        struct[0].g[59,0] = i_R3_b_i - 7.89461337514771*v_R11_a_i - 1.58482814504483*v_R11_a_r + 26.5300693004806*v_R11_b_i - 4.84676915585801*v_R11_b_r - 7.89461337514771*v_R11_c_i - 1.58482814504483*v_R11_c_r - 7.72144347466701*v_R11_n_i - 0.679918453285308*v_R11_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r + 105.57176931318*v_R2_b_i - 54.0657727682604*v_R2_b_r - 39.6392229058201*v_R2_c_i + 10.2713736253513*v_R2_c_r - 24.9575997948692*v_R2_n_i + 23.284964480954*v_R2_n_r + 87.173059186788*v_R3_a_i - 18.9579191056579*v_R3_a_r - 237.673607926841*v_R3_b_i + 112.978314692379*v_R3_b_r + 87.173059186788*v_R3_c_i - 18.9579191056579*v_R3_c_r + 57.6366430644054*v_R3_n_i - 45.8900105086227*v_R3_n_r - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r + 105.57176931318*v_R4_b_i - 54.0657727682604*v_R4_b_r - 39.6392229058201*v_R4_c_i + 10.2713736253513*v_R4_c_r - 24.9575997948692*v_R4_n_i + 23.284964480954*v_R4_n_r
        struct[0].g[60,0] = i_R3_c_r + 0.67991845328531*v_R11_a_i - 7.72144347466701*v_R11_a_r + 1.58482814504483*v_R11_b_i - 7.8946133751477*v_R11_b_r + 4.84676915585801*v_R11_c_i + 26.5300693004806*v_R11_c_r + 1.58482814504483*v_R11_n_i - 7.89461337514771*v_R11_n_r - 23.284964480954*v_R2_a_i - 24.9575997948692*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r + 54.0657727682604*v_R2_c_i + 105.57176931318*v_R2_c_r - 10.2713736253514*v_R2_n_i - 39.6392229058201*v_R2_n_r + 45.8900105086227*v_R3_a_i + 57.6366430644054*v_R3_a_r + 18.9579191056578*v_R3_b_i + 87.1730591867881*v_R3_b_r - 112.978314692379*v_R3_c_i - 237.673607926841*v_R3_c_r + 18.9579191056579*v_R3_n_i + 87.173059186788*v_R3_n_r - 23.284964480954*v_R4_a_i - 24.9575997948692*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r + 54.0657727682604*v_R4_c_i + 105.57176931318*v_R4_c_r - 10.2713736253514*v_R4_n_i - 39.6392229058201*v_R4_n_r
        struct[0].g[61,0] = i_R3_c_i - 7.72144347466701*v_R11_a_i - 0.67991845328531*v_R11_a_r - 7.8946133751477*v_R11_b_i - 1.58482814504483*v_R11_b_r + 26.5300693004806*v_R11_c_i - 4.84676915585801*v_R11_c_r - 7.89461337514771*v_R11_n_i - 1.58482814504483*v_R11_n_r - 24.9575997948692*v_R2_a_i + 23.284964480954*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r + 105.57176931318*v_R2_c_i - 54.0657727682604*v_R2_c_r - 39.6392229058201*v_R2_n_i + 10.2713736253514*v_R2_n_r + 57.6366430644054*v_R3_a_i - 45.8900105086227*v_R3_a_r + 87.1730591867881*v_R3_b_i - 18.9579191056578*v_R3_b_r - 237.673607926841*v_R3_c_i + 112.978314692379*v_R3_c_r + 87.173059186788*v_R3_n_i - 18.9579191056579*v_R3_n_r - 24.9575997948692*v_R4_a_i + 23.284964480954*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r + 105.57176931318*v_R4_c_i - 54.0657727682604*v_R4_c_r - 39.6392229058201*v_R4_n_i + 10.2713736253514*v_R4_n_r
        struct[0].g[62,0] = i_R3_n_r + 1.58482814504483*v_R11_a_i - 7.89461337514771*v_R11_a_r + 0.679918453285308*v_R11_b_i - 7.72144347466701*v_R11_b_r + 1.58482814504483*v_R11_c_i - 7.89461337514771*v_R11_c_r + 4.84676915585801*v_R11_n_i + 26.5300693004806*v_R11_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r - 23.284964480954*v_R2_b_i - 24.9575997948692*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058202*v_R2_c_r + 54.0657727682604*v_R2_n_i + 105.57176931318*v_R2_n_r + 18.9579191056578*v_R3_a_i + 87.173059186788*v_R3_a_r + 45.8900105086227*v_R3_b_i + 57.6366430644054*v_R3_b_r + 18.9579191056579*v_R3_c_i + 87.173059186788*v_R3_c_r - 112.978314692379*v_R3_n_i - 237.673607926841*v_R3_n_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r - 23.284964480954*v_R4_b_i - 24.9575997948692*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058202*v_R4_c_r + 54.0657727682604*v_R4_n_i + 105.57176931318*v_R4_n_r
        struct[0].g[63,0] = i_R3_n_i - 7.89461337514771*v_R11_a_i - 1.58482814504483*v_R11_a_r - 7.72144347466701*v_R11_b_i - 0.679918453285308*v_R11_b_r - 7.89461337514771*v_R11_c_i - 1.58482814504483*v_R11_c_r + 26.5300693004806*v_R11_n_i - 4.84676915585801*v_R11_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r - 24.9575997948692*v_R2_b_i + 23.284964480954*v_R2_b_r - 39.6392229058202*v_R2_c_i + 10.2713736253513*v_R2_c_r + 105.57176931318*v_R2_n_i - 54.0657727682604*v_R2_n_r + 87.173059186788*v_R3_a_i - 18.9579191056578*v_R3_a_r + 57.6366430644054*v_R3_b_i - 45.8900105086227*v_R3_b_r + 87.173059186788*v_R3_c_i - 18.9579191056579*v_R3_c_r - 237.673607926841*v_R3_n_i + 112.978314692379*v_R3_n_r - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r - 24.9575997948692*v_R4_b_i + 23.284964480954*v_R4_b_r - 39.6392229058202*v_R4_c_i + 10.2713736253513*v_R4_c_r + 105.57176931318*v_R4_n_i - 54.0657727682604*v_R4_n_r
        struct[0].g[64,0] = i_R4_a_r + 4.84676915585801*v_R12_a_i + 26.5300693004806*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 0.67991845328531*v_R12_c_i - 7.72144347466701*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r + 54.0657727682604*v_R3_a_i + 105.57176931318*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r - 23.284964480954*v_R3_c_i - 24.9575997948692*v_R3_c_r - 10.2713736253513*v_R3_n_i - 39.6392229058202*v_R3_n_r - 112.978314692379*v_R4_a_i - 237.673607926841*v_R4_a_r + 18.9579191056578*v_R4_b_i + 87.173059186788*v_R4_b_r + 45.8900105086227*v_R4_c_i + 57.6366430644055*v_R4_c_r + 18.9579191056579*v_R4_n_i + 87.173059186788*v_R4_n_r + 54.0657727682604*v_R5_a_i + 105.57176931318*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r - 23.284964480954*v_R5_c_i - 24.9575997948692*v_R5_c_r - 10.2713736253513*v_R5_n_i - 39.6392229058202*v_R5_n_r
        struct[0].g[65,0] = i_R4_a_i + 26.5300693004806*v_R12_a_i - 4.84676915585801*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r - 7.72144347466701*v_R12_c_i - 0.67991845328531*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r + 105.57176931318*v_R3_a_i - 54.0657727682604*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r - 24.9575997948692*v_R3_c_i + 23.284964480954*v_R3_c_r - 39.6392229058202*v_R3_n_i + 10.2713736253513*v_R3_n_r - 237.673607926841*v_R4_a_i + 112.978314692379*v_R4_a_r + 87.173059186788*v_R4_b_i - 18.9579191056578*v_R4_b_r + 57.6366430644055*v_R4_c_i - 45.8900105086227*v_R4_c_r + 87.173059186788*v_R4_n_i - 18.9579191056579*v_R4_n_r + 105.57176931318*v_R5_a_i - 54.0657727682604*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r - 24.9575997948692*v_R5_c_i + 23.284964480954*v_R5_c_r - 39.6392229058202*v_R5_n_i + 10.2713736253513*v_R5_n_r
        struct[0].g[66,0] = i_R4_b_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 4.84676915585801*v_R12_b_i + 26.5300693004806*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 0.679918453285308*v_R12_n_i - 7.72144347466701*v_R12_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r + 54.0657727682604*v_R3_b_i + 105.57176931318*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058201*v_R3_c_r - 23.284964480954*v_R3_n_i - 24.9575997948692*v_R3_n_r + 18.9579191056579*v_R4_a_i + 87.173059186788*v_R4_a_r - 112.978314692379*v_R4_b_i - 237.673607926841*v_R4_b_r + 18.9579191056579*v_R4_c_i + 87.173059186788*v_R4_c_r + 45.8900105086227*v_R4_n_i + 57.6366430644054*v_R4_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r + 54.0657727682604*v_R5_b_i + 105.57176931318*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058201*v_R5_c_r - 23.284964480954*v_R5_n_i - 24.9575997948692*v_R5_n_r
        struct[0].g[67,0] = i_R4_b_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r + 26.5300693004806*v_R12_b_i - 4.84676915585801*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r - 7.72144347466701*v_R12_n_i - 0.679918453285308*v_R12_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r + 105.57176931318*v_R3_b_i - 54.0657727682604*v_R3_b_r - 39.6392229058201*v_R3_c_i + 10.2713736253513*v_R3_c_r - 24.9575997948692*v_R3_n_i + 23.284964480954*v_R3_n_r + 87.173059186788*v_R4_a_i - 18.9579191056579*v_R4_a_r - 237.673607926841*v_R4_b_i + 112.978314692379*v_R4_b_r + 87.173059186788*v_R4_c_i - 18.9579191056579*v_R4_c_r + 57.6366430644054*v_R4_n_i - 45.8900105086227*v_R4_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r + 105.57176931318*v_R5_b_i - 54.0657727682604*v_R5_b_r - 39.6392229058201*v_R5_c_i + 10.2713736253513*v_R5_c_r - 24.9575997948692*v_R5_n_i + 23.284964480954*v_R5_n_r
        struct[0].g[68,0] = i_R4_c_r + 0.67991845328531*v_R12_a_i - 7.72144347466701*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 4.84676915585801*v_R12_c_i + 26.5300693004806*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r - 23.284964480954*v_R3_a_i - 24.9575997948692*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r + 54.0657727682604*v_R3_c_i + 105.57176931318*v_R3_c_r - 10.2713736253514*v_R3_n_i - 39.6392229058201*v_R3_n_r + 45.8900105086227*v_R4_a_i + 57.6366430644054*v_R4_a_r + 18.9579191056578*v_R4_b_i + 87.1730591867881*v_R4_b_r - 112.978314692379*v_R4_c_i - 237.673607926841*v_R4_c_r + 18.9579191056579*v_R4_n_i + 87.173059186788*v_R4_n_r - 23.284964480954*v_R5_a_i - 24.9575997948692*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r + 54.0657727682604*v_R5_c_i + 105.57176931318*v_R5_c_r - 10.2713736253514*v_R5_n_i - 39.6392229058201*v_R5_n_r
        struct[0].g[69,0] = i_R4_c_i - 7.72144347466701*v_R12_a_i - 0.67991845328531*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r + 26.5300693004806*v_R12_c_i - 4.84676915585801*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r - 24.9575997948692*v_R3_a_i + 23.284964480954*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r + 105.57176931318*v_R3_c_i - 54.0657727682604*v_R3_c_r - 39.6392229058201*v_R3_n_i + 10.2713736253514*v_R3_n_r + 57.6366430644054*v_R4_a_i - 45.8900105086227*v_R4_a_r + 87.1730591867881*v_R4_b_i - 18.9579191056578*v_R4_b_r - 237.673607926841*v_R4_c_i + 112.978314692379*v_R4_c_r + 87.173059186788*v_R4_n_i - 18.9579191056579*v_R4_n_r - 24.9575997948692*v_R5_a_i + 23.284964480954*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r + 105.57176931318*v_R5_c_i - 54.0657727682604*v_R5_c_r - 39.6392229058201*v_R5_n_i + 10.2713736253514*v_R5_n_r
        struct[0].g[70,0] = i_R4_n_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 0.679918453285308*v_R12_b_i - 7.72144347466701*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 4.84676915585801*v_R12_n_i + 26.5300693004806*v_R12_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r - 23.284964480954*v_R3_b_i - 24.9575997948692*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058202*v_R3_c_r + 54.0657727682604*v_R3_n_i + 105.57176931318*v_R3_n_r + 18.9579191056578*v_R4_a_i + 87.173059186788*v_R4_a_r + 45.8900105086227*v_R4_b_i + 57.6366430644054*v_R4_b_r + 18.9579191056579*v_R4_c_i + 87.173059186788*v_R4_c_r - 112.978314692379*v_R4_n_i - 237.698607926841*v_R4_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r - 23.284964480954*v_R5_b_i - 24.9575997948692*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058202*v_R5_c_r + 54.0657727682604*v_R5_n_i + 105.57176931318*v_R5_n_r
        struct[0].g[71,0] = i_R4_n_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r - 7.72144347466701*v_R12_b_i - 0.679918453285308*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r + 26.5300693004806*v_R12_n_i - 4.84676915585801*v_R12_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r - 24.9575997948692*v_R3_b_i + 23.284964480954*v_R3_b_r - 39.6392229058202*v_R3_c_i + 10.2713736253513*v_R3_c_r + 105.57176931318*v_R3_n_i - 54.0657727682604*v_R3_n_r + 87.173059186788*v_R4_a_i - 18.9579191056578*v_R4_a_r + 57.6366430644054*v_R4_b_i - 45.8900105086227*v_R4_b_r + 87.173059186788*v_R4_c_i - 18.9579191056579*v_R4_c_r - 237.698607926841*v_R4_n_i + 112.978314692379*v_R4_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r - 24.9575997948692*v_R5_b_i + 23.284964480954*v_R5_b_r - 39.6392229058202*v_R5_c_i + 10.2713736253513*v_R5_c_r + 105.57176931318*v_R5_n_i - 54.0657727682604*v_R5_n_r
        struct[0].g[72,0] = i_R5_a_r + 54.0657727682604*v_R4_a_i + 105.57176931318*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r - 23.284964480954*v_R4_c_i - 24.9575997948692*v_R4_c_r - 10.2713736253513*v_R4_n_i - 39.6392229058202*v_R4_n_r - 108.131545536521*v_R5_a_i - 211.14353862636*v_R5_a_r + 20.5427472507027*v_R5_b_i + 79.2784458116403*v_R5_b_r + 46.569928961908*v_R5_c_i + 49.9151995897384*v_R5_c_r + 20.5427472507027*v_R5_n_i + 79.2784458116403*v_R5_n_r + 54.0657727682604*v_R6_a_i + 105.57176931318*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r - 23.284964480954*v_R6_c_i - 24.9575997948692*v_R6_c_r - 10.2713736253513*v_R6_n_i - 39.6392229058202*v_R6_n_r
        struct[0].g[73,0] = i_R5_a_i + 105.57176931318*v_R4_a_i - 54.0657727682604*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r - 24.9575997948692*v_R4_c_i + 23.284964480954*v_R4_c_r - 39.6392229058202*v_R4_n_i + 10.2713736253513*v_R4_n_r - 211.14353862636*v_R5_a_i + 108.131545536521*v_R5_a_r + 79.2784458116403*v_R5_b_i - 20.5427472507027*v_R5_b_r + 49.9151995897384*v_R5_c_i - 46.569928961908*v_R5_c_r + 79.2784458116403*v_R5_n_i - 20.5427472507027*v_R5_n_r + 105.57176931318*v_R6_a_i - 54.0657727682604*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r - 24.9575997948692*v_R6_c_i + 23.284964480954*v_R6_c_r - 39.6392229058202*v_R6_n_i + 10.2713736253513*v_R6_n_r
        struct[0].g[74,0] = i_R5_b_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r + 54.0657727682604*v_R4_b_i + 105.57176931318*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058201*v_R4_c_r - 23.284964480954*v_R4_n_i - 24.9575997948692*v_R4_n_r + 20.5427472507027*v_R5_a_i + 79.2784458116403*v_R5_a_r - 108.131545536521*v_R5_b_i - 211.14353862636*v_R5_b_r + 20.5427472507027*v_R5_c_i + 79.2784458116403*v_R5_c_r + 46.569928961908*v_R5_n_i + 49.9151995897384*v_R5_n_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r + 54.0657727682604*v_R6_b_i + 105.57176931318*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058201*v_R6_c_r - 23.284964480954*v_R6_n_i - 24.9575997948692*v_R6_n_r
        struct[0].g[75,0] = i_R5_b_i - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r + 105.57176931318*v_R4_b_i - 54.0657727682604*v_R4_b_r - 39.6392229058201*v_R4_c_i + 10.2713736253513*v_R4_c_r - 24.9575997948692*v_R4_n_i + 23.284964480954*v_R4_n_r + 79.2784458116403*v_R5_a_i - 20.5427472507027*v_R5_a_r - 211.14353862636*v_R5_b_i + 108.131545536521*v_R5_b_r + 79.2784458116403*v_R5_c_i - 20.5427472507027*v_R5_c_r + 49.9151995897384*v_R5_n_i - 46.569928961908*v_R5_n_r - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r + 105.57176931318*v_R6_b_i - 54.0657727682604*v_R6_b_r - 39.6392229058201*v_R6_c_i + 10.2713736253513*v_R6_c_r - 24.9575997948692*v_R6_n_i + 23.284964480954*v_R6_n_r
        struct[0].g[76,0] = i_R5_c_r - 23.284964480954*v_R4_a_i - 24.9575997948692*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r + 54.0657727682604*v_R4_c_i + 105.57176931318*v_R4_c_r - 10.2713736253514*v_R4_n_i - 39.6392229058201*v_R4_n_r + 46.569928961908*v_R5_a_i + 49.9151995897384*v_R5_a_r + 20.5427472507027*v_R5_b_i + 79.2784458116404*v_R5_b_r - 108.131545536521*v_R5_c_i - 211.14353862636*v_R5_c_r + 20.5427472507027*v_R5_n_i + 79.2784458116403*v_R5_n_r - 23.284964480954*v_R6_a_i - 24.9575997948692*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r + 54.0657727682604*v_R6_c_i + 105.57176931318*v_R6_c_r - 10.2713736253514*v_R6_n_i - 39.6392229058201*v_R6_n_r
        struct[0].g[77,0] = i_R5_c_i - 24.9575997948692*v_R4_a_i + 23.284964480954*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r + 105.57176931318*v_R4_c_i - 54.0657727682604*v_R4_c_r - 39.6392229058201*v_R4_n_i + 10.2713736253514*v_R4_n_r + 49.9151995897384*v_R5_a_i - 46.569928961908*v_R5_a_r + 79.2784458116404*v_R5_b_i - 20.5427472507027*v_R5_b_r - 211.14353862636*v_R5_c_i + 108.131545536521*v_R5_c_r + 79.2784458116403*v_R5_n_i - 20.5427472507027*v_R5_n_r - 24.9575997948692*v_R6_a_i + 23.284964480954*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r + 105.57176931318*v_R6_c_i - 54.0657727682604*v_R6_c_r - 39.6392229058201*v_R6_n_i + 10.2713736253514*v_R6_n_r
        struct[0].g[78,0] = i_R5_n_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r - 23.284964480954*v_R4_b_i - 24.9575997948692*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058202*v_R4_c_r + 54.0657727682604*v_R4_n_i + 105.57176931318*v_R4_n_r + 20.5427472507027*v_R5_a_i + 79.2784458116403*v_R5_a_r + 46.569928961908*v_R5_b_i + 49.9151995897384*v_R5_b_r + 20.5427472507027*v_R5_c_i + 79.2784458116403*v_R5_c_r - 108.131545536521*v_R5_n_i - 211.14353862636*v_R5_n_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r - 23.284964480954*v_R6_b_i - 24.9575997948692*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058202*v_R6_c_r + 54.0657727682604*v_R6_n_i + 105.57176931318*v_R6_n_r
        struct[0].g[79,0] = i_R5_n_i - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r - 24.9575997948692*v_R4_b_i + 23.284964480954*v_R4_b_r - 39.6392229058202*v_R4_c_i + 10.2713736253513*v_R4_c_r + 105.57176931318*v_R4_n_i - 54.0657727682604*v_R4_n_r + 79.2784458116403*v_R5_a_i - 20.5427472507027*v_R5_a_r + 49.9151995897384*v_R5_b_i - 46.569928961908*v_R5_b_r + 79.2784458116403*v_R5_c_i - 20.5427472507027*v_R5_c_r - 211.14353862636*v_R5_n_i + 108.131545536521*v_R5_n_r - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r - 24.9575997948692*v_R6_b_i + 23.284964480954*v_R6_b_r - 39.6392229058202*v_R6_c_i + 10.2713736253513*v_R6_c_r + 105.57176931318*v_R6_n_i - 54.0657727682604*v_R6_n_r
        struct[0].g[80,0] = i_R6_a_r + 5.65456401516768*v_R16_a_i + 30.9517475172273*v_R16_a_r + 1.84896616921897*v_R16_b_i - 9.21038227100566*v_R16_b_r + 0.793238195499529*v_R16_c_i - 9.00835072044485*v_R16_c_r + 1.84896616921897*v_R16_n_i - 9.21038227100566*v_R16_n_r + 54.0657727682604*v_R5_a_i + 105.57176931318*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r - 23.284964480954*v_R5_c_i - 24.9575997948692*v_R5_c_r - 10.2713736253513*v_R5_n_i - 39.6392229058202*v_R5_n_r - 113.786109551688*v_R6_a_i - 242.095286143588*v_R6_a_r + 18.6937810814837*v_R6_b_i + 88.488828082646*v_R6_b_r + 45.7766907664085*v_R6_c_i + 58.9235503101833*v_R6_c_r + 18.6937810814837*v_R6_n_i + 88.488828082646*v_R6_n_r + 54.0657727682604*v_R7_a_i + 105.57176931318*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r - 23.284964480954*v_R7_c_i - 24.9575997948692*v_R7_c_r - 10.2713736253513*v_R7_n_i - 39.6392229058202*v_R7_n_r
        struct[0].g[81,0] = i_R6_a_i + 30.9517475172273*v_R16_a_i - 5.65456401516768*v_R16_a_r - 9.21038227100566*v_R16_b_i - 1.84896616921897*v_R16_b_r - 9.00835072044485*v_R16_c_i - 0.793238195499529*v_R16_c_r - 9.21038227100566*v_R16_n_i - 1.84896616921897*v_R16_n_r + 105.57176931318*v_R5_a_i - 54.0657727682604*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r - 24.9575997948692*v_R5_c_i + 23.284964480954*v_R5_c_r - 39.6392229058202*v_R5_n_i + 10.2713736253513*v_R5_n_r - 242.095286143588*v_R6_a_i + 113.786109551688*v_R6_a_r + 88.488828082646*v_R6_b_i - 18.6937810814837*v_R6_b_r + 58.9235503101833*v_R6_c_i - 45.7766907664085*v_R6_c_r + 88.488828082646*v_R6_n_i - 18.6937810814837*v_R6_n_r + 105.57176931318*v_R7_a_i - 54.0657727682604*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r - 24.9575997948692*v_R7_c_i + 23.284964480954*v_R7_c_r - 39.6392229058202*v_R7_n_i + 10.2713736253513*v_R7_n_r
        struct[0].g[82,0] = i_R6_b_r + 1.84896616921897*v_R16_a_i - 9.21038227100566*v_R16_a_r + 5.65456401516768*v_R16_b_i + 30.9517475172273*v_R16_b_r + 1.84896616921897*v_R16_c_i - 9.21038227100566*v_R16_c_r + 0.793238195499528*v_R16_n_i - 9.00835072044485*v_R16_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r + 54.0657727682604*v_R5_b_i + 105.57176931318*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058201*v_R5_c_r - 23.284964480954*v_R5_n_i - 24.9575997948692*v_R5_n_r + 18.6937810814837*v_R6_a_i + 88.488828082646*v_R6_a_r - 113.786109551688*v_R6_b_i - 242.095286143588*v_R6_b_r + 18.6937810814837*v_R6_c_i + 88.488828082646*v_R6_c_r + 45.7766907664085*v_R6_n_i + 58.9235503101833*v_R6_n_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r + 54.0657727682604*v_R7_b_i + 105.57176931318*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058201*v_R7_c_r - 23.284964480954*v_R7_n_i - 24.9575997948692*v_R7_n_r
        struct[0].g[83,0] = i_R6_b_i - 9.21038227100566*v_R16_a_i - 1.84896616921897*v_R16_a_r + 30.9517475172273*v_R16_b_i - 5.65456401516768*v_R16_b_r - 9.21038227100566*v_R16_c_i - 1.84896616921897*v_R16_c_r - 9.00835072044485*v_R16_n_i - 0.793238195499528*v_R16_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r + 105.57176931318*v_R5_b_i - 54.0657727682604*v_R5_b_r - 39.6392229058201*v_R5_c_i + 10.2713736253513*v_R5_c_r - 24.9575997948692*v_R5_n_i + 23.284964480954*v_R5_n_r + 88.488828082646*v_R6_a_i - 18.6937810814837*v_R6_a_r - 242.095286143588*v_R6_b_i + 113.786109551688*v_R6_b_r + 88.488828082646*v_R6_c_i - 18.6937810814837*v_R6_c_r + 58.9235503101833*v_R6_n_i - 45.7766907664085*v_R6_n_r - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r + 105.57176931318*v_R7_b_i - 54.0657727682604*v_R7_b_r - 39.6392229058201*v_R7_c_i + 10.2713736253513*v_R7_c_r - 24.9575997948692*v_R7_n_i + 23.284964480954*v_R7_n_r
        struct[0].g[84,0] = i_R6_c_r + 0.793238195499527*v_R16_a_i - 9.00835072044484*v_R16_a_r + 1.84896616921897*v_R16_b_i - 9.21038227100566*v_R16_b_r + 5.65456401516768*v_R16_c_i + 30.9517475172273*v_R16_c_r + 1.84896616921897*v_R16_n_i - 9.21038227100566*v_R16_n_r - 23.284964480954*v_R5_a_i - 24.9575997948692*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r + 54.0657727682604*v_R5_c_i + 105.57176931318*v_R5_c_r - 10.2713736253514*v_R5_n_i - 39.6392229058201*v_R5_n_r + 45.7766907664085*v_R6_a_i + 58.9235503101833*v_R6_a_r + 18.6937810814837*v_R6_b_i + 88.4888280826461*v_R6_b_r - 113.786109551688*v_R6_c_i - 242.095286143588*v_R6_c_r + 18.6937810814837*v_R6_n_i + 88.4888280826459*v_R6_n_r - 23.284964480954*v_R7_a_i - 24.9575997948692*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r + 54.0657727682604*v_R7_c_i + 105.57176931318*v_R7_c_r - 10.2713736253514*v_R7_n_i - 39.6392229058201*v_R7_n_r
        struct[0].g[85,0] = i_R6_c_i - 9.00835072044484*v_R16_a_i - 0.793238195499527*v_R16_a_r - 9.21038227100566*v_R16_b_i - 1.84896616921897*v_R16_b_r + 30.9517475172273*v_R16_c_i - 5.65456401516768*v_R16_c_r - 9.21038227100566*v_R16_n_i - 1.84896616921897*v_R16_n_r - 24.9575997948692*v_R5_a_i + 23.284964480954*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r + 105.57176931318*v_R5_c_i - 54.0657727682604*v_R5_c_r - 39.6392229058201*v_R5_n_i + 10.2713736253514*v_R5_n_r + 58.9235503101833*v_R6_a_i - 45.7766907664085*v_R6_a_r + 88.4888280826461*v_R6_b_i - 18.6937810814837*v_R6_b_r - 242.095286143588*v_R6_c_i + 113.786109551688*v_R6_c_r + 88.4888280826459*v_R6_n_i - 18.6937810814837*v_R6_n_r - 24.9575997948692*v_R7_a_i + 23.284964480954*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r + 105.57176931318*v_R7_c_i - 54.0657727682604*v_R7_c_r - 39.6392229058201*v_R7_n_i + 10.2713736253514*v_R7_n_r
        struct[0].g[86,0] = i_R6_n_r + 1.84896616921897*v_R16_a_i - 9.21038227100566*v_R16_a_r + 0.793238195499527*v_R16_b_i - 9.00835072044485*v_R16_b_r + 1.84896616921897*v_R16_c_i - 9.21038227100566*v_R16_c_r + 5.65456401516768*v_R16_n_i + 30.9517475172273*v_R16_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r - 23.284964480954*v_R5_b_i - 24.9575997948692*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058202*v_R5_c_r + 54.0657727682604*v_R5_n_i + 105.57176931318*v_R5_n_r + 18.6937810814837*v_R6_a_i + 88.488828082646*v_R6_a_r + 45.7766907664085*v_R6_b_i + 58.9235503101832*v_R6_b_r + 18.6937810814837*v_R6_c_i + 88.488828082646*v_R6_c_r - 113.786109551688*v_R6_n_i - 242.120286143588*v_R6_n_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r - 23.284964480954*v_R7_b_i - 24.9575997948692*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058202*v_R7_c_r + 54.0657727682604*v_R7_n_i + 105.57176931318*v_R7_n_r
        struct[0].g[87,0] = i_R6_n_i - 9.21038227100566*v_R16_a_i - 1.84896616921897*v_R16_a_r - 9.00835072044485*v_R16_b_i - 0.793238195499527*v_R16_b_r - 9.21038227100566*v_R16_c_i - 1.84896616921897*v_R16_c_r + 30.9517475172273*v_R16_n_i - 5.65456401516768*v_R16_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r - 24.9575997948692*v_R5_b_i + 23.284964480954*v_R5_b_r - 39.6392229058202*v_R5_c_i + 10.2713736253513*v_R5_c_r + 105.57176931318*v_R5_n_i - 54.0657727682604*v_R5_n_r + 88.488828082646*v_R6_a_i - 18.6937810814837*v_R6_a_r + 58.9235503101832*v_R6_b_i - 45.7766907664085*v_R6_b_r + 88.488828082646*v_R6_c_i - 18.6937810814837*v_R6_c_r - 242.120286143588*v_R6_n_i + 113.786109551688*v_R6_n_r - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r - 24.9575997948692*v_R7_b_i + 23.284964480954*v_R7_b_r - 39.6392229058202*v_R7_c_i + 10.2713736253513*v_R7_c_r + 105.57176931318*v_R7_n_i - 54.0657727682604*v_R7_n_r
        struct[0].g[88,0] = i_R7_a_r + 54.0657727682604*v_R6_a_i + 105.57176931318*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r - 23.284964480954*v_R6_c_i - 24.9575997948692*v_R6_c_r - 10.2713736253513*v_R6_n_i - 39.6392229058202*v_R6_n_r - 108.131545536521*v_R7_a_i - 211.14353862636*v_R7_a_r + 20.5427472507027*v_R7_b_i + 79.2784458116403*v_R7_b_r + 46.569928961908*v_R7_c_i + 49.9151995897384*v_R7_c_r + 20.5427472507027*v_R7_n_i + 79.2784458116403*v_R7_n_r + 54.0657727682604*v_R8_a_i + 105.57176931318*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r - 23.284964480954*v_R8_c_i - 24.9575997948692*v_R8_c_r - 10.2713736253513*v_R8_n_i - 39.6392229058202*v_R8_n_r
        struct[0].g[89,0] = i_R7_a_i + 105.57176931318*v_R6_a_i - 54.0657727682604*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r - 24.9575997948692*v_R6_c_i + 23.284964480954*v_R6_c_r - 39.6392229058202*v_R6_n_i + 10.2713736253513*v_R6_n_r - 211.14353862636*v_R7_a_i + 108.131545536521*v_R7_a_r + 79.2784458116403*v_R7_b_i - 20.5427472507027*v_R7_b_r + 49.9151995897384*v_R7_c_i - 46.569928961908*v_R7_c_r + 79.2784458116403*v_R7_n_i - 20.5427472507027*v_R7_n_r + 105.57176931318*v_R8_a_i - 54.0657727682604*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r - 24.9575997948692*v_R8_c_i + 23.284964480954*v_R8_c_r - 39.6392229058202*v_R8_n_i + 10.2713736253513*v_R8_n_r
        struct[0].g[90,0] = i_R7_b_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r + 54.0657727682604*v_R6_b_i + 105.57176931318*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058201*v_R6_c_r - 23.284964480954*v_R6_n_i - 24.9575997948692*v_R6_n_r + 20.5427472507027*v_R7_a_i + 79.2784458116403*v_R7_a_r - 108.131545536521*v_R7_b_i - 211.14353862636*v_R7_b_r + 20.5427472507027*v_R7_c_i + 79.2784458116403*v_R7_c_r + 46.569928961908*v_R7_n_i + 49.9151995897384*v_R7_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r + 54.0657727682604*v_R8_b_i + 105.57176931318*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058201*v_R8_c_r - 23.284964480954*v_R8_n_i - 24.9575997948692*v_R8_n_r
        struct[0].g[91,0] = i_R7_b_i - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r + 105.57176931318*v_R6_b_i - 54.0657727682604*v_R6_b_r - 39.6392229058201*v_R6_c_i + 10.2713736253513*v_R6_c_r - 24.9575997948692*v_R6_n_i + 23.284964480954*v_R6_n_r + 79.2784458116403*v_R7_a_i - 20.5427472507027*v_R7_a_r - 211.14353862636*v_R7_b_i + 108.131545536521*v_R7_b_r + 79.2784458116403*v_R7_c_i - 20.5427472507027*v_R7_c_r + 49.9151995897384*v_R7_n_i - 46.569928961908*v_R7_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r + 105.57176931318*v_R8_b_i - 54.0657727682604*v_R8_b_r - 39.6392229058201*v_R8_c_i + 10.2713736253513*v_R8_c_r - 24.9575997948692*v_R8_n_i + 23.284964480954*v_R8_n_r
        struct[0].g[92,0] = i_R7_c_r - 23.284964480954*v_R6_a_i - 24.9575997948692*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r + 54.0657727682604*v_R6_c_i + 105.57176931318*v_R6_c_r - 10.2713736253514*v_R6_n_i - 39.6392229058201*v_R6_n_r + 46.569928961908*v_R7_a_i + 49.9151995897384*v_R7_a_r + 20.5427472507027*v_R7_b_i + 79.2784458116404*v_R7_b_r - 108.131545536521*v_R7_c_i - 211.14353862636*v_R7_c_r + 20.5427472507027*v_R7_n_i + 79.2784458116403*v_R7_n_r - 23.284964480954*v_R8_a_i - 24.9575997948692*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r + 54.0657727682604*v_R8_c_i + 105.57176931318*v_R8_c_r - 10.2713736253514*v_R8_n_i - 39.6392229058201*v_R8_n_r
        struct[0].g[93,0] = i_R7_c_i - 24.9575997948692*v_R6_a_i + 23.284964480954*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r + 105.57176931318*v_R6_c_i - 54.0657727682604*v_R6_c_r - 39.6392229058201*v_R6_n_i + 10.2713736253514*v_R6_n_r + 49.9151995897384*v_R7_a_i - 46.569928961908*v_R7_a_r + 79.2784458116404*v_R7_b_i - 20.5427472507027*v_R7_b_r - 211.14353862636*v_R7_c_i + 108.131545536521*v_R7_c_r + 79.2784458116403*v_R7_n_i - 20.5427472507027*v_R7_n_r - 24.9575997948692*v_R8_a_i + 23.284964480954*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r + 105.57176931318*v_R8_c_i - 54.0657727682604*v_R8_c_r - 39.6392229058201*v_R8_n_i + 10.2713736253514*v_R8_n_r
        struct[0].g[94,0] = i_R7_n_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r - 23.284964480954*v_R6_b_i - 24.9575997948692*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058202*v_R6_c_r + 54.0657727682604*v_R6_n_i + 105.57176931318*v_R6_n_r + 20.5427472507027*v_R7_a_i + 79.2784458116403*v_R7_a_r + 46.569928961908*v_R7_b_i + 49.9151995897384*v_R7_b_r + 20.5427472507027*v_R7_c_i + 79.2784458116403*v_R7_c_r - 108.131545536521*v_R7_n_i - 211.14353862636*v_R7_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r - 23.284964480954*v_R8_b_i - 24.9575997948692*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058202*v_R8_c_r + 54.0657727682604*v_R8_n_i + 105.57176931318*v_R8_n_r
        struct[0].g[95,0] = i_R7_n_i - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r - 24.9575997948692*v_R6_b_i + 23.284964480954*v_R6_b_r - 39.6392229058202*v_R6_c_i + 10.2713736253513*v_R6_c_r + 105.57176931318*v_R6_n_i - 54.0657727682604*v_R6_n_r + 79.2784458116403*v_R7_a_i - 20.5427472507027*v_R7_a_r + 49.9151995897384*v_R7_b_i - 46.569928961908*v_R7_b_r + 79.2784458116403*v_R7_c_i - 20.5427472507027*v_R7_c_r - 211.14353862636*v_R7_n_i + 108.131545536521*v_R7_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r - 24.9575997948692*v_R8_b_i + 23.284964480954*v_R8_b_r - 39.6392229058202*v_R8_c_i + 10.2713736253513*v_R8_c_r + 105.57176931318*v_R8_n_i - 54.0657727682604*v_R8_n_r
        struct[0].g[96,0] = i_R8_a_r + 54.0657727682604*v_R7_a_i + 105.57176931318*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r - 23.284964480954*v_R7_c_i - 24.9575997948692*v_R7_c_r - 10.2713736253513*v_R7_n_i - 39.6392229058202*v_R7_n_r - 108.131545536521*v_R8_a_i - 211.14353862636*v_R8_a_r + 20.5427472507027*v_R8_b_i + 79.2784458116403*v_R8_b_r + 46.569928961908*v_R8_c_i + 49.9151995897384*v_R8_c_r + 20.5427472507027*v_R8_n_i + 79.2784458116403*v_R8_n_r + 54.0657727682604*v_R9_a_i + 105.57176931318*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r - 23.284964480954*v_R9_c_i - 24.9575997948692*v_R9_c_r - 10.2713736253513*v_R9_n_i - 39.6392229058202*v_R9_n_r
        struct[0].g[97,0] = i_R8_a_i + 105.57176931318*v_R7_a_i - 54.0657727682604*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r - 24.9575997948692*v_R7_c_i + 23.284964480954*v_R7_c_r - 39.6392229058202*v_R7_n_i + 10.2713736253513*v_R7_n_r - 211.14353862636*v_R8_a_i + 108.131545536521*v_R8_a_r + 79.2784458116403*v_R8_b_i - 20.5427472507027*v_R8_b_r + 49.9151995897384*v_R8_c_i - 46.569928961908*v_R8_c_r + 79.2784458116403*v_R8_n_i - 20.5427472507027*v_R8_n_r + 105.57176931318*v_R9_a_i - 54.0657727682604*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r - 24.9575997948692*v_R9_c_i + 23.284964480954*v_R9_c_r - 39.6392229058202*v_R9_n_i + 10.2713736253513*v_R9_n_r
        struct[0].g[98,0] = i_R8_b_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r + 54.0657727682604*v_R7_b_i + 105.57176931318*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058201*v_R7_c_r - 23.284964480954*v_R7_n_i - 24.9575997948692*v_R7_n_r + 20.5427472507027*v_R8_a_i + 79.2784458116403*v_R8_a_r - 108.131545536521*v_R8_b_i - 211.14353862636*v_R8_b_r + 20.5427472507027*v_R8_c_i + 79.2784458116403*v_R8_c_r + 46.569928961908*v_R8_n_i + 49.9151995897384*v_R8_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r + 54.0657727682604*v_R9_b_i + 105.57176931318*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058201*v_R9_c_r - 23.284964480954*v_R9_n_i - 24.9575997948692*v_R9_n_r
        struct[0].g[99,0] = i_R8_b_i - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r + 105.57176931318*v_R7_b_i - 54.0657727682604*v_R7_b_r - 39.6392229058201*v_R7_c_i + 10.2713736253513*v_R7_c_r - 24.9575997948692*v_R7_n_i + 23.284964480954*v_R7_n_r + 79.2784458116403*v_R8_a_i - 20.5427472507027*v_R8_a_r - 211.14353862636*v_R8_b_i + 108.131545536521*v_R8_b_r + 79.2784458116403*v_R8_c_i - 20.5427472507027*v_R8_c_r + 49.9151995897384*v_R8_n_i - 46.569928961908*v_R8_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r + 105.57176931318*v_R9_b_i - 54.0657727682604*v_R9_b_r - 39.6392229058201*v_R9_c_i + 10.2713736253513*v_R9_c_r - 24.9575997948692*v_R9_n_i + 23.284964480954*v_R9_n_r
        struct[0].g[100,0] = i_R8_c_r - 23.284964480954*v_R7_a_i - 24.9575997948692*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r + 54.0657727682604*v_R7_c_i + 105.57176931318*v_R7_c_r - 10.2713736253514*v_R7_n_i - 39.6392229058201*v_R7_n_r + 46.569928961908*v_R8_a_i + 49.9151995897384*v_R8_a_r + 20.5427472507027*v_R8_b_i + 79.2784458116404*v_R8_b_r - 108.131545536521*v_R8_c_i - 211.14353862636*v_R8_c_r + 20.5427472507027*v_R8_n_i + 79.2784458116403*v_R8_n_r - 23.284964480954*v_R9_a_i - 24.9575997948692*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r + 54.0657727682604*v_R9_c_i + 105.57176931318*v_R9_c_r - 10.2713736253514*v_R9_n_i - 39.6392229058201*v_R9_n_r
        struct[0].g[101,0] = i_R8_c_i - 24.9575997948692*v_R7_a_i + 23.284964480954*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r + 105.57176931318*v_R7_c_i - 54.0657727682604*v_R7_c_r - 39.6392229058201*v_R7_n_i + 10.2713736253514*v_R7_n_r + 49.9151995897384*v_R8_a_i - 46.569928961908*v_R8_a_r + 79.2784458116404*v_R8_b_i - 20.5427472507027*v_R8_b_r - 211.14353862636*v_R8_c_i + 108.131545536521*v_R8_c_r + 79.2784458116403*v_R8_n_i - 20.5427472507027*v_R8_n_r - 24.9575997948692*v_R9_a_i + 23.284964480954*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r + 105.57176931318*v_R9_c_i - 54.0657727682604*v_R9_c_r - 39.6392229058201*v_R9_n_i + 10.2713736253514*v_R9_n_r
        struct[0].g[102,0] = i_R8_n_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r - 23.284964480954*v_R7_b_i - 24.9575997948692*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058202*v_R7_c_r + 54.0657727682604*v_R7_n_i + 105.57176931318*v_R7_n_r + 20.5427472507027*v_R8_a_i + 79.2784458116403*v_R8_a_r + 46.569928961908*v_R8_b_i + 49.9151995897384*v_R8_b_r + 20.5427472507027*v_R8_c_i + 79.2784458116403*v_R8_c_r - 108.131545536521*v_R8_n_i - 211.16853862636*v_R8_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r - 23.284964480954*v_R9_b_i - 24.9575997948692*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058202*v_R9_c_r + 54.0657727682604*v_R9_n_i + 105.57176931318*v_R9_n_r
        struct[0].g[103,0] = i_R8_n_i - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r - 24.9575997948692*v_R7_b_i + 23.284964480954*v_R7_b_r - 39.6392229058202*v_R7_c_i + 10.2713736253513*v_R7_c_r + 105.57176931318*v_R7_n_i - 54.0657727682604*v_R7_n_r + 79.2784458116403*v_R8_a_i - 20.5427472507027*v_R8_a_r + 49.9151995897384*v_R8_b_i - 46.569928961908*v_R8_b_r + 79.2784458116403*v_R8_c_i - 20.5427472507027*v_R8_c_r - 211.16853862636*v_R8_n_i + 108.131545536521*v_R8_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r - 24.9575997948692*v_R9_b_i + 23.284964480954*v_R9_b_r - 39.6392229058202*v_R9_c_i + 10.2713736253513*v_R9_c_r + 105.57176931318*v_R9_n_i - 54.0657727682604*v_R9_n_r
        struct[0].g[104,0] = i_R9_a_r + 54.0657727682604*v_R10_a_i + 105.57176931318*v_R10_a_r - 10.2713736253513*v_R10_b_i - 39.6392229058202*v_R10_b_r - 23.284964480954*v_R10_c_i - 24.9575997948692*v_R10_c_r - 10.2713736253513*v_R10_n_i - 39.6392229058202*v_R10_n_r + 5.65456401516768*v_R17_a_i + 30.9517475172273*v_R17_a_r + 1.84896616921897*v_R17_b_i - 9.21038227100566*v_R17_b_r + 0.793238195499529*v_R17_c_i - 9.00835072044485*v_R17_c_r + 1.84896616921897*v_R17_n_i - 9.21038227100566*v_R17_n_r + 54.0657727682604*v_R8_a_i + 105.57176931318*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r - 23.284964480954*v_R8_c_i - 24.9575997948692*v_R8_c_r - 10.2713736253513*v_R8_n_i - 39.6392229058202*v_R8_n_r - 113.786109551688*v_R9_a_i - 242.095286143588*v_R9_a_r + 18.6937810814837*v_R9_b_i + 88.488828082646*v_R9_b_r + 45.7766907664085*v_R9_c_i + 58.9235503101833*v_R9_c_r + 18.6937810814837*v_R9_n_i + 88.488828082646*v_R9_n_r
        struct[0].g[105,0] = i_R9_a_i + 105.57176931318*v_R10_a_i - 54.0657727682604*v_R10_a_r - 39.6392229058202*v_R10_b_i + 10.2713736253513*v_R10_b_r - 24.9575997948692*v_R10_c_i + 23.284964480954*v_R10_c_r - 39.6392229058202*v_R10_n_i + 10.2713736253513*v_R10_n_r + 30.9517475172273*v_R17_a_i - 5.65456401516768*v_R17_a_r - 9.21038227100566*v_R17_b_i - 1.84896616921897*v_R17_b_r - 9.00835072044485*v_R17_c_i - 0.793238195499529*v_R17_c_r - 9.21038227100566*v_R17_n_i - 1.84896616921897*v_R17_n_r + 105.57176931318*v_R8_a_i - 54.0657727682604*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r - 24.9575997948692*v_R8_c_i + 23.284964480954*v_R8_c_r - 39.6392229058202*v_R8_n_i + 10.2713736253513*v_R8_n_r - 242.095286143588*v_R9_a_i + 113.786109551688*v_R9_a_r + 88.488828082646*v_R9_b_i - 18.6937810814837*v_R9_b_r + 58.9235503101833*v_R9_c_i - 45.7766907664085*v_R9_c_r + 88.488828082646*v_R9_n_i - 18.6937810814837*v_R9_n_r
        struct[0].g[106,0] = i_R9_b_r - 10.2713736253513*v_R10_a_i - 39.6392229058202*v_R10_a_r + 54.0657727682604*v_R10_b_i + 105.57176931318*v_R10_b_r - 10.2713736253513*v_R10_c_i - 39.6392229058201*v_R10_c_r - 23.284964480954*v_R10_n_i - 24.9575997948692*v_R10_n_r + 1.84896616921897*v_R17_a_i - 9.21038227100566*v_R17_a_r + 5.65456401516768*v_R17_b_i + 30.9517475172273*v_R17_b_r + 1.84896616921897*v_R17_c_i - 9.21038227100566*v_R17_c_r + 0.793238195499528*v_R17_n_i - 9.00835072044485*v_R17_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r + 54.0657727682604*v_R8_b_i + 105.57176931318*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058201*v_R8_c_r - 23.284964480954*v_R8_n_i - 24.9575997948692*v_R8_n_r + 18.6937810814837*v_R9_a_i + 88.488828082646*v_R9_a_r - 113.786109551688*v_R9_b_i - 242.095286143588*v_R9_b_r + 18.6937810814837*v_R9_c_i + 88.488828082646*v_R9_c_r + 45.7766907664085*v_R9_n_i + 58.9235503101833*v_R9_n_r
        struct[0].g[107,0] = i_R9_b_i - 39.6392229058202*v_R10_a_i + 10.2713736253513*v_R10_a_r + 105.57176931318*v_R10_b_i - 54.0657727682604*v_R10_b_r - 39.6392229058201*v_R10_c_i + 10.2713736253513*v_R10_c_r - 24.9575997948692*v_R10_n_i + 23.284964480954*v_R10_n_r - 9.21038227100566*v_R17_a_i - 1.84896616921897*v_R17_a_r + 30.9517475172273*v_R17_b_i - 5.65456401516768*v_R17_b_r - 9.21038227100566*v_R17_c_i - 1.84896616921897*v_R17_c_r - 9.00835072044485*v_R17_n_i - 0.793238195499528*v_R17_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r + 105.57176931318*v_R8_b_i - 54.0657727682604*v_R8_b_r - 39.6392229058201*v_R8_c_i + 10.2713736253513*v_R8_c_r - 24.9575997948692*v_R8_n_i + 23.284964480954*v_R8_n_r + 88.488828082646*v_R9_a_i - 18.6937810814837*v_R9_a_r - 242.095286143588*v_R9_b_i + 113.786109551688*v_R9_b_r + 88.488828082646*v_R9_c_i - 18.6937810814837*v_R9_c_r + 58.9235503101833*v_R9_n_i - 45.7766907664085*v_R9_n_r
        struct[0].g[108,0] = i_R9_c_r - 23.284964480954*v_R10_a_i - 24.9575997948692*v_R10_a_r - 10.2713736253513*v_R10_b_i - 39.6392229058202*v_R10_b_r + 54.0657727682604*v_R10_c_i + 105.57176931318*v_R10_c_r - 10.2713736253514*v_R10_n_i - 39.6392229058201*v_R10_n_r + 0.793238195499527*v_R17_a_i - 9.00835072044484*v_R17_a_r + 1.84896616921897*v_R17_b_i - 9.21038227100566*v_R17_b_r + 5.65456401516768*v_R17_c_i + 30.9517475172273*v_R17_c_r + 1.84896616921897*v_R17_n_i - 9.21038227100566*v_R17_n_r - 23.284964480954*v_R8_a_i - 24.9575997948692*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r + 54.0657727682604*v_R8_c_i + 105.57176931318*v_R8_c_r - 10.2713736253514*v_R8_n_i - 39.6392229058201*v_R8_n_r + 45.7766907664085*v_R9_a_i + 58.9235503101833*v_R9_a_r + 18.6937810814837*v_R9_b_i + 88.4888280826461*v_R9_b_r - 113.786109551688*v_R9_c_i - 242.095286143588*v_R9_c_r + 18.6937810814837*v_R9_n_i + 88.4888280826459*v_R9_n_r
        struct[0].g[109,0] = i_R9_c_i - 24.9575997948692*v_R10_a_i + 23.284964480954*v_R10_a_r - 39.6392229058202*v_R10_b_i + 10.2713736253513*v_R10_b_r + 105.57176931318*v_R10_c_i - 54.0657727682604*v_R10_c_r - 39.6392229058201*v_R10_n_i + 10.2713736253514*v_R10_n_r - 9.00835072044484*v_R17_a_i - 0.793238195499527*v_R17_a_r - 9.21038227100566*v_R17_b_i - 1.84896616921897*v_R17_b_r + 30.9517475172273*v_R17_c_i - 5.65456401516768*v_R17_c_r - 9.21038227100566*v_R17_n_i - 1.84896616921897*v_R17_n_r - 24.9575997948692*v_R8_a_i + 23.284964480954*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r + 105.57176931318*v_R8_c_i - 54.0657727682604*v_R8_c_r - 39.6392229058201*v_R8_n_i + 10.2713736253514*v_R8_n_r + 58.9235503101833*v_R9_a_i - 45.7766907664085*v_R9_a_r + 88.4888280826461*v_R9_b_i - 18.6937810814837*v_R9_b_r - 242.095286143588*v_R9_c_i + 113.786109551688*v_R9_c_r + 88.4888280826459*v_R9_n_i - 18.6937810814837*v_R9_n_r
        struct[0].g[110,0] = i_R9_n_r - 10.2713736253513*v_R10_a_i - 39.6392229058202*v_R10_a_r - 23.284964480954*v_R10_b_i - 24.9575997948692*v_R10_b_r - 10.2713736253513*v_R10_c_i - 39.6392229058202*v_R10_c_r + 54.0657727682604*v_R10_n_i + 105.57176931318*v_R10_n_r + 1.84896616921897*v_R17_a_i - 9.21038227100566*v_R17_a_r + 0.793238195499527*v_R17_b_i - 9.00835072044485*v_R17_b_r + 1.84896616921897*v_R17_c_i - 9.21038227100566*v_R17_c_r + 5.65456401516768*v_R17_n_i + 30.9517475172273*v_R17_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r - 23.284964480954*v_R8_b_i - 24.9575997948692*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058202*v_R8_c_r + 54.0657727682604*v_R8_n_i + 105.57176931318*v_R8_n_r + 18.6937810814837*v_R9_a_i + 88.488828082646*v_R9_a_r + 45.7766907664085*v_R9_b_i + 58.9235503101832*v_R9_b_r + 18.6937810814837*v_R9_c_i + 88.488828082646*v_R9_c_r - 113.786109551688*v_R9_n_i - 242.095286143588*v_R9_n_r
        struct[0].g[111,0] = i_R9_n_i - 39.6392229058202*v_R10_a_i + 10.2713736253513*v_R10_a_r - 24.9575997948692*v_R10_b_i + 23.284964480954*v_R10_b_r - 39.6392229058202*v_R10_c_i + 10.2713736253513*v_R10_c_r + 105.57176931318*v_R10_n_i - 54.0657727682604*v_R10_n_r - 9.21038227100566*v_R17_a_i - 1.84896616921897*v_R17_a_r - 9.00835072044485*v_R17_b_i - 0.793238195499527*v_R17_b_r - 9.21038227100566*v_R17_c_i - 1.84896616921897*v_R17_c_r + 30.9517475172273*v_R17_n_i - 5.65456401516768*v_R17_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r - 24.9575997948692*v_R8_b_i + 23.284964480954*v_R8_b_r - 39.6392229058202*v_R8_c_i + 10.2713736253513*v_R8_c_r + 105.57176931318*v_R8_n_i - 54.0657727682604*v_R8_n_r + 88.488828082646*v_R9_a_i - 18.6937810814837*v_R9_a_r + 58.9235503101832*v_R9_b_i - 45.7766907664085*v_R9_b_r + 88.488828082646*v_R9_c_i - 18.6937810814837*v_R9_c_r - 242.095286143588*v_R9_n_i + 113.786109551688*v_R9_n_r
        struct[0].g[112,0] = i_R10_a_r - 59.7203367834281*v_R10_a_i - 136.523516830407*v_R10_a_r + 8.42240745613237*v_R10_b_i + 48.8496051768258*v_R10_b_r + 22.4917262854545*v_R10_c_i + 33.9659505153141*v_R10_c_r + 8.42240745613237*v_R10_n_i + 48.8496051768258*v_R10_n_r + 5.65456401516768*v_R18_a_i + 30.9517475172273*v_R18_a_r + 1.84896616921897*v_R18_b_i - 9.21038227100566*v_R18_b_r + 0.793238195499529*v_R18_c_i - 9.00835072044485*v_R18_c_r + 1.84896616921897*v_R18_n_i - 9.21038227100566*v_R18_n_r + 54.0657727682604*v_R9_a_i + 105.57176931318*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r - 23.284964480954*v_R9_c_i - 24.9575997948692*v_R9_c_r - 10.2713736253513*v_R9_n_i - 39.6392229058202*v_R9_n_r
        struct[0].g[113,0] = i_R10_a_i - 136.523516830407*v_R10_a_i + 59.7203367834281*v_R10_a_r + 48.8496051768258*v_R10_b_i - 8.42240745613237*v_R10_b_r + 33.9659505153141*v_R10_c_i - 22.4917262854545*v_R10_c_r + 48.8496051768258*v_R10_n_i - 8.42240745613237*v_R10_n_r + 30.9517475172273*v_R18_a_i - 5.65456401516768*v_R18_a_r - 9.21038227100566*v_R18_b_i - 1.84896616921897*v_R18_b_r - 9.00835072044485*v_R18_c_i - 0.793238195499529*v_R18_c_r - 9.21038227100566*v_R18_n_i - 1.84896616921897*v_R18_n_r + 105.57176931318*v_R9_a_i - 54.0657727682604*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r - 24.9575997948692*v_R9_c_i + 23.284964480954*v_R9_c_r - 39.6392229058202*v_R9_n_i + 10.2713736253513*v_R9_n_r
        struct[0].g[114,0] = i_R10_b_r + 8.42240745613237*v_R10_a_i + 48.8496051768258*v_R10_a_r - 59.720336783428*v_R10_b_i - 136.523516830407*v_R10_b_r + 8.42240745613238*v_R10_c_i + 48.8496051768258*v_R10_c_r + 22.4917262854545*v_R10_n_i + 33.9659505153141*v_R10_n_r + 1.84896616921897*v_R18_a_i - 9.21038227100566*v_R18_a_r + 5.65456401516768*v_R18_b_i + 30.9517475172273*v_R18_b_r + 1.84896616921897*v_R18_c_i - 9.21038227100566*v_R18_c_r + 0.793238195499528*v_R18_n_i - 9.00835072044485*v_R18_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r + 54.0657727682604*v_R9_b_i + 105.57176931318*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058201*v_R9_c_r - 23.284964480954*v_R9_n_i - 24.9575997948692*v_R9_n_r
        struct[0].g[115,0] = i_R10_b_i + 48.8496051768258*v_R10_a_i - 8.42240745613237*v_R10_a_r - 136.523516830407*v_R10_b_i + 59.720336783428*v_R10_b_r + 48.8496051768258*v_R10_c_i - 8.42240745613238*v_R10_c_r + 33.9659505153141*v_R10_n_i - 22.4917262854545*v_R10_n_r - 9.21038227100566*v_R18_a_i - 1.84896616921897*v_R18_a_r + 30.9517475172273*v_R18_b_i - 5.65456401516768*v_R18_b_r - 9.21038227100566*v_R18_c_i - 1.84896616921897*v_R18_c_r - 9.00835072044485*v_R18_n_i - 0.793238195499528*v_R18_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r + 105.57176931318*v_R9_b_i - 54.0657727682604*v_R9_b_r - 39.6392229058201*v_R9_c_i + 10.2713736253513*v_R9_c_r - 24.9575997948692*v_R9_n_i + 23.284964480954*v_R9_n_r
        struct[0].g[116,0] = i_R10_c_r + 22.4917262854545*v_R10_a_i + 33.9659505153141*v_R10_a_r + 8.42240745613236*v_R10_b_i + 48.8496051768259*v_R10_b_r - 59.7203367834281*v_R10_c_i - 136.523516830407*v_R10_c_r + 8.42240745613239*v_R10_n_i + 48.8496051768258*v_R10_n_r + 0.793238195499527*v_R18_a_i - 9.00835072044484*v_R18_a_r + 1.84896616921897*v_R18_b_i - 9.21038227100566*v_R18_b_r + 5.65456401516768*v_R18_c_i + 30.9517475172273*v_R18_c_r + 1.84896616921897*v_R18_n_i - 9.21038227100566*v_R18_n_r - 23.284964480954*v_R9_a_i - 24.9575997948692*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r + 54.0657727682604*v_R9_c_i + 105.57176931318*v_R9_c_r - 10.2713736253514*v_R9_n_i - 39.6392229058201*v_R9_n_r
        struct[0].g[117,0] = i_R10_c_i + 33.9659505153141*v_R10_a_i - 22.4917262854545*v_R10_a_r + 48.8496051768259*v_R10_b_i - 8.42240745613236*v_R10_b_r - 136.523516830407*v_R10_c_i + 59.7203367834281*v_R10_c_r + 48.8496051768258*v_R10_n_i - 8.42240745613239*v_R10_n_r - 9.00835072044484*v_R18_a_i - 0.793238195499527*v_R18_a_r - 9.21038227100566*v_R18_b_i - 1.84896616921897*v_R18_b_r + 30.9517475172273*v_R18_c_i - 5.65456401516768*v_R18_c_r - 9.21038227100566*v_R18_n_i - 1.84896616921897*v_R18_n_r - 24.9575997948692*v_R9_a_i + 23.284964480954*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r + 105.57176931318*v_R9_c_i - 54.0657727682604*v_R9_c_r - 39.6392229058201*v_R9_n_i + 10.2713736253514*v_R9_n_r
        struct[0].g[118,0] = i_R10_n_r + 8.42240745613237*v_R10_a_i + 48.8496051768258*v_R10_a_r + 22.4917262854545*v_R10_b_i + 33.965950515314*v_R10_b_r + 8.42240745613237*v_R10_c_i + 48.8496051768258*v_R10_c_r - 59.7203367834281*v_R10_n_i - 136.548516830407*v_R10_n_r + 1.84896616921897*v_R18_a_i - 9.21038227100566*v_R18_a_r + 0.793238195499527*v_R18_b_i - 9.00835072044485*v_R18_b_r + 1.84896616921897*v_R18_c_i - 9.21038227100566*v_R18_c_r + 5.65456401516768*v_R18_n_i + 30.9517475172273*v_R18_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r - 23.284964480954*v_R9_b_i - 24.9575997948692*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058202*v_R9_c_r + 54.0657727682604*v_R9_n_i + 105.57176931318*v_R9_n_r
        struct[0].g[119,0] = i_R10_n_i + 48.8496051768258*v_R10_a_i - 8.42240745613237*v_R10_a_r + 33.965950515314*v_R10_b_i - 22.4917262854545*v_R10_b_r + 48.8496051768258*v_R10_c_i - 8.42240745613237*v_R10_c_r - 136.548516830407*v_R10_n_i + 59.7203367834281*v_R10_n_r - 9.21038227100566*v_R18_a_i - 1.84896616921897*v_R18_a_r - 9.00835072044485*v_R18_b_i - 0.793238195499527*v_R18_b_r - 9.21038227100566*v_R18_c_i - 1.84896616921897*v_R18_c_r + 30.9517475172273*v_R18_n_i - 5.65456401516768*v_R18_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r - 24.9575997948692*v_R9_b_i + 23.284964480954*v_R9_b_r - 39.6392229058202*v_R9_c_i + 10.2713736253513*v_R9_c_r + 105.57176931318*v_R9_n_i - 54.0657727682604*v_R9_n_r
        struct[0].g[120,0] = i_R12_a_r - 9.69353831171601*v_R12_a_i - 53.0601386009611*v_R12_a_r - 3.16965629008966*v_R12_b_i + 15.7892267502954*v_R12_b_r - 1.35983690657062*v_R12_c_i + 15.442886949334*v_R12_c_r - 3.16965629008966*v_R12_n_i + 15.7892267502954*v_R12_n_r + 4.84676915585801*v_R13_a_i + 26.5300693004806*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 0.67991845328531*v_R13_c_i - 7.72144347466701*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r + 4.84676915585801*v_R4_a_i + 26.5300693004806*v_R4_a_r + 1.58482814504483*v_R4_b_i - 7.8946133751477*v_R4_b_r + 0.67991845328531*v_R4_c_i - 7.72144347466701*v_R4_c_r + 1.58482814504483*v_R4_n_i - 7.89461337514771*v_R4_n_r
        struct[0].g[121,0] = i_R12_a_i - 53.0601386009611*v_R12_a_i + 9.69353831171601*v_R12_a_r + 15.7892267502954*v_R12_b_i + 3.16965629008966*v_R12_b_r + 15.442886949334*v_R12_c_i + 1.35983690657062*v_R12_c_r + 15.7892267502954*v_R12_n_i + 3.16965629008966*v_R12_n_r + 26.5300693004806*v_R13_a_i - 4.84676915585801*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r - 7.72144347466701*v_R13_c_i - 0.67991845328531*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r + 26.5300693004806*v_R4_a_i - 4.84676915585801*v_R4_a_r - 7.8946133751477*v_R4_b_i - 1.58482814504483*v_R4_b_r - 7.72144347466701*v_R4_c_i - 0.67991845328531*v_R4_c_r - 7.89461337514771*v_R4_n_i - 1.58482814504483*v_R4_n_r
        struct[0].g[122,0] = i_R12_b_r - 3.16965629008966*v_R12_a_i + 15.7892267502954*v_R12_a_r - 9.69353831171602*v_R12_b_i - 53.0601386009611*v_R12_b_r - 3.16965629008967*v_R12_c_i + 15.7892267502954*v_R12_c_r - 1.35983690657062*v_R12_n_i + 15.442886949334*v_R12_n_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 4.84676915585801*v_R13_b_i + 26.5300693004806*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 0.679918453285308*v_R13_n_i - 7.72144347466701*v_R13_n_r + 1.58482814504483*v_R4_a_i - 7.89461337514771*v_R4_a_r + 4.84676915585801*v_R4_b_i + 26.5300693004806*v_R4_b_r + 1.58482814504483*v_R4_c_i - 7.89461337514771*v_R4_c_r + 0.679918453285308*v_R4_n_i - 7.72144347466701*v_R4_n_r
        struct[0].g[123,0] = i_R12_b_i + 15.7892267502954*v_R12_a_i + 3.16965629008966*v_R12_a_r - 53.0601386009611*v_R12_b_i + 9.69353831171602*v_R12_b_r + 15.7892267502954*v_R12_c_i + 3.16965629008967*v_R12_c_r + 15.442886949334*v_R12_n_i + 1.35983690657062*v_R12_n_r - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r + 26.5300693004806*v_R13_b_i - 4.84676915585801*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r - 7.72144347466701*v_R13_n_i - 0.679918453285308*v_R13_n_r - 7.89461337514771*v_R4_a_i - 1.58482814504483*v_R4_a_r + 26.5300693004806*v_R4_b_i - 4.84676915585801*v_R4_b_r - 7.89461337514771*v_R4_c_i - 1.58482814504483*v_R4_c_r - 7.72144347466701*v_R4_n_i - 0.679918453285308*v_R4_n_r
        struct[0].g[124,0] = i_R12_c_r - 1.35983690657062*v_R12_a_i + 15.442886949334*v_R12_a_r - 3.16965629008966*v_R12_b_i + 15.7892267502954*v_R12_b_r - 9.69353831171602*v_R12_c_i - 53.0601386009611*v_R12_c_r - 3.16965629008967*v_R12_n_i + 15.7892267502954*v_R12_n_r + 0.67991845328531*v_R13_a_i - 7.72144347466701*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 4.84676915585801*v_R13_c_i + 26.5300693004806*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r + 0.67991845328531*v_R4_a_i - 7.72144347466701*v_R4_a_r + 1.58482814504483*v_R4_b_i - 7.8946133751477*v_R4_b_r + 4.84676915585801*v_R4_c_i + 26.5300693004806*v_R4_c_r + 1.58482814504483*v_R4_n_i - 7.89461337514771*v_R4_n_r
        struct[0].g[125,0] = i_R12_c_i + 15.442886949334*v_R12_a_i + 1.35983690657062*v_R12_a_r + 15.7892267502954*v_R12_b_i + 3.16965629008966*v_R12_b_r - 53.0601386009611*v_R12_c_i + 9.69353831171602*v_R12_c_r + 15.7892267502954*v_R12_n_i + 3.16965629008967*v_R12_n_r - 7.72144347466701*v_R13_a_i - 0.67991845328531*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r + 26.5300693004806*v_R13_c_i - 4.84676915585801*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r - 7.72144347466701*v_R4_a_i - 0.67991845328531*v_R4_a_r - 7.8946133751477*v_R4_b_i - 1.58482814504483*v_R4_b_r + 26.5300693004806*v_R4_c_i - 4.84676915585801*v_R4_c_r - 7.89461337514771*v_R4_n_i - 1.58482814504483*v_R4_n_r
        struct[0].g[126,0] = i_R12_n_r - 3.16965629008966*v_R12_a_i + 15.7892267502954*v_R12_a_r - 1.35983690657062*v_R12_b_i + 15.442886949334*v_R12_b_r - 3.16965629008966*v_R12_c_i + 15.7892267502954*v_R12_c_r - 9.69353831171602*v_R12_n_i - 53.0601386009611*v_R12_n_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 0.679918453285308*v_R13_b_i - 7.72144347466701*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 4.84676915585801*v_R13_n_i + 26.5300693004806*v_R13_n_r + 1.58482814504483*v_R4_a_i - 7.89461337514771*v_R4_a_r + 0.679918453285308*v_R4_b_i - 7.72144347466701*v_R4_b_r + 1.58482814504483*v_R4_c_i - 7.89461337514771*v_R4_c_r + 4.84676915585801*v_R4_n_i + 26.5300693004806*v_R4_n_r
        struct[0].g[127,0] = i_R12_n_i + 15.7892267502954*v_R12_a_i + 3.16965629008966*v_R12_a_r + 15.442886949334*v_R12_b_i + 1.35983690657062*v_R12_b_r + 15.7892267502954*v_R12_c_i + 3.16965629008966*v_R12_c_r - 53.0601386009611*v_R12_n_i + 9.69353831171602*v_R12_n_r - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r - 7.72144347466701*v_R13_b_i - 0.679918453285308*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r + 26.5300693004806*v_R13_n_i - 4.84676915585801*v_R13_n_r - 7.89461337514771*v_R4_a_i - 1.58482814504483*v_R4_a_r - 7.72144347466701*v_R4_b_i - 0.679918453285308*v_R4_b_r - 7.89461337514771*v_R4_c_i - 1.58482814504483*v_R4_c_r + 26.5300693004806*v_R4_n_i - 4.84676915585801*v_R4_n_r
        struct[0].g[128,0] = i_R13_a_r + 4.84676915585801*v_R12_a_i + 26.5300693004806*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 0.67991845328531*v_R12_c_i - 7.72144347466701*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r - 9.69353831171601*v_R13_a_i - 53.0601386009611*v_R13_a_r - 3.16965629008966*v_R13_b_i + 15.7892267502954*v_R13_b_r - 1.35983690657062*v_R13_c_i + 15.442886949334*v_R13_c_r - 3.16965629008966*v_R13_n_i + 15.7892267502954*v_R13_n_r + 4.84676915585801*v_R14_a_i + 26.5300693004806*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 0.67991845328531*v_R14_c_i - 7.72144347466701*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r
        struct[0].g[129,0] = i_R13_a_i + 26.5300693004806*v_R12_a_i - 4.84676915585801*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r - 7.72144347466701*v_R12_c_i - 0.67991845328531*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r - 53.0601386009611*v_R13_a_i + 9.69353831171601*v_R13_a_r + 15.7892267502954*v_R13_b_i + 3.16965629008966*v_R13_b_r + 15.442886949334*v_R13_c_i + 1.35983690657062*v_R13_c_r + 15.7892267502954*v_R13_n_i + 3.16965629008966*v_R13_n_r + 26.5300693004806*v_R14_a_i - 4.84676915585801*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r - 7.72144347466701*v_R14_c_i - 0.67991845328531*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r
        struct[0].g[130,0] = i_R13_b_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 4.84676915585801*v_R12_b_i + 26.5300693004806*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 0.679918453285308*v_R12_n_i - 7.72144347466701*v_R12_n_r - 3.16965629008966*v_R13_a_i + 15.7892267502954*v_R13_a_r - 9.69353831171602*v_R13_b_i - 53.0601386009611*v_R13_b_r - 3.16965629008967*v_R13_c_i + 15.7892267502954*v_R13_c_r - 1.35983690657062*v_R13_n_i + 15.442886949334*v_R13_n_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 4.84676915585801*v_R14_b_i + 26.5300693004806*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 0.679918453285308*v_R14_n_i - 7.72144347466701*v_R14_n_r
        struct[0].g[131,0] = i_R13_b_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r + 26.5300693004806*v_R12_b_i - 4.84676915585801*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r - 7.72144347466701*v_R12_n_i - 0.679918453285308*v_R12_n_r + 15.7892267502954*v_R13_a_i + 3.16965629008966*v_R13_a_r - 53.0601386009611*v_R13_b_i + 9.69353831171602*v_R13_b_r + 15.7892267502954*v_R13_c_i + 3.16965629008967*v_R13_c_r + 15.442886949334*v_R13_n_i + 1.35983690657062*v_R13_n_r - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r + 26.5300693004806*v_R14_b_i - 4.84676915585801*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r - 7.72144347466701*v_R14_n_i - 0.679918453285308*v_R14_n_r
        struct[0].g[132,0] = i_R13_c_r + 0.67991845328531*v_R12_a_i - 7.72144347466701*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 4.84676915585801*v_R12_c_i + 26.5300693004806*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r - 1.35983690657062*v_R13_a_i + 15.442886949334*v_R13_a_r - 3.16965629008966*v_R13_b_i + 15.7892267502954*v_R13_b_r - 9.69353831171602*v_R13_c_i - 53.0601386009611*v_R13_c_r - 3.16965629008967*v_R13_n_i + 15.7892267502954*v_R13_n_r + 0.67991845328531*v_R14_a_i - 7.72144347466701*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 4.84676915585801*v_R14_c_i + 26.5300693004806*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r
        struct[0].g[133,0] = i_R13_c_i - 7.72144347466701*v_R12_a_i - 0.67991845328531*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r + 26.5300693004806*v_R12_c_i - 4.84676915585801*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r + 15.442886949334*v_R13_a_i + 1.35983690657062*v_R13_a_r + 15.7892267502954*v_R13_b_i + 3.16965629008966*v_R13_b_r - 53.0601386009611*v_R13_c_i + 9.69353831171602*v_R13_c_r + 15.7892267502954*v_R13_n_i + 3.16965629008967*v_R13_n_r - 7.72144347466701*v_R14_a_i - 0.67991845328531*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r + 26.5300693004806*v_R14_c_i - 4.84676915585801*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r
        struct[0].g[134,0] = i_R13_n_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 0.679918453285308*v_R12_b_i - 7.72144347466701*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 4.84676915585801*v_R12_n_i + 26.5300693004806*v_R12_n_r - 3.16965629008966*v_R13_a_i + 15.7892267502954*v_R13_a_r - 1.35983690657062*v_R13_b_i + 15.442886949334*v_R13_b_r - 3.16965629008966*v_R13_c_i + 15.7892267502954*v_R13_c_r - 9.69353831171602*v_R13_n_i - 53.0851386009611*v_R13_n_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 0.679918453285308*v_R14_b_i - 7.72144347466701*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 4.84676915585801*v_R14_n_i + 26.5300693004806*v_R14_n_r
        struct[0].g[135,0] = i_R13_n_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r - 7.72144347466701*v_R12_b_i - 0.679918453285308*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r + 26.5300693004806*v_R12_n_i - 4.84676915585801*v_R12_n_r + 15.7892267502954*v_R13_a_i + 3.16965629008966*v_R13_a_r + 15.442886949334*v_R13_b_i + 1.35983690657062*v_R13_b_r + 15.7892267502954*v_R13_c_i + 3.16965629008966*v_R13_c_r - 53.0851386009611*v_R13_n_i + 9.69353831171602*v_R13_n_r - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r - 7.72144347466701*v_R14_b_i - 0.679918453285308*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r + 26.5300693004806*v_R14_n_i - 4.84676915585801*v_R14_n_r
        struct[0].g[136,0] = i_R14_a_r + 4.84676915585801*v_R13_a_i + 26.5300693004806*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 0.67991845328531*v_R13_c_i - 7.72144347466701*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r - 9.69353831171601*v_R14_a_i - 53.0601386009611*v_R14_a_r - 3.16965629008966*v_R14_b_i + 15.7892267502954*v_R14_b_r - 1.35983690657062*v_R14_c_i + 15.442886949334*v_R14_c_r - 3.16965629008966*v_R14_n_i + 15.7892267502954*v_R14_n_r + 4.84676915585801*v_R15_a_i + 26.5300693004806*v_R15_a_r + 1.58482814504483*v_R15_b_i - 7.8946133751477*v_R15_b_r + 0.67991845328531*v_R15_c_i - 7.72144347466701*v_R15_c_r + 1.58482814504483*v_R15_n_i - 7.89461337514771*v_R15_n_r
        struct[0].g[137,0] = i_R14_a_i + 26.5300693004806*v_R13_a_i - 4.84676915585801*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r - 7.72144347466701*v_R13_c_i - 0.67991845328531*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r - 53.0601386009611*v_R14_a_i + 9.69353831171601*v_R14_a_r + 15.7892267502954*v_R14_b_i + 3.16965629008966*v_R14_b_r + 15.442886949334*v_R14_c_i + 1.35983690657062*v_R14_c_r + 15.7892267502954*v_R14_n_i + 3.16965629008966*v_R14_n_r + 26.5300693004806*v_R15_a_i - 4.84676915585801*v_R15_a_r - 7.8946133751477*v_R15_b_i - 1.58482814504483*v_R15_b_r - 7.72144347466701*v_R15_c_i - 0.67991845328531*v_R15_c_r - 7.89461337514771*v_R15_n_i - 1.58482814504483*v_R15_n_r
        struct[0].g[138,0] = i_R14_b_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 4.84676915585801*v_R13_b_i + 26.5300693004806*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 0.679918453285308*v_R13_n_i - 7.72144347466701*v_R13_n_r - 3.16965629008966*v_R14_a_i + 15.7892267502954*v_R14_a_r - 9.69353831171602*v_R14_b_i - 53.0601386009611*v_R14_b_r - 3.16965629008967*v_R14_c_i + 15.7892267502954*v_R14_c_r - 1.35983690657062*v_R14_n_i + 15.442886949334*v_R14_n_r + 1.58482814504483*v_R15_a_i - 7.89461337514771*v_R15_a_r + 4.84676915585801*v_R15_b_i + 26.5300693004806*v_R15_b_r + 1.58482814504483*v_R15_c_i - 7.89461337514771*v_R15_c_r + 0.679918453285308*v_R15_n_i - 7.72144347466701*v_R15_n_r
        struct[0].g[139,0] = i_R14_b_i - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r + 26.5300693004806*v_R13_b_i - 4.84676915585801*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r - 7.72144347466701*v_R13_n_i - 0.679918453285308*v_R13_n_r + 15.7892267502954*v_R14_a_i + 3.16965629008966*v_R14_a_r - 53.0601386009611*v_R14_b_i + 9.69353831171602*v_R14_b_r + 15.7892267502954*v_R14_c_i + 3.16965629008967*v_R14_c_r + 15.442886949334*v_R14_n_i + 1.35983690657062*v_R14_n_r - 7.89461337514771*v_R15_a_i - 1.58482814504483*v_R15_a_r + 26.5300693004806*v_R15_b_i - 4.84676915585801*v_R15_b_r - 7.89461337514771*v_R15_c_i - 1.58482814504483*v_R15_c_r - 7.72144347466701*v_R15_n_i - 0.679918453285308*v_R15_n_r
        struct[0].g[140,0] = i_R14_c_r + 0.67991845328531*v_R13_a_i - 7.72144347466701*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 4.84676915585801*v_R13_c_i + 26.5300693004806*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r - 1.35983690657062*v_R14_a_i + 15.442886949334*v_R14_a_r - 3.16965629008966*v_R14_b_i + 15.7892267502954*v_R14_b_r - 9.69353831171602*v_R14_c_i - 53.0601386009611*v_R14_c_r - 3.16965629008967*v_R14_n_i + 15.7892267502954*v_R14_n_r + 0.67991845328531*v_R15_a_i - 7.72144347466701*v_R15_a_r + 1.58482814504483*v_R15_b_i - 7.8946133751477*v_R15_b_r + 4.84676915585801*v_R15_c_i + 26.5300693004806*v_R15_c_r + 1.58482814504483*v_R15_n_i - 7.89461337514771*v_R15_n_r
        struct[0].g[141,0] = i_R14_c_i - 7.72144347466701*v_R13_a_i - 0.67991845328531*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r + 26.5300693004806*v_R13_c_i - 4.84676915585801*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r + 15.442886949334*v_R14_a_i + 1.35983690657062*v_R14_a_r + 15.7892267502954*v_R14_b_i + 3.16965629008966*v_R14_b_r - 53.0601386009611*v_R14_c_i + 9.69353831171602*v_R14_c_r + 15.7892267502954*v_R14_n_i + 3.16965629008967*v_R14_n_r - 7.72144347466701*v_R15_a_i - 0.67991845328531*v_R15_a_r - 7.8946133751477*v_R15_b_i - 1.58482814504483*v_R15_b_r + 26.5300693004806*v_R15_c_i - 4.84676915585801*v_R15_c_r - 7.89461337514771*v_R15_n_i - 1.58482814504483*v_R15_n_r
        struct[0].g[142,0] = i_R14_n_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 0.679918453285308*v_R13_b_i - 7.72144347466701*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 4.84676915585801*v_R13_n_i + 26.5300693004806*v_R13_n_r - 3.16965629008966*v_R14_a_i + 15.7892267502954*v_R14_a_r - 1.35983690657062*v_R14_b_i + 15.442886949334*v_R14_b_r - 3.16965629008966*v_R14_c_i + 15.7892267502954*v_R14_c_r - 9.69353831171602*v_R14_n_i - 53.0601386009611*v_R14_n_r + 1.58482814504483*v_R15_a_i - 7.89461337514771*v_R15_a_r + 0.679918453285308*v_R15_b_i - 7.72144347466701*v_R15_b_r + 1.58482814504483*v_R15_c_i - 7.89461337514771*v_R15_c_r + 4.84676915585801*v_R15_n_i + 26.5300693004806*v_R15_n_r
        struct[0].g[143,0] = i_R14_n_i - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r - 7.72144347466701*v_R13_b_i - 0.679918453285308*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r + 26.5300693004806*v_R13_n_i - 4.84676915585801*v_R13_n_r + 15.7892267502954*v_R14_a_i + 3.16965629008966*v_R14_a_r + 15.442886949334*v_R14_b_i + 1.35983690657062*v_R14_b_r + 15.7892267502954*v_R14_c_i + 3.16965629008966*v_R14_c_r - 53.0601386009611*v_R14_n_i + 9.69353831171602*v_R14_n_r - 7.89461337514771*v_R15_a_i - 1.58482814504483*v_R15_a_r - 7.72144347466701*v_R15_b_i - 0.679918453285308*v_R15_b_r - 7.89461337514771*v_R15_c_i - 1.58482814504483*v_R15_c_r + 26.5300693004806*v_R15_n_i - 4.84676915585801*v_R15_n_r
        struct[0].g[144,0] = i_R1_a_i*v_R1_a_i - i_R1_a_i*v_R1_n_i + i_R1_a_r*v_R1_a_r - i_R1_a_r*v_R1_n_r - p_R1_a
        struct[0].g[145,0] = i_R1_b_i*v_R1_b_i - i_R1_b_i*v_R1_n_i + i_R1_b_r*v_R1_b_r - i_R1_b_r*v_R1_n_r - p_R1_b
        struct[0].g[146,0] = i_R1_c_i*v_R1_c_i - i_R1_c_i*v_R1_n_i + i_R1_c_r*v_R1_c_r - i_R1_c_r*v_R1_n_r - p_R1_c
        struct[0].g[147,0] = -i_R1_a_i*v_R1_a_r + i_R1_a_i*v_R1_n_r + i_R1_a_r*v_R1_a_i - i_R1_a_r*v_R1_n_i - q_R1_a
        struct[0].g[148,0] = -i_R1_b_i*v_R1_b_r + i_R1_b_i*v_R1_n_r + i_R1_b_r*v_R1_b_i - i_R1_b_r*v_R1_n_i - q_R1_b
        struct[0].g[149,0] = -i_R1_c_i*v_R1_c_r + i_R1_c_i*v_R1_n_r + i_R1_c_r*v_R1_c_i - i_R1_c_r*v_R1_n_i - q_R1_c
        struct[0].g[150,0] = i_R1_a_r + i_R1_b_r + i_R1_c_r + i_R1_n_r
        struct[0].g[151,0] = i_R1_a_i + i_R1_b_i + i_R1_c_i + i_R1_n_i
        struct[0].g[152,0] = i_R11_a_i*v_R11_a_i - i_R11_a_i*v_R11_n_i + i_R11_a_r*v_R11_a_r - i_R11_a_r*v_R11_n_r - p_R11_a
        struct[0].g[153,0] = i_R11_b_i*v_R11_b_i - i_R11_b_i*v_R11_n_i + i_R11_b_r*v_R11_b_r - i_R11_b_r*v_R11_n_r - p_R11_b
        struct[0].g[154,0] = i_R11_c_i*v_R11_c_i - i_R11_c_i*v_R11_n_i + i_R11_c_r*v_R11_c_r - i_R11_c_r*v_R11_n_r - p_R11_c
        struct[0].g[155,0] = -i_R11_a_i*v_R11_a_r + i_R11_a_i*v_R11_n_r + i_R11_a_r*v_R11_a_i - i_R11_a_r*v_R11_n_i - q_R11_a
        struct[0].g[156,0] = -i_R11_b_i*v_R11_b_r + i_R11_b_i*v_R11_n_r + i_R11_b_r*v_R11_b_i - i_R11_b_r*v_R11_n_i - q_R11_b
        struct[0].g[157,0] = -i_R11_c_i*v_R11_c_r + i_R11_c_i*v_R11_n_r + i_R11_c_r*v_R11_c_i - i_R11_c_r*v_R11_n_i - q_R11_c
        struct[0].g[158,0] = i_R11_a_r + i_R11_b_r + i_R11_c_r + i_R11_n_r
        struct[0].g[159,0] = i_R11_a_i + i_R11_b_i + i_R11_c_i + i_R11_n_i
        struct[0].g[160,0] = i_R15_a_i*v_R15_a_i - i_R15_a_i*v_R15_n_i + i_R15_a_r*v_R15_a_r - i_R15_a_r*v_R15_n_r - p_R15_a
        struct[0].g[161,0] = i_R15_b_i*v_R15_b_i - i_R15_b_i*v_R15_n_i + i_R15_b_r*v_R15_b_r - i_R15_b_r*v_R15_n_r - p_R15_b
        struct[0].g[162,0] = i_R15_c_i*v_R15_c_i - i_R15_c_i*v_R15_n_i + i_R15_c_r*v_R15_c_r - i_R15_c_r*v_R15_n_r - p_R15_c
        struct[0].g[163,0] = -i_R15_a_i*v_R15_a_r + i_R15_a_i*v_R15_n_r + i_R15_a_r*v_R15_a_i - i_R15_a_r*v_R15_n_i - q_R15_a
        struct[0].g[164,0] = -i_R15_b_i*v_R15_b_r + i_R15_b_i*v_R15_n_r + i_R15_b_r*v_R15_b_i - i_R15_b_r*v_R15_n_i - q_R15_b
        struct[0].g[165,0] = -i_R15_c_i*v_R15_c_r + i_R15_c_i*v_R15_n_r + i_R15_c_r*v_R15_c_i - i_R15_c_r*v_R15_n_i - q_R15_c
        struct[0].g[166,0] = i_R15_a_r + i_R15_b_r + i_R15_c_r + i_R15_n_r
        struct[0].g[167,0] = i_R15_a_i + i_R15_b_i + i_R15_c_i + i_R15_n_i
        struct[0].g[168,0] = i_R16_a_i*v_R16_a_i - i_R16_a_i*v_R16_n_i + i_R16_a_r*v_R16_a_r - i_R16_a_r*v_R16_n_r - p_R16_a
        struct[0].g[169,0] = i_R16_b_i*v_R16_b_i - i_R16_b_i*v_R16_n_i + i_R16_b_r*v_R16_b_r - i_R16_b_r*v_R16_n_r - p_R16_b
        struct[0].g[170,0] = i_R16_c_i*v_R16_c_i - i_R16_c_i*v_R16_n_i + i_R16_c_r*v_R16_c_r - i_R16_c_r*v_R16_n_r - p_R16_c
        struct[0].g[171,0] = -i_R16_a_i*v_R16_a_r + i_R16_a_i*v_R16_n_r + i_R16_a_r*v_R16_a_i - i_R16_a_r*v_R16_n_i - q_R16_a
        struct[0].g[172,0] = -i_R16_b_i*v_R16_b_r + i_R16_b_i*v_R16_n_r + i_R16_b_r*v_R16_b_i - i_R16_b_r*v_R16_n_i - q_R16_b
        struct[0].g[173,0] = -i_R16_c_i*v_R16_c_r + i_R16_c_i*v_R16_n_r + i_R16_c_r*v_R16_c_i - i_R16_c_r*v_R16_n_i - q_R16_c
        struct[0].g[174,0] = i_R16_a_r + i_R16_b_r + i_R16_c_r + i_R16_n_r
        struct[0].g[175,0] = i_R16_a_i + i_R16_b_i + i_R16_c_i + i_R16_n_i
        struct[0].g[176,0] = i_R17_a_i*v_R17_a_i - i_R17_a_i*v_R17_n_i + i_R17_a_r*v_R17_a_r - i_R17_a_r*v_R17_n_r - p_R17_a
        struct[0].g[177,0] = i_R17_b_i*v_R17_b_i - i_R17_b_i*v_R17_n_i + i_R17_b_r*v_R17_b_r - i_R17_b_r*v_R17_n_r - p_R17_b
        struct[0].g[178,0] = i_R17_c_i*v_R17_c_i - i_R17_c_i*v_R17_n_i + i_R17_c_r*v_R17_c_r - i_R17_c_r*v_R17_n_r - p_R17_c
        struct[0].g[179,0] = -i_R17_a_i*v_R17_a_r + i_R17_a_i*v_R17_n_r + i_R17_a_r*v_R17_a_i - i_R17_a_r*v_R17_n_i - q_R17_a
        struct[0].g[180,0] = -i_R17_b_i*v_R17_b_r + i_R17_b_i*v_R17_n_r + i_R17_b_r*v_R17_b_i - i_R17_b_r*v_R17_n_i - q_R17_b
        struct[0].g[181,0] = -i_R17_c_i*v_R17_c_r + i_R17_c_i*v_R17_n_r + i_R17_c_r*v_R17_c_i - i_R17_c_r*v_R17_n_i - q_R17_c
        struct[0].g[182,0] = i_R17_a_r + i_R17_b_r + i_R17_c_r + i_R17_n_r
        struct[0].g[183,0] = i_R17_a_i + i_R17_b_i + i_R17_c_i + i_R17_n_i
        struct[0].g[184,0] = i_R18_a_i*v_R18_a_i - i_R18_a_i*v_R18_n_i + i_R18_a_r*v_R18_a_r - i_R18_a_r*v_R18_n_r - p_R18_a
        struct[0].g[185,0] = i_R18_b_i*v_R18_b_i - i_R18_b_i*v_R18_n_i + i_R18_b_r*v_R18_b_r - i_R18_b_r*v_R18_n_r - p_R18_b
        struct[0].g[186,0] = i_R18_c_i*v_R18_c_i - i_R18_c_i*v_R18_n_i + i_R18_c_r*v_R18_c_r - i_R18_c_r*v_R18_n_r - p_R18_c
        struct[0].g[187,0] = -i_R18_a_i*v_R18_a_r + i_R18_a_i*v_R18_n_r + i_R18_a_r*v_R18_a_i - i_R18_a_r*v_R18_n_i - q_R18_a
        struct[0].g[188,0] = -i_R18_b_i*v_R18_b_r + i_R18_b_i*v_R18_n_r + i_R18_b_r*v_R18_b_i - i_R18_b_r*v_R18_n_i - q_R18_b
        struct[0].g[189,0] = -i_R18_c_i*v_R18_c_r + i_R18_c_i*v_R18_n_r + i_R18_c_r*v_R18_c_i - i_R18_c_r*v_R18_n_i - q_R18_c
        struct[0].g[190,0] = i_R18_a_r + i_R18_b_r + i_R18_c_r + i_R18_n_r
        struct[0].g[191,0] = i_R18_a_i + i_R18_b_i + i_R18_c_i + i_R18_n_i
    
    # Outputs:
    if mode == 3:

        struct[0].h[0,0] = (v_R0_a_i**2 + v_R0_a_r**2)**0.5
        struct[0].h[1,0] = (v_R0_b_i**2 + v_R0_b_r**2)**0.5
        struct[0].h[2,0] = (v_R0_c_i**2 + v_R0_c_r**2)**0.5
        struct[0].h[3,0] = (v_R1_a_i**2 + v_R1_a_r**2)**0.5
        struct[0].h[4,0] = (v_R1_b_i**2 + v_R1_b_r**2)**0.5
        struct[0].h[5,0] = (v_R1_c_i**2 + v_R1_c_r**2)**0.5
        struct[0].h[6,0] = (v_R1_n_i**2 + v_R1_n_r**2)**0.5
        struct[0].h[7,0] = (v_R11_a_i**2 + v_R11_a_r**2)**0.5
        struct[0].h[8,0] = (v_R11_b_i**2 + v_R11_b_r**2)**0.5
        struct[0].h[9,0] = (v_R11_c_i**2 + v_R11_c_r**2)**0.5
        struct[0].h[10,0] = (v_R11_n_i**2 + v_R11_n_r**2)**0.5
        struct[0].h[11,0] = (v_R15_a_i**2 + v_R15_a_r**2)**0.5
        struct[0].h[12,0] = (v_R15_b_i**2 + v_R15_b_r**2)**0.5
        struct[0].h[13,0] = (v_R15_c_i**2 + v_R15_c_r**2)**0.5
        struct[0].h[14,0] = (v_R15_n_i**2 + v_R15_n_r**2)**0.5
        struct[0].h[15,0] = (v_R16_a_i**2 + v_R16_a_r**2)**0.5
        struct[0].h[16,0] = (v_R16_b_i**2 + v_R16_b_r**2)**0.5
        struct[0].h[17,0] = (v_R16_c_i**2 + v_R16_c_r**2)**0.5
        struct[0].h[18,0] = (v_R16_n_i**2 + v_R16_n_r**2)**0.5
        struct[0].h[19,0] = (v_R17_a_i**2 + v_R17_a_r**2)**0.5
        struct[0].h[20,0] = (v_R17_b_i**2 + v_R17_b_r**2)**0.5
        struct[0].h[21,0] = (v_R17_c_i**2 + v_R17_c_r**2)**0.5
        struct[0].h[22,0] = (v_R17_n_i**2 + v_R17_n_r**2)**0.5
        struct[0].h[23,0] = (v_R18_a_i**2 + v_R18_a_r**2)**0.5
        struct[0].h[24,0] = (v_R18_b_i**2 + v_R18_b_r**2)**0.5
        struct[0].h[25,0] = (v_R18_c_i**2 + v_R18_c_r**2)**0.5
        struct[0].h[26,0] = (v_R18_n_i**2 + v_R18_n_r**2)**0.5
        struct[0].h[27,0] = (v_R2_a_i**2 + v_R2_a_r**2)**0.5
        struct[0].h[28,0] = (v_R2_b_i**2 + v_R2_b_r**2)**0.5
        struct[0].h[29,0] = (v_R2_c_i**2 + v_R2_c_r**2)**0.5
        struct[0].h[30,0] = (v_R2_n_i**2 + v_R2_n_r**2)**0.5
        struct[0].h[31,0] = (v_R3_a_i**2 + v_R3_a_r**2)**0.5
        struct[0].h[32,0] = (v_R3_b_i**2 + v_R3_b_r**2)**0.5
        struct[0].h[33,0] = (v_R3_c_i**2 + v_R3_c_r**2)**0.5
        struct[0].h[34,0] = (v_R3_n_i**2 + v_R3_n_r**2)**0.5
        struct[0].h[35,0] = (v_R4_a_i**2 + v_R4_a_r**2)**0.5
        struct[0].h[36,0] = (v_R4_b_i**2 + v_R4_b_r**2)**0.5
        struct[0].h[37,0] = (v_R4_c_i**2 + v_R4_c_r**2)**0.5
        struct[0].h[38,0] = (v_R4_n_i**2 + v_R4_n_r**2)**0.5
        struct[0].h[39,0] = (v_R5_a_i**2 + v_R5_a_r**2)**0.5
        struct[0].h[40,0] = (v_R5_b_i**2 + v_R5_b_r**2)**0.5
        struct[0].h[41,0] = (v_R5_c_i**2 + v_R5_c_r**2)**0.5
        struct[0].h[42,0] = (v_R5_n_i**2 + v_R5_n_r**2)**0.5
        struct[0].h[43,0] = (v_R6_a_i**2 + v_R6_a_r**2)**0.5
        struct[0].h[44,0] = (v_R6_b_i**2 + v_R6_b_r**2)**0.5
        struct[0].h[45,0] = (v_R6_c_i**2 + v_R6_c_r**2)**0.5
        struct[0].h[46,0] = (v_R6_n_i**2 + v_R6_n_r**2)**0.5
        struct[0].h[47,0] = (v_R7_a_i**2 + v_R7_a_r**2)**0.5
        struct[0].h[48,0] = (v_R7_b_i**2 + v_R7_b_r**2)**0.5
        struct[0].h[49,0] = (v_R7_c_i**2 + v_R7_c_r**2)**0.5
        struct[0].h[50,0] = (v_R7_n_i**2 + v_R7_n_r**2)**0.5
        struct[0].h[51,0] = (v_R8_a_i**2 + v_R8_a_r**2)**0.5
        struct[0].h[52,0] = (v_R8_b_i**2 + v_R8_b_r**2)**0.5
        struct[0].h[53,0] = (v_R8_c_i**2 + v_R8_c_r**2)**0.5
        struct[0].h[54,0] = (v_R8_n_i**2 + v_R8_n_r**2)**0.5
        struct[0].h[55,0] = (v_R9_a_i**2 + v_R9_a_r**2)**0.5
        struct[0].h[56,0] = (v_R9_b_i**2 + v_R9_b_r**2)**0.5
        struct[0].h[57,0] = (v_R9_c_i**2 + v_R9_c_r**2)**0.5
        struct[0].h[58,0] = (v_R9_n_i**2 + v_R9_n_r**2)**0.5
        struct[0].h[59,0] = (v_R10_a_i**2 + v_R10_a_r**2)**0.5
        struct[0].h[60,0] = (v_R10_b_i**2 + v_R10_b_r**2)**0.5
        struct[0].h[61,0] = (v_R10_c_i**2 + v_R10_c_r**2)**0.5
        struct[0].h[62,0] = (v_R10_n_i**2 + v_R10_n_r**2)**0.5
        struct[0].h[63,0] = (v_R12_a_i**2 + v_R12_a_r**2)**0.5
        struct[0].h[64,0] = (v_R12_b_i**2 + v_R12_b_r**2)**0.5
        struct[0].h[65,0] = (v_R12_c_i**2 + v_R12_c_r**2)**0.5
        struct[0].h[66,0] = (v_R12_n_i**2 + v_R12_n_r**2)**0.5
        struct[0].h[67,0] = (v_R13_a_i**2 + v_R13_a_r**2)**0.5
        struct[0].h[68,0] = (v_R13_b_i**2 + v_R13_b_r**2)**0.5
        struct[0].h[69,0] = (v_R13_c_i**2 + v_R13_c_r**2)**0.5
        struct[0].h[70,0] = (v_R13_n_i**2 + v_R13_n_r**2)**0.5
        struct[0].h[71,0] = (v_R14_a_i**2 + v_R14_a_r**2)**0.5
        struct[0].h[72,0] = (v_R14_b_i**2 + v_R14_b_r**2)**0.5
        struct[0].h[73,0] = (v_R14_c_i**2 + v_R14_c_r**2)**0.5
        struct[0].h[74,0] = (v_R14_n_i**2 + v_R14_n_r**2)**0.5
    

    if mode == 10:

        struct[0].Fx_ini[0,0] = -1

    if mode == 11:


        struct[0].Gy_ini[0,0] = -123.954122254357
        struct[0].Gy_ini[0,1] = -127.595184532966
        struct[0].Gy_ini[0,2] = 39.6392229058202
        struct[0].Gy_ini[0,3] = 10.2713736253513
        struct[0].Gy_ini[0,4] = 24.9575997948692
        struct[0].Gy_ini[0,5] = 23.2849644809540
        struct[0].Gy_ini[0,6] = 58.0215758469966
        struct[0].Gy_ini[0,7] = 83.8007853900572
        struct[0].Gy_ini[0,48] = 105.571769313180
        struct[0].Gy_ini[0,49] = 54.0657727682604
        struct[0].Gy_ini[0,50] = -39.6392229058202
        struct[0].Gy_ini[0,51] = -10.2713736253513
        struct[0].Gy_ini[0,52] = -24.9575997948692
        struct[0].Gy_ini[0,53] = -23.2849644809540
        struct[0].Gy_ini[0,54] = -39.6392229058202
        struct[0].Gy_ini[0,55] = -10.2713736253513
        struct[0].Gy_ini[0,144] = 1
        struct[0].Gy_ini[1,0] = 127.595184532966
        struct[0].Gy_ini[1,1] = -123.954122254357
        struct[0].Gy_ini[1,2] = -10.2713736253513
        struct[0].Gy_ini[1,3] = 39.6392229058202
        struct[0].Gy_ini[1,4] = -23.2849644809540
        struct[0].Gy_ini[1,5] = 24.9575997948692
        struct[0].Gy_ini[1,6] = -83.8007853900572
        struct[0].Gy_ini[1,7] = 58.0215758469966
        struct[0].Gy_ini[1,48] = -54.0657727682604
        struct[0].Gy_ini[1,49] = 105.571769313180
        struct[0].Gy_ini[1,50] = 10.2713736253513
        struct[0].Gy_ini[1,51] = -39.6392229058202
        struct[0].Gy_ini[1,52] = 23.2849644809540
        struct[0].Gy_ini[1,53] = -24.9575997948692
        struct[0].Gy_ini[1,54] = 10.2713736253513
        struct[0].Gy_ini[1,55] = -39.6392229058202
        struct[0].Gy_ini[1,145] = 1
        struct[0].Gy_ini[2,0] = 39.6392229058202
        struct[0].Gy_ini[2,1] = 10.2713736253513
        struct[0].Gy_ini[2,2] = -123.954122254357
        struct[0].Gy_ini[2,3] = -127.595184532966
        struct[0].Gy_ini[2,4] = 39.6392229058201
        struct[0].Gy_ini[2,5] = 10.2713736253513
        struct[0].Gy_ini[2,6] = 43.3399527360457
        struct[0].Gy_ini[2,7] = 96.8143762456598
        struct[0].Gy_ini[2,48] = -39.6392229058202
        struct[0].Gy_ini[2,49] = -10.2713736253513
        struct[0].Gy_ini[2,50] = 105.571769313180
        struct[0].Gy_ini[2,51] = 54.0657727682604
        struct[0].Gy_ini[2,52] = -39.6392229058201
        struct[0].Gy_ini[2,53] = -10.2713736253513
        struct[0].Gy_ini[2,54] = -24.9575997948692
        struct[0].Gy_ini[2,55] = -23.2849644809540
        struct[0].Gy_ini[2,146] = 1
        struct[0].Gy_ini[3,0] = -10.2713736253513
        struct[0].Gy_ini[3,1] = 39.6392229058202
        struct[0].Gy_ini[3,2] = 127.595184532966
        struct[0].Gy_ini[3,3] = -123.954122254357
        struct[0].Gy_ini[3,4] = -10.2713736253513
        struct[0].Gy_ini[3,5] = 39.6392229058201
        struct[0].Gy_ini[3,6] = -96.8143762456598
        struct[0].Gy_ini[3,7] = 43.3399527360457
        struct[0].Gy_ini[3,48] = 10.2713736253513
        struct[0].Gy_ini[3,49] = -39.6392229058202
        struct[0].Gy_ini[3,50] = -54.0657727682604
        struct[0].Gy_ini[3,51] = 105.571769313180
        struct[0].Gy_ini[3,52] = 10.2713736253513
        struct[0].Gy_ini[3,53] = -39.6392229058201
        struct[0].Gy_ini[3,54] = 23.2849644809540
        struct[0].Gy_ini[3,55] = -24.9575997948692
        struct[0].Gy_ini[3,147] = 1
        struct[0].Gy_ini[4,0] = 24.9575997948692
        struct[0].Gy_ini[4,1] = 23.2849644809540
        struct[0].Gy_ini[4,2] = 39.6392229058202
        struct[0].Gy_ini[4,3] = 10.2713736253513
        struct[0].Gy_ini[4,4] = -123.954122254357
        struct[0].Gy_ini[4,5] = -127.595184532966
        struct[0].Gy_ini[4,6] = 58.0215758469966
        struct[0].Gy_ini[4,7] = 83.8007853900572
        struct[0].Gy_ini[4,48] = -24.9575997948692
        struct[0].Gy_ini[4,49] = -23.2849644809540
        struct[0].Gy_ini[4,50] = -39.6392229058202
        struct[0].Gy_ini[4,51] = -10.2713736253513
        struct[0].Gy_ini[4,52] = 105.571769313180
        struct[0].Gy_ini[4,53] = 54.0657727682604
        struct[0].Gy_ini[4,54] = -39.6392229058201
        struct[0].Gy_ini[4,55] = -10.2713736253514
        struct[0].Gy_ini[4,148] = 1
        struct[0].Gy_ini[5,0] = -23.2849644809540
        struct[0].Gy_ini[5,1] = 24.9575997948692
        struct[0].Gy_ini[5,2] = -10.2713736253513
        struct[0].Gy_ini[5,3] = 39.6392229058202
        struct[0].Gy_ini[5,4] = 127.595184532966
        struct[0].Gy_ini[5,5] = -123.954122254357
        struct[0].Gy_ini[5,6] = -83.8007853900572
        struct[0].Gy_ini[5,7] = 58.0215758469966
        struct[0].Gy_ini[5,48] = 23.2849644809540
        struct[0].Gy_ini[5,49] = -24.9575997948692
        struct[0].Gy_ini[5,50] = 10.2713736253513
        struct[0].Gy_ini[5,51] = -39.6392229058202
        struct[0].Gy_ini[5,52] = -54.0657727682604
        struct[0].Gy_ini[5,53] = 105.571769313180
        struct[0].Gy_ini[5,54] = 10.2713736253514
        struct[0].Gy_ini[5,55] = -39.6392229058201
        struct[0].Gy_ini[5,149] = 1
        struct[0].Gy_ini[6,0] = 58.0215758469966
        struct[0].Gy_ini[6,1] = 83.8007853900572
        struct[0].Gy_ini[6,2] = 43.3399527360457
        struct[0].Gy_ini[6,3] = 96.8143762456598
        struct[0].Gy_ini[6,4] = 58.0215758469966
        struct[0].Gy_ini[6,5] = 83.8007853900572
        struct[0].Gy_ini[6,6] = -161.052161470043
        struct[0].Gy_ini[6,7] = -274.654008062378
        struct[0].Gy_ini[6,48] = -39.6392229058202
        struct[0].Gy_ini[6,49] = -10.2713736253513
        struct[0].Gy_ini[6,50] = -24.9575997948692
        struct[0].Gy_ini[6,51] = -23.2849644809540
        struct[0].Gy_ini[6,52] = -39.6392229058202
        struct[0].Gy_ini[6,53] = -10.2713736253513
        struct[0].Gy_ini[6,54] = 105.571769313180
        struct[0].Gy_ini[6,55] = 54.0657727682604
        struct[0].Gy_ini[6,150] = 1
        struct[0].Gy_ini[7,0] = -83.8007853900572
        struct[0].Gy_ini[7,1] = 58.0215758469966
        struct[0].Gy_ini[7,2] = -96.8143762456598
        struct[0].Gy_ini[7,3] = 43.3399527360457
        struct[0].Gy_ini[7,4] = -83.8007853900572
        struct[0].Gy_ini[7,5] = 58.0215758469966
        struct[0].Gy_ini[7,6] = 274.654008062378
        struct[0].Gy_ini[7,7] = -161.052161470043
        struct[0].Gy_ini[7,48] = 10.2713736253513
        struct[0].Gy_ini[7,49] = -39.6392229058202
        struct[0].Gy_ini[7,50] = 23.2849644809540
        struct[0].Gy_ini[7,51] = -24.9575997948692
        struct[0].Gy_ini[7,52] = 10.2713736253513
        struct[0].Gy_ini[7,53] = -39.6392229058202
        struct[0].Gy_ini[7,54] = -54.0657727682604
        struct[0].Gy_ini[7,55] = 105.571769313180
        struct[0].Gy_ini[7,151] = 1
        struct[0].Gy_ini[8,8] = -26.5300693004806
        struct[0].Gy_ini[8,9] = -4.84676915585801
        struct[0].Gy_ini[8,10] = 7.89461337514770
        struct[0].Gy_ini[8,11] = -1.58482814504483
        struct[0].Gy_ini[8,12] = 7.72144347466701
        struct[0].Gy_ini[8,13] = -0.679918453285310
        struct[0].Gy_ini[8,14] = 7.89461337514771
        struct[0].Gy_ini[8,15] = -1.58482814504483
        struct[0].Gy_ini[8,56] = 26.5300693004806
        struct[0].Gy_ini[8,57] = 4.84676915585801
        struct[0].Gy_ini[8,58] = -7.89461337514770
        struct[0].Gy_ini[8,59] = 1.58482814504483
        struct[0].Gy_ini[8,60] = -7.72144347466701
        struct[0].Gy_ini[8,61] = 0.679918453285310
        struct[0].Gy_ini[8,62] = -7.89461337514771
        struct[0].Gy_ini[8,63] = 1.58482814504483
        struct[0].Gy_ini[8,152] = 1
        struct[0].Gy_ini[9,8] = 4.84676915585801
        struct[0].Gy_ini[9,9] = -26.5300693004806
        struct[0].Gy_ini[9,10] = 1.58482814504483
        struct[0].Gy_ini[9,11] = 7.89461337514770
        struct[0].Gy_ini[9,12] = 0.679918453285310
        struct[0].Gy_ini[9,13] = 7.72144347466701
        struct[0].Gy_ini[9,14] = 1.58482814504483
        struct[0].Gy_ini[9,15] = 7.89461337514771
        struct[0].Gy_ini[9,56] = -4.84676915585801
        struct[0].Gy_ini[9,57] = 26.5300693004806
        struct[0].Gy_ini[9,58] = -1.58482814504483
        struct[0].Gy_ini[9,59] = -7.89461337514770
        struct[0].Gy_ini[9,60] = -0.679918453285310
        struct[0].Gy_ini[9,61] = -7.72144347466701
        struct[0].Gy_ini[9,62] = -1.58482814504483
        struct[0].Gy_ini[9,63] = -7.89461337514771
        struct[0].Gy_ini[9,153] = 1
        struct[0].Gy_ini[10,8] = 7.89461337514771
        struct[0].Gy_ini[10,9] = -1.58482814504483
        struct[0].Gy_ini[10,10] = -26.5300693004806
        struct[0].Gy_ini[10,11] = -4.84676915585801
        struct[0].Gy_ini[10,12] = 7.89461337514771
        struct[0].Gy_ini[10,13] = -1.58482814504483
        struct[0].Gy_ini[10,14] = 7.72144347466701
        struct[0].Gy_ini[10,15] = -0.679918453285308
        struct[0].Gy_ini[10,56] = -7.89461337514771
        struct[0].Gy_ini[10,57] = 1.58482814504483
        struct[0].Gy_ini[10,58] = 26.5300693004806
        struct[0].Gy_ini[10,59] = 4.84676915585801
        struct[0].Gy_ini[10,60] = -7.89461337514771
        struct[0].Gy_ini[10,61] = 1.58482814504483
        struct[0].Gy_ini[10,62] = -7.72144347466701
        struct[0].Gy_ini[10,63] = 0.679918453285308
        struct[0].Gy_ini[10,154] = 1
        struct[0].Gy_ini[11,8] = 1.58482814504483
        struct[0].Gy_ini[11,9] = 7.89461337514771
        struct[0].Gy_ini[11,10] = 4.84676915585801
        struct[0].Gy_ini[11,11] = -26.5300693004806
        struct[0].Gy_ini[11,12] = 1.58482814504483
        struct[0].Gy_ini[11,13] = 7.89461337514771
        struct[0].Gy_ini[11,14] = 0.679918453285308
        struct[0].Gy_ini[11,15] = 7.72144347466701
        struct[0].Gy_ini[11,56] = -1.58482814504483
        struct[0].Gy_ini[11,57] = -7.89461337514771
        struct[0].Gy_ini[11,58] = -4.84676915585801
        struct[0].Gy_ini[11,59] = 26.5300693004806
        struct[0].Gy_ini[11,60] = -1.58482814504483
        struct[0].Gy_ini[11,61] = -7.89461337514771
        struct[0].Gy_ini[11,62] = -0.679918453285308
        struct[0].Gy_ini[11,63] = -7.72144347466701
        struct[0].Gy_ini[11,155] = 1
        struct[0].Gy_ini[12,8] = 7.72144347466701
        struct[0].Gy_ini[12,9] = -0.679918453285310
        struct[0].Gy_ini[12,10] = 7.89461337514770
        struct[0].Gy_ini[12,11] = -1.58482814504483
        struct[0].Gy_ini[12,12] = -26.5300693004806
        struct[0].Gy_ini[12,13] = -4.84676915585801
        struct[0].Gy_ini[12,14] = 7.89461337514771
        struct[0].Gy_ini[12,15] = -1.58482814504483
        struct[0].Gy_ini[12,56] = -7.72144347466701
        struct[0].Gy_ini[12,57] = 0.679918453285310
        struct[0].Gy_ini[12,58] = -7.89461337514770
        struct[0].Gy_ini[12,59] = 1.58482814504483
        struct[0].Gy_ini[12,60] = 26.5300693004806
        struct[0].Gy_ini[12,61] = 4.84676915585801
        struct[0].Gy_ini[12,62] = -7.89461337514771
        struct[0].Gy_ini[12,63] = 1.58482814504483
        struct[0].Gy_ini[12,156] = 1
        struct[0].Gy_ini[13,8] = 0.679918453285310
        struct[0].Gy_ini[13,9] = 7.72144347466701
        struct[0].Gy_ini[13,10] = 1.58482814504483
        struct[0].Gy_ini[13,11] = 7.89461337514770
        struct[0].Gy_ini[13,12] = 4.84676915585801
        struct[0].Gy_ini[13,13] = -26.5300693004806
        struct[0].Gy_ini[13,14] = 1.58482814504483
        struct[0].Gy_ini[13,15] = 7.89461337514771
        struct[0].Gy_ini[13,56] = -0.679918453285310
        struct[0].Gy_ini[13,57] = -7.72144347466701
        struct[0].Gy_ini[13,58] = -1.58482814504483
        struct[0].Gy_ini[13,59] = -7.89461337514770
        struct[0].Gy_ini[13,60] = -4.84676915585801
        struct[0].Gy_ini[13,61] = 26.5300693004806
        struct[0].Gy_ini[13,62] = -1.58482814504483
        struct[0].Gy_ini[13,63] = -7.89461337514771
        struct[0].Gy_ini[13,157] = 1
        struct[0].Gy_ini[14,8] = 7.89461337514771
        struct[0].Gy_ini[14,9] = -1.58482814504483
        struct[0].Gy_ini[14,10] = 7.72144347466701
        struct[0].Gy_ini[14,11] = -0.679918453285308
        struct[0].Gy_ini[14,12] = 7.89461337514771
        struct[0].Gy_ini[14,13] = -1.58482814504483
        struct[0].Gy_ini[14,14] = -26.5550693004806
        struct[0].Gy_ini[14,15] = -4.84676915585801
        struct[0].Gy_ini[14,56] = -7.89461337514771
        struct[0].Gy_ini[14,57] = 1.58482814504483
        struct[0].Gy_ini[14,58] = -7.72144347466701
        struct[0].Gy_ini[14,59] = 0.679918453285308
        struct[0].Gy_ini[14,60] = -7.89461337514771
        struct[0].Gy_ini[14,61] = 1.58482814504483
        struct[0].Gy_ini[14,62] = 26.5300693004806
        struct[0].Gy_ini[14,63] = 4.84676915585801
        struct[0].Gy_ini[14,158] = 1
        struct[0].Gy_ini[15,8] = 1.58482814504483
        struct[0].Gy_ini[15,9] = 7.89461337514771
        struct[0].Gy_ini[15,10] = 0.679918453285308
        struct[0].Gy_ini[15,11] = 7.72144347466701
        struct[0].Gy_ini[15,12] = 1.58482814504483
        struct[0].Gy_ini[15,13] = 7.89461337514771
        struct[0].Gy_ini[15,14] = 4.84676915585801
        struct[0].Gy_ini[15,15] = -26.5550693004806
        struct[0].Gy_ini[15,56] = -1.58482814504483
        struct[0].Gy_ini[15,57] = -7.89461337514771
        struct[0].Gy_ini[15,58] = -0.679918453285308
        struct[0].Gy_ini[15,59] = -7.72144347466701
        struct[0].Gy_ini[15,60] = -1.58482814504483
        struct[0].Gy_ini[15,61] = -7.89461337514771
        struct[0].Gy_ini[15,62] = -4.84676915585801
        struct[0].Gy_ini[15,63] = 26.5300693004806
        struct[0].Gy_ini[15,159] = 1
        struct[0].Gy_ini[16,16] = -26.5300693004806
        struct[0].Gy_ini[16,17] = -4.84676915585801
        struct[0].Gy_ini[16,18] = 7.89461337514770
        struct[0].Gy_ini[16,19] = -1.58482814504483
        struct[0].Gy_ini[16,20] = 7.72144347466701
        struct[0].Gy_ini[16,21] = -0.679918453285310
        struct[0].Gy_ini[16,22] = 7.89461337514771
        struct[0].Gy_ini[16,23] = -1.58482814504483
        struct[0].Gy_ini[16,136] = 26.5300693004806
        struct[0].Gy_ini[16,137] = 4.84676915585801
        struct[0].Gy_ini[16,138] = -7.89461337514770
        struct[0].Gy_ini[16,139] = 1.58482814504483
        struct[0].Gy_ini[16,140] = -7.72144347466701
        struct[0].Gy_ini[16,141] = 0.679918453285310
        struct[0].Gy_ini[16,142] = -7.89461337514771
        struct[0].Gy_ini[16,143] = 1.58482814504483
        struct[0].Gy_ini[16,160] = 1
        struct[0].Gy_ini[17,16] = 4.84676915585801
        struct[0].Gy_ini[17,17] = -26.5300693004806
        struct[0].Gy_ini[17,18] = 1.58482814504483
        struct[0].Gy_ini[17,19] = 7.89461337514770
        struct[0].Gy_ini[17,20] = 0.679918453285310
        struct[0].Gy_ini[17,21] = 7.72144347466701
        struct[0].Gy_ini[17,22] = 1.58482814504483
        struct[0].Gy_ini[17,23] = 7.89461337514771
        struct[0].Gy_ini[17,136] = -4.84676915585801
        struct[0].Gy_ini[17,137] = 26.5300693004806
        struct[0].Gy_ini[17,138] = -1.58482814504483
        struct[0].Gy_ini[17,139] = -7.89461337514770
        struct[0].Gy_ini[17,140] = -0.679918453285310
        struct[0].Gy_ini[17,141] = -7.72144347466701
        struct[0].Gy_ini[17,142] = -1.58482814504483
        struct[0].Gy_ini[17,143] = -7.89461337514771
        struct[0].Gy_ini[17,161] = 1
        struct[0].Gy_ini[18,16] = 7.89461337514771
        struct[0].Gy_ini[18,17] = -1.58482814504483
        struct[0].Gy_ini[18,18] = -26.5300693004806
        struct[0].Gy_ini[18,19] = -4.84676915585801
        struct[0].Gy_ini[18,20] = 7.89461337514771
        struct[0].Gy_ini[18,21] = -1.58482814504483
        struct[0].Gy_ini[18,22] = 7.72144347466701
        struct[0].Gy_ini[18,23] = -0.679918453285308
        struct[0].Gy_ini[18,136] = -7.89461337514771
        struct[0].Gy_ini[18,137] = 1.58482814504483
        struct[0].Gy_ini[18,138] = 26.5300693004806
        struct[0].Gy_ini[18,139] = 4.84676915585801
        struct[0].Gy_ini[18,140] = -7.89461337514771
        struct[0].Gy_ini[18,141] = 1.58482814504483
        struct[0].Gy_ini[18,142] = -7.72144347466701
        struct[0].Gy_ini[18,143] = 0.679918453285308
        struct[0].Gy_ini[18,162] = 1
        struct[0].Gy_ini[19,16] = 1.58482814504483
        struct[0].Gy_ini[19,17] = 7.89461337514771
        struct[0].Gy_ini[19,18] = 4.84676915585801
        struct[0].Gy_ini[19,19] = -26.5300693004806
        struct[0].Gy_ini[19,20] = 1.58482814504483
        struct[0].Gy_ini[19,21] = 7.89461337514771
        struct[0].Gy_ini[19,22] = 0.679918453285308
        struct[0].Gy_ini[19,23] = 7.72144347466701
        struct[0].Gy_ini[19,136] = -1.58482814504483
        struct[0].Gy_ini[19,137] = -7.89461337514771
        struct[0].Gy_ini[19,138] = -4.84676915585801
        struct[0].Gy_ini[19,139] = 26.5300693004806
        struct[0].Gy_ini[19,140] = -1.58482814504483
        struct[0].Gy_ini[19,141] = -7.89461337514771
        struct[0].Gy_ini[19,142] = -0.679918453285308
        struct[0].Gy_ini[19,143] = -7.72144347466701
        struct[0].Gy_ini[19,163] = 1
        struct[0].Gy_ini[20,16] = 7.72144347466701
        struct[0].Gy_ini[20,17] = -0.679918453285310
        struct[0].Gy_ini[20,18] = 7.89461337514770
        struct[0].Gy_ini[20,19] = -1.58482814504483
        struct[0].Gy_ini[20,20] = -26.5300693004806
        struct[0].Gy_ini[20,21] = -4.84676915585801
        struct[0].Gy_ini[20,22] = 7.89461337514771
        struct[0].Gy_ini[20,23] = -1.58482814504483
        struct[0].Gy_ini[20,136] = -7.72144347466701
        struct[0].Gy_ini[20,137] = 0.679918453285310
        struct[0].Gy_ini[20,138] = -7.89461337514770
        struct[0].Gy_ini[20,139] = 1.58482814504483
        struct[0].Gy_ini[20,140] = 26.5300693004806
        struct[0].Gy_ini[20,141] = 4.84676915585801
        struct[0].Gy_ini[20,142] = -7.89461337514771
        struct[0].Gy_ini[20,143] = 1.58482814504483
        struct[0].Gy_ini[20,164] = 1
        struct[0].Gy_ini[21,16] = 0.679918453285310
        struct[0].Gy_ini[21,17] = 7.72144347466701
        struct[0].Gy_ini[21,18] = 1.58482814504483
        struct[0].Gy_ini[21,19] = 7.89461337514770
        struct[0].Gy_ini[21,20] = 4.84676915585801
        struct[0].Gy_ini[21,21] = -26.5300693004806
        struct[0].Gy_ini[21,22] = 1.58482814504483
        struct[0].Gy_ini[21,23] = 7.89461337514771
        struct[0].Gy_ini[21,136] = -0.679918453285310
        struct[0].Gy_ini[21,137] = -7.72144347466701
        struct[0].Gy_ini[21,138] = -1.58482814504483
        struct[0].Gy_ini[21,139] = -7.89461337514770
        struct[0].Gy_ini[21,140] = -4.84676915585801
        struct[0].Gy_ini[21,141] = 26.5300693004806
        struct[0].Gy_ini[21,142] = -1.58482814504483
        struct[0].Gy_ini[21,143] = -7.89461337514771
        struct[0].Gy_ini[21,165] = 1
        struct[0].Gy_ini[22,16] = 7.89461337514771
        struct[0].Gy_ini[22,17] = -1.58482814504483
        struct[0].Gy_ini[22,18] = 7.72144347466701
        struct[0].Gy_ini[22,19] = -0.679918453285308
        struct[0].Gy_ini[22,20] = 7.89461337514771
        struct[0].Gy_ini[22,21] = -1.58482814504483
        struct[0].Gy_ini[22,22] = -26.5550693004806
        struct[0].Gy_ini[22,23] = -4.84676915585801
        struct[0].Gy_ini[22,136] = -7.89461337514771
        struct[0].Gy_ini[22,137] = 1.58482814504483
        struct[0].Gy_ini[22,138] = -7.72144347466701
        struct[0].Gy_ini[22,139] = 0.679918453285308
        struct[0].Gy_ini[22,140] = -7.89461337514771
        struct[0].Gy_ini[22,141] = 1.58482814504483
        struct[0].Gy_ini[22,142] = 26.5300693004806
        struct[0].Gy_ini[22,143] = 4.84676915585801
        struct[0].Gy_ini[22,166] = 1
        struct[0].Gy_ini[23,16] = 1.58482814504483
        struct[0].Gy_ini[23,17] = 7.89461337514771
        struct[0].Gy_ini[23,18] = 0.679918453285308
        struct[0].Gy_ini[23,19] = 7.72144347466701
        struct[0].Gy_ini[23,20] = 1.58482814504483
        struct[0].Gy_ini[23,21] = 7.89461337514771
        struct[0].Gy_ini[23,22] = 4.84676915585801
        struct[0].Gy_ini[23,23] = -26.5550693004806
        struct[0].Gy_ini[23,136] = -1.58482814504483
        struct[0].Gy_ini[23,137] = -7.89461337514771
        struct[0].Gy_ini[23,138] = -0.679918453285308
        struct[0].Gy_ini[23,139] = -7.72144347466701
        struct[0].Gy_ini[23,140] = -1.58482814504483
        struct[0].Gy_ini[23,141] = -7.89461337514771
        struct[0].Gy_ini[23,142] = -4.84676915585801
        struct[0].Gy_ini[23,143] = 26.5300693004806
        struct[0].Gy_ini[23,167] = 1
        struct[0].Gy_ini[24,24] = -30.9517475172273
        struct[0].Gy_ini[24,25] = -5.65456401516768
        struct[0].Gy_ini[24,26] = 9.21038227100566
        struct[0].Gy_ini[24,27] = -1.84896616921897
        struct[0].Gy_ini[24,28] = 9.00835072044485
        struct[0].Gy_ini[24,29] = -0.793238195499529
        struct[0].Gy_ini[24,30] = 9.21038227100566
        struct[0].Gy_ini[24,31] = -1.84896616921897
        struct[0].Gy_ini[24,80] = 30.9517475172273
        struct[0].Gy_ini[24,81] = 5.65456401516768
        struct[0].Gy_ini[24,82] = -9.21038227100566
        struct[0].Gy_ini[24,83] = 1.84896616921897
        struct[0].Gy_ini[24,84] = -9.00835072044485
        struct[0].Gy_ini[24,85] = 0.793238195499529
        struct[0].Gy_ini[24,86] = -9.21038227100566
        struct[0].Gy_ini[24,87] = 1.84896616921897
        struct[0].Gy_ini[24,168] = 1
        struct[0].Gy_ini[25,24] = 5.65456401516768
        struct[0].Gy_ini[25,25] = -30.9517475172273
        struct[0].Gy_ini[25,26] = 1.84896616921897
        struct[0].Gy_ini[25,27] = 9.21038227100566
        struct[0].Gy_ini[25,28] = 0.793238195499529
        struct[0].Gy_ini[25,29] = 9.00835072044485
        struct[0].Gy_ini[25,30] = 1.84896616921897
        struct[0].Gy_ini[25,31] = 9.21038227100566
        struct[0].Gy_ini[25,80] = -5.65456401516768
        struct[0].Gy_ini[25,81] = 30.9517475172273
        struct[0].Gy_ini[25,82] = -1.84896616921897
        struct[0].Gy_ini[25,83] = -9.21038227100566
        struct[0].Gy_ini[25,84] = -0.793238195499529
        struct[0].Gy_ini[25,85] = -9.00835072044485
        struct[0].Gy_ini[25,86] = -1.84896616921897
        struct[0].Gy_ini[25,87] = -9.21038227100566
        struct[0].Gy_ini[25,169] = 1
        struct[0].Gy_ini[26,24] = 9.21038227100566
        struct[0].Gy_ini[26,25] = -1.84896616921897
        struct[0].Gy_ini[26,26] = -30.9517475172273
        struct[0].Gy_ini[26,27] = -5.65456401516768
        struct[0].Gy_ini[26,28] = 9.21038227100566
        struct[0].Gy_ini[26,29] = -1.84896616921897
        struct[0].Gy_ini[26,30] = 9.00835072044485
        struct[0].Gy_ini[26,31] = -0.793238195499528
        struct[0].Gy_ini[26,80] = -9.21038227100566
        struct[0].Gy_ini[26,81] = 1.84896616921897
        struct[0].Gy_ini[26,82] = 30.9517475172273
        struct[0].Gy_ini[26,83] = 5.65456401516768
        struct[0].Gy_ini[26,84] = -9.21038227100566
        struct[0].Gy_ini[26,85] = 1.84896616921897
        struct[0].Gy_ini[26,86] = -9.00835072044485
        struct[0].Gy_ini[26,87] = 0.793238195499528
        struct[0].Gy_ini[26,170] = 1
        struct[0].Gy_ini[27,24] = 1.84896616921897
        struct[0].Gy_ini[27,25] = 9.21038227100566
        struct[0].Gy_ini[27,26] = 5.65456401516768
        struct[0].Gy_ini[27,27] = -30.9517475172273
        struct[0].Gy_ini[27,28] = 1.84896616921897
        struct[0].Gy_ini[27,29] = 9.21038227100566
        struct[0].Gy_ini[27,30] = 0.793238195499528
        struct[0].Gy_ini[27,31] = 9.00835072044485
        struct[0].Gy_ini[27,80] = -1.84896616921897
        struct[0].Gy_ini[27,81] = -9.21038227100566
        struct[0].Gy_ini[27,82] = -5.65456401516768
        struct[0].Gy_ini[27,83] = 30.9517475172273
        struct[0].Gy_ini[27,84] = -1.84896616921897
        struct[0].Gy_ini[27,85] = -9.21038227100566
        struct[0].Gy_ini[27,86] = -0.793238195499528
        struct[0].Gy_ini[27,87] = -9.00835072044485
        struct[0].Gy_ini[27,171] = 1
        struct[0].Gy_ini[28,24] = 9.00835072044484
        struct[0].Gy_ini[28,25] = -0.793238195499527
        struct[0].Gy_ini[28,26] = 9.21038227100566
        struct[0].Gy_ini[28,27] = -1.84896616921897
        struct[0].Gy_ini[28,28] = -30.9517475172273
        struct[0].Gy_ini[28,29] = -5.65456401516768
        struct[0].Gy_ini[28,30] = 9.21038227100566
        struct[0].Gy_ini[28,31] = -1.84896616921897
        struct[0].Gy_ini[28,80] = -9.00835072044484
        struct[0].Gy_ini[28,81] = 0.793238195499527
        struct[0].Gy_ini[28,82] = -9.21038227100566
        struct[0].Gy_ini[28,83] = 1.84896616921897
        struct[0].Gy_ini[28,84] = 30.9517475172273
        struct[0].Gy_ini[28,85] = 5.65456401516768
        struct[0].Gy_ini[28,86] = -9.21038227100566
        struct[0].Gy_ini[28,87] = 1.84896616921897
        struct[0].Gy_ini[28,172] = 1
        struct[0].Gy_ini[29,24] = 0.793238195499527
        struct[0].Gy_ini[29,25] = 9.00835072044484
        struct[0].Gy_ini[29,26] = 1.84896616921897
        struct[0].Gy_ini[29,27] = 9.21038227100566
        struct[0].Gy_ini[29,28] = 5.65456401516768
        struct[0].Gy_ini[29,29] = -30.9517475172273
        struct[0].Gy_ini[29,30] = 1.84896616921897
        struct[0].Gy_ini[29,31] = 9.21038227100566
        struct[0].Gy_ini[29,80] = -0.793238195499527
        struct[0].Gy_ini[29,81] = -9.00835072044484
        struct[0].Gy_ini[29,82] = -1.84896616921897
        struct[0].Gy_ini[29,83] = -9.21038227100566
        struct[0].Gy_ini[29,84] = -5.65456401516768
        struct[0].Gy_ini[29,85] = 30.9517475172273
        struct[0].Gy_ini[29,86] = -1.84896616921897
        struct[0].Gy_ini[29,87] = -9.21038227100566
        struct[0].Gy_ini[29,173] = 1
        struct[0].Gy_ini[30,24] = 9.21038227100566
        struct[0].Gy_ini[30,25] = -1.84896616921897
        struct[0].Gy_ini[30,26] = 9.00835072044485
        struct[0].Gy_ini[30,27] = -0.793238195499527
        struct[0].Gy_ini[30,28] = 9.21038227100566
        struct[0].Gy_ini[30,29] = -1.84896616921897
        struct[0].Gy_ini[30,30] = -30.9767475172273
        struct[0].Gy_ini[30,31] = -5.65456401516768
        struct[0].Gy_ini[30,80] = -9.21038227100566
        struct[0].Gy_ini[30,81] = 1.84896616921897
        struct[0].Gy_ini[30,82] = -9.00835072044485
        struct[0].Gy_ini[30,83] = 0.793238195499527
        struct[0].Gy_ini[30,84] = -9.21038227100566
        struct[0].Gy_ini[30,85] = 1.84896616921897
        struct[0].Gy_ini[30,86] = 30.9517475172273
        struct[0].Gy_ini[30,87] = 5.65456401516768
        struct[0].Gy_ini[30,174] = 1
        struct[0].Gy_ini[31,24] = 1.84896616921897
        struct[0].Gy_ini[31,25] = 9.21038227100566
        struct[0].Gy_ini[31,26] = 0.793238195499527
        struct[0].Gy_ini[31,27] = 9.00835072044485
        struct[0].Gy_ini[31,28] = 1.84896616921897
        struct[0].Gy_ini[31,29] = 9.21038227100566
        struct[0].Gy_ini[31,30] = 5.65456401516768
        struct[0].Gy_ini[31,31] = -30.9767475172273
        struct[0].Gy_ini[31,80] = -1.84896616921897
        struct[0].Gy_ini[31,81] = -9.21038227100566
        struct[0].Gy_ini[31,82] = -0.793238195499527
        struct[0].Gy_ini[31,83] = -9.00835072044485
        struct[0].Gy_ini[31,84] = -1.84896616921897
        struct[0].Gy_ini[31,85] = -9.21038227100566
        struct[0].Gy_ini[31,86] = -5.65456401516768
        struct[0].Gy_ini[31,87] = 30.9517475172273
        struct[0].Gy_ini[31,175] = 1
        struct[0].Gy_ini[32,32] = -30.9517475172273
        struct[0].Gy_ini[32,33] = -5.65456401516768
        struct[0].Gy_ini[32,34] = 9.21038227100566
        struct[0].Gy_ini[32,35] = -1.84896616921897
        struct[0].Gy_ini[32,36] = 9.00835072044485
        struct[0].Gy_ini[32,37] = -0.793238195499529
        struct[0].Gy_ini[32,38] = 9.21038227100566
        struct[0].Gy_ini[32,39] = -1.84896616921897
        struct[0].Gy_ini[32,104] = 30.9517475172273
        struct[0].Gy_ini[32,105] = 5.65456401516768
        struct[0].Gy_ini[32,106] = -9.21038227100566
        struct[0].Gy_ini[32,107] = 1.84896616921897
        struct[0].Gy_ini[32,108] = -9.00835072044485
        struct[0].Gy_ini[32,109] = 0.793238195499529
        struct[0].Gy_ini[32,110] = -9.21038227100566
        struct[0].Gy_ini[32,111] = 1.84896616921897
        struct[0].Gy_ini[32,176] = 1
        struct[0].Gy_ini[33,32] = 5.65456401516768
        struct[0].Gy_ini[33,33] = -30.9517475172273
        struct[0].Gy_ini[33,34] = 1.84896616921897
        struct[0].Gy_ini[33,35] = 9.21038227100566
        struct[0].Gy_ini[33,36] = 0.793238195499529
        struct[0].Gy_ini[33,37] = 9.00835072044485
        struct[0].Gy_ini[33,38] = 1.84896616921897
        struct[0].Gy_ini[33,39] = 9.21038227100566
        struct[0].Gy_ini[33,104] = -5.65456401516768
        struct[0].Gy_ini[33,105] = 30.9517475172273
        struct[0].Gy_ini[33,106] = -1.84896616921897
        struct[0].Gy_ini[33,107] = -9.21038227100566
        struct[0].Gy_ini[33,108] = -0.793238195499529
        struct[0].Gy_ini[33,109] = -9.00835072044485
        struct[0].Gy_ini[33,110] = -1.84896616921897
        struct[0].Gy_ini[33,111] = -9.21038227100566
        struct[0].Gy_ini[33,177] = 1
        struct[0].Gy_ini[34,32] = 9.21038227100566
        struct[0].Gy_ini[34,33] = -1.84896616921897
        struct[0].Gy_ini[34,34] = -30.9517475172273
        struct[0].Gy_ini[34,35] = -5.65456401516768
        struct[0].Gy_ini[34,36] = 9.21038227100566
        struct[0].Gy_ini[34,37] = -1.84896616921897
        struct[0].Gy_ini[34,38] = 9.00835072044485
        struct[0].Gy_ini[34,39] = -0.793238195499528
        struct[0].Gy_ini[34,104] = -9.21038227100566
        struct[0].Gy_ini[34,105] = 1.84896616921897
        struct[0].Gy_ini[34,106] = 30.9517475172273
        struct[0].Gy_ini[34,107] = 5.65456401516768
        struct[0].Gy_ini[34,108] = -9.21038227100566
        struct[0].Gy_ini[34,109] = 1.84896616921897
        struct[0].Gy_ini[34,110] = -9.00835072044485
        struct[0].Gy_ini[34,111] = 0.793238195499528
        struct[0].Gy_ini[34,178] = 1
        struct[0].Gy_ini[35,32] = 1.84896616921897
        struct[0].Gy_ini[35,33] = 9.21038227100566
        struct[0].Gy_ini[35,34] = 5.65456401516768
        struct[0].Gy_ini[35,35] = -30.9517475172273
        struct[0].Gy_ini[35,36] = 1.84896616921897
        struct[0].Gy_ini[35,37] = 9.21038227100566
        struct[0].Gy_ini[35,38] = 0.793238195499528
        struct[0].Gy_ini[35,39] = 9.00835072044485
        struct[0].Gy_ini[35,104] = -1.84896616921897
        struct[0].Gy_ini[35,105] = -9.21038227100566
        struct[0].Gy_ini[35,106] = -5.65456401516768
        struct[0].Gy_ini[35,107] = 30.9517475172273
        struct[0].Gy_ini[35,108] = -1.84896616921897
        struct[0].Gy_ini[35,109] = -9.21038227100566
        struct[0].Gy_ini[35,110] = -0.793238195499528
        struct[0].Gy_ini[35,111] = -9.00835072044485
        struct[0].Gy_ini[35,179] = 1
        struct[0].Gy_ini[36,32] = 9.00835072044484
        struct[0].Gy_ini[36,33] = -0.793238195499527
        struct[0].Gy_ini[36,34] = 9.21038227100566
        struct[0].Gy_ini[36,35] = -1.84896616921897
        struct[0].Gy_ini[36,36] = -30.9517475172273
        struct[0].Gy_ini[36,37] = -5.65456401516768
        struct[0].Gy_ini[36,38] = 9.21038227100566
        struct[0].Gy_ini[36,39] = -1.84896616921897
        struct[0].Gy_ini[36,104] = -9.00835072044484
        struct[0].Gy_ini[36,105] = 0.793238195499527
        struct[0].Gy_ini[36,106] = -9.21038227100566
        struct[0].Gy_ini[36,107] = 1.84896616921897
        struct[0].Gy_ini[36,108] = 30.9517475172273
        struct[0].Gy_ini[36,109] = 5.65456401516768
        struct[0].Gy_ini[36,110] = -9.21038227100566
        struct[0].Gy_ini[36,111] = 1.84896616921897
        struct[0].Gy_ini[36,180] = 1
        struct[0].Gy_ini[37,32] = 0.793238195499527
        struct[0].Gy_ini[37,33] = 9.00835072044484
        struct[0].Gy_ini[37,34] = 1.84896616921897
        struct[0].Gy_ini[37,35] = 9.21038227100566
        struct[0].Gy_ini[37,36] = 5.65456401516768
        struct[0].Gy_ini[37,37] = -30.9517475172273
        struct[0].Gy_ini[37,38] = 1.84896616921897
        struct[0].Gy_ini[37,39] = 9.21038227100566
        struct[0].Gy_ini[37,104] = -0.793238195499527
        struct[0].Gy_ini[37,105] = -9.00835072044484
        struct[0].Gy_ini[37,106] = -1.84896616921897
        struct[0].Gy_ini[37,107] = -9.21038227100566
        struct[0].Gy_ini[37,108] = -5.65456401516768
        struct[0].Gy_ini[37,109] = 30.9517475172273
        struct[0].Gy_ini[37,110] = -1.84896616921897
        struct[0].Gy_ini[37,111] = -9.21038227100566
        struct[0].Gy_ini[37,181] = 1
        struct[0].Gy_ini[38,32] = 9.21038227100566
        struct[0].Gy_ini[38,33] = -1.84896616921897
        struct[0].Gy_ini[38,34] = 9.00835072044485
        struct[0].Gy_ini[38,35] = -0.793238195499527
        struct[0].Gy_ini[38,36] = 9.21038227100566
        struct[0].Gy_ini[38,37] = -1.84896616921897
        struct[0].Gy_ini[38,38] = -30.9767475172273
        struct[0].Gy_ini[38,39] = -5.65456401516768
        struct[0].Gy_ini[38,104] = -9.21038227100566
        struct[0].Gy_ini[38,105] = 1.84896616921897
        struct[0].Gy_ini[38,106] = -9.00835072044485
        struct[0].Gy_ini[38,107] = 0.793238195499527
        struct[0].Gy_ini[38,108] = -9.21038227100566
        struct[0].Gy_ini[38,109] = 1.84896616921897
        struct[0].Gy_ini[38,110] = 30.9517475172273
        struct[0].Gy_ini[38,111] = 5.65456401516768
        struct[0].Gy_ini[38,182] = 1
        struct[0].Gy_ini[39,32] = 1.84896616921897
        struct[0].Gy_ini[39,33] = 9.21038227100566
        struct[0].Gy_ini[39,34] = 0.793238195499527
        struct[0].Gy_ini[39,35] = 9.00835072044485
        struct[0].Gy_ini[39,36] = 1.84896616921897
        struct[0].Gy_ini[39,37] = 9.21038227100566
        struct[0].Gy_ini[39,38] = 5.65456401516768
        struct[0].Gy_ini[39,39] = -30.9767475172273
        struct[0].Gy_ini[39,104] = -1.84896616921897
        struct[0].Gy_ini[39,105] = -9.21038227100566
        struct[0].Gy_ini[39,106] = -0.793238195499527
        struct[0].Gy_ini[39,107] = -9.00835072044485
        struct[0].Gy_ini[39,108] = -1.84896616921897
        struct[0].Gy_ini[39,109] = -9.21038227100566
        struct[0].Gy_ini[39,110] = -5.65456401516768
        struct[0].Gy_ini[39,111] = 30.9517475172273
        struct[0].Gy_ini[39,183] = 1
        struct[0].Gy_ini[40,40] = -30.9517475172273
        struct[0].Gy_ini[40,41] = -5.65456401516768
        struct[0].Gy_ini[40,42] = 9.21038227100566
        struct[0].Gy_ini[40,43] = -1.84896616921897
        struct[0].Gy_ini[40,44] = 9.00835072044485
        struct[0].Gy_ini[40,45] = -0.793238195499529
        struct[0].Gy_ini[40,46] = 9.21038227100566
        struct[0].Gy_ini[40,47] = -1.84896616921897
        struct[0].Gy_ini[40,112] = 30.9517475172273
        struct[0].Gy_ini[40,113] = 5.65456401516768
        struct[0].Gy_ini[40,114] = -9.21038227100566
        struct[0].Gy_ini[40,115] = 1.84896616921897
        struct[0].Gy_ini[40,116] = -9.00835072044485
        struct[0].Gy_ini[40,117] = 0.793238195499529
        struct[0].Gy_ini[40,118] = -9.21038227100566
        struct[0].Gy_ini[40,119] = 1.84896616921897
        struct[0].Gy_ini[40,184] = 1
        struct[0].Gy_ini[41,40] = 5.65456401516768
        struct[0].Gy_ini[41,41] = -30.9517475172273
        struct[0].Gy_ini[41,42] = 1.84896616921897
        struct[0].Gy_ini[41,43] = 9.21038227100566
        struct[0].Gy_ini[41,44] = 0.793238195499529
        struct[0].Gy_ini[41,45] = 9.00835072044485
        struct[0].Gy_ini[41,46] = 1.84896616921897
        struct[0].Gy_ini[41,47] = 9.21038227100566
        struct[0].Gy_ini[41,112] = -5.65456401516768
        struct[0].Gy_ini[41,113] = 30.9517475172273
        struct[0].Gy_ini[41,114] = -1.84896616921897
        struct[0].Gy_ini[41,115] = -9.21038227100566
        struct[0].Gy_ini[41,116] = -0.793238195499529
        struct[0].Gy_ini[41,117] = -9.00835072044485
        struct[0].Gy_ini[41,118] = -1.84896616921897
        struct[0].Gy_ini[41,119] = -9.21038227100566
        struct[0].Gy_ini[41,185] = 1
        struct[0].Gy_ini[42,40] = 9.21038227100566
        struct[0].Gy_ini[42,41] = -1.84896616921897
        struct[0].Gy_ini[42,42] = -30.9517475172273
        struct[0].Gy_ini[42,43] = -5.65456401516768
        struct[0].Gy_ini[42,44] = 9.21038227100566
        struct[0].Gy_ini[42,45] = -1.84896616921897
        struct[0].Gy_ini[42,46] = 9.00835072044485
        struct[0].Gy_ini[42,47] = -0.793238195499528
        struct[0].Gy_ini[42,112] = -9.21038227100566
        struct[0].Gy_ini[42,113] = 1.84896616921897
        struct[0].Gy_ini[42,114] = 30.9517475172273
        struct[0].Gy_ini[42,115] = 5.65456401516768
        struct[0].Gy_ini[42,116] = -9.21038227100566
        struct[0].Gy_ini[42,117] = 1.84896616921897
        struct[0].Gy_ini[42,118] = -9.00835072044485
        struct[0].Gy_ini[42,119] = 0.793238195499528
        struct[0].Gy_ini[42,186] = 1
        struct[0].Gy_ini[43,40] = 1.84896616921897
        struct[0].Gy_ini[43,41] = 9.21038227100566
        struct[0].Gy_ini[43,42] = 5.65456401516768
        struct[0].Gy_ini[43,43] = -30.9517475172273
        struct[0].Gy_ini[43,44] = 1.84896616921897
        struct[0].Gy_ini[43,45] = 9.21038227100566
        struct[0].Gy_ini[43,46] = 0.793238195499528
        struct[0].Gy_ini[43,47] = 9.00835072044485
        struct[0].Gy_ini[43,112] = -1.84896616921897
        struct[0].Gy_ini[43,113] = -9.21038227100566
        struct[0].Gy_ini[43,114] = -5.65456401516768
        struct[0].Gy_ini[43,115] = 30.9517475172273
        struct[0].Gy_ini[43,116] = -1.84896616921897
        struct[0].Gy_ini[43,117] = -9.21038227100566
        struct[0].Gy_ini[43,118] = -0.793238195499528
        struct[0].Gy_ini[43,119] = -9.00835072044485
        struct[0].Gy_ini[43,187] = 1
        struct[0].Gy_ini[44,40] = 9.00835072044484
        struct[0].Gy_ini[44,41] = -0.793238195499527
        struct[0].Gy_ini[44,42] = 9.21038227100566
        struct[0].Gy_ini[44,43] = -1.84896616921897
        struct[0].Gy_ini[44,44] = -30.9517475172273
        struct[0].Gy_ini[44,45] = -5.65456401516768
        struct[0].Gy_ini[44,46] = 9.21038227100566
        struct[0].Gy_ini[44,47] = -1.84896616921897
        struct[0].Gy_ini[44,112] = -9.00835072044484
        struct[0].Gy_ini[44,113] = 0.793238195499527
        struct[0].Gy_ini[44,114] = -9.21038227100566
        struct[0].Gy_ini[44,115] = 1.84896616921897
        struct[0].Gy_ini[44,116] = 30.9517475172273
        struct[0].Gy_ini[44,117] = 5.65456401516768
        struct[0].Gy_ini[44,118] = -9.21038227100566
        struct[0].Gy_ini[44,119] = 1.84896616921897
        struct[0].Gy_ini[44,188] = 1
        struct[0].Gy_ini[45,40] = 0.793238195499527
        struct[0].Gy_ini[45,41] = 9.00835072044484
        struct[0].Gy_ini[45,42] = 1.84896616921897
        struct[0].Gy_ini[45,43] = 9.21038227100566
        struct[0].Gy_ini[45,44] = 5.65456401516768
        struct[0].Gy_ini[45,45] = -30.9517475172273
        struct[0].Gy_ini[45,46] = 1.84896616921897
        struct[0].Gy_ini[45,47] = 9.21038227100566
        struct[0].Gy_ini[45,112] = -0.793238195499527
        struct[0].Gy_ini[45,113] = -9.00835072044484
        struct[0].Gy_ini[45,114] = -1.84896616921897
        struct[0].Gy_ini[45,115] = -9.21038227100566
        struct[0].Gy_ini[45,116] = -5.65456401516768
        struct[0].Gy_ini[45,117] = 30.9517475172273
        struct[0].Gy_ini[45,118] = -1.84896616921897
        struct[0].Gy_ini[45,119] = -9.21038227100566
        struct[0].Gy_ini[45,189] = 1
        struct[0].Gy_ini[46,40] = 9.21038227100566
        struct[0].Gy_ini[46,41] = -1.84896616921897
        struct[0].Gy_ini[46,42] = 9.00835072044485
        struct[0].Gy_ini[46,43] = -0.793238195499527
        struct[0].Gy_ini[46,44] = 9.21038227100566
        struct[0].Gy_ini[46,45] = -1.84896616921897
        struct[0].Gy_ini[46,46] = -30.9767475172273
        struct[0].Gy_ini[46,47] = -5.65456401516768
        struct[0].Gy_ini[46,112] = -9.21038227100566
        struct[0].Gy_ini[46,113] = 1.84896616921897
        struct[0].Gy_ini[46,114] = -9.00835072044485
        struct[0].Gy_ini[46,115] = 0.793238195499527
        struct[0].Gy_ini[46,116] = -9.21038227100566
        struct[0].Gy_ini[46,117] = 1.84896616921897
        struct[0].Gy_ini[46,118] = 30.9517475172273
        struct[0].Gy_ini[46,119] = 5.65456401516768
        struct[0].Gy_ini[46,190] = 1
        struct[0].Gy_ini[47,40] = 1.84896616921897
        struct[0].Gy_ini[47,41] = 9.21038227100566
        struct[0].Gy_ini[47,42] = 0.793238195499527
        struct[0].Gy_ini[47,43] = 9.00835072044485
        struct[0].Gy_ini[47,44] = 1.84896616921897
        struct[0].Gy_ini[47,45] = 9.21038227100566
        struct[0].Gy_ini[47,46] = 5.65456401516768
        struct[0].Gy_ini[47,47] = -30.9767475172273
        struct[0].Gy_ini[47,112] = -1.84896616921897
        struct[0].Gy_ini[47,113] = -9.21038227100566
        struct[0].Gy_ini[47,114] = -0.793238195499527
        struct[0].Gy_ini[47,115] = -9.00835072044485
        struct[0].Gy_ini[47,116] = -1.84896616921897
        struct[0].Gy_ini[47,117] = -9.21038227100566
        struct[0].Gy_ini[47,118] = -5.65456401516768
        struct[0].Gy_ini[47,119] = 30.9517475172273
        struct[0].Gy_ini[47,191] = 1
        struct[0].Gy_ini[48,0] = 105.571769313180
        struct[0].Gy_ini[48,1] = 54.0657727682604
        struct[0].Gy_ini[48,2] = -39.6392229058202
        struct[0].Gy_ini[48,3] = -10.2713736253513
        struct[0].Gy_ini[48,4] = -24.9575997948692
        struct[0].Gy_ini[48,5] = -23.2849644809540
        struct[0].Gy_ini[48,6] = -39.6392229058202
        struct[0].Gy_ini[48,7] = -10.2713736253513
        struct[0].Gy_ini[48,48] = -211.143538626360
        struct[0].Gy_ini[48,49] = -108.131545536521
        struct[0].Gy_ini[48,50] = 79.2784458116403
        struct[0].Gy_ini[48,51] = 20.5427472507027
        struct[0].Gy_ini[48,52] = 49.9151995897384
        struct[0].Gy_ini[48,53] = 46.5699289619080
        struct[0].Gy_ini[48,54] = 79.2784458116403
        struct[0].Gy_ini[48,55] = 20.5427472507027
        struct[0].Gy_ini[48,56] = 105.571769313180
        struct[0].Gy_ini[48,57] = 54.0657727682604
        struct[0].Gy_ini[48,58] = -39.6392229058202
        struct[0].Gy_ini[48,59] = -10.2713736253513
        struct[0].Gy_ini[48,60] = -24.9575997948692
        struct[0].Gy_ini[48,61] = -23.2849644809540
        struct[0].Gy_ini[48,62] = -39.6392229058202
        struct[0].Gy_ini[48,63] = -10.2713736253513
        struct[0].Gy_ini[49,0] = -54.0657727682604
        struct[0].Gy_ini[49,1] = 105.571769313180
        struct[0].Gy_ini[49,2] = 10.2713736253513
        struct[0].Gy_ini[49,3] = -39.6392229058202
        struct[0].Gy_ini[49,4] = 23.2849644809540
        struct[0].Gy_ini[49,5] = -24.9575997948692
        struct[0].Gy_ini[49,6] = 10.2713736253513
        struct[0].Gy_ini[49,7] = -39.6392229058202
        struct[0].Gy_ini[49,48] = 108.131545536521
        struct[0].Gy_ini[49,49] = -211.143538626360
        struct[0].Gy_ini[49,50] = -20.5427472507027
        struct[0].Gy_ini[49,51] = 79.2784458116403
        struct[0].Gy_ini[49,52] = -46.5699289619080
        struct[0].Gy_ini[49,53] = 49.9151995897384
        struct[0].Gy_ini[49,54] = -20.5427472507027
        struct[0].Gy_ini[49,55] = 79.2784458116403
        struct[0].Gy_ini[49,56] = -54.0657727682604
        struct[0].Gy_ini[49,57] = 105.571769313180
        struct[0].Gy_ini[49,58] = 10.2713736253513
        struct[0].Gy_ini[49,59] = -39.6392229058202
        struct[0].Gy_ini[49,60] = 23.2849644809540
        struct[0].Gy_ini[49,61] = -24.9575997948692
        struct[0].Gy_ini[49,62] = 10.2713736253513
        struct[0].Gy_ini[49,63] = -39.6392229058202
        struct[0].Gy_ini[50,0] = -39.6392229058202
        struct[0].Gy_ini[50,1] = -10.2713736253513
        struct[0].Gy_ini[50,2] = 105.571769313180
        struct[0].Gy_ini[50,3] = 54.0657727682604
        struct[0].Gy_ini[50,4] = -39.6392229058201
        struct[0].Gy_ini[50,5] = -10.2713736253513
        struct[0].Gy_ini[50,6] = -24.9575997948692
        struct[0].Gy_ini[50,7] = -23.2849644809540
        struct[0].Gy_ini[50,48] = 79.2784458116403
        struct[0].Gy_ini[50,49] = 20.5427472507027
        struct[0].Gy_ini[50,50] = -211.143538626360
        struct[0].Gy_ini[50,51] = -108.131545536521
        struct[0].Gy_ini[50,52] = 79.2784458116403
        struct[0].Gy_ini[50,53] = 20.5427472507027
        struct[0].Gy_ini[50,54] = 49.9151995897384
        struct[0].Gy_ini[50,55] = 46.5699289619080
        struct[0].Gy_ini[50,56] = -39.6392229058202
        struct[0].Gy_ini[50,57] = -10.2713736253513
        struct[0].Gy_ini[50,58] = 105.571769313180
        struct[0].Gy_ini[50,59] = 54.0657727682604
        struct[0].Gy_ini[50,60] = -39.6392229058201
        struct[0].Gy_ini[50,61] = -10.2713736253513
        struct[0].Gy_ini[50,62] = -24.9575997948692
        struct[0].Gy_ini[50,63] = -23.2849644809540
        struct[0].Gy_ini[51,0] = 10.2713736253513
        struct[0].Gy_ini[51,1] = -39.6392229058202
        struct[0].Gy_ini[51,2] = -54.0657727682604
        struct[0].Gy_ini[51,3] = 105.571769313180
        struct[0].Gy_ini[51,4] = 10.2713736253513
        struct[0].Gy_ini[51,5] = -39.6392229058201
        struct[0].Gy_ini[51,6] = 23.2849644809540
        struct[0].Gy_ini[51,7] = -24.9575997948692
        struct[0].Gy_ini[51,48] = -20.5427472507027
        struct[0].Gy_ini[51,49] = 79.2784458116403
        struct[0].Gy_ini[51,50] = 108.131545536521
        struct[0].Gy_ini[51,51] = -211.143538626360
        struct[0].Gy_ini[51,52] = -20.5427472507027
        struct[0].Gy_ini[51,53] = 79.2784458116403
        struct[0].Gy_ini[51,54] = -46.5699289619080
        struct[0].Gy_ini[51,55] = 49.9151995897384
        struct[0].Gy_ini[51,56] = 10.2713736253513
        struct[0].Gy_ini[51,57] = -39.6392229058202
        struct[0].Gy_ini[51,58] = -54.0657727682604
        struct[0].Gy_ini[51,59] = 105.571769313180
        struct[0].Gy_ini[51,60] = 10.2713736253513
        struct[0].Gy_ini[51,61] = -39.6392229058201
        struct[0].Gy_ini[51,62] = 23.2849644809540
        struct[0].Gy_ini[51,63] = -24.9575997948692
        struct[0].Gy_ini[52,0] = -24.9575997948692
        struct[0].Gy_ini[52,1] = -23.2849644809540
        struct[0].Gy_ini[52,2] = -39.6392229058202
        struct[0].Gy_ini[52,3] = -10.2713736253513
        struct[0].Gy_ini[52,4] = 105.571769313180
        struct[0].Gy_ini[52,5] = 54.0657727682604
        struct[0].Gy_ini[52,6] = -39.6392229058201
        struct[0].Gy_ini[52,7] = -10.2713736253514
        struct[0].Gy_ini[52,48] = 49.9151995897384
        struct[0].Gy_ini[52,49] = 46.5699289619080
        struct[0].Gy_ini[52,50] = 79.2784458116404
        struct[0].Gy_ini[52,51] = 20.5427472507027
        struct[0].Gy_ini[52,52] = -211.143538626360
        struct[0].Gy_ini[52,53] = -108.131545536521
        struct[0].Gy_ini[52,54] = 79.2784458116403
        struct[0].Gy_ini[52,55] = 20.5427472507027
        struct[0].Gy_ini[52,56] = -24.9575997948692
        struct[0].Gy_ini[52,57] = -23.2849644809540
        struct[0].Gy_ini[52,58] = -39.6392229058202
        struct[0].Gy_ini[52,59] = -10.2713736253513
        struct[0].Gy_ini[52,60] = 105.571769313180
        struct[0].Gy_ini[52,61] = 54.0657727682604
        struct[0].Gy_ini[52,62] = -39.6392229058201
        struct[0].Gy_ini[52,63] = -10.2713736253514
        struct[0].Gy_ini[53,0] = 23.2849644809540
        struct[0].Gy_ini[53,1] = -24.9575997948692
        struct[0].Gy_ini[53,2] = 10.2713736253513
        struct[0].Gy_ini[53,3] = -39.6392229058202
        struct[0].Gy_ini[53,4] = -54.0657727682604
        struct[0].Gy_ini[53,5] = 105.571769313180
        struct[0].Gy_ini[53,6] = 10.2713736253514
        struct[0].Gy_ini[53,7] = -39.6392229058201
        struct[0].Gy_ini[53,48] = -46.5699289619080
        struct[0].Gy_ini[53,49] = 49.9151995897384
        struct[0].Gy_ini[53,50] = -20.5427472507027
        struct[0].Gy_ini[53,51] = 79.2784458116404
        struct[0].Gy_ini[53,52] = 108.131545536521
        struct[0].Gy_ini[53,53] = -211.143538626360
        struct[0].Gy_ini[53,54] = -20.5427472507027
        struct[0].Gy_ini[53,55] = 79.2784458116403
        struct[0].Gy_ini[53,56] = 23.2849644809540
        struct[0].Gy_ini[53,57] = -24.9575997948692
        struct[0].Gy_ini[53,58] = 10.2713736253513
        struct[0].Gy_ini[53,59] = -39.6392229058202
        struct[0].Gy_ini[53,60] = -54.0657727682604
        struct[0].Gy_ini[53,61] = 105.571769313180
        struct[0].Gy_ini[53,62] = 10.2713736253514
        struct[0].Gy_ini[53,63] = -39.6392229058201
        struct[0].Gy_ini[54,0] = -39.6392229058202
        struct[0].Gy_ini[54,1] = -10.2713736253513
        struct[0].Gy_ini[54,2] = -24.9575997948692
        struct[0].Gy_ini[54,3] = -23.2849644809540
        struct[0].Gy_ini[54,4] = -39.6392229058202
        struct[0].Gy_ini[54,5] = -10.2713736253513
        struct[0].Gy_ini[54,6] = 105.571769313180
        struct[0].Gy_ini[54,7] = 54.0657727682604
        struct[0].Gy_ini[54,48] = 79.2784458116403
        struct[0].Gy_ini[54,49] = 20.5427472507027
        struct[0].Gy_ini[54,50] = 49.9151995897384
        struct[0].Gy_ini[54,51] = 46.5699289619080
        struct[0].Gy_ini[54,52] = 79.2784458116403
        struct[0].Gy_ini[54,53] = 20.5427472507027
        struct[0].Gy_ini[54,54] = -211.168538626360
        struct[0].Gy_ini[54,55] = -108.131545536521
        struct[0].Gy_ini[54,56] = -39.6392229058202
        struct[0].Gy_ini[54,57] = -10.2713736253513
        struct[0].Gy_ini[54,58] = -24.9575997948692
        struct[0].Gy_ini[54,59] = -23.2849644809540
        struct[0].Gy_ini[54,60] = -39.6392229058202
        struct[0].Gy_ini[54,61] = -10.2713736253513
        struct[0].Gy_ini[54,62] = 105.571769313180
        struct[0].Gy_ini[54,63] = 54.0657727682604
        struct[0].Gy_ini[55,0] = 10.2713736253513
        struct[0].Gy_ini[55,1] = -39.6392229058202
        struct[0].Gy_ini[55,2] = 23.2849644809540
        struct[0].Gy_ini[55,3] = -24.9575997948692
        struct[0].Gy_ini[55,4] = 10.2713736253513
        struct[0].Gy_ini[55,5] = -39.6392229058202
        struct[0].Gy_ini[55,6] = -54.0657727682604
        struct[0].Gy_ini[55,7] = 105.571769313180
        struct[0].Gy_ini[55,48] = -20.5427472507027
        struct[0].Gy_ini[55,49] = 79.2784458116403
        struct[0].Gy_ini[55,50] = -46.5699289619080
        struct[0].Gy_ini[55,51] = 49.9151995897384
        struct[0].Gy_ini[55,52] = -20.5427472507027
        struct[0].Gy_ini[55,53] = 79.2784458116403
        struct[0].Gy_ini[55,54] = 108.131545536521
        struct[0].Gy_ini[55,55] = -211.168538626360
        struct[0].Gy_ini[55,56] = 10.2713736253513
        struct[0].Gy_ini[55,57] = -39.6392229058202
        struct[0].Gy_ini[55,58] = 23.2849644809540
        struct[0].Gy_ini[55,59] = -24.9575997948692
        struct[0].Gy_ini[55,60] = 10.2713736253513
        struct[0].Gy_ini[55,61] = -39.6392229058202
        struct[0].Gy_ini[55,62] = -54.0657727682604
        struct[0].Gy_ini[55,63] = 105.571769313180
        struct[0].Gy_ini[56,8] = 26.5300693004806
        struct[0].Gy_ini[56,9] = 4.84676915585801
        struct[0].Gy_ini[56,10] = -7.89461337514770
        struct[0].Gy_ini[56,11] = 1.58482814504483
        struct[0].Gy_ini[56,12] = -7.72144347466701
        struct[0].Gy_ini[56,13] = 0.679918453285310
        struct[0].Gy_ini[56,14] = -7.89461337514771
        struct[0].Gy_ini[56,15] = 1.58482814504483
        struct[0].Gy_ini[56,48] = 105.571769313180
        struct[0].Gy_ini[56,49] = 54.0657727682604
        struct[0].Gy_ini[56,50] = -39.6392229058202
        struct[0].Gy_ini[56,51] = -10.2713736253513
        struct[0].Gy_ini[56,52] = -24.9575997948692
        struct[0].Gy_ini[56,53] = -23.2849644809540
        struct[0].Gy_ini[56,54] = -39.6392229058202
        struct[0].Gy_ini[56,55] = -10.2713736253513
        struct[0].Gy_ini[56,56] = -237.673607926841
        struct[0].Gy_ini[56,57] = -112.978314692379
        struct[0].Gy_ini[56,58] = 87.1730591867880
        struct[0].Gy_ini[56,59] = 18.9579191056578
        struct[0].Gy_ini[56,60] = 57.6366430644055
        struct[0].Gy_ini[56,61] = 45.8900105086227
        struct[0].Gy_ini[56,62] = 87.1730591867880
        struct[0].Gy_ini[56,63] = 18.9579191056579
        struct[0].Gy_ini[56,64] = 105.571769313180
        struct[0].Gy_ini[56,65] = 54.0657727682604
        struct[0].Gy_ini[56,66] = -39.6392229058202
        struct[0].Gy_ini[56,67] = -10.2713736253513
        struct[0].Gy_ini[56,68] = -24.9575997948692
        struct[0].Gy_ini[56,69] = -23.2849644809540
        struct[0].Gy_ini[56,70] = -39.6392229058202
        struct[0].Gy_ini[56,71] = -10.2713736253513
        struct[0].Gy_ini[57,8] = -4.84676915585801
        struct[0].Gy_ini[57,9] = 26.5300693004806
        struct[0].Gy_ini[57,10] = -1.58482814504483
        struct[0].Gy_ini[57,11] = -7.89461337514770
        struct[0].Gy_ini[57,12] = -0.679918453285310
        struct[0].Gy_ini[57,13] = -7.72144347466701
        struct[0].Gy_ini[57,14] = -1.58482814504483
        struct[0].Gy_ini[57,15] = -7.89461337514771
        struct[0].Gy_ini[57,48] = -54.0657727682604
        struct[0].Gy_ini[57,49] = 105.571769313180
        struct[0].Gy_ini[57,50] = 10.2713736253513
        struct[0].Gy_ini[57,51] = -39.6392229058202
        struct[0].Gy_ini[57,52] = 23.2849644809540
        struct[0].Gy_ini[57,53] = -24.9575997948692
        struct[0].Gy_ini[57,54] = 10.2713736253513
        struct[0].Gy_ini[57,55] = -39.6392229058202
        struct[0].Gy_ini[57,56] = 112.978314692379
        struct[0].Gy_ini[57,57] = -237.673607926841
        struct[0].Gy_ini[57,58] = -18.9579191056578
        struct[0].Gy_ini[57,59] = 87.1730591867880
        struct[0].Gy_ini[57,60] = -45.8900105086227
        struct[0].Gy_ini[57,61] = 57.6366430644055
        struct[0].Gy_ini[57,62] = -18.9579191056579
        struct[0].Gy_ini[57,63] = 87.1730591867880
        struct[0].Gy_ini[57,64] = -54.0657727682604
        struct[0].Gy_ini[57,65] = 105.571769313180
        struct[0].Gy_ini[57,66] = 10.2713736253513
        struct[0].Gy_ini[57,67] = -39.6392229058202
        struct[0].Gy_ini[57,68] = 23.2849644809540
        struct[0].Gy_ini[57,69] = -24.9575997948692
        struct[0].Gy_ini[57,70] = 10.2713736253513
        struct[0].Gy_ini[57,71] = -39.6392229058202
        struct[0].Gy_ini[58,8] = -7.89461337514771
        struct[0].Gy_ini[58,9] = 1.58482814504483
        struct[0].Gy_ini[58,10] = 26.5300693004806
        struct[0].Gy_ini[58,11] = 4.84676915585801
        struct[0].Gy_ini[58,12] = -7.89461337514771
        struct[0].Gy_ini[58,13] = 1.58482814504483
        struct[0].Gy_ini[58,14] = -7.72144347466701
        struct[0].Gy_ini[58,15] = 0.679918453285308
        struct[0].Gy_ini[58,48] = -39.6392229058202
        struct[0].Gy_ini[58,49] = -10.2713736253513
        struct[0].Gy_ini[58,50] = 105.571769313180
        struct[0].Gy_ini[58,51] = 54.0657727682604
        struct[0].Gy_ini[58,52] = -39.6392229058201
        struct[0].Gy_ini[58,53] = -10.2713736253513
        struct[0].Gy_ini[58,54] = -24.9575997948692
        struct[0].Gy_ini[58,55] = -23.2849644809540
        struct[0].Gy_ini[58,56] = 87.1730591867880
        struct[0].Gy_ini[58,57] = 18.9579191056579
        struct[0].Gy_ini[58,58] = -237.673607926841
        struct[0].Gy_ini[58,59] = -112.978314692379
        struct[0].Gy_ini[58,60] = 87.1730591867880
        struct[0].Gy_ini[58,61] = 18.9579191056579
        struct[0].Gy_ini[58,62] = 57.6366430644054
        struct[0].Gy_ini[58,63] = 45.8900105086227
        struct[0].Gy_ini[58,64] = -39.6392229058202
        struct[0].Gy_ini[58,65] = -10.2713736253513
        struct[0].Gy_ini[58,66] = 105.571769313180
        struct[0].Gy_ini[58,67] = 54.0657727682604
        struct[0].Gy_ini[58,68] = -39.6392229058201
        struct[0].Gy_ini[58,69] = -10.2713736253513
        struct[0].Gy_ini[58,70] = -24.9575997948692
        struct[0].Gy_ini[58,71] = -23.2849644809540
        struct[0].Gy_ini[59,8] = -1.58482814504483
        struct[0].Gy_ini[59,9] = -7.89461337514771
        struct[0].Gy_ini[59,10] = -4.84676915585801
        struct[0].Gy_ini[59,11] = 26.5300693004806
        struct[0].Gy_ini[59,12] = -1.58482814504483
        struct[0].Gy_ini[59,13] = -7.89461337514771
        struct[0].Gy_ini[59,14] = -0.679918453285308
        struct[0].Gy_ini[59,15] = -7.72144347466701
        struct[0].Gy_ini[59,48] = 10.2713736253513
        struct[0].Gy_ini[59,49] = -39.6392229058202
        struct[0].Gy_ini[59,50] = -54.0657727682604
        struct[0].Gy_ini[59,51] = 105.571769313180
        struct[0].Gy_ini[59,52] = 10.2713736253513
        struct[0].Gy_ini[59,53] = -39.6392229058201
        struct[0].Gy_ini[59,54] = 23.2849644809540
        struct[0].Gy_ini[59,55] = -24.9575997948692
        struct[0].Gy_ini[59,56] = -18.9579191056579
        struct[0].Gy_ini[59,57] = 87.1730591867880
        struct[0].Gy_ini[59,58] = 112.978314692379
        struct[0].Gy_ini[59,59] = -237.673607926841
        struct[0].Gy_ini[59,60] = -18.9579191056579
        struct[0].Gy_ini[59,61] = 87.1730591867880
        struct[0].Gy_ini[59,62] = -45.8900105086227
        struct[0].Gy_ini[59,63] = 57.6366430644054
        struct[0].Gy_ini[59,64] = 10.2713736253513
        struct[0].Gy_ini[59,65] = -39.6392229058202
        struct[0].Gy_ini[59,66] = -54.0657727682604
        struct[0].Gy_ini[59,67] = 105.571769313180
        struct[0].Gy_ini[59,68] = 10.2713736253513
        struct[0].Gy_ini[59,69] = -39.6392229058201
        struct[0].Gy_ini[59,70] = 23.2849644809540
        struct[0].Gy_ini[59,71] = -24.9575997948692
        struct[0].Gy_ini[60,8] = -7.72144347466701
        struct[0].Gy_ini[60,9] = 0.679918453285310
        struct[0].Gy_ini[60,10] = -7.89461337514770
        struct[0].Gy_ini[60,11] = 1.58482814504483
        struct[0].Gy_ini[60,12] = 26.5300693004806
        struct[0].Gy_ini[60,13] = 4.84676915585801
        struct[0].Gy_ini[60,14] = -7.89461337514771
        struct[0].Gy_ini[60,15] = 1.58482814504483
        struct[0].Gy_ini[60,48] = -24.9575997948692
        struct[0].Gy_ini[60,49] = -23.2849644809540
        struct[0].Gy_ini[60,50] = -39.6392229058202
        struct[0].Gy_ini[60,51] = -10.2713736253513
        struct[0].Gy_ini[60,52] = 105.571769313180
        struct[0].Gy_ini[60,53] = 54.0657727682604
        struct[0].Gy_ini[60,54] = -39.6392229058201
        struct[0].Gy_ini[60,55] = -10.2713736253514
        struct[0].Gy_ini[60,56] = 57.6366430644054
        struct[0].Gy_ini[60,57] = 45.8900105086227
        struct[0].Gy_ini[60,58] = 87.1730591867881
        struct[0].Gy_ini[60,59] = 18.9579191056578
        struct[0].Gy_ini[60,60] = -237.673607926841
        struct[0].Gy_ini[60,61] = -112.978314692379
        struct[0].Gy_ini[60,62] = 87.1730591867880
        struct[0].Gy_ini[60,63] = 18.9579191056579
        struct[0].Gy_ini[60,64] = -24.9575997948692
        struct[0].Gy_ini[60,65] = -23.2849644809540
        struct[0].Gy_ini[60,66] = -39.6392229058202
        struct[0].Gy_ini[60,67] = -10.2713736253513
        struct[0].Gy_ini[60,68] = 105.571769313180
        struct[0].Gy_ini[60,69] = 54.0657727682604
        struct[0].Gy_ini[60,70] = -39.6392229058201
        struct[0].Gy_ini[60,71] = -10.2713736253514
        struct[0].Gy_ini[61,8] = -0.679918453285310
        struct[0].Gy_ini[61,9] = -7.72144347466701
        struct[0].Gy_ini[61,10] = -1.58482814504483
        struct[0].Gy_ini[61,11] = -7.89461337514770
        struct[0].Gy_ini[61,12] = -4.84676915585801
        struct[0].Gy_ini[61,13] = 26.5300693004806
        struct[0].Gy_ini[61,14] = -1.58482814504483
        struct[0].Gy_ini[61,15] = -7.89461337514771
        struct[0].Gy_ini[61,48] = 23.2849644809540
        struct[0].Gy_ini[61,49] = -24.9575997948692
        struct[0].Gy_ini[61,50] = 10.2713736253513
        struct[0].Gy_ini[61,51] = -39.6392229058202
        struct[0].Gy_ini[61,52] = -54.0657727682604
        struct[0].Gy_ini[61,53] = 105.571769313180
        struct[0].Gy_ini[61,54] = 10.2713736253514
        struct[0].Gy_ini[61,55] = -39.6392229058201
        struct[0].Gy_ini[61,56] = -45.8900105086227
        struct[0].Gy_ini[61,57] = 57.6366430644054
        struct[0].Gy_ini[61,58] = -18.9579191056578
        struct[0].Gy_ini[61,59] = 87.1730591867881
        struct[0].Gy_ini[61,60] = 112.978314692379
        struct[0].Gy_ini[61,61] = -237.673607926841
        struct[0].Gy_ini[61,62] = -18.9579191056579
        struct[0].Gy_ini[61,63] = 87.1730591867880
        struct[0].Gy_ini[61,64] = 23.2849644809540
        struct[0].Gy_ini[61,65] = -24.9575997948692
        struct[0].Gy_ini[61,66] = 10.2713736253513
        struct[0].Gy_ini[61,67] = -39.6392229058202
        struct[0].Gy_ini[61,68] = -54.0657727682604
        struct[0].Gy_ini[61,69] = 105.571769313180
        struct[0].Gy_ini[61,70] = 10.2713736253514
        struct[0].Gy_ini[61,71] = -39.6392229058201
        struct[0].Gy_ini[62,8] = -7.89461337514771
        struct[0].Gy_ini[62,9] = 1.58482814504483
        struct[0].Gy_ini[62,10] = -7.72144347466701
        struct[0].Gy_ini[62,11] = 0.679918453285308
        struct[0].Gy_ini[62,12] = -7.89461337514771
        struct[0].Gy_ini[62,13] = 1.58482814504483
        struct[0].Gy_ini[62,14] = 26.5300693004806
        struct[0].Gy_ini[62,15] = 4.84676915585801
        struct[0].Gy_ini[62,48] = -39.6392229058202
        struct[0].Gy_ini[62,49] = -10.2713736253513
        struct[0].Gy_ini[62,50] = -24.9575997948692
        struct[0].Gy_ini[62,51] = -23.2849644809540
        struct[0].Gy_ini[62,52] = -39.6392229058202
        struct[0].Gy_ini[62,53] = -10.2713736253513
        struct[0].Gy_ini[62,54] = 105.571769313180
        struct[0].Gy_ini[62,55] = 54.0657727682604
        struct[0].Gy_ini[62,56] = 87.1730591867880
        struct[0].Gy_ini[62,57] = 18.9579191056578
        struct[0].Gy_ini[62,58] = 57.6366430644054
        struct[0].Gy_ini[62,59] = 45.8900105086227
        struct[0].Gy_ini[62,60] = 87.1730591867880
        struct[0].Gy_ini[62,61] = 18.9579191056579
        struct[0].Gy_ini[62,62] = -237.673607926841
        struct[0].Gy_ini[62,63] = -112.978314692379
        struct[0].Gy_ini[62,64] = -39.6392229058202
        struct[0].Gy_ini[62,65] = -10.2713736253513
        struct[0].Gy_ini[62,66] = -24.9575997948692
        struct[0].Gy_ini[62,67] = -23.2849644809540
        struct[0].Gy_ini[62,68] = -39.6392229058202
        struct[0].Gy_ini[62,69] = -10.2713736253513
        struct[0].Gy_ini[62,70] = 105.571769313180
        struct[0].Gy_ini[62,71] = 54.0657727682604
        struct[0].Gy_ini[63,8] = -1.58482814504483
        struct[0].Gy_ini[63,9] = -7.89461337514771
        struct[0].Gy_ini[63,10] = -0.679918453285308
        struct[0].Gy_ini[63,11] = -7.72144347466701
        struct[0].Gy_ini[63,12] = -1.58482814504483
        struct[0].Gy_ini[63,13] = -7.89461337514771
        struct[0].Gy_ini[63,14] = -4.84676915585801
        struct[0].Gy_ini[63,15] = 26.5300693004806
        struct[0].Gy_ini[63,48] = 10.2713736253513
        struct[0].Gy_ini[63,49] = -39.6392229058202
        struct[0].Gy_ini[63,50] = 23.2849644809540
        struct[0].Gy_ini[63,51] = -24.9575997948692
        struct[0].Gy_ini[63,52] = 10.2713736253513
        struct[0].Gy_ini[63,53] = -39.6392229058202
        struct[0].Gy_ini[63,54] = -54.0657727682604
        struct[0].Gy_ini[63,55] = 105.571769313180
        struct[0].Gy_ini[63,56] = -18.9579191056578
        struct[0].Gy_ini[63,57] = 87.1730591867880
        struct[0].Gy_ini[63,58] = -45.8900105086227
        struct[0].Gy_ini[63,59] = 57.6366430644054
        struct[0].Gy_ini[63,60] = -18.9579191056579
        struct[0].Gy_ini[63,61] = 87.1730591867880
        struct[0].Gy_ini[63,62] = 112.978314692379
        struct[0].Gy_ini[63,63] = -237.673607926841
        struct[0].Gy_ini[63,64] = 10.2713736253513
        struct[0].Gy_ini[63,65] = -39.6392229058202
        struct[0].Gy_ini[63,66] = 23.2849644809540
        struct[0].Gy_ini[63,67] = -24.9575997948692
        struct[0].Gy_ini[63,68] = 10.2713736253513
        struct[0].Gy_ini[63,69] = -39.6392229058202
        struct[0].Gy_ini[63,70] = -54.0657727682604
        struct[0].Gy_ini[63,71] = 105.571769313180
        struct[0].Gy_ini[64,56] = 105.571769313180
        struct[0].Gy_ini[64,57] = 54.0657727682604
        struct[0].Gy_ini[64,58] = -39.6392229058202
        struct[0].Gy_ini[64,59] = -10.2713736253513
        struct[0].Gy_ini[64,60] = -24.9575997948692
        struct[0].Gy_ini[64,61] = -23.2849644809540
        struct[0].Gy_ini[64,62] = -39.6392229058202
        struct[0].Gy_ini[64,63] = -10.2713736253513
        struct[0].Gy_ini[64,64] = -237.673607926841
        struct[0].Gy_ini[64,65] = -112.978314692379
        struct[0].Gy_ini[64,66] = 87.1730591867880
        struct[0].Gy_ini[64,67] = 18.9579191056578
        struct[0].Gy_ini[64,68] = 57.6366430644055
        struct[0].Gy_ini[64,69] = 45.8900105086227
        struct[0].Gy_ini[64,70] = 87.1730591867880
        struct[0].Gy_ini[64,71] = 18.9579191056579
        struct[0].Gy_ini[64,72] = 105.571769313180
        struct[0].Gy_ini[64,73] = 54.0657727682604
        struct[0].Gy_ini[64,74] = -39.6392229058202
        struct[0].Gy_ini[64,75] = -10.2713736253513
        struct[0].Gy_ini[64,76] = -24.9575997948692
        struct[0].Gy_ini[64,77] = -23.2849644809540
        struct[0].Gy_ini[64,78] = -39.6392229058202
        struct[0].Gy_ini[64,79] = -10.2713736253513
        struct[0].Gy_ini[64,120] = 26.5300693004806
        struct[0].Gy_ini[64,121] = 4.84676915585801
        struct[0].Gy_ini[64,122] = -7.89461337514770
        struct[0].Gy_ini[64,123] = 1.58482814504483
        struct[0].Gy_ini[64,124] = -7.72144347466701
        struct[0].Gy_ini[64,125] = 0.679918453285310
        struct[0].Gy_ini[64,126] = -7.89461337514771
        struct[0].Gy_ini[64,127] = 1.58482814504483
        struct[0].Gy_ini[65,56] = -54.0657727682604
        struct[0].Gy_ini[65,57] = 105.571769313180
        struct[0].Gy_ini[65,58] = 10.2713736253513
        struct[0].Gy_ini[65,59] = -39.6392229058202
        struct[0].Gy_ini[65,60] = 23.2849644809540
        struct[0].Gy_ini[65,61] = -24.9575997948692
        struct[0].Gy_ini[65,62] = 10.2713736253513
        struct[0].Gy_ini[65,63] = -39.6392229058202
        struct[0].Gy_ini[65,64] = 112.978314692379
        struct[0].Gy_ini[65,65] = -237.673607926841
        struct[0].Gy_ini[65,66] = -18.9579191056578
        struct[0].Gy_ini[65,67] = 87.1730591867880
        struct[0].Gy_ini[65,68] = -45.8900105086227
        struct[0].Gy_ini[65,69] = 57.6366430644055
        struct[0].Gy_ini[65,70] = -18.9579191056579
        struct[0].Gy_ini[65,71] = 87.1730591867880
        struct[0].Gy_ini[65,72] = -54.0657727682604
        struct[0].Gy_ini[65,73] = 105.571769313180
        struct[0].Gy_ini[65,74] = 10.2713736253513
        struct[0].Gy_ini[65,75] = -39.6392229058202
        struct[0].Gy_ini[65,76] = 23.2849644809540
        struct[0].Gy_ini[65,77] = -24.9575997948692
        struct[0].Gy_ini[65,78] = 10.2713736253513
        struct[0].Gy_ini[65,79] = -39.6392229058202
        struct[0].Gy_ini[65,120] = -4.84676915585801
        struct[0].Gy_ini[65,121] = 26.5300693004806
        struct[0].Gy_ini[65,122] = -1.58482814504483
        struct[0].Gy_ini[65,123] = -7.89461337514770
        struct[0].Gy_ini[65,124] = -0.679918453285310
        struct[0].Gy_ini[65,125] = -7.72144347466701
        struct[0].Gy_ini[65,126] = -1.58482814504483
        struct[0].Gy_ini[65,127] = -7.89461337514771
        struct[0].Gy_ini[66,56] = -39.6392229058202
        struct[0].Gy_ini[66,57] = -10.2713736253513
        struct[0].Gy_ini[66,58] = 105.571769313180
        struct[0].Gy_ini[66,59] = 54.0657727682604
        struct[0].Gy_ini[66,60] = -39.6392229058201
        struct[0].Gy_ini[66,61] = -10.2713736253513
        struct[0].Gy_ini[66,62] = -24.9575997948692
        struct[0].Gy_ini[66,63] = -23.2849644809540
        struct[0].Gy_ini[66,64] = 87.1730591867880
        struct[0].Gy_ini[66,65] = 18.9579191056579
        struct[0].Gy_ini[66,66] = -237.673607926841
        struct[0].Gy_ini[66,67] = -112.978314692379
        struct[0].Gy_ini[66,68] = 87.1730591867880
        struct[0].Gy_ini[66,69] = 18.9579191056579
        struct[0].Gy_ini[66,70] = 57.6366430644054
        struct[0].Gy_ini[66,71] = 45.8900105086227
        struct[0].Gy_ini[66,72] = -39.6392229058202
        struct[0].Gy_ini[66,73] = -10.2713736253513
        struct[0].Gy_ini[66,74] = 105.571769313180
        struct[0].Gy_ini[66,75] = 54.0657727682604
        struct[0].Gy_ini[66,76] = -39.6392229058201
        struct[0].Gy_ini[66,77] = -10.2713736253513
        struct[0].Gy_ini[66,78] = -24.9575997948692
        struct[0].Gy_ini[66,79] = -23.2849644809540
        struct[0].Gy_ini[66,120] = -7.89461337514771
        struct[0].Gy_ini[66,121] = 1.58482814504483
        struct[0].Gy_ini[66,122] = 26.5300693004806
        struct[0].Gy_ini[66,123] = 4.84676915585801
        struct[0].Gy_ini[66,124] = -7.89461337514771
        struct[0].Gy_ini[66,125] = 1.58482814504483
        struct[0].Gy_ini[66,126] = -7.72144347466701
        struct[0].Gy_ini[66,127] = 0.679918453285308
        struct[0].Gy_ini[67,56] = 10.2713736253513
        struct[0].Gy_ini[67,57] = -39.6392229058202
        struct[0].Gy_ini[67,58] = -54.0657727682604
        struct[0].Gy_ini[67,59] = 105.571769313180
        struct[0].Gy_ini[67,60] = 10.2713736253513
        struct[0].Gy_ini[67,61] = -39.6392229058201
        struct[0].Gy_ini[67,62] = 23.2849644809540
        struct[0].Gy_ini[67,63] = -24.9575997948692
        struct[0].Gy_ini[67,64] = -18.9579191056579
        struct[0].Gy_ini[67,65] = 87.1730591867880
        struct[0].Gy_ini[67,66] = 112.978314692379
        struct[0].Gy_ini[67,67] = -237.673607926841
        struct[0].Gy_ini[67,68] = -18.9579191056579
        struct[0].Gy_ini[67,69] = 87.1730591867880
        struct[0].Gy_ini[67,70] = -45.8900105086227
        struct[0].Gy_ini[67,71] = 57.6366430644054
        struct[0].Gy_ini[67,72] = 10.2713736253513
        struct[0].Gy_ini[67,73] = -39.6392229058202
        struct[0].Gy_ini[67,74] = -54.0657727682604
        struct[0].Gy_ini[67,75] = 105.571769313180
        struct[0].Gy_ini[67,76] = 10.2713736253513
        struct[0].Gy_ini[67,77] = -39.6392229058201
        struct[0].Gy_ini[67,78] = 23.2849644809540
        struct[0].Gy_ini[67,79] = -24.9575997948692
        struct[0].Gy_ini[67,120] = -1.58482814504483
        struct[0].Gy_ini[67,121] = -7.89461337514771
        struct[0].Gy_ini[67,122] = -4.84676915585801
        struct[0].Gy_ini[67,123] = 26.5300693004806
        struct[0].Gy_ini[67,124] = -1.58482814504483
        struct[0].Gy_ini[67,125] = -7.89461337514771
        struct[0].Gy_ini[67,126] = -0.679918453285308
        struct[0].Gy_ini[67,127] = -7.72144347466701
        struct[0].Gy_ini[68,56] = -24.9575997948692
        struct[0].Gy_ini[68,57] = -23.2849644809540
        struct[0].Gy_ini[68,58] = -39.6392229058202
        struct[0].Gy_ini[68,59] = -10.2713736253513
        struct[0].Gy_ini[68,60] = 105.571769313180
        struct[0].Gy_ini[68,61] = 54.0657727682604
        struct[0].Gy_ini[68,62] = -39.6392229058201
        struct[0].Gy_ini[68,63] = -10.2713736253514
        struct[0].Gy_ini[68,64] = 57.6366430644054
        struct[0].Gy_ini[68,65] = 45.8900105086227
        struct[0].Gy_ini[68,66] = 87.1730591867881
        struct[0].Gy_ini[68,67] = 18.9579191056578
        struct[0].Gy_ini[68,68] = -237.673607926841
        struct[0].Gy_ini[68,69] = -112.978314692379
        struct[0].Gy_ini[68,70] = 87.1730591867880
        struct[0].Gy_ini[68,71] = 18.9579191056579
        struct[0].Gy_ini[68,72] = -24.9575997948692
        struct[0].Gy_ini[68,73] = -23.2849644809540
        struct[0].Gy_ini[68,74] = -39.6392229058202
        struct[0].Gy_ini[68,75] = -10.2713736253513
        struct[0].Gy_ini[68,76] = 105.571769313180
        struct[0].Gy_ini[68,77] = 54.0657727682604
        struct[0].Gy_ini[68,78] = -39.6392229058201
        struct[0].Gy_ini[68,79] = -10.2713736253514
        struct[0].Gy_ini[68,120] = -7.72144347466701
        struct[0].Gy_ini[68,121] = 0.679918453285310
        struct[0].Gy_ini[68,122] = -7.89461337514770
        struct[0].Gy_ini[68,123] = 1.58482814504483
        struct[0].Gy_ini[68,124] = 26.5300693004806
        struct[0].Gy_ini[68,125] = 4.84676915585801
        struct[0].Gy_ini[68,126] = -7.89461337514771
        struct[0].Gy_ini[68,127] = 1.58482814504483
        struct[0].Gy_ini[69,56] = 23.2849644809540
        struct[0].Gy_ini[69,57] = -24.9575997948692
        struct[0].Gy_ini[69,58] = 10.2713736253513
        struct[0].Gy_ini[69,59] = -39.6392229058202
        struct[0].Gy_ini[69,60] = -54.0657727682604
        struct[0].Gy_ini[69,61] = 105.571769313180
        struct[0].Gy_ini[69,62] = 10.2713736253514
        struct[0].Gy_ini[69,63] = -39.6392229058201
        struct[0].Gy_ini[69,64] = -45.8900105086227
        struct[0].Gy_ini[69,65] = 57.6366430644054
        struct[0].Gy_ini[69,66] = -18.9579191056578
        struct[0].Gy_ini[69,67] = 87.1730591867881
        struct[0].Gy_ini[69,68] = 112.978314692379
        struct[0].Gy_ini[69,69] = -237.673607926841
        struct[0].Gy_ini[69,70] = -18.9579191056579
        struct[0].Gy_ini[69,71] = 87.1730591867880
        struct[0].Gy_ini[69,72] = 23.2849644809540
        struct[0].Gy_ini[69,73] = -24.9575997948692
        struct[0].Gy_ini[69,74] = 10.2713736253513
        struct[0].Gy_ini[69,75] = -39.6392229058202
        struct[0].Gy_ini[69,76] = -54.0657727682604
        struct[0].Gy_ini[69,77] = 105.571769313180
        struct[0].Gy_ini[69,78] = 10.2713736253514
        struct[0].Gy_ini[69,79] = -39.6392229058201
        struct[0].Gy_ini[69,120] = -0.679918453285310
        struct[0].Gy_ini[69,121] = -7.72144347466701
        struct[0].Gy_ini[69,122] = -1.58482814504483
        struct[0].Gy_ini[69,123] = -7.89461337514770
        struct[0].Gy_ini[69,124] = -4.84676915585801
        struct[0].Gy_ini[69,125] = 26.5300693004806
        struct[0].Gy_ini[69,126] = -1.58482814504483
        struct[0].Gy_ini[69,127] = -7.89461337514771
        struct[0].Gy_ini[70,56] = -39.6392229058202
        struct[0].Gy_ini[70,57] = -10.2713736253513
        struct[0].Gy_ini[70,58] = -24.9575997948692
        struct[0].Gy_ini[70,59] = -23.2849644809540
        struct[0].Gy_ini[70,60] = -39.6392229058202
        struct[0].Gy_ini[70,61] = -10.2713736253513
        struct[0].Gy_ini[70,62] = 105.571769313180
        struct[0].Gy_ini[70,63] = 54.0657727682604
        struct[0].Gy_ini[70,64] = 87.1730591867880
        struct[0].Gy_ini[70,65] = 18.9579191056578
        struct[0].Gy_ini[70,66] = 57.6366430644054
        struct[0].Gy_ini[70,67] = 45.8900105086227
        struct[0].Gy_ini[70,68] = 87.1730591867880
        struct[0].Gy_ini[70,69] = 18.9579191056579
        struct[0].Gy_ini[70,70] = -237.698607926841
        struct[0].Gy_ini[70,71] = -112.978314692379
        struct[0].Gy_ini[70,72] = -39.6392229058202
        struct[0].Gy_ini[70,73] = -10.2713736253513
        struct[0].Gy_ini[70,74] = -24.9575997948692
        struct[0].Gy_ini[70,75] = -23.2849644809540
        struct[0].Gy_ini[70,76] = -39.6392229058202
        struct[0].Gy_ini[70,77] = -10.2713736253513
        struct[0].Gy_ini[70,78] = 105.571769313180
        struct[0].Gy_ini[70,79] = 54.0657727682604
        struct[0].Gy_ini[70,120] = -7.89461337514771
        struct[0].Gy_ini[70,121] = 1.58482814504483
        struct[0].Gy_ini[70,122] = -7.72144347466701
        struct[0].Gy_ini[70,123] = 0.679918453285308
        struct[0].Gy_ini[70,124] = -7.89461337514771
        struct[0].Gy_ini[70,125] = 1.58482814504483
        struct[0].Gy_ini[70,126] = 26.5300693004806
        struct[0].Gy_ini[70,127] = 4.84676915585801
        struct[0].Gy_ini[71,56] = 10.2713736253513
        struct[0].Gy_ini[71,57] = -39.6392229058202
        struct[0].Gy_ini[71,58] = 23.2849644809540
        struct[0].Gy_ini[71,59] = -24.9575997948692
        struct[0].Gy_ini[71,60] = 10.2713736253513
        struct[0].Gy_ini[71,61] = -39.6392229058202
        struct[0].Gy_ini[71,62] = -54.0657727682604
        struct[0].Gy_ini[71,63] = 105.571769313180
        struct[0].Gy_ini[71,64] = -18.9579191056578
        struct[0].Gy_ini[71,65] = 87.1730591867880
        struct[0].Gy_ini[71,66] = -45.8900105086227
        struct[0].Gy_ini[71,67] = 57.6366430644054
        struct[0].Gy_ini[71,68] = -18.9579191056579
        struct[0].Gy_ini[71,69] = 87.1730591867880
        struct[0].Gy_ini[71,70] = 112.978314692379
        struct[0].Gy_ini[71,71] = -237.698607926841
        struct[0].Gy_ini[71,72] = 10.2713736253513
        struct[0].Gy_ini[71,73] = -39.6392229058202
        struct[0].Gy_ini[71,74] = 23.2849644809540
        struct[0].Gy_ini[71,75] = -24.9575997948692
        struct[0].Gy_ini[71,76] = 10.2713736253513
        struct[0].Gy_ini[71,77] = -39.6392229058202
        struct[0].Gy_ini[71,78] = -54.0657727682604
        struct[0].Gy_ini[71,79] = 105.571769313180
        struct[0].Gy_ini[71,120] = -1.58482814504483
        struct[0].Gy_ini[71,121] = -7.89461337514771
        struct[0].Gy_ini[71,122] = -0.679918453285308
        struct[0].Gy_ini[71,123] = -7.72144347466701
        struct[0].Gy_ini[71,124] = -1.58482814504483
        struct[0].Gy_ini[71,125] = -7.89461337514771
        struct[0].Gy_ini[71,126] = -4.84676915585801
        struct[0].Gy_ini[71,127] = 26.5300693004806
        struct[0].Gy_ini[72,64] = 105.571769313180
        struct[0].Gy_ini[72,65] = 54.0657727682604
        struct[0].Gy_ini[72,66] = -39.6392229058202
        struct[0].Gy_ini[72,67] = -10.2713736253513
        struct[0].Gy_ini[72,68] = -24.9575997948692
        struct[0].Gy_ini[72,69] = -23.2849644809540
        struct[0].Gy_ini[72,70] = -39.6392229058202
        struct[0].Gy_ini[72,71] = -10.2713736253513
        struct[0].Gy_ini[72,72] = -211.143538626360
        struct[0].Gy_ini[72,73] = -108.131545536521
        struct[0].Gy_ini[72,74] = 79.2784458116403
        struct[0].Gy_ini[72,75] = 20.5427472507027
        struct[0].Gy_ini[72,76] = 49.9151995897384
        struct[0].Gy_ini[72,77] = 46.5699289619080
        struct[0].Gy_ini[72,78] = 79.2784458116403
        struct[0].Gy_ini[72,79] = 20.5427472507027
        struct[0].Gy_ini[72,80] = 105.571769313180
        struct[0].Gy_ini[72,81] = 54.0657727682604
        struct[0].Gy_ini[72,82] = -39.6392229058202
        struct[0].Gy_ini[72,83] = -10.2713736253513
        struct[0].Gy_ini[72,84] = -24.9575997948692
        struct[0].Gy_ini[72,85] = -23.2849644809540
        struct[0].Gy_ini[72,86] = -39.6392229058202
        struct[0].Gy_ini[72,87] = -10.2713736253513
        struct[0].Gy_ini[73,64] = -54.0657727682604
        struct[0].Gy_ini[73,65] = 105.571769313180
        struct[0].Gy_ini[73,66] = 10.2713736253513
        struct[0].Gy_ini[73,67] = -39.6392229058202
        struct[0].Gy_ini[73,68] = 23.2849644809540
        struct[0].Gy_ini[73,69] = -24.9575997948692
        struct[0].Gy_ini[73,70] = 10.2713736253513
        struct[0].Gy_ini[73,71] = -39.6392229058202
        struct[0].Gy_ini[73,72] = 108.131545536521
        struct[0].Gy_ini[73,73] = -211.143538626360
        struct[0].Gy_ini[73,74] = -20.5427472507027
        struct[0].Gy_ini[73,75] = 79.2784458116403
        struct[0].Gy_ini[73,76] = -46.5699289619080
        struct[0].Gy_ini[73,77] = 49.9151995897384
        struct[0].Gy_ini[73,78] = -20.5427472507027
        struct[0].Gy_ini[73,79] = 79.2784458116403
        struct[0].Gy_ini[73,80] = -54.0657727682604
        struct[0].Gy_ini[73,81] = 105.571769313180
        struct[0].Gy_ini[73,82] = 10.2713736253513
        struct[0].Gy_ini[73,83] = -39.6392229058202
        struct[0].Gy_ini[73,84] = 23.2849644809540
        struct[0].Gy_ini[73,85] = -24.9575997948692
        struct[0].Gy_ini[73,86] = 10.2713736253513
        struct[0].Gy_ini[73,87] = -39.6392229058202
        struct[0].Gy_ini[74,64] = -39.6392229058202
        struct[0].Gy_ini[74,65] = -10.2713736253513
        struct[0].Gy_ini[74,66] = 105.571769313180
        struct[0].Gy_ini[74,67] = 54.0657727682604
        struct[0].Gy_ini[74,68] = -39.6392229058201
        struct[0].Gy_ini[74,69] = -10.2713736253513
        struct[0].Gy_ini[74,70] = -24.9575997948692
        struct[0].Gy_ini[74,71] = -23.2849644809540
        struct[0].Gy_ini[74,72] = 79.2784458116403
        struct[0].Gy_ini[74,73] = 20.5427472507027
        struct[0].Gy_ini[74,74] = -211.143538626360
        struct[0].Gy_ini[74,75] = -108.131545536521
        struct[0].Gy_ini[74,76] = 79.2784458116403
        struct[0].Gy_ini[74,77] = 20.5427472507027
        struct[0].Gy_ini[74,78] = 49.9151995897384
        struct[0].Gy_ini[74,79] = 46.5699289619080
        struct[0].Gy_ini[74,80] = -39.6392229058202
        struct[0].Gy_ini[74,81] = -10.2713736253513
        struct[0].Gy_ini[74,82] = 105.571769313180
        struct[0].Gy_ini[74,83] = 54.0657727682604
        struct[0].Gy_ini[74,84] = -39.6392229058201
        struct[0].Gy_ini[74,85] = -10.2713736253513
        struct[0].Gy_ini[74,86] = -24.9575997948692
        struct[0].Gy_ini[74,87] = -23.2849644809540
        struct[0].Gy_ini[75,64] = 10.2713736253513
        struct[0].Gy_ini[75,65] = -39.6392229058202
        struct[0].Gy_ini[75,66] = -54.0657727682604
        struct[0].Gy_ini[75,67] = 105.571769313180
        struct[0].Gy_ini[75,68] = 10.2713736253513
        struct[0].Gy_ini[75,69] = -39.6392229058201
        struct[0].Gy_ini[75,70] = 23.2849644809540
        struct[0].Gy_ini[75,71] = -24.9575997948692
        struct[0].Gy_ini[75,72] = -20.5427472507027
        struct[0].Gy_ini[75,73] = 79.2784458116403
        struct[0].Gy_ini[75,74] = 108.131545536521
        struct[0].Gy_ini[75,75] = -211.143538626360
        struct[0].Gy_ini[75,76] = -20.5427472507027
        struct[0].Gy_ini[75,77] = 79.2784458116403
        struct[0].Gy_ini[75,78] = -46.5699289619080
        struct[0].Gy_ini[75,79] = 49.9151995897384
        struct[0].Gy_ini[75,80] = 10.2713736253513
        struct[0].Gy_ini[75,81] = -39.6392229058202
        struct[0].Gy_ini[75,82] = -54.0657727682604
        struct[0].Gy_ini[75,83] = 105.571769313180
        struct[0].Gy_ini[75,84] = 10.2713736253513
        struct[0].Gy_ini[75,85] = -39.6392229058201
        struct[0].Gy_ini[75,86] = 23.2849644809540
        struct[0].Gy_ini[75,87] = -24.9575997948692
        struct[0].Gy_ini[76,64] = -24.9575997948692
        struct[0].Gy_ini[76,65] = -23.2849644809540
        struct[0].Gy_ini[76,66] = -39.6392229058202
        struct[0].Gy_ini[76,67] = -10.2713736253513
        struct[0].Gy_ini[76,68] = 105.571769313180
        struct[0].Gy_ini[76,69] = 54.0657727682604
        struct[0].Gy_ini[76,70] = -39.6392229058201
        struct[0].Gy_ini[76,71] = -10.2713736253514
        struct[0].Gy_ini[76,72] = 49.9151995897384
        struct[0].Gy_ini[76,73] = 46.5699289619080
        struct[0].Gy_ini[76,74] = 79.2784458116404
        struct[0].Gy_ini[76,75] = 20.5427472507027
        struct[0].Gy_ini[76,76] = -211.143538626360
        struct[0].Gy_ini[76,77] = -108.131545536521
        struct[0].Gy_ini[76,78] = 79.2784458116403
        struct[0].Gy_ini[76,79] = 20.5427472507027
        struct[0].Gy_ini[76,80] = -24.9575997948692
        struct[0].Gy_ini[76,81] = -23.2849644809540
        struct[0].Gy_ini[76,82] = -39.6392229058202
        struct[0].Gy_ini[76,83] = -10.2713736253513
        struct[0].Gy_ini[76,84] = 105.571769313180
        struct[0].Gy_ini[76,85] = 54.0657727682604
        struct[0].Gy_ini[76,86] = -39.6392229058201
        struct[0].Gy_ini[76,87] = -10.2713736253514
        struct[0].Gy_ini[77,64] = 23.2849644809540
        struct[0].Gy_ini[77,65] = -24.9575997948692
        struct[0].Gy_ini[77,66] = 10.2713736253513
        struct[0].Gy_ini[77,67] = -39.6392229058202
        struct[0].Gy_ini[77,68] = -54.0657727682604
        struct[0].Gy_ini[77,69] = 105.571769313180
        struct[0].Gy_ini[77,70] = 10.2713736253514
        struct[0].Gy_ini[77,71] = -39.6392229058201
        struct[0].Gy_ini[77,72] = -46.5699289619080
        struct[0].Gy_ini[77,73] = 49.9151995897384
        struct[0].Gy_ini[77,74] = -20.5427472507027
        struct[0].Gy_ini[77,75] = 79.2784458116404
        struct[0].Gy_ini[77,76] = 108.131545536521
        struct[0].Gy_ini[77,77] = -211.143538626360
        struct[0].Gy_ini[77,78] = -20.5427472507027
        struct[0].Gy_ini[77,79] = 79.2784458116403
        struct[0].Gy_ini[77,80] = 23.2849644809540
        struct[0].Gy_ini[77,81] = -24.9575997948692
        struct[0].Gy_ini[77,82] = 10.2713736253513
        struct[0].Gy_ini[77,83] = -39.6392229058202
        struct[0].Gy_ini[77,84] = -54.0657727682604
        struct[0].Gy_ini[77,85] = 105.571769313180
        struct[0].Gy_ini[77,86] = 10.2713736253514
        struct[0].Gy_ini[77,87] = -39.6392229058201
        struct[0].Gy_ini[78,64] = -39.6392229058202
        struct[0].Gy_ini[78,65] = -10.2713736253513
        struct[0].Gy_ini[78,66] = -24.9575997948692
        struct[0].Gy_ini[78,67] = -23.2849644809540
        struct[0].Gy_ini[78,68] = -39.6392229058202
        struct[0].Gy_ini[78,69] = -10.2713736253513
        struct[0].Gy_ini[78,70] = 105.571769313180
        struct[0].Gy_ini[78,71] = 54.0657727682604
        struct[0].Gy_ini[78,72] = 79.2784458116403
        struct[0].Gy_ini[78,73] = 20.5427472507027
        struct[0].Gy_ini[78,74] = 49.9151995897384
        struct[0].Gy_ini[78,75] = 46.5699289619080
        struct[0].Gy_ini[78,76] = 79.2784458116403
        struct[0].Gy_ini[78,77] = 20.5427472507027
        struct[0].Gy_ini[78,78] = -211.143538626360
        struct[0].Gy_ini[78,79] = -108.131545536521
        struct[0].Gy_ini[78,80] = -39.6392229058202
        struct[0].Gy_ini[78,81] = -10.2713736253513
        struct[0].Gy_ini[78,82] = -24.9575997948692
        struct[0].Gy_ini[78,83] = -23.2849644809540
        struct[0].Gy_ini[78,84] = -39.6392229058202
        struct[0].Gy_ini[78,85] = -10.2713736253513
        struct[0].Gy_ini[78,86] = 105.571769313180
        struct[0].Gy_ini[78,87] = 54.0657727682604
        struct[0].Gy_ini[79,64] = 10.2713736253513
        struct[0].Gy_ini[79,65] = -39.6392229058202
        struct[0].Gy_ini[79,66] = 23.2849644809540
        struct[0].Gy_ini[79,67] = -24.9575997948692
        struct[0].Gy_ini[79,68] = 10.2713736253513
        struct[0].Gy_ini[79,69] = -39.6392229058202
        struct[0].Gy_ini[79,70] = -54.0657727682604
        struct[0].Gy_ini[79,71] = 105.571769313180
        struct[0].Gy_ini[79,72] = -20.5427472507027
        struct[0].Gy_ini[79,73] = 79.2784458116403
        struct[0].Gy_ini[79,74] = -46.5699289619080
        struct[0].Gy_ini[79,75] = 49.9151995897384
        struct[0].Gy_ini[79,76] = -20.5427472507027
        struct[0].Gy_ini[79,77] = 79.2784458116403
        struct[0].Gy_ini[79,78] = 108.131545536521
        struct[0].Gy_ini[79,79] = -211.143538626360
        struct[0].Gy_ini[79,80] = 10.2713736253513
        struct[0].Gy_ini[79,81] = -39.6392229058202
        struct[0].Gy_ini[79,82] = 23.2849644809540
        struct[0].Gy_ini[79,83] = -24.9575997948692
        struct[0].Gy_ini[79,84] = 10.2713736253513
        struct[0].Gy_ini[79,85] = -39.6392229058202
        struct[0].Gy_ini[79,86] = -54.0657727682604
        struct[0].Gy_ini[79,87] = 105.571769313180
        struct[0].Gy_ini[80,24] = 30.9517475172273
        struct[0].Gy_ini[80,25] = 5.65456401516768
        struct[0].Gy_ini[80,26] = -9.21038227100566
        struct[0].Gy_ini[80,27] = 1.84896616921897
        struct[0].Gy_ini[80,28] = -9.00835072044485
        struct[0].Gy_ini[80,29] = 0.793238195499529
        struct[0].Gy_ini[80,30] = -9.21038227100566
        struct[0].Gy_ini[80,31] = 1.84896616921897
        struct[0].Gy_ini[80,72] = 105.571769313180
        struct[0].Gy_ini[80,73] = 54.0657727682604
        struct[0].Gy_ini[80,74] = -39.6392229058202
        struct[0].Gy_ini[80,75] = -10.2713736253513
        struct[0].Gy_ini[80,76] = -24.9575997948692
        struct[0].Gy_ini[80,77] = -23.2849644809540
        struct[0].Gy_ini[80,78] = -39.6392229058202
        struct[0].Gy_ini[80,79] = -10.2713736253513
        struct[0].Gy_ini[80,80] = -242.095286143588
        struct[0].Gy_ini[80,81] = -113.786109551688
        struct[0].Gy_ini[80,82] = 88.4888280826460
        struct[0].Gy_ini[80,83] = 18.6937810814837
        struct[0].Gy_ini[80,84] = 58.9235503101833
        struct[0].Gy_ini[80,85] = 45.7766907664085
        struct[0].Gy_ini[80,86] = 88.4888280826460
        struct[0].Gy_ini[80,87] = 18.6937810814837
        struct[0].Gy_ini[80,88] = 105.571769313180
        struct[0].Gy_ini[80,89] = 54.0657727682604
        struct[0].Gy_ini[80,90] = -39.6392229058202
        struct[0].Gy_ini[80,91] = -10.2713736253513
        struct[0].Gy_ini[80,92] = -24.9575997948692
        struct[0].Gy_ini[80,93] = -23.2849644809540
        struct[0].Gy_ini[80,94] = -39.6392229058202
        struct[0].Gy_ini[80,95] = -10.2713736253513
        struct[0].Gy_ini[81,24] = -5.65456401516768
        struct[0].Gy_ini[81,25] = 30.9517475172273
        struct[0].Gy_ini[81,26] = -1.84896616921897
        struct[0].Gy_ini[81,27] = -9.21038227100566
        struct[0].Gy_ini[81,28] = -0.793238195499529
        struct[0].Gy_ini[81,29] = -9.00835072044485
        struct[0].Gy_ini[81,30] = -1.84896616921897
        struct[0].Gy_ini[81,31] = -9.21038227100566
        struct[0].Gy_ini[81,72] = -54.0657727682604
        struct[0].Gy_ini[81,73] = 105.571769313180
        struct[0].Gy_ini[81,74] = 10.2713736253513
        struct[0].Gy_ini[81,75] = -39.6392229058202
        struct[0].Gy_ini[81,76] = 23.2849644809540
        struct[0].Gy_ini[81,77] = -24.9575997948692
        struct[0].Gy_ini[81,78] = 10.2713736253513
        struct[0].Gy_ini[81,79] = -39.6392229058202
        struct[0].Gy_ini[81,80] = 113.786109551688
        struct[0].Gy_ini[81,81] = -242.095286143588
        struct[0].Gy_ini[81,82] = -18.6937810814837
        struct[0].Gy_ini[81,83] = 88.4888280826460
        struct[0].Gy_ini[81,84] = -45.7766907664085
        struct[0].Gy_ini[81,85] = 58.9235503101833
        struct[0].Gy_ini[81,86] = -18.6937810814837
        struct[0].Gy_ini[81,87] = 88.4888280826460
        struct[0].Gy_ini[81,88] = -54.0657727682604
        struct[0].Gy_ini[81,89] = 105.571769313180
        struct[0].Gy_ini[81,90] = 10.2713736253513
        struct[0].Gy_ini[81,91] = -39.6392229058202
        struct[0].Gy_ini[81,92] = 23.2849644809540
        struct[0].Gy_ini[81,93] = -24.9575997948692
        struct[0].Gy_ini[81,94] = 10.2713736253513
        struct[0].Gy_ini[81,95] = -39.6392229058202
        struct[0].Gy_ini[82,24] = -9.21038227100566
        struct[0].Gy_ini[82,25] = 1.84896616921897
        struct[0].Gy_ini[82,26] = 30.9517475172273
        struct[0].Gy_ini[82,27] = 5.65456401516768
        struct[0].Gy_ini[82,28] = -9.21038227100566
        struct[0].Gy_ini[82,29] = 1.84896616921897
        struct[0].Gy_ini[82,30] = -9.00835072044485
        struct[0].Gy_ini[82,31] = 0.793238195499528
        struct[0].Gy_ini[82,72] = -39.6392229058202
        struct[0].Gy_ini[82,73] = -10.2713736253513
        struct[0].Gy_ini[82,74] = 105.571769313180
        struct[0].Gy_ini[82,75] = 54.0657727682604
        struct[0].Gy_ini[82,76] = -39.6392229058201
        struct[0].Gy_ini[82,77] = -10.2713736253513
        struct[0].Gy_ini[82,78] = -24.9575997948692
        struct[0].Gy_ini[82,79] = -23.2849644809540
        struct[0].Gy_ini[82,80] = 88.4888280826460
        struct[0].Gy_ini[82,81] = 18.6937810814837
        struct[0].Gy_ini[82,82] = -242.095286143588
        struct[0].Gy_ini[82,83] = -113.786109551688
        struct[0].Gy_ini[82,84] = 88.4888280826460
        struct[0].Gy_ini[82,85] = 18.6937810814837
        struct[0].Gy_ini[82,86] = 58.9235503101833
        struct[0].Gy_ini[82,87] = 45.7766907664085
        struct[0].Gy_ini[82,88] = -39.6392229058202
        struct[0].Gy_ini[82,89] = -10.2713736253513
        struct[0].Gy_ini[82,90] = 105.571769313180
        struct[0].Gy_ini[82,91] = 54.0657727682604
        struct[0].Gy_ini[82,92] = -39.6392229058201
        struct[0].Gy_ini[82,93] = -10.2713736253513
        struct[0].Gy_ini[82,94] = -24.9575997948692
        struct[0].Gy_ini[82,95] = -23.2849644809540
        struct[0].Gy_ini[83,24] = -1.84896616921897
        struct[0].Gy_ini[83,25] = -9.21038227100566
        struct[0].Gy_ini[83,26] = -5.65456401516768
        struct[0].Gy_ini[83,27] = 30.9517475172273
        struct[0].Gy_ini[83,28] = -1.84896616921897
        struct[0].Gy_ini[83,29] = -9.21038227100566
        struct[0].Gy_ini[83,30] = -0.793238195499528
        struct[0].Gy_ini[83,31] = -9.00835072044485
        struct[0].Gy_ini[83,72] = 10.2713736253513
        struct[0].Gy_ini[83,73] = -39.6392229058202
        struct[0].Gy_ini[83,74] = -54.0657727682604
        struct[0].Gy_ini[83,75] = 105.571769313180
        struct[0].Gy_ini[83,76] = 10.2713736253513
        struct[0].Gy_ini[83,77] = -39.6392229058201
        struct[0].Gy_ini[83,78] = 23.2849644809540
        struct[0].Gy_ini[83,79] = -24.9575997948692
        struct[0].Gy_ini[83,80] = -18.6937810814837
        struct[0].Gy_ini[83,81] = 88.4888280826460
        struct[0].Gy_ini[83,82] = 113.786109551688
        struct[0].Gy_ini[83,83] = -242.095286143588
        struct[0].Gy_ini[83,84] = -18.6937810814837
        struct[0].Gy_ini[83,85] = 88.4888280826460
        struct[0].Gy_ini[83,86] = -45.7766907664085
        struct[0].Gy_ini[83,87] = 58.9235503101833
        struct[0].Gy_ini[83,88] = 10.2713736253513
        struct[0].Gy_ini[83,89] = -39.6392229058202
        struct[0].Gy_ini[83,90] = -54.0657727682604
        struct[0].Gy_ini[83,91] = 105.571769313180
        struct[0].Gy_ini[83,92] = 10.2713736253513
        struct[0].Gy_ini[83,93] = -39.6392229058201
        struct[0].Gy_ini[83,94] = 23.2849644809540
        struct[0].Gy_ini[83,95] = -24.9575997948692
        struct[0].Gy_ini[84,24] = -9.00835072044484
        struct[0].Gy_ini[84,25] = 0.793238195499527
        struct[0].Gy_ini[84,26] = -9.21038227100566
        struct[0].Gy_ini[84,27] = 1.84896616921897
        struct[0].Gy_ini[84,28] = 30.9517475172273
        struct[0].Gy_ini[84,29] = 5.65456401516768
        struct[0].Gy_ini[84,30] = -9.21038227100566
        struct[0].Gy_ini[84,31] = 1.84896616921897
        struct[0].Gy_ini[84,72] = -24.9575997948692
        struct[0].Gy_ini[84,73] = -23.2849644809540
        struct[0].Gy_ini[84,74] = -39.6392229058202
        struct[0].Gy_ini[84,75] = -10.2713736253513
        struct[0].Gy_ini[84,76] = 105.571769313180
        struct[0].Gy_ini[84,77] = 54.0657727682604
        struct[0].Gy_ini[84,78] = -39.6392229058201
        struct[0].Gy_ini[84,79] = -10.2713736253514
        struct[0].Gy_ini[84,80] = 58.9235503101833
        struct[0].Gy_ini[84,81] = 45.7766907664085
        struct[0].Gy_ini[84,82] = 88.4888280826461
        struct[0].Gy_ini[84,83] = 18.6937810814837
        struct[0].Gy_ini[84,84] = -242.095286143588
        struct[0].Gy_ini[84,85] = -113.786109551688
        struct[0].Gy_ini[84,86] = 88.4888280826459
        struct[0].Gy_ini[84,87] = 18.6937810814837
        struct[0].Gy_ini[84,88] = -24.9575997948692
        struct[0].Gy_ini[84,89] = -23.2849644809540
        struct[0].Gy_ini[84,90] = -39.6392229058202
        struct[0].Gy_ini[84,91] = -10.2713736253513
        struct[0].Gy_ini[84,92] = 105.571769313180
        struct[0].Gy_ini[84,93] = 54.0657727682604
        struct[0].Gy_ini[84,94] = -39.6392229058201
        struct[0].Gy_ini[84,95] = -10.2713736253514
        struct[0].Gy_ini[85,24] = -0.793238195499527
        struct[0].Gy_ini[85,25] = -9.00835072044484
        struct[0].Gy_ini[85,26] = -1.84896616921897
        struct[0].Gy_ini[85,27] = -9.21038227100566
        struct[0].Gy_ini[85,28] = -5.65456401516768
        struct[0].Gy_ini[85,29] = 30.9517475172273
        struct[0].Gy_ini[85,30] = -1.84896616921897
        struct[0].Gy_ini[85,31] = -9.21038227100566
        struct[0].Gy_ini[85,72] = 23.2849644809540
        struct[0].Gy_ini[85,73] = -24.9575997948692
        struct[0].Gy_ini[85,74] = 10.2713736253513
        struct[0].Gy_ini[85,75] = -39.6392229058202
        struct[0].Gy_ini[85,76] = -54.0657727682604
        struct[0].Gy_ini[85,77] = 105.571769313180
        struct[0].Gy_ini[85,78] = 10.2713736253514
        struct[0].Gy_ini[85,79] = -39.6392229058201
        struct[0].Gy_ini[85,80] = -45.7766907664085
        struct[0].Gy_ini[85,81] = 58.9235503101833
        struct[0].Gy_ini[85,82] = -18.6937810814837
        struct[0].Gy_ini[85,83] = 88.4888280826461
        struct[0].Gy_ini[85,84] = 113.786109551688
        struct[0].Gy_ini[85,85] = -242.095286143588
        struct[0].Gy_ini[85,86] = -18.6937810814837
        struct[0].Gy_ini[85,87] = 88.4888280826459
        struct[0].Gy_ini[85,88] = 23.2849644809540
        struct[0].Gy_ini[85,89] = -24.9575997948692
        struct[0].Gy_ini[85,90] = 10.2713736253513
        struct[0].Gy_ini[85,91] = -39.6392229058202
        struct[0].Gy_ini[85,92] = -54.0657727682604
        struct[0].Gy_ini[85,93] = 105.571769313180
        struct[0].Gy_ini[85,94] = 10.2713736253514
        struct[0].Gy_ini[85,95] = -39.6392229058201
        struct[0].Gy_ini[86,24] = -9.21038227100566
        struct[0].Gy_ini[86,25] = 1.84896616921897
        struct[0].Gy_ini[86,26] = -9.00835072044485
        struct[0].Gy_ini[86,27] = 0.793238195499527
        struct[0].Gy_ini[86,28] = -9.21038227100566
        struct[0].Gy_ini[86,29] = 1.84896616921897
        struct[0].Gy_ini[86,30] = 30.9517475172273
        struct[0].Gy_ini[86,31] = 5.65456401516768
        struct[0].Gy_ini[86,72] = -39.6392229058202
        struct[0].Gy_ini[86,73] = -10.2713736253513
        struct[0].Gy_ini[86,74] = -24.9575997948692
        struct[0].Gy_ini[86,75] = -23.2849644809540
        struct[0].Gy_ini[86,76] = -39.6392229058202
        struct[0].Gy_ini[86,77] = -10.2713736253513
        struct[0].Gy_ini[86,78] = 105.571769313180
        struct[0].Gy_ini[86,79] = 54.0657727682604
        struct[0].Gy_ini[86,80] = 88.4888280826460
        struct[0].Gy_ini[86,81] = 18.6937810814837
        struct[0].Gy_ini[86,82] = 58.9235503101832
        struct[0].Gy_ini[86,83] = 45.7766907664085
        struct[0].Gy_ini[86,84] = 88.4888280826460
        struct[0].Gy_ini[86,85] = 18.6937810814837
        struct[0].Gy_ini[86,86] = -242.120286143588
        struct[0].Gy_ini[86,87] = -113.786109551688
        struct[0].Gy_ini[86,88] = -39.6392229058202
        struct[0].Gy_ini[86,89] = -10.2713736253513
        struct[0].Gy_ini[86,90] = -24.9575997948692
        struct[0].Gy_ini[86,91] = -23.2849644809540
        struct[0].Gy_ini[86,92] = -39.6392229058202
        struct[0].Gy_ini[86,93] = -10.2713736253513
        struct[0].Gy_ini[86,94] = 105.571769313180
        struct[0].Gy_ini[86,95] = 54.0657727682604
        struct[0].Gy_ini[87,24] = -1.84896616921897
        struct[0].Gy_ini[87,25] = -9.21038227100566
        struct[0].Gy_ini[87,26] = -0.793238195499527
        struct[0].Gy_ini[87,27] = -9.00835072044485
        struct[0].Gy_ini[87,28] = -1.84896616921897
        struct[0].Gy_ini[87,29] = -9.21038227100566
        struct[0].Gy_ini[87,30] = -5.65456401516768
        struct[0].Gy_ini[87,31] = 30.9517475172273
        struct[0].Gy_ini[87,72] = 10.2713736253513
        struct[0].Gy_ini[87,73] = -39.6392229058202
        struct[0].Gy_ini[87,74] = 23.2849644809540
        struct[0].Gy_ini[87,75] = -24.9575997948692
        struct[0].Gy_ini[87,76] = 10.2713736253513
        struct[0].Gy_ini[87,77] = -39.6392229058202
        struct[0].Gy_ini[87,78] = -54.0657727682604
        struct[0].Gy_ini[87,79] = 105.571769313180
        struct[0].Gy_ini[87,80] = -18.6937810814837
        struct[0].Gy_ini[87,81] = 88.4888280826460
        struct[0].Gy_ini[87,82] = -45.7766907664085
        struct[0].Gy_ini[87,83] = 58.9235503101832
        struct[0].Gy_ini[87,84] = -18.6937810814837
        struct[0].Gy_ini[87,85] = 88.4888280826460
        struct[0].Gy_ini[87,86] = 113.786109551688
        struct[0].Gy_ini[87,87] = -242.120286143588
        struct[0].Gy_ini[87,88] = 10.2713736253513
        struct[0].Gy_ini[87,89] = -39.6392229058202
        struct[0].Gy_ini[87,90] = 23.2849644809540
        struct[0].Gy_ini[87,91] = -24.9575997948692
        struct[0].Gy_ini[87,92] = 10.2713736253513
        struct[0].Gy_ini[87,93] = -39.6392229058202
        struct[0].Gy_ini[87,94] = -54.0657727682604
        struct[0].Gy_ini[87,95] = 105.571769313180
        struct[0].Gy_ini[88,80] = 105.571769313180
        struct[0].Gy_ini[88,81] = 54.0657727682604
        struct[0].Gy_ini[88,82] = -39.6392229058202
        struct[0].Gy_ini[88,83] = -10.2713736253513
        struct[0].Gy_ini[88,84] = -24.9575997948692
        struct[0].Gy_ini[88,85] = -23.2849644809540
        struct[0].Gy_ini[88,86] = -39.6392229058202
        struct[0].Gy_ini[88,87] = -10.2713736253513
        struct[0].Gy_ini[88,88] = -211.143538626360
        struct[0].Gy_ini[88,89] = -108.131545536521
        struct[0].Gy_ini[88,90] = 79.2784458116403
        struct[0].Gy_ini[88,91] = 20.5427472507027
        struct[0].Gy_ini[88,92] = 49.9151995897384
        struct[0].Gy_ini[88,93] = 46.5699289619080
        struct[0].Gy_ini[88,94] = 79.2784458116403
        struct[0].Gy_ini[88,95] = 20.5427472507027
        struct[0].Gy_ini[88,96] = 105.571769313180
        struct[0].Gy_ini[88,97] = 54.0657727682604
        struct[0].Gy_ini[88,98] = -39.6392229058202
        struct[0].Gy_ini[88,99] = -10.2713736253513
        struct[0].Gy_ini[88,100] = -24.9575997948692
        struct[0].Gy_ini[88,101] = -23.2849644809540
        struct[0].Gy_ini[88,102] = -39.6392229058202
        struct[0].Gy_ini[88,103] = -10.2713736253513
        struct[0].Gy_ini[89,80] = -54.0657727682604
        struct[0].Gy_ini[89,81] = 105.571769313180
        struct[0].Gy_ini[89,82] = 10.2713736253513
        struct[0].Gy_ini[89,83] = -39.6392229058202
        struct[0].Gy_ini[89,84] = 23.2849644809540
        struct[0].Gy_ini[89,85] = -24.9575997948692
        struct[0].Gy_ini[89,86] = 10.2713736253513
        struct[0].Gy_ini[89,87] = -39.6392229058202
        struct[0].Gy_ini[89,88] = 108.131545536521
        struct[0].Gy_ini[89,89] = -211.143538626360
        struct[0].Gy_ini[89,90] = -20.5427472507027
        struct[0].Gy_ini[89,91] = 79.2784458116403
        struct[0].Gy_ini[89,92] = -46.5699289619080
        struct[0].Gy_ini[89,93] = 49.9151995897384
        struct[0].Gy_ini[89,94] = -20.5427472507027
        struct[0].Gy_ini[89,95] = 79.2784458116403
        struct[0].Gy_ini[89,96] = -54.0657727682604
        struct[0].Gy_ini[89,97] = 105.571769313180
        struct[0].Gy_ini[89,98] = 10.2713736253513
        struct[0].Gy_ini[89,99] = -39.6392229058202
        struct[0].Gy_ini[89,100] = 23.2849644809540
        struct[0].Gy_ini[89,101] = -24.9575997948692
        struct[0].Gy_ini[89,102] = 10.2713736253513
        struct[0].Gy_ini[89,103] = -39.6392229058202
        struct[0].Gy_ini[90,80] = -39.6392229058202
        struct[0].Gy_ini[90,81] = -10.2713736253513
        struct[0].Gy_ini[90,82] = 105.571769313180
        struct[0].Gy_ini[90,83] = 54.0657727682604
        struct[0].Gy_ini[90,84] = -39.6392229058201
        struct[0].Gy_ini[90,85] = -10.2713736253513
        struct[0].Gy_ini[90,86] = -24.9575997948692
        struct[0].Gy_ini[90,87] = -23.2849644809540
        struct[0].Gy_ini[90,88] = 79.2784458116403
        struct[0].Gy_ini[90,89] = 20.5427472507027
        struct[0].Gy_ini[90,90] = -211.143538626360
        struct[0].Gy_ini[90,91] = -108.131545536521
        struct[0].Gy_ini[90,92] = 79.2784458116403
        struct[0].Gy_ini[90,93] = 20.5427472507027
        struct[0].Gy_ini[90,94] = 49.9151995897384
        struct[0].Gy_ini[90,95] = 46.5699289619080
        struct[0].Gy_ini[90,96] = -39.6392229058202
        struct[0].Gy_ini[90,97] = -10.2713736253513
        struct[0].Gy_ini[90,98] = 105.571769313180
        struct[0].Gy_ini[90,99] = 54.0657727682604
        struct[0].Gy_ini[90,100] = -39.6392229058201
        struct[0].Gy_ini[90,101] = -10.2713736253513
        struct[0].Gy_ini[90,102] = -24.9575997948692
        struct[0].Gy_ini[90,103] = -23.2849644809540
        struct[0].Gy_ini[91,80] = 10.2713736253513
        struct[0].Gy_ini[91,81] = -39.6392229058202
        struct[0].Gy_ini[91,82] = -54.0657727682604
        struct[0].Gy_ini[91,83] = 105.571769313180
        struct[0].Gy_ini[91,84] = 10.2713736253513
        struct[0].Gy_ini[91,85] = -39.6392229058201
        struct[0].Gy_ini[91,86] = 23.2849644809540
        struct[0].Gy_ini[91,87] = -24.9575997948692
        struct[0].Gy_ini[91,88] = -20.5427472507027
        struct[0].Gy_ini[91,89] = 79.2784458116403
        struct[0].Gy_ini[91,90] = 108.131545536521
        struct[0].Gy_ini[91,91] = -211.143538626360
        struct[0].Gy_ini[91,92] = -20.5427472507027
        struct[0].Gy_ini[91,93] = 79.2784458116403
        struct[0].Gy_ini[91,94] = -46.5699289619080
        struct[0].Gy_ini[91,95] = 49.9151995897384
        struct[0].Gy_ini[91,96] = 10.2713736253513
        struct[0].Gy_ini[91,97] = -39.6392229058202
        struct[0].Gy_ini[91,98] = -54.0657727682604
        struct[0].Gy_ini[91,99] = 105.571769313180
        struct[0].Gy_ini[91,100] = 10.2713736253513
        struct[0].Gy_ini[91,101] = -39.6392229058201
        struct[0].Gy_ini[91,102] = 23.2849644809540
        struct[0].Gy_ini[91,103] = -24.9575997948692
        struct[0].Gy_ini[92,80] = -24.9575997948692
        struct[0].Gy_ini[92,81] = -23.2849644809540
        struct[0].Gy_ini[92,82] = -39.6392229058202
        struct[0].Gy_ini[92,83] = -10.2713736253513
        struct[0].Gy_ini[92,84] = 105.571769313180
        struct[0].Gy_ini[92,85] = 54.0657727682604
        struct[0].Gy_ini[92,86] = -39.6392229058201
        struct[0].Gy_ini[92,87] = -10.2713736253514
        struct[0].Gy_ini[92,88] = 49.9151995897384
        struct[0].Gy_ini[92,89] = 46.5699289619080
        struct[0].Gy_ini[92,90] = 79.2784458116404
        struct[0].Gy_ini[92,91] = 20.5427472507027
        struct[0].Gy_ini[92,92] = -211.143538626360
        struct[0].Gy_ini[92,93] = -108.131545536521
        struct[0].Gy_ini[92,94] = 79.2784458116403
        struct[0].Gy_ini[92,95] = 20.5427472507027
        struct[0].Gy_ini[92,96] = -24.9575997948692
        struct[0].Gy_ini[92,97] = -23.2849644809540
        struct[0].Gy_ini[92,98] = -39.6392229058202
        struct[0].Gy_ini[92,99] = -10.2713736253513
        struct[0].Gy_ini[92,100] = 105.571769313180
        struct[0].Gy_ini[92,101] = 54.0657727682604
        struct[0].Gy_ini[92,102] = -39.6392229058201
        struct[0].Gy_ini[92,103] = -10.2713736253514
        struct[0].Gy_ini[93,80] = 23.2849644809540
        struct[0].Gy_ini[93,81] = -24.9575997948692
        struct[0].Gy_ini[93,82] = 10.2713736253513
        struct[0].Gy_ini[93,83] = -39.6392229058202
        struct[0].Gy_ini[93,84] = -54.0657727682604
        struct[0].Gy_ini[93,85] = 105.571769313180
        struct[0].Gy_ini[93,86] = 10.2713736253514
        struct[0].Gy_ini[93,87] = -39.6392229058201
        struct[0].Gy_ini[93,88] = -46.5699289619080
        struct[0].Gy_ini[93,89] = 49.9151995897384
        struct[0].Gy_ini[93,90] = -20.5427472507027
        struct[0].Gy_ini[93,91] = 79.2784458116404
        struct[0].Gy_ini[93,92] = 108.131545536521
        struct[0].Gy_ini[93,93] = -211.143538626360
        struct[0].Gy_ini[93,94] = -20.5427472507027
        struct[0].Gy_ini[93,95] = 79.2784458116403
        struct[0].Gy_ini[93,96] = 23.2849644809540
        struct[0].Gy_ini[93,97] = -24.9575997948692
        struct[0].Gy_ini[93,98] = 10.2713736253513
        struct[0].Gy_ini[93,99] = -39.6392229058202
        struct[0].Gy_ini[93,100] = -54.0657727682604
        struct[0].Gy_ini[93,101] = 105.571769313180
        struct[0].Gy_ini[93,102] = 10.2713736253514
        struct[0].Gy_ini[93,103] = -39.6392229058201
        struct[0].Gy_ini[94,80] = -39.6392229058202
        struct[0].Gy_ini[94,81] = -10.2713736253513
        struct[0].Gy_ini[94,82] = -24.9575997948692
        struct[0].Gy_ini[94,83] = -23.2849644809540
        struct[0].Gy_ini[94,84] = -39.6392229058202
        struct[0].Gy_ini[94,85] = -10.2713736253513
        struct[0].Gy_ini[94,86] = 105.571769313180
        struct[0].Gy_ini[94,87] = 54.0657727682604
        struct[0].Gy_ini[94,88] = 79.2784458116403
        struct[0].Gy_ini[94,89] = 20.5427472507027
        struct[0].Gy_ini[94,90] = 49.9151995897384
        struct[0].Gy_ini[94,91] = 46.5699289619080
        struct[0].Gy_ini[94,92] = 79.2784458116403
        struct[0].Gy_ini[94,93] = 20.5427472507027
        struct[0].Gy_ini[94,94] = -211.143538626360
        struct[0].Gy_ini[94,95] = -108.131545536521
        struct[0].Gy_ini[94,96] = -39.6392229058202
        struct[0].Gy_ini[94,97] = -10.2713736253513
        struct[0].Gy_ini[94,98] = -24.9575997948692
        struct[0].Gy_ini[94,99] = -23.2849644809540
        struct[0].Gy_ini[94,100] = -39.6392229058202
        struct[0].Gy_ini[94,101] = -10.2713736253513
        struct[0].Gy_ini[94,102] = 105.571769313180
        struct[0].Gy_ini[94,103] = 54.0657727682604
        struct[0].Gy_ini[95,80] = 10.2713736253513
        struct[0].Gy_ini[95,81] = -39.6392229058202
        struct[0].Gy_ini[95,82] = 23.2849644809540
        struct[0].Gy_ini[95,83] = -24.9575997948692
        struct[0].Gy_ini[95,84] = 10.2713736253513
        struct[0].Gy_ini[95,85] = -39.6392229058202
        struct[0].Gy_ini[95,86] = -54.0657727682604
        struct[0].Gy_ini[95,87] = 105.571769313180
        struct[0].Gy_ini[95,88] = -20.5427472507027
        struct[0].Gy_ini[95,89] = 79.2784458116403
        struct[0].Gy_ini[95,90] = -46.5699289619080
        struct[0].Gy_ini[95,91] = 49.9151995897384
        struct[0].Gy_ini[95,92] = -20.5427472507027
        struct[0].Gy_ini[95,93] = 79.2784458116403
        struct[0].Gy_ini[95,94] = 108.131545536521
        struct[0].Gy_ini[95,95] = -211.143538626360
        struct[0].Gy_ini[95,96] = 10.2713736253513
        struct[0].Gy_ini[95,97] = -39.6392229058202
        struct[0].Gy_ini[95,98] = 23.2849644809540
        struct[0].Gy_ini[95,99] = -24.9575997948692
        struct[0].Gy_ini[95,100] = 10.2713736253513
        struct[0].Gy_ini[95,101] = -39.6392229058202
        struct[0].Gy_ini[95,102] = -54.0657727682604
        struct[0].Gy_ini[95,103] = 105.571769313180
        struct[0].Gy_ini[96,88] = 105.571769313180
        struct[0].Gy_ini[96,89] = 54.0657727682604
        struct[0].Gy_ini[96,90] = -39.6392229058202
        struct[0].Gy_ini[96,91] = -10.2713736253513
        struct[0].Gy_ini[96,92] = -24.9575997948692
        struct[0].Gy_ini[96,93] = -23.2849644809540
        struct[0].Gy_ini[96,94] = -39.6392229058202
        struct[0].Gy_ini[96,95] = -10.2713736253513
        struct[0].Gy_ini[96,96] = -211.143538626360
        struct[0].Gy_ini[96,97] = -108.131545536521
        struct[0].Gy_ini[96,98] = 79.2784458116403
        struct[0].Gy_ini[96,99] = 20.5427472507027
        struct[0].Gy_ini[96,100] = 49.9151995897384
        struct[0].Gy_ini[96,101] = 46.5699289619080
        struct[0].Gy_ini[96,102] = 79.2784458116403
        struct[0].Gy_ini[96,103] = 20.5427472507027
        struct[0].Gy_ini[96,104] = 105.571769313180
        struct[0].Gy_ini[96,105] = 54.0657727682604
        struct[0].Gy_ini[96,106] = -39.6392229058202
        struct[0].Gy_ini[96,107] = -10.2713736253513
        struct[0].Gy_ini[96,108] = -24.9575997948692
        struct[0].Gy_ini[96,109] = -23.2849644809540
        struct[0].Gy_ini[96,110] = -39.6392229058202
        struct[0].Gy_ini[96,111] = -10.2713736253513
        struct[0].Gy_ini[97,88] = -54.0657727682604
        struct[0].Gy_ini[97,89] = 105.571769313180
        struct[0].Gy_ini[97,90] = 10.2713736253513
        struct[0].Gy_ini[97,91] = -39.6392229058202
        struct[0].Gy_ini[97,92] = 23.2849644809540
        struct[0].Gy_ini[97,93] = -24.9575997948692
        struct[0].Gy_ini[97,94] = 10.2713736253513
        struct[0].Gy_ini[97,95] = -39.6392229058202
        struct[0].Gy_ini[97,96] = 108.131545536521
        struct[0].Gy_ini[97,97] = -211.143538626360
        struct[0].Gy_ini[97,98] = -20.5427472507027
        struct[0].Gy_ini[97,99] = 79.2784458116403
        struct[0].Gy_ini[97,100] = -46.5699289619080
        struct[0].Gy_ini[97,101] = 49.9151995897384
        struct[0].Gy_ini[97,102] = -20.5427472507027
        struct[0].Gy_ini[97,103] = 79.2784458116403
        struct[0].Gy_ini[97,104] = -54.0657727682604
        struct[0].Gy_ini[97,105] = 105.571769313180
        struct[0].Gy_ini[97,106] = 10.2713736253513
        struct[0].Gy_ini[97,107] = -39.6392229058202
        struct[0].Gy_ini[97,108] = 23.2849644809540
        struct[0].Gy_ini[97,109] = -24.9575997948692
        struct[0].Gy_ini[97,110] = 10.2713736253513
        struct[0].Gy_ini[97,111] = -39.6392229058202
        struct[0].Gy_ini[98,88] = -39.6392229058202
        struct[0].Gy_ini[98,89] = -10.2713736253513
        struct[0].Gy_ini[98,90] = 105.571769313180
        struct[0].Gy_ini[98,91] = 54.0657727682604
        struct[0].Gy_ini[98,92] = -39.6392229058201
        struct[0].Gy_ini[98,93] = -10.2713736253513
        struct[0].Gy_ini[98,94] = -24.9575997948692
        struct[0].Gy_ini[98,95] = -23.2849644809540
        struct[0].Gy_ini[98,96] = 79.2784458116403
        struct[0].Gy_ini[98,97] = 20.5427472507027
        struct[0].Gy_ini[98,98] = -211.143538626360
        struct[0].Gy_ini[98,99] = -108.131545536521
        struct[0].Gy_ini[98,100] = 79.2784458116403
        struct[0].Gy_ini[98,101] = 20.5427472507027
        struct[0].Gy_ini[98,102] = 49.9151995897384
        struct[0].Gy_ini[98,103] = 46.5699289619080
        struct[0].Gy_ini[98,104] = -39.6392229058202
        struct[0].Gy_ini[98,105] = -10.2713736253513
        struct[0].Gy_ini[98,106] = 105.571769313180
        struct[0].Gy_ini[98,107] = 54.0657727682604
        struct[0].Gy_ini[98,108] = -39.6392229058201
        struct[0].Gy_ini[98,109] = -10.2713736253513
        struct[0].Gy_ini[98,110] = -24.9575997948692
        struct[0].Gy_ini[98,111] = -23.2849644809540
        struct[0].Gy_ini[99,88] = 10.2713736253513
        struct[0].Gy_ini[99,89] = -39.6392229058202
        struct[0].Gy_ini[99,90] = -54.0657727682604
        struct[0].Gy_ini[99,91] = 105.571769313180
        struct[0].Gy_ini[99,92] = 10.2713736253513
        struct[0].Gy_ini[99,93] = -39.6392229058201
        struct[0].Gy_ini[99,94] = 23.2849644809540
        struct[0].Gy_ini[99,95] = -24.9575997948692
        struct[0].Gy_ini[99,96] = -20.5427472507027
        struct[0].Gy_ini[99,97] = 79.2784458116403
        struct[0].Gy_ini[99,98] = 108.131545536521
        struct[0].Gy_ini[99,99] = -211.143538626360
        struct[0].Gy_ini[99,100] = -20.5427472507027
        struct[0].Gy_ini[99,101] = 79.2784458116403
        struct[0].Gy_ini[99,102] = -46.5699289619080
        struct[0].Gy_ini[99,103] = 49.9151995897384
        struct[0].Gy_ini[99,104] = 10.2713736253513
        struct[0].Gy_ini[99,105] = -39.6392229058202
        struct[0].Gy_ini[99,106] = -54.0657727682604
        struct[0].Gy_ini[99,107] = 105.571769313180
        struct[0].Gy_ini[99,108] = 10.2713736253513
        struct[0].Gy_ini[99,109] = -39.6392229058201
        struct[0].Gy_ini[99,110] = 23.2849644809540
        struct[0].Gy_ini[99,111] = -24.9575997948692
        struct[0].Gy_ini[100,88] = -24.9575997948692
        struct[0].Gy_ini[100,89] = -23.2849644809540
        struct[0].Gy_ini[100,90] = -39.6392229058202
        struct[0].Gy_ini[100,91] = -10.2713736253513
        struct[0].Gy_ini[100,92] = 105.571769313180
        struct[0].Gy_ini[100,93] = 54.0657727682604
        struct[0].Gy_ini[100,94] = -39.6392229058201
        struct[0].Gy_ini[100,95] = -10.2713736253514
        struct[0].Gy_ini[100,96] = 49.9151995897384
        struct[0].Gy_ini[100,97] = 46.5699289619080
        struct[0].Gy_ini[100,98] = 79.2784458116404
        struct[0].Gy_ini[100,99] = 20.5427472507027
        struct[0].Gy_ini[100,100] = -211.143538626360
        struct[0].Gy_ini[100,101] = -108.131545536521
        struct[0].Gy_ini[100,102] = 79.2784458116403
        struct[0].Gy_ini[100,103] = 20.5427472507027
        struct[0].Gy_ini[100,104] = -24.9575997948692
        struct[0].Gy_ini[100,105] = -23.2849644809540
        struct[0].Gy_ini[100,106] = -39.6392229058202
        struct[0].Gy_ini[100,107] = -10.2713736253513
        struct[0].Gy_ini[100,108] = 105.571769313180
        struct[0].Gy_ini[100,109] = 54.0657727682604
        struct[0].Gy_ini[100,110] = -39.6392229058201
        struct[0].Gy_ini[100,111] = -10.2713736253514
        struct[0].Gy_ini[101,88] = 23.2849644809540
        struct[0].Gy_ini[101,89] = -24.9575997948692
        struct[0].Gy_ini[101,90] = 10.2713736253513
        struct[0].Gy_ini[101,91] = -39.6392229058202
        struct[0].Gy_ini[101,92] = -54.0657727682604
        struct[0].Gy_ini[101,93] = 105.571769313180
        struct[0].Gy_ini[101,94] = 10.2713736253514
        struct[0].Gy_ini[101,95] = -39.6392229058201
        struct[0].Gy_ini[101,96] = -46.5699289619080
        struct[0].Gy_ini[101,97] = 49.9151995897384
        struct[0].Gy_ini[101,98] = -20.5427472507027
        struct[0].Gy_ini[101,99] = 79.2784458116404
        struct[0].Gy_ini[101,100] = 108.131545536521
        struct[0].Gy_ini[101,101] = -211.143538626360
        struct[0].Gy_ini[101,102] = -20.5427472507027
        struct[0].Gy_ini[101,103] = 79.2784458116403
        struct[0].Gy_ini[101,104] = 23.2849644809540
        struct[0].Gy_ini[101,105] = -24.9575997948692
        struct[0].Gy_ini[101,106] = 10.2713736253513
        struct[0].Gy_ini[101,107] = -39.6392229058202
        struct[0].Gy_ini[101,108] = -54.0657727682604
        struct[0].Gy_ini[101,109] = 105.571769313180
        struct[0].Gy_ini[101,110] = 10.2713736253514
        struct[0].Gy_ini[101,111] = -39.6392229058201
        struct[0].Gy_ini[102,88] = -39.6392229058202
        struct[0].Gy_ini[102,89] = -10.2713736253513
        struct[0].Gy_ini[102,90] = -24.9575997948692
        struct[0].Gy_ini[102,91] = -23.2849644809540
        struct[0].Gy_ini[102,92] = -39.6392229058202
        struct[0].Gy_ini[102,93] = -10.2713736253513
        struct[0].Gy_ini[102,94] = 105.571769313180
        struct[0].Gy_ini[102,95] = 54.0657727682604
        struct[0].Gy_ini[102,96] = 79.2784458116403
        struct[0].Gy_ini[102,97] = 20.5427472507027
        struct[0].Gy_ini[102,98] = 49.9151995897384
        struct[0].Gy_ini[102,99] = 46.5699289619080
        struct[0].Gy_ini[102,100] = 79.2784458116403
        struct[0].Gy_ini[102,101] = 20.5427472507027
        struct[0].Gy_ini[102,102] = -211.168538626360
        struct[0].Gy_ini[102,103] = -108.131545536521
        struct[0].Gy_ini[102,104] = -39.6392229058202
        struct[0].Gy_ini[102,105] = -10.2713736253513
        struct[0].Gy_ini[102,106] = -24.9575997948692
        struct[0].Gy_ini[102,107] = -23.2849644809540
        struct[0].Gy_ini[102,108] = -39.6392229058202
        struct[0].Gy_ini[102,109] = -10.2713736253513
        struct[0].Gy_ini[102,110] = 105.571769313180
        struct[0].Gy_ini[102,111] = 54.0657727682604
        struct[0].Gy_ini[103,88] = 10.2713736253513
        struct[0].Gy_ini[103,89] = -39.6392229058202
        struct[0].Gy_ini[103,90] = 23.2849644809540
        struct[0].Gy_ini[103,91] = -24.9575997948692
        struct[0].Gy_ini[103,92] = 10.2713736253513
        struct[0].Gy_ini[103,93] = -39.6392229058202
        struct[0].Gy_ini[103,94] = -54.0657727682604
        struct[0].Gy_ini[103,95] = 105.571769313180
        struct[0].Gy_ini[103,96] = -20.5427472507027
        struct[0].Gy_ini[103,97] = 79.2784458116403
        struct[0].Gy_ini[103,98] = -46.5699289619080
        struct[0].Gy_ini[103,99] = 49.9151995897384
        struct[0].Gy_ini[103,100] = -20.5427472507027
        struct[0].Gy_ini[103,101] = 79.2784458116403
        struct[0].Gy_ini[103,102] = 108.131545536521
        struct[0].Gy_ini[103,103] = -211.168538626360
        struct[0].Gy_ini[103,104] = 10.2713736253513
        struct[0].Gy_ini[103,105] = -39.6392229058202
        struct[0].Gy_ini[103,106] = 23.2849644809540
        struct[0].Gy_ini[103,107] = -24.9575997948692
        struct[0].Gy_ini[103,108] = 10.2713736253513
        struct[0].Gy_ini[103,109] = -39.6392229058202
        struct[0].Gy_ini[103,110] = -54.0657727682604
        struct[0].Gy_ini[103,111] = 105.571769313180
        struct[0].Gy_ini[104,32] = 30.9517475172273
        struct[0].Gy_ini[104,33] = 5.65456401516768
        struct[0].Gy_ini[104,34] = -9.21038227100566
        struct[0].Gy_ini[104,35] = 1.84896616921897
        struct[0].Gy_ini[104,36] = -9.00835072044485
        struct[0].Gy_ini[104,37] = 0.793238195499529
        struct[0].Gy_ini[104,38] = -9.21038227100566
        struct[0].Gy_ini[104,39] = 1.84896616921897
        struct[0].Gy_ini[104,96] = 105.571769313180
        struct[0].Gy_ini[104,97] = 54.0657727682604
        struct[0].Gy_ini[104,98] = -39.6392229058202
        struct[0].Gy_ini[104,99] = -10.2713736253513
        struct[0].Gy_ini[104,100] = -24.9575997948692
        struct[0].Gy_ini[104,101] = -23.2849644809540
        struct[0].Gy_ini[104,102] = -39.6392229058202
        struct[0].Gy_ini[104,103] = -10.2713736253513
        struct[0].Gy_ini[104,104] = -242.095286143588
        struct[0].Gy_ini[104,105] = -113.786109551688
        struct[0].Gy_ini[104,106] = 88.4888280826460
        struct[0].Gy_ini[104,107] = 18.6937810814837
        struct[0].Gy_ini[104,108] = 58.9235503101833
        struct[0].Gy_ini[104,109] = 45.7766907664085
        struct[0].Gy_ini[104,110] = 88.4888280826460
        struct[0].Gy_ini[104,111] = 18.6937810814837
        struct[0].Gy_ini[104,112] = 105.571769313180
        struct[0].Gy_ini[104,113] = 54.0657727682604
        struct[0].Gy_ini[104,114] = -39.6392229058202
        struct[0].Gy_ini[104,115] = -10.2713736253513
        struct[0].Gy_ini[104,116] = -24.9575997948692
        struct[0].Gy_ini[104,117] = -23.2849644809540
        struct[0].Gy_ini[104,118] = -39.6392229058202
        struct[0].Gy_ini[104,119] = -10.2713736253513
        struct[0].Gy_ini[105,32] = -5.65456401516768
        struct[0].Gy_ini[105,33] = 30.9517475172273
        struct[0].Gy_ini[105,34] = -1.84896616921897
        struct[0].Gy_ini[105,35] = -9.21038227100566
        struct[0].Gy_ini[105,36] = -0.793238195499529
        struct[0].Gy_ini[105,37] = -9.00835072044485
        struct[0].Gy_ini[105,38] = -1.84896616921897
        struct[0].Gy_ini[105,39] = -9.21038227100566
        struct[0].Gy_ini[105,96] = -54.0657727682604
        struct[0].Gy_ini[105,97] = 105.571769313180
        struct[0].Gy_ini[105,98] = 10.2713736253513
        struct[0].Gy_ini[105,99] = -39.6392229058202
        struct[0].Gy_ini[105,100] = 23.2849644809540
        struct[0].Gy_ini[105,101] = -24.9575997948692
        struct[0].Gy_ini[105,102] = 10.2713736253513
        struct[0].Gy_ini[105,103] = -39.6392229058202
        struct[0].Gy_ini[105,104] = 113.786109551688
        struct[0].Gy_ini[105,105] = -242.095286143588
        struct[0].Gy_ini[105,106] = -18.6937810814837
        struct[0].Gy_ini[105,107] = 88.4888280826460
        struct[0].Gy_ini[105,108] = -45.7766907664085
        struct[0].Gy_ini[105,109] = 58.9235503101833
        struct[0].Gy_ini[105,110] = -18.6937810814837
        struct[0].Gy_ini[105,111] = 88.4888280826460
        struct[0].Gy_ini[105,112] = -54.0657727682604
        struct[0].Gy_ini[105,113] = 105.571769313180
        struct[0].Gy_ini[105,114] = 10.2713736253513
        struct[0].Gy_ini[105,115] = -39.6392229058202
        struct[0].Gy_ini[105,116] = 23.2849644809540
        struct[0].Gy_ini[105,117] = -24.9575997948692
        struct[0].Gy_ini[105,118] = 10.2713736253513
        struct[0].Gy_ini[105,119] = -39.6392229058202
        struct[0].Gy_ini[106,32] = -9.21038227100566
        struct[0].Gy_ini[106,33] = 1.84896616921897
        struct[0].Gy_ini[106,34] = 30.9517475172273
        struct[0].Gy_ini[106,35] = 5.65456401516768
        struct[0].Gy_ini[106,36] = -9.21038227100566
        struct[0].Gy_ini[106,37] = 1.84896616921897
        struct[0].Gy_ini[106,38] = -9.00835072044485
        struct[0].Gy_ini[106,39] = 0.793238195499528
        struct[0].Gy_ini[106,96] = -39.6392229058202
        struct[0].Gy_ini[106,97] = -10.2713736253513
        struct[0].Gy_ini[106,98] = 105.571769313180
        struct[0].Gy_ini[106,99] = 54.0657727682604
        struct[0].Gy_ini[106,100] = -39.6392229058201
        struct[0].Gy_ini[106,101] = -10.2713736253513
        struct[0].Gy_ini[106,102] = -24.9575997948692
        struct[0].Gy_ini[106,103] = -23.2849644809540
        struct[0].Gy_ini[106,104] = 88.4888280826460
        struct[0].Gy_ini[106,105] = 18.6937810814837
        struct[0].Gy_ini[106,106] = -242.095286143588
        struct[0].Gy_ini[106,107] = -113.786109551688
        struct[0].Gy_ini[106,108] = 88.4888280826460
        struct[0].Gy_ini[106,109] = 18.6937810814837
        struct[0].Gy_ini[106,110] = 58.9235503101833
        struct[0].Gy_ini[106,111] = 45.7766907664085
        struct[0].Gy_ini[106,112] = -39.6392229058202
        struct[0].Gy_ini[106,113] = -10.2713736253513
        struct[0].Gy_ini[106,114] = 105.571769313180
        struct[0].Gy_ini[106,115] = 54.0657727682604
        struct[0].Gy_ini[106,116] = -39.6392229058201
        struct[0].Gy_ini[106,117] = -10.2713736253513
        struct[0].Gy_ini[106,118] = -24.9575997948692
        struct[0].Gy_ini[106,119] = -23.2849644809540
        struct[0].Gy_ini[107,32] = -1.84896616921897
        struct[0].Gy_ini[107,33] = -9.21038227100566
        struct[0].Gy_ini[107,34] = -5.65456401516768
        struct[0].Gy_ini[107,35] = 30.9517475172273
        struct[0].Gy_ini[107,36] = -1.84896616921897
        struct[0].Gy_ini[107,37] = -9.21038227100566
        struct[0].Gy_ini[107,38] = -0.793238195499528
        struct[0].Gy_ini[107,39] = -9.00835072044485
        struct[0].Gy_ini[107,96] = 10.2713736253513
        struct[0].Gy_ini[107,97] = -39.6392229058202
        struct[0].Gy_ini[107,98] = -54.0657727682604
        struct[0].Gy_ini[107,99] = 105.571769313180
        struct[0].Gy_ini[107,100] = 10.2713736253513
        struct[0].Gy_ini[107,101] = -39.6392229058201
        struct[0].Gy_ini[107,102] = 23.2849644809540
        struct[0].Gy_ini[107,103] = -24.9575997948692
        struct[0].Gy_ini[107,104] = -18.6937810814837
        struct[0].Gy_ini[107,105] = 88.4888280826460
        struct[0].Gy_ini[107,106] = 113.786109551688
        struct[0].Gy_ini[107,107] = -242.095286143588
        struct[0].Gy_ini[107,108] = -18.6937810814837
        struct[0].Gy_ini[107,109] = 88.4888280826460
        struct[0].Gy_ini[107,110] = -45.7766907664085
        struct[0].Gy_ini[107,111] = 58.9235503101833
        struct[0].Gy_ini[107,112] = 10.2713736253513
        struct[0].Gy_ini[107,113] = -39.6392229058202
        struct[0].Gy_ini[107,114] = -54.0657727682604
        struct[0].Gy_ini[107,115] = 105.571769313180
        struct[0].Gy_ini[107,116] = 10.2713736253513
        struct[0].Gy_ini[107,117] = -39.6392229058201
        struct[0].Gy_ini[107,118] = 23.2849644809540
        struct[0].Gy_ini[107,119] = -24.9575997948692
        struct[0].Gy_ini[108,32] = -9.00835072044484
        struct[0].Gy_ini[108,33] = 0.793238195499527
        struct[0].Gy_ini[108,34] = -9.21038227100566
        struct[0].Gy_ini[108,35] = 1.84896616921897
        struct[0].Gy_ini[108,36] = 30.9517475172273
        struct[0].Gy_ini[108,37] = 5.65456401516768
        struct[0].Gy_ini[108,38] = -9.21038227100566
        struct[0].Gy_ini[108,39] = 1.84896616921897
        struct[0].Gy_ini[108,96] = -24.9575997948692
        struct[0].Gy_ini[108,97] = -23.2849644809540
        struct[0].Gy_ini[108,98] = -39.6392229058202
        struct[0].Gy_ini[108,99] = -10.2713736253513
        struct[0].Gy_ini[108,100] = 105.571769313180
        struct[0].Gy_ini[108,101] = 54.0657727682604
        struct[0].Gy_ini[108,102] = -39.6392229058201
        struct[0].Gy_ini[108,103] = -10.2713736253514
        struct[0].Gy_ini[108,104] = 58.9235503101833
        struct[0].Gy_ini[108,105] = 45.7766907664085
        struct[0].Gy_ini[108,106] = 88.4888280826461
        struct[0].Gy_ini[108,107] = 18.6937810814837
        struct[0].Gy_ini[108,108] = -242.095286143588
        struct[0].Gy_ini[108,109] = -113.786109551688
        struct[0].Gy_ini[108,110] = 88.4888280826459
        struct[0].Gy_ini[108,111] = 18.6937810814837
        struct[0].Gy_ini[108,112] = -24.9575997948692
        struct[0].Gy_ini[108,113] = -23.2849644809540
        struct[0].Gy_ini[108,114] = -39.6392229058202
        struct[0].Gy_ini[108,115] = -10.2713736253513
        struct[0].Gy_ini[108,116] = 105.571769313180
        struct[0].Gy_ini[108,117] = 54.0657727682604
        struct[0].Gy_ini[108,118] = -39.6392229058201
        struct[0].Gy_ini[108,119] = -10.2713736253514
        struct[0].Gy_ini[109,32] = -0.793238195499527
        struct[0].Gy_ini[109,33] = -9.00835072044484
        struct[0].Gy_ini[109,34] = -1.84896616921897
        struct[0].Gy_ini[109,35] = -9.21038227100566
        struct[0].Gy_ini[109,36] = -5.65456401516768
        struct[0].Gy_ini[109,37] = 30.9517475172273
        struct[0].Gy_ini[109,38] = -1.84896616921897
        struct[0].Gy_ini[109,39] = -9.21038227100566
        struct[0].Gy_ini[109,96] = 23.2849644809540
        struct[0].Gy_ini[109,97] = -24.9575997948692
        struct[0].Gy_ini[109,98] = 10.2713736253513
        struct[0].Gy_ini[109,99] = -39.6392229058202
        struct[0].Gy_ini[109,100] = -54.0657727682604
        struct[0].Gy_ini[109,101] = 105.571769313180
        struct[0].Gy_ini[109,102] = 10.2713736253514
        struct[0].Gy_ini[109,103] = -39.6392229058201
        struct[0].Gy_ini[109,104] = -45.7766907664085
        struct[0].Gy_ini[109,105] = 58.9235503101833
        struct[0].Gy_ini[109,106] = -18.6937810814837
        struct[0].Gy_ini[109,107] = 88.4888280826461
        struct[0].Gy_ini[109,108] = 113.786109551688
        struct[0].Gy_ini[109,109] = -242.095286143588
        struct[0].Gy_ini[109,110] = -18.6937810814837
        struct[0].Gy_ini[109,111] = 88.4888280826459
        struct[0].Gy_ini[109,112] = 23.2849644809540
        struct[0].Gy_ini[109,113] = -24.9575997948692
        struct[0].Gy_ini[109,114] = 10.2713736253513
        struct[0].Gy_ini[109,115] = -39.6392229058202
        struct[0].Gy_ini[109,116] = -54.0657727682604
        struct[0].Gy_ini[109,117] = 105.571769313180
        struct[0].Gy_ini[109,118] = 10.2713736253514
        struct[0].Gy_ini[109,119] = -39.6392229058201
        struct[0].Gy_ini[110,32] = -9.21038227100566
        struct[0].Gy_ini[110,33] = 1.84896616921897
        struct[0].Gy_ini[110,34] = -9.00835072044485
        struct[0].Gy_ini[110,35] = 0.793238195499527
        struct[0].Gy_ini[110,36] = -9.21038227100566
        struct[0].Gy_ini[110,37] = 1.84896616921897
        struct[0].Gy_ini[110,38] = 30.9517475172273
        struct[0].Gy_ini[110,39] = 5.65456401516768
        struct[0].Gy_ini[110,96] = -39.6392229058202
        struct[0].Gy_ini[110,97] = -10.2713736253513
        struct[0].Gy_ini[110,98] = -24.9575997948692
        struct[0].Gy_ini[110,99] = -23.2849644809540
        struct[0].Gy_ini[110,100] = -39.6392229058202
        struct[0].Gy_ini[110,101] = -10.2713736253513
        struct[0].Gy_ini[110,102] = 105.571769313180
        struct[0].Gy_ini[110,103] = 54.0657727682604
        struct[0].Gy_ini[110,104] = 88.4888280826460
        struct[0].Gy_ini[110,105] = 18.6937810814837
        struct[0].Gy_ini[110,106] = 58.9235503101832
        struct[0].Gy_ini[110,107] = 45.7766907664085
        struct[0].Gy_ini[110,108] = 88.4888280826460
        struct[0].Gy_ini[110,109] = 18.6937810814837
        struct[0].Gy_ini[110,110] = -242.095286143588
        struct[0].Gy_ini[110,111] = -113.786109551688
        struct[0].Gy_ini[110,112] = -39.6392229058202
        struct[0].Gy_ini[110,113] = -10.2713736253513
        struct[0].Gy_ini[110,114] = -24.9575997948692
        struct[0].Gy_ini[110,115] = -23.2849644809540
        struct[0].Gy_ini[110,116] = -39.6392229058202
        struct[0].Gy_ini[110,117] = -10.2713736253513
        struct[0].Gy_ini[110,118] = 105.571769313180
        struct[0].Gy_ini[110,119] = 54.0657727682604
        struct[0].Gy_ini[111,32] = -1.84896616921897
        struct[0].Gy_ini[111,33] = -9.21038227100566
        struct[0].Gy_ini[111,34] = -0.793238195499527
        struct[0].Gy_ini[111,35] = -9.00835072044485
        struct[0].Gy_ini[111,36] = -1.84896616921897
        struct[0].Gy_ini[111,37] = -9.21038227100566
        struct[0].Gy_ini[111,38] = -5.65456401516768
        struct[0].Gy_ini[111,39] = 30.9517475172273
        struct[0].Gy_ini[111,96] = 10.2713736253513
        struct[0].Gy_ini[111,97] = -39.6392229058202
        struct[0].Gy_ini[111,98] = 23.2849644809540
        struct[0].Gy_ini[111,99] = -24.9575997948692
        struct[0].Gy_ini[111,100] = 10.2713736253513
        struct[0].Gy_ini[111,101] = -39.6392229058202
        struct[0].Gy_ini[111,102] = -54.0657727682604
        struct[0].Gy_ini[111,103] = 105.571769313180
        struct[0].Gy_ini[111,104] = -18.6937810814837
        struct[0].Gy_ini[111,105] = 88.4888280826460
        struct[0].Gy_ini[111,106] = -45.7766907664085
        struct[0].Gy_ini[111,107] = 58.9235503101832
        struct[0].Gy_ini[111,108] = -18.6937810814837
        struct[0].Gy_ini[111,109] = 88.4888280826460
        struct[0].Gy_ini[111,110] = 113.786109551688
        struct[0].Gy_ini[111,111] = -242.095286143588
        struct[0].Gy_ini[111,112] = 10.2713736253513
        struct[0].Gy_ini[111,113] = -39.6392229058202
        struct[0].Gy_ini[111,114] = 23.2849644809540
        struct[0].Gy_ini[111,115] = -24.9575997948692
        struct[0].Gy_ini[111,116] = 10.2713736253513
        struct[0].Gy_ini[111,117] = -39.6392229058202
        struct[0].Gy_ini[111,118] = -54.0657727682604
        struct[0].Gy_ini[111,119] = 105.571769313180
        struct[0].Gy_ini[112,40] = 30.9517475172273
        struct[0].Gy_ini[112,41] = 5.65456401516768
        struct[0].Gy_ini[112,42] = -9.21038227100566
        struct[0].Gy_ini[112,43] = 1.84896616921897
        struct[0].Gy_ini[112,44] = -9.00835072044485
        struct[0].Gy_ini[112,45] = 0.793238195499529
        struct[0].Gy_ini[112,46] = -9.21038227100566
        struct[0].Gy_ini[112,47] = 1.84896616921897
        struct[0].Gy_ini[112,104] = 105.571769313180
        struct[0].Gy_ini[112,105] = 54.0657727682604
        struct[0].Gy_ini[112,106] = -39.6392229058202
        struct[0].Gy_ini[112,107] = -10.2713736253513
        struct[0].Gy_ini[112,108] = -24.9575997948692
        struct[0].Gy_ini[112,109] = -23.2849644809540
        struct[0].Gy_ini[112,110] = -39.6392229058202
        struct[0].Gy_ini[112,111] = -10.2713736253513
        struct[0].Gy_ini[112,112] = -136.523516830407
        struct[0].Gy_ini[112,113] = -59.7203367834281
        struct[0].Gy_ini[112,114] = 48.8496051768258
        struct[0].Gy_ini[112,115] = 8.42240745613237
        struct[0].Gy_ini[112,116] = 33.9659505153141
        struct[0].Gy_ini[112,117] = 22.4917262854545
        struct[0].Gy_ini[112,118] = 48.8496051768258
        struct[0].Gy_ini[112,119] = 8.42240745613237
        struct[0].Gy_ini[113,40] = -5.65456401516768
        struct[0].Gy_ini[113,41] = 30.9517475172273
        struct[0].Gy_ini[113,42] = -1.84896616921897
        struct[0].Gy_ini[113,43] = -9.21038227100566
        struct[0].Gy_ini[113,44] = -0.793238195499529
        struct[0].Gy_ini[113,45] = -9.00835072044485
        struct[0].Gy_ini[113,46] = -1.84896616921897
        struct[0].Gy_ini[113,47] = -9.21038227100566
        struct[0].Gy_ini[113,104] = -54.0657727682604
        struct[0].Gy_ini[113,105] = 105.571769313180
        struct[0].Gy_ini[113,106] = 10.2713736253513
        struct[0].Gy_ini[113,107] = -39.6392229058202
        struct[0].Gy_ini[113,108] = 23.2849644809540
        struct[0].Gy_ini[113,109] = -24.9575997948692
        struct[0].Gy_ini[113,110] = 10.2713736253513
        struct[0].Gy_ini[113,111] = -39.6392229058202
        struct[0].Gy_ini[113,112] = 59.7203367834281
        struct[0].Gy_ini[113,113] = -136.523516830407
        struct[0].Gy_ini[113,114] = -8.42240745613237
        struct[0].Gy_ini[113,115] = 48.8496051768258
        struct[0].Gy_ini[113,116] = -22.4917262854545
        struct[0].Gy_ini[113,117] = 33.9659505153141
        struct[0].Gy_ini[113,118] = -8.42240745613237
        struct[0].Gy_ini[113,119] = 48.8496051768258
        struct[0].Gy_ini[114,40] = -9.21038227100566
        struct[0].Gy_ini[114,41] = 1.84896616921897
        struct[0].Gy_ini[114,42] = 30.9517475172273
        struct[0].Gy_ini[114,43] = 5.65456401516768
        struct[0].Gy_ini[114,44] = -9.21038227100566
        struct[0].Gy_ini[114,45] = 1.84896616921897
        struct[0].Gy_ini[114,46] = -9.00835072044485
        struct[0].Gy_ini[114,47] = 0.793238195499528
        struct[0].Gy_ini[114,104] = -39.6392229058202
        struct[0].Gy_ini[114,105] = -10.2713736253513
        struct[0].Gy_ini[114,106] = 105.571769313180
        struct[0].Gy_ini[114,107] = 54.0657727682604
        struct[0].Gy_ini[114,108] = -39.6392229058201
        struct[0].Gy_ini[114,109] = -10.2713736253513
        struct[0].Gy_ini[114,110] = -24.9575997948692
        struct[0].Gy_ini[114,111] = -23.2849644809540
        struct[0].Gy_ini[114,112] = 48.8496051768258
        struct[0].Gy_ini[114,113] = 8.42240745613237
        struct[0].Gy_ini[114,114] = -136.523516830407
        struct[0].Gy_ini[114,115] = -59.7203367834280
        struct[0].Gy_ini[114,116] = 48.8496051768258
        struct[0].Gy_ini[114,117] = 8.42240745613238
        struct[0].Gy_ini[114,118] = 33.9659505153141
        struct[0].Gy_ini[114,119] = 22.4917262854545
        struct[0].Gy_ini[115,40] = -1.84896616921897
        struct[0].Gy_ini[115,41] = -9.21038227100566
        struct[0].Gy_ini[115,42] = -5.65456401516768
        struct[0].Gy_ini[115,43] = 30.9517475172273
        struct[0].Gy_ini[115,44] = -1.84896616921897
        struct[0].Gy_ini[115,45] = -9.21038227100566
        struct[0].Gy_ini[115,46] = -0.793238195499528
        struct[0].Gy_ini[115,47] = -9.00835072044485
        struct[0].Gy_ini[115,104] = 10.2713736253513
        struct[0].Gy_ini[115,105] = -39.6392229058202
        struct[0].Gy_ini[115,106] = -54.0657727682604
        struct[0].Gy_ini[115,107] = 105.571769313180
        struct[0].Gy_ini[115,108] = 10.2713736253513
        struct[0].Gy_ini[115,109] = -39.6392229058201
        struct[0].Gy_ini[115,110] = 23.2849644809540
        struct[0].Gy_ini[115,111] = -24.9575997948692
        struct[0].Gy_ini[115,112] = -8.42240745613237
        struct[0].Gy_ini[115,113] = 48.8496051768258
        struct[0].Gy_ini[115,114] = 59.7203367834280
        struct[0].Gy_ini[115,115] = -136.523516830407
        struct[0].Gy_ini[115,116] = -8.42240745613238
        struct[0].Gy_ini[115,117] = 48.8496051768258
        struct[0].Gy_ini[115,118] = -22.4917262854545
        struct[0].Gy_ini[115,119] = 33.9659505153141
        struct[0].Gy_ini[116,40] = -9.00835072044484
        struct[0].Gy_ini[116,41] = 0.793238195499527
        struct[0].Gy_ini[116,42] = -9.21038227100566
        struct[0].Gy_ini[116,43] = 1.84896616921897
        struct[0].Gy_ini[116,44] = 30.9517475172273
        struct[0].Gy_ini[116,45] = 5.65456401516768
        struct[0].Gy_ini[116,46] = -9.21038227100566
        struct[0].Gy_ini[116,47] = 1.84896616921897
        struct[0].Gy_ini[116,104] = -24.9575997948692
        struct[0].Gy_ini[116,105] = -23.2849644809540
        struct[0].Gy_ini[116,106] = -39.6392229058202
        struct[0].Gy_ini[116,107] = -10.2713736253513
        struct[0].Gy_ini[116,108] = 105.571769313180
        struct[0].Gy_ini[116,109] = 54.0657727682604
        struct[0].Gy_ini[116,110] = -39.6392229058201
        struct[0].Gy_ini[116,111] = -10.2713736253514
        struct[0].Gy_ini[116,112] = 33.9659505153141
        struct[0].Gy_ini[116,113] = 22.4917262854545
        struct[0].Gy_ini[116,114] = 48.8496051768259
        struct[0].Gy_ini[116,115] = 8.42240745613236
        struct[0].Gy_ini[116,116] = -136.523516830407
        struct[0].Gy_ini[116,117] = -59.7203367834281
        struct[0].Gy_ini[116,118] = 48.8496051768258
        struct[0].Gy_ini[116,119] = 8.42240745613239
        struct[0].Gy_ini[117,40] = -0.793238195499527
        struct[0].Gy_ini[117,41] = -9.00835072044484
        struct[0].Gy_ini[117,42] = -1.84896616921897
        struct[0].Gy_ini[117,43] = -9.21038227100566
        struct[0].Gy_ini[117,44] = -5.65456401516768
        struct[0].Gy_ini[117,45] = 30.9517475172273
        struct[0].Gy_ini[117,46] = -1.84896616921897
        struct[0].Gy_ini[117,47] = -9.21038227100566
        struct[0].Gy_ini[117,104] = 23.2849644809540
        struct[0].Gy_ini[117,105] = -24.9575997948692
        struct[0].Gy_ini[117,106] = 10.2713736253513
        struct[0].Gy_ini[117,107] = -39.6392229058202
        struct[0].Gy_ini[117,108] = -54.0657727682604
        struct[0].Gy_ini[117,109] = 105.571769313180
        struct[0].Gy_ini[117,110] = 10.2713736253514
        struct[0].Gy_ini[117,111] = -39.6392229058201
        struct[0].Gy_ini[117,112] = -22.4917262854545
        struct[0].Gy_ini[117,113] = 33.9659505153141
        struct[0].Gy_ini[117,114] = -8.42240745613236
        struct[0].Gy_ini[117,115] = 48.8496051768259
        struct[0].Gy_ini[117,116] = 59.7203367834281
        struct[0].Gy_ini[117,117] = -136.523516830407
        struct[0].Gy_ini[117,118] = -8.42240745613239
        struct[0].Gy_ini[117,119] = 48.8496051768258
        struct[0].Gy_ini[118,40] = -9.21038227100566
        struct[0].Gy_ini[118,41] = 1.84896616921897
        struct[0].Gy_ini[118,42] = -9.00835072044485
        struct[0].Gy_ini[118,43] = 0.793238195499527
        struct[0].Gy_ini[118,44] = -9.21038227100566
        struct[0].Gy_ini[118,45] = 1.84896616921897
        struct[0].Gy_ini[118,46] = 30.9517475172273
        struct[0].Gy_ini[118,47] = 5.65456401516768
        struct[0].Gy_ini[118,104] = -39.6392229058202
        struct[0].Gy_ini[118,105] = -10.2713736253513
        struct[0].Gy_ini[118,106] = -24.9575997948692
        struct[0].Gy_ini[118,107] = -23.2849644809540
        struct[0].Gy_ini[118,108] = -39.6392229058202
        struct[0].Gy_ini[118,109] = -10.2713736253513
        struct[0].Gy_ini[118,110] = 105.571769313180
        struct[0].Gy_ini[118,111] = 54.0657727682604
        struct[0].Gy_ini[118,112] = 48.8496051768258
        struct[0].Gy_ini[118,113] = 8.42240745613237
        struct[0].Gy_ini[118,114] = 33.9659505153140
        struct[0].Gy_ini[118,115] = 22.4917262854545
        struct[0].Gy_ini[118,116] = 48.8496051768258
        struct[0].Gy_ini[118,117] = 8.42240745613237
        struct[0].Gy_ini[118,118] = -136.548516830407
        struct[0].Gy_ini[118,119] = -59.7203367834281
        struct[0].Gy_ini[119,40] = -1.84896616921897
        struct[0].Gy_ini[119,41] = -9.21038227100566
        struct[0].Gy_ini[119,42] = -0.793238195499527
        struct[0].Gy_ini[119,43] = -9.00835072044485
        struct[0].Gy_ini[119,44] = -1.84896616921897
        struct[0].Gy_ini[119,45] = -9.21038227100566
        struct[0].Gy_ini[119,46] = -5.65456401516768
        struct[0].Gy_ini[119,47] = 30.9517475172273
        struct[0].Gy_ini[119,104] = 10.2713736253513
        struct[0].Gy_ini[119,105] = -39.6392229058202
        struct[0].Gy_ini[119,106] = 23.2849644809540
        struct[0].Gy_ini[119,107] = -24.9575997948692
        struct[0].Gy_ini[119,108] = 10.2713736253513
        struct[0].Gy_ini[119,109] = -39.6392229058202
        struct[0].Gy_ini[119,110] = -54.0657727682604
        struct[0].Gy_ini[119,111] = 105.571769313180
        struct[0].Gy_ini[119,112] = -8.42240745613237
        struct[0].Gy_ini[119,113] = 48.8496051768258
        struct[0].Gy_ini[119,114] = -22.4917262854545
        struct[0].Gy_ini[119,115] = 33.9659505153140
        struct[0].Gy_ini[119,116] = -8.42240745613237
        struct[0].Gy_ini[119,117] = 48.8496051768258
        struct[0].Gy_ini[119,118] = 59.7203367834281
        struct[0].Gy_ini[119,119] = -136.548516830407
        struct[0].Gy_ini[120,64] = 26.5300693004806
        struct[0].Gy_ini[120,65] = 4.84676915585801
        struct[0].Gy_ini[120,66] = -7.89461337514770
        struct[0].Gy_ini[120,67] = 1.58482814504483
        struct[0].Gy_ini[120,68] = -7.72144347466701
        struct[0].Gy_ini[120,69] = 0.679918453285310
        struct[0].Gy_ini[120,70] = -7.89461337514771
        struct[0].Gy_ini[120,71] = 1.58482814504483
        struct[0].Gy_ini[120,120] = -53.0601386009611
        struct[0].Gy_ini[120,121] = -9.69353831171601
        struct[0].Gy_ini[120,122] = 15.7892267502954
        struct[0].Gy_ini[120,123] = -3.16965629008966
        struct[0].Gy_ini[120,124] = 15.4428869493340
        struct[0].Gy_ini[120,125] = -1.35983690657062
        struct[0].Gy_ini[120,126] = 15.7892267502954
        struct[0].Gy_ini[120,127] = -3.16965629008966
        struct[0].Gy_ini[120,128] = 26.5300693004806
        struct[0].Gy_ini[120,129] = 4.84676915585801
        struct[0].Gy_ini[120,130] = -7.89461337514770
        struct[0].Gy_ini[120,131] = 1.58482814504483
        struct[0].Gy_ini[120,132] = -7.72144347466701
        struct[0].Gy_ini[120,133] = 0.679918453285310
        struct[0].Gy_ini[120,134] = -7.89461337514771
        struct[0].Gy_ini[120,135] = 1.58482814504483
        struct[0].Gy_ini[121,64] = -4.84676915585801
        struct[0].Gy_ini[121,65] = 26.5300693004806
        struct[0].Gy_ini[121,66] = -1.58482814504483
        struct[0].Gy_ini[121,67] = -7.89461337514770
        struct[0].Gy_ini[121,68] = -0.679918453285310
        struct[0].Gy_ini[121,69] = -7.72144347466701
        struct[0].Gy_ini[121,70] = -1.58482814504483
        struct[0].Gy_ini[121,71] = -7.89461337514771
        struct[0].Gy_ini[121,120] = 9.69353831171601
        struct[0].Gy_ini[121,121] = -53.0601386009611
        struct[0].Gy_ini[121,122] = 3.16965629008966
        struct[0].Gy_ini[121,123] = 15.7892267502954
        struct[0].Gy_ini[121,124] = 1.35983690657062
        struct[0].Gy_ini[121,125] = 15.4428869493340
        struct[0].Gy_ini[121,126] = 3.16965629008966
        struct[0].Gy_ini[121,127] = 15.7892267502954
        struct[0].Gy_ini[121,128] = -4.84676915585801
        struct[0].Gy_ini[121,129] = 26.5300693004806
        struct[0].Gy_ini[121,130] = -1.58482814504483
        struct[0].Gy_ini[121,131] = -7.89461337514770
        struct[0].Gy_ini[121,132] = -0.679918453285310
        struct[0].Gy_ini[121,133] = -7.72144347466701
        struct[0].Gy_ini[121,134] = -1.58482814504483
        struct[0].Gy_ini[121,135] = -7.89461337514771
        struct[0].Gy_ini[122,64] = -7.89461337514771
        struct[0].Gy_ini[122,65] = 1.58482814504483
        struct[0].Gy_ini[122,66] = 26.5300693004806
        struct[0].Gy_ini[122,67] = 4.84676915585801
        struct[0].Gy_ini[122,68] = -7.89461337514771
        struct[0].Gy_ini[122,69] = 1.58482814504483
        struct[0].Gy_ini[122,70] = -7.72144347466701
        struct[0].Gy_ini[122,71] = 0.679918453285308
        struct[0].Gy_ini[122,120] = 15.7892267502954
        struct[0].Gy_ini[122,121] = -3.16965629008966
        struct[0].Gy_ini[122,122] = -53.0601386009611
        struct[0].Gy_ini[122,123] = -9.69353831171602
        struct[0].Gy_ini[122,124] = 15.7892267502954
        struct[0].Gy_ini[122,125] = -3.16965629008967
        struct[0].Gy_ini[122,126] = 15.4428869493340
        struct[0].Gy_ini[122,127] = -1.35983690657062
        struct[0].Gy_ini[122,128] = -7.89461337514771
        struct[0].Gy_ini[122,129] = 1.58482814504483
        struct[0].Gy_ini[122,130] = 26.5300693004806
        struct[0].Gy_ini[122,131] = 4.84676915585801
        struct[0].Gy_ini[122,132] = -7.89461337514771
        struct[0].Gy_ini[122,133] = 1.58482814504483
        struct[0].Gy_ini[122,134] = -7.72144347466701
        struct[0].Gy_ini[122,135] = 0.679918453285308
        struct[0].Gy_ini[123,64] = -1.58482814504483
        struct[0].Gy_ini[123,65] = -7.89461337514771
        struct[0].Gy_ini[123,66] = -4.84676915585801
        struct[0].Gy_ini[123,67] = 26.5300693004806
        struct[0].Gy_ini[123,68] = -1.58482814504483
        struct[0].Gy_ini[123,69] = -7.89461337514771
        struct[0].Gy_ini[123,70] = -0.679918453285308
        struct[0].Gy_ini[123,71] = -7.72144347466701
        struct[0].Gy_ini[123,120] = 3.16965629008966
        struct[0].Gy_ini[123,121] = 15.7892267502954
        struct[0].Gy_ini[123,122] = 9.69353831171602
        struct[0].Gy_ini[123,123] = -53.0601386009611
        struct[0].Gy_ini[123,124] = 3.16965629008967
        struct[0].Gy_ini[123,125] = 15.7892267502954
        struct[0].Gy_ini[123,126] = 1.35983690657062
        struct[0].Gy_ini[123,127] = 15.4428869493340
        struct[0].Gy_ini[123,128] = -1.58482814504483
        struct[0].Gy_ini[123,129] = -7.89461337514771
        struct[0].Gy_ini[123,130] = -4.84676915585801
        struct[0].Gy_ini[123,131] = 26.5300693004806
        struct[0].Gy_ini[123,132] = -1.58482814504483
        struct[0].Gy_ini[123,133] = -7.89461337514771
        struct[0].Gy_ini[123,134] = -0.679918453285308
        struct[0].Gy_ini[123,135] = -7.72144347466701
        struct[0].Gy_ini[124,64] = -7.72144347466701
        struct[0].Gy_ini[124,65] = 0.679918453285310
        struct[0].Gy_ini[124,66] = -7.89461337514770
        struct[0].Gy_ini[124,67] = 1.58482814504483
        struct[0].Gy_ini[124,68] = 26.5300693004806
        struct[0].Gy_ini[124,69] = 4.84676915585801
        struct[0].Gy_ini[124,70] = -7.89461337514771
        struct[0].Gy_ini[124,71] = 1.58482814504483
        struct[0].Gy_ini[124,120] = 15.4428869493340
        struct[0].Gy_ini[124,121] = -1.35983690657062
        struct[0].Gy_ini[124,122] = 15.7892267502954
        struct[0].Gy_ini[124,123] = -3.16965629008966
        struct[0].Gy_ini[124,124] = -53.0601386009611
        struct[0].Gy_ini[124,125] = -9.69353831171602
        struct[0].Gy_ini[124,126] = 15.7892267502954
        struct[0].Gy_ini[124,127] = -3.16965629008967
        struct[0].Gy_ini[124,128] = -7.72144347466701
        struct[0].Gy_ini[124,129] = 0.679918453285310
        struct[0].Gy_ini[124,130] = -7.89461337514770
        struct[0].Gy_ini[124,131] = 1.58482814504483
        struct[0].Gy_ini[124,132] = 26.5300693004806
        struct[0].Gy_ini[124,133] = 4.84676915585801
        struct[0].Gy_ini[124,134] = -7.89461337514771
        struct[0].Gy_ini[124,135] = 1.58482814504483
        struct[0].Gy_ini[125,64] = -0.679918453285310
        struct[0].Gy_ini[125,65] = -7.72144347466701
        struct[0].Gy_ini[125,66] = -1.58482814504483
        struct[0].Gy_ini[125,67] = -7.89461337514770
        struct[0].Gy_ini[125,68] = -4.84676915585801
        struct[0].Gy_ini[125,69] = 26.5300693004806
        struct[0].Gy_ini[125,70] = -1.58482814504483
        struct[0].Gy_ini[125,71] = -7.89461337514771
        struct[0].Gy_ini[125,120] = 1.35983690657062
        struct[0].Gy_ini[125,121] = 15.4428869493340
        struct[0].Gy_ini[125,122] = 3.16965629008966
        struct[0].Gy_ini[125,123] = 15.7892267502954
        struct[0].Gy_ini[125,124] = 9.69353831171602
        struct[0].Gy_ini[125,125] = -53.0601386009611
        struct[0].Gy_ini[125,126] = 3.16965629008967
        struct[0].Gy_ini[125,127] = 15.7892267502954
        struct[0].Gy_ini[125,128] = -0.679918453285310
        struct[0].Gy_ini[125,129] = -7.72144347466701
        struct[0].Gy_ini[125,130] = -1.58482814504483
        struct[0].Gy_ini[125,131] = -7.89461337514770
        struct[0].Gy_ini[125,132] = -4.84676915585801
        struct[0].Gy_ini[125,133] = 26.5300693004806
        struct[0].Gy_ini[125,134] = -1.58482814504483
        struct[0].Gy_ini[125,135] = -7.89461337514771
        struct[0].Gy_ini[126,64] = -7.89461337514771
        struct[0].Gy_ini[126,65] = 1.58482814504483
        struct[0].Gy_ini[126,66] = -7.72144347466701
        struct[0].Gy_ini[126,67] = 0.679918453285308
        struct[0].Gy_ini[126,68] = -7.89461337514771
        struct[0].Gy_ini[126,69] = 1.58482814504483
        struct[0].Gy_ini[126,70] = 26.5300693004806
        struct[0].Gy_ini[126,71] = 4.84676915585801
        struct[0].Gy_ini[126,120] = 15.7892267502954
        struct[0].Gy_ini[126,121] = -3.16965629008966
        struct[0].Gy_ini[126,122] = 15.4428869493340
        struct[0].Gy_ini[126,123] = -1.35983690657062
        struct[0].Gy_ini[126,124] = 15.7892267502954
        struct[0].Gy_ini[126,125] = -3.16965629008966
        struct[0].Gy_ini[126,126] = -53.0601386009611
        struct[0].Gy_ini[126,127] = -9.69353831171602
        struct[0].Gy_ini[126,128] = -7.89461337514771
        struct[0].Gy_ini[126,129] = 1.58482814504483
        struct[0].Gy_ini[126,130] = -7.72144347466701
        struct[0].Gy_ini[126,131] = 0.679918453285308
        struct[0].Gy_ini[126,132] = -7.89461337514771
        struct[0].Gy_ini[126,133] = 1.58482814504483
        struct[0].Gy_ini[126,134] = 26.5300693004806
        struct[0].Gy_ini[126,135] = 4.84676915585801
        struct[0].Gy_ini[127,64] = -1.58482814504483
        struct[0].Gy_ini[127,65] = -7.89461337514771
        struct[0].Gy_ini[127,66] = -0.679918453285308
        struct[0].Gy_ini[127,67] = -7.72144347466701
        struct[0].Gy_ini[127,68] = -1.58482814504483
        struct[0].Gy_ini[127,69] = -7.89461337514771
        struct[0].Gy_ini[127,70] = -4.84676915585801
        struct[0].Gy_ini[127,71] = 26.5300693004806
        struct[0].Gy_ini[127,120] = 3.16965629008966
        struct[0].Gy_ini[127,121] = 15.7892267502954
        struct[0].Gy_ini[127,122] = 1.35983690657062
        struct[0].Gy_ini[127,123] = 15.4428869493340
        struct[0].Gy_ini[127,124] = 3.16965629008966
        struct[0].Gy_ini[127,125] = 15.7892267502954
        struct[0].Gy_ini[127,126] = 9.69353831171602
        struct[0].Gy_ini[127,127] = -53.0601386009611
        struct[0].Gy_ini[127,128] = -1.58482814504483
        struct[0].Gy_ini[127,129] = -7.89461337514771
        struct[0].Gy_ini[127,130] = -0.679918453285308
        struct[0].Gy_ini[127,131] = -7.72144347466701
        struct[0].Gy_ini[127,132] = -1.58482814504483
        struct[0].Gy_ini[127,133] = -7.89461337514771
        struct[0].Gy_ini[127,134] = -4.84676915585801
        struct[0].Gy_ini[127,135] = 26.5300693004806
        struct[0].Gy_ini[128,120] = 26.5300693004806
        struct[0].Gy_ini[128,121] = 4.84676915585801
        struct[0].Gy_ini[128,122] = -7.89461337514770
        struct[0].Gy_ini[128,123] = 1.58482814504483
        struct[0].Gy_ini[128,124] = -7.72144347466701
        struct[0].Gy_ini[128,125] = 0.679918453285310
        struct[0].Gy_ini[128,126] = -7.89461337514771
        struct[0].Gy_ini[128,127] = 1.58482814504483
        struct[0].Gy_ini[128,128] = -53.0601386009611
        struct[0].Gy_ini[128,129] = -9.69353831171601
        struct[0].Gy_ini[128,130] = 15.7892267502954
        struct[0].Gy_ini[128,131] = -3.16965629008966
        struct[0].Gy_ini[128,132] = 15.4428869493340
        struct[0].Gy_ini[128,133] = -1.35983690657062
        struct[0].Gy_ini[128,134] = 15.7892267502954
        struct[0].Gy_ini[128,135] = -3.16965629008966
        struct[0].Gy_ini[128,136] = 26.5300693004806
        struct[0].Gy_ini[128,137] = 4.84676915585801
        struct[0].Gy_ini[128,138] = -7.89461337514770
        struct[0].Gy_ini[128,139] = 1.58482814504483
        struct[0].Gy_ini[128,140] = -7.72144347466701
        struct[0].Gy_ini[128,141] = 0.679918453285310
        struct[0].Gy_ini[128,142] = -7.89461337514771
        struct[0].Gy_ini[128,143] = 1.58482814504483
        struct[0].Gy_ini[129,120] = -4.84676915585801
        struct[0].Gy_ini[129,121] = 26.5300693004806
        struct[0].Gy_ini[129,122] = -1.58482814504483
        struct[0].Gy_ini[129,123] = -7.89461337514770
        struct[0].Gy_ini[129,124] = -0.679918453285310
        struct[0].Gy_ini[129,125] = -7.72144347466701
        struct[0].Gy_ini[129,126] = -1.58482814504483
        struct[0].Gy_ini[129,127] = -7.89461337514771
        struct[0].Gy_ini[129,128] = 9.69353831171601
        struct[0].Gy_ini[129,129] = -53.0601386009611
        struct[0].Gy_ini[129,130] = 3.16965629008966
        struct[0].Gy_ini[129,131] = 15.7892267502954
        struct[0].Gy_ini[129,132] = 1.35983690657062
        struct[0].Gy_ini[129,133] = 15.4428869493340
        struct[0].Gy_ini[129,134] = 3.16965629008966
        struct[0].Gy_ini[129,135] = 15.7892267502954
        struct[0].Gy_ini[129,136] = -4.84676915585801
        struct[0].Gy_ini[129,137] = 26.5300693004806
        struct[0].Gy_ini[129,138] = -1.58482814504483
        struct[0].Gy_ini[129,139] = -7.89461337514770
        struct[0].Gy_ini[129,140] = -0.679918453285310
        struct[0].Gy_ini[129,141] = -7.72144347466701
        struct[0].Gy_ini[129,142] = -1.58482814504483
        struct[0].Gy_ini[129,143] = -7.89461337514771
        struct[0].Gy_ini[130,120] = -7.89461337514771
        struct[0].Gy_ini[130,121] = 1.58482814504483
        struct[0].Gy_ini[130,122] = 26.5300693004806
        struct[0].Gy_ini[130,123] = 4.84676915585801
        struct[0].Gy_ini[130,124] = -7.89461337514771
        struct[0].Gy_ini[130,125] = 1.58482814504483
        struct[0].Gy_ini[130,126] = -7.72144347466701
        struct[0].Gy_ini[130,127] = 0.679918453285308
        struct[0].Gy_ini[130,128] = 15.7892267502954
        struct[0].Gy_ini[130,129] = -3.16965629008966
        struct[0].Gy_ini[130,130] = -53.0601386009611
        struct[0].Gy_ini[130,131] = -9.69353831171602
        struct[0].Gy_ini[130,132] = 15.7892267502954
        struct[0].Gy_ini[130,133] = -3.16965629008967
        struct[0].Gy_ini[130,134] = 15.4428869493340
        struct[0].Gy_ini[130,135] = -1.35983690657062
        struct[0].Gy_ini[130,136] = -7.89461337514771
        struct[0].Gy_ini[130,137] = 1.58482814504483
        struct[0].Gy_ini[130,138] = 26.5300693004806
        struct[0].Gy_ini[130,139] = 4.84676915585801
        struct[0].Gy_ini[130,140] = -7.89461337514771
        struct[0].Gy_ini[130,141] = 1.58482814504483
        struct[0].Gy_ini[130,142] = -7.72144347466701
        struct[0].Gy_ini[130,143] = 0.679918453285308
        struct[0].Gy_ini[131,120] = -1.58482814504483
        struct[0].Gy_ini[131,121] = -7.89461337514771
        struct[0].Gy_ini[131,122] = -4.84676915585801
        struct[0].Gy_ini[131,123] = 26.5300693004806
        struct[0].Gy_ini[131,124] = -1.58482814504483
        struct[0].Gy_ini[131,125] = -7.89461337514771
        struct[0].Gy_ini[131,126] = -0.679918453285308
        struct[0].Gy_ini[131,127] = -7.72144347466701
        struct[0].Gy_ini[131,128] = 3.16965629008966
        struct[0].Gy_ini[131,129] = 15.7892267502954
        struct[0].Gy_ini[131,130] = 9.69353831171602
        struct[0].Gy_ini[131,131] = -53.0601386009611
        struct[0].Gy_ini[131,132] = 3.16965629008967
        struct[0].Gy_ini[131,133] = 15.7892267502954
        struct[0].Gy_ini[131,134] = 1.35983690657062
        struct[0].Gy_ini[131,135] = 15.4428869493340
        struct[0].Gy_ini[131,136] = -1.58482814504483
        struct[0].Gy_ini[131,137] = -7.89461337514771
        struct[0].Gy_ini[131,138] = -4.84676915585801
        struct[0].Gy_ini[131,139] = 26.5300693004806
        struct[0].Gy_ini[131,140] = -1.58482814504483
        struct[0].Gy_ini[131,141] = -7.89461337514771
        struct[0].Gy_ini[131,142] = -0.679918453285308
        struct[0].Gy_ini[131,143] = -7.72144347466701
        struct[0].Gy_ini[132,120] = -7.72144347466701
        struct[0].Gy_ini[132,121] = 0.679918453285310
        struct[0].Gy_ini[132,122] = -7.89461337514770
        struct[0].Gy_ini[132,123] = 1.58482814504483
        struct[0].Gy_ini[132,124] = 26.5300693004806
        struct[0].Gy_ini[132,125] = 4.84676915585801
        struct[0].Gy_ini[132,126] = -7.89461337514771
        struct[0].Gy_ini[132,127] = 1.58482814504483
        struct[0].Gy_ini[132,128] = 15.4428869493340
        struct[0].Gy_ini[132,129] = -1.35983690657062
        struct[0].Gy_ini[132,130] = 15.7892267502954
        struct[0].Gy_ini[132,131] = -3.16965629008966
        struct[0].Gy_ini[132,132] = -53.0601386009611
        struct[0].Gy_ini[132,133] = -9.69353831171602
        struct[0].Gy_ini[132,134] = 15.7892267502954
        struct[0].Gy_ini[132,135] = -3.16965629008967
        struct[0].Gy_ini[132,136] = -7.72144347466701
        struct[0].Gy_ini[132,137] = 0.679918453285310
        struct[0].Gy_ini[132,138] = -7.89461337514770
        struct[0].Gy_ini[132,139] = 1.58482814504483
        struct[0].Gy_ini[132,140] = 26.5300693004806
        struct[0].Gy_ini[132,141] = 4.84676915585801
        struct[0].Gy_ini[132,142] = -7.89461337514771
        struct[0].Gy_ini[132,143] = 1.58482814504483
        struct[0].Gy_ini[133,120] = -0.679918453285310
        struct[0].Gy_ini[133,121] = -7.72144347466701
        struct[0].Gy_ini[133,122] = -1.58482814504483
        struct[0].Gy_ini[133,123] = -7.89461337514770
        struct[0].Gy_ini[133,124] = -4.84676915585801
        struct[0].Gy_ini[133,125] = 26.5300693004806
        struct[0].Gy_ini[133,126] = -1.58482814504483
        struct[0].Gy_ini[133,127] = -7.89461337514771
        struct[0].Gy_ini[133,128] = 1.35983690657062
        struct[0].Gy_ini[133,129] = 15.4428869493340
        struct[0].Gy_ini[133,130] = 3.16965629008966
        struct[0].Gy_ini[133,131] = 15.7892267502954
        struct[0].Gy_ini[133,132] = 9.69353831171602
        struct[0].Gy_ini[133,133] = -53.0601386009611
        struct[0].Gy_ini[133,134] = 3.16965629008967
        struct[0].Gy_ini[133,135] = 15.7892267502954
        struct[0].Gy_ini[133,136] = -0.679918453285310
        struct[0].Gy_ini[133,137] = -7.72144347466701
        struct[0].Gy_ini[133,138] = -1.58482814504483
        struct[0].Gy_ini[133,139] = -7.89461337514770
        struct[0].Gy_ini[133,140] = -4.84676915585801
        struct[0].Gy_ini[133,141] = 26.5300693004806
        struct[0].Gy_ini[133,142] = -1.58482814504483
        struct[0].Gy_ini[133,143] = -7.89461337514771
        struct[0].Gy_ini[134,120] = -7.89461337514771
        struct[0].Gy_ini[134,121] = 1.58482814504483
        struct[0].Gy_ini[134,122] = -7.72144347466701
        struct[0].Gy_ini[134,123] = 0.679918453285308
        struct[0].Gy_ini[134,124] = -7.89461337514771
        struct[0].Gy_ini[134,125] = 1.58482814504483
        struct[0].Gy_ini[134,126] = 26.5300693004806
        struct[0].Gy_ini[134,127] = 4.84676915585801
        struct[0].Gy_ini[134,128] = 15.7892267502954
        struct[0].Gy_ini[134,129] = -3.16965629008966
        struct[0].Gy_ini[134,130] = 15.4428869493340
        struct[0].Gy_ini[134,131] = -1.35983690657062
        struct[0].Gy_ini[134,132] = 15.7892267502954
        struct[0].Gy_ini[134,133] = -3.16965629008966
        struct[0].Gy_ini[134,134] = -53.0851386009611
        struct[0].Gy_ini[134,135] = -9.69353831171602
        struct[0].Gy_ini[134,136] = -7.89461337514771
        struct[0].Gy_ini[134,137] = 1.58482814504483
        struct[0].Gy_ini[134,138] = -7.72144347466701
        struct[0].Gy_ini[134,139] = 0.679918453285308
        struct[0].Gy_ini[134,140] = -7.89461337514771
        struct[0].Gy_ini[134,141] = 1.58482814504483
        struct[0].Gy_ini[134,142] = 26.5300693004806
        struct[0].Gy_ini[134,143] = 4.84676915585801
        struct[0].Gy_ini[135,120] = -1.58482814504483
        struct[0].Gy_ini[135,121] = -7.89461337514771
        struct[0].Gy_ini[135,122] = -0.679918453285308
        struct[0].Gy_ini[135,123] = -7.72144347466701
        struct[0].Gy_ini[135,124] = -1.58482814504483
        struct[0].Gy_ini[135,125] = -7.89461337514771
        struct[0].Gy_ini[135,126] = -4.84676915585801
        struct[0].Gy_ini[135,127] = 26.5300693004806
        struct[0].Gy_ini[135,128] = 3.16965629008966
        struct[0].Gy_ini[135,129] = 15.7892267502954
        struct[0].Gy_ini[135,130] = 1.35983690657062
        struct[0].Gy_ini[135,131] = 15.4428869493340
        struct[0].Gy_ini[135,132] = 3.16965629008966
        struct[0].Gy_ini[135,133] = 15.7892267502954
        struct[0].Gy_ini[135,134] = 9.69353831171602
        struct[0].Gy_ini[135,135] = -53.0851386009611
        struct[0].Gy_ini[135,136] = -1.58482814504483
        struct[0].Gy_ini[135,137] = -7.89461337514771
        struct[0].Gy_ini[135,138] = -0.679918453285308
        struct[0].Gy_ini[135,139] = -7.72144347466701
        struct[0].Gy_ini[135,140] = -1.58482814504483
        struct[0].Gy_ini[135,141] = -7.89461337514771
        struct[0].Gy_ini[135,142] = -4.84676915585801
        struct[0].Gy_ini[135,143] = 26.5300693004806
        struct[0].Gy_ini[136,16] = 26.5300693004806
        struct[0].Gy_ini[136,17] = 4.84676915585801
        struct[0].Gy_ini[136,18] = -7.89461337514770
        struct[0].Gy_ini[136,19] = 1.58482814504483
        struct[0].Gy_ini[136,20] = -7.72144347466701
        struct[0].Gy_ini[136,21] = 0.679918453285310
        struct[0].Gy_ini[136,22] = -7.89461337514771
        struct[0].Gy_ini[136,23] = 1.58482814504483
        struct[0].Gy_ini[136,128] = 26.5300693004806
        struct[0].Gy_ini[136,129] = 4.84676915585801
        struct[0].Gy_ini[136,130] = -7.89461337514770
        struct[0].Gy_ini[136,131] = 1.58482814504483
        struct[0].Gy_ini[136,132] = -7.72144347466701
        struct[0].Gy_ini[136,133] = 0.679918453285310
        struct[0].Gy_ini[136,134] = -7.89461337514771
        struct[0].Gy_ini[136,135] = 1.58482814504483
        struct[0].Gy_ini[136,136] = -53.0601386009611
        struct[0].Gy_ini[136,137] = -9.69353831171601
        struct[0].Gy_ini[136,138] = 15.7892267502954
        struct[0].Gy_ini[136,139] = -3.16965629008966
        struct[0].Gy_ini[136,140] = 15.4428869493340
        struct[0].Gy_ini[136,141] = -1.35983690657062
        struct[0].Gy_ini[136,142] = 15.7892267502954
        struct[0].Gy_ini[136,143] = -3.16965629008966
        struct[0].Gy_ini[137,16] = -4.84676915585801
        struct[0].Gy_ini[137,17] = 26.5300693004806
        struct[0].Gy_ini[137,18] = -1.58482814504483
        struct[0].Gy_ini[137,19] = -7.89461337514770
        struct[0].Gy_ini[137,20] = -0.679918453285310
        struct[0].Gy_ini[137,21] = -7.72144347466701
        struct[0].Gy_ini[137,22] = -1.58482814504483
        struct[0].Gy_ini[137,23] = -7.89461337514771
        struct[0].Gy_ini[137,128] = -4.84676915585801
        struct[0].Gy_ini[137,129] = 26.5300693004806
        struct[0].Gy_ini[137,130] = -1.58482814504483
        struct[0].Gy_ini[137,131] = -7.89461337514770
        struct[0].Gy_ini[137,132] = -0.679918453285310
        struct[0].Gy_ini[137,133] = -7.72144347466701
        struct[0].Gy_ini[137,134] = -1.58482814504483
        struct[0].Gy_ini[137,135] = -7.89461337514771
        struct[0].Gy_ini[137,136] = 9.69353831171601
        struct[0].Gy_ini[137,137] = -53.0601386009611
        struct[0].Gy_ini[137,138] = 3.16965629008966
        struct[0].Gy_ini[137,139] = 15.7892267502954
        struct[0].Gy_ini[137,140] = 1.35983690657062
        struct[0].Gy_ini[137,141] = 15.4428869493340
        struct[0].Gy_ini[137,142] = 3.16965629008966
        struct[0].Gy_ini[137,143] = 15.7892267502954
        struct[0].Gy_ini[138,16] = -7.89461337514771
        struct[0].Gy_ini[138,17] = 1.58482814504483
        struct[0].Gy_ini[138,18] = 26.5300693004806
        struct[0].Gy_ini[138,19] = 4.84676915585801
        struct[0].Gy_ini[138,20] = -7.89461337514771
        struct[0].Gy_ini[138,21] = 1.58482814504483
        struct[0].Gy_ini[138,22] = -7.72144347466701
        struct[0].Gy_ini[138,23] = 0.679918453285308
        struct[0].Gy_ini[138,128] = -7.89461337514771
        struct[0].Gy_ini[138,129] = 1.58482814504483
        struct[0].Gy_ini[138,130] = 26.5300693004806
        struct[0].Gy_ini[138,131] = 4.84676915585801
        struct[0].Gy_ini[138,132] = -7.89461337514771
        struct[0].Gy_ini[138,133] = 1.58482814504483
        struct[0].Gy_ini[138,134] = -7.72144347466701
        struct[0].Gy_ini[138,135] = 0.679918453285308
        struct[0].Gy_ini[138,136] = 15.7892267502954
        struct[0].Gy_ini[138,137] = -3.16965629008966
        struct[0].Gy_ini[138,138] = -53.0601386009611
        struct[0].Gy_ini[138,139] = -9.69353831171602
        struct[0].Gy_ini[138,140] = 15.7892267502954
        struct[0].Gy_ini[138,141] = -3.16965629008967
        struct[0].Gy_ini[138,142] = 15.4428869493340
        struct[0].Gy_ini[138,143] = -1.35983690657062
        struct[0].Gy_ini[139,16] = -1.58482814504483
        struct[0].Gy_ini[139,17] = -7.89461337514771
        struct[0].Gy_ini[139,18] = -4.84676915585801
        struct[0].Gy_ini[139,19] = 26.5300693004806
        struct[0].Gy_ini[139,20] = -1.58482814504483
        struct[0].Gy_ini[139,21] = -7.89461337514771
        struct[0].Gy_ini[139,22] = -0.679918453285308
        struct[0].Gy_ini[139,23] = -7.72144347466701
        struct[0].Gy_ini[139,128] = -1.58482814504483
        struct[0].Gy_ini[139,129] = -7.89461337514771
        struct[0].Gy_ini[139,130] = -4.84676915585801
        struct[0].Gy_ini[139,131] = 26.5300693004806
        struct[0].Gy_ini[139,132] = -1.58482814504483
        struct[0].Gy_ini[139,133] = -7.89461337514771
        struct[0].Gy_ini[139,134] = -0.679918453285308
        struct[0].Gy_ini[139,135] = -7.72144347466701
        struct[0].Gy_ini[139,136] = 3.16965629008966
        struct[0].Gy_ini[139,137] = 15.7892267502954
        struct[0].Gy_ini[139,138] = 9.69353831171602
        struct[0].Gy_ini[139,139] = -53.0601386009611
        struct[0].Gy_ini[139,140] = 3.16965629008967
        struct[0].Gy_ini[139,141] = 15.7892267502954
        struct[0].Gy_ini[139,142] = 1.35983690657062
        struct[0].Gy_ini[139,143] = 15.4428869493340
        struct[0].Gy_ini[140,16] = -7.72144347466701
        struct[0].Gy_ini[140,17] = 0.679918453285310
        struct[0].Gy_ini[140,18] = -7.89461337514770
        struct[0].Gy_ini[140,19] = 1.58482814504483
        struct[0].Gy_ini[140,20] = 26.5300693004806
        struct[0].Gy_ini[140,21] = 4.84676915585801
        struct[0].Gy_ini[140,22] = -7.89461337514771
        struct[0].Gy_ini[140,23] = 1.58482814504483
        struct[0].Gy_ini[140,128] = -7.72144347466701
        struct[0].Gy_ini[140,129] = 0.679918453285310
        struct[0].Gy_ini[140,130] = -7.89461337514770
        struct[0].Gy_ini[140,131] = 1.58482814504483
        struct[0].Gy_ini[140,132] = 26.5300693004806
        struct[0].Gy_ini[140,133] = 4.84676915585801
        struct[0].Gy_ini[140,134] = -7.89461337514771
        struct[0].Gy_ini[140,135] = 1.58482814504483
        struct[0].Gy_ini[140,136] = 15.4428869493340
        struct[0].Gy_ini[140,137] = -1.35983690657062
        struct[0].Gy_ini[140,138] = 15.7892267502954
        struct[0].Gy_ini[140,139] = -3.16965629008966
        struct[0].Gy_ini[140,140] = -53.0601386009611
        struct[0].Gy_ini[140,141] = -9.69353831171602
        struct[0].Gy_ini[140,142] = 15.7892267502954
        struct[0].Gy_ini[140,143] = -3.16965629008967
        struct[0].Gy_ini[141,16] = -0.679918453285310
        struct[0].Gy_ini[141,17] = -7.72144347466701
        struct[0].Gy_ini[141,18] = -1.58482814504483
        struct[0].Gy_ini[141,19] = -7.89461337514770
        struct[0].Gy_ini[141,20] = -4.84676915585801
        struct[0].Gy_ini[141,21] = 26.5300693004806
        struct[0].Gy_ini[141,22] = -1.58482814504483
        struct[0].Gy_ini[141,23] = -7.89461337514771
        struct[0].Gy_ini[141,128] = -0.679918453285310
        struct[0].Gy_ini[141,129] = -7.72144347466701
        struct[0].Gy_ini[141,130] = -1.58482814504483
        struct[0].Gy_ini[141,131] = -7.89461337514770
        struct[0].Gy_ini[141,132] = -4.84676915585801
        struct[0].Gy_ini[141,133] = 26.5300693004806
        struct[0].Gy_ini[141,134] = -1.58482814504483
        struct[0].Gy_ini[141,135] = -7.89461337514771
        struct[0].Gy_ini[141,136] = 1.35983690657062
        struct[0].Gy_ini[141,137] = 15.4428869493340
        struct[0].Gy_ini[141,138] = 3.16965629008966
        struct[0].Gy_ini[141,139] = 15.7892267502954
        struct[0].Gy_ini[141,140] = 9.69353831171602
        struct[0].Gy_ini[141,141] = -53.0601386009611
        struct[0].Gy_ini[141,142] = 3.16965629008967
        struct[0].Gy_ini[141,143] = 15.7892267502954
        struct[0].Gy_ini[142,16] = -7.89461337514771
        struct[0].Gy_ini[142,17] = 1.58482814504483
        struct[0].Gy_ini[142,18] = -7.72144347466701
        struct[0].Gy_ini[142,19] = 0.679918453285308
        struct[0].Gy_ini[142,20] = -7.89461337514771
        struct[0].Gy_ini[142,21] = 1.58482814504483
        struct[0].Gy_ini[142,22] = 26.5300693004806
        struct[0].Gy_ini[142,23] = 4.84676915585801
        struct[0].Gy_ini[142,128] = -7.89461337514771
        struct[0].Gy_ini[142,129] = 1.58482814504483
        struct[0].Gy_ini[142,130] = -7.72144347466701
        struct[0].Gy_ini[142,131] = 0.679918453285308
        struct[0].Gy_ini[142,132] = -7.89461337514771
        struct[0].Gy_ini[142,133] = 1.58482814504483
        struct[0].Gy_ini[142,134] = 26.5300693004806
        struct[0].Gy_ini[142,135] = 4.84676915585801
        struct[0].Gy_ini[142,136] = 15.7892267502954
        struct[0].Gy_ini[142,137] = -3.16965629008966
        struct[0].Gy_ini[142,138] = 15.4428869493340
        struct[0].Gy_ini[142,139] = -1.35983690657062
        struct[0].Gy_ini[142,140] = 15.7892267502954
        struct[0].Gy_ini[142,141] = -3.16965629008966
        struct[0].Gy_ini[142,142] = -53.0601386009611
        struct[0].Gy_ini[142,143] = -9.69353831171602
        struct[0].Gy_ini[143,16] = -1.58482814504483
        struct[0].Gy_ini[143,17] = -7.89461337514771
        struct[0].Gy_ini[143,18] = -0.679918453285308
        struct[0].Gy_ini[143,19] = -7.72144347466701
        struct[0].Gy_ini[143,20] = -1.58482814504483
        struct[0].Gy_ini[143,21] = -7.89461337514771
        struct[0].Gy_ini[143,22] = -4.84676915585801
        struct[0].Gy_ini[143,23] = 26.5300693004806
        struct[0].Gy_ini[143,128] = -1.58482814504483
        struct[0].Gy_ini[143,129] = -7.89461337514771
        struct[0].Gy_ini[143,130] = -0.679918453285308
        struct[0].Gy_ini[143,131] = -7.72144347466701
        struct[0].Gy_ini[143,132] = -1.58482814504483
        struct[0].Gy_ini[143,133] = -7.89461337514771
        struct[0].Gy_ini[143,134] = -4.84676915585801
        struct[0].Gy_ini[143,135] = 26.5300693004806
        struct[0].Gy_ini[143,136] = 3.16965629008966
        struct[0].Gy_ini[143,137] = 15.7892267502954
        struct[0].Gy_ini[143,138] = 1.35983690657062
        struct[0].Gy_ini[143,139] = 15.4428869493340
        struct[0].Gy_ini[143,140] = 3.16965629008966
        struct[0].Gy_ini[143,141] = 15.7892267502954
        struct[0].Gy_ini[143,142] = 9.69353831171602
        struct[0].Gy_ini[143,143] = -53.0601386009611
        struct[0].Gy_ini[144,0] = i_R1_a_r
        struct[0].Gy_ini[144,1] = i_R1_a_i
        struct[0].Gy_ini[144,6] = -i_R1_a_r
        struct[0].Gy_ini[144,7] = -i_R1_a_i
        struct[0].Gy_ini[144,144] = v_R1_a_r - v_R1_n_r
        struct[0].Gy_ini[144,145] = v_R1_a_i - v_R1_n_i
        struct[0].Gy_ini[145,2] = i_R1_b_r
        struct[0].Gy_ini[145,3] = i_R1_b_i
        struct[0].Gy_ini[145,6] = -i_R1_b_r
        struct[0].Gy_ini[145,7] = -i_R1_b_i
        struct[0].Gy_ini[145,146] = v_R1_b_r - v_R1_n_r
        struct[0].Gy_ini[145,147] = v_R1_b_i - v_R1_n_i
        struct[0].Gy_ini[146,4] = i_R1_c_r
        struct[0].Gy_ini[146,5] = i_R1_c_i
        struct[0].Gy_ini[146,6] = -i_R1_c_r
        struct[0].Gy_ini[146,7] = -i_R1_c_i
        struct[0].Gy_ini[146,148] = v_R1_c_r - v_R1_n_r
        struct[0].Gy_ini[146,149] = v_R1_c_i - v_R1_n_i
        struct[0].Gy_ini[147,0] = -i_R1_a_i
        struct[0].Gy_ini[147,1] = i_R1_a_r
        struct[0].Gy_ini[147,6] = i_R1_a_i
        struct[0].Gy_ini[147,7] = -i_R1_a_r
        struct[0].Gy_ini[147,144] = v_R1_a_i - v_R1_n_i
        struct[0].Gy_ini[147,145] = -v_R1_a_r + v_R1_n_r
        struct[0].Gy_ini[148,2] = -i_R1_b_i
        struct[0].Gy_ini[148,3] = i_R1_b_r
        struct[0].Gy_ini[148,6] = i_R1_b_i
        struct[0].Gy_ini[148,7] = -i_R1_b_r
        struct[0].Gy_ini[148,146] = v_R1_b_i - v_R1_n_i
        struct[0].Gy_ini[148,147] = -v_R1_b_r + v_R1_n_r
        struct[0].Gy_ini[149,4] = -i_R1_c_i
        struct[0].Gy_ini[149,5] = i_R1_c_r
        struct[0].Gy_ini[149,6] = i_R1_c_i
        struct[0].Gy_ini[149,7] = -i_R1_c_r
        struct[0].Gy_ini[149,148] = v_R1_c_i - v_R1_n_i
        struct[0].Gy_ini[149,149] = -v_R1_c_r + v_R1_n_r
        struct[0].Gy_ini[150,144] = 1
        struct[0].Gy_ini[150,146] = 1
        struct[0].Gy_ini[150,148] = 1
        struct[0].Gy_ini[150,150] = 1
        struct[0].Gy_ini[151,145] = 1
        struct[0].Gy_ini[151,147] = 1
        struct[0].Gy_ini[151,149] = 1
        struct[0].Gy_ini[151,151] = 1
        struct[0].Gy_ini[152,8] = i_R11_a_r
        struct[0].Gy_ini[152,9] = i_R11_a_i
        struct[0].Gy_ini[152,14] = -i_R11_a_r
        struct[0].Gy_ini[152,15] = -i_R11_a_i
        struct[0].Gy_ini[152,152] = v_R11_a_r - v_R11_n_r
        struct[0].Gy_ini[152,153] = v_R11_a_i - v_R11_n_i
        struct[0].Gy_ini[153,10] = i_R11_b_r
        struct[0].Gy_ini[153,11] = i_R11_b_i
        struct[0].Gy_ini[153,14] = -i_R11_b_r
        struct[0].Gy_ini[153,15] = -i_R11_b_i
        struct[0].Gy_ini[153,154] = v_R11_b_r - v_R11_n_r
        struct[0].Gy_ini[153,155] = v_R11_b_i - v_R11_n_i
        struct[0].Gy_ini[154,12] = i_R11_c_r
        struct[0].Gy_ini[154,13] = i_R11_c_i
        struct[0].Gy_ini[154,14] = -i_R11_c_r
        struct[0].Gy_ini[154,15] = -i_R11_c_i
        struct[0].Gy_ini[154,156] = v_R11_c_r - v_R11_n_r
        struct[0].Gy_ini[154,157] = v_R11_c_i - v_R11_n_i
        struct[0].Gy_ini[155,8] = -i_R11_a_i
        struct[0].Gy_ini[155,9] = i_R11_a_r
        struct[0].Gy_ini[155,14] = i_R11_a_i
        struct[0].Gy_ini[155,15] = -i_R11_a_r
        struct[0].Gy_ini[155,152] = v_R11_a_i - v_R11_n_i
        struct[0].Gy_ini[155,153] = -v_R11_a_r + v_R11_n_r
        struct[0].Gy_ini[156,10] = -i_R11_b_i
        struct[0].Gy_ini[156,11] = i_R11_b_r
        struct[0].Gy_ini[156,14] = i_R11_b_i
        struct[0].Gy_ini[156,15] = -i_R11_b_r
        struct[0].Gy_ini[156,154] = v_R11_b_i - v_R11_n_i
        struct[0].Gy_ini[156,155] = -v_R11_b_r + v_R11_n_r
        struct[0].Gy_ini[157,12] = -i_R11_c_i
        struct[0].Gy_ini[157,13] = i_R11_c_r
        struct[0].Gy_ini[157,14] = i_R11_c_i
        struct[0].Gy_ini[157,15] = -i_R11_c_r
        struct[0].Gy_ini[157,156] = v_R11_c_i - v_R11_n_i
        struct[0].Gy_ini[157,157] = -v_R11_c_r + v_R11_n_r
        struct[0].Gy_ini[158,152] = 1
        struct[0].Gy_ini[158,154] = 1
        struct[0].Gy_ini[158,156] = 1
        struct[0].Gy_ini[158,158] = 1
        struct[0].Gy_ini[159,153] = 1
        struct[0].Gy_ini[159,155] = 1
        struct[0].Gy_ini[159,157] = 1
        struct[0].Gy_ini[159,159] = 1
        struct[0].Gy_ini[160,16] = i_R15_a_r
        struct[0].Gy_ini[160,17] = i_R15_a_i
        struct[0].Gy_ini[160,22] = -i_R15_a_r
        struct[0].Gy_ini[160,23] = -i_R15_a_i
        struct[0].Gy_ini[160,160] = v_R15_a_r - v_R15_n_r
        struct[0].Gy_ini[160,161] = v_R15_a_i - v_R15_n_i
        struct[0].Gy_ini[161,18] = i_R15_b_r
        struct[0].Gy_ini[161,19] = i_R15_b_i
        struct[0].Gy_ini[161,22] = -i_R15_b_r
        struct[0].Gy_ini[161,23] = -i_R15_b_i
        struct[0].Gy_ini[161,162] = v_R15_b_r - v_R15_n_r
        struct[0].Gy_ini[161,163] = v_R15_b_i - v_R15_n_i
        struct[0].Gy_ini[162,20] = i_R15_c_r
        struct[0].Gy_ini[162,21] = i_R15_c_i
        struct[0].Gy_ini[162,22] = -i_R15_c_r
        struct[0].Gy_ini[162,23] = -i_R15_c_i
        struct[0].Gy_ini[162,164] = v_R15_c_r - v_R15_n_r
        struct[0].Gy_ini[162,165] = v_R15_c_i - v_R15_n_i
        struct[0].Gy_ini[163,16] = -i_R15_a_i
        struct[0].Gy_ini[163,17] = i_R15_a_r
        struct[0].Gy_ini[163,22] = i_R15_a_i
        struct[0].Gy_ini[163,23] = -i_R15_a_r
        struct[0].Gy_ini[163,160] = v_R15_a_i - v_R15_n_i
        struct[0].Gy_ini[163,161] = -v_R15_a_r + v_R15_n_r
        struct[0].Gy_ini[164,18] = -i_R15_b_i
        struct[0].Gy_ini[164,19] = i_R15_b_r
        struct[0].Gy_ini[164,22] = i_R15_b_i
        struct[0].Gy_ini[164,23] = -i_R15_b_r
        struct[0].Gy_ini[164,162] = v_R15_b_i - v_R15_n_i
        struct[0].Gy_ini[164,163] = -v_R15_b_r + v_R15_n_r
        struct[0].Gy_ini[165,20] = -i_R15_c_i
        struct[0].Gy_ini[165,21] = i_R15_c_r
        struct[0].Gy_ini[165,22] = i_R15_c_i
        struct[0].Gy_ini[165,23] = -i_R15_c_r
        struct[0].Gy_ini[165,164] = v_R15_c_i - v_R15_n_i
        struct[0].Gy_ini[165,165] = -v_R15_c_r + v_R15_n_r
        struct[0].Gy_ini[166,160] = 1
        struct[0].Gy_ini[166,162] = 1
        struct[0].Gy_ini[166,164] = 1
        struct[0].Gy_ini[166,166] = 1
        struct[0].Gy_ini[167,161] = 1
        struct[0].Gy_ini[167,163] = 1
        struct[0].Gy_ini[167,165] = 1
        struct[0].Gy_ini[167,167] = 1
        struct[0].Gy_ini[168,24] = i_R16_a_r
        struct[0].Gy_ini[168,25] = i_R16_a_i
        struct[0].Gy_ini[168,30] = -i_R16_a_r
        struct[0].Gy_ini[168,31] = -i_R16_a_i
        struct[0].Gy_ini[168,168] = v_R16_a_r - v_R16_n_r
        struct[0].Gy_ini[168,169] = v_R16_a_i - v_R16_n_i
        struct[0].Gy_ini[169,26] = i_R16_b_r
        struct[0].Gy_ini[169,27] = i_R16_b_i
        struct[0].Gy_ini[169,30] = -i_R16_b_r
        struct[0].Gy_ini[169,31] = -i_R16_b_i
        struct[0].Gy_ini[169,170] = v_R16_b_r - v_R16_n_r
        struct[0].Gy_ini[169,171] = v_R16_b_i - v_R16_n_i
        struct[0].Gy_ini[170,28] = i_R16_c_r
        struct[0].Gy_ini[170,29] = i_R16_c_i
        struct[0].Gy_ini[170,30] = -i_R16_c_r
        struct[0].Gy_ini[170,31] = -i_R16_c_i
        struct[0].Gy_ini[170,172] = v_R16_c_r - v_R16_n_r
        struct[0].Gy_ini[170,173] = v_R16_c_i - v_R16_n_i
        struct[0].Gy_ini[171,24] = -i_R16_a_i
        struct[0].Gy_ini[171,25] = i_R16_a_r
        struct[0].Gy_ini[171,30] = i_R16_a_i
        struct[0].Gy_ini[171,31] = -i_R16_a_r
        struct[0].Gy_ini[171,168] = v_R16_a_i - v_R16_n_i
        struct[0].Gy_ini[171,169] = -v_R16_a_r + v_R16_n_r
        struct[0].Gy_ini[172,26] = -i_R16_b_i
        struct[0].Gy_ini[172,27] = i_R16_b_r
        struct[0].Gy_ini[172,30] = i_R16_b_i
        struct[0].Gy_ini[172,31] = -i_R16_b_r
        struct[0].Gy_ini[172,170] = v_R16_b_i - v_R16_n_i
        struct[0].Gy_ini[172,171] = -v_R16_b_r + v_R16_n_r
        struct[0].Gy_ini[173,28] = -i_R16_c_i
        struct[0].Gy_ini[173,29] = i_R16_c_r
        struct[0].Gy_ini[173,30] = i_R16_c_i
        struct[0].Gy_ini[173,31] = -i_R16_c_r
        struct[0].Gy_ini[173,172] = v_R16_c_i - v_R16_n_i
        struct[0].Gy_ini[173,173] = -v_R16_c_r + v_R16_n_r
        struct[0].Gy_ini[174,168] = 1
        struct[0].Gy_ini[174,170] = 1
        struct[0].Gy_ini[174,172] = 1
        struct[0].Gy_ini[174,174] = 1
        struct[0].Gy_ini[175,169] = 1
        struct[0].Gy_ini[175,171] = 1
        struct[0].Gy_ini[175,173] = 1
        struct[0].Gy_ini[175,175] = 1
        struct[0].Gy_ini[176,32] = i_R17_a_r
        struct[0].Gy_ini[176,33] = i_R17_a_i
        struct[0].Gy_ini[176,38] = -i_R17_a_r
        struct[0].Gy_ini[176,39] = -i_R17_a_i
        struct[0].Gy_ini[176,176] = v_R17_a_r - v_R17_n_r
        struct[0].Gy_ini[176,177] = v_R17_a_i - v_R17_n_i
        struct[0].Gy_ini[177,34] = i_R17_b_r
        struct[0].Gy_ini[177,35] = i_R17_b_i
        struct[0].Gy_ini[177,38] = -i_R17_b_r
        struct[0].Gy_ini[177,39] = -i_R17_b_i
        struct[0].Gy_ini[177,178] = v_R17_b_r - v_R17_n_r
        struct[0].Gy_ini[177,179] = v_R17_b_i - v_R17_n_i
        struct[0].Gy_ini[178,36] = i_R17_c_r
        struct[0].Gy_ini[178,37] = i_R17_c_i
        struct[0].Gy_ini[178,38] = -i_R17_c_r
        struct[0].Gy_ini[178,39] = -i_R17_c_i
        struct[0].Gy_ini[178,180] = v_R17_c_r - v_R17_n_r
        struct[0].Gy_ini[178,181] = v_R17_c_i - v_R17_n_i
        struct[0].Gy_ini[179,32] = -i_R17_a_i
        struct[0].Gy_ini[179,33] = i_R17_a_r
        struct[0].Gy_ini[179,38] = i_R17_a_i
        struct[0].Gy_ini[179,39] = -i_R17_a_r
        struct[0].Gy_ini[179,176] = v_R17_a_i - v_R17_n_i
        struct[0].Gy_ini[179,177] = -v_R17_a_r + v_R17_n_r
        struct[0].Gy_ini[180,34] = -i_R17_b_i
        struct[0].Gy_ini[180,35] = i_R17_b_r
        struct[0].Gy_ini[180,38] = i_R17_b_i
        struct[0].Gy_ini[180,39] = -i_R17_b_r
        struct[0].Gy_ini[180,178] = v_R17_b_i - v_R17_n_i
        struct[0].Gy_ini[180,179] = -v_R17_b_r + v_R17_n_r
        struct[0].Gy_ini[181,36] = -i_R17_c_i
        struct[0].Gy_ini[181,37] = i_R17_c_r
        struct[0].Gy_ini[181,38] = i_R17_c_i
        struct[0].Gy_ini[181,39] = -i_R17_c_r
        struct[0].Gy_ini[181,180] = v_R17_c_i - v_R17_n_i
        struct[0].Gy_ini[181,181] = -v_R17_c_r + v_R17_n_r
        struct[0].Gy_ini[182,176] = 1
        struct[0].Gy_ini[182,178] = 1
        struct[0].Gy_ini[182,180] = 1
        struct[0].Gy_ini[182,182] = 1
        struct[0].Gy_ini[183,177] = 1
        struct[0].Gy_ini[183,179] = 1
        struct[0].Gy_ini[183,181] = 1
        struct[0].Gy_ini[183,183] = 1
        struct[0].Gy_ini[184,40] = i_R18_a_r
        struct[0].Gy_ini[184,41] = i_R18_a_i
        struct[0].Gy_ini[184,46] = -i_R18_a_r
        struct[0].Gy_ini[184,47] = -i_R18_a_i
        struct[0].Gy_ini[184,184] = v_R18_a_r - v_R18_n_r
        struct[0].Gy_ini[184,185] = v_R18_a_i - v_R18_n_i
        struct[0].Gy_ini[185,42] = i_R18_b_r
        struct[0].Gy_ini[185,43] = i_R18_b_i
        struct[0].Gy_ini[185,46] = -i_R18_b_r
        struct[0].Gy_ini[185,47] = -i_R18_b_i
        struct[0].Gy_ini[185,186] = v_R18_b_r - v_R18_n_r
        struct[0].Gy_ini[185,187] = v_R18_b_i - v_R18_n_i
        struct[0].Gy_ini[186,44] = i_R18_c_r
        struct[0].Gy_ini[186,45] = i_R18_c_i
        struct[0].Gy_ini[186,46] = -i_R18_c_r
        struct[0].Gy_ini[186,47] = -i_R18_c_i
        struct[0].Gy_ini[186,188] = v_R18_c_r - v_R18_n_r
        struct[0].Gy_ini[186,189] = v_R18_c_i - v_R18_n_i
        struct[0].Gy_ini[187,40] = -i_R18_a_i
        struct[0].Gy_ini[187,41] = i_R18_a_r
        struct[0].Gy_ini[187,46] = i_R18_a_i
        struct[0].Gy_ini[187,47] = -i_R18_a_r
        struct[0].Gy_ini[187,184] = v_R18_a_i - v_R18_n_i
        struct[0].Gy_ini[187,185] = -v_R18_a_r + v_R18_n_r
        struct[0].Gy_ini[188,42] = -i_R18_b_i
        struct[0].Gy_ini[188,43] = i_R18_b_r
        struct[0].Gy_ini[188,46] = i_R18_b_i
        struct[0].Gy_ini[188,47] = -i_R18_b_r
        struct[0].Gy_ini[188,186] = v_R18_b_i - v_R18_n_i
        struct[0].Gy_ini[188,187] = -v_R18_b_r + v_R18_n_r
        struct[0].Gy_ini[189,44] = -i_R18_c_i
        struct[0].Gy_ini[189,45] = i_R18_c_r
        struct[0].Gy_ini[189,46] = i_R18_c_i
        struct[0].Gy_ini[189,47] = -i_R18_c_r
        struct[0].Gy_ini[189,188] = v_R18_c_i - v_R18_n_i
        struct[0].Gy_ini[189,189] = -v_R18_c_r + v_R18_n_r
        struct[0].Gy_ini[190,184] = 1
        struct[0].Gy_ini[190,186] = 1
        struct[0].Gy_ini[190,188] = 1
        struct[0].Gy_ini[190,190] = 1
        struct[0].Gy_ini[191,185] = 1
        struct[0].Gy_ini[191,187] = 1
        struct[0].Gy_ini[191,189] = 1
        struct[0].Gy_ini[191,191] = 1



def run_nn(t,struct,mode):

    # Parameters:
    
    # Inputs:
    v_R0_a_r = struct[0].v_R0_a_r
    v_R0_a_i = struct[0].v_R0_a_i
    v_R0_b_r = struct[0].v_R0_b_r
    v_R0_b_i = struct[0].v_R0_b_i
    v_R0_c_r = struct[0].v_R0_c_r
    v_R0_c_i = struct[0].v_R0_c_i
    i_R1_n_r = struct[0].i_R1_n_r
    i_R1_n_i = struct[0].i_R1_n_i
    i_R11_n_r = struct[0].i_R11_n_r
    i_R11_n_i = struct[0].i_R11_n_i
    i_R15_n_r = struct[0].i_R15_n_r
    i_R15_n_i = struct[0].i_R15_n_i
    i_R16_n_r = struct[0].i_R16_n_r
    i_R16_n_i = struct[0].i_R16_n_i
    i_R17_n_r = struct[0].i_R17_n_r
    i_R17_n_i = struct[0].i_R17_n_i
    i_R18_n_r = struct[0].i_R18_n_r
    i_R18_n_i = struct[0].i_R18_n_i
    i_R2_a_r = struct[0].i_R2_a_r
    i_R2_a_i = struct[0].i_R2_a_i
    i_R2_b_r = struct[0].i_R2_b_r
    i_R2_b_i = struct[0].i_R2_b_i
    i_R2_c_r = struct[0].i_R2_c_r
    i_R2_c_i = struct[0].i_R2_c_i
    i_R2_n_r = struct[0].i_R2_n_r
    i_R2_n_i = struct[0].i_R2_n_i
    i_R3_a_r = struct[0].i_R3_a_r
    i_R3_a_i = struct[0].i_R3_a_i
    i_R3_b_r = struct[0].i_R3_b_r
    i_R3_b_i = struct[0].i_R3_b_i
    i_R3_c_r = struct[0].i_R3_c_r
    i_R3_c_i = struct[0].i_R3_c_i
    i_R3_n_r = struct[0].i_R3_n_r
    i_R3_n_i = struct[0].i_R3_n_i
    i_R4_a_r = struct[0].i_R4_a_r
    i_R4_a_i = struct[0].i_R4_a_i
    i_R4_b_r = struct[0].i_R4_b_r
    i_R4_b_i = struct[0].i_R4_b_i
    i_R4_c_r = struct[0].i_R4_c_r
    i_R4_c_i = struct[0].i_R4_c_i
    i_R4_n_r = struct[0].i_R4_n_r
    i_R4_n_i = struct[0].i_R4_n_i
    i_R5_a_r = struct[0].i_R5_a_r
    i_R5_a_i = struct[0].i_R5_a_i
    i_R5_b_r = struct[0].i_R5_b_r
    i_R5_b_i = struct[0].i_R5_b_i
    i_R5_c_r = struct[0].i_R5_c_r
    i_R5_c_i = struct[0].i_R5_c_i
    i_R5_n_r = struct[0].i_R5_n_r
    i_R5_n_i = struct[0].i_R5_n_i
    i_R6_a_r = struct[0].i_R6_a_r
    i_R6_a_i = struct[0].i_R6_a_i
    i_R6_b_r = struct[0].i_R6_b_r
    i_R6_b_i = struct[0].i_R6_b_i
    i_R6_c_r = struct[0].i_R6_c_r
    i_R6_c_i = struct[0].i_R6_c_i
    i_R6_n_r = struct[0].i_R6_n_r
    i_R6_n_i = struct[0].i_R6_n_i
    i_R7_a_r = struct[0].i_R7_a_r
    i_R7_a_i = struct[0].i_R7_a_i
    i_R7_b_r = struct[0].i_R7_b_r
    i_R7_b_i = struct[0].i_R7_b_i
    i_R7_c_r = struct[0].i_R7_c_r
    i_R7_c_i = struct[0].i_R7_c_i
    i_R7_n_r = struct[0].i_R7_n_r
    i_R7_n_i = struct[0].i_R7_n_i
    i_R8_a_r = struct[0].i_R8_a_r
    i_R8_a_i = struct[0].i_R8_a_i
    i_R8_b_r = struct[0].i_R8_b_r
    i_R8_b_i = struct[0].i_R8_b_i
    i_R8_c_r = struct[0].i_R8_c_r
    i_R8_c_i = struct[0].i_R8_c_i
    i_R8_n_r = struct[0].i_R8_n_r
    i_R8_n_i = struct[0].i_R8_n_i
    i_R9_a_r = struct[0].i_R9_a_r
    i_R9_a_i = struct[0].i_R9_a_i
    i_R9_b_r = struct[0].i_R9_b_r
    i_R9_b_i = struct[0].i_R9_b_i
    i_R9_c_r = struct[0].i_R9_c_r
    i_R9_c_i = struct[0].i_R9_c_i
    i_R9_n_r = struct[0].i_R9_n_r
    i_R9_n_i = struct[0].i_R9_n_i
    i_R10_a_r = struct[0].i_R10_a_r
    i_R10_a_i = struct[0].i_R10_a_i
    i_R10_b_r = struct[0].i_R10_b_r
    i_R10_b_i = struct[0].i_R10_b_i
    i_R10_c_r = struct[0].i_R10_c_r
    i_R10_c_i = struct[0].i_R10_c_i
    i_R10_n_r = struct[0].i_R10_n_r
    i_R10_n_i = struct[0].i_R10_n_i
    i_R12_a_r = struct[0].i_R12_a_r
    i_R12_a_i = struct[0].i_R12_a_i
    i_R12_b_r = struct[0].i_R12_b_r
    i_R12_b_i = struct[0].i_R12_b_i
    i_R12_c_r = struct[0].i_R12_c_r
    i_R12_c_i = struct[0].i_R12_c_i
    i_R12_n_r = struct[0].i_R12_n_r
    i_R12_n_i = struct[0].i_R12_n_i
    i_R13_a_r = struct[0].i_R13_a_r
    i_R13_a_i = struct[0].i_R13_a_i
    i_R13_b_r = struct[0].i_R13_b_r
    i_R13_b_i = struct[0].i_R13_b_i
    i_R13_c_r = struct[0].i_R13_c_r
    i_R13_c_i = struct[0].i_R13_c_i
    i_R13_n_r = struct[0].i_R13_n_r
    i_R13_n_i = struct[0].i_R13_n_i
    i_R14_a_r = struct[0].i_R14_a_r
    i_R14_a_i = struct[0].i_R14_a_i
    i_R14_b_r = struct[0].i_R14_b_r
    i_R14_b_i = struct[0].i_R14_b_i
    i_R14_c_r = struct[0].i_R14_c_r
    i_R14_c_i = struct[0].i_R14_c_i
    i_R14_n_r = struct[0].i_R14_n_r
    i_R14_n_i = struct[0].i_R14_n_i
    p_R1_a = struct[0].p_R1_a
    q_R1_a = struct[0].q_R1_a
    p_R1_b = struct[0].p_R1_b
    q_R1_b = struct[0].q_R1_b
    p_R1_c = struct[0].p_R1_c
    q_R1_c = struct[0].q_R1_c
    p_R11_a = struct[0].p_R11_a
    q_R11_a = struct[0].q_R11_a
    p_R11_b = struct[0].p_R11_b
    q_R11_b = struct[0].q_R11_b
    p_R11_c = struct[0].p_R11_c
    q_R11_c = struct[0].q_R11_c
    p_R15_a = struct[0].p_R15_a
    q_R15_a = struct[0].q_R15_a
    p_R15_b = struct[0].p_R15_b
    q_R15_b = struct[0].q_R15_b
    p_R15_c = struct[0].p_R15_c
    q_R15_c = struct[0].q_R15_c
    p_R16_a = struct[0].p_R16_a
    q_R16_a = struct[0].q_R16_a
    p_R16_b = struct[0].p_R16_b
    q_R16_b = struct[0].q_R16_b
    p_R16_c = struct[0].p_R16_c
    q_R16_c = struct[0].q_R16_c
    p_R17_a = struct[0].p_R17_a
    q_R17_a = struct[0].q_R17_a
    p_R17_b = struct[0].p_R17_b
    q_R17_b = struct[0].q_R17_b
    p_R17_c = struct[0].p_R17_c
    q_R17_c = struct[0].q_R17_c
    p_R18_a = struct[0].p_R18_a
    q_R18_a = struct[0].q_R18_a
    p_R18_b = struct[0].p_R18_b
    q_R18_b = struct[0].q_R18_b
    p_R18_c = struct[0].p_R18_c
    q_R18_c = struct[0].q_R18_c
    u_dummy = struct[0].u_dummy
    
    # Dynamical states:
    x_dummy = struct[0].x[0,0]
    
    # Algebraic states:
    v_R1_a_r = struct[0].y_run[0,0]
    v_R1_a_i = struct[0].y_run[1,0]
    v_R1_b_r = struct[0].y_run[2,0]
    v_R1_b_i = struct[0].y_run[3,0]
    v_R1_c_r = struct[0].y_run[4,0]
    v_R1_c_i = struct[0].y_run[5,0]
    v_R1_n_r = struct[0].y_run[6,0]
    v_R1_n_i = struct[0].y_run[7,0]
    v_R11_a_r = struct[0].y_run[8,0]
    v_R11_a_i = struct[0].y_run[9,0]
    v_R11_b_r = struct[0].y_run[10,0]
    v_R11_b_i = struct[0].y_run[11,0]
    v_R11_c_r = struct[0].y_run[12,0]
    v_R11_c_i = struct[0].y_run[13,0]
    v_R11_n_r = struct[0].y_run[14,0]
    v_R11_n_i = struct[0].y_run[15,0]
    v_R15_a_r = struct[0].y_run[16,0]
    v_R15_a_i = struct[0].y_run[17,0]
    v_R15_b_r = struct[0].y_run[18,0]
    v_R15_b_i = struct[0].y_run[19,0]
    v_R15_c_r = struct[0].y_run[20,0]
    v_R15_c_i = struct[0].y_run[21,0]
    v_R15_n_r = struct[0].y_run[22,0]
    v_R15_n_i = struct[0].y_run[23,0]
    v_R16_a_r = struct[0].y_run[24,0]
    v_R16_a_i = struct[0].y_run[25,0]
    v_R16_b_r = struct[0].y_run[26,0]
    v_R16_b_i = struct[0].y_run[27,0]
    v_R16_c_r = struct[0].y_run[28,0]
    v_R16_c_i = struct[0].y_run[29,0]
    v_R16_n_r = struct[0].y_run[30,0]
    v_R16_n_i = struct[0].y_run[31,0]
    v_R17_a_r = struct[0].y_run[32,0]
    v_R17_a_i = struct[0].y_run[33,0]
    v_R17_b_r = struct[0].y_run[34,0]
    v_R17_b_i = struct[0].y_run[35,0]
    v_R17_c_r = struct[0].y_run[36,0]
    v_R17_c_i = struct[0].y_run[37,0]
    v_R17_n_r = struct[0].y_run[38,0]
    v_R17_n_i = struct[0].y_run[39,0]
    v_R18_a_r = struct[0].y_run[40,0]
    v_R18_a_i = struct[0].y_run[41,0]
    v_R18_b_r = struct[0].y_run[42,0]
    v_R18_b_i = struct[0].y_run[43,0]
    v_R18_c_r = struct[0].y_run[44,0]
    v_R18_c_i = struct[0].y_run[45,0]
    v_R18_n_r = struct[0].y_run[46,0]
    v_R18_n_i = struct[0].y_run[47,0]
    v_R2_a_r = struct[0].y_run[48,0]
    v_R2_a_i = struct[0].y_run[49,0]
    v_R2_b_r = struct[0].y_run[50,0]
    v_R2_b_i = struct[0].y_run[51,0]
    v_R2_c_r = struct[0].y_run[52,0]
    v_R2_c_i = struct[0].y_run[53,0]
    v_R2_n_r = struct[0].y_run[54,0]
    v_R2_n_i = struct[0].y_run[55,0]
    v_R3_a_r = struct[0].y_run[56,0]
    v_R3_a_i = struct[0].y_run[57,0]
    v_R3_b_r = struct[0].y_run[58,0]
    v_R3_b_i = struct[0].y_run[59,0]
    v_R3_c_r = struct[0].y_run[60,0]
    v_R3_c_i = struct[0].y_run[61,0]
    v_R3_n_r = struct[0].y_run[62,0]
    v_R3_n_i = struct[0].y_run[63,0]
    v_R4_a_r = struct[0].y_run[64,0]
    v_R4_a_i = struct[0].y_run[65,0]
    v_R4_b_r = struct[0].y_run[66,0]
    v_R4_b_i = struct[0].y_run[67,0]
    v_R4_c_r = struct[0].y_run[68,0]
    v_R4_c_i = struct[0].y_run[69,0]
    v_R4_n_r = struct[0].y_run[70,0]
    v_R4_n_i = struct[0].y_run[71,0]
    v_R5_a_r = struct[0].y_run[72,0]
    v_R5_a_i = struct[0].y_run[73,0]
    v_R5_b_r = struct[0].y_run[74,0]
    v_R5_b_i = struct[0].y_run[75,0]
    v_R5_c_r = struct[0].y_run[76,0]
    v_R5_c_i = struct[0].y_run[77,0]
    v_R5_n_r = struct[0].y_run[78,0]
    v_R5_n_i = struct[0].y_run[79,0]
    v_R6_a_r = struct[0].y_run[80,0]
    v_R6_a_i = struct[0].y_run[81,0]
    v_R6_b_r = struct[0].y_run[82,0]
    v_R6_b_i = struct[0].y_run[83,0]
    v_R6_c_r = struct[0].y_run[84,0]
    v_R6_c_i = struct[0].y_run[85,0]
    v_R6_n_r = struct[0].y_run[86,0]
    v_R6_n_i = struct[0].y_run[87,0]
    v_R7_a_r = struct[0].y_run[88,0]
    v_R7_a_i = struct[0].y_run[89,0]
    v_R7_b_r = struct[0].y_run[90,0]
    v_R7_b_i = struct[0].y_run[91,0]
    v_R7_c_r = struct[0].y_run[92,0]
    v_R7_c_i = struct[0].y_run[93,0]
    v_R7_n_r = struct[0].y_run[94,0]
    v_R7_n_i = struct[0].y_run[95,0]
    v_R8_a_r = struct[0].y_run[96,0]
    v_R8_a_i = struct[0].y_run[97,0]
    v_R8_b_r = struct[0].y_run[98,0]
    v_R8_b_i = struct[0].y_run[99,0]
    v_R8_c_r = struct[0].y_run[100,0]
    v_R8_c_i = struct[0].y_run[101,0]
    v_R8_n_r = struct[0].y_run[102,0]
    v_R8_n_i = struct[0].y_run[103,0]
    v_R9_a_r = struct[0].y_run[104,0]
    v_R9_a_i = struct[0].y_run[105,0]
    v_R9_b_r = struct[0].y_run[106,0]
    v_R9_b_i = struct[0].y_run[107,0]
    v_R9_c_r = struct[0].y_run[108,0]
    v_R9_c_i = struct[0].y_run[109,0]
    v_R9_n_r = struct[0].y_run[110,0]
    v_R9_n_i = struct[0].y_run[111,0]
    v_R10_a_r = struct[0].y_run[112,0]
    v_R10_a_i = struct[0].y_run[113,0]
    v_R10_b_r = struct[0].y_run[114,0]
    v_R10_b_i = struct[0].y_run[115,0]
    v_R10_c_r = struct[0].y_run[116,0]
    v_R10_c_i = struct[0].y_run[117,0]
    v_R10_n_r = struct[0].y_run[118,0]
    v_R10_n_i = struct[0].y_run[119,0]
    v_R12_a_r = struct[0].y_run[120,0]
    v_R12_a_i = struct[0].y_run[121,0]
    v_R12_b_r = struct[0].y_run[122,0]
    v_R12_b_i = struct[0].y_run[123,0]
    v_R12_c_r = struct[0].y_run[124,0]
    v_R12_c_i = struct[0].y_run[125,0]
    v_R12_n_r = struct[0].y_run[126,0]
    v_R12_n_i = struct[0].y_run[127,0]
    v_R13_a_r = struct[0].y_run[128,0]
    v_R13_a_i = struct[0].y_run[129,0]
    v_R13_b_r = struct[0].y_run[130,0]
    v_R13_b_i = struct[0].y_run[131,0]
    v_R13_c_r = struct[0].y_run[132,0]
    v_R13_c_i = struct[0].y_run[133,0]
    v_R13_n_r = struct[0].y_run[134,0]
    v_R13_n_i = struct[0].y_run[135,0]
    v_R14_a_r = struct[0].y_run[136,0]
    v_R14_a_i = struct[0].y_run[137,0]
    v_R14_b_r = struct[0].y_run[138,0]
    v_R14_b_i = struct[0].y_run[139,0]
    v_R14_c_r = struct[0].y_run[140,0]
    v_R14_c_i = struct[0].y_run[141,0]
    v_R14_n_r = struct[0].y_run[142,0]
    v_R14_n_i = struct[0].y_run[143,0]
    i_R1_a_r = struct[0].y_run[144,0]
    i_R1_a_i = struct[0].y_run[145,0]
    i_R1_b_r = struct[0].y_run[146,0]
    i_R1_b_i = struct[0].y_run[147,0]
    i_R1_c_r = struct[0].y_run[148,0]
    i_R1_c_i = struct[0].y_run[149,0]
    i_R1_n_r = struct[0].y_run[150,0]
    i_R1_n_i = struct[0].y_run[151,0]
    i_R11_a_r = struct[0].y_run[152,0]
    i_R11_a_i = struct[0].y_run[153,0]
    i_R11_b_r = struct[0].y_run[154,0]
    i_R11_b_i = struct[0].y_run[155,0]
    i_R11_c_r = struct[0].y_run[156,0]
    i_R11_c_i = struct[0].y_run[157,0]
    i_R11_n_r = struct[0].y_run[158,0]
    i_R11_n_i = struct[0].y_run[159,0]
    i_R15_a_r = struct[0].y_run[160,0]
    i_R15_a_i = struct[0].y_run[161,0]
    i_R15_b_r = struct[0].y_run[162,0]
    i_R15_b_i = struct[0].y_run[163,0]
    i_R15_c_r = struct[0].y_run[164,0]
    i_R15_c_i = struct[0].y_run[165,0]
    i_R15_n_r = struct[0].y_run[166,0]
    i_R15_n_i = struct[0].y_run[167,0]
    i_R16_a_r = struct[0].y_run[168,0]
    i_R16_a_i = struct[0].y_run[169,0]
    i_R16_b_r = struct[0].y_run[170,0]
    i_R16_b_i = struct[0].y_run[171,0]
    i_R16_c_r = struct[0].y_run[172,0]
    i_R16_c_i = struct[0].y_run[173,0]
    i_R16_n_r = struct[0].y_run[174,0]
    i_R16_n_i = struct[0].y_run[175,0]
    i_R17_a_r = struct[0].y_run[176,0]
    i_R17_a_i = struct[0].y_run[177,0]
    i_R17_b_r = struct[0].y_run[178,0]
    i_R17_b_i = struct[0].y_run[179,0]
    i_R17_c_r = struct[0].y_run[180,0]
    i_R17_c_i = struct[0].y_run[181,0]
    i_R17_n_r = struct[0].y_run[182,0]
    i_R17_n_i = struct[0].y_run[183,0]
    i_R18_a_r = struct[0].y_run[184,0]
    i_R18_a_i = struct[0].y_run[185,0]
    i_R18_b_r = struct[0].y_run[186,0]
    i_R18_b_i = struct[0].y_run[187,0]
    i_R18_c_r = struct[0].y_run[188,0]
    i_R18_c_i = struct[0].y_run[189,0]
    i_R18_n_r = struct[0].y_run[190,0]
    i_R18_n_i = struct[0].y_run[191,0]
    
    # Differential equations:
    if mode == 2:


        struct[0].f[0,0] = u_dummy - x_dummy
    
    # Algebraic equations:
    if mode == 3:


        struct[0].g[0,0] = i_R1_a_r + 0.849044513514155*v_R0_a_i + 0.212261128378539*v_R0_a_r - 0.849044513514155*v_R0_c_i - 0.212261128378539*v_R0_c_r - 127.595184532966*v_R1_a_i - 123.954122254357*v_R1_a_r + 10.2713736253513*v_R1_b_i + 39.6392229058202*v_R1_b_r + 23.284964480954*v_R1_c_i + 24.9575997948692*v_R1_c_r + 83.8007853900572*v_R1_n_i + 58.0215758469966*v_R1_n_r + 54.0657727682604*v_R2_a_i + 105.57176931318*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r - 23.284964480954*v_R2_c_i - 24.9575997948692*v_R2_c_r - 10.2713736253513*v_R2_n_i - 39.6392229058202*v_R2_n_r
        struct[0].g[1,0] = i_R1_a_i + 0.212261128378539*v_R0_a_i - 0.849044513514155*v_R0_a_r - 0.212261128378539*v_R0_c_i + 0.849044513514155*v_R0_c_r - 123.954122254357*v_R1_a_i + 127.595184532966*v_R1_a_r + 39.6392229058202*v_R1_b_i - 10.2713736253513*v_R1_b_r + 24.9575997948692*v_R1_c_i - 23.284964480954*v_R1_c_r + 58.0215758469966*v_R1_n_i - 83.8007853900572*v_R1_n_r + 105.57176931318*v_R2_a_i - 54.0657727682604*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r - 24.9575997948692*v_R2_c_i + 23.284964480954*v_R2_c_r - 39.6392229058202*v_R2_n_i + 10.2713736253513*v_R2_n_r
        struct[0].g[2,0] = i_R1_b_r - 0.849044513514155*v_R0_a_i - 0.212261128378539*v_R0_a_r + 0.849044513514155*v_R0_b_i + 0.212261128378539*v_R0_b_r + 10.2713736253513*v_R1_a_i + 39.6392229058202*v_R1_a_r - 127.595184532966*v_R1_b_i - 123.954122254357*v_R1_b_r + 10.2713736253513*v_R1_c_i + 39.6392229058201*v_R1_c_r + 96.8143762456598*v_R1_n_i + 43.3399527360457*v_R1_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r + 54.0657727682604*v_R2_b_i + 105.57176931318*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058201*v_R2_c_r - 23.284964480954*v_R2_n_i - 24.9575997948692*v_R2_n_r
        struct[0].g[3,0] = i_R1_b_i - 0.212261128378539*v_R0_a_i + 0.849044513514155*v_R0_a_r + 0.212261128378539*v_R0_b_i - 0.849044513514155*v_R0_b_r + 39.6392229058202*v_R1_a_i - 10.2713736253513*v_R1_a_r - 123.954122254357*v_R1_b_i + 127.595184532966*v_R1_b_r + 39.6392229058201*v_R1_c_i - 10.2713736253513*v_R1_c_r + 43.3399527360457*v_R1_n_i - 96.8143762456598*v_R1_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r + 105.57176931318*v_R2_b_i - 54.0657727682604*v_R2_b_r - 39.6392229058201*v_R2_c_i + 10.2713736253513*v_R2_c_r - 24.9575997948692*v_R2_n_i + 23.284964480954*v_R2_n_r
        struct[0].g[4,0] = i_R1_c_r - 0.849044513514155*v_R0_b_i - 0.212261128378539*v_R0_b_r + 0.849044513514155*v_R0_c_i + 0.212261128378539*v_R0_c_r + 23.284964480954*v_R1_a_i + 24.9575997948692*v_R1_a_r + 10.2713736253513*v_R1_b_i + 39.6392229058202*v_R1_b_r - 127.595184532966*v_R1_c_i - 123.954122254357*v_R1_c_r + 83.8007853900572*v_R1_n_i + 58.0215758469966*v_R1_n_r - 23.284964480954*v_R2_a_i - 24.9575997948692*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r + 54.0657727682604*v_R2_c_i + 105.57176931318*v_R2_c_r - 10.2713736253514*v_R2_n_i - 39.6392229058201*v_R2_n_r
        struct[0].g[5,0] = i_R1_c_i - 0.212261128378539*v_R0_b_i + 0.849044513514155*v_R0_b_r + 0.212261128378539*v_R0_c_i - 0.849044513514155*v_R0_c_r + 24.9575997948692*v_R1_a_i - 23.284964480954*v_R1_a_r + 39.6392229058202*v_R1_b_i - 10.2713736253513*v_R1_b_r - 123.954122254357*v_R1_c_i + 127.595184532966*v_R1_c_r + 58.0215758469966*v_R1_n_i - 83.8007853900572*v_R1_n_r - 24.9575997948692*v_R2_a_i + 23.284964480954*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r + 105.57176931318*v_R2_c_i - 54.0657727682604*v_R2_c_r - 39.6392229058201*v_R2_n_i + 10.2713736253514*v_R2_n_r
        struct[0].g[6,0] = i_R1_n_r + 83.8007853900572*v_R1_a_i + 58.0215758469966*v_R1_a_r + 96.8143762456598*v_R1_b_i + 43.3399527360457*v_R1_b_r + 83.8007853900572*v_R1_c_i + 58.0215758469966*v_R1_c_r - 274.654008062378*v_R1_n_i - 161.052161470043*v_R1_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r - 23.284964480954*v_R2_b_i - 24.9575997948692*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058202*v_R2_c_r + 54.0657727682604*v_R2_n_i + 105.57176931318*v_R2_n_r
        struct[0].g[7,0] = i_R1_n_i + 58.0215758469966*v_R1_a_i - 83.8007853900572*v_R1_a_r + 43.3399527360457*v_R1_b_i - 96.8143762456598*v_R1_b_r + 58.0215758469966*v_R1_c_i - 83.8007853900572*v_R1_c_r - 161.052161470043*v_R1_n_i + 274.654008062378*v_R1_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r - 24.9575997948692*v_R2_b_i + 23.284964480954*v_R2_b_r - 39.6392229058202*v_R2_c_i + 10.2713736253513*v_R2_c_r + 105.57176931318*v_R2_n_i - 54.0657727682604*v_R2_n_r
        struct[0].g[8,0] = i_R11_a_r - 4.84676915585801*v_R11_a_i - 26.5300693004806*v_R11_a_r - 1.58482814504483*v_R11_b_i + 7.8946133751477*v_R11_b_r - 0.67991845328531*v_R11_c_i + 7.72144347466701*v_R11_c_r - 1.58482814504483*v_R11_n_i + 7.89461337514771*v_R11_n_r + 4.84676915585801*v_R3_a_i + 26.5300693004806*v_R3_a_r + 1.58482814504483*v_R3_b_i - 7.8946133751477*v_R3_b_r + 0.67991845328531*v_R3_c_i - 7.72144347466701*v_R3_c_r + 1.58482814504483*v_R3_n_i - 7.89461337514771*v_R3_n_r
        struct[0].g[9,0] = i_R11_a_i - 26.5300693004806*v_R11_a_i + 4.84676915585801*v_R11_a_r + 7.8946133751477*v_R11_b_i + 1.58482814504483*v_R11_b_r + 7.72144347466701*v_R11_c_i + 0.67991845328531*v_R11_c_r + 7.89461337514771*v_R11_n_i + 1.58482814504483*v_R11_n_r + 26.5300693004806*v_R3_a_i - 4.84676915585801*v_R3_a_r - 7.8946133751477*v_R3_b_i - 1.58482814504483*v_R3_b_r - 7.72144347466701*v_R3_c_i - 0.67991845328531*v_R3_c_r - 7.89461337514771*v_R3_n_i - 1.58482814504483*v_R3_n_r
        struct[0].g[10,0] = i_R11_b_r - 1.58482814504483*v_R11_a_i + 7.89461337514771*v_R11_a_r - 4.84676915585801*v_R11_b_i - 26.5300693004806*v_R11_b_r - 1.58482814504483*v_R11_c_i + 7.89461337514771*v_R11_c_r - 0.679918453285308*v_R11_n_i + 7.72144347466701*v_R11_n_r + 1.58482814504483*v_R3_a_i - 7.89461337514771*v_R3_a_r + 4.84676915585801*v_R3_b_i + 26.5300693004806*v_R3_b_r + 1.58482814504483*v_R3_c_i - 7.89461337514771*v_R3_c_r + 0.679918453285308*v_R3_n_i - 7.72144347466701*v_R3_n_r
        struct[0].g[11,0] = i_R11_b_i + 7.89461337514771*v_R11_a_i + 1.58482814504483*v_R11_a_r - 26.5300693004806*v_R11_b_i + 4.84676915585801*v_R11_b_r + 7.89461337514771*v_R11_c_i + 1.58482814504483*v_R11_c_r + 7.72144347466701*v_R11_n_i + 0.679918453285308*v_R11_n_r - 7.89461337514771*v_R3_a_i - 1.58482814504483*v_R3_a_r + 26.5300693004806*v_R3_b_i - 4.84676915585801*v_R3_b_r - 7.89461337514771*v_R3_c_i - 1.58482814504483*v_R3_c_r - 7.72144347466701*v_R3_n_i - 0.679918453285308*v_R3_n_r
        struct[0].g[12,0] = i_R11_c_r - 0.67991845328531*v_R11_a_i + 7.72144347466701*v_R11_a_r - 1.58482814504483*v_R11_b_i + 7.8946133751477*v_R11_b_r - 4.84676915585801*v_R11_c_i - 26.5300693004806*v_R11_c_r - 1.58482814504483*v_R11_n_i + 7.89461337514771*v_R11_n_r + 0.67991845328531*v_R3_a_i - 7.72144347466701*v_R3_a_r + 1.58482814504483*v_R3_b_i - 7.8946133751477*v_R3_b_r + 4.84676915585801*v_R3_c_i + 26.5300693004806*v_R3_c_r + 1.58482814504483*v_R3_n_i - 7.89461337514771*v_R3_n_r
        struct[0].g[13,0] = i_R11_c_i + 7.72144347466701*v_R11_a_i + 0.67991845328531*v_R11_a_r + 7.8946133751477*v_R11_b_i + 1.58482814504483*v_R11_b_r - 26.5300693004806*v_R11_c_i + 4.84676915585801*v_R11_c_r + 7.89461337514771*v_R11_n_i + 1.58482814504483*v_R11_n_r - 7.72144347466701*v_R3_a_i - 0.67991845328531*v_R3_a_r - 7.8946133751477*v_R3_b_i - 1.58482814504483*v_R3_b_r + 26.5300693004806*v_R3_c_i - 4.84676915585801*v_R3_c_r - 7.89461337514771*v_R3_n_i - 1.58482814504483*v_R3_n_r
        struct[0].g[14,0] = i_R11_n_r - 1.58482814504483*v_R11_a_i + 7.89461337514771*v_R11_a_r - 0.679918453285308*v_R11_b_i + 7.72144347466701*v_R11_b_r - 1.58482814504483*v_R11_c_i + 7.89461337514771*v_R11_c_r - 4.84676915585801*v_R11_n_i - 26.5550693004806*v_R11_n_r + 1.58482814504483*v_R3_a_i - 7.89461337514771*v_R3_a_r + 0.679918453285308*v_R3_b_i - 7.72144347466701*v_R3_b_r + 1.58482814504483*v_R3_c_i - 7.89461337514771*v_R3_c_r + 4.84676915585801*v_R3_n_i + 26.5300693004806*v_R3_n_r
        struct[0].g[15,0] = i_R11_n_i + 7.89461337514771*v_R11_a_i + 1.58482814504483*v_R11_a_r + 7.72144347466701*v_R11_b_i + 0.679918453285308*v_R11_b_r + 7.89461337514771*v_R11_c_i + 1.58482814504483*v_R11_c_r - 26.5550693004806*v_R11_n_i + 4.84676915585801*v_R11_n_r - 7.89461337514771*v_R3_a_i - 1.58482814504483*v_R3_a_r - 7.72144347466701*v_R3_b_i - 0.679918453285308*v_R3_b_r - 7.89461337514771*v_R3_c_i - 1.58482814504483*v_R3_c_r + 26.5300693004806*v_R3_n_i - 4.84676915585801*v_R3_n_r
        struct[0].g[16,0] = i_R15_a_r + 4.84676915585801*v_R14_a_i + 26.5300693004806*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 0.67991845328531*v_R14_c_i - 7.72144347466701*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r - 4.84676915585801*v_R15_a_i - 26.5300693004806*v_R15_a_r - 1.58482814504483*v_R15_b_i + 7.8946133751477*v_R15_b_r - 0.67991845328531*v_R15_c_i + 7.72144347466701*v_R15_c_r - 1.58482814504483*v_R15_n_i + 7.89461337514771*v_R15_n_r
        struct[0].g[17,0] = i_R15_a_i + 26.5300693004806*v_R14_a_i - 4.84676915585801*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r - 7.72144347466701*v_R14_c_i - 0.67991845328531*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r - 26.5300693004806*v_R15_a_i + 4.84676915585801*v_R15_a_r + 7.8946133751477*v_R15_b_i + 1.58482814504483*v_R15_b_r + 7.72144347466701*v_R15_c_i + 0.67991845328531*v_R15_c_r + 7.89461337514771*v_R15_n_i + 1.58482814504483*v_R15_n_r
        struct[0].g[18,0] = i_R15_b_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 4.84676915585801*v_R14_b_i + 26.5300693004806*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 0.679918453285308*v_R14_n_i - 7.72144347466701*v_R14_n_r - 1.58482814504483*v_R15_a_i + 7.89461337514771*v_R15_a_r - 4.84676915585801*v_R15_b_i - 26.5300693004806*v_R15_b_r - 1.58482814504483*v_R15_c_i + 7.89461337514771*v_R15_c_r - 0.679918453285308*v_R15_n_i + 7.72144347466701*v_R15_n_r
        struct[0].g[19,0] = i_R15_b_i - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r + 26.5300693004806*v_R14_b_i - 4.84676915585801*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r - 7.72144347466701*v_R14_n_i - 0.679918453285308*v_R14_n_r + 7.89461337514771*v_R15_a_i + 1.58482814504483*v_R15_a_r - 26.5300693004806*v_R15_b_i + 4.84676915585801*v_R15_b_r + 7.89461337514771*v_R15_c_i + 1.58482814504483*v_R15_c_r + 7.72144347466701*v_R15_n_i + 0.679918453285308*v_R15_n_r
        struct[0].g[20,0] = i_R15_c_r + 0.67991845328531*v_R14_a_i - 7.72144347466701*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 4.84676915585801*v_R14_c_i + 26.5300693004806*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r - 0.67991845328531*v_R15_a_i + 7.72144347466701*v_R15_a_r - 1.58482814504483*v_R15_b_i + 7.8946133751477*v_R15_b_r - 4.84676915585801*v_R15_c_i - 26.5300693004806*v_R15_c_r - 1.58482814504483*v_R15_n_i + 7.89461337514771*v_R15_n_r
        struct[0].g[21,0] = i_R15_c_i - 7.72144347466701*v_R14_a_i - 0.67991845328531*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r + 26.5300693004806*v_R14_c_i - 4.84676915585801*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r + 7.72144347466701*v_R15_a_i + 0.67991845328531*v_R15_a_r + 7.8946133751477*v_R15_b_i + 1.58482814504483*v_R15_b_r - 26.5300693004806*v_R15_c_i + 4.84676915585801*v_R15_c_r + 7.89461337514771*v_R15_n_i + 1.58482814504483*v_R15_n_r
        struct[0].g[22,0] = i_R15_n_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 0.679918453285308*v_R14_b_i - 7.72144347466701*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 4.84676915585801*v_R14_n_i + 26.5300693004806*v_R14_n_r - 1.58482814504483*v_R15_a_i + 7.89461337514771*v_R15_a_r - 0.679918453285308*v_R15_b_i + 7.72144347466701*v_R15_b_r - 1.58482814504483*v_R15_c_i + 7.89461337514771*v_R15_c_r - 4.84676915585801*v_R15_n_i - 26.5550693004806*v_R15_n_r
        struct[0].g[23,0] = i_R15_n_i - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r - 7.72144347466701*v_R14_b_i - 0.679918453285308*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r + 26.5300693004806*v_R14_n_i - 4.84676915585801*v_R14_n_r + 7.89461337514771*v_R15_a_i + 1.58482814504483*v_R15_a_r + 7.72144347466701*v_R15_b_i + 0.679918453285308*v_R15_b_r + 7.89461337514771*v_R15_c_i + 1.58482814504483*v_R15_c_r - 26.5550693004806*v_R15_n_i + 4.84676915585801*v_R15_n_r
        struct[0].g[24,0] = i_R16_a_r - 5.65456401516768*v_R16_a_i - 30.9517475172273*v_R16_a_r - 1.84896616921897*v_R16_b_i + 9.21038227100566*v_R16_b_r - 0.793238195499529*v_R16_c_i + 9.00835072044485*v_R16_c_r - 1.84896616921897*v_R16_n_i + 9.21038227100566*v_R16_n_r + 5.65456401516768*v_R6_a_i + 30.9517475172273*v_R6_a_r + 1.84896616921897*v_R6_b_i - 9.21038227100566*v_R6_b_r + 0.793238195499529*v_R6_c_i - 9.00835072044485*v_R6_c_r + 1.84896616921897*v_R6_n_i - 9.21038227100566*v_R6_n_r
        struct[0].g[25,0] = i_R16_a_i - 30.9517475172273*v_R16_a_i + 5.65456401516768*v_R16_a_r + 9.21038227100566*v_R16_b_i + 1.84896616921897*v_R16_b_r + 9.00835072044485*v_R16_c_i + 0.793238195499529*v_R16_c_r + 9.21038227100566*v_R16_n_i + 1.84896616921897*v_R16_n_r + 30.9517475172273*v_R6_a_i - 5.65456401516768*v_R6_a_r - 9.21038227100566*v_R6_b_i - 1.84896616921897*v_R6_b_r - 9.00835072044485*v_R6_c_i - 0.793238195499529*v_R6_c_r - 9.21038227100566*v_R6_n_i - 1.84896616921897*v_R6_n_r
        struct[0].g[26,0] = i_R16_b_r - 1.84896616921897*v_R16_a_i + 9.21038227100566*v_R16_a_r - 5.65456401516768*v_R16_b_i - 30.9517475172273*v_R16_b_r - 1.84896616921897*v_R16_c_i + 9.21038227100566*v_R16_c_r - 0.793238195499528*v_R16_n_i + 9.00835072044485*v_R16_n_r + 1.84896616921897*v_R6_a_i - 9.21038227100566*v_R6_a_r + 5.65456401516768*v_R6_b_i + 30.9517475172273*v_R6_b_r + 1.84896616921897*v_R6_c_i - 9.21038227100566*v_R6_c_r + 0.793238195499528*v_R6_n_i - 9.00835072044485*v_R6_n_r
        struct[0].g[27,0] = i_R16_b_i + 9.21038227100566*v_R16_a_i + 1.84896616921897*v_R16_a_r - 30.9517475172273*v_R16_b_i + 5.65456401516768*v_R16_b_r + 9.21038227100566*v_R16_c_i + 1.84896616921897*v_R16_c_r + 9.00835072044485*v_R16_n_i + 0.793238195499528*v_R16_n_r - 9.21038227100566*v_R6_a_i - 1.84896616921897*v_R6_a_r + 30.9517475172273*v_R6_b_i - 5.65456401516768*v_R6_b_r - 9.21038227100566*v_R6_c_i - 1.84896616921897*v_R6_c_r - 9.00835072044485*v_R6_n_i - 0.793238195499528*v_R6_n_r
        struct[0].g[28,0] = i_R16_c_r - 0.793238195499527*v_R16_a_i + 9.00835072044484*v_R16_a_r - 1.84896616921897*v_R16_b_i + 9.21038227100566*v_R16_b_r - 5.65456401516768*v_R16_c_i - 30.9517475172273*v_R16_c_r - 1.84896616921897*v_R16_n_i + 9.21038227100566*v_R16_n_r + 0.793238195499527*v_R6_a_i - 9.00835072044484*v_R6_a_r + 1.84896616921897*v_R6_b_i - 9.21038227100566*v_R6_b_r + 5.65456401516768*v_R6_c_i + 30.9517475172273*v_R6_c_r + 1.84896616921897*v_R6_n_i - 9.21038227100566*v_R6_n_r
        struct[0].g[29,0] = i_R16_c_i + 9.00835072044484*v_R16_a_i + 0.793238195499527*v_R16_a_r + 9.21038227100566*v_R16_b_i + 1.84896616921897*v_R16_b_r - 30.9517475172273*v_R16_c_i + 5.65456401516768*v_R16_c_r + 9.21038227100566*v_R16_n_i + 1.84896616921897*v_R16_n_r - 9.00835072044484*v_R6_a_i - 0.793238195499527*v_R6_a_r - 9.21038227100566*v_R6_b_i - 1.84896616921897*v_R6_b_r + 30.9517475172273*v_R6_c_i - 5.65456401516768*v_R6_c_r - 9.21038227100566*v_R6_n_i - 1.84896616921897*v_R6_n_r
        struct[0].g[30,0] = i_R16_n_r - 1.84896616921897*v_R16_a_i + 9.21038227100566*v_R16_a_r - 0.793238195499527*v_R16_b_i + 9.00835072044485*v_R16_b_r - 1.84896616921897*v_R16_c_i + 9.21038227100566*v_R16_c_r - 5.65456401516768*v_R16_n_i - 30.9767475172273*v_R16_n_r + 1.84896616921897*v_R6_a_i - 9.21038227100566*v_R6_a_r + 0.793238195499527*v_R6_b_i - 9.00835072044485*v_R6_b_r + 1.84896616921897*v_R6_c_i - 9.21038227100566*v_R6_c_r + 5.65456401516768*v_R6_n_i + 30.9517475172273*v_R6_n_r
        struct[0].g[31,0] = i_R16_n_i + 9.21038227100566*v_R16_a_i + 1.84896616921897*v_R16_a_r + 9.00835072044485*v_R16_b_i + 0.793238195499527*v_R16_b_r + 9.21038227100566*v_R16_c_i + 1.84896616921897*v_R16_c_r - 30.9767475172273*v_R16_n_i + 5.65456401516768*v_R16_n_r - 9.21038227100566*v_R6_a_i - 1.84896616921897*v_R6_a_r - 9.00835072044485*v_R6_b_i - 0.793238195499527*v_R6_b_r - 9.21038227100566*v_R6_c_i - 1.84896616921897*v_R6_c_r + 30.9517475172273*v_R6_n_i - 5.65456401516768*v_R6_n_r
        struct[0].g[32,0] = i_R17_a_r - 5.65456401516768*v_R17_a_i - 30.9517475172273*v_R17_a_r - 1.84896616921897*v_R17_b_i + 9.21038227100566*v_R17_b_r - 0.793238195499529*v_R17_c_i + 9.00835072044485*v_R17_c_r - 1.84896616921897*v_R17_n_i + 9.21038227100566*v_R17_n_r + 5.65456401516768*v_R9_a_i + 30.9517475172273*v_R9_a_r + 1.84896616921897*v_R9_b_i - 9.21038227100566*v_R9_b_r + 0.793238195499529*v_R9_c_i - 9.00835072044485*v_R9_c_r + 1.84896616921897*v_R9_n_i - 9.21038227100566*v_R9_n_r
        struct[0].g[33,0] = i_R17_a_i - 30.9517475172273*v_R17_a_i + 5.65456401516768*v_R17_a_r + 9.21038227100566*v_R17_b_i + 1.84896616921897*v_R17_b_r + 9.00835072044485*v_R17_c_i + 0.793238195499529*v_R17_c_r + 9.21038227100566*v_R17_n_i + 1.84896616921897*v_R17_n_r + 30.9517475172273*v_R9_a_i - 5.65456401516768*v_R9_a_r - 9.21038227100566*v_R9_b_i - 1.84896616921897*v_R9_b_r - 9.00835072044485*v_R9_c_i - 0.793238195499529*v_R9_c_r - 9.21038227100566*v_R9_n_i - 1.84896616921897*v_R9_n_r
        struct[0].g[34,0] = i_R17_b_r - 1.84896616921897*v_R17_a_i + 9.21038227100566*v_R17_a_r - 5.65456401516768*v_R17_b_i - 30.9517475172273*v_R17_b_r - 1.84896616921897*v_R17_c_i + 9.21038227100566*v_R17_c_r - 0.793238195499528*v_R17_n_i + 9.00835072044485*v_R17_n_r + 1.84896616921897*v_R9_a_i - 9.21038227100566*v_R9_a_r + 5.65456401516768*v_R9_b_i + 30.9517475172273*v_R9_b_r + 1.84896616921897*v_R9_c_i - 9.21038227100566*v_R9_c_r + 0.793238195499528*v_R9_n_i - 9.00835072044485*v_R9_n_r
        struct[0].g[35,0] = i_R17_b_i + 9.21038227100566*v_R17_a_i + 1.84896616921897*v_R17_a_r - 30.9517475172273*v_R17_b_i + 5.65456401516768*v_R17_b_r + 9.21038227100566*v_R17_c_i + 1.84896616921897*v_R17_c_r + 9.00835072044485*v_R17_n_i + 0.793238195499528*v_R17_n_r - 9.21038227100566*v_R9_a_i - 1.84896616921897*v_R9_a_r + 30.9517475172273*v_R9_b_i - 5.65456401516768*v_R9_b_r - 9.21038227100566*v_R9_c_i - 1.84896616921897*v_R9_c_r - 9.00835072044485*v_R9_n_i - 0.793238195499528*v_R9_n_r
        struct[0].g[36,0] = i_R17_c_r - 0.793238195499527*v_R17_a_i + 9.00835072044484*v_R17_a_r - 1.84896616921897*v_R17_b_i + 9.21038227100566*v_R17_b_r - 5.65456401516768*v_R17_c_i - 30.9517475172273*v_R17_c_r - 1.84896616921897*v_R17_n_i + 9.21038227100566*v_R17_n_r + 0.793238195499527*v_R9_a_i - 9.00835072044484*v_R9_a_r + 1.84896616921897*v_R9_b_i - 9.21038227100566*v_R9_b_r + 5.65456401516768*v_R9_c_i + 30.9517475172273*v_R9_c_r + 1.84896616921897*v_R9_n_i - 9.21038227100566*v_R9_n_r
        struct[0].g[37,0] = i_R17_c_i + 9.00835072044484*v_R17_a_i + 0.793238195499527*v_R17_a_r + 9.21038227100566*v_R17_b_i + 1.84896616921897*v_R17_b_r - 30.9517475172273*v_R17_c_i + 5.65456401516768*v_R17_c_r + 9.21038227100566*v_R17_n_i + 1.84896616921897*v_R17_n_r - 9.00835072044484*v_R9_a_i - 0.793238195499527*v_R9_a_r - 9.21038227100566*v_R9_b_i - 1.84896616921897*v_R9_b_r + 30.9517475172273*v_R9_c_i - 5.65456401516768*v_R9_c_r - 9.21038227100566*v_R9_n_i - 1.84896616921897*v_R9_n_r
        struct[0].g[38,0] = i_R17_n_r - 1.84896616921897*v_R17_a_i + 9.21038227100566*v_R17_a_r - 0.793238195499527*v_R17_b_i + 9.00835072044485*v_R17_b_r - 1.84896616921897*v_R17_c_i + 9.21038227100566*v_R17_c_r - 5.65456401516768*v_R17_n_i - 30.9767475172273*v_R17_n_r + 1.84896616921897*v_R9_a_i - 9.21038227100566*v_R9_a_r + 0.793238195499527*v_R9_b_i - 9.00835072044485*v_R9_b_r + 1.84896616921897*v_R9_c_i - 9.21038227100566*v_R9_c_r + 5.65456401516768*v_R9_n_i + 30.9517475172273*v_R9_n_r
        struct[0].g[39,0] = i_R17_n_i + 9.21038227100566*v_R17_a_i + 1.84896616921897*v_R17_a_r + 9.00835072044485*v_R17_b_i + 0.793238195499527*v_R17_b_r + 9.21038227100566*v_R17_c_i + 1.84896616921897*v_R17_c_r - 30.9767475172273*v_R17_n_i + 5.65456401516768*v_R17_n_r - 9.21038227100566*v_R9_a_i - 1.84896616921897*v_R9_a_r - 9.00835072044485*v_R9_b_i - 0.793238195499527*v_R9_b_r - 9.21038227100566*v_R9_c_i - 1.84896616921897*v_R9_c_r + 30.9517475172273*v_R9_n_i - 5.65456401516768*v_R9_n_r
        struct[0].g[40,0] = i_R18_a_r + 5.65456401516768*v_R10_a_i + 30.9517475172273*v_R10_a_r + 1.84896616921897*v_R10_b_i - 9.21038227100566*v_R10_b_r + 0.793238195499529*v_R10_c_i - 9.00835072044485*v_R10_c_r + 1.84896616921897*v_R10_n_i - 9.21038227100566*v_R10_n_r - 5.65456401516768*v_R18_a_i - 30.9517475172273*v_R18_a_r - 1.84896616921897*v_R18_b_i + 9.21038227100566*v_R18_b_r - 0.793238195499529*v_R18_c_i + 9.00835072044485*v_R18_c_r - 1.84896616921897*v_R18_n_i + 9.21038227100566*v_R18_n_r
        struct[0].g[41,0] = i_R18_a_i + 30.9517475172273*v_R10_a_i - 5.65456401516768*v_R10_a_r - 9.21038227100566*v_R10_b_i - 1.84896616921897*v_R10_b_r - 9.00835072044485*v_R10_c_i - 0.793238195499529*v_R10_c_r - 9.21038227100566*v_R10_n_i - 1.84896616921897*v_R10_n_r - 30.9517475172273*v_R18_a_i + 5.65456401516768*v_R18_a_r + 9.21038227100566*v_R18_b_i + 1.84896616921897*v_R18_b_r + 9.00835072044485*v_R18_c_i + 0.793238195499529*v_R18_c_r + 9.21038227100566*v_R18_n_i + 1.84896616921897*v_R18_n_r
        struct[0].g[42,0] = i_R18_b_r + 1.84896616921897*v_R10_a_i - 9.21038227100566*v_R10_a_r + 5.65456401516768*v_R10_b_i + 30.9517475172273*v_R10_b_r + 1.84896616921897*v_R10_c_i - 9.21038227100566*v_R10_c_r + 0.793238195499528*v_R10_n_i - 9.00835072044485*v_R10_n_r - 1.84896616921897*v_R18_a_i + 9.21038227100566*v_R18_a_r - 5.65456401516768*v_R18_b_i - 30.9517475172273*v_R18_b_r - 1.84896616921897*v_R18_c_i + 9.21038227100566*v_R18_c_r - 0.793238195499528*v_R18_n_i + 9.00835072044485*v_R18_n_r
        struct[0].g[43,0] = i_R18_b_i - 9.21038227100566*v_R10_a_i - 1.84896616921897*v_R10_a_r + 30.9517475172273*v_R10_b_i - 5.65456401516768*v_R10_b_r - 9.21038227100566*v_R10_c_i - 1.84896616921897*v_R10_c_r - 9.00835072044485*v_R10_n_i - 0.793238195499528*v_R10_n_r + 9.21038227100566*v_R18_a_i + 1.84896616921897*v_R18_a_r - 30.9517475172273*v_R18_b_i + 5.65456401516768*v_R18_b_r + 9.21038227100566*v_R18_c_i + 1.84896616921897*v_R18_c_r + 9.00835072044485*v_R18_n_i + 0.793238195499528*v_R18_n_r
        struct[0].g[44,0] = i_R18_c_r + 0.793238195499527*v_R10_a_i - 9.00835072044484*v_R10_a_r + 1.84896616921897*v_R10_b_i - 9.21038227100566*v_R10_b_r + 5.65456401516768*v_R10_c_i + 30.9517475172273*v_R10_c_r + 1.84896616921897*v_R10_n_i - 9.21038227100566*v_R10_n_r - 0.793238195499527*v_R18_a_i + 9.00835072044484*v_R18_a_r - 1.84896616921897*v_R18_b_i + 9.21038227100566*v_R18_b_r - 5.65456401516768*v_R18_c_i - 30.9517475172273*v_R18_c_r - 1.84896616921897*v_R18_n_i + 9.21038227100566*v_R18_n_r
        struct[0].g[45,0] = i_R18_c_i - 9.00835072044484*v_R10_a_i - 0.793238195499527*v_R10_a_r - 9.21038227100566*v_R10_b_i - 1.84896616921897*v_R10_b_r + 30.9517475172273*v_R10_c_i - 5.65456401516768*v_R10_c_r - 9.21038227100566*v_R10_n_i - 1.84896616921897*v_R10_n_r + 9.00835072044484*v_R18_a_i + 0.793238195499527*v_R18_a_r + 9.21038227100566*v_R18_b_i + 1.84896616921897*v_R18_b_r - 30.9517475172273*v_R18_c_i + 5.65456401516768*v_R18_c_r + 9.21038227100566*v_R18_n_i + 1.84896616921897*v_R18_n_r
        struct[0].g[46,0] = i_R18_n_r + 1.84896616921897*v_R10_a_i - 9.21038227100566*v_R10_a_r + 0.793238195499527*v_R10_b_i - 9.00835072044485*v_R10_b_r + 1.84896616921897*v_R10_c_i - 9.21038227100566*v_R10_c_r + 5.65456401516768*v_R10_n_i + 30.9517475172273*v_R10_n_r - 1.84896616921897*v_R18_a_i + 9.21038227100566*v_R18_a_r - 0.793238195499527*v_R18_b_i + 9.00835072044485*v_R18_b_r - 1.84896616921897*v_R18_c_i + 9.21038227100566*v_R18_c_r - 5.65456401516768*v_R18_n_i - 30.9767475172273*v_R18_n_r
        struct[0].g[47,0] = i_R18_n_i - 9.21038227100566*v_R10_a_i - 1.84896616921897*v_R10_a_r - 9.00835072044485*v_R10_b_i - 0.793238195499527*v_R10_b_r - 9.21038227100566*v_R10_c_i - 1.84896616921897*v_R10_c_r + 30.9517475172273*v_R10_n_i - 5.65456401516768*v_R10_n_r + 9.21038227100566*v_R18_a_i + 1.84896616921897*v_R18_a_r + 9.00835072044485*v_R18_b_i + 0.793238195499527*v_R18_b_r + 9.21038227100566*v_R18_c_i + 1.84896616921897*v_R18_c_r - 30.9767475172273*v_R18_n_i + 5.65456401516768*v_R18_n_r
        struct[0].g[48,0] = i_R2_a_r + 54.0657727682604*v_R1_a_i + 105.57176931318*v_R1_a_r - 10.2713736253513*v_R1_b_i - 39.6392229058202*v_R1_b_r - 23.284964480954*v_R1_c_i - 24.9575997948692*v_R1_c_r - 10.2713736253513*v_R1_n_i - 39.6392229058202*v_R1_n_r - 108.131545536521*v_R2_a_i - 211.14353862636*v_R2_a_r + 20.5427472507027*v_R2_b_i + 79.2784458116403*v_R2_b_r + 46.569928961908*v_R2_c_i + 49.9151995897384*v_R2_c_r + 20.5427472507027*v_R2_n_i + 79.2784458116403*v_R2_n_r + 54.0657727682604*v_R3_a_i + 105.57176931318*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r - 23.284964480954*v_R3_c_i - 24.9575997948692*v_R3_c_r - 10.2713736253513*v_R3_n_i - 39.6392229058202*v_R3_n_r
        struct[0].g[49,0] = i_R2_a_i + 105.57176931318*v_R1_a_i - 54.0657727682604*v_R1_a_r - 39.6392229058202*v_R1_b_i + 10.2713736253513*v_R1_b_r - 24.9575997948692*v_R1_c_i + 23.284964480954*v_R1_c_r - 39.6392229058202*v_R1_n_i + 10.2713736253513*v_R1_n_r - 211.14353862636*v_R2_a_i + 108.131545536521*v_R2_a_r + 79.2784458116403*v_R2_b_i - 20.5427472507027*v_R2_b_r + 49.9151995897384*v_R2_c_i - 46.569928961908*v_R2_c_r + 79.2784458116403*v_R2_n_i - 20.5427472507027*v_R2_n_r + 105.57176931318*v_R3_a_i - 54.0657727682604*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r - 24.9575997948692*v_R3_c_i + 23.284964480954*v_R3_c_r - 39.6392229058202*v_R3_n_i + 10.2713736253513*v_R3_n_r
        struct[0].g[50,0] = i_R2_b_r - 10.2713736253513*v_R1_a_i - 39.6392229058202*v_R1_a_r + 54.0657727682604*v_R1_b_i + 105.57176931318*v_R1_b_r - 10.2713736253513*v_R1_c_i - 39.6392229058201*v_R1_c_r - 23.284964480954*v_R1_n_i - 24.9575997948692*v_R1_n_r + 20.5427472507027*v_R2_a_i + 79.2784458116403*v_R2_a_r - 108.131545536521*v_R2_b_i - 211.14353862636*v_R2_b_r + 20.5427472507027*v_R2_c_i + 79.2784458116403*v_R2_c_r + 46.569928961908*v_R2_n_i + 49.9151995897384*v_R2_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r + 54.0657727682604*v_R3_b_i + 105.57176931318*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058201*v_R3_c_r - 23.284964480954*v_R3_n_i - 24.9575997948692*v_R3_n_r
        struct[0].g[51,0] = i_R2_b_i - 39.6392229058202*v_R1_a_i + 10.2713736253513*v_R1_a_r + 105.57176931318*v_R1_b_i - 54.0657727682604*v_R1_b_r - 39.6392229058201*v_R1_c_i + 10.2713736253513*v_R1_c_r - 24.9575997948692*v_R1_n_i + 23.284964480954*v_R1_n_r + 79.2784458116403*v_R2_a_i - 20.5427472507027*v_R2_a_r - 211.14353862636*v_R2_b_i + 108.131545536521*v_R2_b_r + 79.2784458116403*v_R2_c_i - 20.5427472507027*v_R2_c_r + 49.9151995897384*v_R2_n_i - 46.569928961908*v_R2_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r + 105.57176931318*v_R3_b_i - 54.0657727682604*v_R3_b_r - 39.6392229058201*v_R3_c_i + 10.2713736253513*v_R3_c_r - 24.9575997948692*v_R3_n_i + 23.284964480954*v_R3_n_r
        struct[0].g[52,0] = i_R2_c_r - 23.284964480954*v_R1_a_i - 24.9575997948692*v_R1_a_r - 10.2713736253513*v_R1_b_i - 39.6392229058202*v_R1_b_r + 54.0657727682604*v_R1_c_i + 105.57176931318*v_R1_c_r - 10.2713736253514*v_R1_n_i - 39.6392229058201*v_R1_n_r + 46.569928961908*v_R2_a_i + 49.9151995897384*v_R2_a_r + 20.5427472507027*v_R2_b_i + 79.2784458116404*v_R2_b_r - 108.131545536521*v_R2_c_i - 211.14353862636*v_R2_c_r + 20.5427472507027*v_R2_n_i + 79.2784458116403*v_R2_n_r - 23.284964480954*v_R3_a_i - 24.9575997948692*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r + 54.0657727682604*v_R3_c_i + 105.57176931318*v_R3_c_r - 10.2713736253514*v_R3_n_i - 39.6392229058201*v_R3_n_r
        struct[0].g[53,0] = i_R2_c_i - 24.9575997948692*v_R1_a_i + 23.284964480954*v_R1_a_r - 39.6392229058202*v_R1_b_i + 10.2713736253513*v_R1_b_r + 105.57176931318*v_R1_c_i - 54.0657727682604*v_R1_c_r - 39.6392229058201*v_R1_n_i + 10.2713736253514*v_R1_n_r + 49.9151995897384*v_R2_a_i - 46.569928961908*v_R2_a_r + 79.2784458116404*v_R2_b_i - 20.5427472507027*v_R2_b_r - 211.14353862636*v_R2_c_i + 108.131545536521*v_R2_c_r + 79.2784458116403*v_R2_n_i - 20.5427472507027*v_R2_n_r - 24.9575997948692*v_R3_a_i + 23.284964480954*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r + 105.57176931318*v_R3_c_i - 54.0657727682604*v_R3_c_r - 39.6392229058201*v_R3_n_i + 10.2713736253514*v_R3_n_r
        struct[0].g[54,0] = i_R2_n_r - 10.2713736253513*v_R1_a_i - 39.6392229058202*v_R1_a_r - 23.284964480954*v_R1_b_i - 24.9575997948692*v_R1_b_r - 10.2713736253513*v_R1_c_i - 39.6392229058202*v_R1_c_r + 54.0657727682604*v_R1_n_i + 105.57176931318*v_R1_n_r + 20.5427472507027*v_R2_a_i + 79.2784458116403*v_R2_a_r + 46.569928961908*v_R2_b_i + 49.9151995897384*v_R2_b_r + 20.5427472507027*v_R2_c_i + 79.2784458116403*v_R2_c_r - 108.131545536521*v_R2_n_i - 211.16853862636*v_R2_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r - 23.284964480954*v_R3_b_i - 24.9575997948692*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058202*v_R3_c_r + 54.0657727682604*v_R3_n_i + 105.57176931318*v_R3_n_r
        struct[0].g[55,0] = i_R2_n_i - 39.6392229058202*v_R1_a_i + 10.2713736253513*v_R1_a_r - 24.9575997948692*v_R1_b_i + 23.284964480954*v_R1_b_r - 39.6392229058202*v_R1_c_i + 10.2713736253513*v_R1_c_r + 105.57176931318*v_R1_n_i - 54.0657727682604*v_R1_n_r + 79.2784458116403*v_R2_a_i - 20.5427472507027*v_R2_a_r + 49.9151995897384*v_R2_b_i - 46.569928961908*v_R2_b_r + 79.2784458116403*v_R2_c_i - 20.5427472507027*v_R2_c_r - 211.16853862636*v_R2_n_i + 108.131545536521*v_R2_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r - 24.9575997948692*v_R3_b_i + 23.284964480954*v_R3_b_r - 39.6392229058202*v_R3_c_i + 10.2713736253513*v_R3_c_r + 105.57176931318*v_R3_n_i - 54.0657727682604*v_R3_n_r
        struct[0].g[56,0] = i_R3_a_r + 4.84676915585801*v_R11_a_i + 26.5300693004806*v_R11_a_r + 1.58482814504483*v_R11_b_i - 7.8946133751477*v_R11_b_r + 0.67991845328531*v_R11_c_i - 7.72144347466701*v_R11_c_r + 1.58482814504483*v_R11_n_i - 7.89461337514771*v_R11_n_r + 54.0657727682604*v_R2_a_i + 105.57176931318*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r - 23.284964480954*v_R2_c_i - 24.9575997948692*v_R2_c_r - 10.2713736253513*v_R2_n_i - 39.6392229058202*v_R2_n_r - 112.978314692379*v_R3_a_i - 237.673607926841*v_R3_a_r + 18.9579191056578*v_R3_b_i + 87.173059186788*v_R3_b_r + 45.8900105086227*v_R3_c_i + 57.6366430644055*v_R3_c_r + 18.9579191056579*v_R3_n_i + 87.173059186788*v_R3_n_r + 54.0657727682604*v_R4_a_i + 105.57176931318*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r - 23.284964480954*v_R4_c_i - 24.9575997948692*v_R4_c_r - 10.2713736253513*v_R4_n_i - 39.6392229058202*v_R4_n_r
        struct[0].g[57,0] = i_R3_a_i + 26.5300693004806*v_R11_a_i - 4.84676915585801*v_R11_a_r - 7.8946133751477*v_R11_b_i - 1.58482814504483*v_R11_b_r - 7.72144347466701*v_R11_c_i - 0.67991845328531*v_R11_c_r - 7.89461337514771*v_R11_n_i - 1.58482814504483*v_R11_n_r + 105.57176931318*v_R2_a_i - 54.0657727682604*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r - 24.9575997948692*v_R2_c_i + 23.284964480954*v_R2_c_r - 39.6392229058202*v_R2_n_i + 10.2713736253513*v_R2_n_r - 237.673607926841*v_R3_a_i + 112.978314692379*v_R3_a_r + 87.173059186788*v_R3_b_i - 18.9579191056578*v_R3_b_r + 57.6366430644055*v_R3_c_i - 45.8900105086227*v_R3_c_r + 87.173059186788*v_R3_n_i - 18.9579191056579*v_R3_n_r + 105.57176931318*v_R4_a_i - 54.0657727682604*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r - 24.9575997948692*v_R4_c_i + 23.284964480954*v_R4_c_r - 39.6392229058202*v_R4_n_i + 10.2713736253513*v_R4_n_r
        struct[0].g[58,0] = i_R3_b_r + 1.58482814504483*v_R11_a_i - 7.89461337514771*v_R11_a_r + 4.84676915585801*v_R11_b_i + 26.5300693004806*v_R11_b_r + 1.58482814504483*v_R11_c_i - 7.89461337514771*v_R11_c_r + 0.679918453285308*v_R11_n_i - 7.72144347466701*v_R11_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r + 54.0657727682604*v_R2_b_i + 105.57176931318*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058201*v_R2_c_r - 23.284964480954*v_R2_n_i - 24.9575997948692*v_R2_n_r + 18.9579191056579*v_R3_a_i + 87.173059186788*v_R3_a_r - 112.978314692379*v_R3_b_i - 237.673607926841*v_R3_b_r + 18.9579191056579*v_R3_c_i + 87.173059186788*v_R3_c_r + 45.8900105086227*v_R3_n_i + 57.6366430644054*v_R3_n_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r + 54.0657727682604*v_R4_b_i + 105.57176931318*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058201*v_R4_c_r - 23.284964480954*v_R4_n_i - 24.9575997948692*v_R4_n_r
        struct[0].g[59,0] = i_R3_b_i - 7.89461337514771*v_R11_a_i - 1.58482814504483*v_R11_a_r + 26.5300693004806*v_R11_b_i - 4.84676915585801*v_R11_b_r - 7.89461337514771*v_R11_c_i - 1.58482814504483*v_R11_c_r - 7.72144347466701*v_R11_n_i - 0.679918453285308*v_R11_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r + 105.57176931318*v_R2_b_i - 54.0657727682604*v_R2_b_r - 39.6392229058201*v_R2_c_i + 10.2713736253513*v_R2_c_r - 24.9575997948692*v_R2_n_i + 23.284964480954*v_R2_n_r + 87.173059186788*v_R3_a_i - 18.9579191056579*v_R3_a_r - 237.673607926841*v_R3_b_i + 112.978314692379*v_R3_b_r + 87.173059186788*v_R3_c_i - 18.9579191056579*v_R3_c_r + 57.6366430644054*v_R3_n_i - 45.8900105086227*v_R3_n_r - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r + 105.57176931318*v_R4_b_i - 54.0657727682604*v_R4_b_r - 39.6392229058201*v_R4_c_i + 10.2713736253513*v_R4_c_r - 24.9575997948692*v_R4_n_i + 23.284964480954*v_R4_n_r
        struct[0].g[60,0] = i_R3_c_r + 0.67991845328531*v_R11_a_i - 7.72144347466701*v_R11_a_r + 1.58482814504483*v_R11_b_i - 7.8946133751477*v_R11_b_r + 4.84676915585801*v_R11_c_i + 26.5300693004806*v_R11_c_r + 1.58482814504483*v_R11_n_i - 7.89461337514771*v_R11_n_r - 23.284964480954*v_R2_a_i - 24.9575997948692*v_R2_a_r - 10.2713736253513*v_R2_b_i - 39.6392229058202*v_R2_b_r + 54.0657727682604*v_R2_c_i + 105.57176931318*v_R2_c_r - 10.2713736253514*v_R2_n_i - 39.6392229058201*v_R2_n_r + 45.8900105086227*v_R3_a_i + 57.6366430644054*v_R3_a_r + 18.9579191056578*v_R3_b_i + 87.1730591867881*v_R3_b_r - 112.978314692379*v_R3_c_i - 237.673607926841*v_R3_c_r + 18.9579191056579*v_R3_n_i + 87.173059186788*v_R3_n_r - 23.284964480954*v_R4_a_i - 24.9575997948692*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r + 54.0657727682604*v_R4_c_i + 105.57176931318*v_R4_c_r - 10.2713736253514*v_R4_n_i - 39.6392229058201*v_R4_n_r
        struct[0].g[61,0] = i_R3_c_i - 7.72144347466701*v_R11_a_i - 0.67991845328531*v_R11_a_r - 7.8946133751477*v_R11_b_i - 1.58482814504483*v_R11_b_r + 26.5300693004806*v_R11_c_i - 4.84676915585801*v_R11_c_r - 7.89461337514771*v_R11_n_i - 1.58482814504483*v_R11_n_r - 24.9575997948692*v_R2_a_i + 23.284964480954*v_R2_a_r - 39.6392229058202*v_R2_b_i + 10.2713736253513*v_R2_b_r + 105.57176931318*v_R2_c_i - 54.0657727682604*v_R2_c_r - 39.6392229058201*v_R2_n_i + 10.2713736253514*v_R2_n_r + 57.6366430644054*v_R3_a_i - 45.8900105086227*v_R3_a_r + 87.1730591867881*v_R3_b_i - 18.9579191056578*v_R3_b_r - 237.673607926841*v_R3_c_i + 112.978314692379*v_R3_c_r + 87.173059186788*v_R3_n_i - 18.9579191056579*v_R3_n_r - 24.9575997948692*v_R4_a_i + 23.284964480954*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r + 105.57176931318*v_R4_c_i - 54.0657727682604*v_R4_c_r - 39.6392229058201*v_R4_n_i + 10.2713736253514*v_R4_n_r
        struct[0].g[62,0] = i_R3_n_r + 1.58482814504483*v_R11_a_i - 7.89461337514771*v_R11_a_r + 0.679918453285308*v_R11_b_i - 7.72144347466701*v_R11_b_r + 1.58482814504483*v_R11_c_i - 7.89461337514771*v_R11_c_r + 4.84676915585801*v_R11_n_i + 26.5300693004806*v_R11_n_r - 10.2713736253513*v_R2_a_i - 39.6392229058202*v_R2_a_r - 23.284964480954*v_R2_b_i - 24.9575997948692*v_R2_b_r - 10.2713736253513*v_R2_c_i - 39.6392229058202*v_R2_c_r + 54.0657727682604*v_R2_n_i + 105.57176931318*v_R2_n_r + 18.9579191056578*v_R3_a_i + 87.173059186788*v_R3_a_r + 45.8900105086227*v_R3_b_i + 57.6366430644054*v_R3_b_r + 18.9579191056579*v_R3_c_i + 87.173059186788*v_R3_c_r - 112.978314692379*v_R3_n_i - 237.673607926841*v_R3_n_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r - 23.284964480954*v_R4_b_i - 24.9575997948692*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058202*v_R4_c_r + 54.0657727682604*v_R4_n_i + 105.57176931318*v_R4_n_r
        struct[0].g[63,0] = i_R3_n_i - 7.89461337514771*v_R11_a_i - 1.58482814504483*v_R11_a_r - 7.72144347466701*v_R11_b_i - 0.679918453285308*v_R11_b_r - 7.89461337514771*v_R11_c_i - 1.58482814504483*v_R11_c_r + 26.5300693004806*v_R11_n_i - 4.84676915585801*v_R11_n_r - 39.6392229058202*v_R2_a_i + 10.2713736253513*v_R2_a_r - 24.9575997948692*v_R2_b_i + 23.284964480954*v_R2_b_r - 39.6392229058202*v_R2_c_i + 10.2713736253513*v_R2_c_r + 105.57176931318*v_R2_n_i - 54.0657727682604*v_R2_n_r + 87.173059186788*v_R3_a_i - 18.9579191056578*v_R3_a_r + 57.6366430644054*v_R3_b_i - 45.8900105086227*v_R3_b_r + 87.173059186788*v_R3_c_i - 18.9579191056579*v_R3_c_r - 237.673607926841*v_R3_n_i + 112.978314692379*v_R3_n_r - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r - 24.9575997948692*v_R4_b_i + 23.284964480954*v_R4_b_r - 39.6392229058202*v_R4_c_i + 10.2713736253513*v_R4_c_r + 105.57176931318*v_R4_n_i - 54.0657727682604*v_R4_n_r
        struct[0].g[64,0] = i_R4_a_r + 4.84676915585801*v_R12_a_i + 26.5300693004806*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 0.67991845328531*v_R12_c_i - 7.72144347466701*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r + 54.0657727682604*v_R3_a_i + 105.57176931318*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r - 23.284964480954*v_R3_c_i - 24.9575997948692*v_R3_c_r - 10.2713736253513*v_R3_n_i - 39.6392229058202*v_R3_n_r - 112.978314692379*v_R4_a_i - 237.673607926841*v_R4_a_r + 18.9579191056578*v_R4_b_i + 87.173059186788*v_R4_b_r + 45.8900105086227*v_R4_c_i + 57.6366430644055*v_R4_c_r + 18.9579191056579*v_R4_n_i + 87.173059186788*v_R4_n_r + 54.0657727682604*v_R5_a_i + 105.57176931318*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r - 23.284964480954*v_R5_c_i - 24.9575997948692*v_R5_c_r - 10.2713736253513*v_R5_n_i - 39.6392229058202*v_R5_n_r
        struct[0].g[65,0] = i_R4_a_i + 26.5300693004806*v_R12_a_i - 4.84676915585801*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r - 7.72144347466701*v_R12_c_i - 0.67991845328531*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r + 105.57176931318*v_R3_a_i - 54.0657727682604*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r - 24.9575997948692*v_R3_c_i + 23.284964480954*v_R3_c_r - 39.6392229058202*v_R3_n_i + 10.2713736253513*v_R3_n_r - 237.673607926841*v_R4_a_i + 112.978314692379*v_R4_a_r + 87.173059186788*v_R4_b_i - 18.9579191056578*v_R4_b_r + 57.6366430644055*v_R4_c_i - 45.8900105086227*v_R4_c_r + 87.173059186788*v_R4_n_i - 18.9579191056579*v_R4_n_r + 105.57176931318*v_R5_a_i - 54.0657727682604*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r - 24.9575997948692*v_R5_c_i + 23.284964480954*v_R5_c_r - 39.6392229058202*v_R5_n_i + 10.2713736253513*v_R5_n_r
        struct[0].g[66,0] = i_R4_b_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 4.84676915585801*v_R12_b_i + 26.5300693004806*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 0.679918453285308*v_R12_n_i - 7.72144347466701*v_R12_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r + 54.0657727682604*v_R3_b_i + 105.57176931318*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058201*v_R3_c_r - 23.284964480954*v_R3_n_i - 24.9575997948692*v_R3_n_r + 18.9579191056579*v_R4_a_i + 87.173059186788*v_R4_a_r - 112.978314692379*v_R4_b_i - 237.673607926841*v_R4_b_r + 18.9579191056579*v_R4_c_i + 87.173059186788*v_R4_c_r + 45.8900105086227*v_R4_n_i + 57.6366430644054*v_R4_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r + 54.0657727682604*v_R5_b_i + 105.57176931318*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058201*v_R5_c_r - 23.284964480954*v_R5_n_i - 24.9575997948692*v_R5_n_r
        struct[0].g[67,0] = i_R4_b_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r + 26.5300693004806*v_R12_b_i - 4.84676915585801*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r - 7.72144347466701*v_R12_n_i - 0.679918453285308*v_R12_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r + 105.57176931318*v_R3_b_i - 54.0657727682604*v_R3_b_r - 39.6392229058201*v_R3_c_i + 10.2713736253513*v_R3_c_r - 24.9575997948692*v_R3_n_i + 23.284964480954*v_R3_n_r + 87.173059186788*v_R4_a_i - 18.9579191056579*v_R4_a_r - 237.673607926841*v_R4_b_i + 112.978314692379*v_R4_b_r + 87.173059186788*v_R4_c_i - 18.9579191056579*v_R4_c_r + 57.6366430644054*v_R4_n_i - 45.8900105086227*v_R4_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r + 105.57176931318*v_R5_b_i - 54.0657727682604*v_R5_b_r - 39.6392229058201*v_R5_c_i + 10.2713736253513*v_R5_c_r - 24.9575997948692*v_R5_n_i + 23.284964480954*v_R5_n_r
        struct[0].g[68,0] = i_R4_c_r + 0.67991845328531*v_R12_a_i - 7.72144347466701*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 4.84676915585801*v_R12_c_i + 26.5300693004806*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r - 23.284964480954*v_R3_a_i - 24.9575997948692*v_R3_a_r - 10.2713736253513*v_R3_b_i - 39.6392229058202*v_R3_b_r + 54.0657727682604*v_R3_c_i + 105.57176931318*v_R3_c_r - 10.2713736253514*v_R3_n_i - 39.6392229058201*v_R3_n_r + 45.8900105086227*v_R4_a_i + 57.6366430644054*v_R4_a_r + 18.9579191056578*v_R4_b_i + 87.1730591867881*v_R4_b_r - 112.978314692379*v_R4_c_i - 237.673607926841*v_R4_c_r + 18.9579191056579*v_R4_n_i + 87.173059186788*v_R4_n_r - 23.284964480954*v_R5_a_i - 24.9575997948692*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r + 54.0657727682604*v_R5_c_i + 105.57176931318*v_R5_c_r - 10.2713736253514*v_R5_n_i - 39.6392229058201*v_R5_n_r
        struct[0].g[69,0] = i_R4_c_i - 7.72144347466701*v_R12_a_i - 0.67991845328531*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r + 26.5300693004806*v_R12_c_i - 4.84676915585801*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r - 24.9575997948692*v_R3_a_i + 23.284964480954*v_R3_a_r - 39.6392229058202*v_R3_b_i + 10.2713736253513*v_R3_b_r + 105.57176931318*v_R3_c_i - 54.0657727682604*v_R3_c_r - 39.6392229058201*v_R3_n_i + 10.2713736253514*v_R3_n_r + 57.6366430644054*v_R4_a_i - 45.8900105086227*v_R4_a_r + 87.1730591867881*v_R4_b_i - 18.9579191056578*v_R4_b_r - 237.673607926841*v_R4_c_i + 112.978314692379*v_R4_c_r + 87.173059186788*v_R4_n_i - 18.9579191056579*v_R4_n_r - 24.9575997948692*v_R5_a_i + 23.284964480954*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r + 105.57176931318*v_R5_c_i - 54.0657727682604*v_R5_c_r - 39.6392229058201*v_R5_n_i + 10.2713736253514*v_R5_n_r
        struct[0].g[70,0] = i_R4_n_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 0.679918453285308*v_R12_b_i - 7.72144347466701*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 4.84676915585801*v_R12_n_i + 26.5300693004806*v_R12_n_r - 10.2713736253513*v_R3_a_i - 39.6392229058202*v_R3_a_r - 23.284964480954*v_R3_b_i - 24.9575997948692*v_R3_b_r - 10.2713736253513*v_R3_c_i - 39.6392229058202*v_R3_c_r + 54.0657727682604*v_R3_n_i + 105.57176931318*v_R3_n_r + 18.9579191056578*v_R4_a_i + 87.173059186788*v_R4_a_r + 45.8900105086227*v_R4_b_i + 57.6366430644054*v_R4_b_r + 18.9579191056579*v_R4_c_i + 87.173059186788*v_R4_c_r - 112.978314692379*v_R4_n_i - 237.698607926841*v_R4_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r - 23.284964480954*v_R5_b_i - 24.9575997948692*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058202*v_R5_c_r + 54.0657727682604*v_R5_n_i + 105.57176931318*v_R5_n_r
        struct[0].g[71,0] = i_R4_n_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r - 7.72144347466701*v_R12_b_i - 0.679918453285308*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r + 26.5300693004806*v_R12_n_i - 4.84676915585801*v_R12_n_r - 39.6392229058202*v_R3_a_i + 10.2713736253513*v_R3_a_r - 24.9575997948692*v_R3_b_i + 23.284964480954*v_R3_b_r - 39.6392229058202*v_R3_c_i + 10.2713736253513*v_R3_c_r + 105.57176931318*v_R3_n_i - 54.0657727682604*v_R3_n_r + 87.173059186788*v_R4_a_i - 18.9579191056578*v_R4_a_r + 57.6366430644054*v_R4_b_i - 45.8900105086227*v_R4_b_r + 87.173059186788*v_R4_c_i - 18.9579191056579*v_R4_c_r - 237.698607926841*v_R4_n_i + 112.978314692379*v_R4_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r - 24.9575997948692*v_R5_b_i + 23.284964480954*v_R5_b_r - 39.6392229058202*v_R5_c_i + 10.2713736253513*v_R5_c_r + 105.57176931318*v_R5_n_i - 54.0657727682604*v_R5_n_r
        struct[0].g[72,0] = i_R5_a_r + 54.0657727682604*v_R4_a_i + 105.57176931318*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r - 23.284964480954*v_R4_c_i - 24.9575997948692*v_R4_c_r - 10.2713736253513*v_R4_n_i - 39.6392229058202*v_R4_n_r - 108.131545536521*v_R5_a_i - 211.14353862636*v_R5_a_r + 20.5427472507027*v_R5_b_i + 79.2784458116403*v_R5_b_r + 46.569928961908*v_R5_c_i + 49.9151995897384*v_R5_c_r + 20.5427472507027*v_R5_n_i + 79.2784458116403*v_R5_n_r + 54.0657727682604*v_R6_a_i + 105.57176931318*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r - 23.284964480954*v_R6_c_i - 24.9575997948692*v_R6_c_r - 10.2713736253513*v_R6_n_i - 39.6392229058202*v_R6_n_r
        struct[0].g[73,0] = i_R5_a_i + 105.57176931318*v_R4_a_i - 54.0657727682604*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r - 24.9575997948692*v_R4_c_i + 23.284964480954*v_R4_c_r - 39.6392229058202*v_R4_n_i + 10.2713736253513*v_R4_n_r - 211.14353862636*v_R5_a_i + 108.131545536521*v_R5_a_r + 79.2784458116403*v_R5_b_i - 20.5427472507027*v_R5_b_r + 49.9151995897384*v_R5_c_i - 46.569928961908*v_R5_c_r + 79.2784458116403*v_R5_n_i - 20.5427472507027*v_R5_n_r + 105.57176931318*v_R6_a_i - 54.0657727682604*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r - 24.9575997948692*v_R6_c_i + 23.284964480954*v_R6_c_r - 39.6392229058202*v_R6_n_i + 10.2713736253513*v_R6_n_r
        struct[0].g[74,0] = i_R5_b_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r + 54.0657727682604*v_R4_b_i + 105.57176931318*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058201*v_R4_c_r - 23.284964480954*v_R4_n_i - 24.9575997948692*v_R4_n_r + 20.5427472507027*v_R5_a_i + 79.2784458116403*v_R5_a_r - 108.131545536521*v_R5_b_i - 211.14353862636*v_R5_b_r + 20.5427472507027*v_R5_c_i + 79.2784458116403*v_R5_c_r + 46.569928961908*v_R5_n_i + 49.9151995897384*v_R5_n_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r + 54.0657727682604*v_R6_b_i + 105.57176931318*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058201*v_R6_c_r - 23.284964480954*v_R6_n_i - 24.9575997948692*v_R6_n_r
        struct[0].g[75,0] = i_R5_b_i - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r + 105.57176931318*v_R4_b_i - 54.0657727682604*v_R4_b_r - 39.6392229058201*v_R4_c_i + 10.2713736253513*v_R4_c_r - 24.9575997948692*v_R4_n_i + 23.284964480954*v_R4_n_r + 79.2784458116403*v_R5_a_i - 20.5427472507027*v_R5_a_r - 211.14353862636*v_R5_b_i + 108.131545536521*v_R5_b_r + 79.2784458116403*v_R5_c_i - 20.5427472507027*v_R5_c_r + 49.9151995897384*v_R5_n_i - 46.569928961908*v_R5_n_r - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r + 105.57176931318*v_R6_b_i - 54.0657727682604*v_R6_b_r - 39.6392229058201*v_R6_c_i + 10.2713736253513*v_R6_c_r - 24.9575997948692*v_R6_n_i + 23.284964480954*v_R6_n_r
        struct[0].g[76,0] = i_R5_c_r - 23.284964480954*v_R4_a_i - 24.9575997948692*v_R4_a_r - 10.2713736253513*v_R4_b_i - 39.6392229058202*v_R4_b_r + 54.0657727682604*v_R4_c_i + 105.57176931318*v_R4_c_r - 10.2713736253514*v_R4_n_i - 39.6392229058201*v_R4_n_r + 46.569928961908*v_R5_a_i + 49.9151995897384*v_R5_a_r + 20.5427472507027*v_R5_b_i + 79.2784458116404*v_R5_b_r - 108.131545536521*v_R5_c_i - 211.14353862636*v_R5_c_r + 20.5427472507027*v_R5_n_i + 79.2784458116403*v_R5_n_r - 23.284964480954*v_R6_a_i - 24.9575997948692*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r + 54.0657727682604*v_R6_c_i + 105.57176931318*v_R6_c_r - 10.2713736253514*v_R6_n_i - 39.6392229058201*v_R6_n_r
        struct[0].g[77,0] = i_R5_c_i - 24.9575997948692*v_R4_a_i + 23.284964480954*v_R4_a_r - 39.6392229058202*v_R4_b_i + 10.2713736253513*v_R4_b_r + 105.57176931318*v_R4_c_i - 54.0657727682604*v_R4_c_r - 39.6392229058201*v_R4_n_i + 10.2713736253514*v_R4_n_r + 49.9151995897384*v_R5_a_i - 46.569928961908*v_R5_a_r + 79.2784458116404*v_R5_b_i - 20.5427472507027*v_R5_b_r - 211.14353862636*v_R5_c_i + 108.131545536521*v_R5_c_r + 79.2784458116403*v_R5_n_i - 20.5427472507027*v_R5_n_r - 24.9575997948692*v_R6_a_i + 23.284964480954*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r + 105.57176931318*v_R6_c_i - 54.0657727682604*v_R6_c_r - 39.6392229058201*v_R6_n_i + 10.2713736253514*v_R6_n_r
        struct[0].g[78,0] = i_R5_n_r - 10.2713736253513*v_R4_a_i - 39.6392229058202*v_R4_a_r - 23.284964480954*v_R4_b_i - 24.9575997948692*v_R4_b_r - 10.2713736253513*v_R4_c_i - 39.6392229058202*v_R4_c_r + 54.0657727682604*v_R4_n_i + 105.57176931318*v_R4_n_r + 20.5427472507027*v_R5_a_i + 79.2784458116403*v_R5_a_r + 46.569928961908*v_R5_b_i + 49.9151995897384*v_R5_b_r + 20.5427472507027*v_R5_c_i + 79.2784458116403*v_R5_c_r - 108.131545536521*v_R5_n_i - 211.14353862636*v_R5_n_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r - 23.284964480954*v_R6_b_i - 24.9575997948692*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058202*v_R6_c_r + 54.0657727682604*v_R6_n_i + 105.57176931318*v_R6_n_r
        struct[0].g[79,0] = i_R5_n_i - 39.6392229058202*v_R4_a_i + 10.2713736253513*v_R4_a_r - 24.9575997948692*v_R4_b_i + 23.284964480954*v_R4_b_r - 39.6392229058202*v_R4_c_i + 10.2713736253513*v_R4_c_r + 105.57176931318*v_R4_n_i - 54.0657727682604*v_R4_n_r + 79.2784458116403*v_R5_a_i - 20.5427472507027*v_R5_a_r + 49.9151995897384*v_R5_b_i - 46.569928961908*v_R5_b_r + 79.2784458116403*v_R5_c_i - 20.5427472507027*v_R5_c_r - 211.14353862636*v_R5_n_i + 108.131545536521*v_R5_n_r - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r - 24.9575997948692*v_R6_b_i + 23.284964480954*v_R6_b_r - 39.6392229058202*v_R6_c_i + 10.2713736253513*v_R6_c_r + 105.57176931318*v_R6_n_i - 54.0657727682604*v_R6_n_r
        struct[0].g[80,0] = i_R6_a_r + 5.65456401516768*v_R16_a_i + 30.9517475172273*v_R16_a_r + 1.84896616921897*v_R16_b_i - 9.21038227100566*v_R16_b_r + 0.793238195499529*v_R16_c_i - 9.00835072044485*v_R16_c_r + 1.84896616921897*v_R16_n_i - 9.21038227100566*v_R16_n_r + 54.0657727682604*v_R5_a_i + 105.57176931318*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r - 23.284964480954*v_R5_c_i - 24.9575997948692*v_R5_c_r - 10.2713736253513*v_R5_n_i - 39.6392229058202*v_R5_n_r - 113.786109551688*v_R6_a_i - 242.095286143588*v_R6_a_r + 18.6937810814837*v_R6_b_i + 88.488828082646*v_R6_b_r + 45.7766907664085*v_R6_c_i + 58.9235503101833*v_R6_c_r + 18.6937810814837*v_R6_n_i + 88.488828082646*v_R6_n_r + 54.0657727682604*v_R7_a_i + 105.57176931318*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r - 23.284964480954*v_R7_c_i - 24.9575997948692*v_R7_c_r - 10.2713736253513*v_R7_n_i - 39.6392229058202*v_R7_n_r
        struct[0].g[81,0] = i_R6_a_i + 30.9517475172273*v_R16_a_i - 5.65456401516768*v_R16_a_r - 9.21038227100566*v_R16_b_i - 1.84896616921897*v_R16_b_r - 9.00835072044485*v_R16_c_i - 0.793238195499529*v_R16_c_r - 9.21038227100566*v_R16_n_i - 1.84896616921897*v_R16_n_r + 105.57176931318*v_R5_a_i - 54.0657727682604*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r - 24.9575997948692*v_R5_c_i + 23.284964480954*v_R5_c_r - 39.6392229058202*v_R5_n_i + 10.2713736253513*v_R5_n_r - 242.095286143588*v_R6_a_i + 113.786109551688*v_R6_a_r + 88.488828082646*v_R6_b_i - 18.6937810814837*v_R6_b_r + 58.9235503101833*v_R6_c_i - 45.7766907664085*v_R6_c_r + 88.488828082646*v_R6_n_i - 18.6937810814837*v_R6_n_r + 105.57176931318*v_R7_a_i - 54.0657727682604*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r - 24.9575997948692*v_R7_c_i + 23.284964480954*v_R7_c_r - 39.6392229058202*v_R7_n_i + 10.2713736253513*v_R7_n_r
        struct[0].g[82,0] = i_R6_b_r + 1.84896616921897*v_R16_a_i - 9.21038227100566*v_R16_a_r + 5.65456401516768*v_R16_b_i + 30.9517475172273*v_R16_b_r + 1.84896616921897*v_R16_c_i - 9.21038227100566*v_R16_c_r + 0.793238195499528*v_R16_n_i - 9.00835072044485*v_R16_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r + 54.0657727682604*v_R5_b_i + 105.57176931318*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058201*v_R5_c_r - 23.284964480954*v_R5_n_i - 24.9575997948692*v_R5_n_r + 18.6937810814837*v_R6_a_i + 88.488828082646*v_R6_a_r - 113.786109551688*v_R6_b_i - 242.095286143588*v_R6_b_r + 18.6937810814837*v_R6_c_i + 88.488828082646*v_R6_c_r + 45.7766907664085*v_R6_n_i + 58.9235503101833*v_R6_n_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r + 54.0657727682604*v_R7_b_i + 105.57176931318*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058201*v_R7_c_r - 23.284964480954*v_R7_n_i - 24.9575997948692*v_R7_n_r
        struct[0].g[83,0] = i_R6_b_i - 9.21038227100566*v_R16_a_i - 1.84896616921897*v_R16_a_r + 30.9517475172273*v_R16_b_i - 5.65456401516768*v_R16_b_r - 9.21038227100566*v_R16_c_i - 1.84896616921897*v_R16_c_r - 9.00835072044485*v_R16_n_i - 0.793238195499528*v_R16_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r + 105.57176931318*v_R5_b_i - 54.0657727682604*v_R5_b_r - 39.6392229058201*v_R5_c_i + 10.2713736253513*v_R5_c_r - 24.9575997948692*v_R5_n_i + 23.284964480954*v_R5_n_r + 88.488828082646*v_R6_a_i - 18.6937810814837*v_R6_a_r - 242.095286143588*v_R6_b_i + 113.786109551688*v_R6_b_r + 88.488828082646*v_R6_c_i - 18.6937810814837*v_R6_c_r + 58.9235503101833*v_R6_n_i - 45.7766907664085*v_R6_n_r - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r + 105.57176931318*v_R7_b_i - 54.0657727682604*v_R7_b_r - 39.6392229058201*v_R7_c_i + 10.2713736253513*v_R7_c_r - 24.9575997948692*v_R7_n_i + 23.284964480954*v_R7_n_r
        struct[0].g[84,0] = i_R6_c_r + 0.793238195499527*v_R16_a_i - 9.00835072044484*v_R16_a_r + 1.84896616921897*v_R16_b_i - 9.21038227100566*v_R16_b_r + 5.65456401516768*v_R16_c_i + 30.9517475172273*v_R16_c_r + 1.84896616921897*v_R16_n_i - 9.21038227100566*v_R16_n_r - 23.284964480954*v_R5_a_i - 24.9575997948692*v_R5_a_r - 10.2713736253513*v_R5_b_i - 39.6392229058202*v_R5_b_r + 54.0657727682604*v_R5_c_i + 105.57176931318*v_R5_c_r - 10.2713736253514*v_R5_n_i - 39.6392229058201*v_R5_n_r + 45.7766907664085*v_R6_a_i + 58.9235503101833*v_R6_a_r + 18.6937810814837*v_R6_b_i + 88.4888280826461*v_R6_b_r - 113.786109551688*v_R6_c_i - 242.095286143588*v_R6_c_r + 18.6937810814837*v_R6_n_i + 88.4888280826459*v_R6_n_r - 23.284964480954*v_R7_a_i - 24.9575997948692*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r + 54.0657727682604*v_R7_c_i + 105.57176931318*v_R7_c_r - 10.2713736253514*v_R7_n_i - 39.6392229058201*v_R7_n_r
        struct[0].g[85,0] = i_R6_c_i - 9.00835072044484*v_R16_a_i - 0.793238195499527*v_R16_a_r - 9.21038227100566*v_R16_b_i - 1.84896616921897*v_R16_b_r + 30.9517475172273*v_R16_c_i - 5.65456401516768*v_R16_c_r - 9.21038227100566*v_R16_n_i - 1.84896616921897*v_R16_n_r - 24.9575997948692*v_R5_a_i + 23.284964480954*v_R5_a_r - 39.6392229058202*v_R5_b_i + 10.2713736253513*v_R5_b_r + 105.57176931318*v_R5_c_i - 54.0657727682604*v_R5_c_r - 39.6392229058201*v_R5_n_i + 10.2713736253514*v_R5_n_r + 58.9235503101833*v_R6_a_i - 45.7766907664085*v_R6_a_r + 88.4888280826461*v_R6_b_i - 18.6937810814837*v_R6_b_r - 242.095286143588*v_R6_c_i + 113.786109551688*v_R6_c_r + 88.4888280826459*v_R6_n_i - 18.6937810814837*v_R6_n_r - 24.9575997948692*v_R7_a_i + 23.284964480954*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r + 105.57176931318*v_R7_c_i - 54.0657727682604*v_R7_c_r - 39.6392229058201*v_R7_n_i + 10.2713736253514*v_R7_n_r
        struct[0].g[86,0] = i_R6_n_r + 1.84896616921897*v_R16_a_i - 9.21038227100566*v_R16_a_r + 0.793238195499527*v_R16_b_i - 9.00835072044485*v_R16_b_r + 1.84896616921897*v_R16_c_i - 9.21038227100566*v_R16_c_r + 5.65456401516768*v_R16_n_i + 30.9517475172273*v_R16_n_r - 10.2713736253513*v_R5_a_i - 39.6392229058202*v_R5_a_r - 23.284964480954*v_R5_b_i - 24.9575997948692*v_R5_b_r - 10.2713736253513*v_R5_c_i - 39.6392229058202*v_R5_c_r + 54.0657727682604*v_R5_n_i + 105.57176931318*v_R5_n_r + 18.6937810814837*v_R6_a_i + 88.488828082646*v_R6_a_r + 45.7766907664085*v_R6_b_i + 58.9235503101832*v_R6_b_r + 18.6937810814837*v_R6_c_i + 88.488828082646*v_R6_c_r - 113.786109551688*v_R6_n_i - 242.120286143588*v_R6_n_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r - 23.284964480954*v_R7_b_i - 24.9575997948692*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058202*v_R7_c_r + 54.0657727682604*v_R7_n_i + 105.57176931318*v_R7_n_r
        struct[0].g[87,0] = i_R6_n_i - 9.21038227100566*v_R16_a_i - 1.84896616921897*v_R16_a_r - 9.00835072044485*v_R16_b_i - 0.793238195499527*v_R16_b_r - 9.21038227100566*v_R16_c_i - 1.84896616921897*v_R16_c_r + 30.9517475172273*v_R16_n_i - 5.65456401516768*v_R16_n_r - 39.6392229058202*v_R5_a_i + 10.2713736253513*v_R5_a_r - 24.9575997948692*v_R5_b_i + 23.284964480954*v_R5_b_r - 39.6392229058202*v_R5_c_i + 10.2713736253513*v_R5_c_r + 105.57176931318*v_R5_n_i - 54.0657727682604*v_R5_n_r + 88.488828082646*v_R6_a_i - 18.6937810814837*v_R6_a_r + 58.9235503101832*v_R6_b_i - 45.7766907664085*v_R6_b_r + 88.488828082646*v_R6_c_i - 18.6937810814837*v_R6_c_r - 242.120286143588*v_R6_n_i + 113.786109551688*v_R6_n_r - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r - 24.9575997948692*v_R7_b_i + 23.284964480954*v_R7_b_r - 39.6392229058202*v_R7_c_i + 10.2713736253513*v_R7_c_r + 105.57176931318*v_R7_n_i - 54.0657727682604*v_R7_n_r
        struct[0].g[88,0] = i_R7_a_r + 54.0657727682604*v_R6_a_i + 105.57176931318*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r - 23.284964480954*v_R6_c_i - 24.9575997948692*v_R6_c_r - 10.2713736253513*v_R6_n_i - 39.6392229058202*v_R6_n_r - 108.131545536521*v_R7_a_i - 211.14353862636*v_R7_a_r + 20.5427472507027*v_R7_b_i + 79.2784458116403*v_R7_b_r + 46.569928961908*v_R7_c_i + 49.9151995897384*v_R7_c_r + 20.5427472507027*v_R7_n_i + 79.2784458116403*v_R7_n_r + 54.0657727682604*v_R8_a_i + 105.57176931318*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r - 23.284964480954*v_R8_c_i - 24.9575997948692*v_R8_c_r - 10.2713736253513*v_R8_n_i - 39.6392229058202*v_R8_n_r
        struct[0].g[89,0] = i_R7_a_i + 105.57176931318*v_R6_a_i - 54.0657727682604*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r - 24.9575997948692*v_R6_c_i + 23.284964480954*v_R6_c_r - 39.6392229058202*v_R6_n_i + 10.2713736253513*v_R6_n_r - 211.14353862636*v_R7_a_i + 108.131545536521*v_R7_a_r + 79.2784458116403*v_R7_b_i - 20.5427472507027*v_R7_b_r + 49.9151995897384*v_R7_c_i - 46.569928961908*v_R7_c_r + 79.2784458116403*v_R7_n_i - 20.5427472507027*v_R7_n_r + 105.57176931318*v_R8_a_i - 54.0657727682604*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r - 24.9575997948692*v_R8_c_i + 23.284964480954*v_R8_c_r - 39.6392229058202*v_R8_n_i + 10.2713736253513*v_R8_n_r
        struct[0].g[90,0] = i_R7_b_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r + 54.0657727682604*v_R6_b_i + 105.57176931318*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058201*v_R6_c_r - 23.284964480954*v_R6_n_i - 24.9575997948692*v_R6_n_r + 20.5427472507027*v_R7_a_i + 79.2784458116403*v_R7_a_r - 108.131545536521*v_R7_b_i - 211.14353862636*v_R7_b_r + 20.5427472507027*v_R7_c_i + 79.2784458116403*v_R7_c_r + 46.569928961908*v_R7_n_i + 49.9151995897384*v_R7_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r + 54.0657727682604*v_R8_b_i + 105.57176931318*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058201*v_R8_c_r - 23.284964480954*v_R8_n_i - 24.9575997948692*v_R8_n_r
        struct[0].g[91,0] = i_R7_b_i - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r + 105.57176931318*v_R6_b_i - 54.0657727682604*v_R6_b_r - 39.6392229058201*v_R6_c_i + 10.2713736253513*v_R6_c_r - 24.9575997948692*v_R6_n_i + 23.284964480954*v_R6_n_r + 79.2784458116403*v_R7_a_i - 20.5427472507027*v_R7_a_r - 211.14353862636*v_R7_b_i + 108.131545536521*v_R7_b_r + 79.2784458116403*v_R7_c_i - 20.5427472507027*v_R7_c_r + 49.9151995897384*v_R7_n_i - 46.569928961908*v_R7_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r + 105.57176931318*v_R8_b_i - 54.0657727682604*v_R8_b_r - 39.6392229058201*v_R8_c_i + 10.2713736253513*v_R8_c_r - 24.9575997948692*v_R8_n_i + 23.284964480954*v_R8_n_r
        struct[0].g[92,0] = i_R7_c_r - 23.284964480954*v_R6_a_i - 24.9575997948692*v_R6_a_r - 10.2713736253513*v_R6_b_i - 39.6392229058202*v_R6_b_r + 54.0657727682604*v_R6_c_i + 105.57176931318*v_R6_c_r - 10.2713736253514*v_R6_n_i - 39.6392229058201*v_R6_n_r + 46.569928961908*v_R7_a_i + 49.9151995897384*v_R7_a_r + 20.5427472507027*v_R7_b_i + 79.2784458116404*v_R7_b_r - 108.131545536521*v_R7_c_i - 211.14353862636*v_R7_c_r + 20.5427472507027*v_R7_n_i + 79.2784458116403*v_R7_n_r - 23.284964480954*v_R8_a_i - 24.9575997948692*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r + 54.0657727682604*v_R8_c_i + 105.57176931318*v_R8_c_r - 10.2713736253514*v_R8_n_i - 39.6392229058201*v_R8_n_r
        struct[0].g[93,0] = i_R7_c_i - 24.9575997948692*v_R6_a_i + 23.284964480954*v_R6_a_r - 39.6392229058202*v_R6_b_i + 10.2713736253513*v_R6_b_r + 105.57176931318*v_R6_c_i - 54.0657727682604*v_R6_c_r - 39.6392229058201*v_R6_n_i + 10.2713736253514*v_R6_n_r + 49.9151995897384*v_R7_a_i - 46.569928961908*v_R7_a_r + 79.2784458116404*v_R7_b_i - 20.5427472507027*v_R7_b_r - 211.14353862636*v_R7_c_i + 108.131545536521*v_R7_c_r + 79.2784458116403*v_R7_n_i - 20.5427472507027*v_R7_n_r - 24.9575997948692*v_R8_a_i + 23.284964480954*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r + 105.57176931318*v_R8_c_i - 54.0657727682604*v_R8_c_r - 39.6392229058201*v_R8_n_i + 10.2713736253514*v_R8_n_r
        struct[0].g[94,0] = i_R7_n_r - 10.2713736253513*v_R6_a_i - 39.6392229058202*v_R6_a_r - 23.284964480954*v_R6_b_i - 24.9575997948692*v_R6_b_r - 10.2713736253513*v_R6_c_i - 39.6392229058202*v_R6_c_r + 54.0657727682604*v_R6_n_i + 105.57176931318*v_R6_n_r + 20.5427472507027*v_R7_a_i + 79.2784458116403*v_R7_a_r + 46.569928961908*v_R7_b_i + 49.9151995897384*v_R7_b_r + 20.5427472507027*v_R7_c_i + 79.2784458116403*v_R7_c_r - 108.131545536521*v_R7_n_i - 211.14353862636*v_R7_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r - 23.284964480954*v_R8_b_i - 24.9575997948692*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058202*v_R8_c_r + 54.0657727682604*v_R8_n_i + 105.57176931318*v_R8_n_r
        struct[0].g[95,0] = i_R7_n_i - 39.6392229058202*v_R6_a_i + 10.2713736253513*v_R6_a_r - 24.9575997948692*v_R6_b_i + 23.284964480954*v_R6_b_r - 39.6392229058202*v_R6_c_i + 10.2713736253513*v_R6_c_r + 105.57176931318*v_R6_n_i - 54.0657727682604*v_R6_n_r + 79.2784458116403*v_R7_a_i - 20.5427472507027*v_R7_a_r + 49.9151995897384*v_R7_b_i - 46.569928961908*v_R7_b_r + 79.2784458116403*v_R7_c_i - 20.5427472507027*v_R7_c_r - 211.14353862636*v_R7_n_i + 108.131545536521*v_R7_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r - 24.9575997948692*v_R8_b_i + 23.284964480954*v_R8_b_r - 39.6392229058202*v_R8_c_i + 10.2713736253513*v_R8_c_r + 105.57176931318*v_R8_n_i - 54.0657727682604*v_R8_n_r
        struct[0].g[96,0] = i_R8_a_r + 54.0657727682604*v_R7_a_i + 105.57176931318*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r - 23.284964480954*v_R7_c_i - 24.9575997948692*v_R7_c_r - 10.2713736253513*v_R7_n_i - 39.6392229058202*v_R7_n_r - 108.131545536521*v_R8_a_i - 211.14353862636*v_R8_a_r + 20.5427472507027*v_R8_b_i + 79.2784458116403*v_R8_b_r + 46.569928961908*v_R8_c_i + 49.9151995897384*v_R8_c_r + 20.5427472507027*v_R8_n_i + 79.2784458116403*v_R8_n_r + 54.0657727682604*v_R9_a_i + 105.57176931318*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r - 23.284964480954*v_R9_c_i - 24.9575997948692*v_R9_c_r - 10.2713736253513*v_R9_n_i - 39.6392229058202*v_R9_n_r
        struct[0].g[97,0] = i_R8_a_i + 105.57176931318*v_R7_a_i - 54.0657727682604*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r - 24.9575997948692*v_R7_c_i + 23.284964480954*v_R7_c_r - 39.6392229058202*v_R7_n_i + 10.2713736253513*v_R7_n_r - 211.14353862636*v_R8_a_i + 108.131545536521*v_R8_a_r + 79.2784458116403*v_R8_b_i - 20.5427472507027*v_R8_b_r + 49.9151995897384*v_R8_c_i - 46.569928961908*v_R8_c_r + 79.2784458116403*v_R8_n_i - 20.5427472507027*v_R8_n_r + 105.57176931318*v_R9_a_i - 54.0657727682604*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r - 24.9575997948692*v_R9_c_i + 23.284964480954*v_R9_c_r - 39.6392229058202*v_R9_n_i + 10.2713736253513*v_R9_n_r
        struct[0].g[98,0] = i_R8_b_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r + 54.0657727682604*v_R7_b_i + 105.57176931318*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058201*v_R7_c_r - 23.284964480954*v_R7_n_i - 24.9575997948692*v_R7_n_r + 20.5427472507027*v_R8_a_i + 79.2784458116403*v_R8_a_r - 108.131545536521*v_R8_b_i - 211.14353862636*v_R8_b_r + 20.5427472507027*v_R8_c_i + 79.2784458116403*v_R8_c_r + 46.569928961908*v_R8_n_i + 49.9151995897384*v_R8_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r + 54.0657727682604*v_R9_b_i + 105.57176931318*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058201*v_R9_c_r - 23.284964480954*v_R9_n_i - 24.9575997948692*v_R9_n_r
        struct[0].g[99,0] = i_R8_b_i - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r + 105.57176931318*v_R7_b_i - 54.0657727682604*v_R7_b_r - 39.6392229058201*v_R7_c_i + 10.2713736253513*v_R7_c_r - 24.9575997948692*v_R7_n_i + 23.284964480954*v_R7_n_r + 79.2784458116403*v_R8_a_i - 20.5427472507027*v_R8_a_r - 211.14353862636*v_R8_b_i + 108.131545536521*v_R8_b_r + 79.2784458116403*v_R8_c_i - 20.5427472507027*v_R8_c_r + 49.9151995897384*v_R8_n_i - 46.569928961908*v_R8_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r + 105.57176931318*v_R9_b_i - 54.0657727682604*v_R9_b_r - 39.6392229058201*v_R9_c_i + 10.2713736253513*v_R9_c_r - 24.9575997948692*v_R9_n_i + 23.284964480954*v_R9_n_r
        struct[0].g[100,0] = i_R8_c_r - 23.284964480954*v_R7_a_i - 24.9575997948692*v_R7_a_r - 10.2713736253513*v_R7_b_i - 39.6392229058202*v_R7_b_r + 54.0657727682604*v_R7_c_i + 105.57176931318*v_R7_c_r - 10.2713736253514*v_R7_n_i - 39.6392229058201*v_R7_n_r + 46.569928961908*v_R8_a_i + 49.9151995897384*v_R8_a_r + 20.5427472507027*v_R8_b_i + 79.2784458116404*v_R8_b_r - 108.131545536521*v_R8_c_i - 211.14353862636*v_R8_c_r + 20.5427472507027*v_R8_n_i + 79.2784458116403*v_R8_n_r - 23.284964480954*v_R9_a_i - 24.9575997948692*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r + 54.0657727682604*v_R9_c_i + 105.57176931318*v_R9_c_r - 10.2713736253514*v_R9_n_i - 39.6392229058201*v_R9_n_r
        struct[0].g[101,0] = i_R8_c_i - 24.9575997948692*v_R7_a_i + 23.284964480954*v_R7_a_r - 39.6392229058202*v_R7_b_i + 10.2713736253513*v_R7_b_r + 105.57176931318*v_R7_c_i - 54.0657727682604*v_R7_c_r - 39.6392229058201*v_R7_n_i + 10.2713736253514*v_R7_n_r + 49.9151995897384*v_R8_a_i - 46.569928961908*v_R8_a_r + 79.2784458116404*v_R8_b_i - 20.5427472507027*v_R8_b_r - 211.14353862636*v_R8_c_i + 108.131545536521*v_R8_c_r + 79.2784458116403*v_R8_n_i - 20.5427472507027*v_R8_n_r - 24.9575997948692*v_R9_a_i + 23.284964480954*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r + 105.57176931318*v_R9_c_i - 54.0657727682604*v_R9_c_r - 39.6392229058201*v_R9_n_i + 10.2713736253514*v_R9_n_r
        struct[0].g[102,0] = i_R8_n_r - 10.2713736253513*v_R7_a_i - 39.6392229058202*v_R7_a_r - 23.284964480954*v_R7_b_i - 24.9575997948692*v_R7_b_r - 10.2713736253513*v_R7_c_i - 39.6392229058202*v_R7_c_r + 54.0657727682604*v_R7_n_i + 105.57176931318*v_R7_n_r + 20.5427472507027*v_R8_a_i + 79.2784458116403*v_R8_a_r + 46.569928961908*v_R8_b_i + 49.9151995897384*v_R8_b_r + 20.5427472507027*v_R8_c_i + 79.2784458116403*v_R8_c_r - 108.131545536521*v_R8_n_i - 211.16853862636*v_R8_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r - 23.284964480954*v_R9_b_i - 24.9575997948692*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058202*v_R9_c_r + 54.0657727682604*v_R9_n_i + 105.57176931318*v_R9_n_r
        struct[0].g[103,0] = i_R8_n_i - 39.6392229058202*v_R7_a_i + 10.2713736253513*v_R7_a_r - 24.9575997948692*v_R7_b_i + 23.284964480954*v_R7_b_r - 39.6392229058202*v_R7_c_i + 10.2713736253513*v_R7_c_r + 105.57176931318*v_R7_n_i - 54.0657727682604*v_R7_n_r + 79.2784458116403*v_R8_a_i - 20.5427472507027*v_R8_a_r + 49.9151995897384*v_R8_b_i - 46.569928961908*v_R8_b_r + 79.2784458116403*v_R8_c_i - 20.5427472507027*v_R8_c_r - 211.16853862636*v_R8_n_i + 108.131545536521*v_R8_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r - 24.9575997948692*v_R9_b_i + 23.284964480954*v_R9_b_r - 39.6392229058202*v_R9_c_i + 10.2713736253513*v_R9_c_r + 105.57176931318*v_R9_n_i - 54.0657727682604*v_R9_n_r
        struct[0].g[104,0] = i_R9_a_r + 54.0657727682604*v_R10_a_i + 105.57176931318*v_R10_a_r - 10.2713736253513*v_R10_b_i - 39.6392229058202*v_R10_b_r - 23.284964480954*v_R10_c_i - 24.9575997948692*v_R10_c_r - 10.2713736253513*v_R10_n_i - 39.6392229058202*v_R10_n_r + 5.65456401516768*v_R17_a_i + 30.9517475172273*v_R17_a_r + 1.84896616921897*v_R17_b_i - 9.21038227100566*v_R17_b_r + 0.793238195499529*v_R17_c_i - 9.00835072044485*v_R17_c_r + 1.84896616921897*v_R17_n_i - 9.21038227100566*v_R17_n_r + 54.0657727682604*v_R8_a_i + 105.57176931318*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r - 23.284964480954*v_R8_c_i - 24.9575997948692*v_R8_c_r - 10.2713736253513*v_R8_n_i - 39.6392229058202*v_R8_n_r - 113.786109551688*v_R9_a_i - 242.095286143588*v_R9_a_r + 18.6937810814837*v_R9_b_i + 88.488828082646*v_R9_b_r + 45.7766907664085*v_R9_c_i + 58.9235503101833*v_R9_c_r + 18.6937810814837*v_R9_n_i + 88.488828082646*v_R9_n_r
        struct[0].g[105,0] = i_R9_a_i + 105.57176931318*v_R10_a_i - 54.0657727682604*v_R10_a_r - 39.6392229058202*v_R10_b_i + 10.2713736253513*v_R10_b_r - 24.9575997948692*v_R10_c_i + 23.284964480954*v_R10_c_r - 39.6392229058202*v_R10_n_i + 10.2713736253513*v_R10_n_r + 30.9517475172273*v_R17_a_i - 5.65456401516768*v_R17_a_r - 9.21038227100566*v_R17_b_i - 1.84896616921897*v_R17_b_r - 9.00835072044485*v_R17_c_i - 0.793238195499529*v_R17_c_r - 9.21038227100566*v_R17_n_i - 1.84896616921897*v_R17_n_r + 105.57176931318*v_R8_a_i - 54.0657727682604*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r - 24.9575997948692*v_R8_c_i + 23.284964480954*v_R8_c_r - 39.6392229058202*v_R8_n_i + 10.2713736253513*v_R8_n_r - 242.095286143588*v_R9_a_i + 113.786109551688*v_R9_a_r + 88.488828082646*v_R9_b_i - 18.6937810814837*v_R9_b_r + 58.9235503101833*v_R9_c_i - 45.7766907664085*v_R9_c_r + 88.488828082646*v_R9_n_i - 18.6937810814837*v_R9_n_r
        struct[0].g[106,0] = i_R9_b_r - 10.2713736253513*v_R10_a_i - 39.6392229058202*v_R10_a_r + 54.0657727682604*v_R10_b_i + 105.57176931318*v_R10_b_r - 10.2713736253513*v_R10_c_i - 39.6392229058201*v_R10_c_r - 23.284964480954*v_R10_n_i - 24.9575997948692*v_R10_n_r + 1.84896616921897*v_R17_a_i - 9.21038227100566*v_R17_a_r + 5.65456401516768*v_R17_b_i + 30.9517475172273*v_R17_b_r + 1.84896616921897*v_R17_c_i - 9.21038227100566*v_R17_c_r + 0.793238195499528*v_R17_n_i - 9.00835072044485*v_R17_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r + 54.0657727682604*v_R8_b_i + 105.57176931318*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058201*v_R8_c_r - 23.284964480954*v_R8_n_i - 24.9575997948692*v_R8_n_r + 18.6937810814837*v_R9_a_i + 88.488828082646*v_R9_a_r - 113.786109551688*v_R9_b_i - 242.095286143588*v_R9_b_r + 18.6937810814837*v_R9_c_i + 88.488828082646*v_R9_c_r + 45.7766907664085*v_R9_n_i + 58.9235503101833*v_R9_n_r
        struct[0].g[107,0] = i_R9_b_i - 39.6392229058202*v_R10_a_i + 10.2713736253513*v_R10_a_r + 105.57176931318*v_R10_b_i - 54.0657727682604*v_R10_b_r - 39.6392229058201*v_R10_c_i + 10.2713736253513*v_R10_c_r - 24.9575997948692*v_R10_n_i + 23.284964480954*v_R10_n_r - 9.21038227100566*v_R17_a_i - 1.84896616921897*v_R17_a_r + 30.9517475172273*v_R17_b_i - 5.65456401516768*v_R17_b_r - 9.21038227100566*v_R17_c_i - 1.84896616921897*v_R17_c_r - 9.00835072044485*v_R17_n_i - 0.793238195499528*v_R17_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r + 105.57176931318*v_R8_b_i - 54.0657727682604*v_R8_b_r - 39.6392229058201*v_R8_c_i + 10.2713736253513*v_R8_c_r - 24.9575997948692*v_R8_n_i + 23.284964480954*v_R8_n_r + 88.488828082646*v_R9_a_i - 18.6937810814837*v_R9_a_r - 242.095286143588*v_R9_b_i + 113.786109551688*v_R9_b_r + 88.488828082646*v_R9_c_i - 18.6937810814837*v_R9_c_r + 58.9235503101833*v_R9_n_i - 45.7766907664085*v_R9_n_r
        struct[0].g[108,0] = i_R9_c_r - 23.284964480954*v_R10_a_i - 24.9575997948692*v_R10_a_r - 10.2713736253513*v_R10_b_i - 39.6392229058202*v_R10_b_r + 54.0657727682604*v_R10_c_i + 105.57176931318*v_R10_c_r - 10.2713736253514*v_R10_n_i - 39.6392229058201*v_R10_n_r + 0.793238195499527*v_R17_a_i - 9.00835072044484*v_R17_a_r + 1.84896616921897*v_R17_b_i - 9.21038227100566*v_R17_b_r + 5.65456401516768*v_R17_c_i + 30.9517475172273*v_R17_c_r + 1.84896616921897*v_R17_n_i - 9.21038227100566*v_R17_n_r - 23.284964480954*v_R8_a_i - 24.9575997948692*v_R8_a_r - 10.2713736253513*v_R8_b_i - 39.6392229058202*v_R8_b_r + 54.0657727682604*v_R8_c_i + 105.57176931318*v_R8_c_r - 10.2713736253514*v_R8_n_i - 39.6392229058201*v_R8_n_r + 45.7766907664085*v_R9_a_i + 58.9235503101833*v_R9_a_r + 18.6937810814837*v_R9_b_i + 88.4888280826461*v_R9_b_r - 113.786109551688*v_R9_c_i - 242.095286143588*v_R9_c_r + 18.6937810814837*v_R9_n_i + 88.4888280826459*v_R9_n_r
        struct[0].g[109,0] = i_R9_c_i - 24.9575997948692*v_R10_a_i + 23.284964480954*v_R10_a_r - 39.6392229058202*v_R10_b_i + 10.2713736253513*v_R10_b_r + 105.57176931318*v_R10_c_i - 54.0657727682604*v_R10_c_r - 39.6392229058201*v_R10_n_i + 10.2713736253514*v_R10_n_r - 9.00835072044484*v_R17_a_i - 0.793238195499527*v_R17_a_r - 9.21038227100566*v_R17_b_i - 1.84896616921897*v_R17_b_r + 30.9517475172273*v_R17_c_i - 5.65456401516768*v_R17_c_r - 9.21038227100566*v_R17_n_i - 1.84896616921897*v_R17_n_r - 24.9575997948692*v_R8_a_i + 23.284964480954*v_R8_a_r - 39.6392229058202*v_R8_b_i + 10.2713736253513*v_R8_b_r + 105.57176931318*v_R8_c_i - 54.0657727682604*v_R8_c_r - 39.6392229058201*v_R8_n_i + 10.2713736253514*v_R8_n_r + 58.9235503101833*v_R9_a_i - 45.7766907664085*v_R9_a_r + 88.4888280826461*v_R9_b_i - 18.6937810814837*v_R9_b_r - 242.095286143588*v_R9_c_i + 113.786109551688*v_R9_c_r + 88.4888280826459*v_R9_n_i - 18.6937810814837*v_R9_n_r
        struct[0].g[110,0] = i_R9_n_r - 10.2713736253513*v_R10_a_i - 39.6392229058202*v_R10_a_r - 23.284964480954*v_R10_b_i - 24.9575997948692*v_R10_b_r - 10.2713736253513*v_R10_c_i - 39.6392229058202*v_R10_c_r + 54.0657727682604*v_R10_n_i + 105.57176931318*v_R10_n_r + 1.84896616921897*v_R17_a_i - 9.21038227100566*v_R17_a_r + 0.793238195499527*v_R17_b_i - 9.00835072044485*v_R17_b_r + 1.84896616921897*v_R17_c_i - 9.21038227100566*v_R17_c_r + 5.65456401516768*v_R17_n_i + 30.9517475172273*v_R17_n_r - 10.2713736253513*v_R8_a_i - 39.6392229058202*v_R8_a_r - 23.284964480954*v_R8_b_i - 24.9575997948692*v_R8_b_r - 10.2713736253513*v_R8_c_i - 39.6392229058202*v_R8_c_r + 54.0657727682604*v_R8_n_i + 105.57176931318*v_R8_n_r + 18.6937810814837*v_R9_a_i + 88.488828082646*v_R9_a_r + 45.7766907664085*v_R9_b_i + 58.9235503101832*v_R9_b_r + 18.6937810814837*v_R9_c_i + 88.488828082646*v_R9_c_r - 113.786109551688*v_R9_n_i - 242.095286143588*v_R9_n_r
        struct[0].g[111,0] = i_R9_n_i - 39.6392229058202*v_R10_a_i + 10.2713736253513*v_R10_a_r - 24.9575997948692*v_R10_b_i + 23.284964480954*v_R10_b_r - 39.6392229058202*v_R10_c_i + 10.2713736253513*v_R10_c_r + 105.57176931318*v_R10_n_i - 54.0657727682604*v_R10_n_r - 9.21038227100566*v_R17_a_i - 1.84896616921897*v_R17_a_r - 9.00835072044485*v_R17_b_i - 0.793238195499527*v_R17_b_r - 9.21038227100566*v_R17_c_i - 1.84896616921897*v_R17_c_r + 30.9517475172273*v_R17_n_i - 5.65456401516768*v_R17_n_r - 39.6392229058202*v_R8_a_i + 10.2713736253513*v_R8_a_r - 24.9575997948692*v_R8_b_i + 23.284964480954*v_R8_b_r - 39.6392229058202*v_R8_c_i + 10.2713736253513*v_R8_c_r + 105.57176931318*v_R8_n_i - 54.0657727682604*v_R8_n_r + 88.488828082646*v_R9_a_i - 18.6937810814837*v_R9_a_r + 58.9235503101832*v_R9_b_i - 45.7766907664085*v_R9_b_r + 88.488828082646*v_R9_c_i - 18.6937810814837*v_R9_c_r - 242.095286143588*v_R9_n_i + 113.786109551688*v_R9_n_r
        struct[0].g[112,0] = i_R10_a_r - 59.7203367834281*v_R10_a_i - 136.523516830407*v_R10_a_r + 8.42240745613237*v_R10_b_i + 48.8496051768258*v_R10_b_r + 22.4917262854545*v_R10_c_i + 33.9659505153141*v_R10_c_r + 8.42240745613237*v_R10_n_i + 48.8496051768258*v_R10_n_r + 5.65456401516768*v_R18_a_i + 30.9517475172273*v_R18_a_r + 1.84896616921897*v_R18_b_i - 9.21038227100566*v_R18_b_r + 0.793238195499529*v_R18_c_i - 9.00835072044485*v_R18_c_r + 1.84896616921897*v_R18_n_i - 9.21038227100566*v_R18_n_r + 54.0657727682604*v_R9_a_i + 105.57176931318*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r - 23.284964480954*v_R9_c_i - 24.9575997948692*v_R9_c_r - 10.2713736253513*v_R9_n_i - 39.6392229058202*v_R9_n_r
        struct[0].g[113,0] = i_R10_a_i - 136.523516830407*v_R10_a_i + 59.7203367834281*v_R10_a_r + 48.8496051768258*v_R10_b_i - 8.42240745613237*v_R10_b_r + 33.9659505153141*v_R10_c_i - 22.4917262854545*v_R10_c_r + 48.8496051768258*v_R10_n_i - 8.42240745613237*v_R10_n_r + 30.9517475172273*v_R18_a_i - 5.65456401516768*v_R18_a_r - 9.21038227100566*v_R18_b_i - 1.84896616921897*v_R18_b_r - 9.00835072044485*v_R18_c_i - 0.793238195499529*v_R18_c_r - 9.21038227100566*v_R18_n_i - 1.84896616921897*v_R18_n_r + 105.57176931318*v_R9_a_i - 54.0657727682604*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r - 24.9575997948692*v_R9_c_i + 23.284964480954*v_R9_c_r - 39.6392229058202*v_R9_n_i + 10.2713736253513*v_R9_n_r
        struct[0].g[114,0] = i_R10_b_r + 8.42240745613237*v_R10_a_i + 48.8496051768258*v_R10_a_r - 59.720336783428*v_R10_b_i - 136.523516830407*v_R10_b_r + 8.42240745613238*v_R10_c_i + 48.8496051768258*v_R10_c_r + 22.4917262854545*v_R10_n_i + 33.9659505153141*v_R10_n_r + 1.84896616921897*v_R18_a_i - 9.21038227100566*v_R18_a_r + 5.65456401516768*v_R18_b_i + 30.9517475172273*v_R18_b_r + 1.84896616921897*v_R18_c_i - 9.21038227100566*v_R18_c_r + 0.793238195499528*v_R18_n_i - 9.00835072044485*v_R18_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r + 54.0657727682604*v_R9_b_i + 105.57176931318*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058201*v_R9_c_r - 23.284964480954*v_R9_n_i - 24.9575997948692*v_R9_n_r
        struct[0].g[115,0] = i_R10_b_i + 48.8496051768258*v_R10_a_i - 8.42240745613237*v_R10_a_r - 136.523516830407*v_R10_b_i + 59.720336783428*v_R10_b_r + 48.8496051768258*v_R10_c_i - 8.42240745613238*v_R10_c_r + 33.9659505153141*v_R10_n_i - 22.4917262854545*v_R10_n_r - 9.21038227100566*v_R18_a_i - 1.84896616921897*v_R18_a_r + 30.9517475172273*v_R18_b_i - 5.65456401516768*v_R18_b_r - 9.21038227100566*v_R18_c_i - 1.84896616921897*v_R18_c_r - 9.00835072044485*v_R18_n_i - 0.793238195499528*v_R18_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r + 105.57176931318*v_R9_b_i - 54.0657727682604*v_R9_b_r - 39.6392229058201*v_R9_c_i + 10.2713736253513*v_R9_c_r - 24.9575997948692*v_R9_n_i + 23.284964480954*v_R9_n_r
        struct[0].g[116,0] = i_R10_c_r + 22.4917262854545*v_R10_a_i + 33.9659505153141*v_R10_a_r + 8.42240745613236*v_R10_b_i + 48.8496051768259*v_R10_b_r - 59.7203367834281*v_R10_c_i - 136.523516830407*v_R10_c_r + 8.42240745613239*v_R10_n_i + 48.8496051768258*v_R10_n_r + 0.793238195499527*v_R18_a_i - 9.00835072044484*v_R18_a_r + 1.84896616921897*v_R18_b_i - 9.21038227100566*v_R18_b_r + 5.65456401516768*v_R18_c_i + 30.9517475172273*v_R18_c_r + 1.84896616921897*v_R18_n_i - 9.21038227100566*v_R18_n_r - 23.284964480954*v_R9_a_i - 24.9575997948692*v_R9_a_r - 10.2713736253513*v_R9_b_i - 39.6392229058202*v_R9_b_r + 54.0657727682604*v_R9_c_i + 105.57176931318*v_R9_c_r - 10.2713736253514*v_R9_n_i - 39.6392229058201*v_R9_n_r
        struct[0].g[117,0] = i_R10_c_i + 33.9659505153141*v_R10_a_i - 22.4917262854545*v_R10_a_r + 48.8496051768259*v_R10_b_i - 8.42240745613236*v_R10_b_r - 136.523516830407*v_R10_c_i + 59.7203367834281*v_R10_c_r + 48.8496051768258*v_R10_n_i - 8.42240745613239*v_R10_n_r - 9.00835072044484*v_R18_a_i - 0.793238195499527*v_R18_a_r - 9.21038227100566*v_R18_b_i - 1.84896616921897*v_R18_b_r + 30.9517475172273*v_R18_c_i - 5.65456401516768*v_R18_c_r - 9.21038227100566*v_R18_n_i - 1.84896616921897*v_R18_n_r - 24.9575997948692*v_R9_a_i + 23.284964480954*v_R9_a_r - 39.6392229058202*v_R9_b_i + 10.2713736253513*v_R9_b_r + 105.57176931318*v_R9_c_i - 54.0657727682604*v_R9_c_r - 39.6392229058201*v_R9_n_i + 10.2713736253514*v_R9_n_r
        struct[0].g[118,0] = i_R10_n_r + 8.42240745613237*v_R10_a_i + 48.8496051768258*v_R10_a_r + 22.4917262854545*v_R10_b_i + 33.965950515314*v_R10_b_r + 8.42240745613237*v_R10_c_i + 48.8496051768258*v_R10_c_r - 59.7203367834281*v_R10_n_i - 136.548516830407*v_R10_n_r + 1.84896616921897*v_R18_a_i - 9.21038227100566*v_R18_a_r + 0.793238195499527*v_R18_b_i - 9.00835072044485*v_R18_b_r + 1.84896616921897*v_R18_c_i - 9.21038227100566*v_R18_c_r + 5.65456401516768*v_R18_n_i + 30.9517475172273*v_R18_n_r - 10.2713736253513*v_R9_a_i - 39.6392229058202*v_R9_a_r - 23.284964480954*v_R9_b_i - 24.9575997948692*v_R9_b_r - 10.2713736253513*v_R9_c_i - 39.6392229058202*v_R9_c_r + 54.0657727682604*v_R9_n_i + 105.57176931318*v_R9_n_r
        struct[0].g[119,0] = i_R10_n_i + 48.8496051768258*v_R10_a_i - 8.42240745613237*v_R10_a_r + 33.965950515314*v_R10_b_i - 22.4917262854545*v_R10_b_r + 48.8496051768258*v_R10_c_i - 8.42240745613237*v_R10_c_r - 136.548516830407*v_R10_n_i + 59.7203367834281*v_R10_n_r - 9.21038227100566*v_R18_a_i - 1.84896616921897*v_R18_a_r - 9.00835072044485*v_R18_b_i - 0.793238195499527*v_R18_b_r - 9.21038227100566*v_R18_c_i - 1.84896616921897*v_R18_c_r + 30.9517475172273*v_R18_n_i - 5.65456401516768*v_R18_n_r - 39.6392229058202*v_R9_a_i + 10.2713736253513*v_R9_a_r - 24.9575997948692*v_R9_b_i + 23.284964480954*v_R9_b_r - 39.6392229058202*v_R9_c_i + 10.2713736253513*v_R9_c_r + 105.57176931318*v_R9_n_i - 54.0657727682604*v_R9_n_r
        struct[0].g[120,0] = i_R12_a_r - 9.69353831171601*v_R12_a_i - 53.0601386009611*v_R12_a_r - 3.16965629008966*v_R12_b_i + 15.7892267502954*v_R12_b_r - 1.35983690657062*v_R12_c_i + 15.442886949334*v_R12_c_r - 3.16965629008966*v_R12_n_i + 15.7892267502954*v_R12_n_r + 4.84676915585801*v_R13_a_i + 26.5300693004806*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 0.67991845328531*v_R13_c_i - 7.72144347466701*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r + 4.84676915585801*v_R4_a_i + 26.5300693004806*v_R4_a_r + 1.58482814504483*v_R4_b_i - 7.8946133751477*v_R4_b_r + 0.67991845328531*v_R4_c_i - 7.72144347466701*v_R4_c_r + 1.58482814504483*v_R4_n_i - 7.89461337514771*v_R4_n_r
        struct[0].g[121,0] = i_R12_a_i - 53.0601386009611*v_R12_a_i + 9.69353831171601*v_R12_a_r + 15.7892267502954*v_R12_b_i + 3.16965629008966*v_R12_b_r + 15.442886949334*v_R12_c_i + 1.35983690657062*v_R12_c_r + 15.7892267502954*v_R12_n_i + 3.16965629008966*v_R12_n_r + 26.5300693004806*v_R13_a_i - 4.84676915585801*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r - 7.72144347466701*v_R13_c_i - 0.67991845328531*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r + 26.5300693004806*v_R4_a_i - 4.84676915585801*v_R4_a_r - 7.8946133751477*v_R4_b_i - 1.58482814504483*v_R4_b_r - 7.72144347466701*v_R4_c_i - 0.67991845328531*v_R4_c_r - 7.89461337514771*v_R4_n_i - 1.58482814504483*v_R4_n_r
        struct[0].g[122,0] = i_R12_b_r - 3.16965629008966*v_R12_a_i + 15.7892267502954*v_R12_a_r - 9.69353831171602*v_R12_b_i - 53.0601386009611*v_R12_b_r - 3.16965629008967*v_R12_c_i + 15.7892267502954*v_R12_c_r - 1.35983690657062*v_R12_n_i + 15.442886949334*v_R12_n_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 4.84676915585801*v_R13_b_i + 26.5300693004806*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 0.679918453285308*v_R13_n_i - 7.72144347466701*v_R13_n_r + 1.58482814504483*v_R4_a_i - 7.89461337514771*v_R4_a_r + 4.84676915585801*v_R4_b_i + 26.5300693004806*v_R4_b_r + 1.58482814504483*v_R4_c_i - 7.89461337514771*v_R4_c_r + 0.679918453285308*v_R4_n_i - 7.72144347466701*v_R4_n_r
        struct[0].g[123,0] = i_R12_b_i + 15.7892267502954*v_R12_a_i + 3.16965629008966*v_R12_a_r - 53.0601386009611*v_R12_b_i + 9.69353831171602*v_R12_b_r + 15.7892267502954*v_R12_c_i + 3.16965629008967*v_R12_c_r + 15.442886949334*v_R12_n_i + 1.35983690657062*v_R12_n_r - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r + 26.5300693004806*v_R13_b_i - 4.84676915585801*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r - 7.72144347466701*v_R13_n_i - 0.679918453285308*v_R13_n_r - 7.89461337514771*v_R4_a_i - 1.58482814504483*v_R4_a_r + 26.5300693004806*v_R4_b_i - 4.84676915585801*v_R4_b_r - 7.89461337514771*v_R4_c_i - 1.58482814504483*v_R4_c_r - 7.72144347466701*v_R4_n_i - 0.679918453285308*v_R4_n_r
        struct[0].g[124,0] = i_R12_c_r - 1.35983690657062*v_R12_a_i + 15.442886949334*v_R12_a_r - 3.16965629008966*v_R12_b_i + 15.7892267502954*v_R12_b_r - 9.69353831171602*v_R12_c_i - 53.0601386009611*v_R12_c_r - 3.16965629008967*v_R12_n_i + 15.7892267502954*v_R12_n_r + 0.67991845328531*v_R13_a_i - 7.72144347466701*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 4.84676915585801*v_R13_c_i + 26.5300693004806*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r + 0.67991845328531*v_R4_a_i - 7.72144347466701*v_R4_a_r + 1.58482814504483*v_R4_b_i - 7.8946133751477*v_R4_b_r + 4.84676915585801*v_R4_c_i + 26.5300693004806*v_R4_c_r + 1.58482814504483*v_R4_n_i - 7.89461337514771*v_R4_n_r
        struct[0].g[125,0] = i_R12_c_i + 15.442886949334*v_R12_a_i + 1.35983690657062*v_R12_a_r + 15.7892267502954*v_R12_b_i + 3.16965629008966*v_R12_b_r - 53.0601386009611*v_R12_c_i + 9.69353831171602*v_R12_c_r + 15.7892267502954*v_R12_n_i + 3.16965629008967*v_R12_n_r - 7.72144347466701*v_R13_a_i - 0.67991845328531*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r + 26.5300693004806*v_R13_c_i - 4.84676915585801*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r - 7.72144347466701*v_R4_a_i - 0.67991845328531*v_R4_a_r - 7.8946133751477*v_R4_b_i - 1.58482814504483*v_R4_b_r + 26.5300693004806*v_R4_c_i - 4.84676915585801*v_R4_c_r - 7.89461337514771*v_R4_n_i - 1.58482814504483*v_R4_n_r
        struct[0].g[126,0] = i_R12_n_r - 3.16965629008966*v_R12_a_i + 15.7892267502954*v_R12_a_r - 1.35983690657062*v_R12_b_i + 15.442886949334*v_R12_b_r - 3.16965629008966*v_R12_c_i + 15.7892267502954*v_R12_c_r - 9.69353831171602*v_R12_n_i - 53.0601386009611*v_R12_n_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 0.679918453285308*v_R13_b_i - 7.72144347466701*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 4.84676915585801*v_R13_n_i + 26.5300693004806*v_R13_n_r + 1.58482814504483*v_R4_a_i - 7.89461337514771*v_R4_a_r + 0.679918453285308*v_R4_b_i - 7.72144347466701*v_R4_b_r + 1.58482814504483*v_R4_c_i - 7.89461337514771*v_R4_c_r + 4.84676915585801*v_R4_n_i + 26.5300693004806*v_R4_n_r
        struct[0].g[127,0] = i_R12_n_i + 15.7892267502954*v_R12_a_i + 3.16965629008966*v_R12_a_r + 15.442886949334*v_R12_b_i + 1.35983690657062*v_R12_b_r + 15.7892267502954*v_R12_c_i + 3.16965629008966*v_R12_c_r - 53.0601386009611*v_R12_n_i + 9.69353831171602*v_R12_n_r - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r - 7.72144347466701*v_R13_b_i - 0.679918453285308*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r + 26.5300693004806*v_R13_n_i - 4.84676915585801*v_R13_n_r - 7.89461337514771*v_R4_a_i - 1.58482814504483*v_R4_a_r - 7.72144347466701*v_R4_b_i - 0.679918453285308*v_R4_b_r - 7.89461337514771*v_R4_c_i - 1.58482814504483*v_R4_c_r + 26.5300693004806*v_R4_n_i - 4.84676915585801*v_R4_n_r
        struct[0].g[128,0] = i_R13_a_r + 4.84676915585801*v_R12_a_i + 26.5300693004806*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 0.67991845328531*v_R12_c_i - 7.72144347466701*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r - 9.69353831171601*v_R13_a_i - 53.0601386009611*v_R13_a_r - 3.16965629008966*v_R13_b_i + 15.7892267502954*v_R13_b_r - 1.35983690657062*v_R13_c_i + 15.442886949334*v_R13_c_r - 3.16965629008966*v_R13_n_i + 15.7892267502954*v_R13_n_r + 4.84676915585801*v_R14_a_i + 26.5300693004806*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 0.67991845328531*v_R14_c_i - 7.72144347466701*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r
        struct[0].g[129,0] = i_R13_a_i + 26.5300693004806*v_R12_a_i - 4.84676915585801*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r - 7.72144347466701*v_R12_c_i - 0.67991845328531*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r - 53.0601386009611*v_R13_a_i + 9.69353831171601*v_R13_a_r + 15.7892267502954*v_R13_b_i + 3.16965629008966*v_R13_b_r + 15.442886949334*v_R13_c_i + 1.35983690657062*v_R13_c_r + 15.7892267502954*v_R13_n_i + 3.16965629008966*v_R13_n_r + 26.5300693004806*v_R14_a_i - 4.84676915585801*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r - 7.72144347466701*v_R14_c_i - 0.67991845328531*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r
        struct[0].g[130,0] = i_R13_b_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 4.84676915585801*v_R12_b_i + 26.5300693004806*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 0.679918453285308*v_R12_n_i - 7.72144347466701*v_R12_n_r - 3.16965629008966*v_R13_a_i + 15.7892267502954*v_R13_a_r - 9.69353831171602*v_R13_b_i - 53.0601386009611*v_R13_b_r - 3.16965629008967*v_R13_c_i + 15.7892267502954*v_R13_c_r - 1.35983690657062*v_R13_n_i + 15.442886949334*v_R13_n_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 4.84676915585801*v_R14_b_i + 26.5300693004806*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 0.679918453285308*v_R14_n_i - 7.72144347466701*v_R14_n_r
        struct[0].g[131,0] = i_R13_b_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r + 26.5300693004806*v_R12_b_i - 4.84676915585801*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r - 7.72144347466701*v_R12_n_i - 0.679918453285308*v_R12_n_r + 15.7892267502954*v_R13_a_i + 3.16965629008966*v_R13_a_r - 53.0601386009611*v_R13_b_i + 9.69353831171602*v_R13_b_r + 15.7892267502954*v_R13_c_i + 3.16965629008967*v_R13_c_r + 15.442886949334*v_R13_n_i + 1.35983690657062*v_R13_n_r - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r + 26.5300693004806*v_R14_b_i - 4.84676915585801*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r - 7.72144347466701*v_R14_n_i - 0.679918453285308*v_R14_n_r
        struct[0].g[132,0] = i_R13_c_r + 0.67991845328531*v_R12_a_i - 7.72144347466701*v_R12_a_r + 1.58482814504483*v_R12_b_i - 7.8946133751477*v_R12_b_r + 4.84676915585801*v_R12_c_i + 26.5300693004806*v_R12_c_r + 1.58482814504483*v_R12_n_i - 7.89461337514771*v_R12_n_r - 1.35983690657062*v_R13_a_i + 15.442886949334*v_R13_a_r - 3.16965629008966*v_R13_b_i + 15.7892267502954*v_R13_b_r - 9.69353831171602*v_R13_c_i - 53.0601386009611*v_R13_c_r - 3.16965629008967*v_R13_n_i + 15.7892267502954*v_R13_n_r + 0.67991845328531*v_R14_a_i - 7.72144347466701*v_R14_a_r + 1.58482814504483*v_R14_b_i - 7.8946133751477*v_R14_b_r + 4.84676915585801*v_R14_c_i + 26.5300693004806*v_R14_c_r + 1.58482814504483*v_R14_n_i - 7.89461337514771*v_R14_n_r
        struct[0].g[133,0] = i_R13_c_i - 7.72144347466701*v_R12_a_i - 0.67991845328531*v_R12_a_r - 7.8946133751477*v_R12_b_i - 1.58482814504483*v_R12_b_r + 26.5300693004806*v_R12_c_i - 4.84676915585801*v_R12_c_r - 7.89461337514771*v_R12_n_i - 1.58482814504483*v_R12_n_r + 15.442886949334*v_R13_a_i + 1.35983690657062*v_R13_a_r + 15.7892267502954*v_R13_b_i + 3.16965629008966*v_R13_b_r - 53.0601386009611*v_R13_c_i + 9.69353831171602*v_R13_c_r + 15.7892267502954*v_R13_n_i + 3.16965629008967*v_R13_n_r - 7.72144347466701*v_R14_a_i - 0.67991845328531*v_R14_a_r - 7.8946133751477*v_R14_b_i - 1.58482814504483*v_R14_b_r + 26.5300693004806*v_R14_c_i - 4.84676915585801*v_R14_c_r - 7.89461337514771*v_R14_n_i - 1.58482814504483*v_R14_n_r
        struct[0].g[134,0] = i_R13_n_r + 1.58482814504483*v_R12_a_i - 7.89461337514771*v_R12_a_r + 0.679918453285308*v_R12_b_i - 7.72144347466701*v_R12_b_r + 1.58482814504483*v_R12_c_i - 7.89461337514771*v_R12_c_r + 4.84676915585801*v_R12_n_i + 26.5300693004806*v_R12_n_r - 3.16965629008966*v_R13_a_i + 15.7892267502954*v_R13_a_r - 1.35983690657062*v_R13_b_i + 15.442886949334*v_R13_b_r - 3.16965629008966*v_R13_c_i + 15.7892267502954*v_R13_c_r - 9.69353831171602*v_R13_n_i - 53.0851386009611*v_R13_n_r + 1.58482814504483*v_R14_a_i - 7.89461337514771*v_R14_a_r + 0.679918453285308*v_R14_b_i - 7.72144347466701*v_R14_b_r + 1.58482814504483*v_R14_c_i - 7.89461337514771*v_R14_c_r + 4.84676915585801*v_R14_n_i + 26.5300693004806*v_R14_n_r
        struct[0].g[135,0] = i_R13_n_i - 7.89461337514771*v_R12_a_i - 1.58482814504483*v_R12_a_r - 7.72144347466701*v_R12_b_i - 0.679918453285308*v_R12_b_r - 7.89461337514771*v_R12_c_i - 1.58482814504483*v_R12_c_r + 26.5300693004806*v_R12_n_i - 4.84676915585801*v_R12_n_r + 15.7892267502954*v_R13_a_i + 3.16965629008966*v_R13_a_r + 15.442886949334*v_R13_b_i + 1.35983690657062*v_R13_b_r + 15.7892267502954*v_R13_c_i + 3.16965629008966*v_R13_c_r - 53.0851386009611*v_R13_n_i + 9.69353831171602*v_R13_n_r - 7.89461337514771*v_R14_a_i - 1.58482814504483*v_R14_a_r - 7.72144347466701*v_R14_b_i - 0.679918453285308*v_R14_b_r - 7.89461337514771*v_R14_c_i - 1.58482814504483*v_R14_c_r + 26.5300693004806*v_R14_n_i - 4.84676915585801*v_R14_n_r
        struct[0].g[136,0] = i_R14_a_r + 4.84676915585801*v_R13_a_i + 26.5300693004806*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 0.67991845328531*v_R13_c_i - 7.72144347466701*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r - 9.69353831171601*v_R14_a_i - 53.0601386009611*v_R14_a_r - 3.16965629008966*v_R14_b_i + 15.7892267502954*v_R14_b_r - 1.35983690657062*v_R14_c_i + 15.442886949334*v_R14_c_r - 3.16965629008966*v_R14_n_i + 15.7892267502954*v_R14_n_r + 4.84676915585801*v_R15_a_i + 26.5300693004806*v_R15_a_r + 1.58482814504483*v_R15_b_i - 7.8946133751477*v_R15_b_r + 0.67991845328531*v_R15_c_i - 7.72144347466701*v_R15_c_r + 1.58482814504483*v_R15_n_i - 7.89461337514771*v_R15_n_r
        struct[0].g[137,0] = i_R14_a_i + 26.5300693004806*v_R13_a_i - 4.84676915585801*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r - 7.72144347466701*v_R13_c_i - 0.67991845328531*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r - 53.0601386009611*v_R14_a_i + 9.69353831171601*v_R14_a_r + 15.7892267502954*v_R14_b_i + 3.16965629008966*v_R14_b_r + 15.442886949334*v_R14_c_i + 1.35983690657062*v_R14_c_r + 15.7892267502954*v_R14_n_i + 3.16965629008966*v_R14_n_r + 26.5300693004806*v_R15_a_i - 4.84676915585801*v_R15_a_r - 7.8946133751477*v_R15_b_i - 1.58482814504483*v_R15_b_r - 7.72144347466701*v_R15_c_i - 0.67991845328531*v_R15_c_r - 7.89461337514771*v_R15_n_i - 1.58482814504483*v_R15_n_r
        struct[0].g[138,0] = i_R14_b_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 4.84676915585801*v_R13_b_i + 26.5300693004806*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 0.679918453285308*v_R13_n_i - 7.72144347466701*v_R13_n_r - 3.16965629008966*v_R14_a_i + 15.7892267502954*v_R14_a_r - 9.69353831171602*v_R14_b_i - 53.0601386009611*v_R14_b_r - 3.16965629008967*v_R14_c_i + 15.7892267502954*v_R14_c_r - 1.35983690657062*v_R14_n_i + 15.442886949334*v_R14_n_r + 1.58482814504483*v_R15_a_i - 7.89461337514771*v_R15_a_r + 4.84676915585801*v_R15_b_i + 26.5300693004806*v_R15_b_r + 1.58482814504483*v_R15_c_i - 7.89461337514771*v_R15_c_r + 0.679918453285308*v_R15_n_i - 7.72144347466701*v_R15_n_r
        struct[0].g[139,0] = i_R14_b_i - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r + 26.5300693004806*v_R13_b_i - 4.84676915585801*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r - 7.72144347466701*v_R13_n_i - 0.679918453285308*v_R13_n_r + 15.7892267502954*v_R14_a_i + 3.16965629008966*v_R14_a_r - 53.0601386009611*v_R14_b_i + 9.69353831171602*v_R14_b_r + 15.7892267502954*v_R14_c_i + 3.16965629008967*v_R14_c_r + 15.442886949334*v_R14_n_i + 1.35983690657062*v_R14_n_r - 7.89461337514771*v_R15_a_i - 1.58482814504483*v_R15_a_r + 26.5300693004806*v_R15_b_i - 4.84676915585801*v_R15_b_r - 7.89461337514771*v_R15_c_i - 1.58482814504483*v_R15_c_r - 7.72144347466701*v_R15_n_i - 0.679918453285308*v_R15_n_r
        struct[0].g[140,0] = i_R14_c_r + 0.67991845328531*v_R13_a_i - 7.72144347466701*v_R13_a_r + 1.58482814504483*v_R13_b_i - 7.8946133751477*v_R13_b_r + 4.84676915585801*v_R13_c_i + 26.5300693004806*v_R13_c_r + 1.58482814504483*v_R13_n_i - 7.89461337514771*v_R13_n_r - 1.35983690657062*v_R14_a_i + 15.442886949334*v_R14_a_r - 3.16965629008966*v_R14_b_i + 15.7892267502954*v_R14_b_r - 9.69353831171602*v_R14_c_i - 53.0601386009611*v_R14_c_r - 3.16965629008967*v_R14_n_i + 15.7892267502954*v_R14_n_r + 0.67991845328531*v_R15_a_i - 7.72144347466701*v_R15_a_r + 1.58482814504483*v_R15_b_i - 7.8946133751477*v_R15_b_r + 4.84676915585801*v_R15_c_i + 26.5300693004806*v_R15_c_r + 1.58482814504483*v_R15_n_i - 7.89461337514771*v_R15_n_r
        struct[0].g[141,0] = i_R14_c_i - 7.72144347466701*v_R13_a_i - 0.67991845328531*v_R13_a_r - 7.8946133751477*v_R13_b_i - 1.58482814504483*v_R13_b_r + 26.5300693004806*v_R13_c_i - 4.84676915585801*v_R13_c_r - 7.89461337514771*v_R13_n_i - 1.58482814504483*v_R13_n_r + 15.442886949334*v_R14_a_i + 1.35983690657062*v_R14_a_r + 15.7892267502954*v_R14_b_i + 3.16965629008966*v_R14_b_r - 53.0601386009611*v_R14_c_i + 9.69353831171602*v_R14_c_r + 15.7892267502954*v_R14_n_i + 3.16965629008967*v_R14_n_r - 7.72144347466701*v_R15_a_i - 0.67991845328531*v_R15_a_r - 7.8946133751477*v_R15_b_i - 1.58482814504483*v_R15_b_r + 26.5300693004806*v_R15_c_i - 4.84676915585801*v_R15_c_r - 7.89461337514771*v_R15_n_i - 1.58482814504483*v_R15_n_r
        struct[0].g[142,0] = i_R14_n_r + 1.58482814504483*v_R13_a_i - 7.89461337514771*v_R13_a_r + 0.679918453285308*v_R13_b_i - 7.72144347466701*v_R13_b_r + 1.58482814504483*v_R13_c_i - 7.89461337514771*v_R13_c_r + 4.84676915585801*v_R13_n_i + 26.5300693004806*v_R13_n_r - 3.16965629008966*v_R14_a_i + 15.7892267502954*v_R14_a_r - 1.35983690657062*v_R14_b_i + 15.442886949334*v_R14_b_r - 3.16965629008966*v_R14_c_i + 15.7892267502954*v_R14_c_r - 9.69353831171602*v_R14_n_i - 53.0601386009611*v_R14_n_r + 1.58482814504483*v_R15_a_i - 7.89461337514771*v_R15_a_r + 0.679918453285308*v_R15_b_i - 7.72144347466701*v_R15_b_r + 1.58482814504483*v_R15_c_i - 7.89461337514771*v_R15_c_r + 4.84676915585801*v_R15_n_i + 26.5300693004806*v_R15_n_r
        struct[0].g[143,0] = i_R14_n_i - 7.89461337514771*v_R13_a_i - 1.58482814504483*v_R13_a_r - 7.72144347466701*v_R13_b_i - 0.679918453285308*v_R13_b_r - 7.89461337514771*v_R13_c_i - 1.58482814504483*v_R13_c_r + 26.5300693004806*v_R13_n_i - 4.84676915585801*v_R13_n_r + 15.7892267502954*v_R14_a_i + 3.16965629008966*v_R14_a_r + 15.442886949334*v_R14_b_i + 1.35983690657062*v_R14_b_r + 15.7892267502954*v_R14_c_i + 3.16965629008966*v_R14_c_r - 53.0601386009611*v_R14_n_i + 9.69353831171602*v_R14_n_r - 7.89461337514771*v_R15_a_i - 1.58482814504483*v_R15_a_r - 7.72144347466701*v_R15_b_i - 0.679918453285308*v_R15_b_r - 7.89461337514771*v_R15_c_i - 1.58482814504483*v_R15_c_r + 26.5300693004806*v_R15_n_i - 4.84676915585801*v_R15_n_r
        struct[0].g[144,0] = i_R1_a_i*v_R1_a_i - i_R1_a_i*v_R1_n_i + i_R1_a_r*v_R1_a_r - i_R1_a_r*v_R1_n_r - p_R1_a
        struct[0].g[145,0] = i_R1_b_i*v_R1_b_i - i_R1_b_i*v_R1_n_i + i_R1_b_r*v_R1_b_r - i_R1_b_r*v_R1_n_r - p_R1_b
        struct[0].g[146,0] = i_R1_c_i*v_R1_c_i - i_R1_c_i*v_R1_n_i + i_R1_c_r*v_R1_c_r - i_R1_c_r*v_R1_n_r - p_R1_c
        struct[0].g[147,0] = -i_R1_a_i*v_R1_a_r + i_R1_a_i*v_R1_n_r + i_R1_a_r*v_R1_a_i - i_R1_a_r*v_R1_n_i - q_R1_a
        struct[0].g[148,0] = -i_R1_b_i*v_R1_b_r + i_R1_b_i*v_R1_n_r + i_R1_b_r*v_R1_b_i - i_R1_b_r*v_R1_n_i - q_R1_b
        struct[0].g[149,0] = -i_R1_c_i*v_R1_c_r + i_R1_c_i*v_R1_n_r + i_R1_c_r*v_R1_c_i - i_R1_c_r*v_R1_n_i - q_R1_c
        struct[0].g[150,0] = i_R1_a_r + i_R1_b_r + i_R1_c_r + i_R1_n_r
        struct[0].g[151,0] = i_R1_a_i + i_R1_b_i + i_R1_c_i + i_R1_n_i
        struct[0].g[152,0] = i_R11_a_i*v_R11_a_i - i_R11_a_i*v_R11_n_i + i_R11_a_r*v_R11_a_r - i_R11_a_r*v_R11_n_r - p_R11_a
        struct[0].g[153,0] = i_R11_b_i*v_R11_b_i - i_R11_b_i*v_R11_n_i + i_R11_b_r*v_R11_b_r - i_R11_b_r*v_R11_n_r - p_R11_b
        struct[0].g[154,0] = i_R11_c_i*v_R11_c_i - i_R11_c_i*v_R11_n_i + i_R11_c_r*v_R11_c_r - i_R11_c_r*v_R11_n_r - p_R11_c
        struct[0].g[155,0] = -i_R11_a_i*v_R11_a_r + i_R11_a_i*v_R11_n_r + i_R11_a_r*v_R11_a_i - i_R11_a_r*v_R11_n_i - q_R11_a
        struct[0].g[156,0] = -i_R11_b_i*v_R11_b_r + i_R11_b_i*v_R11_n_r + i_R11_b_r*v_R11_b_i - i_R11_b_r*v_R11_n_i - q_R11_b
        struct[0].g[157,0] = -i_R11_c_i*v_R11_c_r + i_R11_c_i*v_R11_n_r + i_R11_c_r*v_R11_c_i - i_R11_c_r*v_R11_n_i - q_R11_c
        struct[0].g[158,0] = i_R11_a_r + i_R11_b_r + i_R11_c_r + i_R11_n_r
        struct[0].g[159,0] = i_R11_a_i + i_R11_b_i + i_R11_c_i + i_R11_n_i
        struct[0].g[160,0] = i_R15_a_i*v_R15_a_i - i_R15_a_i*v_R15_n_i + i_R15_a_r*v_R15_a_r - i_R15_a_r*v_R15_n_r - p_R15_a
        struct[0].g[161,0] = i_R15_b_i*v_R15_b_i - i_R15_b_i*v_R15_n_i + i_R15_b_r*v_R15_b_r - i_R15_b_r*v_R15_n_r - p_R15_b
        struct[0].g[162,0] = i_R15_c_i*v_R15_c_i - i_R15_c_i*v_R15_n_i + i_R15_c_r*v_R15_c_r - i_R15_c_r*v_R15_n_r - p_R15_c
        struct[0].g[163,0] = -i_R15_a_i*v_R15_a_r + i_R15_a_i*v_R15_n_r + i_R15_a_r*v_R15_a_i - i_R15_a_r*v_R15_n_i - q_R15_a
        struct[0].g[164,0] = -i_R15_b_i*v_R15_b_r + i_R15_b_i*v_R15_n_r + i_R15_b_r*v_R15_b_i - i_R15_b_r*v_R15_n_i - q_R15_b
        struct[0].g[165,0] = -i_R15_c_i*v_R15_c_r + i_R15_c_i*v_R15_n_r + i_R15_c_r*v_R15_c_i - i_R15_c_r*v_R15_n_i - q_R15_c
        struct[0].g[166,0] = i_R15_a_r + i_R15_b_r + i_R15_c_r + i_R15_n_r
        struct[0].g[167,0] = i_R15_a_i + i_R15_b_i + i_R15_c_i + i_R15_n_i
        struct[0].g[168,0] = i_R16_a_i*v_R16_a_i - i_R16_a_i*v_R16_n_i + i_R16_a_r*v_R16_a_r - i_R16_a_r*v_R16_n_r - p_R16_a
        struct[0].g[169,0] = i_R16_b_i*v_R16_b_i - i_R16_b_i*v_R16_n_i + i_R16_b_r*v_R16_b_r - i_R16_b_r*v_R16_n_r - p_R16_b
        struct[0].g[170,0] = i_R16_c_i*v_R16_c_i - i_R16_c_i*v_R16_n_i + i_R16_c_r*v_R16_c_r - i_R16_c_r*v_R16_n_r - p_R16_c
        struct[0].g[171,0] = -i_R16_a_i*v_R16_a_r + i_R16_a_i*v_R16_n_r + i_R16_a_r*v_R16_a_i - i_R16_a_r*v_R16_n_i - q_R16_a
        struct[0].g[172,0] = -i_R16_b_i*v_R16_b_r + i_R16_b_i*v_R16_n_r + i_R16_b_r*v_R16_b_i - i_R16_b_r*v_R16_n_i - q_R16_b
        struct[0].g[173,0] = -i_R16_c_i*v_R16_c_r + i_R16_c_i*v_R16_n_r + i_R16_c_r*v_R16_c_i - i_R16_c_r*v_R16_n_i - q_R16_c
        struct[0].g[174,0] = i_R16_a_r + i_R16_b_r + i_R16_c_r + i_R16_n_r
        struct[0].g[175,0] = i_R16_a_i + i_R16_b_i + i_R16_c_i + i_R16_n_i
        struct[0].g[176,0] = i_R17_a_i*v_R17_a_i - i_R17_a_i*v_R17_n_i + i_R17_a_r*v_R17_a_r - i_R17_a_r*v_R17_n_r - p_R17_a
        struct[0].g[177,0] = i_R17_b_i*v_R17_b_i - i_R17_b_i*v_R17_n_i + i_R17_b_r*v_R17_b_r - i_R17_b_r*v_R17_n_r - p_R17_b
        struct[0].g[178,0] = i_R17_c_i*v_R17_c_i - i_R17_c_i*v_R17_n_i + i_R17_c_r*v_R17_c_r - i_R17_c_r*v_R17_n_r - p_R17_c
        struct[0].g[179,0] = -i_R17_a_i*v_R17_a_r + i_R17_a_i*v_R17_n_r + i_R17_a_r*v_R17_a_i - i_R17_a_r*v_R17_n_i - q_R17_a
        struct[0].g[180,0] = -i_R17_b_i*v_R17_b_r + i_R17_b_i*v_R17_n_r + i_R17_b_r*v_R17_b_i - i_R17_b_r*v_R17_n_i - q_R17_b
        struct[0].g[181,0] = -i_R17_c_i*v_R17_c_r + i_R17_c_i*v_R17_n_r + i_R17_c_r*v_R17_c_i - i_R17_c_r*v_R17_n_i - q_R17_c
        struct[0].g[182,0] = i_R17_a_r + i_R17_b_r + i_R17_c_r + i_R17_n_r
        struct[0].g[183,0] = i_R17_a_i + i_R17_b_i + i_R17_c_i + i_R17_n_i
        struct[0].g[184,0] = i_R18_a_i*v_R18_a_i - i_R18_a_i*v_R18_n_i + i_R18_a_r*v_R18_a_r - i_R18_a_r*v_R18_n_r - p_R18_a
        struct[0].g[185,0] = i_R18_b_i*v_R18_b_i - i_R18_b_i*v_R18_n_i + i_R18_b_r*v_R18_b_r - i_R18_b_r*v_R18_n_r - p_R18_b
        struct[0].g[186,0] = i_R18_c_i*v_R18_c_i - i_R18_c_i*v_R18_n_i + i_R18_c_r*v_R18_c_r - i_R18_c_r*v_R18_n_r - p_R18_c
        struct[0].g[187,0] = -i_R18_a_i*v_R18_a_r + i_R18_a_i*v_R18_n_r + i_R18_a_r*v_R18_a_i - i_R18_a_r*v_R18_n_i - q_R18_a
        struct[0].g[188,0] = -i_R18_b_i*v_R18_b_r + i_R18_b_i*v_R18_n_r + i_R18_b_r*v_R18_b_i - i_R18_b_r*v_R18_n_i - q_R18_b
        struct[0].g[189,0] = -i_R18_c_i*v_R18_c_r + i_R18_c_i*v_R18_n_r + i_R18_c_r*v_R18_c_i - i_R18_c_r*v_R18_n_i - q_R18_c
        struct[0].g[190,0] = i_R18_a_r + i_R18_b_r + i_R18_c_r + i_R18_n_r
        struct[0].g[191,0] = i_R18_a_i + i_R18_b_i + i_R18_c_i + i_R18_n_i
    
    # Outputs:
    if mode == 3:

        struct[0].h[0,0] = (v_R0_a_i**2 + v_R0_a_r**2)**0.5
        struct[0].h[1,0] = (v_R0_b_i**2 + v_R0_b_r**2)**0.5
        struct[0].h[2,0] = (v_R0_c_i**2 + v_R0_c_r**2)**0.5
        struct[0].h[3,0] = (v_R1_a_i**2 + v_R1_a_r**2)**0.5
        struct[0].h[4,0] = (v_R1_b_i**2 + v_R1_b_r**2)**0.5
        struct[0].h[5,0] = (v_R1_c_i**2 + v_R1_c_r**2)**0.5
        struct[0].h[6,0] = (v_R1_n_i**2 + v_R1_n_r**2)**0.5
        struct[0].h[7,0] = (v_R11_a_i**2 + v_R11_a_r**2)**0.5
        struct[0].h[8,0] = (v_R11_b_i**2 + v_R11_b_r**2)**0.5
        struct[0].h[9,0] = (v_R11_c_i**2 + v_R11_c_r**2)**0.5
        struct[0].h[10,0] = (v_R11_n_i**2 + v_R11_n_r**2)**0.5
        struct[0].h[11,0] = (v_R15_a_i**2 + v_R15_a_r**2)**0.5
        struct[0].h[12,0] = (v_R15_b_i**2 + v_R15_b_r**2)**0.5
        struct[0].h[13,0] = (v_R15_c_i**2 + v_R15_c_r**2)**0.5
        struct[0].h[14,0] = (v_R15_n_i**2 + v_R15_n_r**2)**0.5
        struct[0].h[15,0] = (v_R16_a_i**2 + v_R16_a_r**2)**0.5
        struct[0].h[16,0] = (v_R16_b_i**2 + v_R16_b_r**2)**0.5
        struct[0].h[17,0] = (v_R16_c_i**2 + v_R16_c_r**2)**0.5
        struct[0].h[18,0] = (v_R16_n_i**2 + v_R16_n_r**2)**0.5
        struct[0].h[19,0] = (v_R17_a_i**2 + v_R17_a_r**2)**0.5
        struct[0].h[20,0] = (v_R17_b_i**2 + v_R17_b_r**2)**0.5
        struct[0].h[21,0] = (v_R17_c_i**2 + v_R17_c_r**2)**0.5
        struct[0].h[22,0] = (v_R17_n_i**2 + v_R17_n_r**2)**0.5
        struct[0].h[23,0] = (v_R18_a_i**2 + v_R18_a_r**2)**0.5
        struct[0].h[24,0] = (v_R18_b_i**2 + v_R18_b_r**2)**0.5
        struct[0].h[25,0] = (v_R18_c_i**2 + v_R18_c_r**2)**0.5
        struct[0].h[26,0] = (v_R18_n_i**2 + v_R18_n_r**2)**0.5
        struct[0].h[27,0] = (v_R2_a_i**2 + v_R2_a_r**2)**0.5
        struct[0].h[28,0] = (v_R2_b_i**2 + v_R2_b_r**2)**0.5
        struct[0].h[29,0] = (v_R2_c_i**2 + v_R2_c_r**2)**0.5
        struct[0].h[30,0] = (v_R2_n_i**2 + v_R2_n_r**2)**0.5
        struct[0].h[31,0] = (v_R3_a_i**2 + v_R3_a_r**2)**0.5
        struct[0].h[32,0] = (v_R3_b_i**2 + v_R3_b_r**2)**0.5
        struct[0].h[33,0] = (v_R3_c_i**2 + v_R3_c_r**2)**0.5
        struct[0].h[34,0] = (v_R3_n_i**2 + v_R3_n_r**2)**0.5
        struct[0].h[35,0] = (v_R4_a_i**2 + v_R4_a_r**2)**0.5
        struct[0].h[36,0] = (v_R4_b_i**2 + v_R4_b_r**2)**0.5
        struct[0].h[37,0] = (v_R4_c_i**2 + v_R4_c_r**2)**0.5
        struct[0].h[38,0] = (v_R4_n_i**2 + v_R4_n_r**2)**0.5
        struct[0].h[39,0] = (v_R5_a_i**2 + v_R5_a_r**2)**0.5
        struct[0].h[40,0] = (v_R5_b_i**2 + v_R5_b_r**2)**0.5
        struct[0].h[41,0] = (v_R5_c_i**2 + v_R5_c_r**2)**0.5
        struct[0].h[42,0] = (v_R5_n_i**2 + v_R5_n_r**2)**0.5
        struct[0].h[43,0] = (v_R6_a_i**2 + v_R6_a_r**2)**0.5
        struct[0].h[44,0] = (v_R6_b_i**2 + v_R6_b_r**2)**0.5
        struct[0].h[45,0] = (v_R6_c_i**2 + v_R6_c_r**2)**0.5
        struct[0].h[46,0] = (v_R6_n_i**2 + v_R6_n_r**2)**0.5
        struct[0].h[47,0] = (v_R7_a_i**2 + v_R7_a_r**2)**0.5
        struct[0].h[48,0] = (v_R7_b_i**2 + v_R7_b_r**2)**0.5
        struct[0].h[49,0] = (v_R7_c_i**2 + v_R7_c_r**2)**0.5
        struct[0].h[50,0] = (v_R7_n_i**2 + v_R7_n_r**2)**0.5
        struct[0].h[51,0] = (v_R8_a_i**2 + v_R8_a_r**2)**0.5
        struct[0].h[52,0] = (v_R8_b_i**2 + v_R8_b_r**2)**0.5
        struct[0].h[53,0] = (v_R8_c_i**2 + v_R8_c_r**2)**0.5
        struct[0].h[54,0] = (v_R8_n_i**2 + v_R8_n_r**2)**0.5
        struct[0].h[55,0] = (v_R9_a_i**2 + v_R9_a_r**2)**0.5
        struct[0].h[56,0] = (v_R9_b_i**2 + v_R9_b_r**2)**0.5
        struct[0].h[57,0] = (v_R9_c_i**2 + v_R9_c_r**2)**0.5
        struct[0].h[58,0] = (v_R9_n_i**2 + v_R9_n_r**2)**0.5
        struct[0].h[59,0] = (v_R10_a_i**2 + v_R10_a_r**2)**0.5
        struct[0].h[60,0] = (v_R10_b_i**2 + v_R10_b_r**2)**0.5
        struct[0].h[61,0] = (v_R10_c_i**2 + v_R10_c_r**2)**0.5
        struct[0].h[62,0] = (v_R10_n_i**2 + v_R10_n_r**2)**0.5
        struct[0].h[63,0] = (v_R12_a_i**2 + v_R12_a_r**2)**0.5
        struct[0].h[64,0] = (v_R12_b_i**2 + v_R12_b_r**2)**0.5
        struct[0].h[65,0] = (v_R12_c_i**2 + v_R12_c_r**2)**0.5
        struct[0].h[66,0] = (v_R12_n_i**2 + v_R12_n_r**2)**0.5
        struct[0].h[67,0] = (v_R13_a_i**2 + v_R13_a_r**2)**0.5
        struct[0].h[68,0] = (v_R13_b_i**2 + v_R13_b_r**2)**0.5
        struct[0].h[69,0] = (v_R13_c_i**2 + v_R13_c_r**2)**0.5
        struct[0].h[70,0] = (v_R13_n_i**2 + v_R13_n_r**2)**0.5
        struct[0].h[71,0] = (v_R14_a_i**2 + v_R14_a_r**2)**0.5
        struct[0].h[72,0] = (v_R14_b_i**2 + v_R14_b_r**2)**0.5
        struct[0].h[73,0] = (v_R14_c_i**2 + v_R14_c_r**2)**0.5
        struct[0].h[74,0] = (v_R14_n_i**2 + v_R14_n_r**2)**0.5
    

    if mode == 10:

        struct[0].Fx[0,0] = -1

    if mode == 11:


        struct[0].Gy[0,0] = -123.954122254357
        struct[0].Gy[0,1] = -127.595184532966
        struct[0].Gy[0,2] = 39.6392229058202
        struct[0].Gy[0,3] = 10.2713736253513
        struct[0].Gy[0,4] = 24.9575997948692
        struct[0].Gy[0,5] = 23.2849644809540
        struct[0].Gy[0,6] = 58.0215758469966
        struct[0].Gy[0,7] = 83.8007853900572
        struct[0].Gy[0,48] = 105.571769313180
        struct[0].Gy[0,49] = 54.0657727682604
        struct[0].Gy[0,50] = -39.6392229058202
        struct[0].Gy[0,51] = -10.2713736253513
        struct[0].Gy[0,52] = -24.9575997948692
        struct[0].Gy[0,53] = -23.2849644809540
        struct[0].Gy[0,54] = -39.6392229058202
        struct[0].Gy[0,55] = -10.2713736253513
        struct[0].Gy[0,144] = 1
        struct[0].Gy[1,0] = 127.595184532966
        struct[0].Gy[1,1] = -123.954122254357
        struct[0].Gy[1,2] = -10.2713736253513
        struct[0].Gy[1,3] = 39.6392229058202
        struct[0].Gy[1,4] = -23.2849644809540
        struct[0].Gy[1,5] = 24.9575997948692
        struct[0].Gy[1,6] = -83.8007853900572
        struct[0].Gy[1,7] = 58.0215758469966
        struct[0].Gy[1,48] = -54.0657727682604
        struct[0].Gy[1,49] = 105.571769313180
        struct[0].Gy[1,50] = 10.2713736253513
        struct[0].Gy[1,51] = -39.6392229058202
        struct[0].Gy[1,52] = 23.2849644809540
        struct[0].Gy[1,53] = -24.9575997948692
        struct[0].Gy[1,54] = 10.2713736253513
        struct[0].Gy[1,55] = -39.6392229058202
        struct[0].Gy[1,145] = 1
        struct[0].Gy[2,0] = 39.6392229058202
        struct[0].Gy[2,1] = 10.2713736253513
        struct[0].Gy[2,2] = -123.954122254357
        struct[0].Gy[2,3] = -127.595184532966
        struct[0].Gy[2,4] = 39.6392229058201
        struct[0].Gy[2,5] = 10.2713736253513
        struct[0].Gy[2,6] = 43.3399527360457
        struct[0].Gy[2,7] = 96.8143762456598
        struct[0].Gy[2,48] = -39.6392229058202
        struct[0].Gy[2,49] = -10.2713736253513
        struct[0].Gy[2,50] = 105.571769313180
        struct[0].Gy[2,51] = 54.0657727682604
        struct[0].Gy[2,52] = -39.6392229058201
        struct[0].Gy[2,53] = -10.2713736253513
        struct[0].Gy[2,54] = -24.9575997948692
        struct[0].Gy[2,55] = -23.2849644809540
        struct[0].Gy[2,146] = 1
        struct[0].Gy[3,0] = -10.2713736253513
        struct[0].Gy[3,1] = 39.6392229058202
        struct[0].Gy[3,2] = 127.595184532966
        struct[0].Gy[3,3] = -123.954122254357
        struct[0].Gy[3,4] = -10.2713736253513
        struct[0].Gy[3,5] = 39.6392229058201
        struct[0].Gy[3,6] = -96.8143762456598
        struct[0].Gy[3,7] = 43.3399527360457
        struct[0].Gy[3,48] = 10.2713736253513
        struct[0].Gy[3,49] = -39.6392229058202
        struct[0].Gy[3,50] = -54.0657727682604
        struct[0].Gy[3,51] = 105.571769313180
        struct[0].Gy[3,52] = 10.2713736253513
        struct[0].Gy[3,53] = -39.6392229058201
        struct[0].Gy[3,54] = 23.2849644809540
        struct[0].Gy[3,55] = -24.9575997948692
        struct[0].Gy[3,147] = 1
        struct[0].Gy[4,0] = 24.9575997948692
        struct[0].Gy[4,1] = 23.2849644809540
        struct[0].Gy[4,2] = 39.6392229058202
        struct[0].Gy[4,3] = 10.2713736253513
        struct[0].Gy[4,4] = -123.954122254357
        struct[0].Gy[4,5] = -127.595184532966
        struct[0].Gy[4,6] = 58.0215758469966
        struct[0].Gy[4,7] = 83.8007853900572
        struct[0].Gy[4,48] = -24.9575997948692
        struct[0].Gy[4,49] = -23.2849644809540
        struct[0].Gy[4,50] = -39.6392229058202
        struct[0].Gy[4,51] = -10.2713736253513
        struct[0].Gy[4,52] = 105.571769313180
        struct[0].Gy[4,53] = 54.0657727682604
        struct[0].Gy[4,54] = -39.6392229058201
        struct[0].Gy[4,55] = -10.2713736253514
        struct[0].Gy[4,148] = 1
        struct[0].Gy[5,0] = -23.2849644809540
        struct[0].Gy[5,1] = 24.9575997948692
        struct[0].Gy[5,2] = -10.2713736253513
        struct[0].Gy[5,3] = 39.6392229058202
        struct[0].Gy[5,4] = 127.595184532966
        struct[0].Gy[5,5] = -123.954122254357
        struct[0].Gy[5,6] = -83.8007853900572
        struct[0].Gy[5,7] = 58.0215758469966
        struct[0].Gy[5,48] = 23.2849644809540
        struct[0].Gy[5,49] = -24.9575997948692
        struct[0].Gy[5,50] = 10.2713736253513
        struct[0].Gy[5,51] = -39.6392229058202
        struct[0].Gy[5,52] = -54.0657727682604
        struct[0].Gy[5,53] = 105.571769313180
        struct[0].Gy[5,54] = 10.2713736253514
        struct[0].Gy[5,55] = -39.6392229058201
        struct[0].Gy[5,149] = 1
        struct[0].Gy[6,0] = 58.0215758469966
        struct[0].Gy[6,1] = 83.8007853900572
        struct[0].Gy[6,2] = 43.3399527360457
        struct[0].Gy[6,3] = 96.8143762456598
        struct[0].Gy[6,4] = 58.0215758469966
        struct[0].Gy[6,5] = 83.8007853900572
        struct[0].Gy[6,6] = -161.052161470043
        struct[0].Gy[6,7] = -274.654008062378
        struct[0].Gy[6,48] = -39.6392229058202
        struct[0].Gy[6,49] = -10.2713736253513
        struct[0].Gy[6,50] = -24.9575997948692
        struct[0].Gy[6,51] = -23.2849644809540
        struct[0].Gy[6,52] = -39.6392229058202
        struct[0].Gy[6,53] = -10.2713736253513
        struct[0].Gy[6,54] = 105.571769313180
        struct[0].Gy[6,55] = 54.0657727682604
        struct[0].Gy[6,150] = 1
        struct[0].Gy[7,0] = -83.8007853900572
        struct[0].Gy[7,1] = 58.0215758469966
        struct[0].Gy[7,2] = -96.8143762456598
        struct[0].Gy[7,3] = 43.3399527360457
        struct[0].Gy[7,4] = -83.8007853900572
        struct[0].Gy[7,5] = 58.0215758469966
        struct[0].Gy[7,6] = 274.654008062378
        struct[0].Gy[7,7] = -161.052161470043
        struct[0].Gy[7,48] = 10.2713736253513
        struct[0].Gy[7,49] = -39.6392229058202
        struct[0].Gy[7,50] = 23.2849644809540
        struct[0].Gy[7,51] = -24.9575997948692
        struct[0].Gy[7,52] = 10.2713736253513
        struct[0].Gy[7,53] = -39.6392229058202
        struct[0].Gy[7,54] = -54.0657727682604
        struct[0].Gy[7,55] = 105.571769313180
        struct[0].Gy[7,151] = 1
        struct[0].Gy[8,8] = -26.5300693004806
        struct[0].Gy[8,9] = -4.84676915585801
        struct[0].Gy[8,10] = 7.89461337514770
        struct[0].Gy[8,11] = -1.58482814504483
        struct[0].Gy[8,12] = 7.72144347466701
        struct[0].Gy[8,13] = -0.679918453285310
        struct[0].Gy[8,14] = 7.89461337514771
        struct[0].Gy[8,15] = -1.58482814504483
        struct[0].Gy[8,56] = 26.5300693004806
        struct[0].Gy[8,57] = 4.84676915585801
        struct[0].Gy[8,58] = -7.89461337514770
        struct[0].Gy[8,59] = 1.58482814504483
        struct[0].Gy[8,60] = -7.72144347466701
        struct[0].Gy[8,61] = 0.679918453285310
        struct[0].Gy[8,62] = -7.89461337514771
        struct[0].Gy[8,63] = 1.58482814504483
        struct[0].Gy[8,152] = 1
        struct[0].Gy[9,8] = 4.84676915585801
        struct[0].Gy[9,9] = -26.5300693004806
        struct[0].Gy[9,10] = 1.58482814504483
        struct[0].Gy[9,11] = 7.89461337514770
        struct[0].Gy[9,12] = 0.679918453285310
        struct[0].Gy[9,13] = 7.72144347466701
        struct[0].Gy[9,14] = 1.58482814504483
        struct[0].Gy[9,15] = 7.89461337514771
        struct[0].Gy[9,56] = -4.84676915585801
        struct[0].Gy[9,57] = 26.5300693004806
        struct[0].Gy[9,58] = -1.58482814504483
        struct[0].Gy[9,59] = -7.89461337514770
        struct[0].Gy[9,60] = -0.679918453285310
        struct[0].Gy[9,61] = -7.72144347466701
        struct[0].Gy[9,62] = -1.58482814504483
        struct[0].Gy[9,63] = -7.89461337514771
        struct[0].Gy[9,153] = 1
        struct[0].Gy[10,8] = 7.89461337514771
        struct[0].Gy[10,9] = -1.58482814504483
        struct[0].Gy[10,10] = -26.5300693004806
        struct[0].Gy[10,11] = -4.84676915585801
        struct[0].Gy[10,12] = 7.89461337514771
        struct[0].Gy[10,13] = -1.58482814504483
        struct[0].Gy[10,14] = 7.72144347466701
        struct[0].Gy[10,15] = -0.679918453285308
        struct[0].Gy[10,56] = -7.89461337514771
        struct[0].Gy[10,57] = 1.58482814504483
        struct[0].Gy[10,58] = 26.5300693004806
        struct[0].Gy[10,59] = 4.84676915585801
        struct[0].Gy[10,60] = -7.89461337514771
        struct[0].Gy[10,61] = 1.58482814504483
        struct[0].Gy[10,62] = -7.72144347466701
        struct[0].Gy[10,63] = 0.679918453285308
        struct[0].Gy[10,154] = 1
        struct[0].Gy[11,8] = 1.58482814504483
        struct[0].Gy[11,9] = 7.89461337514771
        struct[0].Gy[11,10] = 4.84676915585801
        struct[0].Gy[11,11] = -26.5300693004806
        struct[0].Gy[11,12] = 1.58482814504483
        struct[0].Gy[11,13] = 7.89461337514771
        struct[0].Gy[11,14] = 0.679918453285308
        struct[0].Gy[11,15] = 7.72144347466701
        struct[0].Gy[11,56] = -1.58482814504483
        struct[0].Gy[11,57] = -7.89461337514771
        struct[0].Gy[11,58] = -4.84676915585801
        struct[0].Gy[11,59] = 26.5300693004806
        struct[0].Gy[11,60] = -1.58482814504483
        struct[0].Gy[11,61] = -7.89461337514771
        struct[0].Gy[11,62] = -0.679918453285308
        struct[0].Gy[11,63] = -7.72144347466701
        struct[0].Gy[11,155] = 1
        struct[0].Gy[12,8] = 7.72144347466701
        struct[0].Gy[12,9] = -0.679918453285310
        struct[0].Gy[12,10] = 7.89461337514770
        struct[0].Gy[12,11] = -1.58482814504483
        struct[0].Gy[12,12] = -26.5300693004806
        struct[0].Gy[12,13] = -4.84676915585801
        struct[0].Gy[12,14] = 7.89461337514771
        struct[0].Gy[12,15] = -1.58482814504483
        struct[0].Gy[12,56] = -7.72144347466701
        struct[0].Gy[12,57] = 0.679918453285310
        struct[0].Gy[12,58] = -7.89461337514770
        struct[0].Gy[12,59] = 1.58482814504483
        struct[0].Gy[12,60] = 26.5300693004806
        struct[0].Gy[12,61] = 4.84676915585801
        struct[0].Gy[12,62] = -7.89461337514771
        struct[0].Gy[12,63] = 1.58482814504483
        struct[0].Gy[12,156] = 1
        struct[0].Gy[13,8] = 0.679918453285310
        struct[0].Gy[13,9] = 7.72144347466701
        struct[0].Gy[13,10] = 1.58482814504483
        struct[0].Gy[13,11] = 7.89461337514770
        struct[0].Gy[13,12] = 4.84676915585801
        struct[0].Gy[13,13] = -26.5300693004806
        struct[0].Gy[13,14] = 1.58482814504483
        struct[0].Gy[13,15] = 7.89461337514771
        struct[0].Gy[13,56] = -0.679918453285310
        struct[0].Gy[13,57] = -7.72144347466701
        struct[0].Gy[13,58] = -1.58482814504483
        struct[0].Gy[13,59] = -7.89461337514770
        struct[0].Gy[13,60] = -4.84676915585801
        struct[0].Gy[13,61] = 26.5300693004806
        struct[0].Gy[13,62] = -1.58482814504483
        struct[0].Gy[13,63] = -7.89461337514771
        struct[0].Gy[13,157] = 1
        struct[0].Gy[14,8] = 7.89461337514771
        struct[0].Gy[14,9] = -1.58482814504483
        struct[0].Gy[14,10] = 7.72144347466701
        struct[0].Gy[14,11] = -0.679918453285308
        struct[0].Gy[14,12] = 7.89461337514771
        struct[0].Gy[14,13] = -1.58482814504483
        struct[0].Gy[14,14] = -26.5550693004806
        struct[0].Gy[14,15] = -4.84676915585801
        struct[0].Gy[14,56] = -7.89461337514771
        struct[0].Gy[14,57] = 1.58482814504483
        struct[0].Gy[14,58] = -7.72144347466701
        struct[0].Gy[14,59] = 0.679918453285308
        struct[0].Gy[14,60] = -7.89461337514771
        struct[0].Gy[14,61] = 1.58482814504483
        struct[0].Gy[14,62] = 26.5300693004806
        struct[0].Gy[14,63] = 4.84676915585801
        struct[0].Gy[14,158] = 1
        struct[0].Gy[15,8] = 1.58482814504483
        struct[0].Gy[15,9] = 7.89461337514771
        struct[0].Gy[15,10] = 0.679918453285308
        struct[0].Gy[15,11] = 7.72144347466701
        struct[0].Gy[15,12] = 1.58482814504483
        struct[0].Gy[15,13] = 7.89461337514771
        struct[0].Gy[15,14] = 4.84676915585801
        struct[0].Gy[15,15] = -26.5550693004806
        struct[0].Gy[15,56] = -1.58482814504483
        struct[0].Gy[15,57] = -7.89461337514771
        struct[0].Gy[15,58] = -0.679918453285308
        struct[0].Gy[15,59] = -7.72144347466701
        struct[0].Gy[15,60] = -1.58482814504483
        struct[0].Gy[15,61] = -7.89461337514771
        struct[0].Gy[15,62] = -4.84676915585801
        struct[0].Gy[15,63] = 26.5300693004806
        struct[0].Gy[15,159] = 1
        struct[0].Gy[16,16] = -26.5300693004806
        struct[0].Gy[16,17] = -4.84676915585801
        struct[0].Gy[16,18] = 7.89461337514770
        struct[0].Gy[16,19] = -1.58482814504483
        struct[0].Gy[16,20] = 7.72144347466701
        struct[0].Gy[16,21] = -0.679918453285310
        struct[0].Gy[16,22] = 7.89461337514771
        struct[0].Gy[16,23] = -1.58482814504483
        struct[0].Gy[16,136] = 26.5300693004806
        struct[0].Gy[16,137] = 4.84676915585801
        struct[0].Gy[16,138] = -7.89461337514770
        struct[0].Gy[16,139] = 1.58482814504483
        struct[0].Gy[16,140] = -7.72144347466701
        struct[0].Gy[16,141] = 0.679918453285310
        struct[0].Gy[16,142] = -7.89461337514771
        struct[0].Gy[16,143] = 1.58482814504483
        struct[0].Gy[16,160] = 1
        struct[0].Gy[17,16] = 4.84676915585801
        struct[0].Gy[17,17] = -26.5300693004806
        struct[0].Gy[17,18] = 1.58482814504483
        struct[0].Gy[17,19] = 7.89461337514770
        struct[0].Gy[17,20] = 0.679918453285310
        struct[0].Gy[17,21] = 7.72144347466701
        struct[0].Gy[17,22] = 1.58482814504483
        struct[0].Gy[17,23] = 7.89461337514771
        struct[0].Gy[17,136] = -4.84676915585801
        struct[0].Gy[17,137] = 26.5300693004806
        struct[0].Gy[17,138] = -1.58482814504483
        struct[0].Gy[17,139] = -7.89461337514770
        struct[0].Gy[17,140] = -0.679918453285310
        struct[0].Gy[17,141] = -7.72144347466701
        struct[0].Gy[17,142] = -1.58482814504483
        struct[0].Gy[17,143] = -7.89461337514771
        struct[0].Gy[17,161] = 1
        struct[0].Gy[18,16] = 7.89461337514771
        struct[0].Gy[18,17] = -1.58482814504483
        struct[0].Gy[18,18] = -26.5300693004806
        struct[0].Gy[18,19] = -4.84676915585801
        struct[0].Gy[18,20] = 7.89461337514771
        struct[0].Gy[18,21] = -1.58482814504483
        struct[0].Gy[18,22] = 7.72144347466701
        struct[0].Gy[18,23] = -0.679918453285308
        struct[0].Gy[18,136] = -7.89461337514771
        struct[0].Gy[18,137] = 1.58482814504483
        struct[0].Gy[18,138] = 26.5300693004806
        struct[0].Gy[18,139] = 4.84676915585801
        struct[0].Gy[18,140] = -7.89461337514771
        struct[0].Gy[18,141] = 1.58482814504483
        struct[0].Gy[18,142] = -7.72144347466701
        struct[0].Gy[18,143] = 0.679918453285308
        struct[0].Gy[18,162] = 1
        struct[0].Gy[19,16] = 1.58482814504483
        struct[0].Gy[19,17] = 7.89461337514771
        struct[0].Gy[19,18] = 4.84676915585801
        struct[0].Gy[19,19] = -26.5300693004806
        struct[0].Gy[19,20] = 1.58482814504483
        struct[0].Gy[19,21] = 7.89461337514771
        struct[0].Gy[19,22] = 0.679918453285308
        struct[0].Gy[19,23] = 7.72144347466701
        struct[0].Gy[19,136] = -1.58482814504483
        struct[0].Gy[19,137] = -7.89461337514771
        struct[0].Gy[19,138] = -4.84676915585801
        struct[0].Gy[19,139] = 26.5300693004806
        struct[0].Gy[19,140] = -1.58482814504483
        struct[0].Gy[19,141] = -7.89461337514771
        struct[0].Gy[19,142] = -0.679918453285308
        struct[0].Gy[19,143] = -7.72144347466701
        struct[0].Gy[19,163] = 1
        struct[0].Gy[20,16] = 7.72144347466701
        struct[0].Gy[20,17] = -0.679918453285310
        struct[0].Gy[20,18] = 7.89461337514770
        struct[0].Gy[20,19] = -1.58482814504483
        struct[0].Gy[20,20] = -26.5300693004806
        struct[0].Gy[20,21] = -4.84676915585801
        struct[0].Gy[20,22] = 7.89461337514771
        struct[0].Gy[20,23] = -1.58482814504483
        struct[0].Gy[20,136] = -7.72144347466701
        struct[0].Gy[20,137] = 0.679918453285310
        struct[0].Gy[20,138] = -7.89461337514770
        struct[0].Gy[20,139] = 1.58482814504483
        struct[0].Gy[20,140] = 26.5300693004806
        struct[0].Gy[20,141] = 4.84676915585801
        struct[0].Gy[20,142] = -7.89461337514771
        struct[0].Gy[20,143] = 1.58482814504483
        struct[0].Gy[20,164] = 1
        struct[0].Gy[21,16] = 0.679918453285310
        struct[0].Gy[21,17] = 7.72144347466701
        struct[0].Gy[21,18] = 1.58482814504483
        struct[0].Gy[21,19] = 7.89461337514770
        struct[0].Gy[21,20] = 4.84676915585801
        struct[0].Gy[21,21] = -26.5300693004806
        struct[0].Gy[21,22] = 1.58482814504483
        struct[0].Gy[21,23] = 7.89461337514771
        struct[0].Gy[21,136] = -0.679918453285310
        struct[0].Gy[21,137] = -7.72144347466701
        struct[0].Gy[21,138] = -1.58482814504483
        struct[0].Gy[21,139] = -7.89461337514770
        struct[0].Gy[21,140] = -4.84676915585801
        struct[0].Gy[21,141] = 26.5300693004806
        struct[0].Gy[21,142] = -1.58482814504483
        struct[0].Gy[21,143] = -7.89461337514771
        struct[0].Gy[21,165] = 1
        struct[0].Gy[22,16] = 7.89461337514771
        struct[0].Gy[22,17] = -1.58482814504483
        struct[0].Gy[22,18] = 7.72144347466701
        struct[0].Gy[22,19] = -0.679918453285308
        struct[0].Gy[22,20] = 7.89461337514771
        struct[0].Gy[22,21] = -1.58482814504483
        struct[0].Gy[22,22] = -26.5550693004806
        struct[0].Gy[22,23] = -4.84676915585801
        struct[0].Gy[22,136] = -7.89461337514771
        struct[0].Gy[22,137] = 1.58482814504483
        struct[0].Gy[22,138] = -7.72144347466701
        struct[0].Gy[22,139] = 0.679918453285308
        struct[0].Gy[22,140] = -7.89461337514771
        struct[0].Gy[22,141] = 1.58482814504483
        struct[0].Gy[22,142] = 26.5300693004806
        struct[0].Gy[22,143] = 4.84676915585801
        struct[0].Gy[22,166] = 1
        struct[0].Gy[23,16] = 1.58482814504483
        struct[0].Gy[23,17] = 7.89461337514771
        struct[0].Gy[23,18] = 0.679918453285308
        struct[0].Gy[23,19] = 7.72144347466701
        struct[0].Gy[23,20] = 1.58482814504483
        struct[0].Gy[23,21] = 7.89461337514771
        struct[0].Gy[23,22] = 4.84676915585801
        struct[0].Gy[23,23] = -26.5550693004806
        struct[0].Gy[23,136] = -1.58482814504483
        struct[0].Gy[23,137] = -7.89461337514771
        struct[0].Gy[23,138] = -0.679918453285308
        struct[0].Gy[23,139] = -7.72144347466701
        struct[0].Gy[23,140] = -1.58482814504483
        struct[0].Gy[23,141] = -7.89461337514771
        struct[0].Gy[23,142] = -4.84676915585801
        struct[0].Gy[23,143] = 26.5300693004806
        struct[0].Gy[23,167] = 1
        struct[0].Gy[24,24] = -30.9517475172273
        struct[0].Gy[24,25] = -5.65456401516768
        struct[0].Gy[24,26] = 9.21038227100566
        struct[0].Gy[24,27] = -1.84896616921897
        struct[0].Gy[24,28] = 9.00835072044485
        struct[0].Gy[24,29] = -0.793238195499529
        struct[0].Gy[24,30] = 9.21038227100566
        struct[0].Gy[24,31] = -1.84896616921897
        struct[0].Gy[24,80] = 30.9517475172273
        struct[0].Gy[24,81] = 5.65456401516768
        struct[0].Gy[24,82] = -9.21038227100566
        struct[0].Gy[24,83] = 1.84896616921897
        struct[0].Gy[24,84] = -9.00835072044485
        struct[0].Gy[24,85] = 0.793238195499529
        struct[0].Gy[24,86] = -9.21038227100566
        struct[0].Gy[24,87] = 1.84896616921897
        struct[0].Gy[24,168] = 1
        struct[0].Gy[25,24] = 5.65456401516768
        struct[0].Gy[25,25] = -30.9517475172273
        struct[0].Gy[25,26] = 1.84896616921897
        struct[0].Gy[25,27] = 9.21038227100566
        struct[0].Gy[25,28] = 0.793238195499529
        struct[0].Gy[25,29] = 9.00835072044485
        struct[0].Gy[25,30] = 1.84896616921897
        struct[0].Gy[25,31] = 9.21038227100566
        struct[0].Gy[25,80] = -5.65456401516768
        struct[0].Gy[25,81] = 30.9517475172273
        struct[0].Gy[25,82] = -1.84896616921897
        struct[0].Gy[25,83] = -9.21038227100566
        struct[0].Gy[25,84] = -0.793238195499529
        struct[0].Gy[25,85] = -9.00835072044485
        struct[0].Gy[25,86] = -1.84896616921897
        struct[0].Gy[25,87] = -9.21038227100566
        struct[0].Gy[25,169] = 1
        struct[0].Gy[26,24] = 9.21038227100566
        struct[0].Gy[26,25] = -1.84896616921897
        struct[0].Gy[26,26] = -30.9517475172273
        struct[0].Gy[26,27] = -5.65456401516768
        struct[0].Gy[26,28] = 9.21038227100566
        struct[0].Gy[26,29] = -1.84896616921897
        struct[0].Gy[26,30] = 9.00835072044485
        struct[0].Gy[26,31] = -0.793238195499528
        struct[0].Gy[26,80] = -9.21038227100566
        struct[0].Gy[26,81] = 1.84896616921897
        struct[0].Gy[26,82] = 30.9517475172273
        struct[0].Gy[26,83] = 5.65456401516768
        struct[0].Gy[26,84] = -9.21038227100566
        struct[0].Gy[26,85] = 1.84896616921897
        struct[0].Gy[26,86] = -9.00835072044485
        struct[0].Gy[26,87] = 0.793238195499528
        struct[0].Gy[26,170] = 1
        struct[0].Gy[27,24] = 1.84896616921897
        struct[0].Gy[27,25] = 9.21038227100566
        struct[0].Gy[27,26] = 5.65456401516768
        struct[0].Gy[27,27] = -30.9517475172273
        struct[0].Gy[27,28] = 1.84896616921897
        struct[0].Gy[27,29] = 9.21038227100566
        struct[0].Gy[27,30] = 0.793238195499528
        struct[0].Gy[27,31] = 9.00835072044485
        struct[0].Gy[27,80] = -1.84896616921897
        struct[0].Gy[27,81] = -9.21038227100566
        struct[0].Gy[27,82] = -5.65456401516768
        struct[0].Gy[27,83] = 30.9517475172273
        struct[0].Gy[27,84] = -1.84896616921897
        struct[0].Gy[27,85] = -9.21038227100566
        struct[0].Gy[27,86] = -0.793238195499528
        struct[0].Gy[27,87] = -9.00835072044485
        struct[0].Gy[27,171] = 1
        struct[0].Gy[28,24] = 9.00835072044484
        struct[0].Gy[28,25] = -0.793238195499527
        struct[0].Gy[28,26] = 9.21038227100566
        struct[0].Gy[28,27] = -1.84896616921897
        struct[0].Gy[28,28] = -30.9517475172273
        struct[0].Gy[28,29] = -5.65456401516768
        struct[0].Gy[28,30] = 9.21038227100566
        struct[0].Gy[28,31] = -1.84896616921897
        struct[0].Gy[28,80] = -9.00835072044484
        struct[0].Gy[28,81] = 0.793238195499527
        struct[0].Gy[28,82] = -9.21038227100566
        struct[0].Gy[28,83] = 1.84896616921897
        struct[0].Gy[28,84] = 30.9517475172273
        struct[0].Gy[28,85] = 5.65456401516768
        struct[0].Gy[28,86] = -9.21038227100566
        struct[0].Gy[28,87] = 1.84896616921897
        struct[0].Gy[28,172] = 1
        struct[0].Gy[29,24] = 0.793238195499527
        struct[0].Gy[29,25] = 9.00835072044484
        struct[0].Gy[29,26] = 1.84896616921897
        struct[0].Gy[29,27] = 9.21038227100566
        struct[0].Gy[29,28] = 5.65456401516768
        struct[0].Gy[29,29] = -30.9517475172273
        struct[0].Gy[29,30] = 1.84896616921897
        struct[0].Gy[29,31] = 9.21038227100566
        struct[0].Gy[29,80] = -0.793238195499527
        struct[0].Gy[29,81] = -9.00835072044484
        struct[0].Gy[29,82] = -1.84896616921897
        struct[0].Gy[29,83] = -9.21038227100566
        struct[0].Gy[29,84] = -5.65456401516768
        struct[0].Gy[29,85] = 30.9517475172273
        struct[0].Gy[29,86] = -1.84896616921897
        struct[0].Gy[29,87] = -9.21038227100566
        struct[0].Gy[29,173] = 1
        struct[0].Gy[30,24] = 9.21038227100566
        struct[0].Gy[30,25] = -1.84896616921897
        struct[0].Gy[30,26] = 9.00835072044485
        struct[0].Gy[30,27] = -0.793238195499527
        struct[0].Gy[30,28] = 9.21038227100566
        struct[0].Gy[30,29] = -1.84896616921897
        struct[0].Gy[30,30] = -30.9767475172273
        struct[0].Gy[30,31] = -5.65456401516768
        struct[0].Gy[30,80] = -9.21038227100566
        struct[0].Gy[30,81] = 1.84896616921897
        struct[0].Gy[30,82] = -9.00835072044485
        struct[0].Gy[30,83] = 0.793238195499527
        struct[0].Gy[30,84] = -9.21038227100566
        struct[0].Gy[30,85] = 1.84896616921897
        struct[0].Gy[30,86] = 30.9517475172273
        struct[0].Gy[30,87] = 5.65456401516768
        struct[0].Gy[30,174] = 1
        struct[0].Gy[31,24] = 1.84896616921897
        struct[0].Gy[31,25] = 9.21038227100566
        struct[0].Gy[31,26] = 0.793238195499527
        struct[0].Gy[31,27] = 9.00835072044485
        struct[0].Gy[31,28] = 1.84896616921897
        struct[0].Gy[31,29] = 9.21038227100566
        struct[0].Gy[31,30] = 5.65456401516768
        struct[0].Gy[31,31] = -30.9767475172273
        struct[0].Gy[31,80] = -1.84896616921897
        struct[0].Gy[31,81] = -9.21038227100566
        struct[0].Gy[31,82] = -0.793238195499527
        struct[0].Gy[31,83] = -9.00835072044485
        struct[0].Gy[31,84] = -1.84896616921897
        struct[0].Gy[31,85] = -9.21038227100566
        struct[0].Gy[31,86] = -5.65456401516768
        struct[0].Gy[31,87] = 30.9517475172273
        struct[0].Gy[31,175] = 1
        struct[0].Gy[32,32] = -30.9517475172273
        struct[0].Gy[32,33] = -5.65456401516768
        struct[0].Gy[32,34] = 9.21038227100566
        struct[0].Gy[32,35] = -1.84896616921897
        struct[0].Gy[32,36] = 9.00835072044485
        struct[0].Gy[32,37] = -0.793238195499529
        struct[0].Gy[32,38] = 9.21038227100566
        struct[0].Gy[32,39] = -1.84896616921897
        struct[0].Gy[32,104] = 30.9517475172273
        struct[0].Gy[32,105] = 5.65456401516768
        struct[0].Gy[32,106] = -9.21038227100566
        struct[0].Gy[32,107] = 1.84896616921897
        struct[0].Gy[32,108] = -9.00835072044485
        struct[0].Gy[32,109] = 0.793238195499529
        struct[0].Gy[32,110] = -9.21038227100566
        struct[0].Gy[32,111] = 1.84896616921897
        struct[0].Gy[32,176] = 1
        struct[0].Gy[33,32] = 5.65456401516768
        struct[0].Gy[33,33] = -30.9517475172273
        struct[0].Gy[33,34] = 1.84896616921897
        struct[0].Gy[33,35] = 9.21038227100566
        struct[0].Gy[33,36] = 0.793238195499529
        struct[0].Gy[33,37] = 9.00835072044485
        struct[0].Gy[33,38] = 1.84896616921897
        struct[0].Gy[33,39] = 9.21038227100566
        struct[0].Gy[33,104] = -5.65456401516768
        struct[0].Gy[33,105] = 30.9517475172273
        struct[0].Gy[33,106] = -1.84896616921897
        struct[0].Gy[33,107] = -9.21038227100566
        struct[0].Gy[33,108] = -0.793238195499529
        struct[0].Gy[33,109] = -9.00835072044485
        struct[0].Gy[33,110] = -1.84896616921897
        struct[0].Gy[33,111] = -9.21038227100566
        struct[0].Gy[33,177] = 1
        struct[0].Gy[34,32] = 9.21038227100566
        struct[0].Gy[34,33] = -1.84896616921897
        struct[0].Gy[34,34] = -30.9517475172273
        struct[0].Gy[34,35] = -5.65456401516768
        struct[0].Gy[34,36] = 9.21038227100566
        struct[0].Gy[34,37] = -1.84896616921897
        struct[0].Gy[34,38] = 9.00835072044485
        struct[0].Gy[34,39] = -0.793238195499528
        struct[0].Gy[34,104] = -9.21038227100566
        struct[0].Gy[34,105] = 1.84896616921897
        struct[0].Gy[34,106] = 30.9517475172273
        struct[0].Gy[34,107] = 5.65456401516768
        struct[0].Gy[34,108] = -9.21038227100566
        struct[0].Gy[34,109] = 1.84896616921897
        struct[0].Gy[34,110] = -9.00835072044485
        struct[0].Gy[34,111] = 0.793238195499528
        struct[0].Gy[34,178] = 1
        struct[0].Gy[35,32] = 1.84896616921897
        struct[0].Gy[35,33] = 9.21038227100566
        struct[0].Gy[35,34] = 5.65456401516768
        struct[0].Gy[35,35] = -30.9517475172273
        struct[0].Gy[35,36] = 1.84896616921897
        struct[0].Gy[35,37] = 9.21038227100566
        struct[0].Gy[35,38] = 0.793238195499528
        struct[0].Gy[35,39] = 9.00835072044485
        struct[0].Gy[35,104] = -1.84896616921897
        struct[0].Gy[35,105] = -9.21038227100566
        struct[0].Gy[35,106] = -5.65456401516768
        struct[0].Gy[35,107] = 30.9517475172273
        struct[0].Gy[35,108] = -1.84896616921897
        struct[0].Gy[35,109] = -9.21038227100566
        struct[0].Gy[35,110] = -0.793238195499528
        struct[0].Gy[35,111] = -9.00835072044485
        struct[0].Gy[35,179] = 1
        struct[0].Gy[36,32] = 9.00835072044484
        struct[0].Gy[36,33] = -0.793238195499527
        struct[0].Gy[36,34] = 9.21038227100566
        struct[0].Gy[36,35] = -1.84896616921897
        struct[0].Gy[36,36] = -30.9517475172273
        struct[0].Gy[36,37] = -5.65456401516768
        struct[0].Gy[36,38] = 9.21038227100566
        struct[0].Gy[36,39] = -1.84896616921897
        struct[0].Gy[36,104] = -9.00835072044484
        struct[0].Gy[36,105] = 0.793238195499527
        struct[0].Gy[36,106] = -9.21038227100566
        struct[0].Gy[36,107] = 1.84896616921897
        struct[0].Gy[36,108] = 30.9517475172273
        struct[0].Gy[36,109] = 5.65456401516768
        struct[0].Gy[36,110] = -9.21038227100566
        struct[0].Gy[36,111] = 1.84896616921897
        struct[0].Gy[36,180] = 1
        struct[0].Gy[37,32] = 0.793238195499527
        struct[0].Gy[37,33] = 9.00835072044484
        struct[0].Gy[37,34] = 1.84896616921897
        struct[0].Gy[37,35] = 9.21038227100566
        struct[0].Gy[37,36] = 5.65456401516768
        struct[0].Gy[37,37] = -30.9517475172273
        struct[0].Gy[37,38] = 1.84896616921897
        struct[0].Gy[37,39] = 9.21038227100566
        struct[0].Gy[37,104] = -0.793238195499527
        struct[0].Gy[37,105] = -9.00835072044484
        struct[0].Gy[37,106] = -1.84896616921897
        struct[0].Gy[37,107] = -9.21038227100566
        struct[0].Gy[37,108] = -5.65456401516768
        struct[0].Gy[37,109] = 30.9517475172273
        struct[0].Gy[37,110] = -1.84896616921897
        struct[0].Gy[37,111] = -9.21038227100566
        struct[0].Gy[37,181] = 1
        struct[0].Gy[38,32] = 9.21038227100566
        struct[0].Gy[38,33] = -1.84896616921897
        struct[0].Gy[38,34] = 9.00835072044485
        struct[0].Gy[38,35] = -0.793238195499527
        struct[0].Gy[38,36] = 9.21038227100566
        struct[0].Gy[38,37] = -1.84896616921897
        struct[0].Gy[38,38] = -30.9767475172273
        struct[0].Gy[38,39] = -5.65456401516768
        struct[0].Gy[38,104] = -9.21038227100566
        struct[0].Gy[38,105] = 1.84896616921897
        struct[0].Gy[38,106] = -9.00835072044485
        struct[0].Gy[38,107] = 0.793238195499527
        struct[0].Gy[38,108] = -9.21038227100566
        struct[0].Gy[38,109] = 1.84896616921897
        struct[0].Gy[38,110] = 30.9517475172273
        struct[0].Gy[38,111] = 5.65456401516768
        struct[0].Gy[38,182] = 1
        struct[0].Gy[39,32] = 1.84896616921897
        struct[0].Gy[39,33] = 9.21038227100566
        struct[0].Gy[39,34] = 0.793238195499527
        struct[0].Gy[39,35] = 9.00835072044485
        struct[0].Gy[39,36] = 1.84896616921897
        struct[0].Gy[39,37] = 9.21038227100566
        struct[0].Gy[39,38] = 5.65456401516768
        struct[0].Gy[39,39] = -30.9767475172273
        struct[0].Gy[39,104] = -1.84896616921897
        struct[0].Gy[39,105] = -9.21038227100566
        struct[0].Gy[39,106] = -0.793238195499527
        struct[0].Gy[39,107] = -9.00835072044485
        struct[0].Gy[39,108] = -1.84896616921897
        struct[0].Gy[39,109] = -9.21038227100566
        struct[0].Gy[39,110] = -5.65456401516768
        struct[0].Gy[39,111] = 30.9517475172273
        struct[0].Gy[39,183] = 1
        struct[0].Gy[40,40] = -30.9517475172273
        struct[0].Gy[40,41] = -5.65456401516768
        struct[0].Gy[40,42] = 9.21038227100566
        struct[0].Gy[40,43] = -1.84896616921897
        struct[0].Gy[40,44] = 9.00835072044485
        struct[0].Gy[40,45] = -0.793238195499529
        struct[0].Gy[40,46] = 9.21038227100566
        struct[0].Gy[40,47] = -1.84896616921897
        struct[0].Gy[40,112] = 30.9517475172273
        struct[0].Gy[40,113] = 5.65456401516768
        struct[0].Gy[40,114] = -9.21038227100566
        struct[0].Gy[40,115] = 1.84896616921897
        struct[0].Gy[40,116] = -9.00835072044485
        struct[0].Gy[40,117] = 0.793238195499529
        struct[0].Gy[40,118] = -9.21038227100566
        struct[0].Gy[40,119] = 1.84896616921897
        struct[0].Gy[40,184] = 1
        struct[0].Gy[41,40] = 5.65456401516768
        struct[0].Gy[41,41] = -30.9517475172273
        struct[0].Gy[41,42] = 1.84896616921897
        struct[0].Gy[41,43] = 9.21038227100566
        struct[0].Gy[41,44] = 0.793238195499529
        struct[0].Gy[41,45] = 9.00835072044485
        struct[0].Gy[41,46] = 1.84896616921897
        struct[0].Gy[41,47] = 9.21038227100566
        struct[0].Gy[41,112] = -5.65456401516768
        struct[0].Gy[41,113] = 30.9517475172273
        struct[0].Gy[41,114] = -1.84896616921897
        struct[0].Gy[41,115] = -9.21038227100566
        struct[0].Gy[41,116] = -0.793238195499529
        struct[0].Gy[41,117] = -9.00835072044485
        struct[0].Gy[41,118] = -1.84896616921897
        struct[0].Gy[41,119] = -9.21038227100566
        struct[0].Gy[41,185] = 1
        struct[0].Gy[42,40] = 9.21038227100566
        struct[0].Gy[42,41] = -1.84896616921897
        struct[0].Gy[42,42] = -30.9517475172273
        struct[0].Gy[42,43] = -5.65456401516768
        struct[0].Gy[42,44] = 9.21038227100566
        struct[0].Gy[42,45] = -1.84896616921897
        struct[0].Gy[42,46] = 9.00835072044485
        struct[0].Gy[42,47] = -0.793238195499528
        struct[0].Gy[42,112] = -9.21038227100566
        struct[0].Gy[42,113] = 1.84896616921897
        struct[0].Gy[42,114] = 30.9517475172273
        struct[0].Gy[42,115] = 5.65456401516768
        struct[0].Gy[42,116] = -9.21038227100566
        struct[0].Gy[42,117] = 1.84896616921897
        struct[0].Gy[42,118] = -9.00835072044485
        struct[0].Gy[42,119] = 0.793238195499528
        struct[0].Gy[42,186] = 1
        struct[0].Gy[43,40] = 1.84896616921897
        struct[0].Gy[43,41] = 9.21038227100566
        struct[0].Gy[43,42] = 5.65456401516768
        struct[0].Gy[43,43] = -30.9517475172273
        struct[0].Gy[43,44] = 1.84896616921897
        struct[0].Gy[43,45] = 9.21038227100566
        struct[0].Gy[43,46] = 0.793238195499528
        struct[0].Gy[43,47] = 9.00835072044485
        struct[0].Gy[43,112] = -1.84896616921897
        struct[0].Gy[43,113] = -9.21038227100566
        struct[0].Gy[43,114] = -5.65456401516768
        struct[0].Gy[43,115] = 30.9517475172273
        struct[0].Gy[43,116] = -1.84896616921897
        struct[0].Gy[43,117] = -9.21038227100566
        struct[0].Gy[43,118] = -0.793238195499528
        struct[0].Gy[43,119] = -9.00835072044485
        struct[0].Gy[43,187] = 1
        struct[0].Gy[44,40] = 9.00835072044484
        struct[0].Gy[44,41] = -0.793238195499527
        struct[0].Gy[44,42] = 9.21038227100566
        struct[0].Gy[44,43] = -1.84896616921897
        struct[0].Gy[44,44] = -30.9517475172273
        struct[0].Gy[44,45] = -5.65456401516768
        struct[0].Gy[44,46] = 9.21038227100566
        struct[0].Gy[44,47] = -1.84896616921897
        struct[0].Gy[44,112] = -9.00835072044484
        struct[0].Gy[44,113] = 0.793238195499527
        struct[0].Gy[44,114] = -9.21038227100566
        struct[0].Gy[44,115] = 1.84896616921897
        struct[0].Gy[44,116] = 30.9517475172273
        struct[0].Gy[44,117] = 5.65456401516768
        struct[0].Gy[44,118] = -9.21038227100566
        struct[0].Gy[44,119] = 1.84896616921897
        struct[0].Gy[44,188] = 1
        struct[0].Gy[45,40] = 0.793238195499527
        struct[0].Gy[45,41] = 9.00835072044484
        struct[0].Gy[45,42] = 1.84896616921897
        struct[0].Gy[45,43] = 9.21038227100566
        struct[0].Gy[45,44] = 5.65456401516768
        struct[0].Gy[45,45] = -30.9517475172273
        struct[0].Gy[45,46] = 1.84896616921897
        struct[0].Gy[45,47] = 9.21038227100566
        struct[0].Gy[45,112] = -0.793238195499527
        struct[0].Gy[45,113] = -9.00835072044484
        struct[0].Gy[45,114] = -1.84896616921897
        struct[0].Gy[45,115] = -9.21038227100566
        struct[0].Gy[45,116] = -5.65456401516768
        struct[0].Gy[45,117] = 30.9517475172273
        struct[0].Gy[45,118] = -1.84896616921897
        struct[0].Gy[45,119] = -9.21038227100566
        struct[0].Gy[45,189] = 1
        struct[0].Gy[46,40] = 9.21038227100566
        struct[0].Gy[46,41] = -1.84896616921897
        struct[0].Gy[46,42] = 9.00835072044485
        struct[0].Gy[46,43] = -0.793238195499527
        struct[0].Gy[46,44] = 9.21038227100566
        struct[0].Gy[46,45] = -1.84896616921897
        struct[0].Gy[46,46] = -30.9767475172273
        struct[0].Gy[46,47] = -5.65456401516768
        struct[0].Gy[46,112] = -9.21038227100566
        struct[0].Gy[46,113] = 1.84896616921897
        struct[0].Gy[46,114] = -9.00835072044485
        struct[0].Gy[46,115] = 0.793238195499527
        struct[0].Gy[46,116] = -9.21038227100566
        struct[0].Gy[46,117] = 1.84896616921897
        struct[0].Gy[46,118] = 30.9517475172273
        struct[0].Gy[46,119] = 5.65456401516768
        struct[0].Gy[46,190] = 1
        struct[0].Gy[47,40] = 1.84896616921897
        struct[0].Gy[47,41] = 9.21038227100566
        struct[0].Gy[47,42] = 0.793238195499527
        struct[0].Gy[47,43] = 9.00835072044485
        struct[0].Gy[47,44] = 1.84896616921897
        struct[0].Gy[47,45] = 9.21038227100566
        struct[0].Gy[47,46] = 5.65456401516768
        struct[0].Gy[47,47] = -30.9767475172273
        struct[0].Gy[47,112] = -1.84896616921897
        struct[0].Gy[47,113] = -9.21038227100566
        struct[0].Gy[47,114] = -0.793238195499527
        struct[0].Gy[47,115] = -9.00835072044485
        struct[0].Gy[47,116] = -1.84896616921897
        struct[0].Gy[47,117] = -9.21038227100566
        struct[0].Gy[47,118] = -5.65456401516768
        struct[0].Gy[47,119] = 30.9517475172273
        struct[0].Gy[47,191] = 1
        struct[0].Gy[48,0] = 105.571769313180
        struct[0].Gy[48,1] = 54.0657727682604
        struct[0].Gy[48,2] = -39.6392229058202
        struct[0].Gy[48,3] = -10.2713736253513
        struct[0].Gy[48,4] = -24.9575997948692
        struct[0].Gy[48,5] = -23.2849644809540
        struct[0].Gy[48,6] = -39.6392229058202
        struct[0].Gy[48,7] = -10.2713736253513
        struct[0].Gy[48,48] = -211.143538626360
        struct[0].Gy[48,49] = -108.131545536521
        struct[0].Gy[48,50] = 79.2784458116403
        struct[0].Gy[48,51] = 20.5427472507027
        struct[0].Gy[48,52] = 49.9151995897384
        struct[0].Gy[48,53] = 46.5699289619080
        struct[0].Gy[48,54] = 79.2784458116403
        struct[0].Gy[48,55] = 20.5427472507027
        struct[0].Gy[48,56] = 105.571769313180
        struct[0].Gy[48,57] = 54.0657727682604
        struct[0].Gy[48,58] = -39.6392229058202
        struct[0].Gy[48,59] = -10.2713736253513
        struct[0].Gy[48,60] = -24.9575997948692
        struct[0].Gy[48,61] = -23.2849644809540
        struct[0].Gy[48,62] = -39.6392229058202
        struct[0].Gy[48,63] = -10.2713736253513
        struct[0].Gy[49,0] = -54.0657727682604
        struct[0].Gy[49,1] = 105.571769313180
        struct[0].Gy[49,2] = 10.2713736253513
        struct[0].Gy[49,3] = -39.6392229058202
        struct[0].Gy[49,4] = 23.2849644809540
        struct[0].Gy[49,5] = -24.9575997948692
        struct[0].Gy[49,6] = 10.2713736253513
        struct[0].Gy[49,7] = -39.6392229058202
        struct[0].Gy[49,48] = 108.131545536521
        struct[0].Gy[49,49] = -211.143538626360
        struct[0].Gy[49,50] = -20.5427472507027
        struct[0].Gy[49,51] = 79.2784458116403
        struct[0].Gy[49,52] = -46.5699289619080
        struct[0].Gy[49,53] = 49.9151995897384
        struct[0].Gy[49,54] = -20.5427472507027
        struct[0].Gy[49,55] = 79.2784458116403
        struct[0].Gy[49,56] = -54.0657727682604
        struct[0].Gy[49,57] = 105.571769313180
        struct[0].Gy[49,58] = 10.2713736253513
        struct[0].Gy[49,59] = -39.6392229058202
        struct[0].Gy[49,60] = 23.2849644809540
        struct[0].Gy[49,61] = -24.9575997948692
        struct[0].Gy[49,62] = 10.2713736253513
        struct[0].Gy[49,63] = -39.6392229058202
        struct[0].Gy[50,0] = -39.6392229058202
        struct[0].Gy[50,1] = -10.2713736253513
        struct[0].Gy[50,2] = 105.571769313180
        struct[0].Gy[50,3] = 54.0657727682604
        struct[0].Gy[50,4] = -39.6392229058201
        struct[0].Gy[50,5] = -10.2713736253513
        struct[0].Gy[50,6] = -24.9575997948692
        struct[0].Gy[50,7] = -23.2849644809540
        struct[0].Gy[50,48] = 79.2784458116403
        struct[0].Gy[50,49] = 20.5427472507027
        struct[0].Gy[50,50] = -211.143538626360
        struct[0].Gy[50,51] = -108.131545536521
        struct[0].Gy[50,52] = 79.2784458116403
        struct[0].Gy[50,53] = 20.5427472507027
        struct[0].Gy[50,54] = 49.9151995897384
        struct[0].Gy[50,55] = 46.5699289619080
        struct[0].Gy[50,56] = -39.6392229058202
        struct[0].Gy[50,57] = -10.2713736253513
        struct[0].Gy[50,58] = 105.571769313180
        struct[0].Gy[50,59] = 54.0657727682604
        struct[0].Gy[50,60] = -39.6392229058201
        struct[0].Gy[50,61] = -10.2713736253513
        struct[0].Gy[50,62] = -24.9575997948692
        struct[0].Gy[50,63] = -23.2849644809540
        struct[0].Gy[51,0] = 10.2713736253513
        struct[0].Gy[51,1] = -39.6392229058202
        struct[0].Gy[51,2] = -54.0657727682604
        struct[0].Gy[51,3] = 105.571769313180
        struct[0].Gy[51,4] = 10.2713736253513
        struct[0].Gy[51,5] = -39.6392229058201
        struct[0].Gy[51,6] = 23.2849644809540
        struct[0].Gy[51,7] = -24.9575997948692
        struct[0].Gy[51,48] = -20.5427472507027
        struct[0].Gy[51,49] = 79.2784458116403
        struct[0].Gy[51,50] = 108.131545536521
        struct[0].Gy[51,51] = -211.143538626360
        struct[0].Gy[51,52] = -20.5427472507027
        struct[0].Gy[51,53] = 79.2784458116403
        struct[0].Gy[51,54] = -46.5699289619080
        struct[0].Gy[51,55] = 49.9151995897384
        struct[0].Gy[51,56] = 10.2713736253513
        struct[0].Gy[51,57] = -39.6392229058202
        struct[0].Gy[51,58] = -54.0657727682604
        struct[0].Gy[51,59] = 105.571769313180
        struct[0].Gy[51,60] = 10.2713736253513
        struct[0].Gy[51,61] = -39.6392229058201
        struct[0].Gy[51,62] = 23.2849644809540
        struct[0].Gy[51,63] = -24.9575997948692
        struct[0].Gy[52,0] = -24.9575997948692
        struct[0].Gy[52,1] = -23.2849644809540
        struct[0].Gy[52,2] = -39.6392229058202
        struct[0].Gy[52,3] = -10.2713736253513
        struct[0].Gy[52,4] = 105.571769313180
        struct[0].Gy[52,5] = 54.0657727682604
        struct[0].Gy[52,6] = -39.6392229058201
        struct[0].Gy[52,7] = -10.2713736253514
        struct[0].Gy[52,48] = 49.9151995897384
        struct[0].Gy[52,49] = 46.5699289619080
        struct[0].Gy[52,50] = 79.2784458116404
        struct[0].Gy[52,51] = 20.5427472507027
        struct[0].Gy[52,52] = -211.143538626360
        struct[0].Gy[52,53] = -108.131545536521
        struct[0].Gy[52,54] = 79.2784458116403
        struct[0].Gy[52,55] = 20.5427472507027
        struct[0].Gy[52,56] = -24.9575997948692
        struct[0].Gy[52,57] = -23.2849644809540
        struct[0].Gy[52,58] = -39.6392229058202
        struct[0].Gy[52,59] = -10.2713736253513
        struct[0].Gy[52,60] = 105.571769313180
        struct[0].Gy[52,61] = 54.0657727682604
        struct[0].Gy[52,62] = -39.6392229058201
        struct[0].Gy[52,63] = -10.2713736253514
        struct[0].Gy[53,0] = 23.2849644809540
        struct[0].Gy[53,1] = -24.9575997948692
        struct[0].Gy[53,2] = 10.2713736253513
        struct[0].Gy[53,3] = -39.6392229058202
        struct[0].Gy[53,4] = -54.0657727682604
        struct[0].Gy[53,5] = 105.571769313180
        struct[0].Gy[53,6] = 10.2713736253514
        struct[0].Gy[53,7] = -39.6392229058201
        struct[0].Gy[53,48] = -46.5699289619080
        struct[0].Gy[53,49] = 49.9151995897384
        struct[0].Gy[53,50] = -20.5427472507027
        struct[0].Gy[53,51] = 79.2784458116404
        struct[0].Gy[53,52] = 108.131545536521
        struct[0].Gy[53,53] = -211.143538626360
        struct[0].Gy[53,54] = -20.5427472507027
        struct[0].Gy[53,55] = 79.2784458116403
        struct[0].Gy[53,56] = 23.2849644809540
        struct[0].Gy[53,57] = -24.9575997948692
        struct[0].Gy[53,58] = 10.2713736253513
        struct[0].Gy[53,59] = -39.6392229058202
        struct[0].Gy[53,60] = -54.0657727682604
        struct[0].Gy[53,61] = 105.571769313180
        struct[0].Gy[53,62] = 10.2713736253514
        struct[0].Gy[53,63] = -39.6392229058201
        struct[0].Gy[54,0] = -39.6392229058202
        struct[0].Gy[54,1] = -10.2713736253513
        struct[0].Gy[54,2] = -24.9575997948692
        struct[0].Gy[54,3] = -23.2849644809540
        struct[0].Gy[54,4] = -39.6392229058202
        struct[0].Gy[54,5] = -10.2713736253513
        struct[0].Gy[54,6] = 105.571769313180
        struct[0].Gy[54,7] = 54.0657727682604
        struct[0].Gy[54,48] = 79.2784458116403
        struct[0].Gy[54,49] = 20.5427472507027
        struct[0].Gy[54,50] = 49.9151995897384
        struct[0].Gy[54,51] = 46.5699289619080
        struct[0].Gy[54,52] = 79.2784458116403
        struct[0].Gy[54,53] = 20.5427472507027
        struct[0].Gy[54,54] = -211.168538626360
        struct[0].Gy[54,55] = -108.131545536521
        struct[0].Gy[54,56] = -39.6392229058202
        struct[0].Gy[54,57] = -10.2713736253513
        struct[0].Gy[54,58] = -24.9575997948692
        struct[0].Gy[54,59] = -23.2849644809540
        struct[0].Gy[54,60] = -39.6392229058202
        struct[0].Gy[54,61] = -10.2713736253513
        struct[0].Gy[54,62] = 105.571769313180
        struct[0].Gy[54,63] = 54.0657727682604
        struct[0].Gy[55,0] = 10.2713736253513
        struct[0].Gy[55,1] = -39.6392229058202
        struct[0].Gy[55,2] = 23.2849644809540
        struct[0].Gy[55,3] = -24.9575997948692
        struct[0].Gy[55,4] = 10.2713736253513
        struct[0].Gy[55,5] = -39.6392229058202
        struct[0].Gy[55,6] = -54.0657727682604
        struct[0].Gy[55,7] = 105.571769313180
        struct[0].Gy[55,48] = -20.5427472507027
        struct[0].Gy[55,49] = 79.2784458116403
        struct[0].Gy[55,50] = -46.5699289619080
        struct[0].Gy[55,51] = 49.9151995897384
        struct[0].Gy[55,52] = -20.5427472507027
        struct[0].Gy[55,53] = 79.2784458116403
        struct[0].Gy[55,54] = 108.131545536521
        struct[0].Gy[55,55] = -211.168538626360
        struct[0].Gy[55,56] = 10.2713736253513
        struct[0].Gy[55,57] = -39.6392229058202
        struct[0].Gy[55,58] = 23.2849644809540
        struct[0].Gy[55,59] = -24.9575997948692
        struct[0].Gy[55,60] = 10.2713736253513
        struct[0].Gy[55,61] = -39.6392229058202
        struct[0].Gy[55,62] = -54.0657727682604
        struct[0].Gy[55,63] = 105.571769313180
        struct[0].Gy[56,8] = 26.5300693004806
        struct[0].Gy[56,9] = 4.84676915585801
        struct[0].Gy[56,10] = -7.89461337514770
        struct[0].Gy[56,11] = 1.58482814504483
        struct[0].Gy[56,12] = -7.72144347466701
        struct[0].Gy[56,13] = 0.679918453285310
        struct[0].Gy[56,14] = -7.89461337514771
        struct[0].Gy[56,15] = 1.58482814504483
        struct[0].Gy[56,48] = 105.571769313180
        struct[0].Gy[56,49] = 54.0657727682604
        struct[0].Gy[56,50] = -39.6392229058202
        struct[0].Gy[56,51] = -10.2713736253513
        struct[0].Gy[56,52] = -24.9575997948692
        struct[0].Gy[56,53] = -23.2849644809540
        struct[0].Gy[56,54] = -39.6392229058202
        struct[0].Gy[56,55] = -10.2713736253513
        struct[0].Gy[56,56] = -237.673607926841
        struct[0].Gy[56,57] = -112.978314692379
        struct[0].Gy[56,58] = 87.1730591867880
        struct[0].Gy[56,59] = 18.9579191056578
        struct[0].Gy[56,60] = 57.6366430644055
        struct[0].Gy[56,61] = 45.8900105086227
        struct[0].Gy[56,62] = 87.1730591867880
        struct[0].Gy[56,63] = 18.9579191056579
        struct[0].Gy[56,64] = 105.571769313180
        struct[0].Gy[56,65] = 54.0657727682604
        struct[0].Gy[56,66] = -39.6392229058202
        struct[0].Gy[56,67] = -10.2713736253513
        struct[0].Gy[56,68] = -24.9575997948692
        struct[0].Gy[56,69] = -23.2849644809540
        struct[0].Gy[56,70] = -39.6392229058202
        struct[0].Gy[56,71] = -10.2713736253513
        struct[0].Gy[57,8] = -4.84676915585801
        struct[0].Gy[57,9] = 26.5300693004806
        struct[0].Gy[57,10] = -1.58482814504483
        struct[0].Gy[57,11] = -7.89461337514770
        struct[0].Gy[57,12] = -0.679918453285310
        struct[0].Gy[57,13] = -7.72144347466701
        struct[0].Gy[57,14] = -1.58482814504483
        struct[0].Gy[57,15] = -7.89461337514771
        struct[0].Gy[57,48] = -54.0657727682604
        struct[0].Gy[57,49] = 105.571769313180
        struct[0].Gy[57,50] = 10.2713736253513
        struct[0].Gy[57,51] = -39.6392229058202
        struct[0].Gy[57,52] = 23.2849644809540
        struct[0].Gy[57,53] = -24.9575997948692
        struct[0].Gy[57,54] = 10.2713736253513
        struct[0].Gy[57,55] = -39.6392229058202
        struct[0].Gy[57,56] = 112.978314692379
        struct[0].Gy[57,57] = -237.673607926841
        struct[0].Gy[57,58] = -18.9579191056578
        struct[0].Gy[57,59] = 87.1730591867880
        struct[0].Gy[57,60] = -45.8900105086227
        struct[0].Gy[57,61] = 57.6366430644055
        struct[0].Gy[57,62] = -18.9579191056579
        struct[0].Gy[57,63] = 87.1730591867880
        struct[0].Gy[57,64] = -54.0657727682604
        struct[0].Gy[57,65] = 105.571769313180
        struct[0].Gy[57,66] = 10.2713736253513
        struct[0].Gy[57,67] = -39.6392229058202
        struct[0].Gy[57,68] = 23.2849644809540
        struct[0].Gy[57,69] = -24.9575997948692
        struct[0].Gy[57,70] = 10.2713736253513
        struct[0].Gy[57,71] = -39.6392229058202
        struct[0].Gy[58,8] = -7.89461337514771
        struct[0].Gy[58,9] = 1.58482814504483
        struct[0].Gy[58,10] = 26.5300693004806
        struct[0].Gy[58,11] = 4.84676915585801
        struct[0].Gy[58,12] = -7.89461337514771
        struct[0].Gy[58,13] = 1.58482814504483
        struct[0].Gy[58,14] = -7.72144347466701
        struct[0].Gy[58,15] = 0.679918453285308
        struct[0].Gy[58,48] = -39.6392229058202
        struct[0].Gy[58,49] = -10.2713736253513
        struct[0].Gy[58,50] = 105.571769313180
        struct[0].Gy[58,51] = 54.0657727682604
        struct[0].Gy[58,52] = -39.6392229058201
        struct[0].Gy[58,53] = -10.2713736253513
        struct[0].Gy[58,54] = -24.9575997948692
        struct[0].Gy[58,55] = -23.2849644809540
        struct[0].Gy[58,56] = 87.1730591867880
        struct[0].Gy[58,57] = 18.9579191056579
        struct[0].Gy[58,58] = -237.673607926841
        struct[0].Gy[58,59] = -112.978314692379
        struct[0].Gy[58,60] = 87.1730591867880
        struct[0].Gy[58,61] = 18.9579191056579
        struct[0].Gy[58,62] = 57.6366430644054
        struct[0].Gy[58,63] = 45.8900105086227
        struct[0].Gy[58,64] = -39.6392229058202
        struct[0].Gy[58,65] = -10.2713736253513
        struct[0].Gy[58,66] = 105.571769313180
        struct[0].Gy[58,67] = 54.0657727682604
        struct[0].Gy[58,68] = -39.6392229058201
        struct[0].Gy[58,69] = -10.2713736253513
        struct[0].Gy[58,70] = -24.9575997948692
        struct[0].Gy[58,71] = -23.2849644809540
        struct[0].Gy[59,8] = -1.58482814504483
        struct[0].Gy[59,9] = -7.89461337514771
        struct[0].Gy[59,10] = -4.84676915585801
        struct[0].Gy[59,11] = 26.5300693004806
        struct[0].Gy[59,12] = -1.58482814504483
        struct[0].Gy[59,13] = -7.89461337514771
        struct[0].Gy[59,14] = -0.679918453285308
        struct[0].Gy[59,15] = -7.72144347466701
        struct[0].Gy[59,48] = 10.2713736253513
        struct[0].Gy[59,49] = -39.6392229058202
        struct[0].Gy[59,50] = -54.0657727682604
        struct[0].Gy[59,51] = 105.571769313180
        struct[0].Gy[59,52] = 10.2713736253513
        struct[0].Gy[59,53] = -39.6392229058201
        struct[0].Gy[59,54] = 23.2849644809540
        struct[0].Gy[59,55] = -24.9575997948692
        struct[0].Gy[59,56] = -18.9579191056579
        struct[0].Gy[59,57] = 87.1730591867880
        struct[0].Gy[59,58] = 112.978314692379
        struct[0].Gy[59,59] = -237.673607926841
        struct[0].Gy[59,60] = -18.9579191056579
        struct[0].Gy[59,61] = 87.1730591867880
        struct[0].Gy[59,62] = -45.8900105086227
        struct[0].Gy[59,63] = 57.6366430644054
        struct[0].Gy[59,64] = 10.2713736253513
        struct[0].Gy[59,65] = -39.6392229058202
        struct[0].Gy[59,66] = -54.0657727682604
        struct[0].Gy[59,67] = 105.571769313180
        struct[0].Gy[59,68] = 10.2713736253513
        struct[0].Gy[59,69] = -39.6392229058201
        struct[0].Gy[59,70] = 23.2849644809540
        struct[0].Gy[59,71] = -24.9575997948692
        struct[0].Gy[60,8] = -7.72144347466701
        struct[0].Gy[60,9] = 0.679918453285310
        struct[0].Gy[60,10] = -7.89461337514770
        struct[0].Gy[60,11] = 1.58482814504483
        struct[0].Gy[60,12] = 26.5300693004806
        struct[0].Gy[60,13] = 4.84676915585801
        struct[0].Gy[60,14] = -7.89461337514771
        struct[0].Gy[60,15] = 1.58482814504483
        struct[0].Gy[60,48] = -24.9575997948692
        struct[0].Gy[60,49] = -23.2849644809540
        struct[0].Gy[60,50] = -39.6392229058202
        struct[0].Gy[60,51] = -10.2713736253513
        struct[0].Gy[60,52] = 105.571769313180
        struct[0].Gy[60,53] = 54.0657727682604
        struct[0].Gy[60,54] = -39.6392229058201
        struct[0].Gy[60,55] = -10.2713736253514
        struct[0].Gy[60,56] = 57.6366430644054
        struct[0].Gy[60,57] = 45.8900105086227
        struct[0].Gy[60,58] = 87.1730591867881
        struct[0].Gy[60,59] = 18.9579191056578
        struct[0].Gy[60,60] = -237.673607926841
        struct[0].Gy[60,61] = -112.978314692379
        struct[0].Gy[60,62] = 87.1730591867880
        struct[0].Gy[60,63] = 18.9579191056579
        struct[0].Gy[60,64] = -24.9575997948692
        struct[0].Gy[60,65] = -23.2849644809540
        struct[0].Gy[60,66] = -39.6392229058202
        struct[0].Gy[60,67] = -10.2713736253513
        struct[0].Gy[60,68] = 105.571769313180
        struct[0].Gy[60,69] = 54.0657727682604
        struct[0].Gy[60,70] = -39.6392229058201
        struct[0].Gy[60,71] = -10.2713736253514
        struct[0].Gy[61,8] = -0.679918453285310
        struct[0].Gy[61,9] = -7.72144347466701
        struct[0].Gy[61,10] = -1.58482814504483
        struct[0].Gy[61,11] = -7.89461337514770
        struct[0].Gy[61,12] = -4.84676915585801
        struct[0].Gy[61,13] = 26.5300693004806
        struct[0].Gy[61,14] = -1.58482814504483
        struct[0].Gy[61,15] = -7.89461337514771
        struct[0].Gy[61,48] = 23.2849644809540
        struct[0].Gy[61,49] = -24.9575997948692
        struct[0].Gy[61,50] = 10.2713736253513
        struct[0].Gy[61,51] = -39.6392229058202
        struct[0].Gy[61,52] = -54.0657727682604
        struct[0].Gy[61,53] = 105.571769313180
        struct[0].Gy[61,54] = 10.2713736253514
        struct[0].Gy[61,55] = -39.6392229058201
        struct[0].Gy[61,56] = -45.8900105086227
        struct[0].Gy[61,57] = 57.6366430644054
        struct[0].Gy[61,58] = -18.9579191056578
        struct[0].Gy[61,59] = 87.1730591867881
        struct[0].Gy[61,60] = 112.978314692379
        struct[0].Gy[61,61] = -237.673607926841
        struct[0].Gy[61,62] = -18.9579191056579
        struct[0].Gy[61,63] = 87.1730591867880
        struct[0].Gy[61,64] = 23.2849644809540
        struct[0].Gy[61,65] = -24.9575997948692
        struct[0].Gy[61,66] = 10.2713736253513
        struct[0].Gy[61,67] = -39.6392229058202
        struct[0].Gy[61,68] = -54.0657727682604
        struct[0].Gy[61,69] = 105.571769313180
        struct[0].Gy[61,70] = 10.2713736253514
        struct[0].Gy[61,71] = -39.6392229058201
        struct[0].Gy[62,8] = -7.89461337514771
        struct[0].Gy[62,9] = 1.58482814504483
        struct[0].Gy[62,10] = -7.72144347466701
        struct[0].Gy[62,11] = 0.679918453285308
        struct[0].Gy[62,12] = -7.89461337514771
        struct[0].Gy[62,13] = 1.58482814504483
        struct[0].Gy[62,14] = 26.5300693004806
        struct[0].Gy[62,15] = 4.84676915585801
        struct[0].Gy[62,48] = -39.6392229058202
        struct[0].Gy[62,49] = -10.2713736253513
        struct[0].Gy[62,50] = -24.9575997948692
        struct[0].Gy[62,51] = -23.2849644809540
        struct[0].Gy[62,52] = -39.6392229058202
        struct[0].Gy[62,53] = -10.2713736253513
        struct[0].Gy[62,54] = 105.571769313180
        struct[0].Gy[62,55] = 54.0657727682604
        struct[0].Gy[62,56] = 87.1730591867880
        struct[0].Gy[62,57] = 18.9579191056578
        struct[0].Gy[62,58] = 57.6366430644054
        struct[0].Gy[62,59] = 45.8900105086227
        struct[0].Gy[62,60] = 87.1730591867880
        struct[0].Gy[62,61] = 18.9579191056579
        struct[0].Gy[62,62] = -237.673607926841
        struct[0].Gy[62,63] = -112.978314692379
        struct[0].Gy[62,64] = -39.6392229058202
        struct[0].Gy[62,65] = -10.2713736253513
        struct[0].Gy[62,66] = -24.9575997948692
        struct[0].Gy[62,67] = -23.2849644809540
        struct[0].Gy[62,68] = -39.6392229058202
        struct[0].Gy[62,69] = -10.2713736253513
        struct[0].Gy[62,70] = 105.571769313180
        struct[0].Gy[62,71] = 54.0657727682604
        struct[0].Gy[63,8] = -1.58482814504483
        struct[0].Gy[63,9] = -7.89461337514771
        struct[0].Gy[63,10] = -0.679918453285308
        struct[0].Gy[63,11] = -7.72144347466701
        struct[0].Gy[63,12] = -1.58482814504483
        struct[0].Gy[63,13] = -7.89461337514771
        struct[0].Gy[63,14] = -4.84676915585801
        struct[0].Gy[63,15] = 26.5300693004806
        struct[0].Gy[63,48] = 10.2713736253513
        struct[0].Gy[63,49] = -39.6392229058202
        struct[0].Gy[63,50] = 23.2849644809540
        struct[0].Gy[63,51] = -24.9575997948692
        struct[0].Gy[63,52] = 10.2713736253513
        struct[0].Gy[63,53] = -39.6392229058202
        struct[0].Gy[63,54] = -54.0657727682604
        struct[0].Gy[63,55] = 105.571769313180
        struct[0].Gy[63,56] = -18.9579191056578
        struct[0].Gy[63,57] = 87.1730591867880
        struct[0].Gy[63,58] = -45.8900105086227
        struct[0].Gy[63,59] = 57.6366430644054
        struct[0].Gy[63,60] = -18.9579191056579
        struct[0].Gy[63,61] = 87.1730591867880
        struct[0].Gy[63,62] = 112.978314692379
        struct[0].Gy[63,63] = -237.673607926841
        struct[0].Gy[63,64] = 10.2713736253513
        struct[0].Gy[63,65] = -39.6392229058202
        struct[0].Gy[63,66] = 23.2849644809540
        struct[0].Gy[63,67] = -24.9575997948692
        struct[0].Gy[63,68] = 10.2713736253513
        struct[0].Gy[63,69] = -39.6392229058202
        struct[0].Gy[63,70] = -54.0657727682604
        struct[0].Gy[63,71] = 105.571769313180
        struct[0].Gy[64,56] = 105.571769313180
        struct[0].Gy[64,57] = 54.0657727682604
        struct[0].Gy[64,58] = -39.6392229058202
        struct[0].Gy[64,59] = -10.2713736253513
        struct[0].Gy[64,60] = -24.9575997948692
        struct[0].Gy[64,61] = -23.2849644809540
        struct[0].Gy[64,62] = -39.6392229058202
        struct[0].Gy[64,63] = -10.2713736253513
        struct[0].Gy[64,64] = -237.673607926841
        struct[0].Gy[64,65] = -112.978314692379
        struct[0].Gy[64,66] = 87.1730591867880
        struct[0].Gy[64,67] = 18.9579191056578
        struct[0].Gy[64,68] = 57.6366430644055
        struct[0].Gy[64,69] = 45.8900105086227
        struct[0].Gy[64,70] = 87.1730591867880
        struct[0].Gy[64,71] = 18.9579191056579
        struct[0].Gy[64,72] = 105.571769313180
        struct[0].Gy[64,73] = 54.0657727682604
        struct[0].Gy[64,74] = -39.6392229058202
        struct[0].Gy[64,75] = -10.2713736253513
        struct[0].Gy[64,76] = -24.9575997948692
        struct[0].Gy[64,77] = -23.2849644809540
        struct[0].Gy[64,78] = -39.6392229058202
        struct[0].Gy[64,79] = -10.2713736253513
        struct[0].Gy[64,120] = 26.5300693004806
        struct[0].Gy[64,121] = 4.84676915585801
        struct[0].Gy[64,122] = -7.89461337514770
        struct[0].Gy[64,123] = 1.58482814504483
        struct[0].Gy[64,124] = -7.72144347466701
        struct[0].Gy[64,125] = 0.679918453285310
        struct[0].Gy[64,126] = -7.89461337514771
        struct[0].Gy[64,127] = 1.58482814504483
        struct[0].Gy[65,56] = -54.0657727682604
        struct[0].Gy[65,57] = 105.571769313180
        struct[0].Gy[65,58] = 10.2713736253513
        struct[0].Gy[65,59] = -39.6392229058202
        struct[0].Gy[65,60] = 23.2849644809540
        struct[0].Gy[65,61] = -24.9575997948692
        struct[0].Gy[65,62] = 10.2713736253513
        struct[0].Gy[65,63] = -39.6392229058202
        struct[0].Gy[65,64] = 112.978314692379
        struct[0].Gy[65,65] = -237.673607926841
        struct[0].Gy[65,66] = -18.9579191056578
        struct[0].Gy[65,67] = 87.1730591867880
        struct[0].Gy[65,68] = -45.8900105086227
        struct[0].Gy[65,69] = 57.6366430644055
        struct[0].Gy[65,70] = -18.9579191056579
        struct[0].Gy[65,71] = 87.1730591867880
        struct[0].Gy[65,72] = -54.0657727682604
        struct[0].Gy[65,73] = 105.571769313180
        struct[0].Gy[65,74] = 10.2713736253513
        struct[0].Gy[65,75] = -39.6392229058202
        struct[0].Gy[65,76] = 23.2849644809540
        struct[0].Gy[65,77] = -24.9575997948692
        struct[0].Gy[65,78] = 10.2713736253513
        struct[0].Gy[65,79] = -39.6392229058202
        struct[0].Gy[65,120] = -4.84676915585801
        struct[0].Gy[65,121] = 26.5300693004806
        struct[0].Gy[65,122] = -1.58482814504483
        struct[0].Gy[65,123] = -7.89461337514770
        struct[0].Gy[65,124] = -0.679918453285310
        struct[0].Gy[65,125] = -7.72144347466701
        struct[0].Gy[65,126] = -1.58482814504483
        struct[0].Gy[65,127] = -7.89461337514771
        struct[0].Gy[66,56] = -39.6392229058202
        struct[0].Gy[66,57] = -10.2713736253513
        struct[0].Gy[66,58] = 105.571769313180
        struct[0].Gy[66,59] = 54.0657727682604
        struct[0].Gy[66,60] = -39.6392229058201
        struct[0].Gy[66,61] = -10.2713736253513
        struct[0].Gy[66,62] = -24.9575997948692
        struct[0].Gy[66,63] = -23.2849644809540
        struct[0].Gy[66,64] = 87.1730591867880
        struct[0].Gy[66,65] = 18.9579191056579
        struct[0].Gy[66,66] = -237.673607926841
        struct[0].Gy[66,67] = -112.978314692379
        struct[0].Gy[66,68] = 87.1730591867880
        struct[0].Gy[66,69] = 18.9579191056579
        struct[0].Gy[66,70] = 57.6366430644054
        struct[0].Gy[66,71] = 45.8900105086227
        struct[0].Gy[66,72] = -39.6392229058202
        struct[0].Gy[66,73] = -10.2713736253513
        struct[0].Gy[66,74] = 105.571769313180
        struct[0].Gy[66,75] = 54.0657727682604
        struct[0].Gy[66,76] = -39.6392229058201
        struct[0].Gy[66,77] = -10.2713736253513
        struct[0].Gy[66,78] = -24.9575997948692
        struct[0].Gy[66,79] = -23.2849644809540
        struct[0].Gy[66,120] = -7.89461337514771
        struct[0].Gy[66,121] = 1.58482814504483
        struct[0].Gy[66,122] = 26.5300693004806
        struct[0].Gy[66,123] = 4.84676915585801
        struct[0].Gy[66,124] = -7.89461337514771
        struct[0].Gy[66,125] = 1.58482814504483
        struct[0].Gy[66,126] = -7.72144347466701
        struct[0].Gy[66,127] = 0.679918453285308
        struct[0].Gy[67,56] = 10.2713736253513
        struct[0].Gy[67,57] = -39.6392229058202
        struct[0].Gy[67,58] = -54.0657727682604
        struct[0].Gy[67,59] = 105.571769313180
        struct[0].Gy[67,60] = 10.2713736253513
        struct[0].Gy[67,61] = -39.6392229058201
        struct[0].Gy[67,62] = 23.2849644809540
        struct[0].Gy[67,63] = -24.9575997948692
        struct[0].Gy[67,64] = -18.9579191056579
        struct[0].Gy[67,65] = 87.1730591867880
        struct[0].Gy[67,66] = 112.978314692379
        struct[0].Gy[67,67] = -237.673607926841
        struct[0].Gy[67,68] = -18.9579191056579
        struct[0].Gy[67,69] = 87.1730591867880
        struct[0].Gy[67,70] = -45.8900105086227
        struct[0].Gy[67,71] = 57.6366430644054
        struct[0].Gy[67,72] = 10.2713736253513
        struct[0].Gy[67,73] = -39.6392229058202
        struct[0].Gy[67,74] = -54.0657727682604
        struct[0].Gy[67,75] = 105.571769313180
        struct[0].Gy[67,76] = 10.2713736253513
        struct[0].Gy[67,77] = -39.6392229058201
        struct[0].Gy[67,78] = 23.2849644809540
        struct[0].Gy[67,79] = -24.9575997948692
        struct[0].Gy[67,120] = -1.58482814504483
        struct[0].Gy[67,121] = -7.89461337514771
        struct[0].Gy[67,122] = -4.84676915585801
        struct[0].Gy[67,123] = 26.5300693004806
        struct[0].Gy[67,124] = -1.58482814504483
        struct[0].Gy[67,125] = -7.89461337514771
        struct[0].Gy[67,126] = -0.679918453285308
        struct[0].Gy[67,127] = -7.72144347466701
        struct[0].Gy[68,56] = -24.9575997948692
        struct[0].Gy[68,57] = -23.2849644809540
        struct[0].Gy[68,58] = -39.6392229058202
        struct[0].Gy[68,59] = -10.2713736253513
        struct[0].Gy[68,60] = 105.571769313180
        struct[0].Gy[68,61] = 54.0657727682604
        struct[0].Gy[68,62] = -39.6392229058201
        struct[0].Gy[68,63] = -10.2713736253514
        struct[0].Gy[68,64] = 57.6366430644054
        struct[0].Gy[68,65] = 45.8900105086227
        struct[0].Gy[68,66] = 87.1730591867881
        struct[0].Gy[68,67] = 18.9579191056578
        struct[0].Gy[68,68] = -237.673607926841
        struct[0].Gy[68,69] = -112.978314692379
        struct[0].Gy[68,70] = 87.1730591867880
        struct[0].Gy[68,71] = 18.9579191056579
        struct[0].Gy[68,72] = -24.9575997948692
        struct[0].Gy[68,73] = -23.2849644809540
        struct[0].Gy[68,74] = -39.6392229058202
        struct[0].Gy[68,75] = -10.2713736253513
        struct[0].Gy[68,76] = 105.571769313180
        struct[0].Gy[68,77] = 54.0657727682604
        struct[0].Gy[68,78] = -39.6392229058201
        struct[0].Gy[68,79] = -10.2713736253514
        struct[0].Gy[68,120] = -7.72144347466701
        struct[0].Gy[68,121] = 0.679918453285310
        struct[0].Gy[68,122] = -7.89461337514770
        struct[0].Gy[68,123] = 1.58482814504483
        struct[0].Gy[68,124] = 26.5300693004806
        struct[0].Gy[68,125] = 4.84676915585801
        struct[0].Gy[68,126] = -7.89461337514771
        struct[0].Gy[68,127] = 1.58482814504483
        struct[0].Gy[69,56] = 23.2849644809540
        struct[0].Gy[69,57] = -24.9575997948692
        struct[0].Gy[69,58] = 10.2713736253513
        struct[0].Gy[69,59] = -39.6392229058202
        struct[0].Gy[69,60] = -54.0657727682604
        struct[0].Gy[69,61] = 105.571769313180
        struct[0].Gy[69,62] = 10.2713736253514
        struct[0].Gy[69,63] = -39.6392229058201
        struct[0].Gy[69,64] = -45.8900105086227
        struct[0].Gy[69,65] = 57.6366430644054
        struct[0].Gy[69,66] = -18.9579191056578
        struct[0].Gy[69,67] = 87.1730591867881
        struct[0].Gy[69,68] = 112.978314692379
        struct[0].Gy[69,69] = -237.673607926841
        struct[0].Gy[69,70] = -18.9579191056579
        struct[0].Gy[69,71] = 87.1730591867880
        struct[0].Gy[69,72] = 23.2849644809540
        struct[0].Gy[69,73] = -24.9575997948692
        struct[0].Gy[69,74] = 10.2713736253513
        struct[0].Gy[69,75] = -39.6392229058202
        struct[0].Gy[69,76] = -54.0657727682604
        struct[0].Gy[69,77] = 105.571769313180
        struct[0].Gy[69,78] = 10.2713736253514
        struct[0].Gy[69,79] = -39.6392229058201
        struct[0].Gy[69,120] = -0.679918453285310
        struct[0].Gy[69,121] = -7.72144347466701
        struct[0].Gy[69,122] = -1.58482814504483
        struct[0].Gy[69,123] = -7.89461337514770
        struct[0].Gy[69,124] = -4.84676915585801
        struct[0].Gy[69,125] = 26.5300693004806
        struct[0].Gy[69,126] = -1.58482814504483
        struct[0].Gy[69,127] = -7.89461337514771
        struct[0].Gy[70,56] = -39.6392229058202
        struct[0].Gy[70,57] = -10.2713736253513
        struct[0].Gy[70,58] = -24.9575997948692
        struct[0].Gy[70,59] = -23.2849644809540
        struct[0].Gy[70,60] = -39.6392229058202
        struct[0].Gy[70,61] = -10.2713736253513
        struct[0].Gy[70,62] = 105.571769313180
        struct[0].Gy[70,63] = 54.0657727682604
        struct[0].Gy[70,64] = 87.1730591867880
        struct[0].Gy[70,65] = 18.9579191056578
        struct[0].Gy[70,66] = 57.6366430644054
        struct[0].Gy[70,67] = 45.8900105086227
        struct[0].Gy[70,68] = 87.1730591867880
        struct[0].Gy[70,69] = 18.9579191056579
        struct[0].Gy[70,70] = -237.698607926841
        struct[0].Gy[70,71] = -112.978314692379
        struct[0].Gy[70,72] = -39.6392229058202
        struct[0].Gy[70,73] = -10.2713736253513
        struct[0].Gy[70,74] = -24.9575997948692
        struct[0].Gy[70,75] = -23.2849644809540
        struct[0].Gy[70,76] = -39.6392229058202
        struct[0].Gy[70,77] = -10.2713736253513
        struct[0].Gy[70,78] = 105.571769313180
        struct[0].Gy[70,79] = 54.0657727682604
        struct[0].Gy[70,120] = -7.89461337514771
        struct[0].Gy[70,121] = 1.58482814504483
        struct[0].Gy[70,122] = -7.72144347466701
        struct[0].Gy[70,123] = 0.679918453285308
        struct[0].Gy[70,124] = -7.89461337514771
        struct[0].Gy[70,125] = 1.58482814504483
        struct[0].Gy[70,126] = 26.5300693004806
        struct[0].Gy[70,127] = 4.84676915585801
        struct[0].Gy[71,56] = 10.2713736253513
        struct[0].Gy[71,57] = -39.6392229058202
        struct[0].Gy[71,58] = 23.2849644809540
        struct[0].Gy[71,59] = -24.9575997948692
        struct[0].Gy[71,60] = 10.2713736253513
        struct[0].Gy[71,61] = -39.6392229058202
        struct[0].Gy[71,62] = -54.0657727682604
        struct[0].Gy[71,63] = 105.571769313180
        struct[0].Gy[71,64] = -18.9579191056578
        struct[0].Gy[71,65] = 87.1730591867880
        struct[0].Gy[71,66] = -45.8900105086227
        struct[0].Gy[71,67] = 57.6366430644054
        struct[0].Gy[71,68] = -18.9579191056579
        struct[0].Gy[71,69] = 87.1730591867880
        struct[0].Gy[71,70] = 112.978314692379
        struct[0].Gy[71,71] = -237.698607926841
        struct[0].Gy[71,72] = 10.2713736253513
        struct[0].Gy[71,73] = -39.6392229058202
        struct[0].Gy[71,74] = 23.2849644809540
        struct[0].Gy[71,75] = -24.9575997948692
        struct[0].Gy[71,76] = 10.2713736253513
        struct[0].Gy[71,77] = -39.6392229058202
        struct[0].Gy[71,78] = -54.0657727682604
        struct[0].Gy[71,79] = 105.571769313180
        struct[0].Gy[71,120] = -1.58482814504483
        struct[0].Gy[71,121] = -7.89461337514771
        struct[0].Gy[71,122] = -0.679918453285308
        struct[0].Gy[71,123] = -7.72144347466701
        struct[0].Gy[71,124] = -1.58482814504483
        struct[0].Gy[71,125] = -7.89461337514771
        struct[0].Gy[71,126] = -4.84676915585801
        struct[0].Gy[71,127] = 26.5300693004806
        struct[0].Gy[72,64] = 105.571769313180
        struct[0].Gy[72,65] = 54.0657727682604
        struct[0].Gy[72,66] = -39.6392229058202
        struct[0].Gy[72,67] = -10.2713736253513
        struct[0].Gy[72,68] = -24.9575997948692
        struct[0].Gy[72,69] = -23.2849644809540
        struct[0].Gy[72,70] = -39.6392229058202
        struct[0].Gy[72,71] = -10.2713736253513
        struct[0].Gy[72,72] = -211.143538626360
        struct[0].Gy[72,73] = -108.131545536521
        struct[0].Gy[72,74] = 79.2784458116403
        struct[0].Gy[72,75] = 20.5427472507027
        struct[0].Gy[72,76] = 49.9151995897384
        struct[0].Gy[72,77] = 46.5699289619080
        struct[0].Gy[72,78] = 79.2784458116403
        struct[0].Gy[72,79] = 20.5427472507027
        struct[0].Gy[72,80] = 105.571769313180
        struct[0].Gy[72,81] = 54.0657727682604
        struct[0].Gy[72,82] = -39.6392229058202
        struct[0].Gy[72,83] = -10.2713736253513
        struct[0].Gy[72,84] = -24.9575997948692
        struct[0].Gy[72,85] = -23.2849644809540
        struct[0].Gy[72,86] = -39.6392229058202
        struct[0].Gy[72,87] = -10.2713736253513
        struct[0].Gy[73,64] = -54.0657727682604
        struct[0].Gy[73,65] = 105.571769313180
        struct[0].Gy[73,66] = 10.2713736253513
        struct[0].Gy[73,67] = -39.6392229058202
        struct[0].Gy[73,68] = 23.2849644809540
        struct[0].Gy[73,69] = -24.9575997948692
        struct[0].Gy[73,70] = 10.2713736253513
        struct[0].Gy[73,71] = -39.6392229058202
        struct[0].Gy[73,72] = 108.131545536521
        struct[0].Gy[73,73] = -211.143538626360
        struct[0].Gy[73,74] = -20.5427472507027
        struct[0].Gy[73,75] = 79.2784458116403
        struct[0].Gy[73,76] = -46.5699289619080
        struct[0].Gy[73,77] = 49.9151995897384
        struct[0].Gy[73,78] = -20.5427472507027
        struct[0].Gy[73,79] = 79.2784458116403
        struct[0].Gy[73,80] = -54.0657727682604
        struct[0].Gy[73,81] = 105.571769313180
        struct[0].Gy[73,82] = 10.2713736253513
        struct[0].Gy[73,83] = -39.6392229058202
        struct[0].Gy[73,84] = 23.2849644809540
        struct[0].Gy[73,85] = -24.9575997948692
        struct[0].Gy[73,86] = 10.2713736253513
        struct[0].Gy[73,87] = -39.6392229058202
        struct[0].Gy[74,64] = -39.6392229058202
        struct[0].Gy[74,65] = -10.2713736253513
        struct[0].Gy[74,66] = 105.571769313180
        struct[0].Gy[74,67] = 54.0657727682604
        struct[0].Gy[74,68] = -39.6392229058201
        struct[0].Gy[74,69] = -10.2713736253513
        struct[0].Gy[74,70] = -24.9575997948692
        struct[0].Gy[74,71] = -23.2849644809540
        struct[0].Gy[74,72] = 79.2784458116403
        struct[0].Gy[74,73] = 20.5427472507027
        struct[0].Gy[74,74] = -211.143538626360
        struct[0].Gy[74,75] = -108.131545536521
        struct[0].Gy[74,76] = 79.2784458116403
        struct[0].Gy[74,77] = 20.5427472507027
        struct[0].Gy[74,78] = 49.9151995897384
        struct[0].Gy[74,79] = 46.5699289619080
        struct[0].Gy[74,80] = -39.6392229058202
        struct[0].Gy[74,81] = -10.2713736253513
        struct[0].Gy[74,82] = 105.571769313180
        struct[0].Gy[74,83] = 54.0657727682604
        struct[0].Gy[74,84] = -39.6392229058201
        struct[0].Gy[74,85] = -10.2713736253513
        struct[0].Gy[74,86] = -24.9575997948692
        struct[0].Gy[74,87] = -23.2849644809540
        struct[0].Gy[75,64] = 10.2713736253513
        struct[0].Gy[75,65] = -39.6392229058202
        struct[0].Gy[75,66] = -54.0657727682604
        struct[0].Gy[75,67] = 105.571769313180
        struct[0].Gy[75,68] = 10.2713736253513
        struct[0].Gy[75,69] = -39.6392229058201
        struct[0].Gy[75,70] = 23.2849644809540
        struct[0].Gy[75,71] = -24.9575997948692
        struct[0].Gy[75,72] = -20.5427472507027
        struct[0].Gy[75,73] = 79.2784458116403
        struct[0].Gy[75,74] = 108.131545536521
        struct[0].Gy[75,75] = -211.143538626360
        struct[0].Gy[75,76] = -20.5427472507027
        struct[0].Gy[75,77] = 79.2784458116403
        struct[0].Gy[75,78] = -46.5699289619080
        struct[0].Gy[75,79] = 49.9151995897384
        struct[0].Gy[75,80] = 10.2713736253513
        struct[0].Gy[75,81] = -39.6392229058202
        struct[0].Gy[75,82] = -54.0657727682604
        struct[0].Gy[75,83] = 105.571769313180
        struct[0].Gy[75,84] = 10.2713736253513
        struct[0].Gy[75,85] = -39.6392229058201
        struct[0].Gy[75,86] = 23.2849644809540
        struct[0].Gy[75,87] = -24.9575997948692
        struct[0].Gy[76,64] = -24.9575997948692
        struct[0].Gy[76,65] = -23.2849644809540
        struct[0].Gy[76,66] = -39.6392229058202
        struct[0].Gy[76,67] = -10.2713736253513
        struct[0].Gy[76,68] = 105.571769313180
        struct[0].Gy[76,69] = 54.0657727682604
        struct[0].Gy[76,70] = -39.6392229058201
        struct[0].Gy[76,71] = -10.2713736253514
        struct[0].Gy[76,72] = 49.9151995897384
        struct[0].Gy[76,73] = 46.5699289619080
        struct[0].Gy[76,74] = 79.2784458116404
        struct[0].Gy[76,75] = 20.5427472507027
        struct[0].Gy[76,76] = -211.143538626360
        struct[0].Gy[76,77] = -108.131545536521
        struct[0].Gy[76,78] = 79.2784458116403
        struct[0].Gy[76,79] = 20.5427472507027
        struct[0].Gy[76,80] = -24.9575997948692
        struct[0].Gy[76,81] = -23.2849644809540
        struct[0].Gy[76,82] = -39.6392229058202
        struct[0].Gy[76,83] = -10.2713736253513
        struct[0].Gy[76,84] = 105.571769313180
        struct[0].Gy[76,85] = 54.0657727682604
        struct[0].Gy[76,86] = -39.6392229058201
        struct[0].Gy[76,87] = -10.2713736253514
        struct[0].Gy[77,64] = 23.2849644809540
        struct[0].Gy[77,65] = -24.9575997948692
        struct[0].Gy[77,66] = 10.2713736253513
        struct[0].Gy[77,67] = -39.6392229058202
        struct[0].Gy[77,68] = -54.0657727682604
        struct[0].Gy[77,69] = 105.571769313180
        struct[0].Gy[77,70] = 10.2713736253514
        struct[0].Gy[77,71] = -39.6392229058201
        struct[0].Gy[77,72] = -46.5699289619080
        struct[0].Gy[77,73] = 49.9151995897384
        struct[0].Gy[77,74] = -20.5427472507027
        struct[0].Gy[77,75] = 79.2784458116404
        struct[0].Gy[77,76] = 108.131545536521
        struct[0].Gy[77,77] = -211.143538626360
        struct[0].Gy[77,78] = -20.5427472507027
        struct[0].Gy[77,79] = 79.2784458116403
        struct[0].Gy[77,80] = 23.2849644809540
        struct[0].Gy[77,81] = -24.9575997948692
        struct[0].Gy[77,82] = 10.2713736253513
        struct[0].Gy[77,83] = -39.6392229058202
        struct[0].Gy[77,84] = -54.0657727682604
        struct[0].Gy[77,85] = 105.571769313180
        struct[0].Gy[77,86] = 10.2713736253514
        struct[0].Gy[77,87] = -39.6392229058201
        struct[0].Gy[78,64] = -39.6392229058202
        struct[0].Gy[78,65] = -10.2713736253513
        struct[0].Gy[78,66] = -24.9575997948692
        struct[0].Gy[78,67] = -23.2849644809540
        struct[0].Gy[78,68] = -39.6392229058202
        struct[0].Gy[78,69] = -10.2713736253513
        struct[0].Gy[78,70] = 105.571769313180
        struct[0].Gy[78,71] = 54.0657727682604
        struct[0].Gy[78,72] = 79.2784458116403
        struct[0].Gy[78,73] = 20.5427472507027
        struct[0].Gy[78,74] = 49.9151995897384
        struct[0].Gy[78,75] = 46.5699289619080
        struct[0].Gy[78,76] = 79.2784458116403
        struct[0].Gy[78,77] = 20.5427472507027
        struct[0].Gy[78,78] = -211.143538626360
        struct[0].Gy[78,79] = -108.131545536521
        struct[0].Gy[78,80] = -39.6392229058202
        struct[0].Gy[78,81] = -10.2713736253513
        struct[0].Gy[78,82] = -24.9575997948692
        struct[0].Gy[78,83] = -23.2849644809540
        struct[0].Gy[78,84] = -39.6392229058202
        struct[0].Gy[78,85] = -10.2713736253513
        struct[0].Gy[78,86] = 105.571769313180
        struct[0].Gy[78,87] = 54.0657727682604
        struct[0].Gy[79,64] = 10.2713736253513
        struct[0].Gy[79,65] = -39.6392229058202
        struct[0].Gy[79,66] = 23.2849644809540
        struct[0].Gy[79,67] = -24.9575997948692
        struct[0].Gy[79,68] = 10.2713736253513
        struct[0].Gy[79,69] = -39.6392229058202
        struct[0].Gy[79,70] = -54.0657727682604
        struct[0].Gy[79,71] = 105.571769313180
        struct[0].Gy[79,72] = -20.5427472507027
        struct[0].Gy[79,73] = 79.2784458116403
        struct[0].Gy[79,74] = -46.5699289619080
        struct[0].Gy[79,75] = 49.9151995897384
        struct[0].Gy[79,76] = -20.5427472507027
        struct[0].Gy[79,77] = 79.2784458116403
        struct[0].Gy[79,78] = 108.131545536521
        struct[0].Gy[79,79] = -211.143538626360
        struct[0].Gy[79,80] = 10.2713736253513
        struct[0].Gy[79,81] = -39.6392229058202
        struct[0].Gy[79,82] = 23.2849644809540
        struct[0].Gy[79,83] = -24.9575997948692
        struct[0].Gy[79,84] = 10.2713736253513
        struct[0].Gy[79,85] = -39.6392229058202
        struct[0].Gy[79,86] = -54.0657727682604
        struct[0].Gy[79,87] = 105.571769313180
        struct[0].Gy[80,24] = 30.9517475172273
        struct[0].Gy[80,25] = 5.65456401516768
        struct[0].Gy[80,26] = -9.21038227100566
        struct[0].Gy[80,27] = 1.84896616921897
        struct[0].Gy[80,28] = -9.00835072044485
        struct[0].Gy[80,29] = 0.793238195499529
        struct[0].Gy[80,30] = -9.21038227100566
        struct[0].Gy[80,31] = 1.84896616921897
        struct[0].Gy[80,72] = 105.571769313180
        struct[0].Gy[80,73] = 54.0657727682604
        struct[0].Gy[80,74] = -39.6392229058202
        struct[0].Gy[80,75] = -10.2713736253513
        struct[0].Gy[80,76] = -24.9575997948692
        struct[0].Gy[80,77] = -23.2849644809540
        struct[0].Gy[80,78] = -39.6392229058202
        struct[0].Gy[80,79] = -10.2713736253513
        struct[0].Gy[80,80] = -242.095286143588
        struct[0].Gy[80,81] = -113.786109551688
        struct[0].Gy[80,82] = 88.4888280826460
        struct[0].Gy[80,83] = 18.6937810814837
        struct[0].Gy[80,84] = 58.9235503101833
        struct[0].Gy[80,85] = 45.7766907664085
        struct[0].Gy[80,86] = 88.4888280826460
        struct[0].Gy[80,87] = 18.6937810814837
        struct[0].Gy[80,88] = 105.571769313180
        struct[0].Gy[80,89] = 54.0657727682604
        struct[0].Gy[80,90] = -39.6392229058202
        struct[0].Gy[80,91] = -10.2713736253513
        struct[0].Gy[80,92] = -24.9575997948692
        struct[0].Gy[80,93] = -23.2849644809540
        struct[0].Gy[80,94] = -39.6392229058202
        struct[0].Gy[80,95] = -10.2713736253513
        struct[0].Gy[81,24] = -5.65456401516768
        struct[0].Gy[81,25] = 30.9517475172273
        struct[0].Gy[81,26] = -1.84896616921897
        struct[0].Gy[81,27] = -9.21038227100566
        struct[0].Gy[81,28] = -0.793238195499529
        struct[0].Gy[81,29] = -9.00835072044485
        struct[0].Gy[81,30] = -1.84896616921897
        struct[0].Gy[81,31] = -9.21038227100566
        struct[0].Gy[81,72] = -54.0657727682604
        struct[0].Gy[81,73] = 105.571769313180
        struct[0].Gy[81,74] = 10.2713736253513
        struct[0].Gy[81,75] = -39.6392229058202
        struct[0].Gy[81,76] = 23.2849644809540
        struct[0].Gy[81,77] = -24.9575997948692
        struct[0].Gy[81,78] = 10.2713736253513
        struct[0].Gy[81,79] = -39.6392229058202
        struct[0].Gy[81,80] = 113.786109551688
        struct[0].Gy[81,81] = -242.095286143588
        struct[0].Gy[81,82] = -18.6937810814837
        struct[0].Gy[81,83] = 88.4888280826460
        struct[0].Gy[81,84] = -45.7766907664085
        struct[0].Gy[81,85] = 58.9235503101833
        struct[0].Gy[81,86] = -18.6937810814837
        struct[0].Gy[81,87] = 88.4888280826460
        struct[0].Gy[81,88] = -54.0657727682604
        struct[0].Gy[81,89] = 105.571769313180
        struct[0].Gy[81,90] = 10.2713736253513
        struct[0].Gy[81,91] = -39.6392229058202
        struct[0].Gy[81,92] = 23.2849644809540
        struct[0].Gy[81,93] = -24.9575997948692
        struct[0].Gy[81,94] = 10.2713736253513
        struct[0].Gy[81,95] = -39.6392229058202
        struct[0].Gy[82,24] = -9.21038227100566
        struct[0].Gy[82,25] = 1.84896616921897
        struct[0].Gy[82,26] = 30.9517475172273
        struct[0].Gy[82,27] = 5.65456401516768
        struct[0].Gy[82,28] = -9.21038227100566
        struct[0].Gy[82,29] = 1.84896616921897
        struct[0].Gy[82,30] = -9.00835072044485
        struct[0].Gy[82,31] = 0.793238195499528
        struct[0].Gy[82,72] = -39.6392229058202
        struct[0].Gy[82,73] = -10.2713736253513
        struct[0].Gy[82,74] = 105.571769313180
        struct[0].Gy[82,75] = 54.0657727682604
        struct[0].Gy[82,76] = -39.6392229058201
        struct[0].Gy[82,77] = -10.2713736253513
        struct[0].Gy[82,78] = -24.9575997948692
        struct[0].Gy[82,79] = -23.2849644809540
        struct[0].Gy[82,80] = 88.4888280826460
        struct[0].Gy[82,81] = 18.6937810814837
        struct[0].Gy[82,82] = -242.095286143588
        struct[0].Gy[82,83] = -113.786109551688
        struct[0].Gy[82,84] = 88.4888280826460
        struct[0].Gy[82,85] = 18.6937810814837
        struct[0].Gy[82,86] = 58.9235503101833
        struct[0].Gy[82,87] = 45.7766907664085
        struct[0].Gy[82,88] = -39.6392229058202
        struct[0].Gy[82,89] = -10.2713736253513
        struct[0].Gy[82,90] = 105.571769313180
        struct[0].Gy[82,91] = 54.0657727682604
        struct[0].Gy[82,92] = -39.6392229058201
        struct[0].Gy[82,93] = -10.2713736253513
        struct[0].Gy[82,94] = -24.9575997948692
        struct[0].Gy[82,95] = -23.2849644809540
        struct[0].Gy[83,24] = -1.84896616921897
        struct[0].Gy[83,25] = -9.21038227100566
        struct[0].Gy[83,26] = -5.65456401516768
        struct[0].Gy[83,27] = 30.9517475172273
        struct[0].Gy[83,28] = -1.84896616921897
        struct[0].Gy[83,29] = -9.21038227100566
        struct[0].Gy[83,30] = -0.793238195499528
        struct[0].Gy[83,31] = -9.00835072044485
        struct[0].Gy[83,72] = 10.2713736253513
        struct[0].Gy[83,73] = -39.6392229058202
        struct[0].Gy[83,74] = -54.0657727682604
        struct[0].Gy[83,75] = 105.571769313180
        struct[0].Gy[83,76] = 10.2713736253513
        struct[0].Gy[83,77] = -39.6392229058201
        struct[0].Gy[83,78] = 23.2849644809540
        struct[0].Gy[83,79] = -24.9575997948692
        struct[0].Gy[83,80] = -18.6937810814837
        struct[0].Gy[83,81] = 88.4888280826460
        struct[0].Gy[83,82] = 113.786109551688
        struct[0].Gy[83,83] = -242.095286143588
        struct[0].Gy[83,84] = -18.6937810814837
        struct[0].Gy[83,85] = 88.4888280826460
        struct[0].Gy[83,86] = -45.7766907664085
        struct[0].Gy[83,87] = 58.9235503101833
        struct[0].Gy[83,88] = 10.2713736253513
        struct[0].Gy[83,89] = -39.6392229058202
        struct[0].Gy[83,90] = -54.0657727682604
        struct[0].Gy[83,91] = 105.571769313180
        struct[0].Gy[83,92] = 10.2713736253513
        struct[0].Gy[83,93] = -39.6392229058201
        struct[0].Gy[83,94] = 23.2849644809540
        struct[0].Gy[83,95] = -24.9575997948692
        struct[0].Gy[84,24] = -9.00835072044484
        struct[0].Gy[84,25] = 0.793238195499527
        struct[0].Gy[84,26] = -9.21038227100566
        struct[0].Gy[84,27] = 1.84896616921897
        struct[0].Gy[84,28] = 30.9517475172273
        struct[0].Gy[84,29] = 5.65456401516768
        struct[0].Gy[84,30] = -9.21038227100566
        struct[0].Gy[84,31] = 1.84896616921897
        struct[0].Gy[84,72] = -24.9575997948692
        struct[0].Gy[84,73] = -23.2849644809540
        struct[0].Gy[84,74] = -39.6392229058202
        struct[0].Gy[84,75] = -10.2713736253513
        struct[0].Gy[84,76] = 105.571769313180
        struct[0].Gy[84,77] = 54.0657727682604
        struct[0].Gy[84,78] = -39.6392229058201
        struct[0].Gy[84,79] = -10.2713736253514
        struct[0].Gy[84,80] = 58.9235503101833
        struct[0].Gy[84,81] = 45.7766907664085
        struct[0].Gy[84,82] = 88.4888280826461
        struct[0].Gy[84,83] = 18.6937810814837
        struct[0].Gy[84,84] = -242.095286143588
        struct[0].Gy[84,85] = -113.786109551688
        struct[0].Gy[84,86] = 88.4888280826459
        struct[0].Gy[84,87] = 18.6937810814837
        struct[0].Gy[84,88] = -24.9575997948692
        struct[0].Gy[84,89] = -23.2849644809540
        struct[0].Gy[84,90] = -39.6392229058202
        struct[0].Gy[84,91] = -10.2713736253513
        struct[0].Gy[84,92] = 105.571769313180
        struct[0].Gy[84,93] = 54.0657727682604
        struct[0].Gy[84,94] = -39.6392229058201
        struct[0].Gy[84,95] = -10.2713736253514
        struct[0].Gy[85,24] = -0.793238195499527
        struct[0].Gy[85,25] = -9.00835072044484
        struct[0].Gy[85,26] = -1.84896616921897
        struct[0].Gy[85,27] = -9.21038227100566
        struct[0].Gy[85,28] = -5.65456401516768
        struct[0].Gy[85,29] = 30.9517475172273
        struct[0].Gy[85,30] = -1.84896616921897
        struct[0].Gy[85,31] = -9.21038227100566
        struct[0].Gy[85,72] = 23.2849644809540
        struct[0].Gy[85,73] = -24.9575997948692
        struct[0].Gy[85,74] = 10.2713736253513
        struct[0].Gy[85,75] = -39.6392229058202
        struct[0].Gy[85,76] = -54.0657727682604
        struct[0].Gy[85,77] = 105.571769313180
        struct[0].Gy[85,78] = 10.2713736253514
        struct[0].Gy[85,79] = -39.6392229058201
        struct[0].Gy[85,80] = -45.7766907664085
        struct[0].Gy[85,81] = 58.9235503101833
        struct[0].Gy[85,82] = -18.6937810814837
        struct[0].Gy[85,83] = 88.4888280826461
        struct[0].Gy[85,84] = 113.786109551688
        struct[0].Gy[85,85] = -242.095286143588
        struct[0].Gy[85,86] = -18.6937810814837
        struct[0].Gy[85,87] = 88.4888280826459
        struct[0].Gy[85,88] = 23.2849644809540
        struct[0].Gy[85,89] = -24.9575997948692
        struct[0].Gy[85,90] = 10.2713736253513
        struct[0].Gy[85,91] = -39.6392229058202
        struct[0].Gy[85,92] = -54.0657727682604
        struct[0].Gy[85,93] = 105.571769313180
        struct[0].Gy[85,94] = 10.2713736253514
        struct[0].Gy[85,95] = -39.6392229058201
        struct[0].Gy[86,24] = -9.21038227100566
        struct[0].Gy[86,25] = 1.84896616921897
        struct[0].Gy[86,26] = -9.00835072044485
        struct[0].Gy[86,27] = 0.793238195499527
        struct[0].Gy[86,28] = -9.21038227100566
        struct[0].Gy[86,29] = 1.84896616921897
        struct[0].Gy[86,30] = 30.9517475172273
        struct[0].Gy[86,31] = 5.65456401516768
        struct[0].Gy[86,72] = -39.6392229058202
        struct[0].Gy[86,73] = -10.2713736253513
        struct[0].Gy[86,74] = -24.9575997948692
        struct[0].Gy[86,75] = -23.2849644809540
        struct[0].Gy[86,76] = -39.6392229058202
        struct[0].Gy[86,77] = -10.2713736253513
        struct[0].Gy[86,78] = 105.571769313180
        struct[0].Gy[86,79] = 54.0657727682604
        struct[0].Gy[86,80] = 88.4888280826460
        struct[0].Gy[86,81] = 18.6937810814837
        struct[0].Gy[86,82] = 58.9235503101832
        struct[0].Gy[86,83] = 45.7766907664085
        struct[0].Gy[86,84] = 88.4888280826460
        struct[0].Gy[86,85] = 18.6937810814837
        struct[0].Gy[86,86] = -242.120286143588
        struct[0].Gy[86,87] = -113.786109551688
        struct[0].Gy[86,88] = -39.6392229058202
        struct[0].Gy[86,89] = -10.2713736253513
        struct[0].Gy[86,90] = -24.9575997948692
        struct[0].Gy[86,91] = -23.2849644809540
        struct[0].Gy[86,92] = -39.6392229058202
        struct[0].Gy[86,93] = -10.2713736253513
        struct[0].Gy[86,94] = 105.571769313180
        struct[0].Gy[86,95] = 54.0657727682604
        struct[0].Gy[87,24] = -1.84896616921897
        struct[0].Gy[87,25] = -9.21038227100566
        struct[0].Gy[87,26] = -0.793238195499527
        struct[0].Gy[87,27] = -9.00835072044485
        struct[0].Gy[87,28] = -1.84896616921897
        struct[0].Gy[87,29] = -9.21038227100566
        struct[0].Gy[87,30] = -5.65456401516768
        struct[0].Gy[87,31] = 30.9517475172273
        struct[0].Gy[87,72] = 10.2713736253513
        struct[0].Gy[87,73] = -39.6392229058202
        struct[0].Gy[87,74] = 23.2849644809540
        struct[0].Gy[87,75] = -24.9575997948692
        struct[0].Gy[87,76] = 10.2713736253513
        struct[0].Gy[87,77] = -39.6392229058202
        struct[0].Gy[87,78] = -54.0657727682604
        struct[0].Gy[87,79] = 105.571769313180
        struct[0].Gy[87,80] = -18.6937810814837
        struct[0].Gy[87,81] = 88.4888280826460
        struct[0].Gy[87,82] = -45.7766907664085
        struct[0].Gy[87,83] = 58.9235503101832
        struct[0].Gy[87,84] = -18.6937810814837
        struct[0].Gy[87,85] = 88.4888280826460
        struct[0].Gy[87,86] = 113.786109551688
        struct[0].Gy[87,87] = -242.120286143588
        struct[0].Gy[87,88] = 10.2713736253513
        struct[0].Gy[87,89] = -39.6392229058202
        struct[0].Gy[87,90] = 23.2849644809540
        struct[0].Gy[87,91] = -24.9575997948692
        struct[0].Gy[87,92] = 10.2713736253513
        struct[0].Gy[87,93] = -39.6392229058202
        struct[0].Gy[87,94] = -54.0657727682604
        struct[0].Gy[87,95] = 105.571769313180
        struct[0].Gy[88,80] = 105.571769313180
        struct[0].Gy[88,81] = 54.0657727682604
        struct[0].Gy[88,82] = -39.6392229058202
        struct[0].Gy[88,83] = -10.2713736253513
        struct[0].Gy[88,84] = -24.9575997948692
        struct[0].Gy[88,85] = -23.2849644809540
        struct[0].Gy[88,86] = -39.6392229058202
        struct[0].Gy[88,87] = -10.2713736253513
        struct[0].Gy[88,88] = -211.143538626360
        struct[0].Gy[88,89] = -108.131545536521
        struct[0].Gy[88,90] = 79.2784458116403
        struct[0].Gy[88,91] = 20.5427472507027
        struct[0].Gy[88,92] = 49.9151995897384
        struct[0].Gy[88,93] = 46.5699289619080
        struct[0].Gy[88,94] = 79.2784458116403
        struct[0].Gy[88,95] = 20.5427472507027
        struct[0].Gy[88,96] = 105.571769313180
        struct[0].Gy[88,97] = 54.0657727682604
        struct[0].Gy[88,98] = -39.6392229058202
        struct[0].Gy[88,99] = -10.2713736253513
        struct[0].Gy[88,100] = -24.9575997948692
        struct[0].Gy[88,101] = -23.2849644809540
        struct[0].Gy[88,102] = -39.6392229058202
        struct[0].Gy[88,103] = -10.2713736253513
        struct[0].Gy[89,80] = -54.0657727682604
        struct[0].Gy[89,81] = 105.571769313180
        struct[0].Gy[89,82] = 10.2713736253513
        struct[0].Gy[89,83] = -39.6392229058202
        struct[0].Gy[89,84] = 23.2849644809540
        struct[0].Gy[89,85] = -24.9575997948692
        struct[0].Gy[89,86] = 10.2713736253513
        struct[0].Gy[89,87] = -39.6392229058202
        struct[0].Gy[89,88] = 108.131545536521
        struct[0].Gy[89,89] = -211.143538626360
        struct[0].Gy[89,90] = -20.5427472507027
        struct[0].Gy[89,91] = 79.2784458116403
        struct[0].Gy[89,92] = -46.5699289619080
        struct[0].Gy[89,93] = 49.9151995897384
        struct[0].Gy[89,94] = -20.5427472507027
        struct[0].Gy[89,95] = 79.2784458116403
        struct[0].Gy[89,96] = -54.0657727682604
        struct[0].Gy[89,97] = 105.571769313180
        struct[0].Gy[89,98] = 10.2713736253513
        struct[0].Gy[89,99] = -39.6392229058202
        struct[0].Gy[89,100] = 23.2849644809540
        struct[0].Gy[89,101] = -24.9575997948692
        struct[0].Gy[89,102] = 10.2713736253513
        struct[0].Gy[89,103] = -39.6392229058202
        struct[0].Gy[90,80] = -39.6392229058202
        struct[0].Gy[90,81] = -10.2713736253513
        struct[0].Gy[90,82] = 105.571769313180
        struct[0].Gy[90,83] = 54.0657727682604
        struct[0].Gy[90,84] = -39.6392229058201
        struct[0].Gy[90,85] = -10.2713736253513
        struct[0].Gy[90,86] = -24.9575997948692
        struct[0].Gy[90,87] = -23.2849644809540
        struct[0].Gy[90,88] = 79.2784458116403
        struct[0].Gy[90,89] = 20.5427472507027
        struct[0].Gy[90,90] = -211.143538626360
        struct[0].Gy[90,91] = -108.131545536521
        struct[0].Gy[90,92] = 79.2784458116403
        struct[0].Gy[90,93] = 20.5427472507027
        struct[0].Gy[90,94] = 49.9151995897384
        struct[0].Gy[90,95] = 46.5699289619080
        struct[0].Gy[90,96] = -39.6392229058202
        struct[0].Gy[90,97] = -10.2713736253513
        struct[0].Gy[90,98] = 105.571769313180
        struct[0].Gy[90,99] = 54.0657727682604
        struct[0].Gy[90,100] = -39.6392229058201
        struct[0].Gy[90,101] = -10.2713736253513
        struct[0].Gy[90,102] = -24.9575997948692
        struct[0].Gy[90,103] = -23.2849644809540
        struct[0].Gy[91,80] = 10.2713736253513
        struct[0].Gy[91,81] = -39.6392229058202
        struct[0].Gy[91,82] = -54.0657727682604
        struct[0].Gy[91,83] = 105.571769313180
        struct[0].Gy[91,84] = 10.2713736253513
        struct[0].Gy[91,85] = -39.6392229058201
        struct[0].Gy[91,86] = 23.2849644809540
        struct[0].Gy[91,87] = -24.9575997948692
        struct[0].Gy[91,88] = -20.5427472507027
        struct[0].Gy[91,89] = 79.2784458116403
        struct[0].Gy[91,90] = 108.131545536521
        struct[0].Gy[91,91] = -211.143538626360
        struct[0].Gy[91,92] = -20.5427472507027
        struct[0].Gy[91,93] = 79.2784458116403
        struct[0].Gy[91,94] = -46.5699289619080
        struct[0].Gy[91,95] = 49.9151995897384
        struct[0].Gy[91,96] = 10.2713736253513
        struct[0].Gy[91,97] = -39.6392229058202
        struct[0].Gy[91,98] = -54.0657727682604
        struct[0].Gy[91,99] = 105.571769313180
        struct[0].Gy[91,100] = 10.2713736253513
        struct[0].Gy[91,101] = -39.6392229058201
        struct[0].Gy[91,102] = 23.2849644809540
        struct[0].Gy[91,103] = -24.9575997948692
        struct[0].Gy[92,80] = -24.9575997948692
        struct[0].Gy[92,81] = -23.2849644809540
        struct[0].Gy[92,82] = -39.6392229058202
        struct[0].Gy[92,83] = -10.2713736253513
        struct[0].Gy[92,84] = 105.571769313180
        struct[0].Gy[92,85] = 54.0657727682604
        struct[0].Gy[92,86] = -39.6392229058201
        struct[0].Gy[92,87] = -10.2713736253514
        struct[0].Gy[92,88] = 49.9151995897384
        struct[0].Gy[92,89] = 46.5699289619080
        struct[0].Gy[92,90] = 79.2784458116404
        struct[0].Gy[92,91] = 20.5427472507027
        struct[0].Gy[92,92] = -211.143538626360
        struct[0].Gy[92,93] = -108.131545536521
        struct[0].Gy[92,94] = 79.2784458116403
        struct[0].Gy[92,95] = 20.5427472507027
        struct[0].Gy[92,96] = -24.9575997948692
        struct[0].Gy[92,97] = -23.2849644809540
        struct[0].Gy[92,98] = -39.6392229058202
        struct[0].Gy[92,99] = -10.2713736253513
        struct[0].Gy[92,100] = 105.571769313180
        struct[0].Gy[92,101] = 54.0657727682604
        struct[0].Gy[92,102] = -39.6392229058201
        struct[0].Gy[92,103] = -10.2713736253514
        struct[0].Gy[93,80] = 23.2849644809540
        struct[0].Gy[93,81] = -24.9575997948692
        struct[0].Gy[93,82] = 10.2713736253513
        struct[0].Gy[93,83] = -39.6392229058202
        struct[0].Gy[93,84] = -54.0657727682604
        struct[0].Gy[93,85] = 105.571769313180
        struct[0].Gy[93,86] = 10.2713736253514
        struct[0].Gy[93,87] = -39.6392229058201
        struct[0].Gy[93,88] = -46.5699289619080
        struct[0].Gy[93,89] = 49.9151995897384
        struct[0].Gy[93,90] = -20.5427472507027
        struct[0].Gy[93,91] = 79.2784458116404
        struct[0].Gy[93,92] = 108.131545536521
        struct[0].Gy[93,93] = -211.143538626360
        struct[0].Gy[93,94] = -20.5427472507027
        struct[0].Gy[93,95] = 79.2784458116403
        struct[0].Gy[93,96] = 23.2849644809540
        struct[0].Gy[93,97] = -24.9575997948692
        struct[0].Gy[93,98] = 10.2713736253513
        struct[0].Gy[93,99] = -39.6392229058202
        struct[0].Gy[93,100] = -54.0657727682604
        struct[0].Gy[93,101] = 105.571769313180
        struct[0].Gy[93,102] = 10.2713736253514
        struct[0].Gy[93,103] = -39.6392229058201
        struct[0].Gy[94,80] = -39.6392229058202
        struct[0].Gy[94,81] = -10.2713736253513
        struct[0].Gy[94,82] = -24.9575997948692
        struct[0].Gy[94,83] = -23.2849644809540
        struct[0].Gy[94,84] = -39.6392229058202
        struct[0].Gy[94,85] = -10.2713736253513
        struct[0].Gy[94,86] = 105.571769313180
        struct[0].Gy[94,87] = 54.0657727682604
        struct[0].Gy[94,88] = 79.2784458116403
        struct[0].Gy[94,89] = 20.5427472507027
        struct[0].Gy[94,90] = 49.9151995897384
        struct[0].Gy[94,91] = 46.5699289619080
        struct[0].Gy[94,92] = 79.2784458116403
        struct[0].Gy[94,93] = 20.5427472507027
        struct[0].Gy[94,94] = -211.143538626360
        struct[0].Gy[94,95] = -108.131545536521
        struct[0].Gy[94,96] = -39.6392229058202
        struct[0].Gy[94,97] = -10.2713736253513
        struct[0].Gy[94,98] = -24.9575997948692
        struct[0].Gy[94,99] = -23.2849644809540
        struct[0].Gy[94,100] = -39.6392229058202
        struct[0].Gy[94,101] = -10.2713736253513
        struct[0].Gy[94,102] = 105.571769313180
        struct[0].Gy[94,103] = 54.0657727682604
        struct[0].Gy[95,80] = 10.2713736253513
        struct[0].Gy[95,81] = -39.6392229058202
        struct[0].Gy[95,82] = 23.2849644809540
        struct[0].Gy[95,83] = -24.9575997948692
        struct[0].Gy[95,84] = 10.2713736253513
        struct[0].Gy[95,85] = -39.6392229058202
        struct[0].Gy[95,86] = -54.0657727682604
        struct[0].Gy[95,87] = 105.571769313180
        struct[0].Gy[95,88] = -20.5427472507027
        struct[0].Gy[95,89] = 79.2784458116403
        struct[0].Gy[95,90] = -46.5699289619080
        struct[0].Gy[95,91] = 49.9151995897384
        struct[0].Gy[95,92] = -20.5427472507027
        struct[0].Gy[95,93] = 79.2784458116403
        struct[0].Gy[95,94] = 108.131545536521
        struct[0].Gy[95,95] = -211.143538626360
        struct[0].Gy[95,96] = 10.2713736253513
        struct[0].Gy[95,97] = -39.6392229058202
        struct[0].Gy[95,98] = 23.2849644809540
        struct[0].Gy[95,99] = -24.9575997948692
        struct[0].Gy[95,100] = 10.2713736253513
        struct[0].Gy[95,101] = -39.6392229058202
        struct[0].Gy[95,102] = -54.0657727682604
        struct[0].Gy[95,103] = 105.571769313180
        struct[0].Gy[96,88] = 105.571769313180
        struct[0].Gy[96,89] = 54.0657727682604
        struct[0].Gy[96,90] = -39.6392229058202
        struct[0].Gy[96,91] = -10.2713736253513
        struct[0].Gy[96,92] = -24.9575997948692
        struct[0].Gy[96,93] = -23.2849644809540
        struct[0].Gy[96,94] = -39.6392229058202
        struct[0].Gy[96,95] = -10.2713736253513
        struct[0].Gy[96,96] = -211.143538626360
        struct[0].Gy[96,97] = -108.131545536521
        struct[0].Gy[96,98] = 79.2784458116403
        struct[0].Gy[96,99] = 20.5427472507027
        struct[0].Gy[96,100] = 49.9151995897384
        struct[0].Gy[96,101] = 46.5699289619080
        struct[0].Gy[96,102] = 79.2784458116403
        struct[0].Gy[96,103] = 20.5427472507027
        struct[0].Gy[96,104] = 105.571769313180
        struct[0].Gy[96,105] = 54.0657727682604
        struct[0].Gy[96,106] = -39.6392229058202
        struct[0].Gy[96,107] = -10.2713736253513
        struct[0].Gy[96,108] = -24.9575997948692
        struct[0].Gy[96,109] = -23.2849644809540
        struct[0].Gy[96,110] = -39.6392229058202
        struct[0].Gy[96,111] = -10.2713736253513
        struct[0].Gy[97,88] = -54.0657727682604
        struct[0].Gy[97,89] = 105.571769313180
        struct[0].Gy[97,90] = 10.2713736253513
        struct[0].Gy[97,91] = -39.6392229058202
        struct[0].Gy[97,92] = 23.2849644809540
        struct[0].Gy[97,93] = -24.9575997948692
        struct[0].Gy[97,94] = 10.2713736253513
        struct[0].Gy[97,95] = -39.6392229058202
        struct[0].Gy[97,96] = 108.131545536521
        struct[0].Gy[97,97] = -211.143538626360
        struct[0].Gy[97,98] = -20.5427472507027
        struct[0].Gy[97,99] = 79.2784458116403
        struct[0].Gy[97,100] = -46.5699289619080
        struct[0].Gy[97,101] = 49.9151995897384
        struct[0].Gy[97,102] = -20.5427472507027
        struct[0].Gy[97,103] = 79.2784458116403
        struct[0].Gy[97,104] = -54.0657727682604
        struct[0].Gy[97,105] = 105.571769313180
        struct[0].Gy[97,106] = 10.2713736253513
        struct[0].Gy[97,107] = -39.6392229058202
        struct[0].Gy[97,108] = 23.2849644809540
        struct[0].Gy[97,109] = -24.9575997948692
        struct[0].Gy[97,110] = 10.2713736253513
        struct[0].Gy[97,111] = -39.6392229058202
        struct[0].Gy[98,88] = -39.6392229058202
        struct[0].Gy[98,89] = -10.2713736253513
        struct[0].Gy[98,90] = 105.571769313180
        struct[0].Gy[98,91] = 54.0657727682604
        struct[0].Gy[98,92] = -39.6392229058201
        struct[0].Gy[98,93] = -10.2713736253513
        struct[0].Gy[98,94] = -24.9575997948692
        struct[0].Gy[98,95] = -23.2849644809540
        struct[0].Gy[98,96] = 79.2784458116403
        struct[0].Gy[98,97] = 20.5427472507027
        struct[0].Gy[98,98] = -211.143538626360
        struct[0].Gy[98,99] = -108.131545536521
        struct[0].Gy[98,100] = 79.2784458116403
        struct[0].Gy[98,101] = 20.5427472507027
        struct[0].Gy[98,102] = 49.9151995897384
        struct[0].Gy[98,103] = 46.5699289619080
        struct[0].Gy[98,104] = -39.6392229058202
        struct[0].Gy[98,105] = -10.2713736253513
        struct[0].Gy[98,106] = 105.571769313180
        struct[0].Gy[98,107] = 54.0657727682604
        struct[0].Gy[98,108] = -39.6392229058201
        struct[0].Gy[98,109] = -10.2713736253513
        struct[0].Gy[98,110] = -24.9575997948692
        struct[0].Gy[98,111] = -23.2849644809540
        struct[0].Gy[99,88] = 10.2713736253513
        struct[0].Gy[99,89] = -39.6392229058202
        struct[0].Gy[99,90] = -54.0657727682604
        struct[0].Gy[99,91] = 105.571769313180
        struct[0].Gy[99,92] = 10.2713736253513
        struct[0].Gy[99,93] = -39.6392229058201
        struct[0].Gy[99,94] = 23.2849644809540
        struct[0].Gy[99,95] = -24.9575997948692
        struct[0].Gy[99,96] = -20.5427472507027
        struct[0].Gy[99,97] = 79.2784458116403
        struct[0].Gy[99,98] = 108.131545536521
        struct[0].Gy[99,99] = -211.143538626360
        struct[0].Gy[99,100] = -20.5427472507027
        struct[0].Gy[99,101] = 79.2784458116403
        struct[0].Gy[99,102] = -46.5699289619080
        struct[0].Gy[99,103] = 49.9151995897384
        struct[0].Gy[99,104] = 10.2713736253513
        struct[0].Gy[99,105] = -39.6392229058202
        struct[0].Gy[99,106] = -54.0657727682604
        struct[0].Gy[99,107] = 105.571769313180
        struct[0].Gy[99,108] = 10.2713736253513
        struct[0].Gy[99,109] = -39.6392229058201
        struct[0].Gy[99,110] = 23.2849644809540
        struct[0].Gy[99,111] = -24.9575997948692
        struct[0].Gy[100,88] = -24.9575997948692
        struct[0].Gy[100,89] = -23.2849644809540
        struct[0].Gy[100,90] = -39.6392229058202
        struct[0].Gy[100,91] = -10.2713736253513
        struct[0].Gy[100,92] = 105.571769313180
        struct[0].Gy[100,93] = 54.0657727682604
        struct[0].Gy[100,94] = -39.6392229058201
        struct[0].Gy[100,95] = -10.2713736253514
        struct[0].Gy[100,96] = 49.9151995897384
        struct[0].Gy[100,97] = 46.5699289619080
        struct[0].Gy[100,98] = 79.2784458116404
        struct[0].Gy[100,99] = 20.5427472507027
        struct[0].Gy[100,100] = -211.143538626360
        struct[0].Gy[100,101] = -108.131545536521
        struct[0].Gy[100,102] = 79.2784458116403
        struct[0].Gy[100,103] = 20.5427472507027
        struct[0].Gy[100,104] = -24.9575997948692
        struct[0].Gy[100,105] = -23.2849644809540
        struct[0].Gy[100,106] = -39.6392229058202
        struct[0].Gy[100,107] = -10.2713736253513
        struct[0].Gy[100,108] = 105.571769313180
        struct[0].Gy[100,109] = 54.0657727682604
        struct[0].Gy[100,110] = -39.6392229058201
        struct[0].Gy[100,111] = -10.2713736253514
        struct[0].Gy[101,88] = 23.2849644809540
        struct[0].Gy[101,89] = -24.9575997948692
        struct[0].Gy[101,90] = 10.2713736253513
        struct[0].Gy[101,91] = -39.6392229058202
        struct[0].Gy[101,92] = -54.0657727682604
        struct[0].Gy[101,93] = 105.571769313180
        struct[0].Gy[101,94] = 10.2713736253514
        struct[0].Gy[101,95] = -39.6392229058201
        struct[0].Gy[101,96] = -46.5699289619080
        struct[0].Gy[101,97] = 49.9151995897384
        struct[0].Gy[101,98] = -20.5427472507027
        struct[0].Gy[101,99] = 79.2784458116404
        struct[0].Gy[101,100] = 108.131545536521
        struct[0].Gy[101,101] = -211.143538626360
        struct[0].Gy[101,102] = -20.5427472507027
        struct[0].Gy[101,103] = 79.2784458116403
        struct[0].Gy[101,104] = 23.2849644809540
        struct[0].Gy[101,105] = -24.9575997948692
        struct[0].Gy[101,106] = 10.2713736253513
        struct[0].Gy[101,107] = -39.6392229058202
        struct[0].Gy[101,108] = -54.0657727682604
        struct[0].Gy[101,109] = 105.571769313180
        struct[0].Gy[101,110] = 10.2713736253514
        struct[0].Gy[101,111] = -39.6392229058201
        struct[0].Gy[102,88] = -39.6392229058202
        struct[0].Gy[102,89] = -10.2713736253513
        struct[0].Gy[102,90] = -24.9575997948692
        struct[0].Gy[102,91] = -23.2849644809540
        struct[0].Gy[102,92] = -39.6392229058202
        struct[0].Gy[102,93] = -10.2713736253513
        struct[0].Gy[102,94] = 105.571769313180
        struct[0].Gy[102,95] = 54.0657727682604
        struct[0].Gy[102,96] = 79.2784458116403
        struct[0].Gy[102,97] = 20.5427472507027
        struct[0].Gy[102,98] = 49.9151995897384
        struct[0].Gy[102,99] = 46.5699289619080
        struct[0].Gy[102,100] = 79.2784458116403
        struct[0].Gy[102,101] = 20.5427472507027
        struct[0].Gy[102,102] = -211.168538626360
        struct[0].Gy[102,103] = -108.131545536521
        struct[0].Gy[102,104] = -39.6392229058202
        struct[0].Gy[102,105] = -10.2713736253513
        struct[0].Gy[102,106] = -24.9575997948692
        struct[0].Gy[102,107] = -23.2849644809540
        struct[0].Gy[102,108] = -39.6392229058202
        struct[0].Gy[102,109] = -10.2713736253513
        struct[0].Gy[102,110] = 105.571769313180
        struct[0].Gy[102,111] = 54.0657727682604
        struct[0].Gy[103,88] = 10.2713736253513
        struct[0].Gy[103,89] = -39.6392229058202
        struct[0].Gy[103,90] = 23.2849644809540
        struct[0].Gy[103,91] = -24.9575997948692
        struct[0].Gy[103,92] = 10.2713736253513
        struct[0].Gy[103,93] = -39.6392229058202
        struct[0].Gy[103,94] = -54.0657727682604
        struct[0].Gy[103,95] = 105.571769313180
        struct[0].Gy[103,96] = -20.5427472507027
        struct[0].Gy[103,97] = 79.2784458116403
        struct[0].Gy[103,98] = -46.5699289619080
        struct[0].Gy[103,99] = 49.9151995897384
        struct[0].Gy[103,100] = -20.5427472507027
        struct[0].Gy[103,101] = 79.2784458116403
        struct[0].Gy[103,102] = 108.131545536521
        struct[0].Gy[103,103] = -211.168538626360
        struct[0].Gy[103,104] = 10.2713736253513
        struct[0].Gy[103,105] = -39.6392229058202
        struct[0].Gy[103,106] = 23.2849644809540
        struct[0].Gy[103,107] = -24.9575997948692
        struct[0].Gy[103,108] = 10.2713736253513
        struct[0].Gy[103,109] = -39.6392229058202
        struct[0].Gy[103,110] = -54.0657727682604
        struct[0].Gy[103,111] = 105.571769313180
        struct[0].Gy[104,32] = 30.9517475172273
        struct[0].Gy[104,33] = 5.65456401516768
        struct[0].Gy[104,34] = -9.21038227100566
        struct[0].Gy[104,35] = 1.84896616921897
        struct[0].Gy[104,36] = -9.00835072044485
        struct[0].Gy[104,37] = 0.793238195499529
        struct[0].Gy[104,38] = -9.21038227100566
        struct[0].Gy[104,39] = 1.84896616921897
        struct[0].Gy[104,96] = 105.571769313180
        struct[0].Gy[104,97] = 54.0657727682604
        struct[0].Gy[104,98] = -39.6392229058202
        struct[0].Gy[104,99] = -10.2713736253513
        struct[0].Gy[104,100] = -24.9575997948692
        struct[0].Gy[104,101] = -23.2849644809540
        struct[0].Gy[104,102] = -39.6392229058202
        struct[0].Gy[104,103] = -10.2713736253513
        struct[0].Gy[104,104] = -242.095286143588
        struct[0].Gy[104,105] = -113.786109551688
        struct[0].Gy[104,106] = 88.4888280826460
        struct[0].Gy[104,107] = 18.6937810814837
        struct[0].Gy[104,108] = 58.9235503101833
        struct[0].Gy[104,109] = 45.7766907664085
        struct[0].Gy[104,110] = 88.4888280826460
        struct[0].Gy[104,111] = 18.6937810814837
        struct[0].Gy[104,112] = 105.571769313180
        struct[0].Gy[104,113] = 54.0657727682604
        struct[0].Gy[104,114] = -39.6392229058202
        struct[0].Gy[104,115] = -10.2713736253513
        struct[0].Gy[104,116] = -24.9575997948692
        struct[0].Gy[104,117] = -23.2849644809540
        struct[0].Gy[104,118] = -39.6392229058202
        struct[0].Gy[104,119] = -10.2713736253513
        struct[0].Gy[105,32] = -5.65456401516768
        struct[0].Gy[105,33] = 30.9517475172273
        struct[0].Gy[105,34] = -1.84896616921897
        struct[0].Gy[105,35] = -9.21038227100566
        struct[0].Gy[105,36] = -0.793238195499529
        struct[0].Gy[105,37] = -9.00835072044485
        struct[0].Gy[105,38] = -1.84896616921897
        struct[0].Gy[105,39] = -9.21038227100566
        struct[0].Gy[105,96] = -54.0657727682604
        struct[0].Gy[105,97] = 105.571769313180
        struct[0].Gy[105,98] = 10.2713736253513
        struct[0].Gy[105,99] = -39.6392229058202
        struct[0].Gy[105,100] = 23.2849644809540
        struct[0].Gy[105,101] = -24.9575997948692
        struct[0].Gy[105,102] = 10.2713736253513
        struct[0].Gy[105,103] = -39.6392229058202
        struct[0].Gy[105,104] = 113.786109551688
        struct[0].Gy[105,105] = -242.095286143588
        struct[0].Gy[105,106] = -18.6937810814837
        struct[0].Gy[105,107] = 88.4888280826460
        struct[0].Gy[105,108] = -45.7766907664085
        struct[0].Gy[105,109] = 58.9235503101833
        struct[0].Gy[105,110] = -18.6937810814837
        struct[0].Gy[105,111] = 88.4888280826460
        struct[0].Gy[105,112] = -54.0657727682604
        struct[0].Gy[105,113] = 105.571769313180
        struct[0].Gy[105,114] = 10.2713736253513
        struct[0].Gy[105,115] = -39.6392229058202
        struct[0].Gy[105,116] = 23.2849644809540
        struct[0].Gy[105,117] = -24.9575997948692
        struct[0].Gy[105,118] = 10.2713736253513
        struct[0].Gy[105,119] = -39.6392229058202
        struct[0].Gy[106,32] = -9.21038227100566
        struct[0].Gy[106,33] = 1.84896616921897
        struct[0].Gy[106,34] = 30.9517475172273
        struct[0].Gy[106,35] = 5.65456401516768
        struct[0].Gy[106,36] = -9.21038227100566
        struct[0].Gy[106,37] = 1.84896616921897
        struct[0].Gy[106,38] = -9.00835072044485
        struct[0].Gy[106,39] = 0.793238195499528
        struct[0].Gy[106,96] = -39.6392229058202
        struct[0].Gy[106,97] = -10.2713736253513
        struct[0].Gy[106,98] = 105.571769313180
        struct[0].Gy[106,99] = 54.0657727682604
        struct[0].Gy[106,100] = -39.6392229058201
        struct[0].Gy[106,101] = -10.2713736253513
        struct[0].Gy[106,102] = -24.9575997948692
        struct[0].Gy[106,103] = -23.2849644809540
        struct[0].Gy[106,104] = 88.4888280826460
        struct[0].Gy[106,105] = 18.6937810814837
        struct[0].Gy[106,106] = -242.095286143588
        struct[0].Gy[106,107] = -113.786109551688
        struct[0].Gy[106,108] = 88.4888280826460
        struct[0].Gy[106,109] = 18.6937810814837
        struct[0].Gy[106,110] = 58.9235503101833
        struct[0].Gy[106,111] = 45.7766907664085
        struct[0].Gy[106,112] = -39.6392229058202
        struct[0].Gy[106,113] = -10.2713736253513
        struct[0].Gy[106,114] = 105.571769313180
        struct[0].Gy[106,115] = 54.0657727682604
        struct[0].Gy[106,116] = -39.6392229058201
        struct[0].Gy[106,117] = -10.2713736253513
        struct[0].Gy[106,118] = -24.9575997948692
        struct[0].Gy[106,119] = -23.2849644809540
        struct[0].Gy[107,32] = -1.84896616921897
        struct[0].Gy[107,33] = -9.21038227100566
        struct[0].Gy[107,34] = -5.65456401516768
        struct[0].Gy[107,35] = 30.9517475172273
        struct[0].Gy[107,36] = -1.84896616921897
        struct[0].Gy[107,37] = -9.21038227100566
        struct[0].Gy[107,38] = -0.793238195499528
        struct[0].Gy[107,39] = -9.00835072044485
        struct[0].Gy[107,96] = 10.2713736253513
        struct[0].Gy[107,97] = -39.6392229058202
        struct[0].Gy[107,98] = -54.0657727682604
        struct[0].Gy[107,99] = 105.571769313180
        struct[0].Gy[107,100] = 10.2713736253513
        struct[0].Gy[107,101] = -39.6392229058201
        struct[0].Gy[107,102] = 23.2849644809540
        struct[0].Gy[107,103] = -24.9575997948692
        struct[0].Gy[107,104] = -18.6937810814837
        struct[0].Gy[107,105] = 88.4888280826460
        struct[0].Gy[107,106] = 113.786109551688
        struct[0].Gy[107,107] = -242.095286143588
        struct[0].Gy[107,108] = -18.6937810814837
        struct[0].Gy[107,109] = 88.4888280826460
        struct[0].Gy[107,110] = -45.7766907664085
        struct[0].Gy[107,111] = 58.9235503101833
        struct[0].Gy[107,112] = 10.2713736253513
        struct[0].Gy[107,113] = -39.6392229058202
        struct[0].Gy[107,114] = -54.0657727682604
        struct[0].Gy[107,115] = 105.571769313180
        struct[0].Gy[107,116] = 10.2713736253513
        struct[0].Gy[107,117] = -39.6392229058201
        struct[0].Gy[107,118] = 23.2849644809540
        struct[0].Gy[107,119] = -24.9575997948692
        struct[0].Gy[108,32] = -9.00835072044484
        struct[0].Gy[108,33] = 0.793238195499527
        struct[0].Gy[108,34] = -9.21038227100566
        struct[0].Gy[108,35] = 1.84896616921897
        struct[0].Gy[108,36] = 30.9517475172273
        struct[0].Gy[108,37] = 5.65456401516768
        struct[0].Gy[108,38] = -9.21038227100566
        struct[0].Gy[108,39] = 1.84896616921897
        struct[0].Gy[108,96] = -24.9575997948692
        struct[0].Gy[108,97] = -23.2849644809540
        struct[0].Gy[108,98] = -39.6392229058202
        struct[0].Gy[108,99] = -10.2713736253513
        struct[0].Gy[108,100] = 105.571769313180
        struct[0].Gy[108,101] = 54.0657727682604
        struct[0].Gy[108,102] = -39.6392229058201
        struct[0].Gy[108,103] = -10.2713736253514
        struct[0].Gy[108,104] = 58.9235503101833
        struct[0].Gy[108,105] = 45.7766907664085
        struct[0].Gy[108,106] = 88.4888280826461
        struct[0].Gy[108,107] = 18.6937810814837
        struct[0].Gy[108,108] = -242.095286143588
        struct[0].Gy[108,109] = -113.786109551688
        struct[0].Gy[108,110] = 88.4888280826459
        struct[0].Gy[108,111] = 18.6937810814837
        struct[0].Gy[108,112] = -24.9575997948692
        struct[0].Gy[108,113] = -23.2849644809540
        struct[0].Gy[108,114] = -39.6392229058202
        struct[0].Gy[108,115] = -10.2713736253513
        struct[0].Gy[108,116] = 105.571769313180
        struct[0].Gy[108,117] = 54.0657727682604
        struct[0].Gy[108,118] = -39.6392229058201
        struct[0].Gy[108,119] = -10.2713736253514
        struct[0].Gy[109,32] = -0.793238195499527
        struct[0].Gy[109,33] = -9.00835072044484
        struct[0].Gy[109,34] = -1.84896616921897
        struct[0].Gy[109,35] = -9.21038227100566
        struct[0].Gy[109,36] = -5.65456401516768
        struct[0].Gy[109,37] = 30.9517475172273
        struct[0].Gy[109,38] = -1.84896616921897
        struct[0].Gy[109,39] = -9.21038227100566
        struct[0].Gy[109,96] = 23.2849644809540
        struct[0].Gy[109,97] = -24.9575997948692
        struct[0].Gy[109,98] = 10.2713736253513
        struct[0].Gy[109,99] = -39.6392229058202
        struct[0].Gy[109,100] = -54.0657727682604
        struct[0].Gy[109,101] = 105.571769313180
        struct[0].Gy[109,102] = 10.2713736253514
        struct[0].Gy[109,103] = -39.6392229058201
        struct[0].Gy[109,104] = -45.7766907664085
        struct[0].Gy[109,105] = 58.9235503101833
        struct[0].Gy[109,106] = -18.6937810814837
        struct[0].Gy[109,107] = 88.4888280826461
        struct[0].Gy[109,108] = 113.786109551688
        struct[0].Gy[109,109] = -242.095286143588
        struct[0].Gy[109,110] = -18.6937810814837
        struct[0].Gy[109,111] = 88.4888280826459
        struct[0].Gy[109,112] = 23.2849644809540
        struct[0].Gy[109,113] = -24.9575997948692
        struct[0].Gy[109,114] = 10.2713736253513
        struct[0].Gy[109,115] = -39.6392229058202
        struct[0].Gy[109,116] = -54.0657727682604
        struct[0].Gy[109,117] = 105.571769313180
        struct[0].Gy[109,118] = 10.2713736253514
        struct[0].Gy[109,119] = -39.6392229058201
        struct[0].Gy[110,32] = -9.21038227100566
        struct[0].Gy[110,33] = 1.84896616921897
        struct[0].Gy[110,34] = -9.00835072044485
        struct[0].Gy[110,35] = 0.793238195499527
        struct[0].Gy[110,36] = -9.21038227100566
        struct[0].Gy[110,37] = 1.84896616921897
        struct[0].Gy[110,38] = 30.9517475172273
        struct[0].Gy[110,39] = 5.65456401516768
        struct[0].Gy[110,96] = -39.6392229058202
        struct[0].Gy[110,97] = -10.2713736253513
        struct[0].Gy[110,98] = -24.9575997948692
        struct[0].Gy[110,99] = -23.2849644809540
        struct[0].Gy[110,100] = -39.6392229058202
        struct[0].Gy[110,101] = -10.2713736253513
        struct[0].Gy[110,102] = 105.571769313180
        struct[0].Gy[110,103] = 54.0657727682604
        struct[0].Gy[110,104] = 88.4888280826460
        struct[0].Gy[110,105] = 18.6937810814837
        struct[0].Gy[110,106] = 58.9235503101832
        struct[0].Gy[110,107] = 45.7766907664085
        struct[0].Gy[110,108] = 88.4888280826460
        struct[0].Gy[110,109] = 18.6937810814837
        struct[0].Gy[110,110] = -242.095286143588
        struct[0].Gy[110,111] = -113.786109551688
        struct[0].Gy[110,112] = -39.6392229058202
        struct[0].Gy[110,113] = -10.2713736253513
        struct[0].Gy[110,114] = -24.9575997948692
        struct[0].Gy[110,115] = -23.2849644809540
        struct[0].Gy[110,116] = -39.6392229058202
        struct[0].Gy[110,117] = -10.2713736253513
        struct[0].Gy[110,118] = 105.571769313180
        struct[0].Gy[110,119] = 54.0657727682604
        struct[0].Gy[111,32] = -1.84896616921897
        struct[0].Gy[111,33] = -9.21038227100566
        struct[0].Gy[111,34] = -0.793238195499527
        struct[0].Gy[111,35] = -9.00835072044485
        struct[0].Gy[111,36] = -1.84896616921897
        struct[0].Gy[111,37] = -9.21038227100566
        struct[0].Gy[111,38] = -5.65456401516768
        struct[0].Gy[111,39] = 30.9517475172273
        struct[0].Gy[111,96] = 10.2713736253513
        struct[0].Gy[111,97] = -39.6392229058202
        struct[0].Gy[111,98] = 23.2849644809540
        struct[0].Gy[111,99] = -24.9575997948692
        struct[0].Gy[111,100] = 10.2713736253513
        struct[0].Gy[111,101] = -39.6392229058202
        struct[0].Gy[111,102] = -54.0657727682604
        struct[0].Gy[111,103] = 105.571769313180
        struct[0].Gy[111,104] = -18.6937810814837
        struct[0].Gy[111,105] = 88.4888280826460
        struct[0].Gy[111,106] = -45.7766907664085
        struct[0].Gy[111,107] = 58.9235503101832
        struct[0].Gy[111,108] = -18.6937810814837
        struct[0].Gy[111,109] = 88.4888280826460
        struct[0].Gy[111,110] = 113.786109551688
        struct[0].Gy[111,111] = -242.095286143588
        struct[0].Gy[111,112] = 10.2713736253513
        struct[0].Gy[111,113] = -39.6392229058202
        struct[0].Gy[111,114] = 23.2849644809540
        struct[0].Gy[111,115] = -24.9575997948692
        struct[0].Gy[111,116] = 10.2713736253513
        struct[0].Gy[111,117] = -39.6392229058202
        struct[0].Gy[111,118] = -54.0657727682604
        struct[0].Gy[111,119] = 105.571769313180
        struct[0].Gy[112,40] = 30.9517475172273
        struct[0].Gy[112,41] = 5.65456401516768
        struct[0].Gy[112,42] = -9.21038227100566
        struct[0].Gy[112,43] = 1.84896616921897
        struct[0].Gy[112,44] = -9.00835072044485
        struct[0].Gy[112,45] = 0.793238195499529
        struct[0].Gy[112,46] = -9.21038227100566
        struct[0].Gy[112,47] = 1.84896616921897
        struct[0].Gy[112,104] = 105.571769313180
        struct[0].Gy[112,105] = 54.0657727682604
        struct[0].Gy[112,106] = -39.6392229058202
        struct[0].Gy[112,107] = -10.2713736253513
        struct[0].Gy[112,108] = -24.9575997948692
        struct[0].Gy[112,109] = -23.2849644809540
        struct[0].Gy[112,110] = -39.6392229058202
        struct[0].Gy[112,111] = -10.2713736253513
        struct[0].Gy[112,112] = -136.523516830407
        struct[0].Gy[112,113] = -59.7203367834281
        struct[0].Gy[112,114] = 48.8496051768258
        struct[0].Gy[112,115] = 8.42240745613237
        struct[0].Gy[112,116] = 33.9659505153141
        struct[0].Gy[112,117] = 22.4917262854545
        struct[0].Gy[112,118] = 48.8496051768258
        struct[0].Gy[112,119] = 8.42240745613237
        struct[0].Gy[113,40] = -5.65456401516768
        struct[0].Gy[113,41] = 30.9517475172273
        struct[0].Gy[113,42] = -1.84896616921897
        struct[0].Gy[113,43] = -9.21038227100566
        struct[0].Gy[113,44] = -0.793238195499529
        struct[0].Gy[113,45] = -9.00835072044485
        struct[0].Gy[113,46] = -1.84896616921897
        struct[0].Gy[113,47] = -9.21038227100566
        struct[0].Gy[113,104] = -54.0657727682604
        struct[0].Gy[113,105] = 105.571769313180
        struct[0].Gy[113,106] = 10.2713736253513
        struct[0].Gy[113,107] = -39.6392229058202
        struct[0].Gy[113,108] = 23.2849644809540
        struct[0].Gy[113,109] = -24.9575997948692
        struct[0].Gy[113,110] = 10.2713736253513
        struct[0].Gy[113,111] = -39.6392229058202
        struct[0].Gy[113,112] = 59.7203367834281
        struct[0].Gy[113,113] = -136.523516830407
        struct[0].Gy[113,114] = -8.42240745613237
        struct[0].Gy[113,115] = 48.8496051768258
        struct[0].Gy[113,116] = -22.4917262854545
        struct[0].Gy[113,117] = 33.9659505153141
        struct[0].Gy[113,118] = -8.42240745613237
        struct[0].Gy[113,119] = 48.8496051768258
        struct[0].Gy[114,40] = -9.21038227100566
        struct[0].Gy[114,41] = 1.84896616921897
        struct[0].Gy[114,42] = 30.9517475172273
        struct[0].Gy[114,43] = 5.65456401516768
        struct[0].Gy[114,44] = -9.21038227100566
        struct[0].Gy[114,45] = 1.84896616921897
        struct[0].Gy[114,46] = -9.00835072044485
        struct[0].Gy[114,47] = 0.793238195499528
        struct[0].Gy[114,104] = -39.6392229058202
        struct[0].Gy[114,105] = -10.2713736253513
        struct[0].Gy[114,106] = 105.571769313180
        struct[0].Gy[114,107] = 54.0657727682604
        struct[0].Gy[114,108] = -39.6392229058201
        struct[0].Gy[114,109] = -10.2713736253513
        struct[0].Gy[114,110] = -24.9575997948692
        struct[0].Gy[114,111] = -23.2849644809540
        struct[0].Gy[114,112] = 48.8496051768258
        struct[0].Gy[114,113] = 8.42240745613237
        struct[0].Gy[114,114] = -136.523516830407
        struct[0].Gy[114,115] = -59.7203367834280
        struct[0].Gy[114,116] = 48.8496051768258
        struct[0].Gy[114,117] = 8.42240745613238
        struct[0].Gy[114,118] = 33.9659505153141
        struct[0].Gy[114,119] = 22.4917262854545
        struct[0].Gy[115,40] = -1.84896616921897
        struct[0].Gy[115,41] = -9.21038227100566
        struct[0].Gy[115,42] = -5.65456401516768
        struct[0].Gy[115,43] = 30.9517475172273
        struct[0].Gy[115,44] = -1.84896616921897
        struct[0].Gy[115,45] = -9.21038227100566
        struct[0].Gy[115,46] = -0.793238195499528
        struct[0].Gy[115,47] = -9.00835072044485
        struct[0].Gy[115,104] = 10.2713736253513
        struct[0].Gy[115,105] = -39.6392229058202
        struct[0].Gy[115,106] = -54.0657727682604
        struct[0].Gy[115,107] = 105.571769313180
        struct[0].Gy[115,108] = 10.2713736253513
        struct[0].Gy[115,109] = -39.6392229058201
        struct[0].Gy[115,110] = 23.2849644809540
        struct[0].Gy[115,111] = -24.9575997948692
        struct[0].Gy[115,112] = -8.42240745613237
        struct[0].Gy[115,113] = 48.8496051768258
        struct[0].Gy[115,114] = 59.7203367834280
        struct[0].Gy[115,115] = -136.523516830407
        struct[0].Gy[115,116] = -8.42240745613238
        struct[0].Gy[115,117] = 48.8496051768258
        struct[0].Gy[115,118] = -22.4917262854545
        struct[0].Gy[115,119] = 33.9659505153141
        struct[0].Gy[116,40] = -9.00835072044484
        struct[0].Gy[116,41] = 0.793238195499527
        struct[0].Gy[116,42] = -9.21038227100566
        struct[0].Gy[116,43] = 1.84896616921897
        struct[0].Gy[116,44] = 30.9517475172273
        struct[0].Gy[116,45] = 5.65456401516768
        struct[0].Gy[116,46] = -9.21038227100566
        struct[0].Gy[116,47] = 1.84896616921897
        struct[0].Gy[116,104] = -24.9575997948692
        struct[0].Gy[116,105] = -23.2849644809540
        struct[0].Gy[116,106] = -39.6392229058202
        struct[0].Gy[116,107] = -10.2713736253513
        struct[0].Gy[116,108] = 105.571769313180
        struct[0].Gy[116,109] = 54.0657727682604
        struct[0].Gy[116,110] = -39.6392229058201
        struct[0].Gy[116,111] = -10.2713736253514
        struct[0].Gy[116,112] = 33.9659505153141
        struct[0].Gy[116,113] = 22.4917262854545
        struct[0].Gy[116,114] = 48.8496051768259
        struct[0].Gy[116,115] = 8.42240745613236
        struct[0].Gy[116,116] = -136.523516830407
        struct[0].Gy[116,117] = -59.7203367834281
        struct[0].Gy[116,118] = 48.8496051768258
        struct[0].Gy[116,119] = 8.42240745613239
        struct[0].Gy[117,40] = -0.793238195499527
        struct[0].Gy[117,41] = -9.00835072044484
        struct[0].Gy[117,42] = -1.84896616921897
        struct[0].Gy[117,43] = -9.21038227100566
        struct[0].Gy[117,44] = -5.65456401516768
        struct[0].Gy[117,45] = 30.9517475172273
        struct[0].Gy[117,46] = -1.84896616921897
        struct[0].Gy[117,47] = -9.21038227100566
        struct[0].Gy[117,104] = 23.2849644809540
        struct[0].Gy[117,105] = -24.9575997948692
        struct[0].Gy[117,106] = 10.2713736253513
        struct[0].Gy[117,107] = -39.6392229058202
        struct[0].Gy[117,108] = -54.0657727682604
        struct[0].Gy[117,109] = 105.571769313180
        struct[0].Gy[117,110] = 10.2713736253514
        struct[0].Gy[117,111] = -39.6392229058201
        struct[0].Gy[117,112] = -22.4917262854545
        struct[0].Gy[117,113] = 33.9659505153141
        struct[0].Gy[117,114] = -8.42240745613236
        struct[0].Gy[117,115] = 48.8496051768259
        struct[0].Gy[117,116] = 59.7203367834281
        struct[0].Gy[117,117] = -136.523516830407
        struct[0].Gy[117,118] = -8.42240745613239
        struct[0].Gy[117,119] = 48.8496051768258
        struct[0].Gy[118,40] = -9.21038227100566
        struct[0].Gy[118,41] = 1.84896616921897
        struct[0].Gy[118,42] = -9.00835072044485
        struct[0].Gy[118,43] = 0.793238195499527
        struct[0].Gy[118,44] = -9.21038227100566
        struct[0].Gy[118,45] = 1.84896616921897
        struct[0].Gy[118,46] = 30.9517475172273
        struct[0].Gy[118,47] = 5.65456401516768
        struct[0].Gy[118,104] = -39.6392229058202
        struct[0].Gy[118,105] = -10.2713736253513
        struct[0].Gy[118,106] = -24.9575997948692
        struct[0].Gy[118,107] = -23.2849644809540
        struct[0].Gy[118,108] = -39.6392229058202
        struct[0].Gy[118,109] = -10.2713736253513
        struct[0].Gy[118,110] = 105.571769313180
        struct[0].Gy[118,111] = 54.0657727682604
        struct[0].Gy[118,112] = 48.8496051768258
        struct[0].Gy[118,113] = 8.42240745613237
        struct[0].Gy[118,114] = 33.9659505153140
        struct[0].Gy[118,115] = 22.4917262854545
        struct[0].Gy[118,116] = 48.8496051768258
        struct[0].Gy[118,117] = 8.42240745613237
        struct[0].Gy[118,118] = -136.548516830407
        struct[0].Gy[118,119] = -59.7203367834281
        struct[0].Gy[119,40] = -1.84896616921897
        struct[0].Gy[119,41] = -9.21038227100566
        struct[0].Gy[119,42] = -0.793238195499527
        struct[0].Gy[119,43] = -9.00835072044485
        struct[0].Gy[119,44] = -1.84896616921897
        struct[0].Gy[119,45] = -9.21038227100566
        struct[0].Gy[119,46] = -5.65456401516768
        struct[0].Gy[119,47] = 30.9517475172273
        struct[0].Gy[119,104] = 10.2713736253513
        struct[0].Gy[119,105] = -39.6392229058202
        struct[0].Gy[119,106] = 23.2849644809540
        struct[0].Gy[119,107] = -24.9575997948692
        struct[0].Gy[119,108] = 10.2713736253513
        struct[0].Gy[119,109] = -39.6392229058202
        struct[0].Gy[119,110] = -54.0657727682604
        struct[0].Gy[119,111] = 105.571769313180
        struct[0].Gy[119,112] = -8.42240745613237
        struct[0].Gy[119,113] = 48.8496051768258
        struct[0].Gy[119,114] = -22.4917262854545
        struct[0].Gy[119,115] = 33.9659505153140
        struct[0].Gy[119,116] = -8.42240745613237
        struct[0].Gy[119,117] = 48.8496051768258
        struct[0].Gy[119,118] = 59.7203367834281
        struct[0].Gy[119,119] = -136.548516830407
        struct[0].Gy[120,64] = 26.5300693004806
        struct[0].Gy[120,65] = 4.84676915585801
        struct[0].Gy[120,66] = -7.89461337514770
        struct[0].Gy[120,67] = 1.58482814504483
        struct[0].Gy[120,68] = -7.72144347466701
        struct[0].Gy[120,69] = 0.679918453285310
        struct[0].Gy[120,70] = -7.89461337514771
        struct[0].Gy[120,71] = 1.58482814504483
        struct[0].Gy[120,120] = -53.0601386009611
        struct[0].Gy[120,121] = -9.69353831171601
        struct[0].Gy[120,122] = 15.7892267502954
        struct[0].Gy[120,123] = -3.16965629008966
        struct[0].Gy[120,124] = 15.4428869493340
        struct[0].Gy[120,125] = -1.35983690657062
        struct[0].Gy[120,126] = 15.7892267502954
        struct[0].Gy[120,127] = -3.16965629008966
        struct[0].Gy[120,128] = 26.5300693004806
        struct[0].Gy[120,129] = 4.84676915585801
        struct[0].Gy[120,130] = -7.89461337514770
        struct[0].Gy[120,131] = 1.58482814504483
        struct[0].Gy[120,132] = -7.72144347466701
        struct[0].Gy[120,133] = 0.679918453285310
        struct[0].Gy[120,134] = -7.89461337514771
        struct[0].Gy[120,135] = 1.58482814504483
        struct[0].Gy[121,64] = -4.84676915585801
        struct[0].Gy[121,65] = 26.5300693004806
        struct[0].Gy[121,66] = -1.58482814504483
        struct[0].Gy[121,67] = -7.89461337514770
        struct[0].Gy[121,68] = -0.679918453285310
        struct[0].Gy[121,69] = -7.72144347466701
        struct[0].Gy[121,70] = -1.58482814504483
        struct[0].Gy[121,71] = -7.89461337514771
        struct[0].Gy[121,120] = 9.69353831171601
        struct[0].Gy[121,121] = -53.0601386009611
        struct[0].Gy[121,122] = 3.16965629008966
        struct[0].Gy[121,123] = 15.7892267502954
        struct[0].Gy[121,124] = 1.35983690657062
        struct[0].Gy[121,125] = 15.4428869493340
        struct[0].Gy[121,126] = 3.16965629008966
        struct[0].Gy[121,127] = 15.7892267502954
        struct[0].Gy[121,128] = -4.84676915585801
        struct[0].Gy[121,129] = 26.5300693004806
        struct[0].Gy[121,130] = -1.58482814504483
        struct[0].Gy[121,131] = -7.89461337514770
        struct[0].Gy[121,132] = -0.679918453285310
        struct[0].Gy[121,133] = -7.72144347466701
        struct[0].Gy[121,134] = -1.58482814504483
        struct[0].Gy[121,135] = -7.89461337514771
        struct[0].Gy[122,64] = -7.89461337514771
        struct[0].Gy[122,65] = 1.58482814504483
        struct[0].Gy[122,66] = 26.5300693004806
        struct[0].Gy[122,67] = 4.84676915585801
        struct[0].Gy[122,68] = -7.89461337514771
        struct[0].Gy[122,69] = 1.58482814504483
        struct[0].Gy[122,70] = -7.72144347466701
        struct[0].Gy[122,71] = 0.679918453285308
        struct[0].Gy[122,120] = 15.7892267502954
        struct[0].Gy[122,121] = -3.16965629008966
        struct[0].Gy[122,122] = -53.0601386009611
        struct[0].Gy[122,123] = -9.69353831171602
        struct[0].Gy[122,124] = 15.7892267502954
        struct[0].Gy[122,125] = -3.16965629008967
        struct[0].Gy[122,126] = 15.4428869493340
        struct[0].Gy[122,127] = -1.35983690657062
        struct[0].Gy[122,128] = -7.89461337514771
        struct[0].Gy[122,129] = 1.58482814504483
        struct[0].Gy[122,130] = 26.5300693004806
        struct[0].Gy[122,131] = 4.84676915585801
        struct[0].Gy[122,132] = -7.89461337514771
        struct[0].Gy[122,133] = 1.58482814504483
        struct[0].Gy[122,134] = -7.72144347466701
        struct[0].Gy[122,135] = 0.679918453285308
        struct[0].Gy[123,64] = -1.58482814504483
        struct[0].Gy[123,65] = -7.89461337514771
        struct[0].Gy[123,66] = -4.84676915585801
        struct[0].Gy[123,67] = 26.5300693004806
        struct[0].Gy[123,68] = -1.58482814504483
        struct[0].Gy[123,69] = -7.89461337514771
        struct[0].Gy[123,70] = -0.679918453285308
        struct[0].Gy[123,71] = -7.72144347466701
        struct[0].Gy[123,120] = 3.16965629008966
        struct[0].Gy[123,121] = 15.7892267502954
        struct[0].Gy[123,122] = 9.69353831171602
        struct[0].Gy[123,123] = -53.0601386009611
        struct[0].Gy[123,124] = 3.16965629008967
        struct[0].Gy[123,125] = 15.7892267502954
        struct[0].Gy[123,126] = 1.35983690657062
        struct[0].Gy[123,127] = 15.4428869493340
        struct[0].Gy[123,128] = -1.58482814504483
        struct[0].Gy[123,129] = -7.89461337514771
        struct[0].Gy[123,130] = -4.84676915585801
        struct[0].Gy[123,131] = 26.5300693004806
        struct[0].Gy[123,132] = -1.58482814504483
        struct[0].Gy[123,133] = -7.89461337514771
        struct[0].Gy[123,134] = -0.679918453285308
        struct[0].Gy[123,135] = -7.72144347466701
        struct[0].Gy[124,64] = -7.72144347466701
        struct[0].Gy[124,65] = 0.679918453285310
        struct[0].Gy[124,66] = -7.89461337514770
        struct[0].Gy[124,67] = 1.58482814504483
        struct[0].Gy[124,68] = 26.5300693004806
        struct[0].Gy[124,69] = 4.84676915585801
        struct[0].Gy[124,70] = -7.89461337514771
        struct[0].Gy[124,71] = 1.58482814504483
        struct[0].Gy[124,120] = 15.4428869493340
        struct[0].Gy[124,121] = -1.35983690657062
        struct[0].Gy[124,122] = 15.7892267502954
        struct[0].Gy[124,123] = -3.16965629008966
        struct[0].Gy[124,124] = -53.0601386009611
        struct[0].Gy[124,125] = -9.69353831171602
        struct[0].Gy[124,126] = 15.7892267502954
        struct[0].Gy[124,127] = -3.16965629008967
        struct[0].Gy[124,128] = -7.72144347466701
        struct[0].Gy[124,129] = 0.679918453285310
        struct[0].Gy[124,130] = -7.89461337514770
        struct[0].Gy[124,131] = 1.58482814504483
        struct[0].Gy[124,132] = 26.5300693004806
        struct[0].Gy[124,133] = 4.84676915585801
        struct[0].Gy[124,134] = -7.89461337514771
        struct[0].Gy[124,135] = 1.58482814504483
        struct[0].Gy[125,64] = -0.679918453285310
        struct[0].Gy[125,65] = -7.72144347466701
        struct[0].Gy[125,66] = -1.58482814504483
        struct[0].Gy[125,67] = -7.89461337514770
        struct[0].Gy[125,68] = -4.84676915585801
        struct[0].Gy[125,69] = 26.5300693004806
        struct[0].Gy[125,70] = -1.58482814504483
        struct[0].Gy[125,71] = -7.89461337514771
        struct[0].Gy[125,120] = 1.35983690657062
        struct[0].Gy[125,121] = 15.4428869493340
        struct[0].Gy[125,122] = 3.16965629008966
        struct[0].Gy[125,123] = 15.7892267502954
        struct[0].Gy[125,124] = 9.69353831171602
        struct[0].Gy[125,125] = -53.0601386009611
        struct[0].Gy[125,126] = 3.16965629008967
        struct[0].Gy[125,127] = 15.7892267502954
        struct[0].Gy[125,128] = -0.679918453285310
        struct[0].Gy[125,129] = -7.72144347466701
        struct[0].Gy[125,130] = -1.58482814504483
        struct[0].Gy[125,131] = -7.89461337514770
        struct[0].Gy[125,132] = -4.84676915585801
        struct[0].Gy[125,133] = 26.5300693004806
        struct[0].Gy[125,134] = -1.58482814504483
        struct[0].Gy[125,135] = -7.89461337514771
        struct[0].Gy[126,64] = -7.89461337514771
        struct[0].Gy[126,65] = 1.58482814504483
        struct[0].Gy[126,66] = -7.72144347466701
        struct[0].Gy[126,67] = 0.679918453285308
        struct[0].Gy[126,68] = -7.89461337514771
        struct[0].Gy[126,69] = 1.58482814504483
        struct[0].Gy[126,70] = 26.5300693004806
        struct[0].Gy[126,71] = 4.84676915585801
        struct[0].Gy[126,120] = 15.7892267502954
        struct[0].Gy[126,121] = -3.16965629008966
        struct[0].Gy[126,122] = 15.4428869493340
        struct[0].Gy[126,123] = -1.35983690657062
        struct[0].Gy[126,124] = 15.7892267502954
        struct[0].Gy[126,125] = -3.16965629008966
        struct[0].Gy[126,126] = -53.0601386009611
        struct[0].Gy[126,127] = -9.69353831171602
        struct[0].Gy[126,128] = -7.89461337514771
        struct[0].Gy[126,129] = 1.58482814504483
        struct[0].Gy[126,130] = -7.72144347466701
        struct[0].Gy[126,131] = 0.679918453285308
        struct[0].Gy[126,132] = -7.89461337514771
        struct[0].Gy[126,133] = 1.58482814504483
        struct[0].Gy[126,134] = 26.5300693004806
        struct[0].Gy[126,135] = 4.84676915585801
        struct[0].Gy[127,64] = -1.58482814504483
        struct[0].Gy[127,65] = -7.89461337514771
        struct[0].Gy[127,66] = -0.679918453285308
        struct[0].Gy[127,67] = -7.72144347466701
        struct[0].Gy[127,68] = -1.58482814504483
        struct[0].Gy[127,69] = -7.89461337514771
        struct[0].Gy[127,70] = -4.84676915585801
        struct[0].Gy[127,71] = 26.5300693004806
        struct[0].Gy[127,120] = 3.16965629008966
        struct[0].Gy[127,121] = 15.7892267502954
        struct[0].Gy[127,122] = 1.35983690657062
        struct[0].Gy[127,123] = 15.4428869493340
        struct[0].Gy[127,124] = 3.16965629008966
        struct[0].Gy[127,125] = 15.7892267502954
        struct[0].Gy[127,126] = 9.69353831171602
        struct[0].Gy[127,127] = -53.0601386009611
        struct[0].Gy[127,128] = -1.58482814504483
        struct[0].Gy[127,129] = -7.89461337514771
        struct[0].Gy[127,130] = -0.679918453285308
        struct[0].Gy[127,131] = -7.72144347466701
        struct[0].Gy[127,132] = -1.58482814504483
        struct[0].Gy[127,133] = -7.89461337514771
        struct[0].Gy[127,134] = -4.84676915585801
        struct[0].Gy[127,135] = 26.5300693004806
        struct[0].Gy[128,120] = 26.5300693004806
        struct[0].Gy[128,121] = 4.84676915585801
        struct[0].Gy[128,122] = -7.89461337514770
        struct[0].Gy[128,123] = 1.58482814504483
        struct[0].Gy[128,124] = -7.72144347466701
        struct[0].Gy[128,125] = 0.679918453285310
        struct[0].Gy[128,126] = -7.89461337514771
        struct[0].Gy[128,127] = 1.58482814504483
        struct[0].Gy[128,128] = -53.0601386009611
        struct[0].Gy[128,129] = -9.69353831171601
        struct[0].Gy[128,130] = 15.7892267502954
        struct[0].Gy[128,131] = -3.16965629008966
        struct[0].Gy[128,132] = 15.4428869493340
        struct[0].Gy[128,133] = -1.35983690657062
        struct[0].Gy[128,134] = 15.7892267502954
        struct[0].Gy[128,135] = -3.16965629008966
        struct[0].Gy[128,136] = 26.5300693004806
        struct[0].Gy[128,137] = 4.84676915585801
        struct[0].Gy[128,138] = -7.89461337514770
        struct[0].Gy[128,139] = 1.58482814504483
        struct[0].Gy[128,140] = -7.72144347466701
        struct[0].Gy[128,141] = 0.679918453285310
        struct[0].Gy[128,142] = -7.89461337514771
        struct[0].Gy[128,143] = 1.58482814504483
        struct[0].Gy[129,120] = -4.84676915585801
        struct[0].Gy[129,121] = 26.5300693004806
        struct[0].Gy[129,122] = -1.58482814504483
        struct[0].Gy[129,123] = -7.89461337514770
        struct[0].Gy[129,124] = -0.679918453285310
        struct[0].Gy[129,125] = -7.72144347466701
        struct[0].Gy[129,126] = -1.58482814504483
        struct[0].Gy[129,127] = -7.89461337514771
        struct[0].Gy[129,128] = 9.69353831171601
        struct[0].Gy[129,129] = -53.0601386009611
        struct[0].Gy[129,130] = 3.16965629008966
        struct[0].Gy[129,131] = 15.7892267502954
        struct[0].Gy[129,132] = 1.35983690657062
        struct[0].Gy[129,133] = 15.4428869493340
        struct[0].Gy[129,134] = 3.16965629008966
        struct[0].Gy[129,135] = 15.7892267502954
        struct[0].Gy[129,136] = -4.84676915585801
        struct[0].Gy[129,137] = 26.5300693004806
        struct[0].Gy[129,138] = -1.58482814504483
        struct[0].Gy[129,139] = -7.89461337514770
        struct[0].Gy[129,140] = -0.679918453285310
        struct[0].Gy[129,141] = -7.72144347466701
        struct[0].Gy[129,142] = -1.58482814504483
        struct[0].Gy[129,143] = -7.89461337514771
        struct[0].Gy[130,120] = -7.89461337514771
        struct[0].Gy[130,121] = 1.58482814504483
        struct[0].Gy[130,122] = 26.5300693004806
        struct[0].Gy[130,123] = 4.84676915585801
        struct[0].Gy[130,124] = -7.89461337514771
        struct[0].Gy[130,125] = 1.58482814504483
        struct[0].Gy[130,126] = -7.72144347466701
        struct[0].Gy[130,127] = 0.679918453285308
        struct[0].Gy[130,128] = 15.7892267502954
        struct[0].Gy[130,129] = -3.16965629008966
        struct[0].Gy[130,130] = -53.0601386009611
        struct[0].Gy[130,131] = -9.69353831171602
        struct[0].Gy[130,132] = 15.7892267502954
        struct[0].Gy[130,133] = -3.16965629008967
        struct[0].Gy[130,134] = 15.4428869493340
        struct[0].Gy[130,135] = -1.35983690657062
        struct[0].Gy[130,136] = -7.89461337514771
        struct[0].Gy[130,137] = 1.58482814504483
        struct[0].Gy[130,138] = 26.5300693004806
        struct[0].Gy[130,139] = 4.84676915585801
        struct[0].Gy[130,140] = -7.89461337514771
        struct[0].Gy[130,141] = 1.58482814504483
        struct[0].Gy[130,142] = -7.72144347466701
        struct[0].Gy[130,143] = 0.679918453285308
        struct[0].Gy[131,120] = -1.58482814504483
        struct[0].Gy[131,121] = -7.89461337514771
        struct[0].Gy[131,122] = -4.84676915585801
        struct[0].Gy[131,123] = 26.5300693004806
        struct[0].Gy[131,124] = -1.58482814504483
        struct[0].Gy[131,125] = -7.89461337514771
        struct[0].Gy[131,126] = -0.679918453285308
        struct[0].Gy[131,127] = -7.72144347466701
        struct[0].Gy[131,128] = 3.16965629008966
        struct[0].Gy[131,129] = 15.7892267502954
        struct[0].Gy[131,130] = 9.69353831171602
        struct[0].Gy[131,131] = -53.0601386009611
        struct[0].Gy[131,132] = 3.16965629008967
        struct[0].Gy[131,133] = 15.7892267502954
        struct[0].Gy[131,134] = 1.35983690657062
        struct[0].Gy[131,135] = 15.4428869493340
        struct[0].Gy[131,136] = -1.58482814504483
        struct[0].Gy[131,137] = -7.89461337514771
        struct[0].Gy[131,138] = -4.84676915585801
        struct[0].Gy[131,139] = 26.5300693004806
        struct[0].Gy[131,140] = -1.58482814504483
        struct[0].Gy[131,141] = -7.89461337514771
        struct[0].Gy[131,142] = -0.679918453285308
        struct[0].Gy[131,143] = -7.72144347466701
        struct[0].Gy[132,120] = -7.72144347466701
        struct[0].Gy[132,121] = 0.679918453285310
        struct[0].Gy[132,122] = -7.89461337514770
        struct[0].Gy[132,123] = 1.58482814504483
        struct[0].Gy[132,124] = 26.5300693004806
        struct[0].Gy[132,125] = 4.84676915585801
        struct[0].Gy[132,126] = -7.89461337514771
        struct[0].Gy[132,127] = 1.58482814504483
        struct[0].Gy[132,128] = 15.4428869493340
        struct[0].Gy[132,129] = -1.35983690657062
        struct[0].Gy[132,130] = 15.7892267502954
        struct[0].Gy[132,131] = -3.16965629008966
        struct[0].Gy[132,132] = -53.0601386009611
        struct[0].Gy[132,133] = -9.69353831171602
        struct[0].Gy[132,134] = 15.7892267502954
        struct[0].Gy[132,135] = -3.16965629008967
        struct[0].Gy[132,136] = -7.72144347466701
        struct[0].Gy[132,137] = 0.679918453285310
        struct[0].Gy[132,138] = -7.89461337514770
        struct[0].Gy[132,139] = 1.58482814504483
        struct[0].Gy[132,140] = 26.5300693004806
        struct[0].Gy[132,141] = 4.84676915585801
        struct[0].Gy[132,142] = -7.89461337514771
        struct[0].Gy[132,143] = 1.58482814504483
        struct[0].Gy[133,120] = -0.679918453285310
        struct[0].Gy[133,121] = -7.72144347466701
        struct[0].Gy[133,122] = -1.58482814504483
        struct[0].Gy[133,123] = -7.89461337514770
        struct[0].Gy[133,124] = -4.84676915585801
        struct[0].Gy[133,125] = 26.5300693004806
        struct[0].Gy[133,126] = -1.58482814504483
        struct[0].Gy[133,127] = -7.89461337514771
        struct[0].Gy[133,128] = 1.35983690657062
        struct[0].Gy[133,129] = 15.4428869493340
        struct[0].Gy[133,130] = 3.16965629008966
        struct[0].Gy[133,131] = 15.7892267502954
        struct[0].Gy[133,132] = 9.69353831171602
        struct[0].Gy[133,133] = -53.0601386009611
        struct[0].Gy[133,134] = 3.16965629008967
        struct[0].Gy[133,135] = 15.7892267502954
        struct[0].Gy[133,136] = -0.679918453285310
        struct[0].Gy[133,137] = -7.72144347466701
        struct[0].Gy[133,138] = -1.58482814504483
        struct[0].Gy[133,139] = -7.89461337514770
        struct[0].Gy[133,140] = -4.84676915585801
        struct[0].Gy[133,141] = 26.5300693004806
        struct[0].Gy[133,142] = -1.58482814504483
        struct[0].Gy[133,143] = -7.89461337514771
        struct[0].Gy[134,120] = -7.89461337514771
        struct[0].Gy[134,121] = 1.58482814504483
        struct[0].Gy[134,122] = -7.72144347466701
        struct[0].Gy[134,123] = 0.679918453285308
        struct[0].Gy[134,124] = -7.89461337514771
        struct[0].Gy[134,125] = 1.58482814504483
        struct[0].Gy[134,126] = 26.5300693004806
        struct[0].Gy[134,127] = 4.84676915585801
        struct[0].Gy[134,128] = 15.7892267502954
        struct[0].Gy[134,129] = -3.16965629008966
        struct[0].Gy[134,130] = 15.4428869493340
        struct[0].Gy[134,131] = -1.35983690657062
        struct[0].Gy[134,132] = 15.7892267502954
        struct[0].Gy[134,133] = -3.16965629008966
        struct[0].Gy[134,134] = -53.0851386009611
        struct[0].Gy[134,135] = -9.69353831171602
        struct[0].Gy[134,136] = -7.89461337514771
        struct[0].Gy[134,137] = 1.58482814504483
        struct[0].Gy[134,138] = -7.72144347466701
        struct[0].Gy[134,139] = 0.679918453285308
        struct[0].Gy[134,140] = -7.89461337514771
        struct[0].Gy[134,141] = 1.58482814504483
        struct[0].Gy[134,142] = 26.5300693004806
        struct[0].Gy[134,143] = 4.84676915585801
        struct[0].Gy[135,120] = -1.58482814504483
        struct[0].Gy[135,121] = -7.89461337514771
        struct[0].Gy[135,122] = -0.679918453285308
        struct[0].Gy[135,123] = -7.72144347466701
        struct[0].Gy[135,124] = -1.58482814504483
        struct[0].Gy[135,125] = -7.89461337514771
        struct[0].Gy[135,126] = -4.84676915585801
        struct[0].Gy[135,127] = 26.5300693004806
        struct[0].Gy[135,128] = 3.16965629008966
        struct[0].Gy[135,129] = 15.7892267502954
        struct[0].Gy[135,130] = 1.35983690657062
        struct[0].Gy[135,131] = 15.4428869493340
        struct[0].Gy[135,132] = 3.16965629008966
        struct[0].Gy[135,133] = 15.7892267502954
        struct[0].Gy[135,134] = 9.69353831171602
        struct[0].Gy[135,135] = -53.0851386009611
        struct[0].Gy[135,136] = -1.58482814504483
        struct[0].Gy[135,137] = -7.89461337514771
        struct[0].Gy[135,138] = -0.679918453285308
        struct[0].Gy[135,139] = -7.72144347466701
        struct[0].Gy[135,140] = -1.58482814504483
        struct[0].Gy[135,141] = -7.89461337514771
        struct[0].Gy[135,142] = -4.84676915585801
        struct[0].Gy[135,143] = 26.5300693004806
        struct[0].Gy[136,16] = 26.5300693004806
        struct[0].Gy[136,17] = 4.84676915585801
        struct[0].Gy[136,18] = -7.89461337514770
        struct[0].Gy[136,19] = 1.58482814504483
        struct[0].Gy[136,20] = -7.72144347466701
        struct[0].Gy[136,21] = 0.679918453285310
        struct[0].Gy[136,22] = -7.89461337514771
        struct[0].Gy[136,23] = 1.58482814504483
        struct[0].Gy[136,128] = 26.5300693004806
        struct[0].Gy[136,129] = 4.84676915585801
        struct[0].Gy[136,130] = -7.89461337514770
        struct[0].Gy[136,131] = 1.58482814504483
        struct[0].Gy[136,132] = -7.72144347466701
        struct[0].Gy[136,133] = 0.679918453285310
        struct[0].Gy[136,134] = -7.89461337514771
        struct[0].Gy[136,135] = 1.58482814504483
        struct[0].Gy[136,136] = -53.0601386009611
        struct[0].Gy[136,137] = -9.69353831171601
        struct[0].Gy[136,138] = 15.7892267502954
        struct[0].Gy[136,139] = -3.16965629008966
        struct[0].Gy[136,140] = 15.4428869493340
        struct[0].Gy[136,141] = -1.35983690657062
        struct[0].Gy[136,142] = 15.7892267502954
        struct[0].Gy[136,143] = -3.16965629008966
        struct[0].Gy[137,16] = -4.84676915585801
        struct[0].Gy[137,17] = 26.5300693004806
        struct[0].Gy[137,18] = -1.58482814504483
        struct[0].Gy[137,19] = -7.89461337514770
        struct[0].Gy[137,20] = -0.679918453285310
        struct[0].Gy[137,21] = -7.72144347466701
        struct[0].Gy[137,22] = -1.58482814504483
        struct[0].Gy[137,23] = -7.89461337514771
        struct[0].Gy[137,128] = -4.84676915585801
        struct[0].Gy[137,129] = 26.5300693004806
        struct[0].Gy[137,130] = -1.58482814504483
        struct[0].Gy[137,131] = -7.89461337514770
        struct[0].Gy[137,132] = -0.679918453285310
        struct[0].Gy[137,133] = -7.72144347466701
        struct[0].Gy[137,134] = -1.58482814504483
        struct[0].Gy[137,135] = -7.89461337514771
        struct[0].Gy[137,136] = 9.69353831171601
        struct[0].Gy[137,137] = -53.0601386009611
        struct[0].Gy[137,138] = 3.16965629008966
        struct[0].Gy[137,139] = 15.7892267502954
        struct[0].Gy[137,140] = 1.35983690657062
        struct[0].Gy[137,141] = 15.4428869493340
        struct[0].Gy[137,142] = 3.16965629008966
        struct[0].Gy[137,143] = 15.7892267502954
        struct[0].Gy[138,16] = -7.89461337514771
        struct[0].Gy[138,17] = 1.58482814504483
        struct[0].Gy[138,18] = 26.5300693004806
        struct[0].Gy[138,19] = 4.84676915585801
        struct[0].Gy[138,20] = -7.89461337514771
        struct[0].Gy[138,21] = 1.58482814504483
        struct[0].Gy[138,22] = -7.72144347466701
        struct[0].Gy[138,23] = 0.679918453285308
        struct[0].Gy[138,128] = -7.89461337514771
        struct[0].Gy[138,129] = 1.58482814504483
        struct[0].Gy[138,130] = 26.5300693004806
        struct[0].Gy[138,131] = 4.84676915585801
        struct[0].Gy[138,132] = -7.89461337514771
        struct[0].Gy[138,133] = 1.58482814504483
        struct[0].Gy[138,134] = -7.72144347466701
        struct[0].Gy[138,135] = 0.679918453285308
        struct[0].Gy[138,136] = 15.7892267502954
        struct[0].Gy[138,137] = -3.16965629008966
        struct[0].Gy[138,138] = -53.0601386009611
        struct[0].Gy[138,139] = -9.69353831171602
        struct[0].Gy[138,140] = 15.7892267502954
        struct[0].Gy[138,141] = -3.16965629008967
        struct[0].Gy[138,142] = 15.4428869493340
        struct[0].Gy[138,143] = -1.35983690657062
        struct[0].Gy[139,16] = -1.58482814504483
        struct[0].Gy[139,17] = -7.89461337514771
        struct[0].Gy[139,18] = -4.84676915585801
        struct[0].Gy[139,19] = 26.5300693004806
        struct[0].Gy[139,20] = -1.58482814504483
        struct[0].Gy[139,21] = -7.89461337514771
        struct[0].Gy[139,22] = -0.679918453285308
        struct[0].Gy[139,23] = -7.72144347466701
        struct[0].Gy[139,128] = -1.58482814504483
        struct[0].Gy[139,129] = -7.89461337514771
        struct[0].Gy[139,130] = -4.84676915585801
        struct[0].Gy[139,131] = 26.5300693004806
        struct[0].Gy[139,132] = -1.58482814504483
        struct[0].Gy[139,133] = -7.89461337514771
        struct[0].Gy[139,134] = -0.679918453285308
        struct[0].Gy[139,135] = -7.72144347466701
        struct[0].Gy[139,136] = 3.16965629008966
        struct[0].Gy[139,137] = 15.7892267502954
        struct[0].Gy[139,138] = 9.69353831171602
        struct[0].Gy[139,139] = -53.0601386009611
        struct[0].Gy[139,140] = 3.16965629008967
        struct[0].Gy[139,141] = 15.7892267502954
        struct[0].Gy[139,142] = 1.35983690657062
        struct[0].Gy[139,143] = 15.4428869493340
        struct[0].Gy[140,16] = -7.72144347466701
        struct[0].Gy[140,17] = 0.679918453285310
        struct[0].Gy[140,18] = -7.89461337514770
        struct[0].Gy[140,19] = 1.58482814504483
        struct[0].Gy[140,20] = 26.5300693004806
        struct[0].Gy[140,21] = 4.84676915585801
        struct[0].Gy[140,22] = -7.89461337514771
        struct[0].Gy[140,23] = 1.58482814504483
        struct[0].Gy[140,128] = -7.72144347466701
        struct[0].Gy[140,129] = 0.679918453285310
        struct[0].Gy[140,130] = -7.89461337514770
        struct[0].Gy[140,131] = 1.58482814504483
        struct[0].Gy[140,132] = 26.5300693004806
        struct[0].Gy[140,133] = 4.84676915585801
        struct[0].Gy[140,134] = -7.89461337514771
        struct[0].Gy[140,135] = 1.58482814504483
        struct[0].Gy[140,136] = 15.4428869493340
        struct[0].Gy[140,137] = -1.35983690657062
        struct[0].Gy[140,138] = 15.7892267502954
        struct[0].Gy[140,139] = -3.16965629008966
        struct[0].Gy[140,140] = -53.0601386009611
        struct[0].Gy[140,141] = -9.69353831171602
        struct[0].Gy[140,142] = 15.7892267502954
        struct[0].Gy[140,143] = -3.16965629008967
        struct[0].Gy[141,16] = -0.679918453285310
        struct[0].Gy[141,17] = -7.72144347466701
        struct[0].Gy[141,18] = -1.58482814504483
        struct[0].Gy[141,19] = -7.89461337514770
        struct[0].Gy[141,20] = -4.84676915585801
        struct[0].Gy[141,21] = 26.5300693004806
        struct[0].Gy[141,22] = -1.58482814504483
        struct[0].Gy[141,23] = -7.89461337514771
        struct[0].Gy[141,128] = -0.679918453285310
        struct[0].Gy[141,129] = -7.72144347466701
        struct[0].Gy[141,130] = -1.58482814504483
        struct[0].Gy[141,131] = -7.89461337514770
        struct[0].Gy[141,132] = -4.84676915585801
        struct[0].Gy[141,133] = 26.5300693004806
        struct[0].Gy[141,134] = -1.58482814504483
        struct[0].Gy[141,135] = -7.89461337514771
        struct[0].Gy[141,136] = 1.35983690657062
        struct[0].Gy[141,137] = 15.4428869493340
        struct[0].Gy[141,138] = 3.16965629008966
        struct[0].Gy[141,139] = 15.7892267502954
        struct[0].Gy[141,140] = 9.69353831171602
        struct[0].Gy[141,141] = -53.0601386009611
        struct[0].Gy[141,142] = 3.16965629008967
        struct[0].Gy[141,143] = 15.7892267502954
        struct[0].Gy[142,16] = -7.89461337514771
        struct[0].Gy[142,17] = 1.58482814504483
        struct[0].Gy[142,18] = -7.72144347466701
        struct[0].Gy[142,19] = 0.679918453285308
        struct[0].Gy[142,20] = -7.89461337514771
        struct[0].Gy[142,21] = 1.58482814504483
        struct[0].Gy[142,22] = 26.5300693004806
        struct[0].Gy[142,23] = 4.84676915585801
        struct[0].Gy[142,128] = -7.89461337514771
        struct[0].Gy[142,129] = 1.58482814504483
        struct[0].Gy[142,130] = -7.72144347466701
        struct[0].Gy[142,131] = 0.679918453285308
        struct[0].Gy[142,132] = -7.89461337514771
        struct[0].Gy[142,133] = 1.58482814504483
        struct[0].Gy[142,134] = 26.5300693004806
        struct[0].Gy[142,135] = 4.84676915585801
        struct[0].Gy[142,136] = 15.7892267502954
        struct[0].Gy[142,137] = -3.16965629008966
        struct[0].Gy[142,138] = 15.4428869493340
        struct[0].Gy[142,139] = -1.35983690657062
        struct[0].Gy[142,140] = 15.7892267502954
        struct[0].Gy[142,141] = -3.16965629008966
        struct[0].Gy[142,142] = -53.0601386009611
        struct[0].Gy[142,143] = -9.69353831171602
        struct[0].Gy[143,16] = -1.58482814504483
        struct[0].Gy[143,17] = -7.89461337514771
        struct[0].Gy[143,18] = -0.679918453285308
        struct[0].Gy[143,19] = -7.72144347466701
        struct[0].Gy[143,20] = -1.58482814504483
        struct[0].Gy[143,21] = -7.89461337514771
        struct[0].Gy[143,22] = -4.84676915585801
        struct[0].Gy[143,23] = 26.5300693004806
        struct[0].Gy[143,128] = -1.58482814504483
        struct[0].Gy[143,129] = -7.89461337514771
        struct[0].Gy[143,130] = -0.679918453285308
        struct[0].Gy[143,131] = -7.72144347466701
        struct[0].Gy[143,132] = -1.58482814504483
        struct[0].Gy[143,133] = -7.89461337514771
        struct[0].Gy[143,134] = -4.84676915585801
        struct[0].Gy[143,135] = 26.5300693004806
        struct[0].Gy[143,136] = 3.16965629008966
        struct[0].Gy[143,137] = 15.7892267502954
        struct[0].Gy[143,138] = 1.35983690657062
        struct[0].Gy[143,139] = 15.4428869493340
        struct[0].Gy[143,140] = 3.16965629008966
        struct[0].Gy[143,141] = 15.7892267502954
        struct[0].Gy[143,142] = 9.69353831171602
        struct[0].Gy[143,143] = -53.0601386009611
        struct[0].Gy[144,0] = i_R1_a_r
        struct[0].Gy[144,1] = i_R1_a_i
        struct[0].Gy[144,6] = -i_R1_a_r
        struct[0].Gy[144,7] = -i_R1_a_i
        struct[0].Gy[144,144] = v_R1_a_r - v_R1_n_r
        struct[0].Gy[144,145] = v_R1_a_i - v_R1_n_i
        struct[0].Gy[145,2] = i_R1_b_r
        struct[0].Gy[145,3] = i_R1_b_i
        struct[0].Gy[145,6] = -i_R1_b_r
        struct[0].Gy[145,7] = -i_R1_b_i
        struct[0].Gy[145,146] = v_R1_b_r - v_R1_n_r
        struct[0].Gy[145,147] = v_R1_b_i - v_R1_n_i
        struct[0].Gy[146,4] = i_R1_c_r
        struct[0].Gy[146,5] = i_R1_c_i
        struct[0].Gy[146,6] = -i_R1_c_r
        struct[0].Gy[146,7] = -i_R1_c_i
        struct[0].Gy[146,148] = v_R1_c_r - v_R1_n_r
        struct[0].Gy[146,149] = v_R1_c_i - v_R1_n_i
        struct[0].Gy[147,0] = -i_R1_a_i
        struct[0].Gy[147,1] = i_R1_a_r
        struct[0].Gy[147,6] = i_R1_a_i
        struct[0].Gy[147,7] = -i_R1_a_r
        struct[0].Gy[147,144] = v_R1_a_i - v_R1_n_i
        struct[0].Gy[147,145] = -v_R1_a_r + v_R1_n_r
        struct[0].Gy[148,2] = -i_R1_b_i
        struct[0].Gy[148,3] = i_R1_b_r
        struct[0].Gy[148,6] = i_R1_b_i
        struct[0].Gy[148,7] = -i_R1_b_r
        struct[0].Gy[148,146] = v_R1_b_i - v_R1_n_i
        struct[0].Gy[148,147] = -v_R1_b_r + v_R1_n_r
        struct[0].Gy[149,4] = -i_R1_c_i
        struct[0].Gy[149,5] = i_R1_c_r
        struct[0].Gy[149,6] = i_R1_c_i
        struct[0].Gy[149,7] = -i_R1_c_r
        struct[0].Gy[149,148] = v_R1_c_i - v_R1_n_i
        struct[0].Gy[149,149] = -v_R1_c_r + v_R1_n_r
        struct[0].Gy[150,144] = 1
        struct[0].Gy[150,146] = 1
        struct[0].Gy[150,148] = 1
        struct[0].Gy[150,150] = 1
        struct[0].Gy[151,145] = 1
        struct[0].Gy[151,147] = 1
        struct[0].Gy[151,149] = 1
        struct[0].Gy[151,151] = 1
        struct[0].Gy[152,8] = i_R11_a_r
        struct[0].Gy[152,9] = i_R11_a_i
        struct[0].Gy[152,14] = -i_R11_a_r
        struct[0].Gy[152,15] = -i_R11_a_i
        struct[0].Gy[152,152] = v_R11_a_r - v_R11_n_r
        struct[0].Gy[152,153] = v_R11_a_i - v_R11_n_i
        struct[0].Gy[153,10] = i_R11_b_r
        struct[0].Gy[153,11] = i_R11_b_i
        struct[0].Gy[153,14] = -i_R11_b_r
        struct[0].Gy[153,15] = -i_R11_b_i
        struct[0].Gy[153,154] = v_R11_b_r - v_R11_n_r
        struct[0].Gy[153,155] = v_R11_b_i - v_R11_n_i
        struct[0].Gy[154,12] = i_R11_c_r
        struct[0].Gy[154,13] = i_R11_c_i
        struct[0].Gy[154,14] = -i_R11_c_r
        struct[0].Gy[154,15] = -i_R11_c_i
        struct[0].Gy[154,156] = v_R11_c_r - v_R11_n_r
        struct[0].Gy[154,157] = v_R11_c_i - v_R11_n_i
        struct[0].Gy[155,8] = -i_R11_a_i
        struct[0].Gy[155,9] = i_R11_a_r
        struct[0].Gy[155,14] = i_R11_a_i
        struct[0].Gy[155,15] = -i_R11_a_r
        struct[0].Gy[155,152] = v_R11_a_i - v_R11_n_i
        struct[0].Gy[155,153] = -v_R11_a_r + v_R11_n_r
        struct[0].Gy[156,10] = -i_R11_b_i
        struct[0].Gy[156,11] = i_R11_b_r
        struct[0].Gy[156,14] = i_R11_b_i
        struct[0].Gy[156,15] = -i_R11_b_r
        struct[0].Gy[156,154] = v_R11_b_i - v_R11_n_i
        struct[0].Gy[156,155] = -v_R11_b_r + v_R11_n_r
        struct[0].Gy[157,12] = -i_R11_c_i
        struct[0].Gy[157,13] = i_R11_c_r
        struct[0].Gy[157,14] = i_R11_c_i
        struct[0].Gy[157,15] = -i_R11_c_r
        struct[0].Gy[157,156] = v_R11_c_i - v_R11_n_i
        struct[0].Gy[157,157] = -v_R11_c_r + v_R11_n_r
        struct[0].Gy[158,152] = 1
        struct[0].Gy[158,154] = 1
        struct[0].Gy[158,156] = 1
        struct[0].Gy[158,158] = 1
        struct[0].Gy[159,153] = 1
        struct[0].Gy[159,155] = 1
        struct[0].Gy[159,157] = 1
        struct[0].Gy[159,159] = 1
        struct[0].Gy[160,16] = i_R15_a_r
        struct[0].Gy[160,17] = i_R15_a_i
        struct[0].Gy[160,22] = -i_R15_a_r
        struct[0].Gy[160,23] = -i_R15_a_i
        struct[0].Gy[160,160] = v_R15_a_r - v_R15_n_r
        struct[0].Gy[160,161] = v_R15_a_i - v_R15_n_i
        struct[0].Gy[161,18] = i_R15_b_r
        struct[0].Gy[161,19] = i_R15_b_i
        struct[0].Gy[161,22] = -i_R15_b_r
        struct[0].Gy[161,23] = -i_R15_b_i
        struct[0].Gy[161,162] = v_R15_b_r - v_R15_n_r
        struct[0].Gy[161,163] = v_R15_b_i - v_R15_n_i
        struct[0].Gy[162,20] = i_R15_c_r
        struct[0].Gy[162,21] = i_R15_c_i
        struct[0].Gy[162,22] = -i_R15_c_r
        struct[0].Gy[162,23] = -i_R15_c_i
        struct[0].Gy[162,164] = v_R15_c_r - v_R15_n_r
        struct[0].Gy[162,165] = v_R15_c_i - v_R15_n_i
        struct[0].Gy[163,16] = -i_R15_a_i
        struct[0].Gy[163,17] = i_R15_a_r
        struct[0].Gy[163,22] = i_R15_a_i
        struct[0].Gy[163,23] = -i_R15_a_r
        struct[0].Gy[163,160] = v_R15_a_i - v_R15_n_i
        struct[0].Gy[163,161] = -v_R15_a_r + v_R15_n_r
        struct[0].Gy[164,18] = -i_R15_b_i
        struct[0].Gy[164,19] = i_R15_b_r
        struct[0].Gy[164,22] = i_R15_b_i
        struct[0].Gy[164,23] = -i_R15_b_r
        struct[0].Gy[164,162] = v_R15_b_i - v_R15_n_i
        struct[0].Gy[164,163] = -v_R15_b_r + v_R15_n_r
        struct[0].Gy[165,20] = -i_R15_c_i
        struct[0].Gy[165,21] = i_R15_c_r
        struct[0].Gy[165,22] = i_R15_c_i
        struct[0].Gy[165,23] = -i_R15_c_r
        struct[0].Gy[165,164] = v_R15_c_i - v_R15_n_i
        struct[0].Gy[165,165] = -v_R15_c_r + v_R15_n_r
        struct[0].Gy[166,160] = 1
        struct[0].Gy[166,162] = 1
        struct[0].Gy[166,164] = 1
        struct[0].Gy[166,166] = 1
        struct[0].Gy[167,161] = 1
        struct[0].Gy[167,163] = 1
        struct[0].Gy[167,165] = 1
        struct[0].Gy[167,167] = 1
        struct[0].Gy[168,24] = i_R16_a_r
        struct[0].Gy[168,25] = i_R16_a_i
        struct[0].Gy[168,30] = -i_R16_a_r
        struct[0].Gy[168,31] = -i_R16_a_i
        struct[0].Gy[168,168] = v_R16_a_r - v_R16_n_r
        struct[0].Gy[168,169] = v_R16_a_i - v_R16_n_i
        struct[0].Gy[169,26] = i_R16_b_r
        struct[0].Gy[169,27] = i_R16_b_i
        struct[0].Gy[169,30] = -i_R16_b_r
        struct[0].Gy[169,31] = -i_R16_b_i
        struct[0].Gy[169,170] = v_R16_b_r - v_R16_n_r
        struct[0].Gy[169,171] = v_R16_b_i - v_R16_n_i
        struct[0].Gy[170,28] = i_R16_c_r
        struct[0].Gy[170,29] = i_R16_c_i
        struct[0].Gy[170,30] = -i_R16_c_r
        struct[0].Gy[170,31] = -i_R16_c_i
        struct[0].Gy[170,172] = v_R16_c_r - v_R16_n_r
        struct[0].Gy[170,173] = v_R16_c_i - v_R16_n_i
        struct[0].Gy[171,24] = -i_R16_a_i
        struct[0].Gy[171,25] = i_R16_a_r
        struct[0].Gy[171,30] = i_R16_a_i
        struct[0].Gy[171,31] = -i_R16_a_r
        struct[0].Gy[171,168] = v_R16_a_i - v_R16_n_i
        struct[0].Gy[171,169] = -v_R16_a_r + v_R16_n_r
        struct[0].Gy[172,26] = -i_R16_b_i
        struct[0].Gy[172,27] = i_R16_b_r
        struct[0].Gy[172,30] = i_R16_b_i
        struct[0].Gy[172,31] = -i_R16_b_r
        struct[0].Gy[172,170] = v_R16_b_i - v_R16_n_i
        struct[0].Gy[172,171] = -v_R16_b_r + v_R16_n_r
        struct[0].Gy[173,28] = -i_R16_c_i
        struct[0].Gy[173,29] = i_R16_c_r
        struct[0].Gy[173,30] = i_R16_c_i
        struct[0].Gy[173,31] = -i_R16_c_r
        struct[0].Gy[173,172] = v_R16_c_i - v_R16_n_i
        struct[0].Gy[173,173] = -v_R16_c_r + v_R16_n_r
        struct[0].Gy[174,168] = 1
        struct[0].Gy[174,170] = 1
        struct[0].Gy[174,172] = 1
        struct[0].Gy[174,174] = 1
        struct[0].Gy[175,169] = 1
        struct[0].Gy[175,171] = 1
        struct[0].Gy[175,173] = 1
        struct[0].Gy[175,175] = 1
        struct[0].Gy[176,32] = i_R17_a_r
        struct[0].Gy[176,33] = i_R17_a_i
        struct[0].Gy[176,38] = -i_R17_a_r
        struct[0].Gy[176,39] = -i_R17_a_i
        struct[0].Gy[176,176] = v_R17_a_r - v_R17_n_r
        struct[0].Gy[176,177] = v_R17_a_i - v_R17_n_i
        struct[0].Gy[177,34] = i_R17_b_r
        struct[0].Gy[177,35] = i_R17_b_i
        struct[0].Gy[177,38] = -i_R17_b_r
        struct[0].Gy[177,39] = -i_R17_b_i
        struct[0].Gy[177,178] = v_R17_b_r - v_R17_n_r
        struct[0].Gy[177,179] = v_R17_b_i - v_R17_n_i
        struct[0].Gy[178,36] = i_R17_c_r
        struct[0].Gy[178,37] = i_R17_c_i
        struct[0].Gy[178,38] = -i_R17_c_r
        struct[0].Gy[178,39] = -i_R17_c_i
        struct[0].Gy[178,180] = v_R17_c_r - v_R17_n_r
        struct[0].Gy[178,181] = v_R17_c_i - v_R17_n_i
        struct[0].Gy[179,32] = -i_R17_a_i
        struct[0].Gy[179,33] = i_R17_a_r
        struct[0].Gy[179,38] = i_R17_a_i
        struct[0].Gy[179,39] = -i_R17_a_r
        struct[0].Gy[179,176] = v_R17_a_i - v_R17_n_i
        struct[0].Gy[179,177] = -v_R17_a_r + v_R17_n_r
        struct[0].Gy[180,34] = -i_R17_b_i
        struct[0].Gy[180,35] = i_R17_b_r
        struct[0].Gy[180,38] = i_R17_b_i
        struct[0].Gy[180,39] = -i_R17_b_r
        struct[0].Gy[180,178] = v_R17_b_i - v_R17_n_i
        struct[0].Gy[180,179] = -v_R17_b_r + v_R17_n_r
        struct[0].Gy[181,36] = -i_R17_c_i
        struct[0].Gy[181,37] = i_R17_c_r
        struct[0].Gy[181,38] = i_R17_c_i
        struct[0].Gy[181,39] = -i_R17_c_r
        struct[0].Gy[181,180] = v_R17_c_i - v_R17_n_i
        struct[0].Gy[181,181] = -v_R17_c_r + v_R17_n_r
        struct[0].Gy[182,176] = 1
        struct[0].Gy[182,178] = 1
        struct[0].Gy[182,180] = 1
        struct[0].Gy[182,182] = 1
        struct[0].Gy[183,177] = 1
        struct[0].Gy[183,179] = 1
        struct[0].Gy[183,181] = 1
        struct[0].Gy[183,183] = 1
        struct[0].Gy[184,40] = i_R18_a_r
        struct[0].Gy[184,41] = i_R18_a_i
        struct[0].Gy[184,46] = -i_R18_a_r
        struct[0].Gy[184,47] = -i_R18_a_i
        struct[0].Gy[184,184] = v_R18_a_r - v_R18_n_r
        struct[0].Gy[184,185] = v_R18_a_i - v_R18_n_i
        struct[0].Gy[185,42] = i_R18_b_r
        struct[0].Gy[185,43] = i_R18_b_i
        struct[0].Gy[185,46] = -i_R18_b_r
        struct[0].Gy[185,47] = -i_R18_b_i
        struct[0].Gy[185,186] = v_R18_b_r - v_R18_n_r
        struct[0].Gy[185,187] = v_R18_b_i - v_R18_n_i
        struct[0].Gy[186,44] = i_R18_c_r
        struct[0].Gy[186,45] = i_R18_c_i
        struct[0].Gy[186,46] = -i_R18_c_r
        struct[0].Gy[186,47] = -i_R18_c_i
        struct[0].Gy[186,188] = v_R18_c_r - v_R18_n_r
        struct[0].Gy[186,189] = v_R18_c_i - v_R18_n_i
        struct[0].Gy[187,40] = -i_R18_a_i
        struct[0].Gy[187,41] = i_R18_a_r
        struct[0].Gy[187,46] = i_R18_a_i
        struct[0].Gy[187,47] = -i_R18_a_r
        struct[0].Gy[187,184] = v_R18_a_i - v_R18_n_i
        struct[0].Gy[187,185] = -v_R18_a_r + v_R18_n_r
        struct[0].Gy[188,42] = -i_R18_b_i
        struct[0].Gy[188,43] = i_R18_b_r
        struct[0].Gy[188,46] = i_R18_b_i
        struct[0].Gy[188,47] = -i_R18_b_r
        struct[0].Gy[188,186] = v_R18_b_i - v_R18_n_i
        struct[0].Gy[188,187] = -v_R18_b_r + v_R18_n_r
        struct[0].Gy[189,44] = -i_R18_c_i
        struct[0].Gy[189,45] = i_R18_c_r
        struct[0].Gy[189,46] = i_R18_c_i
        struct[0].Gy[189,47] = -i_R18_c_r
        struct[0].Gy[189,188] = v_R18_c_i - v_R18_n_i
        struct[0].Gy[189,189] = -v_R18_c_r + v_R18_n_r
        struct[0].Gy[190,184] = 1
        struct[0].Gy[190,186] = 1
        struct[0].Gy[190,188] = 1
        struct[0].Gy[190,190] = 1
        struct[0].Gy[191,185] = 1
        struct[0].Gy[191,187] = 1
        struct[0].Gy[191,189] = 1
        struct[0].Gy[191,191] = 1

        struct[0].Gu[0,0] = 0.212261128378539
        struct[0].Gu[0,1] = 0.849044513514155
        struct[0].Gu[0,4] = -0.212261128378539
        struct[0].Gu[0,5] = -0.849044513514155
        struct[0].Gu[1,0] = -0.849044513514155
        struct[0].Gu[1,1] = 0.212261128378539
        struct[0].Gu[1,4] = 0.849044513514155
        struct[0].Gu[1,5] = -0.212261128378539
        struct[0].Gu[2,0] = -0.212261128378539
        struct[0].Gu[2,1] = -0.849044513514155
        struct[0].Gu[2,2] = 0.212261128378539
        struct[0].Gu[2,3] = 0.849044513514155
        struct[0].Gu[3,0] = 0.849044513514155
        struct[0].Gu[3,1] = -0.212261128378539
        struct[0].Gu[3,2] = -0.849044513514155
        struct[0].Gu[3,3] = 0.212261128378539
        struct[0].Gu[4,2] = -0.212261128378539
        struct[0].Gu[4,3] = -0.849044513514155
        struct[0].Gu[4,4] = 0.212261128378539
        struct[0].Gu[4,5] = 0.849044513514155
        struct[0].Gu[5,2] = 0.849044513514155
        struct[0].Gu[5,3] = -0.212261128378539
        struct[0].Gu[5,4] = -0.849044513514155
        struct[0].Gu[5,5] = 0.212261128378539
        struct[0].Gu[6,6] = 1
        struct[0].Gu[7,7] = 1
        struct[0].Gu[14,8] = 1
        struct[0].Gu[15,9] = 1
        struct[0].Gu[22,10] = 1
        struct[0].Gu[23,11] = 1
        struct[0].Gu[30,12] = 1
        struct[0].Gu[31,13] = 1
        struct[0].Gu[38,14] = 1
        struct[0].Gu[39,15] = 1
        struct[0].Gu[46,16] = 1
        struct[0].Gu[47,17] = 1
        struct[0].Gu[48,18] = 1
        struct[0].Gu[49,19] = 1
        struct[0].Gu[50,20] = 1
        struct[0].Gu[51,21] = 1
        struct[0].Gu[52,22] = 1
        struct[0].Gu[53,23] = 1
        struct[0].Gu[54,24] = 1
        struct[0].Gu[55,25] = 1
        struct[0].Gu[56,26] = 1
        struct[0].Gu[57,27] = 1
        struct[0].Gu[58,28] = 1
        struct[0].Gu[59,29] = 1
        struct[0].Gu[60,30] = 1
        struct[0].Gu[61,31] = 1
        struct[0].Gu[62,32] = 1
        struct[0].Gu[63,33] = 1
        struct[0].Gu[64,34] = 1
        struct[0].Gu[65,35] = 1
        struct[0].Gu[66,36] = 1
        struct[0].Gu[67,37] = 1
        struct[0].Gu[68,38] = 1
        struct[0].Gu[69,39] = 1
        struct[0].Gu[70,40] = 1
        struct[0].Gu[71,41] = 1
        struct[0].Gu[72,42] = 1
        struct[0].Gu[73,43] = 1
        struct[0].Gu[74,44] = 1
        struct[0].Gu[75,45] = 1
        struct[0].Gu[76,46] = 1
        struct[0].Gu[77,47] = 1
        struct[0].Gu[78,48] = 1
        struct[0].Gu[79,49] = 1
        struct[0].Gu[80,50] = 1
        struct[0].Gu[81,51] = 1
        struct[0].Gu[82,52] = 1
        struct[0].Gu[83,53] = 1
        struct[0].Gu[84,54] = 1
        struct[0].Gu[85,55] = 1
        struct[0].Gu[86,56] = 1
        struct[0].Gu[87,57] = 1
        struct[0].Gu[88,58] = 1
        struct[0].Gu[89,59] = 1
        struct[0].Gu[90,60] = 1
        struct[0].Gu[91,61] = 1
        struct[0].Gu[92,62] = 1
        struct[0].Gu[93,63] = 1
        struct[0].Gu[94,64] = 1
        struct[0].Gu[95,65] = 1
        struct[0].Gu[96,66] = 1
        struct[0].Gu[97,67] = 1
        struct[0].Gu[98,68] = 1
        struct[0].Gu[99,69] = 1
        struct[0].Gu[100,70] = 1
        struct[0].Gu[101,71] = 1
        struct[0].Gu[102,72] = 1
        struct[0].Gu[103,73] = 1
        struct[0].Gu[104,74] = 1
        struct[0].Gu[105,75] = 1
        struct[0].Gu[106,76] = 1
        struct[0].Gu[107,77] = 1
        struct[0].Gu[108,78] = 1
        struct[0].Gu[109,79] = 1
        struct[0].Gu[110,80] = 1
        struct[0].Gu[111,81] = 1
        struct[0].Gu[112,82] = 1
        struct[0].Gu[113,83] = 1
        struct[0].Gu[114,84] = 1
        struct[0].Gu[115,85] = 1
        struct[0].Gu[116,86] = 1
        struct[0].Gu[117,87] = 1
        struct[0].Gu[118,88] = 1
        struct[0].Gu[119,89] = 1
        struct[0].Gu[120,90] = 1
        struct[0].Gu[121,91] = 1
        struct[0].Gu[122,92] = 1
        struct[0].Gu[123,93] = 1
        struct[0].Gu[124,94] = 1
        struct[0].Gu[125,95] = 1
        struct[0].Gu[126,96] = 1
        struct[0].Gu[127,97] = 1
        struct[0].Gu[128,98] = 1
        struct[0].Gu[129,99] = 1
        struct[0].Gu[130,100] = 1
        struct[0].Gu[131,101] = 1
        struct[0].Gu[132,102] = 1
        struct[0].Gu[133,103] = 1
        struct[0].Gu[134,104] = 1
        struct[0].Gu[135,105] = 1
        struct[0].Gu[136,106] = 1
        struct[0].Gu[137,107] = 1
        struct[0].Gu[138,108] = 1
        struct[0].Gu[139,109] = 1
        struct[0].Gu[140,110] = 1
        struct[0].Gu[141,111] = 1
        struct[0].Gu[142,112] = 1
        struct[0].Gu[143,113] = 1
        struct[0].Gu[144,114] = -1
        struct[0].Gu[145,116] = -1
        struct[0].Gu[146,118] = -1
        struct[0].Gu[147,115] = -1
        struct[0].Gu[148,117] = -1
        struct[0].Gu[149,119] = -1
        struct[0].Gu[150,6] = 1
        struct[0].Gu[151,7] = 1
        struct[0].Gu[152,120] = -1
        struct[0].Gu[153,122] = -1
        struct[0].Gu[154,124] = -1
        struct[0].Gu[155,121] = -1
        struct[0].Gu[156,123] = -1
        struct[0].Gu[157,125] = -1
        struct[0].Gu[158,8] = 1
        struct[0].Gu[159,9] = 1
        struct[0].Gu[160,126] = -1
        struct[0].Gu[161,128] = -1
        struct[0].Gu[162,130] = -1
        struct[0].Gu[163,127] = -1
        struct[0].Gu[164,129] = -1
        struct[0].Gu[165,131] = -1
        struct[0].Gu[166,10] = 1
        struct[0].Gu[167,11] = 1
        struct[0].Gu[168,132] = -1
        struct[0].Gu[169,134] = -1
        struct[0].Gu[170,136] = -1
        struct[0].Gu[171,133] = -1
        struct[0].Gu[172,135] = -1
        struct[0].Gu[173,137] = -1
        struct[0].Gu[174,12] = 1
        struct[0].Gu[175,13] = 1
        struct[0].Gu[176,138] = -1
        struct[0].Gu[177,140] = -1
        struct[0].Gu[178,142] = -1
        struct[0].Gu[179,139] = -1
        struct[0].Gu[180,141] = -1
        struct[0].Gu[181,143] = -1
        struct[0].Gu[182,14] = 1
        struct[0].Gu[183,15] = 1
        struct[0].Gu[184,144] = -1
        struct[0].Gu[185,146] = -1
        struct[0].Gu[186,148] = -1
        struct[0].Gu[187,145] = -1
        struct[0].Gu[188,147] = -1
        struct[0].Gu[189,149] = -1
        struct[0].Gu[190,16] = 1
        struct[0].Gu[191,17] = 1





@numba.njit(cache=True)
def Piecewise(arg):
    out = arg[0][1]
    N = len(arg)
    for it in range(N-1,-1,-1):
        if arg[it][1]: out = arg[it][0]
    return out

@numba.njit(cache=True)
def ITE(arg):
    out = arg[0][1]
    N = len(arg)
    for it in range(N-1,-1,-1):
        if arg[it][1]: out = arg[it][0]
    return out


@numba.njit(cache=True)
def Abs(x):
    return np.abs(x)



@numba.njit(cache=True) 
def daesolver(struct): 
    sin = np.sin
    cos = np.cos
    sqrt = np.sqrt
    i = 0 
    
    Dt = struct[i].Dt 

    N_x = struct[i].N_x
    N_y = struct[i].N_y
    N_z = struct[i].N_z

    decimation = struct[i].decimation 
    eye = np.eye(N_x)
    t = struct[i].t 
    t_end = struct[i].t_end 
    if struct[i].it == 0:
        run(t,struct, 1) 
        struct[i].it_store = 0  
        struct[i]['T'][0] = t 
        struct[i].X[0,:] = struct[i].x[:,0]  
        struct[i].Y[0,:] = struct[i].y_run[:,0]  
        struct[i].Z[0,:] = struct[i].h[:,0]  

    solver = struct[i].solvern 
    while t<t_end: 
        struct[i].it += 1
        struct[i].t += Dt
        
        t = struct[i].t


            
        if solver == 5: # Teapezoidal DAE as in Milano's book

            run(t,struct, 2) 
            run(t,struct, 3) 

            x = np.copy(struct[i].x[:]) 
            y = np.copy(struct[i].y_run[:]) 
            f = np.copy(struct[i].f[:]) 
            g = np.copy(struct[i].g[:]) 
            
            for iter in range(struct[i].imax):
                run(t,struct, 2) 
                run(t,struct, 3) 
                run(t,struct,10) 
                run(t,struct,11) 
                
                x_i = struct[i].x[:] 
                y_i = struct[i].y_run[:]  
                f_i = struct[i].f[:] 
                g_i = struct[i].g[:]                 
                F_x_i = struct[i].Fx[:,:]
                F_y_i = struct[i].Fy[:,:] 
                G_x_i = struct[i].Gx[:,:] 
                G_y_i = struct[i].Gy[:,:]                

                A_c_i = np.vstack((np.hstack((eye-0.5*Dt*F_x_i, -0.5*Dt*F_y_i)),
                                   np.hstack((G_x_i,         G_y_i))))
                     
                f_n_i = x_i - x - 0.5*Dt*(f_i+f) 
                # print(t,iter,g_i)
                Dxy_i = np.linalg.solve(-A_c_i,np.vstack((f_n_i,g_i))) 
                
                x_i = x_i + Dxy_i[0:N_x]
                y_i = y_i + Dxy_i[N_x:(N_x+N_y)]

                struct[i].x[:] = x_i
                struct[i].y_run[:] = y_i

        # [f_i,g_i,F_x_i,F_y_i,G_x_i,G_y_i] =  smib_transient(x_i,y_i,u);
        
        # A_c_i = [[eye(N_x)-0.5*Dt*F_x_i, -0.5*Dt*F_y_i],
        #          [                G_x_i,         G_y_i]];
             
        # f_n_i = x_i - x - 0.5*Dt*(f_i+f);
        
        # Dxy_i = -A_c_i\[f_n_i.',g_i.'].';
        
        # x_i = x_i + Dxy_i(1:N_x);
        # y_i = y_i + Dxy_i(N_x+1:N_x+N_y);
                
                xy = np.vstack((x_i,y_i))
                max_relative = 0.0
                for it_var in range(N_x+N_y):
                    abs_value = np.abs(xy[it_var,0])
                    if abs_value < 0.001:
                        abs_value = 0.001
                                             
                    relative_error = np.abs(Dxy_i[it_var,0])/abs_value
                    
                    if relative_error > max_relative: max_relative = relative_error
                    
                if max_relative<struct[i].itol:
                    
                    break
                
                # if iter>struct[i].imax-2:
                    
                #     print('Convergence problem')

            struct[i].x[:] = x_i
            struct[i].y_run[:] = y_i
                
        # channels 
        if struct[i].store == 1:
            it_store = struct[i].it_store
            if struct[i].it >= it_store*decimation: 
                struct[i]['T'][it_store+1] = t 
                struct[i].X[it_store+1,:] = struct[i].x[:,0] 
                struct[i].Y[it_store+1,:] = struct[i].y_run[:,0]
                struct[i].Z[it_store+1,:] = struct[i].h[:,0]
                struct[i].iters[it_store+1,0] = iter
                struct[i].it_store += 1 
            
    struct[i].t = t

    return t


