import numpy as np
import numba
import scipy.optimize as sopt
import json

sin = np.sin
cos = np.cos
atan2 = np.arctan2
sqrt = np.sqrt 
sign = np.sign 
exp = np.exp


class grid_railway_5vsc_class: 

    def __init__(self): 

        self.t_end = 10.000000 
        self.Dt = 0.0010000 
        self.decimation = 10.000000 
        self.itol = 1e-6 
        self.Dt_max = 0.001000 
        self.Dt_min = 0.001000 
        self.solvern = 5 
        self.imax = 100 
        self.N_x = 5
        self.N_y = 254 
        self.N_z = 10 
        self.N_store = 10000 
        self.params_list = ['R_1112', 'R_1213', 'R_1314', 'R_1415', 'R_1521', 'R_2122', 'R_2223', 'R_2324', 'R_2425', 'R_2531', 'R_3132', 'R_3233', 'R_3334', 'R_3435', 'R_3541', 'R_4142', 'R_4243', 'R_4344', 'R_4445', 'R_4551', 'R_5152', 'R_5253', 'R_5354', 'R_5455', 'p_11', 'p_12', 'p_14', 'p_15', 'p_21', 'p_22', 'p_24', 'p_25', 'p_31', 'p_32', 'p_34', 'p_35', 'p_41', 'p_42', 'p_44', 'p_45', 'p_51', 'p_52', 'p_54', 'p_55'] 
        self.params_values_list  = [0.06306666666666667, 0.06306666666666667, 0.07961686626133334, 0.008762450101333334, 0.008762450101333334, 0.008762450101333334, 0.008762450101333334, 0.018346666666666667, 0.018346666666666667, 0.018346666666666667, 0.018346666666666667, 0.018346666666666667, 0.029813333333333334, 0.029813333333333334, 0.029813333333333334, 0.029813333333333334, 0.029813333333333334, 0.07803063134933337, 0.02922567549599999, 0.02922567549599999, 0.02922567549599999, 0.02922567549599999, 0.0344, 0.0344, 0.0, 0.0, -1932995.075, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -1438308.138, 0.0, 0.0, 0.0, 0.0, 0.0] 
        self.inputs_ini_list = ['v_B1_a_r', 'v_B1_a_i', 'v_B1_b_r', 'v_B1_b_i', 'v_B1_c_r', 'v_B1_c_i', 'v_B7_a_r', 'v_B7_a_i', 'v_B7_b_r', 'v_B7_b_i', 'v_B7_c_r', 'v_B7_c_i', 'i_B2lv_n_r', 'i_B2lv_n_i', 'i_B3lv_n_r', 'i_B3lv_n_i', 'i_B4lv_n_r', 'i_B4lv_n_i', 'i_B5lv_n_r', 'i_B5lv_n_i', 'i_B6lv_n_r', 'i_B6lv_n_i', 'i_B2_a_r', 'i_B2_a_i', 'i_B2_b_r', 'i_B2_b_i', 'i_B2_c_r', 'i_B2_c_i', 'i_B3_a_r', 'i_B3_a_i', 'i_B3_b_r', 'i_B3_b_i', 'i_B3_c_r', 'i_B3_c_i', 'i_B4_a_r', 'i_B4_a_i', 'i_B4_b_r', 'i_B4_b_i', 'i_B4_c_r', 'i_B4_c_i', 'i_B5_a_r', 'i_B5_a_i', 'i_B5_b_r', 'i_B5_b_i', 'i_B5_c_r', 'i_B5_c_i', 'i_B6_a_r', 'i_B6_a_i', 'i_B6_b_r', 'i_B6_b_i', 'i_B6_c_r', 'i_B6_c_i', 'q_B2lv_a', 'q_B2lv_b', 'q_B2lv_c', 'q_B3lv_a', 'q_B3lv_b', 'q_B3lv_c', 'q_B4lv_a', 'q_B4lv_b', 'q_B4lv_c', 'q_B5lv_a', 'q_B5lv_b', 'q_B5lv_c', 'q_B6lv_a', 'q_B6lv_b', 'q_B6lv_c', 'u_dummy', 'Dv_r_13', 'Dv_r_23', 'Dv_r_33', 'Dv_r_43', 'Dv_r_53', 'v_nom', 'T_v', 'K_r', 'a_B2lv', 'b_B2lv', 'c_B2lv', 'a_B3lv', 'b_B3lv', 'c_B3lv', 'a_B4lv', 'b_B4lv', 'c_B4lv', 'a_B5lv', 'b_B5lv', 'c_B5lv', 'a_B6lv', 'b_B6lv', 'c_B6lv'] 
        self.inputs_ini_values_list  = [11547.005383792517, 0.0, -5773.502691896256, -10000.000000000002, -5773.502691896264, 10000.0, 11547.005383792517, 0.0, -5773.502691896256, -10000.000000000002, -5773.502691896264, 10000.0, 5.684341886080802e-14, 1.2505552149377763e-12, -3.979039320256561e-13, 1.2505552149377763e-12, -3.979039320256561e-13, 1.1368683772161603e-12, -8.526512829121202e-13, 3.410605131648481e-13, -2.8421709430404007e-13, 4.547473508864641e-13, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -322558.5103381794, -322558.5103381794, -322558.5103381795, -322558.50336306763, -322558.5033630677, -322558.5033630677, -322558.4994660576, -322558.49946605746, -322558.4994660576, -322558.51338830066, -322558.51338830066, -322558.51338830055, -322558.51542340475, -322558.51542340487, -322558.5154234048, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3000.0, 0.02, 0.0003, 2.92075867, 0.44961794, 0.02717091, 2.92075867, 0.44961794, 0.02717091, 2.92075867, 0.44961794, 0.02717091, 2.92075867, 0.44961794, 0.02717091, 2.92075867, 0.44961794, 0.02717091] 
        self.inputs_run_list = ['v_B1_a_r', 'v_B1_a_i', 'v_B1_b_r', 'v_B1_b_i', 'v_B1_c_r', 'v_B1_c_i', 'v_B7_a_r', 'v_B7_a_i', 'v_B7_b_r', 'v_B7_b_i', 'v_B7_c_r', 'v_B7_c_i', 'i_B2lv_n_r', 'i_B2lv_n_i', 'i_B3lv_n_r', 'i_B3lv_n_i', 'i_B4lv_n_r', 'i_B4lv_n_i', 'i_B5lv_n_r', 'i_B5lv_n_i', 'i_B6lv_n_r', 'i_B6lv_n_i', 'i_B2_a_r', 'i_B2_a_i', 'i_B2_b_r', 'i_B2_b_i', 'i_B2_c_r', 'i_B2_c_i', 'i_B3_a_r', 'i_B3_a_i', 'i_B3_b_r', 'i_B3_b_i', 'i_B3_c_r', 'i_B3_c_i', 'i_B4_a_r', 'i_B4_a_i', 'i_B4_b_r', 'i_B4_b_i', 'i_B4_c_r', 'i_B4_c_i', 'i_B5_a_r', 'i_B5_a_i', 'i_B5_b_r', 'i_B5_b_i', 'i_B5_c_r', 'i_B5_c_i', 'i_B6_a_r', 'i_B6_a_i', 'i_B6_b_r', 'i_B6_b_i', 'i_B6_c_r', 'i_B6_c_i', 'q_B2lv_a', 'q_B2lv_b', 'q_B2lv_c', 'q_B3lv_a', 'q_B3lv_b', 'q_B3lv_c', 'q_B4lv_a', 'q_B4lv_b', 'q_B4lv_c', 'q_B5lv_a', 'q_B5lv_b', 'q_B5lv_c', 'q_B6lv_a', 'q_B6lv_b', 'q_B6lv_c', 'u_dummy', 'Dv_r_13', 'Dv_r_23', 'Dv_r_33', 'Dv_r_43', 'Dv_r_53', 'v_nom', 'T_v', 'K_r', 'a_B2lv', 'b_B2lv', 'c_B2lv', 'a_B3lv', 'b_B3lv', 'c_B3lv', 'a_B4lv', 'b_B4lv', 'c_B4lv', 'a_B5lv', 'b_B5lv', 'c_B5lv', 'a_B6lv', 'b_B6lv', 'c_B6lv'] 
        self.inputs_run_values_list = [11547.005383792517, 0.0, -5773.502691896256, -10000.000000000002, -5773.502691896264, 10000.0, 11547.005383792517, 0.0, -5773.502691896256, -10000.000000000002, -5773.502691896264, 10000.0, 5.684341886080802e-14, 1.2505552149377763e-12, -3.979039320256561e-13, 1.2505552149377763e-12, -3.979039320256561e-13, 1.1368683772161603e-12, -8.526512829121202e-13, 3.410605131648481e-13, -2.8421709430404007e-13, 4.547473508864641e-13, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, -322558.5103381794, -322558.5103381794, -322558.5103381795, -322558.50336306763, -322558.5033630677, -322558.5033630677, -322558.4994660576, -322558.49946605746, -322558.4994660576, -322558.51338830066, -322558.51338830066, -322558.51338830055, -322558.51542340475, -322558.51542340487, -322558.5154234048, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 3000.0, 0.02, 0.0003, 2.92075867, 0.44961794, 0.02717091, 2.92075867, 0.44961794, 0.02717091, 2.92075867, 0.44961794, 0.02717091, 2.92075867, 0.44961794, 0.02717091, 2.92075867, 0.44961794, 0.02717091] 
        self.outputs_list = ['p_13', 'v_13', 'p_23', 'v_23', 'p_33', 'v_33', 'p_43', 'v_43', 'p_53', 'v_53'] 
        self.x_list = ['v_13', 'v_23', 'v_33', 'v_43', 'v_53'] 
        self.y_run_list = ['v_B2lv_a_r', 'v_B2lv_a_i', 'v_B2lv_b_r', 'v_B2lv_b_i', 'v_B2lv_c_r', 'v_B2lv_c_i', 'v_B2lv_n_r', 'v_B2lv_n_i', 'v_B3lv_a_r', 'v_B3lv_a_i', 'v_B3lv_b_r', 'v_B3lv_b_i', 'v_B3lv_c_r', 'v_B3lv_c_i', 'v_B3lv_n_r', 'v_B3lv_n_i', 'v_B4lv_a_r', 'v_B4lv_a_i', 'v_B4lv_b_r', 'v_B4lv_b_i', 'v_B4lv_c_r', 'v_B4lv_c_i', 'v_B4lv_n_r', 'v_B4lv_n_i', 'v_B5lv_a_r', 'v_B5lv_a_i', 'v_B5lv_b_r', 'v_B5lv_b_i', 'v_B5lv_c_r', 'v_B5lv_c_i', 'v_B5lv_n_r', 'v_B5lv_n_i', 'v_B6lv_a_r', 'v_B6lv_a_i', 'v_B6lv_b_r', 'v_B6lv_b_i', 'v_B6lv_c_r', 'v_B6lv_c_i', 'v_B6lv_n_r', 'v_B6lv_n_i', 'v_B2_a_r', 'v_B2_a_i', 'v_B2_b_r', 'v_B2_b_i', 'v_B2_c_r', 'v_B2_c_i', 'v_B3_a_r', 'v_B3_a_i', 'v_B3_b_r', 'v_B3_b_i', 'v_B3_c_r', 'v_B3_c_i', 'v_B4_a_r', 'v_B4_a_i', 'v_B4_b_r', 'v_B4_b_i', 'v_B4_c_r', 'v_B4_c_i', 'v_B5_a_r', 'v_B5_a_i', 'v_B5_b_r', 'v_B5_b_i', 'v_B5_c_r', 'v_B5_c_i', 'v_B6_a_r', 'v_B6_a_i', 'v_B6_b_r', 'v_B6_b_i', 'v_B6_c_r', 'v_B6_c_i', 'i_t_B2_B2lv_a_r', 'i_t_B2_B2lv_a_i', 'i_t_B2_B2lv_b_r', 'i_t_B2_B2lv_b_i', 'i_t_B2_B2lv_c_r', 'i_t_B2_B2lv_c_i', 'i_t_B3_B3lv_a_r', 'i_t_B3_B3lv_a_i', 'i_t_B3_B3lv_b_r', 'i_t_B3_B3lv_b_i', 'i_t_B3_B3lv_c_r', 'i_t_B3_B3lv_c_i', 'i_t_B4_B4lv_a_r', 'i_t_B4_B4lv_a_i', 'i_t_B4_B4lv_b_r', 'i_t_B4_B4lv_b_i', 'i_t_B4_B4lv_c_r', 'i_t_B4_B4lv_c_i', 'i_t_B5_B5lv_a_r', 'i_t_B5_B5lv_a_i', 'i_t_B5_B5lv_b_r', 'i_t_B5_B5lv_b_i', 'i_t_B5_B5lv_c_r', 'i_t_B5_B5lv_c_i', 'i_t_B6_B6lv_a_r', 'i_t_B6_B6lv_a_i', 'i_t_B6_B6lv_b_r', 'i_t_B6_B6lv_b_i', 'i_t_B6_B6lv_c_r', 'i_t_B6_B6lv_c_i', 'i_l_B1_B2_a_r', 'i_l_B1_B2_a_i', 'i_l_B1_B2_b_r', 'i_l_B1_B2_b_i', 'i_l_B1_B2_c_r', 'i_l_B1_B2_c_i', 'i_l_B2_B3_a_r', 'i_l_B2_B3_a_i', 'i_l_B2_B3_b_r', 'i_l_B2_B3_b_i', 'i_l_B2_B3_c_r', 'i_l_B2_B3_c_i', 'i_l_B3_B4_a_r', 'i_l_B3_B4_a_i', 'i_l_B3_B4_b_r', 'i_l_B3_B4_b_i', 'i_l_B3_B4_c_r', 'i_l_B3_B4_c_i', 'i_l_B5_B6_a_r', 'i_l_B5_B6_a_i', 'i_l_B5_B6_b_r', 'i_l_B5_B6_b_i', 'i_l_B5_B6_c_r', 'i_l_B5_B6_c_i', 'i_l_B6_B7_a_r', 'i_l_B6_B7_a_i', 'i_l_B6_B7_b_r', 'i_l_B6_B7_b_i', 'i_l_B6_B7_c_r', 'i_l_B6_B7_c_i', 'i_load_B2lv_a_r', 'i_load_B2lv_a_i', 'i_load_B2lv_b_r', 'i_load_B2lv_b_i', 'i_load_B2lv_c_r', 'i_load_B2lv_c_i', 'i_load_B2lv_n_r', 'i_load_B2lv_n_i', 'i_load_B3lv_a_r', 'i_load_B3lv_a_i', 'i_load_B3lv_b_r', 'i_load_B3lv_b_i', 'i_load_B3lv_c_r', 'i_load_B3lv_c_i', 'i_load_B3lv_n_r', 'i_load_B3lv_n_i', 'i_load_B4lv_a_r', 'i_load_B4lv_a_i', 'i_load_B4lv_b_r', 'i_load_B4lv_b_i', 'i_load_B4lv_c_r', 'i_load_B4lv_c_i', 'i_load_B4lv_n_r', 'i_load_B4lv_n_i', 'i_load_B5lv_a_r', 'i_load_B5lv_a_i', 'i_load_B5lv_b_r', 'i_load_B5lv_b_i', 'i_load_B5lv_c_r', 'i_load_B5lv_c_i', 'i_load_B5lv_n_r', 'i_load_B5lv_n_i', 'i_load_B6lv_a_r', 'i_load_B6lv_a_i', 'i_load_B6lv_b_r', 'i_load_B6lv_b_i', 'i_load_B6lv_c_r', 'i_load_B6lv_c_i', 'i_load_B6lv_n_r', 'i_load_B6lv_n_i', 'i_l_1112', 'i_l_1213', 'i_l_1314', 'i_l_1415', 'i_l_2122', 'i_l_2223', 'i_l_2324', 'i_l_2425', 'i_l_3132', 'i_l_3233', 'i_l_3334', 'i_l_3435', 'i_l_4142', 'i_l_4243', 'i_l_4344', 'i_l_4445', 'i_l_5152', 'i_l_5253', 'i_l_5354', 'i_l_5455', 'i_l_1521', 'i_l_2531', 'i_l_3541', 'i_l_4551', 'v_11', 'v_12', 'i_13', 'v_14', 'v_15', 'v_21', 'v_22', 'i_23', 'v_24', 'v_25', 'v_31', 'v_32', 'i_33', 'v_34', 'v_35', 'v_41', 'v_42', 'i_43', 'v_44', 'v_45', 'v_51', 'v_52', 'i_53', 'v_54', 'v_55', 'i_11', 'i_12', 'i_14', 'i_15', 'i_21', 'i_22', 'i_24', 'i_25', 'i_31', 'i_32', 'i_34', 'i_35', 'i_41', 'i_42', 'i_44', 'i_45', 'i_51', 'i_52', 'i_54', 'i_55', 'p_B2lv_a', 'p_B2lv_b', 'p_B2lv_c', 'p_B3lv_a', 'p_B3lv_b', 'p_B3lv_c', 'p_B4lv_a', 'p_B4lv_b', 'p_B4lv_c', 'p_B5lv_a', 'p_B5lv_b', 'p_B5lv_c', 'p_B6lv_a', 'p_B6lv_b', 'p_B6lv_c'] 
        self.xy_list = self.x_list + self.y_run_list 
        self.y_ini_list = ['v_B2lv_a_r', 'v_B2lv_a_i', 'v_B2lv_b_r', 'v_B2lv_b_i', 'v_B2lv_c_r', 'v_B2lv_c_i', 'v_B2lv_n_r', 'v_B2lv_n_i', 'v_B3lv_a_r', 'v_B3lv_a_i', 'v_B3lv_b_r', 'v_B3lv_b_i', 'v_B3lv_c_r', 'v_B3lv_c_i', 'v_B3lv_n_r', 'v_B3lv_n_i', 'v_B4lv_a_r', 'v_B4lv_a_i', 'v_B4lv_b_r', 'v_B4lv_b_i', 'v_B4lv_c_r', 'v_B4lv_c_i', 'v_B4lv_n_r', 'v_B4lv_n_i', 'v_B5lv_a_r', 'v_B5lv_a_i', 'v_B5lv_b_r', 'v_B5lv_b_i', 'v_B5lv_c_r', 'v_B5lv_c_i', 'v_B5lv_n_r', 'v_B5lv_n_i', 'v_B6lv_a_r', 'v_B6lv_a_i', 'v_B6lv_b_r', 'v_B6lv_b_i', 'v_B6lv_c_r', 'v_B6lv_c_i', 'v_B6lv_n_r', 'v_B6lv_n_i', 'v_B2_a_r', 'v_B2_a_i', 'v_B2_b_r', 'v_B2_b_i', 'v_B2_c_r', 'v_B2_c_i', 'v_B3_a_r', 'v_B3_a_i', 'v_B3_b_r', 'v_B3_b_i', 'v_B3_c_r', 'v_B3_c_i', 'v_B4_a_r', 'v_B4_a_i', 'v_B4_b_r', 'v_B4_b_i', 'v_B4_c_r', 'v_B4_c_i', 'v_B5_a_r', 'v_B5_a_i', 'v_B5_b_r', 'v_B5_b_i', 'v_B5_c_r', 'v_B5_c_i', 'v_B6_a_r', 'v_B6_a_i', 'v_B6_b_r', 'v_B6_b_i', 'v_B6_c_r', 'v_B6_c_i', 'i_t_B2_B2lv_a_r', 'i_t_B2_B2lv_a_i', 'i_t_B2_B2lv_b_r', 'i_t_B2_B2lv_b_i', 'i_t_B2_B2lv_c_r', 'i_t_B2_B2lv_c_i', 'i_t_B3_B3lv_a_r', 'i_t_B3_B3lv_a_i', 'i_t_B3_B3lv_b_r', 'i_t_B3_B3lv_b_i', 'i_t_B3_B3lv_c_r', 'i_t_B3_B3lv_c_i', 'i_t_B4_B4lv_a_r', 'i_t_B4_B4lv_a_i', 'i_t_B4_B4lv_b_r', 'i_t_B4_B4lv_b_i', 'i_t_B4_B4lv_c_r', 'i_t_B4_B4lv_c_i', 'i_t_B5_B5lv_a_r', 'i_t_B5_B5lv_a_i', 'i_t_B5_B5lv_b_r', 'i_t_B5_B5lv_b_i', 'i_t_B5_B5lv_c_r', 'i_t_B5_B5lv_c_i', 'i_t_B6_B6lv_a_r', 'i_t_B6_B6lv_a_i', 'i_t_B6_B6lv_b_r', 'i_t_B6_B6lv_b_i', 'i_t_B6_B6lv_c_r', 'i_t_B6_B6lv_c_i', 'i_l_B1_B2_a_r', 'i_l_B1_B2_a_i', 'i_l_B1_B2_b_r', 'i_l_B1_B2_b_i', 'i_l_B1_B2_c_r', 'i_l_B1_B2_c_i', 'i_l_B2_B3_a_r', 'i_l_B2_B3_a_i', 'i_l_B2_B3_b_r', 'i_l_B2_B3_b_i', 'i_l_B2_B3_c_r', 'i_l_B2_B3_c_i', 'i_l_B3_B4_a_r', 'i_l_B3_B4_a_i', 'i_l_B3_B4_b_r', 'i_l_B3_B4_b_i', 'i_l_B3_B4_c_r', 'i_l_B3_B4_c_i', 'i_l_B5_B6_a_r', 'i_l_B5_B6_a_i', 'i_l_B5_B6_b_r', 'i_l_B5_B6_b_i', 'i_l_B5_B6_c_r', 'i_l_B5_B6_c_i', 'i_l_B6_B7_a_r', 'i_l_B6_B7_a_i', 'i_l_B6_B7_b_r', 'i_l_B6_B7_b_i', 'i_l_B6_B7_c_r', 'i_l_B6_B7_c_i', 'i_load_B2lv_a_r', 'i_load_B2lv_a_i', 'i_load_B2lv_b_r', 'i_load_B2lv_b_i', 'i_load_B2lv_c_r', 'i_load_B2lv_c_i', 'i_load_B2lv_n_r', 'i_load_B2lv_n_i', 'i_load_B3lv_a_r', 'i_load_B3lv_a_i', 'i_load_B3lv_b_r', 'i_load_B3lv_b_i', 'i_load_B3lv_c_r', 'i_load_B3lv_c_i', 'i_load_B3lv_n_r', 'i_load_B3lv_n_i', 'i_load_B4lv_a_r', 'i_load_B4lv_a_i', 'i_load_B4lv_b_r', 'i_load_B4lv_b_i', 'i_load_B4lv_c_r', 'i_load_B4lv_c_i', 'i_load_B4lv_n_r', 'i_load_B4lv_n_i', 'i_load_B5lv_a_r', 'i_load_B5lv_a_i', 'i_load_B5lv_b_r', 'i_load_B5lv_b_i', 'i_load_B5lv_c_r', 'i_load_B5lv_c_i', 'i_load_B5lv_n_r', 'i_load_B5lv_n_i', 'i_load_B6lv_a_r', 'i_load_B6lv_a_i', 'i_load_B6lv_b_r', 'i_load_B6lv_b_i', 'i_load_B6lv_c_r', 'i_load_B6lv_c_i', 'i_load_B6lv_n_r', 'i_load_B6lv_n_i', 'i_l_1112', 'i_l_1213', 'i_l_1314', 'i_l_1415', 'i_l_2122', 'i_l_2223', 'i_l_2324', 'i_l_2425', 'i_l_3132', 'i_l_3233', 'i_l_3334', 'i_l_3435', 'i_l_4142', 'i_l_4243', 'i_l_4344', 'i_l_4445', 'i_l_5152', 'i_l_5253', 'i_l_5354', 'i_l_5455', 'i_l_1521', 'i_l_2531', 'i_l_3541', 'i_l_4551', 'v_11', 'v_12', 'i_13', 'v_14', 'v_15', 'v_21', 'v_22', 'i_23', 'v_24', 'v_25', 'v_31', 'v_32', 'i_33', 'v_34', 'v_35', 'v_41', 'v_42', 'i_43', 'v_44', 'v_45', 'v_51', 'v_52', 'i_53', 'v_54', 'v_55', 'i_11', 'i_12', 'i_14', 'i_15', 'i_21', 'i_22', 'i_24', 'i_25', 'i_31', 'i_32', 'i_34', 'i_35', 'i_41', 'i_42', 'i_44', 'i_45', 'i_51', 'i_52', 'i_54', 'i_55', 'p_B2lv_a', 'p_B2lv_b', 'p_B2lv_c', 'p_B3lv_a', 'p_B3lv_b', 'p_B3lv_c', 'p_B4lv_a', 'p_B4lv_b', 'p_B4lv_c', 'p_B5lv_a', 'p_B5lv_b', 'p_B5lv_c', 'p_B6lv_a', 'p_B6lv_b', 'p_B6lv_c'] 
        self.xy_ini_list = self.x_list + self.y_ini_list 
        self.t = 0.0
        self.it = 0
        self.it_store = 0
        self.xy_prev = np.zeros((self.N_x+self.N_y,1))
        self.initialization_tol = 1e-6
        self.N_u = len(self.inputs_run_list) 
        self.sopt_root_method='hybr'
        self.sopt_root_jac=True
        self.u_ini_list = self.inputs_ini_list
        self.u_ini_values_list = self.inputs_ini_values_list
        self.u_run_list = self.inputs_run_list
        self.u_run_values_list = self.inputs_run_values_list
        self.N_u = len(self.u_run_list)
        Fx_ini_rows,Fx_ini_cols,Fy_ini_rows,Fy_ini_cols,Gx_ini_rows,Gx_ini_cols,Gy_ini_rows,Gy_ini_cols = nonzeros()

        self.Fx_ini_rows = np.array(Fx_ini_rows) 
        if len(Fx_ini_rows) == 1: 
            self.Fx_ini_rows = np.array([[Fx_ini_rows]]).reshape(1,) 
            self.Fx_ini_cols = np.array([[Fx_ini_cols]]).reshape(1,)  
            
        self.Fx_ini_cols = np.array(Fx_ini_cols)
        self.Fy_ini_rows = np.array(Fy_ini_rows)        
        self.Fy_ini_cols = np.array(Fy_ini_cols)
        self.Gx_ini_rows = np.array(Gx_ini_rows)        
        self.Gx_ini_cols = np.array(Gx_ini_cols)
        self.Gy_ini_rows = np.array(Gy_ini_rows)        
        self.Gy_ini_cols = np.array(Gy_ini_cols)
        
        
        self.yini2urun = list(set(self.inputs_run_list).intersection(set(self.y_ini_list)))
        self.uini2yrun = list(set(self.y_run_list).intersection(set(self.inputs_ini_list)))

        self.update() 
                
    def update(self): 

        self.N_steps = int(np.ceil(self.t_end/self.Dt)) 
        dt = [  
              ('t_end', np.float64),
              ('Dt', np.float64),
              ('decimation', np.float64),
              ('itol', np.float64),
              ('Dt_max', np.float64),
              ('Dt_min', np.float64),
              ('solvern', np.int64),
              ('imax', np.int64),
              ('N_steps', np.int64),
              ('N_store', np.int64),
              ('N_x', np.int64),
              ('N_y', np.int64),
              ('N_z', np.int64),
              ('t', np.float64),
              ('it', np.int64),
              ('it_store', np.int64),
              ('idx', np.int64),
              ('idy', np.int64),
              ('f', np.float64, (self.N_x,1)),
              ('x', np.float64, (self.N_x,1)),
              ('x_0', np.float64, (self.N_x,1)),
              ('g', np.float64, (self.N_y,1)),
              ('y_run', np.float64, (self.N_y,1)),
              ('y_ini', np.float64, (self.N_y,1)),
              ('u_run', np.float64, (self.N_u,1)),
              ('y_0', np.float64, (self.N_y,1)),
              ('h', np.float64, (self.N_z,1)),
              ('Fx', np.float64, (self.N_x,self.N_x)),
              ('Fy', np.float64, (self.N_x,self.N_y)),
              ('Gx', np.float64, (self.N_y,self.N_x)),
              ('Gy', np.float64, (self.N_y,self.N_y)),
              ('Fu', np.float64, (self.N_x,self.N_u)),
              ('Gu', np.float64, (self.N_y,self.N_u)),
              ('Hx', np.float64, (self.N_z,self.N_x)),
              ('Hy', np.float64, (self.N_z,self.N_y)),
              ('Hu', np.float64, (self.N_z,self.N_u)),
              ('Fx_ini', np.float64, (self.N_x,self.N_x)),
              ('Fy_ini', np.float64, (self.N_x,self.N_y)),
              ('Gx_ini', np.float64, (self.N_y,self.N_x)),
              ('Gy_ini', np.float64, (self.N_y,self.N_y)),
              ('T', np.float64, (self.N_store+1,1)),
              ('X', np.float64, (self.N_store+1,self.N_x)),
              ('Y', np.float64, (self.N_store+1,self.N_y)),
              ('Z', np.float64, (self.N_store+1,self.N_z)),
              ('iters', np.float64, (self.N_store+1,1)),
              ('store', np.int64),
              ('Fx_ini_rows', np.int64, self.Fx_ini_rows.shape),
              ('Fx_ini_cols', np.int64, self.Fx_ini_cols.shape),
              ('Fy_ini_rows', np.int64, self.Fy_ini_rows.shape),
              ('Fy_ini_cols', np.int64, self.Fy_ini_cols.shape),
              ('Gx_ini_rows', np.int64, self.Gx_ini_rows.shape),
              ('Gx_ini_cols', np.int64, self.Gx_ini_cols.shape),
              ('Gy_ini_rows', np.int64, self.Gy_ini_rows.shape),
              ('Gy_ini_cols', np.int64, self.Gy_ini_cols.shape),
              ('Ac_ini', np.float64, ((self.N_x+self.N_y,self.N_x+self.N_y))),   
              ('fg', np.float64, ((self.N_x+self.N_y,1))),  
             ]



        
        
        values = [
                self.t_end,                          
                self.Dt,
                self.decimation,
                self.itol,
                self.Dt_max,
                self.Dt_min,
                self.solvern,
                self.imax,
                self.N_steps,
                self.N_store,
                self.N_x,
                self.N_y,
                self.N_z,
                self.t,
                self.it,
                self.it_store,
                0,                                     # idx
                0,                                     # idy
                np.zeros((self.N_x,1)),                # f
                np.zeros((self.N_x,1)),                # x
                np.zeros((self.N_x,1)),                # x_0
                np.zeros((self.N_y,1)),                # g
                np.zeros((self.N_y,1)),                # y_run
                np.zeros((self.N_y,1)),                # y_ini
                np.zeros((self.N_u,1)),                # u_run
                np.zeros((self.N_y,1)),                # y_0
                np.zeros((self.N_z,1)),                # h
                np.zeros((self.N_x,self.N_x)),         # Fx   
                np.zeros((self.N_x,self.N_y)),         # Fy 
                np.zeros((self.N_y,self.N_x)),         # Gx 
                np.zeros((self.N_y,self.N_y)),         # Fy
                np.zeros((self.N_x,self.N_u)),         # Fu 
                np.zeros((self.N_y,self.N_u)),         # Gu 
                np.zeros((self.N_z,self.N_x)),         # Hx 
                np.zeros((self.N_z,self.N_y)),         # Hy 
                np.zeros((self.N_z,self.N_u)),         # Hu 
                np.zeros((self.N_x,self.N_x)),         # Fx_ini  
                np.zeros((self.N_x,self.N_y)),         # Fy_ini 
                np.zeros((self.N_y,self.N_x)),         # Gx_ini 
                np.zeros((self.N_y,self.N_y)),         # Fy_ini 
                np.zeros((self.N_store+1,1)),          # T
                np.zeros((self.N_store+1,self.N_x)),   # X
                np.zeros((self.N_store+1,self.N_y)),   # Y
                np.zeros((self.N_store+1,self.N_z)),   # Z
                np.zeros((self.N_store+1,1)),          # iters
                1,
                self.Fx_ini_rows,       
                self.Fx_ini_cols,
                self.Fy_ini_rows,       
                self.Fy_ini_cols,
                self.Gx_ini_rows,        
                self.Gx_ini_cols,
                self.Gy_ini_rows,       
                self.Gy_ini_cols,
                np.zeros((self.N_x+self.N_y,self.N_x+self.N_y)),  
                np.zeros((self.N_x+self.N_y,1)),
                ]  

        dt += [(item,np.float64) for item in self.params_list]
        values += [item for item in self.params_values_list]

        for item_id,item_val in zip(self.inputs_ini_list,self.inputs_ini_values_list):
            if item_id in self.inputs_run_list: continue
            dt += [(item_id,np.float64)]
            values += [item_val]

        dt += [(item,np.float64) for item in self.inputs_run_list]
        values += [item for item in self.inputs_run_values_list]

        self.struct = np.rec.array([tuple(values)], dtype=np.dtype(dt))
        
        xy0 = np.zeros((self.N_x+self.N_y,))
        self.ini_dae_jacobian_nn(xy0)
        self.run_dae_jacobian_nn(xy0)
        


    def load_params(self,data_input):

        if type(data_input) == str:
            json_file = data_input
            self.json_file = json_file
            self.json_data = open(json_file).read().replace("'",'"')
            data = json.loads(self.json_data)
        elif type(data_input) == dict:
            data = data_input

        self.data = data
        for item in self.data:
            self.struct[0][item] = self.data[item]
            if item in self.params_list:
                self.params_values_list[self.params_list.index(item)] = self.data[item]
            elif item in self.inputs_ini_list:
                self.inputs_ini_values_list[self.inputs_ini_list.index(item)] = self.data[item]
            elif item in self.inputs_run_list:
                self.inputs_run_values_list[self.inputs_run_list.index(item)] = self.data[item]
            else: 
                print(f'parameter or input {item} not found')



    def ini_problem(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_ini[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        if self.compile:
            ini(self.struct,2)
            ini(self.struct,3)       
        else:
            ini.py_func(self.struct,2)
            ini.py_func(self.struct,3)                   
        fg = np.vstack((self.struct[0].f,self.struct[0].g))[:,0]
        return fg

    def run_problem(self,x):
        t = self.struct[0].t
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_run[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        
        if self.compile:
            run(t,self.struct,2)
            run(t,self.struct,3)
            run(t,self.struct,10)
            run(t,self.struct,11)
            run(t,self.struct,12)
            run(t,self.struct,13)
        else:
            run.py_func(t,self.struct,2)
            run.py_func(t,self.struct,3)
            run.py_func(t,self.struct,10)
            run.py_func(t,self.struct,11)
            run.py_func(t,self.struct,12)
            run.py_func(t,self.struct,13)            
        
        fg = np.vstack((self.struct[0].f,self.struct[0].g))[:,0]
        return fg
    

    def run_dae_jacobian(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_run[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        run(0.0,self.struct,10)
        run(0.0,self.struct,11)     
        run(0.0,self.struct,12)
        run(0.0,self.struct,13)
        A_c = np.block([[self.struct[0].Fx,self.struct[0].Fy],
                        [self.struct[0].Gx,self.struct[0].Gy]])
        return A_c

    def run_dae_jacobian_nn(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_run[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        run_nn(0.0,self.struct,10)
        run_nn(0.0,self.struct,11)     
        run_nn(0.0,self.struct,12)
        run_nn(0.0,self.struct,13)
 

    
    def eval_jacobians(self):

        run(0.0,self.struct,10)
        run(0.0,self.struct,11)  
        run(0.0,self.struct,12) 

        return 1


    def ini_dae_jacobian(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_ini[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        if self.compile:
            ini(self.struct,10)
            ini(self.struct,11) 
        else:
            ini.py_func(self.struct,10)
            ini.py_func(self.struct,11)             
        A_c = np.block([[self.struct[0].Fx_ini,self.struct[0].Fy_ini],
                        [self.struct[0].Gx_ini,self.struct[0].Gy_ini]])
        return A_c

    def ini_dae_jacobian_nn(self,x):
        self.struct[0].x[:,0] = x[0:self.N_x]
        self.struct[0].y_ini[:,0] = x[self.N_x:(self.N_x+self.N_y)]
        ini_nn(self.struct,10)
        ini_nn(self.struct,11)       
 

    def f_ode(self,x):
        self.struct[0].x[:,0] = x
        run(self.struct,1)
        return self.struct[0].f[:,0]

    def f_odeint(self,x,t):
        self.struct[0].x[:,0] = x
        run(self.struct,1)
        return self.struct[0].f[:,0]

    def f_ivp(self,t,x):
        self.struct[0].x[:,0] = x
        run(self.struct,1)
        return self.struct[0].f[:,0]

    def Fx_ode(self,x):
        self.struct[0].x[:,0] = x
        run(self.struct,10)
        return self.struct[0].Fx

    def eval_A(self):
        
        Fx = self.struct[0].Fx
        Fy = self.struct[0].Fy
        Gx = self.struct[0].Gx
        Gy = self.struct[0].Gy
        
        A = Fx - Fy @ np.linalg.solve(Gy,Gx)
        
        self.A = A
        
        return A

    def eval_A_ini(self):
        
        Fx = self.struct[0].Fx_ini
        Fy = self.struct[0].Fy_ini
        Gx = self.struct[0].Gx_ini
        Gy = self.struct[0].Gy_ini
        
        A = Fx - Fy @ np.linalg.solve(Gy,Gx)
        
        
        return A
    
    def reset(self):
        for param,param_value in zip(self.params_list,self.params_values_list):
            self.struct[0][param] = param_value
        for input_name,input_value in zip(self.inputs_ini_list,self.inputs_ini_values_list):
            self.struct[0][input_name] = input_value   
        for input_name,input_value in zip(self.inputs_run_list,self.inputs_run_values_list):
            self.struct[0][input_name] = input_value  

    def simulate(self,events,xy0=0):
        
        # initialize both the ini and the run system
        self.initialize(events,xy0=xy0)
        
        # simulation run
        for event in events:  
            # make all the desired changes
            self.run([event]) 
            
        # post process
        T,X,Y,Z = self.post()
        
        return T,X,Y,Z
    

    
    def run(self,events):
        

        # simulation run
        for event in events:  
            # make all the desired changes
            for item in event:
                self.struct[0][item] = event[item]
            daesolver(self.struct)    # run until next event
            
        return 1
 
    def rtrun(self,events):
        

        # simulation run
        for event in events:  
            # make all the desired changes
            for item in event:
                self.struct[0][item] = event[item]
            self.struct[0].it_store = self.struct[0].N_store-1
            daesolver(self.struct)    # run until next event
            
            
        return 1
    
    def post(self):
        
        # post process result    
        T = self.struct[0]['T'][:self.struct[0].it_store]
        X = self.struct[0]['X'][:self.struct[0].it_store,:]
        Y = self.struct[0]['Y'][:self.struct[0].it_store,:]
        Z = self.struct[0]['Z'][:self.struct[0].it_store,:]
        iters = self.struct[0]['iters'][:self.struct[0].it_store,:]
    
        self.T = T
        self.X = X
        self.Y = Y
        self.Z = Z
        self.iters = iters
        
        return T,X,Y,Z
        
    def save_0(self,file_name = 'xy_0.json'):
        xy_0_dict = {}
        for item in self.x_list:
            xy_0_dict.update({item:self.get_value(item)})
        for item in self.y_ini_list:
            xy_0_dict.update({item:self.get_value(item)})
    
        xy_0_str = json.dumps(xy_0_dict, indent=4)
        with open(file_name,'w') as fobj:
            fobj.write(xy_0_str)

    def load_0(self,file_name = 'xy_0.json'):
        with open(file_name) as fobj:
            xy_0_str = fobj.read()
        xy_0_dict = json.loads(xy_0_str)
    
        for item in xy_0_dict:
            if item in self.x_list:
                self.xy_prev[self.x_list.index(item)] = xy_0_dict[item]
            if item in self.y_ini_list:
                self.xy_prev[self.y_ini_list.index(item)+self.N_x] = xy_0_dict[item]
                
            
    def initialize(self,events=[{}],xy0=0,compile=True):
        '''
        

        Parameters
        ----------
        events : dictionary 
            Dictionary with at least 't_end' and all inputs and parameters 
            that need to be changed.
        xy0 : float or string, optional
            0 means all states should be zero as initial guess. 
            If not zero all the states initial guess are the given input.
            If 'prev' it uses the last known initialization result as initial guess.

        Returns
        -------
        T : TYPE
            DESCRIPTION.
        X : TYPE
            DESCRIPTION.
        Y : TYPE
            DESCRIPTION.
        Z : TYPE
            DESCRIPTION.

        '''
        
        self.compile = compile
        
        # simulation parameters
        self.struct[0].it = 0       # set time step to zero
        self.struct[0].it_store = 0 # set storage to zero
        self.struct[0].t = 0.0      # set time to zero
                    
        # initialization
        it_event = 0
        event = events[it_event]
        for item in event:
            self.struct[0][item] = event[item]
            
        
        ## compute initial conditions using x and y_ini 
        if type(xy0) == str:
            if xy0 == 'prev':
                xy0 = self.xy_prev
            else:
                self.load_0(xy0)
                xy0 = self.xy_prev
        elif type(xy0) == dict:
            with open('xy_0.json','w') as fobj:
                fobj.write(json.dumps(xy0))
            self.load_0('xy_0.json')
            xy0 = self.xy_prev            
        else:
            if xy0 == 0:
                xy0 = np.zeros(self.N_x+self.N_y)
            elif xy0 == 1:
                xy0 = np.ones(self.N_x+self.N_y)
            else:
                xy0 = xy0*np.ones(self.N_x+self.N_y)

        #xy = sopt.fsolve(self.ini_problem,xy0, jac=self.ini_dae_jacobian )

        
        if self.sopt_root_jac:
            sol = sopt.root(self.ini_problem, xy0, 
                            jac=self.ini_dae_jacobian, 
                            method=self.sopt_root_method, tol=self.initialization_tol)
        else:
            sol = sopt.root(self.ini_problem, xy0, method=self.sopt_root_method)

        self.initialization_ok = True
        if sol.success == False:
            print('initialization not found!')
            self.initialization_ok = False

            T = self.struct[0]['T'][:self.struct[0].it_store]
            X = self.struct[0]['X'][:self.struct[0].it_store,:]
            Y = self.struct[0]['Y'][:self.struct[0].it_store,:]
            Z = self.struct[0]['Z'][:self.struct[0].it_store,:]
            iters = self.struct[0]['iters'][:self.struct[0].it_store,:]

        if self.initialization_ok:
            xy = sol.x
            self.xy_prev = xy
            self.struct[0].x[:,0] = xy[0:self.N_x]
            self.struct[0].y_run[:,0] = xy[self.N_x:]

            ## y_ini to u_run
            for item in self.inputs_run_list:
                if item in self.y_ini_list:
                    self.struct[0][item] = self.struct[0].y_ini[self.y_ini_list.index(item)]

            ## u_ini to y_run
            for item in self.inputs_ini_list:
                if item in self.y_run_list:
                    self.struct[0].y_run[self.y_run_list.index(item)] = self.struct[0][item]


            #xy = sopt.fsolve(self.ini_problem,xy0, jac=self.ini_dae_jacobian )
            if self.sopt_root_jac:
                sol = sopt.root(self.run_problem, xy0, 
                                jac=self.run_dae_jacobian, 
                                method=self.sopt_root_method, tol=self.initialization_tol)
            else:
                sol = sopt.root(self.run_problem, xy0, method=self.sopt_root_method)

            if self.compile:
                # evaluate f and g
                run(0.0,self.struct,2)
                run(0.0,self.struct,3)                
    
                # evaluate run jacobians 
                run(0.0,self.struct,10)
                run(0.0,self.struct,11)                
                run(0.0,self.struct,12) 
                run(0.0,self.struct,14) 
                
            else:
                # evaluate f and g
                run.py_func(0.0,self.struct,2)
                run.py_func(0.0,self.struct,3)                
    
                # evaluate run jacobians 
                run.py_func(0.0,self.struct,10)
                run.py_func(0.0,self.struct,11)                
                run.py_func(0.0,self.struct,12) 
                run.py_func(0.0,self.struct,14)                 
                
             
            # post process result    
            T = self.struct[0]['T'][:self.struct[0].it_store]
            X = self.struct[0]['X'][:self.struct[0].it_store,:]
            Y = self.struct[0]['Y'][:self.struct[0].it_store,:]
            Z = self.struct[0]['Z'][:self.struct[0].it_store,:]
            iters = self.struct[0]['iters'][:self.struct[0].it_store,:]
        
            self.T = T
            self.X = X
            self.Y = Y
            self.Z = Z
            self.iters = iters
            
        return self.initialization_ok
    
    
    def get_value(self,name):
        if name in self.inputs_run_list:
            value = self.struct[0][name]
        if name in self.x_list:
            idx = self.x_list.index(name)
            value = self.struct[0].x[idx,0]
        if name in self.y_run_list:
            idy = self.y_run_list.index(name)
            value = self.struct[0].y_run[idy,0]
        if name in self.params_list:
            value = self.struct[0][name]
        if name in self.outputs_list:
            value = self.struct[0].h[self.outputs_list.index(name),0] 

        return value
    
    def get_values(self,name):
        if name in self.x_list:
            values = self.X[:,self.x_list.index(name)]
        if name in self.y_run_list:
            values = self.Y[:,self.y_run_list.index(name)]
        if name in self.outputs_list:
            values = self.Z[:,self.outputs_list.index(name)]
                        
        return values

    def get_mvalue(self,names):
        '''

        Parameters
        ----------
        names : list
            list of variables names to return each value.

        Returns
        -------
        mvalue : TYPE
            list of value of each variable.

        '''
        mvalue = []
        for name in names:
            mvalue += [self.get_value(name)]
                        
        return mvalue
    
    def set_value(self,name_,value):
        if name_ in self.inputs_run_list:
            self.struct[0][name_] = value
            return
        elif name_ in self.params_list:
            self.struct[0][name_] = value
            return
        elif name_ in self.inputs_ini_list:
            self.struct[0][name_] = value
            return 
        else:
            print(f'Input or parameter {name_} not found.')

    def set_values(self,dictionary):
        
        for item in dictionary:
            self.set_value(item,dictionary[item])
            
            
    def report_x(self,value_format='5.2f', decimals=2):
        for item in self.x_list:
            print(f'{item:5s} = {self.get_value(item):5.{decimals}f}')

    def report_y(self,value_format='5.2f', decimals=2):
        for item in self.y_run_list:
            print(f'{item:5s} = {self.get_value(item):5.{decimals}f}')
            
    def report_u(self,value_format='5.2f', decimals=2):
        for item in self.inputs_run_list:
            print(f'{item:5s} = {self.get_value(item):5.{decimals}f}')

    def report_z(self,value_format='5.2f', decimals=2):
        for item in self.outputs_list:
            print(f'{item:5s} = {self.get_value(item):5.{decimals}f}')

    def report_params(self,value_format='5.2f', decimals=2):
        for item in self.params_list:
            print(f'{item:5s} = {self.get_value(item):5.{decimals}f}')
            
    def get_x(self):
        return self.struct[0].x
    
    def ss(self):
        
        ssate(self.struct,self.xy_prev.reshape(len(self.xy_prev),1))
        
        ## y_ini to y_run
        self.struct[0].y_run = self.struct[0].y_ini
        
        ## y_ini to u_run
        for item in self.yini2urun:
            self.struct[0][item] = self.struct[0].y_ini[self.y_ini_list.index(item)]
                
        ## u_ini to y_run
        for item in self.uini2yrun:
            self.struct[0].y_run[self.y_run_list.index(item)] = self.struct[0][item]






@numba.njit(cache=True)
def ini(struct,mode):

    # Parameters:
    R_1112 = struct[0].R_1112
    R_1213 = struct[0].R_1213
    R_1314 = struct[0].R_1314
    R_1415 = struct[0].R_1415
    R_1521 = struct[0].R_1521
    R_2122 = struct[0].R_2122
    R_2223 = struct[0].R_2223
    R_2324 = struct[0].R_2324
    R_2425 = struct[0].R_2425
    R_2531 = struct[0].R_2531
    R_3132 = struct[0].R_3132
    R_3233 = struct[0].R_3233
    R_3334 = struct[0].R_3334
    R_3435 = struct[0].R_3435
    R_3541 = struct[0].R_3541
    R_4142 = struct[0].R_4142
    R_4243 = struct[0].R_4243
    R_4344 = struct[0].R_4344
    R_4445 = struct[0].R_4445
    R_4551 = struct[0].R_4551
    R_5152 = struct[0].R_5152
    R_5253 = struct[0].R_5253
    R_5354 = struct[0].R_5354
    R_5455 = struct[0].R_5455
    p_11 = struct[0].p_11
    p_12 = struct[0].p_12
    p_14 = struct[0].p_14
    p_15 = struct[0].p_15
    p_21 = struct[0].p_21
    p_22 = struct[0].p_22
    p_24 = struct[0].p_24
    p_25 = struct[0].p_25
    p_31 = struct[0].p_31
    p_32 = struct[0].p_32
    p_34 = struct[0].p_34
    p_35 = struct[0].p_35
    p_41 = struct[0].p_41
    p_42 = struct[0].p_42
    p_44 = struct[0].p_44
    p_45 = struct[0].p_45
    p_51 = struct[0].p_51
    p_52 = struct[0].p_52
    p_54 = struct[0].p_54
    p_55 = struct[0].p_55
    
    # Inputs:
    v_B1_a_r = struct[0].v_B1_a_r
    v_B1_a_i = struct[0].v_B1_a_i
    v_B1_b_r = struct[0].v_B1_b_r
    v_B1_b_i = struct[0].v_B1_b_i
    v_B1_c_r = struct[0].v_B1_c_r
    v_B1_c_i = struct[0].v_B1_c_i
    v_B7_a_r = struct[0].v_B7_a_r
    v_B7_a_i = struct[0].v_B7_a_i
    v_B7_b_r = struct[0].v_B7_b_r
    v_B7_b_i = struct[0].v_B7_b_i
    v_B7_c_r = struct[0].v_B7_c_r
    v_B7_c_i = struct[0].v_B7_c_i
    i_B2lv_n_r = struct[0].i_B2lv_n_r
    i_B2lv_n_i = struct[0].i_B2lv_n_i
    i_B3lv_n_r = struct[0].i_B3lv_n_r
    i_B3lv_n_i = struct[0].i_B3lv_n_i
    i_B4lv_n_r = struct[0].i_B4lv_n_r
    i_B4lv_n_i = struct[0].i_B4lv_n_i
    i_B5lv_n_r = struct[0].i_B5lv_n_r
    i_B5lv_n_i = struct[0].i_B5lv_n_i
    i_B6lv_n_r = struct[0].i_B6lv_n_r
    i_B6lv_n_i = struct[0].i_B6lv_n_i
    i_B2_a_r = struct[0].i_B2_a_r
    i_B2_a_i = struct[0].i_B2_a_i
    i_B2_b_r = struct[0].i_B2_b_r
    i_B2_b_i = struct[0].i_B2_b_i
    i_B2_c_r = struct[0].i_B2_c_r
    i_B2_c_i = struct[0].i_B2_c_i
    i_B3_a_r = struct[0].i_B3_a_r
    i_B3_a_i = struct[0].i_B3_a_i
    i_B3_b_r = struct[0].i_B3_b_r
    i_B3_b_i = struct[0].i_B3_b_i
    i_B3_c_r = struct[0].i_B3_c_r
    i_B3_c_i = struct[0].i_B3_c_i
    i_B4_a_r = struct[0].i_B4_a_r
    i_B4_a_i = struct[0].i_B4_a_i
    i_B4_b_r = struct[0].i_B4_b_r
    i_B4_b_i = struct[0].i_B4_b_i
    i_B4_c_r = struct[0].i_B4_c_r
    i_B4_c_i = struct[0].i_B4_c_i
    i_B5_a_r = struct[0].i_B5_a_r
    i_B5_a_i = struct[0].i_B5_a_i
    i_B5_b_r = struct[0].i_B5_b_r
    i_B5_b_i = struct[0].i_B5_b_i
    i_B5_c_r = struct[0].i_B5_c_r
    i_B5_c_i = struct[0].i_B5_c_i
    i_B6_a_r = struct[0].i_B6_a_r
    i_B6_a_i = struct[0].i_B6_a_i
    i_B6_b_r = struct[0].i_B6_b_r
    i_B6_b_i = struct[0].i_B6_b_i
    i_B6_c_r = struct[0].i_B6_c_r
    i_B6_c_i = struct[0].i_B6_c_i
    q_B2lv_a = struct[0].q_B2lv_a
    q_B2lv_b = struct[0].q_B2lv_b
    q_B2lv_c = struct[0].q_B2lv_c
    q_B3lv_a = struct[0].q_B3lv_a
    q_B3lv_b = struct[0].q_B3lv_b
    q_B3lv_c = struct[0].q_B3lv_c
    q_B4lv_a = struct[0].q_B4lv_a
    q_B4lv_b = struct[0].q_B4lv_b
    q_B4lv_c = struct[0].q_B4lv_c
    q_B5lv_a = struct[0].q_B5lv_a
    q_B5lv_b = struct[0].q_B5lv_b
    q_B5lv_c = struct[0].q_B5lv_c
    q_B6lv_a = struct[0].q_B6lv_a
    q_B6lv_b = struct[0].q_B6lv_b
    q_B6lv_c = struct[0].q_B6lv_c
    u_dummy = struct[0].u_dummy
    Dv_r_13 = struct[0].Dv_r_13
    Dv_r_23 = struct[0].Dv_r_23
    Dv_r_33 = struct[0].Dv_r_33
    Dv_r_43 = struct[0].Dv_r_43
    Dv_r_53 = struct[0].Dv_r_53
    v_nom = struct[0].v_nom
    T_v = struct[0].T_v
    K_r = struct[0].K_r
    a_B2lv = struct[0].a_B2lv
    b_B2lv = struct[0].b_B2lv
    c_B2lv = struct[0].c_B2lv
    a_B3lv = struct[0].a_B3lv
    b_B3lv = struct[0].b_B3lv
    c_B3lv = struct[0].c_B3lv
    a_B4lv = struct[0].a_B4lv
    b_B4lv = struct[0].b_B4lv
    c_B4lv = struct[0].c_B4lv
    a_B5lv = struct[0].a_B5lv
    b_B5lv = struct[0].b_B5lv
    c_B5lv = struct[0].c_B5lv
    a_B6lv = struct[0].a_B6lv
    b_B6lv = struct[0].b_B6lv
    c_B6lv = struct[0].c_B6lv
    
    # Dynamical states:
    v_13 = struct[0].x[0,0]
    v_23 = struct[0].x[1,0]
    v_33 = struct[0].x[2,0]
    v_43 = struct[0].x[3,0]
    v_53 = struct[0].x[4,0]
    
    # Algebraic states:
    v_B2lv_a_r = struct[0].y_ini[0,0]
    v_B2lv_a_i = struct[0].y_ini[1,0]
    v_B2lv_b_r = struct[0].y_ini[2,0]
    v_B2lv_b_i = struct[0].y_ini[3,0]
    v_B2lv_c_r = struct[0].y_ini[4,0]
    v_B2lv_c_i = struct[0].y_ini[5,0]
    v_B2lv_n_r = struct[0].y_ini[6,0]
    v_B2lv_n_i = struct[0].y_ini[7,0]
    v_B3lv_a_r = struct[0].y_ini[8,0]
    v_B3lv_a_i = struct[0].y_ini[9,0]
    v_B3lv_b_r = struct[0].y_ini[10,0]
    v_B3lv_b_i = struct[0].y_ini[11,0]
    v_B3lv_c_r = struct[0].y_ini[12,0]
    v_B3lv_c_i = struct[0].y_ini[13,0]
    v_B3lv_n_r = struct[0].y_ini[14,0]
    v_B3lv_n_i = struct[0].y_ini[15,0]
    v_B4lv_a_r = struct[0].y_ini[16,0]
    v_B4lv_a_i = struct[0].y_ini[17,0]
    v_B4lv_b_r = struct[0].y_ini[18,0]
    v_B4lv_b_i = struct[0].y_ini[19,0]
    v_B4lv_c_r = struct[0].y_ini[20,0]
    v_B4lv_c_i = struct[0].y_ini[21,0]
    v_B4lv_n_r = struct[0].y_ini[22,0]
    v_B4lv_n_i = struct[0].y_ini[23,0]
    v_B5lv_a_r = struct[0].y_ini[24,0]
    v_B5lv_a_i = struct[0].y_ini[25,0]
    v_B5lv_b_r = struct[0].y_ini[26,0]
    v_B5lv_b_i = struct[0].y_ini[27,0]
    v_B5lv_c_r = struct[0].y_ini[28,0]
    v_B5lv_c_i = struct[0].y_ini[29,0]
    v_B5lv_n_r = struct[0].y_ini[30,0]
    v_B5lv_n_i = struct[0].y_ini[31,0]
    v_B6lv_a_r = struct[0].y_ini[32,0]
    v_B6lv_a_i = struct[0].y_ini[33,0]
    v_B6lv_b_r = struct[0].y_ini[34,0]
    v_B6lv_b_i = struct[0].y_ini[35,0]
    v_B6lv_c_r = struct[0].y_ini[36,0]
    v_B6lv_c_i = struct[0].y_ini[37,0]
    v_B6lv_n_r = struct[0].y_ini[38,0]
    v_B6lv_n_i = struct[0].y_ini[39,0]
    v_B2_a_r = struct[0].y_ini[40,0]
    v_B2_a_i = struct[0].y_ini[41,0]
    v_B2_b_r = struct[0].y_ini[42,0]
    v_B2_b_i = struct[0].y_ini[43,0]
    v_B2_c_r = struct[0].y_ini[44,0]
    v_B2_c_i = struct[0].y_ini[45,0]
    v_B3_a_r = struct[0].y_ini[46,0]
    v_B3_a_i = struct[0].y_ini[47,0]
    v_B3_b_r = struct[0].y_ini[48,0]
    v_B3_b_i = struct[0].y_ini[49,0]
    v_B3_c_r = struct[0].y_ini[50,0]
    v_B3_c_i = struct[0].y_ini[51,0]
    v_B4_a_r = struct[0].y_ini[52,0]
    v_B4_a_i = struct[0].y_ini[53,0]
    v_B4_b_r = struct[0].y_ini[54,0]
    v_B4_b_i = struct[0].y_ini[55,0]
    v_B4_c_r = struct[0].y_ini[56,0]
    v_B4_c_i = struct[0].y_ini[57,0]
    v_B5_a_r = struct[0].y_ini[58,0]
    v_B5_a_i = struct[0].y_ini[59,0]
    v_B5_b_r = struct[0].y_ini[60,0]
    v_B5_b_i = struct[0].y_ini[61,0]
    v_B5_c_r = struct[0].y_ini[62,0]
    v_B5_c_i = struct[0].y_ini[63,0]
    v_B6_a_r = struct[0].y_ini[64,0]
    v_B6_a_i = struct[0].y_ini[65,0]
    v_B6_b_r = struct[0].y_ini[66,0]
    v_B6_b_i = struct[0].y_ini[67,0]
    v_B6_c_r = struct[0].y_ini[68,0]
    v_B6_c_i = struct[0].y_ini[69,0]
    i_t_B2_B2lv_a_r = struct[0].y_ini[70,0]
    i_t_B2_B2lv_a_i = struct[0].y_ini[71,0]
    i_t_B2_B2lv_b_r = struct[0].y_ini[72,0]
    i_t_B2_B2lv_b_i = struct[0].y_ini[73,0]
    i_t_B2_B2lv_c_r = struct[0].y_ini[74,0]
    i_t_B2_B2lv_c_i = struct[0].y_ini[75,0]
    i_t_B3_B3lv_a_r = struct[0].y_ini[76,0]
    i_t_B3_B3lv_a_i = struct[0].y_ini[77,0]
    i_t_B3_B3lv_b_r = struct[0].y_ini[78,0]
    i_t_B3_B3lv_b_i = struct[0].y_ini[79,0]
    i_t_B3_B3lv_c_r = struct[0].y_ini[80,0]
    i_t_B3_B3lv_c_i = struct[0].y_ini[81,0]
    i_t_B4_B4lv_a_r = struct[0].y_ini[82,0]
    i_t_B4_B4lv_a_i = struct[0].y_ini[83,0]
    i_t_B4_B4lv_b_r = struct[0].y_ini[84,0]
    i_t_B4_B4lv_b_i = struct[0].y_ini[85,0]
    i_t_B4_B4lv_c_r = struct[0].y_ini[86,0]
    i_t_B4_B4lv_c_i = struct[0].y_ini[87,0]
    i_t_B5_B5lv_a_r = struct[0].y_ini[88,0]
    i_t_B5_B5lv_a_i = struct[0].y_ini[89,0]
    i_t_B5_B5lv_b_r = struct[0].y_ini[90,0]
    i_t_B5_B5lv_b_i = struct[0].y_ini[91,0]
    i_t_B5_B5lv_c_r = struct[0].y_ini[92,0]
    i_t_B5_B5lv_c_i = struct[0].y_ini[93,0]
    i_t_B6_B6lv_a_r = struct[0].y_ini[94,0]
    i_t_B6_B6lv_a_i = struct[0].y_ini[95,0]
    i_t_B6_B6lv_b_r = struct[0].y_ini[96,0]
    i_t_B6_B6lv_b_i = struct[0].y_ini[97,0]
    i_t_B6_B6lv_c_r = struct[0].y_ini[98,0]
    i_t_B6_B6lv_c_i = struct[0].y_ini[99,0]
    i_l_B1_B2_a_r = struct[0].y_ini[100,0]
    i_l_B1_B2_a_i = struct[0].y_ini[101,0]
    i_l_B1_B2_b_r = struct[0].y_ini[102,0]
    i_l_B1_B2_b_i = struct[0].y_ini[103,0]
    i_l_B1_B2_c_r = struct[0].y_ini[104,0]
    i_l_B1_B2_c_i = struct[0].y_ini[105,0]
    i_l_B2_B3_a_r = struct[0].y_ini[106,0]
    i_l_B2_B3_a_i = struct[0].y_ini[107,0]
    i_l_B2_B3_b_r = struct[0].y_ini[108,0]
    i_l_B2_B3_b_i = struct[0].y_ini[109,0]
    i_l_B2_B3_c_r = struct[0].y_ini[110,0]
    i_l_B2_B3_c_i = struct[0].y_ini[111,0]
    i_l_B3_B4_a_r = struct[0].y_ini[112,0]
    i_l_B3_B4_a_i = struct[0].y_ini[113,0]
    i_l_B3_B4_b_r = struct[0].y_ini[114,0]
    i_l_B3_B4_b_i = struct[0].y_ini[115,0]
    i_l_B3_B4_c_r = struct[0].y_ini[116,0]
    i_l_B3_B4_c_i = struct[0].y_ini[117,0]
    i_l_B5_B6_a_r = struct[0].y_ini[118,0]
    i_l_B5_B6_a_i = struct[0].y_ini[119,0]
    i_l_B5_B6_b_r = struct[0].y_ini[120,0]
    i_l_B5_B6_b_i = struct[0].y_ini[121,0]
    i_l_B5_B6_c_r = struct[0].y_ini[122,0]
    i_l_B5_B6_c_i = struct[0].y_ini[123,0]
    i_l_B6_B7_a_r = struct[0].y_ini[124,0]
    i_l_B6_B7_a_i = struct[0].y_ini[125,0]
    i_l_B6_B7_b_r = struct[0].y_ini[126,0]
    i_l_B6_B7_b_i = struct[0].y_ini[127,0]
    i_l_B6_B7_c_r = struct[0].y_ini[128,0]
    i_l_B6_B7_c_i = struct[0].y_ini[129,0]
    i_load_B2lv_a_r = struct[0].y_ini[130,0]
    i_load_B2lv_a_i = struct[0].y_ini[131,0]
    i_load_B2lv_b_r = struct[0].y_ini[132,0]
    i_load_B2lv_b_i = struct[0].y_ini[133,0]
    i_load_B2lv_c_r = struct[0].y_ini[134,0]
    i_load_B2lv_c_i = struct[0].y_ini[135,0]
    i_load_B2lv_n_r = struct[0].y_ini[136,0]
    i_load_B2lv_n_i = struct[0].y_ini[137,0]
    i_load_B3lv_a_r = struct[0].y_ini[138,0]
    i_load_B3lv_a_i = struct[0].y_ini[139,0]
    i_load_B3lv_b_r = struct[0].y_ini[140,0]
    i_load_B3lv_b_i = struct[0].y_ini[141,0]
    i_load_B3lv_c_r = struct[0].y_ini[142,0]
    i_load_B3lv_c_i = struct[0].y_ini[143,0]
    i_load_B3lv_n_r = struct[0].y_ini[144,0]
    i_load_B3lv_n_i = struct[0].y_ini[145,0]
    i_load_B4lv_a_r = struct[0].y_ini[146,0]
    i_load_B4lv_a_i = struct[0].y_ini[147,0]
    i_load_B4lv_b_r = struct[0].y_ini[148,0]
    i_load_B4lv_b_i = struct[0].y_ini[149,0]
    i_load_B4lv_c_r = struct[0].y_ini[150,0]
    i_load_B4lv_c_i = struct[0].y_ini[151,0]
    i_load_B4lv_n_r = struct[0].y_ini[152,0]
    i_load_B4lv_n_i = struct[0].y_ini[153,0]
    i_load_B5lv_a_r = struct[0].y_ini[154,0]
    i_load_B5lv_a_i = struct[0].y_ini[155,0]
    i_load_B5lv_b_r = struct[0].y_ini[156,0]
    i_load_B5lv_b_i = struct[0].y_ini[157,0]
    i_load_B5lv_c_r = struct[0].y_ini[158,0]
    i_load_B5lv_c_i = struct[0].y_ini[159,0]
    i_load_B5lv_n_r = struct[0].y_ini[160,0]
    i_load_B5lv_n_i = struct[0].y_ini[161,0]
    i_load_B6lv_a_r = struct[0].y_ini[162,0]
    i_load_B6lv_a_i = struct[0].y_ini[163,0]
    i_load_B6lv_b_r = struct[0].y_ini[164,0]
    i_load_B6lv_b_i = struct[0].y_ini[165,0]
    i_load_B6lv_c_r = struct[0].y_ini[166,0]
    i_load_B6lv_c_i = struct[0].y_ini[167,0]
    i_load_B6lv_n_r = struct[0].y_ini[168,0]
    i_load_B6lv_n_i = struct[0].y_ini[169,0]
    i_l_1112 = struct[0].y_ini[170,0]
    i_l_1213 = struct[0].y_ini[171,0]
    i_l_1314 = struct[0].y_ini[172,0]
    i_l_1415 = struct[0].y_ini[173,0]
    i_l_2122 = struct[0].y_ini[174,0]
    i_l_2223 = struct[0].y_ini[175,0]
    i_l_2324 = struct[0].y_ini[176,0]
    i_l_2425 = struct[0].y_ini[177,0]
    i_l_3132 = struct[0].y_ini[178,0]
    i_l_3233 = struct[0].y_ini[179,0]
    i_l_3334 = struct[0].y_ini[180,0]
    i_l_3435 = struct[0].y_ini[181,0]
    i_l_4142 = struct[0].y_ini[182,0]
    i_l_4243 = struct[0].y_ini[183,0]
    i_l_4344 = struct[0].y_ini[184,0]
    i_l_4445 = struct[0].y_ini[185,0]
    i_l_5152 = struct[0].y_ini[186,0]
    i_l_5253 = struct[0].y_ini[187,0]
    i_l_5354 = struct[0].y_ini[188,0]
    i_l_5455 = struct[0].y_ini[189,0]
    i_l_1521 = struct[0].y_ini[190,0]
    i_l_2531 = struct[0].y_ini[191,0]
    i_l_3541 = struct[0].y_ini[192,0]
    i_l_4551 = struct[0].y_ini[193,0]
    v_11 = struct[0].y_ini[194,0]
    v_12 = struct[0].y_ini[195,0]
    i_13 = struct[0].y_ini[196,0]
    v_14 = struct[0].y_ini[197,0]
    v_15 = struct[0].y_ini[198,0]
    v_21 = struct[0].y_ini[199,0]
    v_22 = struct[0].y_ini[200,0]
    i_23 = struct[0].y_ini[201,0]
    v_24 = struct[0].y_ini[202,0]
    v_25 = struct[0].y_ini[203,0]
    v_31 = struct[0].y_ini[204,0]
    v_32 = struct[0].y_ini[205,0]
    i_33 = struct[0].y_ini[206,0]
    v_34 = struct[0].y_ini[207,0]
    v_35 = struct[0].y_ini[208,0]
    v_41 = struct[0].y_ini[209,0]
    v_42 = struct[0].y_ini[210,0]
    i_43 = struct[0].y_ini[211,0]
    v_44 = struct[0].y_ini[212,0]
    v_45 = struct[0].y_ini[213,0]
    v_51 = struct[0].y_ini[214,0]
    v_52 = struct[0].y_ini[215,0]
    i_53 = struct[0].y_ini[216,0]
    v_54 = struct[0].y_ini[217,0]
    v_55 = struct[0].y_ini[218,0]
    i_11 = struct[0].y_ini[219,0]
    i_12 = struct[0].y_ini[220,0]
    i_14 = struct[0].y_ini[221,0]
    i_15 = struct[0].y_ini[222,0]
    i_21 = struct[0].y_ini[223,0]
    i_22 = struct[0].y_ini[224,0]
    i_24 = struct[0].y_ini[225,0]
    i_25 = struct[0].y_ini[226,0]
    i_31 = struct[0].y_ini[227,0]
    i_32 = struct[0].y_ini[228,0]
    i_34 = struct[0].y_ini[229,0]
    i_35 = struct[0].y_ini[230,0]
    i_41 = struct[0].y_ini[231,0]
    i_42 = struct[0].y_ini[232,0]
    i_44 = struct[0].y_ini[233,0]
    i_45 = struct[0].y_ini[234,0]
    i_51 = struct[0].y_ini[235,0]
    i_52 = struct[0].y_ini[236,0]
    i_54 = struct[0].y_ini[237,0]
    i_55 = struct[0].y_ini[238,0]
    p_B2lv_a = struct[0].y_ini[239,0]
    p_B2lv_b = struct[0].y_ini[240,0]
    p_B2lv_c = struct[0].y_ini[241,0]
    p_B3lv_a = struct[0].y_ini[242,0]
    p_B3lv_b = struct[0].y_ini[243,0]
    p_B3lv_c = struct[0].y_ini[244,0]
    p_B4lv_a = struct[0].y_ini[245,0]
    p_B4lv_b = struct[0].y_ini[246,0]
    p_B4lv_c = struct[0].y_ini[247,0]
    p_B5lv_a = struct[0].y_ini[248,0]
    p_B5lv_b = struct[0].y_ini[249,0]
    p_B5lv_c = struct[0].y_ini[250,0]
    p_B6lv_a = struct[0].y_ini[251,0]
    p_B6lv_b = struct[0].y_ini[252,0]
    p_B6lv_c = struct[0].y_ini[253,0]
    
    # Differential equations:
    if mode == 2:


        struct[0].f[0,0] = (-Dv_r_13 - K_r*i_13*v_13 - v_13 + v_nom)/T_v
        struct[0].f[1,0] = (-Dv_r_23 - K_r*i_23*v_23 - v_23 + v_nom)/T_v
        struct[0].f[2,0] = (-Dv_r_33 - K_r*i_33*v_33 - v_33 + v_nom)/T_v
        struct[0].f[3,0] = (-Dv_r_43 - K_r*i_43*v_43 - v_43 + v_nom)/T_v
        struct[0].f[4,0] = (-Dv_r_53 - K_r*i_53*v_53 - v_53 + v_nom)/T_v
    
    # Algebraic equations:
    if mode == 3:

        struct[0].g[:,:] = np.ascontiguousarray(struct[0].Gy_ini) @ np.ascontiguousarray(struct[0].y_ini)

        struct[0].g[40,0] = 0.598820527961361*v_B1_a_i + 1.10755301189314*v_B1_a_r - 0.171091579417532*v_B1_b_i - 0.316443717683753*v_B1_b_r - 0.171091579417532*v_B1_c_i - 0.316443717683753*v_B1_c_r - 1.28353302446119*v_B2_a_i - 2.23667465123725*v_B2_a_r + 0.385473430243205*v_B2_b_i + 0.643671749092996*v_B2_b_r + 0.385473430243205*v_B2_c_i + 0.643671749092997*v_B2_c_r + 0.996212229189942*v_B2lv_a_i + 0.249053057297486*v_B2lv_a_r - 0.996212229189942*v_B2lv_b_i - 0.249053057297486*v_B2lv_b_r + 0.598820527961361*v_B3_a_i + 1.10755301189314*v_B3_a_r - 0.171091579417532*v_B3_b_i - 0.316443717683753*v_B3_b_r - 0.171091579417532*v_B3_c_i - 0.316443717683753*v_B3_c_r
        struct[0].g[41,0] = 1.10755301189314*v_B1_a_i - 0.598820527961361*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417532*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417532*v_B1_c_r - 2.23667465123725*v_B2_a_i + 1.28353302446119*v_B2_a_r + 0.643671749092996*v_B2_b_i - 0.385473430243205*v_B2_b_r + 0.643671749092997*v_B2_c_i - 0.385473430243205*v_B2_c_r + 0.249053057297486*v_B2lv_a_i - 0.996212229189942*v_B2lv_a_r - 0.249053057297486*v_B2lv_b_i + 0.996212229189942*v_B2lv_b_r + 1.10755301189314*v_B3_a_i - 0.598820527961361*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417532*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417532*v_B3_c_r
        struct[0].g[42,0] = -0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r + 0.59882052796136*v_B1_b_i + 1.10755301189314*v_B1_b_r - 0.171091579417531*v_B1_c_i - 0.316443717683753*v_B1_c_r + 0.385473430243205*v_B2_a_i + 0.643671749092996*v_B2_a_r - 1.28353302446119*v_B2_b_i - 2.23667465123725*v_B2_b_r + 0.385473430243204*v_B2_c_i + 0.643671749092997*v_B2_c_r + 0.996212229189942*v_B2lv_b_i + 0.249053057297486*v_B2lv_b_r - 0.996212229189942*v_B2lv_c_i - 0.249053057297486*v_B2lv_c_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r + 0.59882052796136*v_B3_b_i + 1.10755301189314*v_B3_b_r - 0.171091579417531*v_B3_c_i - 0.316443717683753*v_B3_c_r
        struct[0].g[43,0] = -0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r + 1.10755301189314*v_B1_b_i - 0.59882052796136*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417531*v_B1_c_r + 0.643671749092996*v_B2_a_i - 0.385473430243205*v_B2_a_r - 2.23667465123725*v_B2_b_i + 1.28353302446119*v_B2_b_r + 0.643671749092997*v_B2_c_i - 0.385473430243204*v_B2_c_r + 0.249053057297486*v_B2lv_b_i - 0.996212229189942*v_B2lv_b_r - 0.249053057297486*v_B2lv_c_i + 0.996212229189942*v_B2lv_c_r - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r + 1.10755301189314*v_B3_b_i - 0.59882052796136*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417531*v_B3_c_r
        struct[0].g[44,0] = -0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r - 0.171091579417531*v_B1_b_i - 0.316443717683753*v_B1_b_r + 0.59882052796136*v_B1_c_i + 1.10755301189314*v_B1_c_r + 0.385473430243205*v_B2_a_i + 0.643671749092997*v_B2_a_r + 0.385473430243204*v_B2_b_i + 0.643671749092997*v_B2_b_r - 1.28353302446119*v_B2_c_i - 2.23667465123725*v_B2_c_r - 0.996212229189942*v_B2lv_a_i - 0.249053057297486*v_B2lv_a_r + 0.996212229189942*v_B2lv_c_i + 0.249053057297486*v_B2lv_c_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r - 0.171091579417531*v_B3_b_i - 0.316443717683753*v_B3_b_r + 0.59882052796136*v_B3_c_i + 1.10755301189314*v_B3_c_r
        struct[0].g[45,0] = -0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417531*v_B1_b_r + 1.10755301189314*v_B1_c_i - 0.59882052796136*v_B1_c_r + 0.643671749092997*v_B2_a_i - 0.385473430243205*v_B2_a_r + 0.643671749092997*v_B2_b_i - 0.385473430243204*v_B2_b_r - 2.23667465123725*v_B2_c_i + 1.28353302446119*v_B2_c_r - 0.249053057297486*v_B2lv_a_i + 0.996212229189942*v_B2lv_a_r + 0.249053057297486*v_B2lv_c_i - 0.996212229189942*v_B2lv_c_r - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417531*v_B3_b_r + 1.10755301189314*v_B3_c_i - 0.59882052796136*v_B3_c_r
        struct[0].g[64,0] = 0.598820527961361*v_B5_a_i + 1.10755301189314*v_B5_a_r - 0.171091579417532*v_B5_b_i - 0.316443717683753*v_B5_b_r - 0.171091579417532*v_B5_c_i - 0.316443717683753*v_B5_c_r - 1.28353302446119*v_B6_a_i - 2.23667465123725*v_B6_a_r + 0.385473430243205*v_B6_b_i + 0.643671749092996*v_B6_b_r + 0.385473430243205*v_B6_c_i + 0.643671749092997*v_B6_c_r + 0.996212229189942*v_B6lv_a_i + 0.249053057297486*v_B6lv_a_r - 0.996212229189942*v_B6lv_b_i - 0.249053057297486*v_B6lv_b_r + 0.598820527961361*v_B7_a_i + 1.10755301189314*v_B7_a_r - 0.171091579417532*v_B7_b_i - 0.316443717683753*v_B7_b_r - 0.171091579417532*v_B7_c_i - 0.316443717683753*v_B7_c_r
        struct[0].g[65,0] = 1.10755301189314*v_B5_a_i - 0.598820527961361*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417532*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417532*v_B5_c_r - 2.23667465123725*v_B6_a_i + 1.28353302446119*v_B6_a_r + 0.643671749092996*v_B6_b_i - 0.385473430243205*v_B6_b_r + 0.643671749092997*v_B6_c_i - 0.385473430243205*v_B6_c_r + 0.249053057297486*v_B6lv_a_i - 0.996212229189942*v_B6lv_a_r - 0.249053057297486*v_B6lv_b_i + 0.996212229189942*v_B6lv_b_r + 1.10755301189314*v_B7_a_i - 0.598820527961361*v_B7_a_r - 0.316443717683753*v_B7_b_i + 0.171091579417532*v_B7_b_r - 0.316443717683753*v_B7_c_i + 0.171091579417532*v_B7_c_r
        struct[0].g[66,0] = -0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r + 0.59882052796136*v_B5_b_i + 1.10755301189314*v_B5_b_r - 0.171091579417531*v_B5_c_i - 0.316443717683753*v_B5_c_r + 0.385473430243205*v_B6_a_i + 0.643671749092996*v_B6_a_r - 1.28353302446119*v_B6_b_i - 2.23667465123725*v_B6_b_r + 0.385473430243204*v_B6_c_i + 0.643671749092997*v_B6_c_r + 0.996212229189942*v_B6lv_b_i + 0.249053057297486*v_B6lv_b_r - 0.996212229189942*v_B6lv_c_i - 0.249053057297486*v_B6lv_c_r - 0.171091579417532*v_B7_a_i - 0.316443717683753*v_B7_a_r + 0.59882052796136*v_B7_b_i + 1.10755301189314*v_B7_b_r - 0.171091579417531*v_B7_c_i - 0.316443717683753*v_B7_c_r
        struct[0].g[67,0] = -0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r + 1.10755301189314*v_B5_b_i - 0.59882052796136*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417531*v_B5_c_r + 0.643671749092996*v_B6_a_i - 0.385473430243205*v_B6_a_r - 2.23667465123725*v_B6_b_i + 1.28353302446119*v_B6_b_r + 0.643671749092997*v_B6_c_i - 0.385473430243204*v_B6_c_r + 0.249053057297486*v_B6lv_b_i - 0.996212229189942*v_B6lv_b_r - 0.249053057297486*v_B6lv_c_i + 0.996212229189942*v_B6lv_c_r - 0.316443717683753*v_B7_a_i + 0.171091579417532*v_B7_a_r + 1.10755301189314*v_B7_b_i - 0.59882052796136*v_B7_b_r - 0.316443717683753*v_B7_c_i + 0.171091579417531*v_B7_c_r
        struct[0].g[68,0] = -0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r - 0.171091579417531*v_B5_b_i - 0.316443717683753*v_B5_b_r + 0.59882052796136*v_B5_c_i + 1.10755301189314*v_B5_c_r + 0.385473430243205*v_B6_a_i + 0.643671749092997*v_B6_a_r + 0.385473430243204*v_B6_b_i + 0.643671749092997*v_B6_b_r - 1.28353302446119*v_B6_c_i - 2.23667465123725*v_B6_c_r - 0.996212229189942*v_B6lv_a_i - 0.249053057297486*v_B6lv_a_r + 0.996212229189942*v_B6lv_c_i + 0.249053057297486*v_B6lv_c_r - 0.171091579417532*v_B7_a_i - 0.316443717683753*v_B7_a_r - 0.171091579417531*v_B7_b_i - 0.316443717683753*v_B7_b_r + 0.59882052796136*v_B7_c_i + 1.10755301189314*v_B7_c_r
        struct[0].g[69,0] = -0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417531*v_B5_b_r + 1.10755301189314*v_B5_c_i - 0.59882052796136*v_B5_c_r + 0.643671749092997*v_B6_a_i - 0.385473430243205*v_B6_a_r + 0.643671749092997*v_B6_b_i - 0.385473430243204*v_B6_b_r - 2.23667465123725*v_B6_c_i + 1.28353302446119*v_B6_c_r - 0.249053057297486*v_B6lv_a_i + 0.996212229189942*v_B6lv_a_r + 0.249053057297486*v_B6lv_c_i - 0.996212229189942*v_B6lv_c_r - 0.316443717683753*v_B7_a_i + 0.171091579417532*v_B7_a_r - 0.316443717683753*v_B7_b_i + 0.171091579417531*v_B7_b_r + 1.10755301189314*v_B7_c_i - 0.59882052796136*v_B7_c_r
        struct[0].g[100,0] = -i_l_B1_B2_a_r + 0.598820527961361*v_B1_a_i + 1.10755301189314*v_B1_a_r - 0.171091579417532*v_B1_b_i - 0.316443717683753*v_B1_b_r - 0.171091579417532*v_B1_c_i - 0.316443717683753*v_B1_c_r - 0.598820527961361*v_B2_a_i - 1.10755301189314*v_B2_a_r + 0.171091579417532*v_B2_b_i + 0.316443717683753*v_B2_b_r + 0.171091579417532*v_B2_c_i + 0.316443717683753*v_B2_c_r
        struct[0].g[101,0] = -i_l_B1_B2_a_i + 1.10755301189314*v_B1_a_i - 0.598820527961361*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417532*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417532*v_B1_c_r - 1.10755301189314*v_B2_a_i + 0.598820527961361*v_B2_a_r + 0.316443717683753*v_B2_b_i - 0.171091579417532*v_B2_b_r + 0.316443717683753*v_B2_c_i - 0.171091579417532*v_B2_c_r
        struct[0].g[102,0] = -i_l_B1_B2_b_r - 0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r + 0.59882052796136*v_B1_b_i + 1.10755301189314*v_B1_b_r - 0.171091579417531*v_B1_c_i - 0.316443717683753*v_B1_c_r + 0.171091579417532*v_B2_a_i + 0.316443717683753*v_B2_a_r - 0.59882052796136*v_B2_b_i - 1.10755301189314*v_B2_b_r + 0.171091579417531*v_B2_c_i + 0.316443717683753*v_B2_c_r
        struct[0].g[103,0] = -i_l_B1_B2_b_i - 0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r + 1.10755301189314*v_B1_b_i - 0.59882052796136*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417531*v_B1_c_r + 0.316443717683753*v_B2_a_i - 0.171091579417532*v_B2_a_r - 1.10755301189314*v_B2_b_i + 0.59882052796136*v_B2_b_r + 0.316443717683753*v_B2_c_i - 0.171091579417531*v_B2_c_r
        struct[0].g[104,0] = -i_l_B1_B2_c_r - 0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r - 0.171091579417531*v_B1_b_i - 0.316443717683753*v_B1_b_r + 0.59882052796136*v_B1_c_i + 1.10755301189314*v_B1_c_r + 0.171091579417532*v_B2_a_i + 0.316443717683753*v_B2_a_r + 0.171091579417531*v_B2_b_i + 0.316443717683753*v_B2_b_r - 0.59882052796136*v_B2_c_i - 1.10755301189314*v_B2_c_r
        struct[0].g[105,0] = -i_l_B1_B2_c_i - 0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417531*v_B1_b_r + 1.10755301189314*v_B1_c_i - 0.59882052796136*v_B1_c_r + 0.316443717683753*v_B2_a_i - 0.171091579417532*v_B2_a_r + 0.316443717683753*v_B2_b_i - 0.171091579417531*v_B2_b_r - 1.10755301189314*v_B2_c_i + 0.59882052796136*v_B2_c_r
        struct[0].g[124,0] = -i_l_B6_B7_a_r + 0.598820527961361*v_B6_a_i + 1.10755301189314*v_B6_a_r - 0.171091579417532*v_B6_b_i - 0.316443717683753*v_B6_b_r - 0.171091579417532*v_B6_c_i - 0.316443717683753*v_B6_c_r - 0.598820527961361*v_B7_a_i - 1.10755301189314*v_B7_a_r + 0.171091579417532*v_B7_b_i + 0.316443717683753*v_B7_b_r + 0.171091579417532*v_B7_c_i + 0.316443717683753*v_B7_c_r
        struct[0].g[125,0] = -i_l_B6_B7_a_i + 1.10755301189314*v_B6_a_i - 0.598820527961361*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417532*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417532*v_B6_c_r - 1.10755301189314*v_B7_a_i + 0.598820527961361*v_B7_a_r + 0.316443717683753*v_B7_b_i - 0.171091579417532*v_B7_b_r + 0.316443717683753*v_B7_c_i - 0.171091579417532*v_B7_c_r
        struct[0].g[126,0] = -i_l_B6_B7_b_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r + 0.59882052796136*v_B6_b_i + 1.10755301189314*v_B6_b_r - 0.171091579417531*v_B6_c_i - 0.316443717683753*v_B6_c_r + 0.171091579417532*v_B7_a_i + 0.316443717683753*v_B7_a_r - 0.59882052796136*v_B7_b_i - 1.10755301189314*v_B7_b_r + 0.171091579417531*v_B7_c_i + 0.316443717683753*v_B7_c_r
        struct[0].g[127,0] = -i_l_B6_B7_b_i - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r + 1.10755301189314*v_B6_b_i - 0.59882052796136*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417531*v_B6_c_r + 0.316443717683753*v_B7_a_i - 0.171091579417532*v_B7_a_r - 1.10755301189314*v_B7_b_i + 0.59882052796136*v_B7_b_r + 0.316443717683753*v_B7_c_i - 0.171091579417531*v_B7_c_r
        struct[0].g[128,0] = -i_l_B6_B7_c_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r - 0.171091579417531*v_B6_b_i - 0.316443717683753*v_B6_b_r + 0.59882052796136*v_B6_c_i + 1.10755301189314*v_B6_c_r + 0.171091579417532*v_B7_a_i + 0.316443717683753*v_B7_a_r + 0.171091579417531*v_B7_b_i + 0.316443717683753*v_B7_b_r - 0.59882052796136*v_B7_c_i - 1.10755301189314*v_B7_c_r
        struct[0].g[129,0] = -i_l_B6_B7_c_i - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417531*v_B6_b_r + 1.10755301189314*v_B6_c_i - 0.59882052796136*v_B6_c_r + 0.316443717683753*v_B7_a_i - 0.171091579417532*v_B7_a_r + 0.316443717683753*v_B7_b_i - 0.171091579417531*v_B7_b_r - 1.10755301189314*v_B7_c_i + 0.59882052796136*v_B7_c_r
        struct[0].g[130,0] = i_load_B2lv_a_i*v_B2lv_a_i - i_load_B2lv_a_i*v_B2lv_n_i + i_load_B2lv_a_r*v_B2lv_a_r - i_load_B2lv_a_r*v_B2lv_n_r - p_B2lv_a
        struct[0].g[131,0] = i_load_B2lv_b_i*v_B2lv_b_i - i_load_B2lv_b_i*v_B2lv_n_i + i_load_B2lv_b_r*v_B2lv_b_r - i_load_B2lv_b_r*v_B2lv_n_r - p_B2lv_b
        struct[0].g[132,0] = i_load_B2lv_c_i*v_B2lv_c_i - i_load_B2lv_c_i*v_B2lv_n_i + i_load_B2lv_c_r*v_B2lv_c_r - i_load_B2lv_c_r*v_B2lv_n_r - p_B2lv_c
        struct[0].g[133,0] = -i_load_B2lv_a_i*v_B2lv_a_r + i_load_B2lv_a_i*v_B2lv_n_r + i_load_B2lv_a_r*v_B2lv_a_i - i_load_B2lv_a_r*v_B2lv_n_i - q_B2lv_a
        struct[0].g[134,0] = -i_load_B2lv_b_i*v_B2lv_b_r + i_load_B2lv_b_i*v_B2lv_n_r + i_load_B2lv_b_r*v_B2lv_b_i - i_load_B2lv_b_r*v_B2lv_n_i - q_B2lv_b
        struct[0].g[135,0] = -i_load_B2lv_c_i*v_B2lv_c_r + i_load_B2lv_c_i*v_B2lv_n_r + i_load_B2lv_c_r*v_B2lv_c_i - i_load_B2lv_c_r*v_B2lv_n_i - q_B2lv_c
        struct[0].g[138,0] = i_load_B3lv_a_i*v_B3lv_a_i - i_load_B3lv_a_i*v_B3lv_n_i + i_load_B3lv_a_r*v_B3lv_a_r - i_load_B3lv_a_r*v_B3lv_n_r - p_B3lv_a
        struct[0].g[139,0] = i_load_B3lv_b_i*v_B3lv_b_i - i_load_B3lv_b_i*v_B3lv_n_i + i_load_B3lv_b_r*v_B3lv_b_r - i_load_B3lv_b_r*v_B3lv_n_r - p_B3lv_b
        struct[0].g[140,0] = i_load_B3lv_c_i*v_B3lv_c_i - i_load_B3lv_c_i*v_B3lv_n_i + i_load_B3lv_c_r*v_B3lv_c_r - i_load_B3lv_c_r*v_B3lv_n_r - p_B3lv_c
        struct[0].g[141,0] = -i_load_B3lv_a_i*v_B3lv_a_r + i_load_B3lv_a_i*v_B3lv_n_r + i_load_B3lv_a_r*v_B3lv_a_i - i_load_B3lv_a_r*v_B3lv_n_i - q_B3lv_a
        struct[0].g[142,0] = -i_load_B3lv_b_i*v_B3lv_b_r + i_load_B3lv_b_i*v_B3lv_n_r + i_load_B3lv_b_r*v_B3lv_b_i - i_load_B3lv_b_r*v_B3lv_n_i - q_B3lv_b
        struct[0].g[143,0] = -i_load_B3lv_c_i*v_B3lv_c_r + i_load_B3lv_c_i*v_B3lv_n_r + i_load_B3lv_c_r*v_B3lv_c_i - i_load_B3lv_c_r*v_B3lv_n_i - q_B3lv_c
        struct[0].g[146,0] = i_load_B4lv_a_i*v_B4lv_a_i - i_load_B4lv_a_i*v_B4lv_n_i + i_load_B4lv_a_r*v_B4lv_a_r - i_load_B4lv_a_r*v_B4lv_n_r - p_B4lv_a
        struct[0].g[147,0] = i_load_B4lv_b_i*v_B4lv_b_i - i_load_B4lv_b_i*v_B4lv_n_i + i_load_B4lv_b_r*v_B4lv_b_r - i_load_B4lv_b_r*v_B4lv_n_r - p_B4lv_b
        struct[0].g[148,0] = i_load_B4lv_c_i*v_B4lv_c_i - i_load_B4lv_c_i*v_B4lv_n_i + i_load_B4lv_c_r*v_B4lv_c_r - i_load_B4lv_c_r*v_B4lv_n_r - p_B4lv_c
        struct[0].g[149,0] = -i_load_B4lv_a_i*v_B4lv_a_r + i_load_B4lv_a_i*v_B4lv_n_r + i_load_B4lv_a_r*v_B4lv_a_i - i_load_B4lv_a_r*v_B4lv_n_i - q_B4lv_a
        struct[0].g[150,0] = -i_load_B4lv_b_i*v_B4lv_b_r + i_load_B4lv_b_i*v_B4lv_n_r + i_load_B4lv_b_r*v_B4lv_b_i - i_load_B4lv_b_r*v_B4lv_n_i - q_B4lv_b
        struct[0].g[151,0] = -i_load_B4lv_c_i*v_B4lv_c_r + i_load_B4lv_c_i*v_B4lv_n_r + i_load_B4lv_c_r*v_B4lv_c_i - i_load_B4lv_c_r*v_B4lv_n_i - q_B4lv_c
        struct[0].g[154,0] = i_load_B5lv_a_i*v_B5lv_a_i - i_load_B5lv_a_i*v_B5lv_n_i + i_load_B5lv_a_r*v_B5lv_a_r - i_load_B5lv_a_r*v_B5lv_n_r - p_B5lv_a
        struct[0].g[155,0] = i_load_B5lv_b_i*v_B5lv_b_i - i_load_B5lv_b_i*v_B5lv_n_i + i_load_B5lv_b_r*v_B5lv_b_r - i_load_B5lv_b_r*v_B5lv_n_r - p_B5lv_b
        struct[0].g[156,0] = i_load_B5lv_c_i*v_B5lv_c_i - i_load_B5lv_c_i*v_B5lv_n_i + i_load_B5lv_c_r*v_B5lv_c_r - i_load_B5lv_c_r*v_B5lv_n_r - p_B5lv_c
        struct[0].g[157,0] = -i_load_B5lv_a_i*v_B5lv_a_r + i_load_B5lv_a_i*v_B5lv_n_r + i_load_B5lv_a_r*v_B5lv_a_i - i_load_B5lv_a_r*v_B5lv_n_i - q_B5lv_a
        struct[0].g[158,0] = -i_load_B5lv_b_i*v_B5lv_b_r + i_load_B5lv_b_i*v_B5lv_n_r + i_load_B5lv_b_r*v_B5lv_b_i - i_load_B5lv_b_r*v_B5lv_n_i - q_B5lv_b
        struct[0].g[159,0] = -i_load_B5lv_c_i*v_B5lv_c_r + i_load_B5lv_c_i*v_B5lv_n_r + i_load_B5lv_c_r*v_B5lv_c_i - i_load_B5lv_c_r*v_B5lv_n_i - q_B5lv_c
        struct[0].g[162,0] = i_load_B6lv_a_i*v_B6lv_a_i - i_load_B6lv_a_i*v_B6lv_n_i + i_load_B6lv_a_r*v_B6lv_a_r - i_load_B6lv_a_r*v_B6lv_n_r - p_B6lv_a
        struct[0].g[163,0] = i_load_B6lv_b_i*v_B6lv_b_i - i_load_B6lv_b_i*v_B6lv_n_i + i_load_B6lv_b_r*v_B6lv_b_r - i_load_B6lv_b_r*v_B6lv_n_r - p_B6lv_b
        struct[0].g[164,0] = i_load_B6lv_c_i*v_B6lv_c_i - i_load_B6lv_c_i*v_B6lv_n_i + i_load_B6lv_c_r*v_B6lv_c_r - i_load_B6lv_c_r*v_B6lv_n_r - p_B6lv_c
        struct[0].g[165,0] = -i_load_B6lv_a_i*v_B6lv_a_r + i_load_B6lv_a_i*v_B6lv_n_r + i_load_B6lv_a_r*v_B6lv_a_i - i_load_B6lv_a_r*v_B6lv_n_i - q_B6lv_a
        struct[0].g[166,0] = -i_load_B6lv_b_i*v_B6lv_b_r + i_load_B6lv_b_i*v_B6lv_n_r + i_load_B6lv_b_r*v_B6lv_b_i - i_load_B6lv_b_r*v_B6lv_n_i - q_B6lv_b
        struct[0].g[167,0] = -i_load_B6lv_c_i*v_B6lv_c_r + i_load_B6lv_c_i*v_B6lv_n_r + i_load_B6lv_c_r*v_B6lv_c_i - i_load_B6lv_c_r*v_B6lv_n_i - q_B6lv_c
        struct[0].g[171,0] = -R_1213*i_l_1213 + v_12 - v_13
        struct[0].g[172,0] = -R_1314*i_l_1314 + v_13 - v_14
        struct[0].g[175,0] = -R_2223*i_l_2223 + v_22 - v_23
        struct[0].g[176,0] = -R_2324*i_l_2324 + v_23 - v_24
        struct[0].g[179,0] = -R_3233*i_l_3233 + v_32 - v_33
        struct[0].g[180,0] = -R_3334*i_l_3334 + v_33 - v_34
        struct[0].g[183,0] = -R_4243*i_l_4243 + v_42 - v_43
        struct[0].g[184,0] = -R_4344*i_l_4344 + v_43 - v_44
        struct[0].g[187,0] = -R_5253*i_l_5253 + v_52 - v_53
        struct[0].g[188,0] = -R_5354*i_l_5354 + v_53 - v_54
        struct[0].g[219,0] = i_11*v_11 - p_11
        struct[0].g[220,0] = i_12*v_12 - p_12
        struct[0].g[221,0] = i_14*v_14 - p_14
        struct[0].g[222,0] = i_15*v_15 - p_15
        struct[0].g[223,0] = i_21*v_21 - p_21
        struct[0].g[224,0] = i_22*v_22 - p_22
        struct[0].g[225,0] = i_24*v_24 - p_24
        struct[0].g[226,0] = i_25*v_25 - p_25
        struct[0].g[227,0] = i_31*v_31 - p_31
        struct[0].g[228,0] = i_32*v_32 - p_32
        struct[0].g[229,0] = i_34*v_34 - p_34
        struct[0].g[230,0] = i_35*v_35 - p_35
        struct[0].g[231,0] = i_41*v_41 - p_41
        struct[0].g[232,0] = i_42*v_42 - p_42
        struct[0].g[233,0] = i_44*v_44 - p_44
        struct[0].g[234,0] = i_45*v_45 - p_45
        struct[0].g[235,0] = i_51*v_51 - p_51
        struct[0].g[236,0] = i_52*v_52 - p_52
        struct[0].g[237,0] = i_54*v_54 - p_54
        struct[0].g[238,0] = i_55*v_55 - p_55
        struct[0].g[239,0] = i_13*v_13/3 + p_B2lv_a + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[240,0] = i_13*v_13/3 + p_B2lv_b + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[241,0] = i_13*v_13/3 + p_B2lv_c + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[242,0] = i_23*v_23/3 + p_B3lv_a + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[243,0] = i_23*v_23/3 + p_B3lv_b + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[244,0] = i_23*v_23/3 + p_B3lv_c + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[245,0] = i_33*v_33/3 + p_B4lv_a + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[246,0] = i_33*v_33/3 + p_B4lv_b + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[247,0] = i_33*v_33/3 + p_B4lv_c + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[248,0] = i_43*v_43/3 + p_B5lv_a + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[249,0] = i_43*v_43/3 + p_B5lv_b + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[250,0] = i_43*v_43/3 + p_B5lv_c + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[251,0] = i_53*v_53/3 + p_B6lv_a + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
        struct[0].g[252,0] = i_53*v_53/3 + p_B6lv_b + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
        struct[0].g[253,0] = i_53*v_53/3 + p_B6lv_c + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
    
    # Outputs:
    if mode == 3:

        struct[0].h[0,0] = i_13*v_13
        struct[0].h[1,0] = v_13
        struct[0].h[2,0] = i_23*v_23
        struct[0].h[3,0] = v_23
        struct[0].h[4,0] = i_33*v_33
        struct[0].h[5,0] = v_33
        struct[0].h[6,0] = i_43*v_43
        struct[0].h[7,0] = v_43
        struct[0].h[8,0] = i_53*v_53
        struct[0].h[9,0] = v_53
    

    if mode == 10:

        struct[0].Fx_ini[0,0] = (-K_r*i_13 - 1)/T_v
        struct[0].Fx_ini[1,1] = (-K_r*i_23 - 1)/T_v
        struct[0].Fx_ini[2,2] = (-K_r*i_33 - 1)/T_v
        struct[0].Fx_ini[3,3] = (-K_r*i_43 - 1)/T_v
        struct[0].Fx_ini[4,4] = (-K_r*i_53 - 1)/T_v

    if mode == 11:

        struct[0].Fy_ini[0,196] = -K_r*v_13/T_v 
        struct[0].Fy_ini[1,201] = -K_r*v_23/T_v 
        struct[0].Fy_ini[2,206] = -K_r*v_33/T_v 
        struct[0].Fy_ini[3,211] = -K_r*v_43/T_v 
        struct[0].Fy_ini[4,216] = -K_r*v_53/T_v 

        struct[0].Gx_ini[171,0] = -1
        struct[0].Gx_ini[172,0] = 1
        struct[0].Gx_ini[175,1] = -1
        struct[0].Gx_ini[176,1] = 1
        struct[0].Gx_ini[179,2] = -1
        struct[0].Gx_ini[180,2] = 1
        struct[0].Gx_ini[183,3] = -1
        struct[0].Gx_ini[184,3] = 1
        struct[0].Gx_ini[187,4] = -1
        struct[0].Gx_ini[188,4] = 1
        struct[0].Gx_ini[239,0] = i_13/3
        struct[0].Gx_ini[240,0] = i_13/3
        struct[0].Gx_ini[241,0] = i_13/3
        struct[0].Gx_ini[242,1] = i_23/3
        struct[0].Gx_ini[243,1] = i_23/3
        struct[0].Gx_ini[244,1] = i_23/3
        struct[0].Gx_ini[245,2] = i_33/3
        struct[0].Gx_ini[246,2] = i_33/3
        struct[0].Gx_ini[247,2] = i_33/3
        struct[0].Gx_ini[248,3] = i_43/3
        struct[0].Gx_ini[249,3] = i_43/3
        struct[0].Gx_ini[250,3] = i_43/3
        struct[0].Gx_ini[251,4] = i_53/3
        struct[0].Gx_ini[252,4] = i_53/3
        struct[0].Gx_ini[253,4] = i_53/3

        struct[0].Gy_ini[130,0] = i_load_B2lv_a_r
        struct[0].Gy_ini[130,1] = i_load_B2lv_a_i
        struct[0].Gy_ini[130,6] = -i_load_B2lv_a_r
        struct[0].Gy_ini[130,7] = -i_load_B2lv_a_i
        struct[0].Gy_ini[130,130] = v_B2lv_a_r - v_B2lv_n_r
        struct[0].Gy_ini[130,131] = v_B2lv_a_i - v_B2lv_n_i
        struct[0].Gy_ini[131,2] = i_load_B2lv_b_r
        struct[0].Gy_ini[131,3] = i_load_B2lv_b_i
        struct[0].Gy_ini[131,6] = -i_load_B2lv_b_r
        struct[0].Gy_ini[131,7] = -i_load_B2lv_b_i
        struct[0].Gy_ini[131,132] = v_B2lv_b_r - v_B2lv_n_r
        struct[0].Gy_ini[131,133] = v_B2lv_b_i - v_B2lv_n_i
        struct[0].Gy_ini[132,4] = i_load_B2lv_c_r
        struct[0].Gy_ini[132,5] = i_load_B2lv_c_i
        struct[0].Gy_ini[132,6] = -i_load_B2lv_c_r
        struct[0].Gy_ini[132,7] = -i_load_B2lv_c_i
        struct[0].Gy_ini[132,134] = v_B2lv_c_r - v_B2lv_n_r
        struct[0].Gy_ini[132,135] = v_B2lv_c_i - v_B2lv_n_i
        struct[0].Gy_ini[133,0] = -i_load_B2lv_a_i
        struct[0].Gy_ini[133,1] = i_load_B2lv_a_r
        struct[0].Gy_ini[133,6] = i_load_B2lv_a_i
        struct[0].Gy_ini[133,7] = -i_load_B2lv_a_r
        struct[0].Gy_ini[133,130] = v_B2lv_a_i - v_B2lv_n_i
        struct[0].Gy_ini[133,131] = -v_B2lv_a_r + v_B2lv_n_r
        struct[0].Gy_ini[134,2] = -i_load_B2lv_b_i
        struct[0].Gy_ini[134,3] = i_load_B2lv_b_r
        struct[0].Gy_ini[134,6] = i_load_B2lv_b_i
        struct[0].Gy_ini[134,7] = -i_load_B2lv_b_r
        struct[0].Gy_ini[134,132] = v_B2lv_b_i - v_B2lv_n_i
        struct[0].Gy_ini[134,133] = -v_B2lv_b_r + v_B2lv_n_r
        struct[0].Gy_ini[135,4] = -i_load_B2lv_c_i
        struct[0].Gy_ini[135,5] = i_load_B2lv_c_r
        struct[0].Gy_ini[135,6] = i_load_B2lv_c_i
        struct[0].Gy_ini[135,7] = -i_load_B2lv_c_r
        struct[0].Gy_ini[135,134] = v_B2lv_c_i - v_B2lv_n_i
        struct[0].Gy_ini[135,135] = -v_B2lv_c_r + v_B2lv_n_r
        struct[0].Gy_ini[138,8] = i_load_B3lv_a_r
        struct[0].Gy_ini[138,9] = i_load_B3lv_a_i
        struct[0].Gy_ini[138,14] = -i_load_B3lv_a_r
        struct[0].Gy_ini[138,15] = -i_load_B3lv_a_i
        struct[0].Gy_ini[138,138] = v_B3lv_a_r - v_B3lv_n_r
        struct[0].Gy_ini[138,139] = v_B3lv_a_i - v_B3lv_n_i
        struct[0].Gy_ini[139,10] = i_load_B3lv_b_r
        struct[0].Gy_ini[139,11] = i_load_B3lv_b_i
        struct[0].Gy_ini[139,14] = -i_load_B3lv_b_r
        struct[0].Gy_ini[139,15] = -i_load_B3lv_b_i
        struct[0].Gy_ini[139,140] = v_B3lv_b_r - v_B3lv_n_r
        struct[0].Gy_ini[139,141] = v_B3lv_b_i - v_B3lv_n_i
        struct[0].Gy_ini[140,12] = i_load_B3lv_c_r
        struct[0].Gy_ini[140,13] = i_load_B3lv_c_i
        struct[0].Gy_ini[140,14] = -i_load_B3lv_c_r
        struct[0].Gy_ini[140,15] = -i_load_B3lv_c_i
        struct[0].Gy_ini[140,142] = v_B3lv_c_r - v_B3lv_n_r
        struct[0].Gy_ini[140,143] = v_B3lv_c_i - v_B3lv_n_i
        struct[0].Gy_ini[141,8] = -i_load_B3lv_a_i
        struct[0].Gy_ini[141,9] = i_load_B3lv_a_r
        struct[0].Gy_ini[141,14] = i_load_B3lv_a_i
        struct[0].Gy_ini[141,15] = -i_load_B3lv_a_r
        struct[0].Gy_ini[141,138] = v_B3lv_a_i - v_B3lv_n_i
        struct[0].Gy_ini[141,139] = -v_B3lv_a_r + v_B3lv_n_r
        struct[0].Gy_ini[142,10] = -i_load_B3lv_b_i
        struct[0].Gy_ini[142,11] = i_load_B3lv_b_r
        struct[0].Gy_ini[142,14] = i_load_B3lv_b_i
        struct[0].Gy_ini[142,15] = -i_load_B3lv_b_r
        struct[0].Gy_ini[142,140] = v_B3lv_b_i - v_B3lv_n_i
        struct[0].Gy_ini[142,141] = -v_B3lv_b_r + v_B3lv_n_r
        struct[0].Gy_ini[143,12] = -i_load_B3lv_c_i
        struct[0].Gy_ini[143,13] = i_load_B3lv_c_r
        struct[0].Gy_ini[143,14] = i_load_B3lv_c_i
        struct[0].Gy_ini[143,15] = -i_load_B3lv_c_r
        struct[0].Gy_ini[143,142] = v_B3lv_c_i - v_B3lv_n_i
        struct[0].Gy_ini[143,143] = -v_B3lv_c_r + v_B3lv_n_r
        struct[0].Gy_ini[146,16] = i_load_B4lv_a_r
        struct[0].Gy_ini[146,17] = i_load_B4lv_a_i
        struct[0].Gy_ini[146,22] = -i_load_B4lv_a_r
        struct[0].Gy_ini[146,23] = -i_load_B4lv_a_i
        struct[0].Gy_ini[146,146] = v_B4lv_a_r - v_B4lv_n_r
        struct[0].Gy_ini[146,147] = v_B4lv_a_i - v_B4lv_n_i
        struct[0].Gy_ini[147,18] = i_load_B4lv_b_r
        struct[0].Gy_ini[147,19] = i_load_B4lv_b_i
        struct[0].Gy_ini[147,22] = -i_load_B4lv_b_r
        struct[0].Gy_ini[147,23] = -i_load_B4lv_b_i
        struct[0].Gy_ini[147,148] = v_B4lv_b_r - v_B4lv_n_r
        struct[0].Gy_ini[147,149] = v_B4lv_b_i - v_B4lv_n_i
        struct[0].Gy_ini[148,20] = i_load_B4lv_c_r
        struct[0].Gy_ini[148,21] = i_load_B4lv_c_i
        struct[0].Gy_ini[148,22] = -i_load_B4lv_c_r
        struct[0].Gy_ini[148,23] = -i_load_B4lv_c_i
        struct[0].Gy_ini[148,150] = v_B4lv_c_r - v_B4lv_n_r
        struct[0].Gy_ini[148,151] = v_B4lv_c_i - v_B4lv_n_i
        struct[0].Gy_ini[149,16] = -i_load_B4lv_a_i
        struct[0].Gy_ini[149,17] = i_load_B4lv_a_r
        struct[0].Gy_ini[149,22] = i_load_B4lv_a_i
        struct[0].Gy_ini[149,23] = -i_load_B4lv_a_r
        struct[0].Gy_ini[149,146] = v_B4lv_a_i - v_B4lv_n_i
        struct[0].Gy_ini[149,147] = -v_B4lv_a_r + v_B4lv_n_r
        struct[0].Gy_ini[150,18] = -i_load_B4lv_b_i
        struct[0].Gy_ini[150,19] = i_load_B4lv_b_r
        struct[0].Gy_ini[150,22] = i_load_B4lv_b_i
        struct[0].Gy_ini[150,23] = -i_load_B4lv_b_r
        struct[0].Gy_ini[150,148] = v_B4lv_b_i - v_B4lv_n_i
        struct[0].Gy_ini[150,149] = -v_B4lv_b_r + v_B4lv_n_r
        struct[0].Gy_ini[151,20] = -i_load_B4lv_c_i
        struct[0].Gy_ini[151,21] = i_load_B4lv_c_r
        struct[0].Gy_ini[151,22] = i_load_B4lv_c_i
        struct[0].Gy_ini[151,23] = -i_load_B4lv_c_r
        struct[0].Gy_ini[151,150] = v_B4lv_c_i - v_B4lv_n_i
        struct[0].Gy_ini[151,151] = -v_B4lv_c_r + v_B4lv_n_r
        struct[0].Gy_ini[154,24] = i_load_B5lv_a_r
        struct[0].Gy_ini[154,25] = i_load_B5lv_a_i
        struct[0].Gy_ini[154,30] = -i_load_B5lv_a_r
        struct[0].Gy_ini[154,31] = -i_load_B5lv_a_i
        struct[0].Gy_ini[154,154] = v_B5lv_a_r - v_B5lv_n_r
        struct[0].Gy_ini[154,155] = v_B5lv_a_i - v_B5lv_n_i
        struct[0].Gy_ini[155,26] = i_load_B5lv_b_r
        struct[0].Gy_ini[155,27] = i_load_B5lv_b_i
        struct[0].Gy_ini[155,30] = -i_load_B5lv_b_r
        struct[0].Gy_ini[155,31] = -i_load_B5lv_b_i
        struct[0].Gy_ini[155,156] = v_B5lv_b_r - v_B5lv_n_r
        struct[0].Gy_ini[155,157] = v_B5lv_b_i - v_B5lv_n_i
        struct[0].Gy_ini[156,28] = i_load_B5lv_c_r
        struct[0].Gy_ini[156,29] = i_load_B5lv_c_i
        struct[0].Gy_ini[156,30] = -i_load_B5lv_c_r
        struct[0].Gy_ini[156,31] = -i_load_B5lv_c_i
        struct[0].Gy_ini[156,158] = v_B5lv_c_r - v_B5lv_n_r
        struct[0].Gy_ini[156,159] = v_B5lv_c_i - v_B5lv_n_i
        struct[0].Gy_ini[157,24] = -i_load_B5lv_a_i
        struct[0].Gy_ini[157,25] = i_load_B5lv_a_r
        struct[0].Gy_ini[157,30] = i_load_B5lv_a_i
        struct[0].Gy_ini[157,31] = -i_load_B5lv_a_r
        struct[0].Gy_ini[157,154] = v_B5lv_a_i - v_B5lv_n_i
        struct[0].Gy_ini[157,155] = -v_B5lv_a_r + v_B5lv_n_r
        struct[0].Gy_ini[158,26] = -i_load_B5lv_b_i
        struct[0].Gy_ini[158,27] = i_load_B5lv_b_r
        struct[0].Gy_ini[158,30] = i_load_B5lv_b_i
        struct[0].Gy_ini[158,31] = -i_load_B5lv_b_r
        struct[0].Gy_ini[158,156] = v_B5lv_b_i - v_B5lv_n_i
        struct[0].Gy_ini[158,157] = -v_B5lv_b_r + v_B5lv_n_r
        struct[0].Gy_ini[159,28] = -i_load_B5lv_c_i
        struct[0].Gy_ini[159,29] = i_load_B5lv_c_r
        struct[0].Gy_ini[159,30] = i_load_B5lv_c_i
        struct[0].Gy_ini[159,31] = -i_load_B5lv_c_r
        struct[0].Gy_ini[159,158] = v_B5lv_c_i - v_B5lv_n_i
        struct[0].Gy_ini[159,159] = -v_B5lv_c_r + v_B5lv_n_r
        struct[0].Gy_ini[162,32] = i_load_B6lv_a_r
        struct[0].Gy_ini[162,33] = i_load_B6lv_a_i
        struct[0].Gy_ini[162,38] = -i_load_B6lv_a_r
        struct[0].Gy_ini[162,39] = -i_load_B6lv_a_i
        struct[0].Gy_ini[162,162] = v_B6lv_a_r - v_B6lv_n_r
        struct[0].Gy_ini[162,163] = v_B6lv_a_i - v_B6lv_n_i
        struct[0].Gy_ini[163,34] = i_load_B6lv_b_r
        struct[0].Gy_ini[163,35] = i_load_B6lv_b_i
        struct[0].Gy_ini[163,38] = -i_load_B6lv_b_r
        struct[0].Gy_ini[163,39] = -i_load_B6lv_b_i
        struct[0].Gy_ini[163,164] = v_B6lv_b_r - v_B6lv_n_r
        struct[0].Gy_ini[163,165] = v_B6lv_b_i - v_B6lv_n_i
        struct[0].Gy_ini[164,36] = i_load_B6lv_c_r
        struct[0].Gy_ini[164,37] = i_load_B6lv_c_i
        struct[0].Gy_ini[164,38] = -i_load_B6lv_c_r
        struct[0].Gy_ini[164,39] = -i_load_B6lv_c_i
        struct[0].Gy_ini[164,166] = v_B6lv_c_r - v_B6lv_n_r
        struct[0].Gy_ini[164,167] = v_B6lv_c_i - v_B6lv_n_i
        struct[0].Gy_ini[165,32] = -i_load_B6lv_a_i
        struct[0].Gy_ini[165,33] = i_load_B6lv_a_r
        struct[0].Gy_ini[165,38] = i_load_B6lv_a_i
        struct[0].Gy_ini[165,39] = -i_load_B6lv_a_r
        struct[0].Gy_ini[165,162] = v_B6lv_a_i - v_B6lv_n_i
        struct[0].Gy_ini[165,163] = -v_B6lv_a_r + v_B6lv_n_r
        struct[0].Gy_ini[166,34] = -i_load_B6lv_b_i
        struct[0].Gy_ini[166,35] = i_load_B6lv_b_r
        struct[0].Gy_ini[166,38] = i_load_B6lv_b_i
        struct[0].Gy_ini[166,39] = -i_load_B6lv_b_r
        struct[0].Gy_ini[166,164] = v_B6lv_b_i - v_B6lv_n_i
        struct[0].Gy_ini[166,165] = -v_B6lv_b_r + v_B6lv_n_r
        struct[0].Gy_ini[167,36] = -i_load_B6lv_c_i
        struct[0].Gy_ini[167,37] = i_load_B6lv_c_r
        struct[0].Gy_ini[167,38] = i_load_B6lv_c_i
        struct[0].Gy_ini[167,39] = -i_load_B6lv_c_r
        struct[0].Gy_ini[167,166] = v_B6lv_c_i - v_B6lv_n_i
        struct[0].Gy_ini[167,167] = -v_B6lv_c_r + v_B6lv_n_r
        struct[0].Gy_ini[170,170] = -R_1112
        struct[0].Gy_ini[171,171] = -R_1213
        struct[0].Gy_ini[172,172] = -R_1314
        struct[0].Gy_ini[173,173] = -R_1415
        struct[0].Gy_ini[174,174] = -R_2122
        struct[0].Gy_ini[175,175] = -R_2223
        struct[0].Gy_ini[176,176] = -R_2324
        struct[0].Gy_ini[177,177] = -R_2425
        struct[0].Gy_ini[178,178] = -R_3132
        struct[0].Gy_ini[179,179] = -R_3233
        struct[0].Gy_ini[180,180] = -R_3334
        struct[0].Gy_ini[181,181] = -R_3435
        struct[0].Gy_ini[182,182] = -R_4142
        struct[0].Gy_ini[183,183] = -R_4243
        struct[0].Gy_ini[184,184] = -R_4344
        struct[0].Gy_ini[185,185] = -R_4445
        struct[0].Gy_ini[186,186] = -R_5152
        struct[0].Gy_ini[187,187] = -R_5253
        struct[0].Gy_ini[188,188] = -R_5354
        struct[0].Gy_ini[189,189] = -R_5455
        struct[0].Gy_ini[190,190] = -R_1521
        struct[0].Gy_ini[191,191] = -R_2531
        struct[0].Gy_ini[192,192] = -R_3541
        struct[0].Gy_ini[193,193] = -R_4551
        struct[0].Gy_ini[219,194] = i_11
        struct[0].Gy_ini[219,219] = v_11
        struct[0].Gy_ini[220,195] = i_12
        struct[0].Gy_ini[220,220] = v_12
        struct[0].Gy_ini[221,197] = i_14
        struct[0].Gy_ini[221,221] = v_14
        struct[0].Gy_ini[222,198] = i_15
        struct[0].Gy_ini[222,222] = v_15
        struct[0].Gy_ini[223,199] = i_21
        struct[0].Gy_ini[223,223] = v_21
        struct[0].Gy_ini[224,200] = i_22
        struct[0].Gy_ini[224,224] = v_22
        struct[0].Gy_ini[225,202] = i_24
        struct[0].Gy_ini[225,225] = v_24
        struct[0].Gy_ini[226,203] = i_25
        struct[0].Gy_ini[226,226] = v_25
        struct[0].Gy_ini[227,204] = i_31
        struct[0].Gy_ini[227,227] = v_31
        struct[0].Gy_ini[228,205] = i_32
        struct[0].Gy_ini[228,228] = v_32
        struct[0].Gy_ini[229,207] = i_34
        struct[0].Gy_ini[229,229] = v_34
        struct[0].Gy_ini[230,208] = i_35
        struct[0].Gy_ini[230,230] = v_35
        struct[0].Gy_ini[231,209] = i_41
        struct[0].Gy_ini[231,231] = v_41
        struct[0].Gy_ini[232,210] = i_42
        struct[0].Gy_ini[232,232] = v_42
        struct[0].Gy_ini[233,212] = i_44
        struct[0].Gy_ini[233,233] = v_44
        struct[0].Gy_ini[234,213] = i_45
        struct[0].Gy_ini[234,234] = v_45
        struct[0].Gy_ini[235,214] = i_51
        struct[0].Gy_ini[235,235] = v_51
        struct[0].Gy_ini[236,215] = i_52
        struct[0].Gy_ini[236,236] = v_52
        struct[0].Gy_ini[237,217] = i_54
        struct[0].Gy_ini[237,237] = v_54
        struct[0].Gy_ini[238,218] = i_55
        struct[0].Gy_ini[238,238] = v_55
        struct[0].Gy_ini[239,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[239,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[239,196] = v_13/3
        struct[0].Gy_ini[240,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[240,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[240,196] = v_13/3
        struct[0].Gy_ini[241,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[241,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[241,196] = v_13/3
        struct[0].Gy_ini[242,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[242,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[242,201] = v_23/3
        struct[0].Gy_ini[243,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[243,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[243,201] = v_23/3
        struct[0].Gy_ini[244,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[244,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[244,201] = v_23/3
        struct[0].Gy_ini[245,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[245,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[245,206] = v_33/3
        struct[0].Gy_ini[246,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[246,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[246,206] = v_33/3
        struct[0].Gy_ini[247,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[247,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[247,206] = v_33/3
        struct[0].Gy_ini[248,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[248,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[248,211] = v_43/3
        struct[0].Gy_ini[249,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[249,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[249,211] = v_43/3
        struct[0].Gy_ini[250,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[250,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[250,211] = v_43/3
        struct[0].Gy_ini[251,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[251,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[251,216] = v_53/3
        struct[0].Gy_ini[252,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[252,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[252,216] = v_53/3
        struct[0].Gy_ini[253,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[253,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[253,216] = v_53/3



@numba.njit(cache=True)
def run(t,struct,mode):

    # Parameters:
    R_1112 = struct[0].R_1112
    R_1213 = struct[0].R_1213
    R_1314 = struct[0].R_1314
    R_1415 = struct[0].R_1415
    R_1521 = struct[0].R_1521
    R_2122 = struct[0].R_2122
    R_2223 = struct[0].R_2223
    R_2324 = struct[0].R_2324
    R_2425 = struct[0].R_2425
    R_2531 = struct[0].R_2531
    R_3132 = struct[0].R_3132
    R_3233 = struct[0].R_3233
    R_3334 = struct[0].R_3334
    R_3435 = struct[0].R_3435
    R_3541 = struct[0].R_3541
    R_4142 = struct[0].R_4142
    R_4243 = struct[0].R_4243
    R_4344 = struct[0].R_4344
    R_4445 = struct[0].R_4445
    R_4551 = struct[0].R_4551
    R_5152 = struct[0].R_5152
    R_5253 = struct[0].R_5253
    R_5354 = struct[0].R_5354
    R_5455 = struct[0].R_5455
    p_11 = struct[0].p_11
    p_12 = struct[0].p_12
    p_14 = struct[0].p_14
    p_15 = struct[0].p_15
    p_21 = struct[0].p_21
    p_22 = struct[0].p_22
    p_24 = struct[0].p_24
    p_25 = struct[0].p_25
    p_31 = struct[0].p_31
    p_32 = struct[0].p_32
    p_34 = struct[0].p_34
    p_35 = struct[0].p_35
    p_41 = struct[0].p_41
    p_42 = struct[0].p_42
    p_44 = struct[0].p_44
    p_45 = struct[0].p_45
    p_51 = struct[0].p_51
    p_52 = struct[0].p_52
    p_54 = struct[0].p_54
    p_55 = struct[0].p_55
    
    # Inputs:
    v_B1_a_r = struct[0].v_B1_a_r
    v_B1_a_i = struct[0].v_B1_a_i
    v_B1_b_r = struct[0].v_B1_b_r
    v_B1_b_i = struct[0].v_B1_b_i
    v_B1_c_r = struct[0].v_B1_c_r
    v_B1_c_i = struct[0].v_B1_c_i
    v_B7_a_r = struct[0].v_B7_a_r
    v_B7_a_i = struct[0].v_B7_a_i
    v_B7_b_r = struct[0].v_B7_b_r
    v_B7_b_i = struct[0].v_B7_b_i
    v_B7_c_r = struct[0].v_B7_c_r
    v_B7_c_i = struct[0].v_B7_c_i
    i_B2lv_n_r = struct[0].i_B2lv_n_r
    i_B2lv_n_i = struct[0].i_B2lv_n_i
    i_B3lv_n_r = struct[0].i_B3lv_n_r
    i_B3lv_n_i = struct[0].i_B3lv_n_i
    i_B4lv_n_r = struct[0].i_B4lv_n_r
    i_B4lv_n_i = struct[0].i_B4lv_n_i
    i_B5lv_n_r = struct[0].i_B5lv_n_r
    i_B5lv_n_i = struct[0].i_B5lv_n_i
    i_B6lv_n_r = struct[0].i_B6lv_n_r
    i_B6lv_n_i = struct[0].i_B6lv_n_i
    i_B2_a_r = struct[0].i_B2_a_r
    i_B2_a_i = struct[0].i_B2_a_i
    i_B2_b_r = struct[0].i_B2_b_r
    i_B2_b_i = struct[0].i_B2_b_i
    i_B2_c_r = struct[0].i_B2_c_r
    i_B2_c_i = struct[0].i_B2_c_i
    i_B3_a_r = struct[0].i_B3_a_r
    i_B3_a_i = struct[0].i_B3_a_i
    i_B3_b_r = struct[0].i_B3_b_r
    i_B3_b_i = struct[0].i_B3_b_i
    i_B3_c_r = struct[0].i_B3_c_r
    i_B3_c_i = struct[0].i_B3_c_i
    i_B4_a_r = struct[0].i_B4_a_r
    i_B4_a_i = struct[0].i_B4_a_i
    i_B4_b_r = struct[0].i_B4_b_r
    i_B4_b_i = struct[0].i_B4_b_i
    i_B4_c_r = struct[0].i_B4_c_r
    i_B4_c_i = struct[0].i_B4_c_i
    i_B5_a_r = struct[0].i_B5_a_r
    i_B5_a_i = struct[0].i_B5_a_i
    i_B5_b_r = struct[0].i_B5_b_r
    i_B5_b_i = struct[0].i_B5_b_i
    i_B5_c_r = struct[0].i_B5_c_r
    i_B5_c_i = struct[0].i_B5_c_i
    i_B6_a_r = struct[0].i_B6_a_r
    i_B6_a_i = struct[0].i_B6_a_i
    i_B6_b_r = struct[0].i_B6_b_r
    i_B6_b_i = struct[0].i_B6_b_i
    i_B6_c_r = struct[0].i_B6_c_r
    i_B6_c_i = struct[0].i_B6_c_i
    q_B2lv_a = struct[0].q_B2lv_a
    q_B2lv_b = struct[0].q_B2lv_b
    q_B2lv_c = struct[0].q_B2lv_c
    q_B3lv_a = struct[0].q_B3lv_a
    q_B3lv_b = struct[0].q_B3lv_b
    q_B3lv_c = struct[0].q_B3lv_c
    q_B4lv_a = struct[0].q_B4lv_a
    q_B4lv_b = struct[0].q_B4lv_b
    q_B4lv_c = struct[0].q_B4lv_c
    q_B5lv_a = struct[0].q_B5lv_a
    q_B5lv_b = struct[0].q_B5lv_b
    q_B5lv_c = struct[0].q_B5lv_c
    q_B6lv_a = struct[0].q_B6lv_a
    q_B6lv_b = struct[0].q_B6lv_b
    q_B6lv_c = struct[0].q_B6lv_c
    u_dummy = struct[0].u_dummy
    Dv_r_13 = struct[0].Dv_r_13
    Dv_r_23 = struct[0].Dv_r_23
    Dv_r_33 = struct[0].Dv_r_33
    Dv_r_43 = struct[0].Dv_r_43
    Dv_r_53 = struct[0].Dv_r_53
    v_nom = struct[0].v_nom
    T_v = struct[0].T_v
    K_r = struct[0].K_r
    a_B2lv = struct[0].a_B2lv
    b_B2lv = struct[0].b_B2lv
    c_B2lv = struct[0].c_B2lv
    a_B3lv = struct[0].a_B3lv
    b_B3lv = struct[0].b_B3lv
    c_B3lv = struct[0].c_B3lv
    a_B4lv = struct[0].a_B4lv
    b_B4lv = struct[0].b_B4lv
    c_B4lv = struct[0].c_B4lv
    a_B5lv = struct[0].a_B5lv
    b_B5lv = struct[0].b_B5lv
    c_B5lv = struct[0].c_B5lv
    a_B6lv = struct[0].a_B6lv
    b_B6lv = struct[0].b_B6lv
    c_B6lv = struct[0].c_B6lv
    
    # Dynamical states:
    v_13 = struct[0].x[0,0]
    v_23 = struct[0].x[1,0]
    v_33 = struct[0].x[2,0]
    v_43 = struct[0].x[3,0]
    v_53 = struct[0].x[4,0]
    
    # Algebraic states:
    v_B2lv_a_r = struct[0].y_run[0,0]
    v_B2lv_a_i = struct[0].y_run[1,0]
    v_B2lv_b_r = struct[0].y_run[2,0]
    v_B2lv_b_i = struct[0].y_run[3,0]
    v_B2lv_c_r = struct[0].y_run[4,0]
    v_B2lv_c_i = struct[0].y_run[5,0]
    v_B2lv_n_r = struct[0].y_run[6,0]
    v_B2lv_n_i = struct[0].y_run[7,0]
    v_B3lv_a_r = struct[0].y_run[8,0]
    v_B3lv_a_i = struct[0].y_run[9,0]
    v_B3lv_b_r = struct[0].y_run[10,0]
    v_B3lv_b_i = struct[0].y_run[11,0]
    v_B3lv_c_r = struct[0].y_run[12,0]
    v_B3lv_c_i = struct[0].y_run[13,0]
    v_B3lv_n_r = struct[0].y_run[14,0]
    v_B3lv_n_i = struct[0].y_run[15,0]
    v_B4lv_a_r = struct[0].y_run[16,0]
    v_B4lv_a_i = struct[0].y_run[17,0]
    v_B4lv_b_r = struct[0].y_run[18,0]
    v_B4lv_b_i = struct[0].y_run[19,0]
    v_B4lv_c_r = struct[0].y_run[20,0]
    v_B4lv_c_i = struct[0].y_run[21,0]
    v_B4lv_n_r = struct[0].y_run[22,0]
    v_B4lv_n_i = struct[0].y_run[23,0]
    v_B5lv_a_r = struct[0].y_run[24,0]
    v_B5lv_a_i = struct[0].y_run[25,0]
    v_B5lv_b_r = struct[0].y_run[26,0]
    v_B5lv_b_i = struct[0].y_run[27,0]
    v_B5lv_c_r = struct[0].y_run[28,0]
    v_B5lv_c_i = struct[0].y_run[29,0]
    v_B5lv_n_r = struct[0].y_run[30,0]
    v_B5lv_n_i = struct[0].y_run[31,0]
    v_B6lv_a_r = struct[0].y_run[32,0]
    v_B6lv_a_i = struct[0].y_run[33,0]
    v_B6lv_b_r = struct[0].y_run[34,0]
    v_B6lv_b_i = struct[0].y_run[35,0]
    v_B6lv_c_r = struct[0].y_run[36,0]
    v_B6lv_c_i = struct[0].y_run[37,0]
    v_B6lv_n_r = struct[0].y_run[38,0]
    v_B6lv_n_i = struct[0].y_run[39,0]
    v_B2_a_r = struct[0].y_run[40,0]
    v_B2_a_i = struct[0].y_run[41,0]
    v_B2_b_r = struct[0].y_run[42,0]
    v_B2_b_i = struct[0].y_run[43,0]
    v_B2_c_r = struct[0].y_run[44,0]
    v_B2_c_i = struct[0].y_run[45,0]
    v_B3_a_r = struct[0].y_run[46,0]
    v_B3_a_i = struct[0].y_run[47,0]
    v_B3_b_r = struct[0].y_run[48,0]
    v_B3_b_i = struct[0].y_run[49,0]
    v_B3_c_r = struct[0].y_run[50,0]
    v_B3_c_i = struct[0].y_run[51,0]
    v_B4_a_r = struct[0].y_run[52,0]
    v_B4_a_i = struct[0].y_run[53,0]
    v_B4_b_r = struct[0].y_run[54,0]
    v_B4_b_i = struct[0].y_run[55,0]
    v_B4_c_r = struct[0].y_run[56,0]
    v_B4_c_i = struct[0].y_run[57,0]
    v_B5_a_r = struct[0].y_run[58,0]
    v_B5_a_i = struct[0].y_run[59,0]
    v_B5_b_r = struct[0].y_run[60,0]
    v_B5_b_i = struct[0].y_run[61,0]
    v_B5_c_r = struct[0].y_run[62,0]
    v_B5_c_i = struct[0].y_run[63,0]
    v_B6_a_r = struct[0].y_run[64,0]
    v_B6_a_i = struct[0].y_run[65,0]
    v_B6_b_r = struct[0].y_run[66,0]
    v_B6_b_i = struct[0].y_run[67,0]
    v_B6_c_r = struct[0].y_run[68,0]
    v_B6_c_i = struct[0].y_run[69,0]
    i_t_B2_B2lv_a_r = struct[0].y_run[70,0]
    i_t_B2_B2lv_a_i = struct[0].y_run[71,0]
    i_t_B2_B2lv_b_r = struct[0].y_run[72,0]
    i_t_B2_B2lv_b_i = struct[0].y_run[73,0]
    i_t_B2_B2lv_c_r = struct[0].y_run[74,0]
    i_t_B2_B2lv_c_i = struct[0].y_run[75,0]
    i_t_B3_B3lv_a_r = struct[0].y_run[76,0]
    i_t_B3_B3lv_a_i = struct[0].y_run[77,0]
    i_t_B3_B3lv_b_r = struct[0].y_run[78,0]
    i_t_B3_B3lv_b_i = struct[0].y_run[79,0]
    i_t_B3_B3lv_c_r = struct[0].y_run[80,0]
    i_t_B3_B3lv_c_i = struct[0].y_run[81,0]
    i_t_B4_B4lv_a_r = struct[0].y_run[82,0]
    i_t_B4_B4lv_a_i = struct[0].y_run[83,0]
    i_t_B4_B4lv_b_r = struct[0].y_run[84,0]
    i_t_B4_B4lv_b_i = struct[0].y_run[85,0]
    i_t_B4_B4lv_c_r = struct[0].y_run[86,0]
    i_t_B4_B4lv_c_i = struct[0].y_run[87,0]
    i_t_B5_B5lv_a_r = struct[0].y_run[88,0]
    i_t_B5_B5lv_a_i = struct[0].y_run[89,0]
    i_t_B5_B5lv_b_r = struct[0].y_run[90,0]
    i_t_B5_B5lv_b_i = struct[0].y_run[91,0]
    i_t_B5_B5lv_c_r = struct[0].y_run[92,0]
    i_t_B5_B5lv_c_i = struct[0].y_run[93,0]
    i_t_B6_B6lv_a_r = struct[0].y_run[94,0]
    i_t_B6_B6lv_a_i = struct[0].y_run[95,0]
    i_t_B6_B6lv_b_r = struct[0].y_run[96,0]
    i_t_B6_B6lv_b_i = struct[0].y_run[97,0]
    i_t_B6_B6lv_c_r = struct[0].y_run[98,0]
    i_t_B6_B6lv_c_i = struct[0].y_run[99,0]
    i_l_B1_B2_a_r = struct[0].y_run[100,0]
    i_l_B1_B2_a_i = struct[0].y_run[101,0]
    i_l_B1_B2_b_r = struct[0].y_run[102,0]
    i_l_B1_B2_b_i = struct[0].y_run[103,0]
    i_l_B1_B2_c_r = struct[0].y_run[104,0]
    i_l_B1_B2_c_i = struct[0].y_run[105,0]
    i_l_B2_B3_a_r = struct[0].y_run[106,0]
    i_l_B2_B3_a_i = struct[0].y_run[107,0]
    i_l_B2_B3_b_r = struct[0].y_run[108,0]
    i_l_B2_B3_b_i = struct[0].y_run[109,0]
    i_l_B2_B3_c_r = struct[0].y_run[110,0]
    i_l_B2_B3_c_i = struct[0].y_run[111,0]
    i_l_B3_B4_a_r = struct[0].y_run[112,0]
    i_l_B3_B4_a_i = struct[0].y_run[113,0]
    i_l_B3_B4_b_r = struct[0].y_run[114,0]
    i_l_B3_B4_b_i = struct[0].y_run[115,0]
    i_l_B3_B4_c_r = struct[0].y_run[116,0]
    i_l_B3_B4_c_i = struct[0].y_run[117,0]
    i_l_B5_B6_a_r = struct[0].y_run[118,0]
    i_l_B5_B6_a_i = struct[0].y_run[119,0]
    i_l_B5_B6_b_r = struct[0].y_run[120,0]
    i_l_B5_B6_b_i = struct[0].y_run[121,0]
    i_l_B5_B6_c_r = struct[0].y_run[122,0]
    i_l_B5_B6_c_i = struct[0].y_run[123,0]
    i_l_B6_B7_a_r = struct[0].y_run[124,0]
    i_l_B6_B7_a_i = struct[0].y_run[125,0]
    i_l_B6_B7_b_r = struct[0].y_run[126,0]
    i_l_B6_B7_b_i = struct[0].y_run[127,0]
    i_l_B6_B7_c_r = struct[0].y_run[128,0]
    i_l_B6_B7_c_i = struct[0].y_run[129,0]
    i_load_B2lv_a_r = struct[0].y_run[130,0]
    i_load_B2lv_a_i = struct[0].y_run[131,0]
    i_load_B2lv_b_r = struct[0].y_run[132,0]
    i_load_B2lv_b_i = struct[0].y_run[133,0]
    i_load_B2lv_c_r = struct[0].y_run[134,0]
    i_load_B2lv_c_i = struct[0].y_run[135,0]
    i_load_B2lv_n_r = struct[0].y_run[136,0]
    i_load_B2lv_n_i = struct[0].y_run[137,0]
    i_load_B3lv_a_r = struct[0].y_run[138,0]
    i_load_B3lv_a_i = struct[0].y_run[139,0]
    i_load_B3lv_b_r = struct[0].y_run[140,0]
    i_load_B3lv_b_i = struct[0].y_run[141,0]
    i_load_B3lv_c_r = struct[0].y_run[142,0]
    i_load_B3lv_c_i = struct[0].y_run[143,0]
    i_load_B3lv_n_r = struct[0].y_run[144,0]
    i_load_B3lv_n_i = struct[0].y_run[145,0]
    i_load_B4lv_a_r = struct[0].y_run[146,0]
    i_load_B4lv_a_i = struct[0].y_run[147,0]
    i_load_B4lv_b_r = struct[0].y_run[148,0]
    i_load_B4lv_b_i = struct[0].y_run[149,0]
    i_load_B4lv_c_r = struct[0].y_run[150,0]
    i_load_B4lv_c_i = struct[0].y_run[151,0]
    i_load_B4lv_n_r = struct[0].y_run[152,0]
    i_load_B4lv_n_i = struct[0].y_run[153,0]
    i_load_B5lv_a_r = struct[0].y_run[154,0]
    i_load_B5lv_a_i = struct[0].y_run[155,0]
    i_load_B5lv_b_r = struct[0].y_run[156,0]
    i_load_B5lv_b_i = struct[0].y_run[157,0]
    i_load_B5lv_c_r = struct[0].y_run[158,0]
    i_load_B5lv_c_i = struct[0].y_run[159,0]
    i_load_B5lv_n_r = struct[0].y_run[160,0]
    i_load_B5lv_n_i = struct[0].y_run[161,0]
    i_load_B6lv_a_r = struct[0].y_run[162,0]
    i_load_B6lv_a_i = struct[0].y_run[163,0]
    i_load_B6lv_b_r = struct[0].y_run[164,0]
    i_load_B6lv_b_i = struct[0].y_run[165,0]
    i_load_B6lv_c_r = struct[0].y_run[166,0]
    i_load_B6lv_c_i = struct[0].y_run[167,0]
    i_load_B6lv_n_r = struct[0].y_run[168,0]
    i_load_B6lv_n_i = struct[0].y_run[169,0]
    i_l_1112 = struct[0].y_run[170,0]
    i_l_1213 = struct[0].y_run[171,0]
    i_l_1314 = struct[0].y_run[172,0]
    i_l_1415 = struct[0].y_run[173,0]
    i_l_2122 = struct[0].y_run[174,0]
    i_l_2223 = struct[0].y_run[175,0]
    i_l_2324 = struct[0].y_run[176,0]
    i_l_2425 = struct[0].y_run[177,0]
    i_l_3132 = struct[0].y_run[178,0]
    i_l_3233 = struct[0].y_run[179,0]
    i_l_3334 = struct[0].y_run[180,0]
    i_l_3435 = struct[0].y_run[181,0]
    i_l_4142 = struct[0].y_run[182,0]
    i_l_4243 = struct[0].y_run[183,0]
    i_l_4344 = struct[0].y_run[184,0]
    i_l_4445 = struct[0].y_run[185,0]
    i_l_5152 = struct[0].y_run[186,0]
    i_l_5253 = struct[0].y_run[187,0]
    i_l_5354 = struct[0].y_run[188,0]
    i_l_5455 = struct[0].y_run[189,0]
    i_l_1521 = struct[0].y_run[190,0]
    i_l_2531 = struct[0].y_run[191,0]
    i_l_3541 = struct[0].y_run[192,0]
    i_l_4551 = struct[0].y_run[193,0]
    v_11 = struct[0].y_run[194,0]
    v_12 = struct[0].y_run[195,0]
    i_13 = struct[0].y_run[196,0]
    v_14 = struct[0].y_run[197,0]
    v_15 = struct[0].y_run[198,0]
    v_21 = struct[0].y_run[199,0]
    v_22 = struct[0].y_run[200,0]
    i_23 = struct[0].y_run[201,0]
    v_24 = struct[0].y_run[202,0]
    v_25 = struct[0].y_run[203,0]
    v_31 = struct[0].y_run[204,0]
    v_32 = struct[0].y_run[205,0]
    i_33 = struct[0].y_run[206,0]
    v_34 = struct[0].y_run[207,0]
    v_35 = struct[0].y_run[208,0]
    v_41 = struct[0].y_run[209,0]
    v_42 = struct[0].y_run[210,0]
    i_43 = struct[0].y_run[211,0]
    v_44 = struct[0].y_run[212,0]
    v_45 = struct[0].y_run[213,0]
    v_51 = struct[0].y_run[214,0]
    v_52 = struct[0].y_run[215,0]
    i_53 = struct[0].y_run[216,0]
    v_54 = struct[0].y_run[217,0]
    v_55 = struct[0].y_run[218,0]
    i_11 = struct[0].y_run[219,0]
    i_12 = struct[0].y_run[220,0]
    i_14 = struct[0].y_run[221,0]
    i_15 = struct[0].y_run[222,0]
    i_21 = struct[0].y_run[223,0]
    i_22 = struct[0].y_run[224,0]
    i_24 = struct[0].y_run[225,0]
    i_25 = struct[0].y_run[226,0]
    i_31 = struct[0].y_run[227,0]
    i_32 = struct[0].y_run[228,0]
    i_34 = struct[0].y_run[229,0]
    i_35 = struct[0].y_run[230,0]
    i_41 = struct[0].y_run[231,0]
    i_42 = struct[0].y_run[232,0]
    i_44 = struct[0].y_run[233,0]
    i_45 = struct[0].y_run[234,0]
    i_51 = struct[0].y_run[235,0]
    i_52 = struct[0].y_run[236,0]
    i_54 = struct[0].y_run[237,0]
    i_55 = struct[0].y_run[238,0]
    p_B2lv_a = struct[0].y_run[239,0]
    p_B2lv_b = struct[0].y_run[240,0]
    p_B2lv_c = struct[0].y_run[241,0]
    p_B3lv_a = struct[0].y_run[242,0]
    p_B3lv_b = struct[0].y_run[243,0]
    p_B3lv_c = struct[0].y_run[244,0]
    p_B4lv_a = struct[0].y_run[245,0]
    p_B4lv_b = struct[0].y_run[246,0]
    p_B4lv_c = struct[0].y_run[247,0]
    p_B5lv_a = struct[0].y_run[248,0]
    p_B5lv_b = struct[0].y_run[249,0]
    p_B5lv_c = struct[0].y_run[250,0]
    p_B6lv_a = struct[0].y_run[251,0]
    p_B6lv_b = struct[0].y_run[252,0]
    p_B6lv_c = struct[0].y_run[253,0]
    
    struct[0].u_run[0,0] = v_B1_a_r
    struct[0].u_run[1,0] = v_B1_a_i
    struct[0].u_run[2,0] = v_B1_b_r
    struct[0].u_run[3,0] = v_B1_b_i
    struct[0].u_run[4,0] = v_B1_c_r
    struct[0].u_run[5,0] = v_B1_c_i
    struct[0].u_run[6,0] = v_B7_a_r
    struct[0].u_run[7,0] = v_B7_a_i
    struct[0].u_run[8,0] = v_B7_b_r
    struct[0].u_run[9,0] = v_B7_b_i
    struct[0].u_run[10,0] = v_B7_c_r
    struct[0].u_run[11,0] = v_B7_c_i
    struct[0].u_run[12,0] = i_B2lv_n_r
    struct[0].u_run[13,0] = i_B2lv_n_i
    struct[0].u_run[14,0] = i_B3lv_n_r
    struct[0].u_run[15,0] = i_B3lv_n_i
    struct[0].u_run[16,0] = i_B4lv_n_r
    struct[0].u_run[17,0] = i_B4lv_n_i
    struct[0].u_run[18,0] = i_B5lv_n_r
    struct[0].u_run[19,0] = i_B5lv_n_i
    struct[0].u_run[20,0] = i_B6lv_n_r
    struct[0].u_run[21,0] = i_B6lv_n_i
    struct[0].u_run[22,0] = i_B2_a_r
    struct[0].u_run[23,0] = i_B2_a_i
    struct[0].u_run[24,0] = i_B2_b_r
    struct[0].u_run[25,0] = i_B2_b_i
    struct[0].u_run[26,0] = i_B2_c_r
    struct[0].u_run[27,0] = i_B2_c_i
    struct[0].u_run[28,0] = i_B3_a_r
    struct[0].u_run[29,0] = i_B3_a_i
    struct[0].u_run[30,0] = i_B3_b_r
    struct[0].u_run[31,0] = i_B3_b_i
    struct[0].u_run[32,0] = i_B3_c_r
    struct[0].u_run[33,0] = i_B3_c_i
    struct[0].u_run[34,0] = i_B4_a_r
    struct[0].u_run[35,0] = i_B4_a_i
    struct[0].u_run[36,0] = i_B4_b_r
    struct[0].u_run[37,0] = i_B4_b_i
    struct[0].u_run[38,0] = i_B4_c_r
    struct[0].u_run[39,0] = i_B4_c_i
    struct[0].u_run[40,0] = i_B5_a_r
    struct[0].u_run[41,0] = i_B5_a_i
    struct[0].u_run[42,0] = i_B5_b_r
    struct[0].u_run[43,0] = i_B5_b_i
    struct[0].u_run[44,0] = i_B5_c_r
    struct[0].u_run[45,0] = i_B5_c_i
    struct[0].u_run[46,0] = i_B6_a_r
    struct[0].u_run[47,0] = i_B6_a_i
    struct[0].u_run[48,0] = i_B6_b_r
    struct[0].u_run[49,0] = i_B6_b_i
    struct[0].u_run[50,0] = i_B6_c_r
    struct[0].u_run[51,0] = i_B6_c_i
    struct[0].u_run[52,0] = q_B2lv_a
    struct[0].u_run[53,0] = q_B2lv_b
    struct[0].u_run[54,0] = q_B2lv_c
    struct[0].u_run[55,0] = q_B3lv_a
    struct[0].u_run[56,0] = q_B3lv_b
    struct[0].u_run[57,0] = q_B3lv_c
    struct[0].u_run[58,0] = q_B4lv_a
    struct[0].u_run[59,0] = q_B4lv_b
    struct[0].u_run[60,0] = q_B4lv_c
    struct[0].u_run[61,0] = q_B5lv_a
    struct[0].u_run[62,0] = q_B5lv_b
    struct[0].u_run[63,0] = q_B5lv_c
    struct[0].u_run[64,0] = q_B6lv_a
    struct[0].u_run[65,0] = q_B6lv_b
    struct[0].u_run[66,0] = q_B6lv_c
    struct[0].u_run[67,0] = u_dummy
    struct[0].u_run[68,0] = Dv_r_13
    struct[0].u_run[69,0] = Dv_r_23
    struct[0].u_run[70,0] = Dv_r_33
    struct[0].u_run[71,0] = Dv_r_43
    struct[0].u_run[72,0] = Dv_r_53
    struct[0].u_run[73,0] = v_nom
    struct[0].u_run[74,0] = T_v
    struct[0].u_run[75,0] = K_r
    struct[0].u_run[76,0] = a_B2lv
    struct[0].u_run[77,0] = b_B2lv
    struct[0].u_run[78,0] = c_B2lv
    struct[0].u_run[79,0] = a_B3lv
    struct[0].u_run[80,0] = b_B3lv
    struct[0].u_run[81,0] = c_B3lv
    struct[0].u_run[82,0] = a_B4lv
    struct[0].u_run[83,0] = b_B4lv
    struct[0].u_run[84,0] = c_B4lv
    struct[0].u_run[85,0] = a_B5lv
    struct[0].u_run[86,0] = b_B5lv
    struct[0].u_run[87,0] = c_B5lv
    struct[0].u_run[88,0] = a_B6lv
    struct[0].u_run[89,0] = b_B6lv
    struct[0].u_run[90,0] = c_B6lv
    # Differential equations:
    if mode == 2:


        struct[0].f[0,0] = (-Dv_r_13 - K_r*i_13*v_13 - v_13 + v_nom)/T_v
        struct[0].f[1,0] = (-Dv_r_23 - K_r*i_23*v_23 - v_23 + v_nom)/T_v
        struct[0].f[2,0] = (-Dv_r_33 - K_r*i_33*v_33 - v_33 + v_nom)/T_v
        struct[0].f[3,0] = (-Dv_r_43 - K_r*i_43*v_43 - v_43 + v_nom)/T_v
        struct[0].f[4,0] = (-Dv_r_53 - K_r*i_53*v_53 - v_53 + v_nom)/T_v
    
    # Algebraic equations:
    if mode == 3:

        struct[0].g[:,:] = np.ascontiguousarray(struct[0].Gy) @ np.ascontiguousarray(struct[0].y_run) + np.ascontiguousarray(struct[0].Gu) @ np.ascontiguousarray(struct[0].u_run)

        struct[0].g[130,0] = i_load_B2lv_a_i*v_B2lv_a_i - i_load_B2lv_a_i*v_B2lv_n_i + i_load_B2lv_a_r*v_B2lv_a_r - i_load_B2lv_a_r*v_B2lv_n_r - p_B2lv_a
        struct[0].g[131,0] = i_load_B2lv_b_i*v_B2lv_b_i - i_load_B2lv_b_i*v_B2lv_n_i + i_load_B2lv_b_r*v_B2lv_b_r - i_load_B2lv_b_r*v_B2lv_n_r - p_B2lv_b
        struct[0].g[132,0] = i_load_B2lv_c_i*v_B2lv_c_i - i_load_B2lv_c_i*v_B2lv_n_i + i_load_B2lv_c_r*v_B2lv_c_r - i_load_B2lv_c_r*v_B2lv_n_r - p_B2lv_c
        struct[0].g[133,0] = -i_load_B2lv_a_i*v_B2lv_a_r + i_load_B2lv_a_i*v_B2lv_n_r + i_load_B2lv_a_r*v_B2lv_a_i - i_load_B2lv_a_r*v_B2lv_n_i - q_B2lv_a
        struct[0].g[134,0] = -i_load_B2lv_b_i*v_B2lv_b_r + i_load_B2lv_b_i*v_B2lv_n_r + i_load_B2lv_b_r*v_B2lv_b_i - i_load_B2lv_b_r*v_B2lv_n_i - q_B2lv_b
        struct[0].g[135,0] = -i_load_B2lv_c_i*v_B2lv_c_r + i_load_B2lv_c_i*v_B2lv_n_r + i_load_B2lv_c_r*v_B2lv_c_i - i_load_B2lv_c_r*v_B2lv_n_i - q_B2lv_c
        struct[0].g[138,0] = i_load_B3lv_a_i*v_B3lv_a_i - i_load_B3lv_a_i*v_B3lv_n_i + i_load_B3lv_a_r*v_B3lv_a_r - i_load_B3lv_a_r*v_B3lv_n_r - p_B3lv_a
        struct[0].g[139,0] = i_load_B3lv_b_i*v_B3lv_b_i - i_load_B3lv_b_i*v_B3lv_n_i + i_load_B3lv_b_r*v_B3lv_b_r - i_load_B3lv_b_r*v_B3lv_n_r - p_B3lv_b
        struct[0].g[140,0] = i_load_B3lv_c_i*v_B3lv_c_i - i_load_B3lv_c_i*v_B3lv_n_i + i_load_B3lv_c_r*v_B3lv_c_r - i_load_B3lv_c_r*v_B3lv_n_r - p_B3lv_c
        struct[0].g[141,0] = -i_load_B3lv_a_i*v_B3lv_a_r + i_load_B3lv_a_i*v_B3lv_n_r + i_load_B3lv_a_r*v_B3lv_a_i - i_load_B3lv_a_r*v_B3lv_n_i - q_B3lv_a
        struct[0].g[142,0] = -i_load_B3lv_b_i*v_B3lv_b_r + i_load_B3lv_b_i*v_B3lv_n_r + i_load_B3lv_b_r*v_B3lv_b_i - i_load_B3lv_b_r*v_B3lv_n_i - q_B3lv_b
        struct[0].g[143,0] = -i_load_B3lv_c_i*v_B3lv_c_r + i_load_B3lv_c_i*v_B3lv_n_r + i_load_B3lv_c_r*v_B3lv_c_i - i_load_B3lv_c_r*v_B3lv_n_i - q_B3lv_c
        struct[0].g[146,0] = i_load_B4lv_a_i*v_B4lv_a_i - i_load_B4lv_a_i*v_B4lv_n_i + i_load_B4lv_a_r*v_B4lv_a_r - i_load_B4lv_a_r*v_B4lv_n_r - p_B4lv_a
        struct[0].g[147,0] = i_load_B4lv_b_i*v_B4lv_b_i - i_load_B4lv_b_i*v_B4lv_n_i + i_load_B4lv_b_r*v_B4lv_b_r - i_load_B4lv_b_r*v_B4lv_n_r - p_B4lv_b
        struct[0].g[148,0] = i_load_B4lv_c_i*v_B4lv_c_i - i_load_B4lv_c_i*v_B4lv_n_i + i_load_B4lv_c_r*v_B4lv_c_r - i_load_B4lv_c_r*v_B4lv_n_r - p_B4lv_c
        struct[0].g[149,0] = -i_load_B4lv_a_i*v_B4lv_a_r + i_load_B4lv_a_i*v_B4lv_n_r + i_load_B4lv_a_r*v_B4lv_a_i - i_load_B4lv_a_r*v_B4lv_n_i - q_B4lv_a
        struct[0].g[150,0] = -i_load_B4lv_b_i*v_B4lv_b_r + i_load_B4lv_b_i*v_B4lv_n_r + i_load_B4lv_b_r*v_B4lv_b_i - i_load_B4lv_b_r*v_B4lv_n_i - q_B4lv_b
        struct[0].g[151,0] = -i_load_B4lv_c_i*v_B4lv_c_r + i_load_B4lv_c_i*v_B4lv_n_r + i_load_B4lv_c_r*v_B4lv_c_i - i_load_B4lv_c_r*v_B4lv_n_i - q_B4lv_c
        struct[0].g[154,0] = i_load_B5lv_a_i*v_B5lv_a_i - i_load_B5lv_a_i*v_B5lv_n_i + i_load_B5lv_a_r*v_B5lv_a_r - i_load_B5lv_a_r*v_B5lv_n_r - p_B5lv_a
        struct[0].g[155,0] = i_load_B5lv_b_i*v_B5lv_b_i - i_load_B5lv_b_i*v_B5lv_n_i + i_load_B5lv_b_r*v_B5lv_b_r - i_load_B5lv_b_r*v_B5lv_n_r - p_B5lv_b
        struct[0].g[156,0] = i_load_B5lv_c_i*v_B5lv_c_i - i_load_B5lv_c_i*v_B5lv_n_i + i_load_B5lv_c_r*v_B5lv_c_r - i_load_B5lv_c_r*v_B5lv_n_r - p_B5lv_c
        struct[0].g[157,0] = -i_load_B5lv_a_i*v_B5lv_a_r + i_load_B5lv_a_i*v_B5lv_n_r + i_load_B5lv_a_r*v_B5lv_a_i - i_load_B5lv_a_r*v_B5lv_n_i - q_B5lv_a
        struct[0].g[158,0] = -i_load_B5lv_b_i*v_B5lv_b_r + i_load_B5lv_b_i*v_B5lv_n_r + i_load_B5lv_b_r*v_B5lv_b_i - i_load_B5lv_b_r*v_B5lv_n_i - q_B5lv_b
        struct[0].g[159,0] = -i_load_B5lv_c_i*v_B5lv_c_r + i_load_B5lv_c_i*v_B5lv_n_r + i_load_B5lv_c_r*v_B5lv_c_i - i_load_B5lv_c_r*v_B5lv_n_i - q_B5lv_c
        struct[0].g[162,0] = i_load_B6lv_a_i*v_B6lv_a_i - i_load_B6lv_a_i*v_B6lv_n_i + i_load_B6lv_a_r*v_B6lv_a_r - i_load_B6lv_a_r*v_B6lv_n_r - p_B6lv_a
        struct[0].g[163,0] = i_load_B6lv_b_i*v_B6lv_b_i - i_load_B6lv_b_i*v_B6lv_n_i + i_load_B6lv_b_r*v_B6lv_b_r - i_load_B6lv_b_r*v_B6lv_n_r - p_B6lv_b
        struct[0].g[164,0] = i_load_B6lv_c_i*v_B6lv_c_i - i_load_B6lv_c_i*v_B6lv_n_i + i_load_B6lv_c_r*v_B6lv_c_r - i_load_B6lv_c_r*v_B6lv_n_r - p_B6lv_c
        struct[0].g[165,0] = -i_load_B6lv_a_i*v_B6lv_a_r + i_load_B6lv_a_i*v_B6lv_n_r + i_load_B6lv_a_r*v_B6lv_a_i - i_load_B6lv_a_r*v_B6lv_n_i - q_B6lv_a
        struct[0].g[166,0] = -i_load_B6lv_b_i*v_B6lv_b_r + i_load_B6lv_b_i*v_B6lv_n_r + i_load_B6lv_b_r*v_B6lv_b_i - i_load_B6lv_b_r*v_B6lv_n_i - q_B6lv_b
        struct[0].g[167,0] = -i_load_B6lv_c_i*v_B6lv_c_r + i_load_B6lv_c_i*v_B6lv_n_r + i_load_B6lv_c_r*v_B6lv_c_i - i_load_B6lv_c_r*v_B6lv_n_i - q_B6lv_c
        struct[0].g[171,0] = -R_1213*i_l_1213 + v_12 - v_13
        struct[0].g[172,0] = -R_1314*i_l_1314 + v_13 - v_14
        struct[0].g[175,0] = -R_2223*i_l_2223 + v_22 - v_23
        struct[0].g[176,0] = -R_2324*i_l_2324 + v_23 - v_24
        struct[0].g[179,0] = -R_3233*i_l_3233 + v_32 - v_33
        struct[0].g[180,0] = -R_3334*i_l_3334 + v_33 - v_34
        struct[0].g[183,0] = -R_4243*i_l_4243 + v_42 - v_43
        struct[0].g[184,0] = -R_4344*i_l_4344 + v_43 - v_44
        struct[0].g[187,0] = -R_5253*i_l_5253 + v_52 - v_53
        struct[0].g[188,0] = -R_5354*i_l_5354 + v_53 - v_54
        struct[0].g[219,0] = i_11*v_11 - p_11
        struct[0].g[220,0] = i_12*v_12 - p_12
        struct[0].g[221,0] = i_14*v_14 - p_14
        struct[0].g[222,0] = i_15*v_15 - p_15
        struct[0].g[223,0] = i_21*v_21 - p_21
        struct[0].g[224,0] = i_22*v_22 - p_22
        struct[0].g[225,0] = i_24*v_24 - p_24
        struct[0].g[226,0] = i_25*v_25 - p_25
        struct[0].g[227,0] = i_31*v_31 - p_31
        struct[0].g[228,0] = i_32*v_32 - p_32
        struct[0].g[229,0] = i_34*v_34 - p_34
        struct[0].g[230,0] = i_35*v_35 - p_35
        struct[0].g[231,0] = i_41*v_41 - p_41
        struct[0].g[232,0] = i_42*v_42 - p_42
        struct[0].g[233,0] = i_44*v_44 - p_44
        struct[0].g[234,0] = i_45*v_45 - p_45
        struct[0].g[235,0] = i_51*v_51 - p_51
        struct[0].g[236,0] = i_52*v_52 - p_52
        struct[0].g[237,0] = i_54*v_54 - p_54
        struct[0].g[238,0] = i_55*v_55 - p_55
        struct[0].g[239,0] = i_13*v_13/3 + p_B2lv_a + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[240,0] = i_13*v_13/3 + p_B2lv_b + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[241,0] = i_13*v_13/3 + p_B2lv_c + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[242,0] = i_23*v_23/3 + p_B3lv_a + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[243,0] = i_23*v_23/3 + p_B3lv_b + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[244,0] = i_23*v_23/3 + p_B3lv_c + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[245,0] = i_33*v_33/3 + p_B4lv_a + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[246,0] = i_33*v_33/3 + p_B4lv_b + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[247,0] = i_33*v_33/3 + p_B4lv_c + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[248,0] = i_43*v_43/3 + p_B5lv_a + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[249,0] = i_43*v_43/3 + p_B5lv_b + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[250,0] = i_43*v_43/3 + p_B5lv_c + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[251,0] = i_53*v_53/3 + p_B6lv_a + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
        struct[0].g[252,0] = i_53*v_53/3 + p_B6lv_b + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
        struct[0].g[253,0] = i_53*v_53/3 + p_B6lv_c + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
    
    # Outputs:
    if mode == 3:

        struct[0].h[0,0] = i_13*v_13
        struct[0].h[1,0] = v_13
        struct[0].h[2,0] = i_23*v_23
        struct[0].h[3,0] = v_23
        struct[0].h[4,0] = i_33*v_33
        struct[0].h[5,0] = v_33
        struct[0].h[6,0] = i_43*v_43
        struct[0].h[7,0] = v_43
        struct[0].h[8,0] = i_53*v_53
        struct[0].h[9,0] = v_53
    

    if mode == 10:

        struct[0].Fx[0,0] = (-K_r*i_13 - 1)/T_v
        struct[0].Fx[1,1] = (-K_r*i_23 - 1)/T_v
        struct[0].Fx[2,2] = (-K_r*i_33 - 1)/T_v
        struct[0].Fx[3,3] = (-K_r*i_43 - 1)/T_v
        struct[0].Fx[4,4] = (-K_r*i_53 - 1)/T_v

    if mode == 11:

        struct[0].Fy[0,196] = -K_r*v_13/T_v
        struct[0].Fy[1,201] = -K_r*v_23/T_v
        struct[0].Fy[2,206] = -K_r*v_33/T_v
        struct[0].Fy[3,211] = -K_r*v_43/T_v
        struct[0].Fy[4,216] = -K_r*v_53/T_v

        struct[0].Gx[171,0] = -1
        struct[0].Gx[172,0] = 1
        struct[0].Gx[175,1] = -1
        struct[0].Gx[176,1] = 1
        struct[0].Gx[179,2] = -1
        struct[0].Gx[180,2] = 1
        struct[0].Gx[183,3] = -1
        struct[0].Gx[184,3] = 1
        struct[0].Gx[187,4] = -1
        struct[0].Gx[188,4] = 1
        struct[0].Gx[239,0] = i_13/3
        struct[0].Gx[240,0] = i_13/3
        struct[0].Gx[241,0] = i_13/3
        struct[0].Gx[242,1] = i_23/3
        struct[0].Gx[243,1] = i_23/3
        struct[0].Gx[244,1] = i_23/3
        struct[0].Gx[245,2] = i_33/3
        struct[0].Gx[246,2] = i_33/3
        struct[0].Gx[247,2] = i_33/3
        struct[0].Gx[248,3] = i_43/3
        struct[0].Gx[249,3] = i_43/3
        struct[0].Gx[250,3] = i_43/3
        struct[0].Gx[251,4] = i_53/3
        struct[0].Gx[252,4] = i_53/3
        struct[0].Gx[253,4] = i_53/3

        struct[0].Gy[130,0] = i_load_B2lv_a_r
        struct[0].Gy[130,1] = i_load_B2lv_a_i
        struct[0].Gy[130,6] = -i_load_B2lv_a_r
        struct[0].Gy[130,7] = -i_load_B2lv_a_i
        struct[0].Gy[130,130] = v_B2lv_a_r - v_B2lv_n_r
        struct[0].Gy[130,131] = v_B2lv_a_i - v_B2lv_n_i
        struct[0].Gy[131,2] = i_load_B2lv_b_r
        struct[0].Gy[131,3] = i_load_B2lv_b_i
        struct[0].Gy[131,6] = -i_load_B2lv_b_r
        struct[0].Gy[131,7] = -i_load_B2lv_b_i
        struct[0].Gy[131,132] = v_B2lv_b_r - v_B2lv_n_r
        struct[0].Gy[131,133] = v_B2lv_b_i - v_B2lv_n_i
        struct[0].Gy[132,4] = i_load_B2lv_c_r
        struct[0].Gy[132,5] = i_load_B2lv_c_i
        struct[0].Gy[132,6] = -i_load_B2lv_c_r
        struct[0].Gy[132,7] = -i_load_B2lv_c_i
        struct[0].Gy[132,134] = v_B2lv_c_r - v_B2lv_n_r
        struct[0].Gy[132,135] = v_B2lv_c_i - v_B2lv_n_i
        struct[0].Gy[133,0] = -i_load_B2lv_a_i
        struct[0].Gy[133,1] = i_load_B2lv_a_r
        struct[0].Gy[133,6] = i_load_B2lv_a_i
        struct[0].Gy[133,7] = -i_load_B2lv_a_r
        struct[0].Gy[133,130] = v_B2lv_a_i - v_B2lv_n_i
        struct[0].Gy[133,131] = -v_B2lv_a_r + v_B2lv_n_r
        struct[0].Gy[134,2] = -i_load_B2lv_b_i
        struct[0].Gy[134,3] = i_load_B2lv_b_r
        struct[0].Gy[134,6] = i_load_B2lv_b_i
        struct[0].Gy[134,7] = -i_load_B2lv_b_r
        struct[0].Gy[134,132] = v_B2lv_b_i - v_B2lv_n_i
        struct[0].Gy[134,133] = -v_B2lv_b_r + v_B2lv_n_r
        struct[0].Gy[135,4] = -i_load_B2lv_c_i
        struct[0].Gy[135,5] = i_load_B2lv_c_r
        struct[0].Gy[135,6] = i_load_B2lv_c_i
        struct[0].Gy[135,7] = -i_load_B2lv_c_r
        struct[0].Gy[135,134] = v_B2lv_c_i - v_B2lv_n_i
        struct[0].Gy[135,135] = -v_B2lv_c_r + v_B2lv_n_r
        struct[0].Gy[138,8] = i_load_B3lv_a_r
        struct[0].Gy[138,9] = i_load_B3lv_a_i
        struct[0].Gy[138,14] = -i_load_B3lv_a_r
        struct[0].Gy[138,15] = -i_load_B3lv_a_i
        struct[0].Gy[138,138] = v_B3lv_a_r - v_B3lv_n_r
        struct[0].Gy[138,139] = v_B3lv_a_i - v_B3lv_n_i
        struct[0].Gy[139,10] = i_load_B3lv_b_r
        struct[0].Gy[139,11] = i_load_B3lv_b_i
        struct[0].Gy[139,14] = -i_load_B3lv_b_r
        struct[0].Gy[139,15] = -i_load_B3lv_b_i
        struct[0].Gy[139,140] = v_B3lv_b_r - v_B3lv_n_r
        struct[0].Gy[139,141] = v_B3lv_b_i - v_B3lv_n_i
        struct[0].Gy[140,12] = i_load_B3lv_c_r
        struct[0].Gy[140,13] = i_load_B3lv_c_i
        struct[0].Gy[140,14] = -i_load_B3lv_c_r
        struct[0].Gy[140,15] = -i_load_B3lv_c_i
        struct[0].Gy[140,142] = v_B3lv_c_r - v_B3lv_n_r
        struct[0].Gy[140,143] = v_B3lv_c_i - v_B3lv_n_i
        struct[0].Gy[141,8] = -i_load_B3lv_a_i
        struct[0].Gy[141,9] = i_load_B3lv_a_r
        struct[0].Gy[141,14] = i_load_B3lv_a_i
        struct[0].Gy[141,15] = -i_load_B3lv_a_r
        struct[0].Gy[141,138] = v_B3lv_a_i - v_B3lv_n_i
        struct[0].Gy[141,139] = -v_B3lv_a_r + v_B3lv_n_r
        struct[0].Gy[142,10] = -i_load_B3lv_b_i
        struct[0].Gy[142,11] = i_load_B3lv_b_r
        struct[0].Gy[142,14] = i_load_B3lv_b_i
        struct[0].Gy[142,15] = -i_load_B3lv_b_r
        struct[0].Gy[142,140] = v_B3lv_b_i - v_B3lv_n_i
        struct[0].Gy[142,141] = -v_B3lv_b_r + v_B3lv_n_r
        struct[0].Gy[143,12] = -i_load_B3lv_c_i
        struct[0].Gy[143,13] = i_load_B3lv_c_r
        struct[0].Gy[143,14] = i_load_B3lv_c_i
        struct[0].Gy[143,15] = -i_load_B3lv_c_r
        struct[0].Gy[143,142] = v_B3lv_c_i - v_B3lv_n_i
        struct[0].Gy[143,143] = -v_B3lv_c_r + v_B3lv_n_r
        struct[0].Gy[146,16] = i_load_B4lv_a_r
        struct[0].Gy[146,17] = i_load_B4lv_a_i
        struct[0].Gy[146,22] = -i_load_B4lv_a_r
        struct[0].Gy[146,23] = -i_load_B4lv_a_i
        struct[0].Gy[146,146] = v_B4lv_a_r - v_B4lv_n_r
        struct[0].Gy[146,147] = v_B4lv_a_i - v_B4lv_n_i
        struct[0].Gy[147,18] = i_load_B4lv_b_r
        struct[0].Gy[147,19] = i_load_B4lv_b_i
        struct[0].Gy[147,22] = -i_load_B4lv_b_r
        struct[0].Gy[147,23] = -i_load_B4lv_b_i
        struct[0].Gy[147,148] = v_B4lv_b_r - v_B4lv_n_r
        struct[0].Gy[147,149] = v_B4lv_b_i - v_B4lv_n_i
        struct[0].Gy[148,20] = i_load_B4lv_c_r
        struct[0].Gy[148,21] = i_load_B4lv_c_i
        struct[0].Gy[148,22] = -i_load_B4lv_c_r
        struct[0].Gy[148,23] = -i_load_B4lv_c_i
        struct[0].Gy[148,150] = v_B4lv_c_r - v_B4lv_n_r
        struct[0].Gy[148,151] = v_B4lv_c_i - v_B4lv_n_i
        struct[0].Gy[149,16] = -i_load_B4lv_a_i
        struct[0].Gy[149,17] = i_load_B4lv_a_r
        struct[0].Gy[149,22] = i_load_B4lv_a_i
        struct[0].Gy[149,23] = -i_load_B4lv_a_r
        struct[0].Gy[149,146] = v_B4lv_a_i - v_B4lv_n_i
        struct[0].Gy[149,147] = -v_B4lv_a_r + v_B4lv_n_r
        struct[0].Gy[150,18] = -i_load_B4lv_b_i
        struct[0].Gy[150,19] = i_load_B4lv_b_r
        struct[0].Gy[150,22] = i_load_B4lv_b_i
        struct[0].Gy[150,23] = -i_load_B4lv_b_r
        struct[0].Gy[150,148] = v_B4lv_b_i - v_B4lv_n_i
        struct[0].Gy[150,149] = -v_B4lv_b_r + v_B4lv_n_r
        struct[0].Gy[151,20] = -i_load_B4lv_c_i
        struct[0].Gy[151,21] = i_load_B4lv_c_r
        struct[0].Gy[151,22] = i_load_B4lv_c_i
        struct[0].Gy[151,23] = -i_load_B4lv_c_r
        struct[0].Gy[151,150] = v_B4lv_c_i - v_B4lv_n_i
        struct[0].Gy[151,151] = -v_B4lv_c_r + v_B4lv_n_r
        struct[0].Gy[154,24] = i_load_B5lv_a_r
        struct[0].Gy[154,25] = i_load_B5lv_a_i
        struct[0].Gy[154,30] = -i_load_B5lv_a_r
        struct[0].Gy[154,31] = -i_load_B5lv_a_i
        struct[0].Gy[154,154] = v_B5lv_a_r - v_B5lv_n_r
        struct[0].Gy[154,155] = v_B5lv_a_i - v_B5lv_n_i
        struct[0].Gy[155,26] = i_load_B5lv_b_r
        struct[0].Gy[155,27] = i_load_B5lv_b_i
        struct[0].Gy[155,30] = -i_load_B5lv_b_r
        struct[0].Gy[155,31] = -i_load_B5lv_b_i
        struct[0].Gy[155,156] = v_B5lv_b_r - v_B5lv_n_r
        struct[0].Gy[155,157] = v_B5lv_b_i - v_B5lv_n_i
        struct[0].Gy[156,28] = i_load_B5lv_c_r
        struct[0].Gy[156,29] = i_load_B5lv_c_i
        struct[0].Gy[156,30] = -i_load_B5lv_c_r
        struct[0].Gy[156,31] = -i_load_B5lv_c_i
        struct[0].Gy[156,158] = v_B5lv_c_r - v_B5lv_n_r
        struct[0].Gy[156,159] = v_B5lv_c_i - v_B5lv_n_i
        struct[0].Gy[157,24] = -i_load_B5lv_a_i
        struct[0].Gy[157,25] = i_load_B5lv_a_r
        struct[0].Gy[157,30] = i_load_B5lv_a_i
        struct[0].Gy[157,31] = -i_load_B5lv_a_r
        struct[0].Gy[157,154] = v_B5lv_a_i - v_B5lv_n_i
        struct[0].Gy[157,155] = -v_B5lv_a_r + v_B5lv_n_r
        struct[0].Gy[158,26] = -i_load_B5lv_b_i
        struct[0].Gy[158,27] = i_load_B5lv_b_r
        struct[0].Gy[158,30] = i_load_B5lv_b_i
        struct[0].Gy[158,31] = -i_load_B5lv_b_r
        struct[0].Gy[158,156] = v_B5lv_b_i - v_B5lv_n_i
        struct[0].Gy[158,157] = -v_B5lv_b_r + v_B5lv_n_r
        struct[0].Gy[159,28] = -i_load_B5lv_c_i
        struct[0].Gy[159,29] = i_load_B5lv_c_r
        struct[0].Gy[159,30] = i_load_B5lv_c_i
        struct[0].Gy[159,31] = -i_load_B5lv_c_r
        struct[0].Gy[159,158] = v_B5lv_c_i - v_B5lv_n_i
        struct[0].Gy[159,159] = -v_B5lv_c_r + v_B5lv_n_r
        struct[0].Gy[162,32] = i_load_B6lv_a_r
        struct[0].Gy[162,33] = i_load_B6lv_a_i
        struct[0].Gy[162,38] = -i_load_B6lv_a_r
        struct[0].Gy[162,39] = -i_load_B6lv_a_i
        struct[0].Gy[162,162] = v_B6lv_a_r - v_B6lv_n_r
        struct[0].Gy[162,163] = v_B6lv_a_i - v_B6lv_n_i
        struct[0].Gy[163,34] = i_load_B6lv_b_r
        struct[0].Gy[163,35] = i_load_B6lv_b_i
        struct[0].Gy[163,38] = -i_load_B6lv_b_r
        struct[0].Gy[163,39] = -i_load_B6lv_b_i
        struct[0].Gy[163,164] = v_B6lv_b_r - v_B6lv_n_r
        struct[0].Gy[163,165] = v_B6lv_b_i - v_B6lv_n_i
        struct[0].Gy[164,36] = i_load_B6lv_c_r
        struct[0].Gy[164,37] = i_load_B6lv_c_i
        struct[0].Gy[164,38] = -i_load_B6lv_c_r
        struct[0].Gy[164,39] = -i_load_B6lv_c_i
        struct[0].Gy[164,166] = v_B6lv_c_r - v_B6lv_n_r
        struct[0].Gy[164,167] = v_B6lv_c_i - v_B6lv_n_i
        struct[0].Gy[165,32] = -i_load_B6lv_a_i
        struct[0].Gy[165,33] = i_load_B6lv_a_r
        struct[0].Gy[165,38] = i_load_B6lv_a_i
        struct[0].Gy[165,39] = -i_load_B6lv_a_r
        struct[0].Gy[165,162] = v_B6lv_a_i - v_B6lv_n_i
        struct[0].Gy[165,163] = -v_B6lv_a_r + v_B6lv_n_r
        struct[0].Gy[166,34] = -i_load_B6lv_b_i
        struct[0].Gy[166,35] = i_load_B6lv_b_r
        struct[0].Gy[166,38] = i_load_B6lv_b_i
        struct[0].Gy[166,39] = -i_load_B6lv_b_r
        struct[0].Gy[166,164] = v_B6lv_b_i - v_B6lv_n_i
        struct[0].Gy[166,165] = -v_B6lv_b_r + v_B6lv_n_r
        struct[0].Gy[167,36] = -i_load_B6lv_c_i
        struct[0].Gy[167,37] = i_load_B6lv_c_r
        struct[0].Gy[167,38] = i_load_B6lv_c_i
        struct[0].Gy[167,39] = -i_load_B6lv_c_r
        struct[0].Gy[167,166] = v_B6lv_c_i - v_B6lv_n_i
        struct[0].Gy[167,167] = -v_B6lv_c_r + v_B6lv_n_r
        struct[0].Gy[170,170] = -R_1112
        struct[0].Gy[171,171] = -R_1213
        struct[0].Gy[172,172] = -R_1314
        struct[0].Gy[173,173] = -R_1415
        struct[0].Gy[174,174] = -R_2122
        struct[0].Gy[175,175] = -R_2223
        struct[0].Gy[176,176] = -R_2324
        struct[0].Gy[177,177] = -R_2425
        struct[0].Gy[178,178] = -R_3132
        struct[0].Gy[179,179] = -R_3233
        struct[0].Gy[180,180] = -R_3334
        struct[0].Gy[181,181] = -R_3435
        struct[0].Gy[182,182] = -R_4142
        struct[0].Gy[183,183] = -R_4243
        struct[0].Gy[184,184] = -R_4344
        struct[0].Gy[185,185] = -R_4445
        struct[0].Gy[186,186] = -R_5152
        struct[0].Gy[187,187] = -R_5253
        struct[0].Gy[188,188] = -R_5354
        struct[0].Gy[189,189] = -R_5455
        struct[0].Gy[190,190] = -R_1521
        struct[0].Gy[191,191] = -R_2531
        struct[0].Gy[192,192] = -R_3541
        struct[0].Gy[193,193] = -R_4551
        struct[0].Gy[219,194] = i_11
        struct[0].Gy[219,219] = v_11
        struct[0].Gy[220,195] = i_12
        struct[0].Gy[220,220] = v_12
        struct[0].Gy[221,197] = i_14
        struct[0].Gy[221,221] = v_14
        struct[0].Gy[222,198] = i_15
        struct[0].Gy[222,222] = v_15
        struct[0].Gy[223,199] = i_21
        struct[0].Gy[223,223] = v_21
        struct[0].Gy[224,200] = i_22
        struct[0].Gy[224,224] = v_22
        struct[0].Gy[225,202] = i_24
        struct[0].Gy[225,225] = v_24
        struct[0].Gy[226,203] = i_25
        struct[0].Gy[226,226] = v_25
        struct[0].Gy[227,204] = i_31
        struct[0].Gy[227,227] = v_31
        struct[0].Gy[228,205] = i_32
        struct[0].Gy[228,228] = v_32
        struct[0].Gy[229,207] = i_34
        struct[0].Gy[229,229] = v_34
        struct[0].Gy[230,208] = i_35
        struct[0].Gy[230,230] = v_35
        struct[0].Gy[231,209] = i_41
        struct[0].Gy[231,231] = v_41
        struct[0].Gy[232,210] = i_42
        struct[0].Gy[232,232] = v_42
        struct[0].Gy[233,212] = i_44
        struct[0].Gy[233,233] = v_44
        struct[0].Gy[234,213] = i_45
        struct[0].Gy[234,234] = v_45
        struct[0].Gy[235,214] = i_51
        struct[0].Gy[235,235] = v_51
        struct[0].Gy[236,215] = i_52
        struct[0].Gy[236,236] = v_52
        struct[0].Gy[237,217] = i_54
        struct[0].Gy[237,237] = v_54
        struct[0].Gy[238,218] = i_55
        struct[0].Gy[238,238] = v_55
        struct[0].Gy[239,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[239,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[239,196] = v_13/3
        struct[0].Gy[240,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[240,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[240,196] = v_13/3
        struct[0].Gy[241,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[241,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[241,196] = v_13/3
        struct[0].Gy[242,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[242,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[242,201] = v_23/3
        struct[0].Gy[243,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[243,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[243,201] = v_23/3
        struct[0].Gy[244,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[244,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[244,201] = v_23/3
        struct[0].Gy[245,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[245,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[245,206] = v_33/3
        struct[0].Gy[246,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[246,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[246,206] = v_33/3
        struct[0].Gy[247,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[247,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[247,206] = v_33/3
        struct[0].Gy[248,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[248,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[248,211] = v_43/3
        struct[0].Gy[249,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[249,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[249,211] = v_43/3
        struct[0].Gy[250,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[250,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[250,211] = v_43/3
        struct[0].Gy[251,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[251,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[251,216] = v_53/3
        struct[0].Gy[252,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[252,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[252,216] = v_53/3
        struct[0].Gy[253,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[253,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[253,216] = v_53/3

    if mode > 12:

        struct[0].Fu[0,68] = -1/T_v
        struct[0].Fu[0,73] = 1/T_v
        struct[0].Fu[0,74] = -(-Dv_r_13 - K_r*i_13*v_13 - v_13 + v_nom)/T_v**2
        struct[0].Fu[0,75] = -i_13*v_13/T_v
        struct[0].Fu[1,69] = -1/T_v
        struct[0].Fu[1,73] = 1/T_v
        struct[0].Fu[1,74] = -(-Dv_r_23 - K_r*i_23*v_23 - v_23 + v_nom)/T_v**2
        struct[0].Fu[1,75] = -i_23*v_23/T_v
        struct[0].Fu[2,70] = -1/T_v
        struct[0].Fu[2,73] = 1/T_v
        struct[0].Fu[2,74] = -(-Dv_r_33 - K_r*i_33*v_33 - v_33 + v_nom)/T_v**2
        struct[0].Fu[2,75] = -i_33*v_33/T_v
        struct[0].Fu[3,71] = -1/T_v
        struct[0].Fu[3,73] = 1/T_v
        struct[0].Fu[3,74] = -(-Dv_r_43 - K_r*i_43*v_43 - v_43 + v_nom)/T_v**2
        struct[0].Fu[3,75] = -i_43*v_43/T_v
        struct[0].Fu[4,72] = -1/T_v
        struct[0].Fu[4,73] = 1/T_v
        struct[0].Fu[4,74] = -(-Dv_r_53 - K_r*i_53*v_53 - v_53 + v_nom)/T_v**2
        struct[0].Fu[4,75] = -i_53*v_53/T_v

        struct[0].Gu[239,76] = Piecewise(np.array([(6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[239,77] = Piecewise(np.array([(6*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[239,78] = Piecewise(np.array([(6*i_load_B2lv_a_i**2 + 6*i_load_B2lv_a_r**2 + 6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[240,76] = Piecewise(np.array([(6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[240,77] = Piecewise(np.array([(6*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[240,78] = Piecewise(np.array([(6*i_load_B2lv_a_i**2 + 6*i_load_B2lv_a_r**2 + 6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[241,76] = Piecewise(np.array([(6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[241,77] = Piecewise(np.array([(6*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[241,78] = Piecewise(np.array([(6*i_load_B2lv_a_i**2 + 6*i_load_B2lv_a_r**2 + 6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[242,79] = Piecewise(np.array([(6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[242,80] = Piecewise(np.array([(6*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[242,81] = Piecewise(np.array([(6*i_load_B3lv_a_i**2 + 6*i_load_B3lv_a_r**2 + 6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[243,79] = Piecewise(np.array([(6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[243,80] = Piecewise(np.array([(6*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[243,81] = Piecewise(np.array([(6*i_load_B3lv_a_i**2 + 6*i_load_B3lv_a_r**2 + 6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[244,79] = Piecewise(np.array([(6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[244,80] = Piecewise(np.array([(6*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[244,81] = Piecewise(np.array([(6*i_load_B3lv_a_i**2 + 6*i_load_B3lv_a_r**2 + 6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[245,82] = Piecewise(np.array([(6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[245,83] = Piecewise(np.array([(6*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[245,84] = Piecewise(np.array([(6*i_load_B4lv_a_i**2 + 6*i_load_B4lv_a_r**2 + 6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[246,82] = Piecewise(np.array([(6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[246,83] = Piecewise(np.array([(6*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[246,84] = Piecewise(np.array([(6*i_load_B4lv_a_i**2 + 6*i_load_B4lv_a_r**2 + 6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[247,82] = Piecewise(np.array([(6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[247,83] = Piecewise(np.array([(6*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[247,84] = Piecewise(np.array([(6*i_load_B4lv_a_i**2 + 6*i_load_B4lv_a_r**2 + 6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[248,85] = Piecewise(np.array([(6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[248,86] = Piecewise(np.array([(6*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[248,87] = Piecewise(np.array([(6*i_load_B5lv_a_i**2 + 6*i_load_B5lv_a_r**2 + 6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[249,85] = Piecewise(np.array([(6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[249,86] = Piecewise(np.array([(6*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[249,87] = Piecewise(np.array([(6*i_load_B5lv_a_i**2 + 6*i_load_B5lv_a_r**2 + 6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[250,85] = Piecewise(np.array([(6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[250,86] = Piecewise(np.array([(6*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[250,87] = Piecewise(np.array([(6*i_load_B5lv_a_i**2 + 6*i_load_B5lv_a_r**2 + 6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[251,88] = Piecewise(np.array([(6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[251,89] = Piecewise(np.array([(6*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[251,90] = Piecewise(np.array([(6*i_load_B6lv_a_i**2 + 6*i_load_B6lv_a_r**2 + 6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[252,88] = Piecewise(np.array([(6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[252,89] = Piecewise(np.array([(6*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[252,90] = Piecewise(np.array([(6*i_load_B6lv_a_i**2 + 6*i_load_B6lv_a_r**2 + 6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[253,88] = Piecewise(np.array([(6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[253,89] = Piecewise(np.array([(6*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[253,90] = Piecewise(np.array([(6*i_load_B6lv_a_i**2 + 6*i_load_B6lv_a_r**2 + 6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3

        struct[0].Hx[0,0] = i_13
        struct[0].Hx[1,0] = 1
        struct[0].Hx[2,1] = i_23
        struct[0].Hx[3,1] = 1
        struct[0].Hx[4,2] = i_33
        struct[0].Hx[5,2] = 1
        struct[0].Hx[6,3] = i_43
        struct[0].Hx[7,3] = 1
        struct[0].Hx[8,4] = i_53
        struct[0].Hx[9,4] = 1

        struct[0].Hy[0,196] = v_13
        struct[0].Hy[2,201] = v_23
        struct[0].Hy[4,206] = v_33
        struct[0].Hy[6,211] = v_43
        struct[0].Hy[8,216] = v_53




def ini_nn(struct,mode):

    # Parameters:
    R_1112 = struct[0].R_1112
    R_1213 = struct[0].R_1213
    R_1314 = struct[0].R_1314
    R_1415 = struct[0].R_1415
    R_1521 = struct[0].R_1521
    R_2122 = struct[0].R_2122
    R_2223 = struct[0].R_2223
    R_2324 = struct[0].R_2324
    R_2425 = struct[0].R_2425
    R_2531 = struct[0].R_2531
    R_3132 = struct[0].R_3132
    R_3233 = struct[0].R_3233
    R_3334 = struct[0].R_3334
    R_3435 = struct[0].R_3435
    R_3541 = struct[0].R_3541
    R_4142 = struct[0].R_4142
    R_4243 = struct[0].R_4243
    R_4344 = struct[0].R_4344
    R_4445 = struct[0].R_4445
    R_4551 = struct[0].R_4551
    R_5152 = struct[0].R_5152
    R_5253 = struct[0].R_5253
    R_5354 = struct[0].R_5354
    R_5455 = struct[0].R_5455
    p_11 = struct[0].p_11
    p_12 = struct[0].p_12
    p_14 = struct[0].p_14
    p_15 = struct[0].p_15
    p_21 = struct[0].p_21
    p_22 = struct[0].p_22
    p_24 = struct[0].p_24
    p_25 = struct[0].p_25
    p_31 = struct[0].p_31
    p_32 = struct[0].p_32
    p_34 = struct[0].p_34
    p_35 = struct[0].p_35
    p_41 = struct[0].p_41
    p_42 = struct[0].p_42
    p_44 = struct[0].p_44
    p_45 = struct[0].p_45
    p_51 = struct[0].p_51
    p_52 = struct[0].p_52
    p_54 = struct[0].p_54
    p_55 = struct[0].p_55
    
    # Inputs:
    v_B1_a_r = struct[0].v_B1_a_r
    v_B1_a_i = struct[0].v_B1_a_i
    v_B1_b_r = struct[0].v_B1_b_r
    v_B1_b_i = struct[0].v_B1_b_i
    v_B1_c_r = struct[0].v_B1_c_r
    v_B1_c_i = struct[0].v_B1_c_i
    v_B7_a_r = struct[0].v_B7_a_r
    v_B7_a_i = struct[0].v_B7_a_i
    v_B7_b_r = struct[0].v_B7_b_r
    v_B7_b_i = struct[0].v_B7_b_i
    v_B7_c_r = struct[0].v_B7_c_r
    v_B7_c_i = struct[0].v_B7_c_i
    i_B2lv_n_r = struct[0].i_B2lv_n_r
    i_B2lv_n_i = struct[0].i_B2lv_n_i
    i_B3lv_n_r = struct[0].i_B3lv_n_r
    i_B3lv_n_i = struct[0].i_B3lv_n_i
    i_B4lv_n_r = struct[0].i_B4lv_n_r
    i_B4lv_n_i = struct[0].i_B4lv_n_i
    i_B5lv_n_r = struct[0].i_B5lv_n_r
    i_B5lv_n_i = struct[0].i_B5lv_n_i
    i_B6lv_n_r = struct[0].i_B6lv_n_r
    i_B6lv_n_i = struct[0].i_B6lv_n_i
    i_B2_a_r = struct[0].i_B2_a_r
    i_B2_a_i = struct[0].i_B2_a_i
    i_B2_b_r = struct[0].i_B2_b_r
    i_B2_b_i = struct[0].i_B2_b_i
    i_B2_c_r = struct[0].i_B2_c_r
    i_B2_c_i = struct[0].i_B2_c_i
    i_B3_a_r = struct[0].i_B3_a_r
    i_B3_a_i = struct[0].i_B3_a_i
    i_B3_b_r = struct[0].i_B3_b_r
    i_B3_b_i = struct[0].i_B3_b_i
    i_B3_c_r = struct[0].i_B3_c_r
    i_B3_c_i = struct[0].i_B3_c_i
    i_B4_a_r = struct[0].i_B4_a_r
    i_B4_a_i = struct[0].i_B4_a_i
    i_B4_b_r = struct[0].i_B4_b_r
    i_B4_b_i = struct[0].i_B4_b_i
    i_B4_c_r = struct[0].i_B4_c_r
    i_B4_c_i = struct[0].i_B4_c_i
    i_B5_a_r = struct[0].i_B5_a_r
    i_B5_a_i = struct[0].i_B5_a_i
    i_B5_b_r = struct[0].i_B5_b_r
    i_B5_b_i = struct[0].i_B5_b_i
    i_B5_c_r = struct[0].i_B5_c_r
    i_B5_c_i = struct[0].i_B5_c_i
    i_B6_a_r = struct[0].i_B6_a_r
    i_B6_a_i = struct[0].i_B6_a_i
    i_B6_b_r = struct[0].i_B6_b_r
    i_B6_b_i = struct[0].i_B6_b_i
    i_B6_c_r = struct[0].i_B6_c_r
    i_B6_c_i = struct[0].i_B6_c_i
    q_B2lv_a = struct[0].q_B2lv_a
    q_B2lv_b = struct[0].q_B2lv_b
    q_B2lv_c = struct[0].q_B2lv_c
    q_B3lv_a = struct[0].q_B3lv_a
    q_B3lv_b = struct[0].q_B3lv_b
    q_B3lv_c = struct[0].q_B3lv_c
    q_B4lv_a = struct[0].q_B4lv_a
    q_B4lv_b = struct[0].q_B4lv_b
    q_B4lv_c = struct[0].q_B4lv_c
    q_B5lv_a = struct[0].q_B5lv_a
    q_B5lv_b = struct[0].q_B5lv_b
    q_B5lv_c = struct[0].q_B5lv_c
    q_B6lv_a = struct[0].q_B6lv_a
    q_B6lv_b = struct[0].q_B6lv_b
    q_B6lv_c = struct[0].q_B6lv_c
    u_dummy = struct[0].u_dummy
    Dv_r_13 = struct[0].Dv_r_13
    Dv_r_23 = struct[0].Dv_r_23
    Dv_r_33 = struct[0].Dv_r_33
    Dv_r_43 = struct[0].Dv_r_43
    Dv_r_53 = struct[0].Dv_r_53
    v_nom = struct[0].v_nom
    T_v = struct[0].T_v
    K_r = struct[0].K_r
    a_B2lv = struct[0].a_B2lv
    b_B2lv = struct[0].b_B2lv
    c_B2lv = struct[0].c_B2lv
    a_B3lv = struct[0].a_B3lv
    b_B3lv = struct[0].b_B3lv
    c_B3lv = struct[0].c_B3lv
    a_B4lv = struct[0].a_B4lv
    b_B4lv = struct[0].b_B4lv
    c_B4lv = struct[0].c_B4lv
    a_B5lv = struct[0].a_B5lv
    b_B5lv = struct[0].b_B5lv
    c_B5lv = struct[0].c_B5lv
    a_B6lv = struct[0].a_B6lv
    b_B6lv = struct[0].b_B6lv
    c_B6lv = struct[0].c_B6lv
    
    # Dynamical states:
    v_13 = struct[0].x[0,0]
    v_23 = struct[0].x[1,0]
    v_33 = struct[0].x[2,0]
    v_43 = struct[0].x[3,0]
    v_53 = struct[0].x[4,0]
    
    # Algebraic states:
    v_B2lv_a_r = struct[0].y_ini[0,0]
    v_B2lv_a_i = struct[0].y_ini[1,0]
    v_B2lv_b_r = struct[0].y_ini[2,0]
    v_B2lv_b_i = struct[0].y_ini[3,0]
    v_B2lv_c_r = struct[0].y_ini[4,0]
    v_B2lv_c_i = struct[0].y_ini[5,0]
    v_B2lv_n_r = struct[0].y_ini[6,0]
    v_B2lv_n_i = struct[0].y_ini[7,0]
    v_B3lv_a_r = struct[0].y_ini[8,0]
    v_B3lv_a_i = struct[0].y_ini[9,0]
    v_B3lv_b_r = struct[0].y_ini[10,0]
    v_B3lv_b_i = struct[0].y_ini[11,0]
    v_B3lv_c_r = struct[0].y_ini[12,0]
    v_B3lv_c_i = struct[0].y_ini[13,0]
    v_B3lv_n_r = struct[0].y_ini[14,0]
    v_B3lv_n_i = struct[0].y_ini[15,0]
    v_B4lv_a_r = struct[0].y_ini[16,0]
    v_B4lv_a_i = struct[0].y_ini[17,0]
    v_B4lv_b_r = struct[0].y_ini[18,0]
    v_B4lv_b_i = struct[0].y_ini[19,0]
    v_B4lv_c_r = struct[0].y_ini[20,0]
    v_B4lv_c_i = struct[0].y_ini[21,0]
    v_B4lv_n_r = struct[0].y_ini[22,0]
    v_B4lv_n_i = struct[0].y_ini[23,0]
    v_B5lv_a_r = struct[0].y_ini[24,0]
    v_B5lv_a_i = struct[0].y_ini[25,0]
    v_B5lv_b_r = struct[0].y_ini[26,0]
    v_B5lv_b_i = struct[0].y_ini[27,0]
    v_B5lv_c_r = struct[0].y_ini[28,0]
    v_B5lv_c_i = struct[0].y_ini[29,0]
    v_B5lv_n_r = struct[0].y_ini[30,0]
    v_B5lv_n_i = struct[0].y_ini[31,0]
    v_B6lv_a_r = struct[0].y_ini[32,0]
    v_B6lv_a_i = struct[0].y_ini[33,0]
    v_B6lv_b_r = struct[0].y_ini[34,0]
    v_B6lv_b_i = struct[0].y_ini[35,0]
    v_B6lv_c_r = struct[0].y_ini[36,0]
    v_B6lv_c_i = struct[0].y_ini[37,0]
    v_B6lv_n_r = struct[0].y_ini[38,0]
    v_B6lv_n_i = struct[0].y_ini[39,0]
    v_B2_a_r = struct[0].y_ini[40,0]
    v_B2_a_i = struct[0].y_ini[41,0]
    v_B2_b_r = struct[0].y_ini[42,0]
    v_B2_b_i = struct[0].y_ini[43,0]
    v_B2_c_r = struct[0].y_ini[44,0]
    v_B2_c_i = struct[0].y_ini[45,0]
    v_B3_a_r = struct[0].y_ini[46,0]
    v_B3_a_i = struct[0].y_ini[47,0]
    v_B3_b_r = struct[0].y_ini[48,0]
    v_B3_b_i = struct[0].y_ini[49,0]
    v_B3_c_r = struct[0].y_ini[50,0]
    v_B3_c_i = struct[0].y_ini[51,0]
    v_B4_a_r = struct[0].y_ini[52,0]
    v_B4_a_i = struct[0].y_ini[53,0]
    v_B4_b_r = struct[0].y_ini[54,0]
    v_B4_b_i = struct[0].y_ini[55,0]
    v_B4_c_r = struct[0].y_ini[56,0]
    v_B4_c_i = struct[0].y_ini[57,0]
    v_B5_a_r = struct[0].y_ini[58,0]
    v_B5_a_i = struct[0].y_ini[59,0]
    v_B5_b_r = struct[0].y_ini[60,0]
    v_B5_b_i = struct[0].y_ini[61,0]
    v_B5_c_r = struct[0].y_ini[62,0]
    v_B5_c_i = struct[0].y_ini[63,0]
    v_B6_a_r = struct[0].y_ini[64,0]
    v_B6_a_i = struct[0].y_ini[65,0]
    v_B6_b_r = struct[0].y_ini[66,0]
    v_B6_b_i = struct[0].y_ini[67,0]
    v_B6_c_r = struct[0].y_ini[68,0]
    v_B6_c_i = struct[0].y_ini[69,0]
    i_t_B2_B2lv_a_r = struct[0].y_ini[70,0]
    i_t_B2_B2lv_a_i = struct[0].y_ini[71,0]
    i_t_B2_B2lv_b_r = struct[0].y_ini[72,0]
    i_t_B2_B2lv_b_i = struct[0].y_ini[73,0]
    i_t_B2_B2lv_c_r = struct[0].y_ini[74,0]
    i_t_B2_B2lv_c_i = struct[0].y_ini[75,0]
    i_t_B3_B3lv_a_r = struct[0].y_ini[76,0]
    i_t_B3_B3lv_a_i = struct[0].y_ini[77,0]
    i_t_B3_B3lv_b_r = struct[0].y_ini[78,0]
    i_t_B3_B3lv_b_i = struct[0].y_ini[79,0]
    i_t_B3_B3lv_c_r = struct[0].y_ini[80,0]
    i_t_B3_B3lv_c_i = struct[0].y_ini[81,0]
    i_t_B4_B4lv_a_r = struct[0].y_ini[82,0]
    i_t_B4_B4lv_a_i = struct[0].y_ini[83,0]
    i_t_B4_B4lv_b_r = struct[0].y_ini[84,0]
    i_t_B4_B4lv_b_i = struct[0].y_ini[85,0]
    i_t_B4_B4lv_c_r = struct[0].y_ini[86,0]
    i_t_B4_B4lv_c_i = struct[0].y_ini[87,0]
    i_t_B5_B5lv_a_r = struct[0].y_ini[88,0]
    i_t_B5_B5lv_a_i = struct[0].y_ini[89,0]
    i_t_B5_B5lv_b_r = struct[0].y_ini[90,0]
    i_t_B5_B5lv_b_i = struct[0].y_ini[91,0]
    i_t_B5_B5lv_c_r = struct[0].y_ini[92,0]
    i_t_B5_B5lv_c_i = struct[0].y_ini[93,0]
    i_t_B6_B6lv_a_r = struct[0].y_ini[94,0]
    i_t_B6_B6lv_a_i = struct[0].y_ini[95,0]
    i_t_B6_B6lv_b_r = struct[0].y_ini[96,0]
    i_t_B6_B6lv_b_i = struct[0].y_ini[97,0]
    i_t_B6_B6lv_c_r = struct[0].y_ini[98,0]
    i_t_B6_B6lv_c_i = struct[0].y_ini[99,0]
    i_l_B1_B2_a_r = struct[0].y_ini[100,0]
    i_l_B1_B2_a_i = struct[0].y_ini[101,0]
    i_l_B1_B2_b_r = struct[0].y_ini[102,0]
    i_l_B1_B2_b_i = struct[0].y_ini[103,0]
    i_l_B1_B2_c_r = struct[0].y_ini[104,0]
    i_l_B1_B2_c_i = struct[0].y_ini[105,0]
    i_l_B2_B3_a_r = struct[0].y_ini[106,0]
    i_l_B2_B3_a_i = struct[0].y_ini[107,0]
    i_l_B2_B3_b_r = struct[0].y_ini[108,0]
    i_l_B2_B3_b_i = struct[0].y_ini[109,0]
    i_l_B2_B3_c_r = struct[0].y_ini[110,0]
    i_l_B2_B3_c_i = struct[0].y_ini[111,0]
    i_l_B3_B4_a_r = struct[0].y_ini[112,0]
    i_l_B3_B4_a_i = struct[0].y_ini[113,0]
    i_l_B3_B4_b_r = struct[0].y_ini[114,0]
    i_l_B3_B4_b_i = struct[0].y_ini[115,0]
    i_l_B3_B4_c_r = struct[0].y_ini[116,0]
    i_l_B3_B4_c_i = struct[0].y_ini[117,0]
    i_l_B5_B6_a_r = struct[0].y_ini[118,0]
    i_l_B5_B6_a_i = struct[0].y_ini[119,0]
    i_l_B5_B6_b_r = struct[0].y_ini[120,0]
    i_l_B5_B6_b_i = struct[0].y_ini[121,0]
    i_l_B5_B6_c_r = struct[0].y_ini[122,0]
    i_l_B5_B6_c_i = struct[0].y_ini[123,0]
    i_l_B6_B7_a_r = struct[0].y_ini[124,0]
    i_l_B6_B7_a_i = struct[0].y_ini[125,0]
    i_l_B6_B7_b_r = struct[0].y_ini[126,0]
    i_l_B6_B7_b_i = struct[0].y_ini[127,0]
    i_l_B6_B7_c_r = struct[0].y_ini[128,0]
    i_l_B6_B7_c_i = struct[0].y_ini[129,0]
    i_load_B2lv_a_r = struct[0].y_ini[130,0]
    i_load_B2lv_a_i = struct[0].y_ini[131,0]
    i_load_B2lv_b_r = struct[0].y_ini[132,0]
    i_load_B2lv_b_i = struct[0].y_ini[133,0]
    i_load_B2lv_c_r = struct[0].y_ini[134,0]
    i_load_B2lv_c_i = struct[0].y_ini[135,0]
    i_load_B2lv_n_r = struct[0].y_ini[136,0]
    i_load_B2lv_n_i = struct[0].y_ini[137,0]
    i_load_B3lv_a_r = struct[0].y_ini[138,0]
    i_load_B3lv_a_i = struct[0].y_ini[139,0]
    i_load_B3lv_b_r = struct[0].y_ini[140,0]
    i_load_B3lv_b_i = struct[0].y_ini[141,0]
    i_load_B3lv_c_r = struct[0].y_ini[142,0]
    i_load_B3lv_c_i = struct[0].y_ini[143,0]
    i_load_B3lv_n_r = struct[0].y_ini[144,0]
    i_load_B3lv_n_i = struct[0].y_ini[145,0]
    i_load_B4lv_a_r = struct[0].y_ini[146,0]
    i_load_B4lv_a_i = struct[0].y_ini[147,0]
    i_load_B4lv_b_r = struct[0].y_ini[148,0]
    i_load_B4lv_b_i = struct[0].y_ini[149,0]
    i_load_B4lv_c_r = struct[0].y_ini[150,0]
    i_load_B4lv_c_i = struct[0].y_ini[151,0]
    i_load_B4lv_n_r = struct[0].y_ini[152,0]
    i_load_B4lv_n_i = struct[0].y_ini[153,0]
    i_load_B5lv_a_r = struct[0].y_ini[154,0]
    i_load_B5lv_a_i = struct[0].y_ini[155,0]
    i_load_B5lv_b_r = struct[0].y_ini[156,0]
    i_load_B5lv_b_i = struct[0].y_ini[157,0]
    i_load_B5lv_c_r = struct[0].y_ini[158,0]
    i_load_B5lv_c_i = struct[0].y_ini[159,0]
    i_load_B5lv_n_r = struct[0].y_ini[160,0]
    i_load_B5lv_n_i = struct[0].y_ini[161,0]
    i_load_B6lv_a_r = struct[0].y_ini[162,0]
    i_load_B6lv_a_i = struct[0].y_ini[163,0]
    i_load_B6lv_b_r = struct[0].y_ini[164,0]
    i_load_B6lv_b_i = struct[0].y_ini[165,0]
    i_load_B6lv_c_r = struct[0].y_ini[166,0]
    i_load_B6lv_c_i = struct[0].y_ini[167,0]
    i_load_B6lv_n_r = struct[0].y_ini[168,0]
    i_load_B6lv_n_i = struct[0].y_ini[169,0]
    i_l_1112 = struct[0].y_ini[170,0]
    i_l_1213 = struct[0].y_ini[171,0]
    i_l_1314 = struct[0].y_ini[172,0]
    i_l_1415 = struct[0].y_ini[173,0]
    i_l_2122 = struct[0].y_ini[174,0]
    i_l_2223 = struct[0].y_ini[175,0]
    i_l_2324 = struct[0].y_ini[176,0]
    i_l_2425 = struct[0].y_ini[177,0]
    i_l_3132 = struct[0].y_ini[178,0]
    i_l_3233 = struct[0].y_ini[179,0]
    i_l_3334 = struct[0].y_ini[180,0]
    i_l_3435 = struct[0].y_ini[181,0]
    i_l_4142 = struct[0].y_ini[182,0]
    i_l_4243 = struct[0].y_ini[183,0]
    i_l_4344 = struct[0].y_ini[184,0]
    i_l_4445 = struct[0].y_ini[185,0]
    i_l_5152 = struct[0].y_ini[186,0]
    i_l_5253 = struct[0].y_ini[187,0]
    i_l_5354 = struct[0].y_ini[188,0]
    i_l_5455 = struct[0].y_ini[189,0]
    i_l_1521 = struct[0].y_ini[190,0]
    i_l_2531 = struct[0].y_ini[191,0]
    i_l_3541 = struct[0].y_ini[192,0]
    i_l_4551 = struct[0].y_ini[193,0]
    v_11 = struct[0].y_ini[194,0]
    v_12 = struct[0].y_ini[195,0]
    i_13 = struct[0].y_ini[196,0]
    v_14 = struct[0].y_ini[197,0]
    v_15 = struct[0].y_ini[198,0]
    v_21 = struct[0].y_ini[199,0]
    v_22 = struct[0].y_ini[200,0]
    i_23 = struct[0].y_ini[201,0]
    v_24 = struct[0].y_ini[202,0]
    v_25 = struct[0].y_ini[203,0]
    v_31 = struct[0].y_ini[204,0]
    v_32 = struct[0].y_ini[205,0]
    i_33 = struct[0].y_ini[206,0]
    v_34 = struct[0].y_ini[207,0]
    v_35 = struct[0].y_ini[208,0]
    v_41 = struct[0].y_ini[209,0]
    v_42 = struct[0].y_ini[210,0]
    i_43 = struct[0].y_ini[211,0]
    v_44 = struct[0].y_ini[212,0]
    v_45 = struct[0].y_ini[213,0]
    v_51 = struct[0].y_ini[214,0]
    v_52 = struct[0].y_ini[215,0]
    i_53 = struct[0].y_ini[216,0]
    v_54 = struct[0].y_ini[217,0]
    v_55 = struct[0].y_ini[218,0]
    i_11 = struct[0].y_ini[219,0]
    i_12 = struct[0].y_ini[220,0]
    i_14 = struct[0].y_ini[221,0]
    i_15 = struct[0].y_ini[222,0]
    i_21 = struct[0].y_ini[223,0]
    i_22 = struct[0].y_ini[224,0]
    i_24 = struct[0].y_ini[225,0]
    i_25 = struct[0].y_ini[226,0]
    i_31 = struct[0].y_ini[227,0]
    i_32 = struct[0].y_ini[228,0]
    i_34 = struct[0].y_ini[229,0]
    i_35 = struct[0].y_ini[230,0]
    i_41 = struct[0].y_ini[231,0]
    i_42 = struct[0].y_ini[232,0]
    i_44 = struct[0].y_ini[233,0]
    i_45 = struct[0].y_ini[234,0]
    i_51 = struct[0].y_ini[235,0]
    i_52 = struct[0].y_ini[236,0]
    i_54 = struct[0].y_ini[237,0]
    i_55 = struct[0].y_ini[238,0]
    p_B2lv_a = struct[0].y_ini[239,0]
    p_B2lv_b = struct[0].y_ini[240,0]
    p_B2lv_c = struct[0].y_ini[241,0]
    p_B3lv_a = struct[0].y_ini[242,0]
    p_B3lv_b = struct[0].y_ini[243,0]
    p_B3lv_c = struct[0].y_ini[244,0]
    p_B4lv_a = struct[0].y_ini[245,0]
    p_B4lv_b = struct[0].y_ini[246,0]
    p_B4lv_c = struct[0].y_ini[247,0]
    p_B5lv_a = struct[0].y_ini[248,0]
    p_B5lv_b = struct[0].y_ini[249,0]
    p_B5lv_c = struct[0].y_ini[250,0]
    p_B6lv_a = struct[0].y_ini[251,0]
    p_B6lv_b = struct[0].y_ini[252,0]
    p_B6lv_c = struct[0].y_ini[253,0]
    
    # Differential equations:
    if mode == 2:


        struct[0].f[0,0] = (-Dv_r_13 - K_r*i_13*v_13 - v_13 + v_nom)/T_v
        struct[0].f[1,0] = (-Dv_r_23 - K_r*i_23*v_23 - v_23 + v_nom)/T_v
        struct[0].f[2,0] = (-Dv_r_33 - K_r*i_33*v_33 - v_33 + v_nom)/T_v
        struct[0].f[3,0] = (-Dv_r_43 - K_r*i_43*v_43 - v_43 + v_nom)/T_v
        struct[0].f[4,0] = (-Dv_r_53 - K_r*i_53*v_53 - v_53 + v_nom)/T_v
    
    # Algebraic equations:
    if mode == 3:


        struct[0].g[0,0] = i_load_B2lv_a_r + 0.996212229189942*v_B2_a_i + 0.249053057297486*v_B2_a_r - 0.996212229189942*v_B2_c_i - 0.249053057297486*v_B2_c_r - 23.0065359477124*v_B2lv_a_i - 5.7516339869281*v_B2lv_a_r + 23.0065359477124*v_B2lv_n_i + 5.7516339869281*v_B2lv_n_r
        struct[0].g[1,0] = i_load_B2lv_a_i + 0.249053057297486*v_B2_a_i - 0.996212229189942*v_B2_a_r - 0.249053057297486*v_B2_c_i + 0.996212229189942*v_B2_c_r - 5.7516339869281*v_B2lv_a_i + 23.0065359477124*v_B2lv_a_r + 5.7516339869281*v_B2lv_n_i - 23.0065359477124*v_B2lv_n_r
        struct[0].g[2,0] = i_load_B2lv_b_r - 0.996212229189942*v_B2_a_i - 0.249053057297486*v_B2_a_r + 0.996212229189942*v_B2_b_i + 0.249053057297486*v_B2_b_r - 23.0065359477124*v_B2lv_b_i - 5.7516339869281*v_B2lv_b_r + 23.0065359477124*v_B2lv_n_i + 5.7516339869281*v_B2lv_n_r
        struct[0].g[3,0] = i_load_B2lv_b_i - 0.249053057297486*v_B2_a_i + 0.996212229189942*v_B2_a_r + 0.249053057297486*v_B2_b_i - 0.996212229189942*v_B2_b_r - 5.7516339869281*v_B2lv_b_i + 23.0065359477124*v_B2lv_b_r + 5.7516339869281*v_B2lv_n_i - 23.0065359477124*v_B2lv_n_r
        struct[0].g[4,0] = i_load_B2lv_c_r - 0.996212229189942*v_B2_b_i - 0.249053057297486*v_B2_b_r + 0.996212229189942*v_B2_c_i + 0.249053057297486*v_B2_c_r - 23.0065359477124*v_B2lv_c_i - 5.7516339869281*v_B2lv_c_r + 23.0065359477124*v_B2lv_n_i + 5.7516339869281*v_B2lv_n_r
        struct[0].g[5,0] = i_load_B2lv_c_i - 0.249053057297486*v_B2_b_i + 0.996212229189942*v_B2_b_r + 0.249053057297486*v_B2_c_i - 0.996212229189942*v_B2_c_r - 5.7516339869281*v_B2lv_c_i + 23.0065359477124*v_B2lv_c_r + 5.7516339869281*v_B2lv_n_i - 23.0065359477124*v_B2lv_n_r
        struct[0].g[6,0] = 23.0065359477124*v_B2lv_a_i + 5.7516339869281*v_B2lv_a_r + 23.0065359477124*v_B2lv_b_i + 5.7516339869281*v_B2lv_b_r + 23.0065359477124*v_B2lv_c_i + 5.7516339869281*v_B2lv_c_r - 69.0196078431372*v_B2lv_n_i - 1017.25490196078*v_B2lv_n_r
        struct[0].g[7,0] = 5.7516339869281*v_B2lv_a_i - 23.0065359477124*v_B2lv_a_r + 5.7516339869281*v_B2lv_b_i - 23.0065359477124*v_B2lv_b_r + 5.7516339869281*v_B2lv_c_i - 23.0065359477124*v_B2lv_c_r - 1017.25490196078*v_B2lv_n_i + 69.0196078431372*v_B2lv_n_r
        struct[0].g[8,0] = i_load_B3lv_a_r + 0.996212229189942*v_B3_a_i + 0.249053057297486*v_B3_a_r - 0.996212229189942*v_B3_c_i - 0.249053057297486*v_B3_c_r - 23.0065359477124*v_B3lv_a_i - 5.7516339869281*v_B3lv_a_r + 23.0065359477124*v_B3lv_n_i + 5.7516339869281*v_B3lv_n_r
        struct[0].g[9,0] = i_load_B3lv_a_i + 0.249053057297486*v_B3_a_i - 0.996212229189942*v_B3_a_r - 0.249053057297486*v_B3_c_i + 0.996212229189942*v_B3_c_r - 5.7516339869281*v_B3lv_a_i + 23.0065359477124*v_B3lv_a_r + 5.7516339869281*v_B3lv_n_i - 23.0065359477124*v_B3lv_n_r
        struct[0].g[10,0] = i_load_B3lv_b_r - 0.996212229189942*v_B3_a_i - 0.249053057297486*v_B3_a_r + 0.996212229189942*v_B3_b_i + 0.249053057297486*v_B3_b_r - 23.0065359477124*v_B3lv_b_i - 5.7516339869281*v_B3lv_b_r + 23.0065359477124*v_B3lv_n_i + 5.7516339869281*v_B3lv_n_r
        struct[0].g[11,0] = i_load_B3lv_b_i - 0.249053057297486*v_B3_a_i + 0.996212229189942*v_B3_a_r + 0.249053057297486*v_B3_b_i - 0.996212229189942*v_B3_b_r - 5.7516339869281*v_B3lv_b_i + 23.0065359477124*v_B3lv_b_r + 5.7516339869281*v_B3lv_n_i - 23.0065359477124*v_B3lv_n_r
        struct[0].g[12,0] = i_load_B3lv_c_r - 0.996212229189942*v_B3_b_i - 0.249053057297486*v_B3_b_r + 0.996212229189942*v_B3_c_i + 0.249053057297486*v_B3_c_r - 23.0065359477124*v_B3lv_c_i - 5.7516339869281*v_B3lv_c_r + 23.0065359477124*v_B3lv_n_i + 5.7516339869281*v_B3lv_n_r
        struct[0].g[13,0] = i_load_B3lv_c_i - 0.249053057297486*v_B3_b_i + 0.996212229189942*v_B3_b_r + 0.249053057297486*v_B3_c_i - 0.996212229189942*v_B3_c_r - 5.7516339869281*v_B3lv_c_i + 23.0065359477124*v_B3lv_c_r + 5.7516339869281*v_B3lv_n_i - 23.0065359477124*v_B3lv_n_r
        struct[0].g[14,0] = 23.0065359477124*v_B3lv_a_i + 5.7516339869281*v_B3lv_a_r + 23.0065359477124*v_B3lv_b_i + 5.7516339869281*v_B3lv_b_r + 23.0065359477124*v_B3lv_c_i + 5.7516339869281*v_B3lv_c_r - 69.0196078431372*v_B3lv_n_i - 1017.25490196078*v_B3lv_n_r
        struct[0].g[15,0] = 5.7516339869281*v_B3lv_a_i - 23.0065359477124*v_B3lv_a_r + 5.7516339869281*v_B3lv_b_i - 23.0065359477124*v_B3lv_b_r + 5.7516339869281*v_B3lv_c_i - 23.0065359477124*v_B3lv_c_r - 1017.25490196078*v_B3lv_n_i + 69.0196078431372*v_B3lv_n_r
        struct[0].g[16,0] = i_load_B4lv_a_r + 0.996212229189942*v_B4_a_i + 0.249053057297486*v_B4_a_r - 0.996212229189942*v_B4_c_i - 0.249053057297486*v_B4_c_r - 23.0065359477124*v_B4lv_a_i - 5.7516339869281*v_B4lv_a_r + 23.0065359477124*v_B4lv_n_i + 5.7516339869281*v_B4lv_n_r
        struct[0].g[17,0] = i_load_B4lv_a_i + 0.249053057297486*v_B4_a_i - 0.996212229189942*v_B4_a_r - 0.249053057297486*v_B4_c_i + 0.996212229189942*v_B4_c_r - 5.7516339869281*v_B4lv_a_i + 23.0065359477124*v_B4lv_a_r + 5.7516339869281*v_B4lv_n_i - 23.0065359477124*v_B4lv_n_r
        struct[0].g[18,0] = i_load_B4lv_b_r - 0.996212229189942*v_B4_a_i - 0.249053057297486*v_B4_a_r + 0.996212229189942*v_B4_b_i + 0.249053057297486*v_B4_b_r - 23.0065359477124*v_B4lv_b_i - 5.7516339869281*v_B4lv_b_r + 23.0065359477124*v_B4lv_n_i + 5.7516339869281*v_B4lv_n_r
        struct[0].g[19,0] = i_load_B4lv_b_i - 0.249053057297486*v_B4_a_i + 0.996212229189942*v_B4_a_r + 0.249053057297486*v_B4_b_i - 0.996212229189942*v_B4_b_r - 5.7516339869281*v_B4lv_b_i + 23.0065359477124*v_B4lv_b_r + 5.7516339869281*v_B4lv_n_i - 23.0065359477124*v_B4lv_n_r
        struct[0].g[20,0] = i_load_B4lv_c_r - 0.996212229189942*v_B4_b_i - 0.249053057297486*v_B4_b_r + 0.996212229189942*v_B4_c_i + 0.249053057297486*v_B4_c_r - 23.0065359477124*v_B4lv_c_i - 5.7516339869281*v_B4lv_c_r + 23.0065359477124*v_B4lv_n_i + 5.7516339869281*v_B4lv_n_r
        struct[0].g[21,0] = i_load_B4lv_c_i - 0.249053057297486*v_B4_b_i + 0.996212229189942*v_B4_b_r + 0.249053057297486*v_B4_c_i - 0.996212229189942*v_B4_c_r - 5.7516339869281*v_B4lv_c_i + 23.0065359477124*v_B4lv_c_r + 5.7516339869281*v_B4lv_n_i - 23.0065359477124*v_B4lv_n_r
        struct[0].g[22,0] = 23.0065359477124*v_B4lv_a_i + 5.7516339869281*v_B4lv_a_r + 23.0065359477124*v_B4lv_b_i + 5.7516339869281*v_B4lv_b_r + 23.0065359477124*v_B4lv_c_i + 5.7516339869281*v_B4lv_c_r - 69.0196078431372*v_B4lv_n_i - 1017.25490196078*v_B4lv_n_r
        struct[0].g[23,0] = 5.7516339869281*v_B4lv_a_i - 23.0065359477124*v_B4lv_a_r + 5.7516339869281*v_B4lv_b_i - 23.0065359477124*v_B4lv_b_r + 5.7516339869281*v_B4lv_c_i - 23.0065359477124*v_B4lv_c_r - 1017.25490196078*v_B4lv_n_i + 69.0196078431372*v_B4lv_n_r
        struct[0].g[24,0] = i_load_B5lv_a_r + 0.996212229189942*v_B5_a_i + 0.249053057297486*v_B5_a_r - 0.996212229189942*v_B5_c_i - 0.249053057297486*v_B5_c_r - 23.0065359477124*v_B5lv_a_i - 5.7516339869281*v_B5lv_a_r + 23.0065359477124*v_B5lv_n_i + 5.7516339869281*v_B5lv_n_r
        struct[0].g[25,0] = i_load_B5lv_a_i + 0.249053057297486*v_B5_a_i - 0.996212229189942*v_B5_a_r - 0.249053057297486*v_B5_c_i + 0.996212229189942*v_B5_c_r - 5.7516339869281*v_B5lv_a_i + 23.0065359477124*v_B5lv_a_r + 5.7516339869281*v_B5lv_n_i - 23.0065359477124*v_B5lv_n_r
        struct[0].g[26,0] = i_load_B5lv_b_r - 0.996212229189942*v_B5_a_i - 0.249053057297486*v_B5_a_r + 0.996212229189942*v_B5_b_i + 0.249053057297486*v_B5_b_r - 23.0065359477124*v_B5lv_b_i - 5.7516339869281*v_B5lv_b_r + 23.0065359477124*v_B5lv_n_i + 5.7516339869281*v_B5lv_n_r
        struct[0].g[27,0] = i_load_B5lv_b_i - 0.249053057297486*v_B5_a_i + 0.996212229189942*v_B5_a_r + 0.249053057297486*v_B5_b_i - 0.996212229189942*v_B5_b_r - 5.7516339869281*v_B5lv_b_i + 23.0065359477124*v_B5lv_b_r + 5.7516339869281*v_B5lv_n_i - 23.0065359477124*v_B5lv_n_r
        struct[0].g[28,0] = i_load_B5lv_c_r - 0.996212229189942*v_B5_b_i - 0.249053057297486*v_B5_b_r + 0.996212229189942*v_B5_c_i + 0.249053057297486*v_B5_c_r - 23.0065359477124*v_B5lv_c_i - 5.7516339869281*v_B5lv_c_r + 23.0065359477124*v_B5lv_n_i + 5.7516339869281*v_B5lv_n_r
        struct[0].g[29,0] = i_load_B5lv_c_i - 0.249053057297486*v_B5_b_i + 0.996212229189942*v_B5_b_r + 0.249053057297486*v_B5_c_i - 0.996212229189942*v_B5_c_r - 5.7516339869281*v_B5lv_c_i + 23.0065359477124*v_B5lv_c_r + 5.7516339869281*v_B5lv_n_i - 23.0065359477124*v_B5lv_n_r
        struct[0].g[30,0] = 23.0065359477124*v_B5lv_a_i + 5.7516339869281*v_B5lv_a_r + 23.0065359477124*v_B5lv_b_i + 5.7516339869281*v_B5lv_b_r + 23.0065359477124*v_B5lv_c_i + 5.7516339869281*v_B5lv_c_r - 69.0196078431372*v_B5lv_n_i - 1017.25490196078*v_B5lv_n_r
        struct[0].g[31,0] = 5.7516339869281*v_B5lv_a_i - 23.0065359477124*v_B5lv_a_r + 5.7516339869281*v_B5lv_b_i - 23.0065359477124*v_B5lv_b_r + 5.7516339869281*v_B5lv_c_i - 23.0065359477124*v_B5lv_c_r - 1017.25490196078*v_B5lv_n_i + 69.0196078431372*v_B5lv_n_r
        struct[0].g[32,0] = i_load_B6lv_a_r + 0.996212229189942*v_B6_a_i + 0.249053057297486*v_B6_a_r - 0.996212229189942*v_B6_c_i - 0.249053057297486*v_B6_c_r - 23.0065359477124*v_B6lv_a_i - 5.7516339869281*v_B6lv_a_r + 23.0065359477124*v_B6lv_n_i + 5.7516339869281*v_B6lv_n_r
        struct[0].g[33,0] = i_load_B6lv_a_i + 0.249053057297486*v_B6_a_i - 0.996212229189942*v_B6_a_r - 0.249053057297486*v_B6_c_i + 0.996212229189942*v_B6_c_r - 5.7516339869281*v_B6lv_a_i + 23.0065359477124*v_B6lv_a_r + 5.7516339869281*v_B6lv_n_i - 23.0065359477124*v_B6lv_n_r
        struct[0].g[34,0] = i_load_B6lv_b_r - 0.996212229189942*v_B6_a_i - 0.249053057297486*v_B6_a_r + 0.996212229189942*v_B6_b_i + 0.249053057297486*v_B6_b_r - 23.0065359477124*v_B6lv_b_i - 5.7516339869281*v_B6lv_b_r + 23.0065359477124*v_B6lv_n_i + 5.7516339869281*v_B6lv_n_r
        struct[0].g[35,0] = i_load_B6lv_b_i - 0.249053057297486*v_B6_a_i + 0.996212229189942*v_B6_a_r + 0.249053057297486*v_B6_b_i - 0.996212229189942*v_B6_b_r - 5.7516339869281*v_B6lv_b_i + 23.0065359477124*v_B6lv_b_r + 5.7516339869281*v_B6lv_n_i - 23.0065359477124*v_B6lv_n_r
        struct[0].g[36,0] = i_load_B6lv_c_r - 0.996212229189942*v_B6_b_i - 0.249053057297486*v_B6_b_r + 0.996212229189942*v_B6_c_i + 0.249053057297486*v_B6_c_r - 23.0065359477124*v_B6lv_c_i - 5.7516339869281*v_B6lv_c_r + 23.0065359477124*v_B6lv_n_i + 5.7516339869281*v_B6lv_n_r
        struct[0].g[37,0] = i_load_B6lv_c_i - 0.249053057297486*v_B6_b_i + 0.996212229189942*v_B6_b_r + 0.249053057297486*v_B6_c_i - 0.996212229189942*v_B6_c_r - 5.7516339869281*v_B6lv_c_i + 23.0065359477124*v_B6lv_c_r + 5.7516339869281*v_B6lv_n_i - 23.0065359477124*v_B6lv_n_r
        struct[0].g[38,0] = 23.0065359477124*v_B6lv_a_i + 5.7516339869281*v_B6lv_a_r + 23.0065359477124*v_B6lv_b_i + 5.7516339869281*v_B6lv_b_r + 23.0065359477124*v_B6lv_c_i + 5.7516339869281*v_B6lv_c_r - 69.0196078431372*v_B6lv_n_i - 1017.25490196078*v_B6lv_n_r
        struct[0].g[39,0] = 5.7516339869281*v_B6lv_a_i - 23.0065359477124*v_B6lv_a_r + 5.7516339869281*v_B6lv_b_i - 23.0065359477124*v_B6lv_b_r + 5.7516339869281*v_B6lv_c_i - 23.0065359477124*v_B6lv_c_r - 1017.25490196078*v_B6lv_n_i + 69.0196078431372*v_B6lv_n_r
        struct[0].g[40,0] = 0.598820527961361*v_B1_a_i + 1.10755301189314*v_B1_a_r - 0.171091579417532*v_B1_b_i - 0.316443717683753*v_B1_b_r - 0.171091579417532*v_B1_c_i - 0.316443717683753*v_B1_c_r - 1.28353302446119*v_B2_a_i - 2.23667465123725*v_B2_a_r + 0.385473430243205*v_B2_b_i + 0.643671749092996*v_B2_b_r + 0.385473430243205*v_B2_c_i + 0.643671749092997*v_B2_c_r + 0.996212229189942*v_B2lv_a_i + 0.249053057297486*v_B2lv_a_r - 0.996212229189942*v_B2lv_b_i - 0.249053057297486*v_B2lv_b_r + 0.598820527961361*v_B3_a_i + 1.10755301189314*v_B3_a_r - 0.171091579417532*v_B3_b_i - 0.316443717683753*v_B3_b_r - 0.171091579417532*v_B3_c_i - 0.316443717683753*v_B3_c_r
        struct[0].g[41,0] = 1.10755301189314*v_B1_a_i - 0.598820527961361*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417532*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417532*v_B1_c_r - 2.23667465123725*v_B2_a_i + 1.28353302446119*v_B2_a_r + 0.643671749092996*v_B2_b_i - 0.385473430243205*v_B2_b_r + 0.643671749092997*v_B2_c_i - 0.385473430243205*v_B2_c_r + 0.249053057297486*v_B2lv_a_i - 0.996212229189942*v_B2lv_a_r - 0.249053057297486*v_B2lv_b_i + 0.996212229189942*v_B2lv_b_r + 1.10755301189314*v_B3_a_i - 0.598820527961361*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417532*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417532*v_B3_c_r
        struct[0].g[42,0] = -0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r + 0.59882052796136*v_B1_b_i + 1.10755301189314*v_B1_b_r - 0.171091579417531*v_B1_c_i - 0.316443717683753*v_B1_c_r + 0.385473430243205*v_B2_a_i + 0.643671749092996*v_B2_a_r - 1.28353302446119*v_B2_b_i - 2.23667465123725*v_B2_b_r + 0.385473430243204*v_B2_c_i + 0.643671749092997*v_B2_c_r + 0.996212229189942*v_B2lv_b_i + 0.249053057297486*v_B2lv_b_r - 0.996212229189942*v_B2lv_c_i - 0.249053057297486*v_B2lv_c_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r + 0.59882052796136*v_B3_b_i + 1.10755301189314*v_B3_b_r - 0.171091579417531*v_B3_c_i - 0.316443717683753*v_B3_c_r
        struct[0].g[43,0] = -0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r + 1.10755301189314*v_B1_b_i - 0.59882052796136*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417531*v_B1_c_r + 0.643671749092996*v_B2_a_i - 0.385473430243205*v_B2_a_r - 2.23667465123725*v_B2_b_i + 1.28353302446119*v_B2_b_r + 0.643671749092997*v_B2_c_i - 0.385473430243204*v_B2_c_r + 0.249053057297486*v_B2lv_b_i - 0.996212229189942*v_B2lv_b_r - 0.249053057297486*v_B2lv_c_i + 0.996212229189942*v_B2lv_c_r - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r + 1.10755301189314*v_B3_b_i - 0.59882052796136*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417531*v_B3_c_r
        struct[0].g[44,0] = -0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r - 0.171091579417531*v_B1_b_i - 0.316443717683753*v_B1_b_r + 0.59882052796136*v_B1_c_i + 1.10755301189314*v_B1_c_r + 0.385473430243205*v_B2_a_i + 0.643671749092997*v_B2_a_r + 0.385473430243204*v_B2_b_i + 0.643671749092997*v_B2_b_r - 1.28353302446119*v_B2_c_i - 2.23667465123725*v_B2_c_r - 0.996212229189942*v_B2lv_a_i - 0.249053057297486*v_B2lv_a_r + 0.996212229189942*v_B2lv_c_i + 0.249053057297486*v_B2lv_c_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r - 0.171091579417531*v_B3_b_i - 0.316443717683753*v_B3_b_r + 0.59882052796136*v_B3_c_i + 1.10755301189314*v_B3_c_r
        struct[0].g[45,0] = -0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417531*v_B1_b_r + 1.10755301189314*v_B1_c_i - 0.59882052796136*v_B1_c_r + 0.643671749092997*v_B2_a_i - 0.385473430243205*v_B2_a_r + 0.643671749092997*v_B2_b_i - 0.385473430243204*v_B2_b_r - 2.23667465123725*v_B2_c_i + 1.28353302446119*v_B2_c_r - 0.249053057297486*v_B2lv_a_i + 0.996212229189942*v_B2lv_a_r + 0.249053057297486*v_B2lv_c_i - 0.996212229189942*v_B2lv_c_r - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417531*v_B3_b_r + 1.10755301189314*v_B3_c_i - 0.59882052796136*v_B3_c_r
        struct[0].g[46,0] = 0.598820527961361*v_B2_a_i + 1.10755301189314*v_B2_a_r - 0.171091579417532*v_B2_b_i - 0.316443717683753*v_B2_b_r - 0.171091579417532*v_B2_c_i - 0.316443717683753*v_B2_c_r - 1.28353302446119*v_B3_a_i - 2.23667465123725*v_B3_a_r + 0.385473430243205*v_B3_b_i + 0.643671749092996*v_B3_b_r + 0.385473430243205*v_B3_c_i + 0.643671749092997*v_B3_c_r + 0.996212229189942*v_B3lv_a_i + 0.249053057297486*v_B3lv_a_r - 0.996212229189942*v_B3lv_b_i - 0.249053057297486*v_B3lv_b_r + 0.598820527961361*v_B4_a_i + 1.10755301189314*v_B4_a_r - 0.171091579417532*v_B4_b_i - 0.316443717683753*v_B4_b_r - 0.171091579417532*v_B4_c_i - 0.316443717683753*v_B4_c_r
        struct[0].g[47,0] = 1.10755301189314*v_B2_a_i - 0.598820527961361*v_B2_a_r - 0.316443717683753*v_B2_b_i + 0.171091579417532*v_B2_b_r - 0.316443717683753*v_B2_c_i + 0.171091579417532*v_B2_c_r - 2.23667465123725*v_B3_a_i + 1.28353302446119*v_B3_a_r + 0.643671749092996*v_B3_b_i - 0.385473430243205*v_B3_b_r + 0.643671749092997*v_B3_c_i - 0.385473430243205*v_B3_c_r + 0.249053057297486*v_B3lv_a_i - 0.996212229189942*v_B3lv_a_r - 0.249053057297486*v_B3lv_b_i + 0.996212229189942*v_B3lv_b_r + 1.10755301189314*v_B4_a_i - 0.598820527961361*v_B4_a_r - 0.316443717683753*v_B4_b_i + 0.171091579417532*v_B4_b_r - 0.316443717683753*v_B4_c_i + 0.171091579417532*v_B4_c_r
        struct[0].g[48,0] = -0.171091579417532*v_B2_a_i - 0.316443717683753*v_B2_a_r + 0.59882052796136*v_B2_b_i + 1.10755301189314*v_B2_b_r - 0.171091579417531*v_B2_c_i - 0.316443717683753*v_B2_c_r + 0.385473430243205*v_B3_a_i + 0.643671749092996*v_B3_a_r - 1.28353302446119*v_B3_b_i - 2.23667465123725*v_B3_b_r + 0.385473430243204*v_B3_c_i + 0.643671749092997*v_B3_c_r + 0.996212229189942*v_B3lv_b_i + 0.249053057297486*v_B3lv_b_r - 0.996212229189942*v_B3lv_c_i - 0.249053057297486*v_B3lv_c_r - 0.171091579417532*v_B4_a_i - 0.316443717683753*v_B4_a_r + 0.59882052796136*v_B4_b_i + 1.10755301189314*v_B4_b_r - 0.171091579417531*v_B4_c_i - 0.316443717683753*v_B4_c_r
        struct[0].g[49,0] = -0.316443717683753*v_B2_a_i + 0.171091579417532*v_B2_a_r + 1.10755301189314*v_B2_b_i - 0.59882052796136*v_B2_b_r - 0.316443717683753*v_B2_c_i + 0.171091579417531*v_B2_c_r + 0.643671749092996*v_B3_a_i - 0.385473430243205*v_B3_a_r - 2.23667465123725*v_B3_b_i + 1.28353302446119*v_B3_b_r + 0.643671749092997*v_B3_c_i - 0.385473430243204*v_B3_c_r + 0.249053057297486*v_B3lv_b_i - 0.996212229189942*v_B3lv_b_r - 0.249053057297486*v_B3lv_c_i + 0.996212229189942*v_B3lv_c_r - 0.316443717683753*v_B4_a_i + 0.171091579417532*v_B4_a_r + 1.10755301189314*v_B4_b_i - 0.59882052796136*v_B4_b_r - 0.316443717683753*v_B4_c_i + 0.171091579417531*v_B4_c_r
        struct[0].g[50,0] = -0.171091579417532*v_B2_a_i - 0.316443717683753*v_B2_a_r - 0.171091579417531*v_B2_b_i - 0.316443717683753*v_B2_b_r + 0.59882052796136*v_B2_c_i + 1.10755301189314*v_B2_c_r + 0.385473430243205*v_B3_a_i + 0.643671749092997*v_B3_a_r + 0.385473430243204*v_B3_b_i + 0.643671749092997*v_B3_b_r - 1.28353302446119*v_B3_c_i - 2.23667465123725*v_B3_c_r - 0.996212229189942*v_B3lv_a_i - 0.249053057297486*v_B3lv_a_r + 0.996212229189942*v_B3lv_c_i + 0.249053057297486*v_B3lv_c_r - 0.171091579417532*v_B4_a_i - 0.316443717683753*v_B4_a_r - 0.171091579417531*v_B4_b_i - 0.316443717683753*v_B4_b_r + 0.59882052796136*v_B4_c_i + 1.10755301189314*v_B4_c_r
        struct[0].g[51,0] = -0.316443717683753*v_B2_a_i + 0.171091579417532*v_B2_a_r - 0.316443717683753*v_B2_b_i + 0.171091579417531*v_B2_b_r + 1.10755301189314*v_B2_c_i - 0.59882052796136*v_B2_c_r + 0.643671749092997*v_B3_a_i - 0.385473430243205*v_B3_a_r + 0.643671749092997*v_B3_b_i - 0.385473430243204*v_B3_b_r - 2.23667465123725*v_B3_c_i + 1.28353302446119*v_B3_c_r - 0.249053057297486*v_B3lv_a_i + 0.996212229189942*v_B3lv_a_r + 0.249053057297486*v_B3lv_c_i - 0.996212229189942*v_B3lv_c_r - 0.316443717683753*v_B4_a_i + 0.171091579417532*v_B4_a_r - 0.316443717683753*v_B4_b_i + 0.171091579417531*v_B4_b_r + 1.10755301189314*v_B4_c_i - 0.59882052796136*v_B4_c_r
        struct[0].g[52,0] = 0.598820527961361*v_B3_a_i + 1.10755301189314*v_B3_a_r - 0.171091579417532*v_B3_b_i - 0.316443717683753*v_B3_b_r - 0.171091579417532*v_B3_c_i - 0.316443717683753*v_B3_c_r - 0.684903767132556*v_B4_a_i - 1.12912163934412*v_B4_a_r + 0.214305342572583*v_B4_b_i + 0.327228031409243*v_B4_b_r + 0.214305342572583*v_B4_c_i + 0.327228031409244*v_B4_c_r + 0.996212229189942*v_B4lv_a_i + 0.249053057297486*v_B4lv_a_r - 0.996212229189942*v_B4lv_b_i - 0.249053057297486*v_B4lv_b_r
        struct[0].g[53,0] = 1.10755301189314*v_B3_a_i - 0.598820527961361*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417532*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417532*v_B3_c_r - 1.12912163934412*v_B4_a_i + 0.684903767132556*v_B4_a_r + 0.327228031409243*v_B4_b_i - 0.214305342572583*v_B4_b_r + 0.327228031409244*v_B4_c_i - 0.214305342572583*v_B4_c_r + 0.249053057297486*v_B4lv_a_i - 0.996212229189942*v_B4lv_a_r - 0.249053057297486*v_B4lv_b_i + 0.996212229189942*v_B4lv_b_r
        struct[0].g[54,0] = -0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r + 0.59882052796136*v_B3_b_i + 1.10755301189314*v_B3_b_r - 0.171091579417531*v_B3_c_i - 0.316443717683753*v_B3_c_r + 0.214305342572583*v_B4_a_i + 0.327228031409243*v_B4_a_r - 0.684903767132556*v_B4_b_i - 1.12912163934412*v_B4_b_r + 0.214305342572582*v_B4_c_i + 0.327228031409244*v_B4_c_r + 0.996212229189942*v_B4lv_b_i + 0.249053057297486*v_B4lv_b_r - 0.996212229189942*v_B4lv_c_i - 0.249053057297486*v_B4lv_c_r
        struct[0].g[55,0] = -0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r + 1.10755301189314*v_B3_b_i - 0.59882052796136*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417531*v_B3_c_r + 0.327228031409243*v_B4_a_i - 0.214305342572583*v_B4_a_r - 1.12912163934412*v_B4_b_i + 0.684903767132556*v_B4_b_r + 0.327228031409244*v_B4_c_i - 0.214305342572582*v_B4_c_r + 0.249053057297486*v_B4lv_b_i - 0.996212229189942*v_B4lv_b_r - 0.249053057297486*v_B4lv_c_i + 0.996212229189942*v_B4lv_c_r
        struct[0].g[56,0] = -0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r - 0.171091579417531*v_B3_b_i - 0.316443717683753*v_B3_b_r + 0.59882052796136*v_B3_c_i + 1.10755301189314*v_B3_c_r + 0.214305342572583*v_B4_a_i + 0.327228031409243*v_B4_a_r + 0.214305342572582*v_B4_b_i + 0.327228031409244*v_B4_b_r - 0.684903767132556*v_B4_c_i - 1.12912163934412*v_B4_c_r - 0.996212229189942*v_B4lv_a_i - 0.249053057297486*v_B4lv_a_r + 0.996212229189942*v_B4lv_c_i + 0.249053057297486*v_B4lv_c_r
        struct[0].g[57,0] = -0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417531*v_B3_b_r + 1.10755301189314*v_B3_c_i - 0.59882052796136*v_B3_c_r + 0.327228031409243*v_B4_a_i - 0.214305342572583*v_B4_a_r + 0.327228031409244*v_B4_b_i - 0.214305342572582*v_B4_b_r - 1.12912163934412*v_B4_c_i + 0.684903767132556*v_B4_c_r - 0.249053057297486*v_B4lv_a_i + 0.996212229189942*v_B4lv_a_r + 0.249053057297486*v_B4lv_c_i - 0.996212229189942*v_B4lv_c_r
        struct[0].g[58,0] = -0.684903767132556*v_B5_a_i - 1.12912163934412*v_B5_a_r + 0.214305342572583*v_B5_b_i + 0.327228031409243*v_B5_b_r + 0.214305342572583*v_B5_c_i + 0.327228031409244*v_B5_c_r + 0.996212229189942*v_B5lv_a_i + 0.249053057297486*v_B5lv_a_r - 0.996212229189942*v_B5lv_b_i - 0.249053057297486*v_B5lv_b_r + 0.598820527961361*v_B6_a_i + 1.10755301189314*v_B6_a_r - 0.171091579417532*v_B6_b_i - 0.316443717683753*v_B6_b_r - 0.171091579417532*v_B6_c_i - 0.316443717683753*v_B6_c_r
        struct[0].g[59,0] = -1.12912163934412*v_B5_a_i + 0.684903767132556*v_B5_a_r + 0.327228031409243*v_B5_b_i - 0.214305342572583*v_B5_b_r + 0.327228031409244*v_B5_c_i - 0.214305342572583*v_B5_c_r + 0.249053057297486*v_B5lv_a_i - 0.996212229189942*v_B5lv_a_r - 0.249053057297486*v_B5lv_b_i + 0.996212229189942*v_B5lv_b_r + 1.10755301189314*v_B6_a_i - 0.598820527961361*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417532*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417532*v_B6_c_r
        struct[0].g[60,0] = 0.214305342572583*v_B5_a_i + 0.327228031409243*v_B5_a_r - 0.684903767132556*v_B5_b_i - 1.12912163934412*v_B5_b_r + 0.214305342572582*v_B5_c_i + 0.327228031409244*v_B5_c_r + 0.996212229189942*v_B5lv_b_i + 0.249053057297486*v_B5lv_b_r - 0.996212229189942*v_B5lv_c_i - 0.249053057297486*v_B5lv_c_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r + 0.59882052796136*v_B6_b_i + 1.10755301189314*v_B6_b_r - 0.171091579417531*v_B6_c_i - 0.316443717683753*v_B6_c_r
        struct[0].g[61,0] = 0.327228031409243*v_B5_a_i - 0.214305342572583*v_B5_a_r - 1.12912163934412*v_B5_b_i + 0.684903767132556*v_B5_b_r + 0.327228031409244*v_B5_c_i - 0.214305342572582*v_B5_c_r + 0.249053057297486*v_B5lv_b_i - 0.996212229189942*v_B5lv_b_r - 0.249053057297486*v_B5lv_c_i + 0.996212229189942*v_B5lv_c_r - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r + 1.10755301189314*v_B6_b_i - 0.59882052796136*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417531*v_B6_c_r
        struct[0].g[62,0] = 0.214305342572583*v_B5_a_i + 0.327228031409243*v_B5_a_r + 0.214305342572582*v_B5_b_i + 0.327228031409244*v_B5_b_r - 0.684903767132556*v_B5_c_i - 1.12912163934412*v_B5_c_r - 0.996212229189942*v_B5lv_a_i - 0.249053057297486*v_B5lv_a_r + 0.996212229189942*v_B5lv_c_i + 0.249053057297486*v_B5lv_c_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r - 0.171091579417531*v_B6_b_i - 0.316443717683753*v_B6_b_r + 0.59882052796136*v_B6_c_i + 1.10755301189314*v_B6_c_r
        struct[0].g[63,0] = 0.327228031409243*v_B5_a_i - 0.214305342572583*v_B5_a_r + 0.327228031409244*v_B5_b_i - 0.214305342572582*v_B5_b_r - 1.12912163934412*v_B5_c_i + 0.684903767132556*v_B5_c_r - 0.249053057297486*v_B5lv_a_i + 0.996212229189942*v_B5lv_a_r + 0.249053057297486*v_B5lv_c_i - 0.996212229189942*v_B5lv_c_r - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417531*v_B6_b_r + 1.10755301189314*v_B6_c_i - 0.59882052796136*v_B6_c_r
        struct[0].g[64,0] = 0.598820527961361*v_B5_a_i + 1.10755301189314*v_B5_a_r - 0.171091579417532*v_B5_b_i - 0.316443717683753*v_B5_b_r - 0.171091579417532*v_B5_c_i - 0.316443717683753*v_B5_c_r - 1.28353302446119*v_B6_a_i - 2.23667465123725*v_B6_a_r + 0.385473430243205*v_B6_b_i + 0.643671749092996*v_B6_b_r + 0.385473430243205*v_B6_c_i + 0.643671749092997*v_B6_c_r + 0.996212229189942*v_B6lv_a_i + 0.249053057297486*v_B6lv_a_r - 0.996212229189942*v_B6lv_b_i - 0.249053057297486*v_B6lv_b_r + 0.598820527961361*v_B7_a_i + 1.10755301189314*v_B7_a_r - 0.171091579417532*v_B7_b_i - 0.316443717683753*v_B7_b_r - 0.171091579417532*v_B7_c_i - 0.316443717683753*v_B7_c_r
        struct[0].g[65,0] = 1.10755301189314*v_B5_a_i - 0.598820527961361*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417532*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417532*v_B5_c_r - 2.23667465123725*v_B6_a_i + 1.28353302446119*v_B6_a_r + 0.643671749092996*v_B6_b_i - 0.385473430243205*v_B6_b_r + 0.643671749092997*v_B6_c_i - 0.385473430243205*v_B6_c_r + 0.249053057297486*v_B6lv_a_i - 0.996212229189942*v_B6lv_a_r - 0.249053057297486*v_B6lv_b_i + 0.996212229189942*v_B6lv_b_r + 1.10755301189314*v_B7_a_i - 0.598820527961361*v_B7_a_r - 0.316443717683753*v_B7_b_i + 0.171091579417532*v_B7_b_r - 0.316443717683753*v_B7_c_i + 0.171091579417532*v_B7_c_r
        struct[0].g[66,0] = -0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r + 0.59882052796136*v_B5_b_i + 1.10755301189314*v_B5_b_r - 0.171091579417531*v_B5_c_i - 0.316443717683753*v_B5_c_r + 0.385473430243205*v_B6_a_i + 0.643671749092996*v_B6_a_r - 1.28353302446119*v_B6_b_i - 2.23667465123725*v_B6_b_r + 0.385473430243204*v_B6_c_i + 0.643671749092997*v_B6_c_r + 0.996212229189942*v_B6lv_b_i + 0.249053057297486*v_B6lv_b_r - 0.996212229189942*v_B6lv_c_i - 0.249053057297486*v_B6lv_c_r - 0.171091579417532*v_B7_a_i - 0.316443717683753*v_B7_a_r + 0.59882052796136*v_B7_b_i + 1.10755301189314*v_B7_b_r - 0.171091579417531*v_B7_c_i - 0.316443717683753*v_B7_c_r
        struct[0].g[67,0] = -0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r + 1.10755301189314*v_B5_b_i - 0.59882052796136*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417531*v_B5_c_r + 0.643671749092996*v_B6_a_i - 0.385473430243205*v_B6_a_r - 2.23667465123725*v_B6_b_i + 1.28353302446119*v_B6_b_r + 0.643671749092997*v_B6_c_i - 0.385473430243204*v_B6_c_r + 0.249053057297486*v_B6lv_b_i - 0.996212229189942*v_B6lv_b_r - 0.249053057297486*v_B6lv_c_i + 0.996212229189942*v_B6lv_c_r - 0.316443717683753*v_B7_a_i + 0.171091579417532*v_B7_a_r + 1.10755301189314*v_B7_b_i - 0.59882052796136*v_B7_b_r - 0.316443717683753*v_B7_c_i + 0.171091579417531*v_B7_c_r
        struct[0].g[68,0] = -0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r - 0.171091579417531*v_B5_b_i - 0.316443717683753*v_B5_b_r + 0.59882052796136*v_B5_c_i + 1.10755301189314*v_B5_c_r + 0.385473430243205*v_B6_a_i + 0.643671749092997*v_B6_a_r + 0.385473430243204*v_B6_b_i + 0.643671749092997*v_B6_b_r - 1.28353302446119*v_B6_c_i - 2.23667465123725*v_B6_c_r - 0.996212229189942*v_B6lv_a_i - 0.249053057297486*v_B6lv_a_r + 0.996212229189942*v_B6lv_c_i + 0.249053057297486*v_B6lv_c_r - 0.171091579417532*v_B7_a_i - 0.316443717683753*v_B7_a_r - 0.171091579417531*v_B7_b_i - 0.316443717683753*v_B7_b_r + 0.59882052796136*v_B7_c_i + 1.10755301189314*v_B7_c_r
        struct[0].g[69,0] = -0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417531*v_B5_b_r + 1.10755301189314*v_B5_c_i - 0.59882052796136*v_B5_c_r + 0.643671749092997*v_B6_a_i - 0.385473430243205*v_B6_a_r + 0.643671749092997*v_B6_b_i - 0.385473430243204*v_B6_b_r - 2.23667465123725*v_B6_c_i + 1.28353302446119*v_B6_c_r - 0.249053057297486*v_B6lv_a_i + 0.996212229189942*v_B6lv_a_r + 0.249053057297486*v_B6lv_c_i - 0.996212229189942*v_B6lv_c_r - 0.316443717683753*v_B7_a_i + 0.171091579417532*v_B7_a_r - 0.316443717683753*v_B7_b_i + 0.171091579417531*v_B7_b_r + 1.10755301189314*v_B7_c_i - 0.59882052796136*v_B7_c_r
        struct[0].g[70,0] = -i_t_B2_B2lv_a_r + 0.0862745098039216*v_B2_a_i + 0.0215686274509804*v_B2_a_r - 0.0431372549019608*v_B2_b_i - 0.0107843137254902*v_B2_b_r - 0.0431372549019608*v_B2_c_i - 0.0107843137254902*v_B2_c_r - 0.996212229189942*v_B2lv_a_i - 0.249053057297486*v_B2lv_a_r + 0.996212229189942*v_B2lv_b_i + 0.249053057297486*v_B2lv_b_r
        struct[0].g[71,0] = -i_t_B2_B2lv_a_i + 0.0215686274509804*v_B2_a_i - 0.0862745098039216*v_B2_a_r - 0.0107843137254902*v_B2_b_i + 0.0431372549019608*v_B2_b_r - 0.0107843137254902*v_B2_c_i + 0.0431372549019608*v_B2_c_r - 0.249053057297486*v_B2lv_a_i + 0.996212229189942*v_B2lv_a_r + 0.249053057297486*v_B2lv_b_i - 0.996212229189942*v_B2lv_b_r
        struct[0].g[72,0] = -i_t_B2_B2lv_b_r - 0.0431372549019608*v_B2_a_i - 0.0107843137254902*v_B2_a_r + 0.0862745098039216*v_B2_b_i + 0.0215686274509804*v_B2_b_r - 0.0431372549019608*v_B2_c_i - 0.0107843137254902*v_B2_c_r - 0.996212229189942*v_B2lv_b_i - 0.249053057297486*v_B2lv_b_r + 0.996212229189942*v_B2lv_c_i + 0.249053057297486*v_B2lv_c_r
        struct[0].g[73,0] = -i_t_B2_B2lv_b_i - 0.0107843137254902*v_B2_a_i + 0.0431372549019608*v_B2_a_r + 0.0215686274509804*v_B2_b_i - 0.0862745098039216*v_B2_b_r - 0.0107843137254902*v_B2_c_i + 0.0431372549019608*v_B2_c_r - 0.249053057297486*v_B2lv_b_i + 0.996212229189942*v_B2lv_b_r + 0.249053057297486*v_B2lv_c_i - 0.996212229189942*v_B2lv_c_r
        struct[0].g[74,0] = -i_t_B2_B2lv_c_r - 0.0431372549019608*v_B2_a_i - 0.0107843137254902*v_B2_a_r - 0.0431372549019608*v_B2_b_i - 0.0107843137254902*v_B2_b_r + 0.0862745098039216*v_B2_c_i + 0.0215686274509804*v_B2_c_r + 0.996212229189942*v_B2lv_a_i + 0.249053057297486*v_B2lv_a_r - 0.996212229189942*v_B2lv_c_i - 0.249053057297486*v_B2lv_c_r
        struct[0].g[75,0] = -i_t_B2_B2lv_c_i - 0.0107843137254902*v_B2_a_i + 0.0431372549019608*v_B2_a_r - 0.0107843137254902*v_B2_b_i + 0.0431372549019608*v_B2_b_r + 0.0215686274509804*v_B2_c_i - 0.0862745098039216*v_B2_c_r + 0.249053057297486*v_B2lv_a_i - 0.996212229189942*v_B2lv_a_r - 0.249053057297486*v_B2lv_c_i + 0.996212229189942*v_B2lv_c_r
        struct[0].g[76,0] = -i_t_B3_B3lv_a_r + 0.0862745098039216*v_B3_a_i + 0.0215686274509804*v_B3_a_r - 0.0431372549019608*v_B3_b_i - 0.0107843137254902*v_B3_b_r - 0.0431372549019608*v_B3_c_i - 0.0107843137254902*v_B3_c_r - 0.996212229189942*v_B3lv_a_i - 0.249053057297486*v_B3lv_a_r + 0.996212229189942*v_B3lv_b_i + 0.249053057297486*v_B3lv_b_r
        struct[0].g[77,0] = -i_t_B3_B3lv_a_i + 0.0215686274509804*v_B3_a_i - 0.0862745098039216*v_B3_a_r - 0.0107843137254902*v_B3_b_i + 0.0431372549019608*v_B3_b_r - 0.0107843137254902*v_B3_c_i + 0.0431372549019608*v_B3_c_r - 0.249053057297486*v_B3lv_a_i + 0.996212229189942*v_B3lv_a_r + 0.249053057297486*v_B3lv_b_i - 0.996212229189942*v_B3lv_b_r
        struct[0].g[78,0] = -i_t_B3_B3lv_b_r - 0.0431372549019608*v_B3_a_i - 0.0107843137254902*v_B3_a_r + 0.0862745098039216*v_B3_b_i + 0.0215686274509804*v_B3_b_r - 0.0431372549019608*v_B3_c_i - 0.0107843137254902*v_B3_c_r - 0.996212229189942*v_B3lv_b_i - 0.249053057297486*v_B3lv_b_r + 0.996212229189942*v_B3lv_c_i + 0.249053057297486*v_B3lv_c_r
        struct[0].g[79,0] = -i_t_B3_B3lv_b_i - 0.0107843137254902*v_B3_a_i + 0.0431372549019608*v_B3_a_r + 0.0215686274509804*v_B3_b_i - 0.0862745098039216*v_B3_b_r - 0.0107843137254902*v_B3_c_i + 0.0431372549019608*v_B3_c_r - 0.249053057297486*v_B3lv_b_i + 0.996212229189942*v_B3lv_b_r + 0.249053057297486*v_B3lv_c_i - 0.996212229189942*v_B3lv_c_r
        struct[0].g[80,0] = -i_t_B3_B3lv_c_r - 0.0431372549019608*v_B3_a_i - 0.0107843137254902*v_B3_a_r - 0.0431372549019608*v_B3_b_i - 0.0107843137254902*v_B3_b_r + 0.0862745098039216*v_B3_c_i + 0.0215686274509804*v_B3_c_r + 0.996212229189942*v_B3lv_a_i + 0.249053057297486*v_B3lv_a_r - 0.996212229189942*v_B3lv_c_i - 0.249053057297486*v_B3lv_c_r
        struct[0].g[81,0] = -i_t_B3_B3lv_c_i - 0.0107843137254902*v_B3_a_i + 0.0431372549019608*v_B3_a_r - 0.0107843137254902*v_B3_b_i + 0.0431372549019608*v_B3_b_r + 0.0215686274509804*v_B3_c_i - 0.0862745098039216*v_B3_c_r + 0.249053057297486*v_B3lv_a_i - 0.996212229189942*v_B3lv_a_r - 0.249053057297486*v_B3lv_c_i + 0.996212229189942*v_B3lv_c_r
        struct[0].g[82,0] = -i_t_B4_B4lv_a_r + 0.0862745098039216*v_B4_a_i + 0.0215686274509804*v_B4_a_r - 0.0431372549019608*v_B4_b_i - 0.0107843137254902*v_B4_b_r - 0.0431372549019608*v_B4_c_i - 0.0107843137254902*v_B4_c_r - 0.996212229189942*v_B4lv_a_i - 0.249053057297486*v_B4lv_a_r + 0.996212229189942*v_B4lv_b_i + 0.249053057297486*v_B4lv_b_r
        struct[0].g[83,0] = -i_t_B4_B4lv_a_i + 0.0215686274509804*v_B4_a_i - 0.0862745098039216*v_B4_a_r - 0.0107843137254902*v_B4_b_i + 0.0431372549019608*v_B4_b_r - 0.0107843137254902*v_B4_c_i + 0.0431372549019608*v_B4_c_r - 0.249053057297486*v_B4lv_a_i + 0.996212229189942*v_B4lv_a_r + 0.249053057297486*v_B4lv_b_i - 0.996212229189942*v_B4lv_b_r
        struct[0].g[84,0] = -i_t_B4_B4lv_b_r - 0.0431372549019608*v_B4_a_i - 0.0107843137254902*v_B4_a_r + 0.0862745098039216*v_B4_b_i + 0.0215686274509804*v_B4_b_r - 0.0431372549019608*v_B4_c_i - 0.0107843137254902*v_B4_c_r - 0.996212229189942*v_B4lv_b_i - 0.249053057297486*v_B4lv_b_r + 0.996212229189942*v_B4lv_c_i + 0.249053057297486*v_B4lv_c_r
        struct[0].g[85,0] = -i_t_B4_B4lv_b_i - 0.0107843137254902*v_B4_a_i + 0.0431372549019608*v_B4_a_r + 0.0215686274509804*v_B4_b_i - 0.0862745098039216*v_B4_b_r - 0.0107843137254902*v_B4_c_i + 0.0431372549019608*v_B4_c_r - 0.249053057297486*v_B4lv_b_i + 0.996212229189942*v_B4lv_b_r + 0.249053057297486*v_B4lv_c_i - 0.996212229189942*v_B4lv_c_r
        struct[0].g[86,0] = -i_t_B4_B4lv_c_r - 0.0431372549019608*v_B4_a_i - 0.0107843137254902*v_B4_a_r - 0.0431372549019608*v_B4_b_i - 0.0107843137254902*v_B4_b_r + 0.0862745098039216*v_B4_c_i + 0.0215686274509804*v_B4_c_r + 0.996212229189942*v_B4lv_a_i + 0.249053057297486*v_B4lv_a_r - 0.996212229189942*v_B4lv_c_i - 0.249053057297486*v_B4lv_c_r
        struct[0].g[87,0] = -i_t_B4_B4lv_c_i - 0.0107843137254902*v_B4_a_i + 0.0431372549019608*v_B4_a_r - 0.0107843137254902*v_B4_b_i + 0.0431372549019608*v_B4_b_r + 0.0215686274509804*v_B4_c_i - 0.0862745098039216*v_B4_c_r + 0.249053057297486*v_B4lv_a_i - 0.996212229189942*v_B4lv_a_r - 0.249053057297486*v_B4lv_c_i + 0.996212229189942*v_B4lv_c_r
        struct[0].g[88,0] = -i_t_B5_B5lv_a_r + 0.0862745098039216*v_B5_a_i + 0.0215686274509804*v_B5_a_r - 0.0431372549019608*v_B5_b_i - 0.0107843137254902*v_B5_b_r - 0.0431372549019608*v_B5_c_i - 0.0107843137254902*v_B5_c_r - 0.996212229189942*v_B5lv_a_i - 0.249053057297486*v_B5lv_a_r + 0.996212229189942*v_B5lv_b_i + 0.249053057297486*v_B5lv_b_r
        struct[0].g[89,0] = -i_t_B5_B5lv_a_i + 0.0215686274509804*v_B5_a_i - 0.0862745098039216*v_B5_a_r - 0.0107843137254902*v_B5_b_i + 0.0431372549019608*v_B5_b_r - 0.0107843137254902*v_B5_c_i + 0.0431372549019608*v_B5_c_r - 0.249053057297486*v_B5lv_a_i + 0.996212229189942*v_B5lv_a_r + 0.249053057297486*v_B5lv_b_i - 0.996212229189942*v_B5lv_b_r
        struct[0].g[90,0] = -i_t_B5_B5lv_b_r - 0.0431372549019608*v_B5_a_i - 0.0107843137254902*v_B5_a_r + 0.0862745098039216*v_B5_b_i + 0.0215686274509804*v_B5_b_r - 0.0431372549019608*v_B5_c_i - 0.0107843137254902*v_B5_c_r - 0.996212229189942*v_B5lv_b_i - 0.249053057297486*v_B5lv_b_r + 0.996212229189942*v_B5lv_c_i + 0.249053057297486*v_B5lv_c_r
        struct[0].g[91,0] = -i_t_B5_B5lv_b_i - 0.0107843137254902*v_B5_a_i + 0.0431372549019608*v_B5_a_r + 0.0215686274509804*v_B5_b_i - 0.0862745098039216*v_B5_b_r - 0.0107843137254902*v_B5_c_i + 0.0431372549019608*v_B5_c_r - 0.249053057297486*v_B5lv_b_i + 0.996212229189942*v_B5lv_b_r + 0.249053057297486*v_B5lv_c_i - 0.996212229189942*v_B5lv_c_r
        struct[0].g[92,0] = -i_t_B5_B5lv_c_r - 0.0431372549019608*v_B5_a_i - 0.0107843137254902*v_B5_a_r - 0.0431372549019608*v_B5_b_i - 0.0107843137254902*v_B5_b_r + 0.0862745098039216*v_B5_c_i + 0.0215686274509804*v_B5_c_r + 0.996212229189942*v_B5lv_a_i + 0.249053057297486*v_B5lv_a_r - 0.996212229189942*v_B5lv_c_i - 0.249053057297486*v_B5lv_c_r
        struct[0].g[93,0] = -i_t_B5_B5lv_c_i - 0.0107843137254902*v_B5_a_i + 0.0431372549019608*v_B5_a_r - 0.0107843137254902*v_B5_b_i + 0.0431372549019608*v_B5_b_r + 0.0215686274509804*v_B5_c_i - 0.0862745098039216*v_B5_c_r + 0.249053057297486*v_B5lv_a_i - 0.996212229189942*v_B5lv_a_r - 0.249053057297486*v_B5lv_c_i + 0.996212229189942*v_B5lv_c_r
        struct[0].g[94,0] = -i_t_B6_B6lv_a_r + 0.0862745098039216*v_B6_a_i + 0.0215686274509804*v_B6_a_r - 0.0431372549019608*v_B6_b_i - 0.0107843137254902*v_B6_b_r - 0.0431372549019608*v_B6_c_i - 0.0107843137254902*v_B6_c_r - 0.996212229189942*v_B6lv_a_i - 0.249053057297486*v_B6lv_a_r + 0.996212229189942*v_B6lv_b_i + 0.249053057297486*v_B6lv_b_r
        struct[0].g[95,0] = -i_t_B6_B6lv_a_i + 0.0215686274509804*v_B6_a_i - 0.0862745098039216*v_B6_a_r - 0.0107843137254902*v_B6_b_i + 0.0431372549019608*v_B6_b_r - 0.0107843137254902*v_B6_c_i + 0.0431372549019608*v_B6_c_r - 0.249053057297486*v_B6lv_a_i + 0.996212229189942*v_B6lv_a_r + 0.249053057297486*v_B6lv_b_i - 0.996212229189942*v_B6lv_b_r
        struct[0].g[96,0] = -i_t_B6_B6lv_b_r - 0.0431372549019608*v_B6_a_i - 0.0107843137254902*v_B6_a_r + 0.0862745098039216*v_B6_b_i + 0.0215686274509804*v_B6_b_r - 0.0431372549019608*v_B6_c_i - 0.0107843137254902*v_B6_c_r - 0.996212229189942*v_B6lv_b_i - 0.249053057297486*v_B6lv_b_r + 0.996212229189942*v_B6lv_c_i + 0.249053057297486*v_B6lv_c_r
        struct[0].g[97,0] = -i_t_B6_B6lv_b_i - 0.0107843137254902*v_B6_a_i + 0.0431372549019608*v_B6_a_r + 0.0215686274509804*v_B6_b_i - 0.0862745098039216*v_B6_b_r - 0.0107843137254902*v_B6_c_i + 0.0431372549019608*v_B6_c_r - 0.249053057297486*v_B6lv_b_i + 0.996212229189942*v_B6lv_b_r + 0.249053057297486*v_B6lv_c_i - 0.996212229189942*v_B6lv_c_r
        struct[0].g[98,0] = -i_t_B6_B6lv_c_r - 0.0431372549019608*v_B6_a_i - 0.0107843137254902*v_B6_a_r - 0.0431372549019608*v_B6_b_i - 0.0107843137254902*v_B6_b_r + 0.0862745098039216*v_B6_c_i + 0.0215686274509804*v_B6_c_r + 0.996212229189942*v_B6lv_a_i + 0.249053057297486*v_B6lv_a_r - 0.996212229189942*v_B6lv_c_i - 0.249053057297486*v_B6lv_c_r
        struct[0].g[99,0] = -i_t_B6_B6lv_c_i - 0.0107843137254902*v_B6_a_i + 0.0431372549019608*v_B6_a_r - 0.0107843137254902*v_B6_b_i + 0.0431372549019608*v_B6_b_r + 0.0215686274509804*v_B6_c_i - 0.0862745098039216*v_B6_c_r + 0.249053057297486*v_B6lv_a_i - 0.996212229189942*v_B6lv_a_r - 0.249053057297486*v_B6lv_c_i + 0.996212229189942*v_B6lv_c_r
        struct[0].g[100,0] = -i_l_B1_B2_a_r + 0.598820527961361*v_B1_a_i + 1.10755301189314*v_B1_a_r - 0.171091579417532*v_B1_b_i - 0.316443717683753*v_B1_b_r - 0.171091579417532*v_B1_c_i - 0.316443717683753*v_B1_c_r - 0.598820527961361*v_B2_a_i - 1.10755301189314*v_B2_a_r + 0.171091579417532*v_B2_b_i + 0.316443717683753*v_B2_b_r + 0.171091579417532*v_B2_c_i + 0.316443717683753*v_B2_c_r
        struct[0].g[101,0] = -i_l_B1_B2_a_i + 1.10755301189314*v_B1_a_i - 0.598820527961361*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417532*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417532*v_B1_c_r - 1.10755301189314*v_B2_a_i + 0.598820527961361*v_B2_a_r + 0.316443717683753*v_B2_b_i - 0.171091579417532*v_B2_b_r + 0.316443717683753*v_B2_c_i - 0.171091579417532*v_B2_c_r
        struct[0].g[102,0] = -i_l_B1_B2_b_r - 0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r + 0.59882052796136*v_B1_b_i + 1.10755301189314*v_B1_b_r - 0.171091579417531*v_B1_c_i - 0.316443717683753*v_B1_c_r + 0.171091579417532*v_B2_a_i + 0.316443717683753*v_B2_a_r - 0.59882052796136*v_B2_b_i - 1.10755301189314*v_B2_b_r + 0.171091579417531*v_B2_c_i + 0.316443717683753*v_B2_c_r
        struct[0].g[103,0] = -i_l_B1_B2_b_i - 0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r + 1.10755301189314*v_B1_b_i - 0.59882052796136*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417531*v_B1_c_r + 0.316443717683753*v_B2_a_i - 0.171091579417532*v_B2_a_r - 1.10755301189314*v_B2_b_i + 0.59882052796136*v_B2_b_r + 0.316443717683753*v_B2_c_i - 0.171091579417531*v_B2_c_r
        struct[0].g[104,0] = -i_l_B1_B2_c_r - 0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r - 0.171091579417531*v_B1_b_i - 0.316443717683753*v_B1_b_r + 0.59882052796136*v_B1_c_i + 1.10755301189314*v_B1_c_r + 0.171091579417532*v_B2_a_i + 0.316443717683753*v_B2_a_r + 0.171091579417531*v_B2_b_i + 0.316443717683753*v_B2_b_r - 0.59882052796136*v_B2_c_i - 1.10755301189314*v_B2_c_r
        struct[0].g[105,0] = -i_l_B1_B2_c_i - 0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417531*v_B1_b_r + 1.10755301189314*v_B1_c_i - 0.59882052796136*v_B1_c_r + 0.316443717683753*v_B2_a_i - 0.171091579417532*v_B2_a_r + 0.316443717683753*v_B2_b_i - 0.171091579417531*v_B2_b_r - 1.10755301189314*v_B2_c_i + 0.59882052796136*v_B2_c_r
        struct[0].g[106,0] = -i_l_B2_B3_a_r + 0.598820527961361*v_B2_a_i + 1.10755301189314*v_B2_a_r - 0.171091579417532*v_B2_b_i - 0.316443717683753*v_B2_b_r - 0.171091579417532*v_B2_c_i - 0.316443717683753*v_B2_c_r - 0.598820527961361*v_B3_a_i - 1.10755301189314*v_B3_a_r + 0.171091579417532*v_B3_b_i + 0.316443717683753*v_B3_b_r + 0.171091579417532*v_B3_c_i + 0.316443717683753*v_B3_c_r
        struct[0].g[107,0] = -i_l_B2_B3_a_i + 1.10755301189314*v_B2_a_i - 0.598820527961361*v_B2_a_r - 0.316443717683753*v_B2_b_i + 0.171091579417532*v_B2_b_r - 0.316443717683753*v_B2_c_i + 0.171091579417532*v_B2_c_r - 1.10755301189314*v_B3_a_i + 0.598820527961361*v_B3_a_r + 0.316443717683753*v_B3_b_i - 0.171091579417532*v_B3_b_r + 0.316443717683753*v_B3_c_i - 0.171091579417532*v_B3_c_r
        struct[0].g[108,0] = -i_l_B2_B3_b_r - 0.171091579417532*v_B2_a_i - 0.316443717683753*v_B2_a_r + 0.59882052796136*v_B2_b_i + 1.10755301189314*v_B2_b_r - 0.171091579417531*v_B2_c_i - 0.316443717683753*v_B2_c_r + 0.171091579417532*v_B3_a_i + 0.316443717683753*v_B3_a_r - 0.59882052796136*v_B3_b_i - 1.10755301189314*v_B3_b_r + 0.171091579417531*v_B3_c_i + 0.316443717683753*v_B3_c_r
        struct[0].g[109,0] = -i_l_B2_B3_b_i - 0.316443717683753*v_B2_a_i + 0.171091579417532*v_B2_a_r + 1.10755301189314*v_B2_b_i - 0.59882052796136*v_B2_b_r - 0.316443717683753*v_B2_c_i + 0.171091579417531*v_B2_c_r + 0.316443717683753*v_B3_a_i - 0.171091579417532*v_B3_a_r - 1.10755301189314*v_B3_b_i + 0.59882052796136*v_B3_b_r + 0.316443717683753*v_B3_c_i - 0.171091579417531*v_B3_c_r
        struct[0].g[110,0] = -i_l_B2_B3_c_r - 0.171091579417532*v_B2_a_i - 0.316443717683753*v_B2_a_r - 0.171091579417531*v_B2_b_i - 0.316443717683753*v_B2_b_r + 0.59882052796136*v_B2_c_i + 1.10755301189314*v_B2_c_r + 0.171091579417532*v_B3_a_i + 0.316443717683753*v_B3_a_r + 0.171091579417531*v_B3_b_i + 0.316443717683753*v_B3_b_r - 0.59882052796136*v_B3_c_i - 1.10755301189314*v_B3_c_r
        struct[0].g[111,0] = -i_l_B2_B3_c_i - 0.316443717683753*v_B2_a_i + 0.171091579417532*v_B2_a_r - 0.316443717683753*v_B2_b_i + 0.171091579417531*v_B2_b_r + 1.10755301189314*v_B2_c_i - 0.59882052796136*v_B2_c_r + 0.316443717683753*v_B3_a_i - 0.171091579417532*v_B3_a_r + 0.316443717683753*v_B3_b_i - 0.171091579417531*v_B3_b_r - 1.10755301189314*v_B3_c_i + 0.59882052796136*v_B3_c_r
        struct[0].g[112,0] = -i_l_B3_B4_a_r + 0.598820527961361*v_B3_a_i + 1.10755301189314*v_B3_a_r - 0.171091579417532*v_B3_b_i - 0.316443717683753*v_B3_b_r - 0.171091579417532*v_B3_c_i - 0.316443717683753*v_B3_c_r - 0.598820527961361*v_B4_a_i - 1.10755301189314*v_B4_a_r + 0.171091579417532*v_B4_b_i + 0.316443717683753*v_B4_b_r + 0.171091579417532*v_B4_c_i + 0.316443717683753*v_B4_c_r
        struct[0].g[113,0] = -i_l_B3_B4_a_i + 1.10755301189314*v_B3_a_i - 0.598820527961361*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417532*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417532*v_B3_c_r - 1.10755301189314*v_B4_a_i + 0.598820527961361*v_B4_a_r + 0.316443717683753*v_B4_b_i - 0.171091579417532*v_B4_b_r + 0.316443717683753*v_B4_c_i - 0.171091579417532*v_B4_c_r
        struct[0].g[114,0] = -i_l_B3_B4_b_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r + 0.59882052796136*v_B3_b_i + 1.10755301189314*v_B3_b_r - 0.171091579417531*v_B3_c_i - 0.316443717683753*v_B3_c_r + 0.171091579417532*v_B4_a_i + 0.316443717683753*v_B4_a_r - 0.59882052796136*v_B4_b_i - 1.10755301189314*v_B4_b_r + 0.171091579417531*v_B4_c_i + 0.316443717683753*v_B4_c_r
        struct[0].g[115,0] = -i_l_B3_B4_b_i - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r + 1.10755301189314*v_B3_b_i - 0.59882052796136*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417531*v_B3_c_r + 0.316443717683753*v_B4_a_i - 0.171091579417532*v_B4_a_r - 1.10755301189314*v_B4_b_i + 0.59882052796136*v_B4_b_r + 0.316443717683753*v_B4_c_i - 0.171091579417531*v_B4_c_r
        struct[0].g[116,0] = -i_l_B3_B4_c_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r - 0.171091579417531*v_B3_b_i - 0.316443717683753*v_B3_b_r + 0.59882052796136*v_B3_c_i + 1.10755301189314*v_B3_c_r + 0.171091579417532*v_B4_a_i + 0.316443717683753*v_B4_a_r + 0.171091579417531*v_B4_b_i + 0.316443717683753*v_B4_b_r - 0.59882052796136*v_B4_c_i - 1.10755301189314*v_B4_c_r
        struct[0].g[117,0] = -i_l_B3_B4_c_i - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417531*v_B3_b_r + 1.10755301189314*v_B3_c_i - 0.59882052796136*v_B3_c_r + 0.316443717683753*v_B4_a_i - 0.171091579417532*v_B4_a_r + 0.316443717683753*v_B4_b_i - 0.171091579417531*v_B4_b_r - 1.10755301189314*v_B4_c_i + 0.59882052796136*v_B4_c_r
        struct[0].g[118,0] = -i_l_B5_B6_a_r + 0.598820527961361*v_B5_a_i + 1.10755301189314*v_B5_a_r - 0.171091579417532*v_B5_b_i - 0.316443717683753*v_B5_b_r - 0.171091579417532*v_B5_c_i - 0.316443717683753*v_B5_c_r - 0.598820527961361*v_B6_a_i - 1.10755301189314*v_B6_a_r + 0.171091579417532*v_B6_b_i + 0.316443717683753*v_B6_b_r + 0.171091579417532*v_B6_c_i + 0.316443717683753*v_B6_c_r
        struct[0].g[119,0] = -i_l_B5_B6_a_i + 1.10755301189314*v_B5_a_i - 0.598820527961361*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417532*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417532*v_B5_c_r - 1.10755301189314*v_B6_a_i + 0.598820527961361*v_B6_a_r + 0.316443717683753*v_B6_b_i - 0.171091579417532*v_B6_b_r + 0.316443717683753*v_B6_c_i - 0.171091579417532*v_B6_c_r
        struct[0].g[120,0] = -i_l_B5_B6_b_r - 0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r + 0.59882052796136*v_B5_b_i + 1.10755301189314*v_B5_b_r - 0.171091579417531*v_B5_c_i - 0.316443717683753*v_B5_c_r + 0.171091579417532*v_B6_a_i + 0.316443717683753*v_B6_a_r - 0.59882052796136*v_B6_b_i - 1.10755301189314*v_B6_b_r + 0.171091579417531*v_B6_c_i + 0.316443717683753*v_B6_c_r
        struct[0].g[121,0] = -i_l_B5_B6_b_i - 0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r + 1.10755301189314*v_B5_b_i - 0.59882052796136*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417531*v_B5_c_r + 0.316443717683753*v_B6_a_i - 0.171091579417532*v_B6_a_r - 1.10755301189314*v_B6_b_i + 0.59882052796136*v_B6_b_r + 0.316443717683753*v_B6_c_i - 0.171091579417531*v_B6_c_r
        struct[0].g[122,0] = -i_l_B5_B6_c_r - 0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r - 0.171091579417531*v_B5_b_i - 0.316443717683753*v_B5_b_r + 0.59882052796136*v_B5_c_i + 1.10755301189314*v_B5_c_r + 0.171091579417532*v_B6_a_i + 0.316443717683753*v_B6_a_r + 0.171091579417531*v_B6_b_i + 0.316443717683753*v_B6_b_r - 0.59882052796136*v_B6_c_i - 1.10755301189314*v_B6_c_r
        struct[0].g[123,0] = -i_l_B5_B6_c_i - 0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417531*v_B5_b_r + 1.10755301189314*v_B5_c_i - 0.59882052796136*v_B5_c_r + 0.316443717683753*v_B6_a_i - 0.171091579417532*v_B6_a_r + 0.316443717683753*v_B6_b_i - 0.171091579417531*v_B6_b_r - 1.10755301189314*v_B6_c_i + 0.59882052796136*v_B6_c_r
        struct[0].g[124,0] = -i_l_B6_B7_a_r + 0.598820527961361*v_B6_a_i + 1.10755301189314*v_B6_a_r - 0.171091579417532*v_B6_b_i - 0.316443717683753*v_B6_b_r - 0.171091579417532*v_B6_c_i - 0.316443717683753*v_B6_c_r - 0.598820527961361*v_B7_a_i - 1.10755301189314*v_B7_a_r + 0.171091579417532*v_B7_b_i + 0.316443717683753*v_B7_b_r + 0.171091579417532*v_B7_c_i + 0.316443717683753*v_B7_c_r
        struct[0].g[125,0] = -i_l_B6_B7_a_i + 1.10755301189314*v_B6_a_i - 0.598820527961361*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417532*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417532*v_B6_c_r - 1.10755301189314*v_B7_a_i + 0.598820527961361*v_B7_a_r + 0.316443717683753*v_B7_b_i - 0.171091579417532*v_B7_b_r + 0.316443717683753*v_B7_c_i - 0.171091579417532*v_B7_c_r
        struct[0].g[126,0] = -i_l_B6_B7_b_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r + 0.59882052796136*v_B6_b_i + 1.10755301189314*v_B6_b_r - 0.171091579417531*v_B6_c_i - 0.316443717683753*v_B6_c_r + 0.171091579417532*v_B7_a_i + 0.316443717683753*v_B7_a_r - 0.59882052796136*v_B7_b_i - 1.10755301189314*v_B7_b_r + 0.171091579417531*v_B7_c_i + 0.316443717683753*v_B7_c_r
        struct[0].g[127,0] = -i_l_B6_B7_b_i - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r + 1.10755301189314*v_B6_b_i - 0.59882052796136*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417531*v_B6_c_r + 0.316443717683753*v_B7_a_i - 0.171091579417532*v_B7_a_r - 1.10755301189314*v_B7_b_i + 0.59882052796136*v_B7_b_r + 0.316443717683753*v_B7_c_i - 0.171091579417531*v_B7_c_r
        struct[0].g[128,0] = -i_l_B6_B7_c_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r - 0.171091579417531*v_B6_b_i - 0.316443717683753*v_B6_b_r + 0.59882052796136*v_B6_c_i + 1.10755301189314*v_B6_c_r + 0.171091579417532*v_B7_a_i + 0.316443717683753*v_B7_a_r + 0.171091579417531*v_B7_b_i + 0.316443717683753*v_B7_b_r - 0.59882052796136*v_B7_c_i - 1.10755301189314*v_B7_c_r
        struct[0].g[129,0] = -i_l_B6_B7_c_i - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417531*v_B6_b_r + 1.10755301189314*v_B6_c_i - 0.59882052796136*v_B6_c_r + 0.316443717683753*v_B7_a_i - 0.171091579417532*v_B7_a_r + 0.316443717683753*v_B7_b_i - 0.171091579417531*v_B7_b_r - 1.10755301189314*v_B7_c_i + 0.59882052796136*v_B7_c_r
        struct[0].g[130,0] = i_load_B2lv_a_i*v_B2lv_a_i - i_load_B2lv_a_i*v_B2lv_n_i + i_load_B2lv_a_r*v_B2lv_a_r - i_load_B2lv_a_r*v_B2lv_n_r - p_B2lv_a
        struct[0].g[131,0] = i_load_B2lv_b_i*v_B2lv_b_i - i_load_B2lv_b_i*v_B2lv_n_i + i_load_B2lv_b_r*v_B2lv_b_r - i_load_B2lv_b_r*v_B2lv_n_r - p_B2lv_b
        struct[0].g[132,0] = i_load_B2lv_c_i*v_B2lv_c_i - i_load_B2lv_c_i*v_B2lv_n_i + i_load_B2lv_c_r*v_B2lv_c_r - i_load_B2lv_c_r*v_B2lv_n_r - p_B2lv_c
        struct[0].g[133,0] = -i_load_B2lv_a_i*v_B2lv_a_r + i_load_B2lv_a_i*v_B2lv_n_r + i_load_B2lv_a_r*v_B2lv_a_i - i_load_B2lv_a_r*v_B2lv_n_i - q_B2lv_a
        struct[0].g[134,0] = -i_load_B2lv_b_i*v_B2lv_b_r + i_load_B2lv_b_i*v_B2lv_n_r + i_load_B2lv_b_r*v_B2lv_b_i - i_load_B2lv_b_r*v_B2lv_n_i - q_B2lv_b
        struct[0].g[135,0] = -i_load_B2lv_c_i*v_B2lv_c_r + i_load_B2lv_c_i*v_B2lv_n_r + i_load_B2lv_c_r*v_B2lv_c_i - i_load_B2lv_c_r*v_B2lv_n_i - q_B2lv_c
        struct[0].g[136,0] = i_load_B2lv_a_r + i_load_B2lv_b_r + i_load_B2lv_c_r + i_load_B2lv_n_r
        struct[0].g[137,0] = i_load_B2lv_a_i + i_load_B2lv_b_i + i_load_B2lv_c_i + i_load_B2lv_n_i
        struct[0].g[138,0] = i_load_B3lv_a_i*v_B3lv_a_i - i_load_B3lv_a_i*v_B3lv_n_i + i_load_B3lv_a_r*v_B3lv_a_r - i_load_B3lv_a_r*v_B3lv_n_r - p_B3lv_a
        struct[0].g[139,0] = i_load_B3lv_b_i*v_B3lv_b_i - i_load_B3lv_b_i*v_B3lv_n_i + i_load_B3lv_b_r*v_B3lv_b_r - i_load_B3lv_b_r*v_B3lv_n_r - p_B3lv_b
        struct[0].g[140,0] = i_load_B3lv_c_i*v_B3lv_c_i - i_load_B3lv_c_i*v_B3lv_n_i + i_load_B3lv_c_r*v_B3lv_c_r - i_load_B3lv_c_r*v_B3lv_n_r - p_B3lv_c
        struct[0].g[141,0] = -i_load_B3lv_a_i*v_B3lv_a_r + i_load_B3lv_a_i*v_B3lv_n_r + i_load_B3lv_a_r*v_B3lv_a_i - i_load_B3lv_a_r*v_B3lv_n_i - q_B3lv_a
        struct[0].g[142,0] = -i_load_B3lv_b_i*v_B3lv_b_r + i_load_B3lv_b_i*v_B3lv_n_r + i_load_B3lv_b_r*v_B3lv_b_i - i_load_B3lv_b_r*v_B3lv_n_i - q_B3lv_b
        struct[0].g[143,0] = -i_load_B3lv_c_i*v_B3lv_c_r + i_load_B3lv_c_i*v_B3lv_n_r + i_load_B3lv_c_r*v_B3lv_c_i - i_load_B3lv_c_r*v_B3lv_n_i - q_B3lv_c
        struct[0].g[144,0] = i_load_B3lv_a_r + i_load_B3lv_b_r + i_load_B3lv_c_r + i_load_B3lv_n_r
        struct[0].g[145,0] = i_load_B3lv_a_i + i_load_B3lv_b_i + i_load_B3lv_c_i + i_load_B3lv_n_i
        struct[0].g[146,0] = i_load_B4lv_a_i*v_B4lv_a_i - i_load_B4lv_a_i*v_B4lv_n_i + i_load_B4lv_a_r*v_B4lv_a_r - i_load_B4lv_a_r*v_B4lv_n_r - p_B4lv_a
        struct[0].g[147,0] = i_load_B4lv_b_i*v_B4lv_b_i - i_load_B4lv_b_i*v_B4lv_n_i + i_load_B4lv_b_r*v_B4lv_b_r - i_load_B4lv_b_r*v_B4lv_n_r - p_B4lv_b
        struct[0].g[148,0] = i_load_B4lv_c_i*v_B4lv_c_i - i_load_B4lv_c_i*v_B4lv_n_i + i_load_B4lv_c_r*v_B4lv_c_r - i_load_B4lv_c_r*v_B4lv_n_r - p_B4lv_c
        struct[0].g[149,0] = -i_load_B4lv_a_i*v_B4lv_a_r + i_load_B4lv_a_i*v_B4lv_n_r + i_load_B4lv_a_r*v_B4lv_a_i - i_load_B4lv_a_r*v_B4lv_n_i - q_B4lv_a
        struct[0].g[150,0] = -i_load_B4lv_b_i*v_B4lv_b_r + i_load_B4lv_b_i*v_B4lv_n_r + i_load_B4lv_b_r*v_B4lv_b_i - i_load_B4lv_b_r*v_B4lv_n_i - q_B4lv_b
        struct[0].g[151,0] = -i_load_B4lv_c_i*v_B4lv_c_r + i_load_B4lv_c_i*v_B4lv_n_r + i_load_B4lv_c_r*v_B4lv_c_i - i_load_B4lv_c_r*v_B4lv_n_i - q_B4lv_c
        struct[0].g[152,0] = i_load_B4lv_a_r + i_load_B4lv_b_r + i_load_B4lv_c_r + i_load_B4lv_n_r
        struct[0].g[153,0] = i_load_B4lv_a_i + i_load_B4lv_b_i + i_load_B4lv_c_i + i_load_B4lv_n_i
        struct[0].g[154,0] = i_load_B5lv_a_i*v_B5lv_a_i - i_load_B5lv_a_i*v_B5lv_n_i + i_load_B5lv_a_r*v_B5lv_a_r - i_load_B5lv_a_r*v_B5lv_n_r - p_B5lv_a
        struct[0].g[155,0] = i_load_B5lv_b_i*v_B5lv_b_i - i_load_B5lv_b_i*v_B5lv_n_i + i_load_B5lv_b_r*v_B5lv_b_r - i_load_B5lv_b_r*v_B5lv_n_r - p_B5lv_b
        struct[0].g[156,0] = i_load_B5lv_c_i*v_B5lv_c_i - i_load_B5lv_c_i*v_B5lv_n_i + i_load_B5lv_c_r*v_B5lv_c_r - i_load_B5lv_c_r*v_B5lv_n_r - p_B5lv_c
        struct[0].g[157,0] = -i_load_B5lv_a_i*v_B5lv_a_r + i_load_B5lv_a_i*v_B5lv_n_r + i_load_B5lv_a_r*v_B5lv_a_i - i_load_B5lv_a_r*v_B5lv_n_i - q_B5lv_a
        struct[0].g[158,0] = -i_load_B5lv_b_i*v_B5lv_b_r + i_load_B5lv_b_i*v_B5lv_n_r + i_load_B5lv_b_r*v_B5lv_b_i - i_load_B5lv_b_r*v_B5lv_n_i - q_B5lv_b
        struct[0].g[159,0] = -i_load_B5lv_c_i*v_B5lv_c_r + i_load_B5lv_c_i*v_B5lv_n_r + i_load_B5lv_c_r*v_B5lv_c_i - i_load_B5lv_c_r*v_B5lv_n_i - q_B5lv_c
        struct[0].g[160,0] = i_load_B5lv_a_r + i_load_B5lv_b_r + i_load_B5lv_c_r + i_load_B5lv_n_r
        struct[0].g[161,0] = i_load_B5lv_a_i + i_load_B5lv_b_i + i_load_B5lv_c_i + i_load_B5lv_n_i
        struct[0].g[162,0] = i_load_B6lv_a_i*v_B6lv_a_i - i_load_B6lv_a_i*v_B6lv_n_i + i_load_B6lv_a_r*v_B6lv_a_r - i_load_B6lv_a_r*v_B6lv_n_r - p_B6lv_a
        struct[0].g[163,0] = i_load_B6lv_b_i*v_B6lv_b_i - i_load_B6lv_b_i*v_B6lv_n_i + i_load_B6lv_b_r*v_B6lv_b_r - i_load_B6lv_b_r*v_B6lv_n_r - p_B6lv_b
        struct[0].g[164,0] = i_load_B6lv_c_i*v_B6lv_c_i - i_load_B6lv_c_i*v_B6lv_n_i + i_load_B6lv_c_r*v_B6lv_c_r - i_load_B6lv_c_r*v_B6lv_n_r - p_B6lv_c
        struct[0].g[165,0] = -i_load_B6lv_a_i*v_B6lv_a_r + i_load_B6lv_a_i*v_B6lv_n_r + i_load_B6lv_a_r*v_B6lv_a_i - i_load_B6lv_a_r*v_B6lv_n_i - q_B6lv_a
        struct[0].g[166,0] = -i_load_B6lv_b_i*v_B6lv_b_r + i_load_B6lv_b_i*v_B6lv_n_r + i_load_B6lv_b_r*v_B6lv_b_i - i_load_B6lv_b_r*v_B6lv_n_i - q_B6lv_b
        struct[0].g[167,0] = -i_load_B6lv_c_i*v_B6lv_c_r + i_load_B6lv_c_i*v_B6lv_n_r + i_load_B6lv_c_r*v_B6lv_c_i - i_load_B6lv_c_r*v_B6lv_n_i - q_B6lv_c
        struct[0].g[168,0] = i_load_B6lv_a_r + i_load_B6lv_b_r + i_load_B6lv_c_r + i_load_B6lv_n_r
        struct[0].g[169,0] = i_load_B6lv_a_i + i_load_B6lv_b_i + i_load_B6lv_c_i + i_load_B6lv_n_i
        struct[0].g[170,0] = -R_1112*i_l_1112 + v_11 - v_12
        struct[0].g[171,0] = -R_1213*i_l_1213 + v_12 - v_13
        struct[0].g[172,0] = -R_1314*i_l_1314 + v_13 - v_14
        struct[0].g[173,0] = -R_1415*i_l_1415 + v_14 - v_15
        struct[0].g[174,0] = -R_2122*i_l_2122 + v_21 - v_22
        struct[0].g[175,0] = -R_2223*i_l_2223 + v_22 - v_23
        struct[0].g[176,0] = -R_2324*i_l_2324 + v_23 - v_24
        struct[0].g[177,0] = -R_2425*i_l_2425 + v_24 - v_25
        struct[0].g[178,0] = -R_3132*i_l_3132 + v_31 - v_32
        struct[0].g[179,0] = -R_3233*i_l_3233 + v_32 - v_33
        struct[0].g[180,0] = -R_3334*i_l_3334 + v_33 - v_34
        struct[0].g[181,0] = -R_3435*i_l_3435 + v_34 - v_35
        struct[0].g[182,0] = -R_4142*i_l_4142 + v_41 - v_42
        struct[0].g[183,0] = -R_4243*i_l_4243 + v_42 - v_43
        struct[0].g[184,0] = -R_4344*i_l_4344 + v_43 - v_44
        struct[0].g[185,0] = -R_4445*i_l_4445 + v_44 - v_45
        struct[0].g[186,0] = -R_5152*i_l_5152 + v_51 - v_52
        struct[0].g[187,0] = -R_5253*i_l_5253 + v_52 - v_53
        struct[0].g[188,0] = -R_5354*i_l_5354 + v_53 - v_54
        struct[0].g[189,0] = -R_5455*i_l_5455 + v_54 - v_55
        struct[0].g[190,0] = -R_1521*i_l_1521 + v_15 - v_21
        struct[0].g[191,0] = -R_2531*i_l_2531 + v_25 - v_31
        struct[0].g[192,0] = -R_3541*i_l_3541 + v_35 - v_41
        struct[0].g[193,0] = -R_4551*i_l_4551 + v_45 - v_51
        struct[0].g[194,0] = i_11 - i_l_1112
        struct[0].g[195,0] = i_12 + i_l_1112 - i_l_1213
        struct[0].g[196,0] = i_13 + i_l_1213 - i_l_1314
        struct[0].g[197,0] = i_14 + i_l_1314 - i_l_1415
        struct[0].g[198,0] = i_15 + i_l_1415 - i_l_1521
        struct[0].g[199,0] = i_21 + i_l_1521 - i_l_2122
        struct[0].g[200,0] = i_22 + i_l_2122 - i_l_2223
        struct[0].g[201,0] = i_23 + i_l_2223 - i_l_2324
        struct[0].g[202,0] = i_24 + i_l_2324 - i_l_2425
        struct[0].g[203,0] = i_25 + i_l_2425 - i_l_2531
        struct[0].g[204,0] = i_31 + i_l_2531 - i_l_3132
        struct[0].g[205,0] = i_32 + i_l_3132 - i_l_3233
        struct[0].g[206,0] = i_33 + i_l_3233 - i_l_3334
        struct[0].g[207,0] = i_34 + i_l_3334 - i_l_3435
        struct[0].g[208,0] = i_35 + i_l_3435 - i_l_3541
        struct[0].g[209,0] = i_41 + i_l_3541 - i_l_4142
        struct[0].g[210,0] = i_42 + i_l_4142 - i_l_4243
        struct[0].g[211,0] = i_43 + i_l_4243 - i_l_4344
        struct[0].g[212,0] = i_44 + i_l_4344 - i_l_4445
        struct[0].g[213,0] = i_45 + i_l_4445 - i_l_4551
        struct[0].g[214,0] = i_51 + i_l_4551 - i_l_5152
        struct[0].g[215,0] = i_52 + i_l_5152 - i_l_5253
        struct[0].g[216,0] = i_53 + i_l_5253 - i_l_5354
        struct[0].g[217,0] = i_54 + i_l_5354 - i_l_5455
        struct[0].g[218,0] = i_55 + i_l_5455
        struct[0].g[219,0] = i_11*v_11 - p_11
        struct[0].g[220,0] = i_12*v_12 - p_12
        struct[0].g[221,0] = i_14*v_14 - p_14
        struct[0].g[222,0] = i_15*v_15 - p_15
        struct[0].g[223,0] = i_21*v_21 - p_21
        struct[0].g[224,0] = i_22*v_22 - p_22
        struct[0].g[225,0] = i_24*v_24 - p_24
        struct[0].g[226,0] = i_25*v_25 - p_25
        struct[0].g[227,0] = i_31*v_31 - p_31
        struct[0].g[228,0] = i_32*v_32 - p_32
        struct[0].g[229,0] = i_34*v_34 - p_34
        struct[0].g[230,0] = i_35*v_35 - p_35
        struct[0].g[231,0] = i_41*v_41 - p_41
        struct[0].g[232,0] = i_42*v_42 - p_42
        struct[0].g[233,0] = i_44*v_44 - p_44
        struct[0].g[234,0] = i_45*v_45 - p_45
        struct[0].g[235,0] = i_51*v_51 - p_51
        struct[0].g[236,0] = i_52*v_52 - p_52
        struct[0].g[237,0] = i_54*v_54 - p_54
        struct[0].g[238,0] = i_55*v_55 - p_55
        struct[0].g[239,0] = i_13*v_13/3 + p_B2lv_a + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[240,0] = i_13*v_13/3 + p_B2lv_b + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[241,0] = i_13*v_13/3 + p_B2lv_c + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[242,0] = i_23*v_23/3 + p_B3lv_a + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[243,0] = i_23*v_23/3 + p_B3lv_b + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[244,0] = i_23*v_23/3 + p_B3lv_c + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[245,0] = i_33*v_33/3 + p_B4lv_a + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[246,0] = i_33*v_33/3 + p_B4lv_b + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[247,0] = i_33*v_33/3 + p_B4lv_c + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[248,0] = i_43*v_43/3 + p_B5lv_a + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[249,0] = i_43*v_43/3 + p_B5lv_b + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[250,0] = i_43*v_43/3 + p_B5lv_c + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[251,0] = i_53*v_53/3 + p_B6lv_a + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
        struct[0].g[252,0] = i_53*v_53/3 + p_B6lv_b + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
        struct[0].g[253,0] = i_53*v_53/3 + p_B6lv_c + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
    
    # Outputs:
    if mode == 3:

        struct[0].h[0,0] = i_13*v_13
        struct[0].h[1,0] = v_13
        struct[0].h[2,0] = i_23*v_23
        struct[0].h[3,0] = v_23
        struct[0].h[4,0] = i_33*v_33
        struct[0].h[5,0] = v_33
        struct[0].h[6,0] = i_43*v_43
        struct[0].h[7,0] = v_43
        struct[0].h[8,0] = i_53*v_53
        struct[0].h[9,0] = v_53
    

    if mode == 10:

        struct[0].Fx_ini[0,0] = (-K_r*i_13 - 1)/T_v
        struct[0].Fx_ini[1,1] = (-K_r*i_23 - 1)/T_v
        struct[0].Fx_ini[2,2] = (-K_r*i_33 - 1)/T_v
        struct[0].Fx_ini[3,3] = (-K_r*i_43 - 1)/T_v
        struct[0].Fx_ini[4,4] = (-K_r*i_53 - 1)/T_v

    if mode == 11:

        struct[0].Fy_ini[0,196] = -K_r*v_13/T_v 
        struct[0].Fy_ini[1,201] = -K_r*v_23/T_v 
        struct[0].Fy_ini[2,206] = -K_r*v_33/T_v 
        struct[0].Fy_ini[3,211] = -K_r*v_43/T_v 
        struct[0].Fy_ini[4,216] = -K_r*v_53/T_v 

        struct[0].Gy_ini[0,0] = -5.75163398692810
        struct[0].Gy_ini[0,1] = -23.0065359477124
        struct[0].Gy_ini[0,6] = 5.75163398692810
        struct[0].Gy_ini[0,7] = 23.0065359477124
        struct[0].Gy_ini[0,40] = 0.249053057297486
        struct[0].Gy_ini[0,41] = 0.996212229189942
        struct[0].Gy_ini[0,44] = -0.249053057297486
        struct[0].Gy_ini[0,45] = -0.996212229189942
        struct[0].Gy_ini[0,130] = 1
        struct[0].Gy_ini[1,0] = 23.0065359477124
        struct[0].Gy_ini[1,1] = -5.75163398692810
        struct[0].Gy_ini[1,6] = -23.0065359477124
        struct[0].Gy_ini[1,7] = 5.75163398692810
        struct[0].Gy_ini[1,40] = -0.996212229189942
        struct[0].Gy_ini[1,41] = 0.249053057297486
        struct[0].Gy_ini[1,44] = 0.996212229189942
        struct[0].Gy_ini[1,45] = -0.249053057297486
        struct[0].Gy_ini[1,131] = 1
        struct[0].Gy_ini[2,2] = -5.75163398692810
        struct[0].Gy_ini[2,3] = -23.0065359477124
        struct[0].Gy_ini[2,6] = 5.75163398692810
        struct[0].Gy_ini[2,7] = 23.0065359477124
        struct[0].Gy_ini[2,40] = -0.249053057297486
        struct[0].Gy_ini[2,41] = -0.996212229189942
        struct[0].Gy_ini[2,42] = 0.249053057297486
        struct[0].Gy_ini[2,43] = 0.996212229189942
        struct[0].Gy_ini[2,132] = 1
        struct[0].Gy_ini[3,2] = 23.0065359477124
        struct[0].Gy_ini[3,3] = -5.75163398692810
        struct[0].Gy_ini[3,6] = -23.0065359477124
        struct[0].Gy_ini[3,7] = 5.75163398692810
        struct[0].Gy_ini[3,40] = 0.996212229189942
        struct[0].Gy_ini[3,41] = -0.249053057297486
        struct[0].Gy_ini[3,42] = -0.996212229189942
        struct[0].Gy_ini[3,43] = 0.249053057297486
        struct[0].Gy_ini[3,133] = 1
        struct[0].Gy_ini[4,4] = -5.75163398692810
        struct[0].Gy_ini[4,5] = -23.0065359477124
        struct[0].Gy_ini[4,6] = 5.75163398692810
        struct[0].Gy_ini[4,7] = 23.0065359477124
        struct[0].Gy_ini[4,42] = -0.249053057297486
        struct[0].Gy_ini[4,43] = -0.996212229189942
        struct[0].Gy_ini[4,44] = 0.249053057297486
        struct[0].Gy_ini[4,45] = 0.996212229189942
        struct[0].Gy_ini[4,134] = 1
        struct[0].Gy_ini[5,4] = 23.0065359477124
        struct[0].Gy_ini[5,5] = -5.75163398692810
        struct[0].Gy_ini[5,6] = -23.0065359477124
        struct[0].Gy_ini[5,7] = 5.75163398692810
        struct[0].Gy_ini[5,42] = 0.996212229189942
        struct[0].Gy_ini[5,43] = -0.249053057297486
        struct[0].Gy_ini[5,44] = -0.996212229189942
        struct[0].Gy_ini[5,45] = 0.249053057297486
        struct[0].Gy_ini[5,135] = 1
        struct[0].Gy_ini[6,0] = 5.75163398692810
        struct[0].Gy_ini[6,1] = 23.0065359477124
        struct[0].Gy_ini[6,2] = 5.75163398692810
        struct[0].Gy_ini[6,3] = 23.0065359477124
        struct[0].Gy_ini[6,4] = 5.75163398692810
        struct[0].Gy_ini[6,5] = 23.0065359477124
        struct[0].Gy_ini[6,6] = -1017.25490196078
        struct[0].Gy_ini[6,7] = -69.0196078431372
        struct[0].Gy_ini[7,0] = -23.0065359477124
        struct[0].Gy_ini[7,1] = 5.75163398692810
        struct[0].Gy_ini[7,2] = -23.0065359477124
        struct[0].Gy_ini[7,3] = 5.75163398692810
        struct[0].Gy_ini[7,4] = -23.0065359477124
        struct[0].Gy_ini[7,5] = 5.75163398692810
        struct[0].Gy_ini[7,6] = 69.0196078431372
        struct[0].Gy_ini[7,7] = -1017.25490196078
        struct[0].Gy_ini[8,8] = -5.75163398692810
        struct[0].Gy_ini[8,9] = -23.0065359477124
        struct[0].Gy_ini[8,14] = 5.75163398692810
        struct[0].Gy_ini[8,15] = 23.0065359477124
        struct[0].Gy_ini[8,46] = 0.249053057297486
        struct[0].Gy_ini[8,47] = 0.996212229189942
        struct[0].Gy_ini[8,50] = -0.249053057297486
        struct[0].Gy_ini[8,51] = -0.996212229189942
        struct[0].Gy_ini[8,138] = 1
        struct[0].Gy_ini[9,8] = 23.0065359477124
        struct[0].Gy_ini[9,9] = -5.75163398692810
        struct[0].Gy_ini[9,14] = -23.0065359477124
        struct[0].Gy_ini[9,15] = 5.75163398692810
        struct[0].Gy_ini[9,46] = -0.996212229189942
        struct[0].Gy_ini[9,47] = 0.249053057297486
        struct[0].Gy_ini[9,50] = 0.996212229189942
        struct[0].Gy_ini[9,51] = -0.249053057297486
        struct[0].Gy_ini[9,139] = 1
        struct[0].Gy_ini[10,10] = -5.75163398692810
        struct[0].Gy_ini[10,11] = -23.0065359477124
        struct[0].Gy_ini[10,14] = 5.75163398692810
        struct[0].Gy_ini[10,15] = 23.0065359477124
        struct[0].Gy_ini[10,46] = -0.249053057297486
        struct[0].Gy_ini[10,47] = -0.996212229189942
        struct[0].Gy_ini[10,48] = 0.249053057297486
        struct[0].Gy_ini[10,49] = 0.996212229189942
        struct[0].Gy_ini[10,140] = 1
        struct[0].Gy_ini[11,10] = 23.0065359477124
        struct[0].Gy_ini[11,11] = -5.75163398692810
        struct[0].Gy_ini[11,14] = -23.0065359477124
        struct[0].Gy_ini[11,15] = 5.75163398692810
        struct[0].Gy_ini[11,46] = 0.996212229189942
        struct[0].Gy_ini[11,47] = -0.249053057297486
        struct[0].Gy_ini[11,48] = -0.996212229189942
        struct[0].Gy_ini[11,49] = 0.249053057297486
        struct[0].Gy_ini[11,141] = 1
        struct[0].Gy_ini[12,12] = -5.75163398692810
        struct[0].Gy_ini[12,13] = -23.0065359477124
        struct[0].Gy_ini[12,14] = 5.75163398692810
        struct[0].Gy_ini[12,15] = 23.0065359477124
        struct[0].Gy_ini[12,48] = -0.249053057297486
        struct[0].Gy_ini[12,49] = -0.996212229189942
        struct[0].Gy_ini[12,50] = 0.249053057297486
        struct[0].Gy_ini[12,51] = 0.996212229189942
        struct[0].Gy_ini[12,142] = 1
        struct[0].Gy_ini[13,12] = 23.0065359477124
        struct[0].Gy_ini[13,13] = -5.75163398692810
        struct[0].Gy_ini[13,14] = -23.0065359477124
        struct[0].Gy_ini[13,15] = 5.75163398692810
        struct[0].Gy_ini[13,48] = 0.996212229189942
        struct[0].Gy_ini[13,49] = -0.249053057297486
        struct[0].Gy_ini[13,50] = -0.996212229189942
        struct[0].Gy_ini[13,51] = 0.249053057297486
        struct[0].Gy_ini[13,143] = 1
        struct[0].Gy_ini[14,8] = 5.75163398692810
        struct[0].Gy_ini[14,9] = 23.0065359477124
        struct[0].Gy_ini[14,10] = 5.75163398692810
        struct[0].Gy_ini[14,11] = 23.0065359477124
        struct[0].Gy_ini[14,12] = 5.75163398692810
        struct[0].Gy_ini[14,13] = 23.0065359477124
        struct[0].Gy_ini[14,14] = -1017.25490196078
        struct[0].Gy_ini[14,15] = -69.0196078431372
        struct[0].Gy_ini[15,8] = -23.0065359477124
        struct[0].Gy_ini[15,9] = 5.75163398692810
        struct[0].Gy_ini[15,10] = -23.0065359477124
        struct[0].Gy_ini[15,11] = 5.75163398692810
        struct[0].Gy_ini[15,12] = -23.0065359477124
        struct[0].Gy_ini[15,13] = 5.75163398692810
        struct[0].Gy_ini[15,14] = 69.0196078431372
        struct[0].Gy_ini[15,15] = -1017.25490196078
        struct[0].Gy_ini[16,16] = -5.75163398692810
        struct[0].Gy_ini[16,17] = -23.0065359477124
        struct[0].Gy_ini[16,22] = 5.75163398692810
        struct[0].Gy_ini[16,23] = 23.0065359477124
        struct[0].Gy_ini[16,52] = 0.249053057297486
        struct[0].Gy_ini[16,53] = 0.996212229189942
        struct[0].Gy_ini[16,56] = -0.249053057297486
        struct[0].Gy_ini[16,57] = -0.996212229189942
        struct[0].Gy_ini[16,146] = 1
        struct[0].Gy_ini[17,16] = 23.0065359477124
        struct[0].Gy_ini[17,17] = -5.75163398692810
        struct[0].Gy_ini[17,22] = -23.0065359477124
        struct[0].Gy_ini[17,23] = 5.75163398692810
        struct[0].Gy_ini[17,52] = -0.996212229189942
        struct[0].Gy_ini[17,53] = 0.249053057297486
        struct[0].Gy_ini[17,56] = 0.996212229189942
        struct[0].Gy_ini[17,57] = -0.249053057297486
        struct[0].Gy_ini[17,147] = 1
        struct[0].Gy_ini[18,18] = -5.75163398692810
        struct[0].Gy_ini[18,19] = -23.0065359477124
        struct[0].Gy_ini[18,22] = 5.75163398692810
        struct[0].Gy_ini[18,23] = 23.0065359477124
        struct[0].Gy_ini[18,52] = -0.249053057297486
        struct[0].Gy_ini[18,53] = -0.996212229189942
        struct[0].Gy_ini[18,54] = 0.249053057297486
        struct[0].Gy_ini[18,55] = 0.996212229189942
        struct[0].Gy_ini[18,148] = 1
        struct[0].Gy_ini[19,18] = 23.0065359477124
        struct[0].Gy_ini[19,19] = -5.75163398692810
        struct[0].Gy_ini[19,22] = -23.0065359477124
        struct[0].Gy_ini[19,23] = 5.75163398692810
        struct[0].Gy_ini[19,52] = 0.996212229189942
        struct[0].Gy_ini[19,53] = -0.249053057297486
        struct[0].Gy_ini[19,54] = -0.996212229189942
        struct[0].Gy_ini[19,55] = 0.249053057297486
        struct[0].Gy_ini[19,149] = 1
        struct[0].Gy_ini[20,20] = -5.75163398692810
        struct[0].Gy_ini[20,21] = -23.0065359477124
        struct[0].Gy_ini[20,22] = 5.75163398692810
        struct[0].Gy_ini[20,23] = 23.0065359477124
        struct[0].Gy_ini[20,54] = -0.249053057297486
        struct[0].Gy_ini[20,55] = -0.996212229189942
        struct[0].Gy_ini[20,56] = 0.249053057297486
        struct[0].Gy_ini[20,57] = 0.996212229189942
        struct[0].Gy_ini[20,150] = 1
        struct[0].Gy_ini[21,20] = 23.0065359477124
        struct[0].Gy_ini[21,21] = -5.75163398692810
        struct[0].Gy_ini[21,22] = -23.0065359477124
        struct[0].Gy_ini[21,23] = 5.75163398692810
        struct[0].Gy_ini[21,54] = 0.996212229189942
        struct[0].Gy_ini[21,55] = -0.249053057297486
        struct[0].Gy_ini[21,56] = -0.996212229189942
        struct[0].Gy_ini[21,57] = 0.249053057297486
        struct[0].Gy_ini[21,151] = 1
        struct[0].Gy_ini[22,16] = 5.75163398692810
        struct[0].Gy_ini[22,17] = 23.0065359477124
        struct[0].Gy_ini[22,18] = 5.75163398692810
        struct[0].Gy_ini[22,19] = 23.0065359477124
        struct[0].Gy_ini[22,20] = 5.75163398692810
        struct[0].Gy_ini[22,21] = 23.0065359477124
        struct[0].Gy_ini[22,22] = -1017.25490196078
        struct[0].Gy_ini[22,23] = -69.0196078431372
        struct[0].Gy_ini[23,16] = -23.0065359477124
        struct[0].Gy_ini[23,17] = 5.75163398692810
        struct[0].Gy_ini[23,18] = -23.0065359477124
        struct[0].Gy_ini[23,19] = 5.75163398692810
        struct[0].Gy_ini[23,20] = -23.0065359477124
        struct[0].Gy_ini[23,21] = 5.75163398692810
        struct[0].Gy_ini[23,22] = 69.0196078431372
        struct[0].Gy_ini[23,23] = -1017.25490196078
        struct[0].Gy_ini[24,24] = -5.75163398692810
        struct[0].Gy_ini[24,25] = -23.0065359477124
        struct[0].Gy_ini[24,30] = 5.75163398692810
        struct[0].Gy_ini[24,31] = 23.0065359477124
        struct[0].Gy_ini[24,58] = 0.249053057297486
        struct[0].Gy_ini[24,59] = 0.996212229189942
        struct[0].Gy_ini[24,62] = -0.249053057297486
        struct[0].Gy_ini[24,63] = -0.996212229189942
        struct[0].Gy_ini[24,154] = 1
        struct[0].Gy_ini[25,24] = 23.0065359477124
        struct[0].Gy_ini[25,25] = -5.75163398692810
        struct[0].Gy_ini[25,30] = -23.0065359477124
        struct[0].Gy_ini[25,31] = 5.75163398692810
        struct[0].Gy_ini[25,58] = -0.996212229189942
        struct[0].Gy_ini[25,59] = 0.249053057297486
        struct[0].Gy_ini[25,62] = 0.996212229189942
        struct[0].Gy_ini[25,63] = -0.249053057297486
        struct[0].Gy_ini[25,155] = 1
        struct[0].Gy_ini[26,26] = -5.75163398692810
        struct[0].Gy_ini[26,27] = -23.0065359477124
        struct[0].Gy_ini[26,30] = 5.75163398692810
        struct[0].Gy_ini[26,31] = 23.0065359477124
        struct[0].Gy_ini[26,58] = -0.249053057297486
        struct[0].Gy_ini[26,59] = -0.996212229189942
        struct[0].Gy_ini[26,60] = 0.249053057297486
        struct[0].Gy_ini[26,61] = 0.996212229189942
        struct[0].Gy_ini[26,156] = 1
        struct[0].Gy_ini[27,26] = 23.0065359477124
        struct[0].Gy_ini[27,27] = -5.75163398692810
        struct[0].Gy_ini[27,30] = -23.0065359477124
        struct[0].Gy_ini[27,31] = 5.75163398692810
        struct[0].Gy_ini[27,58] = 0.996212229189942
        struct[0].Gy_ini[27,59] = -0.249053057297486
        struct[0].Gy_ini[27,60] = -0.996212229189942
        struct[0].Gy_ini[27,61] = 0.249053057297486
        struct[0].Gy_ini[27,157] = 1
        struct[0].Gy_ini[28,28] = -5.75163398692810
        struct[0].Gy_ini[28,29] = -23.0065359477124
        struct[0].Gy_ini[28,30] = 5.75163398692810
        struct[0].Gy_ini[28,31] = 23.0065359477124
        struct[0].Gy_ini[28,60] = -0.249053057297486
        struct[0].Gy_ini[28,61] = -0.996212229189942
        struct[0].Gy_ini[28,62] = 0.249053057297486
        struct[0].Gy_ini[28,63] = 0.996212229189942
        struct[0].Gy_ini[28,158] = 1
        struct[0].Gy_ini[29,28] = 23.0065359477124
        struct[0].Gy_ini[29,29] = -5.75163398692810
        struct[0].Gy_ini[29,30] = -23.0065359477124
        struct[0].Gy_ini[29,31] = 5.75163398692810
        struct[0].Gy_ini[29,60] = 0.996212229189942
        struct[0].Gy_ini[29,61] = -0.249053057297486
        struct[0].Gy_ini[29,62] = -0.996212229189942
        struct[0].Gy_ini[29,63] = 0.249053057297486
        struct[0].Gy_ini[29,159] = 1
        struct[0].Gy_ini[30,24] = 5.75163398692810
        struct[0].Gy_ini[30,25] = 23.0065359477124
        struct[0].Gy_ini[30,26] = 5.75163398692810
        struct[0].Gy_ini[30,27] = 23.0065359477124
        struct[0].Gy_ini[30,28] = 5.75163398692810
        struct[0].Gy_ini[30,29] = 23.0065359477124
        struct[0].Gy_ini[30,30] = -1017.25490196078
        struct[0].Gy_ini[30,31] = -69.0196078431372
        struct[0].Gy_ini[31,24] = -23.0065359477124
        struct[0].Gy_ini[31,25] = 5.75163398692810
        struct[0].Gy_ini[31,26] = -23.0065359477124
        struct[0].Gy_ini[31,27] = 5.75163398692810
        struct[0].Gy_ini[31,28] = -23.0065359477124
        struct[0].Gy_ini[31,29] = 5.75163398692810
        struct[0].Gy_ini[31,30] = 69.0196078431372
        struct[0].Gy_ini[31,31] = -1017.25490196078
        struct[0].Gy_ini[32,32] = -5.75163398692810
        struct[0].Gy_ini[32,33] = -23.0065359477124
        struct[0].Gy_ini[32,38] = 5.75163398692810
        struct[0].Gy_ini[32,39] = 23.0065359477124
        struct[0].Gy_ini[32,64] = 0.249053057297486
        struct[0].Gy_ini[32,65] = 0.996212229189942
        struct[0].Gy_ini[32,68] = -0.249053057297486
        struct[0].Gy_ini[32,69] = -0.996212229189942
        struct[0].Gy_ini[32,162] = 1
        struct[0].Gy_ini[33,32] = 23.0065359477124
        struct[0].Gy_ini[33,33] = -5.75163398692810
        struct[0].Gy_ini[33,38] = -23.0065359477124
        struct[0].Gy_ini[33,39] = 5.75163398692810
        struct[0].Gy_ini[33,64] = -0.996212229189942
        struct[0].Gy_ini[33,65] = 0.249053057297486
        struct[0].Gy_ini[33,68] = 0.996212229189942
        struct[0].Gy_ini[33,69] = -0.249053057297486
        struct[0].Gy_ini[33,163] = 1
        struct[0].Gy_ini[34,34] = -5.75163398692810
        struct[0].Gy_ini[34,35] = -23.0065359477124
        struct[0].Gy_ini[34,38] = 5.75163398692810
        struct[0].Gy_ini[34,39] = 23.0065359477124
        struct[0].Gy_ini[34,64] = -0.249053057297486
        struct[0].Gy_ini[34,65] = -0.996212229189942
        struct[0].Gy_ini[34,66] = 0.249053057297486
        struct[0].Gy_ini[34,67] = 0.996212229189942
        struct[0].Gy_ini[34,164] = 1
        struct[0].Gy_ini[35,34] = 23.0065359477124
        struct[0].Gy_ini[35,35] = -5.75163398692810
        struct[0].Gy_ini[35,38] = -23.0065359477124
        struct[0].Gy_ini[35,39] = 5.75163398692810
        struct[0].Gy_ini[35,64] = 0.996212229189942
        struct[0].Gy_ini[35,65] = -0.249053057297486
        struct[0].Gy_ini[35,66] = -0.996212229189942
        struct[0].Gy_ini[35,67] = 0.249053057297486
        struct[0].Gy_ini[35,165] = 1
        struct[0].Gy_ini[36,36] = -5.75163398692810
        struct[0].Gy_ini[36,37] = -23.0065359477124
        struct[0].Gy_ini[36,38] = 5.75163398692810
        struct[0].Gy_ini[36,39] = 23.0065359477124
        struct[0].Gy_ini[36,66] = -0.249053057297486
        struct[0].Gy_ini[36,67] = -0.996212229189942
        struct[0].Gy_ini[36,68] = 0.249053057297486
        struct[0].Gy_ini[36,69] = 0.996212229189942
        struct[0].Gy_ini[36,166] = 1
        struct[0].Gy_ini[37,36] = 23.0065359477124
        struct[0].Gy_ini[37,37] = -5.75163398692810
        struct[0].Gy_ini[37,38] = -23.0065359477124
        struct[0].Gy_ini[37,39] = 5.75163398692810
        struct[0].Gy_ini[37,66] = 0.996212229189942
        struct[0].Gy_ini[37,67] = -0.249053057297486
        struct[0].Gy_ini[37,68] = -0.996212229189942
        struct[0].Gy_ini[37,69] = 0.249053057297486
        struct[0].Gy_ini[37,167] = 1
        struct[0].Gy_ini[38,32] = 5.75163398692810
        struct[0].Gy_ini[38,33] = 23.0065359477124
        struct[0].Gy_ini[38,34] = 5.75163398692810
        struct[0].Gy_ini[38,35] = 23.0065359477124
        struct[0].Gy_ini[38,36] = 5.75163398692810
        struct[0].Gy_ini[38,37] = 23.0065359477124
        struct[0].Gy_ini[38,38] = -1017.25490196078
        struct[0].Gy_ini[38,39] = -69.0196078431372
        struct[0].Gy_ini[39,32] = -23.0065359477124
        struct[0].Gy_ini[39,33] = 5.75163398692810
        struct[0].Gy_ini[39,34] = -23.0065359477124
        struct[0].Gy_ini[39,35] = 5.75163398692810
        struct[0].Gy_ini[39,36] = -23.0065359477124
        struct[0].Gy_ini[39,37] = 5.75163398692810
        struct[0].Gy_ini[39,38] = 69.0196078431372
        struct[0].Gy_ini[39,39] = -1017.25490196078
        struct[0].Gy_ini[40,0] = 0.249053057297486
        struct[0].Gy_ini[40,1] = 0.996212229189942
        struct[0].Gy_ini[40,2] = -0.249053057297486
        struct[0].Gy_ini[40,3] = -0.996212229189942
        struct[0].Gy_ini[40,40] = -2.23667465123725
        struct[0].Gy_ini[40,41] = -1.28353302446119
        struct[0].Gy_ini[40,42] = 0.643671749092996
        struct[0].Gy_ini[40,43] = 0.385473430243205
        struct[0].Gy_ini[40,44] = 0.643671749092997
        struct[0].Gy_ini[40,45] = 0.385473430243205
        struct[0].Gy_ini[40,46] = 1.10755301189314
        struct[0].Gy_ini[40,47] = 0.598820527961361
        struct[0].Gy_ini[40,48] = -0.316443717683753
        struct[0].Gy_ini[40,49] = -0.171091579417532
        struct[0].Gy_ini[40,50] = -0.316443717683753
        struct[0].Gy_ini[40,51] = -0.171091579417532
        struct[0].Gy_ini[41,0] = -0.996212229189942
        struct[0].Gy_ini[41,1] = 0.249053057297486
        struct[0].Gy_ini[41,2] = 0.996212229189942
        struct[0].Gy_ini[41,3] = -0.249053057297486
        struct[0].Gy_ini[41,40] = 1.28353302446119
        struct[0].Gy_ini[41,41] = -2.23667465123725
        struct[0].Gy_ini[41,42] = -0.385473430243205
        struct[0].Gy_ini[41,43] = 0.643671749092996
        struct[0].Gy_ini[41,44] = -0.385473430243205
        struct[0].Gy_ini[41,45] = 0.643671749092997
        struct[0].Gy_ini[41,46] = -0.598820527961361
        struct[0].Gy_ini[41,47] = 1.10755301189314
        struct[0].Gy_ini[41,48] = 0.171091579417532
        struct[0].Gy_ini[41,49] = -0.316443717683753
        struct[0].Gy_ini[41,50] = 0.171091579417532
        struct[0].Gy_ini[41,51] = -0.316443717683753
        struct[0].Gy_ini[42,2] = 0.249053057297486
        struct[0].Gy_ini[42,3] = 0.996212229189942
        struct[0].Gy_ini[42,4] = -0.249053057297486
        struct[0].Gy_ini[42,5] = -0.996212229189942
        struct[0].Gy_ini[42,40] = 0.643671749092996
        struct[0].Gy_ini[42,41] = 0.385473430243205
        struct[0].Gy_ini[42,42] = -2.23667465123725
        struct[0].Gy_ini[42,43] = -1.28353302446119
        struct[0].Gy_ini[42,44] = 0.643671749092997
        struct[0].Gy_ini[42,45] = 0.385473430243204
        struct[0].Gy_ini[42,46] = -0.316443717683753
        struct[0].Gy_ini[42,47] = -0.171091579417532
        struct[0].Gy_ini[42,48] = 1.10755301189314
        struct[0].Gy_ini[42,49] = 0.598820527961360
        struct[0].Gy_ini[42,50] = -0.316443717683753
        struct[0].Gy_ini[42,51] = -0.171091579417531
        struct[0].Gy_ini[43,2] = -0.996212229189942
        struct[0].Gy_ini[43,3] = 0.249053057297486
        struct[0].Gy_ini[43,4] = 0.996212229189942
        struct[0].Gy_ini[43,5] = -0.249053057297486
        struct[0].Gy_ini[43,40] = -0.385473430243205
        struct[0].Gy_ini[43,41] = 0.643671749092996
        struct[0].Gy_ini[43,42] = 1.28353302446119
        struct[0].Gy_ini[43,43] = -2.23667465123725
        struct[0].Gy_ini[43,44] = -0.385473430243204
        struct[0].Gy_ini[43,45] = 0.643671749092997
        struct[0].Gy_ini[43,46] = 0.171091579417532
        struct[0].Gy_ini[43,47] = -0.316443717683753
        struct[0].Gy_ini[43,48] = -0.598820527961360
        struct[0].Gy_ini[43,49] = 1.10755301189314
        struct[0].Gy_ini[43,50] = 0.171091579417531
        struct[0].Gy_ini[43,51] = -0.316443717683753
        struct[0].Gy_ini[44,0] = -0.249053057297486
        struct[0].Gy_ini[44,1] = -0.996212229189942
        struct[0].Gy_ini[44,4] = 0.249053057297486
        struct[0].Gy_ini[44,5] = 0.996212229189942
        struct[0].Gy_ini[44,40] = 0.643671749092997
        struct[0].Gy_ini[44,41] = 0.385473430243205
        struct[0].Gy_ini[44,42] = 0.643671749092997
        struct[0].Gy_ini[44,43] = 0.385473430243204
        struct[0].Gy_ini[44,44] = -2.23667465123725
        struct[0].Gy_ini[44,45] = -1.28353302446119
        struct[0].Gy_ini[44,46] = -0.316443717683753
        struct[0].Gy_ini[44,47] = -0.171091579417532
        struct[0].Gy_ini[44,48] = -0.316443717683753
        struct[0].Gy_ini[44,49] = -0.171091579417531
        struct[0].Gy_ini[44,50] = 1.10755301189314
        struct[0].Gy_ini[44,51] = 0.598820527961360
        struct[0].Gy_ini[45,0] = 0.996212229189942
        struct[0].Gy_ini[45,1] = -0.249053057297486
        struct[0].Gy_ini[45,4] = -0.996212229189942
        struct[0].Gy_ini[45,5] = 0.249053057297486
        struct[0].Gy_ini[45,40] = -0.385473430243205
        struct[0].Gy_ini[45,41] = 0.643671749092997
        struct[0].Gy_ini[45,42] = -0.385473430243204
        struct[0].Gy_ini[45,43] = 0.643671749092997
        struct[0].Gy_ini[45,44] = 1.28353302446119
        struct[0].Gy_ini[45,45] = -2.23667465123725
        struct[0].Gy_ini[45,46] = 0.171091579417532
        struct[0].Gy_ini[45,47] = -0.316443717683753
        struct[0].Gy_ini[45,48] = 0.171091579417531
        struct[0].Gy_ini[45,49] = -0.316443717683753
        struct[0].Gy_ini[45,50] = -0.598820527961360
        struct[0].Gy_ini[45,51] = 1.10755301189314
        struct[0].Gy_ini[46,8] = 0.249053057297486
        struct[0].Gy_ini[46,9] = 0.996212229189942
        struct[0].Gy_ini[46,10] = -0.249053057297486
        struct[0].Gy_ini[46,11] = -0.996212229189942
        struct[0].Gy_ini[46,40] = 1.10755301189314
        struct[0].Gy_ini[46,41] = 0.598820527961361
        struct[0].Gy_ini[46,42] = -0.316443717683753
        struct[0].Gy_ini[46,43] = -0.171091579417532
        struct[0].Gy_ini[46,44] = -0.316443717683753
        struct[0].Gy_ini[46,45] = -0.171091579417532
        struct[0].Gy_ini[46,46] = -2.23667465123725
        struct[0].Gy_ini[46,47] = -1.28353302446119
        struct[0].Gy_ini[46,48] = 0.643671749092996
        struct[0].Gy_ini[46,49] = 0.385473430243205
        struct[0].Gy_ini[46,50] = 0.643671749092997
        struct[0].Gy_ini[46,51] = 0.385473430243205
        struct[0].Gy_ini[46,52] = 1.10755301189314
        struct[0].Gy_ini[46,53] = 0.598820527961361
        struct[0].Gy_ini[46,54] = -0.316443717683753
        struct[0].Gy_ini[46,55] = -0.171091579417532
        struct[0].Gy_ini[46,56] = -0.316443717683753
        struct[0].Gy_ini[46,57] = -0.171091579417532
        struct[0].Gy_ini[47,8] = -0.996212229189942
        struct[0].Gy_ini[47,9] = 0.249053057297486
        struct[0].Gy_ini[47,10] = 0.996212229189942
        struct[0].Gy_ini[47,11] = -0.249053057297486
        struct[0].Gy_ini[47,40] = -0.598820527961361
        struct[0].Gy_ini[47,41] = 1.10755301189314
        struct[0].Gy_ini[47,42] = 0.171091579417532
        struct[0].Gy_ini[47,43] = -0.316443717683753
        struct[0].Gy_ini[47,44] = 0.171091579417532
        struct[0].Gy_ini[47,45] = -0.316443717683753
        struct[0].Gy_ini[47,46] = 1.28353302446119
        struct[0].Gy_ini[47,47] = -2.23667465123725
        struct[0].Gy_ini[47,48] = -0.385473430243205
        struct[0].Gy_ini[47,49] = 0.643671749092996
        struct[0].Gy_ini[47,50] = -0.385473430243205
        struct[0].Gy_ini[47,51] = 0.643671749092997
        struct[0].Gy_ini[47,52] = -0.598820527961361
        struct[0].Gy_ini[47,53] = 1.10755301189314
        struct[0].Gy_ini[47,54] = 0.171091579417532
        struct[0].Gy_ini[47,55] = -0.316443717683753
        struct[0].Gy_ini[47,56] = 0.171091579417532
        struct[0].Gy_ini[47,57] = -0.316443717683753
        struct[0].Gy_ini[48,10] = 0.249053057297486
        struct[0].Gy_ini[48,11] = 0.996212229189942
        struct[0].Gy_ini[48,12] = -0.249053057297486
        struct[0].Gy_ini[48,13] = -0.996212229189942
        struct[0].Gy_ini[48,40] = -0.316443717683753
        struct[0].Gy_ini[48,41] = -0.171091579417532
        struct[0].Gy_ini[48,42] = 1.10755301189314
        struct[0].Gy_ini[48,43] = 0.598820527961360
        struct[0].Gy_ini[48,44] = -0.316443717683753
        struct[0].Gy_ini[48,45] = -0.171091579417531
        struct[0].Gy_ini[48,46] = 0.643671749092996
        struct[0].Gy_ini[48,47] = 0.385473430243205
        struct[0].Gy_ini[48,48] = -2.23667465123725
        struct[0].Gy_ini[48,49] = -1.28353302446119
        struct[0].Gy_ini[48,50] = 0.643671749092997
        struct[0].Gy_ini[48,51] = 0.385473430243204
        struct[0].Gy_ini[48,52] = -0.316443717683753
        struct[0].Gy_ini[48,53] = -0.171091579417532
        struct[0].Gy_ini[48,54] = 1.10755301189314
        struct[0].Gy_ini[48,55] = 0.598820527961360
        struct[0].Gy_ini[48,56] = -0.316443717683753
        struct[0].Gy_ini[48,57] = -0.171091579417531
        struct[0].Gy_ini[49,10] = -0.996212229189942
        struct[0].Gy_ini[49,11] = 0.249053057297486
        struct[0].Gy_ini[49,12] = 0.996212229189942
        struct[0].Gy_ini[49,13] = -0.249053057297486
        struct[0].Gy_ini[49,40] = 0.171091579417532
        struct[0].Gy_ini[49,41] = -0.316443717683753
        struct[0].Gy_ini[49,42] = -0.598820527961360
        struct[0].Gy_ini[49,43] = 1.10755301189314
        struct[0].Gy_ini[49,44] = 0.171091579417531
        struct[0].Gy_ini[49,45] = -0.316443717683753
        struct[0].Gy_ini[49,46] = -0.385473430243205
        struct[0].Gy_ini[49,47] = 0.643671749092996
        struct[0].Gy_ini[49,48] = 1.28353302446119
        struct[0].Gy_ini[49,49] = -2.23667465123725
        struct[0].Gy_ini[49,50] = -0.385473430243204
        struct[0].Gy_ini[49,51] = 0.643671749092997
        struct[0].Gy_ini[49,52] = 0.171091579417532
        struct[0].Gy_ini[49,53] = -0.316443717683753
        struct[0].Gy_ini[49,54] = -0.598820527961360
        struct[0].Gy_ini[49,55] = 1.10755301189314
        struct[0].Gy_ini[49,56] = 0.171091579417531
        struct[0].Gy_ini[49,57] = -0.316443717683753
        struct[0].Gy_ini[50,8] = -0.249053057297486
        struct[0].Gy_ini[50,9] = -0.996212229189942
        struct[0].Gy_ini[50,12] = 0.249053057297486
        struct[0].Gy_ini[50,13] = 0.996212229189942
        struct[0].Gy_ini[50,40] = -0.316443717683753
        struct[0].Gy_ini[50,41] = -0.171091579417532
        struct[0].Gy_ini[50,42] = -0.316443717683753
        struct[0].Gy_ini[50,43] = -0.171091579417531
        struct[0].Gy_ini[50,44] = 1.10755301189314
        struct[0].Gy_ini[50,45] = 0.598820527961360
        struct[0].Gy_ini[50,46] = 0.643671749092997
        struct[0].Gy_ini[50,47] = 0.385473430243205
        struct[0].Gy_ini[50,48] = 0.643671749092997
        struct[0].Gy_ini[50,49] = 0.385473430243204
        struct[0].Gy_ini[50,50] = -2.23667465123725
        struct[0].Gy_ini[50,51] = -1.28353302446119
        struct[0].Gy_ini[50,52] = -0.316443717683753
        struct[0].Gy_ini[50,53] = -0.171091579417532
        struct[0].Gy_ini[50,54] = -0.316443717683753
        struct[0].Gy_ini[50,55] = -0.171091579417531
        struct[0].Gy_ini[50,56] = 1.10755301189314
        struct[0].Gy_ini[50,57] = 0.598820527961360
        struct[0].Gy_ini[51,8] = 0.996212229189942
        struct[0].Gy_ini[51,9] = -0.249053057297486
        struct[0].Gy_ini[51,12] = -0.996212229189942
        struct[0].Gy_ini[51,13] = 0.249053057297486
        struct[0].Gy_ini[51,40] = 0.171091579417532
        struct[0].Gy_ini[51,41] = -0.316443717683753
        struct[0].Gy_ini[51,42] = 0.171091579417531
        struct[0].Gy_ini[51,43] = -0.316443717683753
        struct[0].Gy_ini[51,44] = -0.598820527961360
        struct[0].Gy_ini[51,45] = 1.10755301189314
        struct[0].Gy_ini[51,46] = -0.385473430243205
        struct[0].Gy_ini[51,47] = 0.643671749092997
        struct[0].Gy_ini[51,48] = -0.385473430243204
        struct[0].Gy_ini[51,49] = 0.643671749092997
        struct[0].Gy_ini[51,50] = 1.28353302446119
        struct[0].Gy_ini[51,51] = -2.23667465123725
        struct[0].Gy_ini[51,52] = 0.171091579417532
        struct[0].Gy_ini[51,53] = -0.316443717683753
        struct[0].Gy_ini[51,54] = 0.171091579417531
        struct[0].Gy_ini[51,55] = -0.316443717683753
        struct[0].Gy_ini[51,56] = -0.598820527961360
        struct[0].Gy_ini[51,57] = 1.10755301189314
        struct[0].Gy_ini[52,16] = 0.249053057297486
        struct[0].Gy_ini[52,17] = 0.996212229189942
        struct[0].Gy_ini[52,18] = -0.249053057297486
        struct[0].Gy_ini[52,19] = -0.996212229189942
        struct[0].Gy_ini[52,46] = 1.10755301189314
        struct[0].Gy_ini[52,47] = 0.598820527961361
        struct[0].Gy_ini[52,48] = -0.316443717683753
        struct[0].Gy_ini[52,49] = -0.171091579417532
        struct[0].Gy_ini[52,50] = -0.316443717683753
        struct[0].Gy_ini[52,51] = -0.171091579417532
        struct[0].Gy_ini[52,52] = -1.12912163934412
        struct[0].Gy_ini[52,53] = -0.684903767132556
        struct[0].Gy_ini[52,54] = 0.327228031409243
        struct[0].Gy_ini[52,55] = 0.214305342572583
        struct[0].Gy_ini[52,56] = 0.327228031409244
        struct[0].Gy_ini[52,57] = 0.214305342572583
        struct[0].Gy_ini[53,16] = -0.996212229189942
        struct[0].Gy_ini[53,17] = 0.249053057297486
        struct[0].Gy_ini[53,18] = 0.996212229189942
        struct[0].Gy_ini[53,19] = -0.249053057297486
        struct[0].Gy_ini[53,46] = -0.598820527961361
        struct[0].Gy_ini[53,47] = 1.10755301189314
        struct[0].Gy_ini[53,48] = 0.171091579417532
        struct[0].Gy_ini[53,49] = -0.316443717683753
        struct[0].Gy_ini[53,50] = 0.171091579417532
        struct[0].Gy_ini[53,51] = -0.316443717683753
        struct[0].Gy_ini[53,52] = 0.684903767132556
        struct[0].Gy_ini[53,53] = -1.12912163934412
        struct[0].Gy_ini[53,54] = -0.214305342572583
        struct[0].Gy_ini[53,55] = 0.327228031409243
        struct[0].Gy_ini[53,56] = -0.214305342572583
        struct[0].Gy_ini[53,57] = 0.327228031409244
        struct[0].Gy_ini[54,18] = 0.249053057297486
        struct[0].Gy_ini[54,19] = 0.996212229189942
        struct[0].Gy_ini[54,20] = -0.249053057297486
        struct[0].Gy_ini[54,21] = -0.996212229189942
        struct[0].Gy_ini[54,46] = -0.316443717683753
        struct[0].Gy_ini[54,47] = -0.171091579417532
        struct[0].Gy_ini[54,48] = 1.10755301189314
        struct[0].Gy_ini[54,49] = 0.598820527961360
        struct[0].Gy_ini[54,50] = -0.316443717683753
        struct[0].Gy_ini[54,51] = -0.171091579417531
        struct[0].Gy_ini[54,52] = 0.327228031409243
        struct[0].Gy_ini[54,53] = 0.214305342572583
        struct[0].Gy_ini[54,54] = -1.12912163934412
        struct[0].Gy_ini[54,55] = -0.684903767132556
        struct[0].Gy_ini[54,56] = 0.327228031409244
        struct[0].Gy_ini[54,57] = 0.214305342572582
        struct[0].Gy_ini[55,18] = -0.996212229189942
        struct[0].Gy_ini[55,19] = 0.249053057297486
        struct[0].Gy_ini[55,20] = 0.996212229189942
        struct[0].Gy_ini[55,21] = -0.249053057297486
        struct[0].Gy_ini[55,46] = 0.171091579417532
        struct[0].Gy_ini[55,47] = -0.316443717683753
        struct[0].Gy_ini[55,48] = -0.598820527961360
        struct[0].Gy_ini[55,49] = 1.10755301189314
        struct[0].Gy_ini[55,50] = 0.171091579417531
        struct[0].Gy_ini[55,51] = -0.316443717683753
        struct[0].Gy_ini[55,52] = -0.214305342572583
        struct[0].Gy_ini[55,53] = 0.327228031409243
        struct[0].Gy_ini[55,54] = 0.684903767132556
        struct[0].Gy_ini[55,55] = -1.12912163934412
        struct[0].Gy_ini[55,56] = -0.214305342572582
        struct[0].Gy_ini[55,57] = 0.327228031409244
        struct[0].Gy_ini[56,16] = -0.249053057297486
        struct[0].Gy_ini[56,17] = -0.996212229189942
        struct[0].Gy_ini[56,20] = 0.249053057297486
        struct[0].Gy_ini[56,21] = 0.996212229189942
        struct[0].Gy_ini[56,46] = -0.316443717683753
        struct[0].Gy_ini[56,47] = -0.171091579417532
        struct[0].Gy_ini[56,48] = -0.316443717683753
        struct[0].Gy_ini[56,49] = -0.171091579417531
        struct[0].Gy_ini[56,50] = 1.10755301189314
        struct[0].Gy_ini[56,51] = 0.598820527961360
        struct[0].Gy_ini[56,52] = 0.327228031409243
        struct[0].Gy_ini[56,53] = 0.214305342572583
        struct[0].Gy_ini[56,54] = 0.327228031409244
        struct[0].Gy_ini[56,55] = 0.214305342572582
        struct[0].Gy_ini[56,56] = -1.12912163934412
        struct[0].Gy_ini[56,57] = -0.684903767132556
        struct[0].Gy_ini[57,16] = 0.996212229189942
        struct[0].Gy_ini[57,17] = -0.249053057297486
        struct[0].Gy_ini[57,20] = -0.996212229189942
        struct[0].Gy_ini[57,21] = 0.249053057297486
        struct[0].Gy_ini[57,46] = 0.171091579417532
        struct[0].Gy_ini[57,47] = -0.316443717683753
        struct[0].Gy_ini[57,48] = 0.171091579417531
        struct[0].Gy_ini[57,49] = -0.316443717683753
        struct[0].Gy_ini[57,50] = -0.598820527961360
        struct[0].Gy_ini[57,51] = 1.10755301189314
        struct[0].Gy_ini[57,52] = -0.214305342572583
        struct[0].Gy_ini[57,53] = 0.327228031409243
        struct[0].Gy_ini[57,54] = -0.214305342572582
        struct[0].Gy_ini[57,55] = 0.327228031409244
        struct[0].Gy_ini[57,56] = 0.684903767132556
        struct[0].Gy_ini[57,57] = -1.12912163934412
        struct[0].Gy_ini[58,24] = 0.249053057297486
        struct[0].Gy_ini[58,25] = 0.996212229189942
        struct[0].Gy_ini[58,26] = -0.249053057297486
        struct[0].Gy_ini[58,27] = -0.996212229189942
        struct[0].Gy_ini[58,58] = -1.12912163934412
        struct[0].Gy_ini[58,59] = -0.684903767132556
        struct[0].Gy_ini[58,60] = 0.327228031409243
        struct[0].Gy_ini[58,61] = 0.214305342572583
        struct[0].Gy_ini[58,62] = 0.327228031409244
        struct[0].Gy_ini[58,63] = 0.214305342572583
        struct[0].Gy_ini[58,64] = 1.10755301189314
        struct[0].Gy_ini[58,65] = 0.598820527961361
        struct[0].Gy_ini[58,66] = -0.316443717683753
        struct[0].Gy_ini[58,67] = -0.171091579417532
        struct[0].Gy_ini[58,68] = -0.316443717683753
        struct[0].Gy_ini[58,69] = -0.171091579417532
        struct[0].Gy_ini[59,24] = -0.996212229189942
        struct[0].Gy_ini[59,25] = 0.249053057297486
        struct[0].Gy_ini[59,26] = 0.996212229189942
        struct[0].Gy_ini[59,27] = -0.249053057297486
        struct[0].Gy_ini[59,58] = 0.684903767132556
        struct[0].Gy_ini[59,59] = -1.12912163934412
        struct[0].Gy_ini[59,60] = -0.214305342572583
        struct[0].Gy_ini[59,61] = 0.327228031409243
        struct[0].Gy_ini[59,62] = -0.214305342572583
        struct[0].Gy_ini[59,63] = 0.327228031409244
        struct[0].Gy_ini[59,64] = -0.598820527961361
        struct[0].Gy_ini[59,65] = 1.10755301189314
        struct[0].Gy_ini[59,66] = 0.171091579417532
        struct[0].Gy_ini[59,67] = -0.316443717683753
        struct[0].Gy_ini[59,68] = 0.171091579417532
        struct[0].Gy_ini[59,69] = -0.316443717683753
        struct[0].Gy_ini[60,26] = 0.249053057297486
        struct[0].Gy_ini[60,27] = 0.996212229189942
        struct[0].Gy_ini[60,28] = -0.249053057297486
        struct[0].Gy_ini[60,29] = -0.996212229189942
        struct[0].Gy_ini[60,58] = 0.327228031409243
        struct[0].Gy_ini[60,59] = 0.214305342572583
        struct[0].Gy_ini[60,60] = -1.12912163934412
        struct[0].Gy_ini[60,61] = -0.684903767132556
        struct[0].Gy_ini[60,62] = 0.327228031409244
        struct[0].Gy_ini[60,63] = 0.214305342572582
        struct[0].Gy_ini[60,64] = -0.316443717683753
        struct[0].Gy_ini[60,65] = -0.171091579417532
        struct[0].Gy_ini[60,66] = 1.10755301189314
        struct[0].Gy_ini[60,67] = 0.598820527961360
        struct[0].Gy_ini[60,68] = -0.316443717683753
        struct[0].Gy_ini[60,69] = -0.171091579417531
        struct[0].Gy_ini[61,26] = -0.996212229189942
        struct[0].Gy_ini[61,27] = 0.249053057297486
        struct[0].Gy_ini[61,28] = 0.996212229189942
        struct[0].Gy_ini[61,29] = -0.249053057297486
        struct[0].Gy_ini[61,58] = -0.214305342572583
        struct[0].Gy_ini[61,59] = 0.327228031409243
        struct[0].Gy_ini[61,60] = 0.684903767132556
        struct[0].Gy_ini[61,61] = -1.12912163934412
        struct[0].Gy_ini[61,62] = -0.214305342572582
        struct[0].Gy_ini[61,63] = 0.327228031409244
        struct[0].Gy_ini[61,64] = 0.171091579417532
        struct[0].Gy_ini[61,65] = -0.316443717683753
        struct[0].Gy_ini[61,66] = -0.598820527961360
        struct[0].Gy_ini[61,67] = 1.10755301189314
        struct[0].Gy_ini[61,68] = 0.171091579417531
        struct[0].Gy_ini[61,69] = -0.316443717683753
        struct[0].Gy_ini[62,24] = -0.249053057297486
        struct[0].Gy_ini[62,25] = -0.996212229189942
        struct[0].Gy_ini[62,28] = 0.249053057297486
        struct[0].Gy_ini[62,29] = 0.996212229189942
        struct[0].Gy_ini[62,58] = 0.327228031409243
        struct[0].Gy_ini[62,59] = 0.214305342572583
        struct[0].Gy_ini[62,60] = 0.327228031409244
        struct[0].Gy_ini[62,61] = 0.214305342572582
        struct[0].Gy_ini[62,62] = -1.12912163934412
        struct[0].Gy_ini[62,63] = -0.684903767132556
        struct[0].Gy_ini[62,64] = -0.316443717683753
        struct[0].Gy_ini[62,65] = -0.171091579417532
        struct[0].Gy_ini[62,66] = -0.316443717683753
        struct[0].Gy_ini[62,67] = -0.171091579417531
        struct[0].Gy_ini[62,68] = 1.10755301189314
        struct[0].Gy_ini[62,69] = 0.598820527961360
        struct[0].Gy_ini[63,24] = 0.996212229189942
        struct[0].Gy_ini[63,25] = -0.249053057297486
        struct[0].Gy_ini[63,28] = -0.996212229189942
        struct[0].Gy_ini[63,29] = 0.249053057297486
        struct[0].Gy_ini[63,58] = -0.214305342572583
        struct[0].Gy_ini[63,59] = 0.327228031409243
        struct[0].Gy_ini[63,60] = -0.214305342572582
        struct[0].Gy_ini[63,61] = 0.327228031409244
        struct[0].Gy_ini[63,62] = 0.684903767132556
        struct[0].Gy_ini[63,63] = -1.12912163934412
        struct[0].Gy_ini[63,64] = 0.171091579417532
        struct[0].Gy_ini[63,65] = -0.316443717683753
        struct[0].Gy_ini[63,66] = 0.171091579417531
        struct[0].Gy_ini[63,67] = -0.316443717683753
        struct[0].Gy_ini[63,68] = -0.598820527961360
        struct[0].Gy_ini[63,69] = 1.10755301189314
        struct[0].Gy_ini[64,32] = 0.249053057297486
        struct[0].Gy_ini[64,33] = 0.996212229189942
        struct[0].Gy_ini[64,34] = -0.249053057297486
        struct[0].Gy_ini[64,35] = -0.996212229189942
        struct[0].Gy_ini[64,58] = 1.10755301189314
        struct[0].Gy_ini[64,59] = 0.598820527961361
        struct[0].Gy_ini[64,60] = -0.316443717683753
        struct[0].Gy_ini[64,61] = -0.171091579417532
        struct[0].Gy_ini[64,62] = -0.316443717683753
        struct[0].Gy_ini[64,63] = -0.171091579417532
        struct[0].Gy_ini[64,64] = -2.23667465123725
        struct[0].Gy_ini[64,65] = -1.28353302446119
        struct[0].Gy_ini[64,66] = 0.643671749092996
        struct[0].Gy_ini[64,67] = 0.385473430243205
        struct[0].Gy_ini[64,68] = 0.643671749092997
        struct[0].Gy_ini[64,69] = 0.385473430243205
        struct[0].Gy_ini[65,32] = -0.996212229189942
        struct[0].Gy_ini[65,33] = 0.249053057297486
        struct[0].Gy_ini[65,34] = 0.996212229189942
        struct[0].Gy_ini[65,35] = -0.249053057297486
        struct[0].Gy_ini[65,58] = -0.598820527961361
        struct[0].Gy_ini[65,59] = 1.10755301189314
        struct[0].Gy_ini[65,60] = 0.171091579417532
        struct[0].Gy_ini[65,61] = -0.316443717683753
        struct[0].Gy_ini[65,62] = 0.171091579417532
        struct[0].Gy_ini[65,63] = -0.316443717683753
        struct[0].Gy_ini[65,64] = 1.28353302446119
        struct[0].Gy_ini[65,65] = -2.23667465123725
        struct[0].Gy_ini[65,66] = -0.385473430243205
        struct[0].Gy_ini[65,67] = 0.643671749092996
        struct[0].Gy_ini[65,68] = -0.385473430243205
        struct[0].Gy_ini[65,69] = 0.643671749092997
        struct[0].Gy_ini[66,34] = 0.249053057297486
        struct[0].Gy_ini[66,35] = 0.996212229189942
        struct[0].Gy_ini[66,36] = -0.249053057297486
        struct[0].Gy_ini[66,37] = -0.996212229189942
        struct[0].Gy_ini[66,58] = -0.316443717683753
        struct[0].Gy_ini[66,59] = -0.171091579417532
        struct[0].Gy_ini[66,60] = 1.10755301189314
        struct[0].Gy_ini[66,61] = 0.598820527961360
        struct[0].Gy_ini[66,62] = -0.316443717683753
        struct[0].Gy_ini[66,63] = -0.171091579417531
        struct[0].Gy_ini[66,64] = 0.643671749092996
        struct[0].Gy_ini[66,65] = 0.385473430243205
        struct[0].Gy_ini[66,66] = -2.23667465123725
        struct[0].Gy_ini[66,67] = -1.28353302446119
        struct[0].Gy_ini[66,68] = 0.643671749092997
        struct[0].Gy_ini[66,69] = 0.385473430243204
        struct[0].Gy_ini[67,34] = -0.996212229189942
        struct[0].Gy_ini[67,35] = 0.249053057297486
        struct[0].Gy_ini[67,36] = 0.996212229189942
        struct[0].Gy_ini[67,37] = -0.249053057297486
        struct[0].Gy_ini[67,58] = 0.171091579417532
        struct[0].Gy_ini[67,59] = -0.316443717683753
        struct[0].Gy_ini[67,60] = -0.598820527961360
        struct[0].Gy_ini[67,61] = 1.10755301189314
        struct[0].Gy_ini[67,62] = 0.171091579417531
        struct[0].Gy_ini[67,63] = -0.316443717683753
        struct[0].Gy_ini[67,64] = -0.385473430243205
        struct[0].Gy_ini[67,65] = 0.643671749092996
        struct[0].Gy_ini[67,66] = 1.28353302446119
        struct[0].Gy_ini[67,67] = -2.23667465123725
        struct[0].Gy_ini[67,68] = -0.385473430243204
        struct[0].Gy_ini[67,69] = 0.643671749092997
        struct[0].Gy_ini[68,32] = -0.249053057297486
        struct[0].Gy_ini[68,33] = -0.996212229189942
        struct[0].Gy_ini[68,36] = 0.249053057297486
        struct[0].Gy_ini[68,37] = 0.996212229189942
        struct[0].Gy_ini[68,58] = -0.316443717683753
        struct[0].Gy_ini[68,59] = -0.171091579417532
        struct[0].Gy_ini[68,60] = -0.316443717683753
        struct[0].Gy_ini[68,61] = -0.171091579417531
        struct[0].Gy_ini[68,62] = 1.10755301189314
        struct[0].Gy_ini[68,63] = 0.598820527961360
        struct[0].Gy_ini[68,64] = 0.643671749092997
        struct[0].Gy_ini[68,65] = 0.385473430243205
        struct[0].Gy_ini[68,66] = 0.643671749092997
        struct[0].Gy_ini[68,67] = 0.385473430243204
        struct[0].Gy_ini[68,68] = -2.23667465123725
        struct[0].Gy_ini[68,69] = -1.28353302446119
        struct[0].Gy_ini[69,32] = 0.996212229189942
        struct[0].Gy_ini[69,33] = -0.249053057297486
        struct[0].Gy_ini[69,36] = -0.996212229189942
        struct[0].Gy_ini[69,37] = 0.249053057297486
        struct[0].Gy_ini[69,58] = 0.171091579417532
        struct[0].Gy_ini[69,59] = -0.316443717683753
        struct[0].Gy_ini[69,60] = 0.171091579417531
        struct[0].Gy_ini[69,61] = -0.316443717683753
        struct[0].Gy_ini[69,62] = -0.598820527961360
        struct[0].Gy_ini[69,63] = 1.10755301189314
        struct[0].Gy_ini[69,64] = -0.385473430243205
        struct[0].Gy_ini[69,65] = 0.643671749092997
        struct[0].Gy_ini[69,66] = -0.385473430243204
        struct[0].Gy_ini[69,67] = 0.643671749092997
        struct[0].Gy_ini[69,68] = 1.28353302446119
        struct[0].Gy_ini[69,69] = -2.23667465123725
        struct[0].Gy_ini[70,0] = -0.249053057297486
        struct[0].Gy_ini[70,1] = -0.996212229189942
        struct[0].Gy_ini[70,2] = 0.249053057297486
        struct[0].Gy_ini[70,3] = 0.996212229189942
        struct[0].Gy_ini[70,40] = 0.0215686274509804
        struct[0].Gy_ini[70,41] = 0.0862745098039216
        struct[0].Gy_ini[70,42] = -0.0107843137254902
        struct[0].Gy_ini[70,43] = -0.0431372549019608
        struct[0].Gy_ini[70,44] = -0.0107843137254902
        struct[0].Gy_ini[70,45] = -0.0431372549019608
        struct[0].Gy_ini[70,70] = -1
        struct[0].Gy_ini[71,0] = 0.996212229189942
        struct[0].Gy_ini[71,1] = -0.249053057297486
        struct[0].Gy_ini[71,2] = -0.996212229189942
        struct[0].Gy_ini[71,3] = 0.249053057297486
        struct[0].Gy_ini[71,40] = -0.0862745098039216
        struct[0].Gy_ini[71,41] = 0.0215686274509804
        struct[0].Gy_ini[71,42] = 0.0431372549019608
        struct[0].Gy_ini[71,43] = -0.0107843137254902
        struct[0].Gy_ini[71,44] = 0.0431372549019608
        struct[0].Gy_ini[71,45] = -0.0107843137254902
        struct[0].Gy_ini[71,71] = -1
        struct[0].Gy_ini[72,2] = -0.249053057297486
        struct[0].Gy_ini[72,3] = -0.996212229189942
        struct[0].Gy_ini[72,4] = 0.249053057297486
        struct[0].Gy_ini[72,5] = 0.996212229189942
        struct[0].Gy_ini[72,40] = -0.0107843137254902
        struct[0].Gy_ini[72,41] = -0.0431372549019608
        struct[0].Gy_ini[72,42] = 0.0215686274509804
        struct[0].Gy_ini[72,43] = 0.0862745098039216
        struct[0].Gy_ini[72,44] = -0.0107843137254902
        struct[0].Gy_ini[72,45] = -0.0431372549019608
        struct[0].Gy_ini[72,72] = -1
        struct[0].Gy_ini[73,2] = 0.996212229189942
        struct[0].Gy_ini[73,3] = -0.249053057297486
        struct[0].Gy_ini[73,4] = -0.996212229189942
        struct[0].Gy_ini[73,5] = 0.249053057297486
        struct[0].Gy_ini[73,40] = 0.0431372549019608
        struct[0].Gy_ini[73,41] = -0.0107843137254902
        struct[0].Gy_ini[73,42] = -0.0862745098039216
        struct[0].Gy_ini[73,43] = 0.0215686274509804
        struct[0].Gy_ini[73,44] = 0.0431372549019608
        struct[0].Gy_ini[73,45] = -0.0107843137254902
        struct[0].Gy_ini[73,73] = -1
        struct[0].Gy_ini[74,0] = 0.249053057297486
        struct[0].Gy_ini[74,1] = 0.996212229189942
        struct[0].Gy_ini[74,4] = -0.249053057297486
        struct[0].Gy_ini[74,5] = -0.996212229189942
        struct[0].Gy_ini[74,40] = -0.0107843137254902
        struct[0].Gy_ini[74,41] = -0.0431372549019608
        struct[0].Gy_ini[74,42] = -0.0107843137254902
        struct[0].Gy_ini[74,43] = -0.0431372549019608
        struct[0].Gy_ini[74,44] = 0.0215686274509804
        struct[0].Gy_ini[74,45] = 0.0862745098039216
        struct[0].Gy_ini[74,74] = -1
        struct[0].Gy_ini[75,0] = -0.996212229189942
        struct[0].Gy_ini[75,1] = 0.249053057297486
        struct[0].Gy_ini[75,4] = 0.996212229189942
        struct[0].Gy_ini[75,5] = -0.249053057297486
        struct[0].Gy_ini[75,40] = 0.0431372549019608
        struct[0].Gy_ini[75,41] = -0.0107843137254902
        struct[0].Gy_ini[75,42] = 0.0431372549019608
        struct[0].Gy_ini[75,43] = -0.0107843137254902
        struct[0].Gy_ini[75,44] = -0.0862745098039216
        struct[0].Gy_ini[75,45] = 0.0215686274509804
        struct[0].Gy_ini[75,75] = -1
        struct[0].Gy_ini[76,8] = -0.249053057297486
        struct[0].Gy_ini[76,9] = -0.996212229189942
        struct[0].Gy_ini[76,10] = 0.249053057297486
        struct[0].Gy_ini[76,11] = 0.996212229189942
        struct[0].Gy_ini[76,46] = 0.0215686274509804
        struct[0].Gy_ini[76,47] = 0.0862745098039216
        struct[0].Gy_ini[76,48] = -0.0107843137254902
        struct[0].Gy_ini[76,49] = -0.0431372549019608
        struct[0].Gy_ini[76,50] = -0.0107843137254902
        struct[0].Gy_ini[76,51] = -0.0431372549019608
        struct[0].Gy_ini[76,76] = -1
        struct[0].Gy_ini[77,8] = 0.996212229189942
        struct[0].Gy_ini[77,9] = -0.249053057297486
        struct[0].Gy_ini[77,10] = -0.996212229189942
        struct[0].Gy_ini[77,11] = 0.249053057297486
        struct[0].Gy_ini[77,46] = -0.0862745098039216
        struct[0].Gy_ini[77,47] = 0.0215686274509804
        struct[0].Gy_ini[77,48] = 0.0431372549019608
        struct[0].Gy_ini[77,49] = -0.0107843137254902
        struct[0].Gy_ini[77,50] = 0.0431372549019608
        struct[0].Gy_ini[77,51] = -0.0107843137254902
        struct[0].Gy_ini[77,77] = -1
        struct[0].Gy_ini[78,10] = -0.249053057297486
        struct[0].Gy_ini[78,11] = -0.996212229189942
        struct[0].Gy_ini[78,12] = 0.249053057297486
        struct[0].Gy_ini[78,13] = 0.996212229189942
        struct[0].Gy_ini[78,46] = -0.0107843137254902
        struct[0].Gy_ini[78,47] = -0.0431372549019608
        struct[0].Gy_ini[78,48] = 0.0215686274509804
        struct[0].Gy_ini[78,49] = 0.0862745098039216
        struct[0].Gy_ini[78,50] = -0.0107843137254902
        struct[0].Gy_ini[78,51] = -0.0431372549019608
        struct[0].Gy_ini[78,78] = -1
        struct[0].Gy_ini[79,10] = 0.996212229189942
        struct[0].Gy_ini[79,11] = -0.249053057297486
        struct[0].Gy_ini[79,12] = -0.996212229189942
        struct[0].Gy_ini[79,13] = 0.249053057297486
        struct[0].Gy_ini[79,46] = 0.0431372549019608
        struct[0].Gy_ini[79,47] = -0.0107843137254902
        struct[0].Gy_ini[79,48] = -0.0862745098039216
        struct[0].Gy_ini[79,49] = 0.0215686274509804
        struct[0].Gy_ini[79,50] = 0.0431372549019608
        struct[0].Gy_ini[79,51] = -0.0107843137254902
        struct[0].Gy_ini[79,79] = -1
        struct[0].Gy_ini[80,8] = 0.249053057297486
        struct[0].Gy_ini[80,9] = 0.996212229189942
        struct[0].Gy_ini[80,12] = -0.249053057297486
        struct[0].Gy_ini[80,13] = -0.996212229189942
        struct[0].Gy_ini[80,46] = -0.0107843137254902
        struct[0].Gy_ini[80,47] = -0.0431372549019608
        struct[0].Gy_ini[80,48] = -0.0107843137254902
        struct[0].Gy_ini[80,49] = -0.0431372549019608
        struct[0].Gy_ini[80,50] = 0.0215686274509804
        struct[0].Gy_ini[80,51] = 0.0862745098039216
        struct[0].Gy_ini[80,80] = -1
        struct[0].Gy_ini[81,8] = -0.996212229189942
        struct[0].Gy_ini[81,9] = 0.249053057297486
        struct[0].Gy_ini[81,12] = 0.996212229189942
        struct[0].Gy_ini[81,13] = -0.249053057297486
        struct[0].Gy_ini[81,46] = 0.0431372549019608
        struct[0].Gy_ini[81,47] = -0.0107843137254902
        struct[0].Gy_ini[81,48] = 0.0431372549019608
        struct[0].Gy_ini[81,49] = -0.0107843137254902
        struct[0].Gy_ini[81,50] = -0.0862745098039216
        struct[0].Gy_ini[81,51] = 0.0215686274509804
        struct[0].Gy_ini[81,81] = -1
        struct[0].Gy_ini[82,16] = -0.249053057297486
        struct[0].Gy_ini[82,17] = -0.996212229189942
        struct[0].Gy_ini[82,18] = 0.249053057297486
        struct[0].Gy_ini[82,19] = 0.996212229189942
        struct[0].Gy_ini[82,52] = 0.0215686274509804
        struct[0].Gy_ini[82,53] = 0.0862745098039216
        struct[0].Gy_ini[82,54] = -0.0107843137254902
        struct[0].Gy_ini[82,55] = -0.0431372549019608
        struct[0].Gy_ini[82,56] = -0.0107843137254902
        struct[0].Gy_ini[82,57] = -0.0431372549019608
        struct[0].Gy_ini[82,82] = -1
        struct[0].Gy_ini[83,16] = 0.996212229189942
        struct[0].Gy_ini[83,17] = -0.249053057297486
        struct[0].Gy_ini[83,18] = -0.996212229189942
        struct[0].Gy_ini[83,19] = 0.249053057297486
        struct[0].Gy_ini[83,52] = -0.0862745098039216
        struct[0].Gy_ini[83,53] = 0.0215686274509804
        struct[0].Gy_ini[83,54] = 0.0431372549019608
        struct[0].Gy_ini[83,55] = -0.0107843137254902
        struct[0].Gy_ini[83,56] = 0.0431372549019608
        struct[0].Gy_ini[83,57] = -0.0107843137254902
        struct[0].Gy_ini[83,83] = -1
        struct[0].Gy_ini[84,18] = -0.249053057297486
        struct[0].Gy_ini[84,19] = -0.996212229189942
        struct[0].Gy_ini[84,20] = 0.249053057297486
        struct[0].Gy_ini[84,21] = 0.996212229189942
        struct[0].Gy_ini[84,52] = -0.0107843137254902
        struct[0].Gy_ini[84,53] = -0.0431372549019608
        struct[0].Gy_ini[84,54] = 0.0215686274509804
        struct[0].Gy_ini[84,55] = 0.0862745098039216
        struct[0].Gy_ini[84,56] = -0.0107843137254902
        struct[0].Gy_ini[84,57] = -0.0431372549019608
        struct[0].Gy_ini[84,84] = -1
        struct[0].Gy_ini[85,18] = 0.996212229189942
        struct[0].Gy_ini[85,19] = -0.249053057297486
        struct[0].Gy_ini[85,20] = -0.996212229189942
        struct[0].Gy_ini[85,21] = 0.249053057297486
        struct[0].Gy_ini[85,52] = 0.0431372549019608
        struct[0].Gy_ini[85,53] = -0.0107843137254902
        struct[0].Gy_ini[85,54] = -0.0862745098039216
        struct[0].Gy_ini[85,55] = 0.0215686274509804
        struct[0].Gy_ini[85,56] = 0.0431372549019608
        struct[0].Gy_ini[85,57] = -0.0107843137254902
        struct[0].Gy_ini[85,85] = -1
        struct[0].Gy_ini[86,16] = 0.249053057297486
        struct[0].Gy_ini[86,17] = 0.996212229189942
        struct[0].Gy_ini[86,20] = -0.249053057297486
        struct[0].Gy_ini[86,21] = -0.996212229189942
        struct[0].Gy_ini[86,52] = -0.0107843137254902
        struct[0].Gy_ini[86,53] = -0.0431372549019608
        struct[0].Gy_ini[86,54] = -0.0107843137254902
        struct[0].Gy_ini[86,55] = -0.0431372549019608
        struct[0].Gy_ini[86,56] = 0.0215686274509804
        struct[0].Gy_ini[86,57] = 0.0862745098039216
        struct[0].Gy_ini[86,86] = -1
        struct[0].Gy_ini[87,16] = -0.996212229189942
        struct[0].Gy_ini[87,17] = 0.249053057297486
        struct[0].Gy_ini[87,20] = 0.996212229189942
        struct[0].Gy_ini[87,21] = -0.249053057297486
        struct[0].Gy_ini[87,52] = 0.0431372549019608
        struct[0].Gy_ini[87,53] = -0.0107843137254902
        struct[0].Gy_ini[87,54] = 0.0431372549019608
        struct[0].Gy_ini[87,55] = -0.0107843137254902
        struct[0].Gy_ini[87,56] = -0.0862745098039216
        struct[0].Gy_ini[87,57] = 0.0215686274509804
        struct[0].Gy_ini[87,87] = -1
        struct[0].Gy_ini[88,24] = -0.249053057297486
        struct[0].Gy_ini[88,25] = -0.996212229189942
        struct[0].Gy_ini[88,26] = 0.249053057297486
        struct[0].Gy_ini[88,27] = 0.996212229189942
        struct[0].Gy_ini[88,58] = 0.0215686274509804
        struct[0].Gy_ini[88,59] = 0.0862745098039216
        struct[0].Gy_ini[88,60] = -0.0107843137254902
        struct[0].Gy_ini[88,61] = -0.0431372549019608
        struct[0].Gy_ini[88,62] = -0.0107843137254902
        struct[0].Gy_ini[88,63] = -0.0431372549019608
        struct[0].Gy_ini[88,88] = -1
        struct[0].Gy_ini[89,24] = 0.996212229189942
        struct[0].Gy_ini[89,25] = -0.249053057297486
        struct[0].Gy_ini[89,26] = -0.996212229189942
        struct[0].Gy_ini[89,27] = 0.249053057297486
        struct[0].Gy_ini[89,58] = -0.0862745098039216
        struct[0].Gy_ini[89,59] = 0.0215686274509804
        struct[0].Gy_ini[89,60] = 0.0431372549019608
        struct[0].Gy_ini[89,61] = -0.0107843137254902
        struct[0].Gy_ini[89,62] = 0.0431372549019608
        struct[0].Gy_ini[89,63] = -0.0107843137254902
        struct[0].Gy_ini[89,89] = -1
        struct[0].Gy_ini[90,26] = -0.249053057297486
        struct[0].Gy_ini[90,27] = -0.996212229189942
        struct[0].Gy_ini[90,28] = 0.249053057297486
        struct[0].Gy_ini[90,29] = 0.996212229189942
        struct[0].Gy_ini[90,58] = -0.0107843137254902
        struct[0].Gy_ini[90,59] = -0.0431372549019608
        struct[0].Gy_ini[90,60] = 0.0215686274509804
        struct[0].Gy_ini[90,61] = 0.0862745098039216
        struct[0].Gy_ini[90,62] = -0.0107843137254902
        struct[0].Gy_ini[90,63] = -0.0431372549019608
        struct[0].Gy_ini[90,90] = -1
        struct[0].Gy_ini[91,26] = 0.996212229189942
        struct[0].Gy_ini[91,27] = -0.249053057297486
        struct[0].Gy_ini[91,28] = -0.996212229189942
        struct[0].Gy_ini[91,29] = 0.249053057297486
        struct[0].Gy_ini[91,58] = 0.0431372549019608
        struct[0].Gy_ini[91,59] = -0.0107843137254902
        struct[0].Gy_ini[91,60] = -0.0862745098039216
        struct[0].Gy_ini[91,61] = 0.0215686274509804
        struct[0].Gy_ini[91,62] = 0.0431372549019608
        struct[0].Gy_ini[91,63] = -0.0107843137254902
        struct[0].Gy_ini[91,91] = -1
        struct[0].Gy_ini[92,24] = 0.249053057297486
        struct[0].Gy_ini[92,25] = 0.996212229189942
        struct[0].Gy_ini[92,28] = -0.249053057297486
        struct[0].Gy_ini[92,29] = -0.996212229189942
        struct[0].Gy_ini[92,58] = -0.0107843137254902
        struct[0].Gy_ini[92,59] = -0.0431372549019608
        struct[0].Gy_ini[92,60] = -0.0107843137254902
        struct[0].Gy_ini[92,61] = -0.0431372549019608
        struct[0].Gy_ini[92,62] = 0.0215686274509804
        struct[0].Gy_ini[92,63] = 0.0862745098039216
        struct[0].Gy_ini[92,92] = -1
        struct[0].Gy_ini[93,24] = -0.996212229189942
        struct[0].Gy_ini[93,25] = 0.249053057297486
        struct[0].Gy_ini[93,28] = 0.996212229189942
        struct[0].Gy_ini[93,29] = -0.249053057297486
        struct[0].Gy_ini[93,58] = 0.0431372549019608
        struct[0].Gy_ini[93,59] = -0.0107843137254902
        struct[0].Gy_ini[93,60] = 0.0431372549019608
        struct[0].Gy_ini[93,61] = -0.0107843137254902
        struct[0].Gy_ini[93,62] = -0.0862745098039216
        struct[0].Gy_ini[93,63] = 0.0215686274509804
        struct[0].Gy_ini[93,93] = -1
        struct[0].Gy_ini[94,32] = -0.249053057297486
        struct[0].Gy_ini[94,33] = -0.996212229189942
        struct[0].Gy_ini[94,34] = 0.249053057297486
        struct[0].Gy_ini[94,35] = 0.996212229189942
        struct[0].Gy_ini[94,64] = 0.0215686274509804
        struct[0].Gy_ini[94,65] = 0.0862745098039216
        struct[0].Gy_ini[94,66] = -0.0107843137254902
        struct[0].Gy_ini[94,67] = -0.0431372549019608
        struct[0].Gy_ini[94,68] = -0.0107843137254902
        struct[0].Gy_ini[94,69] = -0.0431372549019608
        struct[0].Gy_ini[94,94] = -1
        struct[0].Gy_ini[95,32] = 0.996212229189942
        struct[0].Gy_ini[95,33] = -0.249053057297486
        struct[0].Gy_ini[95,34] = -0.996212229189942
        struct[0].Gy_ini[95,35] = 0.249053057297486
        struct[0].Gy_ini[95,64] = -0.0862745098039216
        struct[0].Gy_ini[95,65] = 0.0215686274509804
        struct[0].Gy_ini[95,66] = 0.0431372549019608
        struct[0].Gy_ini[95,67] = -0.0107843137254902
        struct[0].Gy_ini[95,68] = 0.0431372549019608
        struct[0].Gy_ini[95,69] = -0.0107843137254902
        struct[0].Gy_ini[95,95] = -1
        struct[0].Gy_ini[96,34] = -0.249053057297486
        struct[0].Gy_ini[96,35] = -0.996212229189942
        struct[0].Gy_ini[96,36] = 0.249053057297486
        struct[0].Gy_ini[96,37] = 0.996212229189942
        struct[0].Gy_ini[96,64] = -0.0107843137254902
        struct[0].Gy_ini[96,65] = -0.0431372549019608
        struct[0].Gy_ini[96,66] = 0.0215686274509804
        struct[0].Gy_ini[96,67] = 0.0862745098039216
        struct[0].Gy_ini[96,68] = -0.0107843137254902
        struct[0].Gy_ini[96,69] = -0.0431372549019608
        struct[0].Gy_ini[96,96] = -1
        struct[0].Gy_ini[97,34] = 0.996212229189942
        struct[0].Gy_ini[97,35] = -0.249053057297486
        struct[0].Gy_ini[97,36] = -0.996212229189942
        struct[0].Gy_ini[97,37] = 0.249053057297486
        struct[0].Gy_ini[97,64] = 0.0431372549019608
        struct[0].Gy_ini[97,65] = -0.0107843137254902
        struct[0].Gy_ini[97,66] = -0.0862745098039216
        struct[0].Gy_ini[97,67] = 0.0215686274509804
        struct[0].Gy_ini[97,68] = 0.0431372549019608
        struct[0].Gy_ini[97,69] = -0.0107843137254902
        struct[0].Gy_ini[97,97] = -1
        struct[0].Gy_ini[98,32] = 0.249053057297486
        struct[0].Gy_ini[98,33] = 0.996212229189942
        struct[0].Gy_ini[98,36] = -0.249053057297486
        struct[0].Gy_ini[98,37] = -0.996212229189942
        struct[0].Gy_ini[98,64] = -0.0107843137254902
        struct[0].Gy_ini[98,65] = -0.0431372549019608
        struct[0].Gy_ini[98,66] = -0.0107843137254902
        struct[0].Gy_ini[98,67] = -0.0431372549019608
        struct[0].Gy_ini[98,68] = 0.0215686274509804
        struct[0].Gy_ini[98,69] = 0.0862745098039216
        struct[0].Gy_ini[98,98] = -1
        struct[0].Gy_ini[99,32] = -0.996212229189942
        struct[0].Gy_ini[99,33] = 0.249053057297486
        struct[0].Gy_ini[99,36] = 0.996212229189942
        struct[0].Gy_ini[99,37] = -0.249053057297486
        struct[0].Gy_ini[99,64] = 0.0431372549019608
        struct[0].Gy_ini[99,65] = -0.0107843137254902
        struct[0].Gy_ini[99,66] = 0.0431372549019608
        struct[0].Gy_ini[99,67] = -0.0107843137254902
        struct[0].Gy_ini[99,68] = -0.0862745098039216
        struct[0].Gy_ini[99,69] = 0.0215686274509804
        struct[0].Gy_ini[99,99] = -1
        struct[0].Gy_ini[100,40] = -1.10755301189314
        struct[0].Gy_ini[100,41] = -0.598820527961361
        struct[0].Gy_ini[100,42] = 0.316443717683753
        struct[0].Gy_ini[100,43] = 0.171091579417532
        struct[0].Gy_ini[100,44] = 0.316443717683753
        struct[0].Gy_ini[100,45] = 0.171091579417532
        struct[0].Gy_ini[100,100] = -1
        struct[0].Gy_ini[101,40] = 0.598820527961361
        struct[0].Gy_ini[101,41] = -1.10755301189314
        struct[0].Gy_ini[101,42] = -0.171091579417532
        struct[0].Gy_ini[101,43] = 0.316443717683753
        struct[0].Gy_ini[101,44] = -0.171091579417532
        struct[0].Gy_ini[101,45] = 0.316443717683753
        struct[0].Gy_ini[101,101] = -1
        struct[0].Gy_ini[102,40] = 0.316443717683753
        struct[0].Gy_ini[102,41] = 0.171091579417532
        struct[0].Gy_ini[102,42] = -1.10755301189314
        struct[0].Gy_ini[102,43] = -0.598820527961360
        struct[0].Gy_ini[102,44] = 0.316443717683753
        struct[0].Gy_ini[102,45] = 0.171091579417531
        struct[0].Gy_ini[102,102] = -1
        struct[0].Gy_ini[103,40] = -0.171091579417532
        struct[0].Gy_ini[103,41] = 0.316443717683753
        struct[0].Gy_ini[103,42] = 0.598820527961360
        struct[0].Gy_ini[103,43] = -1.10755301189314
        struct[0].Gy_ini[103,44] = -0.171091579417531
        struct[0].Gy_ini[103,45] = 0.316443717683753
        struct[0].Gy_ini[103,103] = -1
        struct[0].Gy_ini[104,40] = 0.316443717683753
        struct[0].Gy_ini[104,41] = 0.171091579417532
        struct[0].Gy_ini[104,42] = 0.316443717683753
        struct[0].Gy_ini[104,43] = 0.171091579417531
        struct[0].Gy_ini[104,44] = -1.10755301189314
        struct[0].Gy_ini[104,45] = -0.598820527961360
        struct[0].Gy_ini[104,104] = -1
        struct[0].Gy_ini[105,40] = -0.171091579417532
        struct[0].Gy_ini[105,41] = 0.316443717683753
        struct[0].Gy_ini[105,42] = -0.171091579417531
        struct[0].Gy_ini[105,43] = 0.316443717683753
        struct[0].Gy_ini[105,44] = 0.598820527961360
        struct[0].Gy_ini[105,45] = -1.10755301189314
        struct[0].Gy_ini[105,105] = -1
        struct[0].Gy_ini[106,40] = 1.10755301189314
        struct[0].Gy_ini[106,41] = 0.598820527961361
        struct[0].Gy_ini[106,42] = -0.316443717683753
        struct[0].Gy_ini[106,43] = -0.171091579417532
        struct[0].Gy_ini[106,44] = -0.316443717683753
        struct[0].Gy_ini[106,45] = -0.171091579417532
        struct[0].Gy_ini[106,46] = -1.10755301189314
        struct[0].Gy_ini[106,47] = -0.598820527961361
        struct[0].Gy_ini[106,48] = 0.316443717683753
        struct[0].Gy_ini[106,49] = 0.171091579417532
        struct[0].Gy_ini[106,50] = 0.316443717683753
        struct[0].Gy_ini[106,51] = 0.171091579417532
        struct[0].Gy_ini[106,106] = -1
        struct[0].Gy_ini[107,40] = -0.598820527961361
        struct[0].Gy_ini[107,41] = 1.10755301189314
        struct[0].Gy_ini[107,42] = 0.171091579417532
        struct[0].Gy_ini[107,43] = -0.316443717683753
        struct[0].Gy_ini[107,44] = 0.171091579417532
        struct[0].Gy_ini[107,45] = -0.316443717683753
        struct[0].Gy_ini[107,46] = 0.598820527961361
        struct[0].Gy_ini[107,47] = -1.10755301189314
        struct[0].Gy_ini[107,48] = -0.171091579417532
        struct[0].Gy_ini[107,49] = 0.316443717683753
        struct[0].Gy_ini[107,50] = -0.171091579417532
        struct[0].Gy_ini[107,51] = 0.316443717683753
        struct[0].Gy_ini[107,107] = -1
        struct[0].Gy_ini[108,40] = -0.316443717683753
        struct[0].Gy_ini[108,41] = -0.171091579417532
        struct[0].Gy_ini[108,42] = 1.10755301189314
        struct[0].Gy_ini[108,43] = 0.598820527961360
        struct[0].Gy_ini[108,44] = -0.316443717683753
        struct[0].Gy_ini[108,45] = -0.171091579417531
        struct[0].Gy_ini[108,46] = 0.316443717683753
        struct[0].Gy_ini[108,47] = 0.171091579417532
        struct[0].Gy_ini[108,48] = -1.10755301189314
        struct[0].Gy_ini[108,49] = -0.598820527961360
        struct[0].Gy_ini[108,50] = 0.316443717683753
        struct[0].Gy_ini[108,51] = 0.171091579417531
        struct[0].Gy_ini[108,108] = -1
        struct[0].Gy_ini[109,40] = 0.171091579417532
        struct[0].Gy_ini[109,41] = -0.316443717683753
        struct[0].Gy_ini[109,42] = -0.598820527961360
        struct[0].Gy_ini[109,43] = 1.10755301189314
        struct[0].Gy_ini[109,44] = 0.171091579417531
        struct[0].Gy_ini[109,45] = -0.316443717683753
        struct[0].Gy_ini[109,46] = -0.171091579417532
        struct[0].Gy_ini[109,47] = 0.316443717683753
        struct[0].Gy_ini[109,48] = 0.598820527961360
        struct[0].Gy_ini[109,49] = -1.10755301189314
        struct[0].Gy_ini[109,50] = -0.171091579417531
        struct[0].Gy_ini[109,51] = 0.316443717683753
        struct[0].Gy_ini[109,109] = -1
        struct[0].Gy_ini[110,40] = -0.316443717683753
        struct[0].Gy_ini[110,41] = -0.171091579417532
        struct[0].Gy_ini[110,42] = -0.316443717683753
        struct[0].Gy_ini[110,43] = -0.171091579417531
        struct[0].Gy_ini[110,44] = 1.10755301189314
        struct[0].Gy_ini[110,45] = 0.598820527961360
        struct[0].Gy_ini[110,46] = 0.316443717683753
        struct[0].Gy_ini[110,47] = 0.171091579417532
        struct[0].Gy_ini[110,48] = 0.316443717683753
        struct[0].Gy_ini[110,49] = 0.171091579417531
        struct[0].Gy_ini[110,50] = -1.10755301189314
        struct[0].Gy_ini[110,51] = -0.598820527961360
        struct[0].Gy_ini[110,110] = -1
        struct[0].Gy_ini[111,40] = 0.171091579417532
        struct[0].Gy_ini[111,41] = -0.316443717683753
        struct[0].Gy_ini[111,42] = 0.171091579417531
        struct[0].Gy_ini[111,43] = -0.316443717683753
        struct[0].Gy_ini[111,44] = -0.598820527961360
        struct[0].Gy_ini[111,45] = 1.10755301189314
        struct[0].Gy_ini[111,46] = -0.171091579417532
        struct[0].Gy_ini[111,47] = 0.316443717683753
        struct[0].Gy_ini[111,48] = -0.171091579417531
        struct[0].Gy_ini[111,49] = 0.316443717683753
        struct[0].Gy_ini[111,50] = 0.598820527961360
        struct[0].Gy_ini[111,51] = -1.10755301189314
        struct[0].Gy_ini[111,111] = -1
        struct[0].Gy_ini[112,46] = 1.10755301189314
        struct[0].Gy_ini[112,47] = 0.598820527961361
        struct[0].Gy_ini[112,48] = -0.316443717683753
        struct[0].Gy_ini[112,49] = -0.171091579417532
        struct[0].Gy_ini[112,50] = -0.316443717683753
        struct[0].Gy_ini[112,51] = -0.171091579417532
        struct[0].Gy_ini[112,52] = -1.10755301189314
        struct[0].Gy_ini[112,53] = -0.598820527961361
        struct[0].Gy_ini[112,54] = 0.316443717683753
        struct[0].Gy_ini[112,55] = 0.171091579417532
        struct[0].Gy_ini[112,56] = 0.316443717683753
        struct[0].Gy_ini[112,57] = 0.171091579417532
        struct[0].Gy_ini[112,112] = -1
        struct[0].Gy_ini[113,46] = -0.598820527961361
        struct[0].Gy_ini[113,47] = 1.10755301189314
        struct[0].Gy_ini[113,48] = 0.171091579417532
        struct[0].Gy_ini[113,49] = -0.316443717683753
        struct[0].Gy_ini[113,50] = 0.171091579417532
        struct[0].Gy_ini[113,51] = -0.316443717683753
        struct[0].Gy_ini[113,52] = 0.598820527961361
        struct[0].Gy_ini[113,53] = -1.10755301189314
        struct[0].Gy_ini[113,54] = -0.171091579417532
        struct[0].Gy_ini[113,55] = 0.316443717683753
        struct[0].Gy_ini[113,56] = -0.171091579417532
        struct[0].Gy_ini[113,57] = 0.316443717683753
        struct[0].Gy_ini[113,113] = -1
        struct[0].Gy_ini[114,46] = -0.316443717683753
        struct[0].Gy_ini[114,47] = -0.171091579417532
        struct[0].Gy_ini[114,48] = 1.10755301189314
        struct[0].Gy_ini[114,49] = 0.598820527961360
        struct[0].Gy_ini[114,50] = -0.316443717683753
        struct[0].Gy_ini[114,51] = -0.171091579417531
        struct[0].Gy_ini[114,52] = 0.316443717683753
        struct[0].Gy_ini[114,53] = 0.171091579417532
        struct[0].Gy_ini[114,54] = -1.10755301189314
        struct[0].Gy_ini[114,55] = -0.598820527961360
        struct[0].Gy_ini[114,56] = 0.316443717683753
        struct[0].Gy_ini[114,57] = 0.171091579417531
        struct[0].Gy_ini[114,114] = -1
        struct[0].Gy_ini[115,46] = 0.171091579417532
        struct[0].Gy_ini[115,47] = -0.316443717683753
        struct[0].Gy_ini[115,48] = -0.598820527961360
        struct[0].Gy_ini[115,49] = 1.10755301189314
        struct[0].Gy_ini[115,50] = 0.171091579417531
        struct[0].Gy_ini[115,51] = -0.316443717683753
        struct[0].Gy_ini[115,52] = -0.171091579417532
        struct[0].Gy_ini[115,53] = 0.316443717683753
        struct[0].Gy_ini[115,54] = 0.598820527961360
        struct[0].Gy_ini[115,55] = -1.10755301189314
        struct[0].Gy_ini[115,56] = -0.171091579417531
        struct[0].Gy_ini[115,57] = 0.316443717683753
        struct[0].Gy_ini[115,115] = -1
        struct[0].Gy_ini[116,46] = -0.316443717683753
        struct[0].Gy_ini[116,47] = -0.171091579417532
        struct[0].Gy_ini[116,48] = -0.316443717683753
        struct[0].Gy_ini[116,49] = -0.171091579417531
        struct[0].Gy_ini[116,50] = 1.10755301189314
        struct[0].Gy_ini[116,51] = 0.598820527961360
        struct[0].Gy_ini[116,52] = 0.316443717683753
        struct[0].Gy_ini[116,53] = 0.171091579417532
        struct[0].Gy_ini[116,54] = 0.316443717683753
        struct[0].Gy_ini[116,55] = 0.171091579417531
        struct[0].Gy_ini[116,56] = -1.10755301189314
        struct[0].Gy_ini[116,57] = -0.598820527961360
        struct[0].Gy_ini[116,116] = -1
        struct[0].Gy_ini[117,46] = 0.171091579417532
        struct[0].Gy_ini[117,47] = -0.316443717683753
        struct[0].Gy_ini[117,48] = 0.171091579417531
        struct[0].Gy_ini[117,49] = -0.316443717683753
        struct[0].Gy_ini[117,50] = -0.598820527961360
        struct[0].Gy_ini[117,51] = 1.10755301189314
        struct[0].Gy_ini[117,52] = -0.171091579417532
        struct[0].Gy_ini[117,53] = 0.316443717683753
        struct[0].Gy_ini[117,54] = -0.171091579417531
        struct[0].Gy_ini[117,55] = 0.316443717683753
        struct[0].Gy_ini[117,56] = 0.598820527961360
        struct[0].Gy_ini[117,57] = -1.10755301189314
        struct[0].Gy_ini[117,117] = -1
        struct[0].Gy_ini[118,58] = 1.10755301189314
        struct[0].Gy_ini[118,59] = 0.598820527961361
        struct[0].Gy_ini[118,60] = -0.316443717683753
        struct[0].Gy_ini[118,61] = -0.171091579417532
        struct[0].Gy_ini[118,62] = -0.316443717683753
        struct[0].Gy_ini[118,63] = -0.171091579417532
        struct[0].Gy_ini[118,64] = -1.10755301189314
        struct[0].Gy_ini[118,65] = -0.598820527961361
        struct[0].Gy_ini[118,66] = 0.316443717683753
        struct[0].Gy_ini[118,67] = 0.171091579417532
        struct[0].Gy_ini[118,68] = 0.316443717683753
        struct[0].Gy_ini[118,69] = 0.171091579417532
        struct[0].Gy_ini[118,118] = -1
        struct[0].Gy_ini[119,58] = -0.598820527961361
        struct[0].Gy_ini[119,59] = 1.10755301189314
        struct[0].Gy_ini[119,60] = 0.171091579417532
        struct[0].Gy_ini[119,61] = -0.316443717683753
        struct[0].Gy_ini[119,62] = 0.171091579417532
        struct[0].Gy_ini[119,63] = -0.316443717683753
        struct[0].Gy_ini[119,64] = 0.598820527961361
        struct[0].Gy_ini[119,65] = -1.10755301189314
        struct[0].Gy_ini[119,66] = -0.171091579417532
        struct[0].Gy_ini[119,67] = 0.316443717683753
        struct[0].Gy_ini[119,68] = -0.171091579417532
        struct[0].Gy_ini[119,69] = 0.316443717683753
        struct[0].Gy_ini[119,119] = -1
        struct[0].Gy_ini[120,58] = -0.316443717683753
        struct[0].Gy_ini[120,59] = -0.171091579417532
        struct[0].Gy_ini[120,60] = 1.10755301189314
        struct[0].Gy_ini[120,61] = 0.598820527961360
        struct[0].Gy_ini[120,62] = -0.316443717683753
        struct[0].Gy_ini[120,63] = -0.171091579417531
        struct[0].Gy_ini[120,64] = 0.316443717683753
        struct[0].Gy_ini[120,65] = 0.171091579417532
        struct[0].Gy_ini[120,66] = -1.10755301189314
        struct[0].Gy_ini[120,67] = -0.598820527961360
        struct[0].Gy_ini[120,68] = 0.316443717683753
        struct[0].Gy_ini[120,69] = 0.171091579417531
        struct[0].Gy_ini[120,120] = -1
        struct[0].Gy_ini[121,58] = 0.171091579417532
        struct[0].Gy_ini[121,59] = -0.316443717683753
        struct[0].Gy_ini[121,60] = -0.598820527961360
        struct[0].Gy_ini[121,61] = 1.10755301189314
        struct[0].Gy_ini[121,62] = 0.171091579417531
        struct[0].Gy_ini[121,63] = -0.316443717683753
        struct[0].Gy_ini[121,64] = -0.171091579417532
        struct[0].Gy_ini[121,65] = 0.316443717683753
        struct[0].Gy_ini[121,66] = 0.598820527961360
        struct[0].Gy_ini[121,67] = -1.10755301189314
        struct[0].Gy_ini[121,68] = -0.171091579417531
        struct[0].Gy_ini[121,69] = 0.316443717683753
        struct[0].Gy_ini[121,121] = -1
        struct[0].Gy_ini[122,58] = -0.316443717683753
        struct[0].Gy_ini[122,59] = -0.171091579417532
        struct[0].Gy_ini[122,60] = -0.316443717683753
        struct[0].Gy_ini[122,61] = -0.171091579417531
        struct[0].Gy_ini[122,62] = 1.10755301189314
        struct[0].Gy_ini[122,63] = 0.598820527961360
        struct[0].Gy_ini[122,64] = 0.316443717683753
        struct[0].Gy_ini[122,65] = 0.171091579417532
        struct[0].Gy_ini[122,66] = 0.316443717683753
        struct[0].Gy_ini[122,67] = 0.171091579417531
        struct[0].Gy_ini[122,68] = -1.10755301189314
        struct[0].Gy_ini[122,69] = -0.598820527961360
        struct[0].Gy_ini[122,122] = -1
        struct[0].Gy_ini[123,58] = 0.171091579417532
        struct[0].Gy_ini[123,59] = -0.316443717683753
        struct[0].Gy_ini[123,60] = 0.171091579417531
        struct[0].Gy_ini[123,61] = -0.316443717683753
        struct[0].Gy_ini[123,62] = -0.598820527961360
        struct[0].Gy_ini[123,63] = 1.10755301189314
        struct[0].Gy_ini[123,64] = -0.171091579417532
        struct[0].Gy_ini[123,65] = 0.316443717683753
        struct[0].Gy_ini[123,66] = -0.171091579417531
        struct[0].Gy_ini[123,67] = 0.316443717683753
        struct[0].Gy_ini[123,68] = 0.598820527961360
        struct[0].Gy_ini[123,69] = -1.10755301189314
        struct[0].Gy_ini[123,123] = -1
        struct[0].Gy_ini[124,64] = 1.10755301189314
        struct[0].Gy_ini[124,65] = 0.598820527961361
        struct[0].Gy_ini[124,66] = -0.316443717683753
        struct[0].Gy_ini[124,67] = -0.171091579417532
        struct[0].Gy_ini[124,68] = -0.316443717683753
        struct[0].Gy_ini[124,69] = -0.171091579417532
        struct[0].Gy_ini[124,124] = -1
        struct[0].Gy_ini[125,64] = -0.598820527961361
        struct[0].Gy_ini[125,65] = 1.10755301189314
        struct[0].Gy_ini[125,66] = 0.171091579417532
        struct[0].Gy_ini[125,67] = -0.316443717683753
        struct[0].Gy_ini[125,68] = 0.171091579417532
        struct[0].Gy_ini[125,69] = -0.316443717683753
        struct[0].Gy_ini[125,125] = -1
        struct[0].Gy_ini[126,64] = -0.316443717683753
        struct[0].Gy_ini[126,65] = -0.171091579417532
        struct[0].Gy_ini[126,66] = 1.10755301189314
        struct[0].Gy_ini[126,67] = 0.598820527961360
        struct[0].Gy_ini[126,68] = -0.316443717683753
        struct[0].Gy_ini[126,69] = -0.171091579417531
        struct[0].Gy_ini[126,126] = -1
        struct[0].Gy_ini[127,64] = 0.171091579417532
        struct[0].Gy_ini[127,65] = -0.316443717683753
        struct[0].Gy_ini[127,66] = -0.598820527961360
        struct[0].Gy_ini[127,67] = 1.10755301189314
        struct[0].Gy_ini[127,68] = 0.171091579417531
        struct[0].Gy_ini[127,69] = -0.316443717683753
        struct[0].Gy_ini[127,127] = -1
        struct[0].Gy_ini[128,64] = -0.316443717683753
        struct[0].Gy_ini[128,65] = -0.171091579417532
        struct[0].Gy_ini[128,66] = -0.316443717683753
        struct[0].Gy_ini[128,67] = -0.171091579417531
        struct[0].Gy_ini[128,68] = 1.10755301189314
        struct[0].Gy_ini[128,69] = 0.598820527961360
        struct[0].Gy_ini[128,128] = -1
        struct[0].Gy_ini[129,64] = 0.171091579417532
        struct[0].Gy_ini[129,65] = -0.316443717683753
        struct[0].Gy_ini[129,66] = 0.171091579417531
        struct[0].Gy_ini[129,67] = -0.316443717683753
        struct[0].Gy_ini[129,68] = -0.598820527961360
        struct[0].Gy_ini[129,69] = 1.10755301189314
        struct[0].Gy_ini[129,129] = -1
        struct[0].Gy_ini[130,0] = i_load_B2lv_a_r
        struct[0].Gy_ini[130,1] = i_load_B2lv_a_i
        struct[0].Gy_ini[130,6] = -i_load_B2lv_a_r
        struct[0].Gy_ini[130,7] = -i_load_B2lv_a_i
        struct[0].Gy_ini[130,130] = v_B2lv_a_r - v_B2lv_n_r
        struct[0].Gy_ini[130,131] = v_B2lv_a_i - v_B2lv_n_i
        struct[0].Gy_ini[130,239] = -1
        struct[0].Gy_ini[131,2] = i_load_B2lv_b_r
        struct[0].Gy_ini[131,3] = i_load_B2lv_b_i
        struct[0].Gy_ini[131,6] = -i_load_B2lv_b_r
        struct[0].Gy_ini[131,7] = -i_load_B2lv_b_i
        struct[0].Gy_ini[131,132] = v_B2lv_b_r - v_B2lv_n_r
        struct[0].Gy_ini[131,133] = v_B2lv_b_i - v_B2lv_n_i
        struct[0].Gy_ini[131,240] = -1
        struct[0].Gy_ini[132,4] = i_load_B2lv_c_r
        struct[0].Gy_ini[132,5] = i_load_B2lv_c_i
        struct[0].Gy_ini[132,6] = -i_load_B2lv_c_r
        struct[0].Gy_ini[132,7] = -i_load_B2lv_c_i
        struct[0].Gy_ini[132,134] = v_B2lv_c_r - v_B2lv_n_r
        struct[0].Gy_ini[132,135] = v_B2lv_c_i - v_B2lv_n_i
        struct[0].Gy_ini[132,241] = -1
        struct[0].Gy_ini[133,0] = -i_load_B2lv_a_i
        struct[0].Gy_ini[133,1] = i_load_B2lv_a_r
        struct[0].Gy_ini[133,6] = i_load_B2lv_a_i
        struct[0].Gy_ini[133,7] = -i_load_B2lv_a_r
        struct[0].Gy_ini[133,130] = v_B2lv_a_i - v_B2lv_n_i
        struct[0].Gy_ini[133,131] = -v_B2lv_a_r + v_B2lv_n_r
        struct[0].Gy_ini[134,2] = -i_load_B2lv_b_i
        struct[0].Gy_ini[134,3] = i_load_B2lv_b_r
        struct[0].Gy_ini[134,6] = i_load_B2lv_b_i
        struct[0].Gy_ini[134,7] = -i_load_B2lv_b_r
        struct[0].Gy_ini[134,132] = v_B2lv_b_i - v_B2lv_n_i
        struct[0].Gy_ini[134,133] = -v_B2lv_b_r + v_B2lv_n_r
        struct[0].Gy_ini[135,4] = -i_load_B2lv_c_i
        struct[0].Gy_ini[135,5] = i_load_B2lv_c_r
        struct[0].Gy_ini[135,6] = i_load_B2lv_c_i
        struct[0].Gy_ini[135,7] = -i_load_B2lv_c_r
        struct[0].Gy_ini[135,134] = v_B2lv_c_i - v_B2lv_n_i
        struct[0].Gy_ini[135,135] = -v_B2lv_c_r + v_B2lv_n_r
        struct[0].Gy_ini[136,130] = 1
        struct[0].Gy_ini[136,132] = 1
        struct[0].Gy_ini[136,134] = 1
        struct[0].Gy_ini[136,136] = 1
        struct[0].Gy_ini[137,131] = 1
        struct[0].Gy_ini[137,133] = 1
        struct[0].Gy_ini[137,135] = 1
        struct[0].Gy_ini[137,137] = 1
        struct[0].Gy_ini[138,8] = i_load_B3lv_a_r
        struct[0].Gy_ini[138,9] = i_load_B3lv_a_i
        struct[0].Gy_ini[138,14] = -i_load_B3lv_a_r
        struct[0].Gy_ini[138,15] = -i_load_B3lv_a_i
        struct[0].Gy_ini[138,138] = v_B3lv_a_r - v_B3lv_n_r
        struct[0].Gy_ini[138,139] = v_B3lv_a_i - v_B3lv_n_i
        struct[0].Gy_ini[138,242] = -1
        struct[0].Gy_ini[139,10] = i_load_B3lv_b_r
        struct[0].Gy_ini[139,11] = i_load_B3lv_b_i
        struct[0].Gy_ini[139,14] = -i_load_B3lv_b_r
        struct[0].Gy_ini[139,15] = -i_load_B3lv_b_i
        struct[0].Gy_ini[139,140] = v_B3lv_b_r - v_B3lv_n_r
        struct[0].Gy_ini[139,141] = v_B3lv_b_i - v_B3lv_n_i
        struct[0].Gy_ini[139,243] = -1
        struct[0].Gy_ini[140,12] = i_load_B3lv_c_r
        struct[0].Gy_ini[140,13] = i_load_B3lv_c_i
        struct[0].Gy_ini[140,14] = -i_load_B3lv_c_r
        struct[0].Gy_ini[140,15] = -i_load_B3lv_c_i
        struct[0].Gy_ini[140,142] = v_B3lv_c_r - v_B3lv_n_r
        struct[0].Gy_ini[140,143] = v_B3lv_c_i - v_B3lv_n_i
        struct[0].Gy_ini[140,244] = -1
        struct[0].Gy_ini[141,8] = -i_load_B3lv_a_i
        struct[0].Gy_ini[141,9] = i_load_B3lv_a_r
        struct[0].Gy_ini[141,14] = i_load_B3lv_a_i
        struct[0].Gy_ini[141,15] = -i_load_B3lv_a_r
        struct[0].Gy_ini[141,138] = v_B3lv_a_i - v_B3lv_n_i
        struct[0].Gy_ini[141,139] = -v_B3lv_a_r + v_B3lv_n_r
        struct[0].Gy_ini[142,10] = -i_load_B3lv_b_i
        struct[0].Gy_ini[142,11] = i_load_B3lv_b_r
        struct[0].Gy_ini[142,14] = i_load_B3lv_b_i
        struct[0].Gy_ini[142,15] = -i_load_B3lv_b_r
        struct[0].Gy_ini[142,140] = v_B3lv_b_i - v_B3lv_n_i
        struct[0].Gy_ini[142,141] = -v_B3lv_b_r + v_B3lv_n_r
        struct[0].Gy_ini[143,12] = -i_load_B3lv_c_i
        struct[0].Gy_ini[143,13] = i_load_B3lv_c_r
        struct[0].Gy_ini[143,14] = i_load_B3lv_c_i
        struct[0].Gy_ini[143,15] = -i_load_B3lv_c_r
        struct[0].Gy_ini[143,142] = v_B3lv_c_i - v_B3lv_n_i
        struct[0].Gy_ini[143,143] = -v_B3lv_c_r + v_B3lv_n_r
        struct[0].Gy_ini[144,138] = 1
        struct[0].Gy_ini[144,140] = 1
        struct[0].Gy_ini[144,142] = 1
        struct[0].Gy_ini[144,144] = 1
        struct[0].Gy_ini[145,139] = 1
        struct[0].Gy_ini[145,141] = 1
        struct[0].Gy_ini[145,143] = 1
        struct[0].Gy_ini[145,145] = 1
        struct[0].Gy_ini[146,16] = i_load_B4lv_a_r
        struct[0].Gy_ini[146,17] = i_load_B4lv_a_i
        struct[0].Gy_ini[146,22] = -i_load_B4lv_a_r
        struct[0].Gy_ini[146,23] = -i_load_B4lv_a_i
        struct[0].Gy_ini[146,146] = v_B4lv_a_r - v_B4lv_n_r
        struct[0].Gy_ini[146,147] = v_B4lv_a_i - v_B4lv_n_i
        struct[0].Gy_ini[146,245] = -1
        struct[0].Gy_ini[147,18] = i_load_B4lv_b_r
        struct[0].Gy_ini[147,19] = i_load_B4lv_b_i
        struct[0].Gy_ini[147,22] = -i_load_B4lv_b_r
        struct[0].Gy_ini[147,23] = -i_load_B4lv_b_i
        struct[0].Gy_ini[147,148] = v_B4lv_b_r - v_B4lv_n_r
        struct[0].Gy_ini[147,149] = v_B4lv_b_i - v_B4lv_n_i
        struct[0].Gy_ini[147,246] = -1
        struct[0].Gy_ini[148,20] = i_load_B4lv_c_r
        struct[0].Gy_ini[148,21] = i_load_B4lv_c_i
        struct[0].Gy_ini[148,22] = -i_load_B4lv_c_r
        struct[0].Gy_ini[148,23] = -i_load_B4lv_c_i
        struct[0].Gy_ini[148,150] = v_B4lv_c_r - v_B4lv_n_r
        struct[0].Gy_ini[148,151] = v_B4lv_c_i - v_B4lv_n_i
        struct[0].Gy_ini[148,247] = -1
        struct[0].Gy_ini[149,16] = -i_load_B4lv_a_i
        struct[0].Gy_ini[149,17] = i_load_B4lv_a_r
        struct[0].Gy_ini[149,22] = i_load_B4lv_a_i
        struct[0].Gy_ini[149,23] = -i_load_B4lv_a_r
        struct[0].Gy_ini[149,146] = v_B4lv_a_i - v_B4lv_n_i
        struct[0].Gy_ini[149,147] = -v_B4lv_a_r + v_B4lv_n_r
        struct[0].Gy_ini[150,18] = -i_load_B4lv_b_i
        struct[0].Gy_ini[150,19] = i_load_B4lv_b_r
        struct[0].Gy_ini[150,22] = i_load_B4lv_b_i
        struct[0].Gy_ini[150,23] = -i_load_B4lv_b_r
        struct[0].Gy_ini[150,148] = v_B4lv_b_i - v_B4lv_n_i
        struct[0].Gy_ini[150,149] = -v_B4lv_b_r + v_B4lv_n_r
        struct[0].Gy_ini[151,20] = -i_load_B4lv_c_i
        struct[0].Gy_ini[151,21] = i_load_B4lv_c_r
        struct[0].Gy_ini[151,22] = i_load_B4lv_c_i
        struct[0].Gy_ini[151,23] = -i_load_B4lv_c_r
        struct[0].Gy_ini[151,150] = v_B4lv_c_i - v_B4lv_n_i
        struct[0].Gy_ini[151,151] = -v_B4lv_c_r + v_B4lv_n_r
        struct[0].Gy_ini[152,146] = 1
        struct[0].Gy_ini[152,148] = 1
        struct[0].Gy_ini[152,150] = 1
        struct[0].Gy_ini[152,152] = 1
        struct[0].Gy_ini[153,147] = 1
        struct[0].Gy_ini[153,149] = 1
        struct[0].Gy_ini[153,151] = 1
        struct[0].Gy_ini[153,153] = 1
        struct[0].Gy_ini[154,24] = i_load_B5lv_a_r
        struct[0].Gy_ini[154,25] = i_load_B5lv_a_i
        struct[0].Gy_ini[154,30] = -i_load_B5lv_a_r
        struct[0].Gy_ini[154,31] = -i_load_B5lv_a_i
        struct[0].Gy_ini[154,154] = v_B5lv_a_r - v_B5lv_n_r
        struct[0].Gy_ini[154,155] = v_B5lv_a_i - v_B5lv_n_i
        struct[0].Gy_ini[154,248] = -1
        struct[0].Gy_ini[155,26] = i_load_B5lv_b_r
        struct[0].Gy_ini[155,27] = i_load_B5lv_b_i
        struct[0].Gy_ini[155,30] = -i_load_B5lv_b_r
        struct[0].Gy_ini[155,31] = -i_load_B5lv_b_i
        struct[0].Gy_ini[155,156] = v_B5lv_b_r - v_B5lv_n_r
        struct[0].Gy_ini[155,157] = v_B5lv_b_i - v_B5lv_n_i
        struct[0].Gy_ini[155,249] = -1
        struct[0].Gy_ini[156,28] = i_load_B5lv_c_r
        struct[0].Gy_ini[156,29] = i_load_B5lv_c_i
        struct[0].Gy_ini[156,30] = -i_load_B5lv_c_r
        struct[0].Gy_ini[156,31] = -i_load_B5lv_c_i
        struct[0].Gy_ini[156,158] = v_B5lv_c_r - v_B5lv_n_r
        struct[0].Gy_ini[156,159] = v_B5lv_c_i - v_B5lv_n_i
        struct[0].Gy_ini[156,250] = -1
        struct[0].Gy_ini[157,24] = -i_load_B5lv_a_i
        struct[0].Gy_ini[157,25] = i_load_B5lv_a_r
        struct[0].Gy_ini[157,30] = i_load_B5lv_a_i
        struct[0].Gy_ini[157,31] = -i_load_B5lv_a_r
        struct[0].Gy_ini[157,154] = v_B5lv_a_i - v_B5lv_n_i
        struct[0].Gy_ini[157,155] = -v_B5lv_a_r + v_B5lv_n_r
        struct[0].Gy_ini[158,26] = -i_load_B5lv_b_i
        struct[0].Gy_ini[158,27] = i_load_B5lv_b_r
        struct[0].Gy_ini[158,30] = i_load_B5lv_b_i
        struct[0].Gy_ini[158,31] = -i_load_B5lv_b_r
        struct[0].Gy_ini[158,156] = v_B5lv_b_i - v_B5lv_n_i
        struct[0].Gy_ini[158,157] = -v_B5lv_b_r + v_B5lv_n_r
        struct[0].Gy_ini[159,28] = -i_load_B5lv_c_i
        struct[0].Gy_ini[159,29] = i_load_B5lv_c_r
        struct[0].Gy_ini[159,30] = i_load_B5lv_c_i
        struct[0].Gy_ini[159,31] = -i_load_B5lv_c_r
        struct[0].Gy_ini[159,158] = v_B5lv_c_i - v_B5lv_n_i
        struct[0].Gy_ini[159,159] = -v_B5lv_c_r + v_B5lv_n_r
        struct[0].Gy_ini[160,154] = 1
        struct[0].Gy_ini[160,156] = 1
        struct[0].Gy_ini[160,158] = 1
        struct[0].Gy_ini[160,160] = 1
        struct[0].Gy_ini[161,155] = 1
        struct[0].Gy_ini[161,157] = 1
        struct[0].Gy_ini[161,159] = 1
        struct[0].Gy_ini[161,161] = 1
        struct[0].Gy_ini[162,32] = i_load_B6lv_a_r
        struct[0].Gy_ini[162,33] = i_load_B6lv_a_i
        struct[0].Gy_ini[162,38] = -i_load_B6lv_a_r
        struct[0].Gy_ini[162,39] = -i_load_B6lv_a_i
        struct[0].Gy_ini[162,162] = v_B6lv_a_r - v_B6lv_n_r
        struct[0].Gy_ini[162,163] = v_B6lv_a_i - v_B6lv_n_i
        struct[0].Gy_ini[162,251] = -1
        struct[0].Gy_ini[163,34] = i_load_B6lv_b_r
        struct[0].Gy_ini[163,35] = i_load_B6lv_b_i
        struct[0].Gy_ini[163,38] = -i_load_B6lv_b_r
        struct[0].Gy_ini[163,39] = -i_load_B6lv_b_i
        struct[0].Gy_ini[163,164] = v_B6lv_b_r - v_B6lv_n_r
        struct[0].Gy_ini[163,165] = v_B6lv_b_i - v_B6lv_n_i
        struct[0].Gy_ini[163,252] = -1
        struct[0].Gy_ini[164,36] = i_load_B6lv_c_r
        struct[0].Gy_ini[164,37] = i_load_B6lv_c_i
        struct[0].Gy_ini[164,38] = -i_load_B6lv_c_r
        struct[0].Gy_ini[164,39] = -i_load_B6lv_c_i
        struct[0].Gy_ini[164,166] = v_B6lv_c_r - v_B6lv_n_r
        struct[0].Gy_ini[164,167] = v_B6lv_c_i - v_B6lv_n_i
        struct[0].Gy_ini[164,253] = -1
        struct[0].Gy_ini[165,32] = -i_load_B6lv_a_i
        struct[0].Gy_ini[165,33] = i_load_B6lv_a_r
        struct[0].Gy_ini[165,38] = i_load_B6lv_a_i
        struct[0].Gy_ini[165,39] = -i_load_B6lv_a_r
        struct[0].Gy_ini[165,162] = v_B6lv_a_i - v_B6lv_n_i
        struct[0].Gy_ini[165,163] = -v_B6lv_a_r + v_B6lv_n_r
        struct[0].Gy_ini[166,34] = -i_load_B6lv_b_i
        struct[0].Gy_ini[166,35] = i_load_B6lv_b_r
        struct[0].Gy_ini[166,38] = i_load_B6lv_b_i
        struct[0].Gy_ini[166,39] = -i_load_B6lv_b_r
        struct[0].Gy_ini[166,164] = v_B6lv_b_i - v_B6lv_n_i
        struct[0].Gy_ini[166,165] = -v_B6lv_b_r + v_B6lv_n_r
        struct[0].Gy_ini[167,36] = -i_load_B6lv_c_i
        struct[0].Gy_ini[167,37] = i_load_B6lv_c_r
        struct[0].Gy_ini[167,38] = i_load_B6lv_c_i
        struct[0].Gy_ini[167,39] = -i_load_B6lv_c_r
        struct[0].Gy_ini[167,166] = v_B6lv_c_i - v_B6lv_n_i
        struct[0].Gy_ini[167,167] = -v_B6lv_c_r + v_B6lv_n_r
        struct[0].Gy_ini[168,162] = 1
        struct[0].Gy_ini[168,164] = 1
        struct[0].Gy_ini[168,166] = 1
        struct[0].Gy_ini[168,168] = 1
        struct[0].Gy_ini[169,163] = 1
        struct[0].Gy_ini[169,165] = 1
        struct[0].Gy_ini[169,167] = 1
        struct[0].Gy_ini[169,169] = 1
        struct[0].Gy_ini[170,170] = -R_1112
        struct[0].Gy_ini[170,194] = 1
        struct[0].Gy_ini[170,195] = -1
        struct[0].Gy_ini[171,171] = -R_1213
        struct[0].Gy_ini[171,195] = 1
        struct[0].Gy_ini[172,172] = -R_1314
        struct[0].Gy_ini[172,197] = -1
        struct[0].Gy_ini[173,173] = -R_1415
        struct[0].Gy_ini[173,197] = 1
        struct[0].Gy_ini[173,198] = -1
        struct[0].Gy_ini[174,174] = -R_2122
        struct[0].Gy_ini[174,199] = 1
        struct[0].Gy_ini[174,200] = -1
        struct[0].Gy_ini[175,175] = -R_2223
        struct[0].Gy_ini[175,200] = 1
        struct[0].Gy_ini[176,176] = -R_2324
        struct[0].Gy_ini[176,202] = -1
        struct[0].Gy_ini[177,177] = -R_2425
        struct[0].Gy_ini[177,202] = 1
        struct[0].Gy_ini[177,203] = -1
        struct[0].Gy_ini[178,178] = -R_3132
        struct[0].Gy_ini[178,204] = 1
        struct[0].Gy_ini[178,205] = -1
        struct[0].Gy_ini[179,179] = -R_3233
        struct[0].Gy_ini[179,205] = 1
        struct[0].Gy_ini[180,180] = -R_3334
        struct[0].Gy_ini[180,207] = -1
        struct[0].Gy_ini[181,181] = -R_3435
        struct[0].Gy_ini[181,207] = 1
        struct[0].Gy_ini[181,208] = -1
        struct[0].Gy_ini[182,182] = -R_4142
        struct[0].Gy_ini[182,209] = 1
        struct[0].Gy_ini[182,210] = -1
        struct[0].Gy_ini[183,183] = -R_4243
        struct[0].Gy_ini[183,210] = 1
        struct[0].Gy_ini[184,184] = -R_4344
        struct[0].Gy_ini[184,212] = -1
        struct[0].Gy_ini[185,185] = -R_4445
        struct[0].Gy_ini[185,212] = 1
        struct[0].Gy_ini[185,213] = -1
        struct[0].Gy_ini[186,186] = -R_5152
        struct[0].Gy_ini[186,214] = 1
        struct[0].Gy_ini[186,215] = -1
        struct[0].Gy_ini[187,187] = -R_5253
        struct[0].Gy_ini[187,215] = 1
        struct[0].Gy_ini[188,188] = -R_5354
        struct[0].Gy_ini[188,217] = -1
        struct[0].Gy_ini[189,189] = -R_5455
        struct[0].Gy_ini[189,217] = 1
        struct[0].Gy_ini[189,218] = -1
        struct[0].Gy_ini[190,190] = -R_1521
        struct[0].Gy_ini[190,198] = 1
        struct[0].Gy_ini[190,199] = -1
        struct[0].Gy_ini[191,191] = -R_2531
        struct[0].Gy_ini[191,203] = 1
        struct[0].Gy_ini[191,204] = -1
        struct[0].Gy_ini[192,192] = -R_3541
        struct[0].Gy_ini[192,208] = 1
        struct[0].Gy_ini[192,209] = -1
        struct[0].Gy_ini[193,193] = -R_4551
        struct[0].Gy_ini[193,213] = 1
        struct[0].Gy_ini[193,214] = -1
        struct[0].Gy_ini[194,170] = -1
        struct[0].Gy_ini[194,219] = 1
        struct[0].Gy_ini[195,170] = 1
        struct[0].Gy_ini[195,171] = -1
        struct[0].Gy_ini[195,220] = 1
        struct[0].Gy_ini[196,171] = 1
        struct[0].Gy_ini[196,172] = -1
        struct[0].Gy_ini[196,196] = 1
        struct[0].Gy_ini[197,172] = 1
        struct[0].Gy_ini[197,173] = -1
        struct[0].Gy_ini[197,221] = 1
        struct[0].Gy_ini[198,173] = 1
        struct[0].Gy_ini[198,190] = -1
        struct[0].Gy_ini[198,222] = 1
        struct[0].Gy_ini[199,174] = -1
        struct[0].Gy_ini[199,190] = 1
        struct[0].Gy_ini[199,223] = 1
        struct[0].Gy_ini[200,174] = 1
        struct[0].Gy_ini[200,175] = -1
        struct[0].Gy_ini[200,224] = 1
        struct[0].Gy_ini[201,175] = 1
        struct[0].Gy_ini[201,176] = -1
        struct[0].Gy_ini[201,201] = 1
        struct[0].Gy_ini[202,176] = 1
        struct[0].Gy_ini[202,177] = -1
        struct[0].Gy_ini[202,225] = 1
        struct[0].Gy_ini[203,177] = 1
        struct[0].Gy_ini[203,191] = -1
        struct[0].Gy_ini[203,226] = 1
        struct[0].Gy_ini[204,178] = -1
        struct[0].Gy_ini[204,191] = 1
        struct[0].Gy_ini[204,227] = 1
        struct[0].Gy_ini[205,178] = 1
        struct[0].Gy_ini[205,179] = -1
        struct[0].Gy_ini[205,228] = 1
        struct[0].Gy_ini[206,179] = 1
        struct[0].Gy_ini[206,180] = -1
        struct[0].Gy_ini[206,206] = 1
        struct[0].Gy_ini[207,180] = 1
        struct[0].Gy_ini[207,181] = -1
        struct[0].Gy_ini[207,229] = 1
        struct[0].Gy_ini[208,181] = 1
        struct[0].Gy_ini[208,192] = -1
        struct[0].Gy_ini[208,230] = 1
        struct[0].Gy_ini[209,182] = -1
        struct[0].Gy_ini[209,192] = 1
        struct[0].Gy_ini[209,231] = 1
        struct[0].Gy_ini[210,182] = 1
        struct[0].Gy_ini[210,183] = -1
        struct[0].Gy_ini[210,232] = 1
        struct[0].Gy_ini[211,183] = 1
        struct[0].Gy_ini[211,184] = -1
        struct[0].Gy_ini[211,211] = 1
        struct[0].Gy_ini[212,184] = 1
        struct[0].Gy_ini[212,185] = -1
        struct[0].Gy_ini[212,233] = 1
        struct[0].Gy_ini[213,185] = 1
        struct[0].Gy_ini[213,193] = -1
        struct[0].Gy_ini[213,234] = 1
        struct[0].Gy_ini[214,186] = -1
        struct[0].Gy_ini[214,193] = 1
        struct[0].Gy_ini[214,235] = 1
        struct[0].Gy_ini[215,186] = 1
        struct[0].Gy_ini[215,187] = -1
        struct[0].Gy_ini[215,236] = 1
        struct[0].Gy_ini[216,187] = 1
        struct[0].Gy_ini[216,188] = -1
        struct[0].Gy_ini[216,216] = 1
        struct[0].Gy_ini[217,188] = 1
        struct[0].Gy_ini[217,189] = -1
        struct[0].Gy_ini[217,237] = 1
        struct[0].Gy_ini[218,189] = 1
        struct[0].Gy_ini[218,238] = 1
        struct[0].Gy_ini[219,194] = i_11
        struct[0].Gy_ini[219,219] = v_11
        struct[0].Gy_ini[220,195] = i_12
        struct[0].Gy_ini[220,220] = v_12
        struct[0].Gy_ini[221,197] = i_14
        struct[0].Gy_ini[221,221] = v_14
        struct[0].Gy_ini[222,198] = i_15
        struct[0].Gy_ini[222,222] = v_15
        struct[0].Gy_ini[223,199] = i_21
        struct[0].Gy_ini[223,223] = v_21
        struct[0].Gy_ini[224,200] = i_22
        struct[0].Gy_ini[224,224] = v_22
        struct[0].Gy_ini[225,202] = i_24
        struct[0].Gy_ini[225,225] = v_24
        struct[0].Gy_ini[226,203] = i_25
        struct[0].Gy_ini[226,226] = v_25
        struct[0].Gy_ini[227,204] = i_31
        struct[0].Gy_ini[227,227] = v_31
        struct[0].Gy_ini[228,205] = i_32
        struct[0].Gy_ini[228,228] = v_32
        struct[0].Gy_ini[229,207] = i_34
        struct[0].Gy_ini[229,229] = v_34
        struct[0].Gy_ini[230,208] = i_35
        struct[0].Gy_ini[230,230] = v_35
        struct[0].Gy_ini[231,209] = i_41
        struct[0].Gy_ini[231,231] = v_41
        struct[0].Gy_ini[232,210] = i_42
        struct[0].Gy_ini[232,232] = v_42
        struct[0].Gy_ini[233,212] = i_44
        struct[0].Gy_ini[233,233] = v_44
        struct[0].Gy_ini[234,213] = i_45
        struct[0].Gy_ini[234,234] = v_45
        struct[0].Gy_ini[235,214] = i_51
        struct[0].Gy_ini[235,235] = v_51
        struct[0].Gy_ini[236,215] = i_52
        struct[0].Gy_ini[236,236] = v_52
        struct[0].Gy_ini[237,217] = i_54
        struct[0].Gy_ini[237,237] = v_54
        struct[0].Gy_ini[238,218] = i_55
        struct[0].Gy_ini[238,238] = v_55
        struct[0].Gy_ini[239,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[239,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[239,196] = v_13/3
        struct[0].Gy_ini[239,239] = 1
        struct[0].Gy_ini[240,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[240,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[240,196] = v_13/3
        struct[0].Gy_ini[240,240] = 1
        struct[0].Gy_ini[241,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[241,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy_ini[241,196] = v_13/3
        struct[0].Gy_ini[241,241] = 1
        struct[0].Gy_ini[242,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[242,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[242,201] = v_23/3
        struct[0].Gy_ini[242,242] = 1
        struct[0].Gy_ini[243,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[243,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[243,201] = v_23/3
        struct[0].Gy_ini[243,243] = 1
        struct[0].Gy_ini[244,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[244,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy_ini[244,201] = v_23/3
        struct[0].Gy_ini[244,244] = 1
        struct[0].Gy_ini[245,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[245,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[245,206] = v_33/3
        struct[0].Gy_ini[245,245] = 1
        struct[0].Gy_ini[246,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[246,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[246,206] = v_33/3
        struct[0].Gy_ini[246,246] = 1
        struct[0].Gy_ini[247,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[247,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy_ini[247,206] = v_33/3
        struct[0].Gy_ini[247,247] = 1
        struct[0].Gy_ini[248,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[248,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[248,211] = v_43/3
        struct[0].Gy_ini[248,248] = 1
        struct[0].Gy_ini[249,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[249,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[249,211] = v_43/3
        struct[0].Gy_ini[249,249] = 1
        struct[0].Gy_ini[250,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[250,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy_ini[250,211] = v_43/3
        struct[0].Gy_ini[250,250] = 1
        struct[0].Gy_ini[251,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[251,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[251,216] = v_53/3
        struct[0].Gy_ini[251,251] = 1
        struct[0].Gy_ini[252,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[252,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[252,216] = v_53/3
        struct[0].Gy_ini[252,252] = 1
        struct[0].Gy_ini[253,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[253,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy_ini[253,216] = v_53/3
        struct[0].Gy_ini[253,253] = 1



def run_nn(t,struct,mode):

    # Parameters:
    R_1112 = struct[0].R_1112
    R_1213 = struct[0].R_1213
    R_1314 = struct[0].R_1314
    R_1415 = struct[0].R_1415
    R_1521 = struct[0].R_1521
    R_2122 = struct[0].R_2122
    R_2223 = struct[0].R_2223
    R_2324 = struct[0].R_2324
    R_2425 = struct[0].R_2425
    R_2531 = struct[0].R_2531
    R_3132 = struct[0].R_3132
    R_3233 = struct[0].R_3233
    R_3334 = struct[0].R_3334
    R_3435 = struct[0].R_3435
    R_3541 = struct[0].R_3541
    R_4142 = struct[0].R_4142
    R_4243 = struct[0].R_4243
    R_4344 = struct[0].R_4344
    R_4445 = struct[0].R_4445
    R_4551 = struct[0].R_4551
    R_5152 = struct[0].R_5152
    R_5253 = struct[0].R_5253
    R_5354 = struct[0].R_5354
    R_5455 = struct[0].R_5455
    p_11 = struct[0].p_11
    p_12 = struct[0].p_12
    p_14 = struct[0].p_14
    p_15 = struct[0].p_15
    p_21 = struct[0].p_21
    p_22 = struct[0].p_22
    p_24 = struct[0].p_24
    p_25 = struct[0].p_25
    p_31 = struct[0].p_31
    p_32 = struct[0].p_32
    p_34 = struct[0].p_34
    p_35 = struct[0].p_35
    p_41 = struct[0].p_41
    p_42 = struct[0].p_42
    p_44 = struct[0].p_44
    p_45 = struct[0].p_45
    p_51 = struct[0].p_51
    p_52 = struct[0].p_52
    p_54 = struct[0].p_54
    p_55 = struct[0].p_55
    
    # Inputs:
    v_B1_a_r = struct[0].v_B1_a_r
    v_B1_a_i = struct[0].v_B1_a_i
    v_B1_b_r = struct[0].v_B1_b_r
    v_B1_b_i = struct[0].v_B1_b_i
    v_B1_c_r = struct[0].v_B1_c_r
    v_B1_c_i = struct[0].v_B1_c_i
    v_B7_a_r = struct[0].v_B7_a_r
    v_B7_a_i = struct[0].v_B7_a_i
    v_B7_b_r = struct[0].v_B7_b_r
    v_B7_b_i = struct[0].v_B7_b_i
    v_B7_c_r = struct[0].v_B7_c_r
    v_B7_c_i = struct[0].v_B7_c_i
    i_B2lv_n_r = struct[0].i_B2lv_n_r
    i_B2lv_n_i = struct[0].i_B2lv_n_i
    i_B3lv_n_r = struct[0].i_B3lv_n_r
    i_B3lv_n_i = struct[0].i_B3lv_n_i
    i_B4lv_n_r = struct[0].i_B4lv_n_r
    i_B4lv_n_i = struct[0].i_B4lv_n_i
    i_B5lv_n_r = struct[0].i_B5lv_n_r
    i_B5lv_n_i = struct[0].i_B5lv_n_i
    i_B6lv_n_r = struct[0].i_B6lv_n_r
    i_B6lv_n_i = struct[0].i_B6lv_n_i
    i_B2_a_r = struct[0].i_B2_a_r
    i_B2_a_i = struct[0].i_B2_a_i
    i_B2_b_r = struct[0].i_B2_b_r
    i_B2_b_i = struct[0].i_B2_b_i
    i_B2_c_r = struct[0].i_B2_c_r
    i_B2_c_i = struct[0].i_B2_c_i
    i_B3_a_r = struct[0].i_B3_a_r
    i_B3_a_i = struct[0].i_B3_a_i
    i_B3_b_r = struct[0].i_B3_b_r
    i_B3_b_i = struct[0].i_B3_b_i
    i_B3_c_r = struct[0].i_B3_c_r
    i_B3_c_i = struct[0].i_B3_c_i
    i_B4_a_r = struct[0].i_B4_a_r
    i_B4_a_i = struct[0].i_B4_a_i
    i_B4_b_r = struct[0].i_B4_b_r
    i_B4_b_i = struct[0].i_B4_b_i
    i_B4_c_r = struct[0].i_B4_c_r
    i_B4_c_i = struct[0].i_B4_c_i
    i_B5_a_r = struct[0].i_B5_a_r
    i_B5_a_i = struct[0].i_B5_a_i
    i_B5_b_r = struct[0].i_B5_b_r
    i_B5_b_i = struct[0].i_B5_b_i
    i_B5_c_r = struct[0].i_B5_c_r
    i_B5_c_i = struct[0].i_B5_c_i
    i_B6_a_r = struct[0].i_B6_a_r
    i_B6_a_i = struct[0].i_B6_a_i
    i_B6_b_r = struct[0].i_B6_b_r
    i_B6_b_i = struct[0].i_B6_b_i
    i_B6_c_r = struct[0].i_B6_c_r
    i_B6_c_i = struct[0].i_B6_c_i
    q_B2lv_a = struct[0].q_B2lv_a
    q_B2lv_b = struct[0].q_B2lv_b
    q_B2lv_c = struct[0].q_B2lv_c
    q_B3lv_a = struct[0].q_B3lv_a
    q_B3lv_b = struct[0].q_B3lv_b
    q_B3lv_c = struct[0].q_B3lv_c
    q_B4lv_a = struct[0].q_B4lv_a
    q_B4lv_b = struct[0].q_B4lv_b
    q_B4lv_c = struct[0].q_B4lv_c
    q_B5lv_a = struct[0].q_B5lv_a
    q_B5lv_b = struct[0].q_B5lv_b
    q_B5lv_c = struct[0].q_B5lv_c
    q_B6lv_a = struct[0].q_B6lv_a
    q_B6lv_b = struct[0].q_B6lv_b
    q_B6lv_c = struct[0].q_B6lv_c
    u_dummy = struct[0].u_dummy
    Dv_r_13 = struct[0].Dv_r_13
    Dv_r_23 = struct[0].Dv_r_23
    Dv_r_33 = struct[0].Dv_r_33
    Dv_r_43 = struct[0].Dv_r_43
    Dv_r_53 = struct[0].Dv_r_53
    v_nom = struct[0].v_nom
    T_v = struct[0].T_v
    K_r = struct[0].K_r
    a_B2lv = struct[0].a_B2lv
    b_B2lv = struct[0].b_B2lv
    c_B2lv = struct[0].c_B2lv
    a_B3lv = struct[0].a_B3lv
    b_B3lv = struct[0].b_B3lv
    c_B3lv = struct[0].c_B3lv
    a_B4lv = struct[0].a_B4lv
    b_B4lv = struct[0].b_B4lv
    c_B4lv = struct[0].c_B4lv
    a_B5lv = struct[0].a_B5lv
    b_B5lv = struct[0].b_B5lv
    c_B5lv = struct[0].c_B5lv
    a_B6lv = struct[0].a_B6lv
    b_B6lv = struct[0].b_B6lv
    c_B6lv = struct[0].c_B6lv
    
    # Dynamical states:
    v_13 = struct[0].x[0,0]
    v_23 = struct[0].x[1,0]
    v_33 = struct[0].x[2,0]
    v_43 = struct[0].x[3,0]
    v_53 = struct[0].x[4,0]
    
    # Algebraic states:
    v_B2lv_a_r = struct[0].y_run[0,0]
    v_B2lv_a_i = struct[0].y_run[1,0]
    v_B2lv_b_r = struct[0].y_run[2,0]
    v_B2lv_b_i = struct[0].y_run[3,0]
    v_B2lv_c_r = struct[0].y_run[4,0]
    v_B2lv_c_i = struct[0].y_run[5,0]
    v_B2lv_n_r = struct[0].y_run[6,0]
    v_B2lv_n_i = struct[0].y_run[7,0]
    v_B3lv_a_r = struct[0].y_run[8,0]
    v_B3lv_a_i = struct[0].y_run[9,0]
    v_B3lv_b_r = struct[0].y_run[10,0]
    v_B3lv_b_i = struct[0].y_run[11,0]
    v_B3lv_c_r = struct[0].y_run[12,0]
    v_B3lv_c_i = struct[0].y_run[13,0]
    v_B3lv_n_r = struct[0].y_run[14,0]
    v_B3lv_n_i = struct[0].y_run[15,0]
    v_B4lv_a_r = struct[0].y_run[16,0]
    v_B4lv_a_i = struct[0].y_run[17,0]
    v_B4lv_b_r = struct[0].y_run[18,0]
    v_B4lv_b_i = struct[0].y_run[19,0]
    v_B4lv_c_r = struct[0].y_run[20,0]
    v_B4lv_c_i = struct[0].y_run[21,0]
    v_B4lv_n_r = struct[0].y_run[22,0]
    v_B4lv_n_i = struct[0].y_run[23,0]
    v_B5lv_a_r = struct[0].y_run[24,0]
    v_B5lv_a_i = struct[0].y_run[25,0]
    v_B5lv_b_r = struct[0].y_run[26,0]
    v_B5lv_b_i = struct[0].y_run[27,0]
    v_B5lv_c_r = struct[0].y_run[28,0]
    v_B5lv_c_i = struct[0].y_run[29,0]
    v_B5lv_n_r = struct[0].y_run[30,0]
    v_B5lv_n_i = struct[0].y_run[31,0]
    v_B6lv_a_r = struct[0].y_run[32,0]
    v_B6lv_a_i = struct[0].y_run[33,0]
    v_B6lv_b_r = struct[0].y_run[34,0]
    v_B6lv_b_i = struct[0].y_run[35,0]
    v_B6lv_c_r = struct[0].y_run[36,0]
    v_B6lv_c_i = struct[0].y_run[37,0]
    v_B6lv_n_r = struct[0].y_run[38,0]
    v_B6lv_n_i = struct[0].y_run[39,0]
    v_B2_a_r = struct[0].y_run[40,0]
    v_B2_a_i = struct[0].y_run[41,0]
    v_B2_b_r = struct[0].y_run[42,0]
    v_B2_b_i = struct[0].y_run[43,0]
    v_B2_c_r = struct[0].y_run[44,0]
    v_B2_c_i = struct[0].y_run[45,0]
    v_B3_a_r = struct[0].y_run[46,0]
    v_B3_a_i = struct[0].y_run[47,0]
    v_B3_b_r = struct[0].y_run[48,0]
    v_B3_b_i = struct[0].y_run[49,0]
    v_B3_c_r = struct[0].y_run[50,0]
    v_B3_c_i = struct[0].y_run[51,0]
    v_B4_a_r = struct[0].y_run[52,0]
    v_B4_a_i = struct[0].y_run[53,0]
    v_B4_b_r = struct[0].y_run[54,0]
    v_B4_b_i = struct[0].y_run[55,0]
    v_B4_c_r = struct[0].y_run[56,0]
    v_B4_c_i = struct[0].y_run[57,0]
    v_B5_a_r = struct[0].y_run[58,0]
    v_B5_a_i = struct[0].y_run[59,0]
    v_B5_b_r = struct[0].y_run[60,0]
    v_B5_b_i = struct[0].y_run[61,0]
    v_B5_c_r = struct[0].y_run[62,0]
    v_B5_c_i = struct[0].y_run[63,0]
    v_B6_a_r = struct[0].y_run[64,0]
    v_B6_a_i = struct[0].y_run[65,0]
    v_B6_b_r = struct[0].y_run[66,0]
    v_B6_b_i = struct[0].y_run[67,0]
    v_B6_c_r = struct[0].y_run[68,0]
    v_B6_c_i = struct[0].y_run[69,0]
    i_t_B2_B2lv_a_r = struct[0].y_run[70,0]
    i_t_B2_B2lv_a_i = struct[0].y_run[71,0]
    i_t_B2_B2lv_b_r = struct[0].y_run[72,0]
    i_t_B2_B2lv_b_i = struct[0].y_run[73,0]
    i_t_B2_B2lv_c_r = struct[0].y_run[74,0]
    i_t_B2_B2lv_c_i = struct[0].y_run[75,0]
    i_t_B3_B3lv_a_r = struct[0].y_run[76,0]
    i_t_B3_B3lv_a_i = struct[0].y_run[77,0]
    i_t_B3_B3lv_b_r = struct[0].y_run[78,0]
    i_t_B3_B3lv_b_i = struct[0].y_run[79,0]
    i_t_B3_B3lv_c_r = struct[0].y_run[80,0]
    i_t_B3_B3lv_c_i = struct[0].y_run[81,0]
    i_t_B4_B4lv_a_r = struct[0].y_run[82,0]
    i_t_B4_B4lv_a_i = struct[0].y_run[83,0]
    i_t_B4_B4lv_b_r = struct[0].y_run[84,0]
    i_t_B4_B4lv_b_i = struct[0].y_run[85,0]
    i_t_B4_B4lv_c_r = struct[0].y_run[86,0]
    i_t_B4_B4lv_c_i = struct[0].y_run[87,0]
    i_t_B5_B5lv_a_r = struct[0].y_run[88,0]
    i_t_B5_B5lv_a_i = struct[0].y_run[89,0]
    i_t_B5_B5lv_b_r = struct[0].y_run[90,0]
    i_t_B5_B5lv_b_i = struct[0].y_run[91,0]
    i_t_B5_B5lv_c_r = struct[0].y_run[92,0]
    i_t_B5_B5lv_c_i = struct[0].y_run[93,0]
    i_t_B6_B6lv_a_r = struct[0].y_run[94,0]
    i_t_B6_B6lv_a_i = struct[0].y_run[95,0]
    i_t_B6_B6lv_b_r = struct[0].y_run[96,0]
    i_t_B6_B6lv_b_i = struct[0].y_run[97,0]
    i_t_B6_B6lv_c_r = struct[0].y_run[98,0]
    i_t_B6_B6lv_c_i = struct[0].y_run[99,0]
    i_l_B1_B2_a_r = struct[0].y_run[100,0]
    i_l_B1_B2_a_i = struct[0].y_run[101,0]
    i_l_B1_B2_b_r = struct[0].y_run[102,0]
    i_l_B1_B2_b_i = struct[0].y_run[103,0]
    i_l_B1_B2_c_r = struct[0].y_run[104,0]
    i_l_B1_B2_c_i = struct[0].y_run[105,0]
    i_l_B2_B3_a_r = struct[0].y_run[106,0]
    i_l_B2_B3_a_i = struct[0].y_run[107,0]
    i_l_B2_B3_b_r = struct[0].y_run[108,0]
    i_l_B2_B3_b_i = struct[0].y_run[109,0]
    i_l_B2_B3_c_r = struct[0].y_run[110,0]
    i_l_B2_B3_c_i = struct[0].y_run[111,0]
    i_l_B3_B4_a_r = struct[0].y_run[112,0]
    i_l_B3_B4_a_i = struct[0].y_run[113,0]
    i_l_B3_B4_b_r = struct[0].y_run[114,0]
    i_l_B3_B4_b_i = struct[0].y_run[115,0]
    i_l_B3_B4_c_r = struct[0].y_run[116,0]
    i_l_B3_B4_c_i = struct[0].y_run[117,0]
    i_l_B5_B6_a_r = struct[0].y_run[118,0]
    i_l_B5_B6_a_i = struct[0].y_run[119,0]
    i_l_B5_B6_b_r = struct[0].y_run[120,0]
    i_l_B5_B6_b_i = struct[0].y_run[121,0]
    i_l_B5_B6_c_r = struct[0].y_run[122,0]
    i_l_B5_B6_c_i = struct[0].y_run[123,0]
    i_l_B6_B7_a_r = struct[0].y_run[124,0]
    i_l_B6_B7_a_i = struct[0].y_run[125,0]
    i_l_B6_B7_b_r = struct[0].y_run[126,0]
    i_l_B6_B7_b_i = struct[0].y_run[127,0]
    i_l_B6_B7_c_r = struct[0].y_run[128,0]
    i_l_B6_B7_c_i = struct[0].y_run[129,0]
    i_load_B2lv_a_r = struct[0].y_run[130,0]
    i_load_B2lv_a_i = struct[0].y_run[131,0]
    i_load_B2lv_b_r = struct[0].y_run[132,0]
    i_load_B2lv_b_i = struct[0].y_run[133,0]
    i_load_B2lv_c_r = struct[0].y_run[134,0]
    i_load_B2lv_c_i = struct[0].y_run[135,0]
    i_load_B2lv_n_r = struct[0].y_run[136,0]
    i_load_B2lv_n_i = struct[0].y_run[137,0]
    i_load_B3lv_a_r = struct[0].y_run[138,0]
    i_load_B3lv_a_i = struct[0].y_run[139,0]
    i_load_B3lv_b_r = struct[0].y_run[140,0]
    i_load_B3lv_b_i = struct[0].y_run[141,0]
    i_load_B3lv_c_r = struct[0].y_run[142,0]
    i_load_B3lv_c_i = struct[0].y_run[143,0]
    i_load_B3lv_n_r = struct[0].y_run[144,0]
    i_load_B3lv_n_i = struct[0].y_run[145,0]
    i_load_B4lv_a_r = struct[0].y_run[146,0]
    i_load_B4lv_a_i = struct[0].y_run[147,0]
    i_load_B4lv_b_r = struct[0].y_run[148,0]
    i_load_B4lv_b_i = struct[0].y_run[149,0]
    i_load_B4lv_c_r = struct[0].y_run[150,0]
    i_load_B4lv_c_i = struct[0].y_run[151,0]
    i_load_B4lv_n_r = struct[0].y_run[152,0]
    i_load_B4lv_n_i = struct[0].y_run[153,0]
    i_load_B5lv_a_r = struct[0].y_run[154,0]
    i_load_B5lv_a_i = struct[0].y_run[155,0]
    i_load_B5lv_b_r = struct[0].y_run[156,0]
    i_load_B5lv_b_i = struct[0].y_run[157,0]
    i_load_B5lv_c_r = struct[0].y_run[158,0]
    i_load_B5lv_c_i = struct[0].y_run[159,0]
    i_load_B5lv_n_r = struct[0].y_run[160,0]
    i_load_B5lv_n_i = struct[0].y_run[161,0]
    i_load_B6lv_a_r = struct[0].y_run[162,0]
    i_load_B6lv_a_i = struct[0].y_run[163,0]
    i_load_B6lv_b_r = struct[0].y_run[164,0]
    i_load_B6lv_b_i = struct[0].y_run[165,0]
    i_load_B6lv_c_r = struct[0].y_run[166,0]
    i_load_B6lv_c_i = struct[0].y_run[167,0]
    i_load_B6lv_n_r = struct[0].y_run[168,0]
    i_load_B6lv_n_i = struct[0].y_run[169,0]
    i_l_1112 = struct[0].y_run[170,0]
    i_l_1213 = struct[0].y_run[171,0]
    i_l_1314 = struct[0].y_run[172,0]
    i_l_1415 = struct[0].y_run[173,0]
    i_l_2122 = struct[0].y_run[174,0]
    i_l_2223 = struct[0].y_run[175,0]
    i_l_2324 = struct[0].y_run[176,0]
    i_l_2425 = struct[0].y_run[177,0]
    i_l_3132 = struct[0].y_run[178,0]
    i_l_3233 = struct[0].y_run[179,0]
    i_l_3334 = struct[0].y_run[180,0]
    i_l_3435 = struct[0].y_run[181,0]
    i_l_4142 = struct[0].y_run[182,0]
    i_l_4243 = struct[0].y_run[183,0]
    i_l_4344 = struct[0].y_run[184,0]
    i_l_4445 = struct[0].y_run[185,0]
    i_l_5152 = struct[0].y_run[186,0]
    i_l_5253 = struct[0].y_run[187,0]
    i_l_5354 = struct[0].y_run[188,0]
    i_l_5455 = struct[0].y_run[189,0]
    i_l_1521 = struct[0].y_run[190,0]
    i_l_2531 = struct[0].y_run[191,0]
    i_l_3541 = struct[0].y_run[192,0]
    i_l_4551 = struct[0].y_run[193,0]
    v_11 = struct[0].y_run[194,0]
    v_12 = struct[0].y_run[195,0]
    i_13 = struct[0].y_run[196,0]
    v_14 = struct[0].y_run[197,0]
    v_15 = struct[0].y_run[198,0]
    v_21 = struct[0].y_run[199,0]
    v_22 = struct[0].y_run[200,0]
    i_23 = struct[0].y_run[201,0]
    v_24 = struct[0].y_run[202,0]
    v_25 = struct[0].y_run[203,0]
    v_31 = struct[0].y_run[204,0]
    v_32 = struct[0].y_run[205,0]
    i_33 = struct[0].y_run[206,0]
    v_34 = struct[0].y_run[207,0]
    v_35 = struct[0].y_run[208,0]
    v_41 = struct[0].y_run[209,0]
    v_42 = struct[0].y_run[210,0]
    i_43 = struct[0].y_run[211,0]
    v_44 = struct[0].y_run[212,0]
    v_45 = struct[0].y_run[213,0]
    v_51 = struct[0].y_run[214,0]
    v_52 = struct[0].y_run[215,0]
    i_53 = struct[0].y_run[216,0]
    v_54 = struct[0].y_run[217,0]
    v_55 = struct[0].y_run[218,0]
    i_11 = struct[0].y_run[219,0]
    i_12 = struct[0].y_run[220,0]
    i_14 = struct[0].y_run[221,0]
    i_15 = struct[0].y_run[222,0]
    i_21 = struct[0].y_run[223,0]
    i_22 = struct[0].y_run[224,0]
    i_24 = struct[0].y_run[225,0]
    i_25 = struct[0].y_run[226,0]
    i_31 = struct[0].y_run[227,0]
    i_32 = struct[0].y_run[228,0]
    i_34 = struct[0].y_run[229,0]
    i_35 = struct[0].y_run[230,0]
    i_41 = struct[0].y_run[231,0]
    i_42 = struct[0].y_run[232,0]
    i_44 = struct[0].y_run[233,0]
    i_45 = struct[0].y_run[234,0]
    i_51 = struct[0].y_run[235,0]
    i_52 = struct[0].y_run[236,0]
    i_54 = struct[0].y_run[237,0]
    i_55 = struct[0].y_run[238,0]
    p_B2lv_a = struct[0].y_run[239,0]
    p_B2lv_b = struct[0].y_run[240,0]
    p_B2lv_c = struct[0].y_run[241,0]
    p_B3lv_a = struct[0].y_run[242,0]
    p_B3lv_b = struct[0].y_run[243,0]
    p_B3lv_c = struct[0].y_run[244,0]
    p_B4lv_a = struct[0].y_run[245,0]
    p_B4lv_b = struct[0].y_run[246,0]
    p_B4lv_c = struct[0].y_run[247,0]
    p_B5lv_a = struct[0].y_run[248,0]
    p_B5lv_b = struct[0].y_run[249,0]
    p_B5lv_c = struct[0].y_run[250,0]
    p_B6lv_a = struct[0].y_run[251,0]
    p_B6lv_b = struct[0].y_run[252,0]
    p_B6lv_c = struct[0].y_run[253,0]
    
    # Differential equations:
    if mode == 2:


        struct[0].f[0,0] = (-Dv_r_13 - K_r*i_13*v_13 - v_13 + v_nom)/T_v
        struct[0].f[1,0] = (-Dv_r_23 - K_r*i_23*v_23 - v_23 + v_nom)/T_v
        struct[0].f[2,0] = (-Dv_r_33 - K_r*i_33*v_33 - v_33 + v_nom)/T_v
        struct[0].f[3,0] = (-Dv_r_43 - K_r*i_43*v_43 - v_43 + v_nom)/T_v
        struct[0].f[4,0] = (-Dv_r_53 - K_r*i_53*v_53 - v_53 + v_nom)/T_v
    
    # Algebraic equations:
    if mode == 3:


        struct[0].g[0,0] = i_load_B2lv_a_r + 0.996212229189942*v_B2_a_i + 0.249053057297486*v_B2_a_r - 0.996212229189942*v_B2_c_i - 0.249053057297486*v_B2_c_r - 23.0065359477124*v_B2lv_a_i - 5.7516339869281*v_B2lv_a_r + 23.0065359477124*v_B2lv_n_i + 5.7516339869281*v_B2lv_n_r
        struct[0].g[1,0] = i_load_B2lv_a_i + 0.249053057297486*v_B2_a_i - 0.996212229189942*v_B2_a_r - 0.249053057297486*v_B2_c_i + 0.996212229189942*v_B2_c_r - 5.7516339869281*v_B2lv_a_i + 23.0065359477124*v_B2lv_a_r + 5.7516339869281*v_B2lv_n_i - 23.0065359477124*v_B2lv_n_r
        struct[0].g[2,0] = i_load_B2lv_b_r - 0.996212229189942*v_B2_a_i - 0.249053057297486*v_B2_a_r + 0.996212229189942*v_B2_b_i + 0.249053057297486*v_B2_b_r - 23.0065359477124*v_B2lv_b_i - 5.7516339869281*v_B2lv_b_r + 23.0065359477124*v_B2lv_n_i + 5.7516339869281*v_B2lv_n_r
        struct[0].g[3,0] = i_load_B2lv_b_i - 0.249053057297486*v_B2_a_i + 0.996212229189942*v_B2_a_r + 0.249053057297486*v_B2_b_i - 0.996212229189942*v_B2_b_r - 5.7516339869281*v_B2lv_b_i + 23.0065359477124*v_B2lv_b_r + 5.7516339869281*v_B2lv_n_i - 23.0065359477124*v_B2lv_n_r
        struct[0].g[4,0] = i_load_B2lv_c_r - 0.996212229189942*v_B2_b_i - 0.249053057297486*v_B2_b_r + 0.996212229189942*v_B2_c_i + 0.249053057297486*v_B2_c_r - 23.0065359477124*v_B2lv_c_i - 5.7516339869281*v_B2lv_c_r + 23.0065359477124*v_B2lv_n_i + 5.7516339869281*v_B2lv_n_r
        struct[0].g[5,0] = i_load_B2lv_c_i - 0.249053057297486*v_B2_b_i + 0.996212229189942*v_B2_b_r + 0.249053057297486*v_B2_c_i - 0.996212229189942*v_B2_c_r - 5.7516339869281*v_B2lv_c_i + 23.0065359477124*v_B2lv_c_r + 5.7516339869281*v_B2lv_n_i - 23.0065359477124*v_B2lv_n_r
        struct[0].g[6,0] = 23.0065359477124*v_B2lv_a_i + 5.7516339869281*v_B2lv_a_r + 23.0065359477124*v_B2lv_b_i + 5.7516339869281*v_B2lv_b_r + 23.0065359477124*v_B2lv_c_i + 5.7516339869281*v_B2lv_c_r - 69.0196078431372*v_B2lv_n_i - 1017.25490196078*v_B2lv_n_r
        struct[0].g[7,0] = 5.7516339869281*v_B2lv_a_i - 23.0065359477124*v_B2lv_a_r + 5.7516339869281*v_B2lv_b_i - 23.0065359477124*v_B2lv_b_r + 5.7516339869281*v_B2lv_c_i - 23.0065359477124*v_B2lv_c_r - 1017.25490196078*v_B2lv_n_i + 69.0196078431372*v_B2lv_n_r
        struct[0].g[8,0] = i_load_B3lv_a_r + 0.996212229189942*v_B3_a_i + 0.249053057297486*v_B3_a_r - 0.996212229189942*v_B3_c_i - 0.249053057297486*v_B3_c_r - 23.0065359477124*v_B3lv_a_i - 5.7516339869281*v_B3lv_a_r + 23.0065359477124*v_B3lv_n_i + 5.7516339869281*v_B3lv_n_r
        struct[0].g[9,0] = i_load_B3lv_a_i + 0.249053057297486*v_B3_a_i - 0.996212229189942*v_B3_a_r - 0.249053057297486*v_B3_c_i + 0.996212229189942*v_B3_c_r - 5.7516339869281*v_B3lv_a_i + 23.0065359477124*v_B3lv_a_r + 5.7516339869281*v_B3lv_n_i - 23.0065359477124*v_B3lv_n_r
        struct[0].g[10,0] = i_load_B3lv_b_r - 0.996212229189942*v_B3_a_i - 0.249053057297486*v_B3_a_r + 0.996212229189942*v_B3_b_i + 0.249053057297486*v_B3_b_r - 23.0065359477124*v_B3lv_b_i - 5.7516339869281*v_B3lv_b_r + 23.0065359477124*v_B3lv_n_i + 5.7516339869281*v_B3lv_n_r
        struct[0].g[11,0] = i_load_B3lv_b_i - 0.249053057297486*v_B3_a_i + 0.996212229189942*v_B3_a_r + 0.249053057297486*v_B3_b_i - 0.996212229189942*v_B3_b_r - 5.7516339869281*v_B3lv_b_i + 23.0065359477124*v_B3lv_b_r + 5.7516339869281*v_B3lv_n_i - 23.0065359477124*v_B3lv_n_r
        struct[0].g[12,0] = i_load_B3lv_c_r - 0.996212229189942*v_B3_b_i - 0.249053057297486*v_B3_b_r + 0.996212229189942*v_B3_c_i + 0.249053057297486*v_B3_c_r - 23.0065359477124*v_B3lv_c_i - 5.7516339869281*v_B3lv_c_r + 23.0065359477124*v_B3lv_n_i + 5.7516339869281*v_B3lv_n_r
        struct[0].g[13,0] = i_load_B3lv_c_i - 0.249053057297486*v_B3_b_i + 0.996212229189942*v_B3_b_r + 0.249053057297486*v_B3_c_i - 0.996212229189942*v_B3_c_r - 5.7516339869281*v_B3lv_c_i + 23.0065359477124*v_B3lv_c_r + 5.7516339869281*v_B3lv_n_i - 23.0065359477124*v_B3lv_n_r
        struct[0].g[14,0] = 23.0065359477124*v_B3lv_a_i + 5.7516339869281*v_B3lv_a_r + 23.0065359477124*v_B3lv_b_i + 5.7516339869281*v_B3lv_b_r + 23.0065359477124*v_B3lv_c_i + 5.7516339869281*v_B3lv_c_r - 69.0196078431372*v_B3lv_n_i - 1017.25490196078*v_B3lv_n_r
        struct[0].g[15,0] = 5.7516339869281*v_B3lv_a_i - 23.0065359477124*v_B3lv_a_r + 5.7516339869281*v_B3lv_b_i - 23.0065359477124*v_B3lv_b_r + 5.7516339869281*v_B3lv_c_i - 23.0065359477124*v_B3lv_c_r - 1017.25490196078*v_B3lv_n_i + 69.0196078431372*v_B3lv_n_r
        struct[0].g[16,0] = i_load_B4lv_a_r + 0.996212229189942*v_B4_a_i + 0.249053057297486*v_B4_a_r - 0.996212229189942*v_B4_c_i - 0.249053057297486*v_B4_c_r - 23.0065359477124*v_B4lv_a_i - 5.7516339869281*v_B4lv_a_r + 23.0065359477124*v_B4lv_n_i + 5.7516339869281*v_B4lv_n_r
        struct[0].g[17,0] = i_load_B4lv_a_i + 0.249053057297486*v_B4_a_i - 0.996212229189942*v_B4_a_r - 0.249053057297486*v_B4_c_i + 0.996212229189942*v_B4_c_r - 5.7516339869281*v_B4lv_a_i + 23.0065359477124*v_B4lv_a_r + 5.7516339869281*v_B4lv_n_i - 23.0065359477124*v_B4lv_n_r
        struct[0].g[18,0] = i_load_B4lv_b_r - 0.996212229189942*v_B4_a_i - 0.249053057297486*v_B4_a_r + 0.996212229189942*v_B4_b_i + 0.249053057297486*v_B4_b_r - 23.0065359477124*v_B4lv_b_i - 5.7516339869281*v_B4lv_b_r + 23.0065359477124*v_B4lv_n_i + 5.7516339869281*v_B4lv_n_r
        struct[0].g[19,0] = i_load_B4lv_b_i - 0.249053057297486*v_B4_a_i + 0.996212229189942*v_B4_a_r + 0.249053057297486*v_B4_b_i - 0.996212229189942*v_B4_b_r - 5.7516339869281*v_B4lv_b_i + 23.0065359477124*v_B4lv_b_r + 5.7516339869281*v_B4lv_n_i - 23.0065359477124*v_B4lv_n_r
        struct[0].g[20,0] = i_load_B4lv_c_r - 0.996212229189942*v_B4_b_i - 0.249053057297486*v_B4_b_r + 0.996212229189942*v_B4_c_i + 0.249053057297486*v_B4_c_r - 23.0065359477124*v_B4lv_c_i - 5.7516339869281*v_B4lv_c_r + 23.0065359477124*v_B4lv_n_i + 5.7516339869281*v_B4lv_n_r
        struct[0].g[21,0] = i_load_B4lv_c_i - 0.249053057297486*v_B4_b_i + 0.996212229189942*v_B4_b_r + 0.249053057297486*v_B4_c_i - 0.996212229189942*v_B4_c_r - 5.7516339869281*v_B4lv_c_i + 23.0065359477124*v_B4lv_c_r + 5.7516339869281*v_B4lv_n_i - 23.0065359477124*v_B4lv_n_r
        struct[0].g[22,0] = 23.0065359477124*v_B4lv_a_i + 5.7516339869281*v_B4lv_a_r + 23.0065359477124*v_B4lv_b_i + 5.7516339869281*v_B4lv_b_r + 23.0065359477124*v_B4lv_c_i + 5.7516339869281*v_B4lv_c_r - 69.0196078431372*v_B4lv_n_i - 1017.25490196078*v_B4lv_n_r
        struct[0].g[23,0] = 5.7516339869281*v_B4lv_a_i - 23.0065359477124*v_B4lv_a_r + 5.7516339869281*v_B4lv_b_i - 23.0065359477124*v_B4lv_b_r + 5.7516339869281*v_B4lv_c_i - 23.0065359477124*v_B4lv_c_r - 1017.25490196078*v_B4lv_n_i + 69.0196078431372*v_B4lv_n_r
        struct[0].g[24,0] = i_load_B5lv_a_r + 0.996212229189942*v_B5_a_i + 0.249053057297486*v_B5_a_r - 0.996212229189942*v_B5_c_i - 0.249053057297486*v_B5_c_r - 23.0065359477124*v_B5lv_a_i - 5.7516339869281*v_B5lv_a_r + 23.0065359477124*v_B5lv_n_i + 5.7516339869281*v_B5lv_n_r
        struct[0].g[25,0] = i_load_B5lv_a_i + 0.249053057297486*v_B5_a_i - 0.996212229189942*v_B5_a_r - 0.249053057297486*v_B5_c_i + 0.996212229189942*v_B5_c_r - 5.7516339869281*v_B5lv_a_i + 23.0065359477124*v_B5lv_a_r + 5.7516339869281*v_B5lv_n_i - 23.0065359477124*v_B5lv_n_r
        struct[0].g[26,0] = i_load_B5lv_b_r - 0.996212229189942*v_B5_a_i - 0.249053057297486*v_B5_a_r + 0.996212229189942*v_B5_b_i + 0.249053057297486*v_B5_b_r - 23.0065359477124*v_B5lv_b_i - 5.7516339869281*v_B5lv_b_r + 23.0065359477124*v_B5lv_n_i + 5.7516339869281*v_B5lv_n_r
        struct[0].g[27,0] = i_load_B5lv_b_i - 0.249053057297486*v_B5_a_i + 0.996212229189942*v_B5_a_r + 0.249053057297486*v_B5_b_i - 0.996212229189942*v_B5_b_r - 5.7516339869281*v_B5lv_b_i + 23.0065359477124*v_B5lv_b_r + 5.7516339869281*v_B5lv_n_i - 23.0065359477124*v_B5lv_n_r
        struct[0].g[28,0] = i_load_B5lv_c_r - 0.996212229189942*v_B5_b_i - 0.249053057297486*v_B5_b_r + 0.996212229189942*v_B5_c_i + 0.249053057297486*v_B5_c_r - 23.0065359477124*v_B5lv_c_i - 5.7516339869281*v_B5lv_c_r + 23.0065359477124*v_B5lv_n_i + 5.7516339869281*v_B5lv_n_r
        struct[0].g[29,0] = i_load_B5lv_c_i - 0.249053057297486*v_B5_b_i + 0.996212229189942*v_B5_b_r + 0.249053057297486*v_B5_c_i - 0.996212229189942*v_B5_c_r - 5.7516339869281*v_B5lv_c_i + 23.0065359477124*v_B5lv_c_r + 5.7516339869281*v_B5lv_n_i - 23.0065359477124*v_B5lv_n_r
        struct[0].g[30,0] = 23.0065359477124*v_B5lv_a_i + 5.7516339869281*v_B5lv_a_r + 23.0065359477124*v_B5lv_b_i + 5.7516339869281*v_B5lv_b_r + 23.0065359477124*v_B5lv_c_i + 5.7516339869281*v_B5lv_c_r - 69.0196078431372*v_B5lv_n_i - 1017.25490196078*v_B5lv_n_r
        struct[0].g[31,0] = 5.7516339869281*v_B5lv_a_i - 23.0065359477124*v_B5lv_a_r + 5.7516339869281*v_B5lv_b_i - 23.0065359477124*v_B5lv_b_r + 5.7516339869281*v_B5lv_c_i - 23.0065359477124*v_B5lv_c_r - 1017.25490196078*v_B5lv_n_i + 69.0196078431372*v_B5lv_n_r
        struct[0].g[32,0] = i_load_B6lv_a_r + 0.996212229189942*v_B6_a_i + 0.249053057297486*v_B6_a_r - 0.996212229189942*v_B6_c_i - 0.249053057297486*v_B6_c_r - 23.0065359477124*v_B6lv_a_i - 5.7516339869281*v_B6lv_a_r + 23.0065359477124*v_B6lv_n_i + 5.7516339869281*v_B6lv_n_r
        struct[0].g[33,0] = i_load_B6lv_a_i + 0.249053057297486*v_B6_a_i - 0.996212229189942*v_B6_a_r - 0.249053057297486*v_B6_c_i + 0.996212229189942*v_B6_c_r - 5.7516339869281*v_B6lv_a_i + 23.0065359477124*v_B6lv_a_r + 5.7516339869281*v_B6lv_n_i - 23.0065359477124*v_B6lv_n_r
        struct[0].g[34,0] = i_load_B6lv_b_r - 0.996212229189942*v_B6_a_i - 0.249053057297486*v_B6_a_r + 0.996212229189942*v_B6_b_i + 0.249053057297486*v_B6_b_r - 23.0065359477124*v_B6lv_b_i - 5.7516339869281*v_B6lv_b_r + 23.0065359477124*v_B6lv_n_i + 5.7516339869281*v_B6lv_n_r
        struct[0].g[35,0] = i_load_B6lv_b_i - 0.249053057297486*v_B6_a_i + 0.996212229189942*v_B6_a_r + 0.249053057297486*v_B6_b_i - 0.996212229189942*v_B6_b_r - 5.7516339869281*v_B6lv_b_i + 23.0065359477124*v_B6lv_b_r + 5.7516339869281*v_B6lv_n_i - 23.0065359477124*v_B6lv_n_r
        struct[0].g[36,0] = i_load_B6lv_c_r - 0.996212229189942*v_B6_b_i - 0.249053057297486*v_B6_b_r + 0.996212229189942*v_B6_c_i + 0.249053057297486*v_B6_c_r - 23.0065359477124*v_B6lv_c_i - 5.7516339869281*v_B6lv_c_r + 23.0065359477124*v_B6lv_n_i + 5.7516339869281*v_B6lv_n_r
        struct[0].g[37,0] = i_load_B6lv_c_i - 0.249053057297486*v_B6_b_i + 0.996212229189942*v_B6_b_r + 0.249053057297486*v_B6_c_i - 0.996212229189942*v_B6_c_r - 5.7516339869281*v_B6lv_c_i + 23.0065359477124*v_B6lv_c_r + 5.7516339869281*v_B6lv_n_i - 23.0065359477124*v_B6lv_n_r
        struct[0].g[38,0] = 23.0065359477124*v_B6lv_a_i + 5.7516339869281*v_B6lv_a_r + 23.0065359477124*v_B6lv_b_i + 5.7516339869281*v_B6lv_b_r + 23.0065359477124*v_B6lv_c_i + 5.7516339869281*v_B6lv_c_r - 69.0196078431372*v_B6lv_n_i - 1017.25490196078*v_B6lv_n_r
        struct[0].g[39,0] = 5.7516339869281*v_B6lv_a_i - 23.0065359477124*v_B6lv_a_r + 5.7516339869281*v_B6lv_b_i - 23.0065359477124*v_B6lv_b_r + 5.7516339869281*v_B6lv_c_i - 23.0065359477124*v_B6lv_c_r - 1017.25490196078*v_B6lv_n_i + 69.0196078431372*v_B6lv_n_r
        struct[0].g[40,0] = 0.598820527961361*v_B1_a_i + 1.10755301189314*v_B1_a_r - 0.171091579417532*v_B1_b_i - 0.316443717683753*v_B1_b_r - 0.171091579417532*v_B1_c_i - 0.316443717683753*v_B1_c_r - 1.28353302446119*v_B2_a_i - 2.23667465123725*v_B2_a_r + 0.385473430243205*v_B2_b_i + 0.643671749092996*v_B2_b_r + 0.385473430243205*v_B2_c_i + 0.643671749092997*v_B2_c_r + 0.996212229189942*v_B2lv_a_i + 0.249053057297486*v_B2lv_a_r - 0.996212229189942*v_B2lv_b_i - 0.249053057297486*v_B2lv_b_r + 0.598820527961361*v_B3_a_i + 1.10755301189314*v_B3_a_r - 0.171091579417532*v_B3_b_i - 0.316443717683753*v_B3_b_r - 0.171091579417532*v_B3_c_i - 0.316443717683753*v_B3_c_r
        struct[0].g[41,0] = 1.10755301189314*v_B1_a_i - 0.598820527961361*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417532*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417532*v_B1_c_r - 2.23667465123725*v_B2_a_i + 1.28353302446119*v_B2_a_r + 0.643671749092996*v_B2_b_i - 0.385473430243205*v_B2_b_r + 0.643671749092997*v_B2_c_i - 0.385473430243205*v_B2_c_r + 0.249053057297486*v_B2lv_a_i - 0.996212229189942*v_B2lv_a_r - 0.249053057297486*v_B2lv_b_i + 0.996212229189942*v_B2lv_b_r + 1.10755301189314*v_B3_a_i - 0.598820527961361*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417532*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417532*v_B3_c_r
        struct[0].g[42,0] = -0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r + 0.59882052796136*v_B1_b_i + 1.10755301189314*v_B1_b_r - 0.171091579417531*v_B1_c_i - 0.316443717683753*v_B1_c_r + 0.385473430243205*v_B2_a_i + 0.643671749092996*v_B2_a_r - 1.28353302446119*v_B2_b_i - 2.23667465123725*v_B2_b_r + 0.385473430243204*v_B2_c_i + 0.643671749092997*v_B2_c_r + 0.996212229189942*v_B2lv_b_i + 0.249053057297486*v_B2lv_b_r - 0.996212229189942*v_B2lv_c_i - 0.249053057297486*v_B2lv_c_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r + 0.59882052796136*v_B3_b_i + 1.10755301189314*v_B3_b_r - 0.171091579417531*v_B3_c_i - 0.316443717683753*v_B3_c_r
        struct[0].g[43,0] = -0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r + 1.10755301189314*v_B1_b_i - 0.59882052796136*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417531*v_B1_c_r + 0.643671749092996*v_B2_a_i - 0.385473430243205*v_B2_a_r - 2.23667465123725*v_B2_b_i + 1.28353302446119*v_B2_b_r + 0.643671749092997*v_B2_c_i - 0.385473430243204*v_B2_c_r + 0.249053057297486*v_B2lv_b_i - 0.996212229189942*v_B2lv_b_r - 0.249053057297486*v_B2lv_c_i + 0.996212229189942*v_B2lv_c_r - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r + 1.10755301189314*v_B3_b_i - 0.59882052796136*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417531*v_B3_c_r
        struct[0].g[44,0] = -0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r - 0.171091579417531*v_B1_b_i - 0.316443717683753*v_B1_b_r + 0.59882052796136*v_B1_c_i + 1.10755301189314*v_B1_c_r + 0.385473430243205*v_B2_a_i + 0.643671749092997*v_B2_a_r + 0.385473430243204*v_B2_b_i + 0.643671749092997*v_B2_b_r - 1.28353302446119*v_B2_c_i - 2.23667465123725*v_B2_c_r - 0.996212229189942*v_B2lv_a_i - 0.249053057297486*v_B2lv_a_r + 0.996212229189942*v_B2lv_c_i + 0.249053057297486*v_B2lv_c_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r - 0.171091579417531*v_B3_b_i - 0.316443717683753*v_B3_b_r + 0.59882052796136*v_B3_c_i + 1.10755301189314*v_B3_c_r
        struct[0].g[45,0] = -0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417531*v_B1_b_r + 1.10755301189314*v_B1_c_i - 0.59882052796136*v_B1_c_r + 0.643671749092997*v_B2_a_i - 0.385473430243205*v_B2_a_r + 0.643671749092997*v_B2_b_i - 0.385473430243204*v_B2_b_r - 2.23667465123725*v_B2_c_i + 1.28353302446119*v_B2_c_r - 0.249053057297486*v_B2lv_a_i + 0.996212229189942*v_B2lv_a_r + 0.249053057297486*v_B2lv_c_i - 0.996212229189942*v_B2lv_c_r - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417531*v_B3_b_r + 1.10755301189314*v_B3_c_i - 0.59882052796136*v_B3_c_r
        struct[0].g[46,0] = 0.598820527961361*v_B2_a_i + 1.10755301189314*v_B2_a_r - 0.171091579417532*v_B2_b_i - 0.316443717683753*v_B2_b_r - 0.171091579417532*v_B2_c_i - 0.316443717683753*v_B2_c_r - 1.28353302446119*v_B3_a_i - 2.23667465123725*v_B3_a_r + 0.385473430243205*v_B3_b_i + 0.643671749092996*v_B3_b_r + 0.385473430243205*v_B3_c_i + 0.643671749092997*v_B3_c_r + 0.996212229189942*v_B3lv_a_i + 0.249053057297486*v_B3lv_a_r - 0.996212229189942*v_B3lv_b_i - 0.249053057297486*v_B3lv_b_r + 0.598820527961361*v_B4_a_i + 1.10755301189314*v_B4_a_r - 0.171091579417532*v_B4_b_i - 0.316443717683753*v_B4_b_r - 0.171091579417532*v_B4_c_i - 0.316443717683753*v_B4_c_r
        struct[0].g[47,0] = 1.10755301189314*v_B2_a_i - 0.598820527961361*v_B2_a_r - 0.316443717683753*v_B2_b_i + 0.171091579417532*v_B2_b_r - 0.316443717683753*v_B2_c_i + 0.171091579417532*v_B2_c_r - 2.23667465123725*v_B3_a_i + 1.28353302446119*v_B3_a_r + 0.643671749092996*v_B3_b_i - 0.385473430243205*v_B3_b_r + 0.643671749092997*v_B3_c_i - 0.385473430243205*v_B3_c_r + 0.249053057297486*v_B3lv_a_i - 0.996212229189942*v_B3lv_a_r - 0.249053057297486*v_B3lv_b_i + 0.996212229189942*v_B3lv_b_r + 1.10755301189314*v_B4_a_i - 0.598820527961361*v_B4_a_r - 0.316443717683753*v_B4_b_i + 0.171091579417532*v_B4_b_r - 0.316443717683753*v_B4_c_i + 0.171091579417532*v_B4_c_r
        struct[0].g[48,0] = -0.171091579417532*v_B2_a_i - 0.316443717683753*v_B2_a_r + 0.59882052796136*v_B2_b_i + 1.10755301189314*v_B2_b_r - 0.171091579417531*v_B2_c_i - 0.316443717683753*v_B2_c_r + 0.385473430243205*v_B3_a_i + 0.643671749092996*v_B3_a_r - 1.28353302446119*v_B3_b_i - 2.23667465123725*v_B3_b_r + 0.385473430243204*v_B3_c_i + 0.643671749092997*v_B3_c_r + 0.996212229189942*v_B3lv_b_i + 0.249053057297486*v_B3lv_b_r - 0.996212229189942*v_B3lv_c_i - 0.249053057297486*v_B3lv_c_r - 0.171091579417532*v_B4_a_i - 0.316443717683753*v_B4_a_r + 0.59882052796136*v_B4_b_i + 1.10755301189314*v_B4_b_r - 0.171091579417531*v_B4_c_i - 0.316443717683753*v_B4_c_r
        struct[0].g[49,0] = -0.316443717683753*v_B2_a_i + 0.171091579417532*v_B2_a_r + 1.10755301189314*v_B2_b_i - 0.59882052796136*v_B2_b_r - 0.316443717683753*v_B2_c_i + 0.171091579417531*v_B2_c_r + 0.643671749092996*v_B3_a_i - 0.385473430243205*v_B3_a_r - 2.23667465123725*v_B3_b_i + 1.28353302446119*v_B3_b_r + 0.643671749092997*v_B3_c_i - 0.385473430243204*v_B3_c_r + 0.249053057297486*v_B3lv_b_i - 0.996212229189942*v_B3lv_b_r - 0.249053057297486*v_B3lv_c_i + 0.996212229189942*v_B3lv_c_r - 0.316443717683753*v_B4_a_i + 0.171091579417532*v_B4_a_r + 1.10755301189314*v_B4_b_i - 0.59882052796136*v_B4_b_r - 0.316443717683753*v_B4_c_i + 0.171091579417531*v_B4_c_r
        struct[0].g[50,0] = -0.171091579417532*v_B2_a_i - 0.316443717683753*v_B2_a_r - 0.171091579417531*v_B2_b_i - 0.316443717683753*v_B2_b_r + 0.59882052796136*v_B2_c_i + 1.10755301189314*v_B2_c_r + 0.385473430243205*v_B3_a_i + 0.643671749092997*v_B3_a_r + 0.385473430243204*v_B3_b_i + 0.643671749092997*v_B3_b_r - 1.28353302446119*v_B3_c_i - 2.23667465123725*v_B3_c_r - 0.996212229189942*v_B3lv_a_i - 0.249053057297486*v_B3lv_a_r + 0.996212229189942*v_B3lv_c_i + 0.249053057297486*v_B3lv_c_r - 0.171091579417532*v_B4_a_i - 0.316443717683753*v_B4_a_r - 0.171091579417531*v_B4_b_i - 0.316443717683753*v_B4_b_r + 0.59882052796136*v_B4_c_i + 1.10755301189314*v_B4_c_r
        struct[0].g[51,0] = -0.316443717683753*v_B2_a_i + 0.171091579417532*v_B2_a_r - 0.316443717683753*v_B2_b_i + 0.171091579417531*v_B2_b_r + 1.10755301189314*v_B2_c_i - 0.59882052796136*v_B2_c_r + 0.643671749092997*v_B3_a_i - 0.385473430243205*v_B3_a_r + 0.643671749092997*v_B3_b_i - 0.385473430243204*v_B3_b_r - 2.23667465123725*v_B3_c_i + 1.28353302446119*v_B3_c_r - 0.249053057297486*v_B3lv_a_i + 0.996212229189942*v_B3lv_a_r + 0.249053057297486*v_B3lv_c_i - 0.996212229189942*v_B3lv_c_r - 0.316443717683753*v_B4_a_i + 0.171091579417532*v_B4_a_r - 0.316443717683753*v_B4_b_i + 0.171091579417531*v_B4_b_r + 1.10755301189314*v_B4_c_i - 0.59882052796136*v_B4_c_r
        struct[0].g[52,0] = 0.598820527961361*v_B3_a_i + 1.10755301189314*v_B3_a_r - 0.171091579417532*v_B3_b_i - 0.316443717683753*v_B3_b_r - 0.171091579417532*v_B3_c_i - 0.316443717683753*v_B3_c_r - 0.684903767132556*v_B4_a_i - 1.12912163934412*v_B4_a_r + 0.214305342572583*v_B4_b_i + 0.327228031409243*v_B4_b_r + 0.214305342572583*v_B4_c_i + 0.327228031409244*v_B4_c_r + 0.996212229189942*v_B4lv_a_i + 0.249053057297486*v_B4lv_a_r - 0.996212229189942*v_B4lv_b_i - 0.249053057297486*v_B4lv_b_r
        struct[0].g[53,0] = 1.10755301189314*v_B3_a_i - 0.598820527961361*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417532*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417532*v_B3_c_r - 1.12912163934412*v_B4_a_i + 0.684903767132556*v_B4_a_r + 0.327228031409243*v_B4_b_i - 0.214305342572583*v_B4_b_r + 0.327228031409244*v_B4_c_i - 0.214305342572583*v_B4_c_r + 0.249053057297486*v_B4lv_a_i - 0.996212229189942*v_B4lv_a_r - 0.249053057297486*v_B4lv_b_i + 0.996212229189942*v_B4lv_b_r
        struct[0].g[54,0] = -0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r + 0.59882052796136*v_B3_b_i + 1.10755301189314*v_B3_b_r - 0.171091579417531*v_B3_c_i - 0.316443717683753*v_B3_c_r + 0.214305342572583*v_B4_a_i + 0.327228031409243*v_B4_a_r - 0.684903767132556*v_B4_b_i - 1.12912163934412*v_B4_b_r + 0.214305342572582*v_B4_c_i + 0.327228031409244*v_B4_c_r + 0.996212229189942*v_B4lv_b_i + 0.249053057297486*v_B4lv_b_r - 0.996212229189942*v_B4lv_c_i - 0.249053057297486*v_B4lv_c_r
        struct[0].g[55,0] = -0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r + 1.10755301189314*v_B3_b_i - 0.59882052796136*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417531*v_B3_c_r + 0.327228031409243*v_B4_a_i - 0.214305342572583*v_B4_a_r - 1.12912163934412*v_B4_b_i + 0.684903767132556*v_B4_b_r + 0.327228031409244*v_B4_c_i - 0.214305342572582*v_B4_c_r + 0.249053057297486*v_B4lv_b_i - 0.996212229189942*v_B4lv_b_r - 0.249053057297486*v_B4lv_c_i + 0.996212229189942*v_B4lv_c_r
        struct[0].g[56,0] = -0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r - 0.171091579417531*v_B3_b_i - 0.316443717683753*v_B3_b_r + 0.59882052796136*v_B3_c_i + 1.10755301189314*v_B3_c_r + 0.214305342572583*v_B4_a_i + 0.327228031409243*v_B4_a_r + 0.214305342572582*v_B4_b_i + 0.327228031409244*v_B4_b_r - 0.684903767132556*v_B4_c_i - 1.12912163934412*v_B4_c_r - 0.996212229189942*v_B4lv_a_i - 0.249053057297486*v_B4lv_a_r + 0.996212229189942*v_B4lv_c_i + 0.249053057297486*v_B4lv_c_r
        struct[0].g[57,0] = -0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417531*v_B3_b_r + 1.10755301189314*v_B3_c_i - 0.59882052796136*v_B3_c_r + 0.327228031409243*v_B4_a_i - 0.214305342572583*v_B4_a_r + 0.327228031409244*v_B4_b_i - 0.214305342572582*v_B4_b_r - 1.12912163934412*v_B4_c_i + 0.684903767132556*v_B4_c_r - 0.249053057297486*v_B4lv_a_i + 0.996212229189942*v_B4lv_a_r + 0.249053057297486*v_B4lv_c_i - 0.996212229189942*v_B4lv_c_r
        struct[0].g[58,0] = -0.684903767132556*v_B5_a_i - 1.12912163934412*v_B5_a_r + 0.214305342572583*v_B5_b_i + 0.327228031409243*v_B5_b_r + 0.214305342572583*v_B5_c_i + 0.327228031409244*v_B5_c_r + 0.996212229189942*v_B5lv_a_i + 0.249053057297486*v_B5lv_a_r - 0.996212229189942*v_B5lv_b_i - 0.249053057297486*v_B5lv_b_r + 0.598820527961361*v_B6_a_i + 1.10755301189314*v_B6_a_r - 0.171091579417532*v_B6_b_i - 0.316443717683753*v_B6_b_r - 0.171091579417532*v_B6_c_i - 0.316443717683753*v_B6_c_r
        struct[0].g[59,0] = -1.12912163934412*v_B5_a_i + 0.684903767132556*v_B5_a_r + 0.327228031409243*v_B5_b_i - 0.214305342572583*v_B5_b_r + 0.327228031409244*v_B5_c_i - 0.214305342572583*v_B5_c_r + 0.249053057297486*v_B5lv_a_i - 0.996212229189942*v_B5lv_a_r - 0.249053057297486*v_B5lv_b_i + 0.996212229189942*v_B5lv_b_r + 1.10755301189314*v_B6_a_i - 0.598820527961361*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417532*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417532*v_B6_c_r
        struct[0].g[60,0] = 0.214305342572583*v_B5_a_i + 0.327228031409243*v_B5_a_r - 0.684903767132556*v_B5_b_i - 1.12912163934412*v_B5_b_r + 0.214305342572582*v_B5_c_i + 0.327228031409244*v_B5_c_r + 0.996212229189942*v_B5lv_b_i + 0.249053057297486*v_B5lv_b_r - 0.996212229189942*v_B5lv_c_i - 0.249053057297486*v_B5lv_c_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r + 0.59882052796136*v_B6_b_i + 1.10755301189314*v_B6_b_r - 0.171091579417531*v_B6_c_i - 0.316443717683753*v_B6_c_r
        struct[0].g[61,0] = 0.327228031409243*v_B5_a_i - 0.214305342572583*v_B5_a_r - 1.12912163934412*v_B5_b_i + 0.684903767132556*v_B5_b_r + 0.327228031409244*v_B5_c_i - 0.214305342572582*v_B5_c_r + 0.249053057297486*v_B5lv_b_i - 0.996212229189942*v_B5lv_b_r - 0.249053057297486*v_B5lv_c_i + 0.996212229189942*v_B5lv_c_r - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r + 1.10755301189314*v_B6_b_i - 0.59882052796136*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417531*v_B6_c_r
        struct[0].g[62,0] = 0.214305342572583*v_B5_a_i + 0.327228031409243*v_B5_a_r + 0.214305342572582*v_B5_b_i + 0.327228031409244*v_B5_b_r - 0.684903767132556*v_B5_c_i - 1.12912163934412*v_B5_c_r - 0.996212229189942*v_B5lv_a_i - 0.249053057297486*v_B5lv_a_r + 0.996212229189942*v_B5lv_c_i + 0.249053057297486*v_B5lv_c_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r - 0.171091579417531*v_B6_b_i - 0.316443717683753*v_B6_b_r + 0.59882052796136*v_B6_c_i + 1.10755301189314*v_B6_c_r
        struct[0].g[63,0] = 0.327228031409243*v_B5_a_i - 0.214305342572583*v_B5_a_r + 0.327228031409244*v_B5_b_i - 0.214305342572582*v_B5_b_r - 1.12912163934412*v_B5_c_i + 0.684903767132556*v_B5_c_r - 0.249053057297486*v_B5lv_a_i + 0.996212229189942*v_B5lv_a_r + 0.249053057297486*v_B5lv_c_i - 0.996212229189942*v_B5lv_c_r - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417531*v_B6_b_r + 1.10755301189314*v_B6_c_i - 0.59882052796136*v_B6_c_r
        struct[0].g[64,0] = 0.598820527961361*v_B5_a_i + 1.10755301189314*v_B5_a_r - 0.171091579417532*v_B5_b_i - 0.316443717683753*v_B5_b_r - 0.171091579417532*v_B5_c_i - 0.316443717683753*v_B5_c_r - 1.28353302446119*v_B6_a_i - 2.23667465123725*v_B6_a_r + 0.385473430243205*v_B6_b_i + 0.643671749092996*v_B6_b_r + 0.385473430243205*v_B6_c_i + 0.643671749092997*v_B6_c_r + 0.996212229189942*v_B6lv_a_i + 0.249053057297486*v_B6lv_a_r - 0.996212229189942*v_B6lv_b_i - 0.249053057297486*v_B6lv_b_r + 0.598820527961361*v_B7_a_i + 1.10755301189314*v_B7_a_r - 0.171091579417532*v_B7_b_i - 0.316443717683753*v_B7_b_r - 0.171091579417532*v_B7_c_i - 0.316443717683753*v_B7_c_r
        struct[0].g[65,0] = 1.10755301189314*v_B5_a_i - 0.598820527961361*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417532*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417532*v_B5_c_r - 2.23667465123725*v_B6_a_i + 1.28353302446119*v_B6_a_r + 0.643671749092996*v_B6_b_i - 0.385473430243205*v_B6_b_r + 0.643671749092997*v_B6_c_i - 0.385473430243205*v_B6_c_r + 0.249053057297486*v_B6lv_a_i - 0.996212229189942*v_B6lv_a_r - 0.249053057297486*v_B6lv_b_i + 0.996212229189942*v_B6lv_b_r + 1.10755301189314*v_B7_a_i - 0.598820527961361*v_B7_a_r - 0.316443717683753*v_B7_b_i + 0.171091579417532*v_B7_b_r - 0.316443717683753*v_B7_c_i + 0.171091579417532*v_B7_c_r
        struct[0].g[66,0] = -0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r + 0.59882052796136*v_B5_b_i + 1.10755301189314*v_B5_b_r - 0.171091579417531*v_B5_c_i - 0.316443717683753*v_B5_c_r + 0.385473430243205*v_B6_a_i + 0.643671749092996*v_B6_a_r - 1.28353302446119*v_B6_b_i - 2.23667465123725*v_B6_b_r + 0.385473430243204*v_B6_c_i + 0.643671749092997*v_B6_c_r + 0.996212229189942*v_B6lv_b_i + 0.249053057297486*v_B6lv_b_r - 0.996212229189942*v_B6lv_c_i - 0.249053057297486*v_B6lv_c_r - 0.171091579417532*v_B7_a_i - 0.316443717683753*v_B7_a_r + 0.59882052796136*v_B7_b_i + 1.10755301189314*v_B7_b_r - 0.171091579417531*v_B7_c_i - 0.316443717683753*v_B7_c_r
        struct[0].g[67,0] = -0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r + 1.10755301189314*v_B5_b_i - 0.59882052796136*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417531*v_B5_c_r + 0.643671749092996*v_B6_a_i - 0.385473430243205*v_B6_a_r - 2.23667465123725*v_B6_b_i + 1.28353302446119*v_B6_b_r + 0.643671749092997*v_B6_c_i - 0.385473430243204*v_B6_c_r + 0.249053057297486*v_B6lv_b_i - 0.996212229189942*v_B6lv_b_r - 0.249053057297486*v_B6lv_c_i + 0.996212229189942*v_B6lv_c_r - 0.316443717683753*v_B7_a_i + 0.171091579417532*v_B7_a_r + 1.10755301189314*v_B7_b_i - 0.59882052796136*v_B7_b_r - 0.316443717683753*v_B7_c_i + 0.171091579417531*v_B7_c_r
        struct[0].g[68,0] = -0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r - 0.171091579417531*v_B5_b_i - 0.316443717683753*v_B5_b_r + 0.59882052796136*v_B5_c_i + 1.10755301189314*v_B5_c_r + 0.385473430243205*v_B6_a_i + 0.643671749092997*v_B6_a_r + 0.385473430243204*v_B6_b_i + 0.643671749092997*v_B6_b_r - 1.28353302446119*v_B6_c_i - 2.23667465123725*v_B6_c_r - 0.996212229189942*v_B6lv_a_i - 0.249053057297486*v_B6lv_a_r + 0.996212229189942*v_B6lv_c_i + 0.249053057297486*v_B6lv_c_r - 0.171091579417532*v_B7_a_i - 0.316443717683753*v_B7_a_r - 0.171091579417531*v_B7_b_i - 0.316443717683753*v_B7_b_r + 0.59882052796136*v_B7_c_i + 1.10755301189314*v_B7_c_r
        struct[0].g[69,0] = -0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417531*v_B5_b_r + 1.10755301189314*v_B5_c_i - 0.59882052796136*v_B5_c_r + 0.643671749092997*v_B6_a_i - 0.385473430243205*v_B6_a_r + 0.643671749092997*v_B6_b_i - 0.385473430243204*v_B6_b_r - 2.23667465123725*v_B6_c_i + 1.28353302446119*v_B6_c_r - 0.249053057297486*v_B6lv_a_i + 0.996212229189942*v_B6lv_a_r + 0.249053057297486*v_B6lv_c_i - 0.996212229189942*v_B6lv_c_r - 0.316443717683753*v_B7_a_i + 0.171091579417532*v_B7_a_r - 0.316443717683753*v_B7_b_i + 0.171091579417531*v_B7_b_r + 1.10755301189314*v_B7_c_i - 0.59882052796136*v_B7_c_r
        struct[0].g[70,0] = -i_t_B2_B2lv_a_r + 0.0862745098039216*v_B2_a_i + 0.0215686274509804*v_B2_a_r - 0.0431372549019608*v_B2_b_i - 0.0107843137254902*v_B2_b_r - 0.0431372549019608*v_B2_c_i - 0.0107843137254902*v_B2_c_r - 0.996212229189942*v_B2lv_a_i - 0.249053057297486*v_B2lv_a_r + 0.996212229189942*v_B2lv_b_i + 0.249053057297486*v_B2lv_b_r
        struct[0].g[71,0] = -i_t_B2_B2lv_a_i + 0.0215686274509804*v_B2_a_i - 0.0862745098039216*v_B2_a_r - 0.0107843137254902*v_B2_b_i + 0.0431372549019608*v_B2_b_r - 0.0107843137254902*v_B2_c_i + 0.0431372549019608*v_B2_c_r - 0.249053057297486*v_B2lv_a_i + 0.996212229189942*v_B2lv_a_r + 0.249053057297486*v_B2lv_b_i - 0.996212229189942*v_B2lv_b_r
        struct[0].g[72,0] = -i_t_B2_B2lv_b_r - 0.0431372549019608*v_B2_a_i - 0.0107843137254902*v_B2_a_r + 0.0862745098039216*v_B2_b_i + 0.0215686274509804*v_B2_b_r - 0.0431372549019608*v_B2_c_i - 0.0107843137254902*v_B2_c_r - 0.996212229189942*v_B2lv_b_i - 0.249053057297486*v_B2lv_b_r + 0.996212229189942*v_B2lv_c_i + 0.249053057297486*v_B2lv_c_r
        struct[0].g[73,0] = -i_t_B2_B2lv_b_i - 0.0107843137254902*v_B2_a_i + 0.0431372549019608*v_B2_a_r + 0.0215686274509804*v_B2_b_i - 0.0862745098039216*v_B2_b_r - 0.0107843137254902*v_B2_c_i + 0.0431372549019608*v_B2_c_r - 0.249053057297486*v_B2lv_b_i + 0.996212229189942*v_B2lv_b_r + 0.249053057297486*v_B2lv_c_i - 0.996212229189942*v_B2lv_c_r
        struct[0].g[74,0] = -i_t_B2_B2lv_c_r - 0.0431372549019608*v_B2_a_i - 0.0107843137254902*v_B2_a_r - 0.0431372549019608*v_B2_b_i - 0.0107843137254902*v_B2_b_r + 0.0862745098039216*v_B2_c_i + 0.0215686274509804*v_B2_c_r + 0.996212229189942*v_B2lv_a_i + 0.249053057297486*v_B2lv_a_r - 0.996212229189942*v_B2lv_c_i - 0.249053057297486*v_B2lv_c_r
        struct[0].g[75,0] = -i_t_B2_B2lv_c_i - 0.0107843137254902*v_B2_a_i + 0.0431372549019608*v_B2_a_r - 0.0107843137254902*v_B2_b_i + 0.0431372549019608*v_B2_b_r + 0.0215686274509804*v_B2_c_i - 0.0862745098039216*v_B2_c_r + 0.249053057297486*v_B2lv_a_i - 0.996212229189942*v_B2lv_a_r - 0.249053057297486*v_B2lv_c_i + 0.996212229189942*v_B2lv_c_r
        struct[0].g[76,0] = -i_t_B3_B3lv_a_r + 0.0862745098039216*v_B3_a_i + 0.0215686274509804*v_B3_a_r - 0.0431372549019608*v_B3_b_i - 0.0107843137254902*v_B3_b_r - 0.0431372549019608*v_B3_c_i - 0.0107843137254902*v_B3_c_r - 0.996212229189942*v_B3lv_a_i - 0.249053057297486*v_B3lv_a_r + 0.996212229189942*v_B3lv_b_i + 0.249053057297486*v_B3lv_b_r
        struct[0].g[77,0] = -i_t_B3_B3lv_a_i + 0.0215686274509804*v_B3_a_i - 0.0862745098039216*v_B3_a_r - 0.0107843137254902*v_B3_b_i + 0.0431372549019608*v_B3_b_r - 0.0107843137254902*v_B3_c_i + 0.0431372549019608*v_B3_c_r - 0.249053057297486*v_B3lv_a_i + 0.996212229189942*v_B3lv_a_r + 0.249053057297486*v_B3lv_b_i - 0.996212229189942*v_B3lv_b_r
        struct[0].g[78,0] = -i_t_B3_B3lv_b_r - 0.0431372549019608*v_B3_a_i - 0.0107843137254902*v_B3_a_r + 0.0862745098039216*v_B3_b_i + 0.0215686274509804*v_B3_b_r - 0.0431372549019608*v_B3_c_i - 0.0107843137254902*v_B3_c_r - 0.996212229189942*v_B3lv_b_i - 0.249053057297486*v_B3lv_b_r + 0.996212229189942*v_B3lv_c_i + 0.249053057297486*v_B3lv_c_r
        struct[0].g[79,0] = -i_t_B3_B3lv_b_i - 0.0107843137254902*v_B3_a_i + 0.0431372549019608*v_B3_a_r + 0.0215686274509804*v_B3_b_i - 0.0862745098039216*v_B3_b_r - 0.0107843137254902*v_B3_c_i + 0.0431372549019608*v_B3_c_r - 0.249053057297486*v_B3lv_b_i + 0.996212229189942*v_B3lv_b_r + 0.249053057297486*v_B3lv_c_i - 0.996212229189942*v_B3lv_c_r
        struct[0].g[80,0] = -i_t_B3_B3lv_c_r - 0.0431372549019608*v_B3_a_i - 0.0107843137254902*v_B3_a_r - 0.0431372549019608*v_B3_b_i - 0.0107843137254902*v_B3_b_r + 0.0862745098039216*v_B3_c_i + 0.0215686274509804*v_B3_c_r + 0.996212229189942*v_B3lv_a_i + 0.249053057297486*v_B3lv_a_r - 0.996212229189942*v_B3lv_c_i - 0.249053057297486*v_B3lv_c_r
        struct[0].g[81,0] = -i_t_B3_B3lv_c_i - 0.0107843137254902*v_B3_a_i + 0.0431372549019608*v_B3_a_r - 0.0107843137254902*v_B3_b_i + 0.0431372549019608*v_B3_b_r + 0.0215686274509804*v_B3_c_i - 0.0862745098039216*v_B3_c_r + 0.249053057297486*v_B3lv_a_i - 0.996212229189942*v_B3lv_a_r - 0.249053057297486*v_B3lv_c_i + 0.996212229189942*v_B3lv_c_r
        struct[0].g[82,0] = -i_t_B4_B4lv_a_r + 0.0862745098039216*v_B4_a_i + 0.0215686274509804*v_B4_a_r - 0.0431372549019608*v_B4_b_i - 0.0107843137254902*v_B4_b_r - 0.0431372549019608*v_B4_c_i - 0.0107843137254902*v_B4_c_r - 0.996212229189942*v_B4lv_a_i - 0.249053057297486*v_B4lv_a_r + 0.996212229189942*v_B4lv_b_i + 0.249053057297486*v_B4lv_b_r
        struct[0].g[83,0] = -i_t_B4_B4lv_a_i + 0.0215686274509804*v_B4_a_i - 0.0862745098039216*v_B4_a_r - 0.0107843137254902*v_B4_b_i + 0.0431372549019608*v_B4_b_r - 0.0107843137254902*v_B4_c_i + 0.0431372549019608*v_B4_c_r - 0.249053057297486*v_B4lv_a_i + 0.996212229189942*v_B4lv_a_r + 0.249053057297486*v_B4lv_b_i - 0.996212229189942*v_B4lv_b_r
        struct[0].g[84,0] = -i_t_B4_B4lv_b_r - 0.0431372549019608*v_B4_a_i - 0.0107843137254902*v_B4_a_r + 0.0862745098039216*v_B4_b_i + 0.0215686274509804*v_B4_b_r - 0.0431372549019608*v_B4_c_i - 0.0107843137254902*v_B4_c_r - 0.996212229189942*v_B4lv_b_i - 0.249053057297486*v_B4lv_b_r + 0.996212229189942*v_B4lv_c_i + 0.249053057297486*v_B4lv_c_r
        struct[0].g[85,0] = -i_t_B4_B4lv_b_i - 0.0107843137254902*v_B4_a_i + 0.0431372549019608*v_B4_a_r + 0.0215686274509804*v_B4_b_i - 0.0862745098039216*v_B4_b_r - 0.0107843137254902*v_B4_c_i + 0.0431372549019608*v_B4_c_r - 0.249053057297486*v_B4lv_b_i + 0.996212229189942*v_B4lv_b_r + 0.249053057297486*v_B4lv_c_i - 0.996212229189942*v_B4lv_c_r
        struct[0].g[86,0] = -i_t_B4_B4lv_c_r - 0.0431372549019608*v_B4_a_i - 0.0107843137254902*v_B4_a_r - 0.0431372549019608*v_B4_b_i - 0.0107843137254902*v_B4_b_r + 0.0862745098039216*v_B4_c_i + 0.0215686274509804*v_B4_c_r + 0.996212229189942*v_B4lv_a_i + 0.249053057297486*v_B4lv_a_r - 0.996212229189942*v_B4lv_c_i - 0.249053057297486*v_B4lv_c_r
        struct[0].g[87,0] = -i_t_B4_B4lv_c_i - 0.0107843137254902*v_B4_a_i + 0.0431372549019608*v_B4_a_r - 0.0107843137254902*v_B4_b_i + 0.0431372549019608*v_B4_b_r + 0.0215686274509804*v_B4_c_i - 0.0862745098039216*v_B4_c_r + 0.249053057297486*v_B4lv_a_i - 0.996212229189942*v_B4lv_a_r - 0.249053057297486*v_B4lv_c_i + 0.996212229189942*v_B4lv_c_r
        struct[0].g[88,0] = -i_t_B5_B5lv_a_r + 0.0862745098039216*v_B5_a_i + 0.0215686274509804*v_B5_a_r - 0.0431372549019608*v_B5_b_i - 0.0107843137254902*v_B5_b_r - 0.0431372549019608*v_B5_c_i - 0.0107843137254902*v_B5_c_r - 0.996212229189942*v_B5lv_a_i - 0.249053057297486*v_B5lv_a_r + 0.996212229189942*v_B5lv_b_i + 0.249053057297486*v_B5lv_b_r
        struct[0].g[89,0] = -i_t_B5_B5lv_a_i + 0.0215686274509804*v_B5_a_i - 0.0862745098039216*v_B5_a_r - 0.0107843137254902*v_B5_b_i + 0.0431372549019608*v_B5_b_r - 0.0107843137254902*v_B5_c_i + 0.0431372549019608*v_B5_c_r - 0.249053057297486*v_B5lv_a_i + 0.996212229189942*v_B5lv_a_r + 0.249053057297486*v_B5lv_b_i - 0.996212229189942*v_B5lv_b_r
        struct[0].g[90,0] = -i_t_B5_B5lv_b_r - 0.0431372549019608*v_B5_a_i - 0.0107843137254902*v_B5_a_r + 0.0862745098039216*v_B5_b_i + 0.0215686274509804*v_B5_b_r - 0.0431372549019608*v_B5_c_i - 0.0107843137254902*v_B5_c_r - 0.996212229189942*v_B5lv_b_i - 0.249053057297486*v_B5lv_b_r + 0.996212229189942*v_B5lv_c_i + 0.249053057297486*v_B5lv_c_r
        struct[0].g[91,0] = -i_t_B5_B5lv_b_i - 0.0107843137254902*v_B5_a_i + 0.0431372549019608*v_B5_a_r + 0.0215686274509804*v_B5_b_i - 0.0862745098039216*v_B5_b_r - 0.0107843137254902*v_B5_c_i + 0.0431372549019608*v_B5_c_r - 0.249053057297486*v_B5lv_b_i + 0.996212229189942*v_B5lv_b_r + 0.249053057297486*v_B5lv_c_i - 0.996212229189942*v_B5lv_c_r
        struct[0].g[92,0] = -i_t_B5_B5lv_c_r - 0.0431372549019608*v_B5_a_i - 0.0107843137254902*v_B5_a_r - 0.0431372549019608*v_B5_b_i - 0.0107843137254902*v_B5_b_r + 0.0862745098039216*v_B5_c_i + 0.0215686274509804*v_B5_c_r + 0.996212229189942*v_B5lv_a_i + 0.249053057297486*v_B5lv_a_r - 0.996212229189942*v_B5lv_c_i - 0.249053057297486*v_B5lv_c_r
        struct[0].g[93,0] = -i_t_B5_B5lv_c_i - 0.0107843137254902*v_B5_a_i + 0.0431372549019608*v_B5_a_r - 0.0107843137254902*v_B5_b_i + 0.0431372549019608*v_B5_b_r + 0.0215686274509804*v_B5_c_i - 0.0862745098039216*v_B5_c_r + 0.249053057297486*v_B5lv_a_i - 0.996212229189942*v_B5lv_a_r - 0.249053057297486*v_B5lv_c_i + 0.996212229189942*v_B5lv_c_r
        struct[0].g[94,0] = -i_t_B6_B6lv_a_r + 0.0862745098039216*v_B6_a_i + 0.0215686274509804*v_B6_a_r - 0.0431372549019608*v_B6_b_i - 0.0107843137254902*v_B6_b_r - 0.0431372549019608*v_B6_c_i - 0.0107843137254902*v_B6_c_r - 0.996212229189942*v_B6lv_a_i - 0.249053057297486*v_B6lv_a_r + 0.996212229189942*v_B6lv_b_i + 0.249053057297486*v_B6lv_b_r
        struct[0].g[95,0] = -i_t_B6_B6lv_a_i + 0.0215686274509804*v_B6_a_i - 0.0862745098039216*v_B6_a_r - 0.0107843137254902*v_B6_b_i + 0.0431372549019608*v_B6_b_r - 0.0107843137254902*v_B6_c_i + 0.0431372549019608*v_B6_c_r - 0.249053057297486*v_B6lv_a_i + 0.996212229189942*v_B6lv_a_r + 0.249053057297486*v_B6lv_b_i - 0.996212229189942*v_B6lv_b_r
        struct[0].g[96,0] = -i_t_B6_B6lv_b_r - 0.0431372549019608*v_B6_a_i - 0.0107843137254902*v_B6_a_r + 0.0862745098039216*v_B6_b_i + 0.0215686274509804*v_B6_b_r - 0.0431372549019608*v_B6_c_i - 0.0107843137254902*v_B6_c_r - 0.996212229189942*v_B6lv_b_i - 0.249053057297486*v_B6lv_b_r + 0.996212229189942*v_B6lv_c_i + 0.249053057297486*v_B6lv_c_r
        struct[0].g[97,0] = -i_t_B6_B6lv_b_i - 0.0107843137254902*v_B6_a_i + 0.0431372549019608*v_B6_a_r + 0.0215686274509804*v_B6_b_i - 0.0862745098039216*v_B6_b_r - 0.0107843137254902*v_B6_c_i + 0.0431372549019608*v_B6_c_r - 0.249053057297486*v_B6lv_b_i + 0.996212229189942*v_B6lv_b_r + 0.249053057297486*v_B6lv_c_i - 0.996212229189942*v_B6lv_c_r
        struct[0].g[98,0] = -i_t_B6_B6lv_c_r - 0.0431372549019608*v_B6_a_i - 0.0107843137254902*v_B6_a_r - 0.0431372549019608*v_B6_b_i - 0.0107843137254902*v_B6_b_r + 0.0862745098039216*v_B6_c_i + 0.0215686274509804*v_B6_c_r + 0.996212229189942*v_B6lv_a_i + 0.249053057297486*v_B6lv_a_r - 0.996212229189942*v_B6lv_c_i - 0.249053057297486*v_B6lv_c_r
        struct[0].g[99,0] = -i_t_B6_B6lv_c_i - 0.0107843137254902*v_B6_a_i + 0.0431372549019608*v_B6_a_r - 0.0107843137254902*v_B6_b_i + 0.0431372549019608*v_B6_b_r + 0.0215686274509804*v_B6_c_i - 0.0862745098039216*v_B6_c_r + 0.249053057297486*v_B6lv_a_i - 0.996212229189942*v_B6lv_a_r - 0.249053057297486*v_B6lv_c_i + 0.996212229189942*v_B6lv_c_r
        struct[0].g[100,0] = -i_l_B1_B2_a_r + 0.598820527961361*v_B1_a_i + 1.10755301189314*v_B1_a_r - 0.171091579417532*v_B1_b_i - 0.316443717683753*v_B1_b_r - 0.171091579417532*v_B1_c_i - 0.316443717683753*v_B1_c_r - 0.598820527961361*v_B2_a_i - 1.10755301189314*v_B2_a_r + 0.171091579417532*v_B2_b_i + 0.316443717683753*v_B2_b_r + 0.171091579417532*v_B2_c_i + 0.316443717683753*v_B2_c_r
        struct[0].g[101,0] = -i_l_B1_B2_a_i + 1.10755301189314*v_B1_a_i - 0.598820527961361*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417532*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417532*v_B1_c_r - 1.10755301189314*v_B2_a_i + 0.598820527961361*v_B2_a_r + 0.316443717683753*v_B2_b_i - 0.171091579417532*v_B2_b_r + 0.316443717683753*v_B2_c_i - 0.171091579417532*v_B2_c_r
        struct[0].g[102,0] = -i_l_B1_B2_b_r - 0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r + 0.59882052796136*v_B1_b_i + 1.10755301189314*v_B1_b_r - 0.171091579417531*v_B1_c_i - 0.316443717683753*v_B1_c_r + 0.171091579417532*v_B2_a_i + 0.316443717683753*v_B2_a_r - 0.59882052796136*v_B2_b_i - 1.10755301189314*v_B2_b_r + 0.171091579417531*v_B2_c_i + 0.316443717683753*v_B2_c_r
        struct[0].g[103,0] = -i_l_B1_B2_b_i - 0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r + 1.10755301189314*v_B1_b_i - 0.59882052796136*v_B1_b_r - 0.316443717683753*v_B1_c_i + 0.171091579417531*v_B1_c_r + 0.316443717683753*v_B2_a_i - 0.171091579417532*v_B2_a_r - 1.10755301189314*v_B2_b_i + 0.59882052796136*v_B2_b_r + 0.316443717683753*v_B2_c_i - 0.171091579417531*v_B2_c_r
        struct[0].g[104,0] = -i_l_B1_B2_c_r - 0.171091579417532*v_B1_a_i - 0.316443717683753*v_B1_a_r - 0.171091579417531*v_B1_b_i - 0.316443717683753*v_B1_b_r + 0.59882052796136*v_B1_c_i + 1.10755301189314*v_B1_c_r + 0.171091579417532*v_B2_a_i + 0.316443717683753*v_B2_a_r + 0.171091579417531*v_B2_b_i + 0.316443717683753*v_B2_b_r - 0.59882052796136*v_B2_c_i - 1.10755301189314*v_B2_c_r
        struct[0].g[105,0] = -i_l_B1_B2_c_i - 0.316443717683753*v_B1_a_i + 0.171091579417532*v_B1_a_r - 0.316443717683753*v_B1_b_i + 0.171091579417531*v_B1_b_r + 1.10755301189314*v_B1_c_i - 0.59882052796136*v_B1_c_r + 0.316443717683753*v_B2_a_i - 0.171091579417532*v_B2_a_r + 0.316443717683753*v_B2_b_i - 0.171091579417531*v_B2_b_r - 1.10755301189314*v_B2_c_i + 0.59882052796136*v_B2_c_r
        struct[0].g[106,0] = -i_l_B2_B3_a_r + 0.598820527961361*v_B2_a_i + 1.10755301189314*v_B2_a_r - 0.171091579417532*v_B2_b_i - 0.316443717683753*v_B2_b_r - 0.171091579417532*v_B2_c_i - 0.316443717683753*v_B2_c_r - 0.598820527961361*v_B3_a_i - 1.10755301189314*v_B3_a_r + 0.171091579417532*v_B3_b_i + 0.316443717683753*v_B3_b_r + 0.171091579417532*v_B3_c_i + 0.316443717683753*v_B3_c_r
        struct[0].g[107,0] = -i_l_B2_B3_a_i + 1.10755301189314*v_B2_a_i - 0.598820527961361*v_B2_a_r - 0.316443717683753*v_B2_b_i + 0.171091579417532*v_B2_b_r - 0.316443717683753*v_B2_c_i + 0.171091579417532*v_B2_c_r - 1.10755301189314*v_B3_a_i + 0.598820527961361*v_B3_a_r + 0.316443717683753*v_B3_b_i - 0.171091579417532*v_B3_b_r + 0.316443717683753*v_B3_c_i - 0.171091579417532*v_B3_c_r
        struct[0].g[108,0] = -i_l_B2_B3_b_r - 0.171091579417532*v_B2_a_i - 0.316443717683753*v_B2_a_r + 0.59882052796136*v_B2_b_i + 1.10755301189314*v_B2_b_r - 0.171091579417531*v_B2_c_i - 0.316443717683753*v_B2_c_r + 0.171091579417532*v_B3_a_i + 0.316443717683753*v_B3_a_r - 0.59882052796136*v_B3_b_i - 1.10755301189314*v_B3_b_r + 0.171091579417531*v_B3_c_i + 0.316443717683753*v_B3_c_r
        struct[0].g[109,0] = -i_l_B2_B3_b_i - 0.316443717683753*v_B2_a_i + 0.171091579417532*v_B2_a_r + 1.10755301189314*v_B2_b_i - 0.59882052796136*v_B2_b_r - 0.316443717683753*v_B2_c_i + 0.171091579417531*v_B2_c_r + 0.316443717683753*v_B3_a_i - 0.171091579417532*v_B3_a_r - 1.10755301189314*v_B3_b_i + 0.59882052796136*v_B3_b_r + 0.316443717683753*v_B3_c_i - 0.171091579417531*v_B3_c_r
        struct[0].g[110,0] = -i_l_B2_B3_c_r - 0.171091579417532*v_B2_a_i - 0.316443717683753*v_B2_a_r - 0.171091579417531*v_B2_b_i - 0.316443717683753*v_B2_b_r + 0.59882052796136*v_B2_c_i + 1.10755301189314*v_B2_c_r + 0.171091579417532*v_B3_a_i + 0.316443717683753*v_B3_a_r + 0.171091579417531*v_B3_b_i + 0.316443717683753*v_B3_b_r - 0.59882052796136*v_B3_c_i - 1.10755301189314*v_B3_c_r
        struct[0].g[111,0] = -i_l_B2_B3_c_i - 0.316443717683753*v_B2_a_i + 0.171091579417532*v_B2_a_r - 0.316443717683753*v_B2_b_i + 0.171091579417531*v_B2_b_r + 1.10755301189314*v_B2_c_i - 0.59882052796136*v_B2_c_r + 0.316443717683753*v_B3_a_i - 0.171091579417532*v_B3_a_r + 0.316443717683753*v_B3_b_i - 0.171091579417531*v_B3_b_r - 1.10755301189314*v_B3_c_i + 0.59882052796136*v_B3_c_r
        struct[0].g[112,0] = -i_l_B3_B4_a_r + 0.598820527961361*v_B3_a_i + 1.10755301189314*v_B3_a_r - 0.171091579417532*v_B3_b_i - 0.316443717683753*v_B3_b_r - 0.171091579417532*v_B3_c_i - 0.316443717683753*v_B3_c_r - 0.598820527961361*v_B4_a_i - 1.10755301189314*v_B4_a_r + 0.171091579417532*v_B4_b_i + 0.316443717683753*v_B4_b_r + 0.171091579417532*v_B4_c_i + 0.316443717683753*v_B4_c_r
        struct[0].g[113,0] = -i_l_B3_B4_a_i + 1.10755301189314*v_B3_a_i - 0.598820527961361*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417532*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417532*v_B3_c_r - 1.10755301189314*v_B4_a_i + 0.598820527961361*v_B4_a_r + 0.316443717683753*v_B4_b_i - 0.171091579417532*v_B4_b_r + 0.316443717683753*v_B4_c_i - 0.171091579417532*v_B4_c_r
        struct[0].g[114,0] = -i_l_B3_B4_b_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r + 0.59882052796136*v_B3_b_i + 1.10755301189314*v_B3_b_r - 0.171091579417531*v_B3_c_i - 0.316443717683753*v_B3_c_r + 0.171091579417532*v_B4_a_i + 0.316443717683753*v_B4_a_r - 0.59882052796136*v_B4_b_i - 1.10755301189314*v_B4_b_r + 0.171091579417531*v_B4_c_i + 0.316443717683753*v_B4_c_r
        struct[0].g[115,0] = -i_l_B3_B4_b_i - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r + 1.10755301189314*v_B3_b_i - 0.59882052796136*v_B3_b_r - 0.316443717683753*v_B3_c_i + 0.171091579417531*v_B3_c_r + 0.316443717683753*v_B4_a_i - 0.171091579417532*v_B4_a_r - 1.10755301189314*v_B4_b_i + 0.59882052796136*v_B4_b_r + 0.316443717683753*v_B4_c_i - 0.171091579417531*v_B4_c_r
        struct[0].g[116,0] = -i_l_B3_B4_c_r - 0.171091579417532*v_B3_a_i - 0.316443717683753*v_B3_a_r - 0.171091579417531*v_B3_b_i - 0.316443717683753*v_B3_b_r + 0.59882052796136*v_B3_c_i + 1.10755301189314*v_B3_c_r + 0.171091579417532*v_B4_a_i + 0.316443717683753*v_B4_a_r + 0.171091579417531*v_B4_b_i + 0.316443717683753*v_B4_b_r - 0.59882052796136*v_B4_c_i - 1.10755301189314*v_B4_c_r
        struct[0].g[117,0] = -i_l_B3_B4_c_i - 0.316443717683753*v_B3_a_i + 0.171091579417532*v_B3_a_r - 0.316443717683753*v_B3_b_i + 0.171091579417531*v_B3_b_r + 1.10755301189314*v_B3_c_i - 0.59882052796136*v_B3_c_r + 0.316443717683753*v_B4_a_i - 0.171091579417532*v_B4_a_r + 0.316443717683753*v_B4_b_i - 0.171091579417531*v_B4_b_r - 1.10755301189314*v_B4_c_i + 0.59882052796136*v_B4_c_r
        struct[0].g[118,0] = -i_l_B5_B6_a_r + 0.598820527961361*v_B5_a_i + 1.10755301189314*v_B5_a_r - 0.171091579417532*v_B5_b_i - 0.316443717683753*v_B5_b_r - 0.171091579417532*v_B5_c_i - 0.316443717683753*v_B5_c_r - 0.598820527961361*v_B6_a_i - 1.10755301189314*v_B6_a_r + 0.171091579417532*v_B6_b_i + 0.316443717683753*v_B6_b_r + 0.171091579417532*v_B6_c_i + 0.316443717683753*v_B6_c_r
        struct[0].g[119,0] = -i_l_B5_B6_a_i + 1.10755301189314*v_B5_a_i - 0.598820527961361*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417532*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417532*v_B5_c_r - 1.10755301189314*v_B6_a_i + 0.598820527961361*v_B6_a_r + 0.316443717683753*v_B6_b_i - 0.171091579417532*v_B6_b_r + 0.316443717683753*v_B6_c_i - 0.171091579417532*v_B6_c_r
        struct[0].g[120,0] = -i_l_B5_B6_b_r - 0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r + 0.59882052796136*v_B5_b_i + 1.10755301189314*v_B5_b_r - 0.171091579417531*v_B5_c_i - 0.316443717683753*v_B5_c_r + 0.171091579417532*v_B6_a_i + 0.316443717683753*v_B6_a_r - 0.59882052796136*v_B6_b_i - 1.10755301189314*v_B6_b_r + 0.171091579417531*v_B6_c_i + 0.316443717683753*v_B6_c_r
        struct[0].g[121,0] = -i_l_B5_B6_b_i - 0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r + 1.10755301189314*v_B5_b_i - 0.59882052796136*v_B5_b_r - 0.316443717683753*v_B5_c_i + 0.171091579417531*v_B5_c_r + 0.316443717683753*v_B6_a_i - 0.171091579417532*v_B6_a_r - 1.10755301189314*v_B6_b_i + 0.59882052796136*v_B6_b_r + 0.316443717683753*v_B6_c_i - 0.171091579417531*v_B6_c_r
        struct[0].g[122,0] = -i_l_B5_B6_c_r - 0.171091579417532*v_B5_a_i - 0.316443717683753*v_B5_a_r - 0.171091579417531*v_B5_b_i - 0.316443717683753*v_B5_b_r + 0.59882052796136*v_B5_c_i + 1.10755301189314*v_B5_c_r + 0.171091579417532*v_B6_a_i + 0.316443717683753*v_B6_a_r + 0.171091579417531*v_B6_b_i + 0.316443717683753*v_B6_b_r - 0.59882052796136*v_B6_c_i - 1.10755301189314*v_B6_c_r
        struct[0].g[123,0] = -i_l_B5_B6_c_i - 0.316443717683753*v_B5_a_i + 0.171091579417532*v_B5_a_r - 0.316443717683753*v_B5_b_i + 0.171091579417531*v_B5_b_r + 1.10755301189314*v_B5_c_i - 0.59882052796136*v_B5_c_r + 0.316443717683753*v_B6_a_i - 0.171091579417532*v_B6_a_r + 0.316443717683753*v_B6_b_i - 0.171091579417531*v_B6_b_r - 1.10755301189314*v_B6_c_i + 0.59882052796136*v_B6_c_r
        struct[0].g[124,0] = -i_l_B6_B7_a_r + 0.598820527961361*v_B6_a_i + 1.10755301189314*v_B6_a_r - 0.171091579417532*v_B6_b_i - 0.316443717683753*v_B6_b_r - 0.171091579417532*v_B6_c_i - 0.316443717683753*v_B6_c_r - 0.598820527961361*v_B7_a_i - 1.10755301189314*v_B7_a_r + 0.171091579417532*v_B7_b_i + 0.316443717683753*v_B7_b_r + 0.171091579417532*v_B7_c_i + 0.316443717683753*v_B7_c_r
        struct[0].g[125,0] = -i_l_B6_B7_a_i + 1.10755301189314*v_B6_a_i - 0.598820527961361*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417532*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417532*v_B6_c_r - 1.10755301189314*v_B7_a_i + 0.598820527961361*v_B7_a_r + 0.316443717683753*v_B7_b_i - 0.171091579417532*v_B7_b_r + 0.316443717683753*v_B7_c_i - 0.171091579417532*v_B7_c_r
        struct[0].g[126,0] = -i_l_B6_B7_b_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r + 0.59882052796136*v_B6_b_i + 1.10755301189314*v_B6_b_r - 0.171091579417531*v_B6_c_i - 0.316443717683753*v_B6_c_r + 0.171091579417532*v_B7_a_i + 0.316443717683753*v_B7_a_r - 0.59882052796136*v_B7_b_i - 1.10755301189314*v_B7_b_r + 0.171091579417531*v_B7_c_i + 0.316443717683753*v_B7_c_r
        struct[0].g[127,0] = -i_l_B6_B7_b_i - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r + 1.10755301189314*v_B6_b_i - 0.59882052796136*v_B6_b_r - 0.316443717683753*v_B6_c_i + 0.171091579417531*v_B6_c_r + 0.316443717683753*v_B7_a_i - 0.171091579417532*v_B7_a_r - 1.10755301189314*v_B7_b_i + 0.59882052796136*v_B7_b_r + 0.316443717683753*v_B7_c_i - 0.171091579417531*v_B7_c_r
        struct[0].g[128,0] = -i_l_B6_B7_c_r - 0.171091579417532*v_B6_a_i - 0.316443717683753*v_B6_a_r - 0.171091579417531*v_B6_b_i - 0.316443717683753*v_B6_b_r + 0.59882052796136*v_B6_c_i + 1.10755301189314*v_B6_c_r + 0.171091579417532*v_B7_a_i + 0.316443717683753*v_B7_a_r + 0.171091579417531*v_B7_b_i + 0.316443717683753*v_B7_b_r - 0.59882052796136*v_B7_c_i - 1.10755301189314*v_B7_c_r
        struct[0].g[129,0] = -i_l_B6_B7_c_i - 0.316443717683753*v_B6_a_i + 0.171091579417532*v_B6_a_r - 0.316443717683753*v_B6_b_i + 0.171091579417531*v_B6_b_r + 1.10755301189314*v_B6_c_i - 0.59882052796136*v_B6_c_r + 0.316443717683753*v_B7_a_i - 0.171091579417532*v_B7_a_r + 0.316443717683753*v_B7_b_i - 0.171091579417531*v_B7_b_r - 1.10755301189314*v_B7_c_i + 0.59882052796136*v_B7_c_r
        struct[0].g[130,0] = i_load_B2lv_a_i*v_B2lv_a_i - i_load_B2lv_a_i*v_B2lv_n_i + i_load_B2lv_a_r*v_B2lv_a_r - i_load_B2lv_a_r*v_B2lv_n_r - p_B2lv_a
        struct[0].g[131,0] = i_load_B2lv_b_i*v_B2lv_b_i - i_load_B2lv_b_i*v_B2lv_n_i + i_load_B2lv_b_r*v_B2lv_b_r - i_load_B2lv_b_r*v_B2lv_n_r - p_B2lv_b
        struct[0].g[132,0] = i_load_B2lv_c_i*v_B2lv_c_i - i_load_B2lv_c_i*v_B2lv_n_i + i_load_B2lv_c_r*v_B2lv_c_r - i_load_B2lv_c_r*v_B2lv_n_r - p_B2lv_c
        struct[0].g[133,0] = -i_load_B2lv_a_i*v_B2lv_a_r + i_load_B2lv_a_i*v_B2lv_n_r + i_load_B2lv_a_r*v_B2lv_a_i - i_load_B2lv_a_r*v_B2lv_n_i - q_B2lv_a
        struct[0].g[134,0] = -i_load_B2lv_b_i*v_B2lv_b_r + i_load_B2lv_b_i*v_B2lv_n_r + i_load_B2lv_b_r*v_B2lv_b_i - i_load_B2lv_b_r*v_B2lv_n_i - q_B2lv_b
        struct[0].g[135,0] = -i_load_B2lv_c_i*v_B2lv_c_r + i_load_B2lv_c_i*v_B2lv_n_r + i_load_B2lv_c_r*v_B2lv_c_i - i_load_B2lv_c_r*v_B2lv_n_i - q_B2lv_c
        struct[0].g[136,0] = i_load_B2lv_a_r + i_load_B2lv_b_r + i_load_B2lv_c_r + i_load_B2lv_n_r
        struct[0].g[137,0] = i_load_B2lv_a_i + i_load_B2lv_b_i + i_load_B2lv_c_i + i_load_B2lv_n_i
        struct[0].g[138,0] = i_load_B3lv_a_i*v_B3lv_a_i - i_load_B3lv_a_i*v_B3lv_n_i + i_load_B3lv_a_r*v_B3lv_a_r - i_load_B3lv_a_r*v_B3lv_n_r - p_B3lv_a
        struct[0].g[139,0] = i_load_B3lv_b_i*v_B3lv_b_i - i_load_B3lv_b_i*v_B3lv_n_i + i_load_B3lv_b_r*v_B3lv_b_r - i_load_B3lv_b_r*v_B3lv_n_r - p_B3lv_b
        struct[0].g[140,0] = i_load_B3lv_c_i*v_B3lv_c_i - i_load_B3lv_c_i*v_B3lv_n_i + i_load_B3lv_c_r*v_B3lv_c_r - i_load_B3lv_c_r*v_B3lv_n_r - p_B3lv_c
        struct[0].g[141,0] = -i_load_B3lv_a_i*v_B3lv_a_r + i_load_B3lv_a_i*v_B3lv_n_r + i_load_B3lv_a_r*v_B3lv_a_i - i_load_B3lv_a_r*v_B3lv_n_i - q_B3lv_a
        struct[0].g[142,0] = -i_load_B3lv_b_i*v_B3lv_b_r + i_load_B3lv_b_i*v_B3lv_n_r + i_load_B3lv_b_r*v_B3lv_b_i - i_load_B3lv_b_r*v_B3lv_n_i - q_B3lv_b
        struct[0].g[143,0] = -i_load_B3lv_c_i*v_B3lv_c_r + i_load_B3lv_c_i*v_B3lv_n_r + i_load_B3lv_c_r*v_B3lv_c_i - i_load_B3lv_c_r*v_B3lv_n_i - q_B3lv_c
        struct[0].g[144,0] = i_load_B3lv_a_r + i_load_B3lv_b_r + i_load_B3lv_c_r + i_load_B3lv_n_r
        struct[0].g[145,0] = i_load_B3lv_a_i + i_load_B3lv_b_i + i_load_B3lv_c_i + i_load_B3lv_n_i
        struct[0].g[146,0] = i_load_B4lv_a_i*v_B4lv_a_i - i_load_B4lv_a_i*v_B4lv_n_i + i_load_B4lv_a_r*v_B4lv_a_r - i_load_B4lv_a_r*v_B4lv_n_r - p_B4lv_a
        struct[0].g[147,0] = i_load_B4lv_b_i*v_B4lv_b_i - i_load_B4lv_b_i*v_B4lv_n_i + i_load_B4lv_b_r*v_B4lv_b_r - i_load_B4lv_b_r*v_B4lv_n_r - p_B4lv_b
        struct[0].g[148,0] = i_load_B4lv_c_i*v_B4lv_c_i - i_load_B4lv_c_i*v_B4lv_n_i + i_load_B4lv_c_r*v_B4lv_c_r - i_load_B4lv_c_r*v_B4lv_n_r - p_B4lv_c
        struct[0].g[149,0] = -i_load_B4lv_a_i*v_B4lv_a_r + i_load_B4lv_a_i*v_B4lv_n_r + i_load_B4lv_a_r*v_B4lv_a_i - i_load_B4lv_a_r*v_B4lv_n_i - q_B4lv_a
        struct[0].g[150,0] = -i_load_B4lv_b_i*v_B4lv_b_r + i_load_B4lv_b_i*v_B4lv_n_r + i_load_B4lv_b_r*v_B4lv_b_i - i_load_B4lv_b_r*v_B4lv_n_i - q_B4lv_b
        struct[0].g[151,0] = -i_load_B4lv_c_i*v_B4lv_c_r + i_load_B4lv_c_i*v_B4lv_n_r + i_load_B4lv_c_r*v_B4lv_c_i - i_load_B4lv_c_r*v_B4lv_n_i - q_B4lv_c
        struct[0].g[152,0] = i_load_B4lv_a_r + i_load_B4lv_b_r + i_load_B4lv_c_r + i_load_B4lv_n_r
        struct[0].g[153,0] = i_load_B4lv_a_i + i_load_B4lv_b_i + i_load_B4lv_c_i + i_load_B4lv_n_i
        struct[0].g[154,0] = i_load_B5lv_a_i*v_B5lv_a_i - i_load_B5lv_a_i*v_B5lv_n_i + i_load_B5lv_a_r*v_B5lv_a_r - i_load_B5lv_a_r*v_B5lv_n_r - p_B5lv_a
        struct[0].g[155,0] = i_load_B5lv_b_i*v_B5lv_b_i - i_load_B5lv_b_i*v_B5lv_n_i + i_load_B5lv_b_r*v_B5lv_b_r - i_load_B5lv_b_r*v_B5lv_n_r - p_B5lv_b
        struct[0].g[156,0] = i_load_B5lv_c_i*v_B5lv_c_i - i_load_B5lv_c_i*v_B5lv_n_i + i_load_B5lv_c_r*v_B5lv_c_r - i_load_B5lv_c_r*v_B5lv_n_r - p_B5lv_c
        struct[0].g[157,0] = -i_load_B5lv_a_i*v_B5lv_a_r + i_load_B5lv_a_i*v_B5lv_n_r + i_load_B5lv_a_r*v_B5lv_a_i - i_load_B5lv_a_r*v_B5lv_n_i - q_B5lv_a
        struct[0].g[158,0] = -i_load_B5lv_b_i*v_B5lv_b_r + i_load_B5lv_b_i*v_B5lv_n_r + i_load_B5lv_b_r*v_B5lv_b_i - i_load_B5lv_b_r*v_B5lv_n_i - q_B5lv_b
        struct[0].g[159,0] = -i_load_B5lv_c_i*v_B5lv_c_r + i_load_B5lv_c_i*v_B5lv_n_r + i_load_B5lv_c_r*v_B5lv_c_i - i_load_B5lv_c_r*v_B5lv_n_i - q_B5lv_c
        struct[0].g[160,0] = i_load_B5lv_a_r + i_load_B5lv_b_r + i_load_B5lv_c_r + i_load_B5lv_n_r
        struct[0].g[161,0] = i_load_B5lv_a_i + i_load_B5lv_b_i + i_load_B5lv_c_i + i_load_B5lv_n_i
        struct[0].g[162,0] = i_load_B6lv_a_i*v_B6lv_a_i - i_load_B6lv_a_i*v_B6lv_n_i + i_load_B6lv_a_r*v_B6lv_a_r - i_load_B6lv_a_r*v_B6lv_n_r - p_B6lv_a
        struct[0].g[163,0] = i_load_B6lv_b_i*v_B6lv_b_i - i_load_B6lv_b_i*v_B6lv_n_i + i_load_B6lv_b_r*v_B6lv_b_r - i_load_B6lv_b_r*v_B6lv_n_r - p_B6lv_b
        struct[0].g[164,0] = i_load_B6lv_c_i*v_B6lv_c_i - i_load_B6lv_c_i*v_B6lv_n_i + i_load_B6lv_c_r*v_B6lv_c_r - i_load_B6lv_c_r*v_B6lv_n_r - p_B6lv_c
        struct[0].g[165,0] = -i_load_B6lv_a_i*v_B6lv_a_r + i_load_B6lv_a_i*v_B6lv_n_r + i_load_B6lv_a_r*v_B6lv_a_i - i_load_B6lv_a_r*v_B6lv_n_i - q_B6lv_a
        struct[0].g[166,0] = -i_load_B6lv_b_i*v_B6lv_b_r + i_load_B6lv_b_i*v_B6lv_n_r + i_load_B6lv_b_r*v_B6lv_b_i - i_load_B6lv_b_r*v_B6lv_n_i - q_B6lv_b
        struct[0].g[167,0] = -i_load_B6lv_c_i*v_B6lv_c_r + i_load_B6lv_c_i*v_B6lv_n_r + i_load_B6lv_c_r*v_B6lv_c_i - i_load_B6lv_c_r*v_B6lv_n_i - q_B6lv_c
        struct[0].g[168,0] = i_load_B6lv_a_r + i_load_B6lv_b_r + i_load_B6lv_c_r + i_load_B6lv_n_r
        struct[0].g[169,0] = i_load_B6lv_a_i + i_load_B6lv_b_i + i_load_B6lv_c_i + i_load_B6lv_n_i
        struct[0].g[170,0] = -R_1112*i_l_1112 + v_11 - v_12
        struct[0].g[171,0] = -R_1213*i_l_1213 + v_12 - v_13
        struct[0].g[172,0] = -R_1314*i_l_1314 + v_13 - v_14
        struct[0].g[173,0] = -R_1415*i_l_1415 + v_14 - v_15
        struct[0].g[174,0] = -R_2122*i_l_2122 + v_21 - v_22
        struct[0].g[175,0] = -R_2223*i_l_2223 + v_22 - v_23
        struct[0].g[176,0] = -R_2324*i_l_2324 + v_23 - v_24
        struct[0].g[177,0] = -R_2425*i_l_2425 + v_24 - v_25
        struct[0].g[178,0] = -R_3132*i_l_3132 + v_31 - v_32
        struct[0].g[179,0] = -R_3233*i_l_3233 + v_32 - v_33
        struct[0].g[180,0] = -R_3334*i_l_3334 + v_33 - v_34
        struct[0].g[181,0] = -R_3435*i_l_3435 + v_34 - v_35
        struct[0].g[182,0] = -R_4142*i_l_4142 + v_41 - v_42
        struct[0].g[183,0] = -R_4243*i_l_4243 + v_42 - v_43
        struct[0].g[184,0] = -R_4344*i_l_4344 + v_43 - v_44
        struct[0].g[185,0] = -R_4445*i_l_4445 + v_44 - v_45
        struct[0].g[186,0] = -R_5152*i_l_5152 + v_51 - v_52
        struct[0].g[187,0] = -R_5253*i_l_5253 + v_52 - v_53
        struct[0].g[188,0] = -R_5354*i_l_5354 + v_53 - v_54
        struct[0].g[189,0] = -R_5455*i_l_5455 + v_54 - v_55
        struct[0].g[190,0] = -R_1521*i_l_1521 + v_15 - v_21
        struct[0].g[191,0] = -R_2531*i_l_2531 + v_25 - v_31
        struct[0].g[192,0] = -R_3541*i_l_3541 + v_35 - v_41
        struct[0].g[193,0] = -R_4551*i_l_4551 + v_45 - v_51
        struct[0].g[194,0] = i_11 - i_l_1112
        struct[0].g[195,0] = i_12 + i_l_1112 - i_l_1213
        struct[0].g[196,0] = i_13 + i_l_1213 - i_l_1314
        struct[0].g[197,0] = i_14 + i_l_1314 - i_l_1415
        struct[0].g[198,0] = i_15 + i_l_1415 - i_l_1521
        struct[0].g[199,0] = i_21 + i_l_1521 - i_l_2122
        struct[0].g[200,0] = i_22 + i_l_2122 - i_l_2223
        struct[0].g[201,0] = i_23 + i_l_2223 - i_l_2324
        struct[0].g[202,0] = i_24 + i_l_2324 - i_l_2425
        struct[0].g[203,0] = i_25 + i_l_2425 - i_l_2531
        struct[0].g[204,0] = i_31 + i_l_2531 - i_l_3132
        struct[0].g[205,0] = i_32 + i_l_3132 - i_l_3233
        struct[0].g[206,0] = i_33 + i_l_3233 - i_l_3334
        struct[0].g[207,0] = i_34 + i_l_3334 - i_l_3435
        struct[0].g[208,0] = i_35 + i_l_3435 - i_l_3541
        struct[0].g[209,0] = i_41 + i_l_3541 - i_l_4142
        struct[0].g[210,0] = i_42 + i_l_4142 - i_l_4243
        struct[0].g[211,0] = i_43 + i_l_4243 - i_l_4344
        struct[0].g[212,0] = i_44 + i_l_4344 - i_l_4445
        struct[0].g[213,0] = i_45 + i_l_4445 - i_l_4551
        struct[0].g[214,0] = i_51 + i_l_4551 - i_l_5152
        struct[0].g[215,0] = i_52 + i_l_5152 - i_l_5253
        struct[0].g[216,0] = i_53 + i_l_5253 - i_l_5354
        struct[0].g[217,0] = i_54 + i_l_5354 - i_l_5455
        struct[0].g[218,0] = i_55 + i_l_5455
        struct[0].g[219,0] = i_11*v_11 - p_11
        struct[0].g[220,0] = i_12*v_12 - p_12
        struct[0].g[221,0] = i_14*v_14 - p_14
        struct[0].g[222,0] = i_15*v_15 - p_15
        struct[0].g[223,0] = i_21*v_21 - p_21
        struct[0].g[224,0] = i_22*v_22 - p_22
        struct[0].g[225,0] = i_24*v_24 - p_24
        struct[0].g[226,0] = i_25*v_25 - p_25
        struct[0].g[227,0] = i_31*v_31 - p_31
        struct[0].g[228,0] = i_32*v_32 - p_32
        struct[0].g[229,0] = i_34*v_34 - p_34
        struct[0].g[230,0] = i_35*v_35 - p_35
        struct[0].g[231,0] = i_41*v_41 - p_41
        struct[0].g[232,0] = i_42*v_42 - p_42
        struct[0].g[233,0] = i_44*v_44 - p_44
        struct[0].g[234,0] = i_45*v_45 - p_45
        struct[0].g[235,0] = i_51*v_51 - p_51
        struct[0].g[236,0] = i_52*v_52 - p_52
        struct[0].g[237,0] = i_54*v_54 - p_54
        struct[0].g[238,0] = i_55*v_55 - p_55
        struct[0].g[239,0] = i_13*v_13/3 + p_B2lv_a + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[240,0] = i_13*v_13/3 + p_B2lv_b + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[241,0] = i_13*v_13/3 + p_B2lv_c + Piecewise(np.array([(6*a_B2lv + 6*b_B2lv*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 6*c_B2lv*(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0.0, True)]))/3
        struct[0].g[242,0] = i_23*v_23/3 + p_B3lv_a + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[243,0] = i_23*v_23/3 + p_B3lv_b + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[244,0] = i_23*v_23/3 + p_B3lv_c + Piecewise(np.array([(6*a_B3lv + 6*b_B3lv*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 6*c_B3lv*(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0.0, True)]))/3
        struct[0].g[245,0] = i_33*v_33/3 + p_B4lv_a + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[246,0] = i_33*v_33/3 + p_B4lv_b + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[247,0] = i_33*v_33/3 + p_B4lv_c + Piecewise(np.array([(6*a_B4lv + 6*b_B4lv*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 6*c_B4lv*(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0.0, True)]))/3
        struct[0].g[248,0] = i_43*v_43/3 + p_B5lv_a + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[249,0] = i_43*v_43/3 + p_B5lv_b + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[250,0] = i_43*v_43/3 + p_B5lv_c + Piecewise(np.array([(6*a_B5lv + 6*b_B5lv*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 6*c_B5lv*(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0.0, True)]))/3
        struct[0].g[251,0] = i_53*v_53/3 + p_B6lv_a + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
        struct[0].g[252,0] = i_53*v_53/3 + p_B6lv_b + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
        struct[0].g[253,0] = i_53*v_53/3 + p_B6lv_c + Piecewise(np.array([(6*a_B6lv + 6*b_B6lv*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 6*c_B6lv*(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0.0, True)]))/3
    
    # Outputs:
    if mode == 3:

        struct[0].h[0,0] = i_13*v_13
        struct[0].h[1,0] = v_13
        struct[0].h[2,0] = i_23*v_23
        struct[0].h[3,0] = v_23
        struct[0].h[4,0] = i_33*v_33
        struct[0].h[5,0] = v_33
        struct[0].h[6,0] = i_43*v_43
        struct[0].h[7,0] = v_43
        struct[0].h[8,0] = i_53*v_53
        struct[0].h[9,0] = v_53
    

    if mode == 10:

        struct[0].Fx[0,0] = (-K_r*i_13 - 1)/T_v
        struct[0].Fx[1,1] = (-K_r*i_23 - 1)/T_v
        struct[0].Fx[2,2] = (-K_r*i_33 - 1)/T_v
        struct[0].Fx[3,3] = (-K_r*i_43 - 1)/T_v
        struct[0].Fx[4,4] = (-K_r*i_53 - 1)/T_v

    if mode == 11:

        struct[0].Fy[0,196] = -K_r*v_13/T_v
        struct[0].Fy[1,201] = -K_r*v_23/T_v
        struct[0].Fy[2,206] = -K_r*v_33/T_v
        struct[0].Fy[3,211] = -K_r*v_43/T_v
        struct[0].Fy[4,216] = -K_r*v_53/T_v

        struct[0].Gy[0,0] = -5.75163398692810
        struct[0].Gy[0,1] = -23.0065359477124
        struct[0].Gy[0,6] = 5.75163398692810
        struct[0].Gy[0,7] = 23.0065359477124
        struct[0].Gy[0,40] = 0.249053057297486
        struct[0].Gy[0,41] = 0.996212229189942
        struct[0].Gy[0,44] = -0.249053057297486
        struct[0].Gy[0,45] = -0.996212229189942
        struct[0].Gy[0,130] = 1
        struct[0].Gy[1,0] = 23.0065359477124
        struct[0].Gy[1,1] = -5.75163398692810
        struct[0].Gy[1,6] = -23.0065359477124
        struct[0].Gy[1,7] = 5.75163398692810
        struct[0].Gy[1,40] = -0.996212229189942
        struct[0].Gy[1,41] = 0.249053057297486
        struct[0].Gy[1,44] = 0.996212229189942
        struct[0].Gy[1,45] = -0.249053057297486
        struct[0].Gy[1,131] = 1
        struct[0].Gy[2,2] = -5.75163398692810
        struct[0].Gy[2,3] = -23.0065359477124
        struct[0].Gy[2,6] = 5.75163398692810
        struct[0].Gy[2,7] = 23.0065359477124
        struct[0].Gy[2,40] = -0.249053057297486
        struct[0].Gy[2,41] = -0.996212229189942
        struct[0].Gy[2,42] = 0.249053057297486
        struct[0].Gy[2,43] = 0.996212229189942
        struct[0].Gy[2,132] = 1
        struct[0].Gy[3,2] = 23.0065359477124
        struct[0].Gy[3,3] = -5.75163398692810
        struct[0].Gy[3,6] = -23.0065359477124
        struct[0].Gy[3,7] = 5.75163398692810
        struct[0].Gy[3,40] = 0.996212229189942
        struct[0].Gy[3,41] = -0.249053057297486
        struct[0].Gy[3,42] = -0.996212229189942
        struct[0].Gy[3,43] = 0.249053057297486
        struct[0].Gy[3,133] = 1
        struct[0].Gy[4,4] = -5.75163398692810
        struct[0].Gy[4,5] = -23.0065359477124
        struct[0].Gy[4,6] = 5.75163398692810
        struct[0].Gy[4,7] = 23.0065359477124
        struct[0].Gy[4,42] = -0.249053057297486
        struct[0].Gy[4,43] = -0.996212229189942
        struct[0].Gy[4,44] = 0.249053057297486
        struct[0].Gy[4,45] = 0.996212229189942
        struct[0].Gy[4,134] = 1
        struct[0].Gy[5,4] = 23.0065359477124
        struct[0].Gy[5,5] = -5.75163398692810
        struct[0].Gy[5,6] = -23.0065359477124
        struct[0].Gy[5,7] = 5.75163398692810
        struct[0].Gy[5,42] = 0.996212229189942
        struct[0].Gy[5,43] = -0.249053057297486
        struct[0].Gy[5,44] = -0.996212229189942
        struct[0].Gy[5,45] = 0.249053057297486
        struct[0].Gy[5,135] = 1
        struct[0].Gy[6,0] = 5.75163398692810
        struct[0].Gy[6,1] = 23.0065359477124
        struct[0].Gy[6,2] = 5.75163398692810
        struct[0].Gy[6,3] = 23.0065359477124
        struct[0].Gy[6,4] = 5.75163398692810
        struct[0].Gy[6,5] = 23.0065359477124
        struct[0].Gy[6,6] = -1017.25490196078
        struct[0].Gy[6,7] = -69.0196078431372
        struct[0].Gy[7,0] = -23.0065359477124
        struct[0].Gy[7,1] = 5.75163398692810
        struct[0].Gy[7,2] = -23.0065359477124
        struct[0].Gy[7,3] = 5.75163398692810
        struct[0].Gy[7,4] = -23.0065359477124
        struct[0].Gy[7,5] = 5.75163398692810
        struct[0].Gy[7,6] = 69.0196078431372
        struct[0].Gy[7,7] = -1017.25490196078
        struct[0].Gy[8,8] = -5.75163398692810
        struct[0].Gy[8,9] = -23.0065359477124
        struct[0].Gy[8,14] = 5.75163398692810
        struct[0].Gy[8,15] = 23.0065359477124
        struct[0].Gy[8,46] = 0.249053057297486
        struct[0].Gy[8,47] = 0.996212229189942
        struct[0].Gy[8,50] = -0.249053057297486
        struct[0].Gy[8,51] = -0.996212229189942
        struct[0].Gy[8,138] = 1
        struct[0].Gy[9,8] = 23.0065359477124
        struct[0].Gy[9,9] = -5.75163398692810
        struct[0].Gy[9,14] = -23.0065359477124
        struct[0].Gy[9,15] = 5.75163398692810
        struct[0].Gy[9,46] = -0.996212229189942
        struct[0].Gy[9,47] = 0.249053057297486
        struct[0].Gy[9,50] = 0.996212229189942
        struct[0].Gy[9,51] = -0.249053057297486
        struct[0].Gy[9,139] = 1
        struct[0].Gy[10,10] = -5.75163398692810
        struct[0].Gy[10,11] = -23.0065359477124
        struct[0].Gy[10,14] = 5.75163398692810
        struct[0].Gy[10,15] = 23.0065359477124
        struct[0].Gy[10,46] = -0.249053057297486
        struct[0].Gy[10,47] = -0.996212229189942
        struct[0].Gy[10,48] = 0.249053057297486
        struct[0].Gy[10,49] = 0.996212229189942
        struct[0].Gy[10,140] = 1
        struct[0].Gy[11,10] = 23.0065359477124
        struct[0].Gy[11,11] = -5.75163398692810
        struct[0].Gy[11,14] = -23.0065359477124
        struct[0].Gy[11,15] = 5.75163398692810
        struct[0].Gy[11,46] = 0.996212229189942
        struct[0].Gy[11,47] = -0.249053057297486
        struct[0].Gy[11,48] = -0.996212229189942
        struct[0].Gy[11,49] = 0.249053057297486
        struct[0].Gy[11,141] = 1
        struct[0].Gy[12,12] = -5.75163398692810
        struct[0].Gy[12,13] = -23.0065359477124
        struct[0].Gy[12,14] = 5.75163398692810
        struct[0].Gy[12,15] = 23.0065359477124
        struct[0].Gy[12,48] = -0.249053057297486
        struct[0].Gy[12,49] = -0.996212229189942
        struct[0].Gy[12,50] = 0.249053057297486
        struct[0].Gy[12,51] = 0.996212229189942
        struct[0].Gy[12,142] = 1
        struct[0].Gy[13,12] = 23.0065359477124
        struct[0].Gy[13,13] = -5.75163398692810
        struct[0].Gy[13,14] = -23.0065359477124
        struct[0].Gy[13,15] = 5.75163398692810
        struct[0].Gy[13,48] = 0.996212229189942
        struct[0].Gy[13,49] = -0.249053057297486
        struct[0].Gy[13,50] = -0.996212229189942
        struct[0].Gy[13,51] = 0.249053057297486
        struct[0].Gy[13,143] = 1
        struct[0].Gy[14,8] = 5.75163398692810
        struct[0].Gy[14,9] = 23.0065359477124
        struct[0].Gy[14,10] = 5.75163398692810
        struct[0].Gy[14,11] = 23.0065359477124
        struct[0].Gy[14,12] = 5.75163398692810
        struct[0].Gy[14,13] = 23.0065359477124
        struct[0].Gy[14,14] = -1017.25490196078
        struct[0].Gy[14,15] = -69.0196078431372
        struct[0].Gy[15,8] = -23.0065359477124
        struct[0].Gy[15,9] = 5.75163398692810
        struct[0].Gy[15,10] = -23.0065359477124
        struct[0].Gy[15,11] = 5.75163398692810
        struct[0].Gy[15,12] = -23.0065359477124
        struct[0].Gy[15,13] = 5.75163398692810
        struct[0].Gy[15,14] = 69.0196078431372
        struct[0].Gy[15,15] = -1017.25490196078
        struct[0].Gy[16,16] = -5.75163398692810
        struct[0].Gy[16,17] = -23.0065359477124
        struct[0].Gy[16,22] = 5.75163398692810
        struct[0].Gy[16,23] = 23.0065359477124
        struct[0].Gy[16,52] = 0.249053057297486
        struct[0].Gy[16,53] = 0.996212229189942
        struct[0].Gy[16,56] = -0.249053057297486
        struct[0].Gy[16,57] = -0.996212229189942
        struct[0].Gy[16,146] = 1
        struct[0].Gy[17,16] = 23.0065359477124
        struct[0].Gy[17,17] = -5.75163398692810
        struct[0].Gy[17,22] = -23.0065359477124
        struct[0].Gy[17,23] = 5.75163398692810
        struct[0].Gy[17,52] = -0.996212229189942
        struct[0].Gy[17,53] = 0.249053057297486
        struct[0].Gy[17,56] = 0.996212229189942
        struct[0].Gy[17,57] = -0.249053057297486
        struct[0].Gy[17,147] = 1
        struct[0].Gy[18,18] = -5.75163398692810
        struct[0].Gy[18,19] = -23.0065359477124
        struct[0].Gy[18,22] = 5.75163398692810
        struct[0].Gy[18,23] = 23.0065359477124
        struct[0].Gy[18,52] = -0.249053057297486
        struct[0].Gy[18,53] = -0.996212229189942
        struct[0].Gy[18,54] = 0.249053057297486
        struct[0].Gy[18,55] = 0.996212229189942
        struct[0].Gy[18,148] = 1
        struct[0].Gy[19,18] = 23.0065359477124
        struct[0].Gy[19,19] = -5.75163398692810
        struct[0].Gy[19,22] = -23.0065359477124
        struct[0].Gy[19,23] = 5.75163398692810
        struct[0].Gy[19,52] = 0.996212229189942
        struct[0].Gy[19,53] = -0.249053057297486
        struct[0].Gy[19,54] = -0.996212229189942
        struct[0].Gy[19,55] = 0.249053057297486
        struct[0].Gy[19,149] = 1
        struct[0].Gy[20,20] = -5.75163398692810
        struct[0].Gy[20,21] = -23.0065359477124
        struct[0].Gy[20,22] = 5.75163398692810
        struct[0].Gy[20,23] = 23.0065359477124
        struct[0].Gy[20,54] = -0.249053057297486
        struct[0].Gy[20,55] = -0.996212229189942
        struct[0].Gy[20,56] = 0.249053057297486
        struct[0].Gy[20,57] = 0.996212229189942
        struct[0].Gy[20,150] = 1
        struct[0].Gy[21,20] = 23.0065359477124
        struct[0].Gy[21,21] = -5.75163398692810
        struct[0].Gy[21,22] = -23.0065359477124
        struct[0].Gy[21,23] = 5.75163398692810
        struct[0].Gy[21,54] = 0.996212229189942
        struct[0].Gy[21,55] = -0.249053057297486
        struct[0].Gy[21,56] = -0.996212229189942
        struct[0].Gy[21,57] = 0.249053057297486
        struct[0].Gy[21,151] = 1
        struct[0].Gy[22,16] = 5.75163398692810
        struct[0].Gy[22,17] = 23.0065359477124
        struct[0].Gy[22,18] = 5.75163398692810
        struct[0].Gy[22,19] = 23.0065359477124
        struct[0].Gy[22,20] = 5.75163398692810
        struct[0].Gy[22,21] = 23.0065359477124
        struct[0].Gy[22,22] = -1017.25490196078
        struct[0].Gy[22,23] = -69.0196078431372
        struct[0].Gy[23,16] = -23.0065359477124
        struct[0].Gy[23,17] = 5.75163398692810
        struct[0].Gy[23,18] = -23.0065359477124
        struct[0].Gy[23,19] = 5.75163398692810
        struct[0].Gy[23,20] = -23.0065359477124
        struct[0].Gy[23,21] = 5.75163398692810
        struct[0].Gy[23,22] = 69.0196078431372
        struct[0].Gy[23,23] = -1017.25490196078
        struct[0].Gy[24,24] = -5.75163398692810
        struct[0].Gy[24,25] = -23.0065359477124
        struct[0].Gy[24,30] = 5.75163398692810
        struct[0].Gy[24,31] = 23.0065359477124
        struct[0].Gy[24,58] = 0.249053057297486
        struct[0].Gy[24,59] = 0.996212229189942
        struct[0].Gy[24,62] = -0.249053057297486
        struct[0].Gy[24,63] = -0.996212229189942
        struct[0].Gy[24,154] = 1
        struct[0].Gy[25,24] = 23.0065359477124
        struct[0].Gy[25,25] = -5.75163398692810
        struct[0].Gy[25,30] = -23.0065359477124
        struct[0].Gy[25,31] = 5.75163398692810
        struct[0].Gy[25,58] = -0.996212229189942
        struct[0].Gy[25,59] = 0.249053057297486
        struct[0].Gy[25,62] = 0.996212229189942
        struct[0].Gy[25,63] = -0.249053057297486
        struct[0].Gy[25,155] = 1
        struct[0].Gy[26,26] = -5.75163398692810
        struct[0].Gy[26,27] = -23.0065359477124
        struct[0].Gy[26,30] = 5.75163398692810
        struct[0].Gy[26,31] = 23.0065359477124
        struct[0].Gy[26,58] = -0.249053057297486
        struct[0].Gy[26,59] = -0.996212229189942
        struct[0].Gy[26,60] = 0.249053057297486
        struct[0].Gy[26,61] = 0.996212229189942
        struct[0].Gy[26,156] = 1
        struct[0].Gy[27,26] = 23.0065359477124
        struct[0].Gy[27,27] = -5.75163398692810
        struct[0].Gy[27,30] = -23.0065359477124
        struct[0].Gy[27,31] = 5.75163398692810
        struct[0].Gy[27,58] = 0.996212229189942
        struct[0].Gy[27,59] = -0.249053057297486
        struct[0].Gy[27,60] = -0.996212229189942
        struct[0].Gy[27,61] = 0.249053057297486
        struct[0].Gy[27,157] = 1
        struct[0].Gy[28,28] = -5.75163398692810
        struct[0].Gy[28,29] = -23.0065359477124
        struct[0].Gy[28,30] = 5.75163398692810
        struct[0].Gy[28,31] = 23.0065359477124
        struct[0].Gy[28,60] = -0.249053057297486
        struct[0].Gy[28,61] = -0.996212229189942
        struct[0].Gy[28,62] = 0.249053057297486
        struct[0].Gy[28,63] = 0.996212229189942
        struct[0].Gy[28,158] = 1
        struct[0].Gy[29,28] = 23.0065359477124
        struct[0].Gy[29,29] = -5.75163398692810
        struct[0].Gy[29,30] = -23.0065359477124
        struct[0].Gy[29,31] = 5.75163398692810
        struct[0].Gy[29,60] = 0.996212229189942
        struct[0].Gy[29,61] = -0.249053057297486
        struct[0].Gy[29,62] = -0.996212229189942
        struct[0].Gy[29,63] = 0.249053057297486
        struct[0].Gy[29,159] = 1
        struct[0].Gy[30,24] = 5.75163398692810
        struct[0].Gy[30,25] = 23.0065359477124
        struct[0].Gy[30,26] = 5.75163398692810
        struct[0].Gy[30,27] = 23.0065359477124
        struct[0].Gy[30,28] = 5.75163398692810
        struct[0].Gy[30,29] = 23.0065359477124
        struct[0].Gy[30,30] = -1017.25490196078
        struct[0].Gy[30,31] = -69.0196078431372
        struct[0].Gy[31,24] = -23.0065359477124
        struct[0].Gy[31,25] = 5.75163398692810
        struct[0].Gy[31,26] = -23.0065359477124
        struct[0].Gy[31,27] = 5.75163398692810
        struct[0].Gy[31,28] = -23.0065359477124
        struct[0].Gy[31,29] = 5.75163398692810
        struct[0].Gy[31,30] = 69.0196078431372
        struct[0].Gy[31,31] = -1017.25490196078
        struct[0].Gy[32,32] = -5.75163398692810
        struct[0].Gy[32,33] = -23.0065359477124
        struct[0].Gy[32,38] = 5.75163398692810
        struct[0].Gy[32,39] = 23.0065359477124
        struct[0].Gy[32,64] = 0.249053057297486
        struct[0].Gy[32,65] = 0.996212229189942
        struct[0].Gy[32,68] = -0.249053057297486
        struct[0].Gy[32,69] = -0.996212229189942
        struct[0].Gy[32,162] = 1
        struct[0].Gy[33,32] = 23.0065359477124
        struct[0].Gy[33,33] = -5.75163398692810
        struct[0].Gy[33,38] = -23.0065359477124
        struct[0].Gy[33,39] = 5.75163398692810
        struct[0].Gy[33,64] = -0.996212229189942
        struct[0].Gy[33,65] = 0.249053057297486
        struct[0].Gy[33,68] = 0.996212229189942
        struct[0].Gy[33,69] = -0.249053057297486
        struct[0].Gy[33,163] = 1
        struct[0].Gy[34,34] = -5.75163398692810
        struct[0].Gy[34,35] = -23.0065359477124
        struct[0].Gy[34,38] = 5.75163398692810
        struct[0].Gy[34,39] = 23.0065359477124
        struct[0].Gy[34,64] = -0.249053057297486
        struct[0].Gy[34,65] = -0.996212229189942
        struct[0].Gy[34,66] = 0.249053057297486
        struct[0].Gy[34,67] = 0.996212229189942
        struct[0].Gy[34,164] = 1
        struct[0].Gy[35,34] = 23.0065359477124
        struct[0].Gy[35,35] = -5.75163398692810
        struct[0].Gy[35,38] = -23.0065359477124
        struct[0].Gy[35,39] = 5.75163398692810
        struct[0].Gy[35,64] = 0.996212229189942
        struct[0].Gy[35,65] = -0.249053057297486
        struct[0].Gy[35,66] = -0.996212229189942
        struct[0].Gy[35,67] = 0.249053057297486
        struct[0].Gy[35,165] = 1
        struct[0].Gy[36,36] = -5.75163398692810
        struct[0].Gy[36,37] = -23.0065359477124
        struct[0].Gy[36,38] = 5.75163398692810
        struct[0].Gy[36,39] = 23.0065359477124
        struct[0].Gy[36,66] = -0.249053057297486
        struct[0].Gy[36,67] = -0.996212229189942
        struct[0].Gy[36,68] = 0.249053057297486
        struct[0].Gy[36,69] = 0.996212229189942
        struct[0].Gy[36,166] = 1
        struct[0].Gy[37,36] = 23.0065359477124
        struct[0].Gy[37,37] = -5.75163398692810
        struct[0].Gy[37,38] = -23.0065359477124
        struct[0].Gy[37,39] = 5.75163398692810
        struct[0].Gy[37,66] = 0.996212229189942
        struct[0].Gy[37,67] = -0.249053057297486
        struct[0].Gy[37,68] = -0.996212229189942
        struct[0].Gy[37,69] = 0.249053057297486
        struct[0].Gy[37,167] = 1
        struct[0].Gy[38,32] = 5.75163398692810
        struct[0].Gy[38,33] = 23.0065359477124
        struct[0].Gy[38,34] = 5.75163398692810
        struct[0].Gy[38,35] = 23.0065359477124
        struct[0].Gy[38,36] = 5.75163398692810
        struct[0].Gy[38,37] = 23.0065359477124
        struct[0].Gy[38,38] = -1017.25490196078
        struct[0].Gy[38,39] = -69.0196078431372
        struct[0].Gy[39,32] = -23.0065359477124
        struct[0].Gy[39,33] = 5.75163398692810
        struct[0].Gy[39,34] = -23.0065359477124
        struct[0].Gy[39,35] = 5.75163398692810
        struct[0].Gy[39,36] = -23.0065359477124
        struct[0].Gy[39,37] = 5.75163398692810
        struct[0].Gy[39,38] = 69.0196078431372
        struct[0].Gy[39,39] = -1017.25490196078
        struct[0].Gy[40,0] = 0.249053057297486
        struct[0].Gy[40,1] = 0.996212229189942
        struct[0].Gy[40,2] = -0.249053057297486
        struct[0].Gy[40,3] = -0.996212229189942
        struct[0].Gy[40,40] = -2.23667465123725
        struct[0].Gy[40,41] = -1.28353302446119
        struct[0].Gy[40,42] = 0.643671749092996
        struct[0].Gy[40,43] = 0.385473430243205
        struct[0].Gy[40,44] = 0.643671749092997
        struct[0].Gy[40,45] = 0.385473430243205
        struct[0].Gy[40,46] = 1.10755301189314
        struct[0].Gy[40,47] = 0.598820527961361
        struct[0].Gy[40,48] = -0.316443717683753
        struct[0].Gy[40,49] = -0.171091579417532
        struct[0].Gy[40,50] = -0.316443717683753
        struct[0].Gy[40,51] = -0.171091579417532
        struct[0].Gy[41,0] = -0.996212229189942
        struct[0].Gy[41,1] = 0.249053057297486
        struct[0].Gy[41,2] = 0.996212229189942
        struct[0].Gy[41,3] = -0.249053057297486
        struct[0].Gy[41,40] = 1.28353302446119
        struct[0].Gy[41,41] = -2.23667465123725
        struct[0].Gy[41,42] = -0.385473430243205
        struct[0].Gy[41,43] = 0.643671749092996
        struct[0].Gy[41,44] = -0.385473430243205
        struct[0].Gy[41,45] = 0.643671749092997
        struct[0].Gy[41,46] = -0.598820527961361
        struct[0].Gy[41,47] = 1.10755301189314
        struct[0].Gy[41,48] = 0.171091579417532
        struct[0].Gy[41,49] = -0.316443717683753
        struct[0].Gy[41,50] = 0.171091579417532
        struct[0].Gy[41,51] = -0.316443717683753
        struct[0].Gy[42,2] = 0.249053057297486
        struct[0].Gy[42,3] = 0.996212229189942
        struct[0].Gy[42,4] = -0.249053057297486
        struct[0].Gy[42,5] = -0.996212229189942
        struct[0].Gy[42,40] = 0.643671749092996
        struct[0].Gy[42,41] = 0.385473430243205
        struct[0].Gy[42,42] = -2.23667465123725
        struct[0].Gy[42,43] = -1.28353302446119
        struct[0].Gy[42,44] = 0.643671749092997
        struct[0].Gy[42,45] = 0.385473430243204
        struct[0].Gy[42,46] = -0.316443717683753
        struct[0].Gy[42,47] = -0.171091579417532
        struct[0].Gy[42,48] = 1.10755301189314
        struct[0].Gy[42,49] = 0.598820527961360
        struct[0].Gy[42,50] = -0.316443717683753
        struct[0].Gy[42,51] = -0.171091579417531
        struct[0].Gy[43,2] = -0.996212229189942
        struct[0].Gy[43,3] = 0.249053057297486
        struct[0].Gy[43,4] = 0.996212229189942
        struct[0].Gy[43,5] = -0.249053057297486
        struct[0].Gy[43,40] = -0.385473430243205
        struct[0].Gy[43,41] = 0.643671749092996
        struct[0].Gy[43,42] = 1.28353302446119
        struct[0].Gy[43,43] = -2.23667465123725
        struct[0].Gy[43,44] = -0.385473430243204
        struct[0].Gy[43,45] = 0.643671749092997
        struct[0].Gy[43,46] = 0.171091579417532
        struct[0].Gy[43,47] = -0.316443717683753
        struct[0].Gy[43,48] = -0.598820527961360
        struct[0].Gy[43,49] = 1.10755301189314
        struct[0].Gy[43,50] = 0.171091579417531
        struct[0].Gy[43,51] = -0.316443717683753
        struct[0].Gy[44,0] = -0.249053057297486
        struct[0].Gy[44,1] = -0.996212229189942
        struct[0].Gy[44,4] = 0.249053057297486
        struct[0].Gy[44,5] = 0.996212229189942
        struct[0].Gy[44,40] = 0.643671749092997
        struct[0].Gy[44,41] = 0.385473430243205
        struct[0].Gy[44,42] = 0.643671749092997
        struct[0].Gy[44,43] = 0.385473430243204
        struct[0].Gy[44,44] = -2.23667465123725
        struct[0].Gy[44,45] = -1.28353302446119
        struct[0].Gy[44,46] = -0.316443717683753
        struct[0].Gy[44,47] = -0.171091579417532
        struct[0].Gy[44,48] = -0.316443717683753
        struct[0].Gy[44,49] = -0.171091579417531
        struct[0].Gy[44,50] = 1.10755301189314
        struct[0].Gy[44,51] = 0.598820527961360
        struct[0].Gy[45,0] = 0.996212229189942
        struct[0].Gy[45,1] = -0.249053057297486
        struct[0].Gy[45,4] = -0.996212229189942
        struct[0].Gy[45,5] = 0.249053057297486
        struct[0].Gy[45,40] = -0.385473430243205
        struct[0].Gy[45,41] = 0.643671749092997
        struct[0].Gy[45,42] = -0.385473430243204
        struct[0].Gy[45,43] = 0.643671749092997
        struct[0].Gy[45,44] = 1.28353302446119
        struct[0].Gy[45,45] = -2.23667465123725
        struct[0].Gy[45,46] = 0.171091579417532
        struct[0].Gy[45,47] = -0.316443717683753
        struct[0].Gy[45,48] = 0.171091579417531
        struct[0].Gy[45,49] = -0.316443717683753
        struct[0].Gy[45,50] = -0.598820527961360
        struct[0].Gy[45,51] = 1.10755301189314
        struct[0].Gy[46,8] = 0.249053057297486
        struct[0].Gy[46,9] = 0.996212229189942
        struct[0].Gy[46,10] = -0.249053057297486
        struct[0].Gy[46,11] = -0.996212229189942
        struct[0].Gy[46,40] = 1.10755301189314
        struct[0].Gy[46,41] = 0.598820527961361
        struct[0].Gy[46,42] = -0.316443717683753
        struct[0].Gy[46,43] = -0.171091579417532
        struct[0].Gy[46,44] = -0.316443717683753
        struct[0].Gy[46,45] = -0.171091579417532
        struct[0].Gy[46,46] = -2.23667465123725
        struct[0].Gy[46,47] = -1.28353302446119
        struct[0].Gy[46,48] = 0.643671749092996
        struct[0].Gy[46,49] = 0.385473430243205
        struct[0].Gy[46,50] = 0.643671749092997
        struct[0].Gy[46,51] = 0.385473430243205
        struct[0].Gy[46,52] = 1.10755301189314
        struct[0].Gy[46,53] = 0.598820527961361
        struct[0].Gy[46,54] = -0.316443717683753
        struct[0].Gy[46,55] = -0.171091579417532
        struct[0].Gy[46,56] = -0.316443717683753
        struct[0].Gy[46,57] = -0.171091579417532
        struct[0].Gy[47,8] = -0.996212229189942
        struct[0].Gy[47,9] = 0.249053057297486
        struct[0].Gy[47,10] = 0.996212229189942
        struct[0].Gy[47,11] = -0.249053057297486
        struct[0].Gy[47,40] = -0.598820527961361
        struct[0].Gy[47,41] = 1.10755301189314
        struct[0].Gy[47,42] = 0.171091579417532
        struct[0].Gy[47,43] = -0.316443717683753
        struct[0].Gy[47,44] = 0.171091579417532
        struct[0].Gy[47,45] = -0.316443717683753
        struct[0].Gy[47,46] = 1.28353302446119
        struct[0].Gy[47,47] = -2.23667465123725
        struct[0].Gy[47,48] = -0.385473430243205
        struct[0].Gy[47,49] = 0.643671749092996
        struct[0].Gy[47,50] = -0.385473430243205
        struct[0].Gy[47,51] = 0.643671749092997
        struct[0].Gy[47,52] = -0.598820527961361
        struct[0].Gy[47,53] = 1.10755301189314
        struct[0].Gy[47,54] = 0.171091579417532
        struct[0].Gy[47,55] = -0.316443717683753
        struct[0].Gy[47,56] = 0.171091579417532
        struct[0].Gy[47,57] = -0.316443717683753
        struct[0].Gy[48,10] = 0.249053057297486
        struct[0].Gy[48,11] = 0.996212229189942
        struct[0].Gy[48,12] = -0.249053057297486
        struct[0].Gy[48,13] = -0.996212229189942
        struct[0].Gy[48,40] = -0.316443717683753
        struct[0].Gy[48,41] = -0.171091579417532
        struct[0].Gy[48,42] = 1.10755301189314
        struct[0].Gy[48,43] = 0.598820527961360
        struct[0].Gy[48,44] = -0.316443717683753
        struct[0].Gy[48,45] = -0.171091579417531
        struct[0].Gy[48,46] = 0.643671749092996
        struct[0].Gy[48,47] = 0.385473430243205
        struct[0].Gy[48,48] = -2.23667465123725
        struct[0].Gy[48,49] = -1.28353302446119
        struct[0].Gy[48,50] = 0.643671749092997
        struct[0].Gy[48,51] = 0.385473430243204
        struct[0].Gy[48,52] = -0.316443717683753
        struct[0].Gy[48,53] = -0.171091579417532
        struct[0].Gy[48,54] = 1.10755301189314
        struct[0].Gy[48,55] = 0.598820527961360
        struct[0].Gy[48,56] = -0.316443717683753
        struct[0].Gy[48,57] = -0.171091579417531
        struct[0].Gy[49,10] = -0.996212229189942
        struct[0].Gy[49,11] = 0.249053057297486
        struct[0].Gy[49,12] = 0.996212229189942
        struct[0].Gy[49,13] = -0.249053057297486
        struct[0].Gy[49,40] = 0.171091579417532
        struct[0].Gy[49,41] = -0.316443717683753
        struct[0].Gy[49,42] = -0.598820527961360
        struct[0].Gy[49,43] = 1.10755301189314
        struct[0].Gy[49,44] = 0.171091579417531
        struct[0].Gy[49,45] = -0.316443717683753
        struct[0].Gy[49,46] = -0.385473430243205
        struct[0].Gy[49,47] = 0.643671749092996
        struct[0].Gy[49,48] = 1.28353302446119
        struct[0].Gy[49,49] = -2.23667465123725
        struct[0].Gy[49,50] = -0.385473430243204
        struct[0].Gy[49,51] = 0.643671749092997
        struct[0].Gy[49,52] = 0.171091579417532
        struct[0].Gy[49,53] = -0.316443717683753
        struct[0].Gy[49,54] = -0.598820527961360
        struct[0].Gy[49,55] = 1.10755301189314
        struct[0].Gy[49,56] = 0.171091579417531
        struct[0].Gy[49,57] = -0.316443717683753
        struct[0].Gy[50,8] = -0.249053057297486
        struct[0].Gy[50,9] = -0.996212229189942
        struct[0].Gy[50,12] = 0.249053057297486
        struct[0].Gy[50,13] = 0.996212229189942
        struct[0].Gy[50,40] = -0.316443717683753
        struct[0].Gy[50,41] = -0.171091579417532
        struct[0].Gy[50,42] = -0.316443717683753
        struct[0].Gy[50,43] = -0.171091579417531
        struct[0].Gy[50,44] = 1.10755301189314
        struct[0].Gy[50,45] = 0.598820527961360
        struct[0].Gy[50,46] = 0.643671749092997
        struct[0].Gy[50,47] = 0.385473430243205
        struct[0].Gy[50,48] = 0.643671749092997
        struct[0].Gy[50,49] = 0.385473430243204
        struct[0].Gy[50,50] = -2.23667465123725
        struct[0].Gy[50,51] = -1.28353302446119
        struct[0].Gy[50,52] = -0.316443717683753
        struct[0].Gy[50,53] = -0.171091579417532
        struct[0].Gy[50,54] = -0.316443717683753
        struct[0].Gy[50,55] = -0.171091579417531
        struct[0].Gy[50,56] = 1.10755301189314
        struct[0].Gy[50,57] = 0.598820527961360
        struct[0].Gy[51,8] = 0.996212229189942
        struct[0].Gy[51,9] = -0.249053057297486
        struct[0].Gy[51,12] = -0.996212229189942
        struct[0].Gy[51,13] = 0.249053057297486
        struct[0].Gy[51,40] = 0.171091579417532
        struct[0].Gy[51,41] = -0.316443717683753
        struct[0].Gy[51,42] = 0.171091579417531
        struct[0].Gy[51,43] = -0.316443717683753
        struct[0].Gy[51,44] = -0.598820527961360
        struct[0].Gy[51,45] = 1.10755301189314
        struct[0].Gy[51,46] = -0.385473430243205
        struct[0].Gy[51,47] = 0.643671749092997
        struct[0].Gy[51,48] = -0.385473430243204
        struct[0].Gy[51,49] = 0.643671749092997
        struct[0].Gy[51,50] = 1.28353302446119
        struct[0].Gy[51,51] = -2.23667465123725
        struct[0].Gy[51,52] = 0.171091579417532
        struct[0].Gy[51,53] = -0.316443717683753
        struct[0].Gy[51,54] = 0.171091579417531
        struct[0].Gy[51,55] = -0.316443717683753
        struct[0].Gy[51,56] = -0.598820527961360
        struct[0].Gy[51,57] = 1.10755301189314
        struct[0].Gy[52,16] = 0.249053057297486
        struct[0].Gy[52,17] = 0.996212229189942
        struct[0].Gy[52,18] = -0.249053057297486
        struct[0].Gy[52,19] = -0.996212229189942
        struct[0].Gy[52,46] = 1.10755301189314
        struct[0].Gy[52,47] = 0.598820527961361
        struct[0].Gy[52,48] = -0.316443717683753
        struct[0].Gy[52,49] = -0.171091579417532
        struct[0].Gy[52,50] = -0.316443717683753
        struct[0].Gy[52,51] = -0.171091579417532
        struct[0].Gy[52,52] = -1.12912163934412
        struct[0].Gy[52,53] = -0.684903767132556
        struct[0].Gy[52,54] = 0.327228031409243
        struct[0].Gy[52,55] = 0.214305342572583
        struct[0].Gy[52,56] = 0.327228031409244
        struct[0].Gy[52,57] = 0.214305342572583
        struct[0].Gy[53,16] = -0.996212229189942
        struct[0].Gy[53,17] = 0.249053057297486
        struct[0].Gy[53,18] = 0.996212229189942
        struct[0].Gy[53,19] = -0.249053057297486
        struct[0].Gy[53,46] = -0.598820527961361
        struct[0].Gy[53,47] = 1.10755301189314
        struct[0].Gy[53,48] = 0.171091579417532
        struct[0].Gy[53,49] = -0.316443717683753
        struct[0].Gy[53,50] = 0.171091579417532
        struct[0].Gy[53,51] = -0.316443717683753
        struct[0].Gy[53,52] = 0.684903767132556
        struct[0].Gy[53,53] = -1.12912163934412
        struct[0].Gy[53,54] = -0.214305342572583
        struct[0].Gy[53,55] = 0.327228031409243
        struct[0].Gy[53,56] = -0.214305342572583
        struct[0].Gy[53,57] = 0.327228031409244
        struct[0].Gy[54,18] = 0.249053057297486
        struct[0].Gy[54,19] = 0.996212229189942
        struct[0].Gy[54,20] = -0.249053057297486
        struct[0].Gy[54,21] = -0.996212229189942
        struct[0].Gy[54,46] = -0.316443717683753
        struct[0].Gy[54,47] = -0.171091579417532
        struct[0].Gy[54,48] = 1.10755301189314
        struct[0].Gy[54,49] = 0.598820527961360
        struct[0].Gy[54,50] = -0.316443717683753
        struct[0].Gy[54,51] = -0.171091579417531
        struct[0].Gy[54,52] = 0.327228031409243
        struct[0].Gy[54,53] = 0.214305342572583
        struct[0].Gy[54,54] = -1.12912163934412
        struct[0].Gy[54,55] = -0.684903767132556
        struct[0].Gy[54,56] = 0.327228031409244
        struct[0].Gy[54,57] = 0.214305342572582
        struct[0].Gy[55,18] = -0.996212229189942
        struct[0].Gy[55,19] = 0.249053057297486
        struct[0].Gy[55,20] = 0.996212229189942
        struct[0].Gy[55,21] = -0.249053057297486
        struct[0].Gy[55,46] = 0.171091579417532
        struct[0].Gy[55,47] = -0.316443717683753
        struct[0].Gy[55,48] = -0.598820527961360
        struct[0].Gy[55,49] = 1.10755301189314
        struct[0].Gy[55,50] = 0.171091579417531
        struct[0].Gy[55,51] = -0.316443717683753
        struct[0].Gy[55,52] = -0.214305342572583
        struct[0].Gy[55,53] = 0.327228031409243
        struct[0].Gy[55,54] = 0.684903767132556
        struct[0].Gy[55,55] = -1.12912163934412
        struct[0].Gy[55,56] = -0.214305342572582
        struct[0].Gy[55,57] = 0.327228031409244
        struct[0].Gy[56,16] = -0.249053057297486
        struct[0].Gy[56,17] = -0.996212229189942
        struct[0].Gy[56,20] = 0.249053057297486
        struct[0].Gy[56,21] = 0.996212229189942
        struct[0].Gy[56,46] = -0.316443717683753
        struct[0].Gy[56,47] = -0.171091579417532
        struct[0].Gy[56,48] = -0.316443717683753
        struct[0].Gy[56,49] = -0.171091579417531
        struct[0].Gy[56,50] = 1.10755301189314
        struct[0].Gy[56,51] = 0.598820527961360
        struct[0].Gy[56,52] = 0.327228031409243
        struct[0].Gy[56,53] = 0.214305342572583
        struct[0].Gy[56,54] = 0.327228031409244
        struct[0].Gy[56,55] = 0.214305342572582
        struct[0].Gy[56,56] = -1.12912163934412
        struct[0].Gy[56,57] = -0.684903767132556
        struct[0].Gy[57,16] = 0.996212229189942
        struct[0].Gy[57,17] = -0.249053057297486
        struct[0].Gy[57,20] = -0.996212229189942
        struct[0].Gy[57,21] = 0.249053057297486
        struct[0].Gy[57,46] = 0.171091579417532
        struct[0].Gy[57,47] = -0.316443717683753
        struct[0].Gy[57,48] = 0.171091579417531
        struct[0].Gy[57,49] = -0.316443717683753
        struct[0].Gy[57,50] = -0.598820527961360
        struct[0].Gy[57,51] = 1.10755301189314
        struct[0].Gy[57,52] = -0.214305342572583
        struct[0].Gy[57,53] = 0.327228031409243
        struct[0].Gy[57,54] = -0.214305342572582
        struct[0].Gy[57,55] = 0.327228031409244
        struct[0].Gy[57,56] = 0.684903767132556
        struct[0].Gy[57,57] = -1.12912163934412
        struct[0].Gy[58,24] = 0.249053057297486
        struct[0].Gy[58,25] = 0.996212229189942
        struct[0].Gy[58,26] = -0.249053057297486
        struct[0].Gy[58,27] = -0.996212229189942
        struct[0].Gy[58,58] = -1.12912163934412
        struct[0].Gy[58,59] = -0.684903767132556
        struct[0].Gy[58,60] = 0.327228031409243
        struct[0].Gy[58,61] = 0.214305342572583
        struct[0].Gy[58,62] = 0.327228031409244
        struct[0].Gy[58,63] = 0.214305342572583
        struct[0].Gy[58,64] = 1.10755301189314
        struct[0].Gy[58,65] = 0.598820527961361
        struct[0].Gy[58,66] = -0.316443717683753
        struct[0].Gy[58,67] = -0.171091579417532
        struct[0].Gy[58,68] = -0.316443717683753
        struct[0].Gy[58,69] = -0.171091579417532
        struct[0].Gy[59,24] = -0.996212229189942
        struct[0].Gy[59,25] = 0.249053057297486
        struct[0].Gy[59,26] = 0.996212229189942
        struct[0].Gy[59,27] = -0.249053057297486
        struct[0].Gy[59,58] = 0.684903767132556
        struct[0].Gy[59,59] = -1.12912163934412
        struct[0].Gy[59,60] = -0.214305342572583
        struct[0].Gy[59,61] = 0.327228031409243
        struct[0].Gy[59,62] = -0.214305342572583
        struct[0].Gy[59,63] = 0.327228031409244
        struct[0].Gy[59,64] = -0.598820527961361
        struct[0].Gy[59,65] = 1.10755301189314
        struct[0].Gy[59,66] = 0.171091579417532
        struct[0].Gy[59,67] = -0.316443717683753
        struct[0].Gy[59,68] = 0.171091579417532
        struct[0].Gy[59,69] = -0.316443717683753
        struct[0].Gy[60,26] = 0.249053057297486
        struct[0].Gy[60,27] = 0.996212229189942
        struct[0].Gy[60,28] = -0.249053057297486
        struct[0].Gy[60,29] = -0.996212229189942
        struct[0].Gy[60,58] = 0.327228031409243
        struct[0].Gy[60,59] = 0.214305342572583
        struct[0].Gy[60,60] = -1.12912163934412
        struct[0].Gy[60,61] = -0.684903767132556
        struct[0].Gy[60,62] = 0.327228031409244
        struct[0].Gy[60,63] = 0.214305342572582
        struct[0].Gy[60,64] = -0.316443717683753
        struct[0].Gy[60,65] = -0.171091579417532
        struct[0].Gy[60,66] = 1.10755301189314
        struct[0].Gy[60,67] = 0.598820527961360
        struct[0].Gy[60,68] = -0.316443717683753
        struct[0].Gy[60,69] = -0.171091579417531
        struct[0].Gy[61,26] = -0.996212229189942
        struct[0].Gy[61,27] = 0.249053057297486
        struct[0].Gy[61,28] = 0.996212229189942
        struct[0].Gy[61,29] = -0.249053057297486
        struct[0].Gy[61,58] = -0.214305342572583
        struct[0].Gy[61,59] = 0.327228031409243
        struct[0].Gy[61,60] = 0.684903767132556
        struct[0].Gy[61,61] = -1.12912163934412
        struct[0].Gy[61,62] = -0.214305342572582
        struct[0].Gy[61,63] = 0.327228031409244
        struct[0].Gy[61,64] = 0.171091579417532
        struct[0].Gy[61,65] = -0.316443717683753
        struct[0].Gy[61,66] = -0.598820527961360
        struct[0].Gy[61,67] = 1.10755301189314
        struct[0].Gy[61,68] = 0.171091579417531
        struct[0].Gy[61,69] = -0.316443717683753
        struct[0].Gy[62,24] = -0.249053057297486
        struct[0].Gy[62,25] = -0.996212229189942
        struct[0].Gy[62,28] = 0.249053057297486
        struct[0].Gy[62,29] = 0.996212229189942
        struct[0].Gy[62,58] = 0.327228031409243
        struct[0].Gy[62,59] = 0.214305342572583
        struct[0].Gy[62,60] = 0.327228031409244
        struct[0].Gy[62,61] = 0.214305342572582
        struct[0].Gy[62,62] = -1.12912163934412
        struct[0].Gy[62,63] = -0.684903767132556
        struct[0].Gy[62,64] = -0.316443717683753
        struct[0].Gy[62,65] = -0.171091579417532
        struct[0].Gy[62,66] = -0.316443717683753
        struct[0].Gy[62,67] = -0.171091579417531
        struct[0].Gy[62,68] = 1.10755301189314
        struct[0].Gy[62,69] = 0.598820527961360
        struct[0].Gy[63,24] = 0.996212229189942
        struct[0].Gy[63,25] = -0.249053057297486
        struct[0].Gy[63,28] = -0.996212229189942
        struct[0].Gy[63,29] = 0.249053057297486
        struct[0].Gy[63,58] = -0.214305342572583
        struct[0].Gy[63,59] = 0.327228031409243
        struct[0].Gy[63,60] = -0.214305342572582
        struct[0].Gy[63,61] = 0.327228031409244
        struct[0].Gy[63,62] = 0.684903767132556
        struct[0].Gy[63,63] = -1.12912163934412
        struct[0].Gy[63,64] = 0.171091579417532
        struct[0].Gy[63,65] = -0.316443717683753
        struct[0].Gy[63,66] = 0.171091579417531
        struct[0].Gy[63,67] = -0.316443717683753
        struct[0].Gy[63,68] = -0.598820527961360
        struct[0].Gy[63,69] = 1.10755301189314
        struct[0].Gy[64,32] = 0.249053057297486
        struct[0].Gy[64,33] = 0.996212229189942
        struct[0].Gy[64,34] = -0.249053057297486
        struct[0].Gy[64,35] = -0.996212229189942
        struct[0].Gy[64,58] = 1.10755301189314
        struct[0].Gy[64,59] = 0.598820527961361
        struct[0].Gy[64,60] = -0.316443717683753
        struct[0].Gy[64,61] = -0.171091579417532
        struct[0].Gy[64,62] = -0.316443717683753
        struct[0].Gy[64,63] = -0.171091579417532
        struct[0].Gy[64,64] = -2.23667465123725
        struct[0].Gy[64,65] = -1.28353302446119
        struct[0].Gy[64,66] = 0.643671749092996
        struct[0].Gy[64,67] = 0.385473430243205
        struct[0].Gy[64,68] = 0.643671749092997
        struct[0].Gy[64,69] = 0.385473430243205
        struct[0].Gy[65,32] = -0.996212229189942
        struct[0].Gy[65,33] = 0.249053057297486
        struct[0].Gy[65,34] = 0.996212229189942
        struct[0].Gy[65,35] = -0.249053057297486
        struct[0].Gy[65,58] = -0.598820527961361
        struct[0].Gy[65,59] = 1.10755301189314
        struct[0].Gy[65,60] = 0.171091579417532
        struct[0].Gy[65,61] = -0.316443717683753
        struct[0].Gy[65,62] = 0.171091579417532
        struct[0].Gy[65,63] = -0.316443717683753
        struct[0].Gy[65,64] = 1.28353302446119
        struct[0].Gy[65,65] = -2.23667465123725
        struct[0].Gy[65,66] = -0.385473430243205
        struct[0].Gy[65,67] = 0.643671749092996
        struct[0].Gy[65,68] = -0.385473430243205
        struct[0].Gy[65,69] = 0.643671749092997
        struct[0].Gy[66,34] = 0.249053057297486
        struct[0].Gy[66,35] = 0.996212229189942
        struct[0].Gy[66,36] = -0.249053057297486
        struct[0].Gy[66,37] = -0.996212229189942
        struct[0].Gy[66,58] = -0.316443717683753
        struct[0].Gy[66,59] = -0.171091579417532
        struct[0].Gy[66,60] = 1.10755301189314
        struct[0].Gy[66,61] = 0.598820527961360
        struct[0].Gy[66,62] = -0.316443717683753
        struct[0].Gy[66,63] = -0.171091579417531
        struct[0].Gy[66,64] = 0.643671749092996
        struct[0].Gy[66,65] = 0.385473430243205
        struct[0].Gy[66,66] = -2.23667465123725
        struct[0].Gy[66,67] = -1.28353302446119
        struct[0].Gy[66,68] = 0.643671749092997
        struct[0].Gy[66,69] = 0.385473430243204
        struct[0].Gy[67,34] = -0.996212229189942
        struct[0].Gy[67,35] = 0.249053057297486
        struct[0].Gy[67,36] = 0.996212229189942
        struct[0].Gy[67,37] = -0.249053057297486
        struct[0].Gy[67,58] = 0.171091579417532
        struct[0].Gy[67,59] = -0.316443717683753
        struct[0].Gy[67,60] = -0.598820527961360
        struct[0].Gy[67,61] = 1.10755301189314
        struct[0].Gy[67,62] = 0.171091579417531
        struct[0].Gy[67,63] = -0.316443717683753
        struct[0].Gy[67,64] = -0.385473430243205
        struct[0].Gy[67,65] = 0.643671749092996
        struct[0].Gy[67,66] = 1.28353302446119
        struct[0].Gy[67,67] = -2.23667465123725
        struct[0].Gy[67,68] = -0.385473430243204
        struct[0].Gy[67,69] = 0.643671749092997
        struct[0].Gy[68,32] = -0.249053057297486
        struct[0].Gy[68,33] = -0.996212229189942
        struct[0].Gy[68,36] = 0.249053057297486
        struct[0].Gy[68,37] = 0.996212229189942
        struct[0].Gy[68,58] = -0.316443717683753
        struct[0].Gy[68,59] = -0.171091579417532
        struct[0].Gy[68,60] = -0.316443717683753
        struct[0].Gy[68,61] = -0.171091579417531
        struct[0].Gy[68,62] = 1.10755301189314
        struct[0].Gy[68,63] = 0.598820527961360
        struct[0].Gy[68,64] = 0.643671749092997
        struct[0].Gy[68,65] = 0.385473430243205
        struct[0].Gy[68,66] = 0.643671749092997
        struct[0].Gy[68,67] = 0.385473430243204
        struct[0].Gy[68,68] = -2.23667465123725
        struct[0].Gy[68,69] = -1.28353302446119
        struct[0].Gy[69,32] = 0.996212229189942
        struct[0].Gy[69,33] = -0.249053057297486
        struct[0].Gy[69,36] = -0.996212229189942
        struct[0].Gy[69,37] = 0.249053057297486
        struct[0].Gy[69,58] = 0.171091579417532
        struct[0].Gy[69,59] = -0.316443717683753
        struct[0].Gy[69,60] = 0.171091579417531
        struct[0].Gy[69,61] = -0.316443717683753
        struct[0].Gy[69,62] = -0.598820527961360
        struct[0].Gy[69,63] = 1.10755301189314
        struct[0].Gy[69,64] = -0.385473430243205
        struct[0].Gy[69,65] = 0.643671749092997
        struct[0].Gy[69,66] = -0.385473430243204
        struct[0].Gy[69,67] = 0.643671749092997
        struct[0].Gy[69,68] = 1.28353302446119
        struct[0].Gy[69,69] = -2.23667465123725
        struct[0].Gy[70,0] = -0.249053057297486
        struct[0].Gy[70,1] = -0.996212229189942
        struct[0].Gy[70,2] = 0.249053057297486
        struct[0].Gy[70,3] = 0.996212229189942
        struct[0].Gy[70,40] = 0.0215686274509804
        struct[0].Gy[70,41] = 0.0862745098039216
        struct[0].Gy[70,42] = -0.0107843137254902
        struct[0].Gy[70,43] = -0.0431372549019608
        struct[0].Gy[70,44] = -0.0107843137254902
        struct[0].Gy[70,45] = -0.0431372549019608
        struct[0].Gy[70,70] = -1
        struct[0].Gy[71,0] = 0.996212229189942
        struct[0].Gy[71,1] = -0.249053057297486
        struct[0].Gy[71,2] = -0.996212229189942
        struct[0].Gy[71,3] = 0.249053057297486
        struct[0].Gy[71,40] = -0.0862745098039216
        struct[0].Gy[71,41] = 0.0215686274509804
        struct[0].Gy[71,42] = 0.0431372549019608
        struct[0].Gy[71,43] = -0.0107843137254902
        struct[0].Gy[71,44] = 0.0431372549019608
        struct[0].Gy[71,45] = -0.0107843137254902
        struct[0].Gy[71,71] = -1
        struct[0].Gy[72,2] = -0.249053057297486
        struct[0].Gy[72,3] = -0.996212229189942
        struct[0].Gy[72,4] = 0.249053057297486
        struct[0].Gy[72,5] = 0.996212229189942
        struct[0].Gy[72,40] = -0.0107843137254902
        struct[0].Gy[72,41] = -0.0431372549019608
        struct[0].Gy[72,42] = 0.0215686274509804
        struct[0].Gy[72,43] = 0.0862745098039216
        struct[0].Gy[72,44] = -0.0107843137254902
        struct[0].Gy[72,45] = -0.0431372549019608
        struct[0].Gy[72,72] = -1
        struct[0].Gy[73,2] = 0.996212229189942
        struct[0].Gy[73,3] = -0.249053057297486
        struct[0].Gy[73,4] = -0.996212229189942
        struct[0].Gy[73,5] = 0.249053057297486
        struct[0].Gy[73,40] = 0.0431372549019608
        struct[0].Gy[73,41] = -0.0107843137254902
        struct[0].Gy[73,42] = -0.0862745098039216
        struct[0].Gy[73,43] = 0.0215686274509804
        struct[0].Gy[73,44] = 0.0431372549019608
        struct[0].Gy[73,45] = -0.0107843137254902
        struct[0].Gy[73,73] = -1
        struct[0].Gy[74,0] = 0.249053057297486
        struct[0].Gy[74,1] = 0.996212229189942
        struct[0].Gy[74,4] = -0.249053057297486
        struct[0].Gy[74,5] = -0.996212229189942
        struct[0].Gy[74,40] = -0.0107843137254902
        struct[0].Gy[74,41] = -0.0431372549019608
        struct[0].Gy[74,42] = -0.0107843137254902
        struct[0].Gy[74,43] = -0.0431372549019608
        struct[0].Gy[74,44] = 0.0215686274509804
        struct[0].Gy[74,45] = 0.0862745098039216
        struct[0].Gy[74,74] = -1
        struct[0].Gy[75,0] = -0.996212229189942
        struct[0].Gy[75,1] = 0.249053057297486
        struct[0].Gy[75,4] = 0.996212229189942
        struct[0].Gy[75,5] = -0.249053057297486
        struct[0].Gy[75,40] = 0.0431372549019608
        struct[0].Gy[75,41] = -0.0107843137254902
        struct[0].Gy[75,42] = 0.0431372549019608
        struct[0].Gy[75,43] = -0.0107843137254902
        struct[0].Gy[75,44] = -0.0862745098039216
        struct[0].Gy[75,45] = 0.0215686274509804
        struct[0].Gy[75,75] = -1
        struct[0].Gy[76,8] = -0.249053057297486
        struct[0].Gy[76,9] = -0.996212229189942
        struct[0].Gy[76,10] = 0.249053057297486
        struct[0].Gy[76,11] = 0.996212229189942
        struct[0].Gy[76,46] = 0.0215686274509804
        struct[0].Gy[76,47] = 0.0862745098039216
        struct[0].Gy[76,48] = -0.0107843137254902
        struct[0].Gy[76,49] = -0.0431372549019608
        struct[0].Gy[76,50] = -0.0107843137254902
        struct[0].Gy[76,51] = -0.0431372549019608
        struct[0].Gy[76,76] = -1
        struct[0].Gy[77,8] = 0.996212229189942
        struct[0].Gy[77,9] = -0.249053057297486
        struct[0].Gy[77,10] = -0.996212229189942
        struct[0].Gy[77,11] = 0.249053057297486
        struct[0].Gy[77,46] = -0.0862745098039216
        struct[0].Gy[77,47] = 0.0215686274509804
        struct[0].Gy[77,48] = 0.0431372549019608
        struct[0].Gy[77,49] = -0.0107843137254902
        struct[0].Gy[77,50] = 0.0431372549019608
        struct[0].Gy[77,51] = -0.0107843137254902
        struct[0].Gy[77,77] = -1
        struct[0].Gy[78,10] = -0.249053057297486
        struct[0].Gy[78,11] = -0.996212229189942
        struct[0].Gy[78,12] = 0.249053057297486
        struct[0].Gy[78,13] = 0.996212229189942
        struct[0].Gy[78,46] = -0.0107843137254902
        struct[0].Gy[78,47] = -0.0431372549019608
        struct[0].Gy[78,48] = 0.0215686274509804
        struct[0].Gy[78,49] = 0.0862745098039216
        struct[0].Gy[78,50] = -0.0107843137254902
        struct[0].Gy[78,51] = -0.0431372549019608
        struct[0].Gy[78,78] = -1
        struct[0].Gy[79,10] = 0.996212229189942
        struct[0].Gy[79,11] = -0.249053057297486
        struct[0].Gy[79,12] = -0.996212229189942
        struct[0].Gy[79,13] = 0.249053057297486
        struct[0].Gy[79,46] = 0.0431372549019608
        struct[0].Gy[79,47] = -0.0107843137254902
        struct[0].Gy[79,48] = -0.0862745098039216
        struct[0].Gy[79,49] = 0.0215686274509804
        struct[0].Gy[79,50] = 0.0431372549019608
        struct[0].Gy[79,51] = -0.0107843137254902
        struct[0].Gy[79,79] = -1
        struct[0].Gy[80,8] = 0.249053057297486
        struct[0].Gy[80,9] = 0.996212229189942
        struct[0].Gy[80,12] = -0.249053057297486
        struct[0].Gy[80,13] = -0.996212229189942
        struct[0].Gy[80,46] = -0.0107843137254902
        struct[0].Gy[80,47] = -0.0431372549019608
        struct[0].Gy[80,48] = -0.0107843137254902
        struct[0].Gy[80,49] = -0.0431372549019608
        struct[0].Gy[80,50] = 0.0215686274509804
        struct[0].Gy[80,51] = 0.0862745098039216
        struct[0].Gy[80,80] = -1
        struct[0].Gy[81,8] = -0.996212229189942
        struct[0].Gy[81,9] = 0.249053057297486
        struct[0].Gy[81,12] = 0.996212229189942
        struct[0].Gy[81,13] = -0.249053057297486
        struct[0].Gy[81,46] = 0.0431372549019608
        struct[0].Gy[81,47] = -0.0107843137254902
        struct[0].Gy[81,48] = 0.0431372549019608
        struct[0].Gy[81,49] = -0.0107843137254902
        struct[0].Gy[81,50] = -0.0862745098039216
        struct[0].Gy[81,51] = 0.0215686274509804
        struct[0].Gy[81,81] = -1
        struct[0].Gy[82,16] = -0.249053057297486
        struct[0].Gy[82,17] = -0.996212229189942
        struct[0].Gy[82,18] = 0.249053057297486
        struct[0].Gy[82,19] = 0.996212229189942
        struct[0].Gy[82,52] = 0.0215686274509804
        struct[0].Gy[82,53] = 0.0862745098039216
        struct[0].Gy[82,54] = -0.0107843137254902
        struct[0].Gy[82,55] = -0.0431372549019608
        struct[0].Gy[82,56] = -0.0107843137254902
        struct[0].Gy[82,57] = -0.0431372549019608
        struct[0].Gy[82,82] = -1
        struct[0].Gy[83,16] = 0.996212229189942
        struct[0].Gy[83,17] = -0.249053057297486
        struct[0].Gy[83,18] = -0.996212229189942
        struct[0].Gy[83,19] = 0.249053057297486
        struct[0].Gy[83,52] = -0.0862745098039216
        struct[0].Gy[83,53] = 0.0215686274509804
        struct[0].Gy[83,54] = 0.0431372549019608
        struct[0].Gy[83,55] = -0.0107843137254902
        struct[0].Gy[83,56] = 0.0431372549019608
        struct[0].Gy[83,57] = -0.0107843137254902
        struct[0].Gy[83,83] = -1
        struct[0].Gy[84,18] = -0.249053057297486
        struct[0].Gy[84,19] = -0.996212229189942
        struct[0].Gy[84,20] = 0.249053057297486
        struct[0].Gy[84,21] = 0.996212229189942
        struct[0].Gy[84,52] = -0.0107843137254902
        struct[0].Gy[84,53] = -0.0431372549019608
        struct[0].Gy[84,54] = 0.0215686274509804
        struct[0].Gy[84,55] = 0.0862745098039216
        struct[0].Gy[84,56] = -0.0107843137254902
        struct[0].Gy[84,57] = -0.0431372549019608
        struct[0].Gy[84,84] = -1
        struct[0].Gy[85,18] = 0.996212229189942
        struct[0].Gy[85,19] = -0.249053057297486
        struct[0].Gy[85,20] = -0.996212229189942
        struct[0].Gy[85,21] = 0.249053057297486
        struct[0].Gy[85,52] = 0.0431372549019608
        struct[0].Gy[85,53] = -0.0107843137254902
        struct[0].Gy[85,54] = -0.0862745098039216
        struct[0].Gy[85,55] = 0.0215686274509804
        struct[0].Gy[85,56] = 0.0431372549019608
        struct[0].Gy[85,57] = -0.0107843137254902
        struct[0].Gy[85,85] = -1
        struct[0].Gy[86,16] = 0.249053057297486
        struct[0].Gy[86,17] = 0.996212229189942
        struct[0].Gy[86,20] = -0.249053057297486
        struct[0].Gy[86,21] = -0.996212229189942
        struct[0].Gy[86,52] = -0.0107843137254902
        struct[0].Gy[86,53] = -0.0431372549019608
        struct[0].Gy[86,54] = -0.0107843137254902
        struct[0].Gy[86,55] = -0.0431372549019608
        struct[0].Gy[86,56] = 0.0215686274509804
        struct[0].Gy[86,57] = 0.0862745098039216
        struct[0].Gy[86,86] = -1
        struct[0].Gy[87,16] = -0.996212229189942
        struct[0].Gy[87,17] = 0.249053057297486
        struct[0].Gy[87,20] = 0.996212229189942
        struct[0].Gy[87,21] = -0.249053057297486
        struct[0].Gy[87,52] = 0.0431372549019608
        struct[0].Gy[87,53] = -0.0107843137254902
        struct[0].Gy[87,54] = 0.0431372549019608
        struct[0].Gy[87,55] = -0.0107843137254902
        struct[0].Gy[87,56] = -0.0862745098039216
        struct[0].Gy[87,57] = 0.0215686274509804
        struct[0].Gy[87,87] = -1
        struct[0].Gy[88,24] = -0.249053057297486
        struct[0].Gy[88,25] = -0.996212229189942
        struct[0].Gy[88,26] = 0.249053057297486
        struct[0].Gy[88,27] = 0.996212229189942
        struct[0].Gy[88,58] = 0.0215686274509804
        struct[0].Gy[88,59] = 0.0862745098039216
        struct[0].Gy[88,60] = -0.0107843137254902
        struct[0].Gy[88,61] = -0.0431372549019608
        struct[0].Gy[88,62] = -0.0107843137254902
        struct[0].Gy[88,63] = -0.0431372549019608
        struct[0].Gy[88,88] = -1
        struct[0].Gy[89,24] = 0.996212229189942
        struct[0].Gy[89,25] = -0.249053057297486
        struct[0].Gy[89,26] = -0.996212229189942
        struct[0].Gy[89,27] = 0.249053057297486
        struct[0].Gy[89,58] = -0.0862745098039216
        struct[0].Gy[89,59] = 0.0215686274509804
        struct[0].Gy[89,60] = 0.0431372549019608
        struct[0].Gy[89,61] = -0.0107843137254902
        struct[0].Gy[89,62] = 0.0431372549019608
        struct[0].Gy[89,63] = -0.0107843137254902
        struct[0].Gy[89,89] = -1
        struct[0].Gy[90,26] = -0.249053057297486
        struct[0].Gy[90,27] = -0.996212229189942
        struct[0].Gy[90,28] = 0.249053057297486
        struct[0].Gy[90,29] = 0.996212229189942
        struct[0].Gy[90,58] = -0.0107843137254902
        struct[0].Gy[90,59] = -0.0431372549019608
        struct[0].Gy[90,60] = 0.0215686274509804
        struct[0].Gy[90,61] = 0.0862745098039216
        struct[0].Gy[90,62] = -0.0107843137254902
        struct[0].Gy[90,63] = -0.0431372549019608
        struct[0].Gy[90,90] = -1
        struct[0].Gy[91,26] = 0.996212229189942
        struct[0].Gy[91,27] = -0.249053057297486
        struct[0].Gy[91,28] = -0.996212229189942
        struct[0].Gy[91,29] = 0.249053057297486
        struct[0].Gy[91,58] = 0.0431372549019608
        struct[0].Gy[91,59] = -0.0107843137254902
        struct[0].Gy[91,60] = -0.0862745098039216
        struct[0].Gy[91,61] = 0.0215686274509804
        struct[0].Gy[91,62] = 0.0431372549019608
        struct[0].Gy[91,63] = -0.0107843137254902
        struct[0].Gy[91,91] = -1
        struct[0].Gy[92,24] = 0.249053057297486
        struct[0].Gy[92,25] = 0.996212229189942
        struct[0].Gy[92,28] = -0.249053057297486
        struct[0].Gy[92,29] = -0.996212229189942
        struct[0].Gy[92,58] = -0.0107843137254902
        struct[0].Gy[92,59] = -0.0431372549019608
        struct[0].Gy[92,60] = -0.0107843137254902
        struct[0].Gy[92,61] = -0.0431372549019608
        struct[0].Gy[92,62] = 0.0215686274509804
        struct[0].Gy[92,63] = 0.0862745098039216
        struct[0].Gy[92,92] = -1
        struct[0].Gy[93,24] = -0.996212229189942
        struct[0].Gy[93,25] = 0.249053057297486
        struct[0].Gy[93,28] = 0.996212229189942
        struct[0].Gy[93,29] = -0.249053057297486
        struct[0].Gy[93,58] = 0.0431372549019608
        struct[0].Gy[93,59] = -0.0107843137254902
        struct[0].Gy[93,60] = 0.0431372549019608
        struct[0].Gy[93,61] = -0.0107843137254902
        struct[0].Gy[93,62] = -0.0862745098039216
        struct[0].Gy[93,63] = 0.0215686274509804
        struct[0].Gy[93,93] = -1
        struct[0].Gy[94,32] = -0.249053057297486
        struct[0].Gy[94,33] = -0.996212229189942
        struct[0].Gy[94,34] = 0.249053057297486
        struct[0].Gy[94,35] = 0.996212229189942
        struct[0].Gy[94,64] = 0.0215686274509804
        struct[0].Gy[94,65] = 0.0862745098039216
        struct[0].Gy[94,66] = -0.0107843137254902
        struct[0].Gy[94,67] = -0.0431372549019608
        struct[0].Gy[94,68] = -0.0107843137254902
        struct[0].Gy[94,69] = -0.0431372549019608
        struct[0].Gy[94,94] = -1
        struct[0].Gy[95,32] = 0.996212229189942
        struct[0].Gy[95,33] = -0.249053057297486
        struct[0].Gy[95,34] = -0.996212229189942
        struct[0].Gy[95,35] = 0.249053057297486
        struct[0].Gy[95,64] = -0.0862745098039216
        struct[0].Gy[95,65] = 0.0215686274509804
        struct[0].Gy[95,66] = 0.0431372549019608
        struct[0].Gy[95,67] = -0.0107843137254902
        struct[0].Gy[95,68] = 0.0431372549019608
        struct[0].Gy[95,69] = -0.0107843137254902
        struct[0].Gy[95,95] = -1
        struct[0].Gy[96,34] = -0.249053057297486
        struct[0].Gy[96,35] = -0.996212229189942
        struct[0].Gy[96,36] = 0.249053057297486
        struct[0].Gy[96,37] = 0.996212229189942
        struct[0].Gy[96,64] = -0.0107843137254902
        struct[0].Gy[96,65] = -0.0431372549019608
        struct[0].Gy[96,66] = 0.0215686274509804
        struct[0].Gy[96,67] = 0.0862745098039216
        struct[0].Gy[96,68] = -0.0107843137254902
        struct[0].Gy[96,69] = -0.0431372549019608
        struct[0].Gy[96,96] = -1
        struct[0].Gy[97,34] = 0.996212229189942
        struct[0].Gy[97,35] = -0.249053057297486
        struct[0].Gy[97,36] = -0.996212229189942
        struct[0].Gy[97,37] = 0.249053057297486
        struct[0].Gy[97,64] = 0.0431372549019608
        struct[0].Gy[97,65] = -0.0107843137254902
        struct[0].Gy[97,66] = -0.0862745098039216
        struct[0].Gy[97,67] = 0.0215686274509804
        struct[0].Gy[97,68] = 0.0431372549019608
        struct[0].Gy[97,69] = -0.0107843137254902
        struct[0].Gy[97,97] = -1
        struct[0].Gy[98,32] = 0.249053057297486
        struct[0].Gy[98,33] = 0.996212229189942
        struct[0].Gy[98,36] = -0.249053057297486
        struct[0].Gy[98,37] = -0.996212229189942
        struct[0].Gy[98,64] = -0.0107843137254902
        struct[0].Gy[98,65] = -0.0431372549019608
        struct[0].Gy[98,66] = -0.0107843137254902
        struct[0].Gy[98,67] = -0.0431372549019608
        struct[0].Gy[98,68] = 0.0215686274509804
        struct[0].Gy[98,69] = 0.0862745098039216
        struct[0].Gy[98,98] = -1
        struct[0].Gy[99,32] = -0.996212229189942
        struct[0].Gy[99,33] = 0.249053057297486
        struct[0].Gy[99,36] = 0.996212229189942
        struct[0].Gy[99,37] = -0.249053057297486
        struct[0].Gy[99,64] = 0.0431372549019608
        struct[0].Gy[99,65] = -0.0107843137254902
        struct[0].Gy[99,66] = 0.0431372549019608
        struct[0].Gy[99,67] = -0.0107843137254902
        struct[0].Gy[99,68] = -0.0862745098039216
        struct[0].Gy[99,69] = 0.0215686274509804
        struct[0].Gy[99,99] = -1
        struct[0].Gy[100,40] = -1.10755301189314
        struct[0].Gy[100,41] = -0.598820527961361
        struct[0].Gy[100,42] = 0.316443717683753
        struct[0].Gy[100,43] = 0.171091579417532
        struct[0].Gy[100,44] = 0.316443717683753
        struct[0].Gy[100,45] = 0.171091579417532
        struct[0].Gy[100,100] = -1
        struct[0].Gy[101,40] = 0.598820527961361
        struct[0].Gy[101,41] = -1.10755301189314
        struct[0].Gy[101,42] = -0.171091579417532
        struct[0].Gy[101,43] = 0.316443717683753
        struct[0].Gy[101,44] = -0.171091579417532
        struct[0].Gy[101,45] = 0.316443717683753
        struct[0].Gy[101,101] = -1
        struct[0].Gy[102,40] = 0.316443717683753
        struct[0].Gy[102,41] = 0.171091579417532
        struct[0].Gy[102,42] = -1.10755301189314
        struct[0].Gy[102,43] = -0.598820527961360
        struct[0].Gy[102,44] = 0.316443717683753
        struct[0].Gy[102,45] = 0.171091579417531
        struct[0].Gy[102,102] = -1
        struct[0].Gy[103,40] = -0.171091579417532
        struct[0].Gy[103,41] = 0.316443717683753
        struct[0].Gy[103,42] = 0.598820527961360
        struct[0].Gy[103,43] = -1.10755301189314
        struct[0].Gy[103,44] = -0.171091579417531
        struct[0].Gy[103,45] = 0.316443717683753
        struct[0].Gy[103,103] = -1
        struct[0].Gy[104,40] = 0.316443717683753
        struct[0].Gy[104,41] = 0.171091579417532
        struct[0].Gy[104,42] = 0.316443717683753
        struct[0].Gy[104,43] = 0.171091579417531
        struct[0].Gy[104,44] = -1.10755301189314
        struct[0].Gy[104,45] = -0.598820527961360
        struct[0].Gy[104,104] = -1
        struct[0].Gy[105,40] = -0.171091579417532
        struct[0].Gy[105,41] = 0.316443717683753
        struct[0].Gy[105,42] = -0.171091579417531
        struct[0].Gy[105,43] = 0.316443717683753
        struct[0].Gy[105,44] = 0.598820527961360
        struct[0].Gy[105,45] = -1.10755301189314
        struct[0].Gy[105,105] = -1
        struct[0].Gy[106,40] = 1.10755301189314
        struct[0].Gy[106,41] = 0.598820527961361
        struct[0].Gy[106,42] = -0.316443717683753
        struct[0].Gy[106,43] = -0.171091579417532
        struct[0].Gy[106,44] = -0.316443717683753
        struct[0].Gy[106,45] = -0.171091579417532
        struct[0].Gy[106,46] = -1.10755301189314
        struct[0].Gy[106,47] = -0.598820527961361
        struct[0].Gy[106,48] = 0.316443717683753
        struct[0].Gy[106,49] = 0.171091579417532
        struct[0].Gy[106,50] = 0.316443717683753
        struct[0].Gy[106,51] = 0.171091579417532
        struct[0].Gy[106,106] = -1
        struct[0].Gy[107,40] = -0.598820527961361
        struct[0].Gy[107,41] = 1.10755301189314
        struct[0].Gy[107,42] = 0.171091579417532
        struct[0].Gy[107,43] = -0.316443717683753
        struct[0].Gy[107,44] = 0.171091579417532
        struct[0].Gy[107,45] = -0.316443717683753
        struct[0].Gy[107,46] = 0.598820527961361
        struct[0].Gy[107,47] = -1.10755301189314
        struct[0].Gy[107,48] = -0.171091579417532
        struct[0].Gy[107,49] = 0.316443717683753
        struct[0].Gy[107,50] = -0.171091579417532
        struct[0].Gy[107,51] = 0.316443717683753
        struct[0].Gy[107,107] = -1
        struct[0].Gy[108,40] = -0.316443717683753
        struct[0].Gy[108,41] = -0.171091579417532
        struct[0].Gy[108,42] = 1.10755301189314
        struct[0].Gy[108,43] = 0.598820527961360
        struct[0].Gy[108,44] = -0.316443717683753
        struct[0].Gy[108,45] = -0.171091579417531
        struct[0].Gy[108,46] = 0.316443717683753
        struct[0].Gy[108,47] = 0.171091579417532
        struct[0].Gy[108,48] = -1.10755301189314
        struct[0].Gy[108,49] = -0.598820527961360
        struct[0].Gy[108,50] = 0.316443717683753
        struct[0].Gy[108,51] = 0.171091579417531
        struct[0].Gy[108,108] = -1
        struct[0].Gy[109,40] = 0.171091579417532
        struct[0].Gy[109,41] = -0.316443717683753
        struct[0].Gy[109,42] = -0.598820527961360
        struct[0].Gy[109,43] = 1.10755301189314
        struct[0].Gy[109,44] = 0.171091579417531
        struct[0].Gy[109,45] = -0.316443717683753
        struct[0].Gy[109,46] = -0.171091579417532
        struct[0].Gy[109,47] = 0.316443717683753
        struct[0].Gy[109,48] = 0.598820527961360
        struct[0].Gy[109,49] = -1.10755301189314
        struct[0].Gy[109,50] = -0.171091579417531
        struct[0].Gy[109,51] = 0.316443717683753
        struct[0].Gy[109,109] = -1
        struct[0].Gy[110,40] = -0.316443717683753
        struct[0].Gy[110,41] = -0.171091579417532
        struct[0].Gy[110,42] = -0.316443717683753
        struct[0].Gy[110,43] = -0.171091579417531
        struct[0].Gy[110,44] = 1.10755301189314
        struct[0].Gy[110,45] = 0.598820527961360
        struct[0].Gy[110,46] = 0.316443717683753
        struct[0].Gy[110,47] = 0.171091579417532
        struct[0].Gy[110,48] = 0.316443717683753
        struct[0].Gy[110,49] = 0.171091579417531
        struct[0].Gy[110,50] = -1.10755301189314
        struct[0].Gy[110,51] = -0.598820527961360
        struct[0].Gy[110,110] = -1
        struct[0].Gy[111,40] = 0.171091579417532
        struct[0].Gy[111,41] = -0.316443717683753
        struct[0].Gy[111,42] = 0.171091579417531
        struct[0].Gy[111,43] = -0.316443717683753
        struct[0].Gy[111,44] = -0.598820527961360
        struct[0].Gy[111,45] = 1.10755301189314
        struct[0].Gy[111,46] = -0.171091579417532
        struct[0].Gy[111,47] = 0.316443717683753
        struct[0].Gy[111,48] = -0.171091579417531
        struct[0].Gy[111,49] = 0.316443717683753
        struct[0].Gy[111,50] = 0.598820527961360
        struct[0].Gy[111,51] = -1.10755301189314
        struct[0].Gy[111,111] = -1
        struct[0].Gy[112,46] = 1.10755301189314
        struct[0].Gy[112,47] = 0.598820527961361
        struct[0].Gy[112,48] = -0.316443717683753
        struct[0].Gy[112,49] = -0.171091579417532
        struct[0].Gy[112,50] = -0.316443717683753
        struct[0].Gy[112,51] = -0.171091579417532
        struct[0].Gy[112,52] = -1.10755301189314
        struct[0].Gy[112,53] = -0.598820527961361
        struct[0].Gy[112,54] = 0.316443717683753
        struct[0].Gy[112,55] = 0.171091579417532
        struct[0].Gy[112,56] = 0.316443717683753
        struct[0].Gy[112,57] = 0.171091579417532
        struct[0].Gy[112,112] = -1
        struct[0].Gy[113,46] = -0.598820527961361
        struct[0].Gy[113,47] = 1.10755301189314
        struct[0].Gy[113,48] = 0.171091579417532
        struct[0].Gy[113,49] = -0.316443717683753
        struct[0].Gy[113,50] = 0.171091579417532
        struct[0].Gy[113,51] = -0.316443717683753
        struct[0].Gy[113,52] = 0.598820527961361
        struct[0].Gy[113,53] = -1.10755301189314
        struct[0].Gy[113,54] = -0.171091579417532
        struct[0].Gy[113,55] = 0.316443717683753
        struct[0].Gy[113,56] = -0.171091579417532
        struct[0].Gy[113,57] = 0.316443717683753
        struct[0].Gy[113,113] = -1
        struct[0].Gy[114,46] = -0.316443717683753
        struct[0].Gy[114,47] = -0.171091579417532
        struct[0].Gy[114,48] = 1.10755301189314
        struct[0].Gy[114,49] = 0.598820527961360
        struct[0].Gy[114,50] = -0.316443717683753
        struct[0].Gy[114,51] = -0.171091579417531
        struct[0].Gy[114,52] = 0.316443717683753
        struct[0].Gy[114,53] = 0.171091579417532
        struct[0].Gy[114,54] = -1.10755301189314
        struct[0].Gy[114,55] = -0.598820527961360
        struct[0].Gy[114,56] = 0.316443717683753
        struct[0].Gy[114,57] = 0.171091579417531
        struct[0].Gy[114,114] = -1
        struct[0].Gy[115,46] = 0.171091579417532
        struct[0].Gy[115,47] = -0.316443717683753
        struct[0].Gy[115,48] = -0.598820527961360
        struct[0].Gy[115,49] = 1.10755301189314
        struct[0].Gy[115,50] = 0.171091579417531
        struct[0].Gy[115,51] = -0.316443717683753
        struct[0].Gy[115,52] = -0.171091579417532
        struct[0].Gy[115,53] = 0.316443717683753
        struct[0].Gy[115,54] = 0.598820527961360
        struct[0].Gy[115,55] = -1.10755301189314
        struct[0].Gy[115,56] = -0.171091579417531
        struct[0].Gy[115,57] = 0.316443717683753
        struct[0].Gy[115,115] = -1
        struct[0].Gy[116,46] = -0.316443717683753
        struct[0].Gy[116,47] = -0.171091579417532
        struct[0].Gy[116,48] = -0.316443717683753
        struct[0].Gy[116,49] = -0.171091579417531
        struct[0].Gy[116,50] = 1.10755301189314
        struct[0].Gy[116,51] = 0.598820527961360
        struct[0].Gy[116,52] = 0.316443717683753
        struct[0].Gy[116,53] = 0.171091579417532
        struct[0].Gy[116,54] = 0.316443717683753
        struct[0].Gy[116,55] = 0.171091579417531
        struct[0].Gy[116,56] = -1.10755301189314
        struct[0].Gy[116,57] = -0.598820527961360
        struct[0].Gy[116,116] = -1
        struct[0].Gy[117,46] = 0.171091579417532
        struct[0].Gy[117,47] = -0.316443717683753
        struct[0].Gy[117,48] = 0.171091579417531
        struct[0].Gy[117,49] = -0.316443717683753
        struct[0].Gy[117,50] = -0.598820527961360
        struct[0].Gy[117,51] = 1.10755301189314
        struct[0].Gy[117,52] = -0.171091579417532
        struct[0].Gy[117,53] = 0.316443717683753
        struct[0].Gy[117,54] = -0.171091579417531
        struct[0].Gy[117,55] = 0.316443717683753
        struct[0].Gy[117,56] = 0.598820527961360
        struct[0].Gy[117,57] = -1.10755301189314
        struct[0].Gy[117,117] = -1
        struct[0].Gy[118,58] = 1.10755301189314
        struct[0].Gy[118,59] = 0.598820527961361
        struct[0].Gy[118,60] = -0.316443717683753
        struct[0].Gy[118,61] = -0.171091579417532
        struct[0].Gy[118,62] = -0.316443717683753
        struct[0].Gy[118,63] = -0.171091579417532
        struct[0].Gy[118,64] = -1.10755301189314
        struct[0].Gy[118,65] = -0.598820527961361
        struct[0].Gy[118,66] = 0.316443717683753
        struct[0].Gy[118,67] = 0.171091579417532
        struct[0].Gy[118,68] = 0.316443717683753
        struct[0].Gy[118,69] = 0.171091579417532
        struct[0].Gy[118,118] = -1
        struct[0].Gy[119,58] = -0.598820527961361
        struct[0].Gy[119,59] = 1.10755301189314
        struct[0].Gy[119,60] = 0.171091579417532
        struct[0].Gy[119,61] = -0.316443717683753
        struct[0].Gy[119,62] = 0.171091579417532
        struct[0].Gy[119,63] = -0.316443717683753
        struct[0].Gy[119,64] = 0.598820527961361
        struct[0].Gy[119,65] = -1.10755301189314
        struct[0].Gy[119,66] = -0.171091579417532
        struct[0].Gy[119,67] = 0.316443717683753
        struct[0].Gy[119,68] = -0.171091579417532
        struct[0].Gy[119,69] = 0.316443717683753
        struct[0].Gy[119,119] = -1
        struct[0].Gy[120,58] = -0.316443717683753
        struct[0].Gy[120,59] = -0.171091579417532
        struct[0].Gy[120,60] = 1.10755301189314
        struct[0].Gy[120,61] = 0.598820527961360
        struct[0].Gy[120,62] = -0.316443717683753
        struct[0].Gy[120,63] = -0.171091579417531
        struct[0].Gy[120,64] = 0.316443717683753
        struct[0].Gy[120,65] = 0.171091579417532
        struct[0].Gy[120,66] = -1.10755301189314
        struct[0].Gy[120,67] = -0.598820527961360
        struct[0].Gy[120,68] = 0.316443717683753
        struct[0].Gy[120,69] = 0.171091579417531
        struct[0].Gy[120,120] = -1
        struct[0].Gy[121,58] = 0.171091579417532
        struct[0].Gy[121,59] = -0.316443717683753
        struct[0].Gy[121,60] = -0.598820527961360
        struct[0].Gy[121,61] = 1.10755301189314
        struct[0].Gy[121,62] = 0.171091579417531
        struct[0].Gy[121,63] = -0.316443717683753
        struct[0].Gy[121,64] = -0.171091579417532
        struct[0].Gy[121,65] = 0.316443717683753
        struct[0].Gy[121,66] = 0.598820527961360
        struct[0].Gy[121,67] = -1.10755301189314
        struct[0].Gy[121,68] = -0.171091579417531
        struct[0].Gy[121,69] = 0.316443717683753
        struct[0].Gy[121,121] = -1
        struct[0].Gy[122,58] = -0.316443717683753
        struct[0].Gy[122,59] = -0.171091579417532
        struct[0].Gy[122,60] = -0.316443717683753
        struct[0].Gy[122,61] = -0.171091579417531
        struct[0].Gy[122,62] = 1.10755301189314
        struct[0].Gy[122,63] = 0.598820527961360
        struct[0].Gy[122,64] = 0.316443717683753
        struct[0].Gy[122,65] = 0.171091579417532
        struct[0].Gy[122,66] = 0.316443717683753
        struct[0].Gy[122,67] = 0.171091579417531
        struct[0].Gy[122,68] = -1.10755301189314
        struct[0].Gy[122,69] = -0.598820527961360
        struct[0].Gy[122,122] = -1
        struct[0].Gy[123,58] = 0.171091579417532
        struct[0].Gy[123,59] = -0.316443717683753
        struct[0].Gy[123,60] = 0.171091579417531
        struct[0].Gy[123,61] = -0.316443717683753
        struct[0].Gy[123,62] = -0.598820527961360
        struct[0].Gy[123,63] = 1.10755301189314
        struct[0].Gy[123,64] = -0.171091579417532
        struct[0].Gy[123,65] = 0.316443717683753
        struct[0].Gy[123,66] = -0.171091579417531
        struct[0].Gy[123,67] = 0.316443717683753
        struct[0].Gy[123,68] = 0.598820527961360
        struct[0].Gy[123,69] = -1.10755301189314
        struct[0].Gy[123,123] = -1
        struct[0].Gy[124,64] = 1.10755301189314
        struct[0].Gy[124,65] = 0.598820527961361
        struct[0].Gy[124,66] = -0.316443717683753
        struct[0].Gy[124,67] = -0.171091579417532
        struct[0].Gy[124,68] = -0.316443717683753
        struct[0].Gy[124,69] = -0.171091579417532
        struct[0].Gy[124,124] = -1
        struct[0].Gy[125,64] = -0.598820527961361
        struct[0].Gy[125,65] = 1.10755301189314
        struct[0].Gy[125,66] = 0.171091579417532
        struct[0].Gy[125,67] = -0.316443717683753
        struct[0].Gy[125,68] = 0.171091579417532
        struct[0].Gy[125,69] = -0.316443717683753
        struct[0].Gy[125,125] = -1
        struct[0].Gy[126,64] = -0.316443717683753
        struct[0].Gy[126,65] = -0.171091579417532
        struct[0].Gy[126,66] = 1.10755301189314
        struct[0].Gy[126,67] = 0.598820527961360
        struct[0].Gy[126,68] = -0.316443717683753
        struct[0].Gy[126,69] = -0.171091579417531
        struct[0].Gy[126,126] = -1
        struct[0].Gy[127,64] = 0.171091579417532
        struct[0].Gy[127,65] = -0.316443717683753
        struct[0].Gy[127,66] = -0.598820527961360
        struct[0].Gy[127,67] = 1.10755301189314
        struct[0].Gy[127,68] = 0.171091579417531
        struct[0].Gy[127,69] = -0.316443717683753
        struct[0].Gy[127,127] = -1
        struct[0].Gy[128,64] = -0.316443717683753
        struct[0].Gy[128,65] = -0.171091579417532
        struct[0].Gy[128,66] = -0.316443717683753
        struct[0].Gy[128,67] = -0.171091579417531
        struct[0].Gy[128,68] = 1.10755301189314
        struct[0].Gy[128,69] = 0.598820527961360
        struct[0].Gy[128,128] = -1
        struct[0].Gy[129,64] = 0.171091579417532
        struct[0].Gy[129,65] = -0.316443717683753
        struct[0].Gy[129,66] = 0.171091579417531
        struct[0].Gy[129,67] = -0.316443717683753
        struct[0].Gy[129,68] = -0.598820527961360
        struct[0].Gy[129,69] = 1.10755301189314
        struct[0].Gy[129,129] = -1
        struct[0].Gy[130,0] = i_load_B2lv_a_r
        struct[0].Gy[130,1] = i_load_B2lv_a_i
        struct[0].Gy[130,6] = -i_load_B2lv_a_r
        struct[0].Gy[130,7] = -i_load_B2lv_a_i
        struct[0].Gy[130,130] = v_B2lv_a_r - v_B2lv_n_r
        struct[0].Gy[130,131] = v_B2lv_a_i - v_B2lv_n_i
        struct[0].Gy[130,239] = -1
        struct[0].Gy[131,2] = i_load_B2lv_b_r
        struct[0].Gy[131,3] = i_load_B2lv_b_i
        struct[0].Gy[131,6] = -i_load_B2lv_b_r
        struct[0].Gy[131,7] = -i_load_B2lv_b_i
        struct[0].Gy[131,132] = v_B2lv_b_r - v_B2lv_n_r
        struct[0].Gy[131,133] = v_B2lv_b_i - v_B2lv_n_i
        struct[0].Gy[131,240] = -1
        struct[0].Gy[132,4] = i_load_B2lv_c_r
        struct[0].Gy[132,5] = i_load_B2lv_c_i
        struct[0].Gy[132,6] = -i_load_B2lv_c_r
        struct[0].Gy[132,7] = -i_load_B2lv_c_i
        struct[0].Gy[132,134] = v_B2lv_c_r - v_B2lv_n_r
        struct[0].Gy[132,135] = v_B2lv_c_i - v_B2lv_n_i
        struct[0].Gy[132,241] = -1
        struct[0].Gy[133,0] = -i_load_B2lv_a_i
        struct[0].Gy[133,1] = i_load_B2lv_a_r
        struct[0].Gy[133,6] = i_load_B2lv_a_i
        struct[0].Gy[133,7] = -i_load_B2lv_a_r
        struct[0].Gy[133,130] = v_B2lv_a_i - v_B2lv_n_i
        struct[0].Gy[133,131] = -v_B2lv_a_r + v_B2lv_n_r
        struct[0].Gy[134,2] = -i_load_B2lv_b_i
        struct[0].Gy[134,3] = i_load_B2lv_b_r
        struct[0].Gy[134,6] = i_load_B2lv_b_i
        struct[0].Gy[134,7] = -i_load_B2lv_b_r
        struct[0].Gy[134,132] = v_B2lv_b_i - v_B2lv_n_i
        struct[0].Gy[134,133] = -v_B2lv_b_r + v_B2lv_n_r
        struct[0].Gy[135,4] = -i_load_B2lv_c_i
        struct[0].Gy[135,5] = i_load_B2lv_c_r
        struct[0].Gy[135,6] = i_load_B2lv_c_i
        struct[0].Gy[135,7] = -i_load_B2lv_c_r
        struct[0].Gy[135,134] = v_B2lv_c_i - v_B2lv_n_i
        struct[0].Gy[135,135] = -v_B2lv_c_r + v_B2lv_n_r
        struct[0].Gy[136,130] = 1
        struct[0].Gy[136,132] = 1
        struct[0].Gy[136,134] = 1
        struct[0].Gy[136,136] = 1
        struct[0].Gy[137,131] = 1
        struct[0].Gy[137,133] = 1
        struct[0].Gy[137,135] = 1
        struct[0].Gy[137,137] = 1
        struct[0].Gy[138,8] = i_load_B3lv_a_r
        struct[0].Gy[138,9] = i_load_B3lv_a_i
        struct[0].Gy[138,14] = -i_load_B3lv_a_r
        struct[0].Gy[138,15] = -i_load_B3lv_a_i
        struct[0].Gy[138,138] = v_B3lv_a_r - v_B3lv_n_r
        struct[0].Gy[138,139] = v_B3lv_a_i - v_B3lv_n_i
        struct[0].Gy[138,242] = -1
        struct[0].Gy[139,10] = i_load_B3lv_b_r
        struct[0].Gy[139,11] = i_load_B3lv_b_i
        struct[0].Gy[139,14] = -i_load_B3lv_b_r
        struct[0].Gy[139,15] = -i_load_B3lv_b_i
        struct[0].Gy[139,140] = v_B3lv_b_r - v_B3lv_n_r
        struct[0].Gy[139,141] = v_B3lv_b_i - v_B3lv_n_i
        struct[0].Gy[139,243] = -1
        struct[0].Gy[140,12] = i_load_B3lv_c_r
        struct[0].Gy[140,13] = i_load_B3lv_c_i
        struct[0].Gy[140,14] = -i_load_B3lv_c_r
        struct[0].Gy[140,15] = -i_load_B3lv_c_i
        struct[0].Gy[140,142] = v_B3lv_c_r - v_B3lv_n_r
        struct[0].Gy[140,143] = v_B3lv_c_i - v_B3lv_n_i
        struct[0].Gy[140,244] = -1
        struct[0].Gy[141,8] = -i_load_B3lv_a_i
        struct[0].Gy[141,9] = i_load_B3lv_a_r
        struct[0].Gy[141,14] = i_load_B3lv_a_i
        struct[0].Gy[141,15] = -i_load_B3lv_a_r
        struct[0].Gy[141,138] = v_B3lv_a_i - v_B3lv_n_i
        struct[0].Gy[141,139] = -v_B3lv_a_r + v_B3lv_n_r
        struct[0].Gy[142,10] = -i_load_B3lv_b_i
        struct[0].Gy[142,11] = i_load_B3lv_b_r
        struct[0].Gy[142,14] = i_load_B3lv_b_i
        struct[0].Gy[142,15] = -i_load_B3lv_b_r
        struct[0].Gy[142,140] = v_B3lv_b_i - v_B3lv_n_i
        struct[0].Gy[142,141] = -v_B3lv_b_r + v_B3lv_n_r
        struct[0].Gy[143,12] = -i_load_B3lv_c_i
        struct[0].Gy[143,13] = i_load_B3lv_c_r
        struct[0].Gy[143,14] = i_load_B3lv_c_i
        struct[0].Gy[143,15] = -i_load_B3lv_c_r
        struct[0].Gy[143,142] = v_B3lv_c_i - v_B3lv_n_i
        struct[0].Gy[143,143] = -v_B3lv_c_r + v_B3lv_n_r
        struct[0].Gy[144,138] = 1
        struct[0].Gy[144,140] = 1
        struct[0].Gy[144,142] = 1
        struct[0].Gy[144,144] = 1
        struct[0].Gy[145,139] = 1
        struct[0].Gy[145,141] = 1
        struct[0].Gy[145,143] = 1
        struct[0].Gy[145,145] = 1
        struct[0].Gy[146,16] = i_load_B4lv_a_r
        struct[0].Gy[146,17] = i_load_B4lv_a_i
        struct[0].Gy[146,22] = -i_load_B4lv_a_r
        struct[0].Gy[146,23] = -i_load_B4lv_a_i
        struct[0].Gy[146,146] = v_B4lv_a_r - v_B4lv_n_r
        struct[0].Gy[146,147] = v_B4lv_a_i - v_B4lv_n_i
        struct[0].Gy[146,245] = -1
        struct[0].Gy[147,18] = i_load_B4lv_b_r
        struct[0].Gy[147,19] = i_load_B4lv_b_i
        struct[0].Gy[147,22] = -i_load_B4lv_b_r
        struct[0].Gy[147,23] = -i_load_B4lv_b_i
        struct[0].Gy[147,148] = v_B4lv_b_r - v_B4lv_n_r
        struct[0].Gy[147,149] = v_B4lv_b_i - v_B4lv_n_i
        struct[0].Gy[147,246] = -1
        struct[0].Gy[148,20] = i_load_B4lv_c_r
        struct[0].Gy[148,21] = i_load_B4lv_c_i
        struct[0].Gy[148,22] = -i_load_B4lv_c_r
        struct[0].Gy[148,23] = -i_load_B4lv_c_i
        struct[0].Gy[148,150] = v_B4lv_c_r - v_B4lv_n_r
        struct[0].Gy[148,151] = v_B4lv_c_i - v_B4lv_n_i
        struct[0].Gy[148,247] = -1
        struct[0].Gy[149,16] = -i_load_B4lv_a_i
        struct[0].Gy[149,17] = i_load_B4lv_a_r
        struct[0].Gy[149,22] = i_load_B4lv_a_i
        struct[0].Gy[149,23] = -i_load_B4lv_a_r
        struct[0].Gy[149,146] = v_B4lv_a_i - v_B4lv_n_i
        struct[0].Gy[149,147] = -v_B4lv_a_r + v_B4lv_n_r
        struct[0].Gy[150,18] = -i_load_B4lv_b_i
        struct[0].Gy[150,19] = i_load_B4lv_b_r
        struct[0].Gy[150,22] = i_load_B4lv_b_i
        struct[0].Gy[150,23] = -i_load_B4lv_b_r
        struct[0].Gy[150,148] = v_B4lv_b_i - v_B4lv_n_i
        struct[0].Gy[150,149] = -v_B4lv_b_r + v_B4lv_n_r
        struct[0].Gy[151,20] = -i_load_B4lv_c_i
        struct[0].Gy[151,21] = i_load_B4lv_c_r
        struct[0].Gy[151,22] = i_load_B4lv_c_i
        struct[0].Gy[151,23] = -i_load_B4lv_c_r
        struct[0].Gy[151,150] = v_B4lv_c_i - v_B4lv_n_i
        struct[0].Gy[151,151] = -v_B4lv_c_r + v_B4lv_n_r
        struct[0].Gy[152,146] = 1
        struct[0].Gy[152,148] = 1
        struct[0].Gy[152,150] = 1
        struct[0].Gy[152,152] = 1
        struct[0].Gy[153,147] = 1
        struct[0].Gy[153,149] = 1
        struct[0].Gy[153,151] = 1
        struct[0].Gy[153,153] = 1
        struct[0].Gy[154,24] = i_load_B5lv_a_r
        struct[0].Gy[154,25] = i_load_B5lv_a_i
        struct[0].Gy[154,30] = -i_load_B5lv_a_r
        struct[0].Gy[154,31] = -i_load_B5lv_a_i
        struct[0].Gy[154,154] = v_B5lv_a_r - v_B5lv_n_r
        struct[0].Gy[154,155] = v_B5lv_a_i - v_B5lv_n_i
        struct[0].Gy[154,248] = -1
        struct[0].Gy[155,26] = i_load_B5lv_b_r
        struct[0].Gy[155,27] = i_load_B5lv_b_i
        struct[0].Gy[155,30] = -i_load_B5lv_b_r
        struct[0].Gy[155,31] = -i_load_B5lv_b_i
        struct[0].Gy[155,156] = v_B5lv_b_r - v_B5lv_n_r
        struct[0].Gy[155,157] = v_B5lv_b_i - v_B5lv_n_i
        struct[0].Gy[155,249] = -1
        struct[0].Gy[156,28] = i_load_B5lv_c_r
        struct[0].Gy[156,29] = i_load_B5lv_c_i
        struct[0].Gy[156,30] = -i_load_B5lv_c_r
        struct[0].Gy[156,31] = -i_load_B5lv_c_i
        struct[0].Gy[156,158] = v_B5lv_c_r - v_B5lv_n_r
        struct[0].Gy[156,159] = v_B5lv_c_i - v_B5lv_n_i
        struct[0].Gy[156,250] = -1
        struct[0].Gy[157,24] = -i_load_B5lv_a_i
        struct[0].Gy[157,25] = i_load_B5lv_a_r
        struct[0].Gy[157,30] = i_load_B5lv_a_i
        struct[0].Gy[157,31] = -i_load_B5lv_a_r
        struct[0].Gy[157,154] = v_B5lv_a_i - v_B5lv_n_i
        struct[0].Gy[157,155] = -v_B5lv_a_r + v_B5lv_n_r
        struct[0].Gy[158,26] = -i_load_B5lv_b_i
        struct[0].Gy[158,27] = i_load_B5lv_b_r
        struct[0].Gy[158,30] = i_load_B5lv_b_i
        struct[0].Gy[158,31] = -i_load_B5lv_b_r
        struct[0].Gy[158,156] = v_B5lv_b_i - v_B5lv_n_i
        struct[0].Gy[158,157] = -v_B5lv_b_r + v_B5lv_n_r
        struct[0].Gy[159,28] = -i_load_B5lv_c_i
        struct[0].Gy[159,29] = i_load_B5lv_c_r
        struct[0].Gy[159,30] = i_load_B5lv_c_i
        struct[0].Gy[159,31] = -i_load_B5lv_c_r
        struct[0].Gy[159,158] = v_B5lv_c_i - v_B5lv_n_i
        struct[0].Gy[159,159] = -v_B5lv_c_r + v_B5lv_n_r
        struct[0].Gy[160,154] = 1
        struct[0].Gy[160,156] = 1
        struct[0].Gy[160,158] = 1
        struct[0].Gy[160,160] = 1
        struct[0].Gy[161,155] = 1
        struct[0].Gy[161,157] = 1
        struct[0].Gy[161,159] = 1
        struct[0].Gy[161,161] = 1
        struct[0].Gy[162,32] = i_load_B6lv_a_r
        struct[0].Gy[162,33] = i_load_B6lv_a_i
        struct[0].Gy[162,38] = -i_load_B6lv_a_r
        struct[0].Gy[162,39] = -i_load_B6lv_a_i
        struct[0].Gy[162,162] = v_B6lv_a_r - v_B6lv_n_r
        struct[0].Gy[162,163] = v_B6lv_a_i - v_B6lv_n_i
        struct[0].Gy[162,251] = -1
        struct[0].Gy[163,34] = i_load_B6lv_b_r
        struct[0].Gy[163,35] = i_load_B6lv_b_i
        struct[0].Gy[163,38] = -i_load_B6lv_b_r
        struct[0].Gy[163,39] = -i_load_B6lv_b_i
        struct[0].Gy[163,164] = v_B6lv_b_r - v_B6lv_n_r
        struct[0].Gy[163,165] = v_B6lv_b_i - v_B6lv_n_i
        struct[0].Gy[163,252] = -1
        struct[0].Gy[164,36] = i_load_B6lv_c_r
        struct[0].Gy[164,37] = i_load_B6lv_c_i
        struct[0].Gy[164,38] = -i_load_B6lv_c_r
        struct[0].Gy[164,39] = -i_load_B6lv_c_i
        struct[0].Gy[164,166] = v_B6lv_c_r - v_B6lv_n_r
        struct[0].Gy[164,167] = v_B6lv_c_i - v_B6lv_n_i
        struct[0].Gy[164,253] = -1
        struct[0].Gy[165,32] = -i_load_B6lv_a_i
        struct[0].Gy[165,33] = i_load_B6lv_a_r
        struct[0].Gy[165,38] = i_load_B6lv_a_i
        struct[0].Gy[165,39] = -i_load_B6lv_a_r
        struct[0].Gy[165,162] = v_B6lv_a_i - v_B6lv_n_i
        struct[0].Gy[165,163] = -v_B6lv_a_r + v_B6lv_n_r
        struct[0].Gy[166,34] = -i_load_B6lv_b_i
        struct[0].Gy[166,35] = i_load_B6lv_b_r
        struct[0].Gy[166,38] = i_load_B6lv_b_i
        struct[0].Gy[166,39] = -i_load_B6lv_b_r
        struct[0].Gy[166,164] = v_B6lv_b_i - v_B6lv_n_i
        struct[0].Gy[166,165] = -v_B6lv_b_r + v_B6lv_n_r
        struct[0].Gy[167,36] = -i_load_B6lv_c_i
        struct[0].Gy[167,37] = i_load_B6lv_c_r
        struct[0].Gy[167,38] = i_load_B6lv_c_i
        struct[0].Gy[167,39] = -i_load_B6lv_c_r
        struct[0].Gy[167,166] = v_B6lv_c_i - v_B6lv_n_i
        struct[0].Gy[167,167] = -v_B6lv_c_r + v_B6lv_n_r
        struct[0].Gy[168,162] = 1
        struct[0].Gy[168,164] = 1
        struct[0].Gy[168,166] = 1
        struct[0].Gy[168,168] = 1
        struct[0].Gy[169,163] = 1
        struct[0].Gy[169,165] = 1
        struct[0].Gy[169,167] = 1
        struct[0].Gy[169,169] = 1
        struct[0].Gy[170,170] = -R_1112
        struct[0].Gy[170,194] = 1
        struct[0].Gy[170,195] = -1
        struct[0].Gy[171,171] = -R_1213
        struct[0].Gy[171,195] = 1
        struct[0].Gy[172,172] = -R_1314
        struct[0].Gy[172,197] = -1
        struct[0].Gy[173,173] = -R_1415
        struct[0].Gy[173,197] = 1
        struct[0].Gy[173,198] = -1
        struct[0].Gy[174,174] = -R_2122
        struct[0].Gy[174,199] = 1
        struct[0].Gy[174,200] = -1
        struct[0].Gy[175,175] = -R_2223
        struct[0].Gy[175,200] = 1
        struct[0].Gy[176,176] = -R_2324
        struct[0].Gy[176,202] = -1
        struct[0].Gy[177,177] = -R_2425
        struct[0].Gy[177,202] = 1
        struct[0].Gy[177,203] = -1
        struct[0].Gy[178,178] = -R_3132
        struct[0].Gy[178,204] = 1
        struct[0].Gy[178,205] = -1
        struct[0].Gy[179,179] = -R_3233
        struct[0].Gy[179,205] = 1
        struct[0].Gy[180,180] = -R_3334
        struct[0].Gy[180,207] = -1
        struct[0].Gy[181,181] = -R_3435
        struct[0].Gy[181,207] = 1
        struct[0].Gy[181,208] = -1
        struct[0].Gy[182,182] = -R_4142
        struct[0].Gy[182,209] = 1
        struct[0].Gy[182,210] = -1
        struct[0].Gy[183,183] = -R_4243
        struct[0].Gy[183,210] = 1
        struct[0].Gy[184,184] = -R_4344
        struct[0].Gy[184,212] = -1
        struct[0].Gy[185,185] = -R_4445
        struct[0].Gy[185,212] = 1
        struct[0].Gy[185,213] = -1
        struct[0].Gy[186,186] = -R_5152
        struct[0].Gy[186,214] = 1
        struct[0].Gy[186,215] = -1
        struct[0].Gy[187,187] = -R_5253
        struct[0].Gy[187,215] = 1
        struct[0].Gy[188,188] = -R_5354
        struct[0].Gy[188,217] = -1
        struct[0].Gy[189,189] = -R_5455
        struct[0].Gy[189,217] = 1
        struct[0].Gy[189,218] = -1
        struct[0].Gy[190,190] = -R_1521
        struct[0].Gy[190,198] = 1
        struct[0].Gy[190,199] = -1
        struct[0].Gy[191,191] = -R_2531
        struct[0].Gy[191,203] = 1
        struct[0].Gy[191,204] = -1
        struct[0].Gy[192,192] = -R_3541
        struct[0].Gy[192,208] = 1
        struct[0].Gy[192,209] = -1
        struct[0].Gy[193,193] = -R_4551
        struct[0].Gy[193,213] = 1
        struct[0].Gy[193,214] = -1
        struct[0].Gy[194,170] = -1
        struct[0].Gy[194,219] = 1
        struct[0].Gy[195,170] = 1
        struct[0].Gy[195,171] = -1
        struct[0].Gy[195,220] = 1
        struct[0].Gy[196,171] = 1
        struct[0].Gy[196,172] = -1
        struct[0].Gy[196,196] = 1
        struct[0].Gy[197,172] = 1
        struct[0].Gy[197,173] = -1
        struct[0].Gy[197,221] = 1
        struct[0].Gy[198,173] = 1
        struct[0].Gy[198,190] = -1
        struct[0].Gy[198,222] = 1
        struct[0].Gy[199,174] = -1
        struct[0].Gy[199,190] = 1
        struct[0].Gy[199,223] = 1
        struct[0].Gy[200,174] = 1
        struct[0].Gy[200,175] = -1
        struct[0].Gy[200,224] = 1
        struct[0].Gy[201,175] = 1
        struct[0].Gy[201,176] = -1
        struct[0].Gy[201,201] = 1
        struct[0].Gy[202,176] = 1
        struct[0].Gy[202,177] = -1
        struct[0].Gy[202,225] = 1
        struct[0].Gy[203,177] = 1
        struct[0].Gy[203,191] = -1
        struct[0].Gy[203,226] = 1
        struct[0].Gy[204,178] = -1
        struct[0].Gy[204,191] = 1
        struct[0].Gy[204,227] = 1
        struct[0].Gy[205,178] = 1
        struct[0].Gy[205,179] = -1
        struct[0].Gy[205,228] = 1
        struct[0].Gy[206,179] = 1
        struct[0].Gy[206,180] = -1
        struct[0].Gy[206,206] = 1
        struct[0].Gy[207,180] = 1
        struct[0].Gy[207,181] = -1
        struct[0].Gy[207,229] = 1
        struct[0].Gy[208,181] = 1
        struct[0].Gy[208,192] = -1
        struct[0].Gy[208,230] = 1
        struct[0].Gy[209,182] = -1
        struct[0].Gy[209,192] = 1
        struct[0].Gy[209,231] = 1
        struct[0].Gy[210,182] = 1
        struct[0].Gy[210,183] = -1
        struct[0].Gy[210,232] = 1
        struct[0].Gy[211,183] = 1
        struct[0].Gy[211,184] = -1
        struct[0].Gy[211,211] = 1
        struct[0].Gy[212,184] = 1
        struct[0].Gy[212,185] = -1
        struct[0].Gy[212,233] = 1
        struct[0].Gy[213,185] = 1
        struct[0].Gy[213,193] = -1
        struct[0].Gy[213,234] = 1
        struct[0].Gy[214,186] = -1
        struct[0].Gy[214,193] = 1
        struct[0].Gy[214,235] = 1
        struct[0].Gy[215,186] = 1
        struct[0].Gy[215,187] = -1
        struct[0].Gy[215,236] = 1
        struct[0].Gy[216,187] = 1
        struct[0].Gy[216,188] = -1
        struct[0].Gy[216,216] = 1
        struct[0].Gy[217,188] = 1
        struct[0].Gy[217,189] = -1
        struct[0].Gy[217,237] = 1
        struct[0].Gy[218,189] = 1
        struct[0].Gy[218,238] = 1
        struct[0].Gy[219,194] = i_11
        struct[0].Gy[219,219] = v_11
        struct[0].Gy[220,195] = i_12
        struct[0].Gy[220,220] = v_12
        struct[0].Gy[221,197] = i_14
        struct[0].Gy[221,221] = v_14
        struct[0].Gy[222,198] = i_15
        struct[0].Gy[222,222] = v_15
        struct[0].Gy[223,199] = i_21
        struct[0].Gy[223,223] = v_21
        struct[0].Gy[224,200] = i_22
        struct[0].Gy[224,224] = v_22
        struct[0].Gy[225,202] = i_24
        struct[0].Gy[225,225] = v_24
        struct[0].Gy[226,203] = i_25
        struct[0].Gy[226,226] = v_25
        struct[0].Gy[227,204] = i_31
        struct[0].Gy[227,227] = v_31
        struct[0].Gy[228,205] = i_32
        struct[0].Gy[228,228] = v_32
        struct[0].Gy[229,207] = i_34
        struct[0].Gy[229,229] = v_34
        struct[0].Gy[230,208] = i_35
        struct[0].Gy[230,230] = v_35
        struct[0].Gy[231,209] = i_41
        struct[0].Gy[231,231] = v_41
        struct[0].Gy[232,210] = i_42
        struct[0].Gy[232,232] = v_42
        struct[0].Gy[233,212] = i_44
        struct[0].Gy[233,233] = v_44
        struct[0].Gy[234,213] = i_45
        struct[0].Gy[234,234] = v_45
        struct[0].Gy[235,214] = i_51
        struct[0].Gy[235,235] = v_51
        struct[0].Gy[236,215] = i_52
        struct[0].Gy[236,236] = v_52
        struct[0].Gy[237,217] = i_54
        struct[0].Gy[237,237] = v_54
        struct[0].Gy[238,218] = i_55
        struct[0].Gy[238,238] = v_55
        struct[0].Gy[239,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[239,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[239,196] = v_13/3
        struct[0].Gy[239,239] = 1
        struct[0].Gy[240,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[240,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[240,196] = v_13/3
        struct[0].Gy[240,240] = 1
        struct[0].Gy[241,130] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_r/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_r, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[241,131] = Piecewise(np.array([(6*b_B2lv*i_load_B2lv_a_i/sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1) + 12*c_B2lv*i_load_B2lv_a_i, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gy[241,196] = v_13/3
        struct[0].Gy[241,241] = 1
        struct[0].Gy[242,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[242,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[242,201] = v_23/3
        struct[0].Gy[242,242] = 1
        struct[0].Gy[243,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[243,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[243,201] = v_23/3
        struct[0].Gy[243,243] = 1
        struct[0].Gy[244,138] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_r/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_r, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[244,139] = Piecewise(np.array([(6*b_B3lv*i_load_B3lv_a_i/sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1) + 12*c_B3lv*i_load_B3lv_a_i, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gy[244,201] = v_23/3
        struct[0].Gy[244,244] = 1
        struct[0].Gy[245,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[245,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[245,206] = v_33/3
        struct[0].Gy[245,245] = 1
        struct[0].Gy[246,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[246,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[246,206] = v_33/3
        struct[0].Gy[246,246] = 1
        struct[0].Gy[247,146] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_r/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_r, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[247,147] = Piecewise(np.array([(6*b_B4lv*i_load_B4lv_a_i/sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1) + 12*c_B4lv*i_load_B4lv_a_i, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gy[247,206] = v_33/3
        struct[0].Gy[247,247] = 1
        struct[0].Gy[248,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[248,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[248,211] = v_43/3
        struct[0].Gy[248,248] = 1
        struct[0].Gy[249,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[249,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[249,211] = v_43/3
        struct[0].Gy[249,249] = 1
        struct[0].Gy[250,154] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_r/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_r, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[250,155] = Piecewise(np.array([(6*b_B5lv*i_load_B5lv_a_i/sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1) + 12*c_B5lv*i_load_B5lv_a_i, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gy[250,211] = v_43/3
        struct[0].Gy[250,250] = 1
        struct[0].Gy[251,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[251,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[251,216] = v_53/3
        struct[0].Gy[251,251] = 1
        struct[0].Gy[252,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[252,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[252,216] = v_53/3
        struct[0].Gy[252,252] = 1
        struct[0].Gy[253,162] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_r/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_r, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[253,163] = Piecewise(np.array([(6*b_B6lv*i_load_B6lv_a_i/sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1) + 12*c_B6lv*i_load_B6lv_a_i, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gy[253,216] = v_53/3
        struct[0].Gy[253,253] = 1

        struct[0].Gu[40,0] = 1.10755301189314
        struct[0].Gu[40,1] = 0.598820527961361
        struct[0].Gu[40,2] = -0.316443717683753
        struct[0].Gu[40,3] = -0.171091579417532
        struct[0].Gu[40,4] = -0.316443717683753
        struct[0].Gu[40,5] = -0.171091579417532
        struct[0].Gu[41,0] = -0.598820527961361
        struct[0].Gu[41,1] = 1.10755301189314
        struct[0].Gu[41,2] = 0.171091579417532
        struct[0].Gu[41,3] = -0.316443717683753
        struct[0].Gu[41,4] = 0.171091579417532
        struct[0].Gu[41,5] = -0.316443717683753
        struct[0].Gu[42,0] = -0.316443717683753
        struct[0].Gu[42,1] = -0.171091579417532
        struct[0].Gu[42,2] = 1.10755301189314
        struct[0].Gu[42,3] = 0.598820527961360
        struct[0].Gu[42,4] = -0.316443717683753
        struct[0].Gu[42,5] = -0.171091579417531
        struct[0].Gu[43,0] = 0.171091579417532
        struct[0].Gu[43,1] = -0.316443717683753
        struct[0].Gu[43,2] = -0.598820527961360
        struct[0].Gu[43,3] = 1.10755301189314
        struct[0].Gu[43,4] = 0.171091579417531
        struct[0].Gu[43,5] = -0.316443717683753
        struct[0].Gu[44,0] = -0.316443717683753
        struct[0].Gu[44,1] = -0.171091579417532
        struct[0].Gu[44,2] = -0.316443717683753
        struct[0].Gu[44,3] = -0.171091579417531
        struct[0].Gu[44,4] = 1.10755301189314
        struct[0].Gu[44,5] = 0.598820527961360
        struct[0].Gu[45,0] = 0.171091579417532
        struct[0].Gu[45,1] = -0.316443717683753
        struct[0].Gu[45,2] = 0.171091579417531
        struct[0].Gu[45,3] = -0.316443717683753
        struct[0].Gu[45,4] = -0.598820527961360
        struct[0].Gu[45,5] = 1.10755301189314
        struct[0].Gu[64,6] = 1.10755301189314
        struct[0].Gu[64,7] = 0.598820527961361
        struct[0].Gu[64,8] = -0.316443717683753
        struct[0].Gu[64,9] = -0.171091579417532
        struct[0].Gu[64,10] = -0.316443717683753
        struct[0].Gu[64,11] = -0.171091579417532
        struct[0].Gu[65,6] = -0.598820527961361
        struct[0].Gu[65,7] = 1.10755301189314
        struct[0].Gu[65,8] = 0.171091579417532
        struct[0].Gu[65,9] = -0.316443717683753
        struct[0].Gu[65,10] = 0.171091579417532
        struct[0].Gu[65,11] = -0.316443717683753
        struct[0].Gu[66,6] = -0.316443717683753
        struct[0].Gu[66,7] = -0.171091579417532
        struct[0].Gu[66,8] = 1.10755301189314
        struct[0].Gu[66,9] = 0.598820527961360
        struct[0].Gu[66,10] = -0.316443717683753
        struct[0].Gu[66,11] = -0.171091579417531
        struct[0].Gu[67,6] = 0.171091579417532
        struct[0].Gu[67,7] = -0.316443717683753
        struct[0].Gu[67,8] = -0.598820527961360
        struct[0].Gu[67,9] = 1.10755301189314
        struct[0].Gu[67,10] = 0.171091579417531
        struct[0].Gu[67,11] = -0.316443717683753
        struct[0].Gu[68,6] = -0.316443717683753
        struct[0].Gu[68,7] = -0.171091579417532
        struct[0].Gu[68,8] = -0.316443717683753
        struct[0].Gu[68,9] = -0.171091579417531
        struct[0].Gu[68,10] = 1.10755301189314
        struct[0].Gu[68,11] = 0.598820527961360
        struct[0].Gu[69,6] = 0.171091579417532
        struct[0].Gu[69,7] = -0.316443717683753
        struct[0].Gu[69,8] = 0.171091579417531
        struct[0].Gu[69,9] = -0.316443717683753
        struct[0].Gu[69,10] = -0.598820527961360
        struct[0].Gu[69,11] = 1.10755301189314
        struct[0].Gu[100,0] = 1.10755301189314
        struct[0].Gu[100,1] = 0.598820527961361
        struct[0].Gu[100,2] = -0.316443717683753
        struct[0].Gu[100,3] = -0.171091579417532
        struct[0].Gu[100,4] = -0.316443717683753
        struct[0].Gu[100,5] = -0.171091579417532
        struct[0].Gu[101,0] = -0.598820527961361
        struct[0].Gu[101,1] = 1.10755301189314
        struct[0].Gu[101,2] = 0.171091579417532
        struct[0].Gu[101,3] = -0.316443717683753
        struct[0].Gu[101,4] = 0.171091579417532
        struct[0].Gu[101,5] = -0.316443717683753
        struct[0].Gu[102,0] = -0.316443717683753
        struct[0].Gu[102,1] = -0.171091579417532
        struct[0].Gu[102,2] = 1.10755301189314
        struct[0].Gu[102,3] = 0.598820527961360
        struct[0].Gu[102,4] = -0.316443717683753
        struct[0].Gu[102,5] = -0.171091579417531
        struct[0].Gu[103,0] = 0.171091579417532
        struct[0].Gu[103,1] = -0.316443717683753
        struct[0].Gu[103,2] = -0.598820527961360
        struct[0].Gu[103,3] = 1.10755301189314
        struct[0].Gu[103,4] = 0.171091579417531
        struct[0].Gu[103,5] = -0.316443717683753
        struct[0].Gu[104,0] = -0.316443717683753
        struct[0].Gu[104,1] = -0.171091579417532
        struct[0].Gu[104,2] = -0.316443717683753
        struct[0].Gu[104,3] = -0.171091579417531
        struct[0].Gu[104,4] = 1.10755301189314
        struct[0].Gu[104,5] = 0.598820527961360
        struct[0].Gu[105,0] = 0.171091579417532
        struct[0].Gu[105,1] = -0.316443717683753
        struct[0].Gu[105,2] = 0.171091579417531
        struct[0].Gu[105,3] = -0.316443717683753
        struct[0].Gu[105,4] = -0.598820527961360
        struct[0].Gu[105,5] = 1.10755301189314
        struct[0].Gu[124,6] = -1.10755301189314
        struct[0].Gu[124,7] = -0.598820527961361
        struct[0].Gu[124,8] = 0.316443717683753
        struct[0].Gu[124,9] = 0.171091579417532
        struct[0].Gu[124,10] = 0.316443717683753
        struct[0].Gu[124,11] = 0.171091579417532
        struct[0].Gu[125,6] = 0.598820527961361
        struct[0].Gu[125,7] = -1.10755301189314
        struct[0].Gu[125,8] = -0.171091579417532
        struct[0].Gu[125,9] = 0.316443717683753
        struct[0].Gu[125,10] = -0.171091579417532
        struct[0].Gu[125,11] = 0.316443717683753
        struct[0].Gu[126,6] = 0.316443717683753
        struct[0].Gu[126,7] = 0.171091579417532
        struct[0].Gu[126,8] = -1.10755301189314
        struct[0].Gu[126,9] = -0.598820527961360
        struct[0].Gu[126,10] = 0.316443717683753
        struct[0].Gu[126,11] = 0.171091579417531
        struct[0].Gu[127,6] = -0.171091579417532
        struct[0].Gu[127,7] = 0.316443717683753
        struct[0].Gu[127,8] = 0.598820527961360
        struct[0].Gu[127,9] = -1.10755301189314
        struct[0].Gu[127,10] = -0.171091579417531
        struct[0].Gu[127,11] = 0.316443717683753
        struct[0].Gu[128,6] = 0.316443717683753
        struct[0].Gu[128,7] = 0.171091579417532
        struct[0].Gu[128,8] = 0.316443717683753
        struct[0].Gu[128,9] = 0.171091579417531
        struct[0].Gu[128,10] = -1.10755301189314
        struct[0].Gu[128,11] = -0.598820527961360
        struct[0].Gu[129,6] = -0.171091579417532
        struct[0].Gu[129,7] = 0.316443717683753
        struct[0].Gu[129,8] = -0.171091579417531
        struct[0].Gu[129,9] = 0.316443717683753
        struct[0].Gu[129,10] = 0.598820527961360
        struct[0].Gu[129,11] = -1.10755301189314
        struct[0].Gu[133,52] = -1
        struct[0].Gu[134,53] = -1
        struct[0].Gu[135,54] = -1
        struct[0].Gu[141,55] = -1
        struct[0].Gu[142,56] = -1
        struct[0].Gu[143,57] = -1
        struct[0].Gu[149,58] = -1
        struct[0].Gu[150,59] = -1
        struct[0].Gu[151,60] = -1
        struct[0].Gu[157,61] = -1
        struct[0].Gu[158,62] = -1
        struct[0].Gu[159,63] = -1
        struct[0].Gu[165,64] = -1
        struct[0].Gu[166,65] = -1
        struct[0].Gu[167,66] = -1
        struct[0].Gu[239,76] = Piecewise(np.array([(6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[239,77] = Piecewise(np.array([(6*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[239,78] = Piecewise(np.array([(6*i_load_B2lv_a_i**2 + 6*i_load_B2lv_a_r**2 + 6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[240,76] = Piecewise(np.array([(6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[240,77] = Piecewise(np.array([(6*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[240,78] = Piecewise(np.array([(6*i_load_B2lv_a_i**2 + 6*i_load_B2lv_a_r**2 + 6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[241,76] = Piecewise(np.array([(6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[241,77] = Piecewise(np.array([(6*sqrt(i_load_B2lv_a_i**2 + i_load_B2lv_a_r**2 + 1), (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[241,78] = Piecewise(np.array([(6*i_load_B2lv_a_i**2 + 6*i_load_B2lv_a_r**2 + 6, (i_13*v_13 > 0) | (i_13*v_13 < 1)), (0, True)]))/3
        struct[0].Gu[242,79] = Piecewise(np.array([(6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[242,80] = Piecewise(np.array([(6*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[242,81] = Piecewise(np.array([(6*i_load_B3lv_a_i**2 + 6*i_load_B3lv_a_r**2 + 6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[243,79] = Piecewise(np.array([(6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[243,80] = Piecewise(np.array([(6*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[243,81] = Piecewise(np.array([(6*i_load_B3lv_a_i**2 + 6*i_load_B3lv_a_r**2 + 6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[244,79] = Piecewise(np.array([(6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[244,80] = Piecewise(np.array([(6*sqrt(i_load_B3lv_a_i**2 + i_load_B3lv_a_r**2 + 1), (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[244,81] = Piecewise(np.array([(6*i_load_B3lv_a_i**2 + 6*i_load_B3lv_a_r**2 + 6, (i_23*v_23 > 0) | (i_23*v_23 < 1)), (0, True)]))/3
        struct[0].Gu[245,82] = Piecewise(np.array([(6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[245,83] = Piecewise(np.array([(6*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[245,84] = Piecewise(np.array([(6*i_load_B4lv_a_i**2 + 6*i_load_B4lv_a_r**2 + 6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[246,82] = Piecewise(np.array([(6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[246,83] = Piecewise(np.array([(6*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[246,84] = Piecewise(np.array([(6*i_load_B4lv_a_i**2 + 6*i_load_B4lv_a_r**2 + 6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[247,82] = Piecewise(np.array([(6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[247,83] = Piecewise(np.array([(6*sqrt(i_load_B4lv_a_i**2 + i_load_B4lv_a_r**2 + 1), (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[247,84] = Piecewise(np.array([(6*i_load_B4lv_a_i**2 + 6*i_load_B4lv_a_r**2 + 6, (i_33*v_33 > 0) | (i_33*v_33 < 1)), (0, True)]))/3
        struct[0].Gu[248,85] = Piecewise(np.array([(6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[248,86] = Piecewise(np.array([(6*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[248,87] = Piecewise(np.array([(6*i_load_B5lv_a_i**2 + 6*i_load_B5lv_a_r**2 + 6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[249,85] = Piecewise(np.array([(6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[249,86] = Piecewise(np.array([(6*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[249,87] = Piecewise(np.array([(6*i_load_B5lv_a_i**2 + 6*i_load_B5lv_a_r**2 + 6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[250,85] = Piecewise(np.array([(6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[250,86] = Piecewise(np.array([(6*sqrt(i_load_B5lv_a_i**2 + i_load_B5lv_a_r**2 + 1), (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[250,87] = Piecewise(np.array([(6*i_load_B5lv_a_i**2 + 6*i_load_B5lv_a_r**2 + 6, (i_43*v_43 > 0) | (i_43*v_43 < 1)), (0, True)]))/3
        struct[0].Gu[251,88] = Piecewise(np.array([(6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[251,89] = Piecewise(np.array([(6*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[251,90] = Piecewise(np.array([(6*i_load_B6lv_a_i**2 + 6*i_load_B6lv_a_r**2 + 6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[252,88] = Piecewise(np.array([(6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[252,89] = Piecewise(np.array([(6*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[252,90] = Piecewise(np.array([(6*i_load_B6lv_a_i**2 + 6*i_load_B6lv_a_r**2 + 6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[253,88] = Piecewise(np.array([(6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[253,89] = Piecewise(np.array([(6*sqrt(i_load_B6lv_a_i**2 + i_load_B6lv_a_r**2 + 1), (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3
        struct[0].Gu[253,90] = Piecewise(np.array([(6*i_load_B6lv_a_i**2 + 6*i_load_B6lv_a_r**2 + 6, (i_53*v_53 > 0) | (i_53*v_53 < 1)), (0, True)]))/3





@numba.njit(cache=True)
def Piecewise(arg):
    out = arg[0][1]
    N = len(arg)
    for it in range(N-1,-1,-1):
        if arg[it][1]: out = arg[it][0]
    return out

@numba.njit(cache=True)
def ITE(arg):
    out = arg[0][1]
    N = len(arg)
    for it in range(N-1,-1,-1):
        if arg[it][1]: out = arg[it][0]
    return out


@numba.njit(cache=True)
def Abs(x):
    return np.abs(x)


@numba.njit(cache=True)
def ini_dae_jacobian_numba(struct,x):
    N_x = struct[0].N_x
    N_y = struct[0].N_y
    struct[0].x[:,0] = x[0:N_x]
    struct[0].y_ini[:,0] = x[N_x:(N_x+N_y)]

    ini(struct,10)
    ini(struct,11) 

    for row,col in zip(struct[0].Fx_ini_rows,struct[0].Fx_ini_cols):
        struct[0].Ac_ini[row,col] = struct[0].Fx_ini[row,col]
    for row,col in zip(struct[0].Fy_ini_rows,struct[0].Fy_ini_cols):
        struct[0].Ac_ini[row,col+N_x] = struct[0].Fy_ini[row,col]
    for row,col in zip(struct[0].Gx_ini_rows,struct[0].Gx_ini_cols):
        struct[0].Ac_ini[row+N_x,col] = struct[0].Gx_ini[row,col]
    for row,col in zip(struct[0].Gy_ini_rows,struct[0].Gy_ini_cols):
        struct[0].Ac_ini[row+N_x,col+N_x] = struct[0].Gy_ini[row,col]
        

@numba.njit(cache=True)
def ini_dae_problem(struct,x):
    N_x = struct[0].N_x
    N_y = struct[0].N_y
    struct[0].x[:,0] = x[0:N_x]
    struct[0].y_ini[:,0] = x[N_x:(N_x+N_y)]

    ini(struct,2)
    ini(struct,3) 
    struct[0].fg[:N_x,:] = struct[0].f[:]
    struct[0].fg[N_x:,:] = struct[0].g[:]    
        
@numba.njit(cache=True)
def ssate(struct,xy):
    for it in range(100):
        ini_dae_jacobian_numba(struct,xy[:,0])
        ini_dae_problem(struct,xy[:,0])
        xy[:] += np.linalg.solve(struct[0].Ac_ini,-struct[0].fg)
        if np.max(np.abs(struct[0].fg[:,0]))<1e-8: break
    N_x = struct[0].N_x
    struct[0].x[:,0] = xy[:N_x,0]
    struct[0].y_ini[:,0] = xy[N_x:,0]
    return xy,it


@numba.njit(cache=True) 
def daesolver(struct): 
    sin = np.sin
    cos = np.cos
    sqrt = np.sqrt
    i = 0 
    
    Dt = struct[i].Dt 

    N_x = struct[i].N_x
    N_y = struct[i].N_y
    N_z = struct[i].N_z

    decimation = struct[i].decimation 
    eye = np.eye(N_x)
    t = struct[i].t 
    t_end = struct[i].t_end 
    if struct[i].it == 0:
        run(t,struct, 1) 
        struct[i].it_store = 0  
        struct[i]['T'][0] = t 
        struct[i].X[0,:] = struct[i].x[:,0]  
        struct[i].Y[0,:] = struct[i].y_run[:,0]  
        struct[i].Z[0,:] = struct[i].h[:,0]  

    solver = struct[i].solvern 
    while t<t_end: 
        struct[i].it += 1
        struct[i].t += Dt
        
        t = struct[i].t


            
        if solver == 5: # Teapezoidal DAE as in Milano's book

            run(t,struct, 2) 
            run(t,struct, 3) 

            x = np.copy(struct[i].x[:]) 
            y = np.copy(struct[i].y_run[:]) 
            f = np.copy(struct[i].f[:]) 
            g = np.copy(struct[i].g[:]) 
            
            for iter in range(struct[i].imax):
                run(t,struct, 2) 
                run(t,struct, 3) 
                run(t,struct,10) 
                run(t,struct,11) 
                
                x_i = struct[i].x[:] 
                y_i = struct[i].y_run[:]  
                f_i = struct[i].f[:] 
                g_i = struct[i].g[:]                 
                F_x_i = struct[i].Fx[:,:]
                F_y_i = struct[i].Fy[:,:] 
                G_x_i = struct[i].Gx[:,:] 
                G_y_i = struct[i].Gy[:,:]                

                A_c_i = np.vstack((np.hstack((eye-0.5*Dt*F_x_i, -0.5*Dt*F_y_i)),
                                   np.hstack((G_x_i,         G_y_i))))
                     
                f_n_i = x_i - x - 0.5*Dt*(f_i+f) 
                # print(t,iter,g_i)
                Dxy_i = np.linalg.solve(-A_c_i,np.vstack((f_n_i,g_i))) 
                
                x_i = x_i + Dxy_i[0:N_x]
                y_i = y_i + Dxy_i[N_x:(N_x+N_y)]

                struct[i].x[:] = x_i
                struct[i].y_run[:] = y_i

        # [f_i,g_i,F_x_i,F_y_i,G_x_i,G_y_i] =  smib_transient(x_i,y_i,u);
        
        # A_c_i = [[eye(N_x)-0.5*Dt*F_x_i, -0.5*Dt*F_y_i],
        #          [                G_x_i,         G_y_i]];
             
        # f_n_i = x_i - x - 0.5*Dt*(f_i+f);
        
        # Dxy_i = -A_c_i\[f_n_i.',g_i.'].';
        
        # x_i = x_i + Dxy_i(1:N_x);
        # y_i = y_i + Dxy_i(N_x+1:N_x+N_y);
                
                xy = np.vstack((x_i,y_i))
                max_relative = 0.0
                for it_var in range(N_x+N_y):
                    abs_value = np.abs(xy[it_var,0])
                    if abs_value < 0.001:
                        abs_value = 0.001
                                             
                    relative_error = np.abs(Dxy_i[it_var,0])/abs_value
                    
                    if relative_error > max_relative: max_relative = relative_error
                    
                if max_relative<struct[i].itol:
                    
                    break
                
                # if iter>struct[i].imax-2:
                    
                #     print('Convergence problem')

            struct[i].x[:] = x_i
            struct[i].y_run[:] = y_i
                
        # channels 
        if struct[i].store == 1:
            it_store = struct[i].it_store
            if struct[i].it >= it_store*decimation: 
                struct[i]['T'][it_store+1] = t 
                struct[i].X[it_store+1,:] = struct[i].x[:,0] 
                struct[i].Y[it_store+1,:] = struct[i].y_run[:,0]
                struct[i].Z[it_store+1,:] = struct[i].h[:,0]
                struct[i].iters[it_store+1,0] = iter
                struct[i].it_store += 1 
            
    struct[i].t = t

    return t





def nonzeros():
    Fx_ini_rows = [0, 1, 2, 3, 4]

    Fx_ini_cols = [0, 1, 2, 3, 4]

    Fy_ini_rows = [0, 1, 2, 3, 4]

    Fy_ini_cols = [196, 201, 206, 211, 216]

    Gx_ini_rows = [171, 172, 175, 176, 179, 180, 183, 184, 187, 188, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253]

    Gx_ini_cols = [0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 0, 0, 0, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4]

    Gy_ini_rows = [0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 14, 14, 14, 14, 14, 14, 14, 14, 15, 15, 15, 15, 15, 15, 15, 15, 16, 16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 17, 17, 17, 17, 17, 18, 18, 18, 18, 18, 18, 18, 18, 18, 19, 19, 19, 19, 19, 19, 19, 19, 19, 20, 20, 20, 20, 20, 20, 20, 20, 20, 21, 21, 21, 21, 21, 21, 21, 21, 21, 22, 22, 22, 22, 22, 22, 22, 22, 23, 23, 23, 23, 23, 23, 23, 23, 24, 24, 24, 24, 24, 24, 24, 24, 24, 25, 25, 25, 25, 25, 25, 25, 25, 25, 26, 26, 26, 26, 26, 26, 26, 26, 26, 27, 27, 27, 27, 27, 27, 27, 27, 27, 28, 28, 28, 28, 28, 28, 28, 28, 28, 29, 29, 29, 29, 29, 29, 29, 29, 29, 30, 30, 30, 30, 30, 30, 30, 30, 31, 31, 31, 31, 31, 31, 31, 31, 32, 32, 32, 32, 32, 32, 32, 32, 32, 33, 33, 33, 33, 33, 33, 33, 33, 33, 34, 34, 34, 34, 34, 34, 34, 34, 34, 35, 35, 35, 35, 35, 35, 35, 35, 35, 36, 36, 36, 36, 36, 36, 36, 36, 36, 37, 37, 37, 37, 37, 37, 37, 37, 37, 38, 38, 38, 38, 38, 38, 38, 38, 39, 39, 39, 39, 39, 39, 39, 39, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 53, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 75, 75, 75, 75, 75, 75, 75, 75, 75, 75, 75, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 81, 81, 81, 81, 81, 81, 81, 81, 81, 81, 81, 82, 82, 82, 82, 82, 82, 82, 82, 82, 82, 82, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 89, 89, 89, 89, 89, 89, 89, 89, 89, 89, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 96, 96, 96, 96, 96, 96, 96, 96, 96, 96, 96, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 100, 100, 100, 100, 100, 100, 100, 101, 101, 101, 101, 101, 101, 101, 102, 102, 102, 102, 102, 102, 102, 103, 103, 103, 103, 103, 103, 103, 104, 104, 104, 104, 104, 104, 104, 105, 105, 105, 105, 105, 105, 105, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 117, 117, 117, 117, 117, 117, 117, 117, 117, 117, 117, 117, 117, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 121, 121, 121, 121, 121, 121, 121, 121, 121, 121, 121, 121, 121, 122, 122, 122, 122, 122, 122, 122, 122, 122, 122, 122, 122, 122, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 124, 124, 124, 124, 124, 124, 124, 125, 125, 125, 125, 125, 125, 125, 126, 126, 126, 126, 126, 126, 126, 127, 127, 127, 127, 127, 127, 127, 128, 128, 128, 128, 128, 128, 128, 129, 129, 129, 129, 129, 129, 129, 130, 130, 130, 130, 130, 130, 130, 131, 131, 131, 131, 131, 131, 131, 132, 132, 132, 132, 132, 132, 132, 133, 133, 133, 133, 133, 133, 134, 134, 134, 134, 134, 134, 135, 135, 135, 135, 135, 135, 136, 136, 136, 136, 137, 137, 137, 137, 138, 138, 138, 138, 138, 138, 138, 139, 139, 139, 139, 139, 139, 139, 140, 140, 140, 140, 140, 140, 140, 141, 141, 141, 141, 141, 141, 142, 142, 142, 142, 142, 142, 143, 143, 143, 143, 143, 143, 144, 144, 144, 144, 145, 145, 145, 145, 146, 146, 146, 146, 146, 146, 146, 147, 147, 147, 147, 147, 147, 147, 148, 148, 148, 148, 148, 148, 148, 149, 149, 149, 149, 149, 149, 150, 150, 150, 150, 150, 150, 151, 151, 151, 151, 151, 151, 152, 152, 152, 152, 153, 153, 153, 153, 154, 154, 154, 154, 154, 154, 154, 155, 155, 155, 155, 155, 155, 155, 156, 156, 156, 156, 156, 156, 156, 157, 157, 157, 157, 157, 157, 158, 158, 158, 158, 158, 158, 159, 159, 159, 159, 159, 159, 160, 160, 160, 160, 161, 161, 161, 161, 162, 162, 162, 162, 162, 162, 162, 163, 163, 163, 163, 163, 163, 163, 164, 164, 164, 164, 164, 164, 164, 165, 165, 165, 165, 165, 165, 166, 166, 166, 166, 166, 166, 167, 167, 167, 167, 167, 167, 168, 168, 168, 168, 169, 169, 169, 169, 170, 170, 170, 171, 171, 172, 172, 173, 173, 173, 174, 174, 174, 175, 175, 176, 176, 177, 177, 177, 178, 178, 178, 179, 179, 180, 180, 181, 181, 181, 182, 182, 182, 183, 183, 184, 184, 185, 185, 185, 186, 186, 186, 187, 187, 188, 188, 189, 189, 189, 190, 190, 190, 191, 191, 191, 192, 192, 192, 193, 193, 193, 194, 194, 195, 195, 195, 196, 196, 196, 197, 197, 197, 198, 198, 198, 199, 199, 199, 200, 200, 200, 201, 201, 201, 202, 202, 202, 203, 203, 203, 204, 204, 204, 205, 205, 205, 206, 206, 206, 207, 207, 207, 208, 208, 208, 209, 209, 209, 210, 210, 210, 211, 211, 211, 212, 212, 212, 213, 213, 213, 214, 214, 214, 215, 215, 215, 216, 216, 216, 217, 217, 217, 218, 218, 219, 219, 220, 220, 221, 221, 222, 222, 223, 223, 224, 224, 225, 225, 226, 226, 227, 227, 228, 228, 229, 229, 230, 230, 231, 231, 232, 232, 233, 233, 234, 234, 235, 235, 236, 236, 237, 237, 238, 238, 239, 239, 239, 239, 240, 240, 240, 240, 241, 241, 241, 241, 242, 242, 242, 242, 243, 243, 243, 243, 244, 244, 244, 244, 245, 245, 245, 245, 246, 246, 246, 246, 247, 247, 247, 247, 248, 248, 248, 248, 249, 249, 249, 249, 250, 250, 250, 250, 251, 251, 251, 251, 252, 252, 252, 252, 253, 253, 253, 253]

    Gy_ini_cols = [0, 1, 6, 7, 40, 41, 44, 45, 130, 0, 1, 6, 7, 40, 41, 44, 45, 131, 2, 3, 6, 7, 40, 41, 42, 43, 132, 2, 3, 6, 7, 40, 41, 42, 43, 133, 4, 5, 6, 7, 42, 43, 44, 45, 134, 4, 5, 6, 7, 42, 43, 44, 45, 135, 0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 14, 15, 46, 47, 50, 51, 138, 8, 9, 14, 15, 46, 47, 50, 51, 139, 10, 11, 14, 15, 46, 47, 48, 49, 140, 10, 11, 14, 15, 46, 47, 48, 49, 141, 12, 13, 14, 15, 48, 49, 50, 51, 142, 12, 13, 14, 15, 48, 49, 50, 51, 143, 8, 9, 10, 11, 12, 13, 14, 15, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 22, 23, 52, 53, 56, 57, 146, 16, 17, 22, 23, 52, 53, 56, 57, 147, 18, 19, 22, 23, 52, 53, 54, 55, 148, 18, 19, 22, 23, 52, 53, 54, 55, 149, 20, 21, 22, 23, 54, 55, 56, 57, 150, 20, 21, 22, 23, 54, 55, 56, 57, 151, 16, 17, 18, 19, 20, 21, 22, 23, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 30, 31, 58, 59, 62, 63, 154, 24, 25, 30, 31, 58, 59, 62, 63, 155, 26, 27, 30, 31, 58, 59, 60, 61, 156, 26, 27, 30, 31, 58, 59, 60, 61, 157, 28, 29, 30, 31, 60, 61, 62, 63, 158, 28, 29, 30, 31, 60, 61, 62, 63, 159, 24, 25, 26, 27, 28, 29, 30, 31, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 38, 39, 64, 65, 68, 69, 162, 32, 33, 38, 39, 64, 65, 68, 69, 163, 34, 35, 38, 39, 64, 65, 66, 67, 164, 34, 35, 38, 39, 64, 65, 66, 67, 165, 36, 37, 38, 39, 66, 67, 68, 69, 166, 36, 37, 38, 39, 66, 67, 68, 69, 167, 32, 33, 34, 35, 36, 37, 38, 39, 32, 33, 34, 35, 36, 37, 38, 39, 0, 1, 2, 3, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 0, 1, 2, 3, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 2, 3, 4, 5, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 2, 3, 4, 5, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 0, 1, 4, 5, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 0, 1, 4, 5, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 8, 9, 10, 11, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 8, 9, 10, 11, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 10, 11, 12, 13, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 10, 11, 12, 13, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 8, 9, 12, 13, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 8, 9, 12, 13, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 16, 17, 18, 19, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 16, 17, 18, 19, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 18, 19, 20, 21, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 18, 19, 20, 21, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 16, 17, 20, 21, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 16, 17, 20, 21, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 24, 25, 26, 27, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 24, 25, 26, 27, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 26, 27, 28, 29, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 26, 27, 28, 29, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 24, 25, 28, 29, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 24, 25, 28, 29, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 32, 33, 34, 35, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 32, 33, 34, 35, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 34, 35, 36, 37, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 34, 35, 36, 37, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 32, 33, 36, 37, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 32, 33, 36, 37, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 0, 1, 2, 3, 40, 41, 42, 43, 44, 45, 70, 0, 1, 2, 3, 40, 41, 42, 43, 44, 45, 71, 2, 3, 4, 5, 40, 41, 42, 43, 44, 45, 72, 2, 3, 4, 5, 40, 41, 42, 43, 44, 45, 73, 0, 1, 4, 5, 40, 41, 42, 43, 44, 45, 74, 0, 1, 4, 5, 40, 41, 42, 43, 44, 45, 75, 8, 9, 10, 11, 46, 47, 48, 49, 50, 51, 76, 8, 9, 10, 11, 46, 47, 48, 49, 50, 51, 77, 10, 11, 12, 13, 46, 47, 48, 49, 50, 51, 78, 10, 11, 12, 13, 46, 47, 48, 49, 50, 51, 79, 8, 9, 12, 13, 46, 47, 48, 49, 50, 51, 80, 8, 9, 12, 13, 46, 47, 48, 49, 50, 51, 81, 16, 17, 18, 19, 52, 53, 54, 55, 56, 57, 82, 16, 17, 18, 19, 52, 53, 54, 55, 56, 57, 83, 18, 19, 20, 21, 52, 53, 54, 55, 56, 57, 84, 18, 19, 20, 21, 52, 53, 54, 55, 56, 57, 85, 16, 17, 20, 21, 52, 53, 54, 55, 56, 57, 86, 16, 17, 20, 21, 52, 53, 54, 55, 56, 57, 87, 24, 25, 26, 27, 58, 59, 60, 61, 62, 63, 88, 24, 25, 26, 27, 58, 59, 60, 61, 62, 63, 89, 26, 27, 28, 29, 58, 59, 60, 61, 62, 63, 90, 26, 27, 28, 29, 58, 59, 60, 61, 62, 63, 91, 24, 25, 28, 29, 58, 59, 60, 61, 62, 63, 92, 24, 25, 28, 29, 58, 59, 60, 61, 62, 63, 93, 32, 33, 34, 35, 64, 65, 66, 67, 68, 69, 94, 32, 33, 34, 35, 64, 65, 66, 67, 68, 69, 95, 34, 35, 36, 37, 64, 65, 66, 67, 68, 69, 96, 34, 35, 36, 37, 64, 65, 66, 67, 68, 69, 97, 32, 33, 36, 37, 64, 65, 66, 67, 68, 69, 98, 32, 33, 36, 37, 64, 65, 66, 67, 68, 69, 99, 40, 41, 42, 43, 44, 45, 100, 40, 41, 42, 43, 44, 45, 101, 40, 41, 42, 43, 44, 45, 102, 40, 41, 42, 43, 44, 45, 103, 40, 41, 42, 43, 44, 45, 104, 40, 41, 42, 43, 44, 45, 105, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 106, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 107, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 108, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 109, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 110, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 111, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 112, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 113, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 114, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 115, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 116, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 117, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 118, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 119, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 120, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 121, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 122, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 123, 64, 65, 66, 67, 68, 69, 124, 64, 65, 66, 67, 68, 69, 125, 64, 65, 66, 67, 68, 69, 126, 64, 65, 66, 67, 68, 69, 127, 64, 65, 66, 67, 68, 69, 128, 64, 65, 66, 67, 68, 69, 129, 0, 1, 6, 7, 130, 131, 239, 2, 3, 6, 7, 132, 133, 240, 4, 5, 6, 7, 134, 135, 241, 0, 1, 6, 7, 130, 131, 2, 3, 6, 7, 132, 133, 4, 5, 6, 7, 134, 135, 130, 132, 134, 136, 131, 133, 135, 137, 8, 9, 14, 15, 138, 139, 242, 10, 11, 14, 15, 140, 141, 243, 12, 13, 14, 15, 142, 143, 244, 8, 9, 14, 15, 138, 139, 10, 11, 14, 15, 140, 141, 12, 13, 14, 15, 142, 143, 138, 140, 142, 144, 139, 141, 143, 145, 16, 17, 22, 23, 146, 147, 245, 18, 19, 22, 23, 148, 149, 246, 20, 21, 22, 23, 150, 151, 247, 16, 17, 22, 23, 146, 147, 18, 19, 22, 23, 148, 149, 20, 21, 22, 23, 150, 151, 146, 148, 150, 152, 147, 149, 151, 153, 24, 25, 30, 31, 154, 155, 248, 26, 27, 30, 31, 156, 157, 249, 28, 29, 30, 31, 158, 159, 250, 24, 25, 30, 31, 154, 155, 26, 27, 30, 31, 156, 157, 28, 29, 30, 31, 158, 159, 154, 156, 158, 160, 155, 157, 159, 161, 32, 33, 38, 39, 162, 163, 251, 34, 35, 38, 39, 164, 165, 252, 36, 37, 38, 39, 166, 167, 253, 32, 33, 38, 39, 162, 163, 34, 35, 38, 39, 164, 165, 36, 37, 38, 39, 166, 167, 162, 164, 166, 168, 163, 165, 167, 169, 170, 194, 195, 171, 195, 172, 197, 173, 197, 198, 174, 199, 200, 175, 200, 176, 202, 177, 202, 203, 178, 204, 205, 179, 205, 180, 207, 181, 207, 208, 182, 209, 210, 183, 210, 184, 212, 185, 212, 213, 186, 214, 215, 187, 215, 188, 217, 189, 217, 218, 190, 198, 199, 191, 203, 204, 192, 208, 209, 193, 213, 214, 170, 219, 170, 171, 220, 171, 172, 196, 172, 173, 221, 173, 190, 222, 174, 190, 223, 174, 175, 224, 175, 176, 201, 176, 177, 225, 177, 191, 226, 178, 191, 227, 178, 179, 228, 179, 180, 206, 180, 181, 229, 181, 192, 230, 182, 192, 231, 182, 183, 232, 183, 184, 211, 184, 185, 233, 185, 193, 234, 186, 193, 235, 186, 187, 236, 187, 188, 216, 188, 189, 237, 189, 238, 194, 219, 195, 220, 197, 221, 198, 222, 199, 223, 200, 224, 202, 225, 203, 226, 204, 227, 205, 228, 207, 229, 208, 230, 209, 231, 210, 232, 212, 233, 213, 234, 214, 235, 215, 236, 217, 237, 218, 238, 130, 131, 196, 239, 130, 131, 196, 240, 130, 131, 196, 241, 138, 139, 201, 242, 138, 139, 201, 243, 138, 139, 201, 244, 146, 147, 206, 245, 146, 147, 206, 246, 146, 147, 206, 247, 154, 155, 211, 248, 154, 155, 211, 249, 154, 155, 211, 250, 162, 163, 216, 251, 162, 163, 216, 252, 162, 163, 216, 253]

    return Fx_ini_rows,Fx_ini_cols,Fy_ini_rows,Fy_ini_cols,Gx_ini_rows,Gx_ini_cols,Gy_ini_rows,Gy_ini_cols