# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ebay_feedsdk',
 'ebay_feedsdk.config',
 'ebay_feedsdk.constants',
 'ebay_feedsdk.enums',
 'ebay_feedsdk.errors',
 'ebay_feedsdk.examples',
 'ebay_feedsdk.feed',
 'ebay_feedsdk.filter',
 'ebay_feedsdk.tests',
 'ebay_feedsdk.utils']

package_data = \
{'': ['*'],
 'ebay_feedsdk': ['sample-config/*'],
 'ebay_feedsdk.tests': ['test-data/*']}

setup_kwargs = {
    'name': 'ebay-feedsdk',
    'version': '0.1.1',
    'description': 'Port of https://github.com/eBay/FeedSDK-Python to python3',
    'long_description': 'Feed SDK\n==========\nPython SDK for downloading and filtering item feed files\n\nForked from [https://github.com/eBay/FeedSDK-Python](https://github.com/eBay/FeedSDK-Python) and ported to python3\n\nAvailable as PyPI package under https://pypi.org/project/ebay-feedsdk/\n\n\nTable of contents\n==========\n* [Summary](#summary)\n* [Setup](#setup)\n    - [Setting up in the local environment](#setting-up-in-the-local-environment)\n* [Downloading feed files](#downloading-feed-files)\n    - [Customizing download location](#customizing-download-location)\n* [Filtering feed files](#filtering-feed-files)\n    - [Available filters](#available-filters)\n    - [Combining filter criteria](#combining-filter-criteria)\n    - [Additional filter arguments](#additional-filter-arguments)\n* [Schemas](#schemas)\n    - [GetFeedResponse](#getfeedresponse)\n    - [Response](#response)\n* [Logging](#logging)\n* [Usage](#usage)\n    - [Using command line options](#using-command-line-options)\n    - [Using config file driven approach](#using-config-file-driven-approach)\n    - [Using function calls](#using-function-calls)\n        - [Code samples](#examples)\n* [Performance](#performance)\n* [Important notes](#important-notes)\n\n# Summary\n\nSimilar to [Java Feed SDK](https://github.com/eBay/FeedSDK), this Python SDK facilitates download and filtering of eBay\'s item feed files provided through public [Feed API](https://developer.ebay.com/api-docs/buy/feed/overview.html).\n\nThe feed SDK provides a simple interface to -\n* [Download](#downloading-feed-files)\n* [Filter](#filtering-feed-files)\n\n# Setup\n\nThe the entire repository can be cloned/forked and changes can be made. You are most welcome to collaborate and enhance the existing code base.\n\n## Setting up in the local environment\n\nFor setting up the project in your local environment\n* Clone or download the repository\n* Install the requirements\nTo set up your environment, please see the requirements listed in [requirements.txt](https://github.com/eBay/FeedSDK-Python/blob/master/requirements.txt). You can run $ pip install -r requirements.txt command to install all the requirements.\n\n\n## Downloading feed files\nThe feed files can be as big as several gigabytes. Feed API supports downloading such big feed files in chunks. Chunk size is 100 MB in production environment and is 10 MB in soundbox environment.\n\nThe SDK abstracts the complexity involved in calculating the request header \'__range__\' based on the response header \'__content-range__\' and downloads and appends all the chunks until the whole feed file is downloaded.\n\nTo download a feed file in production which is -\n* __bootstrap__ : (feed_scope = ALL_ACTIVE)\n* __L1 category 1__ : (category_id = 220)\n* __marketplace US__ : (X-EBAY-C-MARKETPLACE-ID: EBAY_US)\ninstantiate a Feed object and call get() function\n\n```\nfeed_obj = Feed(feed_type=\'item\', feed_scope=\'ALL_ACTIVE\', category_id=\'220\', \n\t\t\t       marketplace_id=\'EBAY_US\', token=<TOKEN>, environment=\'PRODUCTION\')\nresult_code, api_status_code, file_path = feed_obj.get()\n\n```\nThe __filePath__ denotes the location where the file was downloaded.\n\n### Customizing download location\n\nThe default download location is ~/Desktop/feed-sdk directory. If the directory does not exist, it will be created.\nThe download location can be changed by specifying the optional \'download_location\' argument when instantiating Feed.\nThe download location should point to a directory. If the directory does not exist, it will be created.\nFor example, to download to the location __/tmp/feed__ - \n\n```\nfeed_obj = Feed(feed_type=\'item\', feed_scope=\'ALL_ACTIVE\', category_id=\'220\', \n                               marketplace_id=\'EBAY_US\', token=<TOKEN>, environment=\'PRODUCTION\',\n\t\t\t       download_location=\'/tmp/feed\')\n```\n---\n\n## Filtering feed files\n\n### Available filters\nThe SDK provides the capability to filter the feed files based on :-\n* List of leaf category ids\n* List of seller usernames\n* List of item locations\n* List of item IDs\n* List of EPIDs\n* List of inferred EPIDs\n* List of GTINs\n* Price range\n* Any other SQL query\n\nOn successful completion of a filter operation, a new __filtered__ file is created in the same directory as the feed file\'s.\n\nTo filter a feed file on leaf category IDs create an object of FeedFilterRequest and call filter() function - \n```\nfeed_filter_obj = FeedFilterRequest(input_fila_path=<absolute path to the feed file>, \n                                    leaf_category_ids=<list of leaf category IDs>)\nfile_path = feed_filter_obj.filter()\n\n```\n\nTo filter on availability threshold type and availability threshold via any_query parameter\n```\nfeed_filter_obj = FeedFilterRequest(input_fila_path=<absolute path to the feed file>,\n                                    any_query=\'AvailabilityThresholdType=\\\'MORE_THAN\\\' AND AvailabilityThreshold==10\')\nfile_path = feed_filter_obj.filter()\n\n```\n\nThe __file_path__ denotes the location of the filtered file. The file_path value can also be read by filter_request.filtered_file_path.\n\n### Combining filter criteria\n\nThe SDK provides the freedom to combine the filter criteria.\n\nTo filter on leaf category IDs and seller user names for listings in the price range of 1 to 100\n\n```\nfeed_filter_obj = FeedFilterRequest(input_fila_path=<absolute path to the feed file>,\n                                    leaf_category_ids=<list of leaf category IDs>, \n\t\t\t\t    seller_names=<list of seller names>,\n                                    price_lower_limit=1, price_upper_limit=100)\nfile_path = feed_filter_obj.filter()\n\n```\n\nTo filter on item location countries for listings that have more than 10 items available\n\n```\nfeed_filter_obj = FeedFilterRequest(input_fila_path=<absolute path to the feed file>,\n                                    item_location_countries=<list of item location countries>, \n                                    any_query=\'AvailabilityThresholdType=\\\'MORE_THAN\\\' AND AvailabilityThreshold=10\')\nfile_path = feed_filter_obj.filter()\n\n```\n\n### Additional filter arguments\nWhen filter function is called, feed data is loaded into a sqlite DB.\nIf keep_db=True argument is passed to filter function, the sqlite db file is kept in the current directory with name sqlite_feed_sdk.db, otherwise it will be deleted after the program execution.\n\nBy default all the columns except Title, ImageUrl, and AdditionalImageUrls are processed. This behaviour can be changed by passing column_name_list argument to filter function and changing IGNORE_COLUMNS set in feed_filter.py. \n\n---\n### Schemas\nThis section provides more detail on what information is contained within the objects returned from the SDK function calls.\n\n### GetFeedResponse\n\nAn instance of GetFeedResponse named tuple is returned from the feed_obj.get() function.\n\n```\n  int status_ode\n  String message\n  String file_path\n  List errors\n\n```\n\n| Field name | Description \n|---|---|\n| status_code | int: 0 indicates a successful response. Any non zero value indicates an error\n| message | String: Detailed information on the status\n| file_path | String: Absolute path of the location of the resulting file\n| errors | List: Detailed error information\n\n\n### Response \n\nAn instance of Response named tuple is returned from feed_filter_object.filter() function.\n\n```\n  int status_code\n  String message\n  String file_path\n  List applied_filters\n```\n| Field name | Description \n|---|---|\n| status_code | int: 0 indicates a successful response. Any non zero value indicates an error\n| message | String: Detailed information on the status\n| file_path | String: Absolute path of the location of the resulting file\n| applied_filters | List: List of queries applied\n\n---\n## Logging\n\nLog files are created in the current directory.\n\n__Ensure that appropriate permissions are present to write to the directory__\n\n* The current log file name is : feed-sdk-log.log\n* Rolling log files are created per day with the pattern : feed-sdk-log.{yyyy-MM-dd}.log\n\n---\n## Usage\n\nThe following sections describe the different ways in which the SDK can be used\n\n### Using command line options\n\nAll the capabilities of the SDK can be invoked using the command line.\n\nTo see the available options and filters , use \'--help\'\n```\nusage: FeedSDK [-h] [-dt DT] -c1 C1 [-scope {ALL_ACTIVE,NEWLY_LISTED}]\n               [-mkt MKT] [-token TOKEN] [-env {SANDBOX,PRODUCTION}]\n               [-lf LF [LF ...]] [-sellerf SELLERF [SELLERF ...]]\n               [-locf LOCF [LOCF ...]] [-pricelf PRICELF] [-priceuf PRICEUF]\n               [-epidf EPIDF [EPIDF ...]] [-iepidf IEPIDF [IEPIDF ...]]\n               [-gtinf GTINF [GTINF ...]] [-itemf ITEMF [ITEMF ...]]\n               [-dl DOWNLOADLOCATION] [--filteronly] [-format FORMAT] [-qf QF]\n\nFeed SDK CLI\n\noptional arguments:\n  -h, --help            show this help message and exit\n  -dt DT                the date when feed file was generated\n  -c1 C1                the l1 category id of the feed file\n  -scope {ALL_ACTIVE,NEWLY_LISTED}\n                        the feed scope. Available scopes are ALL_ACTIVE or\n                        NEWLY_LISTED\n  -mkt MKT              the marketplace id for which feed is being requested.\n                        For example - EBAY_US\n  -token TOKEN          the oauth token for the consumer. Omit the word\n                        \'Bearer\'\n  -env {SANDBOX,PRODUCTION}\n                        environment type. Supported Environments are SANDBOX\n                        and PRODUCTION\n  -lf LF [LF ...]       list of leaf categories which are used to filter the\n                        feed\n  -sellerf SELLERF [SELLERF ...]\n                        list of seller names which are used to filter the feed\n  -locf LOCF [LOCF ...]\n                        list of item locations which are used to filter the\n                        feed\n  -pricelf PRICELF      lower limit of the price range for items in the feed\n  -priceuf PRICEUF      upper limit of the price range for items in the feed\n  -epidf EPIDF [EPIDF ...]\n                        list of epids which are used to filter the feed\n  -iepidf IEPIDF [IEPIDF ...]\n                        list of inferred epids which are used to filter the\n                        feed\n  -gtinf GTINF [GTINF ...]\n                        list of gtins which are used to filter the feed\n  -itemf ITEMF [ITEMF ...]\n                        list of item IDs which are used to filter the feed\n  -dl DOWNLOADLOCATION, --downloadlocation DOWNLOADLOCATION\n                        override for changing the directory where files are\n                        downloaded\n  --filteronly          filter the feed file that already exists in the\n                        default path or the path specified by -dl,\n                        --downloadlocation option. If --filteronly option is\n                        not specified, the feed file will be downloaded again\n  -format FORMAT        feed and filter file format. Default is gzip\n  -qf QF                any other query to filter the feed file. See Python\n                        dataframe query format\n```\nFor example, to use the command line options to\n\nDownload and filter feed files using token\n```\npython feed_cli.py -c1 3252 -scope ALL_ACTIVE -mkt EBAY_DE -env PRODUCTION -qf "AvailabilityThreshold=10" -locf IT GB -dl DIR -token xxx\n```\n\nFilter feed files, no token is needed\n```\npython feed_cli.py --filteronly -c1 260 -pricelf 5 -priceuf 20 -dl FILE_PATH\n```\n\n### Using config file driven approach\n\nAll the capabilities of the SDK can be leveraged via a config file.\nThe feed file download and filter parameters can be specified in the config file for multiple files, and SDK will process them sequentially.\n\nThe structure of the config file\n\n```\n{\n  "requests": [\n    {\n      "feedRequest": {\n        "categoryId": "260",\n        "marketplaceId": "EBAY_US",\n        "feedScope": "ALL_ACTIVE",\n        "type": "ITEM"\n      },\n      "filterRequest": {\n        "itemLocationCountries": [\n          "US",\n          "HK",\n          "CA"\n        ],\n        "priceLowerLimit": 10.0,\n        "priceUpperLimit": 100.0\n      }\n    },\n    {\n      "feedRequest": {\n        "categoryId": "220",\n        "marketplaceId": "EBAY_US",\n        "date": "20190127",\n        "feedScope": "NEWLY_LISTED",\n        "type": "ITEM"\n      }\n    },\n    {\n      "filterRequest": {\n        "inputFilePath": "<Absolute file path to the feed file>",\n        "leafCategoryIds": [\n          "112529",\n          "64619",\n          "111694"\n        ],\n        "itemLocationCountries": [\n          "DE",\n          "GB",\n          "ES"\n        ],\n        "anyQuery": "AvailabilityThresholdType=\'MORE_THAN\' AND AvailabilityThreshold=10",\n        "fileFormat" : "gzip"\n      }\n    }\n  ]\n}\n```\nAn example of using the SDK through a config file is located at \n\n[Example config file - 1](https://github.com/eBay/FeedSDK-Python/blob/master/sample-config/config-file-download)\n\n[Example config file - 2](https://github.com/eBay/FeedSDK-Python/blob/master/sample-config/config-file-download-filter)\n\n[Example config file - 3](https://github.com/eBay/FeedSDK-Python/blob/master/sample-config/config-file-filter)\n\n[Example config file - 4](https://github.com/eBay/FeedSDK-Python/blob/master/sample-config/config-file-query-only)\n\n### Using function calls\n\nSamples showing the usage of available operations and filters.\n\n#### Examples\nAll the examples are located [__here__](https://github.com/eBay/FeedSDK-Python/tree/master/examples)\n[Download and filter by config request](https://github.com/eBay/FeedSDK-Python/blob/master/examples/config_examples.py)\n\n\n---\n## Performance\n|  Category | Type  | Size gz  |  Size unzipped |  Records | Applied Filters | Filter Time | Loading Time | Save Time\n|---|---|---|---|---|---|---|---|---|\n| 11450 | BOOTSTRAP | 4.66 GB | 89.51 GB | 63.2 Million | PriceValue, AvailabilityThresholdType, AvailabilityThreshold | ~ 7 min | ~ 98 min | ~ 2 min\n| 220 | BOOTSTRAP | 867.8 MB | 4.26 GB | 3.3 Million | price, AvailabilityThresholdType, AvailabilityThreshold | ~ 18 sec | ~ 5 min | ~ 37 sec\n| 1281 | BOOTSTRAP | 118.4 MB |  1.06 GB | 812558 | item locations, AcceptedPaymentMethods | ~ 24 sec | ~ 1.2 min | ~ 1.8 min\n| 11232 | BOOTSTRAP | 102.5 MB | 499.9 MB | 405268 | epids, inferredEpids | ~ 0.3 sec | ~ 37 sec | ~ 0.003 sec\n| 550 | BOOTSTRAP | 60.7 MB | 986.5 MB | 1000795 | price, sellers, item locations | ~ 4 sec | ~ 1.4 min | ~ 0.1 sec\n| 260 | BOOTSTRAP | 2.3 MB | 15.6 MB | 24100 | price, AvailabilityThresholdType, AvailabilityThreshold | ~ 0.01 sec | ~ 2 sec | ~ 0.4 sec\n| 220 | DAILY | 13.5 MB | 60.4 MB | 55047 | price, leaf categories, item locations | ~ 0.08 sec | ~ 4 sec | ~ 0.007 sec\n\n\n---\n## Important notes \n\n* Ensure there is enough storage for feed files.\n* Ensure that the file storage directories have appropriate write permissions.\n* In case of failure in downloading due to network issues, the process needs to start again. There is no capability at the moment, to resume.\n\n# License\nCopyright (c) 2018-2019 eBay Inc.\n\nUse of this source code is governed by an Apache 2.0 license that can be found in the LICENSE file or at https://opensource.org/licenses/Apache-2.0.\n',
    'author': 'Lars Erler',
    'author_email': 'lars@xaospage.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/taxaos/FeedSDK-Python',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7.1,<4.0.0',
}


setup(**setup_kwargs)
