import pytest

from df_and_order.df_transform import DfTransformConfig, TRANSFORM_IN_MEMORY_KEY, TRANSFORM_PERMANENT_KEY
from df_and_order.df_transform_step import DfTransformStepConfig, TRANSFORM_STEP_MODULE_PATH_KEY, \
    TRANSFORM_STEP_PARAMS_KEY


def test_properties(mocker):
    transform_id = 'trans_id'
    in_memory_steps = mocker.Mock()
    permanent_steps = mocker.Mock()

    transform = DfTransformConfig(transform_id=transform_id,
                               in_memory_steps=in_memory_steps,
                               permanent_steps=permanent_steps)

    assert transform.transform_id == transform_id
    assert transform.in_memory_steps == in_memory_steps
    assert transform.permanent_steps == permanent_steps


def test_to_dict():
    transform_id = 'trans_id'
    in_memory_steps = [
        DfTransformStepConfig(module_path='inmem1', params={'a': 1}),
        DfTransformStepConfig(module_path='inmem2', params={'b': 2}),
    ]
    permanent_steps = [
        DfTransformStepConfig(module_path='perm3', params={}),
        DfTransformStepConfig(module_path='perm4', params={'c': 3}),
    ]

    transform = DfTransformConfig(transform_id=transform_id,
                                  in_memory_steps=in_memory_steps,
                                  permanent_steps=permanent_steps)

    res_transform_id, res_transform_dict = transform.to_dict()

    test_transform_dict = {
        TRANSFORM_IN_MEMORY_KEY: [
            {
                TRANSFORM_STEP_MODULE_PATH_KEY: 'inmem1',
                TRANSFORM_STEP_PARAMS_KEY: {'a': 1}
            },
            {
                TRANSFORM_STEP_MODULE_PATH_KEY: 'inmem2',
                TRANSFORM_STEP_PARAMS_KEY: {'b': 2}
            }
        ],
        TRANSFORM_PERMANENT_KEY: [
            {
                TRANSFORM_STEP_MODULE_PATH_KEY: 'perm3',
            },
            {
                TRANSFORM_STEP_MODULE_PATH_KEY: 'perm4',
                TRANSFORM_STEP_PARAMS_KEY: {'c': 3}
            }
        ]
    }

    assert res_transform_id == transform_id
    assert res_transform_dict == test_transform_dict


def test_from_dict(mocker):
    in_memory_steps = [
        {
            TRANSFORM_STEP_MODULE_PATH_KEY: 'inmem1',
            TRANSFORM_STEP_PARAMS_KEY: {'a': 1}
        },
        {
            TRANSFORM_STEP_MODULE_PATH_KEY: 'inmem2',
            TRANSFORM_STEP_PARAMS_KEY: {'b': 2}
        }
    ]
    permanent_steps = [
        {
            TRANSFORM_STEP_MODULE_PATH_KEY: 'perm3',
        },
        {
            TRANSFORM_STEP_MODULE_PATH_KEY: 'perm4',
            TRANSFORM_STEP_PARAMS_KEY: {'c': 3}
        }
    ]
    transform_dict = {
        TRANSFORM_IN_MEMORY_KEY: in_memory_steps,
        TRANSFORM_PERMANENT_KEY: permanent_steps
    }

    transform_id = 'trans_id'
    transform = DfTransformConfig.from_dict(transform_id=transform_id,
                                            transform_dict=transform_dict)

    for original_dict, step in zip(in_memory_steps, transform.in_memory_steps):
        assert original_dict[TRANSFORM_STEP_MODULE_PATH_KEY] == step.module_path
        assert original_dict[TRANSFORM_STEP_PARAMS_KEY] == step.params

    for original_dict, step in zip(permanent_steps, transform.permanent_steps):
        assert original_dict[TRANSFORM_STEP_MODULE_PATH_KEY] == step.module_path
        if original_dict.get(TRANSFORM_STEP_PARAMS_KEY):
            assert original_dict[TRANSFORM_STEP_PARAMS_KEY] == step.params

