import functools
import logging
import re
from abc import ABC, abstractmethod
from typing import Sequence, List, Union, Callable, Any, Dict, TYPE_CHECKING, Optional, Hashable

import numpy as np
import pandas as pd

from . import util, data_transformation
from .columngen import ColumnGenerator
from .data_transformation import DFTNormalisation, DFTFromFeatureGenerator, DataFrameTransformer
from .util import flattenArguments
from .util.string import orRegexGroup, ToStringMixin, listString
from .util.typing import PandasNamedTuple

if TYPE_CHECKING:
    from .vector_model import VectorModel

log = logging.getLogger(__name__)


class DuplicateColumnNamesException(Exception):
    pass


class FeatureGenerator(ToStringMixin, ABC):
    """
    Base class for feature generators that create a new DataFrame containing feature values
    from an input DataFrame
    """
    def __init__(self, categoricalFeatureNames: Optional[Union[Sequence[str], str]] = None,
                 normalisationRules: Sequence[data_transformation.DFTNormalisation.Rule] = (),
                 normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None, addCategoricalDefaultRules=True):
        """
        :param categoricalFeatureNames: either a sequence of column names or a regex that is to match all categorical feature names
            (which must not only work for the feature generated by this feature generator, i.e. it should not match feature names generated
            by other feature generators).
            It will be ensured that the respective columns in the generated data frames will have dtype 'category'.
            Furthermore, presence of meta-information can later be leveraged for further transformations, e.g. one-hot encoding.
        :param normalisationRules: Rules to be used by DFTNormalisation (e.g. for constructing an input transformer for a model).
            These rules are only relevant if a DFTNormalisation object consuming them is instantiated and used
            within a data processing pipeline. They do not affect feature generation.
        :param normalisationRuleTemplate: This parameter can be supplied instead of normalisationRules for the case where
            there shall be a single rule that applies to all columns generated by this feature generator that were not labeled as categorical.
        :param addCategoricalDefaultRules:
            If True, normalisation rules for categorical features (which are unsupported by normalisation) and their corresponding one-hot
            encoded features (with "_<index>" appended) will be added.
        """
        # NOTE: While it would be more elegant to not have all of the above constructor arguments and instead provide
        # them later using "with*" methods, this would have the significant drawback that it would enable
        # all such attributes to be provided in all subclasses, even in ones where we know settings exactly
        # and can provide them directly in the subclass constructor implementation. Thus it would enable
        # non-sensical settings which should be avoided.
        if len(normalisationRules) > 0 and normalisationRuleTemplate is not None:
            raise ValueError(f"normalisationRules should be empty when a normalisationRuleTemplate is provided")

        self._generatedColumnNames = None
        self.__categoricalFeatureNames = categoricalFeatureNames

        if type(categoricalFeatureNames) == str:
            categoricalFeatureNameRegex = categoricalFeatureNames
        else:
            if categoricalFeatureNames is not None and len(categoricalFeatureNames) > 0:
                categoricalFeatureNameRegex = orRegexGroup(categoricalFeatureNames)
            else:
                categoricalFeatureNameRegex = None
        self._categoricalFeatureNameRegex: str = categoricalFeatureNameRegex
        self._categoricalFeatureRules = []

        if normalisationRuleTemplate is not None:
            self._normalisationRules = [normalisationRuleTemplate.toPlaceholderRule()]  # placeholder rule's regex will be set in generate
            self._mustUpdateNormalisationRuleBasedOnColumnNames = True
        else:
            self._normalisationRules = list(normalisationRules)
            self._mustUpdateNormalisationRuleBasedOnColumnNames = False

        if addCategoricalDefaultRules:
            if categoricalFeatureNameRegex is not None:
                self._categoricalFeatureRules.append(data_transformation.DFTNormalisation.Rule(categoricalFeatureNameRegex, unsupported=True))
                self._categoricalFeatureRules.append(data_transformation.DFTNormalisation.Rule(categoricalFeatureNameRegex + r"_\d+", skip=True))  # rule for one-hot transformation

        self._name = None
        self._isFitted = False

    # for backwards compatibility with persisted Featuregens based on code prior to commit 7088cbbe
    # They lack the __isFitted attribute and we assume that each such Featuregen was fitted
    def __setstate__(self, d):
        d["_isFitted"] = d.get("_isFitted", True)
        self.__dict__ = d

    def _toStringExcludePrivate(self) -> bool:
        return True

    def _toStringAdditionalEntries(self) -> Dict[str, Any]:
        return dict(name=self.getName())

    def getName(self):
        """
        :return: the name of this feature generator, which may be a default name if the name has not been set. Note that feature generators created
            by a FeatureGeneratorFactory always get the name with which the generator factory was registered.
        """
        if self._name is None:
            return f"{self.__class__.__name__}-{id(self)}"
        return self._name

    def setName(self, name):
        self._name = name

    def getNames(self) -> list:
        """
        :return: the list of names of feature generators; will be a list with a single name for a regular feature generator
        """
        return [self.getName()]

    def info(self):
        return {
            "name": self.getName(),
            "categoricalFeatureNames": self.__categoricalFeatureNames,
            "generatedColumnNames": self.getGeneratedColumnNames(),
            "isFitted": self.isFitted(),
            "normalisationRules": self.getNormalisationRules(),
        }

    def getNormalisationRules(self, includeGeneratedCategoricalRules=True) -> List[data_transformation.DFTNormalisation.Rule]:
        if includeGeneratedCategoricalRules:
            return self._normalisationRules + self._categoricalFeatureRules
        else:
            return self._normalisationRules

    def getCategoricalFeatureNameRegex(self) -> Optional[str]:
        return self._categoricalFeatureNameRegex

    def isCategoricalFeature(self, featureName):
        if self._categoricalFeatureNameRegex is None:
            return False
        return re.fullmatch(self._categoricalFeatureNameRegex, featureName) is not None

    def getGeneratedColumnNames(self) -> Optional[List[str]]:
        """
        :return: Column names of the data frame generated by the most recent call of the feature generators 'generate' method.
            Returns None if generate was never called.
        """
        return self._generatedColumnNames

    def toDFT(self):
        return DFTFromFeatureGenerator(self)

    @abstractmethod
    def _fit(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None):
        """
        Fits the feature generator based on the given data

        :param X: the input/features data frame for the learning problem
        :param Y: the corresponding output data frame for the learning problem
            (which will typically contain regression or classification target columns)
        :param ctx: a context object whose functionality may be required for feature generation;
            this is typically the model instance that this feature generator is to generate inputs for
        """
        pass

    def fit(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None):
        """
        Fits the feature generator based on the given data

        :param X: the input/features data frame for the learning problem
        :param Y: the corresponding output data frame for the learning problem
            (which will typically contain regression or classification target columns)
        :param ctx: a context object whose functionality may be required for feature generation;
            this is typically the model instance that this feature generator is to generate inputs for
        """
        log.debug(f"Fitting {self}")
        self._fit(X, Y=Y, ctx=ctx)
        self._isFitted = True

    def isFitted(self):
        return self._isFitted

    def generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        """
        Generates features for the data points in the given data frame

        :param df: the input data frame for which to generate features
        :param ctx: a context object whose functionality may be required for feature generation;
            this is typically the model instance that this feature generator is to generate inputs for
        :return: a data frame containing the generated features, which uses the same index as X (and Y)
        """
        if not self.isFitted():
            raise Exception(f"Cannot generate features from a FeatureGenerator which is not fitted: "
                            f"the feature generator {self.getName()} requires fitting")

        log.debug(f"Generating features with {self}")
        resultDF = self._generate(df, ctx=ctx)

        isColumnDuplicatedArray = resultDF.columns.duplicated()
        if any(isColumnDuplicatedArray):
            duplicatedColumns = set(resultDF.columns[isColumnDuplicatedArray])
            raise DuplicateColumnNamesException(f"Feature data frame contains duplicate column names: {duplicatedColumns}")

        # ensure that categorical columns have dtype 'category'
        categoricalFeatureNames = []
        if self._categoricalFeatureNameRegex is not None:
            resultDF = resultDF.copy()  # resultDF we got might be a view of some other DF, so before we modify it, we must copy it
            categoricalFeatureNames = [col for col in resultDF.columns if self.isCategoricalFeature(col)]
            for colName in categoricalFeatureNames:
                series = resultDF[colName].copy()
                if series.dtype.name != 'category':
                    resultDF[colName] = series.astype('category', copy=False)

        self._generatedColumnNames = resultDF.columns

        # finalise normalisation rule template (if any) by making it apply to all non-categorical features
        # (a default rule applies to categorical features)
        if self._mustUpdateNormalisationRuleBasedOnColumnNames:
            nonCategoricalFeatures = list(set(self._generatedColumnNames).difference(categoricalFeatureNames))
            # NOTE: We here update the existing rule which was instantiated with a dummy regex because
            # some mechanisms (e.g. MultiFeatureGenerators) retrieve rule instances early on (before generate
            # is ever called) and therefore updating an existing rule is the safe route and should always
            # work, because rules should never actually be applied before generate has indeed been called
            self._normalisationRules[0].setRegex(orRegexGroup(nonCategoricalFeatures))
            self._mustUpdateNormalisationRuleBasedOnColumnNames = False

        return resultDF

    @abstractmethod
    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        """
        Generates features for the data points in the given data frame.

        :param df: the input data frame for which to generate features
        :param ctx: a context object whose functionality may be required for feature generation;
            this is typically the model instance that this feature generator is to generate inputs for
        :return: a data frame containing the generated features, which uses the same index as ``df``.
            The data frame's columns holding categorical columns are not required to have dtype ``category``;
            this will be ensured by the encapsulating call as long as the respective columns' names
            were appropriately provided at construction.
        """
        pass

    def fitGenerate(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None) -> pd.DataFrame:
        """
        Fits the feature generator and subsequently generates features for the data points in the given data frame

        :param X: the input data frame for the learning problem and for which to generate features
        :param Y: the corresponding output data frame for the learning problem
            (which will typically contain regression or classification target columns)
        :param ctx: a context object whose functionality may be required for feature generation;
            this is typically the model instance that this feature generator is to generate inputs for
        :return: a data frame containing the generated features, which uses the same index as X (and Y)
        """
        self.fit(X, Y, ctx)
        return self.generate(X, ctx)

    def flattened(self, columnsToFlatten: List[str] = None,
            normalisationRules=(),
            normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None,
            keepOtherColumns=True) -> "ChainedFeatureGenerator":
        """
        Returns a new feature generator which returns flattened versions of one or more of the vector-valued columns generated
        by this feature generator.

        :param columnsToFlatten: the list of columns to flatten; if None, flatten all columns
        :param normalisationRules: a list of normalisation rules which apply to the flattened columns
        :param normalisationRuleTemplate: a normalisation rule template which applies to all generated flattened columns
        :param keepOtherColumns: if True, any additional columns that are not to be flattened are to be retained
            by the returned feature generator; if False, additional columns are to be discarded
        :return: a feature generator which generates the flattened columns
        """
        return flattenedFeatureGenerator(self, columnsToFlatten=columnsToFlatten, normalisationRules=normalisationRules,
            keepOtherColumns=keepOtherColumns, normalisationRuleTemplate=normalisationRuleTemplate)

    def concat(self, *others: "FeatureGenerator") -> "MultiFeatureGenerator":
        """
        Concatenates this feature generator with one or more other feature generator in order to produce a feature generator that
        jointly generates all features

        :param others: other feature generators
        :return: a :class:`MultiFeatureGenerator`
        """
        if isinstance(self, MultiFeatureGenerator):
            fgens = list(self.featureGenerators)
        else:
            fgens = [self]
        fgens.extend(others)
        return MultiFeatureGenerator(fgens)

    def chain(self, *others: "FeatureGenerator") -> "ChainedFeatureGenerator":
        """
        Chains this feature generator with one or more other feature generators such that each feature generator
        receives as input the output of the preceding feature generator. The resulting feature generator
        produces the features of the last element in the chain.

        :param others: other feature generator
        :return: a :class:`ChainedFeatureGenerator`
        """
        if isinstance(self, ChainedFeatureGenerator):
            fgens = self.featureGenerators
        else:
            fgens = [self]
        fgens.extend(others)
        return ChainedFeatureGenerator(fgens)


class RuleBasedFeatureGenerator(FeatureGenerator, ABC):
    """
    A feature generator which does not require fitting
    """
    def fit(self, X, Y=None, ctx=None):
        pass

    def _fit(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None):
        pass

    def isFitted(self):
        return True


class MultiFeatureGenerator(FeatureGenerator):
    """
    Wrapper for multiple feature generators. Calling generate here applies all given feature generators independently and
    returns the concatenation of their outputs
    """
    def __init__(self, *featureGenerators: Union[FeatureGenerator, List[FeatureGenerator]]):
        self.featureGenerators = featureGenerators = flattenArguments(featureGenerators)
        if len(self.featureGenerators) == 0:
            log.debug("Creating an empty MultiFeatureGenerator. It will generate a data frame without columns.")
        categoricalFeatureNameRegexes = [regex for regex in [fg.getCategoricalFeatureNameRegex() for fg in featureGenerators] if regex is not None]
        if len(categoricalFeatureNameRegexes) > 0:
            categoricalFeatureNames = "|".join(categoricalFeatureNameRegexes)
        else:
            categoricalFeatureNames = ()
        normalisationRules = util.concatSequences([fg.getNormalisationRules() for fg in featureGenerators])
        super().__init__(categoricalFeatureNames=categoricalFeatureNames, normalisationRules=normalisationRules,
            addCategoricalDefaultRules=False)

    def _toStringObjectInfo(self) -> str:
        return f"featureGenerators={listString(self.featureGenerators)}"

    def _generateFromMultiple(self, generateFeatures: Callable[[FeatureGenerator], pd.DataFrame], index) -> pd.DataFrame:
        dfs = []
        for fg in self.featureGenerators:
            df = generateFeatures(fg)
            dfs.append(df)
        if len(dfs) == 0:
            return pd.DataFrame(index=index)
        else:
            combinedDF = pd.concat(dfs, axis=1)
            if len(combinedDF.columns) != len(set(combinedDF.columns)):
                raise Exception(f"At least one column was generated more than once: {list(combinedDF.columns)}; check feature generators for correctness!")
            return combinedDF

    def _generate(self, inputDF: pd.DataFrame, ctx=None):
        def generateFeatures(fg: FeatureGenerator):
            return fg.generate(inputDF, ctx=ctx)
        return self._generateFromMultiple(generateFeatures, inputDF.index)

    def fitGenerate(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None) -> pd.DataFrame:
        log.debug(f"Fitting and generating features with {self}")

        def generateFeatures(fg: FeatureGenerator):
            return fg.fitGenerate(X, Y, ctx)

        return self._generateFromMultiple(generateFeatures, X.index)

    def _fit(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None):
        for fg in self.featureGenerators:
            fg.fit(X, Y)

    def isFitted(self):
        return all([fg.isFitted() for fg in self.featureGenerators])

    def info(self):
        info = super(MultiFeatureGenerator, self).info()
        info["featureGeneratorNames"] = self.getNames()
        return info

    def getNames(self) -> list:
        return functools.reduce(lambda x, y: x + y, [fg.getNames() for fg in self.featureGenerators], [])


class FeatureGeneratorFromNamedTuples(FeatureGenerator, ABC):
    """
    Generates feature values for one data point at a time, creating a dictionary with
    feature values from each named tuple
    """
    def __init__(self, cache: util.cache.PersistentKeyValueCache = None, categoricalFeatureNames: Sequence[str] = (),
                 normalisationRules: Sequence[data_transformation.DFTNormalisation.Rule] = (),
                 normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None):
        super().__init__(categoricalFeatureNames=categoricalFeatureNames, normalisationRules=normalisationRules, normalisationRuleTemplate=normalisationRuleTemplate)
        self.cache = cache

    def _generate(self, df: pd.DataFrame, ctx=None):
        dicts = []
        for idx, nt in enumerate(df.itertuples()):
            nt: PandasNamedTuple
            if idx % 100 == 0:
                log.debug(f"Generating feature via {self.__class__.__name__} for index {idx}")
            value = None
            if self.cache is not None:
                value = self.cache.get(nt.Index)
            if value is None:
                value = self._generateFeatureDict(nt)
                if self.cache is not None:
                    self.cache.set(nt.Index, value)
            dicts.append(value)
        return pd.DataFrame(dicts, index=df.index)

    @abstractmethod
    def _generateFeatureDict(self, namedTuple) -> Dict[str, Any]:
        """
        Creates a dictionary with feature values from a named tuple

        :param namedTuple: the data point for which to generate features
        :return: a dictionary mapping feature names to values
        """
        pass


class FeatureGeneratorTakeColumns(RuleBasedFeatureGenerator):
    def __init__(self, columns: Union[str, List[str]] = None, exceptColumns: Sequence[str] = (),
            categoricalFeatureNames: Optional[Union[Sequence[str], str]] = (),
            normalisationRules: Sequence[data_transformation.DFTNormalisation.Rule] = (),
            normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None,
            verifyColumnNames=True):
        """
        :param columns: name of the column or list of names of columns to be taken. If None, all columns will be taken.
        :param exceptColumns: list of names of columns to not take if present in the input df
        :param categoricalFeatureNames: either a sequence of column names or a regex that is to match all categorical feature names
            (which must not only work for the feature generated by this feature generator, i.e. it should not match feature names generated
            by other feature generators).
            It will be ensured that the respective columns in the generated data frames will have dtype 'category'.
            Furthermore, presence of meta-information can later be leveraged for further transformations, e.g. one-hot encoding.
        :param normalisationRules: Rules to be used by DFTNormalisation (e.g. for constructing an input transformer for a model).
            These rules are only relevant if a DFTNormalisation object consuming them is instantiated and used
            within a data processing pipeline. They do not affect feature generation.
        :param normalisationRuleTemplate: This parameter can be supplied instead of normalisationRules for the case where
            there shall be a single rule that applies to all columns generated by this feature generator that were not labeled as categorical.
        :param verifyColumnNames: if True and columns to take were specified, will raise an error in case said columns
            are missing during feature generation. If False, will log on info level instead
        """
        super().__init__(categoricalFeatureNames=categoricalFeatureNames, normalisationRules=normalisationRules, normalisationRuleTemplate=normalisationRuleTemplate)
        if isinstance(columns, str):
            columns = [columns]
        self.columns = columns
        self.exceptColumns = exceptColumns
        self.verifyColumnNames = verifyColumnNames

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        columnsToTake = self.columns if self.columns is not None else df.columns
        columnsToTake = [col for col in columnsToTake if col not in self.exceptColumns]

        if self.columns is not None:
            missingCols = set(columnsToTake).difference(df.columns)
            if len(missingCols) > 0:
                missing_cols_notification = f"Columns {missingCols} were specified but are not present in data frame. " \
                               f"verifyColumnNames was set to {self.verifyColumnNames}; " \
                               f"available columns: {list(df.columns)}"
                if self.verifyColumnNames:
                    raise RuntimeError(missing_cols_notification)
                log.info(missing_cols_notification)
        return df[columnsToTake]

    def info(self):
        info = super().info()
        info["columns"] = self.columns
        info["exceptColumns"] = self.exceptColumns
        return info


class FeatureGeneratorFlattenColumns(RuleBasedFeatureGenerator):
    """
    Instances of this class take columns with vectors and creates a data frame with columns containing entries of
    these vectors.

    For example, if columns "vec1", "vec2" contain vectors of dimensions dim1, dim2, a data frame with dim1+dim2 new columns
    will be created. It will contain the columns "vec1_<i1>", "vec2_<i2>" with i1, i2 ranging in (0, dim1), (0, dim2).

    """
    def __init__(self, columns: Optional[Union[str, Sequence[str]]] = None, categoricalFeatureNames: Sequence[str] = (),
                 normalisationRules: Sequence[data_transformation.DFTNormalisation.Rule] = (),
                 normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None):
        """

        :param columns: name of the column or list of names of columns to be flattened. If None, all columns will be flattened.
        :param categoricalFeatureNames:
        :param normalisationRules:
        :param normalisationRuleTemplate:
        """
        super().__init__(categoricalFeatureNames=categoricalFeatureNames, normalisationRules=normalisationRules, normalisationRuleTemplate=normalisationRuleTemplate)
        if isinstance(columns, str):
            columns = [columns]
        self.columns = columns

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        resultDf = pd.DataFrame(index=df.index)
        columnsToFlatten = self.columns if self.columns is not None else df.columns
        for col in columnsToFlatten:
            log.debug(f"Flattening column {col}")
            # NOTE: we found the use of np.stack to produce the most runtime-efficient results.
            # Other variants, e.g. based on lists instead of numpy.arrays, perform much worse.
            values = np.stack(df[col].values)
            if len(values.shape) != 2:
                raise ValueError(f"Column {col} was expected to contain one dimensional vectors, something went wrong")
            dimension = values.shape[1]
            new_columns = [f"{col}_{i}" for i in range(dimension)]
            log.debug(f"Flattening resulted in {len(new_columns)} new columns")
            resultDf[new_columns] = pd.DataFrame(values, index=df.index)
        return resultDf

    def info(self):
        info = super().info()
        info["columns"] = self.columns
        return info


class FeatureGeneratorFromColumnGenerator(RuleBasedFeatureGenerator):
    """
    Implements a feature generator via a column generator
    """
    log = log.getChild(__qualname__)

    def __init__(self, columnGen: ColumnGenerator, takeInputColumnIfPresent=False, isCategorical=False,
            normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None):
        """
        :param columnGen: the underlying column generator
        :param takeInputColumnIfPresent: if True, then if a column whose name corresponds to the column to generate exists
            in the input data, simply copy it to generate the output (without using the column generator); if False, always
            apply the columnGen to generate the output
        :param isCategorical: whether the resulting column is categorical
        :param normalisationRuleTemplate: template for a DFTNormalisation for the resulting column.
            This should only be provided if isCategorical is False
        """
        if isCategorical and normalisationRuleTemplate is not None:
            raise ValueError(f"normalisationRuleTemplate should be None when the generated column is categorical")

        categoricalFeatureNames = (columnGen.generatedColumnName, ) if isCategorical else ()
        super().__init__(categoricalFeatureNames=categoricalFeatureNames, normalisationRuleTemplate=normalisationRuleTemplate)

        self.takeInputColumnIfPresent = takeInputColumnIfPresent
        self.columnGen = columnGen

    def info(self):
        info = super().info()
        info["takeInputColumnIfPresent"] = self.takeInputColumnIfPresent
        info["generatedColName"] = self.columnGen.generatedColumnName
        return info

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        colName = self.columnGen.generatedColumnName
        if self.takeInputColumnIfPresent and colName in df.columns:
            self.log.debug(f"Taking column '{colName}' from input data frame")
            series = df[colName]
        else:
            self.log.debug(f"Generating column '{colName}' via {self.columnGen}")
            series = self.columnGen.generateColumn(df)
        return pd.DataFrame({colName: series})


class ChainedFeatureGenerator(FeatureGenerator):
    """
    Chains feature generators such that they are executed one after another. The output of generator i>=1 is the input of
    generator i+1 in the generator sequence.
    """
    def __init__(self, *featureGenerators: Union[FeatureGenerator, List[FeatureGenerator]]):
        """
        :param featureGenerators: feature generators to apply in order; the properties of the last feature generator
            determine the relevant meta-data such as categorical feature names and normalisation rules
        """
        self.featureGenerators = flattenArguments(featureGenerators)
        if len(featureGenerators) == 0:
            raise ValueError("Empty list of feature generators")
        lastFG: FeatureGenerator = self.featureGenerators[-1]
        super().__init__(categoricalFeatureNames=lastFG.getCategoricalFeatureNameRegex(), normalisationRules=lastFG.getNormalisationRules(),
            addCategoricalDefaultRules=False)

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        for featureGen in self.featureGenerators:
            df = featureGen.generate(df, ctx)
        return df

    def _fit(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None):
        self.fitGenerate(X, Y, ctx)

    def isFitted(self):
        return all([fg.isFitted() for fg in self.featureGenerators])

    def info(self):
        info = super().info()
        info["chainedFeatureGeneratorNames"] = self.getNames()

    def fitGenerate(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None) -> pd.DataFrame:
        log.debug(f"Fitting and generating features with {self}")
        for fg in self.featureGenerators:
            X = fg.fitGenerate(X, Y, ctx)
        return X


class FeatureGeneratorTargetDistribution(FeatureGenerator):
    """
    A feature generator, which, for a column T (typically the categorical target column of a classification problem
    or the continuous target column of a regression problem),

    * can ensure that T takes on limited set of values t_1, ..., t_n by allowing the user to apply
      binning using given bin boundaries
    * computes for each value c of a categorical column C the conditional empirical distribution
      P(T | C=c) in the training data during the training phase,
    * generates, for each requested column C and value c in the column, n features
      '<C>_<T>_distribution_<t_i>' = P(T=t_i | C=c) if flatten=True
      or one feature '<C>_<T>_distribution' = [P(T=t_i | C=c), ..., P(T=t_n | C=c)] if flatten=False

    Being probability values, the features generated by this feature generator are already normalised.
    """
    def __init__(self, columns: Union[str, Sequence[str]], targetColumn: str,
            targetColumnBins: Optional[Union[Sequence[float], int, pd.IntervalIndex]], targetColumnInFeaturesDf=False,
            flatten=True):
        """
        :param columns: the categorical columns for which to generate distribution features
        :param targetColumn: the column the distributions over which will make up the features.
            If targetColumnBins is not None, this column will be discretised before computing the conditional distributions
        :param targetColumnBins: if not None, specifies the binning to apply via pandas.cut
            (see https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.cut.html).
            Note that if a value should match no bin, NaN will generated. To avoid this when specifying bin boundaries in a list,
            -inf and +inf should be used as the first and last entries.
        :param targetColumnInFeaturesDf: if True, when fitting will look for targetColumn in the features data frame (X) instead of in target data frame (Y)
        :param flatten: whether to generate a separate scalar feature per distribution value rather than one feature
            with all of the distribution's values
        """
        self.flatten = flatten
        if isinstance(columns, str):
            columns = [columns]
        self.columns = columns
        self.targetColumn = targetColumn
        self.targetColumnInFeaturesDf = targetColumnInFeaturesDf
        self.targetColumnBins = targetColumnBins
        if self.flatten:
            normalisationRuleTemplate = data_transformation.DFTNormalisation.RuleTemplate(skip=True)
        else:
            normalisationRuleTemplate = data_transformation.DFTNormalisation.RuleTemplate(unsupported=True)
        super().__init__(normalisationRuleTemplate=normalisationRuleTemplate)
        self._targetColumnValues = None
        # This will hold the mapping: column -> featureValue -> targetValue -> targetValueEmpiricalProbability
        self._discreteTargetDistributionsByColumn: Optional[Dict[str, Dict[Any, Dict[Any, float]]]] = None

    def info(self):
        info = super().info()
        info["columns"] = self.columns
        info["targetColumn"] = self.targetColumn
        info["targetColumnBins"] = self.targetColumnBins
        info["flatten"] = self.flatten
        return info

    def _fit(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None):
        """
        This will persist the empirical target probability distributions for all unique values in the specified columns
        """
        if self.targetColumnInFeaturesDf:
            target = X[self.targetColumn]
        else:
            target = Y[self.targetColumn]
        if self.targetColumnBins is not None:
            discretisedTarget = pd.cut(target, self.targetColumnBins)
        else:
            discretisedTarget = target
        self._targetColumnValues = discretisedTarget.unique()

        self._discreteTargetDistributionsByColumn = {}
        for column in self.columns:
            self._discreteTargetDistributionsByColumn[column] = {}
            columnTargetDf = pd.DataFrame()
            columnTargetDf[column] = X[column]
            columnTargetDf["target"] = discretisedTarget.values
            for value, valueTargetsDf in columnTargetDf.groupby(column):
                # The normalized value_counts contain targetValue -> targetValueEmpiricalProbability for the current value
                self._discreteTargetDistributionsByColumn[column][value] = valueTargetsDf["target"].value_counts(normalize=True).to_dict()

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        if self._discreteTargetDistributionsByColumn is None:
            raise Exception("Feature generator has not been fitted")
        resultDf = pd.DataFrame(index=df.index)
        for column in self.columns:
            targetDistributionByValue = self._discreteTargetDistributionsByColumn[column]
            if self.flatten:
                for targetValue in self._targetColumnValues:
                    # Important: pd.Series.apply should not be used here, as it would label the resulting column as categorical
                    resultDf[f"{column}_{self.targetColumn}_distribution_{targetValue}"] = [targetDistributionByValue[value].get(targetValue, 0.0) for value in df[column]]
            else:
                distributions = [[targetDistributionByValue[value].get(targetValue, 0.0) for targetValue in self._targetColumnValues] for value in df[column]]
                resultDf[f"{column}_{self.targetColumn}_distribution"] = pd.Series(distributions, index=df[column].index)
        return resultDf


################################
#
# generator registry
#
################################


class FeatureGeneratorRegistry:
    """
    Represents a registry for named feature generators which can be instantiated via factories.

    In addition to functions registerFactory and getFeatureGenerator, feature generators can be registered and retrieved via \n
    registry.<name> = <featureGeneratorFactory> \n
    registry.<name> \n

    Example:
        >>> from sensai.featuregen import FeatureGeneratorRegistry, FeatureGeneratorTakeColumns
        >>> import pandas as pd

        >>> df = pd.DataFrame({"foo": [1, 2, 3], "bar": [7, 8, 9]})
        >>> registry = FeatureGeneratorRegistry()
        >>> registry.testFgen = lambda: FeatureGeneratorTakeColumns("foo")
        >>> registry.testFgen().generate(df)
           foo
        0    1
        1    2
        2    3
    """
    def __init__(self, useSingletons=False):
        """
        :param useSingletons: if True, internally maintain feature generator singletons, such that there is at most one
            instance for each name
        """
        # Important: Don't set public members in init. Since we override setattr this would lead to undesired consequences
        self._featureGeneratorFactories: Dict[Hashable, Callable[[], FeatureGenerator]] = {}
        self._featureGeneratorSingletons: Dict[Hashable, Callable[[], FeatureGenerator]] = {}
        self._useSingletons = useSingletons

    def __setattr__(self, name: str, value):
        if not name.startswith("_"):
            self.registerFactory(name, value)
        else:
            super().__setattr__(name, value)

    def __getattr__(self, item: str):
        factory = self._featureGeneratorFactories.get(item)
        if factory is not None:
            return factory
        else:
            raise AttributeError(item)

    @property
    def availableFeatures(self):
        return list(self._featureGeneratorFactories.keys())

    @staticmethod
    def _name(name: Hashable):
        # for enums, which have .name, use the name only, because it is less problematic to persist
        if hasattr(name, "name"):
            name = name.name
        return name

    def registerFactory(self, name: Hashable, factory: Callable[[], FeatureGenerator]):
        """
        Registers a feature generator factory which can subsequently be referenced by models via their name
        :param name: the name (which can, in particular, be a string or an enum item)
        :param factory: the factory
        """
        name = self._name(name)
        if name in self._featureGeneratorFactories:
            raise ValueError(f"Generator for name '{name}' already registered")
        self._featureGeneratorFactories[name] = factory

    def getFeatureGenerator(self, name) -> FeatureGenerator:
        """
        Creates a feature generator from a name, which must have been previously registered.
        The name of the returned feature generator (as returned by getName()) is set to name.

        :param name: the name (which can, in particular, be a string or an enum item)
        :return: a new feature generator instance (or existing instance for the case where useSingletons is enabled)
        """
        name = self._name(name)
        generator = self._featureGeneratorSingletons.get(name)
        if generator is None:
            factory = self._featureGeneratorFactories.get(name)
            if factory is None:
                raise ValueError(f"No factory registered for name '{name}': known names: {listString(self._featureGeneratorFactories.keys())}. Use registerFeatureGeneratorFactory to register a new feature generator factory.")
            generator = factory()
            generator.setName(name)
            if self._useSingletons:
                self._featureGeneratorSingletons[name] = generator
        return generator


class FeatureCollector(object):
    """
    A feature collector which can provide a multi-feature generator from a list of names/generators and registry
    """

    def __init__(self, *featureGeneratorsOrNames: Union[str, FeatureGenerator], registry:
            FeatureGeneratorRegistry = None):
        """
        :param featureGeneratorsOrNames: generator names (known to the registry) or generator instances.
        :param registry: the feature generator registry for the case where names are passed
        """
        self._featureGeneratorsOrNames = featureGeneratorsOrNames
        self._registry = registry
        self._multiFeatureGenerator = self._createMultiFeatureGenerator()

    def getMultiFeatureGenerator(self) -> MultiFeatureGenerator:
        return self._multiFeatureGenerator

    def getNormalisationRules(self, includeGeneratedCategoricalRules=True):
        return self.getMultiFeatureGenerator().getNormalisationRules(
            includeGeneratedCategoricalRules=includeGeneratedCategoricalRules)

    def getNormalizationRules(self, includeGeneratedCategoricalRules=True):  # for backward compatibility
        return self.getNormalisationRules(includeGeneratedCategoricalRules=includeGeneratedCategoricalRules)

    def getCategoricalFeatureNameRegex(self) -> str:
        return self.getMultiFeatureGenerator().getCategoricalFeatureNameRegex()

    def _createMultiFeatureGenerator(self):
        featureGenerators = []
        for f in self._featureGeneratorsOrNames:
            if isinstance(f, FeatureGenerator):
                featureGenerators.append(f)
            else:
                if self._registry is None:
                    raise Exception(f"Received feature name '{f}' instead of instance but no registry to perform the lookup")
                featureGenerators.append(self._registry.getFeatureGenerator(f))
        return MultiFeatureGenerator(*featureGenerators)


class FeatureGeneratorFromVectorModel(FeatureGenerator):
    def __init__(self, vectorModel: "VectorModel", targetFeatureGenerator: FeatureGenerator, categoricalFeatureNames: Sequence[str] = (),
                 normalisationRules: Sequence[data_transformation.DFTNormalisation.Rule] = (),
                 normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None,
                 inputFeatureGenerator: FeatureGenerator = None, useTargetFeatureGeneratorForTraining=False):
        """
        Provides a feature via predictions of a given model

        :param vectorModel: model used for generate features from predictions
        :param targetFeatureGenerator: generator for target to be predicted
        :param categoricalFeatureNames:
        :param normalisationRules:
        :param normalisationRuleTemplate:
        :param inputFeatureGenerator: optional feature generator to be applied to input of vectorModel's fit and predict
        :param useTargetFeatureGeneratorForTraining: if False, this generator will always apply the model
            to generate features.
            If True, this generator will use targetFeatureGenerator to generate features, bypassing the
            model. This is useful for the case where the model which is
            to receive the generated features shall be trained on the original targets rather than the predictions
            thereof.
        """
        super().__init__(categoricalFeatureNames=categoricalFeatureNames, normalisationRules=normalisationRules, normalisationRuleTemplate=normalisationRuleTemplate)

        self.useTargetFeatureGeneratorForTraining = useTargetFeatureGeneratorForTraining
        self.targetFeatureGenerator = targetFeatureGenerator
        self.inputFeatureGenerator = inputFeatureGenerator
        self.useTargetFeatureGeneratorForTraining = useTargetFeatureGeneratorForTraining
        self.vectorModel = vectorModel

    def _fit(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None):
        targetDF = self.targetFeatureGenerator.fitGenerate(X, Y)
        if self.inputFeatureGenerator:
            X = self.inputFeatureGenerator.fitGenerate(X, Y)
        self.vectorModel.fit(X, targetDF)

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        if self.inputFeatureGenerator:
            df = self.inputFeatureGenerator.generate(df)
        if self.useTargetFeatureGeneratorForTraining and not ctx.isFitted():
            log.debug(f"Using targetFeatureGenerator {self.targetFeatureGenerator.__class__.__name__} to generate target features")
            return self.targetFeatureGenerator.generate(df)
        else:
            log.debug(f"Generating target features via {self.vectorModel.__class__.__name__}")
            return self.vectorModel.predict(df)

    def info(self):
        info = super().info()
        info["wrappedModel"] = str(self.vectorModel)
        return info


class FeatureGeneratorMapColumn(RuleBasedFeatureGenerator, ABC):
    """
    Creates a single feature from a single input column by applying a function to each element of the input column
    """
    def __init__(self, inputColName: str, featureColName: str, categoricalFeatureNames: Optional[Union[Sequence[str], str]] = None,
            normalisationRules: Sequence[data_transformation.DFTNormalisation.Rule] = (),
            normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None, addCategoricalDefaultRules=True):
        super().__init__(categoricalFeatureNames=categoricalFeatureNames, normalisationRules=normalisationRules,
            normalisationRuleTemplate=normalisationRuleTemplate, addCategoricalDefaultRules=addCategoricalDefaultRules)
        self._inputColName = inputColName
        self._featureColName = featureColName

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        if self._inputColName not in df.columns:
            raise ValueError(f"Column '{self._inputColName}' required by feature generator not found in list of columns: {list(df.columns)}")
        inputSeries = df[self._inputColName]
        values = inputSeries.apply(self._createValue)
        return pd.DataFrame({self._featureColName: values}, index=df.index)

    @abstractmethod
    def _createValue(self, value):
        """
        Maps a value from the input column to a feature value

        :param value: a value from the input column
        :return: the feature value
        """
        pass


class FeatureGeneratorMapColumnDict(RuleBasedFeatureGenerator, ABC):
    """
    Creates an arbitrary number of features from a single input column by applying a function to each element of the input column
    """
    def __init__(self, inputColName: str, categoricalFeatureNames: Optional[Union[Sequence[str], str]] = None,
            normalisationRules: Sequence[data_transformation.DFTNormalisation.Rule] = (),
            normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None, addCategoricalDefaultRules=True):
        super().__init__(categoricalFeatureNames=categoricalFeatureNames, normalisationRules=normalisationRules,
            normalisationRuleTemplate=normalisationRuleTemplate, addCategoricalDefaultRules=addCategoricalDefaultRules)
        self._inputColName = inputColName

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        if self._inputColName not in df.columns:
            raise ValueError(f"Column '{self._inputColName}' required by feature generator not found in list of columns: {list(df.columns)}")
        inputSeries = df[self._inputColName]
        values = [self._createFeaturesDict(v) for v in inputSeries]
        return pd.DataFrame(values, index=df.index)

    @abstractmethod
    def _createFeaturesDict(self, value) -> Dict[str, Any]:
        """
        Maps a value from the input column to a dictionary containing one or more features.

        :param value: a value from the input column
        :return: a dictionary mapping feature names to values
        """
        pass


class FeatureGeneratorNAMarker(RuleBasedFeatureGenerator):
    """
    Creates features indicating whether another feature is N/A (not available).
    It can be practical to use this feature generator in conjunction with DFTFillNA for models that cannot handle missing values.
    """
    def __init__(self, columns: List[str], valueA=0, valueNA=1):
        """
        Note: When changing the default values used, use only values that are considered to be normalised when using this
        feature generation in a context where DFTNormalisation is used (no normalisation is applied to features generated
        by this feature generator).

        :param columns: the columns for which to generate
        :param valueA: the feature value if the input feature is available
        :param valueNA: the feature value if the input feature is not available
        """
        super().__init__(normalisationRuleTemplate=DFTNormalisation.RuleTemplate(skip=True))
        self.columns = columns
        self.valueA = valueA
        self.valueNA = valueNA

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        newCols = {}
        valueMap = {True: self.valueNA, False: self.valueA}
        for col in self.columns:
            newCols[f"{col}_na"] = [valueMap[isNA] for isNA in df[col].isna()]
        return pd.DataFrame(newCols, index=df.index)


def flattenedFeatureGenerator(fgen: FeatureGenerator, columnsToFlatten: List[str] = None, keepOtherColumns=True,
        normalisationRules: Sequence[DFTNormalisation.Rule] = (),
        normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None):
    """
    Return a flattening version of the input feature generator.

    :param fgen: the feature generator which generates columns that are to be flattened
    :param columnsToFlatten: list of names of output columns to be flattened; if None, flatten all columns
    :param keepOtherColumns: whether any additional columns that are not to be flattened are to be retained
        by the returned feature generator
    :param normalisationRules: additional normalisation rules for the flattened output columns
    :param normalisationRuleTemplate: This parameter can be supplied instead of normalisationRules for the case where
        there shall be a single rule that applies to all flattened output columns
    :return: FeatureGenerator instance that will generate flattened versions of the specified columns and leave
        all other output columns as is.
    
    Example:
        >>> from sensai.featuregen import FeatureGeneratorTakeColumns, flattenedFeatureGenerator
        >>> import pandas as pd
        >>>
        >>> df = pd.DataFrame({"foo": [[1, 2], [3, 4]], "bar": ["a", "b"]})
        >>> fgen = flattenedFeatureGenerator(FeatureGeneratorTakeColumns(), columnsToFlatten=["foo"])
        >>> fgen.generate(df)
           foo_0  foo_1 bar
        0      1      2   a
        1      3      4   b
    """
    flatteningGenerator = FeatureGeneratorFlattenColumns(columns=columnsToFlatten, normalisationRules=normalisationRules,
        normalisationRuleTemplate=normalisationRuleTemplate)
    if columnsToFlatten is None or not keepOtherColumns:
        return ChainedFeatureGenerator(fgen, flatteningGenerator)
    else:
        return ChainedFeatureGenerator(fgen,
            MultiFeatureGenerator(flatteningGenerator, FeatureGeneratorTakeColumns(exceptColumns=columnsToFlatten)))


class FeatureGeneratorFromDFT(FeatureGenerator):
    def __init__(self, dft: DataFrameTransformer, categoricalFeatureNames: Optional[Union[Sequence[str], str]] = None,
            normalisationRules: Sequence[data_transformation.DFTNormalisation.Rule] = (),
            normalisationRuleTemplate: data_transformation.DFTNormalisation.RuleTemplate = None,
            addCategoricalDefaultRules=True):
        super().__init__(categoricalFeatureNames=categoricalFeatureNames, normalisationRules=normalisationRules,
            normalisationRuleTemplate=normalisationRuleTemplate, addCategoricalDefaultRules=addCategoricalDefaultRules)
        self.dft = dft

    def _fit(self, X: pd.DataFrame, Y: pd.DataFrame = None, ctx=None):
        self.dft.fit(X)

    def _generate(self, df: pd.DataFrame, ctx=None) -> pd.DataFrame:
        return self.dft.apply(df)
