"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitDependencyCheck = void 0;
const core_1 = require("@aws-cdk/core");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cli_1 = require("./cli");
class CodeCommitDependencyCheck extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { schedule, computeType, repository, preCheckCommand = 'echo "No preCheckCommand!"', version = '5.3.2', projectName, failOnCVSS = 0, paths = ['.'], excludes, suppressions, enableExperimental, reportsBucket, } = props;
        const { repositoryName, repositoryCloneUrlHttp, repositoryArn, } = repository;
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_2_0;
        const dependencyCheck = `dependency-check-${version}-release`;
        const cli = new cli_1.Cli();
        this.checkProject = new aws_codebuild_1.Project(this, 'CheckProject', {
            cache: aws_codebuild_1.Cache.local(aws_codebuild_1.LocalCacheMode.CUSTOM),
            environment: {
                buildImage,
                computeType: computeType || buildImage.defaultComputeType,
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    'git-credential-helper': 'yes',
                },
                phases: {
                    install: {
                        commands: [
                            'echo "[===== Install OWASP Dependency Check =====]"',
                            'wget -O public-key.asc https://bintray.com/user/downloadSubjectPublicKey?username=jeremy-long',
                            'gpg --keyid-format long --list-options show-keyring public-key.asc',
                            'gpg --import public-key.asc',
                            `wget https://dl.bintray.com/jeremy-long/owasp/${dependencyCheck}.zip`,
                            `wget https://dl.bintray.com/jeremy-long/owasp/${dependencyCheck}.zip.asc`,
                            `gpg --verify ${dependencyCheck}.zip.asc ${dependencyCheck}.zip`,
                            `unzip ${dependencyCheck}.zip -d /opt`,
                            'chmod +x /opt/dependency-check/bin/dependency-check.sh',
                            'export PATH="$PATH:/opt/dependency-check/bin"',
                        ],
                    },
                    pre_build: {
                        commands: [
                            `echo "[===== Clone repository: ${repositoryName} =====]"`,
                            `git clone "${repositoryCloneUrlHttp}"`,
                            `cd ${repositoryName}`,
                            `${preCheckCommand}`,
                            'SHA=$(git rev-parse HEAD)',
                            'cd ${CODEBUILD_SRC_DIR}',
                        ],
                    },
                    build: {
                        commands: [
                            `echo "[===== Scan repository: ${repositoryName} =====]"`,
                            'echo "[===== SHA: $SHA =====]"',
                            'mkdir reports',
                            cli.version(),
                            cli.scan({
                                projectName: projectName || repositoryName,
                                basedir: repositoryName,
                                paths,
                                failOnCVSS,
                                enableExperimental,
                                suppressions,
                                excludes,
                            }),
                        ],
                        finally: [
                            'echo "[===== Upload reports =====]"',
                            "dt=$(date -u '+%Y_%m_%d_%H_%M')",
                            reportsBucket
                                ? `aws s3 cp reports/dependency-check-report.html s3://${reportsBucket.bucketName}/${repositoryName}/\${dt}_UTC/`
                                : 'echo "No reportsBuckets"',
                        ],
                    },
                },
                reports: {
                    dependencyCheckReport: {
                        files: ['reports/dependency-check-junit.xml'],
                    },
                },
                cache: {
                    paths: ['/opt/dependency-check/data/**/*'],
                },
            }),
        });
        this.checkProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [repositoryArn],
            actions: [
                'codecommit:BatchGet*',
                'codecommit:Get*',
                'codecommit:Describe*',
                'codecommit:List*',
                'codecommit:GitPull',
            ],
        }));
        if (reportsBucket) {
            reportsBucket.grantWrite(this.checkProject);
        }
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule,
            targets: [new aws_events_targets_1.CodeBuildProject(this.checkProject)],
        });
    }
    /**
     * Defines an event rule which triggers when a check fails.
     */
    onCheckFailed(id, options) {
        return this.checkProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a check starts.
     */
    onCheckStarted(id, options) {
        return this.checkProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a check complets successfully.
     */
    onCheckSucceeded(id, options) {
        return this.checkProject.onBuildSucceeded(id, options);
    }
}
exports.CodeCommitDependencyCheck = CodeCommitDependencyCheck;
//# sourceMappingURL=data:application/json;base64,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